-- $Id: maths-literaltovalue.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Maths)
procedure LiteralToValue (Str    : in     LexTokenManager.LexString;
                          Num    :    out Value;
                          Ok     :    out ErrorCode)
is
   DecimalPointFound,
   ExponentFound,
   BaseFound          :  Boolean;
   Base               :  Natural;
   InputString,
   CoreString,
   ExpString          :  EStrings.T;
   ExpSign            :  Character;
   PlacesAfterPoint   :  EStrings.Lengths;
   LegalSyntax        :  Boolean;
   Exponent           :  Integer;
   NumLocal           :  Value;
   OkLocal            :  ErrorCode;

   ---------------------------------------------------------

   procedure MakeInteger
      --# global in     Base;
      --#        in     BaseFound;
      --#        in     CoreString;
      --#        in     Exponent;
      --#        in     ExponentFound;
      --#        in out NumLocal;
      --#        in out OkLocal;
      --# derives NumLocal from *,
      --#                       Base,
      --#                       BaseFound,
      --#                       CoreString,
      --#                       Exponent,
      --#                       ExponentFound &
      --#         OkLocal  from *,
      --#                       CoreString,
      --#                       Exponent,
      --#                       ExponentFound;
   is
      CoreLocal  : EStrings.T;
      PaddedOk   : Boolean;

      -------------------------------

      procedure PadStringWithZeros (Str : in out EStrings.T;
                                    By  : in     Natural;
                                    Ok  :    out Boolean)
         --# derives Ok,
         --#         Str from By,
         --#                  Str;
      is
      begin
         if (Str.Length + By) < EStrings.MaxStringLength then
            Ok := True;
            for i in Integer range 1 .. By loop
               Str.Content (i + Str.Length) := '0';
            end loop;
            Str.Length := Str.Length + By;
         else
            Ok := False;
         end if;
      end PadStringWithZeros;

      -------------------------------

   begin -- MakeInteger
      NumLocal.Sort := IntegerValue;
      CoreLocal := CoreString;
      if ExponentFound then
         PadStringWithZeros (CoreLocal, Exponent, PaddedOk);
      else
         PaddedOk := True;
      end if;
      if PaddedOk then
         if BaseFound then
            NumLocal.Numerator   := StripLeadingZeros (StringToPart (Base, CoreLocal));
         else
            NumLocal.Numerator   := StripLeadingZeros (DecStringToPart (CoreLocal));
         end if;
      else
         NumLocal := NoValue;
         OkLocal  := OverFlow;
      end if;
   end MakeInteger;

   ---------------------------------------------------------

   procedure MakeReal
      --# global in     CoreString;
      --#        in     Exponent;
      --#        in     PlacesAfterPoint;
      --#        in out NumLocal;
      --# derives NumLocal from *,
      --#                       CoreString,
      --#                       Exponent,
      --#                       PlacesAfterPoint;
   is
      ExpLocal : Integer;
      DenPart,
      NumPart  : Part;

   begin
      ExpLocal      := Exponent;
      NumLocal.Sort := RealValue;
      NumPart       := DecStringToPart (CoreString);
      DenPart       := OnePart;
      ExpLocal := ExpLocal - PlacesAfterPoint;
      if ExpLocal > 0 then
         for i in Integer range 1 .. ExpLocal loop
            NumPart := ShiftUpPart (NumPart);
         end loop;
      elsif ExpLocal < 0 then
         ExpLocal := -ExpLocal;
         for i in Integer range 1 .. ExpLocal loop
            DenPart := ShiftUpPart (DenPart);
         end loop;
         --else do nothing about exponent
      end if;
      NumLocal.Numerator   := NumPart;
      NumLocal.Denominator := DenPart;
      Normalize (NumLocal);
   end MakeReal;

   ---------------------------------------------------------

begin -- LiteralToValue
   if Str = LexTokenManager.NullString then
      Num := NoValue;
      OkLocal  := NoError;
   else
      NumLocal := NoValue;  --must return something if error
      LexTokenManager.LexStringToString (Str, InputString);
      ParseString (InputString,
                   DecimalPointFound,
                   ExponentFound,
                   BaseFound,
                   Base,
                   CoreString,
                   ExpString,
                   ExpSign,
                   PlacesAfterPoint,
                   LegalSyntax);

      if LegalSyntax then
         OkLocal := NoError;
         Exponent := StringToNatural (ExpString);
         if ExpSign = '-' then
            Exponent := -Exponent;
         end if;
         if DecimalPointFound then
            MakeReal;
         else
            MakeInteger;
         end if;
      else
         OkLocal  := IllegalValue;
      end if;

      OkLocal  := OverflowCheck (OkLocal, NumLocal);

      if OkLocal = NoError then
         Num := NumLocal;
      else
         Num := NoValue;
      end if;
   end if;
   Ok := OkLocal;
end LiteralToValue;
