-- $Id: mainloop.adb 13146 2009-04-24 17:27:57Z Trevor Jennings $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with SPARK_IO,
  ErrorHandler,
  Error_Types,
  ContextManager,
  EStrings,
  File_Utils,
  FileSystem,
  LexTokenManager,
  Dictionary,
  STree,
  SparkLex,
  CommandLineData,
  SPParser,
  LexTokenLists,
  IndexManager,
  Sem,
  RequiredUnits,
  ScreenEcho,
  Statistics,
  MetaFile,
  ConfigFile,
  SparkHTML,
  VCG,
  XMLReport;

use type SPARK_IO.File_Status;
use type SPARK_IO.File_Type;
use type LexTokenManager.LexString;
use type ErrorHandler.ErrorLevel;
use type ContextManager.UnitDescriptors;
use type ContextManager.FileDescriptors;
use type ContextManager.UnitStatus;
use type ContextManager.FileStatus;
use type ContextManager.UnitTypes;
use type CommandLineData.RuleGenerationPolicies;

package body MainLoop
is

   procedure PrintFileName (File  : in SPARK_IO.File_Type;
                            Name  : in EStrings.T;
                            Plain : in Boolean)
   --  all-from-all derives for data flow analysis only
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                Name,
   --#                                Plain;
   is
      Stat : FileSystem.TypFileSpecStatus;
      FullName : EStrings.T;
   begin
      --# accept F, 10, Stat, "Stat not used" &
      --#        F, 33, Stat, "Stat not used";
      if Plain then
         FullName := EStrings.LowerCase (FileSystem.JustFile (Name, True));
      else
         FileSystem.FindFullFileName (Name, Stat, FullName);
      end if;

      if File = SPARK_IO.Standard_Output then
         ScreenEcho.Put_ExaminerString (FullName);
      else
         EStrings.PutString (File, FullName);
      end if;
   end PrintFileName; -- Stat deliberately unused

   procedure InitialiseProcessing
   --# global in     CommandLineData.Content;
   --#        in out LexTokenManager.StringTable;
   --#        in out SparkHTML.GenerateHTML;
   --#        in out SPARK_IO.File_Sys;
   --#           out Dictionary.Dict;
   --#           out SparkHTML.HTMLWorkDir;
   --#           out SparkHTML.SparkWorkDir;
   --#           out SparkLex.CurrLine;
   --#           out XMLReport.State;
   --# derives Dictionary.Dict,
   --#         SparkHTML.HTMLWorkDir,
   --#         SparkHTML.SparkWorkDir,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          LexTokenManager.StringTable,
   --#                                          SPARK_IO.File_Sys &
   --#         LexTokenManager.StringTable from *,
   --#                                          CommandLineData.Content &
   --#         SparkHTML.GenerateHTML      from *,
   --#                                          CommandLineData.Content,
   --#                                          LexTokenManager.StringTable,
   --#                                          SPARK_IO.File_Sys &
   --#         SparkLex.CurrLine,
   --#         XMLReport.State             from ;
   is
   begin
      SparkLex.ClearLineContext;
      LexTokenManager.InitialiseStringTable;
      Dictionary.Initialize (CommandLineData.Content.WriteDict);
      --# accept F, 602, SparkHTML.HTMLWorkDir, SparkHTML.HTMLWorkDir, "Only used in HTML selected" &
      --#        F, 602, SparkHTML.SparkWorkDir, SparkHTML.SparkWorkDir, "Only used in HTML selected";
      if CommandLineData.Content.HTML then
         SparkHTML.InitSparkHTML;
      end if;
      XMLReport.Init;
   end InitialiseProcessing;

   procedure OpenFile (FileDescriptor : in ContextManager.FileDescriptors)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ContextManager.FileHeap;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SparkLex.CurrLine;
   --#        in out SPARK_IO.File_Sys;
   --# derives ContextManager.FileHeap,
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        ContextManager.FileHeap,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FileDescriptor,
   --#                                        SPARK_IO.File_Sys &
   --#         SparkLex.CurrLine         from *,
   --#                                        ContextManager.FileHeap,
   --#                                        FileDescriptor,
   --#                                        SPARK_IO.File_Sys;
   is
      Status         : SPARK_IO.File_Status;
      FileContext    : SparkLex.LineContext;
      SourceFile     : SPARK_IO.File_Type := SPARK_IO.Null_File;
      SourceFileName : EStrings.T;
      ErrorContext   : ErrorHandler.ErrorContexts;
   begin
      ContextManager.GetSourceFileName (FileDescriptor, SourceFileName);
      FileSystem.OpenSourceFile (SourceFile, SourceFileName, Status);

      if Status = SPARK_IO.Ok then
         ContextManager.SetSourceFile (FileDescriptor, SourceFile);
         SparkLex.ClearLineContext;
         SparkLex.StoreLineContext (FileContext);
         ContextManager.SetLineContext (FileDescriptor, FileContext);
         ErrorHandler.ErrorInit (SourceFileName, CommandLineData.Content.Echo);
         ErrorHandler.GetErrorContext (ErrorContext);
         ContextManager.SetErrorContext (FileDescriptor, ErrorContext);
         ContextManager.SetFileStatus (FileDescriptor, ContextManager.FileOpen);
      else
         ScreenEcho.Put_String ("Cannot open file ");
         PrintFileName (SPARK_IO.Standard_Output, SourceFileName, CommandLineData.Content.PlainOutput);
         ScreenEcho.New_Line (1);
         ContextManager.SetFileStatus (FileDescriptor, ContextManager.UnableToOpen);
         ErrorHandler.SetFileOpenError;
      end if;
   end OpenFile;

   procedure CreateFileContext (SourceFileName : in     EStrings.T;
                                FileDescriptor :    out ContextManager.FileDescriptors)
   --# global in out ContextManager.FileHeap;
   --# derives ContextManager.FileHeap from *,
   --#                                      SourceFileName &
   --#         FileDescriptor          from ContextManager.FileHeap;
   is
      FD2 : ContextManager.FileDescriptors;
   begin
      ContextManager.CreateFileDescriptor (FD2);
      ContextManager.SetSourceFileName (FD2,
                                        SourceFileName);
      ContextManager.SetFileStatus (FD2,
                                    ContextManager.FileCreated);
      FileDescriptor := FD2;
   end CreateFileContext;

   procedure PrepareNextArgumentFile (TheFileName    : in     EStrings.T;
                                      DoListing      : in     Boolean;
                                      TheListingName : in     EStrings.T;
                                      FileDescriptor :    out ContextManager.FileDescriptors)
   --# global in out ContextManager.FileHeap;
   --# derives ContextManager.FileHeap from *,
   --#                                      DoListing,
   --#                                      TheFileName,
   --#                                      TheListingName &
   --#         FileDescriptor          from ContextManager.FileHeap;
   is
      FD2 : ContextManager.FileDescriptors;
   begin
      CreateFileContext (TheFileName,  FD2);
      if DoListing then
         ContextManager.SetListingReq (FD2, True);
         ContextManager.SetListingFileName (FD2, TheListingName);
      end if;
      FileDescriptor := FD2;
   end PrepareNextArgumentFile;

   procedure ShowEndOfFile (ToFile : in SPARK_IO.File_Type)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                ToFile;
   is
   begin

      SPARK_IO.New_Line (ToFile, 2);
      SPARK_IO.Put_Line (ToFile,
                         "--End of file--------------------------------------------------",
                         0);
   end ShowEndOfFile;

   procedure EchoWarningOrNote (AboutFile : in EStrings.T;
                                Msg       : in String)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                AboutFile,
   --#                                CommandLineData.Content,
   --#                                Msg;
   is
   begin
      if CommandLineData.Content.Echo then
         if CommandLineData.Content.Brief then
            -- Echo the first filename from the command-line.  If it starts with an '@'
            -- then chop it off, so the file is recognized by the IDE.
            if AboutFile.Length >= 2 and then
              AboutFile.Content (1) = '@' then

               ScreenEcho.Put_ExaminerString (EStrings.Section (AboutFile, 2, AboutFile.Length - 1));
            else
               ScreenEcho.Put_ExaminerString (AboutFile);
            end if;

            ScreenEcho.Put_String (":1:1: ");
            ScreenEcho.Put_Line (Msg);
         else
            ScreenEcho.New_Line (1);
            ScreenEcho.Put_Line (Msg);
         end if;
      end if;
   end EchoWarningOrNote;


   procedure OutputDictionaryFile
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in out Dictionary.Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict   from *,
   --#                                CommandLineData.Content,
   --#                                SPARK_IO.File_Sys &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.StringTable;
   is
      status : SPARK_IO.File_Status;
      DictMessage  : constant EStrings.T :=
         EStrings.T'
         (26,
          EStrings.Contents'
          ('G', 'e', 'n', 'e', 'r', 'a', 't', 'i', 'n', 'g', ' ',
           'd', 'i', 'c', 't', 'i', 'o', 'n', 'a', 'r', 'y', ' ',
           'f', 'i', 'l', 'e',  others => ' '));
   begin
      if CommandLineData.Content.WriteDict then
         if CommandLineData.Content.Echo and not CommandLineData.Content.Brief then
            ScreenEcho.Echo (DictMessage);
         end if;
         Dictionary.Write (CommandLineData.Content.DictFileName,
                           status);
         if status /= SPARK_IO.Ok then
            ScreenEcho.Put_Line ("Error writing dictionary file: ");
            PrintFileName (SPARK_IO.Standard_Output,
                           CommandLineData.Content.DictFileName,
                           False);
            ScreenEcho.New_Line (1);
         end if;
      end if;
   end OutputDictionaryFile;


   procedure PrintLexTokenList (File : in SPARK_IO.File_Type;
                                List : in LexTokenLists.Lists)
   --# global in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                LexTokenManager.StringTable,
   --#                                List;
   is
      pos : LexTokenLists.Lengths;
      Str : EStrings.T;
   begin
      if List.Length = 0 then
         SPARK_IO.Put_String (File, "Unexpected Empty Lextoken List", 0);
      else
         pos := 1;
         loop
            LexTokenManager.LexStringToString (List.Content (pos), Str);
            EStrings.PutString (File, Str);
            exit when pos = List.Length;
            SPARK_IO.Put_Char (File, '.');
            pos := pos + 1;
         end loop;
      end if;
   end PrintLexTokenList;


   -- Following hidden trace routines enabled by -debug=u -----------------------------------
   procedure Trace (Msg : String)
   --# derives null from Msg;
   is
      --# hide Trace;
   begin
      if CommandLineData.Content.Debug.Units then
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            Msg,
                            0);
      end if;
   end Trace;

   procedure TraceUnit (List  : in LexTokenLists.Lists;
                        Types : in ContextManager.UnitTypeSets)
   --# derives null from List,
   --#                   Types;
   is
      --# hide TraceUnit;
      FirstType : Boolean := True;
   begin
      if CommandLineData.Content.Debug.Units then
         SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                              "   unit name: ",
                              0);
         PrintLexTokenList (SPARK_IO.Standard_Output,
                            List);
         SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                              " which is {",
                              0);
         for I in ContextManager.UnitTypes loop
            if Types (I) then
               if not FirstType then
                  SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                                       ", ",
                                       0);
               end if;
               SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                                    ContextManager.UnitTypes'Image (I),
                                    0);
               FirstType := False;
            end if;
         end loop;
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            "}",
                            0);
      end if;
   end TraceUnit;
   -----------------------------------------------------------------------------------

   function GetUnitName (List : in LexTokenLists.Lists) return EStrings.T
   --# global in LexTokenManager.StringTable;
   is
      pos : LexTokenLists.Lengths;
      Str : EStrings.T;
      ReturnString : EStrings.T;
   begin
      ReturnString := EStrings.EmptyString;

      if List.Length = 0 then
         EStrings.AppendString (ReturnString, "Unexpected Empty Lextoken List");
      else
         pos := 1;
         loop
            LexTokenManager.LexStringToString (List.Content (pos), Str);
            EStrings.AppendExaminerString (ReturnString, Str);
            exit when pos = List.Length;
            EStrings.AppendString (ReturnString, ".");
            pos := pos + 1;
         end loop;
      end if;
      return ReturnString;
   end GetUnitName;

   function GetLexTokenList (List : in LexTokenLists.Lists) return EStrings.T
   --# global in LexTokenManager.StringTable;
   is
      pos : LexTokenLists.Lengths;
      Str : EStrings.T;
      ReturnStr : EStrings.T;
   begin
      pos := 1;
      ReturnStr := EStrings.EmptyString;
      if List.Length > 0 then
         loop
            LexTokenManager.LexStringToString (List.Content (pos), Str);
            EStrings.AppendExaminerString (ReturnStr, Str);
            exit when pos = List.Length;
            EStrings.AppendString (ReturnStr, ".");
            pos := pos + 1;
         end loop;
      end if;
      return ReturnStr;
   end GetLexTokenList;


   procedure PrintUnitType (File     : in SPARK_IO.File_Type;
                            UnitType : in ContextManager.UnitTypes)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                UnitType;
   is
   begin
      case UnitType is
         when ContextManager.MainProgram =>
            SPARK_IO.Put_String (File, "main program", 0);
         when ContextManager.PackageOrGenericDeclaration =>
            SPARK_IO.Put_String (File, "package or generic specification", 0);
         when ContextManager.PackageSpecification =>
            SPARK_IO.Put_String (File, "package specification", 0);
         when ContextManager.PackageBody =>
            SPARK_IO.Put_String (File, "package body", 0);
         when ContextManager.SubUnit =>
            SPARK_IO.Put_String (File, "subunit", 0);
         when ContextManager.GenericDeclaration =>
            SPARK_IO.Put_String (File, "generic declaration", 0);
         when ContextManager.GenericPackageInstantiation =>
            SPARK_IO.Put_String (File, "generic package instantiation", 0);
         when ContextManager.GenericSubprogramBody =>
            SPARK_IO.Put_String (File, "generic subprogram body", 0);
         when ContextManager.InterUnitPragma =>
            null;
         when ContextManager.InvalidUnit =>
            null;
      end case;
   end PrintUnitType;


   function GetUnitType (UnitType : in ContextManager.UnitTypes) return EStrings.T
   is
      ReturnStr : EStrings.T;
   begin
      case UnitType is
         when ContextManager.MainProgram =>
            ReturnStr := XMLReport.XStr ("main program");
         when ContextManager.PackageOrGenericDeclaration =>
            ReturnStr := XMLReport.XStr ("package or generic specification");
         when ContextManager.PackageSpecification =>
            ReturnStr := XMLReport.XStr ("package specification");
         when ContextManager.PackageBody =>
            ReturnStr := XMLReport.XStr ("package body");
         when ContextManager.SubUnit =>
            ReturnStr := XMLReport.XStr ("subunit");
         when ContextManager.GenericDeclaration =>
            ReturnStr := XMLReport.XStr ("generic declaration");
         when ContextManager.GenericPackageInstantiation =>
            ReturnStr := XMLReport.XStr ("generic package instantiation");
         when ContextManager.GenericSubprogramBody =>
            ReturnStr := XMLReport.XStr ("generic subprogram body");
         when ContextManager.InterUnitPragma =>
            ReturnStr := XMLReport.XStr ("InterUnitPragma");
         when ContextManager.InvalidUnit =>
            ReturnStr := XMLReport.XStr ("InvalidUnit");
      end case;
      return ReturnStr;
   end GetUnitType;

   procedure PrintReason (File       : in SPARK_IO.File_Type;
                          UnitStatus : in ContextManager.UnitStatus)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                UnitStatus;
   is
   begin
      case UnitStatus is
         when ContextManager.NoIndexFile =>
            SPARK_IO.Put_String (File, "No index file specified.", 0);
         when ContextManager.NotInIndexFile =>
            SPARK_IO.Put_String (File, "Omitted from index file.", 0);
         when ContextManager.CannotOpenFile =>
            SPARK_IO.Put_String (File, "Cannot open source file specified in index file.", 0);
         when ContextManager.UnableToLocate =>
            SPARK_IO.Put_String (File, "Declaration not found in source code.", 0);
         when others =>
            null;  -- should not happen because only called when unit is not found
      end case;
   end PrintReason;

   -- This function is not currently used but will be required when
   -- XMLReport is updated to contain the reason
   function GetReason (UnitStatus : in ContextManager.UnitStatus) return EStrings.T
   is
      ReturnStr : EStrings.T;
   begin
      case UnitStatus is
         when ContextManager.NoIndexFile =>
            ReturnStr := XMLReport.XStr ("No index file specified.");
         when ContextManager.NotInIndexFile =>
            ReturnStr := XMLReport.XStr ("Omitted from index file.");
         when ContextManager.CannotOpenFile =>
            ReturnStr := XMLReport.XStr ("Cannot open source file specified in index file.");
         when ContextManager.UnableToLocate =>
            ReturnStr := XMLReport.XStr ("Declaration not found in source code.");
         when others =>
            ReturnStr := EStrings.EmptyString;  -- only called when unit is not found
      end case;
      return ReturnStr;
   end GetReason;


   procedure AppendToReportFile (ReportFile     : in SPARK_IO.File_Type;
                                 Purpose        : in Error_Types.ConversionRequestSource;
                                 FileDescriptor : in ContextManager.FileDescriptors)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.UnitHeap;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out ContextManager.FileHeap;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --#        in out XMLReport.State;
   --# derives ContextManager.FileHeap,
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys,
   --#         XMLReport.State           from CommandLineData.Content,
   --#                                        ContextManager.FileHeap,
   --#                                        ContextManager.UnitHeap,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FileDescriptor,
   --#                                        LexTokenManager.StringTable,
   --#                                        Purpose,
   --#                                        ReportFile,
   --#                                        SPARK_IO.File_Sys,
   --#                                        XMLReport.State;
   is
      ErrorContext : ErrorHandler.ErrorContexts;

      function GetSourceFileName (FileDescriptor : in ContextManager.FileDescriptors)
                                 return EStrings.T
      --# global in ContextManager.FileHeap;
      is
         FileName : EStrings.T;
      begin
         ContextManager.GetSourceFileName (FileDescriptor, FileName);
         return FileName;
      end GetSourceFileName;


      procedure PrintSourceFileName (ReportFile     : in SPARK_IO.File_Type;
                                     FileDescriptor : in ContextManager.FileDescriptors)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.FileHeap;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                ContextManager.FileHeap,
      --#                                FileDescriptor,
      --#                                ReportFile;
      is
         FileName : EStrings.T;
      begin
         SPARK_IO.Put_String (ReportFile, "Source Filename:   ", 0);
         ContextManager.GetSourceFileName (FileDescriptor, FileName);
         PrintFileName (ReportFile, FileName, CommandLineData.Content.PlainOutput);
         SPARK_IO.New_Line (ReportFile, 1);
      end PrintSourceFileName;

      procedure PrintListingFileName (ReportFile     : in SPARK_IO.File_Type;
                                      FileDescriptor : in ContextManager.FileDescriptors)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.FileHeap;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                ContextManager.FileHeap,
      --#                                FileDescriptor,
      --#                                ReportFile;
      is
         FileName : EStrings.T;
      begin
         if ContextManager.ListingReqt (FileDescriptor) then
            SPARK_IO.Put_String (ReportFile, "Listing Filename:  ", 0);
            ContextManager.GetListingFileName (FileDescriptor, FileName);
            CommandLineData.Normalize_FileName_To_Output_Directory (FileName);
            PrintFileName (ReportFile, FileName, CommandLineData.Content.PlainOutput);
            SPARK_IO.New_Line (ReportFile, 1);
         else
            SPARK_IO.Put_Line (ReportFile, "No Listing File", 0);
         end if;
      end PrintListingFileName;

      procedure PrintUnitStatus (File           : in SPARK_IO.File_Type;
                                 UnitDescriptor : in ContextManager.UnitDescriptors)
      --# global in     ContextManager.UnitHeap;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                ContextManager.UnitHeap,
      --#                                File,
      --#                                UnitDescriptor;
      is
      begin
         case ContextManager.GetUnitStatus (UnitDescriptor) is
            when ContextManager.NoUnitEntry =>
               null;
            when ContextManager.UnitCreated =>
               null;
            when ContextManager.UnitParsed
               | ContextManager.UnitDeferred =>
               SPARK_IO.Put_String (File, "Unit has been parsed only, " &
                                    "any errors are listed below.", 0);
            when ContextManager.UnitAnalysed =>
               SPARK_IO.Put_String (File, "Unit has been analysed, " &
                                    "any errors are listed below.", 0);
            when ContextManager.NoIndexFile  =>
               SPARK_IO.Put_String (File, "Unit required but not found in source file and " &
                                    "no index file has been specified.", 0);
            when ContextManager.NotInIndexFile =>
               SPARK_IO.Put_String (File, "Unit required but not found in source file or " &
                                    "in index file.", 0);
            when ContextManager.CannotOpenFile =>
               SPARK_IO.Put_String (File, "Source file for required Unit given " &
                                    "in index file cannot be opened.", 0);
            when ContextManager.UnableToLocate =>
               SPARK_IO.Put_String (File, "Unit required but not found.", 0);
         end case;
      end PrintUnitStatus;

      function GetUnitStatus (UnitDescriptor : in ContextManager.UnitDescriptors) return EStrings.T
      --# global in ContextManager.UnitHeap;
      is
         ReturnString : EStrings.T;
      begin
         ReturnString := EStrings.EmptyString;
         case ContextManager.GetUnitStatus (UnitDescriptor) is
            when ContextManager.NoUnitEntry =>
               null;
            when ContextManager.UnitCreated =>
               null;
            when ContextManager.UnitParsed
               | ContextManager.UnitDeferred =>
               EStrings.CopyString (ReturnString,
                                           "parsed");
            when ContextManager.UnitAnalysed =>
               EStrings.CopyString (ReturnString,
                                           "analysed");
            when ContextManager.NoIndexFile  =>
               EStrings.CopyString (ReturnString,
                                           "not found and no index file specified");
            when ContextManager.NotInIndexFile =>
               EStrings.CopyString (ReturnString,
                                           "not found in source file or " &
                                           "in index file.");
            when ContextManager.CannotOpenFile =>
               EStrings.CopyString (ReturnString,
                                           "unit source file name given " &
                                           "in index file cannot be opened.");
            when ContextManager.UnableToLocate =>
               EStrings.CopyString (ReturnString,
                                           "not found");
         end case;
         return ReturnString;
      end GetUnitStatus;

      procedure OutputUnitList (ReportFile     : in SPARK_IO.File_Type;
                                FileDescriptor : in ContextManager.FileDescriptors)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.UnitHeap;
      --#        in     LexTokenManager.StringTable;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                ContextManager.UnitHeap,
      --#                                FileDescriptor,
      --#                                LexTokenManager.StringTable,
      --#                                ReportFile,
      --#                                XMLReport.State &
      --#         XMLReport.State   from *,
      --#                                CommandLineData.Content,
      --#                                ContextManager.UnitHeap,
      --#                                FileDescriptor,
      --#                                LexTokenManager.StringTable;
      is
         UnitDescriptor       : ContextManager.UnitDescriptors;
         UnitName             : LexTokenLists.Lists;
         UnitType             : ContextManager.UnitTypes;
         StoredFileDescriptor : ContextManager.FileDescriptors;
         UnitOutput           : Boolean;
      begin
         UnitOutput := False;
         UnitDescriptor := ContextManager.FirstUnitDescriptor;
         if CommandLineData.Content.XML then
            XMLReport.StartSection (XMLReport.SUnitsInFile,
                                    ReportFile);
         end if;
         loop
            exit when UnitDescriptor = ContextManager.NullUnit;
            ContextManager.GetFileDescriptor (UnitDescriptor,
                                              StoredFileDescriptor);
            if (ContextManager.GetUnitStatus (UnitDescriptor) /=
                ContextManager.NoUnitEntry) and
               FileDescriptor = StoredFileDescriptor
            then
               ContextManager.GetUnitName (UnitDescriptor, UnitName, UnitType);
               if UnitName.Length /= 0 then
                  --# accept F, 41, "Stable expression expected";
                  if CommandLineData.Content.XML then
                     XMLReport.AdaUnit (GetUnitName (UnitName),
                                        GetUnitType (UnitType),
                                        GetUnitStatus (UnitDescriptor),
                                        ReportFile);
                     UnitOutput := True;
                  else
                     UnitOutput := True;
                     SPARK_IO.Put_String (ReportFile, "   Unit name:  ", 0);
                     PrintLexTokenList (ReportFile, UnitName);
                     SPARK_IO.New_Line (ReportFile, 1);
                     SPARK_IO.Put_String (ReportFile, "   Unit type:  ", 0);
                     PrintUnitType (ReportFile, UnitType);
                     SPARK_IO.New_Line (ReportFile, 1);
                     SPARK_IO.Put_String (ReportFile, "   ", 0);
                     PrintUnitStatus (ReportFile, UnitDescriptor);
                     SPARK_IO.New_Line (ReportFile, 2);
                  end if;
                  --# end accept;
               end if;
            end if;
            UnitDescriptor := ContextManager.NextUnitDescriptor (UnitDescriptor);
         end loop;
         if CommandLineData.Content.XML then
            XMLReport.EndSection (XMLReport.SUnitsInFile,
                                  ReportFile);
         end if;
         if not UnitOutput then
            SPARK_IO.Put_Line (ReportFile, "***     No units in file", 0);
            SPARK_IO.New_Line (ReportFile, 2);
         end if;
      end OutputUnitList;

   begin -- AppendToReportFile
      SPARK_IO.New_Line (ReportFile, 1);
      if not CommandLineData.Content.XML then
         PrintSourceFileName (ReportFile, FileDescriptor);
      end if;
      if ContextManager.GetFileStatus (FileDescriptor) = ContextManager.UnableToOpen then
         if CommandLineData.Content.XML then
            XMLReport.StartFile (GetSourceFileName (FileDescriptor),
                                 ReportFile);
            OutputUnitList (ReportFile, FileDescriptor);
            XMLReport.EndFile (ReportFile);
         else
            SPARK_IO.New_Line (ReportFile, 1);
            OutputUnitList (ReportFile, FileDescriptor);
            SPARK_IO.Put_Line (ReportFile, "***     Unable to open source file", 0);
         end if;
      else
         if CommandLineData.Content.XML then
            XMLReport.StartFile (GetSourceFileName (FileDescriptor),
                                 ReportFile);
            OutputUnitList (ReportFile, FileDescriptor);
            ContextManager.GetErrorContext (FileDescriptor, ErrorContext);
            ErrorHandler.SetErrorContext (ErrorContext);
            ErrorHandler.AppendErrors (ReportFile,
                                       Purpose);
            ErrorHandler.GetErrorContext (ErrorContext);
            ContextManager.SetErrorContext (FileDescriptor, ErrorContext);
            XMLReport.EndFile (ReportFile);
         else
            PrintListingFileName (ReportFile, FileDescriptor);
            SPARK_IO.New_Line (ReportFile, 1);
            OutputUnitList (ReportFile, FileDescriptor);
            ContextManager.GetErrorContext (FileDescriptor, ErrorContext);
            ErrorHandler.SetErrorContext (ErrorContext);
            ErrorHandler.AppendErrors (ReportFile,
                                       Purpose);
            ErrorHandler.GetErrorContext (ErrorContext);
            ContextManager.SetErrorContext (FileDescriptor, ErrorContext);
         end if;
      end if;
   end AppendToReportFile;



   procedure OutputReportFile (ReportFileErrorList : in out SPARK_IO.File_Type)
   --# global in     CommandLineData.Content;
   --#        in     ConfigFile.State;
   --#        in     ContextManager.UnitHeap;
   --#        in     Dictionary.Dict;
   --#        in     IndexManager.IndexTable;
   --#        in     SparkHTML.HTMLWorkDir;
   --#        in     SparkHTML.SparkWorkDir;
   --#        in     Statistics.TableUsage;
   --#        in out ContextManager.FileHeap;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SparkHTML.GenerateHTML;
   --#        in out SPARK_IO.File_Sys;
   --#        in out XMLReport.State;
   --# derives ContextManager.FileHeap,
   --#         ErrorHandler.ErrorContext,
   --#         XMLReport.State             from CommandLineData.Content,
   --#                                          ConfigFile.State,
   --#                                          ContextManager.FileHeap,
   --#                                          ContextManager.UnitHeap,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          IndexManager.IndexTable,
   --#                                          LexTokenManager.StringTable,
   --#                                          SPARK_IO.File_Sys,
   --#                                          XMLReport.State &
   --#         LexTokenManager.StringTable from *,
   --#                                          CommandLineData.Content,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          IndexManager.IndexTable,
   --#                                          SPARK_IO.File_Sys,
   --#                                          XMLReport.State &
   --#         ReportFileErrorList         from *,
   --#                                          CommandLineData.Content,
   --#                                          SPARK_IO.File_Sys &
   --#         SparkHTML.GenerateHTML      from *,
   --#                                          CommandLineData.Content,
   --#                                          ConfigFile.State,
   --#                                          ContextManager.FileHeap,
   --#                                          ContextManager.UnitHeap,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          IndexManager.IndexTable,
   --#                                          LexTokenManager.StringTable,
   --#                                          ReportFileErrorList,
   --#                                          SPARK_IO.File_Sys,
   --#                                          Statistics.TableUsage,
   --#                                          XMLReport.State &
   --#         SPARK_IO.File_Sys           from *,
   --#                                          CommandLineData.Content,
   --#                                          ConfigFile.State,
   --#                                          ContextManager.FileHeap,
   --#                                          ContextManager.UnitHeap,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          IndexManager.IndexTable,
   --#                                          LexTokenManager.StringTable,
   --#                                          ReportFileErrorList,
   --#                                          SparkHTML.GenerateHTML,
   --#                                          SparkHTML.HTMLWorkDir,
   --#                                          SparkHTML.SparkWorkDir,
   --#                                          Statistics.TableUsage,
   --#                                          XMLReport.State;
   is
      ReportFile     : SPARK_IO.File_Type;
      OK             : Boolean;

      ReportMessage  : constant EStrings.T :=
         EStrings.T'
         (22,
          EStrings.Contents'('G', 'e', 'n', 'e', 'r', 'a', 't', 'i', 'n', 'g', ' ',
                                    'r', 'e', 'p', 'o', 'r', 't', ' ',
                                    'f', 'i', 'l', 'e',  others => ' '));


      procedure CreateReportFile (ReportFile : out SPARK_IO.File_Type;
                                  OK         : out Boolean)
      --# global in     CommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives OK,
      --#         ReportFile        from CommandLineData.Content,
      --#                                SPARK_IO.File_Sys &
      --#         SPARK_IO.File_Sys,
      --#         XMLReport.State   from CommandLineData.Content,
      --#                                SPARK_IO.File_Sys,
      --#                                XMLReport.State;
      is
         LocalFile : SPARK_IO.File_Type;
         Status    : SPARK_IO.File_Status;
         FileName  : EStrings.T;
      begin
         OK := True;
         LocalFile := SPARK_IO.Null_File;

         FileName := CommandLineData.Content.ReportFileName;
         CommandLineData.Normalize_FileName_To_Output_Directory (FileName);
         FileName := FileSystem.CaseOfFilesForCreate (FileName);

         SPARK_IO.Create (LocalFile,
                          FileName.Length,
                          FileName.Content,
                          "", Status);
         if Status = SPARK_IO.Ok then
            if CommandLineData.Content.XML then
               XMLReport.StartSection (XMLReport.SReport,
                                       LocalFile);
               XMLReport.StartSection (XMLReport.SPrologue,
                                       LocalFile);
            else
               File_Utils.PrintAHeader (LocalFile,
                                        "Report of SPARK Examination",
                                        File_Utils.OtherFile);
            end if;
         else
            OK := False;
            ScreenEcho.Put_Line ("Unable to open report file: ");
            PrintFileName (SPARK_IO.Standard_Output,
                           CommandLineData.Content.ReportFileName,
                           False);
            ScreenEcho.New_Line (1);
         end if;
         ReportFile := LocalFile;
      end CreateReportFile;

      procedure CloseReportFile (ReportFile : in out SPARK_IO.File_Type)
      --# global in out SPARK_IO.File_Sys;
      --# derives ReportFile,
      --#         SPARK_IO.File_Sys from *,
      --#                                ReportFile;
      is
         Status     : SPARK_IO.File_Status;
      begin
         --# accept F, 10, Status, "Status not used here" &
         --#        F, 33, Status, "Status not used here";
         SPARK_IO.Close (ReportFile, Status);
      end CloseReportFile;

      procedure OutputFileList (ReportFile : in SPARK_IO.File_Type)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.FileHeap;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                ContextManager.FileHeap,
      --#                                ReportFile;
      is
         FileDescriptor : ContextManager.FileDescriptors;
         FileName       : EStrings.T;
      begin
         if not CommandLineData.Content.XML then
            FileDescriptor := ContextManager.FirstFileDescriptor;
            SPARK_IO.New_Line (ReportFile, 2);
            if FileDescriptor = ContextManager.NullFile then
               SPARK_IO.Put_Line (ReportFile, "No source files used", 0);
            else
               if not CommandLineData.Content.XML then
                  SPARK_IO.Put_Line (ReportFile, "Source Filename(s) used were:", 0);
               end if;
               loop
                  exit when FileDescriptor = ContextManager.NullFile;
                  SPARK_IO.Put_String (ReportFile, "   ", 0);
                  ContextManager.GetSourceFileName (FileDescriptor, FileName);
                  PrintFileName (ReportFile, FileName, CommandLineData.Content.PlainOutput);
                  SPARK_IO.New_Line (ReportFile, 1);
                  FileDescriptor := ContextManager.NextFileDescriptor (FileDescriptor);
               end loop;
            end if;
         end if;
      end OutputFileList;

      procedure OutputErrorList (ReportFile          : in     SPARK_IO.File_Type;
                                 ReportFileErrorList : in out SPARK_IO.File_Type)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.UnitHeap;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in out ContextManager.FileHeap;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives ContextManager.FileHeap,
      --#         ErrorHandler.ErrorContext,
      --#         XMLReport.State           from CommandLineData.Content,
      --#                                        ContextManager.FileHeap,
      --#                                        ContextManager.UnitHeap,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.StringTable,
      --#                                        ReportFile,
      --#                                        SPARK_IO.File_Sys,
      --#                                        XMLReport.State &
      --#         ReportFileErrorList       from * &
      --#         SPARK_IO.File_Sys         from *,
      --#                                        CommandLineData.Content,
      --#                                        ContextManager.FileHeap,
      --#                                        ContextManager.UnitHeap,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.StringTable,
      --#                                        ReportFile,
      --#                                        ReportFileErrorList,
      --#                                        XMLReport.State;
      is
         subtype LineIndex is Positive range 1 .. 512;
         subtype Lines is String (LineIndex);
         CurrentLine : Lines;
         Stop        : Natural;
         Unused1,
         Unused2     : SPARK_IO.File_Status;
         FileDescriptor : ContextManager.FileDescriptors;
      begin
         --# accept F, 10, Unused1, "Not required" &
         --#        F, 33, Unused1, "Not required" &
         --#        F, 10, Unused2, "Not required" &
         --#        F, 33, Unused2, "Not required";
         if CommandLineData.Content.XML then
            XMLReport.StartSection (XMLReport.SResults,
                                    ReportFile);
         end if;

         --first output errors associated with files picked up from index mechanism
         FileDescriptor := ContextManager.FirstFileDescriptor;
         loop
            exit when FileDescriptor = ContextManager.NullFile;
            if not ContextManager.ErrorsReported (FileDescriptor) then
               AppendToReportFile (ReportFile,
                                   Error_Types.ForReportIndexedFiles,
                                   FileDescriptor);
            end if;
            FileDescriptor := ContextManager.NextFileDescriptor (FileDescriptor);
         end loop;

         --then append errors from files on command line/meta files
         SPARK_IO.Reset (ReportFileErrorList, SPARK_IO.In_File, Unused1);
         while not SPARK_IO.End_Of_File (ReportFileErrorList)
         loop
            SPARK_IO.Get_Line (ReportFileErrorList,
                                 --to get
                               CurrentLine,
                               Stop);
            if Stop = 0 then
               SPARK_IO.New_Line (ReportFile, 1);
            else
               SPARK_IO.Put_Line (ReportFile,
                                  CurrentLine,
                                  Stop);
            end if;
         end loop;
         SPARK_IO.Close (ReportFileErrorList, Unused2);
      end OutputErrorList;

      procedure OutputUnitsRequiredButNotFound (ReportFile : in SPARK_IO.File_Type)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.UnitHeap;
      --#        in     LexTokenManager.StringTable;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                ContextManager.UnitHeap,
      --#                                LexTokenManager.StringTable,
      --#                                ReportFile,
      --#                                XMLReport.State &
      --#         XMLReport.State   from *,
      --#                                CommandLineData.Content,
      --#                                ContextManager.UnitHeap,
      --#                                LexTokenManager.StringTable;
      is
         StatusCol : constant Positive := 50;
         ReasonCol : constant Positive := 72;

         UnitDescriptor : ContextManager.UnitDescriptors;
         UnitName       : LexTokenLists.Lists;
         UnitType       : ContextManager.UnitTypes;

         FirstTime      : Boolean;
      begin
         if CommandLineData.Content.XML then
            XMLReport.StartSection (XMLReport.SUnitsNotFound,
                                    ReportFile);

            UnitDescriptor := ContextManager.FirstUnitDescriptor;

            loop
               exit when UnitDescriptor = ContextManager.NullUnit;
               if ContextManager.GetUnitStatus (UnitDescriptor) in
                 ContextManager.UnitNotFound
               then
                  ContextManager.GetUnitName (UnitDescriptor, UnitName, UnitType);

                  -- XMLReport needs to be updated to include the reason
                  XMLReport.AdaUnit (GetLexTokenList (UnitName),
                                     GetUnitType (UnitType),
                                     GetReason (ContextManager.GetUnitStatus (UnitDescriptor)),
                                     ReportFile);

               end if;
               UnitDescriptor := ContextManager.NextUnitDescriptor (UnitDescriptor);
            end loop;

            XMLReport.EndSection (XMLReport.SUnitsNotFound,
                                  ReportFile);

         else
            FirstTime := True;
            UnitDescriptor := ContextManager.FirstUnitDescriptor;
            loop
               exit when UnitDescriptor = ContextManager.NullUnit;
               if ContextManager.GetUnitStatus (UnitDescriptor) in
                 ContextManager.UnitNotFound
               then
                  SPARK_IO.New_Line (ReportFile, 1);
                  if FirstTime then
                     SPARK_IO.New_Line (ReportFile, 1);
                     SPARK_IO.Put_Line (ReportFile,
                                        "The following were required during " &
                                          "analysis but could not be located:", 0);
                     SPARK_IO.New_Line (ReportFile, 1);
                     SPARK_IO.Put_String (ReportFile, "   Unit name", 0);
                     SPARK_IO.Set_Col (ReportFile, StatusCol);
                     SPARK_IO.Put_String (ReportFile, "Unit type", 0);
                     SPARK_IO.Set_Col (ReportFile, ReasonCol);
                     SPARK_IO.Put_Line (ReportFile, "Reason", 0);
                     SPARK_IO.Put_String (ReportFile, "   ---------", 0);
                     SPARK_IO.Set_Col (ReportFile, StatusCol);
                     SPARK_IO.Put_String (ReportFile, "---------", 0);
                     SPARK_IO.Set_Col (ReportFile, ReasonCol);
                     SPARK_IO.Put_Line (ReportFile, "------", 0);
                     FirstTime := False;
                  end if;
                  ContextManager.GetUnitName (UnitDescriptor, UnitName, UnitType);
                  SPARK_IO.Put_String (ReportFile, "   ", 0);
                  PrintLexTokenList (ReportFile, UnitName);
                  SPARK_IO.Set_Col (ReportFile, StatusCol);
                  PrintUnitType (ReportFile, UnitType);
                  SPARK_IO.Set_Col (ReportFile, ReasonCol);
                  PrintReason (ReportFile, ContextManager.GetUnitStatus (UnitDescriptor));
               end if;
               UnitDescriptor := ContextManager.NextUnitDescriptor (UnitDescriptor);
            end loop;
            SPARK_IO.New_Line (ReportFile, 2);
         end if;
      end OutputUnitsRequiredButNotFound;


      procedure OutputUnitsWithCyclicRequirements (ReportFile : in SPARK_IO.File_Type)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.UnitHeap;
      --#        in     LexTokenManager.StringTable;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                ContextManager.UnitHeap,
      --#                                LexTokenManager.StringTable,
      --#                                ReportFile,
      --#                                XMLReport.State &
      --#         XMLReport.State   from *,
      --#                                CommandLineData.Content,
      --#                                ContextManager.UnitHeap,
      --#                                LexTokenManager.StringTable;
      is
         StatusCol : constant Positive := 50;

         UnitDescriptor : ContextManager.UnitDescriptors;
         UnitName       : LexTokenLists.Lists;
         UnitType       : ContextManager.UnitTypes;

         FirstTime      : Boolean;
      begin
         if CommandLineData.Content.XML then
            XMLReport.StartSection (XMLReport.SCyclicRequirements,
                                    ReportFile);
            UnitDescriptor := ContextManager.FirstUnitDescriptor;
            loop
               exit when UnitDescriptor = ContextManager.NullUnit;
               if ContextManager.UnitInCycle (UnitDescriptor) then
                  ContextManager.GetUnitName (UnitDescriptor, UnitName, UnitType);
                  XMLReport.Unit (GetLexTokenList (UnitName),
                                  GetUnitType (UnitType),
                                  ReportFile);
               end if;
               UnitDescriptor := ContextManager.NextUnitDescriptor (UnitDescriptor);
            end loop;
            XMLReport.EndSection (XMLReport.SCyclicRequirements,
                                  ReportFile);
         else
            FirstTime := True;
            UnitDescriptor := ContextManager.FirstUnitDescriptor;
            loop
               exit when UnitDescriptor = ContextManager.NullUnit;
               if ContextManager.UnitInCycle (UnitDescriptor) then
                  SPARK_IO.New_Line (ReportFile, 1);
                  if FirstTime then
                     SPARK_IO.New_Line (ReportFile, 1);
                     SPARK_IO.Put_Line (ReportFile,
                                        "A cycle was detected when determining the required units " &
                                          "of the following:", 0);
                     SPARK_IO.New_Line (ReportFile, 1);
                     SPARK_IO.Put_String (ReportFile, "   Unit name", 0);
                     SPARK_IO.Set_Col (ReportFile, StatusCol);
                     SPARK_IO.Put_Line (ReportFile, "Unit type", 0);
                     SPARK_IO.Put_String (ReportFile, "   ---------", 0);
                     SPARK_IO.Set_Col (ReportFile, StatusCol);
                     SPARK_IO.Put_Line (ReportFile, "---------", 0);
                     FirstTime := False;
                  end if;
                  ContextManager.GetUnitName (UnitDescriptor, UnitName, UnitType);
                  SPARK_IO.Put_String (ReportFile, "   ", 0);
                  PrintLexTokenList (ReportFile, UnitName);
                  SPARK_IO.Set_Col (ReportFile, StatusCol);
                  PrintUnitType (ReportFile, UnitType);
               end if;
               UnitDescriptor := ContextManager.NextUnitDescriptor (UnitDescriptor);
            end loop;
            if not FirstTime then
               SPARK_IO.New_Line (ReportFile, 2);
            end if;

         end if;
      end OutputUnitsWithCyclicRequirements;



      procedure OutputCommandLine (ReportFile : in SPARK_IO.File_Type)
      --# global in     CommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                ReportFile,
      --#                                XMLReport.State &
      --#         XMLReport.State   from *,
      --#                                CommandLineData.Content;
      is
         TmpString : EStrings.T;
         TmpSuccess : Boolean;

         procedure OutputOption (ForXML     : in Boolean)
         --# global in     CommandLineData.Content;
         --#        in     ReportFile;
         --#        in     TmpString;
         --#        in out SPARK_IO.File_Sys;
         --#        in out XMLReport.State;
         --# derives SPARK_IO.File_Sys from *,
         --#                                CommandLineData.Content,
         --#                                ForXML,
         --#                                ReportFile,
         --#                                TmpString,
         --#                                XMLReport.State &
         --#         XMLReport.State   from *,
         --#                                CommandLineData.Content,
         --#                                ForXML;
         is

         begin
            if CommandLineData.Content.XML then
               if ForXML then
                  XMLReport.Option (TmpString, ReportFile);
               end if;
            else
               SPARK_IO.Put_String (ReportFile, "    ", 0);
               EStrings.PutLine (ReportFile, TmpString);
            end if;
         end OutputOption;

         function PlainOutput (Est : EStrings.T)
                              return EStrings.T
         --# global in CommandLineData.Content;
         is
            Result : EStrings.T;
         begin
            if CommandLineData.Content.PlainOutput and then
              not CommandLineData.Content.XML then
               Result := EStrings.LowerCase (Est);
            else
               Result := Est;
            end if;
            return Result;
         end PlainOutput;

      begin

         if CommandLineData.Content.XML then
            XMLReport.StartSection (XMLReport.SCommandline,
                                    ReportFile);
         else
            SPARK_IO.Put_Line (ReportFile, "Options:", 0);
         end if;

         if CommandLineData.Content.DefaultSwitchFile then
            EStrings.CopyString (TmpString, "default switch file used");
            OutputOption (ForXML => True);
         end if;

         if CommandLineData.Content.Index then
            EStrings.CopyString (TmpString, "index_file=");
            EStrings.AppendExaminerString
              (TmpString, PlainOutput (CommandLineData.Content.IndexFileName));
            OutputOption (ForXML => True);
         else
            EStrings.CopyString (TmpString, "noindex_file");
            OutputOption (ForXML => False);
         end if;

         if CommandLineData.Content.Warning then
            EStrings.CopyString (TmpString, "warning_file=");
            EStrings.AppendExaminerString
              (TmpString, PlainOutput (CommandLineData.Content.WarningFileName));
            OutputOption (ForXML => True);
         else
            EStrings.CopyString (TmpString, "nowarning_file");
            OutputOption (ForXML => False);
         end if;

         if CommandLineData.Content.TargetData then
            EStrings.CopyString (TmpString, "target_compiler_data=");
            EStrings.AppendExaminerString
              (TmpString, PlainOutput (CommandLineData.Content.TargetDataFile));
            OutputOption (ForXML => True);
         else
            EStrings.CopyString (TmpString, "notarget_compiler_data");
            OutputOption (ForXML => False);
         end if;

         if CommandLineData.Content.TargetConfig then
            EStrings.CopyString (TmpString, "config_file=");
            EStrings.AppendExaminerString
              (TmpString, PlainOutput (CommandLineData.Content.TargetConfigFile));
            OutputOption (ForXML => True);
         else
            EStrings.CopyString (TmpString, "noconfig_file");
            OutputOption (ForXML => False);
         end if;

         EStrings.CopyString (TmpString, "source_extension=");
         EStrings.AppendExaminerString
           (TmpString, PlainOutput (CommandLineData.Content.SourceExtension));
         OutputOption (ForXML => True);

         EStrings.CopyString (TmpString, "listing_extension=");
         EStrings.AppendExaminerString
              (TmpString, PlainOutput (CommandLineData.Content.ListingExtension));
         OutputOption (ForXML => True);

         if not CommandLineData.Content.Echo then
            EStrings.CopyString (TmpString, "noecho");
            OutputOption (ForXML => True);
         end if;

         if CommandLineData.Content.WriteDict then
            EStrings.CopyString (TmpString, "dictionary_file=");
            EStrings.AppendExaminerString
              (TmpString, PlainOutput (CommandLineData.Content.DictFileName));
            OutputOption (ForXML => True);
         else
            EStrings.CopyString (TmpString, "nodictionary");
            OutputOption (ForXML => False);
         end if;


         if CommandLineData.Content.Report then
            EStrings.CopyString (TmpString, "report_file=");
            EStrings.AppendExaminerString
              (TmpString, PlainOutput (CommandLineData.Content.ReportFileName));
            OutputOption (ForXML => True);
         else
            EStrings.CopyString (TmpString, "noreport_file");
            OutputOption (ForXML => False);
         end if;

         if CommandLineData.Content.HTML then
            EStrings.CopyString (TmpString, "html");
            OutputOption (ForXML => True);
         else
            EStrings.CopyString (TmpString, "no_html");
            OutputOption (ForXML => False);
         end if;

         if CommandLineData.Content.RedType.PFs then
            EStrings.CopyString (TmpString, "pfs");
            OutputOption (ForXML => True);
         end if;

         if CommandLineData.Content.RedType.ExpChecks then
            EStrings.CopyString (TmpString, "exp_checks");
            OutputOption (ForXML => True);
         end if;

         if CommandLineData.Content.RedType.RTC then
            EStrings.CopyString (TmpString, "rtc");
            OutputOption (ForXML => True);
         end if;

         if CommandLineData.Content.RedType.RealRTCs then
            EStrings.CopyString (TmpString, "RealRTCs");
            OutputOption (ForXML => True);
         end if;

         if CommandLineData.Content.RedType.VCs then
            EStrings.CopyString (TmpString, "vcs");
            OutputOption (ForXML => True);
         end if;

         if CommandLineData.Content.SyntaxOnly then
            EStrings.CopyString (TmpString, "syntax_check");
            OutputOption (ForXML => True);
         end if;

         if CommandLineData.Content.PlainOutput then
            EStrings.CopyString (TmpString, "plain_output");
            OutputOption (ForXML => True);
         end if;

         if CommandLineData.Content.WriteStatistics then
            EStrings.CopyString (TmpString, "statistics");
            OutputOption (ForXML => True);
         else
            EStrings.CopyString (TmpString, "nostatistics");
            OutputOption (ForXML => True);
         end if;

         if CommandLineData.Content.FDLreserved then
            EStrings.CopyString (TmpString, "fdl_identifiers");
            OutputOption (ForXML => True);
         else
            EStrings.CopyString (TmpString, "nofdl_identifiers");
            OutputOption (ForXML => True);
         end if;

         if CommandLineData.Content.DoInformationFlow then
            EStrings.CopyString (TmpString, "flow_analysis=information");
            OutputOption (ForXML => True);
         else
            EStrings.CopyString (TmpString, "flow_analysis=data");
            OutputOption (ForXML => True);
         end if;

         if CommandLineData.IsSpark95 then
            EStrings.CopyString (TmpString, "ada95");
            OutputOption (ForXML => True);
         else
            EStrings.CopyString (TmpString, "ada83");
            OutputOption (ForXML => True);
         end if;

         --# accept F, 10, TmpSuccess, "Not required here" &
         --#        F, 33, TmpSuccess, "Not required here";
         EStrings.CopyString (TmpString, "annotation_character=");
         EStrings.AppendChar
           (TmpString, CommandLineData.Content.AnnoChar, TmpSuccess);
         OutputOption (ForXML => True);

         EStrings.CopyString (TmpString, "profile=");
         case CommandLineData.Content.AnalysisProfile is
            when CommandLineData.Ravenscar =>
               EStrings.AppendString (TmpString, "ravenscar");
            when CommandLineData.Sequential =>
               EStrings.AppendString (TmpString, "sequential");
         end case;
         OutputOption (ForXML => True);


         EStrings.CopyString (TmpString, "rules=");
         case CommandLineData.Content.ConstantRules is
            when CommandLineData.NoRules =>
               EStrings.AppendString (TmpString, "none");
            when CommandLineData.Lazy =>
               EStrings.AppendString (TmpString, "lazy");
            when CommandLineData.Keen =>
               EStrings.AppendString (TmpString, "keen");
            when CommandLineData.AllRules =>
               EStrings.AppendString (TmpString, "all");
         end case;
         OutputOption (ForXML => True);


         if CommandLineData.Content.LegacyErrors then
            EStrings.CopyString (TmpString, "original_flow_errors");
            OutputOption (ForXML => True);
         end if;

         EStrings.CopyString (TmpString, "error_explanations=");
         case CommandLineData.Content.ErrorExplanation is
            when CommandLineData.Off =>
               EStrings.AppendString (TmpString, "off");
            when CommandLineData.FirstOccurrence =>
               EStrings.AppendString (TmpString, "first_occurrence");
            when CommandLineData.EveryOccurrence =>
               EStrings.AppendString (TmpString, "every_occurrence");
         end case;
         OutputOption (ForXML => True);

         EStrings.CopyString (TmpString, "justification_option=");
         case CommandLineData.Content.JustificationOption is
            when CommandLineData.Ignore =>
               EStrings.AppendString (TmpString, "ignore");
            when CommandLineData.Full =>
               EStrings.AppendString (TmpString, "full");
            when CommandLineData.Brief =>
               EStrings.AppendString (TmpString, "brief");
         end case;
         OutputOption (ForXML => True);

         if not CommandLineData.Content.XML then
            SPARK_IO.New_Line (ReportFile, 1);
            SPARK_IO.Put_Line (ReportFile, "Selected files:", 0);
         end if;

         for I in CommandLineData.SourceFileCounts
           range 1 .. CommandLineData.Content.NumberSource
         loop
            TmpString := PlainOutput (CommandLineData.Content.SourceFileList (I).SourceFileName);

            --# accept F, 41, "Stable expression OK here";
            if CommandLineData.Content.XML then
               XMLReport.Filename (TmpString, ReportFile);
            else
               SPARK_IO.Put_String (ReportFile, "   ", 0);
               EStrings.PutLine (ReportFile, TmpString);
            end if;
            --# end accept;
         end loop;


         if CommandLineData.Content.XML then
            XMLReport.EndSection (XMLReport.SCommandline,
                                  ReportFile);
         else
            SPARK_IO.New_Line (ReportFile, 2);
         end if;
      end OutputCommandLine;

      procedure OutputIndexList (ReportFile : in SPARK_IO.File_Type)
      --# global in     CommandLineData.Content;
      --#        in     IndexManager.IndexTable;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                IndexManager.IndexTable,
      --#                                ReportFile,
      --#                                XMLReport.State &
      --#         XMLReport.State   from *,
      --#                                CommandLineData.Content,
      --#                                IndexManager.IndexTable;
      is
      begin
         IndexManager.ListIndexFile (ReportFile);
      end OutputIndexList;

      procedure OutputMetaFileList (ReportFile : in SPARK_IO.File_Type)
      --# global in     CommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives SPARK_IO.File_Sys,
      --#         XMLReport.State   from CommandLineData.Content,
      --#                                ReportFile,
      --#                                SPARK_IO.File_Sys,
      --#                                XMLReport.State;
      is
         MetaFileUsed : Boolean := False;
         FileName : EStrings.T;
      begin
         if CommandLineData.Content.XML then
            XMLReport.StartSection (XMLReport.SMetaFiles,
                                    ReportFile);
         else
            SPARK_IO.New_Line (ReportFile, 1);
         end if;

         for CurrentSource in CommandLineData.SourceFilePositions
            range 1 .. CommandLineData.Content.NumberSource
         loop
            FileName := CommandLineData.Content.SourceFileList (CurrentSource).SourceFileName;
            if FileName.Content (1) = '@' then
               MetaFile.ReportFileContent (ReportFile, FileName, MetaFileUsed);
            end if;
         end loop;
         if not MetaFileUsed and not CommandLineData.Content.XML then
            SPARK_IO.New_Line (ReportFile, 1);
            SPARK_IO.Put_Line (ReportFile, "No Meta Files used", 0);
         end if;

         if CommandLineData.Content.XML then
            XMLReport.EndSection (XMLReport.SMetaFiles,
                                    ReportFile);
         end if;

      end OutputMetaFileList;

   begin  --OutputReportFile

      if CommandLineData.Content.Report then
         CreateReportFile (ReportFile, OK);
         if OK then
            if CommandLineData.Content.Echo and not CommandLineData.Content.Brief then
               ScreenEcho.Echo (ReportMessage);
            end if;
            OutputCommandLine (ReportFile);
            OutputIndexList (ReportFile);
            OutputMetaFileList (ReportFile);
            ErrorHandler.OutputWarningList (ReportFile);
            Dictionary.OutputTargetDataFile (ReportFile);
            ConfigFile.OutputConfigFile (ReportFile);
            OutputFileList (ReportFile);
            OutputUnitsRequiredButNotFound (ReportFile);
            OutputUnitsWithCyclicRequirements (ReportFile);
            if CommandLineData.Content.XML then
               XMLReport.EndSection (XMLReport.SPrologue,
                                     ReportFile);
            end if;
            OutputErrorList (ReportFile,
                             ReportFileErrorList);
            if CommandLineData.Content.SyntaxOnly then
               SPARK_IO.New_Line (ReportFile, 1);
               SPARK_IO.Put_Line (ReportFile,
                                  "Warning: No semantic checks carried out, text " &
                                  "may not be legal SPARK", 0);
            end if;
            if not CommandLineData.Content.DoInformationFlow then
               SPARK_IO.New_Line (ReportFile, 1);
               SPARK_IO.Put_Line (ReportFile,
                                  "Note: Information flow analysis not carried out", 0);
            end if;
            if CommandLineData.Content.WriteStatistics then
               Statistics.WriteOutput (ReportFile);
            end if;
            if not CommandLineData.Content.PlainOutput then
               ErrorHandler.OutputReferenceList (ReportFile);
            end if;
            if CommandLineData.Content.XML then
               XMLReport.EndSection (XMLReport.SReport,
                                     ReportFile);
            else
               ShowEndOfFile (ReportFile);
            end if;
            --# accept F, 10, ReportFile, "Final assignment after close";
            CloseReportFile (ReportFile);
            --# end accept;
            if CommandLineData.Content.HTML then
               SparkHTML.GenReportHTML;
            end if;
         end if;
      end if;
   end OutputReportFile;

   procedure EchoListingFile (FileName : in EStrings.T)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                FileName;
   is
      Message : EStrings.T;
   begin
      if CommandLineData.Content.Echo and not CommandLineData.Content.Brief then
         EStrings.CopyString (Message, "Generating listing file ");
         if CommandLineData.Content.PlainOutput then
            EStrings.AppendExaminerString
               (Message, EStrings.LowerCase (FileName));
         else
            EStrings.AppendExaminerString (Message, FileName);
         end if;
         ScreenEcho.Echo (Message);
      end if;
   end EchoListingFile;

   procedure OutputListingFile (FileDescriptor : in ContextManager.FileDescriptors)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out ContextManager.FileHeap;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ContextManager.FileHeap,
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        ContextManager.FileHeap,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FileDescriptor,
   --#                                        LexTokenManager.StringTable,
   --#                                        SPARK_IO.File_Sys;
   is
      ListingFile          : SPARK_IO.File_Type := SPARK_IO.Null_File;
      PlainListingFileName : EStrings.T;
      ListingFileName      : EStrings.T;
      Status               : SPARK_IO.File_Status;
      ErrorContext         : ErrorHandler.ErrorContexts;
   begin
      ContextManager.GetListingFileName (FileDescriptor, ListingFileName);

      PlainListingFileName := ListingFileName;
      CommandLineData.Normalize_FileName_To_Output_Directory (ListingFileName);

      ListingFileName := FileSystem.CaseOfFilesForCreate (ListingFileName);
      SPARK_IO.Create (ListingFile, ListingFileName.Length,
                       ListingFileName.Content, "", Status);
      if Status /= SPARK_IO.Ok then
         ScreenEcho.Put_String ("Listing file, ");
         PrintFileName (SPARK_IO.Standard_Output, ListingFileName, False);
         ScreenEcho.Put_Line (", could not be created");
      else
         EchoListingFile (PlainListingFileName);
         File_Utils.PrintAHeader (ListingFile,
                                  "Listing of SPARK Text",
                                  File_Utils.OtherFile);
         ContextManager.GetErrorContext (FileDescriptor, ErrorContext);
         ErrorHandler.SetErrorContext (ErrorContext);
         ErrorHandler.PrintErrors (ListingFile, Error_Types.ForListing);
         if CommandLineData.Content.SyntaxOnly then

            -- HTML directives:
            --! <NameFormat> <Name>
            --! <ErrorFormat> <Error>

            -- HTML output
            --! <Name> warning-no-semantic-checks
            --! <Error> Warning : No semantic checks carried out, text may not be legal SPARK.
            --! Issued when the Examiner is used solely to check the syntax of a
            --! SPARK text: this does not check the semantics of a program
            --! (e.g. the correctness of the annotations) and
            --! therefore does not guarantee that a program is legal SPARK.
            SPARK_IO.New_Line (ListingFile, 1);
            SPARK_IO.Put_Line (ListingFile,
                               "Warning: No semantic checks carried out, text " &
                               "may not be legal SPARK", 0);
         end if;
         if not CommandLineData.Content.DoInformationFlow then
            --! <Name> note-information-flow
            --! <Error> Note: Information flow analysis not carried out
            --! This is issued as a reminder that information flow analysis has
            --! not been carried out in this run of the Examiner: information
            --! flow errors may be present undetected in the text analysed
            SPARK_IO.New_Line (ListingFile, 1);
            SPARK_IO.Put_Line (ListingFile,
                               "Note: Information flow analysis not carried out", 0);
         end if;
         if CommandLineData.IsSpark83 then
            --! <Name> note-ada-83
            --! <Error> Note: Ada 83 language rules selected
            --! Issued when the Examiner is used in SPARK 83 mode
            SPARK_IO.New_Line (ListingFile, 1);
            SPARK_IO.Put_Line (ListingFile,
                               "Note: Ada83 language rules selected", 0);
         end if;
         ErrorHandler.GetErrorContext (ErrorContext);
         ContextManager.SetErrorContext (FileDescriptor, ErrorContext);
         ShowEndOfFile (ListingFile);
         --# accept F, 10, ListingFile, "Final assignment after close" &
         --#        F, 10, Status,      "Final assignment after close";
         SPARK_IO.Close (ListingFile, Status);
         --# end accept;
      end if;
   end OutputListingFile;


   procedure CreateUnitContext (FileDescriptor : in     ContextManager.FileDescriptors;
                                UnitDescriptor :    out ContextManager.UnitDescriptors)
   --# global in out ContextManager.FileHeap;
   --#        in out ContextManager.UnitHeap;
   --# derives ContextManager.FileHeap,
   --#         ContextManager.UnitHeap from *,
   --#                                      ContextManager.FileHeap,
   --#                                      FileDescriptor &
   --#         UnitDescriptor          from ContextManager.UnitHeap;
   is
      UD2       : ContextManager.UnitDescriptors;
      UnitCount : Natural;
   begin
      ContextManager.CreateUnitDescriptor (UD2);
      ContextManager.SetUnitStatus (UD2, ContextManager.UnitCreated);
      ContextManager.SetFileDescriptor (UD2, FileDescriptor);
      if FileDescriptor /= ContextManager.NullFile then
         ContextManager.IncUnitCount (FileDescriptor);
         ContextManager.GetUnitCount (FileDescriptor, UnitCount);
         ContextManager.SetUnitNumber (UD2, UnitCount);
      else
         ContextManager.SetUnitNumber (UD2, 0);
      end if;
      ContextManager.SetRedType (UD2, CommandLineData.NoVCG);
      UnitDescriptor := UD2;
   end CreateUnitContext;

   procedure ParseCurrentUnit
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.UnitStack;
   --#        in     Dictionary.Dict;
   --#        in out ContextManager.FileHeap;
   --#        in out ContextManager.UnitHeap;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --#           out SparkLex.CurrLine;
   --# derives ContextManager.FileHeap,
   --#         ContextManager.UnitHeap,
   --#         ErrorHandler.ErrorContext,
   --#         LexTokenManager.StringTable,
   --#         SparkLex.CurrLine,
   --#         SPARK_IO.File_Sys,
   --#         STree.Table                 from CommandLineData.Content,
   --#                                          ContextManager.FileHeap,
   --#                                          ContextManager.UnitHeap,
   --#                                          ContextManager.UnitStack,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table;
   is
      FileDescriptor : ContextManager.FileDescriptors;
      SourceFile     : SPARK_IO.File_Type;
      Success        : SPARK_IO.File_Status;
      MaxStackSize   : Natural;
      FileEnd        : Boolean;
      Severity       : ErrorHandler.ErrorLevel;
      ParseTree      : STree.SyntaxNode;
      FileContext    : SparkLex.LineContext;
      ErrorContext   : ErrorHandler.ErrorContexts;
      AlreadyClosed  : Boolean;
   begin -- ParseCurrentUnit
      --# accept F, 10, MaxStackSize, "Not required here" &
      --#        F, 33, MaxStackSize, "Not required here" &
      --#        F, 10, Success,      "Not required here" &
      --#        F, 33, Success,      "Not required here" &
      --#        F, 10, SourceFile,   "Final assignment in Close";
      ContextManager.GetFileDescriptor (ContextManager.CurrentUnit, FileDescriptor);
      ContextManager.GetSourceFile (FileDescriptor, SourceFile);
      ContextManager.GetLineContext (FileDescriptor, FileContext);
      SparkLex.RestoreLineContext (FileContext);
      ContextManager.GetErrorContext (FileDescriptor, ErrorContext);
      ErrorHandler.SetErrorContext (ErrorContext);
      SPParser.SPParse (SourceFile, MaxStackSize, FileEnd); -- MaxStackSize unused
      STree.RetrieveCurrentRoot (ParseTree);
      ContextManager.SetParseTree (ContextManager.CurrentUnit, ParseTree);
      ContextManager.SetUnitStatus (ContextManager.CurrentUnit, ContextManager.UnitParsed);
      if FileEnd then
         if ContextManager.GetFileStatus (FileDescriptor) = ContextManager.FileOpen then
            ContextManager.SetFileStatus (FileDescriptor, ContextManager.FileEnd);
            SPARK_IO.Close (SourceFile, Success); -- flow error as Success unused
         end if;
         AlreadyClosed := True;
      else
         AlreadyClosed := False;
      end if;
      ErrorHandler.GetErrorSeverity (Severity);
      if Severity = ErrorHandler.Fatal then
         ContextManager.SetUnitStatus (ContextManager.CurrentUnit,
                                       ContextManager.UnitAnalysed);
         ContextManager.SetFileStatus (FileDescriptor, ContextManager.FileEnd);
         if not AlreadyClosed then
            SPARK_IO.Close (SourceFile, Success); -- flow error as Success unused
         end if;
      end if;
      SparkLex.StoreLineContext (FileContext);
      ContextManager.SetLineContext (FileDescriptor, FileContext);
      ErrorHandler.GetErrorContext (ErrorContext);
      ContextManager.SetErrorContext (FileDescriptor, ErrorContext);
   end ParseCurrentUnit;

   procedure EchoCompilationUnit (UnitName : in LexTokenLists.Lists;
                                  UnitType : in ContextManager.UnitTypes)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                LexTokenManager.StringTable,
   --#                                UnitName,
   --#                                UnitType;
   is
      Str,
      Message : EStrings.T;
   begin

      if CommandLineData.Content.Echo and then
        not CommandLineData.Content.Brief and then
        UnitType /= ContextManager.InterUnitPragma then

         EStrings.CopyString (Message, "Examining ");
         case UnitType is
            when ContextManager.MainProgram          =>
               EStrings.AppendString (Message, "main program ");
            when ContextManager.PackageSpecification =>
               EStrings.AppendString (Message, "the specification of package ");
            when ContextManager.PackageBody          =>
               EStrings.AppendString (Message, "the body of package ");
            when ContextManager.SubUnit              =>
               EStrings.AppendString (Message, "subunit ");
            when ContextManager.GenericDeclaration =>
               EStrings.AppendString (Message, "generic declaration ");
            when ContextManager.GenericPackageInstantiation =>
               EStrings.AppendString (Message, "generic package instantiation ");
            when ContextManager.GenericSubprogramBody =>
               EStrings.AppendString (Message, "generic subprogram body ");

            -- following don't get examined so don't need suitable message
            when ContextManager.InterUnitPragma |
              ContextManager.InvalidUnit |
              ContextManager.PackageOrGenericDeclaration =>
               null;
         end case;

         LexTokenManager.LexStringToString (UnitName.Content (UnitName.Length), Str);
         EStrings.AppendExaminerString (Message, Str);

         ScreenEcho.Echo (Message);
      end if;
   end EchoCompilationUnit;

   procedure AnalyseCurrentUnit
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.UnitStack;
   --#        in out ContextManager.FileHeap;
   --#        in out ContextManager.UnitHeap;
   --#        in out Declarations.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Graph.Table;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out StmtStack.S;
   --#        in out STree.Table;
   --#        in out VCG.Invoked;
   --#           out SparkLex.CurrLine;
   --# derives ContextManager.FileHeap,
   --#         Declarations.State,
   --#         Dictionary.Dict,
   --#         ErrorHandler.ErrorContext,
   --#         Graph.Table,
   --#         LexTokenManager.StringTable,
   --#         SPARK_IO.File_Sys,
   --#         Statistics.TableUsage,
   --#         StmtStack.S,
   --#         STree.Table,
   --#         VCG.Invoked                 from *,
   --#                                          CommandLineData.Content,
   --#                                          ContextManager.FileHeap,
   --#                                          ContextManager.UnitHeap,
   --#                                          ContextManager.UnitStack,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table &
   --#         ContextManager.UnitHeap     from *,
   --#                                          ContextManager.UnitStack &
   --#         SparkLex.CurrLine           from ContextManager.FileHeap,
   --#                                          ContextManager.UnitHeap,
   --#                                          ContextManager.UnitStack;
   is
      FileDescriptor : ContextManager.FileDescriptors;
      SourceFile     : SPARK_IO.File_Type;
      SourceFileName : EStrings.T;
      UnitName       : LexTokenLists.Lists;
      UnitType       : ContextManager.UnitTypes;
      Success        : SPARK_IO.File_Status;
      Severity       : ErrorHandler.ErrorLevel;
      FileContext    : SparkLex.LineContext;
      ParseTree      : STree.SyntaxNode;
      ErrorContext   : ErrorHandler.ErrorContexts;
      RedType        : CommandLineData.RedTypes;
   begin
      ContextManager.GetParseTree (ContextManager.CurrentUnit, ParseTree);
      ContextManager.GetFileDescriptor (ContextManager.CurrentUnit, FileDescriptor);
      ContextManager.GetSourceFile (FileDescriptor, SourceFile);
      ContextManager.GetSourceFileName (FileDescriptor, SourceFileName);
      ContextManager.GetLineContext (FileDescriptor, FileContext);
      ContextManager.GetRedType (ContextManager.CurrentUnit, RedType);
      ContextManager.GetUnitName (ContextManager.CurrentUnit, UnitName, UnitType);
      SparkLex.RestoreLineContext (FileContext);
      ContextManager.GetErrorContext (FileDescriptor, ErrorContext);
      ErrorHandler.SetErrorContext (ErrorContext);
      EchoCompilationUnit (UnitName, UnitType);
      Dictionary.SetCurrentFileName (SourceFileName);
      Sem.CompUnit (TopNode => ParseTree,
                    RedType => RedType);
      ErrorHandler.GetErrorSeverity (Severity);

      --# accept F, 10, SourceFile, "Final assignment in Close" &
      --#        F, 10, Success,    "Final assignment in Close" &
      --#        F, 33, Success,    "Final assignment in Close";
      if Severity = ErrorHandler.Fatal then
         ContextManager.SetFileStatus (FileDescriptor, ContextManager.FileEnd);
         SPARK_IO.Close (SourceFile, Success);
      end if;
      SparkLex.StoreLineContext (FileContext);
      ContextManager.SetLineContext (FileDescriptor, FileContext);
      ErrorHandler.GetErrorContext (ErrorContext);
      ContextManager.SetErrorContext (FileDescriptor, ErrorContext);

      -- return the ParseTree to the syntax tree free list, except sub-trees
      -- that are later needed by the VCG.  If the user has selected /rules=none,
      -- then we _can_ return constant declarations as well.
      STree.DeleteSyntaxTree
        (Root          => ParseTree,
         KeepConstants => CommandLineData.Content.ConstantRules /= CommandLineData.NoRules);

      ContextManager.SetUnitStatus (ContextManager.CurrentUnit,
                                    ContextManager.UnitAnalysed);
   end AnalyseCurrentUnit;

   function GetLikelyUnitType (PossibleUnitTypes : ContextManager.UnitTypeSets)
      return ContextManager.UnitTypes
   is
      ActualUnitType : ContextManager.UnitTypes;
   begin
      -- since we didn't find the unit we need to guess what kind of thing we were
      -- looking for so that we can make a sensible report about missing units in the
      -- report file.  We use the "sets" of possible units to decide.
      if PossibleUnitTypes (ContextManager.PackageSpecification) then
         if PossibleUnitTypes (ContextManager.GenericDeclaration) then
            -- both package spec and generic dec are member
            ActualUnitType := ContextManager.PackageOrGenericDeclaration;
         else
            -- just package spec is member
            ActualUnitType := ContextManager.PackageSpecification;
         end if;
      elsif PossibleUnitTypes (ContextManager.GenericDeclaration) then
         -- etc
         ActualUnitType := ContextManager.GenericDeclaration;
      elsif PossibleUnitTypes (ContextManager.SubUnit) then
         ActualUnitType := ContextManager.SubUnit;
      else
         ActualUnitType := ContextManager.PackageBody;
      end if;
      return ActualUnitType;
   end GetLikelyUnitType;

   procedure PrintLookupFailMessage (CurrFile     : in EStrings.T;
                                     RequiredUnit : in EStrings.T;
                                     ExpectedSort : in EStrings.T;
                                     Message      : in String)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                CurrFile,
   --#                                ExpectedSort,
   --#                                Message,
   --#                                RequiredUnit;
   is
      CurrStr : EStrings.T;
   begin
      if CommandLineData.Content.PlainOutput or
         CommandLineData.Content.Brief
      then
         if CurrFile.Length > 0 then
            CurrStr := EStrings.LowerCase (FileSystem.JustFile
                                          (CurrFile, True));
         else
            CurrStr := EStrings.EmptyString;
         end if;
      else
         CurrStr := CurrFile;
      end if;

      if CommandLineData.Content.Brief then
         ScreenEcho.Put_ExaminerString (CurrStr);
         ScreenEcho.Put_Char (':');
         ScreenEcho.Put_Integer (1, 0, 10);
         ScreenEcho.Put_Char (':');
         ScreenEcho.Put_Integer (1, 0, 10);
         ScreenEcho.Put_String (": ");
      else
         ScreenEcho.New_Line (1);
         ScreenEcho.Put_String ("In ");
         ScreenEcho.Put_ExaminerString (CurrStr);
         ScreenEcho.Put_String (": ");
      end if;

      ScreenEcho.Put_ExaminerString (ExpectedSort);
      ScreenEcho.Put_Char (' ');
      ScreenEcho.Put_ExaminerString (RequiredUnit);
      ScreenEcho.Put_String (" is not declared in this file ");
      ScreenEcho.Put_String ("and ");
      ScreenEcho.Put_ExaminerString (ExpectedSort);
      ScreenEcho.Put_Char (' ');
      ScreenEcho.Put_ExaminerString (RequiredUnit);
      ScreenEcho.Put_Char (' ');
      ScreenEcho.Put_Line (Message);
   end PrintLookupFailMessage;

   procedure AddRequiredUnit (RequiredUnit      : in LexTokenLists.Lists;
                              PossibleUnitTypes : in ContextManager.UnitTypeSets)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ContextManager.FileHeap;
   --#        in out ContextManager.UnitHeap;
   --#        in out ContextManager.UnitStack;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out IndexManager.IndexTable;
   --#        in out IndexManager.LastChar;
   --#        in out LexTokenManager.StringTable;
   --#        in out SparkLex.CurrLine;
   --#        in out SPARK_IO.File_Sys;
   --# derives ContextManager.FileHeap,
   --#         ContextManager.UnitHeap,
   --#         ErrorHandler.ErrorContext   from CommandLineData.Content,
   --#                                          ContextManager.FileHeap,
   --#                                          ContextManager.UnitHeap,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          PossibleUnitTypes,
   --#                                          RequiredUnit,
   --#                                          SPARK_IO.File_Sys &
   --#         ContextManager.UnitStack,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          ContextManager.FileHeap,
   --#                                          ContextManager.UnitHeap,
   --#                                          ContextManager.UnitStack,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          PossibleUnitTypes,
   --#                                          RequiredUnit,
   --#                                          SPARK_IO.File_Sys &
   --#         IndexManager.IndexTable,
   --#         IndexManager.LastChar,
   --#         LexTokenManager.StringTable from *,
   --#                                          CommandLineData.Content,
   --#                                          ContextManager.UnitHeap,
   --#                                          LexTokenManager.StringTable,
   --#                                          PossibleUnitTypes,
   --#                                          RequiredUnit,
   --#                                          SPARK_IO.File_Sys &
   --#         SparkLex.CurrLine           from *,
   --#                                          CommandLineData.Content,
   --#                                          ContextManager.FileHeap,
   --#                                          ContextManager.UnitHeap,
   --#                                          LexTokenManager.StringTable,
   --#                                          PossibleUnitTypes,
   --#                                          RequiredUnit,
   --#                                          SPARK_IO.File_Sys;

   is
      UnitDescriptor : ContextManager.UnitDescriptors;
      SourceFileName : EStrings.T;
      ActualUnitType : ContextManager.UnitTypes;
      FileDescriptor : ContextManager.FileDescriptors;
      FileStr        : EStrings.T;
      Found          : Boolean;
   begin
      Trace ("In MainLoop.AddRequiredUnit");
      TraceUnit (RequiredUnit, PossibleUnitTypes);
      ContextManager.GetUnitByName (RequiredUnit, PossibleUnitTypes, UnitDescriptor);
      if UnitDescriptor = ContextManager.NullUnit then
         ContextManager.GetFileDescriptor (ContextManager.CurrentUnit, FileDescriptor);

         -- A unit which was previously referenced, but could not be located
         -- can have a NullFile entry, and therefore won't have an entry
         -- in the SourceFileName table either.  We need to report something, so...
         if FileDescriptor = ContextManager.NullFile then
            EStrings.CopyString (FileStr, "UNKNOWN");
         else
            ContextManager.GetSourceFileName (FileDescriptor, FileStr);
         end if;

         if CommandLineData.Content.Index then
            Trace ("   unit not known, calling IndexManager.LookUp");
            IndexManager.LookUp (RequiredUnit,
                                 PossibleUnitTypes,
                                 SourceFileName,
                                 ActualUnitType,
                                 Found);
            if Found then
               Trace ("      Found");
               ContextManager.GetFileByName (SourceFileName, FileDescriptor);
               if FileDescriptor = ContextManager.NullFile then
                  CreateFileContext (SourceFileName, FileDescriptor);
                  OpenFile (FileDescriptor);
               end if;
               CreateUnitContext (FileDescriptor, UnitDescriptor);
               ContextManager.SetUnitName (UnitDescriptor, RequiredUnit, ActualUnitType);
               if ContextManager.GetFileStatus (FileDescriptor) =
                  ContextManager.UnableToOpen
               then
                  Trace ("      unit name found in index file but cannot open specified file");
                  ActualUnitType := GetLikelyUnitType (PossibleUnitTypes);
                  PrintLookupFailMessage (FileStr,
                                          GetUnitName (RequiredUnit),
                                          GetUnitType (ActualUnitType),
                                          "source file entry in index file cannot be opened");
                  ContextManager.SetUnitStatus (UnitDescriptor,
                                                ContextManager.CannotOpenFile);
               else
                  ContextManager.PushUnit (UnitDescriptor);
               end if;
            else
               Trace ("      not found in Index File");
               ActualUnitType := GetLikelyUnitType (PossibleUnitTypes);
               PrintLookupFailMessage (FileStr,
                                       GetUnitName (RequiredUnit),
                                       GetUnitType (ActualUnitType),
                                       "is not referenced in an index file.");
               CreateUnitContext (ContextManager.NullFile, UnitDescriptor);
               ContextManager.SetUnitName (UnitDescriptor, RequiredUnit, ActualUnitType);
               ContextManager.SetUnitStatus (UnitDescriptor,
                                             ContextManager.NotInIndexFile);
            end if;
         else
            Trace ("   unit not known, and no index file specified");
               ActualUnitType := GetLikelyUnitType (PossibleUnitTypes);
               PrintLookupFailMessage (FileStr,
                                       GetUnitName (RequiredUnit),
                                       GetUnitType (ActualUnitType),
                                       "cannot be located as no index file has been specified.");
            CreateUnitContext (ContextManager.NullFile, UnitDescriptor);
            ContextManager.SetUnitName (UnitDescriptor, RequiredUnit, ActualUnitType);
            ContextManager.SetUnitStatus (UnitDescriptor,
                                          ContextManager.NoIndexFile);
         end if;
      else
         Trace ("   unit already seen");
         if ContextManager.GetUnitStatus (UnitDescriptor) =
            ContextManager.UnitParsed
         then
            ContextManager.MarkUnitInCycle (UnitDescriptor);
         end if;
         ContextManager.PushUnit (UnitDescriptor);
      end if;
   end AddRequiredUnit;

   procedure SetUpRequiredUnits
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     STree.Table;
   --#        in out ContextManager.FileHeap;
   --#        in out ContextManager.UnitHeap;
   --#        in out ContextManager.UnitStack;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out IndexManager.IndexTable;
   --#        in out IndexManager.LastChar;
   --#        in out LexTokenManager.StringTable;
   --#        in out SparkLex.CurrLine;
   --#        in out SPARK_IO.File_Sys;
   --# derives ContextManager.FileHeap,
   --#         ContextManager.UnitHeap,
   --#         ContextManager.UnitStack,
   --#         ErrorHandler.ErrorContext,
   --#         IndexManager.IndexTable,
   --#         IndexManager.LastChar,
   --#         LexTokenManager.StringTable,
   --#         SparkLex.CurrLine,
   --#         SPARK_IO.File_Sys           from *,
   --#                                          CommandLineData.Content,
   --#                                          ContextManager.FileHeap,
   --#                                          ContextManager.UnitHeap,
   --#                                          ContextManager.UnitStack,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table;
   is
      InheritPtr        : STree.SyntaxNode;
      UnitType          : ContextManager.UnitTypes;
      UnitName          : LexTokenLists.Lists;
      CurrentUnitName   : LexTokenLists.Lists;
      CurrentUnitType   : ContextManager.UnitTypes;

      function FoundTopUnit (WantedName : LexTokenLists.Lists;
                             WantedType : ContextManager.UnitTypes;
                             FoundName  : LexTokenLists.Lists;
                             FoundType  : ContextManager.UnitTypes) return Boolean
      is
      begin
         Trace ("In MainLoop.FoundTopUnit");
         return WantedName.Length = 0 or else
            (LexTokenLists.EqUnit (WantedName, FoundName) and
             WantedType = FoundType);
      end FoundTopUnit;

      procedure ReplaceTopUnit (CurrentUnitName : in LexTokenLists.Lists;
                                CurrentUnitType : in ContextManager.UnitTypes)
      --# global in out ContextManager.FileHeap;
      --#        in out ContextManager.UnitHeap;
      --#        in out ContextManager.UnitStack;
      --# derives ContextManager.FileHeap,
      --#         ContextManager.UnitStack from ContextManager.FileHeap,
      --#                                       ContextManager.UnitHeap,
      --#                                       ContextManager.UnitStack,
      --#                                       CurrentUnitName &
      --#         ContextManager.UnitHeap  from *,
      --#                                       ContextManager.FileHeap,
      --#                                       ContextManager.UnitStack,
      --#                                       CurrentUnitName,
      --#                                       CurrentUnitType;
      is
         NewUnitDescriptor,
         OldUnitDescriptor : ContextManager.UnitDescriptors;
         FileDescriptor    : ContextManager.FileDescriptors;
         FoundUnit         : LexTokenLists.Lists;
         FoundUnitType     : ContextManager.UnitTypes;
         FoundUnitTypeSet  : ContextManager.UnitTypeSets;
         FoundUnitDescriptor : ContextManager.UnitDescriptors;
         FoundFileDescriptor : ContextManager.FileDescriptors;
         ParseTree         : STree.SyntaxNode;
      begin
         --# accept F, 10, OldUnitDescriptor, "Not required here" &
         --#        F, 33, OldUnitDescriptor, "Not required here";
         Trace ("In MainLoop.ReplaceTopUnit");
         if CurrentUnitName.Length = 0 then
            ContextManager.PopUnit (OldUnitDescriptor);
         else
            ContextManager.GetFileDescriptor (ContextManager.CurrentUnit,
                                              FileDescriptor);
            FoundFileDescriptor := FileDescriptor;
            ContextManager.GetUnitName (ContextManager.CurrentUnit,
                                        FoundUnit, FoundUnitType);
            FoundUnitTypeSet :=
               ContextManager.UnitTypeSets'(others => False);
            FoundUnitTypeSet (FoundUnitType) := True;
            ContextManager.GetUnitByName (FoundUnit, FoundUnitTypeSet,
                                          FoundUnitDescriptor);
            if ContextManager.GetUnitStatus (FoundUnitDescriptor) =
               ContextManager.UnitCreated then
               ContextManager.GetFileDescriptor (FoundUnitDescriptor,
                                                 FoundFileDescriptor);
            end if;
            if FoundUnitDescriptor /= ContextManager.CurrentUnit and then
               FoundFileDescriptor = FileDescriptor
            then -- special case of earlier pending unit found
               ContextManager.SetUnitStatus (FoundUnitDescriptor,
                                             ContextManager.UnitParsed);
               ContextManager.GetParseTree (ContextManager.CurrentUnit,
                                            ParseTree);
               ContextManager.SetParseTree (FoundUnitDescriptor,
                                            ParseTree);
               ContextManager.SetParseTree (ContextManager.CurrentUnit,
                                            STree.NullNode);
               ContextManager.SetUnitName (ContextManager.CurrentUnit,
                                           CurrentUnitName,
                                           CurrentUnitType);
               case ContextManager.GetFileStatus (FileDescriptor) is
                  when ContextManager.FileEnd =>
                     ContextManager.SetUnitStatus (ContextManager.CurrentUnit,
                                                   ContextManager.UnableToLocate);
                     ContextManager.PopUnit (OldUnitDescriptor);
                  when ContextManager.UnableToOpen =>
                     ContextManager.SetUnitStatus (ContextManager.CurrentUnit,
                                                   ContextManager.CannotOpenFile);
                     ContextManager.PopUnit (OldUnitDescriptor);
                  when others =>
                     ContextManager.SetUnitStatus (ContextManager.CurrentUnit,
                                                   ContextManager.UnitCreated);
               end case;
            else -- 'normal' case
               CreateUnitContext (FileDescriptor, NewUnitDescriptor);
               ContextManager.SetUnitStatus (ContextManager.CurrentUnit,
                                             ContextManager.UnitDeferred);
               ContextManager.PopUnit (OldUnitDescriptor);
               ContextManager.SetUnitName (NewUnitDescriptor,
                                           CurrentUnitName,
                                           CurrentUnitType);
               case ContextManager.GetFileStatus (FileDescriptor) is
                  when ContextManager.FileEnd =>
                     ContextManager.SetUnitStatus (NewUnitDescriptor,
                                                   ContextManager.UnableToLocate);
                  when ContextManager.UnableToOpen =>
                     ContextManager.SetUnitStatus (NewUnitDescriptor,
                                                   ContextManager.CannotOpenFile);
                  when others =>
                     ContextManager.PushUnit (NewUnitDescriptor);
               end case;
            end if;
         end if;
      end ReplaceTopUnit;


      procedure AddInheritedPackages
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     STree.Table;
      --#        in out ContextManager.FileHeap;
      --#        in out ContextManager.UnitHeap;
      --#        in out ContextManager.UnitStack;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out IndexManager.IndexTable;
      --#        in out IndexManager.LastChar;
      --#        in out InheritPtr;
      --#        in out LexTokenManager.StringTable;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --# derives ContextManager.FileHeap,
      --#         ContextManager.UnitHeap,
      --#         ContextManager.UnitStack,
      --#         ErrorHandler.ErrorContext,
      --#         IndexManager.IndexTable,
      --#         IndexManager.LastChar,
      --#         LexTokenManager.StringTable,
      --#         SparkLex.CurrLine,
      --#         SPARK_IO.File_Sys           from *,
      --#                                          CommandLineData.Content,
      --#                                          ContextManager.FileHeap,
      --#                                          ContextManager.UnitHeap,
      --#                                          ContextManager.UnitStack,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          InheritPtr,
      --#                                          LexTokenManager.StringTable,
      --#                                          SPARK_IO.File_Sys,
      --#                                          STree.Table &
      --#         InheritPtr                  from *,
      --#                                          STree.Table;
      is
         RequiredUnit      : LexTokenLists.Lists;
         Found             : Boolean;

         -- If a package is inherited and therefore required we do not want to add it
         -- to the required unit list if it is predefined.  Currently only
         -- Ada.Characters.Latin_1 is predefined. If Ravenscar is selected, package
         -- Ada.Real_Time is also predefined.
         function IsPredefinedPackage return Boolean
         --# global in CommandLineData.Content;
         --#        in RequiredUnit;
         is
            Result : Boolean;
         begin
            case RequiredUnit.Length is
               when 1 =>
                  -- Ada
                  Result :=  RequiredUnit.Content (1) = LexTokenManager.AdaToken;

               when 2 =>
                  -- Ada.Real_Time, Ada.Characters or Ada.Interrupts
                  Result :=
                    RequiredUnit.Content (1) = LexTokenManager.AdaToken and then
                    (RequiredUnit.Content (2) = LexTokenManager.CharactersToken or else
                       (CommandLineData.RavenscarSelected and then
                          (RequiredUnit.Content (2) = LexTokenManager.Real_TimeToken or
                             RequiredUnit.Content (2) = LexTokenManager.InterruptsToken or
                             RequiredUnit.Content (2) = LexTokenManager.Synchronous_Task_ControlToken)));
               when 3 =>
                  -- Ada.Characters.Latin_1
                  Result := RequiredUnit.Content (1) = LexTokenManager.AdaToken and then
                    RequiredUnit.Content (2) = LexTokenManager.CharactersToken   and then
                    RequiredUnit.Content (3) = LexTokenManager.Latin_1Token;
               when others =>
                  Result := False;
            end case;
            return Result;
         end IsPredefinedPackage;

         ------------------------

         function SystemNeededAndProvidedByConfigFile return Boolean
         --# global in Dictionary.Dict;
         --#        in RequiredUnit;
         is
         begin
            return RequiredUnit.Length = 1 and then
              RequiredUnit.Content (1) = LexTokenManager.SystemToken and then
              Dictionary.IsDefined (LexTokenManager.SystemToken,
                                    Dictionary.GlobalScope,
                                    Dictionary.ProgramContext);
         end SystemNeededAndProvidedByConfigFile;

         ------------------------

         function IsPredefinedGeneric return Boolean
         --# global in RequiredUnit;
         is
            Result : Boolean;
         begin
            case RequiredUnit.Length is
               when 1 =>
                  -- Unchecked_Conversion
                  Result :=  RequiredUnit.Content (1) = LexTokenManager.Unchecked_ConversionToken;

               when 2 =>
                  -- Ada.Unchecked_Conversion
                  Result :=
                    RequiredUnit.Content (1) = LexTokenManager.AdaToken and then
                    RequiredUnit.Content (2) = LexTokenManager.Unchecked_ConversionToken;
               when others =>
                  Result := False;
            end case;
            return Result;
         end IsPredefinedGeneric;

         ------------------------

      begin -- AddInheritedPackages
         Trace ("In MainLoop.AddInheritedPackages");
         loop
            RequiredUnits.Next (InheritPtr, RequiredUnit, Found);
            exit when not Found;

            -- Ignore dotted package names in 83 mode and ignore predefined packages
            -- in 95 mode.  Ignore System if provided by config file. Ignore
            -- predefined generic units.
            if not (RequiredUnit.Length > 1 and CommandLineData.IsSpark83)
              and then not (CommandLineData.IsSpark95 and IsPredefinedPackage)
              and then not SystemNeededAndProvidedByConfigFile
              and then not IsPredefinedGeneric
            then
               AddRequiredUnit (RequiredUnit, ContextManager.InheritableItem);
            end if;
         end loop;
      end AddInheritedPackages;

      procedure HandleMainProgram
      --# global in     CommandLineData.Content;
      --#        in     CurrentUnitName;
      --#        in     CurrentUnitType;
      --#        in     Dictionary.Dict;
      --#        in     STree.Table;
      --#        in     UnitName;
      --#        in     UnitType;
      --#        in out ContextManager.FileHeap;
      --#        in out ContextManager.UnitHeap;
      --#        in out ContextManager.UnitStack;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out IndexManager.IndexTable;
      --#        in out IndexManager.LastChar;
      --#        in out InheritPtr;
      --#        in out LexTokenManager.StringTable;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --# derives ContextManager.FileHeap,
      --#         ContextManager.UnitHeap,
      --#         ContextManager.UnitStack,
      --#         ErrorHandler.ErrorContext,
      --#         IndexManager.IndexTable,
      --#         IndexManager.LastChar,
      --#         LexTokenManager.StringTable,
      --#         SparkLex.CurrLine,
      --#         SPARK_IO.File_Sys           from *,
      --#                                          CommandLineData.Content,
      --#                                          ContextManager.FileHeap,
      --#                                          ContextManager.UnitHeap,
      --#                                          ContextManager.UnitStack,
      --#                                          CurrentUnitName,
      --#                                          CurrentUnitType,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          InheritPtr,
      --#                                          LexTokenManager.StringTable,
      --#                                          SPARK_IO.File_Sys,
      --#                                          STree.Table,
      --#                                          UnitName,
      --#                                          UnitType &
      --#         InheritPtr                  from *,
      --#                                          CurrentUnitName,
      --#                                          CurrentUnitType,
      --#                                          STree.Table,
      --#                                          UnitName,
      --#                                          UnitType;
      is
      begin
         Trace ("In MainLoop.HandleMainProgram");
         ContextManager.SetUnitName (ContextManager.CurrentUnit, UnitName, UnitType);
         if FoundTopUnit (CurrentUnitName, CurrentUnitType, UnitName, UnitType) then
            AddInheritedPackages;
         else
            ReplaceTopUnit (CurrentUnitName, CurrentUnitType);
         end if;
      end HandleMainProgram;


      procedure HandleInterUnitPragma
      --# global in     CurrentUnitName;
      --#        in     CurrentUnitType;
      --#        in     UnitName;
      --#        in     UnitType;
      --#        in out ContextManager.FileHeap;
      --#        in out ContextManager.UnitHeap;
      --#        in out ContextManager.UnitStack;
      --# derives ContextManager.FileHeap,
      --#         ContextManager.UnitHeap,
      --#         ContextManager.UnitStack from ContextManager.FileHeap,
      --#                                       ContextManager.UnitHeap,
      --#                                       ContextManager.UnitStack,
      --#                                       CurrentUnitName,
      --#                                       CurrentUnitType,
      --#                                       UnitName,
      --#                                       UnitType;
      is
         NewUnitName       : LexTokenLists.Lists;
      begin
         Trace ("In MainLoop.HandleInterUnitPragma");
         NewUnitName := LexTokenLists.NullList;
         ContextManager.SetUnitName
            (ContextManager.CurrentUnit, NewUnitName, UnitType);
         if not FoundTopUnit (CurrentUnitName, CurrentUnitType, UnitName, UnitType) then
            ReplaceTopUnit (CurrentUnitName, CurrentUnitType);
         end if;
      end HandleInterUnitPragma;

      procedure HandlePackageSpecification
      --# global in     CommandLineData.Content;
      --#        in     CurrentUnitName;
      --#        in     CurrentUnitType;
      --#        in     Dictionary.Dict;
      --#        in     STree.Table;
      --#        in     UnitName;
      --#        in     UnitType;
      --#        in out ContextManager.FileHeap;
      --#        in out ContextManager.UnitHeap;
      --#        in out ContextManager.UnitStack;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out IndexManager.IndexTable;
      --#        in out IndexManager.LastChar;
      --#        in out InheritPtr;
      --#        in out LexTokenManager.StringTable;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --# derives ContextManager.FileHeap,
      --#         ContextManager.UnitHeap,
      --#         ContextManager.UnitStack,
      --#         ErrorHandler.ErrorContext,
      --#         IndexManager.IndexTable,
      --#         IndexManager.LastChar,
      --#         LexTokenManager.StringTable,
      --#         SparkLex.CurrLine,
      --#         SPARK_IO.File_Sys           from *,
      --#                                          CommandLineData.Content,
      --#                                          ContextManager.FileHeap,
      --#                                          ContextManager.UnitHeap,
      --#                                          ContextManager.UnitStack,
      --#                                          CurrentUnitName,
      --#                                          CurrentUnitType,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          InheritPtr,
      --#                                          LexTokenManager.StringTable,
      --#                                          SPARK_IO.File_Sys,
      --#                                          STree.Table,
      --#                                          UnitName,
      --#                                          UnitType &
      --#         InheritPtr                  from *,
      --#                                          CurrentUnitName,
      --#                                          CurrentUnitType,
      --#                                          STree.Table,
      --#                                          UnitName,
      --#                                          UnitType;
      is
         procedure AddParent (UnitName : in LexTokenLists.Lists)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in out ContextManager.FileHeap;
         --#        in out ContextManager.UnitHeap;
         --#        in out ContextManager.UnitStack;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out IndexManager.IndexTable;
         --#        in out IndexManager.LastChar;
         --#        in out LexTokenManager.StringTable;
         --#        in out SparkLex.CurrLine;
         --#        in out SPARK_IO.File_Sys;
         --# derives ContextManager.FileHeap,
         --#         ContextManager.UnitHeap,
         --#         ErrorHandler.ErrorContext   from CommandLineData.Content,
         --#                                          ContextManager.FileHeap,
         --#                                          ContextManager.UnitHeap,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          LexTokenManager.StringTable,
         --#                                          SPARK_IO.File_Sys,
         --#                                          UnitName &
         --#         ContextManager.UnitStack,
         --#         SPARK_IO.File_Sys           from CommandLineData.Content,
         --#                                          ContextManager.FileHeap,
         --#                                          ContextManager.UnitHeap,
         --#                                          ContextManager.UnitStack,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          LexTokenManager.StringTable,
         --#                                          SPARK_IO.File_Sys,
         --#                                          UnitName &
         --#         IndexManager.IndexTable,
         --#         IndexManager.LastChar,
         --#         LexTokenManager.StringTable from *,
         --#                                          CommandLineData.Content,
         --#                                          ContextManager.UnitHeap,
         --#                                          LexTokenManager.StringTable,
         --#                                          SPARK_IO.File_Sys,
         --#                                          UnitName &
         --#         SparkLex.CurrLine           from *,
         --#                                          CommandLineData.Content,
         --#                                          ContextManager.FileHeap,
         --#                                          ContextManager.UnitHeap,
         --#                                          LexTokenManager.StringTable,
         --#                                          SPARK_IO.File_Sys,
         --#                                          UnitName;
         is
            RequiredUnit      : LexTokenLists.Lists;
         begin
            Trace ("In MainLoop.HandlePackageSpecification.AddParent");
            RequiredUnit.Length := UnitName.Length - 1;
            RequiredUnit.Content := UnitName.Content;
            AddRequiredUnit (RequiredUnit, ContextManager.PackageSpecificationSet);
         end AddParent;

      begin -- HandlePackageSpecification
         Trace ("In MainLoop.HandlePackageSpecification");
         ContextManager.SetUnitName (ContextManager.CurrentUnit, UnitName, UnitType);
         if FoundTopUnit (CurrentUnitName, CurrentUnitType, UnitName, UnitType) then
            AddInheritedPackages;
            if CommandLineData.IsSpark95
              and then UnitName.Length > 1
              and then UnitName.Content (1) /= LexTokenManager.AdaToken
              and then UnitName.Content (1) /= LexTokenManager.SystemToken
            then
               AddParent (UnitName);
            end if;
         else
            ReplaceTopUnit (CurrentUnitName, CurrentUnitType);
         end if;
      end HandlePackageSpecification;

      procedure HandlePackageBody
      --# global in     CommandLineData.Content;
      --#        in     CurrentUnitName;
      --#        in     CurrentUnitType;
      --#        in     Dictionary.Dict;
      --#        in     UnitName;
      --#        in out ContextManager.FileHeap;
      --#        in out ContextManager.UnitHeap;
      --#        in out ContextManager.UnitStack;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out IndexManager.IndexTable;
      --#        in out IndexManager.LastChar;
      --#        in out LexTokenManager.StringTable;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --# derives ContextManager.FileHeap,
      --#         ContextManager.UnitHeap,
      --#         ContextManager.UnitStack,
      --#         ErrorHandler.ErrorContext,
      --#         IndexManager.IndexTable,
      --#         IndexManager.LastChar,
      --#         LexTokenManager.StringTable,
      --#         SparkLex.CurrLine,
      --#         SPARK_IO.File_Sys           from *,
      --#                                          CommandLineData.Content,
      --#                                          ContextManager.FileHeap,
      --#                                          ContextManager.UnitHeap,
      --#                                          ContextManager.UnitStack,
      --#                                          CurrentUnitName,
      --#                                          CurrentUnitType,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.StringTable,
      --#                                          SPARK_IO.File_Sys,
      --#                                          UnitName;
      is
         Components : IndexManager.ComponentLists;
         Index      : IndexManager.ComponentIndex;
      begin
         Trace ("In MainLoop.HandlePackageBody");
         ContextManager.SetUnitName (ContextManager.CurrentUnit,
                                     UnitName,
                                     ContextManager.PackageBody);
         if FoundTopUnit (CurrentUnitName, CurrentUnitType,
                          UnitName, ContextManager.PackageBody)
         then
            if CommandLineData.IsSpark95 then -- look for components in index file
               IndexManager.LookUpComponents (UnitName, Components);
               Index := IndexManager.ComponentIndex'First;
               loop
                  exit when LexTokenLists.EqUnit (Components (Index),
                                                  LexTokenLists.NullList);
                  AddRequiredUnit (Components (Index), ContextManager.PackageSpecificationSet);
                  Index := IndexManager.ComponentIndex'Succ (Index);
               end loop;
            end if;
            if not (UnitName.Length > 1 and CommandLineData.IsSpark83)
               and then
               not (UnitName.Content (1) = LexTokenManager.AdaToken
                    and CommandLineData.IsSpark95)
            then
               AddRequiredUnit (UnitName, ContextManager.PackageSpecificationSet);

            end if;
         else
            ReplaceTopUnit (CurrentUnitName, CurrentUnitType);
         end if;
      end HandlePackageBody;

      -- PNA annotation below mangled because I have based it on wf_package_body and it is likely
      -- to end up quite close to that.  For the purposes of CFR 1340 I have commented out the
      -- bits of the annotation that aren't needed.
      procedure HandleGenericDeclaration
      --# global in     ContextManager.UnitStack;
      --#        in     UnitName;
      --#        in out ContextManager.UnitHeap;
      --# derives ContextManager.UnitHeap from *,
      --#                                      ContextManager.UnitStack,
      --#                                      UnitName;
      --#                                          --CurrentUnitName,
      --#                                          --CurrentUnitType; -- &
      --#         --SPARK_IO.FILE_SYS           from *,
      --#         --                                 --Dictionary.Dict,
      --#         --                                 --LexTokenManager.StringTable,
      --#         --                                 --CommandLineData.Content,
      --#         --                                 --ErrorHandler.ErrorContext,
      --#         --                                 --ContextManager.FileHeap,
      --#         --                                 ContextManager.UnitHeap,
      --#         --                                 --IndexManager.IndexTable,
      --#         --                                 ContextManager.UnitStack,
      --#         --                                 UnitName,
      --#         --                                 CurrentUnitName,
      --#         --                                 CurrentUnitType &
      --#         --SparkLex.CurrLine           from *,
      --#         --                                 --SPARK_IO.FILE_SYS,
      --#         --                                 --LexTokenManager.StringTable,
      --#         --                                 CommandLineData.Content,
      --#         --                                 --ContextManager.FileHeap,
      --#         --                                 ContextManager.UnitHeap,
      --#         --                                 --IndexManager.IndexTable,
      --#         --                                 ContextManager.UnitStack,
      --#         --                                 UnitName,
      --#         --                                 CurrentUnitName,
      --#         --                                 CurrentUnitType;
      is
      begin -- HandleGenericDeclaration
         Trace ("In MainLoop.HandleGenericDeclaration");
         ContextManager.SetUnitName (ContextManager.CurrentUnit,
                                     UnitName,
                                     ContextManager.GenericDeclaration);
         -- incomplete, doesn't yet handle required units of a generic unit PNA 21/11/03 GenericTBD
      end HandleGenericDeclaration;

      -- PNA annotation below mangled because I have based it on wf_package_body and it is likely
      -- to end up quite close to that.  For the purposes of CFR 1340 I have commented out the
      -- bits of the annotation that aren't needed.
      procedure HandleGenericPackageInstantiation
      --# global in     ContextManager.UnitStack;
      --#        in     UnitName;
      --#        in out ContextManager.UnitHeap;
      --# derives ContextManager.UnitHeap from *,
      --#                                      ContextManager.UnitStack,
      --#                                      UnitName;
      --#                                          --CurrentUnitName,
      --#                                          --CurrentUnitType; -- &
      --#         --SPARK_IO.FILE_SYS           from *,
      --#         --                                 --Dictionary.Dict,
      --#         --                                 --LexTokenManager.StringTable,
      --#         --                                 --CommandLineData.Content,
      --#         --                                 --ErrorHandler.ErrorContext,
      --#         --                                 --ContextManager.FileHeap,
      --#         --                                 ContextManager.UnitHeap,
      --#         --                                 --IndexManager.IndexTable,
      --#         --                                 ContextManager.UnitStack,
      --#         --                                 UnitName,
      --#         --                                 CurrentUnitName,
      --#         --                                 CurrentUnitType &
      --#         --SparkLex.CurrLine           from *,
      --#         --                                 --SPARK_IO.FILE_SYS,
      --#         --                                 --LexTokenManager.StringTable,
      --#         --                                 CommandLineData.Content,
      --#         --                                 --ContextManager.FileHeap,
      --#         --                                 ContextManager.UnitHeap,
      --#         --                                 --IndexManager.IndexTable,
      --#         --                                 ContextManager.UnitStack,
      --#         --                                 UnitName,
      --#         --                                 CurrentUnitName,
      --#         --                                 CurrentUnitType;
      is
      begin -- HandleGenericPackageInstantiation
         Trace ("In MainLoop.HandleGenericPackageInstantiation");
         ContextManager.SetUnitName (ContextManager.CurrentUnit,
                                     UnitName,
                                     ContextManager.GenericPackageInstantiation);
         -- incomplete, doesn't yet handle required units of a generic unit PNA 21/11/03 GenericTBD
      end HandleGenericPackageInstantiation;

      procedure HandleGenericSubprogramBody
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     UnitName;
      --#        in out ContextManager.FileHeap;
      --#        in out ContextManager.UnitHeap;
      --#        in out ContextManager.UnitStack;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out IndexManager.IndexTable;
      --#        in out IndexManager.LastChar;
      --#        in out LexTokenManager.StringTable;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --# derives ContextManager.FileHeap,
      --#         ContextManager.UnitHeap,
      --#         ContextManager.UnitStack,
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.File_Sys           from CommandLineData.Content,
      --#                                          ContextManager.FileHeap,
      --#                                          ContextManager.UnitHeap,
      --#                                          ContextManager.UnitStack,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.StringTable,
      --#                                          SPARK_IO.File_Sys,
      --#                                          UnitName &
      --#         IndexManager.IndexTable,
      --#         IndexManager.LastChar,
      --#         LexTokenManager.StringTable from *,
      --#                                          CommandLineData.Content,
      --#                                          ContextManager.UnitHeap,
      --#                                          ContextManager.UnitStack,
      --#                                          LexTokenManager.StringTable,
      --#                                          SPARK_IO.File_Sys,
      --#                                          UnitName &
      --#         SparkLex.CurrLine           from *,
      --#                                          CommandLineData.Content,
      --#                                          ContextManager.FileHeap,
      --#                                          ContextManager.UnitHeap,
      --#                                          ContextManager.UnitStack,
      --#                                          LexTokenManager.StringTable,
      --#                                          SPARK_IO.File_Sys,
      --#                                          UnitName;
      is
      begin -- HandleGenericSubprogramBody
         Trace ("In MainLoop.HandleGenericSubprogramBody");
         ContextManager.SetUnitName (ContextManager.CurrentUnit,
                                     UnitName,
                                     ContextManager.GenericSubprogramBody);
         -- generic body found so we need to see its spec
         AddRequiredUnit (UnitName, ContextManager.GenericDeclarationSet);
      end HandleGenericSubprogramBody;

      procedure HandleSubUnit
      --# global in     CommandLineData.Content;
      --#        in     CurrentUnitName;
      --#        in     CurrentUnitType;
      --#        in     Dictionary.Dict;
      --#        in     STree.Table;
      --#        in     UnitName;
      --#        in out ContextManager.FileHeap;
      --#        in out ContextManager.UnitHeap;
      --#        in out ContextManager.UnitStack;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out IndexManager.IndexTable;
      --#        in out IndexManager.LastChar;
      --#        in out InheritPtr;
      --#        in out LexTokenManager.StringTable;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --# derives ContextManager.FileHeap,
      --#         ContextManager.UnitHeap,
      --#         ContextManager.UnitStack,
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.File_Sys           from CommandLineData.Content,
      --#                                          ContextManager.FileHeap,
      --#                                          ContextManager.UnitHeap,
      --#                                          ContextManager.UnitStack,
      --#                                          CurrentUnitName,
      --#                                          CurrentUnitType,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          InheritPtr,
      --#                                          LexTokenManager.StringTable,
      --#                                          SPARK_IO.File_Sys,
      --#                                          STree.Table,
      --#                                          UnitName &
      --#         IndexManager.IndexTable,
      --#         IndexManager.LastChar,
      --#         LexTokenManager.StringTable from *,
      --#                                          CommandLineData.Content,
      --#                                          ContextManager.UnitHeap,
      --#                                          ContextManager.UnitStack,
      --#                                          CurrentUnitName,
      --#                                          CurrentUnitType,
      --#                                          InheritPtr,
      --#                                          LexTokenManager.StringTable,
      --#                                          SPARK_IO.File_Sys,
      --#                                          STree.Table,
      --#                                          UnitName &
      --#         InheritPtr                  from *,
      --#                                          STree.Table &
      --#         SparkLex.CurrLine           from *,
      --#                                          CommandLineData.Content,
      --#                                          ContextManager.FileHeap,
      --#                                          ContextManager.UnitHeap,
      --#                                          ContextManager.UnitStack,
      --#                                          CurrentUnitName,
      --#                                          CurrentUnitType,
      --#                                          InheritPtr,
      --#                                          LexTokenManager.StringTable,
      --#                                          SPARK_IO.File_Sys,
      --#                                          STree.Table,
      --#                                          UnitName;
      is
         RequiredUnit      : LexTokenLists.Lists;
         NewUnitName       : LexTokenLists.Lists;
         Found             : Boolean;
      begin
         Trace ("In MainLoop.HandleSubUnit");
         RequiredUnits.Next (InheritPtr, RequiredUnit, Found);
         if Found then
            NewUnitName := RequiredUnit;
            LexTokenLists.Append (NewUnitName, UnitName.Content (1));
            ContextManager.SetUnitName (ContextManager.CurrentUnit,
                                        NewUnitName,
                                        ContextManager.SubUnit);
            if FoundTopUnit (CurrentUnitName, CurrentUnitType,
                             NewUnitName, ContextManager.SubUnit)
            then
               if RequiredUnit.Length = 1 then
                  AddRequiredUnit (RequiredUnit, ContextManager.BodySet);
               else
                  AddRequiredUnit (RequiredUnit, ContextManager.SubUnitSet);
               end if;
            else
               ReplaceTopUnit (CurrentUnitName, CurrentUnitType);
            end if;
         end if;
      end HandleSubUnit;

      procedure LookAtParseTree
      --# global in     ContextManager.UnitHeap;
      --#        in     ContextManager.UnitStack;
      --#        in     STree.Table;
      --#           out InheritPtr;
      --#           out UnitName;
      --#           out UnitType;
      --# derives InheritPtr,
      --#         UnitName,
      --#         UnitType   from ContextManager.UnitHeap,
      --#                         ContextManager.UnitStack,
      --#                         STree.Table;
      is
         ParseTree         : STree.SyntaxNode;
      begin
         ContextManager.GetParseTree (ContextManager.CurrentUnit, ParseTree);
         RequiredUnits.Init (ParseTree, InheritPtr, UnitType, UnitName);
      end LookAtParseTree;

   begin -- SetUpRequiredUnits
      Trace ("In MainLoop.SetUpRequiredUnits");
      LookAtParseTree;
      ContextManager.GetUnitName (ContextManager.CurrentUnit, CurrentUnitName, CurrentUnitType);

      if LexTokenLists.EqUnit (UnitName, LexTokenLists.NullList)
        and then UnitType /= ContextManager.InterUnitPragma
      then
         ContextManager.SetUnitStatus (ContextManager.CurrentUnit, ContextManager.NoUnitEntry);
         ReplaceTopUnit (CurrentUnitName, CurrentUnitType);
      else
         --# accept F, 10, InheritPtr, "Not required here";
         case UnitType is
            when ContextManager.MainProgram =>
               HandleMainProgram;
            when ContextManager.PackageSpecification =>
               HandlePackageSpecification;
            when ContextManager.PackageBody =>
               HandlePackageBody;
            when ContextManager.SubUnit =>
               HandleSubUnit;
            when ContextManager.GenericDeclaration =>
               HandleGenericDeclaration;
            when ContextManager.GenericPackageInstantiation =>
               HandleGenericPackageInstantiation;
            when ContextManager.GenericSubprogramBody =>
               HandleGenericSubprogramBody;
            when ContextManager.InterUnitPragma =>
               HandleInterUnitPragma;
            when ContextManager.InvalidUnit |
              ContextManager.PackageOrGenericDeclaration =>
               null;
         end case;
         --# end accept;
      end if;
   end SetUpRequiredUnits;

   procedure ProcessUnits (FileDescriptor : in ContextManager.FileDescriptors)
   --# global in     CommandLineData.Content;
   --#        in out ContextManager.FileHeap;
   --#        in out ContextManager.UnitHeap;
   --#        in out ContextManager.UnitStack;
   --#        in out Declarations.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Graph.Table;
   --#        in out IndexManager.IndexTable;
   --#        in out IndexManager.LastChar;
   --#        in out LexTokenManager.StringTable;
   --#        in out SparkLex.CurrLine;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out StmtStack.S;
   --#        in out STree.Table;
   --#        in out VCG.Invoked;
   --# derives ContextManager.FileHeap,
   --#         ContextManager.UnitHeap,
   --#         ContextManager.UnitStack,
   --#         Declarations.State,
   --#         Dictionary.Dict,
   --#         ErrorHandler.ErrorContext,
   --#         Graph.Table,
   --#         IndexManager.IndexTable,
   --#         IndexManager.LastChar,
   --#         LexTokenManager.StringTable,
   --#         SparkLex.CurrLine,
   --#         SPARK_IO.File_Sys,
   --#         Statistics.TableUsage,
   --#         StmtStack.S,
   --#         STree.Table,
   --#         VCG.Invoked                 from *,
   --#                                          CommandLineData.Content,
   --#                                          ContextManager.FileHeap,
   --#                                          ContextManager.UnitHeap,
   --#                                          ContextManager.UnitStack,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          FileDescriptor,
   --#                                          LexTokenManager.StringTable,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table;
   is
      UnitDescriptor : ContextManager.UnitDescriptors;
      Severity       : ErrorHandler.ErrorLevel;
   begin -- ProcessUnits
      --# accept F, 10, UnitDescriptor, "Final assignment ineffective OK";
      loop
         exit when ContextManager.GetFileStatus (FileDescriptor) =
            ContextManager.FileEnd;
         CreateUnitContext (FileDescriptor, UnitDescriptor);
         ContextManager.SetRedType (UnitDescriptor,
                                    CommandLineData.Content.RedType);
         ContextManager.PushUnit (UnitDescriptor);
         loop
            exit when ContextManager.CurrentUnit = ContextManager.NullUnit;
            case ContextManager.GetUnitStatus (ContextManager.CurrentUnit) is
               when ContextManager.NoUnitEntry => null;
               when ContextManager.UnitCreated =>
                  ParseCurrentUnit;
                  ErrorHandler.GetErrorSeverity (Severity);
                  if Severity /= ErrorHandler.Fatal then
                     SetUpRequiredUnits;
                  end if;
               when ContextManager.UnitDeferred =>
                  ContextManager.SetUnitStatus (ContextManager.CurrentUnit,
                                                ContextManager.UnitParsed);
                  SetUpRequiredUnits;
               when ContextManager.UnitParsed =>
                  --# accept F, 41, "Stable expression OK here";
                  if CommandLineData.Content.SyntaxOnly then
                     ContextManager.PopUnit (UnitDescriptor);
                  else
                     AnalyseCurrentUnit;
                  end if;
                  --# end accept;
               when ContextManager.UnitAnalysed |
                    ContextManager.NoIndexFile .. ContextManager.UnableToLocate =>
                  ContextManager.PopUnit (UnitDescriptor);
            end case;
         end loop;
      end loop;
   end ProcessUnits;

   procedure CloseFile (FileDescriptor      : in ContextManager.FileDescriptors;
                        ReportFileErrorList : in SPARK_IO.File_Type;
                        ReportNeeded        : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.UnitHeap;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     SparkHTML.GenerateHTML;
   --#        in out ContextManager.FileHeap;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --#        in out XMLReport.State;
   --# derives ContextManager.FileHeap,
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys,
   --#         XMLReport.State           from CommandLineData.Content,
   --#                                        ContextManager.FileHeap,
   --#                                        ContextManager.UnitHeap,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FileDescriptor,
   --#                                        LexTokenManager.StringTable,
   --#                                        ReportFileErrorList,
   --#                                        ReportNeeded,
   --#                                        SparkHTML.GenerateHTML,
   --#                                        SPARK_IO.File_Sys,
   --#                                        XMLReport.State;
   is
   begin
      if ContextManager.ListingReqt (FileDescriptor) and not CommandLineData.Content.NoListings then
         OutputListingFile (FileDescriptor);
         if CommandLineData.Content.HTML then
            SparkHTML.GenListingHTML (FileDescriptor);
         end if;
      end if;
      if ReportNeeded then
         AppendToReportFile (ReportFileErrorList,
                             Error_Types.ForReportSelectedFiles,
                             FileDescriptor);
         ContextManager.SetErrorsReported (FileDescriptor);
      end if;
   end CloseFile;

   procedure ProcessFiles
   is
      FileDescriptor : ContextManager.FileDescriptors;
      FileFound,
      ConfigSuccess,
      ConfigRead,
      DoListing      : Boolean;
      FileName,
      TheFileName,
      TheListingName : EStrings.T;
      TheMetaFile    : MetaFile.MetaFiles;
      ReportFileErrorList : SPARK_IO.File_Type := SPARK_IO.Null_File;

      procedure OpenErrorList
      --# global in out ReportFileErrorList;
      --#        in out SPARK_IO.File_Sys;
      --# derives ReportFileErrorList,
      --#         SPARK_IO.File_Sys   from ReportFileErrorList,
      --#                                  SPARK_IO.File_Sys;
      is
         Unused : SPARK_IO.File_Status;
      begin
         --# accept F, 10, Unused, "Not required here" &
         --#        F, 33, Unused, "Not required here";
         SPARK_IO.Create (ReportFileErrorList, 0, "", "",
                           --to get
                          Unused);
      end OpenErrorList;
      --assignment to Unused is ineffective, consider testing and calling system errors

      procedure TraceMetaFileNames
      --# global in CommandLineData.Content;
      --#        in DoListing;
      --#        in FileFound;
      --#        in TheFileName;
      --#        in TheListingName;
      --# derives null from CommandLineData.Content,
      --#                   DoListing,
      --#                   FileFound,
      --#                   TheFileName,
      --#                   TheListingName;
      is
         --# hide TraceMetaFileNames;
      begin
         if FileFound and CommandLineData.Content.Debug.FileNames then
            SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                                 "MetaFile.NextName returns:", 0);

            PrintFileName (SPARK_IO.Standard_Output,
                           TheFileName,
                           CommandLineData.Content.PlainOutput);
            if DoListing then
               SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                                    ", ", 0);

               PrintFileName (SPARK_IO.Standard_Output,
                              TheListingName,
                              CommandLineData.Content.PlainOutput);
            else
               SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                                  ", No listing file", 0);
            end if;
            ScreenEcho.New_Line (1);
         end if;
      end TraceMetaFileNames;

   begin --ProcessFiles
      InitialiseProcessing;
      OpenErrorList;
      ErrorHandler.ReadWarningFile;
      Dictionary.ReadTargetDataFile;
      --# accept F, 10, ConfigRead, "Not required here" &
      --#        F, 33, ConfigRead, "Not required here";
      ConfigFile.ReadConfigFile (ConfigRead, ConfigSuccess);
      if ConfigSuccess then
         for CurrentSource in CommandLineData.SourceFilePositions
           range 1 .. CommandLineData.Content.NumberSource
         loop
            FileName := CommandLineData.Content.SourceFileList (CurrentSource).SourceFileName;
            if FileName.Content (1) = '@' then --process a meta file
               MetaFile.Create (FileName,
                                TheMetaFile);
               loop
                  MetaFile.NextName (TheMetaFile,
                                       -- to get
                                     TheFileName,
                                     DoListing,
                                     TheListingName,
                                     FileFound);
                  exit when not FileFound;

                  TraceMetaFileNames;

                  PrepareNextArgumentFile (TheFileName,
                                           DoListing,
                                           TheListingName,
                                             -- to get
                                           FileDescriptor);
                  OpenFile (FileDescriptor);
                  if ContextManager.GetFileStatus (FileDescriptor) /=
                    ContextManager.UnableToOpen
                  then
                     ProcessUnits (FileDescriptor);
                     CloseFile (FileDescriptor,
                                ReportFileErrorList,
                                CommandLineData.Content.Report);
                  end if;
               end loop;

            else --its a single source file
               TheFileName := FileName;
               TheListingName := CommandLineData.Content.SourceFileList (CurrentSource).ListingFileName;
               DoListing := CommandLineData.Content.SourceFileList (CurrentSource).Listing;
               PrepareNextArgumentFile (TheFileName,
                                        DoListing,
                                        TheListingName,
                                          -- to get
                                        FileDescriptor);
               OpenFile (FileDescriptor);
               if ContextManager.GetFileStatus (FileDescriptor) /=
                 ContextManager.UnableToOpen
               then
                  ProcessUnits (FileDescriptor);
                  CloseFile (FileDescriptor,
                             ReportFileErrorList,
                             CommandLineData.Content.Report);
               end if;
            end if;
         end loop;  --each file on command line
      else
         EchoWarningOrNote (CommandLineData.Content.TargetConfigFile,
                            "Warning - analysis aborted due to errors in target configuration file");
      end if;
      LexTokenManager.ReportUsage;
      Dictionary.ReportUsage;
      STree.ReportUsage;
      --# accept F, 10, ReportFileErrorList, "Final assignment on Close";
      OutputReportFile (ReportFileErrorList);
      --# end accept;
      OutputDictionaryFile;
      if CommandLineData.Content.SyntaxOnly then
         EchoWarningOrNote (CommandLineData.Content.SourceFileList (1).SourceFileName,
                            "Warning - No semantic checks carried out, text " &
                            "may not be legal SPARK");
      end if;
      if CommandLineData.Content.RedType.VCs and then -- Any sort of VC generation requested
        not VCG.Invoked and then                      -- and VCG not actually invoked on any body
        ErrorHandler.GetErrorsType < 3 then           -- and no syntax or semantic errors

         EchoWarningOrNote (CommandLineData.Content.SourceFileList (1).SourceFileName,
                            "Warning - VC generation requested but no bodies presented. No VCs generated.");
      end if;


      if not CommandLineData.Content.DoInformationFlow then
         EchoWarningOrNote (CommandLineData.Content.SourceFileList (1).SourceFileName,
                            "Note - Information flow analysis not carried out");
      end if;
      if CommandLineData.IsSpark83 then
         EchoWarningOrNote (CommandLineData.Content.SourceFileList (1).SourceFileName,
                            "Note - Ada83 language rules selected");
      end if;

      ErrorHandler.EchoTotalErrorCount;

      if CommandLineData.Content.Echo and not CommandLineData.Content.Brief then
         ScreenEcho.New_Line (2);
         ScreenEcho.Put_Line ("-----------End of SPARK Examination--------------------------------");
      end if;
   end ProcessFiles;
end MainLoop;
