-- $Id: lists.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


package body Lists
is
   NoNext        : constant List := 0;
   ListIndicator : constant Integer := -1;

   ---------------------------------------------------------------------------

   procedure Init (Heap : out ListHeap)
   is
   begin
      -- Partial initialization for now, since this is expensive on VAX/VMS.
      -- Re-think when VMS is retired.

      --# accept F, 32,        Heap.HeapArray, "Partial initialization OK" &
      --#        F, 31,        Heap.HeapArray, "Partial initialization OK" &
      --#        F, 602, Heap, Heap.HeapArray, "Partial initialization OK";
      Heap.HighMark := 0;
      Heap.FirstFree := NoNext;
   end Init;

   ---------------------------------------------------------------------------

   procedure NewList (Heap    : in out ListHeap;
                      TheList :    out List;
                      Ok      :    out Boolean)
   is
      TheListLocal : List;

   begin
      if Heap.HighMark < List'Last then
         --array not used up yet
         Heap.HighMark := Heap.HighMark + 1;
         TheListLocal := Heap.HighMark;
         Heap.HeapArray (TheListLocal).Next := NoNext;
         Heap.HeapArray (TheListLocal).Value := ListIndicator;
         TheList := TheListLocal;
         Ok := True;

      elsif Heap.FirstFree = NoNext then
         TheList := NullList;
         Ok := False;

      else
         TheListLocal := Heap.FirstFree;
         Heap.FirstFree := Heap.HeapArray (Heap.FirstFree).Next;
         Heap.HeapArray (TheListLocal).Next := NoNext;
         Heap.HeapArray (TheListLocal).Value := ListIndicator;
         TheList := TheListLocal;
         Ok := True;
      end if;
   end NewList;

   ---------------------------------------------------------------------------

   procedure AddValue (Heap           : in out ListHeap;
                       TheList        : in     List;
                       Value          : in     Natural;
                       AlreadyPresent :    out Boolean;
                       Ok             :    out Boolean)
   is
      Try,
      TryNext,
      AddPoint : List;
      Done     : Boolean;
   begin
      AlreadyPresent := False;
      Ok := False;
      if TheList /= NullList then
         Try := TheList;
         if Heap.HeapArray (Try).Value = ListIndicator then  --list is valid
            TryNext := Heap.HeapArray (Try).Next;
            Done := False;
            loop
               if TryNext = NoNext or else
                  Heap.HeapArray (TryNext).Value > Value
               then
                  if Heap.HighMark < List'Last then
                     -- Array not used up yet
                     Ok := True;
                     Done := True;
                     Heap.HighMark := Heap.HighMark + 1;
                     AddPoint := Heap.HighMark;
                     Heap.HeapArray (AddPoint).Value := Value;
                     Heap.HeapArray (AddPoint).Next := TryNext;
                     Heap.HeapArray (Try).Next := AddPoint;
                  elsif Heap.FirstFree = NoNext then --heap is full
                     Done := True;
                  else
                     Ok := True;
                     Done := True;
                     AddPoint := Heap.FirstFree;
                     Heap.FirstFree := Heap.HeapArray (Heap.FirstFree).Next;
                     Heap.HeapArray (AddPoint).Value := Value;
                     Heap.HeapArray (AddPoint).Next := TryNext;
                     Heap.HeapArray (Try).Next := AddPoint;
                  end if;

               elsif Heap.HeapArray (TryNext).Value = Value then
                  Ok := True;
                  Done := True;
                  AlreadyPresent := True;
               end if;

               exit when Done;

               Try := TryNext;
               TryNext := Heap.HeapArray (Try).Next;
            end loop;
         end if;
      end if;
   end AddValue;

   ---------------------------------------------------------------------------

   procedure GetFirst (Heap    : in out ListHeap;
                       TheList : in out List;
                       Value   :    out Natural;
                       Empty   :    out Boolean;
                       Ok      :    out Boolean)
   is
      Ptr : List;
   begin
      if TheList = NullList then -- can't get value from null list
         Ok := False;
         Empty := True;
         Value := 0;
      elsif Heap.HeapArray (TheList).Value /= ListIndicator then -- list invalid
         Ok := False;
         Empty := True;
         Value := 0;
      else -- valid list pointer
         Ptr := Heap.HeapArray (TheList).Next; -- point to first element
         if Ptr = NoNext then -- list is empty
            Ok := True; --list is ok its just empty!
            Empty := True;
            Value := 0;
         else  --list is not empty
            Ok := True;
            Empty := False;
            Value := Heap.HeapArray (Ptr).Value;
            Heap.HeapArray (Ptr).Value := ListIndicator;
            Heap.HeapArray (TheList).Value := 0;
            Heap.HeapArray (TheList).Next  := Heap.FirstFree;
            Heap.FirstFree := TheList;
            TheList := Ptr;
         end if;
      end if;
   end GetFirst;

   ---------------------------------------------------------------------------

   procedure DeleteList (Heap    : in out ListHeap;
                         TheList : in out List)
   is
      Ptr,
      FreePtr : List;
   begin
      if TheList /= NullList then -- can't delete null list
         Ptr := TheList;
         if Heap.HeapArray (Ptr).Value = ListIndicator then
            Heap.HeapArray (Ptr).Value := 0;
            FreePtr := Heap.FirstFree;
            Heap.FirstFree := Ptr;
            while Heap.HeapArray (Ptr).Next /= NoNext loop
               Ptr := Heap.HeapArray (Ptr).Next;
            end loop;
            Heap.HeapArray (Ptr).Next := FreePtr;
         end if;
      end if;
      TheList := NullList;
   end DeleteList;

   ---------------------------------------------------------------------------

   function IsMember (Heap    : ListHeap;
                      TheList : List;
                      Value   : Natural) return Boolean
   is
      Result : Boolean;
      Try    : List;
   begin
      if TheList = NullList then --can't check memebership of null list
         Result := False;
      else
         Try := TheList; -- 782 - Deleted redundant type conversion
         if Heap.HeapArray (Try).Value = ListIndicator then
            Result := False;
            while Heap.HeapArray (Try).Next /= NoNext loop
               Try := Heap.HeapArray (Try).Next;
               if Heap.HeapArray (Try).Value = Value then
                  Result := True;
                  exit;
               end if;
            end loop;
         else -- value supplied was not a valid list pointer
            Result := False;
         end if;
      end if;

      return Result;

   end IsMember;

   ----------------------------------------------------------------------
   -- Temporary Test Procedure
   ----------------------------------------------------------------------

   --procedure PrintList(Heap    : ListHeap;
   --                    TheList : List)
   --is
   --  Ptr : List;
   --  package Int_IO is new Integer_IO(integer); use Int_io;
   --
   --begin
   --  if TheList = NullList then --can't add to null list
   --    Put_Line("Null List");
   --  else
   --    if Heap.HeapArray(TheList).Value /= ListIndicator then  --list invalid
   --      Put_Line("Invalid List");
   --    else -- valid list pointer
   --      if Heap.HeapArray(TheList).Next = NoNext then
   --        Put_Line("Empty List");
   --      else
   --        Ptr := Heap.HeapArray(TheList).Next;
   --        while Ptr /= NoNext loop
   --          Put(Heap.HeapArray(Ptr).Value);
   --          Put(" ");
   --          Ptr := Heap.HeapArray(Ptr).Next;
   --        end loop;
   --      end if;
   --    end if;
   --    New_Line;
   --  end if;
   --end PrintList;
   --
   ----------------------------------------------------------------------
   --
   --procedure PrintHeap(Heap : ListHeap)
   --is
   --  package Int_IO is new Integer_IO(integer); use Int_io;
   --
   --begin
   --  put("FirstFree="); Put(integer(Heap.FirstFree)); New_Line;
   --  for I in List range 1..List'Last loop
   --    put(integer(I));
   --    put(integer(Heap.HeapArray(I).Value));
   --    put(integer(HEap.HeapArray(I).Next));
   --    New_Line;
   --  end loop;
   --end PrintHeap;
   --
   ----------------------------------------------------------------------

end Lists;
