-- $Id: lextokenmanager.adb 11889 2008-12-12 15:49:12Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with Ada.Characters.Handling,
     Ada.Characters.Latin_1,
     SystemErrors,
     Statistics;

package body LexTokenManager
is
   type TableContents is array (LexString) of Character;

   subtype HashIndex is Natural range 0 .. ExaminerConstants.StringHashSize - 1;
   type HashTableStruct is array (HashIndex) of LexString;

   type TableStructure is record
      NextVacant : LexString;
      HashTable  : HashTableStruct;
      Contents   : TableContents;
   end record;

   -- The string table uses hashing with chaining, ie if there is a clash then
   -- the next item is rehashed to a new location and a link is provided to it.
   -- This link occupies the first 3 bytes in each entry (because the max string
   -- table size is 2**21).
   StrLinkLen : constant LexString := 3;

   StringTable : TableStructure;

   procedure Hash (Str      : in     ELStrings.Line;
                   StartPos : in     ELStrings.Positions;
                   EndPos   : in     ELStrings.Positions;
                   HashVal  :    out HashIndex)
      --# derives HashVal from EndPos,
      --#                      StartPos,
      --#                      Str;
   is
      Val : HashIndex;
   begin
      Val := 0;
      for Ix in ELStrings.Positions range StartPos .. EndPos loop
         Val := (Val +
                 HashIndex (Character'Pos
                            (Ada.Characters.Handling.To_Upper
                             (Str (Ix))))) mod ExaminerConstants.StringHashSize;
      end loop;
      HashVal := Val;
   end Hash;

   -- Performs case insensitive comparison of two LexStrings and
   -- returns which string comes first when ordered alphabetically
   -- (See also CompStr and CompStrCaseSensitive, below)
   function LexStringCaseInsensitiveCompare
      (LexStr1 : in LexString;
       LexStr2 : in LexString) return StrCompResult
   is
      Result   : StrCompResult := StrEq;
      Finished : Boolean := False;
      Index1   : LexString;
      Index2   : LexString;

   begin

      if LexStr1 /= NullString and LexStr2 /= NullString then

         -- This check should never fail, but would need precondition and proof to show this.
         if (LexStr1 <= LexString'Last - StrLinkLen and
             LexStr2 <= LexString'Last - StrLinkLen) then

            Index1 := LexStr1 + StrLinkLen;
            Index2 := LexStr2 + StrLinkLen;

         else

            Index1 := NullString;
            Index2 := NullString;
            SystemErrors.FatalError (SystemErrors.StringTableOverflow, "Invalid input string.");

         end if;

         while not Finished loop

            -- If we've reached the end of both strings then they are equal
            if (StringTable.Contents (Index1) = Ada.Characters.Latin_1.NUL and
                StringTable.Contents (Index2) = Ada.Characters.Latin_1.NUL) then

               Result := StrEq;
               Finished := True;

            -- If the current character in the first string is closer to the start of the
            -- alphabet then the first string is alphabetically first. Note that Ada.Characters.Latin_1.NUL
            -- comes before 'A' so this will return StrFirst if we've reached the end of the
            -- first string but not the second.
            elsif (Character'Pos (Ada.Characters.Handling.To_Upper (StringTable.Contents (Index1))) <
                   Character'Pos (Ada.Characters.Handling.To_Upper (StringTable.Contents (Index2)))) then

               Result := StrFirst;
               Finished := True;

            -- Reverse of the previous comparison...
            elsif (Character'Pos (Ada.Characters.Handling.To_Upper (StringTable.Contents (Index1))) >
                   Character'Pos (Ada.Characters.Handling.To_Upper (StringTable.Contents (Index2)))) then

               Result := StrSecond;
               Finished := True;

            else

               -- If we have reached this point and one of the strings has hit the top of the
               -- table without its final char being null then something has gone wrong, but
               -- guard avoids possible RTE.
               if (Index1 < LexString'Last and Index2 < LexString'Last) then
                  Index1 := Index1 + 1;
                  Index2 := Index2 + 1;
               else
                  SystemErrors.FatalError (SystemErrors.StringTableOverflow, "Attempt to index past end of table.");
               end if;

            end if;

         end loop;

      -- Deal with the cases where at least one string is null
      elsif LexStr1 = NullString and LexStr2 = NullString then
         Result := StrEq;
      elsif LexStr1 = NullString then
         Result := StrFirst;
      else
         Result := StrSecond;
      end if;

      return Result;

   end LexStringCaseInsensitiveCompare;

   procedure CompStr (Str          : in     ELStrings.Line;
                      StartPos     : in     ELStrings.Positions;
                      EndPos       : in     ELStrings.Positions;
                      LexStr       : in     LexString;
                      StringsEqual :    out Boolean)
   --# global in StringTable;
   --# derives StringsEqual from EndPos,
   --#                           LexStr,
   --#                           StartPos,
   --#                           Str,
   --#                           StringTable;
   is
      LX : Natural;
      SX : LexString;
      StrEqual : Boolean;
   begin
      if LexStr = NullString then
         StrEqual := False;
      else
         SX := LexStr + StrLinkLen;
         StrEqual := True;
         LX := StartPos;
         while StrEqual and LX <= EndPos loop
            if Ada.Characters.Handling.To_Upper (Str (LX)) =
               Ada.Characters.Handling.To_Upper (StringTable.Contents (SX)) then
               LX := LX + 1; SX := SX + 1;
            else
               StrEqual := False;
            end if;
         end loop;
         if StringTable.Contents (SX) /= Ada.Characters.Latin_1.NUL then
            StrEqual := False;
         end if;
      end if;
      StringsEqual := StrEqual;
   end CompStr;

   procedure CompStrCaseSensitive
      (Str          : in     ELStrings.Line;
       StartPos     : in     ELStrings.Positions;
       EndPos       : in     ELStrings.Positions;
       LexStr       : in     LexString;
       StringsEqual :    out Boolean)
   --# global in StringTable;
   --# derives StringsEqual from EndPos,
   --#                           LexStr,
   --#                           StartPos,
   --#                           Str,
   --#                           StringTable;
   is
      LX : Natural;
      SX : LexString;
      StrEqual : Boolean;
   begin
      if LexStr = NullString then
         StrEqual := False;
      else
         SX := LexStr + StrLinkLen;
         StrEqual := True;
         LX := StartPos;
         while StrEqual and LX <= EndPos loop
            if Str (LX) = StringTable.Contents (SX) then
               LX := LX + 1; SX := SX + 1;
            else
               StrEqual := False;
            end if;
         end loop;
         if StringTable.Contents (SX) /= Ada.Characters.Latin_1.NUL then
            StrEqual := False;
         end if;
      end if;
      StringsEqual := StrEqual;
   end CompStrCaseSensitive;

   procedure SetStrLink (LexStr, StrLink : in LexString)
   --# global in out StringTable;
   --# derives StringTable from *,
   --#                          LexStr,
   --#                          StrLink;
   is
   begin
      StringTable.Contents (LexStr)     := Character'Val (StrLink / 2**14);
      StringTable.Contents (LexStr + 1) := Character'Val ((StrLink / 2**7) mod 2**7);
      StringTable.Contents (LexStr + 2) := Character'Val (StrLink mod 2**7);
   end SetStrLink;

   procedure GetStrLink (LexStr  : in     LexString;
                         StrLink :    out LexString)
   --# global in StringTable;
   --# derives StrLink from LexStr,
   --#                      StringTable;
   is
      Link : LexString;
   begin
      Link := Character'Pos (StringTable.Contents (LexStr)) * 2**14;
      Link := Link + Character'Pos (StringTable.Contents (LexStr + 1)) * 2**7;
      Link := Link + Character'Pos (StringTable.Contents (LexStr + 2));
      StrLink := Link;
   end GetStrLink;

   procedure CopyStr (Str      : in     ELStrings.Line;
                      StartPos : in     ELStrings.Positions;
                      EndPos   : in     ELStrings.Positions;
                      LexStr   :    out LexString)
   --# global in out StringTable;
   --# derives LexStr      from EndPos,
   --#                          StartPos,
   --#                          StringTable &
   --#         StringTable from *,
   --#                          EndPos,
   --#                          StartPos,
   --#                          Str;
   is
      SX       : LexString;
   begin
      if StringTable.NextVacant = NullString or
         ((StringTable.NextVacant + StrLinkLen) +
         LexString ((EndPos - StartPos) + 1)) + 1 > LexString'Last then

         LexStr := NullString;
      else
         -- There is space for link, string and string terminator
         SX := StringTable.NextVacant;
         LexStr := SX;
         SetStrLink (SX, NullString);
         SX := SX + StrLinkLen; -- Skip StringLink.
         for LX in ELStrings.Positions range StartPos .. EndPos loop
            StringTable.Contents (SX) := Str (LX);
            SX := SX + 1;
         end loop;
         StringTable.Contents (SX) := Ada.Characters.Latin_1.NUL;  -- LexString Terminator.
         if SX = LexString'Last then
            StringTable.NextVacant := NullString;
         else
            StringTable.NextVacant := SX + 1;
         end if;
      end if;
   end CopyStr;

   procedure InsertGeneralLexString
      (Str           : in     ELStrings.Line;
       StartPos      : in     ELStrings.Positions;
       EndPos        : in     ELStrings.Positions;
       CaseSensitive : in     Boolean;
       LexStr        :    out LexString)
   --# global in out StringTable;
   --# derives LexStr,
   --#         StringTable from CaseSensitive,
   --#                          EndPos,
   --#                          StartPos,
   --#                          Str,
   --#                          StringTable;
   is
      HashVal                   : HashIndex;
      LOCStr, TokenStr, StrLink : LexString;
      Searching, StrEqual       : Boolean;
   begin
      TokenStr := NullString;
      Hash (Str, StartPos, EndPos, HashVal);
      LOCStr := StringTable.HashTable (HashVal);
      if LOCStr = NullString then  -- No string elements at this entry.
         CopyStr (Str, StartPos, EndPos, TokenStr);
         StringTable.HashTable (HashVal) := TokenStr; -- String element now exists
                                                      -- at this HashTable entry.
      else
         Searching := True;

         if CaseSensitive then
            while Searching loop
               CompStrCaseSensitive (Str, StartPos, EndPos, LOCStr, StrEqual);
               if StrEqual then
                  TokenStr := LOCStr;
                  Searching := False;
               else
                  GetStrLink (LOCStr, StrLink);
                  if StrLink = NullString then
                     CopyStr (Str, StartPos, EndPos, TokenStr);
                     SetStrLink (LOCStr, TokenStr);
                     Searching := False;
                  else
                     LOCStr := StrLink;
                  end if;
               end if;
            end loop;
         else -- not CaseSensitive
            while Searching loop
               CompStr (Str, StartPos, EndPos, LOCStr, StrEqual);
               if StrEqual then
                  TokenStr := LOCStr;
                  Searching := False;
               else
                  GetStrLink (LOCStr, StrLink);
                  if StrLink = NullString then
                     CopyStr (Str, StartPos, EndPos, TokenStr);
                     SetStrLink (LOCStr, TokenStr);
                     Searching := False;
                  else
                     LOCStr := StrLink;
                  end if;
               end if;
            end loop;
         end if; -- CaseSensitive
      end if;
      if TokenStr = NullString then
         SystemErrors.FatalError (SystemErrors.StringTableOverflow, "");
      end if;
      LexStr := TokenStr;
   end InsertGeneralLexString;

   procedure InsertLongLexString
      (Str      : in     ELStrings.Line;
       StartPos : in     ELStrings.Positions;
       EndPos   : in     ELStrings.Positions;
       LexStr   :    out LexString)
   is
   begin
      InsertGeneralLexString (Str           => Str,
                              StartPos      => StartPos,
                              EndPos        => EndPos,
                              CaseSensitive => False,
                              LexStr        => LexStr);
   end InsertLongLexString;

   procedure InsertLexString
      (Str      : in     EStrings.Line;
       StartPos : in     EStrings.Positions;
       EndPos   : in     EStrings.Positions;
       LexStr   :    out LexString)
   is
      LocalStr : ELStrings.Line;
   begin
      LocalStr := ELStrings.EmptyLine;
      for I in EStrings.Lengths range StartPos .. EndPos loop
         LocalStr (I) := Str (I);
      end loop;
      InsertGeneralLexString (Str           => LocalStr,
                              StartPos      => StartPos,
                              EndPos        => EndPos,
                              CaseSensitive => False,
                              LexStr        => LexStr);
   end InsertLexString;

   procedure InsertCaseSensitiveLexString
      (Str      : in     EStrings.Line;
       StartPos : in     EStrings.Positions;
       EndPos   : in     EStrings.Positions;
       LexStr   :    out LexString)
   is
      LocalStr : ELStrings.Line;
   begin
      LocalStr := ELStrings.EmptyLine;
      for I in EStrings.Lengths range StartPos .. EndPos loop
         LocalStr (I) := Str (I);
      end loop;
      InsertGeneralLexString (Str           => LocalStr,
                              StartPos      => StartPos,
                              EndPos        => EndPos,
                              CaseSensitive => True,
                              LexStr        => LexStr);
   end InsertCaseSensitiveLexString;

   procedure InsertExaminerString
      (Str      : in     EStrings.T;
       LexStr   :    out LexString)
   is
      Line : ELStrings.Line;
   begin
      Line := ELStrings.EmptyLine;
      for I in EStrings.Positions range 1 .. Str.Length loop
         Line (I) := Str.Content (I);
         --# assert I >= 1 and I <= Str.Length and Str.Length >= 1;
      end loop;
      InsertLongLexString (Line, 1, Str.Length, LexStr);
   end InsertExaminerString;

   procedure InsertExaminerLongString
      (Str      : in     ELStrings.T;
       LexStr   :    out LexString)
   is
      Line : ELStrings.Line;
   begin
      Line := ELStrings.EmptyLine;
      for I in ELStrings.Positions range 1 .. Str.Length loop
         Line (I) := Str.Content (I);
         --# assert I >= 1 and I <= Str.Length and Str.Length >= 1;
      end loop;
      InsertLongLexString (Line, 1, Str.Length, LexStr);
   end InsertExaminerLongString;

   procedure InsertCaseSensitiveExaminerString
      (Str      : in     EStrings.T;
       LexStr   :    out LexString)
   is
      Line : ELStrings.Line;
   begin
      Line := ELStrings.EmptyLine;
      for I in EStrings.Positions range 1 .. Str.Length loop
         Line (I) := Str.Content (I);
         --# assert I >= 1 and I <= Str.Length and Str.Length >= 1;
      end loop;
      InsertGeneralLexString (Str           => Line,
                              StartPos      => 1,
                              EndPos        => Str.Length,
                              CaseSensitive => True,
                              LexStr        => LexStr);
   end InsertCaseSensitiveExaminerString;

   procedure InsertCaseSensitiveExaminerLongString
      (Str      : in     ELStrings.T;
       LexStr   :    out LexString)
   is
      Line : ELStrings.Line;
   begin
      Line := ELStrings.EmptyLine;
      for I in ELStrings.Positions range 1 .. Str.Length loop
         Line (I) := Str.Content (I);
         --# assert I >= 1 and I <= Str.Length and Str.Length >= 1;
      end loop;
      InsertGeneralLexString (Str           => Line,
                              StartPos      => 1,
                              EndPos        => Str.Length,
                              CaseSensitive => True,
                              LexStr        => LexStr);
   end InsertCaseSensitiveExaminerLongString;

   procedure LexStringToString (LexStr   : in LexString;
                                Str      : out EStrings.T)
   is
      IX : LexString;
      CH : Character;
      LastCH : Natural;
   begin
      Str := EStrings.EmptyString;
      LastCH := 0;
      if LexStr /= NullString then
         IX := LexStr + StrLinkLen;

         loop
            CH := StringTable.Contents (IX);
            exit when CH = Ada.Characters.Latin_1.NUL;  -- LexString Terminator.
            LastCH := LastCH + 1;
            Str.Content (LastCH) := CH;
            exit when LastCH = EStrings.MaxStringLength;
            IX := IX + 1;
         end loop;

      end if;
      Str.Length := LastCH;
   end LexStringToString;

   procedure LexStringToLongString (LexStr   : in LexString;
                                    Str      : out ELStrings.T)
   is
      IX : LexString;
      CH : Character;
      LastCH : Natural;
   begin
      Str := ELStrings.EmptyString;
      LastCH := 0;
      if LexStr /= NullString then
         IX := LexStr + StrLinkLen;

         loop
            CH := StringTable.Contents (IX);
            exit when CH = Ada.Characters.Latin_1.NUL;  -- LexString Terminator.
            LastCH := LastCH + 1;
            Str.Content (LastCH) := CH;
            exit when LastCH = ELStrings.MaxStringLength;
            IX := IX + 1;
         end loop;

      end if;
      Str.Length := LastCH;
   end LexStringToLongString;

   function LexStringRef (LexStr : LexString) return Natural
   is
   begin
      return Natural (LexStr);
   end LexStringRef;

   function ConvertLexStringRef (LexStrRef : Natural) return LexString
   is
      LexStr : LexString;
   begin
      if LexStrRef > Natural (LexString'Last) then
         LexStr := NullString;
      else
         LexStr := LexString (LexStrRef);
      end if;
      return LexStr;
   end ConvertLexStringRef;

   function IsAttributeToken (Tok     : LexString;
                              Spark95 : Boolean) return Boolean
   is
      subtype Index is LexString range TheFirstToken .. TheLastToken;
      type Tables is array (Index, Boolean) of Boolean;

      ------------------------------------------------------------------
      -- This table defines the sets of attributes that are defined
      -- in both SPARK83 and SPARK95 modes.
      --
      -- Seven floating-point and fixed-point attributes are considered
      -- obsolete but remain as implementation-defined attributes
      -- in SPARK95 mode, following the advice of AARM A.5.3 (72.f)
      -- and A.5.4 (4.b)
      ------------------------------------------------------------------
      AttributeTable : constant Tables := Tables'
         --attribute------------------83----95----------
         (AftToken               => (True, True),
          BaseToken              => (True, True),
          DeltaToken             => (True, True),
          DigitsToken            => (True, True),
          EmaxToken              => (True, True), -- obselete in 95
          EpsilonToken           => (True, True), -- obselete in 95
          FirstToken             => (True, True),
          ForeToken              => (True, True),
          LargeToken             => (True, True), -- obselete in 95
          LastToken              => (True, True),
          LengthToken            => (True, True),
          MachineEmaxToken       => (True, True),
          MachineEminToken       => (True, True),
          MachineMantissaToken   => (True, True),
          MachineOverflowsToken  => (True, True),
          MachineRadixToken      => (True, True),
          MachineRoundsToken     => (True, True),
          MantissaToken          => (True, True), -- obselete in 95
          PosToken               => (True, True),
          PredToken              => (True, True),
          RangeToken             => (True, True),
          SafeEmaxToken          => (True, True), -- obselete in 95
          SafeLargeToken         => (True, True), -- obselete in 95
          SafeSmallToken         => (True, True), -- obselete in 95
          SizeToken              => (True, True),
          SmallToken             => (True, True),
          SuccToken              => (True, True),
          ValToken               => (True, True),
          DenormToken            => (False, True),
          Model_EminToken        => (False, True),
          Model_EpsilonToken     => (False, True),
          Model_MantissaToken    => (False, True),
          Model_SmallToken       => (False, True),
          Safe_FirstToken        => (False, True),
          Safe_LastToken         => (False, True),
          Component_SizeToken    => (False, True),
          MinToken               => (False, True),
          MaxToken               => (False, True),
          Signed_ZerosToken      => (False, True),
          ValidToken             => (False, True),
          AdjacentToken          => (False, True),
          ComposeToken           => (False, True),
          Copy_SignToken         => (False, True),
          Leading_PartToken      => (False, True),
          RemainderToken         => (False, True),
          ScalingToken           => (False, True),
          CeilingToken           => (False, True),
          ExponentToken          => (False, True),
          FloorToken             => (False, True),
          FractionToken          => (False, True),
          MachineToken           => (False, True),
          ModelToken             => (False, True),
          RoundingToken          => (False, True),
          TruncationToken        => (False, True),
          Unbiased_RoundingToken => (False, True),
          AddressToken           => (False, False), -- only used in rep. clauses.
          ModulusToken           => (False, True),
          TailToken              => (True,  True),  -- valid in proof context only
          AppendToken            => (True,  True),  -- valid in proof context only
          AccessToken            => (False, True),  -- valid only if Ravenscar selected
          Always_ValidToken      => (True, True),
          others                 => (False, False));

   begin
      return Tok in Index and then
             AttributeTable (Tok, Spark95);
   end IsAttributeToken;

   procedure InitialiseStringTable
   is
      InitOk    : Boolean;

      -----------------------------------------------------------
      -- This procedure inserts the given String into the String
      -- table and checks that the new LexString returned is as
      -- expected.  We use type String here so that callers may
      -- pass a string literal.
      -----------------------------------------------------------
      procedure Ins (Str         : in     String;
                     ExpectedStr : in     LexString)
      --# global in out InitOk;
      --#        in out StringTable;
      --# derives InitOk      from *,
      --#                          ExpectedStr,
      --#                          Str,
      --#                          StringTable &
      --#         StringTable from *,
      --#                          Str;
      --# pre Str'First = 1 and
      --#     Str'Last >= 1 and
      --#     Str'Last <= EStrings.Line'Last;
      is
         LS     : EStrings.Line;
         NewStr : LexString;
      begin
         LS := EStrings.EmptyLine;
         for I in Positive range 1 .. Str'Last loop
            --# assert Str'First = 1 and
            --#        Str'Last >= 1 and
            --#        Str'Last <= EStrings.Line'Last and
            --#        I >= 1 and
            --#        I <= Str'Last;
            LS (I) := Str (I);
         end loop;
         InsertLexString (LS, 1, Str'Last, NewStr);
         InitOk := InitOk and (NewStr = ExpectedStr);
      end Ins;


      procedure InsertStrings
      --# global in out InitOk;
      --#        in out StringTable;
      --# derives InitOk,
      --#         StringTable from *,
      --#                          StringTable;
      is
      begin
         Ins ("AFT", AftToken);
         Ins ("BASE", BaseToken);
         Ins ("DELTA", DeltaToken);
         Ins ("DIGITS", DigitsToken);
         Ins ("EMAX", EmaxToken);
         Ins ("EPSILON", EpsilonToken);
         Ins ("FIRST", FirstToken);
         Ins ("FORE", ForeToken);
         Ins ("LARGE", LargeToken);
         Ins ("LAST", LastToken);

         --# assert True;

         Ins ("LENGTH", LengthToken);
         Ins ("MACHINE_EMAX", MachineEmaxToken);
         Ins ("MACHINE_EMIN", MachineEminToken);
         Ins ("MACHINE_MANTISSA", MachineMantissaToken);
         Ins ("MACHINE_OVERFLOWS", MachineOverflowsToken);
         Ins ("MACHINE_RADIX", MachineRadixToken);
         Ins ("MACHINE_ROUNDS", MachineRoundsToken);
         Ins ("MANTISSA", MantissaToken);
         Ins ("POS", PosToken);
         Ins ("PRED", PredToken);

         --# assert True;

         Ins ("RANGE", RangeToken);
         Ins ("SAFE_EMAX", SafeEmaxToken);
         Ins ("SAFE_LARGE", SafeLargeToken);
         Ins ("SAFE_SMALL", SafeSmallToken);
         Ins ("SIZE", SizeToken);
         Ins ("SMALL", SmallToken);
         Ins ("SUCC", SuccToken);
         Ins ("VAL", ValToken);
         Ins ("LEFT", LeftToken);
         Ins ("RIGHT", RightToken);

         --# assert True;

         Ins ("TRUE", TrueToken);
         Ins ("FALSE", FalseToken);
         Ins ("0", ZeroValue);
         Ins ("1", OneValue);
         Ins ("superindex", SuperIndexToken);
         Ins ("Interface", InterfaceToken);
         Ins ("Import", ImportToken);
         Ins ("Link_Name", Link_NameToken);
         Ins ("External_Name", External_NameToken);
         Ins ("Entity", EntityToken);

         --# assert True;

         Ins ("Convention", ConventionToken);
         Ins ("Elaborate_Body", Elaborate_BodyToken);
         Ins ("Ada", AdaToken);
         Ins ("Denorm", DenormToken);
         Ins ("Model_Emin", Model_EminToken);
         Ins ("Model_Epsilon", Model_EpsilonToken);
         Ins ("Model_Mantissa", Model_MantissaToken);
         Ins ("Model_Small", Model_SmallToken);
         Ins ("Safe_First", Safe_FirstToken);
         Ins ("Safe_Last", Safe_LastToken);

         --# assert True;

         Ins ("Component_Size", Component_SizeToken);
         Ins ("Min", MinToken);
         Ins ("Max", MaxToken);
         Ins ("Signed_Zeros", Signed_ZerosToken);
         Ins ("Valid", ValidToken);
         Ins ("Characters", CharactersToken);
         Ins ("Latin_1", Latin_1Token);
         Ins ("Adjacent", AdjacentToken);
         Ins ("Compose", ComposeToken);
         Ins ("Copy_Sign", Copy_SignToken);

         --# assert True;

         Ins ("Leading_Part", Leading_PartToken);
         Ins ("Remainder", RemainderToken);
         Ins ("Scaling", ScalingToken);
         Ins ("Ceiling", CeilingToken);
         Ins ("Exponent", ExponentToken);
         Ins ("Floor", FloorToken);
         Ins ("Fraction", FractionToken);
         Ins ("Machine", MachineToken);
         Ins ("Model", ModelToken);
         Ins ("Rounding", RoundingToken);

         --# assert True;

         Ins ("Truncation", TruncationToken);
         Ins ("Unbiased_Rounding", Unbiased_RoundingToken);
         Ins ("Address", AddressToken);
         Ins ("Modulus", ModulusToken);
         Ins ("Tail", TailToken);
         Ins ("Append", AppendToken);
         Ins ("System", SystemToken);
         Ins ("Min_Int", Min_IntToken);
         Ins ("Max_Int", Max_IntToken);
         Ins ("Max_Binary_Modulus", Max_Binary_ModulusToken);

         --# assert True;

         Ins ("Max_Base_Digits", Max_Base_DigitsToken);
         Ins ("Max_Digits", Max_DigitsToken);
         Ins ("Max_Mantissa", Max_MantissaToken);
         Ins ("Fine_Delta", Fine_DeltaToken);
         Ins ("Null_Address", Null_AddressToken);
         Ins ("Storage_Unit", Storage_UnitToken);
         Ins ("Word_Size", Word_SizeToken);
         Ins ("Any_Priority", Any_PriorityToken);
         Ins ("Priority", PriorityToken);
         Ins ("Interrupt_Priority", Interrupt_PriorityToken);

         --# assert True;

         Ins ("Default_Priority", Default_PriorityToken);
         Ins ("Atomic", AtomicToken);
         Ins ("Real_Time", Real_TimeToken);
         Ins ("Inherit", InheritToken);
         Ins ("Synchronous_Task_Control", Synchronous_Task_ControlToken);
         Ins ("Attach_Handler", Attach_HandlerToken);
         Ins ("Interrupt_Handler", Interrupt_HandlerToken);
         Ins ("Interrupts", InterruptsToken);
         Ins ("Access", AccessToken);
         Ins ("Atomic_Components", Atomic_ComponentsToken);

         --# assert True;

         Ins ("Volatile_Components", Volatile_ComponentsToken);
         Ins ("main_program", Main_ProgramToken);
         Ins ("Assert", AssertToken);
         Ins ("overriding", OverridingToken);
         Ins ("Unchecked_Conversion", Unchecked_ConversionToken);
         Ins ("Rule", RuleToken);
         Ins ("NoRule", NoRuleToken);
         Ins ("Always_Valid", Always_ValidToken);

         --# assert True;

         Ins ("Bit_Order", Bit_OrderToken);
         Ins ("High_Order_First", High_Order_FirstToken);
         Ins ("Low_Order_First",  Low_Order_FirstToken);
         Ins ("Default_Bit_Order", Default_Bit_OrderToken);

         --# assert True;

         Ins ("All_Calls_Remote", All_Calls_RemoteToken);
         Ins ("Asynchronous", AsynchronousToken);
         Ins ("Controlled", ControlledToken);
         Ins ("Discard_Names", Discard_NamesToken);
         Ins ("Elaborate", ElaborateToken);
         Ins ("Elaborate_All", Elaborate_AllToken);
         Ins ("Export", ExportToken);
         Ins ("Inline", InlineToken);
         Ins ("Inspection_Point", Inspection_PointToken);
         -- Ins ("Interrupt_Priority", Interrupt_PriorityToken);
         Ins ("Linker_Options", Linker_OptionsToken);
         Ins ("List", ListToken);
         Ins ("Locking_Policy", Locking_PolicyToken);
         Ins ("Normalize_Scalars", Normalize_ScalarsToken);
         Ins ("Optimize", OptimizeToken);
         Ins ("Pack", PackToken);
         Ins ("Page", PageToken);
         Ins ("Preelaborate", PreelaborateToken);
         -- Ins ("Priority", PriorityToken);
         Ins ("Pure", PureToken);
         Ins ("Queueing_Policy", Queueing_PolicyToken);
         Ins ("Remote_Call_Interface", Remote_Call_InterfaceToken);
         Ins ("Remote_Types", Remote_TypesToken);
         Ins ("Restrictions", RestrictionsToken);
         Ins ("Reviewable", ReviewableToken);
         Ins ("Shared_Passive", Shared_PassiveToken);
         Ins ("Storage_Size", Storage_SizeToken);
         Ins ("Suppress", SuppressToken);
         Ins ("Task_Dispatching_Policy", Task_Dispatching_PolicyToken);
         Ins ("Volatile", VolatileToken);

         --# assert True;

         Ins ("Memory_Size", Memory_SizeToken);
         Ins ("Shared", SharedToken);
         -- Ins ("Storage_Unit", Storage_UnitToken);
         Ins ("System_Name", System_NameToken);

         ---------------------------------------------------------
         -- Insert new lex strings here
         -- Remember to adjust TheLastToken in lextokenmanager.aps
         ---------------------------------------------------------

      end InsertStrings;

   begin
      InitOk := True;

      InsertStrings;

      if not InitOk then
         SystemErrors.FatalError (SystemErrors.InvalidInit, "");
      end if;
   end InitialiseStringTable;

   procedure ReportUsage
   is
   begin
      -- table fills from bottom up with no reclamation, so NextVacant indicates
      -- the amount of space used
      Statistics.SetTableUsage (Statistics.StringTable,
                                Integer (StringTable.NextVacant - 1));
   end ReportUsage;

   procedure InsertNat (N      : in     Natural;
                        LexStr :    out LexString)
      is separate;

begin
   StringTable.NextVacant := 1;
   StringTable.HashTable  := HashTableStruct'(others => NullString);
   StringTable.Contents   := TableContents'(others => Ada.Characters.Latin_1.NUL);
end LexTokenManager;
