-- $Id: indexmanager.adb 11889 2008-12-12 15:49:12Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


-- Options for handling generic units are marked with -- GenOption --
-- With the code commented out, generic declarations are indexed as "specification"
-- With the code included, they are indicated by "generic_declaration"
-- The former is probably better because if a package body is found, there is no way
-- of knowing whether a generic spec or normal spec is required.  We know this for a
-- generic subprgoram body because any sunprogram body that is not a main program must
-- be generic.  It seems uneccesarily complicated to hcnage the language of index files
-- so as to say something about generic suprgorams that we can't say about generic
-- packages.  Therefore overloading "specification" seems the ebst option.


with FileSystem,
     Ada.Characters.Latin_1,
     XMLReport,
     CommandLineData,
     LexTokenManager,
     ScreenEcho,
     SystemErrors;

use type LexTokenManager.LexString;

package body IndexManager
is

   type EntryTypes is (SuperIndex,
                       AuxIndex,
                       mainp,
                       pspec,
                        -- GenOption -- GenericDec,
                       pbodi,
                       subunit,
                       ComponentList,
                       InvalidEntryType);

   type LibraryManagerErrors is (ESFileLocation, ESIsIn, ESUnitEntry,
                                 EWIndex, EWUnexpectedSuper, ESComment,
                                 ESAre,
                                 ESComponents,
                                 EWIllegalUnitName,
                                 EWAux,
                                 EWSuper,
                                 ESRecursion);

   subtype IndexSizes is Integer range
      0 .. ExaminerConstants.MaxIndexNumber;

   subtype IndexPositions is Integer range
      1 .. ExaminerConstants.MaxIndexNumber;

   type IndexContents is array (IndexPositions) of EStrings.T;

   type IndexTables is record
      Size    : IndexSizes;
      Content : IndexContents;
   end record;

   type FilePosition is record
      Line : Positive;
      Col  : Positive;
   end record;

   IndexTable : IndexTables;

   LastChar   : Character;

   procedure AddEntry (FileName : in EStrings.T)
   --# global in out IndexTable;
   --# derives IndexTable from *,
   --#                         FileName;
   is
      Found : Boolean;
   begin
      Found := False;

      for I in IndexSizes range 1 .. IndexTable.Size loop
         Found := EStrings.EqString (IndexTable.Content (I), FileName);
         exit when Found;
      end loop;

      if not Found then
         if IndexTable.Size < ExaminerConstants.MaxIndexNumber then
            IndexTable.Size := IndexTable.Size + 1;
            IndexTable.Content (IndexTable.Size) := FileName;
         else
            SystemErrors.FatalError (SystemErrors.IndexStackFull, "");
         end if;
      end if;
   end AddEntry;

   procedure PutError (S : in String;
                       N : in SPARK_IO.File_Type;
                       TokenPosition : in FilePosition;
                       TokenString : in EStrings.T)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                N,
   --#                                S,
   --#                                TokenPosition,
   --#                                TokenString;
   is
      NameLen  : EStrings.Lengths;
      NameCon  : EStrings.Contents;
      Str      : EStrings.T;
      Column   : Positive;
   begin
      -- Adjust column number to take account of lookahead
      if TokenPosition.Col > 1 then
         Column := TokenPosition.Col - 1;
      else
         Column := TokenPosition.Col;
      end if;

      SPARK_IO.Name (N, NameCon, NameLen);
      Str := EStrings.T'(NameLen, NameCon);

      if CommandLineData.Content.PlainOutput or else
        CommandLineData.Content.Brief
      then
         Str := EStrings.LowerCase (FileSystem.JustFile (Str, True));
      end if;

      if CommandLineData.Content.Brief then
         ScreenEcho.Put_ExaminerString (Str);
         ScreenEcho.Put_Char (':');
         ScreenEcho.Put_Integer (TokenPosition.Line, 0, 10);
         ScreenEcho.Put_Char (':');
         ScreenEcho.Put_Integer (Column, 0, 10);
         ScreenEcho.Put_String (": ");
         ScreenEcho.Put_String (S);
      else
         ScreenEcho.Put_String ("In index file ");
         ScreenEcho.Put_ExaminerString (Str);
         ScreenEcho.Put_String (" at line ");
         ScreenEcho.Put_Integer (TokenPosition.Line, 0, 10);
         ScreenEcho.Put_String (" column ");
         ScreenEcho.Put_Integer (Column, 0, 10);
         ScreenEcho.Put_String (": ");
         ScreenEcho.Put_String (S);
      end if;

      if TokenString.Length > 0 then
         ScreenEcho.Put_String (": ");
         ScreenEcho.Put_ExaminerLine (TokenString);
      else
         ScreenEcho.New_Line (1);
      end if;

   end PutError;

   procedure OutputError (E : in LibraryManagerErrors;
                          N : in SPARK_IO.File_Type;
                          TokenPosition : in FilePosition;
                          TokenString : in EStrings.T)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                E,
   --#                                N,
   --#                                TokenPosition,
   --#                                TokenString;
   is
   begin
      case E is
         when ESFileLocation    => PutError ("Incorrect syntax in File location", N,
                                            TokenPosition, TokenString);
         when ESIsIn            => PutError ("Incorrect syntax, ""is in"" expected", N,
                                            TokenPosition, TokenString);
         when ESUnitEntry       => PutError ("Incorrect syntax in Entry type", N,
                                            TokenPosition, TokenString);
         when ESComment         => PutError ("Illegal comment, ignored", N,
                                            TokenPosition, TokenString);
         when EWUnexpectedSuper => PutError ("Unexpected superindex", N,
                                            TokenPosition, TokenString);
         when EWIllegalUnitName => PutError ("Illegal Unit name", N,
                                             TokenPosition, TokenString);
         when ESAre             => PutError ("Incorrect syntax ""are"" expected", N,
                                             TokenPosition, TokenString);
         when ESComponents      => PutError ("Illegal syntax in Component entry", N,
                                             TokenPosition, TokenString);
         when EWAux             => PutError ("Unit name in auxindex file is not a suffix of", N,
                                             TokenPosition, TokenString);

         when EWIndex | EWSuper |
              ESRecursion       => null; -- should not happen
      end case;
   end OutputError;

   procedure OutputError2 (E          : in LibraryManagerErrors;
                           SourceFile : in EStrings.T;
                           Position   : in FilePosition;
                           TargetFile : in EStrings.T)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                E,
   --#                                Position,
   --#                                SourceFile,
   --#                                TargetFile;
   is
      SourceStr : EStrings.T;
      TargetStr : EStrings.T;
      LineNo    : Positive;
      ColNo     : Positive;
   begin
      if CommandLineData.Content.PlainOutput or
        CommandLineData.Content.Brief
      then
         if SourceFile.Length > 0 then
            SourceStr := EStrings.LowerCase (FileSystem.JustFile
                                                    (SourceFile, True));
         else
            SourceStr := EStrings.EmptyString;
         end if;
         TargetStr    := EStrings.LowerCase (FileSystem.JustFile
                                                    (TargetFile, True));
      else
         SourceStr := SourceFile;
         TargetStr := TargetFile;
      end if;

      case E is
         when EWIndex | ESRecursion =>
            if SourceStr.Length > 0 then
               -- Compensate for look ahead
               if Position.Col <= 2 and Position.Line > 1 then
                  LineNo := Position.Line - 1;
                  ColNo := 1;
               else
                  if Position.Col > 1 then
                     ColNo := Position.Col - 1;
                  else
                     ColNo := Position.Col;
                  end if;
                  LineNo := Position.Line;
               end if;

               if CommandLineData.Content.Brief then
                  ScreenEcho.Put_ExaminerString (SourceStr);
                  ScreenEcho.Put_Char (':');
                  ScreenEcho.Put_Integer (LineNo, 0, 10);
                  ScreenEcho.Put_Char (':');
                  ScreenEcho.Put_Integer (ColNo, 0, 10);
               else
                  ScreenEcho.Put_String ("In index file ");
                  ScreenEcho.Put_ExaminerString (SourceStr);
                  ScreenEcho.Put_String (" at line ");
                  ScreenEcho.Put_Integer (LineNo, 0, 10);
                  ScreenEcho.Put_String (" column ");
                  ScreenEcho.Put_Integer (ColNo, 0, 10);
               end if;
               ScreenEcho.Put_String (": ");
            end if;

            if E = ESRecursion then
               ScreenEcho.Put_String ("Recursive use of index file ");
            else
               ScreenEcho.Put_String ("Cannot open index file ");
            end if;
            ScreenEcho.Put_ExaminerLine (TargetStr);

         when EWSuper               =>
            if CommandLineData.Content.Brief then
               -- Assume superindex is the first entry in the file
               ScreenEcho.Put_ExaminerString (SourceStr);
               ScreenEcho.Put_String (":1:1");
            else
               ScreenEcho.Put_String ("In index file ");
               ScreenEcho.Put_ExaminerString (SourceStr);
            end if;
            ScreenEcho.Put_String (": Cannot open superindex file ");
            ScreenEcho.Put_ExaminerLine (TargetStr);

         when ESFileLocation | ESIsIn | ESUnitEntry | ESAre |
            ESComponents   |
            ESComment      | EWUnexpectedSuper    | EWIllegalUnitName | EWAux
            => null; -- should not happen
      end case;
   end OutputError2;




   procedure UngetChar (Ch : in Character)
   --# global out LastChar;
   --# derives LastChar from Ch;
   is
   begin
      LastChar := Ch;
   end UngetChar;

   function IsEndOfBufferedLine (File : SPARK_IO.File_Type) return Boolean
   --# global in LastChar;
   --#        in SPARK_IO.File_Sys;
   is
      Answer : Boolean;
   begin
      Answer := False;
      if (SPARK_IO.End_Of_Line (File)) then
         if (LastChar = Ada.Characters.Latin_1.NUL) then
            Answer := True;
         end if;
      end if;
      return Answer;
   end IsEndOfBufferedLine;

   function IsEndOfBufferedFile (File : SPARK_IO.File_Type) return Boolean
   --# global in LastChar;
   --#        in SPARK_IO.File_Sys;
   is
      Answer : Boolean;
   begin
      Answer := False;
      if (SPARK_IO.End_Of_File (File)) then
         if (LastChar = Ada.Characters.Latin_1.NUL) then
            Answer := True;
         end if;
      end if;
      return Answer;
   end IsEndOfBufferedFile;

   procedure ReadBufferedChar (File : in     SPARK_IO.File_Type;
                               Ch   :    out Character)
   --# global in     CommandLineData.Content;
   --#        in out LastChar;
   --#        in out SPARK_IO.File_Sys;
   --# derives Ch,
   --#         LastChar          from File,
   --#                                LastChar,
   --#                                SPARK_IO.File_Sys &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                File,
   --#                                LastChar;
   is
      ReadChar : Character;
      TokenPosition : FilePosition;
   begin
      if LastChar = Ada.Characters.Latin_1.NUL then
         -- No buffered character
         SPARK_IO.Get_Char (File, ReadChar);
         if (ReadChar = Ada.Characters.Latin_1.HT) or (ReadChar = Ada.Characters.Latin_1.CR) then
            ReadChar := ' ';
         end if;
      else
         -- Read the buffered character
         ReadChar := LastChar;
         LastChar := Ada.Characters.Latin_1.NUL;
      end if;

      -- Check for comments; skip line if found
      if ReadChar = '-' then
         if not IsEndOfBufferedFile (File) then
            SPARK_IO.Get_Char (File, ReadChar);
            if ReadChar = '-' then
               --it is a comment so skip to end of line
               SPARK_IO.Skip_Line (File, 1);
               ReadChar := ' ';
            else
               --only one '-' so its not a comment
               UngetChar (ReadChar);
               ReadChar := '-';
            end if;
         else
            -- A single dash then EOF - nonsense
            --error reading; assume no file can end with '-'

            -- In priciple calls to Line and Col may cause a run-time exception
            -- if Line or Col > Count'Last defined in the TEXT_IO package
            -- but in practice this is exception will not be raised because, for
            -- instance, in gnat Count'Last = Natural'Last
            TokenPosition := FilePosition'(Line => SPARK_IO.Line (File),
                                           Col => SPARK_IO.Col (File));
            OutputError (ESComment, File,
                         TokenPosition, EStrings.EmptyString);
         end if;
      end if;

      Ch := ReadChar;
   end ReadBufferedChar;

   -- Get the next single character; unget any whitespace
   procedure GetSingleChar (File           : in     SPARK_IO.File_Type;
                            InQuotedString : in Boolean;
                            Ch             :    out Character;
                            OK             :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in out LastChar;
   --#        in out SPARK_IO.File_Sys;
   --# derives Ch,
   --#         OK                from File,
   --#                                LastChar,
   --#                                SPARK_IO.File_Sys &
   --#         LastChar          from *,
   --#                                File,
   --#                                InQuotedString,
   --#                                SPARK_IO.File_Sys &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                File,
   --#                                LastChar;
   is
      MyChar : Character;
   begin  -- GetSingleChar
      if IsEndOfBufferedFile (File) then
         Ch := ' ';
         OK := False;
      elsif IsEndOfBufferedLine (File) then
         Ch := ' ';
         OK := True;
      else
         ReadBufferedChar (File, MyChar);

         -- Allow for quoted strings containing spaces
         if (MyChar = ' ') and not InQuotedString then
            UngetChar (MyChar);
         end if;

         OK := True;
         Ch := MyChar;
      end if;
   end GetSingleChar;

   -- Get all the contiguous whitespace (including newlines)
   -- up to end of file or a non-whitespace char.  Unget
   -- non-whitespace.
   procedure SkipWhiteSpace (File : in     SPARK_IO.File_Type)
   --# global in     CommandLineData.Content;
   --#        in out LastChar;
   --#        in out SPARK_IO.File_Sys;
   --# derives LastChar,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                File,
   --#                                LastChar,
   --#                                SPARK_IO.File_Sys;
   is
      MyChar : Character;
   begin  -- SkipWhiteSpace
      while IsEndOfBufferedLine (File) and (not IsEndOfBufferedFile (File))
      loop
         SPARK_IO.Skip_Line (File, 1);
      end loop;
      if not IsEndOfBufferedFile (File) then
         ReadBufferedChar (File, MyChar);
         if (MyChar /= ' ') then
            UngetChar (MyChar);
         end if;
         while (MyChar = ' ') and (not IsEndOfBufferedFile (File))
         loop
            if IsEndOfBufferedLine (File) then
               SPARK_IO.Skip_Line (File, 1);
            else
               -- Read the next char, as long as it's whitespace
               ReadBufferedChar (File, MyChar);
               if (MyChar /= ' ') then
                  UngetChar (MyChar);
               end if;
            end if;
         end loop;
      end if;
   end SkipWhiteSpace;

   procedure GetTokenPosition (IndexFile : in     SPARK_IO.File_Type;
                              Position  :     out FilePosition)
   --# global in     CommandLineData.Content;
   --#        in out LastChar;
   --#        in out SPARK_IO.File_Sys;
   --# derives LastChar,
   --#         Position,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                IndexFile,
   --#                                LastChar,
   --#                                SPARK_IO.File_Sys;
   is
   begin
      SkipWhiteSpace (IndexFile);
      Position := FilePosition'(Line => SPARK_IO.Line (IndexFile),
                                Col  => SPARK_IO.Col (IndexFile));
   end GetTokenPosition;

   function TokenListToString (TokenList : LexTokenLists.Lists) return EStrings.T
   --# global in LexTokenManager.StringTable;
   is
      FullStr,
      EStr : EStrings.T;
      StringIsFull : Boolean;
   begin
      StringIsFull := False;
      if TokenList.Length > 0 then
         LexTokenManager.LexStringToString (TokenList.Content (1), FullStr);
         for I in LexTokenLists.Positions range 2 .. TokenList.Length loop
            LexTokenManager.LexStringToString (TokenList.Content (I), EStr);
            if (FullStr.Length + EStr.Length) + 1 in EStrings.Positions then
               EStrings.AppendString (FullStr, ".");
               EStrings.AppendExaminerString (FullStr, EStr);
            else
               if FullStr.Length + 3 in EStrings.Positions then
                  EStrings.AppendString (FullStr, "...");
               else
                  FullStr.Content (FullStr.Length - 2) := '.';
                  FullStr.Content (FullStr.Length - 1) := '.';
                  FullStr.Content (FullStr.Length) := '.';
               end if;
               StringIsFull := True;
            end if;
            exit when StringIsFull;
         end loop;
      else
         FullStr := EStrings.EmptyString;
      end if;
      return FullStr;
   end TokenListToString;

   procedure ReadEntry (IndexFile       : in     SPARK_IO.File_Type;
                        EntryUnitExport :    out LexTokenLists.Lists;
                        EntryType       :    out EntryTypes;
                        EntryFileName   :    out EStrings.T;
                        EntryComponents :    out ComponentLists;
                        ValidEntry      :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in out LastChar;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives EntryComponents,
   --#         EntryFileName,
   --#         EntryType,
   --#         EntryUnitExport,
   --#         LastChar,
   --#         LexTokenManager.StringTable,
   --#         SPARK_IO.File_Sys,
   --#         ValidEntry                  from CommandLineData.Content,
   --#                                          IndexFile,
   --#                                          LastChar,
   --#                                          LexTokenManager.StringTable,
   --#                                          SPARK_IO.File_Sys;
   is
      EntryNameValid,
      IsInValid,
      ComponentsValid,
      FileLocationValid : Boolean;
      EntryUnit : LexTokenLists.Lists;
      EntryTypeLoc : EntryTypes;
      CurrentPosition : FilePosition;

      procedure ReadAnIdentifier (IndexFile  : in     SPARK_IO.File_Type;
                                  Identifier : in out EStrings.T;
                                  Valid      :    out Boolean;
                                  Follower   :    out Character)
      --# global in     CommandLineData.Content;
      --#        in out LastChar;
      --#        in out SPARK_IO.File_Sys;
      --# derives Follower,
      --#         LastChar,
      --#         SPARK_IO.File_Sys,
      --#         Valid             from CommandLineData.Content,
      --#                                IndexFile,
      --#                                LastChar,
      --#                                SPARK_IO.File_Sys &
      --#         Identifier        from *,
      --#                                CommandLineData.Content,
      --#                                IndexFile,
      --#                                LastChar,
      --#                                SPARK_IO.File_Sys;
      is
         Ch       : Character;
         ReadOK   : Boolean;
         AppendOK : Boolean;
      begin    --ReadAnIdentifier
         SkipWhiteSpace (IndexFile);
         GetSingleChar (IndexFile, False, Ch, ReadOK);
         if ReadOK and (Ch in 'a' .. 'z' or Ch in 'A' .. 'Z') then
            Identifier := EStrings.EmptyString;
            loop
               EStrings.AppendChar (Identifier, Ch, AppendOK);
               if not AppendOK then
                  SystemErrors.FatalError (SystemErrors.UnitNameInIndexTooLong, "in ReadAnIdentifier");
               end if;
               GetSingleChar (IndexFile, False, Ch, ReadOK);
               exit when not ReadOK or else
                  (Ch not in 'A' .. 'Z' and
                   Ch not in 'a' .. 'z' and
                   Ch not in '0' .. '9' and
                   Ch /= '_');
            end loop;
            Valid := IsEndOfBufferedFile (IndexFile) or ReadOK;
            Follower := Ch;
         else
            Valid := False;
            Follower := ' ';
         end if;
      end ReadAnIdentifier;

      procedure ReadAUnitName (IndexFile : in     SPARK_IO.File_Type;
                               UnitName  :    out LexTokenLists.Lists;
                               Valid     :    out Boolean)
      --# global in     CommandLineData.Content;
      --#        in out LastChar;
      --#        in out LexTokenManager.StringTable;
      --#        in out SPARK_IO.File_Sys;
      --# derives LastChar,
      --#         LexTokenManager.StringTable,
      --#         SPARK_IO.File_Sys           from *,
      --#                                          CommandLineData.Content,
      --#                                          IndexFile,
      --#                                          LastChar,
      --#                                          SPARK_IO.File_Sys &
      --#         UnitName                    from CommandLineData.Content,
      --#                                          IndexFile,
      --#                                          LastChar,
      --#                                          LexTokenManager.StringTable,
      --#                                          SPARK_IO.File_Sys &
      --#         Valid                       from CommandLineData.Content,
      --#                                          IndexFile,
      --#                                          LastChar,
      --#                                          SPARK_IO.File_Sys;
      is
         Id              : EStrings.T;
         ValidIdentifier : Boolean;
         NextCh          : Character;
         Token           : LexTokenManager.LexString;
         NameList        : LexTokenLists.Lists;

         procedure InsertLexString (EStr   : in     EStrings.T;
                                    LexStr :    out LexTokenManager.LexString)
         --# global in out LexTokenManager.StringTable;
         --# derives LexStr,
         --#         LexTokenManager.StringTable from EStr,
         --#                                          LexTokenManager.StringTable;
         is
            Str : EStrings.Line;
         begin
            Str := EStrings.Line'(others => ' ');
            for I in EStrings.Positions range 1 .. EStr.Length
            loop
               Str (I) := EStr.Content (I);
            end loop;
            --Str is defined because the above for loop must be entered.
            LexTokenManager.InsertLexString (Str, 1, EStr.Length, LexStr);
         end InsertLexString;

      begin -- ReadAUnitName
         Id := EStrings.EmptyString;
         NameList := LexTokenLists.NullList;
         loop
            ReadAnIdentifier (IndexFile, Id, ValidIdentifier, NextCh);
            exit when not ValidIdentifier;
            InsertLexString (Id, Token);
            LexTokenLists.Append (NameList, Token);
            exit when NextCh /= '.';
         end loop;
         Valid := ValidIdentifier and NextCh = ' ';
         UnitName := NameList;
      end ReadAUnitName;

      procedure ReadAnEntryName (IndexFile       : in     SPARK_IO.File_Type;
                                 EntryUnitExport :    out LexTokenLists.Lists;
                                 EntryType       :    out EntryTypes;
                                 EntryNameValid  :    out Boolean)
      --# global in     CommandLineData.Content;
      --#        in out LastChar;
      --#        in out LexTokenManager.StringTable;
      --#        in out SPARK_IO.File_Sys;
      --# derives EntryNameValid,
      --#         EntryType,
      --#         EntryUnitExport,
      --#         LastChar,
      --#         LexTokenManager.StringTable,
      --#         SPARK_IO.File_Sys           from CommandLineData.Content,
      --#                                          IndexFile,
      --#                                          LastChar,
      --#                                          LexTokenManager.StringTable,
      --#                                          SPARK_IO.File_Sys;
      is
         UnitValid           : Boolean;
         AString             : EStrings.T;
         StringValid         : Boolean;
         NextCh              : Character;
         EntryUnit           : LexTokenLists.Lists;
         Position            : FilePosition;
      begin  -- ReadAnEntryName
         EntryType := InvalidEntryType;
         AString := EStrings.EmptyString;
         -- Get position of start of entry
         GetTokenPosition (IndexFile, Position);
         ReadAUnitName (IndexFile, EntryUnit, UnitValid);
         if UnitValid then
            if EntryUnit.Length = 1 and
               EntryUnit.Content (1) = LexTokenManager.SuperIndexToken then
               EntryType := SuperIndex;
               EntryNameValid := True;
            else
               -- Get the position of the start of the unit type
               GetTokenPosition (IndexFile, Position);
               ReadAnIdentifier (IndexFile, AString, StringValid, NextCh);
               if StringValid and NextCh = ' ' then
                  if EStrings.Eq1String (AString, "auxindex") then
                     EntryType := AuxIndex;
                     EntryNameValid := True;
                  elsif EStrings.Eq1String (AString, "main_program") then
                     EntryType := mainp;
                     EntryNameValid := True;
                  elsif EStrings.Eq1String (AString, "specification") or else
                        EStrings.Eq1String (AString, "spec") then
                     EntryType := pspec;
                     EntryNameValid := True;
                  -- GenOption -- elsif EStrings.Eq1String (AString, "generic_declaration") then
                  -- GenOption --    EntryType := GenericDec;
                  -- GenOption --    EntryNameValid := True;
                  elsif EStrings.Eq1String (AString, "body") then
                     EntryType := pbodi;
                     EntryNameValid := True;
                  elsif EStrings.Eq1String (AString, "subunit") then
                     EntryType := subunit;
                     EntryNameValid := True;
                  elsif EStrings.Eq1String (AString, "components") then
                     EntryType := ComponentList;
                     EntryNameValid := True;
                  else
                     EntryNameValid := False;
                     OutputError (ESUnitEntry, IndexFile, Position, AString);
                  end if;
               else
                  EntryNameValid := False;
                  OutputError (ESUnitEntry, IndexFile, Position,
                               EStrings.EmptyString);
               end if;
            end if;
         else
            EntryNameValid := False;
            OutputError (EWIllegalUnitName, IndexFile, Position,
                         EStrings.EmptyString);
         end if;
         EntryUnitExport := EntryUnit;
      end ReadAnEntryName;

      procedure SkipIsIn (IndexFile : in     SPARK_IO.File_Type;
                          IsInValid :    out Boolean)
      --# global in     CommandLineData.Content;
      --#        in out LastChar;
      --#        in out SPARK_IO.File_Sys;
      --# derives IsInValid,
      --#         LastChar,
      --#         SPARK_IO.File_Sys from CommandLineData.Content,
      --#                                IndexFile,
      --#                                LastChar,
      --#                                SPARK_IO.File_Sys;
      is
         Id      : EStrings.T;
         IdValid : Boolean;
         IsValid : Boolean;
         InValid : Boolean;
         NextCh  : Character;
      begin
         Id := EStrings.EmptyString;
         ReadAnIdentifier (IndexFile, Id, IdValid, NextCh);
         if IdValid and NextCh = ' ' then
            IsValid := EStrings.Eq1String (Id, "is");
         else
            IsValid := False;
         end if;

         if IsValid then
            ReadAnIdentifier (IndexFile, Id, IdValid, NextCh);

            if IdValid and NextCh = ' ' then
               InValid := EStrings.Eq1String (Id, "in");
            else
               InValid := False;
            end if;

            if InValid then
               IsInValid := True;
            else
               IsInValid := False;
            end if;
         else
            IsInValid := False;
         end if;
      end SkipIsIn;

      procedure ReadAFileLocation (IndexFile         : in     SPARK_IO.File_Type;
                                   EntryFileName     :    out EStrings.T;
                                   FileLocationValid :    out Boolean)
      --# global in     CommandLineData.Content;
      --#        in out LastChar;
      --#        in out SPARK_IO.File_Sys;
      --# derives EntryFileName,
      --#         FileLocationValid,
      --#         LastChar,
      --#         SPARK_IO.File_Sys from CommandLineData.Content,
      --#                                IndexFile,
      --#                                LastChar,
      --#                                SPARK_IO.File_Sys;
      is
         Ch             : Character;
         ReadOK         : Boolean;
         AppendOK       : Boolean;
         EStr           : EStrings.T;
         InQuotedString : Boolean;
      begin
         InQuotedString := False;

         EStr := EStrings.EmptyString;

         SkipWhiteSpace (IndexFile);
         GetSingleChar (IndexFile, False, Ch, ReadOK);
         loop
            -- Allow for quoted strings containing spaces
            exit when not ReadOK or
               (Ch = ' ' and not InQuotedString);

            if Ch = Ada.Characters.Latin_1.Quotation then
               InQuotedString := not InQuotedString;
            else
               EStrings.AppendChar (EStr, Ch, AppendOK);
               if not AppendOK then
                  SystemErrors.FatalError (SystemErrors.FileNameInIndexTooLong, "in ReadAFIleLocation");
               end if;
            end if;

            GetSingleChar (IndexFile, InQuotedString, Ch, ReadOK);
         end loop;
         SkipWhiteSpace (IndexFile);
         EntryFileName := EStr;
         FileLocationValid := EStr.Length > 0;

      end ReadAFileLocation;

      procedure ReadComponents (IndexFile     : in     SPARK_IO.File_Type;
                                TheComponents :    out ComponentLists;
                                EntryValid    :    out Boolean)
      --# global in     CommandLineData.Content;
      --#        in out LastChar;
      --#        in out LexTokenManager.StringTable;
      --#        in out SPARK_IO.File_Sys;
      --# derives EntryValid                  from CommandLineData.Content,
      --#                                          IndexFile,
      --#                                          LastChar,
      --#                                          SPARK_IO.File_Sys &
      --#         LastChar,
      --#         LexTokenManager.StringTable,
      --#         SPARK_IO.File_Sys           from *,
      --#                                          CommandLineData.Content,
      --#                                          IndexFile,
      --#                                          LastChar,
      --#                                          SPARK_IO.File_Sys &
      --#         TheComponents               from CommandLineData.Content,
      --#                                          IndexFile,
      --#                                          LastChar,
      --#                                          LexTokenManager.StringTable,
      --#                                          SPARK_IO.File_Sys;
      is
         Id        : EStrings.T;
         Token     : LexTokenManager.LexString;
         Index     : ComponentIndex;
         Component : LexTokenLists.Lists;
         Valid     : Boolean;
         NextCh    : Character;
         CurrentPosition : FilePosition;

         procedure InsertLexString (EStr   : in     EStrings.T;
                                    LexStr :    out LexTokenManager.LexString)
         --# global in out LexTokenManager.StringTable;
         --# derives LexStr,
         --#         LexTokenManager.StringTable from EStr,
         --#                                          LexTokenManager.StringTable;
         is
            Str : EStrings.Line;
         begin
            Str := EStrings.Line'(others => ' ');
            for I in EStrings.Positions range 1 .. EStr.Length loop
               Str (I) := EStr.Content (I);
            end loop;
            --Str is defined because the above for loop must be entered.
            LexTokenManager.InsertLexString (Str, 1, EStr.Length, LexStr);
         end InsertLexString;

      begin
         TheComponents := ComponentLists'(others => LexTokenLists.NullList);
         Index := ComponentIndex'First;
         Id    := EStrings.EmptyString;

         -- Locate the position of the "are"
         GetTokenPosition (IndexFile, CurrentPosition);
         ReadAnIdentifier (IndexFile, Id, Valid, NextCh);

         Valid := Valid and then NextCh = ' ' and then
            EStrings.Eq1String (Id, "are");

         if Valid then
            loop --over components
               Id := EStrings.EmptyString;
               Component := LexTokenLists.NullList;
               loop --over component identifiers

                  -- Locate the position of the next identifier
                  GetTokenPosition (IndexFile, CurrentPosition);

                  ReadAnIdentifier (IndexFile, Id, Valid, NextCh);
                  exit when not Valid;
                  InsertLexString (Id, Token);
                  LexTokenLists.Append (Component, Token);
                  exit when NextCh /= '.';
               end loop;
               exit when not Valid;
               TheComponents (Index) := Component;
               if Index < ComponentIndex'Last then
                  Index := ComponentIndex'Succ (Index);
               else
                  SystemErrors.FatalError (SystemErrors.IndexComponentListFull, "");
               end if;
               exit when NextCh /= ',';
            end loop;
            Valid := Valid and NextCh = ' ';
            if not Valid then
               OutputError (ESComponents, IndexFile, CurrentPosition,
                            EStrings.EmptyString);
            end if;
         else
            OutputError (ESAre, IndexFile, CurrentPosition,
                         EStrings.EmptyString);
         end if;
         TheComponents (Index) := LexTokenLists.NullList; -- list terminator
         EntryValid := Valid;
      end ReadComponents;

   begin  -- ReadEntry
      EntryComponents := ComponentLists'(others => LexTokenLists.NullList);

      ReadAnEntryName (IndexFile, EntryUnit, EntryTypeLoc, EntryNameValid);
      EntryType := EntryTypeLoc;

      if EntryNameValid then
         if EntryTypeLoc = ComponentList then
            EntryFileName := EStrings.EmptyString;
            ReadComponents (IndexFile, EntryComponents, ComponentsValid);
            if ComponentsValid then
               ValidEntry := True;
            else
               ValidEntry := False;
               -- Errors are reported by ReadComponents
               SPARK_IO.Skip_Line (IndexFile, 1);
            end if;
         else
            -- Locate the start of "is in"
            GetTokenPosition (IndexFile, CurrentPosition);

            SkipIsIn (IndexFile, IsInValid);
            if IsInValid then
               -- Locate the start of file location
               GetTokenPosition (IndexFile, CurrentPosition);

               ReadAFileLocation (IndexFile, EntryFileName, FileLocationValid);
               if FileLocationValid then
                  ValidEntry := True;
               else
                  ValidEntry := False;
                  EntryFileName := EStrings.EmptyString;
                  OutputError (ESFileLocation, IndexFile,
                               CurrentPosition, EStrings.EmptyString);
               end if;
            else
               ValidEntry := False;
               EntryFileName := EStrings.EmptyString;
               OutputError (ESIsIn, IndexFile,
                            CurrentPosition, EStrings.EmptyString);
               SPARK_IO.Skip_Line (IndexFile, 1);
            end if;
         end if;
      else
         ValidEntry := False;
         EntryFileName := EStrings.EmptyString;
         -- Error reported in call to ReadAnEntry
         SPARK_IO.Skip_Line (IndexFile, 1);
      end if;
      EntryUnitExport := EntryUnit;
   end ReadEntry;

   procedure LookUp (RequiredUnit      : in     LexTokenLists.Lists;
                     PossibleUnitTypes : in     ContextManager.UnitTypeSets;
                     SourceFileName    :    out EStrings.T;
                     ActualUnitType    :    out ContextManager.UnitTypes;
                     Found             :    out Boolean)
   is
      type SearchStatusType is (FoundSource, FoundAux, UseSuper);

      LFound               : Boolean;
      InAuxIndex           : Boolean;
      Done                 : Boolean;
      CurrentIndexFileName : EStrings.T;
      FileSpecStatus       : FileSystem.TypFileSpecStatus;
      ReturnedFileName     : EStrings.T;
      SuperIndexFound      : Boolean;
      SuperIndexFileName   : EStrings.T;
      SearchStatus         : SearchStatusType;
      IndexFile            : SPARK_IO.File_Type;
      FileStatus           : SPARK_IO.File_Status;

      AuxIndexUnit         : LexTokenLists.Lists;

      LocalIndexTable      : IndexTables;
      IndexRevisited       : Boolean;
      LocalSourceFileName  : EStrings.T;

      LastFileName         : EStrings.T;
      Position             : FilePosition;
      PossErrorType        : LibraryManagerErrors;

      procedure Trace (S : in EStrings.T;
                       M : in String)
      --# global in CommandLineData.Content;
      --# derives null from CommandLineData.Content,
      --#                   M,
      --#                   S;
      is
         --# hide Trace;
      begin
         if CommandLineData.Content.Debug.FileNames then
            SPARK_IO.Put_String (SPARK_IO.Standard_Output, M, 0);

            if CommandLineData.Content.PlainOutput then
               EStrings.PutLine (SPARK_IO.Standard_Output,
                                        EStrings.LowerCase (FileSystem.JustFile (S, True)));
            else
               EStrings.PutLine (SPARK_IO.Standard_Output,
                                        S);
            end if;
         end if;
      end Trace;

      procedure AddLocalEntry (FileName       : in     EStrings.T;
                               AlreadyPresent :    out Boolean)
      --# global in out LocalIndexTable;
      --# derives AlreadyPresent,
      --#         LocalIndexTable from FileName,
      --#                              LocalIndexTable;
      is
         Found : Boolean;
      begin
         Found := False;

         for I in IndexSizes range 1 .. LocalIndexTable.Size loop
            Found := EStrings.EqString (LocalIndexTable.Content (I), FileName);
            exit when Found;
         end loop;

         if not Found then
            if LocalIndexTable.Size < ExaminerConstants.MaxIndexNumber then
               LocalIndexTable.Size := LocalIndexTable.Size + 1;
               LocalIndexTable.Content (LocalIndexTable.Size) := FileName;
            else
               SystemErrors.FatalError (SystemErrors.IndexStackFull, "");
            end if;
         end if;
         AlreadyPresent := Found;
      end AddLocalEntry;

      procedure LookInFile (IndexFile         : in     SPARK_IO.File_Type;
                            RequiredUnit      : in     LexTokenLists.Lists;
                            PossibleUnitTypes : in     ContextManager.UnitTypeSets;
                            ReturnedFileName  :    out EStrings.T;
                            ActualUnitType    :    out ContextManager.UnitTypes;
                            Status            :    out SearchStatusType)
      --# global in     CommandLineData.Content;
      --#        in     InAuxIndex;
      --#        in out AuxIndexUnit;
      --#        in out LexTokenManager.StringTable;
      --#        in out SPARK_IO.File_Sys;
      --#           out LastChar;
      --#           out SuperIndexFileName;
      --#           out SuperIndexFound;
      --# derives ActualUnitType,
      --#         AuxIndexUnit,
      --#         LastChar,
      --#         LexTokenManager.StringTable,
      --#         ReturnedFileName,
      --#         SPARK_IO.File_Sys,
      --#         Status,
      --#         SuperIndexFileName,
      --#         SuperIndexFound             from AuxIndexUnit,
      --#                                          CommandLineData.Content,
      --#                                          InAuxIndex,
      --#                                          IndexFile,
      --#                                          LexTokenManager.StringTable,
      --#                                          PossibleUnitTypes,
      --#                                          RequiredUnit,
      --#                                          SPARK_IO.File_Sys;
      is
         EntryUnit     : LexTokenLists.Lists;
         EntryType     : EntryTypes;
         EntryFileName : EStrings.T;
         EntryComponents : ComponentLists;
         ValidEntry    : Boolean;
         FirstInFile   : Boolean;
         LStatus       : SearchStatusType;
         Position      : FilePosition;
         AuxStr        : EStrings.T;

         procedure HandleSuperIndex (Position : in FilePosition)
         --# global in     CommandLineData.Content;
         --#        in     EntryFileName;
         --#        in     FirstInFile;
         --#        in     InAuxIndex;
         --#        in     IndexFile;
         --#        in out SPARK_IO.File_Sys;
         --#        in out SuperIndexFileName;
         --#        in out SuperIndexFound;
         --# derives SPARK_IO.File_Sys  from *,
         --#                                 CommandLineData.Content,
         --#                                 FirstInFile,
         --#                                 InAuxIndex,
         --#                                 IndexFile,
         --#                                 Position,
         --#                                 SuperIndexFound &
         --#         SuperIndexFileName from *,
         --#                                 EntryFileName,
         --#                                 FirstInFile,
         --#                                 InAuxIndex,
         --#                                 SuperIndexFound &
         --#         SuperIndexFound    from *,
         --#                                 FirstInFile,
         --#                                 InAuxIndex;
         is
         begin
            if InAuxIndex or SuperIndexFound or
               not FirstInFile then
                  OutputError (EWUnexpectedSuper, IndexFile,
                              Position, EStrings.EmptyString);
            else
               SuperIndexFound    := True;
               SuperIndexFileName := EntryFileName;
            end if;
         end HandleSuperIndex;

         procedure HandleAuxIndex
         --# global in     EntryFileName;
         --#        in     EntryUnit;
         --#        in     RequiredUnit;
         --#        in out AuxIndexUnit;
         --#        in out LStatus;
         --#           out ReturnedFileName;
         --# derives AuxIndexUnit,
         --#         LStatus          from *,
         --#                               EntryUnit,
         --#                               RequiredUnit &
         --#         ReturnedFileName from EntryFileName,
         --#                               EntryUnit,
         --#                               RequiredUnit;
         is
         begin
            if LexTokenLists.PrefixUnit (EntryUnit, RequiredUnit) then
               LStatus := FoundAux;
               AuxIndexUnit := EntryUnit;
               ReturnedFileName := EntryFileName;
            else
               ReturnedFileName := EStrings.EmptyString;
            end if;
         end HandleAuxIndex;

         procedure HandleMainP
         --# global in     EntryFileName;
         --#        in     EntryUnit;
         --#        in     PossibleUnitTypes;
         --#        in     RequiredUnit;
         --#        in out LStatus;
         --#           out ActualUnitType;
         --#           out ReturnedFileName;
         --# derives ActualUnitType   from EntryUnit,
         --#                               PossibleUnitTypes,
         --#                               RequiredUnit &
         --#         LStatus          from *,
         --#                               EntryUnit,
         --#                               PossibleUnitTypes,
         --#                               RequiredUnit &
         --#         ReturnedFileName from EntryFileName,
         --#                               EntryUnit,
         --#                               PossibleUnitTypes,
         --#                               RequiredUnit;
         is
         begin
            if PossibleUnitTypes (ContextManager.MainProgram) and then
               LexTokenLists.EqUnit (EntryUnit, RequiredUnit)
            then
               LStatus := FoundSource;
               ReturnedFileName := EntryFileName;
               ActualUnitType := ContextManager.MainProgram;
            else
               ReturnedFileName := EStrings.EmptyString;
               ActualUnitType := ContextManager.InvalidUnit;
            end if;
         end HandleMainP;

         procedure HandlePSpec
         --# global in     EntryFileName;
         --#        in     EntryUnit;
         --#        in     PossibleUnitTypes;
         --#        in     RequiredUnit;
         --#        in out LStatus;
         --#           out ActualUnitType;
         --#           out ReturnedFileName;
         --# derives ActualUnitType   from EntryUnit,
         --#                               PossibleUnitTypes,
         --#                               RequiredUnit &
         --#         LStatus          from *,
         --#                               EntryUnit,
         --#                               PossibleUnitTypes,
         --#                               RequiredUnit &
         --#         ReturnedFileName from EntryFileName,
         --#                               EntryUnit,
         --#                               PossibleUnitTypes,
         --#                               RequiredUnit;
         is
         begin
            if PossibleUnitTypes (ContextManager.PackageSpecification) and then
               LexTokenLists.EqUnit (EntryUnit, RequiredUnit)
            then
               LStatus := FoundSource;
               ReturnedFileName := EntryFileName;
               ActualUnitType := ContextManager.PackageSpecification;
            elsif PossibleUnitTypes (ContextManager.GenericDeclaration) and then -- GenOption --
              LexTokenLists.EqUnit (EntryUnit, RequiredUnit)                     -- GenOption --
            then                                                                 -- GenOption --
               LStatus := FoundSource;                                           -- GenOption --
               ReturnedFileName := EntryFileName;                                -- GenOption --
               ActualUnitType := ContextManager.GenericDeclaration;              -- GenOption --
            else
               ReturnedFileName := EStrings.EmptyString;
               ActualUnitType := ContextManager.InvalidUnit;
            end if;
         end HandlePSpec;

         -- GenOption -- procedure HandleGenericDec
         -- GenOption -- --# global in     EntryFileName;
         -- GenOption -- --#        in     RequiredUnit;
         -- GenOption -- --#        in     PossibleUnitTypes;
         -- GenOption -- --#        in     EntryUnit;
         -- GenOption -- --#        in out LStatus;
         -- GenOption -- --#           out ReturnedFileName;
         -- GenOption -- --#           out ActualUnitType;
         -- GenOption -- --# derives ReturnedFileName from EntryFileName,
         -- GenOption -- --#                               RequiredUnit,
         -- GenOption -- --#                               PossibleUnitTypes,
         -- GenOption -- --#                               EntryUnit &
         -- GenOption -- --#         ActualUnitType   from RequiredUnit,
         -- GenOption -- --#                               PossibleUnitTypes,
         -- GenOption -- --#                               EntryUnit &
         -- GenOption -- --#         LStatus          from *,
         -- GenOption -- --#                               RequiredUnit,
         -- GenOption -- --#                               PossibleUnitTypes,
         -- GenOption -- --#                               EntryUnit;
         -- GenOption -- is
         -- GenOption -- begin
         -- GenOption --    if PossibleUnitTypes (ContextManager.GenericDeclaration) and then
         -- GenOption --      LexTokenLists.EqUnit (EntryUnit, RequiredUnit)
         -- GenOption --    then
         -- GenOption --       LStatus := FoundSource;
         -- GenOption --       ReturnedFileName := EntryFileName;
         -- GenOption --       ActualUnitType := ContextManager.GenericDeclaration;
         -- GenOption --    else
         -- GenOption --       ReturnedFileName := EStrings.EmptyString;
         -- GenOption --       ActualUnitType := ContextManager.InvalidUnit;
         -- GenOption --    end if;
         -- GenOption -- end HandleGenericDec;

         procedure HandlePBodi
         --# global in     EntryFileName;
         --#        in     EntryUnit;
         --#        in     PossibleUnitTypes;
         --#        in     RequiredUnit;
         --#        in out LStatus;
         --#           out ActualUnitType;
         --#           out ReturnedFileName;
         --# derives ActualUnitType   from EntryUnit,
         --#                               PossibleUnitTypes,
         --#                               RequiredUnit &
         --#         LStatus          from *,
         --#                               EntryUnit,
         --#                               PossibleUnitTypes,
         --#                               RequiredUnit &
         --#         ReturnedFileName from EntryFileName,
         --#                               EntryUnit,
         --#                               PossibleUnitTypes,
         --#                               RequiredUnit;
         is
         begin
            if PossibleUnitTypes (ContextManager.PackageBody) and then
               LexTokenLists.EqUnit (EntryUnit, RequiredUnit)
            then
               LStatus := FoundSource;
               ReturnedFileName := EntryFileName;
               ActualUnitType := ContextManager.PackageBody;
            else
               ReturnedFileName := EStrings.EmptyString;
               ActualUnitType := ContextManager.InvalidUnit;
            end if;
         end HandlePBodi;

         procedure HandleSubUnit
         --# global in     EntryFileName;
         --#        in     EntryUnit;
         --#        in     PossibleUnitTypes;
         --#        in     RequiredUnit;
         --#        in out LStatus;
         --#           out ActualUnitType;
         --#           out ReturnedFileName;
         --# derives ActualUnitType   from EntryUnit,
         --#                               PossibleUnitTypes,
         --#                               RequiredUnit &
         --#         LStatus          from *,
         --#                               EntryUnit,
         --#                               PossibleUnitTypes,
         --#                               RequiredUnit &
         --#         ReturnedFileName from EntryFileName,
         --#                               EntryUnit,
         --#                               PossibleUnitTypes,
         --#                               RequiredUnit;
         is
         begin
            if PossibleUnitTypes (ContextManager.SubUnit) and then
               LexTokenLists.EqUnit (EntryUnit, RequiredUnit)
            then
               LStatus := FoundSource;
               ReturnedFileName := EntryFileName;
               ActualUnitType := ContextManager.SubUnit;
            else
               ReturnedFileName := EStrings.EmptyString;
               ActualUnitType := ContextManager.InvalidUnit;
            end if;
         end HandleSubUnit;

      begin --LookInFile
         SuperIndexFileName := EStrings.EmptyString;
         ReturnedFileName   := EStrings.EmptyString;
         ActualUnitType     := ContextManager.InvalidUnit;

         SuperIndexFound    := False;

         LStatus := UseSuper;
         FirstInFile := True;
         LastChar := Ada.Characters.Latin_1.NUL;
         loop
            exit when SPARK_IO.End_Of_File (IndexFile);
            -- Get Start of Entry
            GetTokenPosition (IndexFile, Position);
            --# accept Flow, 10, EntryComponents, "Expect EntryComponents Unused";
            ReadEntry (IndexFile,
                       EntryUnit,
                       EntryType,
                       EntryFileName,
                       EntryComponents,
                       ValidEntry);
            --# end accept;

            if ValidEntry then
               if InAuxIndex and then
                  not LexTokenLists.PrefixUnit (AuxIndexUnit, EntryUnit)
               then
                  AuxStr   := TokenListToString (AuxIndexUnit);
                  OutputError (EWAux, IndexFile,
                               Position, AuxStr);
               end if;

               case EntryType is
                  when SuperIndex => HandleSuperIndex (Position);
                  when AuxIndex   => HandleAuxIndex;
                  when mainp      => HandleMainP;
                  when pspec      => HandlePSpec;
                  -- GenOption -- when GenericDec => HandleGenericDec;
                  when pbodi      => HandlePBodi;
                  when subunit    => HandleSubUnit;
                  when ComponentList => null;
                  when InvalidEntryType => null;
               end case;
            end if;
            exit when LStatus = FoundSource or LStatus = FoundAux;
            FirstInFile := False;
         end loop;
         Status := LStatus;
         --# accept Flow, 33, EntryComponents, "Expect EntryComponents Unused";
      end LookInFile;

   begin --LookUp
      LocalIndexTable.Size := 0;
      LocalSourceFileName := EStrings.EmptyString;
      ActualUnitType := ContextManager.InvalidUnit;
      LFound := False;
      LastChar := Ada.Characters.Latin_1.NUL;
      LastFileName := EStrings.EmptyString;
      Position := FilePosition'(Line => 1, Col => 1);
      PossErrorType := EWIndex;
      if CommandLineData.Content.Index then
         AuxIndexUnit := LexTokenLists.NullList;
         InAuxIndex := False;
         Done := False;
         --# accept Flow, 10, FileSpecStatus, "Expect FileSpecStatus Unused";
         FileSystem.FindFullFileName
            (CommandLineData.Content.IndexFileName,
             FileSpecStatus,
             CurrentIndexFileName);
         --# end accept;
         loop
            --# accept Flow, 23, LocalIndexTable.Content, "Expect array df error";
            AddLocalEntry (CurrentIndexFileName, IndexRevisited);
            --# end accept;
            if IndexRevisited then -- recursion in index entries
               OutputError2 (ESRecursion, LastFileName, Position,
                             CurrentIndexFileName);
               LFound := False;
               exit;
            end if;

            IndexFile := SPARK_IO.Null_File;
            SPARK_IO.Open (IndexFile, SPARK_IO.In_File,
                           CurrentIndexFileName.Length, CurrentIndexFileName.Content,
                           "", FileStatus);
            AddEntry (CurrentIndexFileName);

            if FileStatus = SPARK_IO.Ok then
               LookInFile (IndexFile, RequiredUnit, PossibleUnitTypes,
                           ReturnedFileName, ActualUnitType, SearchStatus);

               LastFileName := CurrentIndexFileName;
               GetTokenPosition (IndexFile, Position);

               --# accept Flow, 10, IndexFile, "Expect ineffective assignment" &
               --#        Flow, 10, FileStatus, "Expect ineffective assignment";
               SPARK_IO.Close (IndexFile, FileStatus);
               --# end accept;
               case SearchStatus is
                  when FoundAux    => InAuxIndex := True;
                     FileSystem.CheckExtension (ReturnedFileName,
                                                CommandLineData.DEFAULT_INDEX_EXTENSION);
                     CurrentIndexFileName :=
                       FileSystem.InterpretRelative
                       (ReturnedFileName,
                        CurrentIndexFileName);
                     PossErrorType := EWIndex;
                  when UseSuper    => if SuperIndexFound then
                     InAuxIndex := False;
                     FileSystem.CheckExtension
                       (SuperIndexFileName,
                        CommandLineData.DEFAULT_INDEX_EXTENSION);
                     CurrentIndexFileName :=
                       FileSystem.InterpretRelative
                       (SuperIndexFileName,
                        CurrentIndexFileName);
                     PossErrorType := EWSuper;
                  else
                     LFound := False;
                     Done := True;
                  end if;
                  when FoundSource => LFound := True;
                     Done := True;
               end case;
            else
               LFound := False;
               OutputError2 (PossErrorType, LastFileName, Position,
                             CurrentIndexFileName);
               Done  := True;
            end if;
            exit when Done;
         end loop;
         if LFound then
            --# accept Flow, 504, ReturnedFileName, "ReturnedFileName always defined here";
            Trace (ReturnedFileName,
                   "IndexManager.Lookup ReturnedFileName is:");
            FileSystem.CheckExtension
              (ReturnedFileName,
               CommandLineData.Content.SourceExtension);
            --# end accept;
            LocalSourceFileName := FileSystem.InterpretRelative
              (ReturnedFileName,
               CurrentIndexFileName);
            Trace (LocalSourceFileName, "IndexManager.Lookup SourceFileName is:");
         end if;
      end if;
      Found := LFound;
      SourceFileName := LocalSourceFileName;
      --# accept Flow, 602, SPARK_IO.File_Sys, LocalIndexTable.Content, "Expect array df error" &
      --#        Flow, 602, SourceFileName, LocalIndexTable.Content, "Expect array df error" &
      --#        Flow, 602, LexTokenManager.StringTable, LocalIndexTable.Content, "Expect array df error" &
      --#        Flow, 602, IndexTable, LocalIndexTable.Content, "Expect array df error" &
      --#        Flow, 602, LastChar, LocalIndexTable.Content, "Expect array df error" &
      --#        Flow, 602, ActualUnitType, LocalIndexTable.Content, "Expect array df error" &
      --#        Flow, 602, Found, LocalIndexTable.Content, "Expect array df error" &
      --#        Flow, 602, SourceFileName, ReturnedFileName, "Expect df error for ReturnedFileName" &
      --#        Flow, 33, FileSpecStatus, "Expect FileSpecStatus unused";
   end LookUp;

   procedure LookUpComponents (ParentUnit : in     LexTokenLists.Lists;
                               Components :    out ComponentLists)
   is
      type SearchStatusType is (FoundSource, FoundAux, UseSuper);

      InAuxIndex           : Boolean;
      Done                 : Boolean;
      CurrentIndexFileName : EStrings.T;
      FileSpecStatus       : FileSystem.TypFileSpecStatus;
      ReturnedFileName     : EStrings.T;
      SuperIndexFound      : Boolean;
      SuperIndexFileName   : EStrings.T;
      SearchStatus         : SearchStatusType;
      IndexFile            : SPARK_IO.File_Type;
      FileStatus           : SPARK_IO.File_Status;

      AuxIndexUnit         : LexTokenLists.Lists;

      LastFileName         : EStrings.T;
      Position             : FilePosition;
      PossErrorType        : LibraryManagerErrors;

      procedure LookInFile (IndexFile        : in     SPARK_IO.File_Type;
                            ParentUnit       : in     LexTokenLists.Lists;
                            Components       :    out ComponentLists;
                            ReturnedFileName :    out EStrings.T;
                            Status           :    out SearchStatusType)
      --# global in     CommandLineData.Content;
      --#        in     InAuxIndex;
      --#        in out AuxIndexUnit;
      --#        in out LexTokenManager.StringTable;
      --#        in out SPARK_IO.File_Sys;
      --#           out LastChar;
      --#           out SuperIndexFileName;
      --#           out SuperIndexFound;
      --# derives AuxIndexUnit,
      --#         Components,
      --#         LastChar,
      --#         LexTokenManager.StringTable,
      --#         ReturnedFileName,
      --#         SPARK_IO.File_Sys,
      --#         Status,
      --#         SuperIndexFileName,
      --#         SuperIndexFound             from AuxIndexUnit,
      --#                                          CommandLineData.Content,
      --#                                          InAuxIndex,
      --#                                          IndexFile,
      --#                                          LexTokenManager.StringTable,
      --#                                          ParentUnit,
      --#                                          SPARK_IO.File_Sys;
      is
         EntryUnit     : LexTokenLists.Lists;
         EntryType     : EntryTypes;
         EntryFileName : EStrings.T;
         EntryComponents : ComponentLists;
         ValidEntry    : Boolean;
         FirstInFile   : Boolean;
         LStatus       : SearchStatusType;
         Position      : FilePosition;
         AuxStr        : EStrings.T;

         procedure HandleSuperIndex (Position : FilePosition)
         --# global in     CommandLineData.Content;
         --#        in     EntryFileName;
         --#        in     FirstInFile;
         --#        in     InAuxIndex;
         --#        in     IndexFile;
         --#        in out SPARK_IO.File_Sys;
         --#        in out SuperIndexFileName;
         --#        in out SuperIndexFound;
         --# derives SPARK_IO.File_Sys  from *,
         --#                                 CommandLineData.Content,
         --#                                 FirstInFile,
         --#                                 InAuxIndex,
         --#                                 IndexFile,
         --#                                 Position,
         --#                                 SuperIndexFound &
         --#         SuperIndexFileName from *,
         --#                                 EntryFileName,
         --#                                 FirstInFile,
         --#                                 InAuxIndex,
         --#                                 SuperIndexFound &
         --#         SuperIndexFound    from *,
         --#                                 FirstInFile,
         --#                                 InAuxIndex;
         is
         begin
            if InAuxIndex or SuperIndexFound or
               not FirstInFile
            then
               OutputError (EWUnexpectedSuper, IndexFile,
                           Position, EStrings.EmptyString);
            else
               SuperIndexFound    := True;
               SuperIndexFileName := EntryFileName;
            end if;
         end HandleSuperIndex;

         procedure HandleAuxIndex
         --# global in     EntryFileName;
         --#        in     EntryUnit;
         --#        in     ParentUnit;
         --#        in out AuxIndexUnit;
         --#        in out LStatus;
         --#           out ReturnedFileName;
         --# derives AuxIndexUnit,
         --#         LStatus          from *,
         --#                               EntryUnit,
         --#                               ParentUnit &
         --#         ReturnedFileName from EntryFileName,
         --#                               EntryUnit,
         --#                               ParentUnit;
         is
         begin
            if LexTokenLists.PrefixUnit (EntryUnit, ParentUnit) then
               LStatus := FoundAux;
               AuxIndexUnit := EntryUnit;
               ReturnedFileName := EntryFileName;
            else
               ReturnedFileName := EStrings.EmptyString;
            end if;
         end HandleAuxIndex;

         procedure HandleComponents
         --# global in     EntryComponents;
         --#        in     EntryUnit;
         --#        in     ParentUnit;
         --#        in out Components;
         --#        in out LStatus;
         --# derives Components from *,
         --#                         EntryComponents,
         --#                         EntryUnit,
         --#                         ParentUnit &
         --#         LStatus    from *,
         --#                         EntryUnit,
         --#                         ParentUnit;
         is
         begin
            if LexTokenLists.EqUnit (EntryUnit, ParentUnit) then
               LStatus := FoundSource;
               Components := EntryComponents;
            end if;
         end HandleComponents;

      begin --LookInFile
         SuperIndexFileName := EStrings.EmptyString;

         SuperIndexFound := False;

         Components := ComponentLists'(others => LexTokenLists.NullList);

         ReturnedFileName   := EStrings.EmptyString;
         LStatus := UseSuper;
         FirstInFile := True;
         LastChar := Ada.Characters.Latin_1.NUL;
         loop
            exit when SPARK_IO.End_Of_File (IndexFile);
            Position := FilePosition'(Line => SPARK_IO.Line (IndexFile),
                                      Col  => SPARK_IO.Col (IndexFile));
            ReadEntry (IndexFile, EntryUnit, EntryType, EntryFileName,
                       EntryComponents, ValidEntry);
            if ValidEntry then
               if InAuxIndex and then
                  not LexTokenLists.PrefixUnit (AuxIndexUnit, EntryUnit) then
                  AuxStr := TokenListToString (AuxIndexUnit);
                  OutputError (EWIllegalUnitName, IndexFile,
                              Position, AuxStr);
               end if;

               case EntryType is
                  when SuperIndex => HandleSuperIndex (Position);
                  when AuxIndex   => HandleAuxIndex;
                  when mainp      => null;
                  when pspec      => null;
                  -- GenOption -- when GenericDec => null;
                  when pbodi      => null;
                  when subunit    => null;
                  when ComponentList => HandleComponents;
                  when InvalidEntryType => null;
               end case;
            end if;
            exit when LStatus = FoundSource or LStatus = FoundAux;
            FirstInFile := False;
         end loop;
         Status := LStatus;
      end LookInFile; -- 782 - Expect DF error on Components.

   begin --LookUpComponents
      Components := ComponentLists'(others => LexTokenLists.NullList);

      LastChar := Ada.Characters.Latin_1.NUL;
      LastFileName := EStrings.EmptyString;
      Position := FilePosition'(Line => 1, Col => 1);
      PossErrorType := EWIndex;
      if CommandLineData.Content.Index then
         AuxIndexUnit := LexTokenLists.NullList;
         InAuxIndex := False;
         Done := False;
         --# accept Flow, 10, FileSpecStatus, "Expect FileSpecStatus Unused";
         FileSystem.FindFullFileName
            (CommandLineData.Content.IndexFileName,
             FileSpecStatus,
             CurrentIndexFileName);
         --# end accept;
         loop
            IndexFile := SPARK_IO.Null_File;
            SPARK_IO.Open (IndexFile, SPARK_IO.In_File,
                           CurrentIndexFileName.Length, CurrentIndexFileName.Content,
                           "", FileStatus);
            AddEntry (CurrentIndexFileName);
            if FileStatus = SPARK_IO.Ok then
               LookInFile (IndexFile, ParentUnit,
                           Components, ReturnedFileName, SearchStatus);
               LastFileName := CurrentIndexFileName;
               Position := FilePosition'(Line => SPARK_IO.Line (IndexFile),
                                         Col  => SPARK_IO.Col (IndexFile));
               --# accept Flow, 10, IndexFile, "Expect ineffective assignment" &
               --#        Flow, 10, FileStatus, "Expect ineffective assignment";
               SPARK_IO.Close (IndexFile, FileStatus);
               --# end accept;
               case SearchStatus is
                  when FoundAux    => InAuxIndex := True;
                     FileSystem.CheckExtension
                        (ReturnedFileName,
                         CommandLineData.DEFAULT_INDEX_EXTENSION);
                     CurrentIndexFileName :=
                        FileSystem.InterpretRelative
                        (ReturnedFileName,
                         CurrentIndexFileName);
                     PossErrorType := EWIndex;
                  when UseSuper    => if SuperIndexFound then
                     InAuxIndex := False;
                     FileSystem.CheckExtension
                        (SuperIndexFileName,
                         CommandLineData.DEFAULT_INDEX_EXTENSION);
                     CurrentIndexFileName :=
                        FileSystem.InterpretRelative
                        (SuperIndexFileName,
                         CurrentIndexFileName);
                     PossErrorType := EWSuper;
                  else
                     Done := True;
                  end if;
                  when FoundSource => Done := True;
               end case;
            else
               OutputError2 (PossErrorType, LastFileName, Position,
                             CurrentIndexFileName);
               Done  := True;
            end if;
            exit when Done;
         end loop;
      end if;
      --# accept Flow, 33, FileSpecStatus, "Expect FileSpecStatus unused";
   end LookUpComponents;

   procedure ListIndexFile (ReportFile : in SPARK_IO.File_Type)
   is
   begin
      if IndexTable.Size = 0 then
         if not CommandLineData.Content.XML then
            SPARK_IO.Put_Line (ReportFile, "No Index files were used", 0);
         end if;
      else
         if CommandLineData.Content.XML then
            XMLReport.StartSection (XMLReport.SIndexes,
                                    ReportFile);
         else
            SPARK_IO.Put_Line (ReportFile, "Index Filename(s) used were: ", 0);
         end if;

         for I in IndexSizes range 1 .. IndexTable.Size loop
            --# accept Flow, 41, "Expect stable expression";
            if CommandLineData.Content.XML then
               XMLReport.Index (IndexTable.Content (I),
                                ReportFile);
            elsif CommandLineData.Content.PlainOutput then
            --# end accept;
               SPARK_IO.Put_String (ReportFile, "   ", 0);
               EStrings.PutString
                  (ReportFile,
                   EStrings.LowerCase
                   (FileSystem.JustFile (IndexTable.Content (I), True)));
               SPARK_IO.New_Line (ReportFile, 1);
            else
               SPARK_IO.Put_String (ReportFile, "   ", 0);
               EStrings.PutString
                  (ReportFile, IndexTable.Content (I));
               SPARK_IO.New_Line (ReportFile, 1);
            end if;

         end loop;
         if CommandLineData.Content.XML then
            XMLReport.EndSection (XMLReport.SIndexes,
                                  ReportFile);
         end if;
      end if;
   end ListIndexFile;

begin
   LastChar := Character'First;
   IndexTable.Size := 0;
   --# accept Flow, 32, IndexTable.Content, "Initialization is partial but effective" &
   --#        Flow, 31, IndexTable.Content, "Initialization is partial but effective" &
   --#        Flow, 602, IndexTable, IndexTable.Content, "Initialization is partial but effective";
end IndexManager;
