-- $Id: filesystem.ads 12952 2009-04-09 13:13:58Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with ExaminerConstants,
     EStrings,
     ELStrings,
     SPARK_IO;

use type EStrings.T;

--# inherit ELStrings,
--#         EStrings,
--#         ExaminerConstants,
--#         SPARK_IO;

package FileSystem
is

   -----------------------------------------------------------------
   -- Constants, types, and functions relating to reading
   -- command-line arguments
   -----------------------------------------------------------------

   subtype TypCmdLineRange is Positive
      range 1 .. ExaminerConstants.MaxCommandStringLength;

   subtype TypCmdLine is String (TypCmdLineRange);

   function ArgumentSeparator return Character;

   function IsAnArgumentTerminator (Ch : Character) return Boolean;

   -- The character that introduces a command-line switch.
   -- '-' on UNIX, '/' on NT or VMS for instance
   function SwitchCharacter return Character;

   function UseWindowsCommandLine return Boolean;
   function UseUnixCommandLine return Boolean;

   procedure ReadCmdLine (CmdLineFound : out Boolean;
                          CmdLine      : out TypCmdLine);
   --# global in out SPARK_IO.File_Sys;
   --# derives CmdLine,
   --#         CmdLinefound,
   --#         SPARK_IO.File_Sys from SPARK_IO.File_Sys;

   -----------------------------------------------------------------
   -- Constants, types, and functions relating to the casing
   -- of filenames
   -----------------------------------------------------------------

   function CaseOfVCGFiles (EStr : EStrings.T)
                           return EStrings.T;

   function CaseOfFileExtensions (EStr : EStrings.T)
                                return EStrings.T;

   -- Select default filename casing when Opening files
   function CaseOfFilesForOpen (EStr : EStrings.T)
                              return EStrings.T;

   -- Select default filename casing when Creating files,
   -- for both normal and "Long" EStrings.
   function CaseOfFilesForCreate (EStr : EStrings.T)
                              return EStrings.T;
   function CaseOfFilesForCreateL (EStr : ELStrings.T)
                              return ELStrings.T;

   -----------------------------------------------------------------
   -- Constants, types, and functions relating to Source files
   -----------------------------------------------------------------

   -- Opens a source file for reading, but using the Form required
   -- by the host compiler to enable sharing of the file.
   procedure OpenSourceFile
      (File   : in out SPARK_IO.File_Type;
       Name   : in     EStrings.T;
       Status :    out SPARK_IO.File_Status);
   --# global in out SPARK_IO.File_Sys;
   --# derives File,
   --#         SPARK_IO.File_Sys,
   --#         STATUS            from File,
   --#                                Name,
   --#                                SPARK_IO.File_Sys;

   -----------------------------------------------------------------
   -- Constants, types, and subprograms relating to filenames and
   -- extensions.
   -----------------------------------------------------------------

   type TypFileSpecStatus is (FileFound, FileMissing, FileInvalid);

   -- Returns FileName fully normalized relative to the second argument.
   -- If the second argument is an empty string, then the filename
   -- is interpretated relative to current working directory
   function InterpretRelative (FileName,
                               RelativeToDirectory : EStrings.T)
                              return EStrings.T;

   -- Find the fully normalized pathname for FileSpec relative
   -- to current working directory and returns it in FullFileName.
   -- FileStatus is set to FileFound if this pathname denotes
   -- an existing regular file, or FileMissing otherwise.
   procedure FindFullFileName
      (FileSpec     : in     EStrings.T;
       FileStatus   :    out TypFileSpecStatus;
       FullFileName :    out EStrings.T);
   --# global in SPARK_IO.File_Sys;
   --# derives FileStatus,
   --#         FullFileName from FileSpec,
   --#                           SPARK_IO.File_Sys;

   -- Returns the basename of Fn, with or without the extension. For example:
   -- JustFile ("a/b/c/f.dat", True) = "f.dat"
   -- JustFile ("a/b/c/g.exe", False) = "g"
   function JustFile (Fn  : EStrings.T;
                      Ext : Boolean)
                     return EStrings.T;

   --  Any directory prefix is removed. If Suffix is non-empty and is a
   --  suffix of Fn (1 .. Stop), it is removed.  This implements the
   --  same semantics as GNAT.Directory_Operations.Base_Name on systems
   --  where this library is available, but is legal SPARK.
   procedure BaseName (Fn     : in out String;
                       Suffix : in     String;
                       Stop   : in out Natural);
   --# derives Fn,
   --#         Stop from Fn,
   --#                   Stop,
   --#                   Suffix;

   procedure CheckExtension (Fn  : in out EStrings.T;
                             Ext : in     EStrings.T);
   --# derives Fn from *,
   --#                 Ext;

   --780--special version of CheckExtension just for listing files
   procedure CheckListingExtension (SourceName : in     EStrings.T;
                                    Fn         : in out EStrings.T;
                                    Ext        : in     EStrings.T);
   --# derives Fn from *,
   --#                 Ext,
   --#                 SourceName;

   function IsReadableFile (F : EStrings.T) return Boolean;
   --# global in SPARK_IO.File_Sys;

   function SameFile (F1, F2 : EStrings.T) return Boolean;

   function IsDirectory (F : EStrings.T) return Boolean;
   --# global in SPARK_IO.File_Sys;

   function StartOfDirectory   return EStrings.T;
   function BetweenDirectories return EStrings.T;
   function EndOfPath          return EStrings.T;
   function MaxFileNameLength  return Natural;

   -- if directory name D ends with an EndOfPath string, then
   -- no change, otherwise append EndOfPath
   procedure AppendEndOfPathIfNeeded (D : in out EStrings.T);
   --# derives D from *;

   procedure IdempotentCreateSubdirectory
      (Path : in     EStrings.T;
       Ok   :    out Boolean);
   --# global in out SPARK_IO.File_Sys;
   --# derives Ok,
   --#         SPARK_IO.File_Sys from Path,
   --#                                SPARK_IO.File_Sys;

   -- For Unix/NT does not return a trailing / or \
   function WorkingDirectory return EStrings.T;
   --# global in SPARK_IO.File_Sys;

   -- For Unix/NT does not return a trailing / or \
   function ExaminerLibDirectory return EStrings.T;
   --# global in SPARK_IO.File_Sys;

   function GetEnvironmentVariable (EnvVarName : String)
                                   return EStrings.T;
   --# global in SPARK_IO.File_Sys;

end FileSystem;
