-- $Id: file_utils.adb 11900 2008-12-15 08:43:09Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with Calendar, -- Expect warning here, since only ref'd in hidden code
  CommandLineData,
  EStrings,
  ScreenEcho,
  Version;

package body File_Utils is

   LenDateTime                 : constant Integer := 23;

   subtype TypDateTimeRange    is Integer range 1 .. LenDateTime;
   subtype TypDateTime         is String (TypDateTimeRange);

   procedure DateTime (DateString : out TypDateTime)
   --# derives DateString from ;
   is
      --# hide DateTime;
      --
      -- FUNCTION :
      --
      -- Returns a string indicating the current date and time as a 23 character
      -- string in the format 24-DEC-1988 12:00:00.00
      --
      -- OPERATION :
      --
      -- Uses the standard package Calendar.
      --

      MonthsInYear       : constant := 12;
      DaysInMonth        : constant := 31;
      HoursInDay         : constant := 24;
      MinutesInHour      : constant := 60;
      SecondsInMinute    : constant := 60;
      HundredthsInSecond : constant := 100;

      type YearsType      is range 1991 .. 2099;
      type MonthsType     is range 1 .. MonthsInYear;
      type DaysType       is range 1 .. DaysInMonth;
      type HoursType      is range 0 .. HoursInDay - 1;
      type MinutesType    is range 0 .. MinutesInHour - 1;
      type SecondsType    is range 0 .. SecondsInMinute - 1;
      type HundredthsType is range 0 .. HundredthsInSecond - 1;

      subtype Digit is Natural range 0 .. 9;

      Year       : YearsType;
      Month      : MonthsType;
      Day        : DaysType;
      Hours      : HoursType;
      Minutes    : MinutesType;
      Seconds    : SecondsType;
      Hundredths : HundredthsType;

      procedure DecodeTime (Time       : in     Calendar.Time;
                            Year       :    out YearsType;
                            Month      :    out MonthsType;
                            Day        :    out DaysType;
                            Hours      :    out HoursType;
                            Minutes    :    out MinutesType;
                            Seconds    :    out SecondsType;
                            Hundredths :    out HundredthsType)
      is
         SystemTime : Duration;

         procedure DecodeDuration (Time       : in     Duration;
                                   Hours      :    out HoursType;
                                   Minutes    :    out MinutesType;
                                   Seconds    :    out SecondsType;
                                   Hundredths :    out HundredthsType)
         is
            type TimeOfDay is range 0 .. HoursInDay * MinutesInHour * SecondsInMinute * HundredthsInSecond - 1;
            SystemTime : TimeOfDay;
         begin

            SystemTime := TimeOfDay (Time * Duration'(100.0));

            Hours      := HoursType      (SystemTime / (MinutesInHour * SecondsInMinute * HundredthsInSecond));
            Minutes    := MinutesType    (SystemTime / (SecondsInMinute * HundredthsInSecond) mod MinutesInHour);
            Seconds    := SecondsType    (SystemTime / HundredthsInSecond mod SecondsInMinute);
            Hundredths := HundredthsType (SystemTime mod HundredthsInSecond);

         end DecodeDuration;

      begin

         Calendar.Split (Date    => Time,
                         Year    => Calendar.Year_Number  (Year),
                         Month   => Calendar.Month_Number (Month),
                         Day     => Calendar.Day_Number   (Day),
                         Seconds => SystemTime);

         DecodeDuration (SystemTime, Hours, Minutes, Seconds, Hundredths);

      end DecodeTime;

      function DigitImage (Value : Digit) return Character
      is
         type DigitImages is array (Digit) of Character;

         Image : constant DigitImages :=
           DigitImages'('0', '1', '2', '3', '4', '5', '6', '7', '8', '9');
      begin
         return Image (Value);
      end DigitImage;

      function DateImage (Day   : DaysType;
                          Month : MonthsType;
                          Year  : YearsType) return String
      is

         function YearImage (Year : YearsType) return String
         is

            DigitsInYear : constant := 4;

            subtype DigitInYear is Positive range 1 .. DigitsInYear;
            subtype ResultType  is String (DigitInYear);

            Result : ResultType;

            function GetYearDigit (Year : YearsType;
                                   Pos  : DigitInYear) return Digit
            is
            begin
               return Digit (Year / 10 ** (DigitInYear'Last - Pos) mod 10);
            end GetYearDigit;

         begin

            for I in DigitInYear loop
               Result (I) := DigitImage (GetYearDigit (Year, I));
            end loop;

            return Result;

         end YearImage;

         function MonthImage (Month : MonthsType) return String
         is

            CharactersInMonth : constant := 3;

            subtype CharacterInMonth is Positive range 1 .. CharactersInMonth;
            subtype MonthNames       is String (CharacterInMonth);
            type    NamesOfMonths    is array (MonthsType) of MonthNames;

            Image : constant NamesOfMonths :=
              NamesOfMonths'("JAN", "FEB", "MAR", "APR",
                             "MAY", "JUN", "JUL", "AUG",
                             "SEP", "OCT", "NOV", "DEC");

         begin
            return Image (Month);
         end MonthImage;

         function DayImage (Day : DaysType) return String
         is

            DigitsInDay : constant := 2;

            subtype DigitInDay is Positive range 1 .. DigitsInDay;
            subtype ResultType is String (DigitInDay);

            Result : ResultType;

            function GetDayDigit (Day : DaysType; Pos : DigitInDay) return Digit
            is
            begin
               return Digit (Day / 10 ** (DigitInDay'Last - Pos) mod 10);
            end GetDayDigit;

         begin

            for I in DigitInDay loop
               Result (I) := DigitImage (GetDayDigit (Day, I));
            end loop;

            return Result;

         end DayImage;

      begin
         return DayImage   (Day)   & '-' &
           MonthImage (Month) & '-' &
           YearImage  (Year);
      end DateImage;

      function TimeImage (Hours      : HoursType;
                          Minutes    : MinutesType;
                          Seconds    : SecondsType;
                          Hundredths : HundredthsType) return String
      is

         function HourImage (Hour : HoursType) return String
         is

            DigitsInHour : constant := 2;

            subtype DigitInHour is Positive range 1 .. DigitsInHour;
            subtype ResultType  is String (DigitInHour);

            Result : ResultType;

            function GetHourDigit (Hour : HoursType; Pos : DigitInHour) return Digit
            is
            begin
               return Digit (Hour / 10 ** (DigitInHour'Last - Pos) mod 10);
            end GetHourDigit;

         begin

            for I in DigitInHour loop
               Result (I) := DigitImage (GetHourDigit (Hour, I));
            end loop;

            return Result;

         end HourImage;

         function MinuteImage (Minute : MinutesType) return String
         is

            DigitsInMinute : constant := 2;

            subtype DigitInMinute is Positive range 1 .. DigitsInMinute;
            subtype ResultType    is String (DigitInMinute);

            Result : ResultType;

            function GetMinuteDigit (Minute : MinutesType;
                                     Pos    : DigitInMinute) return Digit
            is
            begin
               return Digit (Minute / 10 ** (DigitInMinute'Last - Pos) mod 10);
            end GetMinuteDigit;

         begin

            for I in DigitInMinute loop
               Result (I) := DigitImage (GetMinuteDigit (Minute, I));
            end loop;

            return Result;

         end MinuteImage;

         function SecondImage (Second : SecondsType) return String
         is

            DigitsInSecond : constant := 2;

            subtype DigitInSecond is Positive range 1 .. DigitsInSecond;
            subtype ResultType    is String (DigitInSecond);

            Result : ResultType;

            function GetSecondDigit (Second : SecondsType;
                                     Pos    : DigitInSecond) return Digit
            is
            begin
               return Digit (Second / 10 ** (DigitInSecond'Last - Pos) mod 10);
            end GetSecondDigit;

         begin

            for I in DigitInSecond loop
               Result (I) := DigitImage (GetSecondDigit (Second, I));
            end loop;

            return Result;

         end SecondImage;

         function HundredthImage (Hundredth : HundredthsType) return String
         is

            DigitsInHundredth : constant := 2;

            subtype DigitInHundredth is Positive range 1 .. DigitsInHundredth;
            subtype ResultType       is String (DigitInHundredth);

            Result : ResultType;

            function GetHundredthDigit (Hundredth : HundredthsType;
                                        Pos       : DigitInHundredth) return Digit
            is
            begin
               return Digit (Hundredth / 10 ** (DigitInHundredth'Last - Pos) mod 10);
            end GetHundredthDigit;

         begin

            for I in DigitInHundredth loop
               Result (I) := DigitImage (GetHundredthDigit (Hundredth, I));
            end loop;

            return Result;

         end HundredthImage;

      begin
         return HourImage      (Hours)   & ':' &
           MinuteImage    (Minutes) & ':' &
           SecondImage    (Seconds) & '.' &
           HundredthImage (Hundredths);
      end TimeImage;

   begin

      DecodeTime (Calendar.Clock, Year, Month, Day, Hours, Minutes, Seconds, Hundredths);

      DateString := DateImage (Day, Month, Year) &
        ' ' &
        TimeImage (Hours, Minutes, Seconds, Hundredths);

   end DateTime;






   ------------------
   -- PrintAHeader --
   ------------------

   procedure PrintAHeader
     (File       : in SPARK_IO.File_Type;
      HeaderLine : in String;
      FileType   : in FileTypes)
   is

      subtype TypStarLineIndex is Positive range 1 .. 55;
      subtype TypStarLine is String (TypStarLineIndex);

      ExtendedDateString : EStrings.T :=
        EStrings.T'
        (Length => 7,
         Content => EStrings.Contents'
           ('D', 'A', 'T', 'E', ' ', ':', ' ', others => ' '));

      DateString : TypDateTime;

      FDLCommentOpen  : constant Character := '{';
      FDLCommentClose : constant Character := '}';

      StarLine : constant TypStarLine :=
        TypStarLine'("*******************************************************");

      procedure Set_Col (File : in SPARK_IO.File_Type;
                         Posn : in Positive)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                file,
      --#                                posn;
      is
      begin
         if File = SPARK_IO.Standard_Output then
            ScreenEcho.Set_Col (Posn);
         else
            SPARK_IO.Set_Col (File, Posn);
         end if;
      end Set_Col;

      procedure Put_Char (File : in SPARK_IO.File_Type;
                          Item : in Character)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                file,
      --#                                Item;
      is
      begin
         if File = SPARK_IO.Standard_Output then
            ScreenEcho.Put_Char (Item);
         else
            SPARK_IO.Put_Char (File, Item);
         end if;
      end Put_Char;

      procedure Put_String (File : in SPARK_IO.File_Type;
                            Item : in String;
                            Stop : in Natural)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                file,
      --#                                Item,
      --#                                Stop;
      is
      begin
         if File = SPARK_IO.Standard_Output then
            ScreenEcho.Put_String (Item);
         else
            SPARK_IO.Put_String (File, Item, Stop);
         end if;
      end Put_String;

      procedure New_Line (File    : in SPARK_IO.File_Type;
                          Spacing : in Positive)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                file,
      --#                                spacing;
      is
      begin
         if File = SPARK_IO.Standard_Output then
            ScreenEcho.New_Line (Spacing);
         else
            SPARK_IO.New_Line (File, Spacing);
         end if;
      end New_Line;

      procedure Center (File     : in SPARK_IO.File_Type;
                        Str      : in String;
                        StartPos : in Integer;
                        Stop     : in Integer)
      --# global in     FileType;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                file,
      --#                                FileType,
      --#                                StartPos,
      --#                                Stop,
      --#                                Str;
      is
         PageWidth : constant Natural := 80;
      begin
         Set_Col (File, (PageWidth - ((Stop - StartPos) + 1)) / 2);
         if FileType = DecFile then
            Put_Char (File, FDLCommentOpen);
            for I in Integer range StartPos .. Stop loop
               Put_Char (File, Str (I));
            end loop;
            Put_Char (File, FDLCommentClose);
            New_Line (File, 1);

         elsif FileType = RuleFile then
            Put_String (File, "/*", 0);
            for I in Integer range StartPos .. Stop loop
               Put_Char (File, Str (I));
            end loop;
            Put_String (File, "*/", 0);
            New_Line (File, 1);
         else
            for I in Integer range StartPos .. Stop loop
               Put_Char (File, Str (I));
            end loop;
            New_Line (File, 1);
         end if;
      end Center;

      procedure OutputVersionLine (File : SPARK_IO.File_Type)
      --# global in     CommandLineData.Content;
      --#        in     FileType;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                file,
      --#                                FileType;
      is
         VersionLine   : EStrings.T;
      begin
         EStrings.CopyString
           (VersionLine,
            "SPARK Examiner " & Version.Toolset_Distribution & " Edition");

         if not CommandLineData.Content.PlainOutput then
            -- append version number, build date, and SVN Repository number
            EStrings.AppendString (VersionLine, ", Version " & Version.Toolset_Version);
            EStrings.AppendString (VersionLine, ", Build Date " & Version.Toolset_Build_Date);
            EStrings.AppendString (VersionLine, ", Build " & Version.Toolset_Build_Stamp);
         end if;

         Center (File, VersionLine.Content, 1, VersionLine.Length);

      end OutputVersionLine;

   begin
      Center (File, StarLine, 1, StarLine'Length);
      if HeaderLine'Length /= 0 then
         Center (File, HeaderLine, 1, HeaderLine'Length);
      end if;

      OutputVersionLine (File);

      -- Display the copyright line if not in plain mode.
      -- Otherwise, display a blank line.
      if not CommandLineData.Content.PlainOutput then
         Center (File, Version.Toolset_Copyright, Version.Toolset_Copyright'First, Version.Toolset_Copyright'Last);
      else
         New_Line (File, 1);
      end if;

      Center (File, StarLine, 1, StarLine'Length);
      New_Line (File, 2);

      if not CommandLineData.Content.PlainOutput then
         DateTime (DateString);
         EStrings.AppendString (ExtendedDateString, DateString);
         Center (File, ExtendedDateString.Content, 1, ExtendedDateString.Length);
         New_Line (File, 1);
      end if;
   end PrintAHeader;

end File_Utils;
