-- $Id: estrings.ads 13056 2009-04-20 17:01:20Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--  EStrings
--
--  Purpose:
--   Provides an ADT for a fixed-length String type.
--
--  In addition to the basic constructors and operations, simple I/O
--  procedure are prodived to Put and Read such strings to/from a SPARK_IO file.
--
--  Clients:
--   Used throughout the Examiner and other tools.
--
--  Use:
--   Declare an EString.T
--
--   Initialization via the constant EmptyString or via
--   constructor CopyString
--
--  Extension:
--   None planned.  See ELStrings for a similar (but longer) String ADT.
--------------------------------------------------------------------------------

with SPARK_IO;

--# inherit Ada.Characters.Latin_1,
--#         CommonString,
--#         SPARK_IO;

package EStrings
is
   -----------------------------------------------------------------------------
   -- Types and subtypes.  Note these are visible, so a client _can_ directly
   -- access the internal structure of EStrings.T as a String subtype
   -----------------------------------------------------------------------------

   subtype Lengths is Integer range 0 .. 255;
   MaxStringLength : constant Lengths := Lengths'Last;

   subtype Positions is Lengths range 1 .. MaxStringLength;
   subtype Contents  is String (Positions);

   type T is record
      Length  : Lengths;
      Content : Contents;
   end record;

   MaxLineLength  : constant Lengths := MaxStringLength;
   type Line is array (Lengths) of Character;

   type OrderTypes is (FirstOneFirst, SecondOneFirst, NeitherFirst);

   -----------------------------------------------------------------------------
   -- Constants
   -----------------------------------------------------------------------------

   EmptyString : constant T :=
     T'(Length => 0, Content => Contents'(Positions => ' '));

   EmptyLine   : constant Line := Line'(others => ' ');


   -----------------------------------------------------------------------------
   -- Operations
   -----------------------------------------------------------------------------

   function EqString (EStr1, EStr2 : in T) return Boolean;

   function Eq1String (M : T;
                       S : String)
                     return Boolean;

   function IsEmpty (EStr : T) return Boolean;

   procedure CopyString (EStr :    out T;
                         Str  : in     String);
   --# derives EStr from Str;
   --# post EStr.Length = Str'Length;

   -- Append Str to EStr, raising a fatal error on overflow
   procedure AppendString (EStr : in out T;
                           Str  : in     String);
   --# derives EStr from *,
   --#                   Str;

   -- Append EStr2 to EStr1, raising a fatal error on overflow
   procedure AppendExaminerString (EStr1 : in out T;
                                   EStr2 : in     T);
   --# derives EStr1 from *,
   --#                    EStr2;

   -- Append Str to EStr, truncating on overflow
   procedure AppendStringTruncate (EStr : in out T;
                                   Str  : in     String);
   --# derives EStr from *,
   --#                   Str;

   -- Append EStr2 to EStr1, truncating on overflow
   procedure AppendExaminerStringTruncate (EStr1 : in out T;
                                           EStr2 : in     T);
   --# derives EStr1 from *,
   --#                    EStr2;

   function LowerCase (EStr : T) return T;

   function UpperCase (EStr : T) return T;

   -- Section added for use in summary tool
   -- returns the specified subsection of the string
   -- if the subsection lies outside the string, empty string returned
   function Section (EStr     : T;
                     StartPos : Positions;
                     Length   : Lengths) return T;

   procedure AppendChar (EStr    : in out T;
                         Ch      : in     Character;
                         Success :    out Boolean);
   --# derives EStr    from *,
   --#                      Ch &
   --#         Success from EStr;

   -- Trim added for use in summary tool
   -- removes ' ', Latin_1.HT, Latin_1.LF, Latin_1.CR from either end of the string
   function Trim (EStr : T) return T;

   -- FindSubString: for use in summary tool
   -- find specified SearchString in the given T
   procedure FindSubString (EStr         : in     T;
                            SearchString : in     String;
                            StringFound  :    out Boolean;
                            StringStart  :    out Positions);
   --# derives StringFound,
   --#         StringStart from EStr,
   --#                          SearchString;

   -- FindSubStringAfter: for use in summary tool
   -- find the specified SearchString, starting at the specified position in
   -- the given T
   procedure FindSubStringAfter (EStr         : in     T;
                                 SearchStart  : in     Positions;
                                 SearchString : in     String;
                                 StringFound  :    out Boolean;
                                 StringStart  :    out Positions);
   --# derives StringFound,
   --#         StringStart from EStr,
   --#                          SearchStart,
   --#                          SearchString;

   -- FindCharAfter: find specified character in EStr, starting at specified
   -- position
   procedure FindCharAfter (EStr        : in     T;
                            SearchStart : in     Positions;
                            SearchChar  : in     Character;
                            CharFound   :    out Boolean;
                            CharPos     :    out Positions);
   --# derives CharFound,
   --#         CharPos   from EStr,
   --#                        SearchChar,
   --#                        SearchStart;

   -- FindChar: find first occurrence of specified character in EStr
   procedure FindChar (EStr       : in     T;
                       SearchChar : in     Character;
                       CharFound  :    out Boolean;
                       CharPos    :    out Positions);
   --# derives CharFound,
   --#         CharPos   from EStr,
   --#                        SearchChar;

   function LexOrder (FirstName, SecondName : T) return OrderTypes;

   -- PopChar takes as input a T.  It removes the first character
   -- from the string and returns it in the Char output parameter.  If the empty
   -- string is passed in then the outputs are:
   --    EStr = EmptyString
   --    Char = ' '
   procedure PopChar (EStr : in out T;
                      Char :    out Character);
   --# derives Char,
   --#         EStr from EStr;


   -----------------------------------------------------------------------------
   -- I/O procedures
   -----------------------------------------------------------------------------

   procedure PutString (File : in SPARK_IO.File_Type;
                        EStr : in T);
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                EStr,
   --#                                File;

   procedure PutLine (File : in SPARK_IO.File_Type;
                      EStr : in T);
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                EStr,
   --#                                File;

   procedure GetLine (File : in     SPARK_IO.File_Type;
                      EStr :    out T);
   --# global in out SPARK_IO.File_Sys;
   --# derives EStr,
   --#         SPARK_IO.File_Sys from File,
   --#                                SPARK_IO.File_Sys;



end EStrings;
