-- $Id: estrings.adb 13056 2009-04-20 17:01:20Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with CommonString,
     Ada.Characters.Latin_1;

package body EStrings
is

   function LowerCase (EStr : T) return T
   is
      Res : T;
   begin
      CommonString.LowerCase (EStr.Content, EStr.Length, Res.Content);
      Res.Length := EStr.Length;

      return Res;
   end LowerCase;

   function UpperCase (EStr : T) return T
   is
      Res : T;
   begin
      CommonString.UpperCase (EStr.Content, EStr.Length, Res.Content);
      Res.Length := EStr.Length;

      return Res;
   end UpperCase;

   --------------------------------------------

   -- Case Insensitive comparison.
   function EqString (EStr1, EStr2 : T) return Boolean
   is
   begin
      return CommonString.EqString (EStr1.Content, EStr1.Length, EStr2.Content, EStr2.Length);
   end EqString;

   -- Case INsensitive comparison.
   function Eq1String (M : T;
                       S : String) return Boolean
   is
   begin
      return CommonString.EqString (M.Content, M.Length, S, S'Length);
   end Eq1String;

   function IsEmpty (EStr : T) return Boolean
   is
   begin
      return EqString (EStr, EmptyString);
   end IsEmpty;

   procedure CopyString (EStr :    out T;
                         Str  : in     String)
   is
      ResultString : Contents;
      ResultLength : Lengths;
   begin
      CommonString.CopyString (ResultString, ResultLength, Str);

      EStr.Content := ResultString;
      EStr.Length  := ResultLength;
   end CopyString;

   procedure AppendString (EStr : in out T;
                           Str  : in     String)
   is
      ResultString : Contents;
      ResultLength : Lengths;
   begin
      ResultString := EStr.Content;
      ResultLength := EStr.Length;

      CommonString.AppendString (ResultString, ResultLength, Str, Str'Length);

      EStr.Content := ResultString;
      EStr.Length  := ResultLength;
   end AppendString;

   procedure AppendStringTruncate (EStr : in out T;
                                   Str  : in     String)
   is
      ResultString : Contents;
      ResultLength : Lengths;
   begin
      ResultString := EStr.Content;
      ResultLength := EStr.Length;

      CommonString.AppendStringTruncate (ResultString, ResultLength, Str, Str'Length);

      EStr.Content := ResultString;
      EStr.Length  := ResultLength;
   end AppendStringTruncate;

   procedure AppendExaminerString (EStr1 : in out T;
                                   EStr2 : in     T)
   is
      ResultString : Contents;
      ResultLength : Lengths;
   begin
      ResultString := EStr1.Content;
      ResultLength := EStr1.Length;

      CommonString.AppendString (ResultString, ResultLength, EStr2.Content, EStr2.Length);

      EStr1.Content := ResultString;
      EStr1.Length  := ResultLength;
   end AppendExaminerString;

   procedure AppendExaminerStringTruncate (EStr1 : in out T;
                                           EStr2 : in     T)
   is
      ResultString : Contents;
      ResultLength : Lengths;
   begin
      ResultString := EStr1.Content;
      ResultLength := EStr1.Length;

      CommonString.AppendStringTruncate (ResultString, ResultLength, EStr2.Content, EStr2.Length);

      EStr1.Content := ResultString;
      EStr1.Length  := ResultLength;
   end AppendExaminerStringTruncate;

   procedure PutString (File : in SPARK_IO.File_Type;
                        EStr : in T)
   is
   begin
      CommonString.PutString (File, EStr.Content, EStr.Length);
   end PutString;

   procedure PutLine (File : in SPARK_IO.File_Type;
                      EStr : in T)
   is
   begin
      CommonString.PutLine (File, EStr.Content, EStr.Length);
   end PutLine;

   procedure GetLine (File : in     SPARK_IO.File_Type;
                      EStr :    out T)
   is
      AStr    : Contents;
      ALength : Lengths;
   begin
      CommonString.GetLine (File, AStr, ALength);
      EStr := T'(Content => AStr, Length => ALength);
   end GetLine;

   procedure AppendChar (EStr    : in out T;
                         Ch      : in     Character;
                         Success :    out Boolean)
   is
      ResultString : Contents;
      ResultLength : Lengths;
   begin
      ResultString := EStr.Content;
      ResultLength := EStr.Length;

      CommonString.AppendChar (ResultString, ResultLength, Ch, Success);

      EStr.Content := ResultString;
      EStr.Length  := ResultLength;
   end AppendChar;

   procedure PopChar (EStr : in out T;
                      Char :    out Character)
   is
      ResultString : Contents;
      ResultLength : Lengths;
   begin
      ResultString := EStr.Content;
      ResultLength := EStr.Length;

      CommonString.PopChar (ResultString, ResultLength, Char);

      EStr.Content := ResultString;
      EStr.Length  := ResultLength;
   end PopChar;


   ----------------------------------------------------------------------------
   -- Section added for use in POGS
   -- returns the specified subsection of the string
   -- if the subsection lies outside the string, empty string returned
   ----------------------------------------------------------------------------
   function Section (EStr     : T;
                     StartPos : Positions;
                     Length   : Lengths) return T
   is
      Finished : Boolean;
      Pos      : Positions;
      Result   : T;
   begin
      Result := EmptyString;
      if (StartPos + Length) - 1 > EStr.Length then
         null;
      else
         Pos := StartPos;
         Finished := False;
         while not Finished and then Pos < StartPos + Length loop
            Result.Content ((Pos + 1) - StartPos) := EStr.Content (Pos);
            if not (Pos = Positions'Last) then
               Pos := Pos + 1;
            else
               Finished := True;
            end if;
         end loop;

         Result.Length := Length;
      end if;

      return Result;
   end Section;

   ----------------------------------------------------------------------------
   -- Trim added for use in POGS
   -- removes ' ', Latin_1.HT, Latin_1.LF, Latin_1.CR from either end of the string
   ----------------------------------------------------------------------------
   function Trim (EStr : T) return T
   is
      StartPos      : Positions;
      EndPos        : Positions;
      Finished      : Boolean;
      TrimmedLength : Lengths;
      TrimmedString : T;

      -----------------------------------------------------------------------
      function IsWhiteSpace (C : Character) return Boolean
      is
         Result : Boolean;
      begin
         if C = ' ' or else
            C = Ada.Characters.Latin_1.HT or else
            C = Ada.Characters.Latin_1.LF or else
            C = Ada.Characters.Latin_1.CR then
            Result := True;
         else
            Result := False;
         end if;

         return Result;
      end IsWhiteSpace;

      -----------------------------------------------------------------------
   begin -- Trim
      -- find first non space Character
      StartPos := 1;
      Finished := False;
      while not Finished loop
         if StartPos > EStr.Length or else
            not IsWhiteSpace (EStr.Content (StartPos)) then

            Finished := True;
         else
            StartPos := StartPos + 1;
         end if;
      end loop;

      -- if end of string then return empty string
      if StartPos > EStr.Length then
         TrimmedString := EmptyString;
      else
         -- find last non space Character
         EndPos := EStr.Length;
         Finished := False;
         while not Finished loop
            if EndPos = StartPos or else
               not IsWhiteSpace (EStr.Content (EndPos)) then
               Finished := True;
            else
               EndPos := EndPos - 1;
            end if;
         end loop;

         -- calculate trimmed string length
         TrimmedLength := (EndPos - StartPos) + 1;

         TrimmedString := Section (EStr, StartPos, TrimmedLength);
      end if;

      return TrimmedString;
   end Trim;

   -------------------------------------------------------------------------
   -- FindSubStringAfter: for use in POGS
   -- find the specified SearchString, starting at the specified position in
   -- the given T
   -------------------------------------------------------------------------
   procedure FindSubStringAfter (EStr         : in     T;
                                 SearchStart  : in     Positions;
                                 SearchString : in     String;
                                 StringFound  :    out Boolean;
                                 StringStart  :    out Positions)
   is
      CurrentStartPos      : Positions;
      CurrentCompareOffset : Integer;
      CouldBeHere          : Boolean;
      Found                : Boolean := False;
      LastStartPos         : Positions;
   begin
      CurrentStartPos := SearchStart;
      LastStartPos := (EStr.Length + 1) - SearchString'Length;

      while not Found and then
            CurrentStartPos <= LastStartPos loop

         CurrentCompareOffset := 0;
         CouldBeHere := True;

         while not Found and then
               CouldBeHere and then
               CurrentCompareOffset < SearchString'Length loop

            if EStr.Content (CurrentStartPos + CurrentCompareOffset) /=
              SearchString (1 + CurrentCompareOffset) then
               CouldBeHere := False;
            else
               if CurrentCompareOffset = SearchString'Length - 1 then
                  Found := True;
               else
                  CurrentCompareOffset := CurrentCompareOffset + 1;
               end if;
            end if;
         end loop;

         if not Found then
            CurrentStartPos := CurrentStartPos + 1;
         end if;

      end loop;

      StringFound := Found;
      StringStart := CurrentStartPos;
   end FindSubStringAfter;

   --------------------------------------------------------------------------
   -- FindSubString: for use in POGS
   -- find specified SearchString in the given T
   --------------------------------------------------------------------------
   procedure FindSubString (EStr         : in     T;
                            SearchString : in     String;
                            StringFound  :    out Boolean;
                            StringStart  :    out Positions)
   is
   begin
      FindSubStringAfter (EStr, 1, SearchString, StringFound, StringStart);
   end FindSubString;

   ---------------------------------------------------------------------------
   -- FindCharAfter: find specified Character in EStr, starting at specified
   -- position
   ---------------------------------------------------------------------------
   procedure FindCharAfter (EStr        : in     T;
                            SearchStart : in     Positions;
                            SearchChar  : in     Character;
                            CharFound   :    out Boolean;
                            CharPos     :    out Positions)
   is
      Position : Positions;
      Found    : Boolean := False;
      Finished   : Boolean := False;
   begin
      Position := SearchStart;
      while not Finished loop
         if           Position <= EStr.Length
           and then EStr.Content (Position) = SearchChar then
            Found := True;
            Finished := True;
         elsif Position < EStr.Length then
            Position := Position + 1;
         else
            Finished := True;
            Found := False;
         end if;
      end loop;

      CharPos := Position;
      CharFound := Found;
   end FindCharAfter;

   --------------------------------------------------------------------------
   -- FindChar: find first occurrence of specified Character in EStr
   --------------------------------------------------------------------------
   procedure FindChar (EStr       : in     T;
                       SearchChar : in     Character;
                       CharFound  :    out Boolean;
                       CharPos    :    out Positions)
   is
   begin
      FindCharAfter (EStr, 1, SearchChar, CharFound, CharPos);
   end FindChar;

   --------------------------------------------------------------------------
   function LexOrder (FirstName, SecondName : T)
                     return OrderTypes
   is
      Pos       : Positions := 1;
      Finished  : Boolean := False;
      LexOrderResult : OrderTypes := NeitherFirst;

      RankOfNextCharInFirst  : Integer;
      RankOfNextCharInSecond : Integer;
   begin

      while not Finished loop
         -- if the remainder of only one of the strings is empty then
         -- that is the first in order
         -- if the remainder of both is empty then neither is first
         if Pos > FirstName.Length and Pos <= SecondName.Length then
            LexOrderResult := FirstOneFirst;
            Finished := True;
         elsif Pos > SecondName.Length and Pos <= FirstName.Length then
            LexOrderResult := SecondOneFirst;
            Finished := True;
         elsif Pos > FirstName.Length and Pos > SecondName.Length then
            LexOrderResult := NeitherFirst;
            Finished := True;
         else
            -- decide using the current Character in the string
            RankOfNextCharInFirst := Character'Pos (FirstName.Content (Pos));
            RankOfNextCharInSecond := Character'Pos (SecondName.Content (Pos));

            if RankOfNextCharInFirst < RankOfNextCharInSecond then
               LexOrderResult := FirstOneFirst;
               Finished := True;
            elsif RankOfNextCharInSecond < RankOfNextCharInFirst then
               LexOrderResult := SecondOneFirst;
               Finished := True;
            else
               if Pos < Positions'Last then
                  Pos := Pos + 1;
               else
                  LexOrderResult := NeitherFirst;
                  Finished := True;
               end if;
            end if;
         end if;
      end loop;

      return LexOrderResult;
   end LexOrder;

end EStrings;
