-- $Id: errorhandler.adb 13146 2009-04-24 17:27:57Z Trevor Jennings $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with
  Ada.Characters.Latin_1,
  SystemErrors,
  SPRelations,
  CommandLineData,
  FileSystem,
  Debug,
  ScreenEcho,
  ELStrings,
  XMLReport,
  CommonStringUtilities;

use type CommandLineData.JustificationOptions;
use type ELStrings.T;

package body ErrorHandler
--# own ErrorContext is ErrorContextRec,
--#                     EchoAccumulator,
--#                     ErrorBuffer.Buffer,
--#                     Conversions.State,
--#                     WarningStatus.SuppressionList,
--#                     TotalErrorCount,
--#                     FileOpenError;
is

   ----------------------------------------------------------------------
   -- Centralized location for wrapper strings used when appending error explanations
   -- to error message strings.  These are used by Conversions and in calls to ErrorAccumulator.StartMsg

   ExplanationPrefix  : constant String := " [Explanatory note: ";
   ExplanationPostfix : constant String := "]";

   ----------------------------------------------------------------------

   --# inherit CommandLineData,
   --#         Dictionary,
   --#         ELStrings,
   --#         ErrorHandler,
   --#         Error_IO,
   --#         Error_Types,
   --#         EStrings,
   --#         ExaminerConstants,
   --#         LexTokenManager,
   --#         SPARK_IO,
   --#         SPSymbols,
   --#         SystemErrors;
   package Conversions
   --# own State;
   --# initializes State;
   is

      procedure ToString (ErrNum  : in     Error_Types.NumericError;
                          Purpose : in     Error_Types.ConversionRequestSource;
                          ErrStr  :    out Error_Types.StringError);
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in out State;
      --# derives ErrStr from CommandLineData.Content,
      --#                     Dictionary.Dict,
      --#                     ErrNum,
      --#                     LexTokenManager.StringTable,
      --#                     Purpose,
      --#                     State &
      --#         State  from *,
      --#                     CommandLineData.Content,
      --#                     ErrNum,
      --#                     Purpose;

      procedure OutputReferenceList (ToFile : in SPARK_IO.File_Type);
      --# global in     CommandLineData.Content;
      --#        in     State;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                State,
      --#                                ToFile;

   end Conversions;

   ----------------------------------------------------------------------

   --# inherit CommandLineData,
   --#         Conversions,
   --#         Dictionary,
   --#         ELStrings,
   --#         ErrorHandler,
   --#         Error_IO,
   --#         Error_Types,
   --#         EStrings,
   --#         ExaminerConstants,
   --#         LexTokenManager,
   --#         SPARK_IO,
   --#         SPSymbols,
   --#         SystemErrors;
   package ErrorBuffer
   --# own Buffer;
   --# initializes Buffer;
   is

      --this is a generic add routines called by more closely focussed procedure
      procedure Add (ErrFile     : in out Error_IO.File_Type;
                     ErrType     : in     Error_Types.Error_Class;
                     Pos         : in     LexTokenManager.TokenPosition;
                     Scope       : in     Dictionary.Scopes;
                     ErrorNumber : in     Natural;
                     Reference   : in     Natural;
                     Name1,
                     Name2,
                     Name3       : in     Error_Types.Names;
                     EchoStr     :    out Error_Types.StringError);
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in out Buffer;
      --#        in out Conversions.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives Buffer            from *,
      --#                                CommandLineData.Content,
      --#                                Dictionary.Dict,
      --#                                ErrorNumber,
      --#                                ErrType,
      --#                                Name1,
      --#                                Name2,
      --#                                Name3,
      --#                                Pos,
      --#                                Reference,
      --#                                Scope &
      --#         Conversions.State from *,
      --#                                Buffer,
      --#                                CommandLineData.Content,
      --#                                ErrorNumber,
      --#                                ErrType,
      --#                                Name1,
      --#                                Name2,
      --#                                Name3,
      --#                                Pos,
      --#                                Reference,
      --#                                Scope &
      --#         EchoStr           from Buffer,
      --#                                CommandLineData.Content,
      --#                                Conversions.State,
      --#                                Dictionary.Dict,
      --#                                ErrorNumber,
      --#                                ErrType,
      --#                                LexTokenManager.StringTable,
      --#                                Name1,
      --#                                Name2,
      --#                                Name3,
      --#                                Pos,
      --#                                Reference,
      --#                                Scope &
      --#         ErrFile           from *,
      --#                                Buffer,
      --#                                CommandLineData.Content,
      --#                                ErrorNumber,
      --#                                ErrType,
      --#                                Name1,
      --#                                Name2,
      --#                                Name3,
      --#                                Pos,
      --#                                Reference,
      --#                                Scope,
      --#                                SPARK_IO.File_Sys &
      --#         SPARK_IO.File_Sys from *,
      --#                                Buffer,
      --#                                CommandLineData.Content,
      --#                                Dictionary.Dict,
      --#                                ErrFile,
      --#                                ErrorNumber,
      --#                                ErrType,
      --#                                Name1,
      --#                                Name2,
      --#                                Name3,
      --#                                Pos,
      --#                                Reference,
      --#                                Scope;

      procedure Flush (ErrFile : in out Error_IO.File_Type);
      --# global in     Dictionary.Dict;
      --#        in out Buffer;
      --#        in out SPARK_IO.File_Sys;
      --# derives Buffer            from * &
      --#         ErrFile           from *,
      --#                                Buffer,
      --#                                SPARK_IO.File_Sys &
      --#         SPARK_IO.File_Sys from *,
      --#                                Buffer,
      --#                                Dictionary.Dict,
      --#                                ErrFile;

   end ErrorBuffer;

   ----------------------------------------------------------------------

   -- This package provides a list of justification (accept) annotations found in the
   -- code.  When errors are added, the list can be consulted to see whether the error
   -- should be displayed or not.  The table is populated for the file currently being examined,
   -- when the error context changes, the table contents are saved to a temorary file (or memory-based
   -- simulation of such a file) so that we can list all the justifications in the report and listing files.

   --# inherit CommandLineData,
   --#         Dictionary,
   --#         ErrorHandler,
   --#         EStrings,
   --#         LexTokenManager,
   --#         SPARK_IO,
   --#         SystemErrors,
   --#         XMLReport;
   package Justifications
   is
      type UnmatchedJustificationIterator is private;

      procedure StartUnit (WhichTable : in out ErrorHandler.JustificationsDataTables);
      --# derives WhichTable from *;

      procedure SetCurrentUnitHasSemanticErrors (WhichTable : in out ErrorHandler.JustificationsDataTables);
      --# derives WhichTable from *;

      procedure FirstUnmatchedJustification (It         :    out UnmatchedJustificationIterator;
                                             WhichTable : in     ErrorHandler.JustificationsDataTables);
      --# global in CommandLineData.Content;
      --# derives It from CommandLineData.Content,
      --#                 WhichTable;

      procedure NextUnmatchedJustification (It         : in out UnmatchedJustificationIterator;
                                            WhichTable : in     ErrorHandler.JustificationsDataTables);
      --# global in CommandLineData.Content;
      --# derives It from *,
      --#                 CommandLineData.Content,
      --#                 WhichTable;

      function ErrorPosition (It         : UnmatchedJustificationIterator) return LexTokenManager.TokenPosition;

      function IsNullIterator (It : UnmatchedJustificationIterator) return Boolean;

      procedure EndUnit (WhichTable : in out ErrorHandler.JustificationsDataTables);
      --# derives WhichTable from *;

      procedure StartJustification (WhichTable                   : in out ErrorHandler.JustificationsDataTables;
                                    Position                     : in     LexTokenManager.TokenPosition;
                                    Line                         : in     LexTokenManager.LineNumbers;
                                    Kind                         : in     ErrorHandler.JustificationKinds;
                                    ErrNum                       : in     Natural;
                                    Identifiers                  : in     ErrorHandler.JustificationIdentifiers;
                                    Explanation                  : in     LexTokenManager.LexString;
                                    MaximumJustificationsReached :    out Boolean);
      --# derives MaximumJustificationsReached from WhichTable &
      --#         WhichTable                   from *,
      --#                                           ErrNum,
      --#                                           Explanation,
      --#                                           Identifiers,
      --#                                           Kind,
      --#                                           Line,
      --#                                           Position;

      procedure EndJustification (WhichTable   : in out ErrorHandler.JustificationsDataTables;
                                  Line         : in     LexTokenManager.LineNumbers;
                                  UnmatchedEnd :    out Boolean);
      --# derives UnmatchedEnd,
      --#         WhichTable   from Line,
      --#                           WhichTable;

      procedure CheckWhetherJustified (WhichTable  : in out ErrorHandler.JustificationsDataTables;
                                       Line        : in     LexTokenManager.TokenPosition;
                                       Kind        : in     ErrorHandler.JustificationKinds;
                                       ErrNum      : in     Natural;
                                       Identifiers : in     ErrorHandler.JustificationIdentifiers;
                                       MatchFound  :    out Boolean);
      --# global in CommandLineData.Content;
      --# derives MatchFound,
      --#         WhichTable from CommandLineData.Content,
      --#                         ErrNum,
      --#                         Identifiers,
      --#                         Kind,
      --#                         Line,
      --#                         WhichTable;

      procedure PrintJustifications (WhichTable : in     ErrorHandler.JustificationsDataTables;
                                     File       : in     SPARK_IO.File_Type);
      --# global in     CommandLineData.Content;
      --#        in     LexTokenManager.StringTable;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                File,
      --#                                LexTokenManager.StringTable,
      --#                                WhichTable;

      procedure PrintJustificationsXML (WhichTable : in     ErrorHandler.JustificationsDataTables;
                                        File       : in     SPARK_IO.File_Type);
      --# global in     CommandLineData.Content;
      --#        in     LexTokenManager.StringTable;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                File,
      --#                                LexTokenManager.StringTable,
      --#                                WhichTable,
      --#                                XMLReport.State &
      --#         XMLReport.State   from *,
      --#                                CommandLineData.Content,
      --#                                WhichTable;
   private
      type UnmatchedJustificationIterator is
         record
            CurrentTableEntry : ErrorHandler.DataTablePtr;
            CurrentPosition : LexTokenManager.TokenPosition;
         end record;
   end Justifications;

   ----------------------------------------------------------------------

   --# inherit Ada.Characters.Handling,
   --#         Ada.Characters.Latin_1,
   --#         CommandLineData,
   --#         ErrorHandler,
   --#         EStrings,
   --#         ExaminerConstants,
   --#         FileSystem,
   --#         LexTokenManager,
   --#         ScreenEcho,
   --#         SPARK_IO,
   --#         SystemErrors,
   --#         XMLReport;
   package WarningStatus
      --# own SuppressionList;
      --# initializes SuppressionList;
   is
      procedure ReadWarningFile;
      --# global in     CommandLineData.Content;
      --#        in out ErrorHandler.FileOpenError;
      --#        in out LexTokenManager.StringTable;
      --#        in out SPARK_IO.File_Sys;
      --#        in out SuppressionList;
      --# derives ErrorHandler.FileOpenError  from *,
      --#                                          CommandLineData.Content,
      --#                                          SPARK_IO.File_Sys &
      --#         LexTokenManager.StringTable,
      --#         SPARK_IO.File_Sys,
      --#         SuppressionList             from CommandLineData.Content,
      --#                                          LexTokenManager.StringTable,
      --#                                          SPARK_IO.File_Sys,
      --#                                          SuppressionList;

      function IsSuppressed (TheElement : ErrorHandler.WarningElements) return Boolean;
      --# global in SuppressionList;
      function PragmaIsSuppressed (PragmaName : LexTokenManager.LexString)
                                  return Boolean;
      --# global in SuppressionList;
      procedure OutputWarningList (ToFile : in SPARK_IO.File_Type);
      --# global in     CommandLineData.Content;
      --#        in     LexTokenManager.StringTable;
      --#        in     SuppressionList;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                LexTokenManager.StringTable,
      --#                                SuppressionList,
      --#                                ToFile,
      --#                                XMLReport.State &
      --#         XMLReport.State   from *,
      --#                                CommandLineData.Content,
      --#                                SuppressionList;

      procedure ReportSuppressedWarnings (ToFile  : in SPARK_IO.File_Type;
                                          Counter : ErrorHandler.Counters);
      --# global in     SuppressionList;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Counter,
      --#                                SuppressionList,
      --#                                ToFile;

   end WarningStatus;

   ----------------------------------------------------------------------
   StmtWidth          : constant Positive := 4;
   SourceLineIndent   : constant Positive := 6;
   ErrorLineLength    : constant Positive := 80;
   XMLErrorLineLength : constant Positive := 256;

   subtype ErrorSetLengths   is Natural range
      0 .. ExaminerConstants.MaxErrorSetSize;
   subtype ErrorSetPositions is Natural range
      1 .. ExaminerConstants.MaxErrorSetSize;
   type ErrorStruct is
      record
         ErrNum : Natural;
         Error  : Error_Types.StringError;
      end record;
   type ErrorArray is array (ErrorSetPositions) of ErrorStruct;
   type ErrorSets is
      record
         Length  : ErrorSetLengths;
         Content : ErrorArray;
      end record;

   EmptyErrorStruct : constant ErrorStruct := ErrorStruct'(0, Error_Types.Empty_StringError);
   EmptyErrorArray  : constant ErrorArray  :=
      ErrorArray'(ErrorSetPositions => EmptyErrorStruct);
   EmptyErrorSet    : constant ErrorSets   := ErrorSets'(0, EmptyErrorArray);
   ----------------------------------------------------------------------

   --# inherit CommandLineData,
   --#         ELStrings,
   --#         ErrorHandler,
   --#         Error_Types,
   --#         EStrings,
   --#         ExaminerConstants,
   --#         SPARK_IO,
   --#         SystemErrors,
   --#         XMLReport;
   package ErrorAccumulator is

      type T is private;

      Clear : constant T;

      function IsErrorContinuation (TheError : Error_Types.StringError) return Boolean;

      function IsErrorStart (TheError : Error_Types.StringError) return Boolean;

      function IsActive (This : T) return Boolean;

      procedure StartMsg (This        :    out T;
                          StartError  : in     ErrorHandler.ErrorStruct;
                          StartIndent : in     Natural;
                          Explanation : in     ELStrings.T;
                          LineLength  : in     Natural;
                          Indent      : in     Natural);
      --# derives This from Explanation,
      --#                   Indent,
      --#                   LineLength,
      --#                   StartError,
      --#                   StartIndent;

      procedure Flush (This     : in out T;
                       Listing  : in     SPARK_IO.File_Type);
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Listing,
      --#                                This &
      --#         This              from *;

      procedure Add (This     : in out T;
                     Error    : in     ErrorHandler.ErrorStruct;
                     EndPos,
                     Indent   : in     Natural;
                     Listing  : in     SPARK_IO.File_Type);
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                EndPos,
      --#                                Error,
      --#                                Indent,
      --#                                Listing,
      --#                                This &
      --#         This              from *,
      --#                                EndPos,
      --#                                Error,
      --#                                Indent;
      --#

   private

      type T is
         record
            Active      : Boolean;
            StartError  : ErrorHandler.ErrorStruct;
            StartIndent : Natural;
            -- following fields only concerned with appending explanation to message after Flush operation
            Explanation : ELStrings.T;
            LineLength  : Natural;
            Indent      : Natural;
         end record;

      Clear : constant T :=
        T'(Active      => False,
           StartError  => ErrorHandler.EmptyErrorStruct,
           StartIndent => 0,
           Explanation => ELStrings.EmptyString,
           LineLength  => 0, -- gets set by ErrorAccumulator.StartMsg
           Indent      => 0  -- gets set by ErrorAccumulator.StartMsg
          );

   end ErrorAccumulator;

   ----------------------------------------------------------------------

   ErrorContextRec : ErrorContexts;
   EchoAccumulator : ErrorAccumulator.T;
   TotalErrorCount : TotalErrorCounts;
   FileOpenError : Boolean := False;

   ----------------------------------------------------------------------

   function DependencyErrNumber (ErrType : FullDependErrType) return Natural
   is
      Result : Natural;
   begin
      case ErrType is
         when NotUsed   |
           NotUsedNew   |
           NotUsedContinue |
           IneffInit |
           IneffLocalInit =>
            Result := DependencyErrType'Pos (ErrType)
              + Error_Types.UncondDependencyErrorOffset;
         when MayBeUsed |
           MayBeUsedNew |
           MayBeUsedContinue |
           Uninitialised |
           IntegrityViolation |
           MayBeIntegrityViolation =>
            Result := DependencyErrType'Pos (ErrType)
              + Error_Types.CondDependencyErrorOffset;
      end case;
      return Result;
   end DependencyErrNumber;

   ----------------------------------------------------------------------

   package body Conversions
      is separate;

   ----------------------------------------------------------------------

   package body ErrorBuffer
      is separate;

   ----------------------------------------------------------------------

   package body WarningStatus
      is separate;

   ----------------------------------------------------------------------

   package body Justifications
      is separate;

   ----------------------------------------------------------------------

   function SymbolToJustificationIdentifier (Sym : Dictionary.Symbol) return JustificationIdentifier
   is
   begin
      return JustificationIdentifier'(StringForm => LexTokenManager.NullString,
                                      SymbolForm => Sym);
   end SymbolToJustificationIdentifier;

   function LexStrToJustificationIdentifier (Str : LexTokenManager.LexString) return JustificationIdentifier
   is
   begin
      return JustificationIdentifier'(StringForm => Str,
                                      SymbolForm => Dictionary.NullSymbol);
   end LexStrToJustificationIdentifier;


   procedure ReadWarningFile
   --# global in     CommandLineData.Content;
   --#        in out FileOpenError;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out WarningStatus.SuppressionList;
   --# derives FileOpenError                 from *,
   --#                                            CommandLineData.Content,
   --#                                            SPARK_IO.File_Sys &
   --#         LexTokenManager.StringTable,
   --#         SPARK_IO.File_Sys,
   --#         WarningStatus.SuppressionList from CommandLineData.Content,
   --#                                            LexTokenManager.StringTable,
   --#                                            SPARK_IO.File_Sys,
   --#                                            WarningStatus.SuppressionList;
   is
   begin
      WarningStatus.ReadWarningFile;
   end ReadWarningFile;

   procedure OutputWarningList (ToFile : in SPARK_IO.File_Type)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     WarningStatus.SuppressionList;
   --#        in out SPARK_IO.File_Sys;
   --#        in out XMLReport.State;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                LexTokenManager.StringTable,
   --#                                ToFile,
   --#                                WarningStatus.SuppressionList,
   --#                                XMLReport.State &
   --#         XMLReport.State   from *,
   --#                                CommandLineData.Content,
   --#                                WarningStatus.SuppressionList;
   is
   begin
      WarningStatus.OutputWarningList (ToFile);
   end OutputWarningList;

   ----------------------------------------------------------------------

   procedure OutputReferenceList (ToFile : in SPARK_IO.File_Type)
   --# global in     CommandLineData.Content;
   --#        in     Conversions.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Conversions.State,
   --#                                ToFile;
   is
   begin
      Conversions.OutputReferenceList (ToFile);
   end OutputReferenceList;

   --------------------------------------------------------------------------

   function LexStringToName (Str : LexTokenManager.LexString) return
      Error_Types.Names
   is
      Result : Error_Types.Names;
   begin
      if Str = LexTokenManager.NullString then
         Result := Error_Types.NoName;
      elsif Str = LexTokenManager.Main_ProgramToken then
         Result := Error_Types.ThePartitionName;
      else
         Result := Error_Types.Names'(Error_Types.LexString,
                                      LexTokenManager.LexStringRef (Str));
      end if;
      return Result;
   end LexStringToName;

   --------------------------------------------------------------------------

   function SymbolToName (Sym : Dictionary.Symbol) return
      Error_Types.Names
   is
      Result : Error_Types.Names;
   begin
      if Sym = Dictionary.NullSymbol then
         Result := Error_Types.NoName;
      else
         Result := Error_Types.Names'(Error_Types.Symbol,
                                      Integer (Dictionary.SymbolRef (Sym)));
      end if;
      return Result;
   end SymbolToName;

   --------------------------------------------------------------------------

   function SPSymbolToName (Sym : SPSymbols.SPSymbol) return
      Error_Types.Names
   is
   begin
      return Error_Types.Names'(Error_Types.ParserSymbol,
                                SPSymbols.SPSymbol'Pos (Sym));
   end SPSymbolToName;

   --------------------------------------------------------------------------

   procedure AppendString (EStr : in out Error_Types.StringError;
                           Str  : in     String)
      --# derives EStr from *,
      --#                   Str;
   is
      pragma Inline (AppendString);
   begin
      ELStrings.AppendString (EStr.Message, Str);
   end AppendString;

   ------------------------------------------------------------------------

   procedure AppendSym (Error : in out Error_Types.StringError;
                        Sym   : in     SPSymbols.SPSymbol)
      --# derives Error from *,
      --#                    Sym;
      is separate;

   ----------------------------------------------------------------------

   procedure AppendLexString (EStr : in out Error_Types.StringError;
                              LStr : in     LexTokenManager.LexString)
   --# global in LexTokenManager.StringTable;
   --# derives EStr from *,
   --#                   LexTokenManager.StringTable,
   --#                   LStr;
   is
      pragma Inline (AppendLexString);
      Str : EStrings.T;
   begin
      LexTokenManager.LexStringToString (LStr, Str);
      ELStrings.AppendExaminerString (EStr.Message, Str);
   end AppendLexString;
   pragma Unreferenced (AppendLexString); -- unused at present

   ----------------------------------------------------------------------

   procedure Set_Col (File : in SPARK_IO.File_Type;
                      Posn : in Positive)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                Posn;
   is
      pragma Inline (Set_Col);
   begin
      if File = SPARK_IO.Standard_Output then
         ScreenEcho.Set_Col (Posn);
      else
         SPARK_IO.Set_Col (File, Posn);
      end if;
   end Set_Col;

   procedure Put_Char (File : in SPARK_IO.File_Type;
                       Item : in Character)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                Item;
   is
      pragma Inline (Put_Char);
   begin
      if File = SPARK_IO.Standard_Output then
         ScreenEcho.Put_Char (Item);
      else
         SPARK_IO.Put_Char (File, Item);
      end if;
   end Put_Char;

   procedure Put_Integer (File  : in SPARK_IO.File_Type;
                          Item  : in Integer;
                          Width : in Natural;
                          Base  : in SPARK_IO.Number_Base)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Base,
   --#                                File,
   --#                                Item,
   --#                                Width;
   is
      pragma Inline (Put_Integer);
   begin
      if File = SPARK_IO.Standard_Output then
         ScreenEcho.Put_Integer (Item, Width, Base);
      else
         SPARK_IO.Put_Integer (File, Item, Width, Base);
      end if;
   end Put_Integer;

   procedure Put_String (File : in SPARK_IO.File_Type;
                         Item : in String;
                         Stop : in Natural)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                Item,
   --#                                Stop;
   is
   begin
      SPARK_IO.Put_String (File, Item, Stop);
   end Put_String;

   procedure New_Line (File    : in SPARK_IO.File_Type;
                       Spacing : in Positive)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                Spacing;
   is
      pragma Inline (New_Line);
   begin
      if File = SPARK_IO.Standard_Output then
         ScreenEcho.New_Line (Spacing);
      else
         SPARK_IO.New_Line (File, Spacing);
      end if;
   end New_Line;


   procedure Put_Line (File : in SPARK_IO.File_Type;
                       Item : in String;
                       Stop : in Natural)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                Item,
   --#                                Stop;
   is
      pragma Inline (Put_Line);
   begin
      if File = SPARK_IO.Standard_Output then
         ScreenEcho.Put_Line (Item);
      else
         SPARK_IO.Put_Line (File, Item, Stop);
      end if;
   end Put_Line;

   procedure PutEString (File : in SPARK_IO.File_Type;
                         EStr : in EStrings.T)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                EStr,
   --#                                File;
   is
      pragma Inline (PutEString);
   begin
      if File = SPARK_IO.Standard_Output then
         ScreenEcho.Put_ExaminerString (EStr);
      else
         EStrings.PutString (File, EStr);
      end if;
   end PutEString;
   pragma Unreferenced (PutEString); -- unused at present

   procedure PutELine (File : in SPARK_IO.File_Type;
                       EStr : in EStrings.T)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                EStr,
   --#                                File;
   is
      pragma Inline (PutELine);
   begin
      if File = SPARK_IO.Standard_Output then
         ScreenEcho.Put_ExaminerLine (EStr);
      else
         EStrings.PutLine (File, EStr);
      end if;
   end PutELine;

   ----------------------------------------------------------------------

   procedure PutSpaces (File : in SPARK_IO.File_Type;
                        N    : in Natural)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                N;
   is
   begin
      for I in Natural range 1 .. N
      loop
         Put_Char (File, ' ');
      end loop;
   end PutSpaces;

   ----------------------------------------------------------------------

   procedure PrintLine (Listing  : in SPARK_IO.File_Type;
                        StartPos,
                           EndPos,
                           Indent   : in Natural;
                        Line     : in ELStrings.T;
                        NewLine  : in Boolean;
                        NewStart : out Natural)
   --# global in out SPARK_IO.File_Sys;
   --# derives NewStart          from EndPos,
   --#                                Indent,
   --#                                Line,
   --#                                StartPos &
   --#         SPARK_IO.File_Sys from *,
   --#                                EndPos,
   --#                                Indent,
   --#                                Line,
   --#                                Listing,
   --#                                NewLine,
   --#                                StartPos;
      is separate;

   ----------------------------------------------------------------------------

   package body ErrorAccumulator
      is separate;

   -----------------------------------------------------------------------------
   -- String utility used in calls to ErrorAccumulator.StartMsg to remove explanations from
   -- strings being stored in the accumulator so that they can be kept and
   -- printed when the accumulator is flushed

   -- e.g. if "rage, rage against the dying of the light [Explanatory note: by Dylan Thomas]" is passed
   -- in in Estr and ErrorHandlerError.Prefix is set to " [Explanatory note: " then after the call,
   -- Estr has "rage, rage against the dying of the light" and
   -- Explanation has " [Explanatory note: by Dylan Thomas]"

   -- If there is no match for ErrorHandlerError.Prefix then EStr is unchanged and Explanation
   -- is the EmptyString;

   procedure SplitStringAtExplanation (EStr        : in out ELStrings.T;
                                       Explanation :    out ELStrings.T)
   --# derives EStr,
   --#         Explanation from EStr;
   is
      StartPoint : Natural;
      FoundIt : Boolean;
   begin
      CommonStringUtilities.FindSubStringAfter (EStr.Content,
                                                EStr.Length,
                                                1,
                                                ExplanationPrefix,
                                                ExplanationPrefix'Length,
                                                FoundIt,
                                                StartPoint);
      if FoundIt then
         -- copy characters from EStr to Explanation
         for I in ELStrings.Positions range StartPoint .. EStr.Length  loop
            --# accept Flow, 23, Explanation.Content, "First write to array";
            Explanation.Content ((I + 1) - StartPoint) := EStr.Content (I); -- expect flow error
            --# end accept;
         end loop;
         Explanation.Length := (EStr.Length - StartPoint) + 1;

         -- truncate EStr to remove Explanation
         EStr.Length := StartPoint - 1;
      else
         Explanation := ELStrings.EmptyString;
      end if;
      --# accept Flow, 602, Explanation, Explanation.Content, "Consequence of first write to array";
   end SplitStringAtExplanation; -- one DF error reported on Explanation.Content

   -----------------------------------------------------------------------------
   procedure FlushBuffer
   --# global in     Dictionary.Dict;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorBuffer.Buffer from * &
   --#         ErrorContextRec    from *,
   --#                                 ErrorBuffer.Buffer,
   --#                                 SPARK_IO.File_Sys &
   --#         SPARK_IO.File_Sys  from *,
   --#                                 Dictionary.Dict,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec;
   is
      ErrFile   : Error_IO.File_Type;
   begin
      ErrFile := ErrorContextRec.Errs;
      ErrorBuffer.Flush (ErrFile);
      ErrorContextRec.Errs := ErrFile;
   end FlushBuffer;

   ----------------------------------------------------------------------

   procedure FlushEchoMessages
   --# global in out EchoAccumulator;
   --#        in out SPARK_IO.File_Sys;
   --# derives EchoAccumulator,
   --#         SPARK_IO.File_Sys from *,
   --#                                EchoAccumulator;
   is
      WasActive : Boolean;
   begin
      WasActive := ErrorAccumulator.IsActive (EchoAccumulator);
      ErrorAccumulator.Flush (EchoAccumulator, SPARK_IO.Standard_Output);
      if WasActive then
         New_Line (SPARK_IO.Standard_Output, 1);
      end if;

   end FlushEchoMessages;
   ----------------------------------------------------------------------

--   procedure PutErrorEntry (File : in Temp_IO.File_Type;
--                            Ent  : in Error_Types.Error_Entry)
--      --# global  SPARK_IO.File_Sys;
--      --# derives SPARK_IO.File_Sys from File, Ent, SPARK_IO.File_Sys;
--   is
--      pragma Inline (PutErrorEntry);
--   begin
--      Temp_IO.Put_Integer (File, Ent.MessageId, 1);
--      Temp_IO.Put_Char (File, ' ');
--
--      Temp_IO.Put_Integer (File, Integer (Ent.Position.StartLineNo), 1);
--      Temp_IO.Put_Char (File, ' ');
--      Temp_IO.Put_Integer (File, Ent.Position.StartPos, 1);
--      Temp_IO.Put_Char (File, ' ');
--      Temp_IO.Put_Integer (File, Error_Types.Error_Class'Pos (Ent.ErrorType), 1);
--      Temp_IO.Put_Char (File, ' ');
--      Temp_IO.New_Line (File, 1);
--      Temp_IO.PutEString (File, Ent.Message);
--      Temp_IO.New_Line (File, 1);
--   end PutErrorEntry;

   ----------------------------------------------------------------------

   procedure IncTotalErrorsOrWarnings (Kind : in CountedErrorKinds)
   --# global in out TotalErrorCount;
   --# derives TotalErrorCount from *,
   --#                              Kind;
   is
   begin
      if TotalErrorCount.GrandTotal = Count'Last then
         SystemErrors.FatalError (SystemErrors.TooManyErrors, "");
      end if;
      TotalErrorCount.GrandTotal := TotalErrorCount.GrandTotal + 1;
      -- We don't need to guard the following increment because a subtotal can't
      -- overflow without the grand total (tested above) overflowing first
      TotalErrorCount.ExplicitErrorCount (Kind) := TotalErrorCount.ExplicitErrorCount (Kind) + 1;
   end IncTotalErrorsOrWarnings;

   ----------------------------------------------------------------------

   procedure IncTotalSuppressedWarnings
   --# global in out TotalErrorCount;
   --# derives TotalErrorCount from *;
   is
   begin
      if TotalErrorCount.SuppressedWarnings = Count'Last then
         SystemErrors.FatalError (SystemErrors.TooManyErrors, "");
      end if;
      TotalErrorCount.SuppressedWarnings := TotalErrorCount.SuppressedWarnings + 1;
   end IncTotalSuppressedWarnings;

   ----------------------------------------------------------------------

   procedure IncTotalJustifiedWarnings
   --# global in out TotalErrorCount;
   --# derives TotalErrorCount from *;
   is
   begin
      if TotalErrorCount.JustifiedWarnings = Count'Last then
         SystemErrors.FatalError (SystemErrors.TooManyErrors, "");
      end if;
      TotalErrorCount.JustifiedWarnings := TotalErrorCount.JustifiedWarnings + 1;
   end IncTotalJustifiedWarnings;

   ----------------------------------------------------------------------

   procedure EchoTotalErrorCount
   --# global in     CommandLineData.Content;
   --#        in     TotalErrorCount;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                TotalErrorCount;
   is
      procedure Indent
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *;
      is
      begin
         ScreenEcho.Put_String ("     ");
      end Indent;

      procedure EndLine (IsPlural : in Boolean)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                IsPlural;
      is
      begin
         if IsPlural then
            ScreenEcho.Put_Line ("s");
         else
            ScreenEcho.New_Line (1);
         end if;
      end EndLine;

   begin -- EchoTotalErrorCount
      if CommandLineData.Content.Echo then
         ScreenEcho.New_Line (1);

         -- Explicit and summarized warnings are handled separately
         -- First explicit counts
         if TotalErrorCount.GrandTotal = 0 then
            ScreenEcho.Put_Line ("   No errors or warnings");
         else
            ScreenEcho.Put_Integer (Integer (TotalErrorCount.GrandTotal), 5, 10);
            if TotalErrorCount.GrandTotal > 1 then
               ScreenEcho.Put_Line (" errors or warnings, comprising:");
            else
               ScreenEcho.Put_Line (" error or warning, comprising:");
            end if;
            -- List subtotals by category
            if TotalErrorCount.ExplicitErrorCount (SyntaxOrSemantic) > 0 then
               Indent;
               ScreenEcho.Put_Integer (Integer (TotalErrorCount.ExplicitErrorCount (SyntaxOrSemantic)), 5, 10);
               ScreenEcho.Put_String (" syntax or semantic error");
               EndLine (TotalErrorCount.ExplicitErrorCount (SyntaxOrSemantic) > 1);
            end if;
            if TotalErrorCount.ExplicitErrorCount (Flow) > 0 then
               Indent;
               ScreenEcho.Put_Integer (Integer (TotalErrorCount.ExplicitErrorCount (Flow)), 5, 10);
               ScreenEcho.Put_String (" flow error");
               EndLine (TotalErrorCount.ExplicitErrorCount (Flow) > 1);
            end if;
            if TotalErrorCount.ExplicitErrorCount (Warning) > 0 then
               Indent;
               ScreenEcho.Put_Integer (Integer (TotalErrorCount.ExplicitErrorCount (Warning)), 5, 10);
               ScreenEcho.Put_String (" warning");
               EndLine (TotalErrorCount.ExplicitErrorCount (Warning) > 1);
            end if;
         end if;

         -- Then append summary of suppressed warnings (say nothing at all if total is 0)
         if TotalErrorCount.SuppressedWarnings > 0 then
            ScreenEcho.Put_Integer (Integer (TotalErrorCount.SuppressedWarnings), 5, 10);
            ScreenEcho.Put_String (" summarized warning");
            EndLine (TotalErrorCount.SuppressedWarnings > 1);
         end if;

         -- Then append summary of justified warnings (say nothing at all if total is 0)
         if TotalErrorCount.JustifiedWarnings > 0 then
            ScreenEcho.Put_Integer (Integer (TotalErrorCount.JustifiedWarnings), 5, 10);
            ScreenEcho.Put_String (" expected (justified) warning");
            EndLine (TotalErrorCount.JustifiedWarnings > 1);
         end if;
      end if;
   end EchoTotalErrorCount;

   ----------------------------------------------------------------------

   procedure IncMessageCount (ErrType : in Error_Types.Error_Class)
   --# global in out ErrorContextRec;
   --#        in out TotalErrorCount;
   --# derives ErrorContextRec,
   --#         TotalErrorCount from *,
   --#                              ErrType;
   is
      pragma Inline (IncMessageCount);
   begin
      if ErrorContextRec.NumMessage = Count'Last then
         SystemErrors.FatalError (SystemErrors.TooManyErrors, "");
      end if;
      ErrorContextRec.NumMessage := ErrorContextRec.NumMessage + 1;
      if ErrType /= Error_Types.NoErr then
         if ErrorContextRec.NumErrs = Count'Last then
            SystemErrors.FatalError (SystemErrors.TooManyErrors, "");
         end if;
         ErrorContextRec.NumErrs := ErrorContextRec.NumErrs + 1;
      end if;
      -- Above lines maintain counts of errors per unit in the current ErrorContext record.

      -- We also need to maintain grand totals (in various subcategories) thus:
      case ErrType is
         when
           Error_Types.LexErr     |
           Error_Types.SyntaxErr  |
           Error_Types.SyntaxRec  |
           Error_Types.SemanticErr => IncTotalErrorsOrWarnings (SyntaxOrSemantic);
         when
           Error_Types.UncondFlowErr       |
           Error_Types.CondlFlowErr        |
           Error_Types.UncondDependencyErr |
           Error_Types.CondlDependencyErr  |
           Error_Types.DepSemanticErr      |  -- refinement inconsistency, treat as a flow error
           Error_Types.ControlFlowErr      |
           Error_Types.IneffectiveStat     |
           Error_Types.StabilityErr        |
           Error_Types.UsageErr  => IncTotalErrorsOrWarnings (Flow);
         when
           Error_Types.WarningWithPosition    |
           Error_Types.WarningWithoutPosition |
           Error_Types.Note  => IncTotalErrorsOrWarnings (Warning);
         when
           Error_Types.NoErr => null;
      end case;
   end IncMessageCount;

   ----------------------------------------------------------------------

--    procedure GetErrorEntry (File : in     Temp_IO.File_Type;
--                             Ent  :    out Error_Types.Error_Entry;
--                             OK   :    out Boolean)
--       --# global SPARK_IO.File_Sys;
--       --# derives Ent               from File, SPARK_IO.File_Sys &
--       --#         OK                from File, SPARK_IO.File_Sys &
--       --#         SPARK_IO.File_Sys from File, SPARK_IO.File_Sys;
--    is
--       Str     : EStrings.Contents;
--       Stop    : EStrings.Lengths;
--       ErrTypeNo,
--       LineNo  : Natural;
--       Pos     : EStrings.Lengths;
--       LOk     : Boolean;
--       Read    : Boolean;
--       MessageId : Natural;
--    begin
--       OK  := False;
--       Ent := Error_Types.Empty_Error_Entry;
--       LOk := not Temp_IO.End_Of_File (File);
--       if LOk then
--          Temp_IO.Get_Integer (File, MessageId, 0, Read);
--          LOk := Read and not Temp_IO.End_Of_File (File);
--          if LOk then
--             Temp_IO.Get_Integer (File, LineNo, 0, Read);
--             LOk := Read and not Temp_IO.End_Of_File (File);
--             if LOk then
--                Temp_IO.Get_Integer (File, Pos, 0, Read);
--                LOk := Read and not Temp_IO.End_Of_File (File);
--                if LOk then
--                   Temp_IO.Get_Integer (File, ErrTypeNo, 0, Read);
--                   LOk := Read and not Temp_IO.End_Of_File (File);
--                   if LOk then
--                      Temp_IO.Skip_Line (File, 1); -- skip space;
--                      Temp_IO.Get_Line (File, Str, Stop);
--                      LOk := Stop /= 0;
--                      if LOk then
--                         OK := True;
--                         Ent := Error_Types.Error_Entry'(
--                                            MessageId,
--                                            Error_Types.Error_Class'Val (ErrTypeNo),
--                                            LexTokenManager.TokenPosition'(
--                                                                           LexTokenManager.LineNumbers (LineNo), Pos),
--                                            EStrings.T'(Stop, Str));
--                      end if;
--                   end if;
--                end if;
--             end if;
--          end if;
--       end if;
--    end GetErrorEntry;

   ----------------------------------------------------------------------

   procedure MoveToIndent (SourceFile : in SPARK_IO.File_Type;
                           Line       :    EStrings.T;
                           Indent     :    Natural;
                           Position   :    Integer)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Indent,
   --#                                Line,
   --#                                Position,
   --#                                SourceFile;
   is
      limit : Integer;
   begin
      PutSpaces (SourceFile, Indent);
      if Position > Line.Length then
         limit := Line.Length;
      else
         limit := Position;
      end if;
      for i in Natural range 1 .. limit
      loop
         if Line.Content (i) = Ada.Characters.Latin_1.HT then
            Put_Char (SourceFile, Ada.Characters.Latin_1.HT);
         else
            Put_Char (SourceFile, ' ');
         end if;
      end loop;
      for i in Natural range 1 .. Position - limit
      loop
         Put_Char (SourceFile, ' ');
      end loop;
   end MoveToIndent;

   ----------------------------------------------------------------------

   procedure GetFileLine
   --# global in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContextRec,
   --#         SPARK_IO.File_Sys from ErrorContextRec,
   --#                                SPARK_IO.File_Sys;
      is separate;

   ----------------------------------------------------------------------

   procedure PrintSourceLine (ToFile : SPARK_IO.File_Type)
   --# global in     CommandLineData.Content;
   --#        in     ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                ErrorContextRec,
   --#                                ToFile;
   is
   begin
      if not CommandLineData.Content.PlainOutput then
         Put_Integer (ToFile, Integer (ErrorContextRec.LineNo), StmtWidth, 10);
      end if;

      Set_Col (ToFile, SourceLineIndent + 1);
      PutELine (ToFile, ErrorContextRec.CurrentLine);
   end PrintSourceLine;

   ----------------------------------------------------------------------

   function ErrorHasPositionInline (ErrType : in Error_Types.Error_Class) return Boolean
   is
   begin

      return not (ErrType = Error_Types.UncondDependencyErr or
                  ErrType = Error_Types.NoErr               or
                  ErrType = Error_Types.CondlDependencyErr  or
                  ErrType = Error_Types.DepSemanticErr      or
                  ErrType = Error_Types.UsageErr            or
                  ErrType = Error_Types.Note                or
                  ErrType = Error_Types.WarningWithoutPosition);
   end ErrorHasPositionInline;

   ----------------------------------------------------------------------

   function WidthOf (ErrCount : in Natural) return Positive
   is
      Size : Positive;
   begin
      if    ErrCount < 10 then
         Size := 1;
      elsif ErrCount < 100 then
         Size := 2;
      elsif ErrCount < 1000 then
         Size := 3;
      else
         Size := 4;
      end if;
      return Size;
   end WidthOf;

   ----------------------------------------------------------------------

   function ConvertMessageId (MessageId : in Natural;
                              ErrType   : in Error_Types.Error_Class)
                             return Natural
   is
      Id : Natural;
   begin
      case ErrType is
         when Error_Types.CondlDependencyErr =>
            if MessageId = DependencyErrNumber (MayBeUsedNew) then
               Id := DependencyErrNumber (MayBeUsed);
            else
               Id := MessageId;
            end if;
         when Error_Types.UncondDependencyErr =>
            if MessageId = DependencyErrNumber (NotUsedNew) then
               Id := DependencyErrNumber (NotUsed);
            else
               Id := MessageId;
            end if;
         when others =>
               Id := MessageId;
      end case;
      return Id;
   end ConvertMessageId;


   procedure PutMessageId (File      : in SPARK_IO.File_Type;
                           MessageId : in Natural;
                           ErrType   : in Error_Types.Error_Class)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                ErrType,
   --#                                File,
   --#                                MessageId;
   is
   begin
      if MessageId = 0 then
         SPARK_IO.Put_Char (File, ' ');
      else
         SPARK_IO.Put_Integer (File,
                               ConvertMessageId (MessageId, ErrType), 3, 10);
         if CommandLineData.Content.Brief then
            SPARK_IO.Put_String (File, " - ", 0);
         else
            SPARK_IO.Put_String (File, ": ", 0);
         end if;
      end if;
   end PutMessageId;

   function GetErrorClass (ErrClass : in Error_Types.Error_Class) return EStrings.T
   is
      TmpString : EStrings.T;
   begin
      case ErrClass is
         when Error_Types.LexErr    =>
            EStrings.CopyString (TmpString, "Lexical Error");
         when Error_Types.SyntaxErr =>
            EStrings.CopyString (TmpString, "Syntax Error");
         when Error_Types.SyntaxRec =>
            EStrings.CopyString (TmpString, "Syntax Recovery");
         when Error_Types.SemanticErr  =>
            EStrings.CopyString (TmpString, "Semantic Error");
         when Error_Types.ControlFlowErr =>
            EStrings.CopyString (TmpString, "Illegal Structure");
         when Error_Types.UncondFlowErr    |
           Error_Types.UncondDependencyErr |
           Error_Types.IneffectiveStat     |
           Error_Types.StabilityErr        |
           Error_Types.UsageErr            |
           Error_Types.DepSemanticErr =>
            EStrings.CopyString (TmpString, "Flow Error");
         when Error_Types.CondlFlowErr     |
           Error_Types.CondlDependencyErr  |
           Error_Types.WarningWithPosition |
           Error_Types.WarningWithoutPosition =>
            EStrings.CopyString (TmpString, "Warning");
         when Error_Types.Note =>
            EStrings.CopyString (TmpString, "Note");
         when Error_Types.NoErr =>
            EStrings.CopyString (TmpString, "No Error");
      end case;

      return TmpString;
   end GetErrorClass;

   procedure OutputFullErrorName (File    : in SPARK_IO.File_Type;
                                  ErrType : in Error_Types.Error_Class)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                ErrType,
   --#                                File;
   is
   begin
      case ErrType is
         when Error_Types.LexErr    =>
            Put_String (File, "Lexical Error     :", 0);
         when Error_Types.SyntaxErr =>
            Put_String (File, "Syntax Error      :", 0);
         when Error_Types.SyntaxRec =>
            Put_String (File, "Syntax Recovery   :", 0);
         when Error_Types.SemanticErr  =>
            Put_String (File, "Semantic Error    :", 0);
         when Error_Types.ControlFlowErr =>
            Put_String (File, "Illegal Structure :", 0);
         when Error_Types.CondlFlowErr      |
           Error_Types.CondlDependencyErr   |
           Error_Types.UncondFlowErr        |
           Error_Types.UncondDependencyErr  |
           Error_Types.IneffectiveStat      |
           Error_Types.StabilityErr         |
           Error_Types.UsageErr             |
           Error_Types.DepSemanticErr =>
            Put_String (File, "Flow Error        :", 0);
         when Error_Types.WarningWithPosition |
           Error_Types.WarningWithoutPosition =>
            Put_String (File, "Warning           :", 0);
         when Error_Types.Note =>
            Put_String (File, "Note              :", 0);
         when Error_Types.NoErr =>
            null;
      end case;
   end OutputFullErrorName;

   procedure OutputBriefErrorName (File    : in SPARK_IO.File_Type;
                                   ErrType : in Error_Types.Error_Class)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                ErrType,
   --#                                File;
   is
   begin
      case ErrType is
         when Error_Types.LexErr    =>
            Put_String (File, "Lexical Error ", 0);
         when Error_Types.SyntaxErr =>
            Put_String (File, "Syntax Error ", 0);
         when Error_Types.SyntaxRec =>
            Put_String (File, "Syntax Recovery ", 0);
         when Error_Types.SemanticErr  =>
            Put_String (File, "Semantic Error ", 0);
         when Error_Types.ControlFlowErr =>
            Put_String (File, "Illegal Structure ", 0);
         when Error_Types.CondlFlowErr      |
           Error_Types.CondlDependencyErr   |
           Error_Types.UncondFlowErr        |
           Error_Types.UncondDependencyErr  |
           Error_Types.IneffectiveStat      |
           Error_Types.StabilityErr         |
           Error_Types.UsageErr             |
           Error_Types.DepSemanticErr =>
            Put_String (File, "Flow Error ", 0);
         when Error_Types.WarningWithPosition |
           Error_Types.WarningWithoutPosition =>
            Put_String (File, "Warning ", 0);
         when Error_Types.Note =>
            Put_String (File, "Note ", 0);
         when Error_Types.NoErr =>
            null;
      end case;
   end OutputBriefErrorName;

   procedure OutputErrorMarker (File      : in SPARK_IO.File_Type;
                                ErrType   : in Error_Types.Error_Class;
                                MessageId : in Natural;
                                ErrCount  : in Natural)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                ErrCount,
   --#                                ErrType,
   --#                                File,
   --#                                MessageId;
   is

      procedure OutputErrorFlash (File    : in SPARK_IO.File_Type;
                                  ErrType : in Error_Types.Error_Class)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                ErrType,
      --#                                File;
      is
      begin
         case ErrType is

            when Error_Types.LexErr               |
                 Error_Types.SyntaxErr            |
                 Error_Types.SyntaxRec            |
                 Error_Types.ControlFlowErr       |
                 Error_Types.SemanticErr =>
               Put_String (File, "***", 0);
            when Error_Types.UncondFlowErr        |
                 Error_Types.UncondDependencyErr  |
                 Error_Types.IneffectiveStat      |
                 Error_Types.StabilityErr         |
                 Error_Types.UsageErr             |
                 Error_Types.DepSemanticErr =>
               Put_String (File, "!!!", 0);
            when Error_Types.CondlFlowErr |
                 Error_Types.CondlDependencyErr =>
               Put_String (File, "???", 0);
            when Error_Types.NoErr =>
               Put_String (File, "+++        ", 0);
            when Error_Types.WarningWithPosition    |
                 Error_Types.WarningWithoutPosition |
                 Error_Types.Note =>
               Put_String (File, "---", 0);
         end case;
      end OutputErrorFlash;

      ----------------------------------------------------------------------

      procedure OutputErrorCount (File     : in SPARK_IO.File_Type;
                                  ErrCount : in Natural)
      --# global in     CommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                ErrCount,
      --#                                File;
      is
      begin
         if (ErrCount = 0) or CommandLineData.Content.PlainOutput then
            Put_String (File, "        ", 0);
         else
            Put_String (File, " (", 0);
            Put_Integer (File, ErrCount, 3, 10);
            Put_String (File, ")  ", 0);
         end if;
      end OutputErrorCount;

   begin --OutputErrorMarker
      OutputErrorFlash (File, ErrType);
      if ErrType /= Error_Types.NoErr then
         OutputErrorCount (File, ErrCount);
         OutputFullErrorName (File, ErrType);
         PutMessageId (File, MessageId, ErrType);
      else
         OutputFullErrorName (File, ErrType);
      end if;
   end OutputErrorMarker;

   procedure OutputBriefErrorMarker (File      : in SPARK_IO.File_Type;
                                     ErrType   : in Error_Types.Error_Class;
                                     MessageId : in Natural)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                ErrType,
   --#                                File,
   --#                                MessageId;
   is
   begin
      if ErrType /= Error_Types.NoErr then
         OutputBriefErrorName (File, ErrType);
         PutMessageId (File, MessageId, ErrType);
      end if;
   end OutputBriefErrorMarker;



   ---------------------------------------------------------------------------

   procedure EchoErrorEntry (EchoFile  : in SPARK_IO.File_Type;
                             Error     : in Error_Types.StringError)
      --# global in     CommandLineData.Content;
      --#        in out EchoAccumulator;
      --#        in out ErrorContextRec;
      --#        in out SPARK_IO.File_Sys;
      --# derives EchoAccumulator   from *,
      --#                                CommandLineData.Content,
      --#                                Error,
      --#                                ErrorContextRec &
      --#         ErrorContextRec,
      --#         SPARK_IO.File_Sys from CommandLineData.Content,
      --#                                EchoAccumulator,
      --#                                EchoFile,
      --#                                Error,
      --#                                ErrorContextRec,
      --#                                SPARK_IO.File_Sys;
      is separate;

   ---------------------------------------------------------------------------

   procedure ErrorInit (SourceFileName : in EStrings.T;
                        Echo           : in Boolean)
   --# global in     Dictionary.Dict;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --# derives EchoAccumulator,
   --#         ErrorBuffer.Buffer from * &
   --#         ErrorContextRec    from *,
   --#                                 Dictionary.Dict,
   --#                                 Echo,
   --#                                 ErrorBuffer.Buffer,
   --#                                 SourceFileName,
   --#                                 SPARK_IO.File_Sys &
   --#         SPARK_IO.File_Sys  from *,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 SourceFileName;
   is
      SourceFile  : SPARK_IO.File_Type := SPARK_IO.Null_File;
      ErrFile     : Error_IO.File_Type  := Error_IO.Null_File;
      Success     : SPARK_IO.File_Status;
      OKTempFile  : Boolean;
      NewContext  : ErrorContexts;

   begin
      Error_IO.Create (ErrFile, Success);
      OKTempFile := Success = SPARK_IO.Ok;
      if not OKTempFile then
         SystemErrors.FatalError (SystemErrors.ErrorHandlerTemporaryFiles, "in ErrorInit");
      end if;
      NewContext.Errs := ErrFile;
      NewContext.LineNo := 0;
      NewContext.SourceFileName := SourceFileName;
      NewContext.NumErrs := 0;
      NewContext.NumMessage := 0;
      NewContext.Severity := NoError;
      NewContext.RecoveryMessages := False;
      NewContext.Echo := Echo;
      NewContext.CurrentLine := EStrings.EmptyString;
      NewContext.Counter := Counters'(others => 0);
      NewContext.Source := SPARK_IO.Null_File;
      NewContext.JustificationsDataTable := EmptyJustificationsDataTable;
      --# accept Flow, 10, ErrorContextRec,  "Flush changes buffer but we need to initialize it anyway";
      FlushBuffer; -- Ineffective assignment to ErrorContextRec expected
      --# end accept;
      ErrorContextRec := NewContext;

      FileSystem.OpenSourceFile (SourceFile, SourceFileName, Success);

      ErrorContextRec.Source   := SourceFile;
      if not (Success = SPARK_IO.Ok) then
         SystemErrors.FatalError (SystemErrors.ErrorHandlerSource, "in ErrorInit");
      end if;
      FlushEchoMessages;
   end ErrorInit;

   ---------------------------------------------------------------------------

   procedure SparkMakeInit
   --# global in     Dictionary.Dict;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --# derives EchoAccumulator,
   --#         ErrorBuffer.Buffer from * &
   --#         ErrorContextRec    from SPARK_IO.File_Sys &
   --#         SPARK_IO.File_Sys  from *,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec;
   is
      ErrFile : Error_IO.File_Type := Error_IO.Null_File;
      Unused : SPARK_IO.File_Status;
      NewContext : ErrorContexts;
   begin
      --# accept Flow, 10, Unused, "Design decision not to check file status";
      Error_IO.Create (File   => ErrFile,  -- expect ineffective statement
                       Status => Unused);
      --# end accept;
      NewContext.Errs := ErrFile;
      NewContext.LineNo := 0;
      NewContext.NumErrs := 0;
      NewContext.NumMessage := 0;
      NewContext.Severity := NoError;
      NewContext.SourceFileName := EStrings.EmptyString;
      NewContext.RecoveryMessages := False;
      NewContext.Echo := False; --Echo;
      NewContext.CurrentLine := EStrings.EmptyString;
      NewContext.Counter := Counters'(others => 0);
      NewContext.Source := SPARK_IO.Null_File;
      NewContext.JustificationsDataTable := EmptyJustificationsDataTable;

      --# accept Flow, 10, ErrorContextRec, "Flush changes buffer but we need to initialize it anyway";
      FlushBuffer; -- Ineffective assignment to ErrorContextRec expected
      --# end accept;
      ErrorContextRec := NewContext;
      FlushEchoMessages;
      --# accept Flow, 33, Unused, "Consequence of earlier deliberate non-use";
   end SparkMakeInit;

   -----------------------------------------------------------------------------

   procedure GetErrorContext (Context : out ErrorContexts)
   --# global in     Dictionary.Dict;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --# derives Context,
   --#         ErrorContextRec    from ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 SPARK_IO.File_Sys &
   --#         EchoAccumulator,
   --#         ErrorBuffer.Buffer from * &
   --#         SPARK_IO.File_Sys  from *,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec;
   is
      SourceFile : SPARK_IO.File_Type;
      Success    : SPARK_IO.File_Status;
   begin
      FlushBuffer;
      SourceFile := ErrorContextRec.Source;
      --# accept Flow, 10, Success, "Intentional non use of file return code";
      SPARK_IO.Close (SourceFile, Success);
      --# end accept;
      ErrorContextRec.Source := SourceFile;
      Context := ErrorContextRec;
      FlushEchoMessages;
      --# accept Flow, 33, Success, "Consequence of earlier non-use";
   end GetErrorContext;

   -----------------------------------------------------------------------------

   procedure SetErrorContext (Context : in ErrorContexts)
   --# global in     Dictionary.Dict;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --# derives EchoAccumulator,
   --#         ErrorBuffer.Buffer from * &
   --#         ErrorContextRec    from *,
   --#                                 Context,
   --#                                 Dictionary.Dict,
   --#                                 ErrorBuffer.Buffer,
   --#                                 SPARK_IO.File_Sys &
   --#         SPARK_IO.File_Sys  from *,
   --#                                 Context,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec;
   is
      Success           : SPARK_IO.File_Status;
      SourceFile        : SPARK_IO.File_Type;

   begin
      --# accept Flow, 10, ErrorContextRec, "Flush changes buffer but we need to initialize it anyway";
      FlushBuffer; -- Ineffective assignment to ErrorContextRec expected
      --# end accept;
      ErrorContextRec := Context;
      SourceFile     := ErrorContextRec.Source;

      --# accept Flow, 10, Success, "Intentional non use of file return code";
      FileSystem.OpenSourceFile (SourceFile, ErrorContextRec.SourceFileName, Success);
      --# end accept;

      ErrorContextRec.Source := SourceFile;
      ErrorContextRec.LineNo := 0;
      ErrorContextRec.CurrentLine := EStrings.EmptyString;
      FlushEchoMessages;
      --# accept Flow, 33, Success, "Consequence of earlier non-use";
   end SetErrorContext;

   -----------------------------------------------------------------------------

   procedure ErrorReset
      --# global in out ErrorContextRec;
      --# derives ErrorContextRec from *;
   is
   begin
      ErrorContextRec.Severity := NoError;
   end ErrorReset;

   -----------------------------------------------------------------------------

   procedure GetErrorSeverity (Severity : out ErrorLevel)
      --# global in ErrorContextRec;
      --# derives Severity from ErrorContextRec;
   is
   begin
      Severity := ErrorContextRec.Severity;
   end GetErrorSeverity;

   -----------------------------------------------------------------------------

   procedure ReportSuccess (Position   : in LexTokenManager.TokenPosition;
                            SubprogStr : in LexTokenManager.LexString)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State,
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrorBuffer.Buffer,
   --#                                 Position,
   --#                                 SubprogStr &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 SPARK_IO.File_Sys,
   --#                                 SubprogStr &
   --#         TotalErrorCount    from *;
   is
      Error          : Error_Types.StringError;
      File           : Error_IO.File_Type;
   begin
      IncMessageCount (Error_Types.NoErr);
      File := ErrorContextRec.Errs;
      ErrorBuffer.Add (File,
                       Error_Types.NoErr,
                       Position,
                       Dictionary.GlobalScope, --not used
                       0,
                       NoReference,
                       LexStringToName (SubprogStr),
                       Error_Types.NoName,
                       Error_Types.NoName,
                        --to get
                       Error);
      ErrorContextRec.Errs := File;

      -- In "Brief" mode we don't echo success messages, since an
      -- IDE like GPS would interpret them as "errors", which they
      -- aren't!
      if not CommandLineData.Content.Brief then
         EchoErrorEntry (SPARK_IO.Standard_Output, Error);
      end if;
   end ReportSuccess;

   ---------------------------------------------------------------------------

   procedure IncSuppressedWarningCounter (WarningType : in WarningElements)
   --# global in out ErrorContextRec;
   --#        in out TotalErrorCount;
   --# derives ErrorContextRec from *,
   --#                              WarningType &
   --#         TotalErrorCount from *;
   is
   begin
      if ErrorContextRec.Counter (WarningType) = Count'Last then
         SystemErrors.FatalError (SystemErrors.TooManySuppressedWarnings, "");
      end if;
      ErrorContextRec.Counter (WarningType) :=
        ErrorContextRec.Counter (WarningType) + 1;
      IncTotalSuppressedWarnings; -- above is per file, per kind count; this one is grand total
   end IncSuppressedWarningCounter;

   ---------------------------------------------------------------------------

   procedure HiddenText (Position : in LexTokenManager.TokenPosition;
                         UnitStr  : in LexTokenManager.LexString;
                         UnitTyp  : in SPSymbols.SPSymbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     WarningStatus.SuppressionList;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State,
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 Position,
   --#                                 UnitStr,
   --#                                 UnitTyp,
   --#                                 WarningStatus.SuppressionList &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 SPARK_IO.File_Sys,
   --#                                 UnitStr,
   --#                                 UnitTyp,
   --#                                 WarningStatus.SuppressionList &
   --#         TotalErrorCount    from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrorContextRec,
   --#                                 Position,
   --#                                 WarningStatus.SuppressionList;
   is
      Error          : Error_Types.StringError;
      File           : Error_IO.File_Type;
      ErrorIsJustified : Boolean;
   begin
      Justifications.CheckWhetherJustified (ErrorContextRec.JustificationsDataTable,
                                            Position,
                                            WarningMessage,
                                            10,
                                            NullJustificationIdentifiers,
                                             -- to get
                                            ErrorIsJustified);
      if ErrorIsJustified then
         IncTotalJustifiedWarnings;
      else
         if WarningStatus.IsSuppressed (HiddenParts) then
            IncSuppressedWarningCounter (HiddenParts);
         else
            IncMessageCount (Error_Types.WarningWithoutPosition);
            File := ErrorContextRec.Errs;

            ErrorBuffer.Add (File,
                             Error_Types.WarningWithoutPosition,
                             Position,
                             Dictionary.GlobalScope,
                             10,   -- arbitrary error number 10 assigned to hidden text
                             NoReference,
                             LexStringToName (UnitStr),
                             SPSymbolToName (UnitTyp),
                             Error_Types.NoName,
                              --to get
                             Error);
            ErrorContextRec.Errs := File;
            EchoErrorEntry (SPARK_IO.Standard_Output, Error);
         end if;
      end if;
   end HiddenText;

   ---------------------------------------------------------------------------

   procedure HiddenHandler (Position : in LexTokenManager.TokenPosition;
                            UnitStr  : in LexTokenManager.LexString;
                            UnitTyp  : in SPSymbols.SPSymbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     WarningStatus.SuppressionList;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State,
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 Position,
   --#                                 UnitStr,
   --#                                 UnitTyp,
   --#                                 WarningStatus.SuppressionList &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 SPARK_IO.File_Sys,
   --#                                 UnitStr,
   --#                                 UnitTyp,
   --#                                 WarningStatus.SuppressionList &
   --#         TotalErrorCount    from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrorContextRec,
   --#                                 Position,
   --#                                 WarningStatus.SuppressionList;
   is
      Error          : Error_Types.StringError;
      File           : Error_IO.File_Type;
      ErrorIsJustified : Boolean;
   begin
      Justifications.CheckWhetherJustified (ErrorContextRec.JustificationsDataTable,
                                            Position,
                                            WarningMessage,
                                            9,
                                            NullJustificationIdentifiers,
                                             -- to get
                                            ErrorIsJustified);
      if ErrorIsJustified then
         IncTotalJustifiedWarnings;
      else
         if WarningStatus.IsSuppressed (HandlerParts) then
            IncSuppressedWarningCounter (HandlerParts);
         else
            IncMessageCount (Error_Types.WarningWithoutPosition);
            File := ErrorContextRec.Errs;

            ErrorBuffer.Add (File,
                             Error_Types.WarningWithoutPosition,
                             Position,
                             Dictionary.GlobalScope,
                             9,   -- arbitrary error number 9 assigned to hidden handlers
                             NoReference,
                             LexStringToName (UnitStr),
                             SPSymbolToName (UnitTyp),
                             Error_Types.NoName,
                              --to get
                             Error);
            ErrorContextRec.Errs := File;
            EchoErrorEntry (SPARK_IO.Standard_Output, Error);
         end if;
      end if;
   end HiddenHandler;

   ---------------------------------------------------------------------------

   procedure RepresentationClause (Position : in LexTokenManager.TokenPosition)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     WarningStatus.SuppressionList;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State,
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 Position,
   --#                                 WarningStatus.SuppressionList &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 SPARK_IO.File_Sys,
   --#                                 WarningStatus.SuppressionList &
   --#         TotalErrorCount    from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrorContextRec,
   --#                                 Position,
   --#                                 WarningStatus.SuppressionList;
   is
      Error : Error_Types.StringError;
      File  : Error_IO.File_Type;
      ErrorIsJustified : Boolean;
   begin
      Justifications.CheckWhetherJustified (ErrorContextRec.JustificationsDataTable,
                                            Position,
                                            WarningMessage,
                                            2,
                                            NullJustificationIdentifiers,
                                             -- to get
                                            ErrorIsJustified);
      if ErrorIsJustified then
         IncTotalJustifiedWarnings;
      else
         if WarningStatus.IsSuppressed (RepresentationClauses) then
            IncSuppressedWarningCounter (RepresentationClauses);
         else
            IncMessageCount (Error_Types.WarningWithPosition);
            File := ErrorContextRec.Errs;
            ErrorBuffer.Add (File,
                             Error_Types.WarningWithPosition,
                             Position,
                             Dictionary.GlobalScope,
                             2,
                             NoReference,
                             Error_Types.NoName,
                             Error_Types.NoName,
                             Error_Types.NoName,
                              --to get
                             Error);
            ErrorContextRec.Errs := File;
            EchoErrorEntry (SPARK_IO.Standard_Output, Error);
         end if;
      end if;
   end RepresentationClause;

   ---------------------------------------------------------------------------

   procedure APragma (PragmaName : in LexTokenManager.LexString;
                      Position : in LexTokenManager.TokenPosition)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     WarningStatus.SuppressionList;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State,
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 Position,
   --#                                 PragmaName,
   --#                                 WarningStatus.SuppressionList &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 PragmaName,
   --#                                 SPARK_IO.File_Sys,
   --#                                 WarningStatus.SuppressionList &
   --#         TotalErrorCount    from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrorContextRec,
   --#                                 Position,
   --#                                 PragmaName,
   --#                                 WarningStatus.SuppressionList;
   is
      Error : Error_Types.StringError;
      File  : Error_IO.File_Type;
      ErrorIsJustified : Boolean;
   begin
      Justifications.CheckWhetherJustified (ErrorContextRec.JustificationsDataTable,
                                            Position,
                                            WarningMessage,
                                            3,
                                            NullJustificationIdentifiers,
                                             -- to get
                                            ErrorIsJustified);
      if ErrorIsJustified then
         IncTotalJustifiedWarnings;
      else
         if WarningStatus.PragmaIsSuppressed (PragmaName) then
            IncSuppressedWarningCounter (Pragmas);
         else
            IncMessageCount (Error_Types.WarningWithPosition);
            File := ErrorContextRec.Errs;
            ErrorBuffer.Add (File,
                             Error_Types.WarningWithPosition,
                             Position,
                             Dictionary.GlobalScope,
                             3,
                             NoReference,
                             Error_Types.NoName,
                             Error_Types.NoName,
                             Error_Types.NoName,
                              --to get
                             Error);
            ErrorContextRec.Errs := File;
            EchoErrorEntry (SPARK_IO.Standard_Output, Error);
         end if;
      end if;
   end APragma;

   --------------------------------------------------------------------------

   procedure AddCutPoint (AtLine : LexTokenManager.LineNumbers)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     WarningStatus.SuppressionList;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State,
   --#         ErrorBuffer.Buffer from *,
   --#                                 AtLine,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrorBuffer.Buffer,
   --#                                 WarningStatus.SuppressionList &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from AtLine,
   --#                                 CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 LexTokenManager.StringTable,
   --#                                 SPARK_IO.File_Sys,
   --#                                 WarningStatus.SuppressionList &
   --#         TotalErrorCount    from *,
   --#                                 WarningStatus.SuppressionList;
   is
      Error : Error_Types.StringError;
      File  : Error_IO.File_Type;
   begin
      if WarningStatus.IsSuppressed (DefaultLoopAssertions) then
         IncSuppressedWarningCounter (DefaultLoopAssertions);
      else
         IncMessageCount (Error_Types.WarningWithoutPosition);
         File := ErrorContextRec.Errs;
         ErrorBuffer.Add (File,
                          Error_Types.WarningWithoutPosition,
                          LexTokenManager.TokenPosition'(AtLine, 0),
                          Dictionary.GlobalScope,
                          402,
                          NoReference,
                          Error_Types.NoName,
                          Error_Types.NoName,
                          Error_Types.NoName,
                           --to get
                          Error);
         ErrorContextRec.Errs := File;
         EchoErrorEntry (SPARK_IO.Standard_Output, Error);
      end if;
   end AddCutPoint;

   --------------------------------------------------------------------------

   -- Previously, LexErrors, SyntaxErrors and SyntaxRecovery didn't use
   -- the error buffer mechanism and were written direct to the error
   -- file.  This meant that attempts to NOT store strings anywhere in the
   -- error handler system were impossible.  The three routines have been
   -- altered below in an elegent bodge to use ErrorBuffer.Add.
   -- The trick has been to build the error string as now but instead of storing
   -- it, we convert it to a lex string, place it in parameter Name1 and then use
   -- the ErrorBuffer.Add mechanism.  This is not wasteful of LexTokenManager
   -- storage space because these errors are fatal anyway.
   --
   -- The result is that all errors are added to the system via ErrorBuffer.Add
   -- and remain in numeric form until PrintErrors or AppendErrors converst them.

   procedure LexError (ErrorMessage, RecoveryMessage : in String;
                       ErrorItem : in LexTokenManager.LexValue)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State,
   --#         ErrorBuffer.Buffer          from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorBuffer.Buffer,
   --#                                          ErrorItem,
   --#                                          ErrorMessage,
   --#                                          LexTokenManager.StringTable,
   --#                                          RecoveryMessage &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          Conversions.State,
   --#                                          Dictionary.Dict,
   --#                                          EchoAccumulator,
   --#                                          ErrorBuffer.Buffer,
   --#                                          ErrorContextRec,
   --#                                          ErrorItem,
   --#                                          ErrorMessage,
   --#                                          LexTokenManager.StringTable,
   --#                                          RecoveryMessage,
   --#                                          SPARK_IO.File_Sys &
   --#         LexTokenManager.StringTable from *,
   --#                                          ErrorItem,
   --#                                          ErrorMessage,
   --#                                          RecoveryMessage &
   --#         TotalErrorCount             from *;
   is
      Error : Error_Types.StringError;
      ErrorLexString : LexTokenManager.LexString; --pna
   begin
      ErrorContextRec.Severity := Fatal;
      IncMessageCount (Error_Types.LexErr);
      Error := Error_Types.StringError'(0, Error_Types.LexErr, ErrorItem.Position, ELStrings.EmptyString);

      AppendString (Error, ErrorMessage);
      AppendString (Error, " - ");
      AppendString (Error, RecoveryMessage);

      AppendString (Error, ".");
      -- convert string to LexString so we can use ErrorBuffer.Add
      LexTokenManager.InsertExaminerLongString (Error.Message,
                                                --to get
                                                ErrorLexString);
      ErrorBuffer.Add (ErrFile => ErrorContextRec.Errs,
                       ErrType => Error_Types.LexErr,
                       Pos => ErrorItem.Position,
                       Scope => Dictionary.GlobalScope,
                       ErrorNumber => 0,
                       Reference => 0,
                       Name1 => LexStringToName (ErrorLexString),
                       Name2 => Error_Types.NoName,
                       Name3 => Error_Types.NoName,
                        -- to get back the error we started with
                       EchoStr => Error);

      EchoErrorEntry (SPARK_IO.Standard_Output, Error);
   end LexError;

   --------------------------------------------------------------------------

   procedure SyntaxError (ErrorItem                       : in LexTokenManager.LexValue;
                          CurrentSym, EntrySymbol         : in SPSymbols.SPSymbol;
                          NoOfTerminals, NoOfNonTerminals : in SPExpectedSymbols.SPEssSymRange;
                          TerminalList, NonTerminalList   : in SPExpectedSymbols.SPExpSymList)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State,
   --#         ErrorBuffer.Buffer          from *,
   --#                                          CommandLineData.Content,
   --#                                          CurrentSym,
   --#                                          Dictionary.Dict,
   --#                                          EntrySymbol,
   --#                                          ErrorBuffer.Buffer,
   --#                                          ErrorItem,
   --#                                          LexTokenManager.StringTable,
   --#                                          NonTerminalList,
   --#                                          NoOfNonTerminals,
   --#                                          NoOfTerminals,
   --#                                          TerminalList &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          Conversions.State,
   --#                                          CurrentSym,
   --#                                          Dictionary.Dict,
   --#                                          EchoAccumulator,
   --#                                          EntrySymbol,
   --#                                          ErrorBuffer.Buffer,
   --#                                          ErrorContextRec,
   --#                                          ErrorItem,
   --#                                          LexTokenManager.StringTable,
   --#                                          NonTerminalList,
   --#                                          NoOfNonTerminals,
   --#                                          NoOfTerminals,
   --#                                          SPARK_IO.File_Sys,
   --#                                          TerminalList &
   --#         LexTokenManager.StringTable from *,
   --#                                          CurrentSym,
   --#                                          EntrySymbol,
   --#                                          ErrorItem,
   --#                                          NonTerminalList,
   --#                                          NoOfNonTerminals,
   --#                                          NoOfTerminals,
   --#                                          TerminalList &
   --#         TotalErrorCount             from *;
   is
      Error : Error_Types.StringError;
      ErrorLexString : LexTokenManager.LexString; --pna

      procedure AppendSymList (Error    : in out Error_Types.StringError;
                               NoInList : in     SPExpectedSymbols.SPEssSymRange;
                               List     : in     SPExpectedSymbols.SPExpSymList;
                               LastSep  : in     String)
         --# derives Error from *,
         --#                    LastSep,
         --#                    List,
         --#                    NoInList;
      is
      begin
         AppendSym (Error, List (1));
         for I in SPExpectedSymbols.SPEssSymRange range 2 .. NoInList - 1 loop
            AppendString (Error, ", ");
            AppendSym (Error, List (I));
         end loop;
         if NoInList > 1 then
            AppendString (Error, LastSep);
            AppendSym (Error, List (NoInList));
         end if;
      end AppendSymList;

      procedure AppendHint
      --# global in     CurrentSym;
      --#        in     EntrySymbol;
      --#        in     NonTerminalList;
      --#        in     NoOfNonTerminals;
      --#        in     NoOfTerminals;
      --#        in     TerminalList;
      --#        in out Error;
      --# derives Error from *,
      --#                    CurrentSym,
      --#                    EntrySymbol,
      --#                    NonTerminalList,
      --#                    NoOfNonTerminals,
      --#                    NoOfTerminals,
      --#                    TerminalList;
      is
         procedure DumpState
         --# derives ;
         is
            --# hide DumpState;
         begin
            if CommandLineData.Content.Debug.Parser then
               Debug.PrintMsg ("In ErrorHandler.SyntaxError.AppendHint we know:", True);
               Debug.PrintInt ("The Number of Terminals is ", Integer (NoOfTerminals));
               Debug.PrintMsg ("The Terminal List is:", True);

               for TLI in SPExpectedSymbols.SPEssSymRange range 1 .. NoOfTerminals loop
                  Debug.PrintMsg (SPSymbols.SPSymbol'Image (TerminalList (TLI)), False);
                  Debug.PrintMsg (" ", False);
               end loop;
               Debug.PrintMsg ("", True);

               Debug.PrintInt ("The Number of Non-Terminals is ", Integer (NoOfNonTerminals));
               Debug.PrintMsg ("The Non-Terminal List is:", True);

               for NTLI in SPExpectedSymbols.SPEssSymRange range 1 .. NoOfNonTerminals loop
                  Debug.PrintMsg (SPSymbols.SPSymbol'Image (NonTerminalList (NTLI)), False);
                  Debug.PrintMsg (" ", False);
               end loop;
               Debug.PrintMsg ("", True);

               Debug.PrintMsg ("The Entry Symbol:", False);
               Debug.PrintMsg (SPSymbols.SPSymbol'Image (EntrySymbol), True);

               Debug.PrintMsg ("The Current Symbol:", False);
               Debug.PrintMsg (SPSymbols.SPSymbol'Image (CurrentSym), True);

               Debug.PrintLexStr ("The current lex token is:", ErrorItem.TokenStr);
            end if;
         end DumpState;


      begin
         DumpState;

         ---------------------------------------------------------------------
         -- Hints for special cases
         --
         -- Here, we search for known special cases where we can give
         -- a more informative error message than that generated by the parser
         -- alone.  In detecting these special cases, take great care to find
         -- _exactly_ the case you're expecting, since issuing a hint which
         -- is wrong is probably just as bad as not issuing the hint at all!
         --
         -- This is a bit of a kludge - the long-term solution is to
         -- re-implement the parser generator to produce better errors in
         -- general.
         ---------------------------------------------------------------------

         if NoOfTerminals = 0 and NoOfNonTerminals = 1 then
            ---------------------------------------------------------------------
            -- Case 1 - A protected type definition where the user forgets to
            -- put a Priority pragma.  This is likely to be a common mistake for
            -- beginner RavenSPARK users, so giving a more informative error here
            -- is important.
            ---------------------------------------------------------------------
            if (NonTerminalList (1) = SPSymbols.protected_definition) and
              (CurrentSym = SPSymbols.RWprocedure or
                 CurrentSym = SPSymbols.RWentry or
                 CurrentSym = SPSymbols.RWfunction) then
               AppendString (Error, " Pragma Priority or Interrupt_Priority is required here.");

            ---------------------------------------------------------------------
            -- Case 2 - Nested procedure body has unexpected semicolon. Can be
            -- legal if followed by a pragma Import, but definitely wrong if
            -- followed by "is".
            ---------------------------------------------------------------------
            elsif (NonTerminalList (1) = SPSymbols.apragma) and
              (CurrentSym = SPSymbols.RWis) and
              (EntrySymbol = SPSymbols.procedure_annotation) then
               AppendString (Error, " Unexpected semicolon on preceding procedure specification.");

            ---------------------------------------------------------------------
            -- Case 11 - A comma has been used instead of a semicolon prior to
            -- an "IN" in a moded global definition.
            ---------------------------------------------------------------------
            elsif (NonTerminalList (1) = SPSymbols.global_variable) and
              (CurrentSym = SPSymbols.RWin) and
              (EntrySymbol = SPSymbols.comma) then
               AppendString (Error, " A semicolon and not a comma should precede an ""IN"" in a moded global definition.");
            ---------------------------------------------------------------------
            -- Case 15 - Empty parenthesis in a subprogram specification.
            ---------------------------------------------------------------------
            elsif (NonTerminalList (1) = SPSymbols.formal_part_rep) and
              (CurrentSym  = SPSymbols.right_paren) and
              (EntrySymbol = SPSymbols.left_paren) then
               AppendString (Error, " A subprogram specification without parameters should not have parentheses.");
            ---------------------------------------------------------------------
            -- Case 16 - Empty parenthesis in a subprogram call.
            ---------------------------------------------------------------------
            elsif (NonTerminalList (1) = SPSymbols.name_argument_list) and
              (CurrentSym  = SPSymbols.right_paren) and
              (EntrySymbol = SPSymbols.left_paren) then
               AppendString (Error, " A subprogram call without parameters should not have parentheses.");
            ---------------------------------------------------------------------
            -- Case 18 - Formal parameter list starts with an FDL identifier
            ---------------------------------------------------------------------
            elsif (NonTerminalList (1) = SPSymbols.formal_part_rep) and
              (CurrentSym  = SPSymbols.predefined_FDL_identifier) and
              (EntrySymbol = SPSymbols.left_paren) then

               AppendString (Error, " This token is a predefined FDL identifier, ");
               AppendString (Error, "and therefore may not be used in this context.");

            end if;

         elsif NoOfTerminals = 1 and NoOfNonTerminals = 0 then
            ---------------------------------------------------------------------
            -- Case 3 - misplaced "is" following procedure or function annotation
            ---------------------------------------------------------------------
            if TerminalList (1) = SPSymbols.RWinherit and
              (CurrentSym = SPSymbols.RWglobal or
               CurrentSym = SPSymbols.RWderives) then
               AppendString (Error, " Subprogram annotation should precede reserved word ""IS"".");
            ---------------------------------------------------------------------
            -- Case 4 - Expecting Identifier, but got a predefined FDL identifier
            -- A common error for beginners!
            ---------------------------------------------------------------------
            elsif TerminalList (1) = SPSymbols.identifier and
              CurrentSym = SPSymbols.predefined_FDL_identifier then
               AppendString (Error, " This token is a predefined FDL identifier, ");
               AppendString (Error, "and therefore may not be used in this context.");
            ---------------------------------------------------------------------
            -- Case 9 - "own" annotation in package body following "is"
            ---------------------------------------------------------------------
            elsif TerminalList (1) = SPSymbols.RWinherit and
              CurrentSym = SPSymbols.RWown then
               AppendString (Error, " Own annotation should precede reserved word ""IS"".");
            ---------------------------------------------------------------------
            -- Case 14 - Incorrect name given in hide directive
            ---------------------------------------------------------------------
            elsif TerminalList (1) = SPSymbols.semicolon and
              EntrySymbol = SPSymbols.hidden_part then
               AppendString (Error, " The name in a preceding ""hide"" directive is probably incorrect.");
            end if;

         elsif NoOfTerminals = 0 and NoOfNonTerminals = 0 then
            ---------------------------------------------------------------------
            -- Case 5 - Misplaced "is" _before_ a mandatory annotation
            ---------------------------------------------------------------------
            if CurrentSym = SPSymbols.annotation_start and
              EntrySymbol = SPSymbols.RWis then
               AppendString (Error, " For packages and subprograms, annotations must ");
               AppendString (Error, "precede ""IS"".");

            ---------------------------------------------------------------------
            -- Case 6 - semicolon missing at end of annotation
            ---------------------------------------------------------------------
            elsif CurrentSym = SPSymbols.annotation_end and
              EntrySymbol = SPSymbols.dotted_simple_name then
               AppendString (Error, " Annotations must end with a ;");

            ---------------------------------------------------------------------
            -- Case 10 - unexpected semicolon at end of subprogram specification
            ---------------------------------------------------------------------
            elsif CurrentSym = SPSymbols.RWis and
              EntrySymbol = SPSymbols.semicolon then
               AppendString (Error, " Unexpected semicolon on preceding subprogram specification.");

            ---------------------------------------------------------------------
            -- Case 12 - reported message is "No complete DOTTED_SIMPLE_NAME can be followed by IDENTIFIER here,"
            -- Rockwell-Collins raised candidate for improvement in relation to
            -- imports in derives annotations but the possible occurance of the
            -- syntax error is much wider than this so we have to be careful with
            -- the wording of the eror message.
            ---------------------------------------------------------------------
            elsif CurrentSym = SPSymbols.identifier and
              EntrySymbol = SPSymbols.dotted_simple_name then
               AppendString (Error, " A separator is missing between two succesive names.");

            ---------------------------------------------------------------------
            -- Case 17 - declaration of an object using an FDL identifier
            ---------------------------------------------------------------------
            elsif CurrentSym = SPSymbols.predefined_FDL_identifier and
              (EntrySymbol = SPSymbols.visible_part_rep or
                 EntrySymbol = SPSymbols.initial_declarative_item_rep) then

               AppendString (Error, " This token is a predefined FDL identifier, ");
               AppendString (Error, "and therefore may not be used in this context.");

            end if;

         elsif NoOfTerminals = 4 and NoOfNonTerminals = 0 then
            ---------------------------------------------------------------------
            -- Case 7 - missing ; on the end of an own variable annotation
            ---------------------------------------------------------------------
            if EntrySymbol = SPSymbols.own_variable_list and
              CurrentSym = SPSymbols.annotation_end and
              TerminalList (1) = SPSymbols.left_paren and
              TerminalList (2) = SPSymbols.comma and
              TerminalList (3) = SPSymbols.colon and
              TerminalList (4) = SPSymbols.semicolon then

               AppendString (Error, " Own variable annotation must end with a ;");
            end if;


         elsif NoOfTerminals = 2 and NoOfNonTerminals = 0 then
            ---------------------------------------------------------------------
            -- Case 8 - missing ; on the end of an initializes annotation
            ---------------------------------------------------------------------
            if EntrySymbol = SPSymbols.package_variable_list and
              CurrentSym = SPSymbols.annotation_end and
              TerminalList (1) = SPSymbols.comma and
              TerminalList (2) = SPSymbols.semicolon then

               AppendString (Error, " Initializes annotation must end with a ;");
            end if;

            ---------------------------------------------------------------------
            -- Case 13 - derives list applied to a function
            -- From the SPARK.LLA grammar, The two expected terminal symbols
            -- RWpre and RWreturn can only occur together in an annotation
            -- as a function constraint so we should be safe to assume that we
            -- are in an annotation associated with a function.
            ---------------------------------------------------------------------
            if EntrySymbol = SPSymbols.annotation_start and
              CurrentSym = SPSymbols.RWderives and
              TerminalList (1) = SPSymbols.RWpre and
              TerminalList (2) = SPSymbols.RWreturn then

               AppendString (Error, " A function never has a ""derives"" annotation." &
                                    " It may have a global definition in which all the global variables are of mode in.");
            end if;

         end if;

      end AppendHint; -- TerminalList not ref at the moment, but likely to be needed in future.

   begin
      ErrorContextRec.Severity := Fatal;
      IncMessageCount (Error_Types.SyntaxErr);
      Error := Error_Types.StringError'(0, Error_Types.SyntaxErr, ErrorItem.Position, ELStrings.EmptyString);

      if NoOfTerminals /= 0 and NoOfNonTerminals = 0 then
         AppendSymList (Error, NoOfTerminals, TerminalList, " or ");
         AppendString (Error, " expected");
      elsif NoOfTerminals = 0 and NoOfNonTerminals = 1 then
         AppendString (Error, "No ");
         AppendSymList (Error, NoOfNonTerminals, NonTerminalList, " nor ");
         AppendString (Error, " can start with ");
         AppendSym (Error, CurrentSym);

      elsif NoOfTerminals = 0 and NoOfNonTerminals > 1 then
         AppendString (Error, "Neither ");
         AppendSymList (Error, NoOfNonTerminals, NonTerminalList, " nor ");
         AppendString (Error, " can start with ");
         AppendSym (Error, CurrentSym);
      elsif NoOfTerminals /= 0 and NoOfNonTerminals /= 0 then
         AppendSymList (Error, NoOfTerminals, TerminalList, ", ");
         AppendString (Error, " or start of ");
         AppendSymList (Error, NoOfNonTerminals, NonTerminalList, " or ");
         AppendString (Error, " expected");
      else
         if SPRelations.SPTerminalLike (EntrySymbol) then
            AppendSym (Error, EntrySymbol);
            AppendString (Error, " cannot be followed by ");
         else
            AppendString (Error, "No complete ");
            AppendSym (Error, EntrySymbol);
            AppendString (Error, " can be followed by ");
         end if;
         AppendSym (Error, CurrentSym);
         AppendString (Error, " here");
      end if;
      AppendString (Error, ".");

      AppendHint;

      -- convert string to LExString so we can use ErrorBuffer.Add
      LexTokenManager.InsertExaminerLongString (Error.Message,
                                                --to get
                                                ErrorLexString);
      ErrorBuffer.Add (ErrFile => ErrorContextRec.Errs,
                       ErrType => Error_Types.SyntaxErr,
                       Pos => ErrorItem.Position,
                       Scope => Dictionary.GlobalScope,
                       ErrorNumber => 0,
                       Reference => 0,
                       Name1 => LexStringToName (ErrorLexString),
                       Name2 => Error_Types.NoName,
                       Name3 => Error_Types.NoName,
                        -- to get
                       EchoStr => Error);
      EchoErrorEntry (SPARK_IO.Standard_Output, Error);
   end SyntaxError;

   ---------------------------------------------------------------------------

   procedure SyntaxRecovery (RecoveryPosn     : in LexTokenManager.LexValue;
                             ReplacementSym   : in SPSymbols.SPSymbol;
                             NextSym          : in SPSymbols.SPTerminal;
                             NoOfSyms         : in Natural;
                             SymList          : in ErrSymList)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out Conversions.State;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State,
   --#         ErrorBuffer.Buffer,
   --#         SPARK_IO.File_Sys           from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorBuffer.Buffer,
   --#                                          ErrorContextRec,
   --#                                          LexTokenManager.StringTable,
   --#                                          NextSym,
   --#                                          NoOfSyms,
   --#                                          RecoveryPosn,
   --#                                          ReplacementSym,
   --#                                          SymList &
   --#         ErrorContextRec             from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorBuffer.Buffer,
   --#                                          LexTokenManager.StringTable,
   --#                                          NextSym,
   --#                                          NoOfSyms,
   --#                                          RecoveryPosn,
   --#                                          ReplacementSym,
   --#                                          SPARK_IO.File_Sys,
   --#                                          SymList &
   --#         LexTokenManager.StringTable from *,
   --#                                          ErrorContextRec,
   --#                                          NextSym,
   --#                                          NoOfSyms,
   --#                                          RecoveryPosn,
   --#                                          ReplacementSym,
   --#                                          SymList &
   --#         TotalErrorCount             from *,
   --#                                          ErrorContextRec;
   is
      Error,
      UnusedError    : Error_Types.StringError;
      ErrorLexString : LexTokenManager.LexString;
      MaxIndex       : ErrSymRange;
   begin
      if ErrorContextRec.RecoveryMessages then
         IncMessageCount (Error_Types.SyntaxRec);
         Error := Error_Types.StringError'(0, Error_Types.SyntaxRec, RecoveryPosn.Position,
                              ELStrings.EmptyString);
         AppendString (Error, "The recovery action was to ");
         if ReplacementSym = SPSymbols.SPDEFAULT then
            AppendString (Error, "delete ");
            AppendSym (Error, SymList (1));
         elsif ReplacementSym in SPSymbols.SPTerminal then
            if NoOfSyms = 0 then
               AppendString (Error, "insert ");
            else
               AppendString (Error, "replace ");
               AppendSym (Error, SymList (1));
               AppendString (Error, " with ");
            end if;
            AppendSym (Error, ReplacementSym);
         else
            if NextSym /= SPSymbols.SPDEFAULT then
               if NextSym = SPSymbols.SPEND then
                  AppendString (Error, "ignore all remaining tokens and ");
               else
                  AppendString (Error, "ignore all tokens up until ");
                  AppendSym (Error, NextSym);
                  AppendString (Error, " and ");
               end if;
            end if;
            if NoOfSyms /= 0 then
               AppendString (Error, "replace ");
               if NoOfSyms > Natural (ErrSymList'Last) then
                  MaxIndex := ErrSymList'Last - 1;
               else
                  MaxIndex := ErrSymRange (NoOfSyms);
               end if;
               for Index in ErrSymRange range 1 .. MaxIndex loop
                  AppendSym (Error, SymList (Index));
                  AppendString (Error, " ");
               end loop;
               if NoOfSyms > Natural (ErrSymList'Last) then
                  AppendString (Error, " .. ");
                  AppendSym (Error, SymList (ErrSymList'Last));
               end if;
               AppendString (Error, " by ");
            else
               AppendString (Error, "insert ");
            end if;
            AppendSym (Error, ReplacementSym);
         end if;

         AppendString (Error, ".");
         -- convert string to LexString so we can use ErrorBuffer.Add
         LexTokenManager.InsertExaminerLongString (Error.Message,
                                                   --to get
                                                   ErrorLexString);
         --# accept Flow, 10, UnusedError, "Only used where we want to echo message to screen";
         ErrorBuffer.Add (ErrFile => ErrorContextRec.Errs,
                          ErrType => Error_Types.SyntaxRec,
                          Pos => RecoveryPosn.Position,
                          Scope => Dictionary.GlobalScope,
                          ErrorNumber => 0,
                          Reference => 0,
                          Name1 => LexStringToName (ErrorLexString),
                          Name2 => Error_Types.NoName,
                          Name3 => Error_Types.NoName,
                           -- to get back the error we started with
                          EchoStr => UnusedError);
         --# end accept;
         -- note, no echo back to screen in this case
      end if;
      --# accept Flow, 33, UnusedError, "Consequence of earlier deliberate non-use";
   end SyntaxRecovery;

   -------------------------------------------------------------------------

   procedure SemanticWarningWithPosition (ErrNum   : in Natural;
                                          Position : in LexTokenManager.TokenPosition;
                                          IdStr    : in LexTokenManager.LexString;
                                          Sym      : in Dictionary.Symbol;
                                          Scope    : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     WarningStatus.SuppressionList;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State,
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 IdStr,
   --#                                 Position,
   --#                                 Scope,
   --#                                 Sym,
   --#                                 WarningStatus.SuppressionList &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 IdStr,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 Scope,
   --#                                 SPARK_IO.File_Sys,
   --#                                 Sym,
   --#                                 WarningStatus.SuppressionList &
   --#         TotalErrorCount    from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrNum,
   --#                                 WarningStatus.SuppressionList;
   is
      Error : Error_Types.StringError;
      File  : Error_IO.File_Type;
   begin
      if ErrNum = 1 and then
         WarningStatus.IsSuppressed (WithClauses)
      then
         IncSuppressedWarningCounter (WithClauses);

      elsif ErrNum = 4 and then
         WarningStatus.IsSuppressed (DeclareAnnotations)
      then
         IncSuppressedWarningCounter (DeclareAnnotations);

      elsif ErrNum = 5 and then
         WarningStatus.IsSuppressed (InterruptHandlers)
      then
         IncSuppressedWarningCounter (InterruptHandlers);

      elsif ErrNum = 7 and then
         WarningStatus.IsSuppressed (Ada2005ReservedWords)
      then
         IncSuppressedWarningCounter (Ada2005ReservedWords);

      elsif ErrNum = 11 and then
         WarningStatus.IsSuppressed (OthersClauses)
      then
         IncSuppressedWarningCounter (OthersClauses);

      elsif ErrNum = 12 and then
         WarningStatus.IsSuppressed (UncheckedConversion)
      then
         IncSuppressedWarningCounter (UncheckedConversion);

      elsif ErrNum = 169 and then
         WarningStatus.IsSuppressed (DirectUpdates)
      then
         IncSuppressedWarningCounter (DirectUpdates);

      elsif (ErrNum = 200 or ErrNum = 201) and then
         WarningStatus.IsSuppressed (StaticExpressions)
      then
         IncSuppressedWarningCounter (StaticExpressions);

      elsif ErrNum = 302 and then
        not CommandLineData.Content.RedType.ExpChecks and then
         WarningStatus.IsSuppressed (ExpressionReordering) then
         -- Suppress ambiguous ordering message if not
         -- generating VCs for Overflow_Check
         IncSuppressedWarningCounter (ExpressionReordering);

      elsif ErrNum = 309 and then
         WarningStatus.IsSuppressed (TypeConversions)
      then
         IncSuppressedWarningCounter (TypeConversions);

      elsif ErrNum = 310 and then
         WarningStatus.IsSuppressed (ObsolescentFeatures)
      then
         IncSuppressedWarningCounter (ObsolescentFeatures);

      elsif ErrNum = 350 and then
         WarningStatus.IsSuppressed (ImportedObjects)
      then
         IncSuppressedWarningCounter (ImportedObjects);

      elsif ErrNum = 392 and then
         WarningStatus.IsSuppressed (ExternalVariableAssignment)
      then
         IncSuppressedWarningCounter (ExternalVariableAssignment);

      elsif ErrNum = 394 and then
         WarningStatus.IsSuppressed (UnuseablePrivateTypes)
      then
         IncSuppressedWarningCounter (UnuseablePrivateTypes);

         -- following two separate warnings share the same supression key word
      elsif ErrNum = 396 and then -- non-moded variable with address clause
         WarningStatus.IsSuppressed (UnexpectedAddressClauses)
      then
         IncSuppressedWarningCounter (UnexpectedAddressClauses);

      elsif ErrNum = 351 and then -- constant with address clause
         WarningStatus.IsSuppressed (UnexpectedAddressClauses)
      then
         IncSuppressedWarningCounter (UnexpectedAddressClauses);

      else
         if ErrNum = 399 then
            --special case, although just a warning we want this to suppress flow analysis
            ErrorContextRec.Severity := SemanticErrs;
         end if;

         IncMessageCount (Error_Types.WarningWithPosition);
         File := ErrorContextRec.Errs;
         if Sym = Dictionary.NullSymbol then
            ErrorBuffer.Add (File,
                             Error_Types.WarningWithPosition,
                             Position,
                             Dictionary.GlobalScope,
                             ErrNum,
                             NoReference,
                             LexStringToName (IdStr),
                             Error_Types.NoName,
                             Error_Types.NoName,
                              --to get
                             Error);
         else
            ErrorBuffer.Add (File,
                             Error_Types.WarningWithPosition,
                             Position,
                             Scope,
                             ErrNum,
                             NoReference,
                             SymbolToName (Sym),
                             Error_Types.NoName,
                             Error_Types.NoName,
                              --to get
                             Error);
         end if;
         ErrorContextRec.Errs := File;
         EchoErrorEntry (SPARK_IO.Standard_Output, Error);
      end if;
   end SemanticWarningWithPosition;

   procedure SemanticWarningWithoutPosition (ErrNum   : in Natural;
                                             Position : in LexTokenManager.TokenPosition;
                                             IdStr    : in LexTokenManager.LexString;
                                             Sym      : in Dictionary.Symbol;
                                             Scope    : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     WarningStatus.SuppressionList;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State,
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 IdStr,
   --#                                 Position,
   --#                                 Scope,
   --#                                 Sym,
   --#                                 WarningStatus.SuppressionList &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 IdStr,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 Scope,
   --#                                 SPARK_IO.File_Sys,
   --#                                 Sym,
   --#                                 WarningStatus.SuppressionList &
   --#         TotalErrorCount    from *,
   --#                                 ErrNum,
   --#                                 WarningStatus.SuppressionList;
   is
      Error : Error_Types.StringError;
      File  : Error_IO.File_Type;
   begin
      if ErrNum = 400 and then
         WarningStatus.IsSuppressed (UnusedVariables)
      then
         IncSuppressedWarningCounter (UnusedVariables);
      elsif ErrNum = 403 and then
         WarningStatus.IsSuppressed (ConstantVariables)
      then
         IncSuppressedWarningCounter (ConstantVariables);
      elsif ErrNum = 405 and then
         WarningStatus.IsSuppressed (RealRTCs)
      then
         IncSuppressedWarningCounter (RealRTCs);
      else
         IncMessageCount (Error_Types.WarningWithoutPosition);
         File := ErrorContextRec.Errs;
         if Sym = Dictionary.NullSymbol then
            ErrorBuffer.Add (File,
                             Error_Types.WarningWithoutPosition,
                             Position,
                             Dictionary.GlobalScope,
                             ErrNum,
                             NoReference,
                             LexStringToName (IdStr),
                             Error_Types.NoName,
                             Error_Types.NoName,
                              --to get
                             Error);
         else
            ErrorBuffer.Add (File,
                             Error_Types.WarningWithoutPosition,
                             Position,
                             Scope,
                             ErrNum,
                             NoReference,
                             SymbolToName (Sym),
                             Error_Types.NoName,
                             Error_Types.NoName,
                              --to get
                             Error);
         end if;
         ErrorContextRec.Errs := File;
         EchoErrorEntry (SPARK_IO.Standard_Output, Error);
      end if;
   end SemanticWarningWithoutPosition;


   function IsNumberWithPosition (ErrNum : in Natural) return Boolean is
   begin
      return ErrNum >= 1 and then ErrNum < 400;
   end IsNumberWithPosition;


   procedure SemanticWarning (ErrNum : in Natural;
                              Position : in LexTokenManager.TokenPosition;
                              IdStr : in LexTokenManager.LexString)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     WarningStatus.SuppressionList;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State,
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 IdStr,
   --#                                 Position,
   --#                                 WarningStatus.SuppressionList &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 IdStr,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 SPARK_IO.File_Sys,
   --#                                 WarningStatus.SuppressionList &
   --#         TotalErrorCount    from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrNum,
   --#                                 ErrorContextRec,
   --#                                 IdStr,
   --#                                 Position,
   --#                                 WarningStatus.SuppressionList;
   is
      ErrorIsJustified : Boolean;
   begin
      Justifications.CheckWhetherJustified (ErrorContextRec.JustificationsDataTable,
                                            Position,
                                            WarningMessage,
                                            ErrNum,
                                            JustificationIdentifiers'
                                              (1 => LexStrToJustificationIdentifier (IdStr),
                                               others => NullJustificationIdentifier),
                                             -- to get
                                            ErrorIsJustified);
      if ErrorIsJustified then
         IncTotalJustifiedWarnings;
      else
         if IsNumberWithPosition (ErrNum) then
            SemanticWarningWithPosition (ErrNum,
                                         Position,
                                         IdStr,
                                         Dictionary.NullSymbol,
                                         Dictionary.GlobalScope);
         else
            SemanticWarningWithoutPosition (ErrNum,
                                            Position,
                                            IdStr,
                                            Dictionary.NullSymbol,
                                            Dictionary.GlobalScope);
         end if;
      end if;
   end SemanticWarning;

   ----------------------------------------------------------------------------------

   procedure SemanticWarningSym (ErrNum   : in Natural;
                                 Position : in LexTokenManager.TokenPosition;
                                 Sym      : in Dictionary.Symbol;
                                 Scope    : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     WarningStatus.SuppressionList;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State,
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 Position,
   --#                                 Scope,
   --#                                 Sym,
   --#                                 WarningStatus.SuppressionList &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 Scope,
   --#                                 SPARK_IO.File_Sys,
   --#                                 Sym,
   --#                                 WarningStatus.SuppressionList &
   --#         TotalErrorCount    from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrNum,
   --#                                 ErrorContextRec,
   --#                                 Position,
   --#                                 Sym,
   --#                                 WarningStatus.SuppressionList;
   is
      ErrorIsJustified : Boolean;
   begin
      Justifications.CheckWhetherJustified (ErrorContextRec.JustificationsDataTable,
                                            Position,
                                            WarningMessage,
                                            ErrNum,
                                            JustificationIdentifiers'
                                              (1 => SymbolToJustificationIdentifier (Sym),
                                               others => NullJustificationIdentifier),
                                             -- to get
                                            ErrorIsJustified);
      if ErrorIsJustified then
         IncTotalJustifiedWarnings;
      else
         if IsNumberWithPosition (ErrNum) then
            SemanticWarningWithPosition (ErrNum,
                                         Position,
                                         LexTokenManager.NullString,
                                         Sym,
                                         Scope);
         else
            SemanticWarningWithoutPosition (ErrNum,
                                            Position,
                                            LexTokenManager.NullString,
                                            Sym,
                                            Scope);
         end if;
      end if;
   end SemanticWarningSym;

   ----------------------------------------------------------------------------------

   procedure SemanticNote (ErrNum : in Natural;
                           Position : in LexTokenManager.TokenPosition;
                           IdStr : in LexTokenManager.LexString)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     WarningStatus.SuppressionList;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State,
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 IdStr,
   --#                                 Position,
   --#                                 WarningStatus.SuppressionList &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 IdStr,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 SPARK_IO.File_Sys,
   --#                                 WarningStatus.SuppressionList &
   --#         TotalErrorCount    from *,
   --#                                 WarningStatus.SuppressionList;
   is
      File  : Error_IO.File_Type;
      Error : Error_Types.StringError;
   begin --SemanticNote
      if WarningStatus.IsSuppressed (Notes) then
         IncSuppressedWarningCounter (Notes);
      else
         IncMessageCount (Error_Types.Note);
         File := ErrorContextRec.Errs;
         ErrorBuffer.Add (File,
                          Error_Types.Note,
                          Position,
                          Dictionary.GlobalScope,
                          ErrNum,
                          NoReference,
                          LexStringToName (IdStr),
                          Error_Types.NoName,
                          Error_Types.NoName,
                           --to get
                          Error);
         ErrorContextRec.Errs := File;
         EchoErrorEntry (SPARK_IO.Standard_Output, Error);
      end if;
   end SemanticNote;

   --------------------------------------------------------------------------

   procedure DepSemanticError (ErrNum   : in Natural;
                               Position : in LexTokenManager.TokenPosition;
                               IdStr1   : in LexTokenManager.LexString;
                               IdStr2   : in LexTokenManager.LexString)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State,
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 IdStr1,
   --#                                 IdStr2,
   --#                                 Position &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 IdStr1,
   --#                                 IdStr2,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 SPARK_IO.File_Sys &
   --#         TotalErrorCount    from *;
   is
      Error : Error_Types.StringError;
      File  : Error_IO.File_Type;
   begin
      IncMessageCount (Error_Types.DepSemanticErr);
      File := ErrorContextRec.Errs;
      ErrorBuffer.Add (File,
                       Error_Types.DepSemanticErr,
                       Position,
                       Dictionary.GlobalScope,
                       ErrNum,
                       NoReference,
                       LexStringToName (IdStr1),
                       LexStringToName (IdStr2),
                       Error_Types.NoName,
                        --to get
                       Error);
      ErrorContextRec.Errs := File;
      EchoErrorEntry (SPARK_IO.Standard_Output, Error);
   end DepSemanticError;

   ---------------------------------------------------------------------------

   procedure DepSemanticErrorSym (ErrNum   : in Natural;
                                  Position : in LexTokenManager.TokenPosition;
                                  Sym1     : in Dictionary.Symbol;
                                  Sym2     : in Dictionary.Symbol;
                                  Scope    : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State  from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 Position,
   --#                                 Scope,
   --#                                 Sym1,
   --#                                 Sym2 &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 Scope,
   --#                                 SPARK_IO.File_Sys,
   --#                                 Sym1,
   --#                                 Sym2 &
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrNum,
   --#                                 Position,
   --#                                 Scope,
   --#                                 Sym1,
   --#                                 Sym2 &
   --#         TotalErrorCount    from *;
   is
      Error : Error_Types.StringError;
      File  : Error_IO.File_Type;
   begin
      IncMessageCount (Error_Types.DepSemanticErr);
      File := ErrorContextRec.Errs;
      ErrorBuffer.Add (File,
                       Error_Types.DepSemanticErr,
                       Position,
                       Scope,
                       ErrNum,
                       NoReference,
                       SymbolToName (Sym1),
                       SymbolToName (Sym2),
                       Error_Types.NoName,
                        --to get
                       Error);
      ErrorContextRec.Errs := File;
      EchoErrorEntry (SPARK_IO.Standard_Output, Error);
   end DepSemanticErrorSym;

   ---------------------------------------------------------------------------

   procedure SemanticError (ErrNum    : in Natural;
                            Reference : in Natural;
                            Position  : in LexTokenManager.TokenPosition;
                            IdStr     : in LexTokenManager.LexString)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State,
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 IdStr,
   --#                                 Position,
   --#                                 Reference &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 IdStr,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 Reference,
   --#                                 SPARK_IO.File_Sys &
   --#         TotalErrorCount    from *;
   is
      Error : Error_Types.StringError;
      File  : Error_IO.File_Type;
   begin
      ErrorContextRec.Severity := SemanticErrs;
      Justifications.SetCurrentUnitHasSemanticErrors (ErrorContextRec.JustificationsDataTable);
      IncMessageCount (Error_Types.SemanticErr);
      File := ErrorContextRec.Errs;
      ErrorBuffer.Add (File,
                       Error_Types.SemanticErr,
                       Position,
                       Dictionary.GlobalScope,
                       ErrNum,
                       Reference,
                       LexStringToName (IdStr),
                       Error_Types.NoName,
                       Error_Types.NoName,
                        --to get
                       Error);
      ErrorContextRec.Errs := File;
      EchoErrorEntry (SPARK_IO.Standard_Output, Error);
   end SemanticError;

   ---------------------------------------------------------------------------

   procedure SemanticError2 (ErrNum    : in Natural;
                             Reference : in Natural;
                             Position  : in LexTokenManager.TokenPosition;
                             IdStr1    : in LexTokenManager.LexString;
                             IdStr2    : in LexTokenManager.LexString)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State,
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 IdStr1,
   --#                                 IdStr2,
   --#                                 Position,
   --#                                 Reference &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 IdStr1,
   --#                                 IdStr2,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 Reference,
   --#                                 SPARK_IO.File_Sys &
   --#         TotalErrorCount    from *;
   is
      Error : Error_Types.StringError;
      File  : Error_IO.File_Type;
   begin
      Justifications.SetCurrentUnitHasSemanticErrors (ErrorContextRec.JustificationsDataTable);
      ErrorContextRec.Severity := SemanticErrs;
      IncMessageCount (Error_Types.SemanticErr);
      File := ErrorContextRec.Errs;
      ErrorBuffer.Add (File,
                       Error_Types.SemanticErr,
                       Position,
                       Dictionary.GlobalScope,
                       ErrNum,
                       Reference,
                       LexStringToName (IdStr1),
                       LexStringToName (IdStr2),
                       Error_Types.NoName,
                        --to get
                       Error);
      ErrorContextRec.Errs := File;
      EchoErrorEntry (SPARK_IO.Standard_Output, Error);
   end SemanticError2;

   ---------------------------------------------------------------------------

   procedure SemanticErrorSym (ErrNum    : in Natural;
                               Reference : in Natural;
                               Position  : in LexTokenManager.TokenPosition;
                               Sym       : in Dictionary.Symbol;
                               Scope     : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State  from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 Position,
   --#                                 Reference,
   --#                                 Scope,
   --#                                 Sym &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 Reference,
   --#                                 Scope,
   --#                                 SPARK_IO.File_Sys,
   --#                                 Sym &
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrNum,
   --#                                 Position,
   --#                                 Reference,
   --#                                 Scope,
   --#                                 Sym &
   --#         TotalErrorCount    from *;
   is
      Error : Error_Types.StringError;
      File  : Error_IO.File_Type;
   begin
      Justifications.SetCurrentUnitHasSemanticErrors (ErrorContextRec.JustificationsDataTable);
      ErrorContextRec.Severity := SemanticErrs;
      IncMessageCount (Error_Types.SemanticErr);
      File := ErrorContextRec.Errs;
      ErrorBuffer.Add (File,
                       Error_Types.SemanticErr,
                       Position,
                       Scope,
                       ErrNum,
                       Reference,
                       SymbolToName (Sym),
                       Error_Types.NoName,
                       Error_Types.NoName,
                        --to get
                       Error);
      ErrorContextRec.Errs := File;
      EchoErrorEntry (SPARK_IO.Standard_Output, Error);
   end SemanticErrorSym;

   ---------------------------------------------------------------------------

   procedure SemanticErrorSym2 (ErrNum    : in Natural;
                                Reference : in Natural;
                                Position  : in LexTokenManager.TokenPosition;
                                Sym       : in Dictionary.Symbol;
                                Sym2      : in Dictionary.Symbol;
                                Scope     : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State  from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 Position,
   --#                                 Reference,
   --#                                 Scope,
   --#                                 Sym,
   --#                                 Sym2 &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 Reference,
   --#                                 Scope,
   --#                                 SPARK_IO.File_Sys,
   --#                                 Sym,
   --#                                 Sym2 &
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrNum,
   --#                                 Position,
   --#                                 Reference,
   --#                                 Scope,
   --#                                 Sym,
   --#                                 Sym2 &
   --#         TotalErrorCount    from *;
   is
      Error : Error_Types.StringError;
      File  : Error_IO.File_Type;
   begin
      Justifications.SetCurrentUnitHasSemanticErrors (ErrorContextRec.JustificationsDataTable);
      ErrorContextRec.Severity := SemanticErrs;
      IncMessageCount (Error_Types.SemanticErr);
      File := ErrorContextRec.Errs;
      ErrorBuffer.Add (File,
                       Error_Types.SemanticErr,
                       Position,
                       Scope,
                       ErrNum,
                       Reference,
                       SymbolToName (Sym),
                       SymbolToName (Sym2),
                       Error_Types.NoName,
                        --to get
                       Error);
      ErrorContextRec.Errs := File;
      EchoErrorEntry (SPARK_IO.Standard_Output, Error);
   end SemanticErrorSym2;

   ---------------------------------------------------------------------------

   procedure SemanticErrorLex1Sym1 (ErrNum    : in Natural;
                                    Reference : in Natural;
                                    Position  : in LexTokenManager.TokenPosition;
                                    IdStr     : in LexTokenManager.LexString;
                                    Sym       : in Dictionary.Symbol;
                                    Scope     : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State  from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 IdStr,
   --#                                 Position,
   --#                                 Reference,
   --#                                 Scope,
   --#                                 Sym &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 IdStr,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 Reference,
   --#                                 Scope,
   --#                                 SPARK_IO.File_Sys,
   --#                                 Sym &
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrNum,
   --#                                 IdStr,
   --#                                 Position,
   --#                                 Reference,
   --#                                 Scope,
   --#                                 Sym &
   --#         TotalErrorCount    from *;
   is
      Error : Error_Types.StringError;
      File  : Error_IO.File_Type;
   begin
      Justifications.SetCurrentUnitHasSemanticErrors (ErrorContextRec.JustificationsDataTable);
      ErrorContextRec.Severity := SemanticErrs;
      IncMessageCount (Error_Types.SemanticErr);
      File := ErrorContextRec.Errs;
      ErrorBuffer.Add (File,
                       Error_Types.SemanticErr,
                       Position,
                       Scope,
                       ErrNum,
                       Reference,
                       LexStringToName (IdStr),
                       SymbolToName (Sym),
                       Error_Types.NoName,
                        --to get
                       Error);
      ErrorContextRec.Errs := File;
      EchoErrorEntry (SPARK_IO.Standard_Output, Error);
   end SemanticErrorLex1Sym1;

   ---------------------------------------------------------------------------

   procedure SemanticErrorLex1Sym2 (ErrNum    : in Natural;
                                    Reference : in Natural;
                                    Position  : in LexTokenManager.TokenPosition;
                                    IdStr     : in LexTokenManager.LexString;
                                    Sym       : in Dictionary.Symbol;
                                    Sym2      : in Dictionary.Symbol;
                                    Scope     : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State  from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 IdStr,
   --#                                 Position,
   --#                                 Reference,
   --#                                 Scope,
   --#                                 Sym,
   --#                                 Sym2 &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 IdStr,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 Reference,
   --#                                 Scope,
   --#                                 SPARK_IO.File_Sys,
   --#                                 Sym,
   --#                                 Sym2 &
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrNum,
   --#                                 IdStr,
   --#                                 Position,
   --#                                 Reference,
   --#                                 Scope,
   --#                                 Sym,
   --#                                 Sym2 &
   --#         TotalErrorCount    from *;
   is
      Error : Error_Types.StringError;
      File  : Error_IO.File_Type;
   begin
      Justifications.SetCurrentUnitHasSemanticErrors (ErrorContextRec.JustificationsDataTable);
      ErrorContextRec.Severity := SemanticErrs;
      IncMessageCount (Error_Types.SemanticErr);
      File := ErrorContextRec.Errs;
      ErrorBuffer.Add (File,
                       Error_Types.SemanticErr,
                       Position,
                       Scope,
                       ErrNum,
                       Reference,
                       LexStringToName (IdStr),
                       SymbolToName (Sym),
                       SymbolToName (Sym2),
                        --to get
                       Error);
      ErrorContextRec.Errs := File;
      EchoErrorEntry (SPARK_IO.Standard_Output, Error);
   end SemanticErrorLex1Sym2;

   ---------------------------------------------------------------------------

   procedure SemanticErrorSym3 (ErrNum    : in Natural;
                                Reference : in Natural;
                                Position  : in LexTokenManager.TokenPosition;
                                Sym       : in Dictionary.Symbol;
                                Sym2      : in Dictionary.Symbol;
                                Sym3      : in Dictionary.Symbol;
                                Scope     : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State  from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 Position,
   --#                                 Reference,
   --#                                 Scope,
   --#                                 Sym,
   --#                                 Sym2,
   --#                                 Sym3 &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrNum,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 Reference,
   --#                                 Scope,
   --#                                 SPARK_IO.File_Sys,
   --#                                 Sym,
   --#                                 Sym2,
   --#                                 Sym3 &
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrNum,
   --#                                 Position,
   --#                                 Reference,
   --#                                 Scope,
   --#                                 Sym,
   --#                                 Sym2,
   --#                                 Sym3 &
   --#         TotalErrorCount    from *;
   is
      Error : Error_Types.StringError;
      File  : Error_IO.File_Type;
   begin
      Justifications.SetCurrentUnitHasSemanticErrors (ErrorContextRec.JustificationsDataTable);
      ErrorContextRec.Severity := SemanticErrs;
      IncMessageCount (Error_Types.SemanticErr);
      File := ErrorContextRec.Errs;
      ErrorBuffer.Add (File,
                       Error_Types.SemanticErr,
                       Position,
                       Scope,
                       ErrNum,
                       Reference,
                       SymbolToName (Sym),
                       SymbolToName (Sym2),
                       SymbolToName (Sym3),
                        --to get
                       Error);
      ErrorContextRec.Errs := File;
      EchoErrorEntry (SPARK_IO.Standard_Output, Error);
   end SemanticErrorSym3;

   ---------------------------------------------------------------------------

   procedure ControlFlowError (ErrType  : in ControlFlowErrType;
                               Position : in LexTokenManager.TokenPosition)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State,
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrType,
   --#                                 Position &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 ErrType,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 SPARK_IO.File_Sys &
   --#         TotalErrorCount    from *;
   is
      Error : Error_Types.StringError;
      File  : Error_IO.File_Type;
   begin
      -- Note, no justification check here, these are serious structural errors in the code
      ErrorContextRec.Severity := FlowErrs;
      IncMessageCount (Error_Types.UncondFlowErr);
      File := ErrorContextRec.Errs;
      ErrorBuffer.Add (File,
                       Error_Types.ControlFlowErr,
                       Position,
                       Dictionary.GlobalScope,
                       ControlFlowErrType'Pos (ErrType) + Error_Types.ControlFlowErrOffset,
                       NoReference,
                       Error_Types.NoName,
                       Error_Types.NoName,
                       Error_Types.NoName,
                        --to get
                       Error);
      ErrorContextRec.Errs := File;
      EchoErrorEntry (SPARK_IO.Standard_Output, Error);
   end ControlFlowError;

   -------------------------------------------------------------------------

   procedure DataFlowError (ErrType  : in DataFlowErrType;
                            Position : in LexTokenManager.TokenPosition;
                            VarSym   : in Dictionary.Symbol;
                            Scope    : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State  from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 ErrType,
   --#                                 Position,
   --#                                 Scope,
   --#                                 VarSym &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 ErrType,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 Scope,
   --#                                 SPARK_IO.File_Sys,
   --#                                 VarSym &
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrorContextRec,
   --#                                 ErrType,
   --#                                 Position,
   --#                                 Scope,
   --#                                 VarSym &
   --#         TotalErrorCount    from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrorContextRec,
   --#                                 ErrType,
   --#                                 Position,
   --#                                 VarSym;
   is
      Error : Error_Types.StringError;
      File  : Error_IO.File_Type;
      ErrorIsJustified : Boolean;

      function ObtainErrorNumber (ErrType : DataFlowErrType) return Natural
      is
         Result : Natural;
      begin
         case ErrType is
            when ExpnUndefined |
              StmtUndefined |
              InvariantExp =>
               Result := DataFlowErrType'Pos (ErrType) + Error_Types.UncondFlowErrorOffset;
            when ExpnMayBeUndefined |
              StmtMayBeUndefined =>
               Result := DataFlowErrType'Pos (ErrType) + Error_Types.CondFlowErrorOffset;
         end case;
         return Result;
      end ObtainErrorNumber;

   begin -- DataFlowError
      Justifications.CheckWhetherJustified (ErrorContextRec.JustificationsDataTable,
                                            Position,
                                            FlowMessage,
                                            ObtainErrorNumber (ErrType),
                                            JustificationIdentifiers'
                                              (1 => SymbolToJustificationIdentifier (VarSym),
                                               others => NullJustificationIdentifier),
                                             -- to get
                                            ErrorIsJustified);
      if ErrorIsJustified then
         IncTotalJustifiedWarnings;

      else
         File := ErrorContextRec.Errs;
         case ErrType is
            when ExpnUndefined |
              StmtUndefined |
              InvariantExp =>
               ErrorContextRec.Severity := FlowErrs;
               IncMessageCount (Error_Types.UncondFlowErr);
               ErrorBuffer.Add (File,
                                Error_Types.UncondFlowErr,
                                Position,
                                Scope,
                                DataFlowErrType'Pos (ErrType) + Error_Types.UncondFlowErrorOffset,
                                NoReference,
                                SymbolToName (VarSym),
                                Error_Types.NoName,
                                Error_Types.NoName,
                                 --to get
                                Error);

            when ExpnMayBeUndefined |
              StmtMayBeUndefined =>
               ErrorContextRec.Severity := FlowWarning;
               IncMessageCount (Error_Types.CondlFlowErr);
               ErrorBuffer.Add (File,
                                Error_Types.CondlFlowErr,
                                Position,
                                Scope,
                                DataFlowErrType'Pos (ErrType) + Error_Types.CondFlowErrorOffset,
                                NoReference,
                                SymbolToName (VarSym),
                                Error_Types.NoName,
                                Error_Types.NoName,
                                 --to get
                                Error);
         end case;
         ErrorContextRec.Errs := File;
         EchoErrorEntry (SPARK_IO.Standard_Output, Error);
      end if;
   end DataFlowError;

   -------------------------------------------------------------------------

   procedure IneffectiveStmt (Position : in LexTokenManager.TokenPosition;
                              VarSym   : in Dictionary.Symbol;
                              Scope    : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State  from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 Position,
   --#                                 Scope,
   --#                                 VarSym &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 Scope,
   --#                                 SPARK_IO.File_Sys,
   --#                                 VarSym &
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrorContextRec,
   --#                                 Position,
   --#                                 Scope,
   --#                                 VarSym &
   --#         TotalErrorCount    from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrorContextRec,
   --#                                 Position,
   --#                                 VarSym;
   is
      Error : Error_Types.StringError;
      File  : Error_IO.File_Type;
      ErrorIsJustified : Boolean;
   begin
      Justifications.CheckWhetherJustified (ErrorContextRec.JustificationsDataTable,
                                            Position,
                                            FlowMessage,
                                            10,
                                            JustificationIdentifiers'
                                              (1 => SymbolToJustificationIdentifier (VarSym),
                                               others => NullJustificationIdentifier),
                                             -- to get
                                            ErrorIsJustified);
      if ErrorIsJustified then
         IncTotalJustifiedWarnings;

      else
         ErrorContextRec.Severity := FlowErrs;
         IncMessageCount (Error_Types.UncondFlowErr);
         File := ErrorContextRec.Errs;
         ErrorBuffer.Add (File,
                          Error_Types.IneffectiveStat,
                          Position,
                          Scope,
                          10,    -- arbitrary error number 10 assigned
                          NoReference,
                          SymbolToName (VarSym),
                          Error_Types.NoName,
                          Error_Types.NoName,
                           --to get
                          Error);
         ErrorContextRec.Errs := File;
         EchoErrorEntry (SPARK_IO.Standard_Output, Error);
      end if;
   end IneffectiveStmt;

   ----------------------------------------------------------------------------

   procedure StabilityError (ErrType        : in StabilityErrType;
                             Position       : in LexTokenManager.TokenPosition;
                             StabilityIndex : in IndexType)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State,
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 ErrType,
   --#                                 Position,
   --#                                 StabilityIndex &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 ErrType,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 SPARK_IO.File_Sys,
   --#                                 StabilityIndex &
   --#         TotalErrorCount    from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrorContextRec,
   --#                                 ErrType,
   --#                                 Position;
   is
      Error : Error_Types.StringError;
      File  : Error_IO.File_Type;
      ErrorIsJustified : Boolean;
   begin
      Justifications.CheckWhetherJustified (ErrorContextRec.JustificationsDataTable,
                                            Position,
                                            FlowMessage,
                                            StabilityErrType'Pos (ErrType) + Error_Types.StabilityErrOffset,
                                            NullJustificationIdentifiers,
                                             -- to get
                                            ErrorIsJustified);

      if ErrorIsJustified then
         IncTotalJustifiedWarnings;

      else
         ErrorContextRec.Severity := FlowErrs;
         IncMessageCount (Error_Types.UncondFlowErr);
         File := ErrorContextRec.Errs;
         ErrorBuffer.Add (File,
                          Error_Types.StabilityErr,
                          Position,
                          Dictionary.GlobalScope,
                          StabilityErrType'Pos (ErrType) + Error_Types.StabilityErrOffset,
                          NoReference,
                          Error_Types.Names'(Error_Types.StabilityIndex,
                                             IndexType'Pos (StabilityIndex)),
                          Error_Types.NoName,
                          Error_Types.NoName,
                           --to get
                          Error);
         ErrorContextRec.Errs := File;
         EchoErrorEntry (SPARK_IO.Standard_Output, Error);
      end if;
   end StabilityError;

   ----------------------------------------------------------------------------

   procedure DependencyError (ErrType      : in DependencyErrType;
                              Position     : in LexTokenManager.TokenPosition;
                              ImportVarSym : in Dictionary.Symbol;
                              ExportVarSym : in Dictionary.Symbol;
                              Scope        : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State  from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 ErrType,
   --#                                 ExportVarSym,
   --#                                 ImportVarSym,
   --#                                 Position,
   --#                                 Scope &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 ErrType,
   --#                                 ExportVarSym,
   --#                                 ImportVarSym,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 Scope,
   --#                                 SPARK_IO.File_Sys &
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrorContextRec,
   --#                                 ErrType,
   --#                                 ExportVarSym,
   --#                                 ImportVarSym,
   --#                                 Position,
   --#                                 Scope &
   --#         TotalErrorCount    from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrorContextRec,
   --#                                 ErrType,
   --#                                 ExportVarSym,
   --#                                 ImportVarSym,
   --#                                 Position;
   is
      Error : Error_Types.StringError;
      File  : Error_IO.File_Type;
      ErrorIsJustified : Boolean;
   begin
      case ErrType is
         when NotUsed   |
            IneffInit |
            IneffLocalInit =>

            Justifications.CheckWhetherJustified (ErrorContextRec.JustificationsDataTable,
                                                  Position,
                                                  FlowMessage,
                                                  DependencyErrType'Pos (ErrType) +
                                                    Error_Types.UncondDependencyErrorOffset,
                                                  JustificationIdentifiers'
                                                    (1 => SymbolToJustificationIdentifier (ExportVarSym),
                                                     2 => SymbolToJustificationIdentifier (ImportVarSym)),
                                                   -- to get
                                                  ErrorIsJustified);

            if ErrorIsJustified then
               IncTotalJustifiedWarnings;

            else
               File := ErrorContextRec.Errs;
               ErrorContextRec.Severity := FlowErrs;
               IncMessageCount (Error_Types.UncondDependencyErr);
               ErrorBuffer.Add (File,
                                Error_Types.UncondDependencyErr,
                                Position,
                                Scope,
                                DependencyErrType'Pos (ErrType) + Error_Types.UncondDependencyErrorOffset,
                                NoReference,
                                SymbolToName (ImportVarSym),
                                SymbolToName (ExportVarSym),
                                Error_Types.NoName,
                                 --to get
                                Error);
               ErrorContextRec.Errs := File;
               EchoErrorEntry (SPARK_IO.Standard_Output, Error);
            end if;

         when MayBeUsed |
           Uninitialised |
           IntegrityViolation |
           MayBeIntegrityViolation =>

            Justifications.CheckWhetherJustified (ErrorContextRec.JustificationsDataTable,
                                                  Position,
                                                  FlowMessage,
                                                  DependencyErrType'Pos (ErrType) +
                                                    Error_Types.CondDependencyErrorOffset,
                                                  JustificationIdentifiers'
                                                    (1 => SymbolToJustificationIdentifier (ExportVarSym),
                                                     2 => SymbolToJustificationIdentifier (ImportVarSym)),
                                                   -- to get
                                                  ErrorIsJustified);

            if ErrorIsJustified then
               IncTotalJustifiedWarnings;

            else
               File := ErrorContextRec.Errs;
               ErrorContextRec.Severity := FlowWarning;
               IncMessageCount (Error_Types.CondlDependencyErr);
               ErrorBuffer.Add (File,
                                Error_Types.CondlDependencyErr,
                                Position,
                                Scope,
                                DependencyErrType'Pos (ErrType) + Error_Types.CondDependencyErrorOffset,
                                NoReference,
                                SymbolToName (ImportVarSym),
                                SymbolToName (ExportVarSym),
                                Error_Types.NoName,
                                 --to get
                                Error);
               ErrorContextRec.Errs := File;
               EchoErrorEntry (SPARK_IO.Standard_Output, Error);
            end if;
      end case;
   end DependencyError;

   --------------------------------------------------------------------------

   procedure UsageError (Errtype  : in UsageErrType;
                         Position : in LexTokenManager.TokenPosition;
                         VarSym   : in Dictionary.Symbol;
                         Scope    : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State  from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 ErrType,
   --#                                 Position,
   --#                                 Scope,
   --#                                 VarSym &
   --#         EchoAccumulator,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys  from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 ErrType,
   --#                                 LexTokenManager.StringTable,
   --#                                 Position,
   --#                                 Scope,
   --#                                 SPARK_IO.File_Sys,
   --#                                 VarSym &
   --#         ErrorBuffer.Buffer from *,
   --#                                 CommandLineData.Content,
   --#                                 Dictionary.Dict,
   --#                                 ErrorContextRec,
   --#                                 ErrType,
   --#                                 Position,
   --#                                 Scope,
   --#                                 VarSym &
   --#         TotalErrorCount    from *,
   --#                                 CommandLineData.Content,
   --#                                 ErrorContextRec,
   --#                                 ErrType,
   --#                                 Position,
   --#                                 VarSym;
   is
      Error : Error_Types.StringError;
      File  : Error_IO.File_Type;
      ErrorIsJustified : Boolean;
   begin
      Justifications.CheckWhetherJustified (ErrorContextRec.JustificationsDataTable,
                                            Position,
                                            FlowMessage,
                                            UsageErrType'Pos (Errtype) + Error_Types.UsageErrOffset,
                                            JustificationIdentifiers'
                                              (1 => SymbolToJustificationIdentifier (VarSym),
                                               others => NullJustificationIdentifier),
                                             -- to get
                                            ErrorIsJustified);

      if ErrorIsJustified then
         IncTotalJustifiedWarnings;

      else
         File := ErrorContextRec.Errs;
         ErrorContextRec.Severity := FlowErrs;
         IncMessageCount (Error_Types.UncondDependencyErr);
         ErrorBuffer.Add (File,
                          Error_Types.UsageErr,
                          Position,
                          Scope,
                          UsageErrType'Pos (Errtype) + Error_Types.UsageErrOffset,
                          NoReference,
                          SymbolToName (VarSym),
                          Error_Types.NoName,
                          Error_Types.NoName,
                           --to get
                          Error);
         ErrorContextRec.Errs := File;
         EchoErrorEntry (SPARK_IO.Standard_Output, Error);
      end if;
   end UsageError;

   -----------------------------------------------------------------------------

   procedure PutErrorPointers (Listing :  in SPARK_IO.File_Type;
                               Errors  :  in ErrorSets)
   --# global in     CommandLineData.Content;
   --#        in     ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                ErrorContextRec,
   --#                                Errors,
   --#                                Listing;
   is
      ErrPos     : Natural;
      LastErrPos : Natural;
      CurrPos    : Natural;
      ErrNum     : Positive;
      FirstOne   : Boolean;
   begin
      LastErrPos := 0;
      CurrPos := 0;
      FirstOne   := True;

      --# assert True; -- for RTC generation

      for I in ErrorSetPositions range 1 .. Errors.Length loop

         --# assert True; -- for RTC generation

         if ErrorHasPositionInline (Errors.Content (I).Error.ErrorType) then
            if FirstOne then
               PutSpaces (Listing, SourceLineIndent);
               FirstOne := False;
            end if;

            ErrPos := Errors.Content (I).Error.Position.StartPos;

            if ErrPos = 0 then
               ErrPos := 1;
            end if;

            --# assert True; -- for RTC generation

            if ErrPos = LastErrPos then
               Put_Char (Listing, ',');
            else
               if CurrPos > ErrPos then
                  New_Line (Listing, 1);
                  MoveToIndent (Listing, ErrorContextRec.CurrentLine,
                                SourceLineIndent, ErrPos - 1);
               elsif CurrPos = 0 then
                  MoveToIndent (Listing, ErrorContextRec.CurrentLine, 0, ErrPos - 1);
               else
                  PutSpaces (Listing, ErrPos - CurrPos);
               end if;
               CurrPos := ErrPos;
               Put_Char (Listing, '^');
            end if;
            ErrNum := Errors.Content (I).ErrNum;

            --# accept Flow, 41, "Mode-specific code";
            if CommandLineData.Content.PlainOutput then -- Expect stable condition
               CurrPos := CurrPos + 1;
            else
               Put_Integer (Listing, ErrNum, WidthOf (ErrNum), 10);
               CurrPos := (CurrPos + WidthOf (ErrNum)) + 1;
            end if;
            --# end accept;

            LastErrPos := ErrPos;
         end if;
      end loop;

      --# assert True; -- for RTC generation

      if not FirstOne then
         New_Line (Listing, 1);
      end if;
   end PutErrorPointers;

   procedure PutErrorMessages (Listing          : in     SPARK_IO.File_Type;
                               Errors           : in     ErrorSets;
                               StartPos         : in     Natural;
                               Accumulator      : in out ErrorAccumulator.T)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives Accumulator       from *,
   --#                                Errors,
   --#                                StartPos &
   --#         SPARK_IO.File_Sys from *,
   --#                                Accumulator,
   --#                                CommandLineData.Content,
   --#                                Errors,
   --#                                Listing,
   --#                                StartPos;
   is
      TheError      : ErrorStruct;
      NewStart      : Natural;
      StartedActive : Boolean;
      Indent        : Natural;
      LineLength    : Natural;
      Explanation   : ELStrings.T;
   begin

      Indent := 11;
      LineLength := ErrorLineLength;

      for I in ErrorSetPositions range 1 .. Errors.Length
      loop
         TheError := Errors.Content (I);

         StartedActive := ErrorAccumulator.IsActive (Accumulator);

         if ErrorAccumulator.IsActive (Accumulator) then
            ErrorAccumulator.Add (Accumulator, TheError,
                                  LineLength, Indent, Listing);
         end if;

         if not ErrorAccumulator.IsActive (Accumulator) then

            -- put a trailing blank line after compound messages.
            if StartedActive then
               New_Line (Listing, 2);
            end if;

            OutputErrorMarker (Listing,
                               TheError.Error.ErrorType,
                               TheError.Error.MessageId,
                               TheError.ErrNum);

            if ErrorAccumulator.IsErrorStart (TheError.Error) then
               -- Message is of akind that will be "accumulated" before printing.  We don't want
               -- the explanation to appear over and over again, only once at the end of the completed
               -- message.  So we extract explanation from message here and pass it into ErrorAccumulator
               -- where it is stored and appended to the error mesage when a later Flush operation is
               -- performed.
               SplitStringAtExplanation (TheError.Error.Message,
                                         Explanation);
               PrintLine (Listing,
                          StartPos,
                          LineLength,
                          Indent,
                          TheError.Error.Message, -- explanation has been removed
                          False,
                          NewStart);

               ErrorAccumulator.StartMsg (Accumulator,
                                          TheError,
                                          NewStart,
                                          Explanation, -- Explanation extracted by Split... above
                                          LineLength, -- we know this and Indent in this scope so pass them
                                          Indent);  -- in rather than recalculate them in error accumulator
            else
               -- leave message text unchanged and display it
               --# accept Flow, 10, NewStart, "Returned parameter not needed in this case";
               PrintLine (Listing, StartPos, LineLength, Indent, -- Expect ineff assignment to NewStart
                          TheError.Error.Message, False, NewStart);
               --# end accept;

               -- Terminate message
               New_Line (Listing, 1);

            end if;

         end if;
      end loop;
   end PutErrorMessages;

   procedure PutErrorMessagesXML (Listing          : in     SPARK_IO.File_Type;
                                  Errors           : in     ErrorSets;
                                  StartPos         : in     Natural;
                                  Accumulator      : in out ErrorAccumulator.T)
   --# global in out SPARK_IO.File_Sys;
   --#        in out XMLReport.State;
   --# derives Accumulator,
   --#         XMLReport.State   from *,
   --#                                Accumulator,
   --#                                Errors,
   --#                                StartPos &
   --#         SPARK_IO.File_Sys from *,
   --#                                Accumulator,
   --#                                Errors,
   --#                                Listing,
   --#                                StartPos,
   --#                                XMLReport.State;
   is
      TheError      : ErrorStruct;
      NewStart      : Natural;
      StartedActive : Boolean;
      Indent        : Natural;
      LineLength    : Natural;
      Explanation   : ELStrings.T;
   begin

      Indent := 0;
      LineLength := XMLErrorLineLength;

      for I in ErrorSetPositions range 1 .. Errors.Length
      loop
         TheError := Errors.Content (I);
         TheError.Error.Message := XMLReport.FilterLongString (TheError.Error.Message);

         StartedActive := ErrorAccumulator.IsActive (Accumulator);

         if ErrorAccumulator.IsActive (Accumulator) then
            ErrorAccumulator.Add (Accumulator, TheError,
                                  LineLength, Indent, Listing);
         end if;

         if not ErrorAccumulator.IsActive (Accumulator) then

            if StartedActive then
               XMLReport.EndMessage (Listing);
               New_Line (Listing, 1);
            end if;

            XMLReport.StartMessage (GetErrorClass (TheError.Error.ErrorType),
                                    ConvertMessageId (TheError.Error.MessageId,
                                                      TheError.Error.ErrorType),
                                    Integer (TheError.Error.Position.StartLineNo),
                                    TheError.Error.Position.StartPos,
                                    Listing);

            if ErrorAccumulator.IsErrorStart (TheError.Error) then
               -- Message is of akind that will be "accumulated" before printing.  We don't want
               -- the explanation to appear over and over again, only once at the end of the completed
               -- message.  So we extract explanation from message here and pass it into ErrorAccumulator
               -- where it is stored and appended to the error mesage when a later Flush operation is
               -- performed.
               SplitStringAtExplanation (TheError.Error.Message,
                                         Explanation);
               PrintLine (Listing,
                          StartPos,
                          LineLength,
                          Indent,
                          TheError.Error.Message, -- explanation has been removed
                          False,
                          NewStart);

               ErrorAccumulator.StartMsg (Accumulator,
                                          TheError,
                                          NewStart,
                                          Explanation, -- Explanation extracted by Split... above
                                          LineLength, -- we know this and Indent in this scope so pass them
                                          Indent);  -- in rather than recalculate them in error accumulator
            else
               -- leave message text unchanged and display it
               --# accept Flow, 10, NewStart, "Returned parameter not needed in this case";
               PrintLine (Listing, StartPos, LineLength, Indent, -- Expect ineff assignment to NewStart
                          TheError.Error.Message, False, NewStart);
               --# end accept;

               -- Terminate message
               XMLReport.EndMessage (Listing);

            end if;
         end if;
      end loop;
   end PutErrorMessagesXML;

   procedure PrintErrors (Listing : in SPARK_IO.File_Type;
                          Purpose : in Error_Types.ConversionRequestSource)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     WarningStatus.SuppressionList;
   --#        in out Conversions.State;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --# derives Conversions.State,
   --#         ErrorContextRec    from CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 LexTokenManager.StringTable,
   --#                                 Listing,
   --#                                 Purpose,
   --#                                 SPARK_IO.File_Sys &
   --#         ErrorBuffer.Buffer from * &
   --#         SPARK_IO.File_Sys  from *,
   --#                                 CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 LexTokenManager.StringTable,
   --#                                 Listing,
   --#                                 Purpose,
   --#                                 WarningStatus.SuppressionList;
      is separate;

   procedure AppendErrors (Report  : in SPARK_IO.File_Type;
                           Purpose : in Error_Types.ConversionRequestSource)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     WarningStatus.SuppressionList;
   --#        in out Conversions.State;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out XMLReport.State;
   --# derives Conversions.State,
   --#         ErrorContextRec,
   --#         XMLReport.State   from CommandLineData.Content,
   --#                                Conversions.State,
   --#                                Dictionary.Dict,
   --#                                ErrorContextRec,
   --#                                LexTokenManager.StringTable,
   --#                                Purpose,
   --#                                Report,
   --#                                SPARK_IO.File_Sys,
   --#                                XMLReport.State &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Conversions.State,
   --#                                Dictionary.Dict,
   --#                                ErrorContextRec,
   --#                                LexTokenManager.StringTable,
   --#                                Purpose,
   --#                                Report,
   --#                                WarningStatus.SuppressionList,
   --#                                XMLReport.State;
      is separate;


   -- exported operations concerned with error justification mechanism ------------------------------

   procedure StartUnit
   --# global in out ErrorContextRec;
   --# derives ErrorContextRec from *;
   is
   begin
      Justifications.StartUnit (ErrorContextRec.JustificationsDataTable);
   end StartUnit;

   procedure EndUnit
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     WarningStatus.SuppressionList;
   --#        in out Conversions.State;
   --#        in out EchoAccumulator;
   --#        in out ErrorBuffer.Buffer;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TotalErrorCount;
   --# derives Conversions.State,
   --#         EchoAccumulator,
   --#         ErrorBuffer.Buffer,
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys,
   --#         TotalErrorCount    from *,
   --#                                 CommandLineData.Content,
   --#                                 Conversions.State,
   --#                                 Dictionary.Dict,
   --#                                 EchoAccumulator,
   --#                                 ErrorBuffer.Buffer,
   --#                                 ErrorContextRec,
   --#                                 LexTokenManager.StringTable,
   --#                                 SPARK_IO.File_Sys,
   --#                                 WarningStatus.SuppressionList;
   is
      It : Justifications.UnmatchedJustificationIterator;
   begin
      -- First report all justifications in the unit just being ended that have failed to match
      -- But only if we aren't ignoring justifications
      if CommandLineData.Content.JustificationOption /= CommandLineData.Ignore then
         Justifications.FirstUnmatchedJustification (It,
                                                     ErrorContextRec.JustificationsDataTable);
         while not Justifications.IsNullIterator (It) loop
            SemanticWarning (121,
                             Justifications.ErrorPosition (It),
                             LexTokenManager.NullString);
            Justifications.NextUnmatchedJustification (It,
                                                       ErrorContextRec.JustificationsDataTable);
         end loop;
      end if;
      -- Then discard data for now-completed unit
      Justifications.EndUnit (ErrorContextRec.JustificationsDataTable);
   end EndUnit;

   procedure StartJustification (Position                     : in     LexTokenManager.TokenPosition;
                                 Line                         : in     LexTokenManager.LineNumbers;
                                 Kind                         : in     JustificationKinds;
                                 ErrNum                       : in     Natural;
                                 Identifiers                  : in     JustificationIdentifiers;
                                 Explanation                  : in     LexTokenManager.LexString;
                                 MaximumJustificationsReached :    out Boolean)
   --# global in out ErrorContextRec;
   --# derives ErrorContextRec              from *,
   --#                                           ErrNum,
   --#                                           Explanation,
   --#                                           Identifiers,
   --#                                           Kind,
   --#                                           Line,
   --#                                           Position &
   --#         MaximumJustificationsReached from ErrorContextRec;
   is
   begin
      Justifications.StartJustification (ErrorContextRec.JustificationsDataTable,
                                         Position,
                                         Line,
                                         Kind,
                                         ErrNum,
                                         Identifiers,
                                         Explanation,
                                          -- to get
                                         MaximumJustificationsReached);
   end StartJustification;

   procedure EndJustification (Line         : in     LexTokenManager.LineNumbers;
                               UnmatchedEnd :    out Boolean)
   --# global in out ErrorContextRec;
   --# derives ErrorContextRec,
   --#         UnmatchedEnd    from ErrorContextRec,
   --#                              Line;
   is
   begin
      Justifications.EndJustification (ErrorContextRec.JustificationsDataTable,
                                       Line,
                                       -- to get
                                       UnmatchedEnd);
   end EndJustification;

   procedure SetFileOpenError
   --# global out FileOpenError;
   --# derives FileOpenError from ;
   is
   begin
      FileOpenError := True;
   end SetFileOpenError;

   function GetErrorsType return Exit_Code
   --# global in FileOpenError;
   --#        in TotalErrorCount;
   is
      ReturnValue : Exit_Code := 0;
   begin
      if FileOpenError then
         ReturnValue := 8;
      elsif TotalErrorCount.ExplicitErrorCount (SyntaxOrSemantic) /= 0 then
         ReturnValue := 3;
      elsif TotalErrorCount.ExplicitErrorCount (Flow) /= 0 then
         ReturnValue := 2;
      elsif TotalErrorCount.ExplicitErrorCount (Warning) /= 0 then
         ReturnValue := 1;
      end if;
      return ReturnValue;
   end GetErrorsType;

begin
   ErrorContextRec := NullErrorContext;
   EchoAccumulator := ErrorAccumulator.Clear;
   TotalErrorCount := NullTotalErrorCounts;
end ErrorHandler;
