-- $Id: errorhandler-warningstatus.adb 13147 2009-04-24 18:20:53Z Trevor Jennings $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with Ada.Characters.Handling;
separate (ErrorHandler)
package body WarningStatus
--# own SuppressionList is SuppressedElement,
--#                        PragmaList,
--#                        SuppressAllPragmas,
--#                        SomethingSuppressed;
is

   subtype PragmaCounts is Integer range
      0 .. ExaminerConstants.MaxPragmasInWarningFile;
   subtype PragmaIndex  is Integer range
      1 .. ExaminerConstants.MaxPragmasInWarningFile;

   type    PragmaArrays is array (PragmaIndex) of LexTokenManager.LexString;
   type    PragmaLists  is record
      PragmaArray : PragmaArrays;
      PragmaCount : PragmaCounts;
   end record;

   type SuppressedElementArray is array (ErrorHandler.WarningElements) of Boolean;


   SuppressedElement     : SuppressedElementArray;
   PragmaList            : PragmaLists;
   SomethingSuppressed   : Boolean;
   SuppressAllPragmas    : Boolean;

   ---------------------------------------------------------------------------

   function PragmaFound (PragmaName : LexTokenManager.LexString)
                        return Boolean
   --# global in PragmaList;
   is
      LookAt,
      Left,
      Right  : Integer;
      Found  : Boolean;
      Sought : Natural;

      type Compares is (LessThan, Equal, GreaterThan);
      MatchRes : Compares;

      function MatchCheck (Pos    : Integer) return Compares
      --# global in PragmaList;
      --#        in Sought;
      is
         Result : Compares;
         Elem   : Natural;
      begin
         --can't order private types so have to convert them using this call
         Elem := LexTokenManager.LexStringRef (PragmaList.PragmaArray (Pos));
         if  Elem < Sought then
            Result := LessThan;
         elsif Elem = Sought then
            Result := Equal;
         else
            Result := GreaterThan;
         end if;
         return Result;
      end MatchCheck;

   begin
      Sought := LexTokenManager.LexStringRef (PragmaName);
      Left := 0;
      Right := PragmaList.PragmaCount + 1;
      Found := False;

      loop
         exit when (Left + 1) = Right;
         LookAt := (Left + Right) / 2;

         MatchRes := MatchCheck (LookAt);

         if MatchRes = Equal then
            Found := True;
            exit;
         end if;

         if MatchRes = LessThan then
            Left := LookAt;
         else
            Right := LookAt;
         end if;
      end loop;

      return Found;
   end PragmaFound;

   ---------------------------------------------------------------------------

   procedure ReadWarningFile
   --# global in     CommandLineData.Content;
   --#        in out ErrorHandler.FileOpenError;
   --#        in out LexTokenManager.StringTable;
   --#        in out PragmaList;
   --#        in out SomethingSuppressed;
   --#        in out SPARK_IO.File_Sys;
   --#        in out SuppressAllPragmas;
   --#        in out SuppressedElement;
   --# derives ErrorHandler.FileOpenError  from *,
   --#                                          CommandLineData.Content,
   --#                                          SPARK_IO.File_Sys &
   --#         LexTokenManager.StringTable,
   --#         SPARK_IO.File_Sys,
   --#         SuppressedElement           from *,
   --#                                          CommandLineData.Content,
   --#                                          LexTokenManager.StringTable,
   --#                                          PragmaList,
   --#                                          SPARK_IO.File_Sys &
   --#         PragmaList,
   --#         SuppressAllPragmas          from CommandLineData.Content,
   --#                                          LexTokenManager.StringTable,
   --#                                          PragmaList,
   --#                                          SPARK_IO.File_Sys,
   --#                                          SuppressAllPragmas &
   --#         SomethingSuppressed         from *,
   --#                                          CommandLineData.Content,
   --#                                          LexTokenManager.StringTable,
   --#                                          PragmaList,
   --#                                          SPARK_IO.File_Sys,
   --#                                          SuppressedElement;
      is separate;

   ---------------------------------------------------------------------------

   function IsSuppressed (TheElement : ErrorHandler.WarningElements) return Boolean
   --# global in SuppressedElement;
   is
   begin
      return SuppressedElement (TheElement);
   end IsSuppressed;

   ---------------------------------------------------------------------------

   function PragmaIsSuppressed (PragmaName : LexTokenManager.LexString)
                               return Boolean
   --# global in PragmaList;
   --#        in SuppressAllPragmas;
   is
      Result : Boolean;
   begin
      Result := SuppressAllPragmas;
      if not Result then
         Result := PragmaFound (PragmaName);
      end if;
      return Result;
   end PragmaIsSuppressed;

   ---------------------------------------------------------------------------

   function GetDescription (Item : in ErrorHandler.WarningElements;
                            P1   : in String;
                            P2   : in String) return EStrings.T
   is
      TmpString,
      Result     : EStrings.T;
      Posn       : EStrings.Lengths;
      AlwaysTrue : Boolean;
   begin
      -- The messages are parmeterised for different message formats.
      -- The substring %%1 is substituted with the string P1 and
      -- the substring %%2 is substituted with the string P2.
      -- Some message formats require a simple "s character to represent a
      -- plural whereas others require the string "(s)".  In one case in one
      -- format has a plural whereas another does not.
      -- This complex scheme is to maintain compatibility with an earlier
      -- message scheme.
      case Item is
         when ErrorHandler.Pragmas               =>
            TmpString := XMLReport.XStr ("Pragma%%1");
         when ErrorHandler.HiddenParts           =>
            TmpString := XMLReport.XStr ("Hidden part%%1");
         when ErrorHandler.HandlerParts           =>
            TmpString := XMLReport.XStr ("Hidden exception handler part%%1");
         when ErrorHandler.RepresentationClauses =>
            TmpString := XMLReport.XStr ("Representation clause%%1");
         when ErrorHandler.DirectUpdates         =>
            TmpString := XMLReport.XStr ("Direct update%%1 of own variable(s) of non-enclosing package%%1");
         when ErrorHandler.WithClauses           =>
            TmpString := XMLReport.XStr ("With clause%%1 lacking a supporting inherit");
         when ErrorHandler.StaticExpressions     =>
            TmpString := XMLReport.XStr ("Static expression%%1 too complex for Examiner");
         when ErrorHandler.UnusedVariables       =>
            TmpString := XMLReport.XStr ("Variable%%1 declared but not used");
         when ErrorHandler.ConstantVariables     =>
            TmpString := XMLReport.XStr ("Variable%%1 used as constants");
         when ErrorHandler.TypeConversions     =>
            TmpString := XMLReport.XStr ("Unnecessary type conversion%%1");
         when ErrorHandler.OthersClauses     =>
            TmpString := XMLReport.XStr ("Unnecessary others clause%%1");
         when ErrorHandler.ImportedObjects     =>
            TmpString := XMLReport.XStr ("Use%%1 of pragma Import on objects");
         when ErrorHandler.UnexpectedAddressClauses     =>
            TmpString := XMLReport.XStr ("Unexpected address clause%%1");
         when ErrorHandler.ExpressionReordering     =>
            TmpString := XMLReport.XStr ("Reordering of expressions");
         when ErrorHandler.Notes     =>
            TmpString := XMLReport.XStr ("Note%%1");
         when ErrorHandler.UnuseablePrivateTypes     =>
            TmpString := XMLReport.XStr ("Private type%%1 lacking method of initialization");
         when ErrorHandler.ExternalVariableAssignment     =>
            TmpString := XMLReport.XStr ("Assignment%%2 or return%%2 of external variables");
         when ErrorHandler.DeclareAnnotations     =>
            TmpString := XMLReport.XStr ("Declare annotations in non Ravenscar programs");
         when ErrorHandler.InterruptHandlers     =>
            TmpString := XMLReport.XStr ("Protected objects that include interrupt handlers");
         when ErrorHandler.UncheckedConversion     =>
            TmpString := XMLReport.XStr ("Use%%1 of instantiations of Unchecked_Conversion");
         when ErrorHandler.Ada2005ReservedWords =>
            TmpString := XMLReport.XStr ("Use%%1 of Ada2005 reserved words");
         when ErrorHandler.ObsolescentFeatures =>
            TmpString := XMLReport.XStr ("Use%%1 of obsolete feature from Ada83 in SPARK 95 mode");
         when ErrorHandler.DefaultLoopAssertions =>
            TmpString := XMLReport.XStr ("Generation of default loop assertions");
         when ErrorHandler.RealRTCs =>
            TmpString := XMLReport.XStr ("Generation of RTCs on real numbers");
      end case;

      Posn := 0;
      Result := EStrings.EmptyString;
      while Posn < TmpString.Length loop
         Posn := Posn + 1;
         if Posn + 2 <= TmpString.Length and then
           TmpString.Content (Posn) = '%' and then
           TmpString.Content (Posn + 1) = '%' and then
           (TmpString.Content (Posn + 2) = '1' or else
            TmpString.Content (Posn + 2) = '2')
         then
            if TmpString.Content (Posn + 2) = '1' then
               EStrings.AppendString (Result, P1);
            else
               EStrings.AppendString (Result, P2);
            end if;
            Posn := Posn + 2;
         else
            --# accept F, 10, AlwaysTrue,
            --#        "The loop conditions ensure that the cheracter will always be appended";
            EStrings.AppendChar (Result,  TmpString.Content (Posn), AlwaysTrue);
            --# end accept;
         end if;
      end loop;

      --# accept F, 33, AlwaysTrue,
      --#        "The loop constraints ensure result of EStrings.AppendChar is always True.";
      return Result;
   end GetDescription;


   ---------------------------------------------------------------------------

   procedure OutputWarningList (ToFile : in SPARK_IO.File_Type)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     PragmaList;
   --#        in     SomethingSuppressed;
   --#        in     SuppressAllPragmas;
   --#        in     SuppressedElement;
   --#        in out SPARK_IO.File_Sys;
   --#        in out XMLReport.State;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                LexTokenManager.StringTable,
   --#                                PragmaList,
   --#                                SomethingSuppressed,
   --#                                SuppressAllPragmas,
   --#                                SuppressedElement,
   --#                                ToFile,
   --#                                XMLReport.State &
   --#         XMLReport.State   from *,
   --#                                CommandLineData.Content,
   --#                                PragmaList,
   --#                                SomethingSuppressed,
   --#                                SuppressAllPragmas,
   --#                                SuppressedElement;
   is
      procedure PutPragmas
      --# global in     CommandLineData.Content;
      --#        in     LexTokenManager.StringTable;
      --#        in     PragmaList;
      --#        in     SuppressAllPragmas;
      --#        in     SuppressedElement;
      --#        in     ToFile;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                LexTokenManager.StringTable,
      --#                                PragmaList,
      --#                                SuppressAllPragmas,
      --#                                SuppressedElement,
      --#                                ToFile,
      --#                                XMLReport.State &
      --#         XMLReport.State   from *,
      --#                                CommandLineData.Content,
      --#                                PragmaList,
      --#                                SuppressAllPragmas,
      --#                                SuppressedElement;
      is
         Wrap   : constant Integer := 72;
         Margin : constant Integer := 14;
         Column : Integer;
         Punct  : Character;

         procedure PutPragmaName (Str : LexTokenManager.LexString)
         --# global in     LexTokenManager.StringTable;
         --#        in     ToFile;
         --#        in out Column;
         --#        in out Punct;
         --#        in out SPARK_IO.File_Sys;
         --# derives Column            from *,
         --#                                LexTokenManager.StringTable,
         --#                                Str &
         --#         Punct             from  &
         --#         SPARK_IO.File_Sys from *,
         --#                                Column,
         --#                                LexTokenManager.StringTable,
         --#                                Punct,
         --#                                Str,
         --#                                ToFile;
         is
            Result      : EStrings.T;
         begin
            LexTokenManager.LexStringToString (Str, Result);
            SPARK_IO.Put_Char (ToFile, Punct);
            SPARK_IO.Put_Char (ToFile, ' ');
            Punct := ',';
            Column := Column + 2;
            if Column + Result.Length > Wrap then
               SPARK_IO.New_Line (ToFile, 1);
               SPARK_IO.Put_String (ToFile, "            ", 0);
               Column := Margin;
            end if;
            EStrings.PutString (ToFile, Result);
            Column := Column + Result.Length;
         end PutPragmaName;

         function GetPragmaName (Str : LexTokenManager.LexString) return EStrings.T
         --# global in LexTokenManager.StringTable;
         is
            Result : EStrings.T;
         begin
            LexTokenManager.LexStringToString (Str, Result);
            return Result;
         end GetPragmaName;


      begin --PutPragmas
         if CommandLineData.Content.XML then
            if SuppressAllPragmas then
               XMLReport.SuppressedPragma (XMLReport.XStr ("all"),
                                           ToFile);
            elsif SuppressedElement (ErrorHandler.Pragmas) then
               for I in Integer range 1 .. PragmaList.PragmaCount loop
                  XMLReport.SuppressedPragma (GetPragmaName (PragmaList.PragmaArray (I)),
                                              ToFile);
               end loop;
            end if;
         else
            if SuppressAllPragmas then
               SPARK_IO.Put_Line (ToFile, "   All pragmas", 0);
            elsif SuppressedElement (ErrorHandler.Pragmas) then
               Column := Margin;
               Punct := ':';
               SPARK_IO.Put_String (File => ToFile,
                                    Item => "   ",
                                    Stop => 0);
               EStrings.PutString (File => ToFile,
                                   EStr =>
                                     GetDescription (Item => ErrorHandler.Pragmas,
                                                     P1   => "s",
                                                     P2   => ""));
               for I in Integer range 1 .. PragmaList.PragmaCount loop
                  PutPragmaName (PragmaList.PragmaArray (I));
               end loop;
            end if;
            SPARK_IO.New_Line (ToFile, 1);
         end if;
      end PutPragmas;

   begin  --OutputWarningList
      if CommandLineData.Content.XML then
         XMLReport.StartSection (XMLReport.SWarningsConfig,
                                 ToFile);
         if SomethingSuppressed then
            for I in ErrorHandler.WarningElements range
              ErrorHandler.HiddenParts .. ErrorHandler.WarningElements'Last
            loop
               if SuppressedElement (I) then
                  XMLReport.Suppressed (GetDescription (Item => I,
                                                        P1   => "s",
                                                        P2   => ""),
                                        ToFile);
               end if;
            end loop;
            PutPragmas;
         end if;
         XMLReport.EndSection (XMLReport.SWarningsConfig,
                                 ToFile);
      else
         SPARK_IO.New_Line (ToFile, 2);
         if SomethingSuppressed then
            SPARK_IO.Put_Line (ToFile, "Summary warning reporting selected for:", 0);
            for I in ErrorHandler.WarningElements range
              ErrorHandler.HiddenParts .. ErrorHandler.WarningElements'Last
            loop
               if SuppressedElement (I) then
                  SPARK_IO.Put_String (File => ToFile,
                                       Item => "   ",
                                       Stop => 0);
                  EStrings.PutLine (File => ToFile,
                                    EStr =>
                                      GetDescription (Item => I,
                                                      P1   => "s",
                                                      P2   => ""));
               end if;
            end loop;
            PutPragmas;
         else
            SPARK_IO.Put_Line (ToFile, "Full warning reporting selected", 0);
         end if;

      end if;

   end OutputWarningList;

   ---------------------------------------------------------------------------

   procedure ReportSuppressedWarnings (ToFile  : in SPARK_IO.File_Type;
                                       Counter : ErrorHandler.Counters)
   --# global in     SomethingSuppressed;
   --#        in     SuppressedElement;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Counter,
   --#                                SomethingSuppressed,
   --#                                SuppressedElement,
   --#                                ToFile;
   is
      indent : constant Integer := 6;
      TotalWarnings : Integer;
      SevereWarning : Boolean := False;
      TmpString : EStrings.T;

      procedure PutCount (Count : in Integer;
                          Width : in Integer)
      --# global in     ToFile;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Count,
      --#                                ToFile,
      --#                                Width;
      is
      begin
         SPARK_IO.Put_Integer (ToFile, Count, Width, 10);
      end PutCount;

   begin  --ReportSuppressedWarnings
      if SomethingSuppressed then

         TotalWarnings := 0;
         for I in ErrorHandler.WarningElements loop
            TotalWarnings := TotalWarnings + Integer (Counter (I));
         end loop;

         if TotalWarnings = 0 then
            SPARK_IO.Put_Line (ToFile, "No summarized warnings", 0);
            SPARK_IO.New_Line (ToFile, 1);
         else
            PutCount (TotalWarnings, 0);
            SPARK_IO.Put_Line (ToFile, " summarized warning(s), comprising:", 0);
            for I in ErrorHandler.WarningElements loop
               if SuppressedElement (I) and then Counter (I) > 0 then
                  PutCount (Integer (Counter (I)), indent);
                  SPARK_IO.Put_Char (File => ToFile, Item => ' ');
                  TmpString := GetDescription (Item => I,
                                               P1   => "(s)",
                                               P2   => "(s)");
                  if TmpString.Length > 0 then
                     TmpString.Content (1) :=
                       Ada.Characters.Handling.To_Lower (TmpString.Content (1));
                  end if;

                  EStrings.PutString (File => ToFile,
                                      EStr => TmpString);

                  if I in ErrorHandler.SevereWarnings then
                     SPARK_IO.Put_Char (ToFile, '*');
                     SevereWarning := True;
                  end if;
                  SPARK_IO.New_Line (ToFile, 1);
               end if;
            end loop;
            if SevereWarning then
               SPARK_IO.Put_Line
                  (ToFile,
                   "(*Note: the above warnings may affect the validity of the analysis.)",
                   0);
            end if;
            SPARK_IO.New_Line (ToFile, 1);
         end if;
      end if;
   end ReportSuppressedWarnings;

   ---------------------------------------------------------------------------

begin
   SuppressedElement      := SuppressedElementArray'(others => False);
   PragmaList.PragmaCount := 0; --will cause flow error
   SuppressAllPragmas     := False;
   SomethingSuppressed    := False;
   --# accept Flow, 32, PragmaList.PragmaArray, "Init. is partial but effective." &
   --#        Flow, 31, PragmaList.PragmaArray, "Init. is partial but effective." &
   --#        Flow, 602, PragmaList, PragmaList.PragmaArray, "Init. is partial but effective.";
end WarningStatus;
