-- $Id: errorhandler-warningstatus-readwarningfile.adb 11887 2008-12-12 14:21:06Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with Ada.Characters.Handling,
     Ada.Characters.Latin_1;

separate (ErrorHandler.WarningStatus)
procedure ReadWarningFile
is
   Option      : EStrings.T;
   FileOk      : Boolean;
   WarningFile : SPARK_IO.File_Type;

   procedure OpenFile
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --#           out FileOk;
   --#           out WarningFile;
   --# derives FileOk,
   --#         SPARK_IO.File_Sys,
   --#         WarningFile       from CommandLineData.Content,
   --#                                SPARK_IO.File_Sys;
   is
      FileName       : EStrings.T;
      FileSpecStatus : FileSystem.TypFileSpecStatus;
      FileStatus     : SPARK_IO.File_Status;

   begin
      --# accept Flow, 10, FileSpecStatus, "Expected ineffective assignment to FileSpecStatus";
      FileSystem.FindFullFileName (CommandLineData.Content.WarningFileName,
                                   FileSpecStatus,
                                   FileName);
      --# end accept;

      WarningFile := SPARK_IO.Null_File; -- to avoid error on opening

      SPARK_IO.Open (WarningFile,
                     SPARK_IO.In_File,
                     FileName.Length,
                     FileName.Content,
                     "",
                     FileStatus);

      if FileStatus = SPARK_IO.Ok then
         FileOk := True;
      else
         FileOk := False;
         ScreenEcho.Put_String ("Cannot open file ");
         if CommandLineData.Content.PlainOutput then
            ScreenEcho.Put_ExaminerLine
               (EStrings.LowerCase
                (FileSystem.JustFile (FileName, True)));
         else
            ScreenEcho.Put_ExaminerLine (FileName);
         end if;
      end if;
      --# accept Flow, 33, FileSpecStatus, "Expected FileSpecStatus to be neither referenced nor exported";
   end OpenFile;

   ------------------------------------------------

   procedure CloseFile
   --# global in     WarningFile;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                WarningFile;
   is
      FileStatus     : SPARK_IO.File_Status;
   begin
      --# accept Flow, 10, FileStatus, "Expected ineffective assignment to FileStatus" &
      --#        Flow, 10, WarningFile, "Not assigned to. Due to Text_IO mode in out";
      SPARK_IO.Close (WarningFile,
                      FileStatus);
      --# end accept;
      --# accept Flow, 33, FileStatus, "Expected FileStatus to be neither referenced nor exported" &
      --#        Flow, 34, WarningFile, "Not assigned to. Due to Text_IO mode in out";
   end CloseFile;

   ------------------------------------------------

   procedure GetString (File : in     SPARK_IO.File_Type;
                        Str  :    out EStrings.T)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys,
   --#         Str               from File,
   --#                                SPARK_IO.File_Sys;
   is
      CharOk : Boolean;
      Ch     : Character;
      AppendOK : Boolean;
      EStr   : EStrings.T;

      procedure GetChar (File : in     SPARK_IO.File_Type;
                         Ch   :    out Character;
                         OK   :    out Boolean)
      --# global in out SPARK_IO.File_Sys;
      --# derives Ch,
      --#         OK,
      --#         SPARK_IO.File_Sys from File,
      --#                                SPARK_IO.File_Sys;
      is
         ChLocal : Character;
      begin
         if SPARK_IO.End_Of_File (File) then
            OK := False;
            Ch := ' ';
         elsif SPARK_IO.End_Of_Line (File) then
            SPARK_IO.Skip_Line (File, 1);
            OK := True;
            Ch := ' ';
         else
            SPARK_IO.Get_Char (File, ChLocal);
            if (ChLocal = Ada.Characters.Latin_1.HT) or (ChLocal = Ada.Characters.Latin_1.CR) then
               ChLocal := ' ';
            end if;
            if ChLocal = '-' then --must be comment start
               SPARK_IO.Skip_Line (File, 1);
               OK := True;
               Ch := ' ';
            else --valid character to return
               OK := True;
               Ch := ChLocal;
            end if;
         end if;
      end GetChar;

   begin --GetString
      EStr := EStrings.EmptyString;

      --skip leading white space
      loop
         GetChar (File, Ch, CharOk);
         exit when Ch /= ' ';
         exit when not CharOk;
      end loop;

      if CharOk then
         loop
            EStrings.AppendChar (EStr, Ch, AppendOK);
            if not AppendOK then
               SystemErrors.FatalError (SystemErrors.WarningNameTooLong, "");
            end if;
            GetChar (File, Ch, CharOk);
            exit when Ch = ' ';
            exit when not CharOk;
         end loop;
      end if;
      Str := EStr;
   end GetString;

   -----------------------------------

   procedure CheckOptionName (OptName : in     EStrings.T;
                              Str     : in     String;
                              OK      :    out Boolean)
   --# derives OK from OptName,
   --#                 Str;
   is
      LOK : Boolean := False;
   begin --CheckOptionName
      if OptName.Length <= Str'Length then
         for I in EStrings.Lengths range 1 .. OptName.Length loop
            LOK := Ada.Characters.Handling.To_Lower (OptName.Content (I)) =
               Ada.Characters.Handling.To_Lower (Str (I));
            exit when not LOK;
         end loop;
      end if;
      OK := LOK;
   end CheckOptionName;

   -------------------------------

   procedure InvalidOption (Opt : EStrings.T)
   --# global in     CommandLineData.Content;
   --#        in     WarningFile;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Opt,
   --#                                WarningFile;
   is
   begin
      if CommandLineData.Content.Brief then
         if CommandLineData.Content.PlainOutput then
            ScreenEcho.Put_ExaminerString
              (EStrings.LowerCase (CommandLineData.Content.WarningFileName));
         else
            ScreenEcho.Put_ExaminerString
              (CommandLineData.Content.WarningFileName);
         end if;

         ScreenEcho.Put_Char (':');
         ScreenEcho.Put_Integer (SPARK_IO.Line (WarningFile), 0, 10);
         ScreenEcho.Put_Char (':');
         ScreenEcho.Put_Integer (1, 0, 10);
         ScreenEcho.Put_Char (':');
         ScreenEcho.Put_String ("Invalid warning option: ");
         ScreenEcho.Put_ExaminerLine (Opt);
      else
         ScreenEcho.Put_String ("Invalid warning option: ");
         ScreenEcho.Put_ExaminerLine (Opt);
      end if;
   end InvalidOption;

   -------------------------------

   procedure ProcessOption (Opt : in EStrings.T)
   --# global in     CommandLineData.Content;
   --#        in     WarningFile;
   --#        in out LexTokenManager.StringTable;
   --#        in out PragmaList;
   --#        in out SPARK_IO.File_Sys;
   --#        in out SuppressAllPragmas;
   --#        in out SuppressedElement;
   --# derives LexTokenManager.StringTable,
   --#         PragmaList                  from LexTokenManager.StringTable,
   --#                                          Opt,
   --#                                          PragmaList,
   --#                                          SPARK_IO.File_Sys,
   --#                                          WarningFile &
   --#         SPARK_IO.File_Sys           from *,
   --#                                          CommandLineData.Content,
   --#                                          Opt,
   --#                                          PragmaList,
   --#                                          WarningFile &
   --#         SuppressAllPragmas          from *,
   --#                                          Opt,
   --#                                          SPARK_IO.File_Sys,
   --#                                          WarningFile &
   --#         SuppressedElement           from *,
   --#                                          Opt;
   is
      OptionMatch : Boolean;

      procedure ProcessPragma
      --# global in     WarningFile;
      --#        in out LexTokenManager.StringTable;
      --#        in out PragmaList;
      --#        in out SPARK_IO.File_Sys;
      --#        in out SuppressAllPragmas;
      --# derives LexTokenManager.StringTable,
      --#         SPARK_IO.File_Sys           from *,
      --#                                          PragmaList,
      --#                                          SPARK_IO.File_Sys,
      --#                                          WarningFile &
      --#         PragmaList                  from *,
      --#                                          LexTokenManager.StringTable,
      --#                                          SPARK_IO.File_Sys,
      --#                                          WarningFile &
      --#         SuppressAllPragmas          from *,
      --#                                          SPARK_IO.File_Sys,
      --#                                          WarningFile;
      is
         PragmaName : EStrings.T;
         Match      : Boolean;

         procedure AddPragmaName (Prag : in EStrings.T)
         --# global in out LexTokenManager.StringTable;
         --#        in out PragmaList;
         --#        in out SPARK_IO.File_Sys;
         --# derives LexTokenManager.StringTable,
         --#         SPARK_IO.File_Sys           from *,
         --#                                          Prag,
         --#                                          PragmaList &
         --#         PragmaList                  from *,
         --#                                          LexTokenManager.StringTable,
         --#                                          Prag;
         is
            LexName    : LexTokenManager.LexString;
            TempLine   : EStrings.Line;
         begin
            if PragmaList.PragmaCount <
               (ExaminerConstants.MaxPragmasInWarningFile - 1)
            then
               for I in Integer range 1 .. Prag.Length loop
                  --# accept Flow, 23, TempLine, "Array initialised in loop";
                  TempLine (I) := Prag.Content (I);
                  --# end accept;
               end loop;
               --# accept Flow, 504, TempLine, "Array initialised in loop";
               LexTokenManager.InsertLexString (TempLine, -- array initialisation
                                                1,
                                                Prag.Length,
                                                LexName);
               --# end accept;
               PragmaList.PragmaCount := PragmaList.PragmaCount + 1;
               PragmaList.PragmaArray (PragmaList.PragmaCount) := LexName;
            else -- too many
               ScreenEcho.Put_String ("Too many pragmas, ignoring: ");
               ScreenEcho.Put_ExaminerLine (Prag);
            end if;
            --# accept Flow, 602, LexTokenManager.StringTable, TempLine, "Array initialised in loop" &
            --#        Flow, 602, PragmaList, TempLine, "Array initialised in loop";
         end AddPragmaName;

      begin --ProcessPragma
         GetString (WarningFile,
                     --to get
                    PragmaName);
         if PragmaName.Length /= 0 then
            CheckOptionName (PragmaName, "all", Match);
            if Match then
               SuppressAllPragmas := True;
            else
               AddPragmaName (PragmaName);
            end if;
         else
            ScreenEcho.Put_Line ("Pragma name missing");
         end if;
      end ProcessPragma;

      --------------------------

   begin --ProcessOption
      OptionMatch := False;
      case Opt.Content (1) is
         when 'p' | 'P' =>
            case Opt.Content (3) is
               when 'a' | 'A' =>
                  CheckOptionName (Opt, "pragma", OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.Pragmas) := True;
                     ProcessPragma;
                  end if;
               when 'i' | 'I' =>
                  CheckOptionName (Opt, "private_types", OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.UnuseablePrivateTypes) := True;
                  end if;
               when others =>
                  null;
            end case;
         when 'h' | 'H' =>
            case Opt.Content (2) is
               when 'a' | 'A' =>
                  CheckOptionName (Opt, "handler_parts", OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.HandlerParts) := True;
                  end if;
               when 'i' | 'I' =>
                  CheckOptionName (Opt, "hidden_parts", OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.HiddenParts) := True;
                  end if;
               when others =>
                  null;
            end case;
         when 'r' | 'R' =>
            case Opt.Content (3) is
               when 'a' | 'A' =>
                  CheckOptionName (Opt, "real_rtcs", OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.RealRTCs) := True;
                  end if;
               when 'p' | 'P' =>
                  CheckOptionName (Opt, "representation_clauses", OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.RepresentationClauses) := True;
                  end if;
               when others =>
                  null;
            end case;
         when 'd' | 'D' =>
            case Opt.Content (2) is
               when 'i' | 'I' =>
                  CheckOptionName (Opt, "direct_updates", OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.DirectUpdates) := True;
                  end if;
               when 'e' | 'E' =>
                  case Opt.Content (3) is
                     when 'c' | 'C' =>
                        CheckOptionName (Opt, "declare_annotations", OptionMatch);
                        if OptionMatch then
                           SuppressedElement (ErrorHandler.DeclareAnnotations) := True;
                        end if;
                     when 'f' | 'F' =>
                        CheckOptionName (Opt, "default_loop_assertions", OptionMatch);
                        if OptionMatch then
                           SuppressedElement (ErrorHandler.DefaultLoopAssertions) := True;
                        end if;
                     when others =>
                        null; -- falls through with OptionMatch false and generates error
                  end case;
               when others =>
                  null;
            end case;
         when 'w' | 'W' =>
            CheckOptionName (Opt, "with_clauses", OptionMatch);
            if OptionMatch then
               SuppressedElement (ErrorHandler.WithClauses) := True;
            end if;
         when 's' | 'S' =>
            CheckOptionName (Opt, "static_expressions", OptionMatch);
            if OptionMatch then
               SuppressedElement (ErrorHandler.StaticExpressions) := True;
            end if;
         when 'u' | 'U' =>
            case Opt.Content (3) is
               when 'u' | 'U' =>
                  CheckOptionName (Opt, "unused_variables", OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.UnusedVariables) := True;
                  end if;
               when 'c' | 'C' =>
                  CheckOptionName (Opt, "unchecked_conversion", OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.UncheckedConversion) := True;
                  end if;
               when others =>
                  null; -- falls through with OptionMatch false and generates error
            end case;
         when 'c' | 'C' =>
            CheckOptionName (Opt, "constant_variables", OptionMatch);
            if OptionMatch then
               SuppressedElement (ErrorHandler.ConstantVariables) := True;
            end if;
         when 't' | 'T' =>
            CheckOptionName (Opt, "type_conversions", OptionMatch);
            if OptionMatch then
               SuppressedElement (ErrorHandler.TypeConversions) := True;
            end if;
         when 'n' | 'N' =>
            CheckOptionName (Opt, "notes", OptionMatch);
            if OptionMatch then
               SuppressedElement (ErrorHandler.Notes) := True;
            end if;
         when 'o' | 'O' =>
            case Opt.Content (2) is
               when 'b' | 'B' =>
                  CheckOptionName (Opt, "obsolescent_features", OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.ObsolescentFeatures) := True;
                  end if;
               when 't' | 'T' =>
                  CheckOptionName (Opt, "others_clauses", OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.OthersClauses) := True;
                  end if;
               when others =>
                  null;
            end case;
         when 'a' | 'A' =>
            case Opt.Content (3) is
               when 'a' | 'A' =>
                  CheckOptionName (Opt, "ada2005_reserved_words", OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.Ada2005ReservedWords) := True;
                  end if;
               when 'd' | 'D' =>
                  CheckOptionName (Opt, "address_clauses", OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.UnexpectedAddressClauses) := True;
                  end if;
               when others =>
                  null;
            end case;
         when 'e' | 'E' =>
            case Opt.Content (3) is
               when 'p' | 'P' =>
                  CheckOptionName (Opt, "expression_reordering", OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.ExpressionReordering) := True;
                  end if;
               when 't' | 'T' =>
                  CheckOptionName (Opt, "external_assignment", OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.ExternalVariableAssignment) := True;
                  end if;
               when others =>
                  null;
            end case;

         when 'i' | 'I' =>
            case Opt.Content (2) is
               when 'n' | 'N' =>
                  CheckOptionName (Opt, "interrupt_handlers", OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.InterruptHandlers) := True;
                  end if;
               when 'm' | 'M' =>
                  CheckOptionName (Opt, "imported_objects", OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.ImportedObjects) := True;
                  end if;
               when others =>
                  null;
            end case;
         when others =>
            null;
      end case;
      if not OptionMatch then
         InvalidOption (Opt);
      end if;
   end ProcessOption;

   -------------------------------

   procedure SortPragmas
   --# global in     SuppressAllPragmas;
   --#        in out PragmaList;
   --# derives PragmaList from *,
   --#                         SuppressAllPragmas;
   is
      J   : Integer;
      Val : LexTokenManager.LexString;

   begin
      if not SuppressAllPragmas and then
         PragmaList.PragmaCount /= 0
      then
         --insert sentinel to mark bounds of sort
         PragmaList.PragmaArray (PragmaList.PragmaCount + 1) :=
            LexTokenManager.ConvertLexStringRef (ExaminerConstants.StringTableSize);

         for I in reverse Integer range 1 .. PragmaList.PragmaCount
         loop
            J := I;
            Val := PragmaList.PragmaArray (J);
            while LexTokenManager.LexStringRef (PragmaList.PragmaArray (J + 1))
               < LexTokenManager.LexStringRef (Val)
            loop
               PragmaList.PragmaArray (J) := PragmaList.PragmaArray (J + 1);
               J := J + 1;
            end loop;
            PragmaList.PragmaArray (J) := Val;
         end loop;
      end if;
   end SortPragmas;

   -------------------------------

begin --ReadWarningFile
   if CommandLineData.Content.Warning then
      OpenFile;
      if FileOk then
         if CommandLineData.Content.Echo and not CommandLineData.Content.Brief then
            ScreenEcho.New_Line (1);
            ScreenEcho.Put_Line ("           Reading warning control file ...");
         end if;

         loop
            GetString (WarningFile,
                        --to get
                       Option);
            exit when Option.Length = 0;
            ProcessOption (Option);
         end loop;

         CloseFile;

         SortPragmas;

         for i in ErrorHandler.WarningElements loop
            SomethingSuppressed := SomethingSuppressed or
               SuppressedElement (i);
         end loop;
      else
         ErrorHandler.FileOpenError := True;
      end if;
   end if;
end ReadWarningFile;
