-- $Id: errorhandler-printerrors.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (ErrorHandler)
procedure PrintErrors (Listing : in SPARK_IO.File_Type;
                       Purpose : in Error_Types.ConversionRequestSource)
is
   ErrCount  : Natural;
   OK        : Boolean;
   NextError : Error_Types.StringError;
   NumErr    : Error_Types.NumericError;

   Accumulator : ErrorAccumulator.T := ErrorAccumulator.Clear;

   procedure CopySourceLines (ToFile : in SPARK_IO.File_Type;
                              LineNo : in LexTokenManager.LineNumbers)
   --# global in     CommandLineData.Content;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContextRec,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                ErrorContextRec,
   --#                                LineNo,
   --#                                SPARK_IO.File_Sys,
   --#                                ToFile;
   is
   begin
      if CommandLineData.Content.XML then
         loop
            exit when ErrorContextRec.LineNo >= LineNo;
            GetFileLine;
         end loop;
      else
         loop
            exit when ErrorContextRec.LineNo >= LineNo;
            GetFileLine;
            PrintSourceLine (ToFile);
         end loop;
      end if;
   end CopySourceLines;

   procedure CopySourceLineToEOF (ToFile : in SPARK_IO.File_Type)
   --# global in     CommandLineData.Content;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContextRec,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                ErrorContextRec,
   --#                                SPARK_IO.File_Sys,
   --#                                ToFile;
   is
   begin
      if not CommandLineData.Content.XML then
         loop
            exit when SPARK_IO.End_Of_File (ErrorContextRec.Source);
            GetFileLine;
            PrintSourceLine (ToFile);
         end loop;
      end if;
   end CopySourceLineToEOF;

   procedure GetErrorSet (ErrorSet  :    out ErrorSets;
                          NextError : in out Error_Types.StringError;
                          OK        :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     ErrorContextRec;
   --#        in     LexTokenManager.StringTable;
   --#        in     Purpose;
   --#        in out Conversions.State;
   --#        in out ErrCount;
   --#        in out SPARK_IO.File_Sys;
   --# derives Conversions.State,
   --#         NextError,
   --#         OK,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Conversions.State,
   --#                                Dictionary.Dict,
   --#                                ErrorContextRec,
   --#                                LexTokenManager.StringTable,
   --#                                Purpose,
   --#                                SPARK_IO.File_Sys &
   --#         ErrCount,
   --#         ErrorSet          from CommandLineData.Content,
   --#                                Conversions.State,
   --#                                Dictionary.Dict,
   --#                                ErrCount,
   --#                                ErrorContextRec,
   --#                                LexTokenManager.StringTable,
   --#                                NextError,
   --#                                Purpose,
   --#                                SPARK_IO.File_Sys;
   is
      LOK    : Boolean;
      Errors : ErrorSets;
      NumErr : Error_Types.NumericError;

   begin
      Errors := EmptyErrorSet;
      loop
         Errors.Length := Errors.Length + 1;
         Errors.Content (Errors.Length).Error := NextError;
         if NextError.ErrorType = Error_Types.NoErr or else
           ErrorAccumulator.IsErrorContinuation (NextError) then
            Errors.Content (Errors.Length).ErrNum := 0;
         else
            ErrCount := ErrCount + 1;
            Errors.Content (Errors.Length).ErrNum := ErrCount;
         end if;

         Error_IO.Get_Numeric_Error (ErrorContextRec.Errs, NumErr);
         Conversions.ToString (NumErr, Purpose, NextError);
         LOK := (NextError /= Error_Types.Empty_StringError);

         exit when not LOK;
         exit when ErrorContextRec.LineNo < NextError.Position.StartLineNo;
         exit when Errors.Length = ExaminerConstants.MaxErrorSetSize;
      end loop;

      OK := LOK;
      ErrorSet := Errors;
   end GetErrorSet;

   procedure ProcessErrorSet (Listing   : in     SPARK_IO.File_Type;
                              NextError : in out Error_Types.StringError;
                              OK        :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     ErrorContextRec;
   --#        in     LexTokenManager.StringTable;
   --#        in     Purpose;
   --#        in out Accumulator;
   --#        in out Conversions.State;
   --#        in out ErrCount;
   --#        in out SPARK_IO.File_Sys;
   --# derives Accumulator,
   --#         ErrCount          from *,
   --#                                CommandLineData.Content,
   --#                                Conversions.State,
   --#                                Dictionary.Dict,
   --#                                ErrCount,
   --#                                ErrorContextRec,
   --#                                LexTokenManager.StringTable,
   --#                                NextError,
   --#                                Purpose,
   --#                                SPARK_IO.File_Sys &
   --#         Conversions.State,
   --#         NextError,
   --#         OK                from CommandLineData.Content,
   --#                                Conversions.State,
   --#                                Dictionary.Dict,
   --#                                ErrorContextRec,
   --#                                LexTokenManager.StringTable,
   --#                                Purpose,
   --#                                SPARK_IO.File_Sys &
   --#         SPARK_IO.File_Sys from *,
   --#                                Accumulator,
   --#                                CommandLineData.Content,
   --#                                Conversions.State,
   --#                                Dictionary.Dict,
   --#                                ErrCount,
   --#                                ErrorContextRec,
   --#                                LexTokenManager.StringTable,
   --#                                Listing,
   --#                                NextError,
   --#                                Purpose;
   is
      ErrorSet : ErrorSets;
   begin
      GetErrorSet (ErrorSet, NextError, OK);
      if not CommandLineData.Content.XML then
         PutErrorPointers (Listing, ErrorSet);
      end if;
      PutErrorMessages (Listing, ErrorSet, 31, Accumulator);
   end ProcessErrorSet;

   procedure ProcessErrorsOnLine (Listing   : in     SPARK_IO.File_Type;
                                  NextError : in out Error_Types.StringError;
                                  OK        :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     ErrorContextRec;
   --#        in     LexTokenManager.StringTable;
   --#        in     Purpose;
   --#        in out Accumulator;
   --#        in out Conversions.State;
   --#        in out ErrCount;
   --#        in out SPARK_IO.File_Sys;
   --# derives Accumulator,
   --#         Conversions.State,
   --#         ErrCount,
   --#         NextError,
   --#         OK,
   --#         SPARK_IO.File_Sys from Accumulator,
   --#                                CommandLineData.Content,
   --#                                Conversions.State,
   --#                                Dictionary.Dict,
   --#                                ErrCount,
   --#                                ErrorContextRec,
   --#                                LexTokenManager.StringTable,
   --#                                Listing,
   --#                                NextError,
   --#                                Purpose,
   --#                                SPARK_IO.File_Sys;
   is
      LOK : Boolean;
      AccumulatorWasActive : Boolean;
   begin
      if not ErrorHasPositionInline (NextError.ErrorType) and then
        not CommandLineData.Content.XML then
         New_Line (Listing, 1);
      end if;
      loop
         ProcessErrorSet (Listing, NextError, LOK);
         exit when not LOK;
         exit when ErrorContextRec.LineNo < NextError.Position.StartLineNo;
      end loop;
      OK := LOK;
      AccumulatorWasActive := ErrorAccumulator.IsActive (Accumulator);
      ErrorAccumulator.Flush (Accumulator, Listing);
      if AccumulatorWasActive then
         New_Line (Listing, 1);
      end if;
   end ProcessErrorsOnLine;

   procedure SetUpFiles (OK : out Boolean)
   --# global in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#           out ErrCount;
   --# derives ErrCount          from  &
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys from *,
   --#                                ErrorContextRec &
   --#         OK                from ErrorContextRec,
   --#                                SPARK_IO.File_Sys;
   is
      LOK        : Boolean;
      Success    : SPARK_IO.File_Status;
      SourceFile : SPARK_IO.File_Type;
      ErrorFile  : Error_IO.File_Type;
   begin
      SourceFile := ErrorContextRec.Source;
      SPARK_IO.Reset (SourceFile, SPARK_IO.In_File, Success);
      ErrorContextRec.Source := SourceFile;
      LOK := Success = SPARK_IO.Ok;
      ErrorContextRec.LineNo := 0;
      ErrCount := 0;
      ErrorFile := ErrorContextRec.Errs;
      Error_IO.Reset (ErrorFile, SPARK_IO.In_File, Success);
      ErrorContextRec.Errs := ErrorFile;
      OK := LOK and Success = SPARK_IO.Ok;
      ErrorContextRec.CurrentLine.Length := 0;
   end SetUpFiles;

begin
   FlushBuffer;
   SetUpFiles (OK);
   if OK then
      if not CommandLineData.Content.XML then
         Put_Line (Listing, "Line", 0);
      end if;

      Error_IO.Get_Numeric_Error (ErrorContextRec.Errs, NumErr);
      Conversions.ToString (NumErr, Purpose, NextError);
      OK := (NextError /= Error_Types.Empty_StringError);

      loop
         exit when not OK;
         CopySourceLines (Listing, NextError.Position.StartLineNo);
         ProcessErrorsOnLine (Listing, NextError, OK);
         --# accept Flow, 41, "Expected stable expression";
         if not CommandLineData.Content.XML then
         --# end accept;
            New_Line (Listing, 1);
         end if;
      end loop;
      CopySourceLineToEOF (Listing);
   end if;
   Justifications.PrintJustifications (ErrorContextRec.JustificationsDataTable,
                                       Listing);

   WarningStatus.ReportSuppressedWarnings (Listing,
                                           ErrorContextRec.Counter);

end PrintErrors;
