-- $Id: errorhandler-echoerrorentry.adb 11889 2008-12-12 15:49:12Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

separate (ErrorHandler)
procedure EchoErrorEntry (EchoFile  : in SPARK_IO.File_Type;
                          Error     : in Error_Types.StringError)
is
   subtype BigStringI is Positive range 1 .. 1024;
   subtype BigString is String (BigStringI);

   SourceFileName : BigString;
   SourceFileStop : Natural;
   ErrStruct      : ErrorStruct;
   NewStart       : Natural;
   WasActive      : Boolean;
   LineLength     : Natural;
   NewLineCount   : Natural;
   LocalError     : Error_Types.StringError;
   Explanation    : ELStrings.T;

   procedure EchoSourceLine (EchoFile  : in SPARK_IO.File_Type;
                             ErrLineNo : in LexTokenManager.LineNumbers)
   --# global in     CommandLineData.Content;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContextRec   from *,
   --#                                ErrLineNo,
   --#                                SPARK_IO.File_Sys &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                EchoFile,
   --#                                ErrLineNo,
   --#                                ErrorContextRec;
   is
      SourceFile   : SPARK_IO.File_Type;
      Success      : SPARK_IO.File_Status;
   begin
      if ErrorContextRec.LineNo > ErrLineNo then
         ErrorContextRec.LineNo := 0;
         ErrorContextRec.CurrentLine.Length := 0;
         SourceFile := ErrorContextRec.Source;
         --# accept Flow, 10, Success, "Expected ineffective assignment to Success";
         SPARK_IO.Reset (SourceFile, SPARK_IO.In_File, Success);
         --# end accept;
         ErrorContextRec.Source := SourceFile;
      end if;
      if ErrLineNo > 0 and then
         ErrorContextRec.LineNo < ErrLineNo - 1
      then
         SPARK_IO.Skip_Line (ErrorContextRec.Source,
                             Integer ((ErrLineNo - ErrorContextRec.LineNo) - 1));
      end if;
      GetFileLine;
      ErrorContextRec.LineNo := ErrLineNo;
      New_Line (EchoFile, 1);
      PrintSourceLine (EchoFile);
      --# accept Flow, 33, Success, "Expected Success to be neither referenced nor exported";
   end EchoSourceLine;

   procedure PutErrorPointer (EchoFile : in SPARK_IO.File_Type;
                              ErrPos   : in Natural)
   --# global in     ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                EchoFile,
   --#                                ErrorContextRec,
   --#                                ErrPos;
   is
   begin
      MoveToIndent (EchoFile, ErrorContextRec.CurrentLine,
                    SourceLineIndent, ErrPos - 1);
      Put_Char (EchoFile, '^');
      New_Line (SPARK_IO.Standard_Output, 1);
   end PutErrorPointer;

   function NeedSourceLine (Error : Error_Types.StringError) return Boolean
   --# global in ErrorContextRec;
   is
   begin
      return Error.Position.StartLineNo /= ErrorContextRec.LineNo and
         Error.ErrorType /= Error_Types.NoErr;
   end NeedSourceLine;

begin

   if ErrorContextRec.Echo then

      if CommandLineData.Content.Brief then
         LineLength   := 0;
         NewLineCount := 1;
      else
         LineLength   := ErrorLineLength;
         NewLineCount := 1;
      end if;

      ErrStruct := ErrorStruct'(ErrNum => 0,
                                Error  => Error);

      WasActive := ErrorAccumulator.IsActive (EchoAccumulator);
      if ErrorAccumulator.IsActive (EchoAccumulator) then
         ErrorAccumulator.Add (EchoAccumulator, ErrStruct,
                               LineLength, 11, EchoFile);
      end if;

      if not ErrorAccumulator.IsActive (EchoAccumulator) then

         if WasActive then
            New_Line (EchoFile, NewLineCount);
         end if;

         if CommandLineData.Content.Brief then
            SPARK_IO.Name (ErrorContextRec.Source,
                           SourceFileName,
                           SourceFileStop);

            FileSystem.BaseName (SourceFileName, "", SourceFileStop);

            Put_String (EchoFile,
                        SourceFileName,
                        SourceFileStop);

            Put_Char (EchoFile, ':');
            Put_Integer (EchoFile, Integer (Error.Position.StartLineNo), 0, 10);
            Put_Char (EchoFile, ':');
            Put_Integer (EchoFile, Error.Position.StartPos, 0, 10);
            Put_Char (EchoFile, ':');
            Put_Char (EchoFile, ' ');
            OutputBriefErrorMarker (EchoFile,
                                    Error.ErrorType,
                                    Error.MessageId);

         else
            if NeedSourceLine (Error) then
               EchoSourceLine (EchoFile, Error.Position.StartLineNo);
            end if;
            if ErrorHasPositionInline (Error.ErrorType) then
               PutErrorPointer (EchoFile, Error.Position.StartPos);
            else
               New_Line (EchoFile, 1);
            end if;
            OutputErrorMarker (EchoFile,
                               Error.ErrorType,
                               Error.MessageId,
                               0);

         end if;

         if ErrorAccumulator.IsErrorStart (Error) then
            -- extract explanation  from message here and pass it into ErrorAccumulator for later Flush --PNA
            LocalError := Error;
            SplitStringAtExplanation (LocalError.Message,
                                      Explanation);

            PrintLine (EchoFile,
                       31,
                       LineLength,
                       11,
                       LocalError.Message,
                       False,
                       NewStart);

            ErrorAccumulator.StartMsg (EchoAccumulator,
                                       ErrStruct,
                                       NewStart,
                                       Explanation, -- Explanation extracted by Split... above
                                       LineLength, -- we know this and next thing in this scope so pass them
                                       11);  -- in rather than recalculate them in error accumulator
         else
            -- leave message text unchanged and display it
            --# accept Flow, 10, NewStart, "Expected ineffective assignment to NewStart";
            PrintLine (EchoFile, 31, LineLength, 11, -- Expect ineff assignment to NewStart
                       Error.Message, False, NewStart);
            --# end accept;
            New_Line (EchoFile, NewLineCount);
         end if;
      end if;

   end if;


end EchoErrorEntry;
