-- $Id: errorhandler-conversions-tostring.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (ErrorHandler.Conversions)
procedure ToString (ErrNum  : in     Error_Types.NumericError;
                    Purpose : in     Error_Types.ConversionRequestSource;
                    ErrStr  :    out Error_Types.StringError)
is
   ErrorString       : ELStrings.T;
   ExplanationNeeded : Boolean;

   procedure AppendReference (EStr      : in out ELStrings.T;
                              Reference : in     Natural)
   --# global in     CommandLineData.Content;
   --#        in out SourceUsed;
   --# derives EStr,
   --#         SourceUsed from *,
   --#                         CommandLineData.Content,
   --#                         Reference;
      is
      separate;

   ------------------------------------------------------------

   procedure AppendLexString (EStr : in out ELStrings.T;
                              LStr : in     LexTokenManager.LexString)
   --# global in LexTokenManager.StringTable;
   --# derives EStr from *,
   --#                   LexTokenManager.StringTable,
   --#                   LStr;
   is
      pragma Inline (AppendLexString);
      Str : EStrings.T;
   begin
      LexTokenManager.LexStringToString (LStr, Str);
      ELStrings.AppendExaminerString (EStr, Str);
   end AppendLexString;

   ------------------------------------------------------------

   procedure AppendSymbol (EStr  : in out ELStrings.T;
                           Sym   : in     Dictionary.Symbol;
                           Scope : in     Dictionary.Scopes)
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --#        in LexTokenManager.StringTable;
   --# derives EStr from *,
   --#                   CommandLineData.Content,
   --#                   Dictionary.Dict,
   --#                   LexTokenManager.StringTable,
   --#                   Scope,
   --#                   Sym;
   is
      PackageExStr   : EStrings.T;
      ExStr          : EStrings.T;

      ---------------------------------------------

   begin --AppendSymbol
      -- put Access on the front of access types before constructing rest of string
      if Dictionary.IsType (Sym) and then Dictionary.TypeIsAccess (Sym) then
         ELStrings.AppendExaminerString
           (EStr,
            EStrings.T'
              (Length => 7,
               Content => EStrings.Contents'('A', 'c', 'c', 'e', 's', 's', ' ', others => ' ')));
      end if;
      -- construct rest of string
      Dictionary.GetAnyPrefixNeeded (Sym, Scope, ".", PackageExStr);
      Dictionary.GenerateSimpleName (Sym, ".", ExStr);
      if PackageExStr.Length > 0 then
         ELStrings.AppendExaminerString (EStr, PackageExStr);
         ELStrings.AppendString (EStr, ".");
      end if;
      ELStrings.AppendExaminerString (EStr, ExStr);
   end AppendSymbol;

   ------------------------------------------------------------------------

   procedure AppendName (EStr  : in out ELStrings.T;
                         Name  : in     Error_Types.Names;
                         Scope : in     Dictionary.Scopes)
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --#        in LexTokenManager.StringTable;
   --# derives EStr from *,
   --#                   CommandLineData.Content,
   --#                   Dictionary.Dict,
   --#                   LexTokenManager.StringTable,
   --#                   Name,
   --#                   Scope;
   is
   begin
      case Name.NameSort is
         when Error_Types.None           => null;
         when Error_Types.LexString      => null;
            AppendLexString (EStr,
                             LexTokenManager.ConvertLexStringRef (Name.NameValue));
         when Error_Types.Entity         => null;
         when Error_Types.Symbol         =>
            AppendSymbol (EStr,
                          Dictionary.ConvertSymbolRef (ExaminerConstants.RefType (Name.NameValue)),
                          Scope);
         when Error_Types.ParserSymbol   => null;
         when Error_Types.StabilityIndex => null;
         when Error_Types.ThePartition   => null;
      end case;
   end AppendName;

   ------------------------------------------------------------

   procedure AppendExportVar (EStr       : in out ELStrings.T;
                              Name       : in     Error_Types.Names;
                              Scope      : in     Dictionary.Scopes;
                              Capitalise : in     Boolean)
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --#        in LexTokenManager.StringTable;
   --# derives EStr from *,
   --#                   Capitalise,
   --#                   CommandLineData.Content,
   --#                   Dictionary.Dict,
   --#                   LexTokenManager.StringTable,
   --#                   Name,
   --#                   Scope;
   is
   begin
      if Name = Error_Types.NoName then
         if Capitalise then
            ELStrings.AppendString (EStr, "T");
         else
            ELStrings.AppendString (EStr, "t");
         end if;
         ELStrings.AppendString (EStr, "he function value");
      else
         AppendName (EStr, Name, Scope);
      end if;
   end AppendExportVar;

   ------------------------------------------------------------

   procedure UncondFlowErr (ErrNum          : in     Error_Types.NumericError;
                            WithExplanation : in     Boolean;
                            EStr            : in out ELStrings.T)
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --#        in LexTokenManager.StringTable;
   --# derives EStr from *,
   --#                   CommandLineData.Content,
   --#                   Dictionary.Dict,
   --#                   ErrNum,
   --#                   LexTokenManager.StringTable,
   --#                   WithExplanation;
      is separate;

   ------------------------------------------------------------

   procedure CondlFlowErr (ErrNum          : in     Error_Types.NumericError;
                           WithExplanation : in     Boolean;
                           EStr            : in out ELStrings.T)
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --#        in LexTokenManager.StringTable;
   --# derives EStr from *,
   --#                   CommandLineData.Content,
   --#                   Dictionary.Dict,
   --#                   ErrNum,
   --#                   LexTokenManager.StringTable,
   --#                   WithExplanation;
      is separate;

   ------------------------------------------------------------

   procedure UncondDependency (ErrNum          : in     Error_Types.NumericError;
                               WithExplanation : in     Boolean;
                               EStr            : in out ELStrings.T)
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --#        in LexTokenManager.StringTable;
   --# derives EStr from *,
   --#                   CommandLineData.Content,
   --#                   Dictionary.Dict,
   --#                   ErrNum,
   --#                   LexTokenManager.StringTable,
   --#                   WithExplanation;
      is separate;

   ------------------------------------------------------------

   procedure CondlDependency (ErrNum          : in     Error_Types.NumericError;
                              WithExplanation : in     Boolean;
                              EStr            : in out ELStrings.T)
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --#        in LexTokenManager.StringTable;
   --# derives EStr from *,
   --#                   CommandLineData.Content,
   --#                   Dictionary.Dict,
   --#                   ErrNum,
   --#                   LexTokenManager.StringTable,
   --#                   WithExplanation;
      is separate;

   ------------------------------------------------------------

   procedure SemanticErr (ErrNum          : in     Error_Types.NumericError;
                          WithExplanation : in     Boolean;
                          EStr            : in out ELStrings.T)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SourceUsed;
   --# derives EStr       from *,
   --#                         CommandLineData.Content,
   --#                         Dictionary.Dict,
   --#                         ErrNum,
   --#                         LexTokenManager.StringTable,
   --#                         WithExplanation &
   --#         SourceUsed from *,
   --#                         CommandLineData.Content,
   --#                         ErrNum;
      is separate;

   ------------------------------------------------------------

   procedure DepSemanticErr (ErrNum          : in     Error_Types.NumericError;
                             WithExplanation : in     Boolean;
                             EStr            : in out ELStrings.T)
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --#        in LexTokenManager.StringTable;
   --# derives EStr from *,
   --#                   CommandLineData.Content,
   --#                   Dictionary.Dict,
   --#                   ErrNum,
   --#                   LexTokenManager.StringTable,
   --#                   WithExplanation;
      is separate;

   ------------------------------------------------------------

   procedure WarningWithPosition (ErrNum          : in     Error_Types.NumericError;
                                  WithExplanation : in     Boolean;
                                  EStr            : in out ELStrings.T)
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --#        in LexTokenManager.StringTable;
   --# derives EStr from *,
   --#                   CommandLineData.Content,
   --#                   Dictionary.Dict,
   --#                   ErrNum,
   --#                   LexTokenManager.StringTable,
   --#                   WithExplanation;
      is separate;

   ------------------------------------------------------------

   procedure WarningWithoutPosition (ErrNum          : in     Error_Types.NumericError;
                                     WithExplanation : in     Boolean;
                                     EStr            : in out ELStrings.T)
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --#        in LexTokenManager.StringTable;
   --# derives EStr from *,
   --#                   CommandLineData.Content,
   --#                   Dictionary.Dict,
   --#                   ErrNum,
   --#                   LexTokenManager.StringTable,
   --#                   WithExplanation;
      is separate;

   ------------------------------------------------------------

   procedure ControlFlowError (ErrNum          : in     Error_Types.NumericError;
                               WithExplanation : in     Boolean;
                               EStr            : in out ELStrings.T)
   --# derives EStr from *,
   --#                   ErrNum,
   --#                   WithExplanation;
      is separate;

   ------------------------------------------------------------

   procedure IneffectiveStatement (ErrNum          : in     Error_Types.NumericError;
                                   WithExplanation : in     Boolean;
                                   EStr            : in out ELStrings.T)
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --#        in LexTokenManager.StringTable;
   --# derives EStr from *,
   --#                   CommandLineData.Content,
   --#                   Dictionary.Dict,
   --#                   ErrNum,
   --#                   LexTokenManager.StringTable,
   --#                   WithExplanation;
      is separate;

   ------------------------------------------------------------

   procedure StabilityError (ErrNum          : in     Error_Types.NumericError;
                             WithExplanation : in     Boolean;
                             EStr            : in out ELStrings.T)
   --# derives EStr from *,
   --#                   ErrNum,
   --#                   WithExplanation;
      is separate;

   ------------------------------------------------------------

   procedure UsageError (ErrNum          : in     Error_Types.NumericError;
                         WithExplanation : in     Boolean;
                         EStr            : in out ELStrings.T)
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --#        in LexTokenManager.StringTable;
   --# derives EStr from *,
   --#                   CommandLineData.Content,
   --#                   Dictionary.Dict,
   --#                   ErrNum,
   --#                   LexTokenManager.StringTable,
   --#                   WithExplanation;
      is separate;

   ------------------------------------------------------------

   procedure NoErr (ErrNum : in     Error_Types.NumericError;
                    EStr   : in out ELStrings.T)
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --#        in LexTokenManager.StringTable;
   --# derives EStr from *,
   --#                   CommandLineData.Content,
   --#                   Dictionary.Dict,
   --#                   ErrNum,
   --#                   LexTokenManager.StringTable;
      is separate;

   ------------------------------------------------------------
   procedure Note (ErrNum          : in     Error_Types.NumericError;
                   WithExplanation : in     Boolean;
                   EStr            : in out ELStrings.T)
   --# derives EStr from *,
   --#                   ErrNum,
   --#                   WithExplanation;
      is separate;

   ------------------------------------------------------------

   procedure SyntaxOrLexError (ErrNum : in     Error_Types.NumericError;
                               EStr   : in out ELStrings.T)
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --#        in LexTokenManager.StringTable;
   --# derives EStr from *,
   --#                   CommandLineData.Content,
   --#                   Dictionary.Dict,
   --#                   ErrNum,
   --#                   LexTokenManager.StringTable;
   is
   begin
      -- When a syntax error has been constructed the entire text string of the error
      -- gets put into the string table and included as Name1 in the numeric form of the
      -- error record.  Conversion back to a string just needs the following:
      AppendName (EStr, ErrNum.Name1, ErrNum.Scope);
   end SyntaxOrLexError;

   ------------------------------------------------------------

   procedure CheckExplanation (ExplanationClass  : in     ExplanationClasses;
                               ErrorNumber       : in     Error_Types.ErrNumRange;
                               Purpose           : in     Error_Types.ConversionRequestSource;
                               ExplanationNeeded :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in out ExplanationTable;
   --# derives ExplanationNeeded,
   --#         ExplanationTable  from CommandLineData.Content,
   --#                                ErrorNumber,
   --#                                ExplanationClass,
   --#                                ExplanationTable,
   --#                                Purpose;
   is
   begin
      -- In general, explanation depend on command line switch setting and whether that explanation
      -- has appeared before.
      case CommandLineData.Content.ErrorExplanation is
         when CommandLineData.Off =>
            ExplanationNeeded := False;
         when CommandLineData.FirstOccurrence =>
            ExplanationNeeded := not ExplanationTable (ExplanationClass)(ErrorNumber)(Purpose);
            ExplanationTable (ExplanationClass)(ErrorNumber)(Purpose) := True;
         when CommandLineData.EveryOccurrence =>
            ExplanationNeeded := True;
      end case;
      -- But we also have a special case where we turn explanations off if (HTML and (Purpose=ForReport)).
      -- This is because explanations are only a click away when looking at HTML report files, so why clutter up screen?
      if CommandLineData.Content.HTML and then Purpose in Error_Types.ForReport then
         ExplanationNeeded := False;
      end if;
      -- We also turn it off for XML generation, at least for now.
      if CommandLineData.Content.XML then
         ExplanationNeeded := False;
      end if;
   end CheckExplanation;

   ------------------------------------------------------------

begin --ToString
   if ErrNum = Error_Types.Empty_NumericError then
      ErrStr := Error_Types.Empty_StringError;
   else
      ErrorString := ELStrings.EmptyString;
      case ErrNum.ErrorType is
         when Error_Types.UncondFlowErr          =>
            CheckExplanation (FlowErrors,
                              ErrNum.ErrorNum,
                              Purpose,
                              -- to get
                              ExplanationNeeded);
            UncondFlowErr (ErrNum, ExplanationNeeded, ErrorString);
         when Error_Types.CondlFlowErr           =>
            CheckExplanation (FlowErrors,
                              ErrNum.ErrorNum,
                              Purpose,
                              -- to get
                              ExplanationNeeded);
            CondlFlowErr (ErrNum, ExplanationNeeded, ErrorString);
         when Error_Types.UncondDependencyErr    =>
            CheckExplanation (DependencyErrs,
                              ErrNum.ErrorNum,
                              Purpose,
                              -- to get
                              ExplanationNeeded);
            UncondDependency (ErrNum, ExplanationNeeded, ErrorString);
         when Error_Types.CondlDependencyErr     =>
            CheckExplanation (DependencyErrs,
                              ErrNum.ErrorNum,
                              Purpose,
                              -- to get
                              ExplanationNeeded);
            CondlDependency (ErrNum, ExplanationNeeded, ErrorString);
         when Error_Types.SemanticErr            =>
            CheckExplanation (SemanticErrs,
                              ErrNum.ErrorNum,
                              Purpose,
                              -- to get
                              ExplanationNeeded);
            SemanticErr (ErrNum, ExplanationNeeded, ErrorString);
         when Error_Types.DepSemanticErr         =>
            CheckExplanation (DepSemanticErrs,
                              ErrNum.ErrorNum,
                              Purpose,
                              -- to get
                              ExplanationNeeded);
            DepSemanticErr (ErrNum, ExplanationNeeded, ErrorString);
         when Error_Types.WarningWithPosition    =>
            CheckExplanation (Warnings,
                              ErrNum.ErrorNum,
                              Purpose,
                              -- to get
                              ExplanationNeeded);
            WarningWithPosition (ErrNum, ExplanationNeeded, ErrorString);
         when Error_Types.WarningWithoutPosition =>
            CheckExplanation (Warnings,
                              ErrNum.ErrorNum,
                              Purpose,
                              -- to get
                              ExplanationNeeded);
            WarningWithoutPosition (ErrNum, ExplanationNeeded, ErrorString);
         when Error_Types.ControlFlowErr         =>
            CheckExplanation (ControlFlows,
                              ErrNum.ErrorNum,
                              Purpose,
                              -- to get
                              ExplanationNeeded);
            ControlFlowError (ErrNum, ExplanationNeeded, ErrorString);
         when Error_Types.NoErr                  =>
            NoErr (ErrNum, ErrorString);
         when Error_Types.IneffectiveStat        =>
            CheckExplanation (IneffectiveStatements,
                              ErrNum.ErrorNum,
                              Purpose,
                              -- to get
                              ExplanationNeeded);
            IneffectiveStatement (ErrNum, ExplanationNeeded, ErrorString);
         when Error_Types.StabilityErr           =>
            CheckExplanation (FlowErrors,
                              ErrNum.ErrorNum,
                              Purpose,
                              -- to get
                              ExplanationNeeded);
            StabilityError (ErrNum, ExplanationNeeded, ErrorString);
         when Error_Types.UsageErr               =>
            CheckExplanation (FlowErrors,
                              ErrNum.ErrorNum,
                              Purpose,
                              -- to get
                              ExplanationNeeded);
            UsageError (ErrNum, ExplanationNeeded, ErrorString);
         when Error_Types.Note                   =>
            CheckExplanation (Notes,
                              ErrNum.ErrorNum,
                              Purpose,
                              -- to get
                              ExplanationNeeded);
            Note (ErrNum, ExplanationNeeded, ErrorString);
         when Error_Types.SyntaxErr |
           Error_Types.LexErr |
           Error_Types.SyntaxRec =>
            SyntaxOrLexError (ErrNum, ErrorString);
      end case;

      ErrStr := Error_Types.StringError'(ErrorType => ErrNum.ErrorType,
                                         Position  => ErrNum.Position,
                                         Message   => ErrorString,
                                         MessageId => ErrNum.ErrorNum);
   end if;
end ToString;
