-- $Id: errorhandler-conversions-tostring-semanticerr.adb 12530 2009-02-20 09:43:11Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (ErrorHandler.Conversions.ToString)
procedure SemanticErr (ErrNum          : in     Error_Types.NumericError;
                       WithExplanation : in     Boolean;
                       EStr            : in out ELStrings.T)
is
   procedure SemanticErrExpl (EStr   : in out ELStrings.T)
   --# global in ErrNum;
   --# derives EStr from *,
   --#                   ErrNum;
      is separate;
   -- Note that the parameter names for this subunit are chosen to make it as easy as
   --      possible to auto-generate the subunit from this, its parent, file.  The
   --      generation requires copying the case statement below, stripping out the
   --      current Append'Thing' statements and adding an AppendString for the
   --      explanatory text that is delineated by --! comments.

   procedure AppendExplanation
   --# global in     ErrNum;
   --#        in     WithExplanation;
   --#        in out EStr;
   --# derives EStr from *,
   --#                   ErrNum,
   --#                   WithExplanation;
   is
      ExplanationString : ELStrings.T := ELStrings.EmptyString;
   begin
      if WithExplanation then
         -- we need to at least look for an explanation
         SemanticErrExpl (ExplanationString);
         if ExplanationString.Length > 0 then
            -- there actually is one
            ELStrings.AppendString (EStr, ErrorHandler.ExplanationPrefix);
            ELStrings.AppendExaminerLongString (EStr, ExplanationString);
            ELStrings.AppendString (EStr, ErrorHandler.ExplanationPostfix);
         end if;
      end if;
   end AppendExplanation;

begin
   -- Directives for HTML output:

   --! <NameFormat> <"semantic-"><Name>
   --! <ErrorFormat> <"*** Semantic Error : "><Name><" : "><Error>

   case ErrNum.ErrorNum is
      when 1 =>
         ELStrings.AppendString
            (EStr, "The identifier ");
         if ErrNum.Name2 /= Error_Types.NoName then
            AppendName
               (EStr, ErrNum.Name2, ErrNum.Scope);
            ELStrings.AppendString
               (EStr, ".");
         end if;
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is either undeclared or not visible at this point");
         --! If the identifier is declared in a
         --! separate (or parent) package, the package must be included in an inherit clause
         --! and the identifier prefixed with the package name.
         --! Ensure that there are no errors in the declaration of the identifier

      when 2 =>
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " does not denote a formal parameter for ");
         AppendName
           (EStr, ErrNum.Name2, ErrNum.Scope);

      when 3 =>
         ELStrings.AppendString
            (EStr,
             "Incorrect number of actual parameters for call of subprogram ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);

      when 4 =>
         ELStrings.AppendString
            (EStr,
             "More than one parameter association is " &
             "given for formal parameter ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);

      when 5 =>
         ELStrings.AppendString
            (EStr, "Illegal use of identifier ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         --! Usually associated with the use of an identifier other than a package
         --! name as a prefix in a selected component.

      when 6 =>
         ELStrings.AppendString
            (EStr, "Identifier ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is not the name of a variable");

      when 7 =>
         ELStrings.AppendString
            (EStr, "Identifier ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is not the name of a procedure");

      when 8 =>
         ELStrings.AppendString
            (EStr, "There is no field named ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " in this entity");
         --! Issued when the selector in a selected component of a record
         --! references a non-existent field.

      when 9 =>
         ELStrings.AppendString
            (EStr, "Selected components are not allowed for ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         --! Occurs if the prefix to a selected component representing a procedure
         --! in a procedure call statement or a type mark is not a package. Also
         --! occurs if a selector is applied in an expression to an object which
         --! is not a record variable.

      when 10 =>
         ELStrings.AppendString
            (EStr, "Illegal redeclaration of identifier ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);

      when 11 =>
         ELStrings.AppendString
            (EStr, "There is no package declaration for ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         --! Issued if a package body is encountered for which there is no
         --! package specification.

      when 12 =>
         ELStrings.AppendString
            (EStr, "Own variable ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " can only be completed by a variable declaration, not a constant");
         --! If the object in question is really a constant, then remove it from
         --! the enclosing package's own variable annotation.

      when 13 =>
         ELStrings.AppendString
            (EStr, "A body for subprogram ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " has already been declared");

      when 14 =>
         ELStrings.AppendString
            (EStr, "Illegal parent unit name");
         --! Issued if the name in a &quot;separate&quot; clause of a subunit does not
         --! correctly identify a compilation unit.  Common causes of this error
         --! are a syntax error in the parent unit or omitting the
         --! parent unit specification and/or parent unit body entries from the index file

      when 15 =>
         ELStrings.AppendString
            (EStr, "The stub for ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is either undeclared or cannot be located");
         --! Common causes of this error are an error in the declaration
         --! of the stub or the omission of the parent unit body from the index file

      when 16 =>
         ELStrings.AppendString
            (EStr, "A body for package ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " has already been declared");

      when 17 =>
         ELStrings.AppendString
            (EStr, "A body stub for package ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " has already been declared");

      when 18 =>
         ELStrings.AppendString
            (EStr, "Identifier ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is not the name of a package");

      when 19 =>
         ELStrings.AppendString
            (EStr, "Identifier ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is not the name of a procedure");

      when 20 =>
         ELStrings.AppendString
            (EStr, "Illegal operator symbol");
         --! Issued if a renaming declaration contains a non-existent operator.

      when 21 =>
         ELStrings.AppendString
            (EStr, "This entity is not an array");
         --! Issued if an attempt is made to index into a name which does not
         --! represent an array.

      when 22 =>
         ELStrings.AppendString
            (EStr,
             "The type in this declaration is not consistent " &
             "with the previous declaration of ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         --! Occurs when the type given in the Ada declaration of an own variable
         --! differs from that
         --! &quot;announced&quot; in the package's own variable clause.

      when 23 =>
         ELStrings.AppendString
            (EStr,
             "No parameter association is given for formal parameter ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);

      when 24 =>
         ELStrings.AppendString
            (EStr, "The identifier ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " (exported by called subprogram) is not visible at this point");
         --! When a procedure is called any global variables exported by that
         --! procedure must be visible at the point of call. This error message
         --! indicates that the global variable concerned is not visible.
         --! It may be that it needs to be added to the global annotation of
         --! the procedure containing the call (or some further enclosing subprogram)
         --! or it may be that an inherit clause is missing from the package containing
         --! the call.

      when 25 =>
         ELStrings.AppendString
            (EStr, "The identifier ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " (imported by called subprogram) is not visible at this point");
         --! When a procedure is called any global variables imported by that
         --! procedure must be visible at the point of call. This error message
         --! indicates that the global variable concerned is not visible.
         --! It may be that it needs to be added to the global annotation of
         --! the subprogram containing the call (or some further enclosing subprogram)
         --! or it may be that an inherit clause is missing from the package containing
         --! the call.

      when 26 =>
         ELStrings.AppendString
            (EStr, "The deferred constant ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " does not have an associated full definition");
         --! Issued at the end of a package specification if no full declaration
         --! has been supplied for a deferred constant declared in the package
         --! specification.

      when 27 =>
         ELStrings.AppendString
            (EStr, "The private type ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " does not have an associated full definition");
         --! Issued at the end of a package specification if no full declaration
         --! has been supplied for a private type declared in the package specification.

      when 28 =>
         ELStrings.AppendString
            (EStr, "The own variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " does not have a definition");
         --! Issued at the end of a package body if an own variable announced in the
         --! package specification has neither been given an Ada declaration nor refined.

      when 29 =>
         ELStrings.AppendString
            (EStr, "The subprogram ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             ", declared in the package specification, " &
             "does not have an associated body");
      when 30 =>
         ELStrings.AppendString
            (EStr, "Attribute ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is not yet implemented in the Examiner");
         --! The attribute is identified in Annex K of the SPARK 95 report as a
         --! valid SPARK 95
         --! attribute but the Examiner does not currently support it.
         --! It is
         --! possible to work round the omission by putting the use of the attribute
         --! inside a
         --! suitable function which is hidden from the Examiner.

      when 31 =>
         ELStrings.AppendString
            (EStr,
             "The prefix of this attribute is not an object or type");

      when 32 =>
         ELStrings.AppendString
            (EStr, "Illegal type conversion");
         --! Likely causes are type conversions involving record types or
         --! non-convertible arrays.

      when 33 =>
         ELStrings.AppendString
            (EStr, "Illegal aggregate");
         --! Issued if the prefix of an aggregate is not a composite type.

      when 34 =>
         ELStrings.AppendString
            (EStr, "Illegal procedure call");
         --! Issued if a call is made to a user-defined subprogram in a
         --! package initialization part.

      when 35 =>
         ELStrings.AppendString
            (EStr, "Operator is not declared for these types");
         --! Indicates use of an undeclared operator; this message means that
         --! the type on each side
         --! of the operator cannot appear with the operator used. e.g.
         --! attempting to add an integer to
         --! an enumeration literal.

      when 36 =>
         ELStrings.AppendString
           (EStr, "Expression is not static");

      when 37 =>
         ELStrings.AppendString
           (EStr, "Expression is not constant");

      when 38 =>
         ELStrings.AppendString
           (EStr, "Expression is not of the expected type");

      when 39 =>
         ELStrings.AppendString
            (EStr, "Illegal use of unconstrained type");
         --! An unconstrained array type or variable of such a type is
         --! illegally used. Use of
         --! unconstrained arrays in SPARK is limited to passing them as
         --! parameters, indexing into them
         --! and taking attributes of them.  This message also arises if a string
         --! literal is used as an actual parameter where the formal parameter is a string subtype.
         --! In this case, the error can be removed by qualifying the string literal with the subtype
         --! name.

      when 40 =>
         ELStrings.AppendString
            (EStr, "Numeric or Time_Span type required");
         --! This operator is only defined for numeric types and, if the
         --! Ravenscar Profile is
         --! selected, for type Ada.Real_Time.Time_Span.

      when 41 =>
         ELStrings.AppendString
            (EStr, "Array type required");
         --! Issued if a subtype declaration taking the form of a constrained
         --! subtype of an
         --! unconstrained array type is encountered but with a type mark which
         --! does not represent an
         --! array.

      when 42 =>
         ELStrings.AppendString
            (EStr, "Incompatible types");
         --! Issued when a name represents an object which is not of the required type.

      when 43 =>
         ELStrings.AppendString
           (EStr, "Range is not constant");

      when 44 =>
         ELStrings.AppendString
            (EStr, "Scalar type required");
         --! The bounds of an explicit range must be scalar types.

      when 45 =>
         ELStrings.AppendString
           (EStr, "Range is not static");

      when 46 =>
         ELStrings.AppendString
           (EStr, "Discrete type required");

      when 47 =>
         ELStrings.AppendString
            (EStr, "The definition of this type contains errors which may make this array definition invalid");
         --! Issued if an array type definition is encountered where one or
         --! more of the index types used in the definition contained errors in its original declaration.  For
         --! example, SPARK requires array index bounds to be constant (known at compile time) so an attempt
         --! to use an illegal subtype with variable bounds as an array index will generate this message.

      when 48 =>
         ELStrings.AppendString
            (EStr, "Subtypes of private types are not permitted");
         --! Issued if an attempt is made to declare a subtype of a private type in a location where the full
         --! view of the type is not visible.

      when 49 =>
         ELStrings.AppendString
            (EStr, "Attribute ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " takes only one argument");
         --! Only SPARK 95 attributes 'Min and 'Max require two arguments.

      when 50 =>
         ELStrings.AppendString
           (EStr, "Initializing expression must be constant");
         --! To assign a non-constant expression to a variable, an assignment statement
         --! in the body of the program unit (following the 'begin') must be used.

      when 51 =>
         ELStrings.AppendString
            (EStr, "Arrays may not be ordered");
         --! Issued if an ordering operator such as &quot;&lt;&quot; is encountered between
         --! objects of an array
         --! type other than string or a constrained subtype of string.

      when 52 =>
         ELStrings.AppendString
            (EStr, "Only Scalar, String and Time types may be ordered");
         --! Ordering operators are only defined for scalar types and type
         --! String plus, if the
         --! Ravenscar Profile is selected, types Time and Time_Span in package
         --! Ada.Real_Time.

      when 53 =>
         ELStrings.AppendString
            (EStr, "Illegal others clause");
         --! In SPARK record aggregates may not contain an others clause.

      when 54 =>
         ELStrings.AppendString
            (EStr, "Illegal attribute: ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         --! Issued when an attribute not supported by SPARK is used.

      when 55 =>
         ELStrings.AppendString
            (EStr, "Attribute ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " takes no argument");

      when 56 =>
         ELStrings.AppendString
           (EStr, "Argument expected");

      when 57 =>
         ELStrings.AppendString
            (EStr,
             "Fixed type definition must have associated range constraint");

      when 58 =>
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " expected, to repeat initial identifier");
         --! Occurs at the end of a package, subprogram, protected type, task type
         --! or loop if the terminal identifier does not
         --! match the name or label originally given.

      when 59 =>
         ELStrings.AppendString
            (EStr,
             "Composite subtype definition may not have " &
             "associated range constraint");
         --! A subtype of the form applicable to a subrange of a scalar type has
         --! been encountered
         --! but the type provided is not a scalar type.

      when 60 =>
         ELStrings.AppendString
            (EStr, "Illegal choice in record aggregate");
         --! In SPARK record aggregates may not contain multiple choices, each
         --! field must be
         --! assigned a value individually.

      when 61 =>
         ELStrings.AppendString
            (EStr,
             "Illegal occurrence of body stub - a body stub may only occur " &
               "in a compilation unit");

      when 62 =>
         ELStrings.AppendString
            (EStr, "A body for the embedded package ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is required");
         --! Issued if an embedded package declares subprograms or own variables
         --! and no body is
         --! provided.

      when 63 =>
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is not a type mark");

      when 64 =>
         ELStrings.AppendString
            (EStr, "Parameters of function subprograms " &
               "must be of mode in");

      when 65 =>
         ELStrings.AppendString
            (EStr, "Formal parameters of renamed operators " &
             "may not be renamed");
         --! The names of the parameters used in renaming declarations may not
         --! be altered from Left,
         --! Right for binary operators and Right for unary operators. These are
         --! the names given for
         --! the parameters in the ARM and the SPARK Definition requires that
         --! parameter names are not
         --! changed.

      when 66 =>
         ELStrings.AppendString
            (EStr, "Unexpected package initialization - no own variables of package ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " require initialization");
         --! Either the package does not have an initializes annotation or all
         --! the own variables
         --! requiring initialization were given values at the point of declaration.

      when 67 =>
         ELStrings.AppendString
           (EStr, "Illegal machine code insertion. Machine code functions are not permitted in SPARK 83.");
         --! This is an Ada 83 rule.  Machine code can only be used in procedures.

      when 68 =>
         ELStrings.AppendString
            (EStr, "Illegal operator renaming - operators are " &
             "defined on types not subtypes");
         --! Issued if an attempt is made to rename an operator using a subtype
         --! of the type for
         --! which it was originally implicitly declared.

      when 69 =>
         ELStrings.AppendString
            (EStr, "pragma ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " has two parameters");

      when 70 =>
         ELStrings.AppendString
            (EStr, "pragma ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " expected");
         --! <Error> pragma Import expected.
         --! <Error2> pragma Interface expected.

      when 71 =>
         ELStrings.AppendString
            (EStr, "This expression does not represent the " &
             "expected subprogram or variable name ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         --! Issued if the name supplied in a pragma interface, import or attach_handler
         --! does not match the name of the associated subprogram or variable.

      when 72 =>
         ELStrings.AppendString
            (EStr, "Unexpected pragma ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         --! <Error> Unexpected pragma Import.
         --! Pragma import may only occur in a body stub, or immediately
         --! after a subprogram declaration in the visible part of a package,
         --! or immediately after a variable declaration.

         --! <Name> 72
         --! <Error> Unexpected pragma Interface.
         --! Pragma interface may only occur in a body stub or immediately
         --! after a subprogram declaration in the visible part of a package.

      when 73 =>
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " has already been declared or refined");
         --! Issued if an Ada declaration is given for an own variable which
         --! has been refined, or in
         --! a refinement clause if an own variable is refined more than once.

      when 74 =>
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " does not occur in the package own variable list");
         --! A subject of a refinement definition of a package must be an own
         --! variable of that
         --! package.

      when 75 =>
         ELStrings.AppendString
            (EStr,
             "Illegal use of inherited package");
         --! Issued if an attempt is made to refine an own variable onto an own
         --! variable of a
         --! non-embedded package.

      when 76 =>
         ELStrings.AppendString
            (EStr, "Identifier ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is already declared and cannot be");
         ELStrings.AppendString
            (EStr, " the name of an embedded package");
         --! Issued when a refinement clause in a package body attempts to name
         --! an embedded package
         --! own variable as a refinement constituent and the name given for the
         --! embedded package is
         --! already in use.

      when 77 =>
         ELStrings.AppendString
            (EStr, "Variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " should occur in this own variable clause");
         --! Occurs in the own variable clause of a package embedded in another
         --! package if an own
         --! variable which is a refinement constituent of an own variable of the
         --! enclosing package is
         --! omitted.

      when 78 =>
         ELStrings.AppendString
            (EStr, "Initialization of own variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is ineffective");
         --! Issued if an own variable occurs in the initialization clause of
         --! an embedded package
         --! and the own variable concerned is a refinement constituent of another
         --! own variable which
         --! is not listed in the initialization specification of its package.

      when 79 =>
         ELStrings.AppendString
            (EStr, "Variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " should occur in this " &
             "initialization specification");
         --! Occurs in the initialization clause of a package embedded in another
         --! package if an own
         --! variable which is a refinement constituent of an initialized own variable
         --! of the enclosing
         --! package is omitted.

      when 80 =>
         ELStrings.AppendString
            (EStr, "Unexpected own variable clause - ");
         ELStrings.AppendString
            (EStr, "no variable in this clause ");
         ELStrings.AppendString
           (EStr, "is a refinement constituent");

      when 81 =>
         ELStrings.AppendString
            (EStr, "Own variable clause expected - ");
         ELStrings.AppendString
            (EStr, "own variables of this package ");
         ELStrings.AppendString
           (EStr, "occur as refinement constituents");

      when 82 =>
         ELStrings.AppendString
            (EStr, "Unexpected initialization specification - ");
         ELStrings.AppendString
            (EStr, "no own variables of this package ");
         ELStrings.AppendString
            (EStr, "require initialization");
         --! An own variable initialization clause and that of its refinement
         --! constituents must be
         --! consistent.

      when 83 =>
         ELStrings.AppendString
            (EStr, "Initialization specification expected - ");
         ELStrings.AppendString
            (EStr, "own variables of this package require ");
         ELStrings.AppendString
            (EStr, "initialization");
         --! Issued if an own variable does not occur in the initialization
         --! clause of an embedded
         --! package and the own variable concerned is a refinement constituent
         --! of another own variable
         --! which is listed in the initialization clause of its package.

      when 84 =>
         ELStrings.AppendString
            (EStr, "The refinement constituent ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " does not have a declaration");
         --! Issued at the end of a package if a refinement constituent of a
         --! refined own variable
         --! has not been given an Ada declaration or further refined.

      when 85 =>
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is not a constituent of any abstract own variable appearing" &
             " in the earlier global definition for this subprogram");
         --! A variable XXX which has occurred in a refined global annotation
         --! is neither a variable
         --! that occurred in the earlier global definition nor a refinement
         --! constituent of any such
         --! variable.

      when 86 =>
         ELStrings.AppendString
            (EStr, "At least one constituent of ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " was expected in this refined global definition");
         --! If the global annotation of a procedure specification contains
         --! an own variable and that
         --! own variable is later refined then at least one refinement constituent
         --! of the own variable
         --! shall appear in the second global annotation supplied for the procedure
         --! body.

      when 87 =>
         ELStrings.AppendString
            (EStr,
             "Refined global definition expected for subprogram ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         --! A global definition containing abstract own variables was given in
         --!  the definition for
         --! subprogram XXX, in a package specification. A refined global definition
         --! is required in the
         --! package body.

      when 88 =>
         ELStrings.AppendString
            (EStr, "Variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is not a refinement constituent");

      when 89 =>
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is not a private type declared in this package");

      when 90 =>
         ELStrings.AppendString
            (EStr, "This operator may not be applied to ranges");

      when 91 =>
         ELStrings.AppendString
           (EStr, "Ranges may not be assigned");

      when 92 =>
         ELStrings.AppendString
           (EStr, "Named association may not be used here");

      when 93 =>
         ELStrings.AppendString
            (EStr,
             "Number of index expressions differs from number of dimensions of array ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);

      when 94 =>
         ELStrings.AppendString
            (EStr, "Condition is not boolean");
         --! Issued anywhere a boolean expression is required (e.g. in<b> if</b>, <b>
         --! exit </b> and <b>while</b> statements) and the expression provided
         --! is not of type boolean.

      when 95 =>
         ELStrings.AppendString
           (EStr, "Type mark expected");

      when 96 =>
         ELStrings.AppendString
            (EStr, "Attribute ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is not valid with this prefix");

      when 97 =>
         ELStrings.AppendString
            (EStr, "Attribute BASE may only appear as a prefix");
         --! 'BASE may only be used as a prefix to another attribute.

      when 98 =>
         ELStrings.AppendString
           (EStr, "This expression is not a range");

      when 99 =>
         ELStrings.AppendString
            (EStr, "Unconstrained array expected");
         --! Occurs if a subtype is declared of an array which is already constrained.

      when 100 =>
         ELStrings.AppendString
           (EStr, "Floating point type mark expected");

      when 101 =>
         ELStrings.AppendString
           (EStr, "Fixed point type mark expected");

      when 102 =>
         ELStrings.AppendString
           (EStr, "This is not the name of a field of record ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);

      when 103 =>
         ELStrings.AppendString
            (EStr, "A value has already been supplied for field ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);

      when 104 =>
         ELStrings.AppendString
            (EStr, "No value has been supplied for field ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
      when 105 =>
         ELStrings.AppendString
            (EStr, "More values have been supplied than " &
             "number of fields in record ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);

      when 106 =>
         ELStrings.AppendString
           (EStr, "Range is not of the expected type");

      when 107 =>
         ELStrings.AppendString
            (EStr, "Expression is not of the expected type. Actual type is ");
         if ErrNum.Name1 /= Error_Types.NoName then
            AppendName
               (EStr, ErrNum.Name1, ErrNum.Scope);
            if ErrNum.Name2 /= Error_Types.NoName then
               ELStrings.AppendString
                 (EStr, ". Expected type is ");
               AppendName
                 (EStr, ErrNum.Name2, ErrNum.Scope);
            end if;
         end if;

      when 108 =>
         ELStrings.AppendString
           (EStr, "Expression is not of the expected type. Expected any Integer type");

      when 109 =>
         ELStrings.AppendString
           (EStr, "Expression is not of the expected type. Expected any Real type");

      when 110 =>
         ELStrings.AppendString
           (EStr, "Use type clauses following an embedded package are not currently " &
              "supported by the Examiner");

      when 111 =>
         ELStrings.AppendString
           (EStr, "Package renaming is not currently supported by the Examiner");

      when 112 =>
         ELStrings.AppendString
           (EStr, "A use type clause may not appear here.  They are only permitted " &
              "as part of a context clause or directly following an embedded " &
              "package specification");

      when 113 =>
         ELStrings.AppendString
           (EStr, "Private subprogram declarations are not permitted in SPARK 83");
         --! Private subprograms would not be callable in SPARK 83 and are therefore not
         --! permitted; they may be declared and called in SPARK 95.

      when 114 =>
         ELStrings.AppendString
           (EStr, "Subtype mark or Range may not be used in an expression in this context");
         --! A subtype mark or an explicit Range attribute may not be used in a context
         --! where a simple expression is expected.

      when 115 =>
         ELStrings.AppendString
           (EStr, "In a package body, an own variable annotation must include one or more refinement constituents");
         --! Annotation should be of the form 'own S is A, B, C;'

      when 120 =>
         ELStrings.AppendString
           (EStr, "Pragma import not allowed here because variable ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is already initialized.  See ALRM B.1(24)");

      when 121 =>
         ELStrings.AppendString
           (EStr, "'Flow_Message' or 'Warning_Message' expected");
         --! The identifier indicating what kind of message to justify must be either
         --! 'Flow_Message' or 'Warning_Message' or some unique abbreviation of them such as
         --! 'Fl' or even 'F'.  Case is ignored.

      when 122 =>
         ELStrings.AppendString
           (EStr, "Error or warning number expected");
         --! This item should be an integer literal representing the error or warning message that
         --! is being marked as expected.

      when 123 =>
         ELStrings.AppendString
           (EStr, "This warning number may not appear in an accept annotation");
         --! It does not make sense to allow certain warnings to be justified with the accept annotation.
         --! In particular, attempting to justify warnings raised by the justification system itself
         --! could lead to some special kind of recursive hell that we would not wish to enter.

      when 124 =>
         ELStrings.AppendString
           (EStr, "Incorrect number of names in accept annotation: should be 0");
         --! This class of error does not reference any variables, and therefore requires
         --! no names.

      when 125 =>
         ELStrings.AppendString
           (EStr, "Incorrect number of names in accept annotation: should be 1");
         --! This class of error references one variable, and therefore requires
         --! one name.

      when 126 =>
         ELStrings.AppendString
           (EStr, "Incorrect number of names in accept annotation: should be 2");
         --! This class of error references two variables, and therefore requires
         --! two names. Two names are need to justify expected information
         --! flow messages such as &quot;X is not derived from Y&quot;.
         --! Note that for messages of this kind the accept annotation should list the names in the order
         --! &quot;export, import&quot;

      when 127 =>
         ELStrings.AppendString
           (EStr, "Incorrect number of names in accept annotation: should be 0 or 1");
         --! This class of error references either zero or one variable, and therefore
         --! requires either zero or one name. An ineffective assignment error requires
         --! the name of variable being assigned to. An ineffective statement error
         --! has no name associated with it.

      when 128 =>
         ELStrings.AppendString
           (EStr, "Incorrect number of names in accept annotation: should be 1 or 2");
         --! This class of error references either one or two variables, and therefore
         --! requires either one or two names. One name is required when the export
         --! is a function return value.

      when 130 =>
         ELStrings.AppendString
            (EStr,
             "A type from the current package should not appear in a use type clause");

      when 131 =>
         ELStrings.AppendString
            (EStr, "The package name ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " should appear in a with clause preceding the use type clause");

      when 132 =>
         ELStrings.AppendString
            (EStr,
             "The unit name or the name of an enclosing package of the unit should not appear in its with clause");
         --! A package should not 'with' itself and a subunit
         --! should not 'with' the package (or main program) which declares its stub

      when 133 =>
         ELStrings.AppendString
            (EStr,
             "Name in with clause is locally redeclared");

      when 134 =>
         ELStrings.AppendString
            (EStr,
             "A package name should not appear in its own inherit clause");

      when 135 =>
         ELStrings.AppendString
            (EStr, "The package ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is either undeclared or not visible at this point");
         --! Possible causes of this error are
         --! an error in the inherited package specification or
         --! omitting an entry for the package specification from the index file

      when 136 =>
         ELStrings.AppendString
            (EStr, "The own variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is not declared in the own variable clause of the corresponding package declaration");
         --! A refinement clause of a package body defines the constituent
         --! parts of own variables given in the own variable clause of the
         --! corresponding package declaration

      when 137 =>
         ELStrings.AppendString
            (EStr, "The child package ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is either undeclared or not visible at this point");
         --! Possible causes of this error are
         --! an error in the child package specification or
         --! omitting the child from the parent's component list in the index file
         --! or omitting the child specification entry from the index file

      when 138 =>
         ELStrings.AppendString
            (EStr, "Child package own variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is does not appear in the own variable clause of the child package");
         --! A constituent of a refinement clause which is defined in a
         --! child package must be an own variable of the child package

      when 139 =>
         ELStrings.AppendString
            (EStr, "The variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is not declared in the own variable clause of this package");
         --! A package can only initialize variables
         --! declared in its own variable clause

      when 140 =>
         ELStrings.AppendString
            (EStr, "The predecessor package ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is either undeclared or not visible at this point");
         --! The parent of a child package must be a library package and
         --! must be declared prior to a child package.  If using an index file
         --! the parent must have an entry in the index file and the child package
         --! must be listed as a component of the parent package

      when 141 =>
         ELStrings.AppendString
            (EStr, "The private type ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is either undeclared or not visible at this point");

      when 142 =>
         ELStrings.AppendString
            (EStr, "The subprogram prefix ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is either undeclared or not visible at this point");
         --! The prefix should appear in the inherit clause of the current package

      when 143 =>
         ELStrings.AppendString
            (EStr, "The subprogram ");
         if ErrNum.Name2 /= Error_Types.NoName then
            AppendName
               (EStr, ErrNum.Name2, ErrNum.Scope);
            ELStrings.AppendString
               (EStr, ".");
         end if;
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is either undeclared or not visible at this point");

      when 144 =>
         ELStrings.AppendString
           (EStr, "The dotted name ");
         if ErrNum.Name2 /= Error_Types.NoName then
            AppendName
              (EStr, ErrNum.Name2, ErrNum.Scope);
            ELStrings.AppendString
              (EStr, ".");
         end if;
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is either undeclared or not visible at this point");
         --! The name must denote an entire variable or an own variable
         --! of a package.  If the variable or own variable is declared in a
         --! separate (or parent) package, the package must be included in an inherit clause
         --! and the identifier prefixed with the package name

      when 145 =>
         ELStrings.AppendString
           (EStr, "The identifier ");
         if ErrNum.Name2 /= Error_Types.NoName then
            AppendName
              (EStr, ErrNum.Name2, ErrNum.Scope);
            ELStrings.AppendString
              (EStr, ".");
         end if;
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is either undeclared or not visible at this point");
         --! The identifier should be a typemark.  If the typemark is declared in a
         --! separate (or parent) package, the package must be included in an inherit clause
         --! and the identifier prefixed with the package name.
         --! Ensure that there are no errors in the declaration of the typemark

      when 148 =>
         ELStrings.AppendString
            (EStr, "The abstract proof type ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " may not be used to define an own variable in another package");
         --! Own variables may be &quot;type announced&quot; as being of an abstract proof
         --! type only where
         --! that type is declared later in the same package. Thus --# own State :
         --! T; is legal if
         --! --# type T is abstract; appears later in the package; however, --# own
         --! State : P.T; is
         --! illegal if T is an abstract proof type declared in remote package P.

      when 149 =>
         ELStrings.AppendString
            (EStr,
             "More than one own variable has been announced as being of type ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " which may not therefore be declared as an abstract proof type");
         --! Occurs when an own variable clause announces more than one own variable
         --! as being of
         --! a type XXX and XXX is later declared as being of an abstract proof type.
         --! Each abstract
         --! own variable must be of a unique type.

      when 150 =>
         ELStrings.AppendString
            (EStr, "Entire variable expected. The names of constants never appear in mandatory annotations");
         --! Issued when a the name of a constant is found in a mandatory annotation such as
         --! a global or derives annotation.  Constants should not appear in such annotations.

      when 151 =>
         ELStrings.AppendString
            (EStr, "The variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " does not occur either in the package own variable list" &
             " or as a refinement constituent");
         --! A variable declared in a package must have been previously announced
         --! as either an own
         --! variable or as a concrete refinement constituent of an own variable.

      when 152 =>
         ELStrings.AppendString
            (EStr,
             "The number of formal parameters is not consistent " &
             "with the previous declaration of ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
      when 153 =>
         ELStrings.AppendString
            (EStr, "The declaration of formal parameter ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is not consistent with the subprogram's previous declaration");
         --! Issued if the name, type or parameter mode of a parameter is different
         --! in the
         --! subprogram body declaration from that declared originally.

      when 154 =>
         ELStrings.AppendString
            (EStr, "The subprogram or task body ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " does not have an annotation");
         --! A subprogram or task body must have a global annotation if it references
         --! global variables; a
         --! procedure or task body must have a dependency relation to perform
         --! information flow analysis.

      when 155 =>
         ELStrings.AppendString
            (EStr,
             "Unexpected annotation - all annotations required for procedure or task " &
              "body ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " have already occurred");
         --! Do not repeat global or derives annotations in the body
         --! (or body stub) of a subprogram, entry or task except for
         --! state (own variable) refinement.

      when 156 =>
         ELStrings.AppendString
            (EStr, "Entire variable expected");
         --! Issued when an identifier which SPARK requires to be an entire
         --! variable represents
         --! something other than this. Most commonly this message occurs when
         --! a component of a
         --! structured variable appears in a core annotation.

      when 157 =>
         ELStrings.AppendString
            (EStr, "The name ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " already appears in the global variable list");

      when 158 =>
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is a formal parameter of this subprogram");
         --! Issued in a global annotation if it names a formal parameter of the
         --! subprogram.

      when 159 =>
         ELStrings.AppendString
            (EStr, "The name ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " has already appeared as an exported variable");

      when 160 =>
         ELStrings.AppendString
            (EStr, "The name ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " already appears in the list of imported variables");

      when 161 =>
         ELStrings.AppendString
            (EStr, "Exportation of ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is incompatible with its parameter mode");
         --! Issued if a parameter appears as an export to a procedure when it is of
         --! parameter mode
         --! in.

      when 162 =>
         ELStrings.AppendString
            (EStr, "Importation of ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is incompatible with its parameter mode");
         --! Issued if a parameter appears as an import to a procedure when it is of
         --! parameter mode
         --! out.

      when 163 =>
         ELStrings.AppendString
            (EStr, "Subprogram ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " cannot be called from here");
         --! SPARK contains rules to prevent construction of programs containing
         --! recursive
         --! subprogram calls; this error message occurs if a procedure or function
         --! is called before
         --! its body has been declared. Re-ordering of subprogram bodies in the
         --! package concerned will
         --! be required.

      when 164 =>
         ELStrings.AppendString
            (EStr,
             "Actual parameter of mode in out or out must be " &
             "the name of a variable");

      when 165 =>
         ELStrings.AppendString
            (EStr,
             "This parameter is overlapped by another one, which is exported");
         --! Violation of the anti-aliasing rule.

      when 166 =>
         ELStrings.AppendString
            (EStr,
             "This parameter is overlapped by an exported global variable");
         --! Violation of the anti-aliasing rule.

      when 167 =>
         ELStrings.AppendString
            (EStr, "Imported variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is not named in the initialization specification of its package");
         --! Issued when an own variable which is imported into the main program
         --! procedure (or a task when the Ravenscar profile is enabled) has not
         --! been declared as being initialized by its package.  At the main program
         --! level the only
         --! imports that are permitted are initialized own variables of inherited
         --! packages.  There are
         --! two possible cases to consider: (1) the main program should be importing
         --! the variable in
         --! which case it should be annotated in its package with --# initializes
         --! (and, of course, actually
         --! initialized in some way) or be an external variable or protected variable
         --! which is implicitly
         --! initialized; or (2) the own variable concerned is not initialized at
         --! elaboration,
         --! should not therefore be considered an import to the main program and
         --! should be removed from the
         --! main program's import list.

      when 168 =>
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is a loop parameter, whose updating is not allowed");

      when 169 =>
         ELStrings.AppendString
            (EStr, "Global variables of function subprograms " &
             "must be of mode in");
         --! It is an important property of SPARK  that functions cannot have
         --! side-effects, therefore
         --! only the reading of global variable is permitted.  It is usually
         --! convenient to omit
         --! modes from function global annotations but use of mode 'in' is
         --! permitted.

      when 170 =>
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is a formal parameter of mode in, whose updating is not allowed");

      when 171 =>
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is a formal parameter of mode out, whose value cannot be read");

      when 172 =>
         ELStrings.AppendString
            (EStr,
             "The actual parameter associated with an exported " &
             "formal parameter must be an entire variable");
         --! Issued if an actual parameter which is an array element is associated
         --! with an exported
         --! formal parameter in a procedure call. Exported parameters must be either
         --! entire variables
         --! or a record field.

      when 173 =>
         ELStrings.AppendString
            (EStr,
             "This exported parameter is named in the global " &
             "definition of the procedure");
         --! Violation of the anti-aliasing rule.

      when 174 =>
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is not an own variable");
         --! Occurs in initialization specifications if something other than a
         --! variable is listed as
         --! being initialized.

      when 175 =>
         ELStrings.AppendString
           (EStr, "Information flow from ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " to ");
         AppendName
           (EStr, ErrNum.Name2, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " violates the selected information flow policy.");

      when 180 =>
         ELStrings.AppendString
            (EStr, "Entire composite constant expected");
         --! Issued when an identifier which SPARK requires to be an entire
         --! composite constant represents something other than this.

      when 181 =>
         ELStrings.AppendString
            (EStr, "Invalid policy for constant proof rule generation");

      when 182 =>
         ELStrings.AppendString
           (EStr, "Rule Policy for ");
         if ErrNum.Name2 /= Error_Types.NoName then
            AppendName
               (EStr, ErrNum.Name2, ErrNum.Scope);
            ELStrings.AppendString
               (EStr, ".");
         end if;
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " already declared in current scope");
         --! Issued when a rule policy has already been declared for this
         --! constant within this declarative region. This rule policy will
         --! be ineffective.

      when 190 =>
         ELStrings.AppendString
            (EStr, "The name ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " already appears in the inherit clause");

      when 191 =>
         ELStrings.AppendString
            (EStr, "The name ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " already appears in the with clause");

      when 200 =>
         ELStrings.AppendString
            (EStr, "The parameter ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is neither imported nor exported");
         --! Each formal parameter of a subprogram shall be imported or exported or both.

      when 201 =>
         ELStrings.AppendString
            (EStr, "The global variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is neither imported nor exported");
         --! Every variable in a global definition must also appear in the associated derives annotation
         --! where it will be either imported or exported or both.

      when 250 =>
         ELStrings.AppendString
            (EStr, "The 'Size value for type ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " has already been set");

      when 251 =>
         ELStrings.AppendString
            (EStr, "The attribute value for ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, "'Size must be of an integer type");

      when 252 =>
         ELStrings.AppendString
            (EStr, "The attribute value for ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, "'Size must be a static simple expression");
         --! The value of 'Size must be static and must be of
         --! an integer type

      when 253 =>
         ELStrings.AppendString
            (EStr, "The attribute value for ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, "'Size must not be negative");
         --! The value of 'Size must be a positive integer or zero

      when 254 =>
         ELStrings.AppendString
            (EStr, "The Size attribute can only be specified for a first subtype");
         --! Setting 'Size for a user-defined non-first subtype is not permitted.
         --! See Ada95 LRM 13.3(48)

      when 255 =>
         ELStrings.AppendString
           (EStr, "The Address attribute can only be specified for a variable, " &
                  "a constant, or a program unit");
         --! Ada95 LRM Annex N.31 defines a program unit to be either a package, a task unit,
         --! a protected unit, a protected entry, a generic unit, or an explicitly
         --! declared subprogram other than an enumeration literal.

      when 273 =>
         ELStrings.AppendString
            (EStr, "Own variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " may not be refined because it was declared with a "     &
             "type mark which has not subsequently been declared as "  &
             "an abstract proof type");
         --! Where a type mark is included in an own variable declaration it
         --! indicates that the own
         --! variable will either be of a concrete type of that name (which may
         --! be either already
         --! declared or be declared later in the package) or of an abstract proof
         --! type declared in
         --! the package specification.  In the former case the refinement is
         --! illegal because own
         --! variables of concrete Ada types may not be refined.  In the latter case
         --! it is legal;
         --! however, no suitable proof type declaration has been found in this case.

      when 300 =>
         ELStrings.AppendString
            (EStr,
             "Renaming declarations are not allowed here");
         --! A renaming declaration must be the first declarative item of a package
         --! body or main
         --! program or it must be placed immediately after the declaration of
         --! an embedded package.

      when 301 =>
         ELStrings.AppendString
            (EStr,
             "Renaming or use type declarations here can only rename subprograms in " &
              "package ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         --! A renaming declaration may be placed immediately after the declaration
         --! of an embedded
         --! package; in this case it may only rename subprograms declared in that
         --! package.

      when 302 =>
         ELStrings.AppendString
            (EStr,
             "The subprogram specification in this renaming declaration " &
             "is not consistent with the declaration of subprogram ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         --! Issued in a subprogram renaming declaration if it contains parameter
         --! names, numbers or
         --! types which differ from those originally declared.

      when 303 =>
         ELStrings.AppendString
            (EStr,
             "An operator can only be renamed by the same operator");
         --! Issued if a renaming declaration has a different operator on each
         --! side of the reserved
         --! word RENAMES.

      when 304 =>
         ELStrings.AppendString
            (EStr, "A renaming declaration for operator ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is not allowed");

      when 305 =>
         ELStrings.AppendString
            (EStr,
             "The specification in this renaming declaration is not " &
             "consistent with the implicit declaration of operator ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         --! Issued in an operator renaming declaration if it contains types
         --! which differ from those
         --! applicable to the operator being renamed.

      when 306 =>
         ELStrings.AppendString
            (EStr, "Operator ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is already visible");
         --! Occurs in an operator renaming declaration if an attempt is made
         --! to rename an operator
         --! which is already visible. (The message will also appear as a secondary
         --! consequence of
         --! trying to rename an operator between undeclared types.)

      when 307 =>
         ELStrings.AppendString
            (EStr,
             "The implicit declaration of this operator does not occur in package ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);

      when 308 =>
         ELStrings.AppendString
            (EStr,
             "Type is limited");
         --! Issued if an attempt is made to assign a variable of a type which is
         --! limited or which
         --! contains a limited type.

      when 309 =>
         ELStrings.AppendString
            (EStr, "Operator not visible for these types");
         --! This message means that the operator exists between the types on each
         --! side of it but
         --! that it is not visible. The most likely cause is that the types
         --! concerned are defined in
         --! another package and that renaming is required to make the operator visible.

      when 310 =>
         ELStrings.AppendString
           (EStr, "The % operator may only appear in an assert or check statement in a for loop");
         --! The % operator is used to indicate the value of a variable on entry to a for loop.  This is because
         --! the variable may be used in the exit expression of the loop and may also be modified in the body
         --! of the loop.  Since the semantics of Ada require the exit expression to be fixed after evaluation
         --! we require a way of reasoning about the original value of a variable prior to any alteration in
         --! the loop body.  No other situation requires this value so % may not be used anywhere else.

      when 311 =>
         ELStrings.AppendString
            (EStr,
             "Announced own variable types may not be implemented as unconstrained " &
              "arrays");
         --! Where an own variable is announced as being of some type,
         --! SPARK requires that type
         --! to be declared; the declaration cannot be in the form of an
         --! unconstrained array because
         --! SPARK prohibits unconstrained variables.

      when 312 =>
         ELStrings.AppendString
            (EStr,
             "A subprogram can only be renamed to the same name " &
             "with the package prefix removed");

      when 313 =>
         ELStrings.AppendString
            (EStr, "Only one main program is permitted");

      when 314 =>
         ELStrings.AppendString
            (EStr, "Own variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " has been refined and may not appear here");
         --! Issued if an attempt is made to use, in a second annotation, an own
         --! variable which has
         --! been refined. Second annotations should use the appropriate refinement
         --! constituents of the
         --! own variable.

      when 315 =>
         ELStrings.AppendString
            (EStr, "Unsupported proof context");
         --! Certain proof contexts have been included in the syntax of SPARK but
         --! are not yet
         --! supported; this error message results if one is found.

      when 317 =>
         ELStrings.AppendString
            (EStr, "Tilde, in a function return annotation, may only be applied to an external variable of mode IN");
         --! The tilde decoration indicates the initial value of a variable or
         --! parameter which is both imported and exported. A function may not have an explicit side effect
         --! on a program variable and so cannot be regarded as exporting such a variable.  For modelling purposes
         --! a read of an external (stream) variable is regarded as having a side effect (outside the SPARK
         --! boundary).  Since it may be necessary to refer to the initial value of the external variable, before
         --! this implicit side effect occurs, the use of tilde is allowed only for external variables of mode IN which
         --! are globally referenced by function.
      when 318 =>
         ELStrings.AppendString
            (EStr, "Tilde or Percent may only be applied to variables");
         --! The tilde decoration indicates the initial value of a variable or
         --! parameter which is
         --! both imported and exported. Percent indicates the value of a variable
         --! on entry to a for loop; this message occurs if either operator is applied
         --! to any other object.

      when 319 =>
         ELStrings.AppendString
            (EStr, "Tilde may only be applied to a variable " &
             "which is both imported and exported");
         --! The tilde decoration indicates the initial value of a variable or
         --! parameter which is
         --! both imported and exported; this message occurs if the variable
         --! concerned is either
         --! exported only or imported only in which case no distinction between
         --! its initial and final
         --! value is required.

      when 320 =>
         ELStrings.AppendString
            (EStr, "Tilde or Percent may only be applied to an entire variable");
         --! Tilde (and %) may not be applied to an element of an array or field of a record.
         --! e.g. to
         --! indicate the initial value of the Ith element of array V use V~(I)
         --! not V(I)~.

      when 321 =>
         ELStrings.AppendString
            (EStr, "Tilde may not appear in pre-conditions");
         --! Since it does not make sense to refer to anything other than the
         --! initial value of a
         --! variable in a pre-condition there is no need to use tilde to distinguish
         --! initial from
         --! final values.

      when 322 =>
         ELStrings.AppendString
            (EStr, "Only imports may be referenced in " &
             "pre-conditions or return expressions");
         --! Pre-conditions are concerned with the initial values of information
         --! carried into a
         --! subprogram. Since only imports can do this only imports can appear in
         --! pre-condition
         --! expressions.

      when 323 =>
         ELStrings.AppendString
            (EStr, "Updates may only be applied to records or arrays");
         --! The extended SPARK update syntax is only used to express changes to
         --! components of a
         --! structured variable.

      when 324 =>
         ELStrings.AppendString
            (EStr, "Only one field name may appear here");
         --! When using the extended SPARK update syntax for a record, you can not
         --! update more than
         --! one element in each clause of the update. For example, you cannot use
         --! [x,y =&gt; z], you
         --! must instead use [x =&gt; z; y =&gt; z].

      when 325 =>
         ELStrings.AppendString
            (EStr, "Type ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " has not been declared");
         --! Occurs if a type is &quot;announced&quot; as part of an own variable
         --! clause and the end
         --! of the package is reached without an Ada declaration for a type of
         --! this name being found.

      when 326 =>
         ELStrings.AppendString
            (EStr, "Predicate is not boolean");
         --! Occurs anywhere where a proof context is found not to be a boolean
         --! expression.

      when 327 =>
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is a global variable which may not be updated in a function subprogram");

      when 328 =>
         ELStrings.AppendString
            (EStr, "The identifier ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " (exported by called subprogram) may not be updated in a function " &
               "subprogram");
         --! Occurs if a function calls a procedure which exports a global variable;
         --! this would
         --! create an illegal side-effect of the function.

      when 329 =>
         ELStrings.AppendString
            (EStr, "Illegal function call");
         --! Issued if a call is made to a user-defined subprogram in a package
         --! initialization part.

      when 330 =>
         ELStrings.AppendString
            (EStr,
             "Illegal use of an own variable not of this package");
         --! Issued if an attempt is made, in a package initialization part, to
         --! update an own
         --! variable of a non-enclosing package.

      when 331 =>
         ELStrings.AppendString
            (EStr,
             "Private types may not be unconstrained arrays");

      when 332 =>
         ELStrings.AppendString
            (EStr,
             "This private type was not declared as limited");
         --! Issued where the type contains a component which is a limited private
         --! type, but where
         --! the declaration of this type in the visible part of the package does
         --! not specify that the
         --! type is limited.

      when 333 =>
         ELStrings.AppendString
            (EStr, "Initialization of ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is not announced in the initialization clause of this package");
         --! Issued when an own variable is initialized either by assignment or
         --! by having a pragma Import attached to it when initialization of the variable
         --! is not announced in its package's own variable initialization specification.

      when 334 =>
         ELStrings.AppendString
            (EStr, "Identifier ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is not the name of a function");

      when 335 =>
         ELStrings.AppendString
            (EStr,
             "This annotation should be placed with the declaration of function ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         --! Issued if a function is declared in a package specification without an
         --! annotation but
         --! one is then supplied on the function body.

      when 336 =>
         ELStrings.AppendString
            (EStr,
             "Unexpected annotation - all annotations required for function ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " have already occurred");

      when 337 =>
         ELStrings.AppendString
            (EStr, "Package ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " may not be used as a prefix here");
         --! Selected component notation may not be used in places where an item is
         --! directly
         --! visible.

      when 338 =>
         ELStrings.AppendString
            (EStr, "Scalar parameter ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is of mode in out and must appear as an import");
         --! Parameters passed as mode in out must be listed as imports in the
         --! subprogram's
         --! dependency relation if they are of scalar types. The rule also applies
         --! to a parameter of a
         --! private type if its full declaration is scalar.

      when 339 =>
         ELStrings.AppendString
            (EStr, "Subprogram ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " was not declared in package ");
         AppendName
            (EStr, ErrNum.Name2, ErrNum.Scope);

      when 340 =>
         ELStrings.AppendString
            (EStr,
             "Only operators may be renamed in " &
             "package specifications");
         --! User-declared subprograms may not be renamed in package specifications
         --! although the
         --! implicitly declared function subprograms associated with operators may be.

      when 341 =>
         ELStrings.AppendString
            (EStr,
             "A range may not appear here");
         --! Issued if a range is found where a single value is expected, for example,
         --! if an array
         --! slice is constructed.

      when 342 =>
         ELStrings.AppendString
            (EStr,
             "This proof annotation should be placed with the declaration of " &
               "subprogram ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);

         --! Like global and derives annotations, proof annotations should be placed
         --! on the first
         --! appearance of a subprogram.  There may also be a requirement for a
         --! second proof annotation
         --! on a subprogram body where it references an abstract own variable.

      when 343 =>
         ELStrings.AppendString
            (EStr,
             "Unexpected proof annotation - all annotations required for subprogram ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " have already occurred");
         --! Issued if a second proof annotation for a subprogram is found but
         --! the subprogram does
         --! not reference any abstract own variables.  A second annotation is
         --! only required where it
         --! is necessary to express both an abstract (external) and a refined
         --! (internal) view of an
         --! operation.

      when 399 =>
         ELStrings.AppendString
            (EStr,
             "Range error in annotation expression");
         --! Issued if a proof annotation contains an expression that would cause
         --! a constraint error
         --! if it were in an executable Ada statement.  For example: &quot;--# post X =
         --! T'Succ(T'Last);&quot;
         --! VCs generated from such malformed predicates would always be unprovable.

      when 400 =>
         ELStrings.AppendString
            (EStr,
             "Expression contains division by zero");
         --! Issued when a static expression, evaluated using perfect arithmetic,
         --! is found to
         --! contain a division by zero.

      when 401 =>
         ELStrings.AppendString
            (EStr,
             "Illegal numeric literal");
         --! Issued when a numeric literal is illegal because it contains, for example,
         --! digits not
         --! compatible with its number base.

      when 402 =>
         ELStrings.AppendString
            (EStr,
             "Constraint_Error will be raised here");
         --! Issued whenever a static expression would cause a constraint error.
         --! e.g. assigning a
         --! value to a constant outside the constant's type range. In SPARK a static
         --! expression may
         --! not yield a value which violates a range constraint.

      when 403 =>
         ELStrings.AppendString
            (EStr,
             "Argument value is inconsistent with the number of " &
             "dimensions of array type ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         --! Issued when an array attribute containing an argument is found and the
         --! value of the
         --! argument is inconsistent with the number of dimensions of the array
         --! type to which it is
         --! being applied.

      when 406 =>
         ELStrings.AppendString
            (EStr,
             "Only scalar or non-tagged record subtypes may be declared without a constraint");
         --! Issued if a subtype declaration of the form </i>subtype S is T<i>
         --! is used where </i>T<i> is not a scalar or non-tagged record type.
         --! Secondly, T must not be private at the point of this declaration.

      when 407 =>
         ELStrings.AppendString
            (EStr,
             "This choice overlaps a previous one");
         --! Choices in case statements and array aggregates may not overlap.

      when 408 =>
         ELStrings.AppendString
            (EStr,
             "Case statement is incomplete");
         --! A case statement must either explicitly supply choices to cover the
         --! whole range of the
         --! (sub)type of the controlling expression, or it must supply an others choice.

      when 409 =>
         ELStrings.AppendString
            (EStr,
             "Empty range specified");
         --! In SPARK, no static range is permitted to be null.

      when 410 =>
         ELStrings.AppendString
            (EStr,
             "Choice out of range");
         --! The choices in case statements and array aggregates must be within
         --! the constraints of
         --! the appropriate (sub)type.

      when 411 =>
         ELStrings.AppendString
            (EStr,
             "Others clause required");
         --! Issued where an others clause is required to satisfy the Ada language rules.

      when 412 =>
         ELStrings.AppendString
            (EStr,
             "Explicit boolean range not permitted");
      when 413 =>
         ELStrings.AppendString
            (EStr,
             "Invalid range constraint");
         --! Issued where a range constraint is outside the range of the (sub)type to
         --! which the
         --! constraint applies.

      when 414 =>
         ELStrings.AppendString
            (EStr,
             "Array aggregate is incomplete");
         --! An array aggregate must either explicitly supply values for all array
         --! elements or
         --! provide an others clause.

      when 415 =>
         ELStrings.AppendString
            (EStr,
             "Too many entries in array aggregate");
         --! Issued where an array aggregate using positional association contains
         --! more entries than
         --! required by the array index type.

      when 416 => ELStrings.AppendString
         (EStr,
          "Type may not have an empty range");

      when 417 => ELStrings.AppendString
         (EStr,
          "String subtypes must have a lower index bound of 1");

      when 418 => ELStrings.AppendString
         (EStr,
          "Index upper and/or lower bounds do not match those expected");
         --! Issued where assignment, association or type conversion is attempted
         --! between two
         --! different constrained subtypes of the same unconstrained array type, and
         --! where the index
         --! bounds do not match.

      when 420 =>
         ELStrings.AppendString
            (EStr, "Array index(es) not convertible");
         --! Issued when an attempt is made to convert between two arrays whose indexes
         --! are neither
         --! of the same type nor numeric.

      when 421 =>
         ELStrings.AppendString
            (EStr, "Array components are not of the expected type");
         --! Issued when a type conversion attempts to convert between two array types
         --! whose
         --! components are of different types.

      when 422 =>
         ELStrings.AppendString
            (EStr, "Array component constraints do not " &
             "match those expected");
         --! Issued when a type conversion attempts to convert between two array types
         --! whose
         --! components are of the same type but do not have constraints which can be
         --! statically
         --! determined to be identical.

      when 423 =>
         ELStrings.AppendString
            (EStr, "Array has different number of dimensions " &
             "from that expected");
         --! Issued when attempting to convert between two array types which have
         --! different numbers
         --! of dimensions.

      when 425 =>  ELStrings.AppendString
         (EStr,
          "String literals may not be converted");
         --! Issued if the argument of a type conversion is a string literal. A common
         --! cause is an
         --! attempt to type qualify a string and accidentally omitting the tick character.

      when 500 =>
         ELStrings.AppendString
            (EStr,
             "Mode expected");
         --! Issued when performing data flow analysis only where a subprogram has no
         --! dependency
         --! clause and its global variables have not been given modes in the global
         --! annotation.

      when 501 =>
         ELStrings.AppendString
            (EStr,
             "Dependency relation expected");
         --! A dependency relation is required for each procedure if information flow
         --! analysis is to
         --! be performed.

      when 502 =>
         ELStrings.AppendString
            (EStr, "Exportation of ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is incompatible with its global mode");
         --! Issued when a procedure has both a global annotation with modes and a
         --! dependency
         --! relation, and a global of mode in is listed as an export in the dependency
         --!  relation.

      when 503 =>
         ELStrings.AppendString
            (EStr, "Importation of ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is incompatible with its global mode");
         --! Issued when a procedure has both a global annotation with modes and a
         --! dependency
         --! relation, and a global of mode out is listed as an import in the
         --! dependency relation.

      when 504 =>
         ELStrings.AppendString
            (EStr, "Parameter ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is of mode in out and must appear as an import");

      when 505 =>
         ELStrings.AppendString
            (EStr, "Global variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is of mode in out and must appear as an import");
         --! Issued where a procedure has both a global annotation with modes and
         --! a dependency
         --! relation, and a global variable of mode in out is not listed as an
         --! import in the
         --! dependency relation.

      when 506 =>
         ELStrings.AppendString
            (EStr, "Parameter ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is of mode in out and must appear as an export");

      when 507 =>
         ELStrings.AppendString
            (EStr, "Global variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is of mode in out and must appear as an export");
         --! Issued where a procedure has both a global annotation with modes
         --! and a dependency
         --! relation, and a global variable of mode in out is not listed as an
         --! export in the
         --! dependency relation.

      when 508 =>
         ELStrings.AppendString
            (EStr,
             "This global variable is a parameter of mode in and" &
             " can only have the global mode in");

      when 550 =>
         ELStrings.AppendString
            (EStr,
             "use type clauses may only be used in SPARK95: " &
             "clause ignored");

      when 551 =>
         ELStrings.AppendString
            (EStr, "All operators for type ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " are already visible");

      when 552 =>
         ELStrings.AppendString
            (EStr, "The type ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " already appears in the use type clause");

      when 554 =>
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is a limited private type for which no " &
             "operators can be made visible");

      when 555 =>
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is not mentioned in an earlier with clause " &
             "of this compilation unit");

      when 600 =>
         ELStrings.AppendString
            (EStr, "pragma Import has a minimum of 2 and " &
             "a maximum of 4 parameters");

      when 601 =>
         ELStrings.AppendString
            (EStr, "Convention, Entity, External_Name or " &
             "Link_Name expected");

      when 602 =>
         ELStrings.AppendString
            (EStr, "An association for ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " has already been given");

      when 603 =>
         ELStrings.AppendString
            (EStr, "No association for ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " was given");

      when 604 =>
         ELStrings.AppendString
            (EStr, "This package may not have a body " &
             "- consider use of pragma Elaborate_Body");
         --! In Ada 95, a package body is illegal unless it is required for the
         --! purpose of providing
         --! a subprogram body, or unless this pragma is used. This error is
         --! issued where a package
         --! body is found for a package whose specification does not require a body.

      when 605 =>
         ELStrings.AppendString
            (EStr, "pragma Elaborate_Body has one parameter");

      when 606 =>
         ELStrings.AppendString
            (EStr, "This expression does not represent the " &
             "expected package name ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         --! Issued when the parameter to a pragma Elaborate_Body is invalid.

      when 607 =>
         ELStrings.AppendString
            (EStr, "This package requires a body and must therefore" &
             " include either pragma Elaborate_Body or a subprogram " &
             "declaration");
         --! Issued where a package specification contains no subprogram declarations,
         --! but whose own
         --! variables (as specified in the package annotation) are not all declared
         --! (and initialized
         --! where appropriate) in the package specification. This is because such a
         --! package is not
         --! allowed a body in Ada 95 unless either the pragma is given or a
         --! subprogram declared.

      when 608 =>
         ELStrings.AppendString
            (EStr, "Reduced accuracy subtypes of real numbers are " &
             "considered obsolescent and are not supported by SPARK");

      when 609 =>
         ELStrings.AppendString
            (EStr, "This entity cannot be assigned to");

      when 610 =>
         ELStrings.AppendString
            (EStr, "Child packages may not be used in SPARK83");

      when 611 =>
         ELStrings.AppendString
            (EStr, "Illegal use of deferred constant prior to its full declaration");

      when 613 =>
         ELStrings.AppendString
            (EStr, "Illegal name for body stub");
         --! Issued if a dotted name appears in a body stub as in &quot;package body
         --! P.Q is
         --! separate&quot;. No legal stub could ever have such a name.

      when 614 =>
         ELStrings.AppendString
            (EStr, "Child packages may be declared only at library level");
         --! Issued if an attempt is made to declare a child package which is
         --! embedded in a package
         --! or subprogram.

      when 615 =>
         ELStrings.AppendString
            (EStr, "Name does not match name of package");
         --! Issued if the closing identifier of a package has a different number of
         --! identifiers
         --! from the name originally given for the package. For example &quot;package
         --! P.Q is ... end
         --! P.Q.R;&quot;

      when 616 =>
         ELStrings.AppendString
            (EStr, "The private package ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is not visible at this point");
         --! Issued if an attempt is made to with or inherit a private package
         --! from the visible part
         --! of a public package.

      when 617 =>
         ELStrings.AppendString
            (EStr, "Public sibling ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is not visible at this point");
         --! Arises from attempting to inherit a public sibling child package
         --! from a private child
         --! package.

      when 618 =>
         ELStrings.AppendString
            (EStr,
             "The owner of the current package does not inherit the package ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         --! A private child package can only inherit a remote package if its
         --! parent also inherits
         --! it; this is a analogous to the behaviour of embedded packages which
         --!  may also only inherit
         --! a remote package if their enclosing package also does so.

      when 619 =>
         ELStrings.AppendString
            (EStr, "The package ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is not owned by the current package");
         --! This message indicates an attempt to claim that own variables of a
         --! package other than a
         --! private child package of the current package are refinement constituents
         --!  of an abstract
         --! own variable of the current package.

      when 620 =>
         ELStrings.AppendString
            (EStr,
             "Own variables here must be refinement constituents in package owner ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         --! Own variables of private child packages must appear as refinement
         --! constituents of the
         --! package which owns the child. If the Examiner has seen the owner
         --! package body before
         --! processing the child and has not found the required refinement
         --! constituent then this
         --! message results on processing the child.

      when 621 =>
         ELStrings.AppendString
            (EStr, "Own variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " expected as a refinement constituent in this package");
         --! Own variables of private child packages must appear as refinement
         --! constituents of the
         --! package which owns the child. If the Examiner has seen a child package
         --! which declares an
         --! own variable before examining its owner&#146;s body then this message
         --! is issued if the
         --! owner lacks the required refinement constituent declaration.

      when 622 =>
         ELStrings.AppendString
            (EStr, "Own variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " did not occur in an initialization specification");
         --! Issued if an own variable appears in an initialization clause and is
         --! also a refinement
         --! constituent of an own variable which is not marked as initialized.

      when 623 =>
         ELStrings.AppendString
            (EStr, "Own variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " occurred in an initialization specification");
         --! Issued if an own variable does not appear in an initialization clause
         --! and is also a
         --! refinement constituent of an own variable that is marked as initialized.

      when 624 =>
         ELStrings.AppendString
            (EStr, "All operators from ancestor package ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " are already visible");
         --! A package must appear in a <b>with</b> clause before types declared in
         --! it can be specified in a <b>use type</b> clause.

         -- nice gap here for generics
      when 630 =>
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is not the name of generic subprogram");
         --! Only generic subprogram can be instantiated.

      when 631 =>
         ELStrings.AppendString
            (EStr, "Generic function found where a generic procedure was expected");
         --! Subprogram kind of generic and its instantiation must match

      when 632 =>
         ELStrings.AppendString
            (EStr, "Generic procedure found where a generic function was expected");
         --! Subprogram kind of generic and its instantiation must match

      when 633 =>
         ELStrings.AppendString
           (EStr, "Generic actual part expected,  generic unit ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " has generic formal parameters");
         --! The number of generic formal and actual parameters must match exactly

      when 634 =>
         ELStrings.AppendString
           (EStr, "Unexpected generic actual part,  generic unit ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " does not have any generic formal parameters");
         --! The number of generic formal and actual parameters must match exactly

      when 635 =>
         ELStrings.AppendString
            (EStr,
             "Incorrect number of generic actual parameters for instantiation of generic unit ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         --! The number of generic formal and actual parameters must match exactly

      when 636 =>
         ELStrings.AppendString
           (EStr,
            "Type ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is not compatible with generic formal parameter ");
         AppendName
           (EStr, ErrNum.Name2, ErrNum.Scope);
         --! See ALRM 12.5.  Each generic formal type parameter must be supplied with an actual type
         --! which is of a compatible class.  Note that SPARK does not have default values for such
         --! associations

      when 637 =>
         ELStrings.AppendString
            (EStr,
             "User-defined generic units are not permitted in SPARK 83");
         --! There are weaknesses in the generic type model of Ada 83 that prevent the implementation
         --! of a safe subset of generics in SPARK 83.  These deficiencies are overcome in Ada 95.
         --! SPARK 83 users may employ the predefined unit Unchecked_Conversion only.

      when 638 =>
         ELStrings.AppendString
            (EStr,
             "Unexpected global annotation.  A generic subprogram  may not reference or update global variables");
         --! A standalone generic subprogram may not have a global annotation.  Note that a subprogram in a
         --! generic package may have a global annotation as long as it only refers to own variables that
         --! are local to the package.

      when 639 =>
         ELStrings.AppendString
            (EStr,
             "A generic formal object may only have default mode or mode in");
         --! SPARK restricts formal objects to being constants in order to avoid concealed information
         --! flows.

      when 640 =>
         ELStrings.AppendString
            (EStr,
             "A generic formal object may only be instantiated with a constant expression");
         --! SPARK restricts formal objects to being constants in order to avoid concealed information
         --! flows.

      when 641 =>
         ELStrings.AppendString
           (EStr,
            "There is no generic subprogram declaration named ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " so a generic body of that name cannot be declared here");
         --! A generic body must be preceded by a generic declaration of the same name.

      when 642 =>
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is not the name of a generic subprogram so a generic body cannot be declared here");
         --! A generic body must be preceded by a generic declaration of the same name.

      when 643 =>
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is a generic function so a generic procedure body cannot be declared here");
         --! A generic body must be preceded by a generic declaration of the same name and kind.

      when 644 =>
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is a generic procedure so a generic function body cannot be declared here");
         --! A generic body must be preceded by a generic declaration of the same name and kind.

      when 645 =>
         ELStrings.AppendString
           (EStr,
            "Actual array element ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is not compatible with the element type ");
         AppendName
           (EStr, ErrNum.Name2, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " of the generic formal parameter");
         --! See ALRM 12.5.  Each generic formal type parameter must be supplied with an actual type
         --! which is of a compatible class.  Note that SPARK does not have default values for such
         --! associations

      when 646 =>
         ELStrings.AppendString
           (EStr,
            "Actual array index ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is not compatible with the index type ");
         AppendName
           (EStr, ErrNum.Name2, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " of the generic formal parameter");
         --! See ALRM 12.5.  Each generic formal type parameter must be supplied with an actual type
         --! which is of a compatible class.  Note that SPARK does not have default values for such
         --! associations

      when 647 =>
         ELStrings.AppendString
           (EStr,
            "Actual array ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " has more dimensions than formal array ");
         AppendName
           (EStr, ErrNum.Name2, ErrNum.Scope);
         --! See ALRM 12.5.  Each generic formal type parameter must be supplied with an actual type
         --! which is of a compatible class.  Note that SPARK does not have default values for such
         --! associations

      when 648 =>
         ELStrings.AppendString
           (EStr,
            "Actual array ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " has fewer dimensions than formal array ");
         AppendName
           (EStr, ErrNum.Name2, ErrNum.Scope);
         --! See ALRM 12.5.  Each generic formal type parameter must be supplied with an actual type
         --! which is of a compatible class.  Note that SPARK does not have default values for such
         --! associations

      when 649 =>
         ELStrings.AppendString
           (EStr,
            "Actual array ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is constrained but the associated formal ");
         AppendName
           (EStr, ErrNum.Name2, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is unconstrained");
         --! See ALRM 12.5.  Each generic formal type parameter must be supplied with an actual type
         --! which is of a compatible class.  Note that SPARK does not have default values for such
         --! associations

      when 650 =>
         ELStrings.AppendString
           (EStr,
            "Actual array ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is unconstrained but the associated formal ");
         AppendName
           (EStr, ErrNum.Name2, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is constrained");
         --! See ALRM 12.5.  Each generic formal type parameter must be supplied with an actual type
         --! which is of a compatible class.  Note that SPARK does not have default values for such
         --! associations

      when 651 =>
         ELStrings.AppendString
           (EStr,
            "Variables of generic types may not be initialized at declaration");
         --! In non-generic code we statically know the value being assigned to the variable and can
         --! check that it is in range.  In the case of a generic we cannot do this because we do not
         --! know the bounds of the variable's type.  The variable may, however, be assigned to in the sequence
         --! of statements in the generic body because generation of run-time checks will provide
         --! suitable protection from out-of-range values.

      when 652 =>
         ELStrings.AppendString
           (EStr,
            "Subtypes of generic types are not permitted");
         --! In non-generic code we statically know the values being used as the range bounds for
         --! a subtype and can check that they are in range.  In the case of a generic we cannot
         --! do this because we do not know the bounds of the variable's type.

      when 653 =>
         ELStrings.AppendString
           (EStr,
            "Constants of generic types are not permitted");
         --! In non-generic code we statically know the value being assigned to the constant and can
         --! check that it is in range.  In the case of a generic we cannot do this because we do not
         --! know the bounds of the constant's type.  A variable, assigned to in the sequence
         --! of statements in the generic body, may be a suitable substitute for such a constant.
      when 654 =>
         AppendName (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is a generic subprogram which must be instantiated before it can be called");
         --! Generic units provide a template for creating callable units and are not directly
         --! callable.
      when 655 =>
         ELStrings.AppendString
           (EStr,
            "Invalid prefix, ");
         AppendName (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is a generic package");
         --! Components of generic packages cannot be accessed directly.  First instantiate the package
         --! and then access components of the instantiation.
      when 656 =>
         ELStrings.AppendString
           (EStr, "The only currently supported attribute in this " &
                  "context is 'Always_Valid");
      when 657 =>
         ELStrings.AppendString
           (EStr, "A 'Always_Valid assertion requires a variable here");
         --! The 'Always_Valid assertion can only be applied to variables or
         --! to components of record variables.
      when 658 =>
         ELStrings.AppendString
           (EStr, "The object in this assertion must have a scalar type");
         --! The 'Always_Valid assertion can only be applied to objects which have
         --! a scalar type. Each field of a record must be marked individually
         --! as always valid.
      when 659 =>
         ELStrings.AppendString
           (EStr, "A 'Always_Valid assertion must be in the same declarative "&
                  "region as contains the declaration of the variable to "&
                  "which it refers");
      when 660 =>
         ELStrings.AppendString
           (EStr, "A 'Always_Valid assertion must not be applied to an "&
                  "object already marked as always valid");
      when 661 =>
         ELStrings.AppendString
           (EStr, "Only a simple variable or a record component can be the "&
                  "subject of an Always_Valid assertion ");
      when 662 =>
         ELStrings.AppendString
           (EStr, "Only Mode in own variables and constituents can be marked "&
                  "using 'Always_Valid");
         --! The 'Always_Valid assertion can only be applied to variables which are
         --! own variables with the mode in, or to subcomponents of records
         --! which are mode in own variables.

      when 700 =>
         ELStrings.AppendString
            (EStr,
             "Mode 'in out' may not be applied to own variables or " &
             "their refinement constituents");
         --! Own variables may be given a mode to indicate that they are system level
         --! inputs
         --! or outputs (i.e. they obtain values from or pass values to the external
         --! environment).  Since effective SPARK design strictly separates inputs from
         --! outputs the mode 'in out' is not permitted.

      when 701 =>
         ELStrings.AppendString
            (EStr, "The mode of this refinement constituent is not consistent with " &
               "its subject:  ");
         AppendName (EStr, ErrNum.Name1, ErrNum.Scope);
         --! If an abstract own variable is given a mode then its refinement
         --! constituents must
         --! all be of the same mode.

      when 702 =>
         ELStrings.AppendString
            (EStr, "Own variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " must be given the mode 'in' to match its earlier announcement ");
         --! Issued if an own variable of an embedded package is not given the
         --! same mode as
         --! the earlier refinement constituent that announced it would exist.

      when 703 =>
         ELStrings.AppendString
            (EStr, "Own variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " must be given the mode 'out' to match its earlier announcement ");
         --! Issued if an own variable of an embedded package is not given the same
         --! mode as
         --! the earlier refinement constituent that announced it would exist.

      when 704 =>
         ELStrings.AppendString
            (EStr, "Own variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " may not have a mode because one was not present in its " &
             "earlier announcement ");
         --! Issued if an own variable of an embedded package is given a mode when
         --! the earlier refinement constituent that announced it would exist did not
         --! have one.

      when 705 =>
         ELStrings.AppendString
            (EStr, "Refinement constituent ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " must be given the mode 'in' to match the child package own " &
             "variable with which it is being associated");
         --! If a refinement constituent is an own variable of a private package then the
         --! constituent must have the same mode as the own variable to which it refers.

      when 706 =>
         ELStrings.AppendString
            (EStr, "Refinement constituent ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " must be given the mode 'out' to match the child package own " &
             "variable with which it is being associated");
         --! If a refinement constituent is an own variable of a private package then the
         --! constituent must have the same mode as the own variable to which it refers.

      when 707 =>
         ELStrings.AppendString
            (EStr, "Refinement constituent ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " may not have a mode because one was not present on the child package " &
               "own variable with which it is being associated");
         --! If a refinement constituent is an own variable of a private package then the
         --! constituent can only be given a mode if the own variable to which it
         --!  refers has one.

      when 708 =>
         ELStrings.AppendString
            (EStr, "Own variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " has a mode and may not appear in an initializes clause");
         --! Mode own variables (stream variables) are implicitly initialized by the
         --! environment
         --! to which they are connected and may not appear in initializes clauses
         --! since this
         --! would require their explicit initialization.

      when 709 =>
         ELStrings.AppendString
            (EStr, "Own variable or constituent ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " has mode 'out' and may not be referenced by a function");
         --! Functions are permitted to reference own variables that are either unmoded
         --! or of
         --! mode 'in'.  Since mode 'out' own variables represent outputs to the
         --! environment,
         --! reading them in a function does not make sense and is not allowed.

      when 710 =>
         ELStrings.AppendString
            (EStr, "The own variable or constituent ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is of mode 'in' and can only have global mode 'in'");
         --! Global modes, if given, must be consistent with the modes of own variables
         --! that appear
         --! in the global list.

      when 711 =>
         ELStrings.AppendString
            (EStr, "The own variable or constituent ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is of mode 'out' and can only have global mode 'out'");
         --! Global modes, if given, must be consistent with the modes of own
         --! variables that appear
         --! in the global list.

      when 712 =>
         ELStrings.AppendString
            (EStr, "The own variable or constituent ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is of either mode 'in' or mode 'out' and  may not have global " &
               "mode 'in out'");
         --! Global modes, if given, must be consistent with the modes of own variables
         --! that appear
         --! in the global list.

      when 713 =>
         ELStrings.AppendString
            (EStr, "The own variable or constituent ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is of mode 'in' and may not appear in a dependency clause as an export");
         --! Own variables with mode 'in' denote system-level inputs; their exportation
         --! is
         --! not allowed.

      when 714 =>
         ELStrings.AppendString
            (EStr, "The own variable or constituent ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is of mode 'out' and may not appear in a dependency clause as an import");
         --! Own variables with mode 'out' denote system-level outputs; their
         --! importation is
         --! not allowed.

      when 715 =>
         ELStrings.AppendString
            (EStr, "Function ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " references external (stream) variables and may only appear directly" &
             " in an assignment or return statement");
         --! To avoid ordering effects, functions which globally access own
         --! variables which
         --! have modes (indicating that they are connected to the external
         --! environment) may
         --! only appear directly in assignment or return statements.
         --! They may not appear as
         --! actual parameters or in any other form of expression.

      when 716 =>
         ELStrings.AppendString
            (EStr, "External (stream) variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " may only appear directly" &
             " in an assignment or return statement; or as an actual parameter to an unchecked conversion");
         --! To avoid ordering effects, own variables which
         --! have modes (indicating that they are connected to the external
         --! environment) may
         --! only appear directly in assignment or return statements. They may not
         --! appear as
         --! actual parameters (other than to instantiations of Unchecked_Conversion) or in any other form of expression.

      when 717 =>
         ELStrings.AppendString
            (EStr, "External (stream) variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is of mode 'in' and may not be assigned to");
         --! Own variables with mode 'in' represent inputs to the system from the
         --! external
         --! environment. As such, assigning to them does not make sense and is not
         --! permitted.

      when 718 =>
         ELStrings.AppendString
            (EStr, "External (stream) variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is of mode 'out' and may not be referenced");
         --! Own variables with mode 'out' represent outputs to the external
         --! environment from the system. As such, referencing them does not make sense
         --! and is not permitted.

      when 719 =>
         ELStrings.AppendString
            (EStr, "External (stream) variables may not be referenced or updated " &
                   "during package elaboration");
         --! Own variables with modes represent inputs and outputs between the external
         --! environment and the system. Referencing or updating them during package
         --! elaboration would introduce ordering effects and is not permitted.

      when 720 =>
         ELStrings.AppendString
            (EStr, "Variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is an external (stream) variable and may not be initialized at " &
               "declaration");
         --! Own variables with modes represent inputs and outputs between the external
         --! environment and the system. Referencing or updating them during package
         --! elaboration would introduce ordering effects and is not permitted.

      when 721 =>
         ELStrings.AppendString
            (EStr, "This refined function global annotation may not reference ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " because it is an external (stream) variable whose abstract subject ");
         AppendName
            (EStr, ErrNum.Name2, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " does not have a mode");
         --! Functions may be used to reference external (stream) variables and the
         --! Examiner
         --! generates the appropriate information flow to show that the value returned
         --!  by
         --! the function is 'volatile'.  If the abstract view of the same function
         --! shows it
         --! referencing an own variable which is not an external stream then the
         --! volatility
         --! of the function is concealed.  The error can be removed either by making the
         --! abstract own variable a mode 'in' stream or by using a procedure instead
         --! of a
         --! function to read the refined stream variable.

      when 722 =>
         ELStrings.AppendString
            (EStr, "The mode on abstract global variable ");
         AppendName
            (EStr, ErrNum.Name2, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " must be made 'in out' to make it consistent with the referencing of" &
             " mode 'in' external (stream) constituent ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " in the refined global annotation");
         --! Where a procedure references an external (stream) variable of mode 'in' the
         --! Examiner constructs appropriate information flow to show that the input
         --! stream
         --! is 'volatile'. If the abstract view shows that the procedure obtains its
         --! result
         --! by simply reading an own variable which is not an external stream then the
         --! volatility is concealed.  The error can be removed either by making the
         --! global
         --! mode of XXX 'in out' or making XXX an external (stream) variable of mode
         --! 'in'.

      when 723 =>
         ELStrings.AppendString
            (EStr, "Variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " must appear in this refined global annotation");
         --! Issued when a global variable which is present in the first (abstract)
         --! global annotation
         --! is omitted from the second (refined) one.

      when 724 =>
         ELStrings.AppendString
            (EStr, "Exit label must match the label of the most closely enclosing " &
               "loop statement");
         --! If an exit statement names a loop label, then the most closely enclosing
         --! loop statement must have a matching label

      when 725 =>
         ELStrings.AppendString
            (EStr, "Protected function or variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " may only appear directly" &
             " in an assignment or return statement");
         --! To avoid ordering effects, protected functions  may
         --! only appear directly in assignment or return statements. They may not
         --! appear as
         --! actual parameters or in any other form of expression.  Ordering effects
         --! occur
         --! because the global state referenced by the protected function may be updated
         --! by another process during expression evaluation.

      when 730 =>
         ELStrings.AppendString
            (EStr, "A loop with no iteration scheme or exit statements may only " &
               "appear as the last " &
               "statement in the outermost scope of the main subprogram (or a task " &
               "body when using the Ravenscar profile)");
         --! If a loop has neither an iteration scheme nor any exit statements then it
         --! will
         --! run forever. Any statements following it will be unreachable. SPARK only
         --! allows
         --! one such loop which must be the last statement of the main program

      when 750 =>
         ELStrings.AppendString
           (EStr, "The identifier ");
         if ErrNum.Name2 /= Error_Types.NoName then
            AppendName
              (EStr, ErrNum.Name2, ErrNum.Scope);
            ELStrings.AppendString
              (EStr, ".");
         end if;
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is either undeclared or not visible at this point");
         ELStrings.AppendString
           (EStr,
            ". An array type may not be used as its own index type");
         --! The type mark used for the index of an array type declaration must not
         --! be the same as the name of the array type being declared

      when 751 =>
         ELStrings.AppendString
           (EStr, "The identifier ");
         if ErrNum.Name2 /= Error_Types.NoName then
            AppendName
              (EStr, ErrNum.Name2, ErrNum.Scope);
            ELStrings.AppendString
              (EStr, ".");
         end if;
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is either undeclared or not visible at this point");
         ELStrings.AppendString
           (EStr,
            ". A record type may not include fields of its own type");
         --! The type mark given for a field in a record type declaration must
         --! not be the same as the name of the record type being declared

      when 752 =>
         ELStrings.AppendString
           (EStr, "The identifier ");
         if ErrNum.Name2 /= Error_Types.NoName then
            AppendName
              (EStr, ErrNum.Name2, ErrNum.Scope);
            ELStrings.AppendString
              (EStr, ".");
         end if;
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is either undeclared or not visible at this point");
         ELStrings.AppendString
           (EStr,
            ". This identifier must appear in a preceding legal global annotation or formal parameter list");
         --! For an identifier to appear legally as an import in a derives annotation,
         --! it must be a formal parameter or must appear legally in a
         --! preceding global annotation and must be of mode 'in' or mode 'in out'

      when 753 =>
         ELStrings.AppendString
           (EStr, "The identifier ");
         if ErrNum.Name2 /= Error_Types.NoName then
            AppendName
              (EStr, ErrNum.Name2, ErrNum.Scope);
            ELStrings.AppendString
              (EStr, ".");
         end if;
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is either undeclared or not visible at this point");
         ELStrings.AppendString
           (EStr,
            ". This identifier must appear in a preceding legal global annotation or formal parameter list");
         --! For an identifier to appear legally as an export in a derives annotation,
         --! it must be a formal parameter or must appear legally in a
         --! preceding global annotation and must be of mode 'out' or mode 'in out'

      when 754 =>
         ELStrings.AppendString
           (EStr, "The identifier ");
         if ErrNum.Name2 /= Error_Types.NoName then
            AppendName
              (EStr, ErrNum.Name2, ErrNum.Scope);
            ELStrings.AppendString
              (EStr, ".");
         end if;
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is either undeclared or not visible at this point");
         ELStrings.AppendString
           (EStr,
            ". This package must be both inherited and withed to be visible here");
         --! For a package name to be visible in Ada context, it must appear in
         --! both the inherit clause and the with clause of the enclosing package

      when 755 =>
         ELStrings.AppendString
           (EStr, "The identifier ");
         if ErrNum.Name2 /= Error_Types.NoName then
            AppendName
              (EStr, ErrNum.Name2, ErrNum.Scope);
            ELStrings.AppendString
              (EStr, ".");
         end if;
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is either undeclared or not visible at this point");
         ELStrings.AppendString
           (EStr,
            ". A parent of a child package must be inherited to be visible here");
         --! A parent of a child package must be inherited (but not withed)
         --! to be visible in that child.

      when 756 =>
         ELStrings.AppendString
           (EStr, "The identifier ");
         if ErrNum.Name2 /= Error_Types.NoName then
            AppendName
              (EStr, ErrNum.Name2, ErrNum.Scope);
            ELStrings.AppendString
              (EStr, ".");
         end if;
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is either undeclared or not visible at this point");
         ELStrings.AppendString
           (EStr,
            ". The grandparent of a child package should not appear in this prefix.");
         --! A grandparent of a child package should not be included in prefixes
         --! referencing a declaration of the child package

      when 757 =>
         ELStrings.AppendString
            (EStr, "The identifer ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is either undeclared or not visible at this point");
         ELStrings.AppendString
           (EStr,
            ". A record field name cannot be the same as its indicated type");

      when 770 =>
         ELStrings.AppendString
            (EStr, "If Any_Priority is defined, Priority and Interrupt_Priority" &
            " must also be defined");
         --! If the type Any_Priority is defined in package System, then the subtypes
         --! Priority and
         --! Interrupt_Priority must also be defined; if support for tasking is not
         --! required, then
         --! the definition of Any_Priority may be removed

      when 771 =>
         ELStrings.AppendString
            (EStr, "The parent type of this subtype must be Any_Priority");
         --! Ada 95 requires that both Priority and Interrupt_Priority be immediate
         --! subtypes
         --! of Any_Priority.

      when 772 =>
         ELStrings.AppendString
            (EStr, "The range of Priority must contain at least 30 values; LRM D.1(26)");
         --! Ada 95 requires that the range of the subtype Priority include at least
         --! 30 values;
         --! this requirement is stated in the Ada 95 Language Reference Manual at
         --! D.1(26)

      when 773 =>
         ELStrings.AppendString
            (EStr, "Priority'First must equal Any_Priority'First; LRM D.1(10)");
         --! Ada 95 requires that task priority types meet the following criteria,
         --! the second of which is relevant to this error:
         --! <li>subtype Any_Priority is Integer range implementation-defined;
         --! <li>subtype Priority is Any_Priority range Any_Priority'First .. implementation-defined;
         --! <li>subtype Interrupt_Priority is Any_Priority range Priority'Last+1 .. Any_Priority'Last

      when 774 =>
         ELStrings.AppendString
            (EStr, "Interrupt_Priority'First must equal Priority'Last + 1; LRM D.1(10)");
         --! Ada 95 requires that task priority types meet the following criteria,
         --! the third of which is relevant to this error:
         --! <li>subtype Any_Priority is Integer range implementation-defined;
         --! <li>subtype Priority is Any_Priority range Any_Priority'First .. implementation-defined;
         --! <li>subtype Interrupt_Priority is Any_Priority range Priority'Last+1 .. Any_Priority'Last

      when 775 =>
         ELStrings.AppendString
            (EStr, "Interrupt_Priority'Last must equal Any_Priority'Last; LRM D.1(10)");
         --! Ada 95 requires that task priority types meet the following criteria,
         --! the third of which is relevant to this error:
         --! <li>subtype Any_Priority is Integer range implementation-defined;
         --! <li>subtype Priority is Any_Priority range Any_Priority'First .. implementation-defined;
         --! <li>subtype Interrupt_Priority is Any_Priority range Priority'Last+1 .. Any_Priority'Last

      when 776 =>
         ELStrings.AppendString
           (EStr, "In SPARK95 mode, only packages Standard, System, Ada.Real_Time " &
              "and Ada.Interrupts may be specified in the config file");
         --! In SPARK95 mode, the packages that may be specified in the target
         --! configuration file are: Standard, System, Ada.Real_Time and Ada.Interrupts.
         --! The latter two are ignored unless the Ravenscar profile is selected.

      when 777 =>
         ELStrings.AppendString
           (EStr, "In package System, Priority must be an immediate subtype of Integer");
         --! Ada 95, and hence SPARK95, defines Priority as being an immediate
         --! subtype of Integer

      when 778 =>
         ELStrings.AppendString
           (EStr, "This identifier is not valid at this point in the target "&
                  "configuration file");
         --! The specified identifier cannot be used here; it is most probably
         --! either not valid in the target configuration file at all, or might
         --! be valid in a different package, but not here.

      when 779 =>
         ELStrings.AppendString
           (EStr, "Definition of this package in the target configuration "&
                  "file is not allowed in SPARK83 mode");
         --! In SPARK83 mode, only package Standard may be specified in the
         --! target configuration file.

      when 780 =>
         ELStrings.AppendString
           (EStr, "Type ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " must be private");
         --! This type may only be declared as private in the target configuration
         --! file.

      when 781 =>
         ELStrings.AppendString
           (EStr, "The lower bound of a signed integer type declaration " &
                  "must be greater than or equal to System.Min_Int");
         --! This error can only be generated in SPARK95 mode when the configuration
         --! file specifies
         --! a value for System.Min_Int

      when 782 =>
         ELStrings.AppendString
           (EStr, "The upper bound of a signed integer type declaration " &
                  "must be less than or equal to System.Max_Int");
         --! This error can only be generated in SPARK95 mode when the configuration
         --! file specifies
         --! a value for System.Max_Int

      when 783 =>
         ELStrings.AppendString
           (EStr, "Modulus must be less than or equal to System.Max_Binary_Modulus");
         --! This error can only be generated in SPARK95 mode when the configuration
         --! file specifies
         --! a value for System.Max_Binary_Modulus

      when 784 =>
         ELStrings.AppendString
           (EStr, "System.Max_Binary_Modulus must be a positive power of 2");

      when 785 =>
         ELStrings.AppendString
           (EStr, "The number of digits specified exceeds the value defined "&
                  "for System.Max_Digits");
         --! The maximum decimal precision for a floating point type, where
         --! a range specification has not been included, is defined
         --! by System.Max_Digits

      when 786 =>
         ELStrings.AppendString
           (EStr, "The number of digits specified exceeds the value defined "&
                  "for System.Max_Base_Digits");
         --! The maximum decimal precision for a floating point type, where
         --! a range specification has been included, is defined
         --! by System.Max_Base_Digits

      when 787 =>
         ELStrings.AppendString
           (EStr, "Digits value must be positive");

      when 788 =>
         ELStrings.AppendString
           (EStr, "Delta value must be positive");

      when 789 =>
         ELStrings.AppendString
           (EStr, "The only currently supported type attribute in this " &
                  "context is 'Base");

      when 790 =>
         ELStrings.AppendString
           (EStr, "A base type assertion requires a type here");

      when 791 =>
         ELStrings.AppendString
           (EStr, "The base type in this assertion must be a predefined type");
         --! Predefined types are those defined either by the language, or in package
         --! Standard, using the configuration file mechanism</i>

      when 792 =>
         ELStrings.AppendString
           (EStr, "The types in this assertion must both be either floating point " &
                  "or signed integer");

      when 793 =>
         ELStrings.AppendString
           (EStr, "The base type in a base type assertion must have a defined range");
         --! The ranges for the predefined types will generally be specified in the
         --! target configuration file</i>

      when 794 =>
         ELStrings.AppendString
           (EStr, "The range of the base type in a base type assertion must be at " &
                  "least as large as the type which is the subject of the assertion");

      when 795 =>
         ELStrings.AppendString
           (EStr, "A base type assertion must be in the same declarative region "&
                  "as that of the full type definition");

      when 796 =>
         ELStrings.AppendString
           (EStr, "This type already has a base type defined: either it already has a " &
                  "base type assertion, or it is a predefined type");
         --! If the type which is the subject of the assertion is not a derived type,
         --! i.e. it is its own base type, then this error is raised</i>

      when 797 =>
         ELStrings.AppendString
           (EStr, "The base type in a floating point base type assertion must " &
              "have a defined accuracy");

      when 798 =>
         ELStrings.AppendString
           (EStr, "The accuracy of the base type in a base type assertion must be at " &
                  "least that of the type which is the subject of the assertion");
      when 799 =>
         ELStrings.AppendString
           (EStr, "Only a simple type can be the subject of a base type assertion ");

      when 800 =>
         ELStrings.AppendString
            (EStr, "Modulus must be a positive power of 2");
         --! In SPARK, modular types must have a modulus which is a positive
         --! power of 2

      when 801 =>
         ELStrings.AppendString
            (EStr,
             "Modular types may only be used in SPARK95");
         --! Ada83 (and hence SPARK83) does not include modular types

      when 803 =>
         ELStrings.AppendString
            (EStr,
             "Unary arithmetic operators are not permitted for modular types");
         --! Unary arithmetic operators are of little value.  The &quot;abs&quot; and &quot;+&quot; operators
         --! have no effect for modular types, and so are not required.  The unary minus
         --! operator is a source of potential confusion, and so is not permitted in
         --! SPARK

      when 804 =>
         ELStrings.AppendString
            (EStr,
             "Universal expression may not be implicitly converted to a modular " &
               "type here. " &
               "Left hand operand requires qualification to type ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         --! A universal expression cannot be used as the left hand operand of a binary
         --! operator if the right hand operand is of a modular type.  Qualification of
         --! the
         --! left hand expression is required in this case.

      when 805 =>
         ELStrings.AppendString
            (EStr,
             "Universal expression may not be implicitly converted to a modular " &
               "type here. " &
               "Right hand operand requires qualification to type ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         --! A universal expression cannot be used as the right hand operand of a binary
         --! operator if the left hand operand is of a modular type.  Qualification of
         --! the
         --! right hand expression is required in this case.

      when 814 =>
         ELStrings.AppendString
            (EStr,
             "Default_Bit_Order must be of type Bit_Order");
         --! The only possible type for the constant System.Default_Bit_Order
         --! is System.Bit_Order when it appears in the configuration file

      when 815 =>
         ELStrings.AppendString
            (EStr,
             "The only allowed values of Default_Bit_Order are Low_Order_First and High_Order_First");
         --! System.Bit_Order is implicity declared in package System when a configuration
         --! file is given. This is an enumeration type with only two literals
         --! Low_Order_First and High_Order_First

      when 820 =>
         ELStrings.AppendString
            (EStr,
             "Abstract types are not currently permitted in SPARK");
         --! Only non-abstract tagged types are currently supported.  It is hoped to
         --! lift this
         --! restriction in a future Examiner release.

      when 821 =>
         ELStrings.AppendString
           (EStr,
            "This type declaration must be a tagged record because it's " &
              "private type is tagged");
         --! If a type is declared as &quot;tagged private&quot; then its full declaration must
         --! be a tagged
         --! record.

      when 822 =>
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is not a tagged type; only tagged types may be extended");
         --! In SPARK, &quot;new&quot; can only be used to declare a type extension; other
         --! derived types are not permitted.

      when 823 =>
         ELStrings.AppendString
           (EStr,
            "This type may not be extended in the same package in which it is declared");
         --! SPARK only permits types from another library package to be extended.
         --! This rule prevents
         --! overloading of inherited operations.

      when 824 =>
         ELStrings.AppendString
           (EStr,
            "This package already extends a type from package ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            ".  Only one type extension per package is permitted");
         --! SPARK only permits one type extension per package.  This rule prevents
         --! overloading of inherited operations.

      when 825 =>
         ELStrings.AppendString
           (EStr,
            "Type ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " expected in order to complete earlier private extension");
         --! Since SPARK only permits one type extension per package it follows that
         --! the declaration
         --! &quot;new XXX with private&quot; in a package visible part must be paired with &quot;new
         --! XXX with record...&quot;
         --! in its private part.  The ancestor type XXX must be the same in both
         --! declarations.

      when 826 =>
         ELStrings.AppendString
           (EStr,
            "Type extension is not permitted in SPARK 83");
         --! Type extension is an Ada 95 feature not included in Ada or SPARK 83.

      when 827 =>
         ELStrings.AppendString
           (EStr,
            "The actual parameter associated with a tagged formal parameter in an " &
            "inherited operation must be an object not an expression");
         --! There are several reasons for this SPARK rule.  Firstly, Ada requires
         --! tagged parameters to be
         --! passed by reference and so an object must exist at least implicitly.
         --! Secondly, in order to
         --! perform flow analysis of inherited subprogram calls, the Examiner needs
         --! identify what subset of the
         --! information available at the point of call
         --! is passed to and from the called subprogram.  Since information can only
         --! flow through objects it
         --! follows that actual parameter must be an object.

      when 828 =>
         ELStrings.AppendString
           (EStr,
            "Tagged types and tagged type extensions may only be declared in " &
            "library-level package specifications");
         --! This SPARK rule facilitates the main uses of tagged types while greatly
         --! simplifying visibility rules.

      when 829 =>
         ELStrings.AppendString
           (EStr,
            "Illegal re-declaration: this subprogram shares the same name as the " &
              "inheritable root operation ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " but does not override it");
         --! To avoid overloading, SPARK prohibits more than one potentially visible
         --! subprogram having the
         --! same name.

      when 830 =>
         ELStrings.AppendString
           (EStr,
            "A private type may not be implemented as a " &
            "tagged type or an extension of a tagged type");
         --! This rule means that a private type can only be implemented as a tagged
         --! type if the private
         --! type itself is tagged.

      when 831 =>
         ELStrings.AppendString
           (EStr,
            "Extended tagged types may only be converted in the direction of their " &
            "root type");
         --! This is an Ada rule: type conversions simply omit unused fields of the
         --! extended type.  It follows
         --! that conversions must be in the direction of the root type.

      when 832 =>
         ELStrings.AppendString
           (EStr,
            "Only tagged objects, not expressions, may be converted");
         --! For flow analysis purposes the Examiner needs to know what subset of the
         --! information in
         --! the unconverted view is available in the converted view.  Since
         --! information can only flow
         --! through objects it follows that only objects can be converted.

      when 833 =>
         ELStrings.AppendString
           (EStr,
            "Invalid record aggregate: type ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " has a private ancestor");
         --! If an extended type has a private ancestor then an extension aggregate
         --!  must be used rather
         --! than a normal aggregate.

      when 834 =>
         ELStrings.AppendString
           (EStr,
            "Null records are only permitted if they are tagged");
         --! An empty record can have no use in a SPARK program others than as a
         --! root type from which
         --! other types can be derived and extended. For this reason, null records
         --! are only allowed
         --! if they are tagged.

      when 835 =>
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is not an extended tagged record type");
         --! An extension aggregate is only appropriate if the record type it is
         --! defining is
         --! an extended record.  A normal aggregate should be used for other record
         --! (and array)
         --! types.

      when 836 =>
         ELStrings.AppendString
           (EStr,
            "This expression does not represent a valid ancestor type of the " &
              "aggregate ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         --! The expression before the reserved word &quot;with&quot; must be of an ancestor type
         --! of the overall aggregate
         --! type.  In SPARK, the ancestor expression may not be a subtype mark.

      when 837 =>
         ELStrings.AppendString
           (EStr,
            "Invalid record aggregate: there is a private ancestor between the " &
              "type of this expression " &
              "and the type of the aggregate ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         --! The ancestor type can be an tagged type with a private extension;
         --! however, there must be no private
         --! extensions between the ancestor type and the type of the aggregate.

      when 838 =>
         ELStrings.AppendString
           (EStr,
            "Incomplete aggregate: null record cannot be used here because fields in ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " require values");
         --! The aggregate form &quot;with null record&quot; can only be used if the type of
         --! the aggregate is a null record
         --! extension of the ancestor type.  If any fields are added between the
         --! ancestor type and the aggregate type then
         --! values need to be supplied for them so &quot;null record&quot; is inappropriate.

      when 839 =>
         ELStrings.AppendString
           (EStr,
            "This package already contains a root tagged type or tagged" &
              " type extension. " &
              "Only one such declaration per package is permitted");
         --! SPARK permits one root tagged type or one tagged type extension per
         --! package, but not both.  This rule prevents
         --! the declaration of illegal operations with more than one controlling
         --! parameter.

      when 840 =>
         ELStrings.AppendString
           (EStr,
            "A tagged or extended type may not appear here. " &
              "SPARK does not permit the declaration of primitive functions " &
              "with controlling results");
         --! A primitive function controlled by its return result would be almost
         --! unusable in SPARK because a data
         --! flow error would occur wherever it was used.

      when 841 =>
         ELStrings.AppendString
           (EStr,
            "The return type in the declaration of this function contained an error. " &
            "It is not possible to check the validity of this return type");
         --! Issued when there is an error in the return type on a function's
         --! initial declaration.  In this situation
         --! we cannot be sure what return type is expected in the function's body.
         --! It would be misleading to simply
         --! report a type mismatch since the types might match perfectly and both
         --! be wrong.  Instead, the Examiner reports
         --! the above error and refuses to analyse the function body until its
         --! specification is corrected.

      when 842 =>
         ELStrings.AppendString
           (EStr,
            "Pragma Atomic_Components is not permitted in SPARK when the " &
              "Ravenscar profile is selected");

      when 843 =>
         ELStrings.AppendString
           (EStr,
            "Pragma Volatile_Components is not permitted in SPARK when the " &
              "Ravenscar profile is selected");

      when 850 =>
         ELStrings.AppendString
            (EStr,
             "This construct may only be used when the Ravenscar profile " &
             "is selected");
         --! Support for concurrent features of the SPARK language, including
         --! protected objects,
         --! tasking, etc. are only supported when the Ravenscar profile is selected.

      when 851 =>
         ELStrings.AppendString
            (EStr,
             "The parameter to pragma Atomic must be a simple_name");
         --! The parameter to pragma Atomic must be a simple_name; and may not be
         --! passed using
         --! a named association

      when 852 =>
         ELStrings.AppendString
            (EStr,
             "pragma Atomic may only appear in the same immediate scope as the " &
               "type to which it applies");
         --! This is an Ada rule (pragma Atomic takes a local name see LRM 13.1(1)).
         --! Note that this precludes the use of pragma Atomic on a predefined type

      when 853 =>
         ELStrings.AppendString
            (EStr,
             "pragma Atomic may only apply to a scalar base type, or to a non-tagged " &
             "record type with exactly 1 field that is a predefined scalar type");
         --! pragma Atomic may only be applied to base types that are scalar. (i.e.
         --! enumeration types, integer types, real types, modular types) or a non-tagged
         --! record type with a single field which is a predefined scalar type, such
         --! as Integer, Character or Boolean. As an additional special case, a
         --! record type with a single field of type System.Address is also allowed.

      when 854 =>
         ELStrings.AppendString
            (EStr,
             "pragma Atomic takes exactly one parameter");

      when 855 =>
         ELStrings.AppendString
            (EStr,
             "The type of own variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is not consistent with its modifier");
         --! An own variable with a task modifier must be of a task type. A task own
         --! variable must have the task modifier. An own variable with a protected
         --! modifier must be a protected object, suspension object or pragma
         --! atomic type. A protected or suspension object own variable must
         --! have the protected modifier.

      when 858 =>
         ELStrings.AppendString
            (EStr,
             "A variable that appears in a protects property list may not appear " &
               "in a refinement clause");
         --! A variable in a protects list is effectively protected and hence
         --! cannot be refined.

      when 859 =>
         ELStrings.AppendString
            (EStr,
             "A protected own variable may not appear in a refinement clause");
         --! Protected state cannot be refined or be used as refinement constituents

      when 860 =>
         ELStrings.AppendString
            (EStr,
             "Own variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " appears in a protects list and hence must appear in the initializes " &
               "clause");
         --! Protected state (including all refinement constituents) must be initialized.

      when 863 =>
         ELStrings.AppendString
            (EStr, "Own variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is protected and may not appear in an initializes clause");
         --! Protected own variables must always be initialized, and should not appear in
         --! initializes annotations.

      when 864 =>
         ELStrings.AppendString
            (EStr, "Unexpected initialization specification - ");
         ELStrings.AppendString
            (EStr, "all own variables of this package ");
         ELStrings.AppendString
            (EStr, "are either implicitly initialized, or do not require " &
               "initialization");
         --! An own variable initialization clause and that of its refinement
         --!  constituents must be
         --! consistent.

      when 865 =>
         ELStrings.AppendString
           (EStr, "Field ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is part of the ancestor part of this aggregate and does not " &
              "require a value here");
         --! An extension aggregate must supply values for all fields that are part of
         --! the overall aggregate
         --! type but not those which are part of the ancestor part.

      when 866 =>
         ELStrings.AppendString
            (EStr, "The expression in a delay_until statement must be of type " &
               "Ada.Real_Time.Time");
         --! When the Ravenscar Profile is selected, the delay until statement may be
         --! used.
         --! The argument of this statement must be of type Ada.Real_Time.Time.

      when 867 =>
         ELStrings.AppendString
           (EStr, "Subprogram ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " contains a delay statement but does not have a delay property");
         --! Any subprogram that may call delay until must have a delay property in a
         --! declare annotation.  Your subprogram is directly or indirectly making a
         --! call to delay until

      when 868 =>
         ELStrings.AppendString
           (EStr, "Protected object ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " may only be declared immediately within a library package");
         --! This error message is issued if a type mark representing a protected type
         --! appears anywhere other than in
         --! a library level variable declaration or library-level own variable type
         --! announcement.

      when 869 =>
         ELStrings.AppendString
           (EStr, "Protected type ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " already contains an Entry declaration; only one Entry is permitted");
         --! The Ravenscar profile prohibits a protected type from declaring more than
         --! one entry.

      when 870 =>
         ELStrings.AppendString
           (EStr, "Protected type ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " does not have any operations, at least one operation must " &
              "be declared");
         --! A protected type which provides no operations can never be used so SPARK
         --! requires the
         --! declaration of at least one.

      when 872 =>
         ELStrings.AppendString
           (EStr, "Variable ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is not protected; only protected items may be globally " &
              "accessed by protected operations");
         --! In order to avoid the possibility of shared data corruption,
         --! SPARK prohibits protected operations
         --! from accessing unprotected data items.

      when 873 =>
         ELStrings.AppendString
           (EStr, "This subprogram requires a global annotation which references" &
              " the protected type name ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         --! In order to statically-detect certain bounded errors defined by the
         --! Ravenscar profile, SPARK
         --! requires every visible operation of protected type to globally
         --! reference the abstract state of
         --! the type.

      when 874 =>
         ELStrings.AppendString
           (EStr, "Protected state ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " must be initialized at declaration");
         --! Because there is no guarantee that a concurrent thread that initializes a
         --! protected object will be executed before one that reads it, the only way
         --! we can be sure that a protected object is properly initialized is to do
         --! so at the point of declaration. You have either declared some protected
         --! state and not included an initialization or you have tried to initialize
         --! some protected state in package body elaboration.

      when 875 =>
         ELStrings.AppendString
           (EStr, "Protected type expected; access discriminants may only " &
              "refer to protected types in SPARK");
         --! Access discriminants have been allowed in SPARK solely to allow devices
         --! made up of co-operating
         --! Ravenscar-compliant units to be constructed.  For this reason only
         --! protected types may appear in
         --! access discriminants.

      when 876 =>
         ELStrings.AppendString
           (EStr, "This protected type or task declaration must include " &
              "either a pragma Priority or " &
              "pragma Interrupt_Priority");
         --! To allow the static detection of certain bounded errors defined by the
         --! Ravenscar profile, SPARK requires
         --! an explicitly-set priority for each protected type, task type or object
         --! of those types.  The System.Default_Priority
         --! may used explicitly provided package System has been defined in the
         --! configuration file.

      when 877 =>
         ELStrings.AppendString
           (EStr, "Priority values require an argument which is an expression " &
              "of type integer");

      when 878 =>
         ELStrings.AppendString
           (EStr, "This protected type declaration contains a pragma Attach_Handler " &
              "and must therefore also " &
              "include a pragma Interrupt_Priority");
         --! To allow the static detection of certain bounded errors defined by the
         --! Ravenscar profile, SPARK requires
         --! an explicitly-set priority for each protected type or object.
         --! The System.Default_Priority may used explicitly
         --! provided package System has been defined in the configuration file.

      when 879 =>
         ELStrings.AppendString
           (EStr, "Unexpected pragma ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, ": this pragma may not appear here");
         --! pragma Interrupt_Priority must be the first item in
         --! a protected type declaration or task type declaration; pragma Priority
         --! must be the first item in
         --! a protected type declaration, task type declaration or the main program.

      when 880 =>
         ELStrings.AppendString
           (EStr, "Pragma Priority or Interrupt_Priority expected here");
         --! Issued when a pragma other than Priority or Interrupt_Priority appears as
         --! the first item in
         --! a protected type or task type declaration.

      when 881 =>
         ELStrings.AppendString
           (EStr, "The priority of ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " must be in the range ");
         AppendName
           (EStr, ErrNum.Name2, ErrNum.Scope);
         --! See LRM D.1(17).

      when 882 =>
         ELStrings.AppendString
           (EStr, "Integrity property requires an argument which is an expression " &
              "of type Natural");

      when 883 =>
         ELStrings.AppendString
           (EStr, "Pragma Interrupt_Handler may not be used; SPARK does not support" &
              " the dynamic attachment of interrupt handlers [LRM C3.1(9)]");
         --! Interrupt_Handler is of no use unless dynamic attachment of interrupt
         --! handlers is to be used.

      when 884 =>
         ELStrings.AppendString
           (EStr, "Pragma Attach_Handler is only permitted immediately after the " &
              "corresponding protected procedure declaration in a protected type " &
              "declaration");

         --! Pragma Attach_Handler may only be used within a protected type
         --! declaration.  Furthermore, it must
         --! immediately follow a protected procedure declaration with the same name
         --! as the first argument to the pragma.

      when 885 =>
         ELStrings.AppendString
           (EStr, "Pragma Attach_Handler may only be applied to a procedure with " &
              "no parameters");
         --! See LRM C.3.1(5).

      when 887 =>
         ELStrings.AppendString
           (EStr, "A discriminant may only appear alone, not in an expression");
         --! Issued when a task or protected type priority is set using an expression
         --! involving a discriminant. The use
         --! of such an expression greatly complicates the static evaluation of the
         --! priority of task or protected subtypes
         --! thus preventing the static elimination of certain Ravenscar bounded errors.

      when 888 =>
         ELStrings.AppendString
           (EStr, "Unexpected Delay, ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " already has a Delay property");
         --! A procedure may only have a maximum of one delay annotation.

      when 889 =>
         ELStrings.AppendString
           (EStr, "The own variable ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " must have the suspendable property");
         --! The type used to declare this object must be a protected type with and
         --! entry or a suspension object type

      when 890 =>
         ELStrings.AppendString
            (EStr, "The name ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " already appears in the suspends list");
         --! Items may not appear more than once in an a suspends list.

      when 891 =>
         ELStrings.AppendString
           (EStr, "Task type or protected type required");
         --! Issued in a subtype declaration where the constraint is a discriminant
         --! constraint.  Only task and protected types
         --! may take a discriminant constraint as part of a subtype declaration.

      when 892 =>
         ELStrings.AppendString
           (EStr, "Array type, task type or protected type required");
         --! Issued in a subtype declaration where the constraint is a either a
         --! discriminant constraint or an index
         --! constraint (these two forms cannot always be distinguished
         --!  syntactically).  Only task and protected types
         --! may take a discriminant constraint and only array types may
         --! take an index constraint as part of a
         --! subtype declaration.

      when 893 =>
         ELStrings.AppendString
           (EStr,
            "Number of discriminant constraints differs from number of known " &
              "discriminants of type ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         --! Issued in a subtype declaration if too many or two few discriminant
         --! constraints are supplied.

      when 894 =>
         ELStrings.AppendString
           (EStr,
            "Only variables of a protected type may be aliased");
         --! SPARK supports the keyword aliased in variable declarations only so
         --! that protected and task types can support
         --! access discriminants.  Since it has no other purpose it may not be
         --! used except in a protected object declaration.

      when 895 =>
         ELStrings.AppendString
           (EStr,
            "Attribute Access may only be applied to variables which are declared " &
              "as aliased, variable ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             " is not aliased");
         --! This is a slightly annoying Ada issue.  Marking a variable as aliased
         --! prevents it being placed in a register
         --! which would make pointing at it hazardous; however, SPARK only permits
         --! 'Access on protected types which are
         --! limited and therefore always passed by reference anyway and immune from
         --! register optimization.   Requiring
         --! aliased on protected objects that will appear in discriminant
         --! constraints is therefore unwanted syntactic sugar
         --! only.

      when 896 =>
         ELStrings.AppendString
            (EStr, "The task type ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " does not have an associated body");
         --! Issued at the end of a package body if a task type declared in its
         --!  specification contains neither a body
         --! nor a body stub for it.

      when 897 =>
         ELStrings.AppendString
            (EStr, "The protected type ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " does not have an associated body");
         --! Issued at the end of a package body if a protected type declared in
         --! its specification contains neither a body
         --! nor a body stub for it.

      when 898 =>
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr, " is not a protected or task type which requires a body");
         --! Issued if a body or body stub for a task or protected type is
         --! encountered and there is no matching specification.

      when 899 =>
         ELStrings.AppendString
           (EStr, "A body for type ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " has already been declared");
         --! Issued if a body or body stub for a task or protected type is
         --! encountered and an earlier body
         --! has already been encountered.

      when 901 =>
         ELStrings.AppendString
           (EStr, "Suspension object ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " may only be declared immediately within a library package " &
              "specification or body");
         --! Suspension objects must be declared at library level. They cannot
         --!  be used in protected type state or as local variables in subprograms.

      when 902 =>
         ELStrings.AppendString
           (EStr, "Recursive use of typemark ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " in known descriminant");

      when 903 =>
         ELStrings.AppendString
            (EStr, "Protected or suspension object types cannot be used to " &
               "declare constants");
         --! Protected and suspension objects are used to ensure integrity of shared
         --! objects. If it is necessary to share constant data then these constructs
         --! should not be used.

      when 904 =>
         ELStrings.AppendString
            (EStr, "Protected or suspension objects cannot be used as subprogram " &
               "parameters");
         --! SPARK does not currently support this feature.

      when 905 =>
         ELStrings.AppendString
            (EStr, "Protected or suspension objects cannot be returned from functions");
         --! SPARK does not currently support this feature.

      when 906 =>
         ELStrings.AppendString
            (EStr, "Protected or suspension objects cannot be used in composite types");
         --! Protected and suspension objects cannot be used in record or array
         --! structures.

      when 907 =>
         ELStrings.AppendString
           (EStr, "Delay until must be called from a task or unprotected " &
              "procedure body");
         --! You are calling delay until from an invalid construct.
         --! Any construct that calls delay until must have a delay property in the
         --! declare annotation. This construct must be one of a task or procedure body
         --!

      when 908 =>
         ELStrings.AppendString
           (EStr, "Blocking properties are not allowed in protected scope");
         --! Procedures in protected scope must not block and therefore blocking
         --! properties are prohibited

      when 909 =>
         ELStrings.AppendString
           (EStr, "Object ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " cannot suspend");
         --! You are either applying the suspendable property to an own variable
         --! that cannot suspend or you have declared a variable (whose own variable
         --! has the suspendable property) which cannot suspend. Or you have used an
         --! item in a suspends list that does not have the suspendable property.
         --! An object can only suspend if it is a suspension object or a protected
         --! type with an entry.

      when 910 =>
         ELStrings.AppendString
           (EStr, "Name ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " must appear in the suspends list property for the enclosing unit");
         --! Protected entry calls and calls to
         --! Ada.Synchronous_Task_Control.Suspend_Until_True may block the currently
         --! executing task. SPARK requires you announce this fact by placing the
         --! actual callee name in the suspends list for the enclosing unit.

      when 911 =>
         ELStrings.AppendString
           (EStr, "The argument in pragma Priority for the main program must be an " &
              "integer literal " &
              "or a local constant of static integer value");
         --! If the main program priority is not an integer literal then you
         --! should declare a constant that has the required value in the
         --! declarative part of the main program prior to the position of the pragma.

      when 912 =>
         ELStrings.AppendString
           (EStr, "This call contains a delay property that is not propagated to the " &
              "enclosing unit");
         --! The call being made has a declare annotation that contains a delay
         --! property. SPARK requires that this property is propagated up
         --! the call chain and hence must appear in a declare annotation
         --! for the enclosing unit.

      when 913 =>
         ELStrings.AppendString
           (EStr, "This call has a name in its suspends list which is not " &
              "propagated to the enclosing unit");
         --! The call being made has a declare annotation that contains a suspends
         --! list. SPARK requires that the entire list is propagated up the call
         --! chain and hence must appear in a declare annotation for the enclosing unit.

      when 914 =>
         ELStrings.AppendString
           (EStr, "The name ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " specified in the suspends list can never be called");
         --! You have specified the name of a protected or suspension object in
         --! the suspends list that can never be called by this procedure or task.

      when 915 =>
         ELStrings.AppendString
            (EStr, "Procedure ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " has a delay property but cannot delay");
         --! You have specified a delay property for this procedure but delay
         --! until can never be called from it.

      when 916 =>
         ELStrings.AppendString
           (EStr, " Protected object ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " has a circular dependency in subprogram ");
         AppendName
           (EStr, ErrNum.Name2, ErrNum.Scope);
         --! The type of the protected object mentions the protected object name in
         --! the derives list for the given subprogram

      when 917 =>
         ELStrings.AppendString
            (EStr, "Procedure ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " cannot be called from a protected action");
         --! The procedure being called may block and hence cannot be called from
         --! a protected action.

      when 918 =>
         ELStrings.AppendString
            (EStr, "The delay property is not allowed for ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         --! The delay property may only be applied to a procedure

      when 919 =>
         ELStrings.AppendString
            (EStr, "The priority property is not allowed for ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         --! The priority property can only be applied to protected own variables
         --! which are type announced. If the type has been declared it must be a
         --! protected type

      when 920 =>
         ELStrings.AppendString
           (EStr, "The suspends property is not allowed for ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         --! The suspends property may only be applied to task type specifications
         --! and procedures

      when 921 =>
         ELStrings.AppendString
           (EStr, "The identifier ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is not recognised as a component of a property list");
         --! The property list can only specify the reserved word delay, suspends or
         --! priority.

      when 922 =>
         ELStrings.AppendString
           (EStr, "The own variable ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " must have the priority property");
         --! In order to perform the ceiling priority checks the priority property must
         --! be given to all own variables of protected type.

      when 923 =>
         ELStrings.AppendString
           (EStr, "The procedure ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " cannot be called from a function as it has a blocking side effect");
         --! Blocking is seen as a side effect and hence procedures that potentially
         --! block cannot be called from functions.

      when 924 =>
         ELStrings.AppendString
           (EStr, "The suspendable property is not allowed for ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         --! Objects that suspend must be declared as own protected variables

      when 925 =>
         ELStrings.AppendString
           (EStr, "The own variable or task ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " must have a type announcement");
         --! Own variables of protected type and own tasks must have a type announcement

      when 926 =>
         ELStrings.AppendString
           (EStr, "Illegal declaration of task ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, ". Task objects must be declared at library level");
         --! Task objects must be declared in library level package specifications or
         --! bodies.

      when 927 =>
         ELStrings.AppendString
           (EStr, "The own task annotation for this task is missing the name ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " in its suspends list");
         --! The task type declaration has name XXX in its list and this must appear
         --! in the own task annotation

      when 928 =>
         ELStrings.AppendString
           (EStr, "Private elements are not allowed for protected type ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         --! Protected type XXX has been used to declare a protected, moded own variable.
         --! Protected, moded own variables are refined onto a set of virtual elements with
         --! the same mode. As such private elements are not allowed.

      when 929 =>
         ELStrings.AppendString
           (EStr, "Unexpected declare annotation. Procedure ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " should have the declare annotation on the specification");
         --! Declare annotations cannot appear on the procedure body if it appears
         --! on the procedure specification

      when 930 =>
         ELStrings.AppendString
           (EStr, "Task ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " does not appear in the own task annotation for this package");
         --! A task has been declared that is not specified as an own task of the
         --! package.

      when 931 =>
         ELStrings.AppendString
           (EStr, "Task ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " does not have a definition");
         --! A task name appears in the own task annotation for this package but
         --! is never declared

      when 932 =>
         ELStrings.AppendString
           (EStr, "The priority for protected object ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " does not match that given in the own variable declaration");
         --! The priority given in the priority property must match that given in
         --! the protected type.

      when 933 =>
         ELStrings.AppendString
           (EStr, "A pragma Priority is required for the main program when " &
              "Ravenscar Profile is enabled");
         --! When SPARK profile Ravenscar is selected, all tasks, protected objects
         --! and the main program must explicitly be assigned a priority.

      when 934 =>
         ELStrings.AppendString
           (EStr, "Priority ceiling check failure: the priority of ");
         AppendName
           (EStr, ErrNum.Name2, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is less than that of ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         --! The active priority of a task is the higher of its base priority
         --! and the ceiling priorities of all protected objects that it is
         --! executing. The active priority at the point of a call to a
         --! protected operation must not exceed the ceiling priority of the
         --! callee.

      when 935 =>
         ELStrings.AppendString
           (EStr, "The own variable ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " must have the interrupt property");
         --! An own variable has been declared using a protected type with a pragma
         --! attach handler. Such objects are used in interrupt processing and must
         --! have the interrupt property specified in their own variable declaration

      when 936 =>
         ELStrings.AppendString
           (EStr, "The interrupt property is not allowed for ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         --! The interrupt property can only be applied to protected own variables
         --! that are type announced. If the type is declared then it must be a
         --! protected type that contains an attach handler

      when 937 =>
         ELStrings.AppendString
           (EStr, "The protects property is not allowed for ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         --! The protects property can only be applied to protected own variables
         --! that are type announced. If the type is declared then it must be a
         --! protected type.

      when 938 =>
         ELStrings.AppendString
           (EStr, "The unprotected variable ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is shared by ");
         AppendName
           (EStr, ErrNum.Name2, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " and ");
         AppendName
           (EStr, ErrNum.Name3, ErrNum.Scope);
         --! XXX is an unprotected variable that appears in the global list of the
         --! threads YYY and ZZZ. Unprotected variables cannot be shared between
         --! threads in SPARK. A thread is one of: the main program, a task, an
         --! interrupt handler.

      when 939 =>
         ELStrings.AppendString
           (EStr, "The suspendable item ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " is referenced by ");
         AppendName
           (EStr, ErrNum.Name2, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " and ");
         AppendName
           (EStr, ErrNum.Name3, ErrNum.Scope);
         --! XXX is an own variable with the suspends property that appears in the
         --! suspends list of the threads YYY and ZZZ. SPARK prohibits this to
         --! prevent more than one thread being suspended on the same item at any
         --! one time. A thread is one of: the main program, a task, an interrupt
         --!  handler.

      when 940 =>
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is a protected own variable. Protected variables may not be used in proof contexts");
         --! The use of protected variables in pre and postconditions or other proof annotations is not (currently)
         --! supported.  Protected variables are volatile because they can be changed at any time by another program
         --! thread and this may invalidate some common proof techniques.  The prohibition of protected variables
         --! does not prevent proof of absence of run-time errors nor proof of protected operation bodies.  See the
         --! manuals &quot;Generation of VCs&quot; and &quot;Generation of RTCs&quot; for more details.

      when 941 =>
         ELStrings.AppendString
           (EStr, "The type of own variable ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr,
            " must be local to this package");
         --! The type used to an announce an own variable with a protects property
         --! must be declared in the same package.

      when 942 =>
         ELStrings.AppendString
           (EStr, "Only one instance of the type ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is allowed");
         --! Type XXX has a protects property. This means there can be only one object
         --! in the package that has this type or any subtype of this type.

      when 943 =>
         ELStrings.AppendString
           (EStr, "The name ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " cannot appear in a protects list");
         --! All items in a protects list must be unprotected own variables owned by
         --! this package

      when 944 =>
         ELStrings.AppendString
           (EStr, "The name ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is already protected by ");
         AppendName
           (EStr, ErrNum.Name2, ErrNum.Scope);
         --! The name XXX appears in more than one protects list. The first time
         --! it appeared was for own variable YYY. XXX should appear in at most
         --! one protects list.

      when 945 =>
         ELStrings.AppendString
           (EStr, "The property ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " must be given a static expression for its value");
         --! This property can only accept a static expression.

      when 946 =>
         ELStrings.AppendString
           (EStr, "The own variable ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " must only ever be accessed from operations in protected type ");
         AppendName
           (EStr, ErrNum.Name2, ErrNum.Scope);
         --! The own variable XXX is protected by the protected type YYY and hence
         --! must never be accessed from anywhere else.

      when 947 =>
         ELStrings.AppendString
           (EStr, "The own variable ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " appears in a protects list for type ");
         AppendName
           (EStr, ErrNum.Name2, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " but is not used in the body");
         --! The protected type YYY claims to protect XXX via a protects property.
         --! However, the variable XXX is not used by any operation in YYY.

      when 948 =>
         ELStrings.AppendString
           (EStr, "The type of own variable or task ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " must be a base type");
         --! Own tasks and protected own variables of a protected type must be
         --! announced using the base type. The
         --! subsequent variable declaration may be a subtype of the base type

      when 949 =>
         ELStrings.AppendString
           (EStr, "Unexpected partition annotation: a global annotation may only " &
              "appear here when " &
           "the Ravenscar profile is selected");
         --! When the sequential SPARK profile is selected, the global and derives
         --! annotation on the main program describes
         --! the entire program's behaviour.  No additional, partition annotation
         --! is required or permitted.  Note that an
         --! annotation must appear here if the Ravenscar profile is selected.

      when 950 =>
         ELStrings.AppendString
           (EStr, "Partition annotation expected: a global and, optionally, a " &
              "derives annotation must " &
              "appear after 'main_program' when the Ravenscar profile is selected");
         --! When the Ravenscar profile is selected the global and derives annotation
         --! on the main program describes the
         --! behaviour of the environment task only, not the entire program.
         --! An additional annotation, called the
         --! partition annotation, is required to describe the entire program's
         --! behaviour; this annotation follows
         --! immediately after 'main_program;'

      when 951 =>
         ELStrings.AppendString
           (EStr, "Inherited package ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " contains tasks and/or interrupt handlers and must therefore " &
              "appear in the preceding WITH clause");
         --! In order to ensure that a Ravenscar program is complete, SPARK requires
         --! that all 'active' packages
         --! inherited by the environment task also appear in a corresponding
         --! with clause.  This check ensures that
         --! any program entities described in the partition annotation are also
         --! linked into the program itself.

      when 952 =>
         ELStrings.AppendString
           (EStr, "Subprogram ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is an interrupt handler and cannot be called");
         --! Interrupt handler operations cannot be called.


         --cfr1203 --error message only needed if we eliminate suspension objects from partition-level annos
         --cfr1203 when 985 =>
         --cfr1203    AppendName
         --cfr1203      (EStr, ErrNum.Name1, ErrNum.Scope);
         --cfr1203    ELStrings.AppendString
         --cfr1203      (EStr, " is a predefined suspension object which must not
         --cfr1203  appear in the partition global annotation");


      when 953 =>
         ELStrings.AppendString
           (EStr, "Interrupt property error for own variable ");
         AppendName
           (EStr, ErrNum.Name2, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, ". ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is not an interrupt handler in type ");
         AppendName
           (EStr, ErrNum.Name3, ErrNum.Scope);
         --! The handler names in an interrupt property must match one in the
         --! protected type of the own variable.

      when 954 =>
         ELStrings.AppendString
           (EStr, "Interrupt property error for own variable ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, ". Interrupt stream name ");
         AppendName
           (EStr, ErrNum.Name2, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is illegal");
         --! The stream name must be unprefixed and not already in use within the
         --! scope of the package.

      when 955 =>
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " can only appear in the partition wide flow annotation");
         --! Interrupt stream variables are used only to enhance the partition
         --! wide flow annotation and must not be used elsewhere.

      when 956 =>
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " already appears in as an interrupt handler in the " &
              "interrupt mappings");
         --! An interrupt handler can be mapped onto exactly one interrupt stream
         --! variable. An interrupt stream variable may be mapped onto many interrupt
         --!  handlers.

      when 957 =>
         ELStrings.AppendString
           (EStr, "Consecutive updates of protected variable ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " are disallowed when they do not depend directly on its preceding value");
         --! A protected variable cannot be updated without direct reference to its
         --! preceding value more than once within a subprogram or task.
         --! Each update of a protected variable may have a wider effect than
         --! just the change of value of the protected variable. The overall
         --! change is considered to be the accumulation of all updates and
         --! reads of the protected variable and to preseve this information flow
         --! successive updates must directly depend on the preceding value of
         --! the variable

      when 958 =>
         ELStrings.AppendString
           (EStr, "A task may not import the unprotected state ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         --! A task may not import unprotected state unless it is mode in.
         --! This is because under the concurrent elaboration policy, the task cannot
         --! rely on the state being initialized before it is run.

      when 959 =>
         ELStrings.AppendString
           (EStr, "Unprotected state ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is exported by a task and hence must not appear in an " &
              "initializes clause");
         --! Own variable XXX is being accessed by a task. The order in which the task
         --! is run and the own variable initialized is non-deterministic under a
         --! concurrent elaboration policy. In this case SPARK forces the task to
         --! perform the initialization and as such the own variable must not appear
         --! in an initializes clause.

      when 960 =>
         ELStrings.AppendString
           (EStr, "The function Ada.Real_Time.Clock " &
              "can only be used directly (1) in an assignment or return statement or (2) " &
              "to initialize a library a level constant");
         --! <li>To avoid ordering effects, functions which globally access own
         --! variables which have modes (indicating that they are connected to
         --! the external environment) may only appear directly in assignment or
         --! return statements. They may not appear as actual parameters or in any
         --! other form of expression.
         --! <li>SPARK relaxes the illegal use of function calls in elaboration code in
         --! the case of the function Ada.Real_Time.Clock. However the function can only
         --! be used to directly initialize a constant value.

      when 961 =>
         ELStrings.AppendString
           (EStr, "This property value is of an incorrect format");
         --! Please check the user manual for valid property value formats.

      when 962 =>
         ELStrings.AppendString
           (EStr, "Error(s) detected by VC Generator. See the .vcg file for more information");
         --! This message is echoed to the screen if an unrecoverable
         --! error occurs which
         --! makes the generation of VCs for the current subprogram impossible.
         --! Another message more precisely identifying the problem will be placed in
         --! the .vcg file.

      when 986 =>
         ELStrings.AppendString
           (EStr, "A protected function may not call a locally-declared " &
              "protected procedure");
         --! See LRM 9.5.1 (2). A protected function has read access to the
         --! protected elements of the type whereas
         --! the called procedure has read-write access. There is no way in which
         --! an Ada compiler can determine whether
         --! the procedure will illegally update the protected state or not so the
         --! call is prohibited by the rules of Ada.
         --! (Of course, in SPARK, we know there is no function side effect but the
         --! rules of Ada must prevail nonetheless).

      when 987 =>
         ELStrings.AppendString
           (EStr, "Task types and protected types may only be declared in " &
              "package specifications");
         --! The Examiner performs certain important checks at the whole program level
         --! such as detection of illegal sharing of
         --! unprotected state and partition-level information flow analysis.
         --! These checks require visibility of task
         --! types and protected types (especially those containing interrupt
         --! handlers).  SPARK therefore requires these
         --! types to be declare in package specifications.  Subtypes and objects
         --! of task types, protected types and their
         --! subtypes may be declared in package bodies.

      when 988 =>
         ELStrings.AppendString
           (EStr, "Illegal re-use of identifier ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, "; this identifier is used in a directly visible protected type");
         --! SPARK does not allow the re-use of operation names which are already
         --! in use in a directly visible
         --! protected type.  The restriction is necessary to avoid overload resolution
         --! issues in the protected
         --! body.  For example, type PT in package P declares operation K.
         --! Package P also declares an operation K.
         --! From inside the body of PT, a call to K could refer to either of the
         --! two Ks since both are directly visible.

      when 989 =>
         ELStrings.AppendString
           (EStr, "The last statement of a task body must be a plain loop " &
              "with no exits");
         --! To prevent any possibility of a task terminating (which can lead to a
         --! bounded error), SPARK requires
         --! each task to end with a non-terminating loop.  The environment task (or
         --! &quot;main program&quot;) does not need
         --! to end in a plain loop provided the program closure includes at least
         --! one other task.  If there are
         --! no other tasks, then the environment task must be made non-terminating
         --! with a plain loop.

      when 990 =>
         ELStrings.AppendString
           (EStr, "Unexpected annotation, a task body may have only global and " &
              "derives annotations");
         --! Issued if a pre, post or declare annotation is attached to a task body.

      when 991 =>
         ELStrings.AppendString
           (EStr, "Unexpected task body, ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is not the name of a task declared in this package specification");
         --! Issued if task body is encountered for which there is no preceding
         --! declaration.

      when 992 =>
         ELStrings.AppendString
           (EStr, "A body for task type ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " has already been declared");
         --! Issued if a duplicate body or body stub is encountered for a task.

      when 993 =>
         ELStrings.AppendString
           (EStr, "There is no protected type declaration for ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         --! Issued if a body is found for a protected types for which there is
         --! no preceding declaration.

      when 994 =>
         ELStrings.AppendString
           (EStr, "Invalid guard, ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is not a Boolean protected element of this protected type");
         --! The SPARK Ravenscar rules require a simple Boolean guard which must
         --! be one of the protected elements
         --! of the type declaring the entry.

      when 995 =>
         ELStrings.AppendString
           (EStr, "Unexpected entry body, ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " is not the name of an entry declared in this protected type");
         --! Local entries are not permitted so a protected body can declare at
         --! most one entry body and that must have
         --! declared in the protected type specification.

      when 996 =>
         ELStrings.AppendString
            (EStr, "The protected operation ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
            (EStr,
             ", declared in this type, does not have an associated body");
         --! Each exported protected operation must have a matching implementation
         --! in the associated protected body.

      when 997 =>
         ELStrings.AppendString
           (EStr, "A body for protected type ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString
           (EStr, " has already been declared");
         --! Each protected type declaration must have exactly one matching
         --! protected body or body stub.

      when 998 =>
         ELStrings.AppendString
           (EStr, "There is no protected type declaration for ");
         AppendName
           (EStr, ErrNum.Name1, ErrNum.Scope);
         --! Issued if a protected body or body stub is found and there is no
         --! matching declaration for it.

      when 999 =>
         ELStrings.AppendString
            (EStr,
             "This feature of Generics is not yet implemented");

      when others =>
         ELStrings.AppendString
            (EStr,
             "UNKNOWN ERROR NUMBER PASSED TO SemanticErr");
   end case;

   AppendReference
     (EStr, ErrNum.Reference);

   AppendExplanation;

   ELStrings.AppendString
      (EStr, ".");
end SemanticErr;
