-- $Id: errorhandler-conversions-tostring-condldependency.adb 12530 2009-02-20 09:43:11Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (ErrorHandler.Conversions.ToString)
procedure CondlDependency (ErrNum          : in     Error_Types.NumericError;
                           WithExplanation : in     Boolean;
                           EStr            : in out ELStrings.T)
is
   ErrType : ErrorHandler.FullDependErrType;

   procedure CondlDependencyExpl (EStr   : in out ELStrings.T)
   --# global in ErrType;
   --# derives EStr from *,
   --#                   ErrType;
      is separate;
   -- Note that the parameter names for this subunit are chosen to make it as easy as
   --      possible to auto-generate the subunit from this, its parent, file.  The
   --      generation requires copying the case statement below, stripping out the
   --      current Append'Thing' statements and adding an AppendString for the
   --      explanatory text that is delineated by --! comments.

   procedure AppendExplanation
   --# global in     ErrType;
   --#        in     WithExplanation;
   --#        in out EStr;
   --# derives EStr from *,
   --#                   ErrType,
   --#                   WithExplanation;
   is
      ExplanationString : ELStrings.T := ELStrings.EmptyString;
   begin
      if WithExplanation then
         -- we need to at least look for an explanation
         CondlDependencyExpl (ExplanationString);
         if ExplanationString.Length > 0 then
            -- there actually is one
            ELStrings.AppendString (EStr, ErrorHandler.ExplanationPrefix);
            ELStrings.AppendExaminerLongString (EStr, ExplanationString);
            ELStrings.AppendString (EStr, ErrorHandler.ExplanationPostfix);
         end if;
      end if;
   end AppendExplanation;

begin
   ErrType := ErrorHandler.DependencyErrType'Val (ErrNum.ErrorNum -
                                                  Error_Types.CondDependencyErrorOffset);
   case ErrType is

      -- HTML Directives
      --! <NameFormat> <"flow-"><Name>
      --! <ErrorFormat> <"??? Flow Error : "><Name><" : "><Error>

      when ErrorHandler.MayBeUsedNew =>
         --! <Name> 601
         AppendExportVar (EStr, ErrNum.Name2, ErrNum.Scope, True);
         ELStrings.AppendString (EStr, " may be derived from the imported value(s) of ");
         AppendExportVar (EStr, ErrNum.Name1, ErrNum.Scope, False);
         --! Here the item on the left of &quot;may be derived from ...&quot; is an exported variable and
         --! the item(s) on the right are imports of a procedure subprogram.
         --! The message reports a possible dependency, found in the code, which
         --! does not appear in the specified dependency relation (derives annotation).
         --! The discrepancy could be caused by an error in the subprogram code which implements
         --! an unintended dependency.  It could also be in an error in the subprogram derives annotation
         --! which omits a necessary and intended dependency.  Finally, the Examiner may be reporting
         --! a false coupling between two items resulting from a non-executable code path or the
         --! sharing of disjoint parts of structured or abstract data (e.g one variable writing to one element
         --! of an array and another variable reading back a different element).
         --! Unexpected dependencies should be investigated carefully and only accepted without modification
         --! of either code or annotation if it is certain they are of &quot;false coupling&quot; kind.

      when ErrorHandler.MayBeUsed =>  -- Legacy
         --! <Name> 601
         ELStrings.AppendString (EStr, "The imported value of ");
         AppendName (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString (EStr, " may be used in the derivation of ");
         AppendExportVar (EStr, ErrNum.Name2, ErrNum.Scope, False);
         --! Here first item is an import and the second is an export of a procedure subprogram.
         --! The message reports a possible dependency, found in the code, which
         --! does not appear in the specified dependency relation.
         --! This version of the message has been retained for backward compatibility.

      when ErrorHandler.MayBeUsedContinue =>
         ELStrings.AppendString (EStr, ", ");
         AppendName (EStr, ErrNum.Name1, ErrNum.Scope);

      when ErrorHandler.Uninitialised =>
         --! <Name> 602
         ELStrings.AppendString (EStr, "The undefined initial value of ");
         AppendName (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString (EStr, " may be used in the derivation of ");
         AppendExportVar (EStr, ErrNum.Name2, ErrNum.Scope, False);
         --! Here XXX is a non-imported variable, and YYY is an export, of a
         --! procedure subprogram.

      when ErrorHandler.IntegrityViolation =>
         --! <Name> 605
         ELStrings.AppendString (EStr, "Information flow from ");
         AppendName (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString (EStr, " to ");
         AppendExportVar (EStr, ErrNum.Name2, ErrNum.Scope, False);
         ELStrings.AppendString (EStr, " violates information flow policy");
         --! This message indicates a violation of security or safety policy, such
         --! as information flow from a Secret input to an Unclassified output.

      when ErrorHandler.MayBeIntegrityViolation =>
         --! <Name> 606
         ELStrings.AppendString (EStr, "The imported value of ");
         AppendName (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.AppendString (EStr, " may be used in the derivation of ");
         AppendExportVar (EStr, ErrNum.Name2, ErrNum.Scope, False);
         ELStrings.AppendString (EStr, ". Furthermore, this information flow violates ");
         ELStrings.AppendString (EStr, "information flow policy.");
         --! Here XXX is an import and YYY is an export of a procedure subprogram.
         --! The message reports a possible dependency, found in the code, which
         --! does not appear in the specified dependency relation. If this dependency
         --! did appear in the dependency relation, then it would also constitute
         --! an integrity violation.

      when others =>
         ELStrings.AppendString (EStr,
                                       "UNKNOWN ERROR NUMBER PASSED TO CondlDependency");
   end case;
   AppendExplanation;
   if ErrType /= ErrorHandler.MayBeUsedContinue and
     ErrType /= ErrorHandler.MayBeUsedNew then
      ELStrings.AppendString (EStr, ".");
   end if;
end CondlDependency;
