-- $Id: error_types.ads 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with LexTokenManager;
with ELStrings;
with Dictionary;

--# inherit Dictionary,
--#         ELStrings,
--#         LexTokenManager;
package Error_Types is

   --------------------------------------------------------
   -- Types common to ErrorHandler and Error_IO packages --
   --------------------------------------------------------

   type Error_Class is (LexErr,
                        SyntaxErr,
                        SyntaxRec,
                        UncondFlowErr,
                        CondlFlowErr,
                        UncondDependencyErr,
                        CondlDependencyErr,
                        SemanticErr,
                        DepSemanticErr,
                        WarningWithPosition,
                        WarningWithoutPosition,
                        Note,
                        ControlFlowErr,
                        IneffectiveStat,
                        StabilityErr,
                        UsageErr,
                        NoErr);

   type NameSorts is (None,
                      Symbol,
                      Entity,
                      LexString,
                      ParserSymbol,
                      StabilityIndex,
                      ThePartition);

   -- a "Name" is a kind of variant record, storing one of various NameSorts is numerical form
   type Names is record
      NameSort  : NameSorts;
      NameValue : Integer;  --converted from Symbol, LexStr etc
   end record;
   NoName : constant Names := Names'(NameSort  => None,
                                        NameValue => 0);

   ThePartitionName : constant Names := Names'(NameSort  => ThePartition,
                                               NameValue => 0);


   -- error messages have a number, curently in the range 0 .. 999;
   subtype ErrNumRange is Integer range 0 .. 999;

   -- Error messages are handled in 2 forms.  The NumericError form is space efficient
   -- and is used for adding and sorting messages into line number order; that is the work
   -- of ErrorBuffer

   type NumericError is record
      ErrorType : Error_Class;
      Position  : LexTokenManager.TokenPosition;
      Scope     : Dictionary.Scopes;
      ErrorNum  : ErrNumRange;
      Reference : Natural;
      Name1,
      Name2,
      Name3     : Names;
   end record;

   Empty_NumericError : constant NumericError :=
     NumericError'(NoErr,
                   LexTokenManager.TokenPosition'(0, 0),
                   Dictionary.NullScope,
                   0,
                   0,
                   NoName,
                   NoName,
                   NoName);

   -- The StringError form includes a conversion of the message into string form and is therefore not
   -- space efficient but is suitable for printing; that is the work of ConvertToString as called
   -- by PrintErrors and AppendErrors (and also as a sid effect of Add).

   type StringError is record
      MessageId : ErrNumRange;
      ErrorType : Error_Class;
      Position  : LexTokenManager.TokenPosition;
      Message   : ELStrings.T;
   end record;

   Empty_StringError : constant StringError :=
      StringError'(0,
                   NoErr,
                   LexTokenManager.TokenPosition'(0, 0),
                   ELStrings.EmptyString);

   -- constants used as offsets to 'val/'pos of various error kind enumerations
   --     so as to ensure that they all have unique error numbers
   UncondFlowErrorOffset       : constant ErrNumRange := 20;
   UncondDependencyErrorOffset : constant ErrNumRange := 50;
   StabilityErrOffset          : constant ErrNumRange := 40;
   UsageErrOffset              : constant ErrNumRange := 30;
   ControlFlowErrOffset        : constant ErrNumRange := 1;
   CondFlowErrorOffset         : constant ErrNumRange := 500;
   CondDependencyErrorOffset   : constant ErrNumRange := 600;

   -- To track when explanations have been given, so that we can restrict such explanations
   -- to the first occurrence of each error number, we need to know whether the conversion
   -- of a numeric error to a string error is being made for one of the three following
   -- purposes.
   type ConversionRequestSource is (ForScreen, ForReportSelectedFiles, ForReportIndexedFiles, ForListing);
   subtype ForReport is ConversionRequestSource range ForReportSelectedFiles .. ForReportIndexedFiles;


end Error_Types;
