-- $Id: elstrings.adb 11889 2008-12-12 15:49:12Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with CommonString,
     CommonStringUtilities;

package body ELStrings
is

   function ToExaminerLongString (EStr : EStrings.T)
                                 return T
   is
      ResultStr : Contents;
      UnusedLen : Lengths;
   begin
      --# accept F, 10, UnusedLen, "UnusedLen unused here" &
      --#        F, 33, UnusedLen, "UnusedLen unused here";
      CommonString.CopyString (StrOut => ResultStr,
                               Length => UnusedLen,
                               StrIn  => EStr.Content);

      return T'(Content => ResultStr,
                Length  => EStr.Length);
   end ToExaminerLongString; -- Flow Error OK on UnusedLen

   function ToExaminerString (EStr : T)
                             return EStrings.T
   is
      L         : EStrings.Positions;
      Result    : EStrings.T;
      ResultStr : EStrings.Contents;
      ResultLen : EStrings.Lengths;
   begin
      if EStr.Length = 0 then
         Result := EStrings.EmptyString;
      else
         if EStr.Length > EStrings.Lengths'Last then
            L := EStrings.Lengths'Last;
         else
            L := EStr.Length;
         end if;

         CommonStringUtilities.Section (StrIn     => EStr.Content,
                                        StrLenIn  => EStr.Length,
                                        StartPos  => EStrings.Positions'First,
                                        Length    => L,
                                        StrOut    => ResultStr,
                                        StrLenOut => ResultLen);

         Result := EStrings.T'(Content => ResultStr,
                                                   Length  => ResultLen);
      end if;

      return Result;

   end ToExaminerString;

   function LowerCase (EStr : T) return T
   is
      Res : T;
   begin
      CommonString.LowerCase (EStr.Content, EStr.Length, Res.Content);
      Res.Length := EStr.Length;

      return Res;
   end LowerCase;

   ------------------------------------

   function UpperCase (EStr : T) return T
   is
      Res : T;
   begin
      CommonString.UpperCase (EStr.Content, EStr.Length, Res.Content);
      Res.Length := EStr.Length;

      return Res;
   end UpperCase;

   --------------------------------------------

   -- Case INsensitive comparison.
   function EqString (EStr1, EStr2 : T) return Boolean
   is
   begin
      return CommonString.EqString
         (EStr1.Content, EStr1.Length, EStr2.Content, EStr2.Length);
   end EqString;

   -- Case INsensitive comparison.
   function Eq1String (M : T;
                       S : String)
                      return Boolean
   is
   begin
      return CommonString.EqString (M.Content, M.Length, S, S'Length);
   end Eq1String;

   function IsEmpty (EStr : T) return Boolean
   is
   begin
      return EqString (EStr, EmptyString);
   end IsEmpty;

   procedure CopyString (EStr :    out T;
                         Str  : in     String)
   is
      ResultString : Contents;
      ResultLength : Lengths;
   begin
      CommonString.CopyString (ResultString, ResultLength, Str);

      EStr.Content := ResultString;
      EStr.Length  := ResultLength;
   end CopyString;

   procedure AppendString (EStr : in out T;
                           Str  : in     String)
   is
      ResultString : Contents;
      ResultLength : Lengths;
   begin
      ResultString := EStr.Content;
      ResultLength := EStr.Length;

      CommonString.AppendString (ResultString, ResultLength, Str, Str'Length);

      EStr.Content := ResultString;
      EStr.Length  := ResultLength;
   end AppendString;

   procedure AppendExaminerString (ELStr : in out T;
                                   EStr  : in     EStrings.T)
   is
      ResultString : Contents;
      ResultLength : Lengths;
   begin
      ResultString := ELStr.Content;
      ResultLength := ELStr.Length;

      CommonString.AppendString
         (ResultString, ResultLength, EStr.Content, EStr.Length);

      ELStr.Content := ResultString;
      ELStr.Length  := ResultLength;

   end AppendExaminerString;


   procedure AppendExaminerLongString (EStr1 : in out T;
                                       EStr2 : in     T)
   is
      ResultString : Contents;
      ResultLength : Lengths;
   begin
      ResultString := EStr1.Content;
      ResultLength := EStr1.Length;

      CommonString.AppendString
         (ResultString, ResultLength, EStr2.Content, EStr2.Length);

      EStr1.Content := ResultString;
      EStr1.Length  := ResultLength;
   end AppendExaminerLongString;

   procedure PutString (File : in SPARK_IO.File_Type;
                        EStr : in T)
   is
   begin
      CommonString.PutString (File, EStr.Content, EStr.Length);
   end PutString;

   procedure PutLine (File : in SPARK_IO.File_Type;
                      EStr : in T)
   is
   begin
      CommonString.PutLine (File, EStr.Content, EStr.Length);
   end PutLine;

   procedure GetLine (File : in  SPARK_IO.File_Type;
                      EStr : out T)
   is
      AStr    : Contents;
      ALength : Lengths;
   begin
      CommonString.GetLine (File, AStr, ALength);
      EStr := T'(Content => AStr, Length => ALength);
   end GetLine;

   procedure AppendChar (EStr    : in out T;
                         Ch      : in     Character;
                         Success :    out Boolean)
   is
      ResultString : Contents;
      ResultLength : Lengths;
   begin
      ResultString := EStr.Content;
      ResultLength := EStr.Length;

      CommonString.AppendChar (ResultString, ResultLength, Ch, Success);

      EStr.Content := ResultString;
      EStr.Length  := ResultLength;
   end AppendChar;

   procedure PopChar  (EStr : in out T;
                       Char :    out Character)
   is
      ResultString : Contents;
      ResultLength : Lengths;
   begin
      ResultString := EStr.Content;
      ResultLength := EStr.Length;

      CommonString.PopChar (ResultString, ResultLength, Char);

      EStr.Content := ResultString;
      EStr.Length  := ResultLength;
   end PopChar;

   ---------------------------------------------------------------------------
   -- FindCharAfter: find specified Character in EStr, starting at specified
   -- position
   procedure FindCharAfter (EStr        : in     T;
                            SearchStart : in     Positions;
                            SearchChar  : in     Character;
                            CharFound   :    out Boolean;
                            CharPos     :    out Positions)
   is
      Position : Positions;
      Found1    : Boolean := False;
      Finish1   : Boolean := False;
   begin
      Position := SearchStart;
      while not Finish1 loop
         if           Position <= EStr.Length
           and then EStr.Content (Position) = SearchChar then
            Found1 := True;
            Finish1 := True;
         elsif Position < EStr.Length then
            Position := Position + 1;
         else
            Finish1 := True;
            Found1 := False;
         end if;
      end loop;

      CharPos := Position;
      CharFound := Found1;
   end FindCharAfter;

   --------------------------------------------------------------------------
   -- FindChar: find first occurrence of specified Character in EStr
   procedure FindChar (EStr       : in     T;
                       SearchChar : in     Character;
                       CharFound  :    out Boolean;
                       CharPos    :    out Positions)
   is
   begin
      FindCharAfter (EStr, 1, SearchChar, CharFound, CharPos);
   end FindChar;

   --------------------------------------------------------------------------
   function LexOrder (FirstName, SecondName : T)
                     return OrderTypes
   is
      Pos       : Positions := 1;
      Finished  : Boolean := False;
      LexOrderResult : OrderTypes := NeitherFirst;

      RankOfNextCharInFirst  : Integer;
      RankOfNextCharInSecond : Integer;
   begin

      while not Finished loop
         -- if the remainder of only one of the strings is empty then
         -- that is the first in order
         -- if the remainder of both is empty then neither is first
         if Pos > FirstName.Length and Pos <= SecondName.Length then
            LexOrderResult := FirstOneFirst;
            Finished := True;
         elsif Pos > SecondName.Length and Pos <= FirstName.Length then
            LexOrderResult := SecondOneFirst;
            Finished := True;
         elsif Pos > FirstName.Length and Pos > SecondName.Length then
            LexOrderResult := NeitherFirst;
            Finished := True;
         else
            -- decide using the current Character in the string
            RankOfNextCharInFirst := Character'Pos (FirstName.Content (Pos));
            RankOfNextCharInSecond := Character'Pos (SecondName.Content (Pos));

            if RankOfNextCharInFirst < RankOfNextCharInSecond then
               LexOrderResult := FirstOneFirst;
               Finished := True;
            elsif RankOfNextCharInSecond < RankOfNextCharInFirst then
               LexOrderResult := SecondOneFirst;
               Finished := True;
            else
               if Pos < Positions'Last then
                  Pos := Pos + 1;
               else
                  LexOrderResult := NeitherFirst;
                  Finished := True;
               end if;
            end if;
         end if;
      end loop;

      return LexOrderResult;
   end LexOrder;

   ----------------------------------------------------------------------------
   -- Section added for use in summary tool
   -- returns the specified subsection of the string
   -- if the subsection lies outside the string, empty string returned
   function Section (EStr     : T;
                     StartPos : Positions;
                     Length   : Lengths) return T
   is
      Finished : Boolean;
      Pos      : Positions;
      Result   : T;
   begin
      Result := EmptyString;
      if (StartPos + Length) - 1 > EStr.Length then
         null;
      else
         Pos := StartPos;
         Finished := False;
         while not Finished and then Pos < StartPos + Length loop
            Result.Content ((Pos + 1) - StartPos) := EStr.Content (Pos);
            if not (Pos = Positions'Last) then
               Pos := Pos + 1;
            else
               Finished := True;
            end if;
         end loop;

         Result.Length := Length;
      end if;

      return Result;
   end Section;

end ELStrings;
