-- $Id: dictionary.ads 13056 2009-04-20 17:01:20Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with ExaminerConstants,
     EStrings,
     LexTokenManager,
     SPARK_IO,
     SPSymbols;

use type ExaminerConstants.RefType;
use type LexTokenManager.LexString;
use type SPSymbols.SPSymbol;
use type SPARK_IO.File_Status;

--# inherit Ada.Characters.Handling,
--#         CommandLineData,
--#         ELStrings,
--#         EStrings,
--#         ExaminerConstants,
--#         FileSystem,
--#         LexTokenManager,
--#         LextokenStacks,
--#         Maths,
--#         ScreenEcho,
--#         SPARK_IO,
--#         SPSymbols,
--#         Statistics,
--#         SystemErrors,
--#         XMLReport;

package Dictionary
--# own Dict : Dictionaries;
is

   --------------------------------------------------------------------------------
   --                             TYPE DEFINITIONS                               --
   --------------------------------------------------------------------------------

   type Symbol is private;
   NullSymbol : constant Symbol;

   type Iterator is private;
   NullIterator : constant Iterator;

   type Location is record
      StartPosition : LexTokenManager.TokenPosition;
      EndPosition   : LexTokenManager.TokenPosition;
   end record;

   type Scopes is private;
   -- following constant is not valid for use other than as a placeholder for constructing a
   -- an aggregate that needs a Scope in it (e.g. EmptyNumericError in Error_Types)
   NullScope : constant Scopes;

   -- following type is used to provide a text writeable representation of a scope
   type ScopeReference is record
      TypeOfScope : Integer;
      Region      : Integer;
   end record;

   -- conversion routines to and from scope follow in access function section

   type Contexts is (ProofContext, ProgramContext);

   type Modes is (DefaultMode, InMode, OutMode, InOutMode, InvalidMode);

   type Abstractions is (IsAbstract, IsRefined);

   type PrefixSort is (AType, ABaseType, AnObject);

   type PackageSort is (Public, PrivateChild);

   type KindsOfOp is (AProcedure, AFunction, NotASubprogram);

   type RavenscarPragmas is (Priority, InterruptPriority, AttachHandler);
   subtype RavenscarPragmasWithValue is RavenscarPragmas range Priority .. InterruptPriority;

   type GenericParameterKind is (GenericTypeParameter, GenericObjectParameter);


   -- The generation of replacement rules for composite constants is governed
   -- by an object_assertion annotation.  This can either explicitly request
   -- that a rule be generated, request that NO rule be generated, or can be
   -- left unspecified.  The action of the Examiner for these cases also depends
   -- on the setting of the /rules=XXX command-line switch declared in
   -- commandlinedata.ads
   type RulePolicies is (Unspecified, RuleRequested, NoRuleRequested);

   --------------------------------------------------------------------------------
   --                               HOUSEKEEPING                                 --
   --------------------------------------------------------------------------------

   procedure Initialize (WriteToFile : in Boolean);
   --# global in     CommandLineData.Content;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#           out Dict;
   --# derives Dict,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          LexTokenManager.StringTable,
   --#                                          SPARK_IO.File_Sys,
   --#                                          WriteToFile &
   --#         LexTokenManager.StringTable from *,
   --#                                          CommandLineData.Content;

   procedure SetCurrentFileName (FileName : in EStrings.T);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   FileName;

   procedure Write (FileName : in     EStrings.T;
                    Status   :    out SPARK_IO.File_Status);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                FileName,
   --#                                SPARK_IO.File_Sys &
   --#         SPARK_IO.File_Sys,
   --#         Status            from Dict,
   --#                                FileName,
   --#                                LexTokenManager.StringTable,
   --#                                SPARK_IO.File_Sys;

   procedure ReportUsage;
   --# global in     Dict;
   --#        in out Statistics.TableUsage;
   --# derives Statistics.TableUsage from *,
   --#                                    Dict;

   procedure ReadTargetDataFile;
   --# global in     CommandLineData.Content;
   --#        in out Dict;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         LexTokenManager.StringTable,
   --#         SPARK_IO.File_Sys           from *,
   --#                                          CommandLineData.Content,
   --#                                          LexTokenManager.StringTable,
   --#                                          SPARK_IO.File_Sys;

   procedure OutputTargetDataFile (ToFile : in SPARK_IO.File_Type);
   --# global in     CommandLineData.Content;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out XMLReport.State;
   --# derives LexTokenManager.StringTable,
   --#         SPARK_IO.File_Sys,
   --#         XMLReport.State             from CommandLineData.Content,
   --#                                          LexTokenManager.StringTable,
   --#                                          SPARK_IO.File_Sys,
   --#                                          ToFile,
   --#                                          XMLReport.State;


   --------------------------------------------------------------------------------
   --                                SYMBOLS and ORDERING                        --
   --------------------------------------------------------------------------------

   function Declared_Before (Left, Right : in Symbol) return Boolean;

   --------------------------------------------------------------------------------
   --                              LOOKUP ROUTINES                               --
   --------------------------------------------------------------------------------

   function GlobalScope return Scopes;
   --# global in Dict;

   function VisibleScope (ThePackage : Symbol) return Scopes;
   -- pre IsPackage (ThePackage);

   function LocalScope (Region : Symbol) return Scopes;
   -- pre IsCompilationUnit (Region) or IsLoop (Region);

   function PrivateScope (Region : Symbol) return Scopes;
   -- pre IsCompilationUnit (Region) or IsLoop (Region);

   function IsLibraryLevel (Scope : Scopes) return Boolean;
   --# global in Dict;

   function IsPredefinedScope (Scope : Scopes) return Boolean;
   --# global in Dict;

   function IsGlobalScope (Scope : Scopes) return Boolean;
   --# global in Dict;

   function IsVisibleScope (Scope : Scopes) return Boolean;

   function IsLocalScope (Scope : Scopes) return Boolean;

   function IsPrivateScope (Scope : Scopes) return Boolean;

   function IsLocalOrPrivateScope (Scope : Scopes) return Boolean;

   function GetRegion (Scope : Scopes) return Symbol;

   function IsLocal (Inner, Outer : Scopes) return Boolean;
   --# global in Dict;

   function GetContext (TheSymbol : Symbol) return Contexts;
   --# global in Dict;
   -- pre TheSymbol /= NullSymbol and not IsUnknownTypeMark (TheSymbol);

   function GetRootPackage (ThePackage : Symbol) return Symbol;
   --# global in Dict;
   --  pre  IsPackage(ThePackage);

   function GetLibraryPackage (Scope : Scopes) return Symbol;
   --# global in Dict;

   function IsPrivatePackage (ThePackage : Symbol) return Boolean;
   --# global in Dict;
   --  pre  IsPackage(ThePackage);

   function GetPackageParent (ThePackage : Symbol) return Symbol;
   --# global in Dict;
   --  pre  IsPackage(ThePackage);

   function PackageDeclaresTaggedType (ThePackage : Symbol) return Boolean;
   --# global in Dict;
   --  pre  IsPackage(ThePackage);

   function PackageExtendsAnotherPackage (ThePackage : Symbol) return Boolean;
   --# global in Dict;
   --  pre  IsPackage(ThePackage);

   function GetPackageThatIsExtended (ThePackage : Symbol) return Symbol;
   --# global in Dict;
   --  pre  IsPackage(ThePackage);

   function GetPackageExtendedType (ThePackage : Symbol) return Symbol;
   --# global in Dict;
   --  pre  IsPackage(ThePackage);

   function GetPackageOwner (ThePackage : Symbol) return Symbol;
   --# global in Dict;
   --  pre  IsPackage(ThePackage);

   function IsProperDescendent (Inner, Outer : Symbol) return Boolean;
   --# global in Dict;
   --  pre  IsPackage(Inner) and IsPackage(Outer);

   function IsPrivateDescendent (Inner, Outer : Symbol) return Boolean;
   --# global in Dict;
   --  pre  IsPackage(Inner) and IsPackage(Outer);

   function IsDescendentOfPrivateSibling (Candidate, ThePackage : Symbol)
                                         return Boolean;
   --# global in Dict;
   --  pre  IsPackage(ThePackage);

   function IsDirectlyDefined (Name    : LexTokenManager.LexString;
                               Scope   : Scopes;
                               Context : Contexts) return Boolean;
   --# global in Dict;

   function IsDefined (Name    : LexTokenManager.LexString;
                       Scope   : Scopes;
                       Context : Contexts) return Boolean;
   --# global in Dict;

   function UnaryOperatorIsDefined (Name    : SPSymbols.SPSymbol;
                                    Operand : Symbol) return Boolean;
   --# global in CommandLineData.Content;
   --#        in Dict;
   -- pre IsType (Operand) or IsUnknownTypeMark (Operand);

   function BinaryOperatorIsDefined (Name  : SPSymbols.SPSymbol;
                                     Left  : Symbol;
                                     Right : Symbol) return Boolean;
   --# global in CommandLineData.Content;
   --#        in Dict;
   -- pre (IsType (Left)  or IsUnknownTypeMark (Left)) and
   --     (IsType (Right) or IsUnknownTypeMark (Right));

   function UnaryOperatorIsVisible (Name    : SPSymbols.SPSymbol;
                                    Operand : Symbol;
                                    Scope   : Scopes) return Boolean;
   --# global in CommandLineData.Content;
   --#        in Dict;
   -- pre UnaryOperatorIsDefined (Name, Operand);

   function BinaryOperatorIsVisible (Name  : SPSymbols.SPSymbol;
                                     Left  : Symbol;
                                     Right : Symbol;
                                     Scope : Scopes) return Boolean;
   --# global in CommandLineData.Content;
   --#        in Dict;
   -- pre BinaryOperatorIsDefined (Name, Left, Right);

   function AttributeIsVisible (Name     : LexTokenManager.LexString;
                                Prefix   : PrefixSort;
                                TypeMark : Symbol;
                                Scope    : Scopes) return Boolean;
   --# global in CommandLineData.Content;
   --#        in Dict;
   -- pre IsTypeMark (TypeMark) or IsUnknowTypeMark (TypeMark);

   function AttributeIsVisibleButObselete (Name     : LexTokenManager.LexString;
                                           Prefix   : PrefixSort;
                                           TypeMark : Symbol;
                                           Scope    : Scopes) return Boolean;
   --# global in CommandLineData.Content;
   --#        in Dict;
   -- pre IsTypeMark (TypeMark) or IsUnknowTypeMark (TypeMark);

   function PrefixAllowed (Prefix : Symbol; Scope : Scopes) return Boolean;
   --# global in Dict;

   function PrefixRequired (Item : Symbol; Scope : Scopes) return Boolean;
   --# global in Dict;
   -- pre not IsUnknownTypeMark (Item);

   function GetPrefix (Item : Symbol) return LexTokenManager.LexString;
   --# global in Dict;
   -- pre not IsUnknownTypeMark (Item);

   function LookupImmediateScope (Name    : LexTokenManager.LexString;
                                  Scope   : Scopes;
                                  Context : Contexts) return Symbol;
   --# global in Dict;

   function LookupItem (Name    : LexTokenManager.LexString;
                        Scope   : Scopes;
                        Context : Contexts) return Symbol;
   --# global in Dict;

   function LookupSelectedItem (Prefix   : Symbol;
                                Selector : LexTokenManager.LexString;
                                Scope    : Scopes;
                                Context  : Contexts) return Symbol;
   --# global in CommandLineData.Content;
   --#        in Dict;
   -- pre IsPackage (Prefix) or
   --     IsRecordType (Prefix) or
   --     IsRecordComponent (Prefix) or
   --     (IsObject (Prefix) and IsRecordType (GetType (Prefix)) or
   --     (IsFunction (Prefix) and IsRecordType (GetType (Prefix)) or
   --     (IsObject (Prefix) and IsProtectedType (GetType (Prefix)));

   -- starting in Scope look for operations inherited as a result of use of
   -- tagged and extended tagged types.  Returns NullSymbol if no match found.
   -- ActualTaggedType is set (on successful return) to the type of the tagged parameter
   -- required in the calling environment.
   procedure SearchForInheritedOperations (Name             : in     LexTokenManager.LexString;
                                           Scope            : in     Scopes;
                                           Prefix           : in     Symbol;
                                           Context          : in     Contexts;
                                           OpSym            :    out Symbol;
                                           KindOfOp         :    out KindsOfOp;
                                           ActualTaggedType :    out Symbol);
   --# global in CommandLineData.Content;
   --#        in Dict;
   --# derives ActualTaggedType,
   --#         KindOfOp,
   --#         OpSym            from CommandLineData.Content,
   --#                               Context,
   --#                               Dict,
   --#                               Name,
   --#                               Prefix,
   --#                               Scope;

   function GetSubprogramControllingType (Subprogram  : Symbol) return Symbol;
   --# global in Dict;
   -- returns null symbol if the subprogram does not have a controlling type (i.e. a parameter of a tagged
   -- type which is declared in the same package as the subprogram)

   function GetOverriddenSubprogram (Subprogram  : Symbol) return Symbol;
   --# global in Dict;
   --  pre IsSubprogram (Subprogram);
   --  post -- returns null symbol is subprogram does not override or symbol of most-closely
   --       -- overriden subprogram if it does

   function IsCallable (Subprogram   : Symbol;
                        PrefixNeeded : Boolean;
                        Scope        : Scopes) return Boolean;
   --# global in Dict;
   -- pre IsSubprogram (Subprogram);

   procedure GenerateSimpleName (Item      : in     Symbol;
                                 Separator : in     String;
                                 Name      :    out EStrings.T);
   --# global in Dict;
   --#        in LexTokenManager.StringTable;
   --# derives Name from Dict,
   --#                   Item,
   --#                   LexTokenManager.StringTable,
   --#                   Separator;

   procedure GetAnyPrefixNeeded (Sym       : in     Symbol;
                                 Scope     : in     Scopes;
                                 Separator : in     String;
                                 Prefix    :    out EStrings.T);
   --# global in CommandLineData.Content;
   --#        in Dict;
   --#        in LexTokenManager.StringTable;
   --# derives Prefix from CommandLineData.Content,
   --#                     Dict,
   --#                     LexTokenManager.StringTable,
   --#                     Scope,
   --#                     Separator,
   --#                     Sym;

   function IsValidGenericTypeAssociation (Formal, Actual : Symbol;
                                           Scope          : Scopes) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (Formal) and IsTypeMark (Actual) and
   --     TypeIsGeneric (Formal) and not TypeIsGeneric (Actual);

   --------------------------------------------------------------------------------
   --                               CONSTRUCTORS                                 --
   --------------------------------------------------------------------------------

   procedure AddDeferredConstant (Name          : in LexTokenManager.LexString;
                                  TypeMark      : in Symbol;
                                  TypeReference : in Location;
                                  Declaration   : in Location;
                                  ThePackage    : in Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Name,
   --#                                ThePackage,
   --#                                TypeMark &
   --#         SPARK_IO.File_Sys from *,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                ThePackage,
   --#                                TypeMark,
   --#                                TypeReference;
   -- pre  not IsReservedWord (Name) and
   --      not IsDirectlyDefined (Name, ThePackage, ProofContext) and
   --      IsTypeMark (TypeMark) and
   --      IsPackage (ThePackage);
   -- post for_some (TheConstant: Symbol,
   --        TheConstant = LookupItem (Name, ThePackage, ProgramContext) and
   --        TypeMark = GetType (TheConstant) and
   --        Declaration = GetDeferredConstantDeclaration (TheConstant));


   procedure PromoteDeferredToFullConstant (TheConstant : in Symbol;
                                            Value       : in LexTokenManager.LexString;
                                            ExpNode     : in ExaminerConstants.RefType;
                                            ThePackage  : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   ExpNode,
   --#                   TheConstant,
   --#                   ThePackage,
   --#                   Value;
   -- pre TheConstant denotes System.Default_Bit_Order - this _can_ be promoted
   --     from deferred to full constant status in the configuration file.


   procedure AddScalarConstant (Name            : in LexTokenManager.LexString;
                                TypeMark        : in Symbol;
                                TypeReference   : in Location;
                                Value           : in LexTokenManager.LexString;
                                ExpIsWellFormed : in Boolean;
                                ExpNode         : in ExaminerConstants.RefType;
                                Static          : in Boolean;
                                Declaration     : in Location;
                                Scope           : in Scopes;
                                Context         : in Contexts);
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                CommandLineData.Content,
   --#                                Context,
   --#                                ExpIsWellFormed,
   --#                                ExpNode,
   --#                                Name,
   --#                                Scope,
   --#                                Static,
   --#                                TypeMark,
   --#                                Value &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope,
   --#                                Static,
   --#                                TypeMark,
   --#                                TypeReference,
   --#                                Value;
   -- pre  not IsReservedWord (Name) and
   --      IsDirectlyDefined (Name, Scope, Context) =>
   --        IsDeferredConstant (LookupItem (Name, Scope, Context)) and
   --      IsScalarTypeMark (TypeMark, Scope);
   -- post for_some (TheConstant: Symbol,
   --        TheConstant = LookupItem (Name, Scope, Context) and
   --        TypeMark = GetType (TheConstant) and
   --        Value = GetValue (TheConstant) and
   --        Static = IsStatic (TheConstant, Scope));

   procedure AddArrayConstant (Name            : in LexTokenManager.LexString;
                               TypeMark        : in Symbol;
                               TypeReference   : in Location;
                               ExpIsWellFormed : in Boolean;
                               ExpNode         : in ExaminerConstants.RefType;
                               Static          : in Boolean; -- String constants are static in Ada95
                               Declaration     : in Location;
                               Scope           : in Scopes;
                               Context         : in Contexts);
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                CommandLineData.Content,
   --#                                Context,
   --#                                ExpIsWellFormed,
   --#                                ExpNode,
   --#                                Name,
   --#                                Scope,
   --#                                Static,
   --#                                TypeMark &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope,
   --#                                Static,
   --#                                TypeMark,
   --#                                TypeReference;
   -- pre  not IsReservedWord (Name) and
   --      IsDirectlyDefined (Name, Scope, Context) =>
   --        IsDeferredConstant (LookupItem (Name, Scope, Context)) and
   --      IsArrayTypeMark (TypeMark, Scope);
   -- post for_some (TheConstant: Symbol,
   --        TheConstant = LookupItem (Name, Scope, Context) and
   --        TypeMark = GetType (TheConstant);

   procedure AddRecordConstant (Name            : in LexTokenManager.LexString;
                                TheType         : in Symbol;
                                TypeReference   : in Location;
                                ExpIsWellFormed : in Boolean;
                                ExpNode         : in ExaminerConstants.RefType;
                                Declaration     : in Location;
                                Scope           : in Scopes;
                                Context         : in Contexts);
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                CommandLineData.Content,
   --#                                Context,
   --#                                ExpIsWellFormed,
   --#                                ExpNode,
   --#                                Name,
   --#                                Scope,
   --#                                TheType &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope,
   --#                                TheType,
   --#                                TypeReference;
   -- pre  not IsReservedWord (Name) and
   --      IsDirectlyDefined (Name, Scope, Context) =>
   --        IsDeferredConstant
   --          (LookupItem (Name, Scope, Context)) and
   --      IsRecordType (TheType);
   -- post for_some (TheConstant: Symbol,
   --        TheConstant = LookupItem (Name, Scope, Context) and
   --        TheType = GetType (TheConstant);

   procedure AddConstantRulePolicy (TheConstant : in Symbol;
                                    TheScope    : in Scopes;
                                    ThePolicy   : in RulePolicies);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   TheConstant,
   --#                   ThePolicy,
   --#                   TheScope;


   procedure AddVariable (Name          : in LexTokenManager.LexString;
                          TypeMark      : in Symbol;
                          Initialized   : in Boolean;
                          IsAliased     : in Boolean;
                          ExpNode       : in ExaminerConstants.RefType;
                          TypeReference : in Location;
                          Declaration   : in Location;
                          Scope         : in Scopes);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                ExpNode,
   --#                                Initialized,
   --#                                IsAliased,
   --#                                Name,
   --#                                Scope,
   --#                                TypeMark &
   --#         SPARK_IO.File_Sys from *,
   --#                                Declaration,
   --#                                Dict,
   --#                                Initialized,
   --#                                IsAliased,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope,
   --#                                TypeMark,
   --#                                TypeReference;
   -- pre  not IsReservedWord (Name) and
   --      IsDirectlyDefined (Name, Scope, ProofContext) =>
   --        IsOwnVariable (LookupItem (Name, Scope, ProofContext)) and
   --      IsTypeMark (TypeMark);
   -- post for_some (Variable: Symbol,
   --        Variable = LookupItem (Name, Scope, ProgramContext) and
   --        IsVariable (Variable) and
   --        TypeMark = GetType (Variable));

   procedure AddVariableAddressClause (Variable : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Variable;
   -- pre  IsEntireVariable (Variable);
   -- post VariableHasAddressClause (Variable);
   -- could also be extended to write location to dictionary file as for other reps

   procedure AddVariablePragmaImport (Variable : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Variable;
   -- pre  IsEntireVariable (Variable);
   -- post VariableHasPragmaImport (Variable);
   -- could also be extended to write location to dictionary file as for other reps

   procedure AddTypeSizeAttribute (TypeMark : in Symbol; SizeVal : in LexTokenManager.LexString);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   SizeVal,
   --#                   TypeMark;
   -- pre  IsType (TypeMark);
   -- post TypeSizeAttribute (TypeMark, Dict) = SizeVal;

   procedure SetTypeAtomic (TypeMark : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   TypeMark;

   procedure AddRecordSubcomponent (Prefix       : in     Symbol;
                                    Component    : in     Symbol;
                                    Subcomponent :    out Symbol);
   --# global in out Dict;
   --# derives Dict,
   --#         Subcomponent from Component,
   --#                           Dict,
   --#                           Prefix;

   procedure AddQuantifiedVariable (Name          : in     LexTokenManager.LexString;
                                    TypeMark      : in     Symbol;
                                    TypeReference : in     Location;
                                    Region        : in     Symbol;
                                    Variable      :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Name,
   --#                                Region,
   --#                                TypeMark &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Region,
   --#                                TypeMark,
   --#                                TypeReference &
   --#         Variable          from Dict;
   -- pre  not IsReservedWord (Name) and
   --      not IsDirectlyDefined (Name, LocalScope (Region), Proof Context) and
   --      IsTypeMark (TypeMark);
   -- post for_some (Variable: Symbol,
   --        Variable = LookupItem (Name, LocalScope (Region), ProofContext) and
   --        IsVariable (Variable) and
   --        TypeMark = GetType (Variable));

   procedure AddTypeAnnouncement (Name         : in     LexTokenManager.LexString;
                                  Announcement : in     Location;
                                  ThePackage   : in     Symbol;
                                  TypeMark     :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Name,
   --#                                ThePackage &
   --#         SPARK_IO.File_Sys from *,
   --#                                Announcement,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                ThePackage &
   --#         TypeMark          from Dict;
   -- pre  not IsReservedWord (Name) and
   --      not IsDirectlyDefined (Name, Scope, ProofContext) and
   --      IsPackage (ThePackage);
   -- post TypeMark = LookupItem (Name, ThePackage, ProgramContext));

   procedure AddPrivateType (Name           : in LexTokenManager.LexString;
                             Declaration    : in Location;
                             ThePackage     : in Symbol;
                             IsLimited      : in Boolean;
                             IsTaggedType   : in Boolean;
                             IsAbstractType : in Boolean;
                             Extends        : in Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Extends,
   --#                                IsAbstractType,
   --#                                IsLimited,
   --#                                IsTaggedType,
   --#                                Name,
   --#                                ThePackage &
   --#         SPARK_IO.File_Sys from *,
   --#                                Declaration,
   --#                                Dict,
   --#                                IsLimited,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                ThePackage;
   -- pre  not IsReservedWord (Name) and
   --      not IsDirectlyDefined (Name, Scope, ProofContext) and
   --      IsPackage (ThePackage);
   -- post for_some (TheType: Symbol,
   --        TheType = LookupItem (Name, ThePackage, ProgramContext) and
   --        TypeIsPrivate (TheType) and
   --        IsLimited = IsLimitedPrivateType (TheType) and
   --        Declaration = GetPrivateTypeDeclaration (TheType));

   procedure AddAbstractProofType (Name        : in     LexTokenManager.LexString;
                                   Declaration : in     Location;
                                   Scope       : in     Scopes;
                                   TypeSym     :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         TypeSym           from Dict,
   --#                                Name,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope;

   procedure AddDefaultAbstractProofType (OwnVarName  : in     LexTokenManager.LexString;
                                          Declaration : in     Location;
                                          Scope       : in     Scopes;
                                          TypeSym     :    out Symbol);
   --# global in out Dict;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         TypeSym                     from Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          OwnVarName,
   --#                                          Scope &
   --#         LexTokenManager.StringTable from *,
   --#                                          OwnVarName &
   --#         SPARK_IO.File_Sys           from *,
   --#                                          Declaration,
   --#                                          Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          OwnVarName,
   --#                                          Scope;


   procedure AddEnumerationType (Name            : in     LexTokenManager.LexString;
                                 Declaration     : in     Location;
                                 Scope           : in     Scopes;
                                 Context         : in     Contexts;
                                 EnumerationType :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         EnumerationType   from Context,
   --#                                Dict,
   --#                                Name,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope;
   -- pre  not IsReservedWord (Name) and
   --      IsDirectlyDefined (Name, Scope, Context) =>
   --        IsPrivateType (LookupItem (Name, Scope, Context), Scope);
   -- post EnumerationType = LookupItem (Name, Scope, Context) and
   --      IsEnumerationType (EnumerationType, GetScope (EnumerationType));

   procedure AddEnumerationLiteral (Name     : in LexTokenManager.LexString;
                                    Position : in LexTokenManager.LexString;
                                    TheType  : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Name,
   --#                   Position,
   --#                   TheType;
   -- pre  not IsReservedWord (Name) and
   --      IsProgramType (TheType) =>
   --        not IsDirectlyDefined (Name, GetScope (TheType), ProgramContext) and
   --      IsProofType (TheType) =>
   --        not IsDirectlyDefined (Name, GetScope (TheType), ProofContext) and
   --      IsEnumerationType (TheType);
   -- post forsome (Literal: Symbol,
   --        IsProgramType (TheType) =>
   --          Literal = LookupItem (Name, GetScope (TheType), ProgramContext) and
   --        IsProofType (TheType) =>
   --          Literal = LookupItem (Name, GetScope (TheType), ProofContext) and
   --        IsEnumerationLiteral (Literal) and
   --        TheType = GetType (Literal));

   procedure AddEnumerationRepresentationClause (TheType : in Symbol;
                                                 Clause  : in Location);
   --# global in     Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Clause,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                TheType;
   -- pre IsEnumerationType (TheType);

   procedure AddEnumerationLiteralRepresentation (Literal : in Symbol;
                                                  Code    : in Integer);
   --# global in     Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Code,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Literal;
   -- pre IsEnumerationLiteral (Literal);

   procedure AddIntegerType (Name        : in LexTokenManager.LexString;
                             Declaration : in Location;
                             Lower       : in LexTokenManager.LexString;
                             Upper       : in LexTokenManager.LexString;
                             Scope       : in Scopes;
                             Context     : in Contexts);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Context,
   --#                                Lower,
   --#                                Name,
   --#                                Scope,
   --#                                Upper &
   --#         SPARK_IO.File_Sys from *,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope;
   -- pre  not IsReservedWord (Name) and
   --      IsDirectlyDefined (Name, Scope, Context) =>
   --        IsPrivateType (LookupItem (Name, Scope, Context), Scope);
   -- post for_some (TheType: Symbol,
   --        TheType = LookupItem (Name, Scope, Context) and
   --        IsIntegerType (TheType, GetScope (TheType)) and
   --        GetFirst (TheType) = Lower and
   --        GetLast (TheType) = Upper;

   procedure AddPreDefIntegerType (Name        : in LexTokenManager.LexString;
                                   Declaration : in Location;
                                   Lower       : in LexTokenManager.LexString;
                                   Upper       : in LexTokenManager.LexString;
                                   Scope       : in Scopes;
                                   Context     : in Contexts);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Context,
   --#                                Lower,
   --#                                Name,
   --#                                Scope,
   --#                                Upper &
   --#         SPARK_IO.File_Sys from *,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope;
   -- pre  not IsReservedWord (Name) and
   --      IsDirectlyDefined (Name, Scope, Context) =>
   --        IsPrivateType (LookupItem (Name, Scope, Context), Scope);
   -- post for_some (TheType: Symbol,
   --        TheType = LookupItem (Name, Scope, Context) and
   --        IsIntegerType (TheType, GetScope (TheType)) and
   --        GetFirst (TheType) = Lower and
   --        GetLast (TheType) = Upper;

   procedure AddModularType (Name        : in LexTokenManager.LexString;
                             Declaration : in Location;
                             Modulus     : in LexTokenManager.LexString;
                             Scope       : in Scopes;
                             Context     : in Contexts);
   --# global in out Dict;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict                        from *,
   --#                                          Context,
   --#                                          LexTokenManager.StringTable,
   --#                                          Modulus,
   --#                                          Name,
   --#                                          Scope &
   --#         LexTokenManager.StringTable from *,
   --#                                          Modulus &
   --#         SPARK_IO.File_Sys           from *,
   --#                                          Context,
   --#                                          Declaration,
   --#                                          Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          Name,
   --#                                          Scope;
   -- pre  not IsReservedWord (Name) and
   --      IsDirectlyDefined (Name, Scope, Context) =>
   --        IsPrivateType (LookupItem (Name, Scope, Context), Scope);
   -- post for_some (TheType: Symbol,
   --        TheType = LookupItem (Name, Scope, Context) and
   --        IsModularType (TheType, GetScope (TheType)) and
   --        GetModulus (TheType) = Modulus;

   procedure AddFloatingPointType (Name        : in LexTokenManager.LexString;
                                   Declaration : in Location;
                                   Lower       : in LexTokenManager.LexString;
                                   Upper       : in LexTokenManager.LexString;
                                   ErrorBound  : in LexTokenManager.LexString;
                                   Scope       : in Scopes;
                                   Context     : in Contexts);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Context,
   --#                                ErrorBound,
   --#                                Lower,
   --#                                Name,
   --#                                Scope,
   --#                                Upper &
   --#         SPARK_IO.File_Sys from *,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope;
   -- pre  not IsReservedWord (Name) and
   --      IsDirectlyDefined (Name, Scope, Context) =>
   --        IsPrivateType (LookupItem (Name, Scope, Context), Scope);
   -- post for_some (TheType: Symbol,
   --        TheType = LookupItem (Name, Scope, Context) and
   --        IsFloatingPointType (TheType, GetScope (TheType)) and
   --        GetFirst (TheType) = Lower and
   --        GetLast (TheType) = Upper and
   --        GetScalarAttributeValue (false, DigitsToken, TheType) = ErrorBound;

   procedure AddPreDefFloatingPointType (Name        : in LexTokenManager.LexString;
                                         Declaration : in Location;
                                         Lower       : in LexTokenManager.LexString;
                                         Upper       : in LexTokenManager.LexString;
                                         ErrorBound  : in LexTokenManager.LexString;
                                         Scope       : in Scopes;
                                         Context     : in Contexts);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Context,
   --#                                ErrorBound,
   --#                                Lower,
   --#                                Name,
   --#                                Scope,
   --#                                Upper &
   --#         SPARK_IO.File_Sys from *,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope;
   -- pre  not IsReservedWord (Name) and
   --      IsDirectlyDefined (Name, Scope, Context) =>
   --        IsPrivateType (LookupItem (Name, Scope, Context), Scope);
   -- post for_some (TheType: Symbol,
   --        TheType = LookupItem (Name, Scope, Context) and
   --        IsFloatingPointType (TheType, GetScope (TheType)) and
   --        GetFirst (TheType) = Lower and
   --        GetLast (TheType) = Upper and
   --        GetScalarAttributeValue (false, DigitsToken, TheType) = ErrorBound;

   procedure AddProtectedType (Name            : in     LexTokenManager.LexString;
                               Declaration     : in     Location;
                               Scope           : in     Scopes;
                               Context         : in     Contexts;
                               Mode            : in     Modes;
                               Constrained     : in     Boolean;
                               ProtectedType   :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Constrained,
   --#                                Context,
   --#                                Mode,
   --#                                Name,
   --#                                Scope &
   --#         ProtectedType     from Context,
   --#                                Dict,
   --#                                Name,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope;
   -- pre  not IsReservedWord (Name) and
   --      IsDirectlyDefined (Name, Scope, Context) =>
   --        IsPrivateType (LookupItem (Name, Scope, Context), Scope);
   -- post ProtectedType = LookupItem (Name, Scope, Context);

   procedure AddTaskType (Name            : in     LexTokenManager.LexString;
                          Declaration     : in     Location;
                          Scope           : in     Scopes;
                          Context         : in     Contexts;
                          Constrained     : in     Boolean;
                          TaskType   :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Constrained,
   --#                                Context,
   --#                                Name,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope &
   --#         TaskType          from Context,
   --#                                Dict,
   --#                                Name,
   --#                                Scope;
   -- pre  not IsReservedWord (Name) and
   --      IsDirectlyDefined (Name, Scope, Context) =>
   --        IsPrivateType (LookupItem (Name, Scope, Context), Scope);
   -- post TaskType = LookupItem (Name, Scope, Context);

   procedure AddTaskOrProtectedSubtype (Name            : in     LexTokenManager.LexString;
                                        Parent          : in     Symbol;
                                        Declaration     : in     Location;
                                        Scope           : in     Scopes;
                                        Context         : in     Contexts;
                                        TheSubtype      :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Context,
   --#                                Name,
   --#                                Parent,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Parent,
   --#                                Scope &
   --#         TheSubtype        from Context,
   --#                                Dict,
   --#                                Name,
   --#                                Scope;
   -- pre  IsTaskType (Parent) or IsProtectedType (Parent);

   procedure SetTypeIsPredefined (TypeMark : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   TypeMark;

   procedure SetTypeIsWellformed (TypeMark   : in Symbol;
                                  Wellformed : in Boolean);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   TypeMark,
   --#                   Wellformed;

   procedure SetBaseType (TypeMark, BaseType : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   BaseType,
   --#                   TypeMark;
   -- pre  not IsSubtype (TypeMark) and IsType (TypeMark)

   procedure AddFixedPointType (Name        : in LexTokenManager.LexString;
                                Declaration : in Location;
                                Lower       : in LexTokenManager.LexString;
                                Upper       : in LexTokenManager.LexString;
                                ErrorBound  : in LexTokenManager.LexString;
                                Scope       : in Scopes;
                                Context     : in Contexts);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Context,
   --#                                ErrorBound,
   --#                                Lower,
   --#                                Name,
   --#                                Scope,
   --#                                Upper &
   --#         SPARK_IO.File_Sys from *,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope;
   -- pre  not IsReservedWord (Name) and
   --      IsDirectlyDefined (Name, Scope, Context) =>
   --        IsPrivateType (LookupItem (Name, Scope, Context), Scope);
   -- post for_some (TheType: Symbol,
   --        TheType = LookupItem (Name, Scope, Context) and
   --        IsFixedPointType (TheType, GetScope (TheType)) and
   --        GetFirst (TheType) = Lower and
   --        GetLast (TheType) = Upper and
   --        GetScalarAttributeValue (false, DeltaToken, TheType) = ErrorBound;


   procedure AddArrayType (Name                   : in     LexTokenManager.LexString;
                           Declaration            : in     Location;
                           Scope                  : in     Scopes;
                           Context                : in     Contexts;
                           Constrained            : in     Boolean;
                           ComponentType          : in     Symbol;
                           ComponentTypeReference : in     Location;
                           IsGeneric              : in     Boolean;
                           TheArrayType           :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                ComponentType,
   --#                                Constrained,
   --#                                Context,
   --#                                IsGeneric,
   --#                                Name,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                ComponentType,
   --#                                ComponentTypeReference,
   --#                                Constrained,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                IsGeneric,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope &
   --#         TheArrayType      from Context,
   --#                                Dict,
   --#                                Name,
   --#                                Scope;
   -- pre  not IsReservedWord (Name) and
   --      IsDirectlyDefined (Name, Scope, Context) =>
   --        IsPrivateType (LookupItem (Name, Scope, Context), Scope) and
   --      IsTypeMark (ComponentType);
   -- post TheArrayType = LookupItem (Name, Scope, Context) and
   --      IsArrayType (TheArrayType) and
   --      Constrained = IsConstrainedArrayType (TheArrayType,
   --                                            GetScope (TheArrayType)) and
   --      not Constrained = IsUnconstrainedArrayType (TheArrayType) and
   --      ComponentType = GetArrayComponent (TheArrayType));

   procedure AddArrayIndex (TheArrayType       : in Symbol;
                            TheIndexType       : in Symbol;
                            IndexTypeReference : in Location);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                TheArrayType,
   --#                                TheIndexType &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                IndexTypeReference,
   --#                                LexTokenManager.StringTable,
   --#                                TheArrayType,
   --#                                TheIndexType;
   -- pre  IsArrayTypeMark (TheArrayType, GetScope (TheArrayType)) and
   --      IsDiscreteTypeMark (IndexType, GetScope (TheArrayType));
   -- post IndexType = GetArrayIndex (TheArrayType, Dimension);

   procedure AddRecordType (Name           : in     LexTokenManager.LexString;
                            IsTaggedType   : in     Boolean;
                            IsAbstractType : in     Boolean;
                            Extends        : in     Symbol;
                            Declaration    : in     Location;
                            Scope          : in     Scopes;
                            Context        : in     Contexts;
                            TheRecordType  :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Context,
   --#                                Extends,
   --#                                IsAbstractType,
   --#                                IsTaggedType,
   --#                                Name,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope &
   --#         TheRecordType     from Context,
   --#                                Dict,
   --#                                Name,
   --#                                Scope;
   -- pre  not IsReservedWord (Name) and
   --      IsDirectlyDefined (Name, Scope, Context) =>
   --        IsPrivateType (LookupItem (Name, Scope, Context), Scope);
   -- post TheRecordType = LookupItem (Name, Scope, Context) and
   --      IsRecordType (TheRecordType, GetScope (TheRecordType)));

   procedure AddRecordComponent (Name                   : in LexTokenManager.LexString;
                                 TheRecordType          : in Symbol;
                                 TheComponentType       : in Symbol;
                                 InheritedField         : in Boolean; --tag
                                 ComponentTypeReference : in Location);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                InheritedField,
   --#                                Name,
   --#                                TheComponentType,
   --#                                TheRecordType &
   --#         SPARK_IO.File_Sys from *,
   --#                                ComponentTypeReference,
   --#                                Dict,
   --#                                InheritedField,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                TheComponentType,
   --#                                TheRecordType;
   -- pre  not IsReservedWord (Name) and
   --      IsProgramDeclarativeItem (TheRecordType) =>
   --        not IsDirectlyDefined (Name, GetScope (TheRecordType), ProgramContext) and
   --      IsProofDeclarativeItem (TheRecordType) =>
   --        not IsDirectlyDefined (Name, GetScope (TheRecordType), ProofContext) and
   --      IsRecordTypeMark (TheRecordType, GetScope (TheRecordType)) and
   --      IsTypeMark (TheComponentType);
   -- post IsProgramDeclarativeItem (TheRecordType) =>
   --        IsDirectlyDefined (Name, GetScope (TheRecordType), ProgramContext) and
   --      IsProofDeclarativeItem (TheRecordType) =>
   --        IsDirectlyDefined (Name, GetScope (TheRecordType), ProofContext);

   -- Generic types

   procedure AddGenericLimitedType (Name        : in     LexTokenManager.LexString;
                                    Declaration : in     Location;
                                    Scope       : in     Scopes;
                                    TypeSym     :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         TypeSym           from Dict,
   --#                                Name,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope;

   procedure AddGenericPrivateType (Name        : in     LexTokenManager.LexString;
                                    Declaration : in     Location;
                                    Scope       : in     Scopes;
                                    TypeSym     :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         TypeSym           from Dict,
   --#                                Name,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope;

   procedure AddGenericOrderedDiscreteType (Name        : in     LexTokenManager.LexString;
                                            Declaration : in     Location;
                                            Scope       : in     Scopes;
                                            TypeSym     :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         TypeSym           from Dict,
   --#                                Name,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope;

   procedure AddGenericUnorderedDiscreteType (Name        : in     LexTokenManager.LexString;
                                              Declaration : in     Location;
                                              Scope       : in     Scopes;
                                              TypeSym     :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         TypeSym           from Dict,
   --#                                Name,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope;

   procedure AddGenericIntegerType (Name        : in     LexTokenManager.LexString;
                                    Declaration : in     Location;
                                    Scope       : in     Scopes;
                                    TypeSym     :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         TypeSym           from Dict,
   --#                                Name,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope;

   procedure AddGenericModularType (Name        : in     LexTokenManager.LexString;
                                    Declaration : in     Location;
                                    Scope       : in     Scopes;
                                    TypeSym     :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         TypeSym           from Dict,
   --#                                Name,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope;

   procedure AddGenericFloatingPointType (Name        : in     LexTokenManager.LexString;
                                          Declaration : in     Location;
                                          Scope       : in     Scopes;
                                          TypeSym     :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         TypeSym           from Dict,
   --#                                Name,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope;

   procedure AddGenericFixedPointType (Name        : in     LexTokenManager.LexString;
                                       Declaration : in     Location;
                                       Scope       : in     Scopes;
                                       TypeSym     :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         TypeSym           from Dict,
   --#                                Name,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope;

   -- rep clauses

   procedure AddRecordRepresentationClause (TheType : in Symbol;
                                            Clause  : in Location);
   --# global in     Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Clause,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                TheType;
   -- pre IsRecordType (TheType);

   procedure AddRecordComponentRepresentation (Component        : in Symbol;
                                               Clause           : in Location;
                                               RelativeAddress  : in Natural;
                                               FirstBitPosition : in Natural;
                                               LastBitPosition  : in Natural);
   --# global in     Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Clause,
   --#                                Component,
   --#                                Dict,
   --#                                FirstBitPosition,
   --#                                LastBitPosition,
   --#                                LexTokenManager.StringTable,
   --#                                RelativeAddress;
   -- pre IsRecordComponent (Component) and FirstBitPosition <= LastBitPosition;

   procedure AddAlignmentClause (TheType : in Symbol; Clause : in Location);
   --# global in     Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Clause,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                TheType;
   -- pre IsRecordType (TheType);

   procedure AddLoop (Scope         : in     Scopes;
                      LoopStatement : in     Location;
                      TheLoop       :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                LoopStatement,
   --#                                Scope &
   --#         TheLoop           from Dict;
   -- pre  IsLocalScope (Scope);
   -- post IsLoop (TheLoop) and not LoophasName (TheLoop);

   procedure AddLoopName (Name    : in LexTokenManager.LexString;
                          TheLoop : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Name,
   --#                   TheLoop;
   -- pre  not IsReservedWord (Name) and
   --      not IsDirectlyVisible (Name, GetScope (TheLoop), ProgramContext) and
   --      IsLoop (TheLoop) and
   --      not LoopHasName (TheLoop);
   -- post LoopHasName (TheLoop) and
   --      Name = GetSimpleName (TheLoop) and
   --      IsDirectlyVisible (Name, GetScope (TheLoop), ProgramContext);

   procedure AddLoopParameter (TheLoop       : in Symbol;
                               Name          : in LexTokenManager.LexString;
                               TypeMark      : in Symbol;
                               StaticRange   : in Boolean;
                               IsReverse     : in Boolean;
                               TypeReference : in Location);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                IsReverse,
   --#                                Name,
   --#                                StaticRange,
   --#                                TheLoop,
   --#                                TypeMark &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                IsReverse,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                StaticRange,
   --#                                TheLoop,
   --#                                TypeMark,
   --#                                TypeReference;
   -- pre  IsLoop (TheLoop) and
   --      not IsReservedWord (Name) and
   --      not IsDirectlyDefined (Name, Scope, ProofContext);
   --      IsTypeMark (TypeMark);
   -- post for_some (LoopParameter: Symbol,
   --        LoopParameter = LookupItem (Name, TheLoop, ProgramContext) and
   --        IsLoopParameter (LoopParameter) and
   --        TypeMark = GetType (LoopParameter));

   procedure MarkLoopHasExits (TheLoop       : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   TheLoop;
   -- pre  IsLoop (TheLoop);
   -- post GetLoopHasExits (TheLoop);

   -- This is used by the VCG to store a Cells.Cell representing the exit expression of
   -- a for loop
   procedure SetLoopExitExpn (ForLoop : in Symbol;
                              Expn    : in Natural);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Expn,
   --#                   ForLoop;
   -- pre  IsForLoop (TheLoop);

   -- Following is used by VCG in for loop modelling to create a variable uniquely associated
   -- with an original variable and a loop.  The original variable is used in the exit expression
   -- of a for loop and the new one is used to store the value on entry to the loop so as to
   -- freeze the loop bounds as required by Ada semantics.
   procedure IdempotentCreateLoopOnEntryVariable (OriginalVariable : in     Symbol;
                                                  TheLoop          : in     Symbol;
                                                  OnEntryVariable  :    out Symbol);
   --# global in out Dict;
   --# derives Dict,
   --#         OnEntryVariable from Dict,
   --#                              OriginalVariable,
   --#                              TheLoop;
   -- pre  IsForLoop (TheLoop);

   -- Provides access to variable created by previous call.
   function GetLoopOnEntryVariable (OriginalVariable : Symbol;
                                    TheLoop          : Symbol) return Symbol;
   --# global in Dict;
   -- pre  IsForLoop (TheLoop); -- (and IdempotentCreateLoopOnEntryVariable has previously been invoked)

   procedure AddEnumerationSubtype (Name            : in LexTokenManager.LexString;
                                    Static          : in Boolean;
                                    Parent          : in Symbol;
                                    ParentReference : in Location;
                                    Lower           : in LexTokenManager.LexString;
                                    Upper           : in LexTokenManager.LexString;
                                    Declaration     : in Location;
                                    Scope           : in Scopes;
                                    Context         : in Contexts);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Context,
   --#                                Lower,
   --#                                Name,
   --#                                Parent,
   --#                                Scope,
   --#                                Static,
   --#                                Upper &
   --#         SPARK_IO.File_Sys from *,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Parent,
   --#                                ParentReference,
   --#                                Scope;
   -- pre  not IsReservedWord (Name) and
   --      not IsDirectlyDefined (Name, Scope, Context) and
   --      IsEnumerationTypeMark (Parent, Scope);
   -- post for_some (TheSubtype: Symbol,
   --        TheSubtype = LookupItem (Name, Scope, Context) and
   --        IsEnumerationSubtype (TheSubtype) and
   --        Static = IsStatic (TheSubtype, Scope) and
   --        Parent = GetParentType (TheSubtype)) and
   --        GetFirst (TheSubtype) = Lower and
   --        GetLast (TheSubType) = Upper;

   procedure AddIntegerSubtype (Name            : in LexTokenManager.LexString;
                                Static          : in Boolean;
                                Parent          : in Symbol;
                                ParentReference : in Location;
                                Lower           : in LexTokenManager.LexString;
                                Upper           : in LexTokenManager.LexString;
                                Declaration     : in Location;
                                Scope           : in Scopes;
                                Context         : in Contexts);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Context,
   --#                                Lower,
   --#                                Name,
   --#                                Parent,
   --#                                Scope,
   --#                                Static,
   --#                                Upper &
   --#         SPARK_IO.File_Sys from *,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Parent,
   --#                                ParentReference,
   --#                                Scope;
   -- pre  not IsReservedWord (Name) and
   --      not IsDirectlyDefined (Name, Scope, Context) and
   --      IsIntegerTypeMark (Parent, Scope);
   -- post for_some (TheSubtype: Symbol,
   --        TheSubtype = LookupItem (Name, Scope, Context) and
   --        IsIntegerSubtype (TheSubtype) and
   --        Static = IsStatic (TheSubtype, Scope) and
   --        Parent = GetParentType (TheSubtype)) and
   --        GetFirst (TheSubtype) = Lower and
   --        GetLast (TheSubType) = Upper;

   procedure AddRecordSubtype (Name            : in LexTokenManager.LexString;
                               Parent          : in Symbol;
                               ParentReference : in Location;
                               Declaration     : in Location;
                               Scope           : in Scopes;
                               Context         : in Contexts);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Context,
   --#                                Name,
   --#                                Parent,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Parent,
   --#                                ParentReference,
   --#                                Scope;
   -- pre  not IsReservedWord (Name) and
   --      not IsDirectlyDefined (Name, Scope, Context) and
   --      IsRecordTypeMark (Parent, Scope) and
   --      not TypeIsTagged (Parent);
   -- post for_some (TheSubtype: Symbol,
   --        TheSubtype = LookupItem (Name, Scope, Context) and
   --        IsRecordSubtype (TheSubtype) and
   --        Parent = GetParentType (TheSubtype));

   procedure AddModularSubtype (Name            : in LexTokenManager.LexString;
                                Parent          : in Symbol;
                                ParentReference : in Location;
                                Lower           : in LexTokenManager.LexString;
                                Upper           : in LexTokenManager.LexString;
                                Declaration     : in Location;
                                Scope           : in Scopes;
                                Context         : in Contexts);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Context,
   --#                                Lower,
   --#                                Name,
   --#                                Parent,
   --#                                Scope,
   --#                                Upper &
   --#         SPARK_IO.File_Sys from *,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Parent,
   --#                                ParentReference,
   --#                                Scope;
   -- pre  not IsReservedWord (Name) and
   --      not IsDirectlyDefined (Name, Scope, Context) and
   --      IsModularTypeMark (Parent, Scope);
   -- post for_some (TheSubtype: Symbol,
   --        TheSubtype = LookupItem (Name, Scope, Context) and
   --        IsModularSubtype (TheSubtype) and
   --        Static = IsStatic (TheSubtype, Scope) and
   --        Parent = GetParentType (TheSubtype)) and
   --        GetFirst (TheSubtype) = Lower and
   --        GetLast (TheSubType) = Upper and
   --        GetModulus (TheType) = Modulus;

   procedure AddFloatingPointSubtype (Name            : in LexTokenManager.LexString;
                                      Static          : in Boolean;
                                      Parent          : in Symbol;
                                      ParentReference : in Location;
                                      Lower           : in LexTokenManager.LexString;
                                      Upper           : in LexTokenManager.LexString;
                                      ErrorBound      : in LexTokenManager.LexString;
                                      Declaration     : in Location;
                                      Scope           : in Scopes;
                                      Context         : in Contexts);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Context,
   --#                                ErrorBound,
   --#                                Lower,
   --#                                Name,
   --#                                Parent,
   --#                                Scope,
   --#                                Static,
   --#                                Upper &
   --#         SPARK_IO.File_Sys from *,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Parent,
   --#                                ParentReference,
   --#                                Scope;
   -- pre  not IsReservedWord (Name) and
   --      not IsDirectlyDefined (Name, Scope, Context) and
   --      IsFloatingPointTypeMark (Parent, Scope);
   -- post for_some (TheSubtype: Symbol,
   --        TheSubtype = LookupItem (Name, Scope, Context) and
   --        IsFloatingPointSubtype (TheSubtype) and
   --        Static = IsStatic (TheSubtype, Scope) and
   --        Parent = GetParentType (TheSubtype)) and
   --        GetFirst (TheSubtype) = Lower and
   --        GetLast (TheSubType) = Upper and
   --        GetScalarAttributeValue (false, DigitsToken, TheSubtype) = ErrorBound;

   procedure AddFixedPointSubtype (Name            : in LexTokenManager.LexString;
                                   Static          : in Boolean;
                                   Parent          : in Symbol;
                                   ParentReference : in Location;
                                   Lower           : in LexTokenManager.LexString;
                                   Upper           : in LexTokenManager.LexString;
                                   ErrorBound      : in LexTokenManager.LexString;
                                   Declaration     : in Location;
                                   Scope           : in Scopes;
                                   Context         : in Contexts);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Context,
   --#                                ErrorBound,
   --#                                Lower,
   --#                                Name,
   --#                                Parent,
   --#                                Scope,
   --#                                Static,
   --#                                Upper &
   --#         SPARK_IO.File_Sys from *,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Parent,
   --#                                ParentReference,
   --#                                Scope;
   -- pre  not IsReservedWord (Name) and
   --      not IsDirectlyDefined (Name, Scope, Context) and
   --      IsFixedPointTypeMark (Parent, Scope);
   -- post for_some (TheSubtype: Symbol,
   --        TheSubtype = LookupItem (Name, Scope, Context) and
   --        IsFixedPointSubtype (TheSubtype) and
   --        Static = IsStatic (TheSubtype, Scope) and
   --        Parent = GetParentType (TheSubtype)) and
   --        GetFirst (TheSubtype) = Lower and
   --        GetLast (TheSubtype) = Upper and
   --        GetScalarAttributeValue (false, DeltaToken, TheSubtype) = ErrorBound;

   procedure AddArraySubtype (Name            : in     LexTokenManager.LexString;
                              Parent          : in     Symbol;
                              ParentReference : in     Location;
                              Declaration     : in     Location;
                              Scope           : in     Scopes;
                              Context         : in     Contexts;
                              Static          : in     Boolean;
                              ArraySubtype    :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives ArraySubtype      from Context,
   --#                                Dict,
   --#                                Name,
   --#                                Scope &
   --#         Dict              from *,
   --#                                Context,
   --#                                Name,
   --#                                Parent,
   --#                                Scope,
   --#                                Static &
   --#         SPARK_IO.File_Sys from *,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Parent,
   --#                                ParentReference,
   --#                                Scope;
   -- pre  not IsReservedWord (Name) and
   --      not IsDirectlyDefined (Name, Scope, Context) and
   --      IsArrayType (Parent, Scope);
   -- post ArraySubtype = LookupItem (Name, Scope, Context) and
   --      IsArraySubtype (ArraySubtype) and
   --      Parent = GetParentType (ArraySubtype));

   procedure AddAssertStatement (Statement : in Location);
   --# global in     Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                Statement;

   procedure AddCheckStatement (Statement : in Location);
   --# global in     Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                Statement;

   procedure AddPrecondition (Abstraction   : in Abstractions;
                              Subprogram    : in Symbol;
                              Predicate     : in ExaminerConstants.RefType;
                              Precondition  : in Location);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Abstraction,
   --#                                Predicate,
   --#                                Subprogram &
   --#         SPARK_IO.File_Sys from *,
   --#                                Abstraction,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Precondition,
   --#                                Predicate,
   --#                                Subprogram;
   -- pre IsSubprogram (Subprogram);

   procedure AddPostcondition (Abstraction     : in Abstractions;
                               CompilationUnit : in Symbol;
                               Predicate       : in ExaminerConstants.RefType;
                               Postcondition   : in Location);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Abstraction,
   --#                                CompilationUnit,
   --#                                Predicate &
   --#         SPARK_IO.File_Sys from *,
   --#                                Abstraction,
   --#                                CompilationUnit,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Postcondition,
   --#                                Predicate;
   -- pre IsCompilationUnit (CompilationUnit);

   procedure AddSubprogram (Name          : in     LexTokenManager.LexString;
                            Specification : in     Location;
                            Scope         : in     Scopes;
                            Context       : in     Contexts;
                            Subprogram    :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Context,
   --#                                Name,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Context,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope,
   --#                                Specification &
   --#         Subprogram        from Dict;
   -- pre  not IsReservedWord (Name) and
   --      not IsDirectlyDefined (Name, Scope, Context);
   -- post Subprogram = LookupItem (Name, Scope, Context) and
   --      IsSubprogram (Subprogram);

   function ActualOfGenericFormal (TheGenericFormalSym : in Symbol;
                                   ActualSubprogramSym : in Symbol) return Symbol;
   --# global in Dict;
   -- returns the symbol that is the generic actual that matches the given generic formal for a
   -- given instantiation

   function ActualParameterOfGenericParameter (TheParameter        : in Symbol;
                                               ActualSubprogramSym : in Symbol) return Symbol;
   --# global in Dict;
   -- returns the symbol of the actual parameter that has the same posiiton number in the instantiated
   -- subprogram as the parameter symbol  has in the generic from which it comes

   procedure AddSubprogramInstantiation (Name          : in     LexTokenManager.LexString;
                                         TheGeneric    : in     Symbol;
                                         Specification : in     Location;
                                         Scope         : in     Scopes;
                                         Context       : in     Contexts;
                                         Subprogram    :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Context,
   --#                                Name,
   --#                                Scope,
   --#                                TheGeneric &
   --#         SPARK_IO.File_Sys from *,
   --#                                Context,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope,
   --#                                Specification &
   --#         Subprogram        from Dict;
   -- pre  not IsReservedWord (Name) and
   --      not IsDirectlyDefined (Name, Scope, Context) and
   --      IsGenericSubprogram (TheGeneric);
   -- post Subprogram = LookupItem (Name, Scope, Context) and
   --      IsSubprogram (Subprogram);

   function IsInstantiation (PackageOrSubProgram : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsSubprogram (PackageOrSubProgram) or IsPackage (PackageOrSubProgram);

   function GetGenericOfInstantiation (PackageOrSubProgram : Symbol) return Symbol;
   --# global in Dict;
   -- pre IsSubprogram (PackageOrSubProgram) or IsPackage (PackageOrSubProgram);


   procedure InstantiateSubprogramParameters (GenericSubprogramSym : in Symbol;
                                              ActualSubprogramSym  : in Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                ActualSubprogramSym,
   --#                                GenericSubprogramSym &
   --#         SPARK_IO.File_Sys from *,
   --#                                ActualSubprogramSym,
   --#                                Dict,
   --#                                GenericSubprogramSym,
   --#                                LexTokenManager.StringTable;
   -- pre IsGenericSubprogram (GenericSubprogramSym) and
   --     IsInstantiationOf (GenericSubprogramSym, ActualSubprogramSym);
   -- Copies the parameters and return types from Generic to Instantiation substituting
   -- actual types as it goes (using generic association linked list of Instantiation)


   procedure SetSubprogramSignatureNotWellformed
      (Abstraction   : in Abstractions;
       Subprogram    : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Abstraction,
   --#                   Subprogram;
   -- pre IsSubprogram (Subprogram) or IsTaskType (Subprogram);

   procedure SetSubprogramIsEntry (Subprogram    : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Subprogram;

   procedure SetSubprogramIsGeneric (Subprogram    : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Subprogram;

   procedure SetSubprogramEntryBarrier (Subprogram, TheBarrier  : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Subprogram,
   --#                   TheBarrier;

   procedure SetIsInterruptHandler (Subprogram    : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Subprogram;

   procedure SetHasDelayProperty (TheProcedure : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   TheProcedure;
   -- pre IsProcedure (TheProcedure);

   procedure SetUsesUnprotectedVariables (Sym : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Sym;
   -- pre IsTaskType (Sym) or IsSubprogram (Sym);

   procedure SetUnprotectedReference (Variable : in Symbol;
                                      ByThread : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   ByThread,
   --#                   Variable;
   -- pre IsOwnVariable (Variable) and IsThread (ByThread);

   procedure SetSuspendsReference (Variable : in Symbol;
                                   ByThread : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   ByThread,
   --#                   Variable;
   -- pre IsOwnVariable (Variable) and IsThread (ByThread);

   procedure SetVirtualElementSeenByOwner (TheVariable : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   TheVariable;
   -- pre IsOwnVariable (TheVariable);

   procedure SetPriorityProperty (OwnVariable : in Symbol;
                                  TheValue    : in LexTokenManager.LexString);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   OwnVariable,
   --#                   TheValue;
   -- pre IsOwnVariable (OwnVariable);

   procedure SetIntegrityProperty (OwnVariable : in Symbol;
                                   TheValue    : in LexTokenManager.LexString);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   OwnVariable,
   --#                   TheValue;
   -- pre IsOwnVariable (OwnVariable);

   procedure SetIsSuspendable (Variable : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Variable;
   -- pre IsOwnVariable (Variable);

   procedure SetHasInterruptProperty (Variable : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Variable;
   -- pre IsOwnVariable (Variable);

   procedure MarkAccountsForDelay (TheProcedure : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   TheProcedure;
   -- pre IsProcedure (TheProcedure);

   procedure MarkAccountsForSuspendsListItem (TheTaskOrProc : in Symbol;
                                              ThePOorSO     : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   ThePOorSO,
   --#                   TheTaskOrProc;
   -- pre IsTaskType (TheTaskOrProc) or IsProcedure (TheTaskOrProc);

   procedure MarkAccountsForSuspendsListItems (TheTaskOrProc       : in Symbol;
                                               TheItemsInProcedure : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   TheItemsInProcedure,
   --#                   TheTaskOrProc;
   -- pre IsProcedure (TheItemsInProcedure) and
   --     (IsTaskType (TheTaskOrProc) or IsProcedure (TheTaskOrProc));

   procedure SetProtectedTypeElementsHidden (TheProtectedType : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   TheProtectedType;

   procedure SetProtectedTypeEntry (TheProtectedType, TheEntry : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   TheEntry,
   --#                   TheProtectedType;

   procedure SetTypeHasPragma (TheProtectedOrTaskType : in Symbol;
                               ThePragma              : in RavenscarPragmas);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   ThePragma,
   --#                   TheProtectedOrTaskType;

   procedure SetTypePragmaValue (TheProtectedOrTaskType : in Symbol;
                                 ThePragma              : in RavenscarPragmasWithValue;
                                 TheValue               : in LexTokenManager.LexString);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   ThePragma,
   --#                   TheProtectedOrTaskType,
   --#                   TheValue;

   procedure SetMainProgramPriority (TheValue : in LexTokenManager.LexString);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   TheValue;
   -- pre MainProgramExists;

   procedure AddReturnType (TheFunction   : in Symbol;
                            TypeMark      : in Symbol;
                            TypeReference : in Location);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                TheFunction,
   --#                                TypeMark &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                TheFunction,
   --#                                TypeMark,
   --#                                TypeReference;
   -- pre IsSubprogram (TheFunction) and IsTypeMark (TypeMark);
   -- post IsFunction (TheFunction) and
   --      TypeMark = GetType (TheFunction);

   procedure AddImplicitReturnVariable (Abstraction : in     Abstractions;
                                        Name        : in     LexTokenManager.LexString;
                                        TheFunction : in     Symbol;
                                        Variable    :    out Symbol);
   --# global in out Dict;
   --# derives Dict     from *,
   --#                       Abstraction,
   --#                       Name,
   --#                       TheFunction &
   --#         Variable from Dict;
   -- pre  not IsReservedWord (Name) and
   --      not IsDirectlyDefined (Name, LocalScope (TheFunction), Proof Context);
   -- post Variable = GetImplicitReturnVariable (Abstraction, TheFunction) and
   --      IsVariable (Variable);

   procedure AddSubprogramParameter (Name          : in LexTokenManager.LexString;
                                     Subprogram    : in Symbol;
                                     TypeMark      : in Symbol;
                                     TypeReference : in Location;
                                     Mode          : in Modes;
                                     Specification : in Location);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Mode,
   --#                                Name,
   --#                                Subprogram,
   --#                                TypeMark &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Mode,
   --#                                Name,
   --#                                Specification,
   --#                                Subprogram,
   --#                                TypeMark,
   --#                                TypeReference;
   -- pre  not IsReservedWord (Name) and
   --      IsProofFunction (Subprogram) =>
   --        Name = LexTokenManager.NullString and
   --      IsAdaFunction (Subprogram) =>
   --        not IsDirectlyDefined (Name, LocalScope (Subprogram), ProgramContext) and
   --      IsSubprogram (TheFunction) and
   --      IsTypeMark (TypeMark);
   -- post for_some (Parameter: Symbol,
   --        IsAdaFunction (Subprogram) =>
   --          Parameter = LookupItem (Name, LocalScope (Subprogram), ProgramContext) and
   --        Parameter = GetSubprogramParameter (Subprogram),
   --          GetNumberOfSubprogramParameters (Subprogram)) and
   --        TypeMark = GetType (Parameter) and
   --        Mode = GetSubprogramParameterMode (Parameter));

   procedure AddGenericFormalParameter (Name                : in LexTokenManager.LexString; -- not used for types
                                        SubprogramOrPackage : in Symbol;
                                        TypeMark            : in Symbol;
                                        Kind                : in GenericParameterKind);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Kind,
   --#                   Name,
   --#                   SubprogramOrPackage,
   --#                   TypeMark;

   procedure AddGenericAssociation (SubprogramOrPackage : in Symbol;
                                    FormalSym           : in Symbol;
                                    ActualSym           : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   ActualSym,
   --#                   FormalSym,
   --#                   SubprogramOrPackage;

   procedure AddKnownDiscriminant (Name                : in LexTokenManager.LexString;
                                   ProtectedOrTaskType : in Symbol;
                                   TypeMark            : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Name,
   --#                   ProtectedOrTaskType,
   --#                   TypeMark;
   -- pre IsProtectedType (ProtectedOrTaskType) or IsTaskType (ProtectedOrTaskType);

   procedure SetDiscriminantSetsPriority (TheDiscriminant : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   TheDiscriminant;
   -- pre IsKnownDiscriminant (TheDiscriminant);

   procedure AddDiscriminantConstraintStaticValue (ProtectedOrTaskSubtype : in Symbol;
                                                   TheValue               : in LexTokenManager.LexString);

   --# global in out Dict;
   --# derives Dict from *,
   --#                   ProtectedOrTaskSubtype,
   --#                   TheValue;
   -- pre IsProtectedType (GetParentType (ProtectedOrTaskType)) or
   --     IsTaskType (GetParentType (ProtectedOrTaskType));

   procedure AddDiscriminantConstraintAccessedObject (ProtectedOrTaskSubtype : in Symbol;
                                                      TheObject              : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   ProtectedOrTaskSubtype,
   --#                   TheObject;
   -- pre IsProtectedType (GetParentType (ProtectedOrTaskType)) or
   --     IsTaskType (GetParentType (ProtectedOrTaskType));

   procedure SetSubtypePriority (ProtectedOrTaskSubtype : in Symbol;
                                 ThePriority            : in LexTokenManager.LexString);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   ProtectedOrTaskSubtype,
   --#                   ThePriority;
   -- pre IsProtectedType (GetParentType (ProtectedOrTaskType)) or
   --     IsTaskType (GetParentType (ProtectedOrTaskType));

   procedure AddBody (CompilationUnit : in Symbol;
                      TheBody         : in Location;
                      Hidden          : in Boolean);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                CompilationUnit,
   --#                                Hidden &
   --#         SPARK_IO.File_Sys from *,
   --#                                CompilationUnit,
   --#                                Dict,
   --#                                Hidden,
   --#                                LexTokenManager.StringTable,
   --#                                TheBody;
   -- pre IsCompilationUnit (CompilationUnit);
   -- post HasBody (CompilationUnit);

   procedure AddBodyStub (CompilationUnit : in Symbol; BodyStub : in Location);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                CompilationUnit &
   --#         SPARK_IO.File_Sys from *,
   --#                                BodyStub,
   --#                                CompilationUnit,
   --#                                Dict,
   --#                                LexTokenManager.StringTable;
   -- pre IsCompilationUnit (CompilationUnit);
   -- post HasBody (CompilationUnit);

   procedure AddMainProgram (Subprogram : in Symbol;
                             Annotation : in Location);
   --# global in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Subprogram &
   --#         SPARK_IO.File_Sys from *,
   --#                                Annotation,
   --#                                Dict,
   --#                                Subprogram;
   -- pre  IsSubprogram (Symbol);
   -- post Subprogram = GetMainProgram;

   procedure AddDependencyRelation (Abstraction        : in Abstractions;
                                    TheProcedure       : in Symbol;
                                    DependencyRelation : in Location);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Abstraction,
   --#                                TheProcedure &
   --#         SPARK_IO.File_Sys from *,
   --#                                Abstraction,
   --#                                DependencyRelation,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                TheProcedure;
   -- pre IsProcedure (TheProcedure);

   procedure RenameSubprogram (Subprogram          : in Symbol;
                               SubprogramReference : in Location;
                               Declaration         : in Location;
                               Scope               : in Scopes);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Scope,
   --#                                Subprogram &
   --#         SPARK_IO.File_Sys from *,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Scope,
   --#                                Subprogram,
   --#                                SubprogramReference;
   -- pre  IsSubprogram (Subprogram);
   -- post IsRenamed (Subprogram, Scope);

   procedure RenameUnaryOperator (Name    : in SPSymbols.SPSymbol;
                                  Operand : in Symbol;
                                  Scope   : in Scopes);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Name,
   --#                                Operand,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Operand,
   --#                                Scope;
   -- pre UnaryOperatorIsDefined (Name, Operand);

   procedure RenameBinaryOperator (Name  : in SPSymbols.SPSymbol;
                                   Left  : in Symbol;
                                   Right : in Symbol;
                                   Scope : in Scopes);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Left,
   --#                                Name,
   --#                                Right,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                Left,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Right,
   --#                                Scope;
   -- pre BinaryOperatorIsDefined (Name, Left, Right);

   procedure AddGlobalAnnotation (Abstraction : in Abstractions;
                                  Subprogram  : in Symbol;
                                  Annotation  : in Location);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Abstraction,
   --#                                Subprogram &
   --#         SPARK_IO.File_Sys from *,
   --#                                Abstraction,
   --#                                Annotation,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Subprogram;
   -- pre IsSubprogram (Subprogram);

   procedure AddGlobalVariable (Abstraction       : in Abstractions;
                                Subprogram        : in Symbol;
                                Variable          : in Symbol;
                                Mode              : in Modes;
                                PrefixNeeded      : in Boolean;
                                VariableReference : in Location);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Abstraction,
   --#                                Mode,
   --#                                PrefixNeeded,
   --#                                Subprogram,
   --#                                Variable &
   --#         SPARK_IO.File_Sys from *,
   --#                                Abstraction,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Mode,
   --#                                PrefixNeeded,
   --#                                Subprogram,
   --#                                Variable,
   --#                                VariableReference;
   -- pre  IsSubprogram (Subprogram) and
   --      IsVariable (Variable) and
   --      IsDefined (Variable, GetScope (Subprogram), ProgramContext);
   -- post IsGlobalVariable (Subprogram, Variable);

   procedure AddExport (Abstraction     : in Abstractions;
                        TheProcedure    : in Symbol;
                        TheExport       : in Symbol;
                        ExportReference : in Location;
                        Annotation      : in Location);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Abstraction,
   --#                                TheExport,
   --#                                TheProcedure &
   --#         SPARK_IO.File_Sys from *,
   --#                                Abstraction,
   --#                                Annotation,
   --#                                Dict,
   --#                                ExportReference,
   --#                                LexTokenManager.StringTable,
   --#                                TheExport,
   --#                                TheProcedure;
   -- pre  IsGlobalVariable (TheProcedure, TheExport) or
   --      TheProcedure := GetSubprogram (TheExport);
   -- post IsExport (TheProcedure, TheExport);

   procedure AddDependency (Abstraction     : in Abstractions;
                            TheProcedure    : in Symbol;
                            TheExport       : in Symbol;
                            TheImport       : in Symbol;
                            ImportReference : in Location);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Abstraction,
   --#                                TheExport,
   --#                                TheImport,
   --#                                TheProcedure &
   --#         SPARK_IO.File_Sys from *,
   --#                                Abstraction,
   --#                                Dict,
   --#                                ImportReference,
   --#                                LexTokenManager.StringTable,
   --#                                TheExport,
   --#                                TheImport,
   --#                                TheProcedure;
   -- pre  IsExport (TheProcedure, TheExport) and
   --      (IsGlobalVariable (TheProcedure, TheImport) or
   --       TheProcedure = GetSubprogram (TheImport));
   -- post DependencyExists (TheProcedure, TheExport, TheImport);

   procedure AddVirtualElement (ToProtectedType   : in Symbol;
                                TheVirtualElement : in Symbol;
                                TheOwner          : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   TheOwner,
   --#                   TheVirtualElement,
   --#                   ToProtectedType;

   procedure AddPOorSOToSuspendsList (TheTaskOrProc : in Symbol;
                                      ThePOorSO     : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   ThePOorSO,
   --#                   TheTaskOrProc;

   procedure AddInterruptStreamMapping
     (Subject             : in     Symbol;
      TheInterruptHandler : in     LexTokenManager.LexString;
      TheInterruptStream  : in     LexTokenManager.LexString);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Subject,
   --#                   TheInterruptHandler,
   --#                   TheInterruptStream;

   procedure CopyDependencyList (Abstraction  : in Abstractions;
                                 TheProcedure : in Symbol;
                                 FromExport   : in Symbol;
                                 ToExport     : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Abstraction,
   --#                   FromExport,
   --#                   TheProcedure,
   --#                   ToExport;
   -- pre  IsExport (Abstraction, TheProcedure, FromExport) and
   --      IsExport (Abstraction, TheProcedure, ToExport);

   procedure ForceImport (Abstraction     : in Abstractions;
                          TheProcedure    : in Symbol;
                          TheImport       : in Symbol;
                          ImportReference : in Location);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Abstraction,
   --#                                TheImport,
   --#                                TheProcedure &
   --#         SPARK_IO.File_Sys from *,
   --#                                Abstraction,
   --#                                Dict,
   --#                                ImportReference,
   --#                                LexTokenManager.StringTable,
   --#                                TheImport,
   --#                                TheProcedure;
   -- pre  IsGlobalVariable (Abstraction, TheProcedure, TheImport) or
   --      TheProcedure = GetSubprogram (TheImport));
   -- post IsImport (Abstraction, TheProcedure, TheImport);

   procedure AddPackage (Name          : in     LexTokenManager.LexString;
                         Specification : in     Location;
                         Scope         : in     Scopes;
                         ThePackage    :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         ThePackage        from Dict,
   --#                                Name,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope,
   --#                                Specification;
   -- pre  not IsReservedWord (Name) and
   --      not IsDirectlyDefined (Name, Scope, ProgramContext);
   -- post ThePackage = LookupItem (Name, Scope, ProgramContext) and
   --      IsPackage (ThePackage);

   procedure AddChildPackage (TheParent     : in     Symbol;
                              Sort          : in     PackageSort;
                              Name          : in     LexTokenManager.LexString;
                              Specification : in     Location;
                              Scope         : in     Scopes;
                              ThePackage    :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Name,
   --#                                Scope,
   --#                                Sort,
   --#                                TheParent &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope,
   --#                                Sort,
   --#                                Specification,
   --#                                TheParent &
   --#         ThePackage        from Dict,
   --#                                Name,
   --#                                Scope,
   --#                                TheParent;
   -- pre  not IsReservedWord (Name) and
   --      IsPackage(TheParent) and
   --      not IsDirectlyDefined (Name, Scope, ProgramContext);
   -- post ThePackage = LookupItem (Name, Scope, ProgramContext) and
   --      IsPackage (ThePackage);

   procedure AddPrivatePackage (Name          : in     LexTokenManager.LexString;
                                Specification : in     Location;
                                Scope         : in     Scopes;
                                ThePackage    :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         ThePackage        from Dict,
   --#                                Name,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Scope,
   --#                                Specification;
   -- pre  not IsReservedWord (Name) and
   --      not IsDirectlyDefined (Name, Scope, ProgramContext);
   -- post ThePackage = LookupItem (Name, Scope, ProgramContext) and
   --      IsPackage (ThePackage);

   procedure AddPrivatePart (ThePackage  : in Symbol;
                             PrivatePart : in Location;
                             Hidden      : in Boolean);
   --# global in     Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                Hidden,
   --#                                LexTokenManager.StringTable,
   --#                                PrivatePart,
   --#                                ThePackage;
   -- pre IsPackage (ThePackage);

   procedure AddOwnAnnotation (ThePackage : in Symbol;
                               Annotation : in Location);
   --# global in     Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Annotation,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                ThePackage;
   -- pre IsPackage (ThePackage);

   procedure AddOwnVariable (Name              : in     LexTokenManager.LexString;
                             ThePackage        : in     Symbol;
                             Mode              : in     Modes;
                             IsProtected       : in     Boolean;
                             IsInterruptStream : in     Boolean;
                             Reference         : in     Location;
                             Variable          :    out Symbol);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                IsInterruptStream,
   --#                                IsProtected,
   --#                                Mode,
   --#                                Name,
   --#                                ThePackage &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                IsInterruptStream,
   --#                                IsProtected,
   --#                                LexTokenManager.StringTable,
   --#                                Mode,
   --#                                Name,
   --#                                Reference,
   --#                                ThePackage &
   --#         Variable          from Dict,
   --#                                Name,
   --#                                ThePackage;
   -- pre  not IsReservedWord (Name) and
   --      not IsDirectlyDefined (Name, ThePackage, ProofContext) and
   --      IsPackage (ThePackage);
   -- post Variable = LookupItem (Name, VisibleScope (ThePackage), ProofContext) and
   --      IsOwnVariable (Variable) and
   --      ThePackage = GetOwner (Variable));

   procedure AddOwnVariableType (OwnVariable   : in Symbol;
                                 TypeMark      : in Symbol;
                                 TypeReference : in Location);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                OwnVariable,
   --#                                TypeMark &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                OwnVariable,
   --#                                TypeMark,
   --#                                TypeReference;
   -- pre  IsOwnVariable (OwnVariable) and IsTypeMark (TypeMark);
   -- post GetType (OwnVariable) = TypeMark;

   procedure AddOwnTask (Name       : in     LexTokenManager.LexString;
                         TypeMark   : in     Symbol;
                         ThePackage : in     Symbol;
                         TaskSym    :    out Symbol);
   --# global in out Dict;
   --# derives Dict    from *,
   --#                      Name,
   --#                      ThePackage,
   --#                      TypeMark &
   --#         TaskSym from Dict;
   -- pre  IsTypeMark (TypeMark);
   -- post GetType (OwnTask) = TypeMark;

   procedure AddRefinementDefinition (ThePackage : in Symbol;
                                      Annotation : in Location);
   --# global in     Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Annotation,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                ThePackage;
   -- pre IsPackage (ThePackage);

   procedure AddConstituent (Name                 : in LexTokenManager.LexString;
                             Subject              : in Symbol;
                             Mode                 : in Modes;
                             SubjectReference     : in Location;
                             ConstituentReference : in Location);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Mode,
   --#                                Name,
   --#                                Subject &
   --#         SPARK_IO.File_Sys from *,
   --#                                ConstituentReference,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Name,
   --#                                Subject,
   --#                                SubjectReference;
   -- pre  not IsReservedWord (Name) and IsOwnVariable (Subject);
   -- post exists (Constituent : Symbol,
   --        Constituent = LookupItem (Name, LocalScope (GetOwner (Subject)), ProofContext) and
   --        IsRefinement (Subject, Constituent));

   procedure AddConstituentSym (ConstituentVariable  : in Symbol;
                                Subject              : in Symbol;
                                SubjectReference     : in Location;
                                ConstituentReference : in Location);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                ConstituentVariable,
   --#                                Subject &
   --#         SPARK_IO.File_Sys from *,
   --#                                ConstituentReference,
   --#                                ConstituentVariable,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Subject,
   --#                                SubjectReference;
   -- pre  IsOwnVariable (ConstituentVariable) and IsOwnVariable (Subject);
   -- post exists (Constituent : Symbol,
   --        Constituent = ConstituentVariable and
   --        IsRefinement (Subject, Constituent));

   procedure AddEmbeddedConstituent (PackageName          : in LexTokenManager.LexString;
                                     VariableName         : in LexTokenManager.LexString;
                                     Subject              : in Symbol;
                                     Mode                 : in Modes;
                                     SubjectReference     : in Location;
                                     ConstituentReference : in Location);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Mode,
   --#                                PackageName,
   --#                                Subject,
   --#                                VariableName &
   --#         SPARK_IO.File_Sys from *,
   --#                                ConstituentReference,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                PackageName,
   --#                                Subject,
   --#                                SubjectReference,
   --#                                VariableName;
   -- pre  not IsReservedWord (PackageName) and
   --      not IsReservedWord (VariableName) and
   --      IsOwnVariable (Subject);
   -- post exists (Constituent : Symbol,
   --        Constituent = LookupSelectedItem (PackageName, VariableName, GetScope (Subject), ProofContext) and
   --        IsRefinement (Subject, Constituent));

   procedure AddChildConstituent (Variable             : in Symbol;
                                  Subject              : in Symbol;
                                  Mode                 : in Modes;
                                  SubjectReference     : in Location;
                                  ConstituentReference : in Location);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Mode,
   --#                                Subject,
   --#                                Variable &
   --#         SPARK_IO.File_Sys from *,
   --#                                ConstituentReference,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Subject,
   --#                                SubjectReference,
   --#                                Variable;
   -- pre  IsOwnVariable (Variable) and
   --      IsOwnVariable (Subject) and
   --      GetPackageOwner(GetOwner(Variable)) = GetOwner(Subject);
   -- post IsRefinement (Subject, Variable);

   procedure AddInitializationSpecification (ThePackage : in Symbol;
                                             Annotation : in Location);
   --# global in     Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Annotation,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                ThePackage;
   -- pre IsPackage (ThePackage);

   procedure AddInitializedOwnVariable (Variable          : in Symbol;
                                        VariableReference : in Location);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Variable &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Variable,
   --#                                VariableReference;
   -- pre  IsOwnVariable (Variable);
   -- post OwnVariableIsInitialized (Variable);

   procedure AddPackageInitialization (ThePackage     : in Symbol;
                                       Initialization : in Location;
                                       Hidden         : in Boolean);
   --# global in     Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                Hidden,
   --#                                Initialization,
   --#                                LexTokenManager.StringTable,
   --#                                ThePackage;
   -- pre IsPackage (ThePackage);

   procedure AddWithClause (Scope  : in Scopes;
                            Clause : in Location);
   --# global in     Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Clause,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Scope;

   procedure AddWithReference (Scope            : in     Scopes;
                               ThePackage       : in     Symbol;
                               Explicit         : in     Boolean;
                               PackageReference : in     Location;
                               AlreadyPresent   :    out Boolean);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives AlreadyPresent,
   --#         Dict              from Dict,
   --#                                Explicit,
   --#                                Scope,
   --#                                ThePackage &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                Explicit,
   --#                                LexTokenManager.StringTable,
   --#                                PackageReference,
   --#                                Scope,
   --#                                ThePackage;
   -- pre  IsPackage (ThePackage);
   -- post IsWithed (ThePackage, Scope);

   procedure AddUseTypeClause (Scope  : in Scopes;
                               Clause : in Location);
   --# global in     Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Clause,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Scope;

   procedure AddUseTypeReference (Scope         : in Scopes;
                                  TheType       : in Symbol;
                                  TypeReference : in Location);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Scope,
   --#                                TheType &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Scope,
   --#                                TheType,
   --#                                TypeReference;
   -- pre  IsTypeMark (TheType);
   -- post IsUsedLocally (TheType, Scope);

   procedure AddInheritsAnnotation (CompilationUnit : in Symbol;
                                    Annotation      : in Location);
   --# global in     Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Annotation,
   --#                                CompilationUnit,
   --#                                Dict,
   --#                                LexTokenManager.StringTable;
   -- pre IsCompilationUnit (CompilationUnit);

   procedure AddInheritsReference (CompilationUnit  : in     Symbol;
                                   ThePackage       : in     Symbol;
                                   Explicit         : in     Boolean;
                                   PackageReference : in     Location;
                                   AlreadyPresent   :    out Boolean);
   --# global in     LexTokenManager.StringTable;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives AlreadyPresent,
   --#         Dict              from CompilationUnit,
   --#                                Dict,
   --#                                Explicit,
   --#                                ThePackage &
   --#         SPARK_IO.File_Sys from *,
   --#                                CompilationUnit,
   --#                                Dict,
   --#                                Explicit,
   --#                                LexTokenManager.StringTable,
   --#                                PackageReference,
   --#                                ThePackage;
   -- pre  IsCompilationUnit (CompilationUnit) and
   --      IsPackage (ThePackage);
   -- post IsInherited (ThePackage, CompilationUnit);

   procedure SetPackageIsGeneric (ThePackage : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   ThePackage;

   procedure SetPackageElaborateBodyFound (ThePackage : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   ThePackage;
   pragma Inline (SetPackageElaborateBodyFound);

   procedure SetPackageAsExtendingAnother (ThePackage          : in Symbol;
                                           ThePackageItExtends : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   ThePackage,
   --#                   ThePackageItExtends;
   pragma Inline (SetPackageAsExtendingAnother);

   procedure AddWriteReference (Variable, CompilationUnit : in Symbol;
                                Reference                 : in Location);
   --# global in     Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CompilationUnit,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Reference,
   --#                                Variable;
   -- pre IsVariable (Variable) and IsCompilationUnit (CompilationUnit);

   procedure AddReadReference (Object, CompilationUnit : in Symbol;
                               Reference               : in Location);
   --# global in     Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CompilationUnit,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Object,
   --#                                Reference;
   -- pre IsObject (Object) and IsCompilationUnit (CompilationUnit);

   procedure AddSubprogramCall (Subprogram, CompilationUnit : in Symbol;
                                Call                        : in Location);
   --# global in     Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Call,
   --#                                CompilationUnit,
   --#                                Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Subprogram;
   -- pre IsSubprogram (Subprogram) and IsCompilationUnit (CompilationUnit);

   procedure AddOtherReference (Item, CompilationUnit : in Symbol;
                                Reference             : in Location);
   --# global in     Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CompilationUnit,
   --#                                Dict,
   --#                                Item,
   --#                                LexTokenManager.StringTable,
   --#                                Reference;
   -- pre not IsUnknownTypeMark (Item) and
   --     IsCompilationUnit (CompilationUnit);

   procedure AddUsesUncheckedConversion (TheUnit : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   TheUnit;
   -- pre IsCompilationUnit (TheUnit) or IsTaskType (TheUnit);

   procedure AddAssignsFromExternal (TheUnit : in Symbol);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   TheUnit;
   -- pre IsCompilationUnit (TheUnit) or IsTaskType (TheUnit);

   --------------------------------------------------------------------------------
   --                             ACCESS FUNCTIONS                               --
   --------------------------------------------------------------------------------

   function SymbolRef (Item : Symbol) return ExaminerConstants.RefType;

   function ConvertSymbolRef (Ref : ExaminerConstants.RefType) return Symbol;

   function ConvertScope (S : Scopes) return ScopeReference;

   function ConvertScopeReference (SR : ScopeReference) return Scopes;

   function GetSimpleName (Item : Symbol) return LexTokenManager.LexString;
   --# global in Dict;
   -- pre Item = NullSymbol or (not IsUnknownTypeMark (Item));

   function GetType (TheSymbol : Symbol) return Symbol;
   --# global in Dict;
   -- pre IsObject (TheSymbol) or
   --     IsFunction (TheSymbol) or
   --     IsEnumerationLiteral (TheSymbol) or
   --     IsRecordComponent (TheSymbol) or
   --     IsSubtype (TheSymbol);

   -- There is no need for a GetUnaryOperatorType function since the result type
   -- of every SPARK unary operator is the same as the operand type

   function GetAccess (TheProtectedType : Symbol) return Symbol;
   --# global in Dict;

   function DeReference (TheAccessType : Symbol) return Symbol;
   --# global in Dict;

   function GetBinaryOperatorType (Name  : SPSymbols.SPSymbol;
                                   Left  : Symbol;
                                   Right : Symbol) return Symbol;
   --# global in CommandLineData.Content;
   --#        in Dict;
   -- pre BinaryOperatorIsDefined (Name, Left, Right);

   function GetScalarAttributeType (Name     : LexTokenManager.LexString;
                                    TypeMark : Symbol) return Symbol;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark);

   function GetArrayAttributeType (Name      : LexTokenManager.LexString;
                                   TypeMark  : Symbol;
                                   Dimension : Positive) return Symbol;
   --# global in Dict;
   -- pre IsArrayAttribute(Name, TypeMark) and
   --     IsTypeMark (TypeMark) and
   --     Dimension <= GetNumberofDimensions (TypeMark);

   function GetScalarAttributeValue (Base     : Boolean;
                                     Name     : LexTokenManager.LexString;
                                     TypeMark : Symbol) return LexTokenManager.LexString;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark);

   function GetArrayAttributeValue (Name      : LexTokenManager.LexString;
                                    TypeMark  : Symbol;
                                    Dimension : Positive) return LexTokenManager.LexString;
   --# global in Dict;
   -- pre IsArrayAttribute(Name, TypeMark) and
   --     IsTypeMark (TypeMark) and
   --     Dimension <= GetNumberofDimensions (TypeMark);

   function GetEnumerationLiteral (EnumerationType : Symbol;
                                   Position        : LexTokenManager.LexString) return Symbol;
   --# global in Dict;
   -- pre IsEnumerationType (EnumerationType);

   function GetPositionNumber (Literal : Symbol) return LexTokenManager.LexString;
   --# global in Dict;
   -- pre IsEnumerationLiteral (Literal);

   function GetRecordType (Component : Symbol) return Symbol;
   --# global in Dict;
   -- pre IsRecordComponent (Component);

   -- Predefined types -------------------------------------------------------------

   function GetUnknownTypeMark return Symbol;
   --# global in Dict;

   function GetUniversalIntegerType return Symbol;
   --# global in Dict;

   function GetUniversalRealType return Symbol;
   --# global in Dict;

   function GetUniversalFixedType return Symbol;
   --# global in Dict;

   function GetPredefinedBooleanType return Symbol;
   --# global in Dict;

   -- The NullVariable is an out stream used as a data sink for null derives etc.
   function GetNullVariable return Symbol;
   --# global in Dict;

   -- ThePartition is a subprogram symbol (predefined when Ravenscar profile is selected)
   -- that is used as place with which to associate the partition annotation.
   function GetThePartition return Symbol;
   --# global in Dict;

   function GetFalse return Symbol;
   --# global in Dict;

   function GetTrue return Symbol;
   --# global in Dict;

   function GetPredefinedIntegerType return Symbol;
   --# global in Dict;

   function GetPredefinedLongIntegerType return Symbol;
   --# global in Dict;

   function GetPredefinedFloatType return Symbol;
   --# global in Dict;

   function GetPredefinedLongFloatType return Symbol;
   --# global in Dict;

   function GetPredefinedCharacterType return Symbol;
   --# global in Dict;

   function GetPredefinedStringType return Symbol;
   --# global in Dict;

   function GetPredefinedNaturalSubtype return Symbol;
   --# global in Dict;

   function GetPredefinedPositiveSubtype return Symbol;
   --# global in Dict;

   function GetPredefinedTimeType return Symbol;
   --# global in Dict;
   -- pre CommandLineData.RavenscarSelected;

   function GetPredefinedTimeSpanType return Symbol;
   --# global in Dict;

   function GetPredefinedSuspensionObjectType return Symbol;
   --# global in Dict;
   -- pre CommandLineData.RavenscarSelected;

   -- Note: following function is to with getting the true base type
   --       of a type which is derived (from say root_integer).
   --       Use GetRootType to get the "first named subtype" of a subtype
   function GetBaseType (TypeMark : Symbol) return Symbol;
   --# global in Dict;
   --  pre IsType (TypeMark);

   function GetRootOfExtendedType (TypeMark : Symbol) return Symbol;
   --# global in Dict;
   -- Applies to types extended from tagged types.  Gives the immediately
   -- preceding ancestor type

   -- The 'root' type is (in Ada terms) the first subtype of the
   -- anonymous type chosen by the compiler when a signed integer
   -- or floating point type is declared
   function GetRootType (TypeMark : Symbol) return Symbol;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark);

   function GetNumberOfDimensions (TypeMark : Symbol) return Positive;
   --# global in Dict;
   -- pre IsArrayTypeMark (TypeMark, GetScope (TypeMark));

   function GetArrayIndex (TypeMark : Symbol; Dimension : Positive) return Symbol;
   --# global in Dict;
   -- pre IsArrayTypeMark (TypeMark, GetScope (TypeMark) and
   --     Dimension <= GetNumberOfDimensions (TypeMark);

   function GetArrayComponent (TypeMark : Symbol) return Symbol;
   --# global in Dict;
   -- pre IsArrayTypeMark (TypeMark, GetScope (TypeMark));

   function GetNumberOfNonExtendedComponents (TheRecordType : Symbol) return Natural;
   --# global in Dict;
   -- pre IsRecordTypeMark (TheRecordType, GetScope (TheRecordType));
   -- Returns the number of record components in a record but does nor traverse a group of
   -- records modelling an extended tagged record.  The count includes the Inherit field of
   -- an extended record.

   function GetNumberOfActualComponents (TheRecordType : Symbol) return Natural;
   --# global in Dict;
   -- pre IsRecordTypeMark (TheRecordType, GetScope (TheRecordType));
   -- As above but ignore Inherit fields of extended tagged records

   function GetNumberOfComponents (TheRecordType : Symbol) return Natural;
   --# global in Dict;
   -- pre IsRecordTypeMark (TheRecordType, GetScope (TheRecordType));
   -- if the type is extended we get all the fields whether locally declared
   -- or inherited

   function GetNonExtendedRecordComponent (TheRecordType : Symbol;
                                           Number        : Positive) return Symbol;
   --# global in Dict;
   -- pre IsRecordTypeMark (TheRecordType, GetScope (TheRecordType));
   -- Returns field N of a single record (i.e ignores structured composition of records used to
   -- model tagged extended records.  If record is an extension, it cann return the "Inherit" field

   function GetRecordComponent (TheRecordType : Symbol;
                                Number        : Positive) return Symbol;
   --# global in Dict;
   -- pre Number <= GetNumberOfRecordComponents (TheRecordType);
   -- Works for extended records as well as non-extended ones; does not ever return Inherit fields

   function GetInheritDepth (FieldName    : LexTokenManager.LexString;
                             RecordType   : Symbol) return Natural;
   --# global in Dict;
   -- Assuming we have used LookUpSelectedItem and established that there is a field
   -- called FieldName in record RecordType we can use this function to find out
   -- how far FieldName is down a chain of inherit-from-root-type pointers.  So if
   -- R.F directly exists we return 0, and if R.F actually represents R.Inherit.F then we get
   -- 1 and so on

   function RecordComponentIsInherited (TheComponent : Symbol) return Boolean;
   --# global in Dict;
   --  pre IsRecordComponent (TheComponent);
   -- returns true for extended types where a field is inherited rather than
   -- explicitly declared

   function GetVariableExpNode (Variable : Symbol) return ExaminerConstants.RefType;
   --# global in Dict;
   -- pre IsEntireVariable (Variable);

   function GetConstantExpNode (TheConstant : Symbol) return ExaminerConstants.RefType;
   --# global in Dict;
   -- pre IsConstant (TheConstant);

   function ConstantExpIsWellFormed (TheConstant : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsConstant (TheConstant);

   function GetValue (TheConstant : Symbol) return LexTokenManager.LexString;
   --# global in Dict;
   -- pre IsConstant (TheConstant);

   function GetConstantRulePolicy (TheConstant : Symbol;
                                   TheScope    : Scopes) return RulePolicies;
   --# global in Dict;
   -- pre IsConstant (TheConstant) and
   --     (IsArrayType ( GetType (TheConstant) or IsRecordType (GetType (TheConstant)));

   function IsConstantRulePolicyPresent (TheConstant : Symbol;
                                         TheScope    : Scopes) return Boolean;
   --# global in Dict;
   -- pre IsConstant (TheConstant) and
   --     (IsArrayType ( GetType (TheConstant) or IsRecordType (GetType (TheConstant)));

   function TypeIsWellformed (TheType : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TheType);

   function GetPredefinedPackageStandard return Symbol;
   --# global in Dict;

   function GetPredefinedPackageASCII return Symbol;
   --# global in Dict;

   function GetOwner (Variable : Symbol) return Symbol;
   --# global in Dict;
   -- pre IsOwnVariable (Variable) or IsConstituent (Variable);

   function LastMostEnclosingLoop (CompilationUnit : Symbol) return Symbol;
   --# global in Dict;
   -- pre IsCompilationUnit (CompilationUnit)

   function GetLoop (CompilationUnit : Symbol; Number : Positive) return Symbol;
   --# global in Dict;
   -- pre IsCompilationUnit (CompilationUnit) and
   --     Number <= GetNumberOfLoops (CompilationUnit);

   function GetLoopParameter (TheLoop : Symbol) return Symbol;
   --# global in Dict;
   -- pre IsForLoop (TheLoop);

   -- Returns a Cells.Cell previously planted by the VCG and representing
   -- the exit expression of a for loop
   function GetLoopExitExpn (TheLoop : Symbol) return Natural;
   --# global in Dict;
   -- pre IsForLoop (TheLoop);

   function GetLoopHasExits (TheLoop : Symbol) return Boolean;
   --# global in Dict;
   -- pre  IsLoop (TheLoop);

   function LoopParameterHasStaticRange (TheLoopParameter : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsLoopParameter (TheLoopParameter);

   function LoopParameterMovesInReverse (TheLoopParameter : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsLoopParameter (TheLoopParameter);

   function GetScope (Item : Symbol) return Scopes;
   --# global in Dict;
   -- pre not IsUnknownTypeMark (Item);

   function GetEnclosingScope (Scope : Scopes) return Scopes;
   --# global in Dict;

   function GetEnclosingPackage (Scope : Scopes) return Symbol;
   --# global in Dict;

   function GetEnclosingProtectedRegion (Scope : Scopes) return Symbol;
   --# global in Dict;

   function IsOrIsInProtectedScope (Scope : Scopes) return Boolean;
   --# global in Dict;

   function SubprogramSignatureIsWellformed (Abstraction : Abstractions;
                                             Subprogram  : Symbol) return Boolean;
   --# global in Dict;
   --  pre IsSubprogram (Subprogram) or IsTaskType (Subprogram);

   function HasPrecondition (Abstraction : Abstractions;
                             Subprogram  : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsSubprogram (Subprogram) or IsTaskType (Subprogram);
   -- NOTE: a task type is allowed but always returns False since tasks don't have preconditions

   function GetPrecondition (Abstraction : Abstractions;
                             Subprogram  : Symbol) return ExaminerConstants.RefType;
   --# global in Dict;
   -- pre IsSubprogram (Subprogram) or IsTaskType (Subprogram);
   -- NOTE: a task type is allowed but always returns 0 since tasks don't have preconditions

   function HasPostcondition (Abstraction     : Abstractions;
                              CompilationUnit : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsCompilationUnit (CompilationUnit)  or IsTaskType (CompilationUnit);
   -- NOTE: a task type is allowed but always returns 0 since tasks don't have postconditions

   function GetPostcondition (Abstraction     : Abstractions;
                              CompilationUnit : Symbol) return ExaminerConstants.RefType;
   --# global in Dict;
   -- pre IsCompilationUnit (CompilationUnit);

   function HasImplicitReturnVariable (Abstraction : Abstractions;
                                       TheFunction : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsFunction (TheFunction);

   function GetImplicitReturnVariable (Abstraction : Abstractions;
                                       TheFunction : Symbol) return Symbol;
   --# global in Dict;
   -- pre HasImplicitReturnVariable (Abstraction, TheFunction);

   function GetAdaFunction (ProofFunction : Symbol) return Symbol;
   --# global in Dict;
   -- pre IsImplicitProofFunction (ProofFunction);

   function GetImplicitProofFunction (Abstraction : Abstractions;
                                      TheFunction : Symbol) return Symbol;
   --# global in Dict;
   -- pre IsAdaFunction (TheFunction);

   function GetSubprogramParameterNumber (Parameter : Symbol) return Positive;
   --# global in Dict;
   -- pre IsSubprogramParameter (Parameter);

   function GetSubprogramParameterMode (Parameter : Symbol) return Modes;
   --# global in Dict;
   -- pre IsSubprogramParameter (Parameter);

   function GetSubprogramParameterConstraint (Parameter : Symbol;
                                              Dimension : Positive) return Symbol;
   --# global in Dict;
   -- pre IsSubprogramParameter (Parameter) and
   --     IsUnconstrainedArrayType (GetType (Parameter));
   -- return M => IsSubprogramParameterConstraint (M) and
   --             GetParameterAssociatedWithParameterConstraint (M) = Parameter and
   --             GetSubprogramParameterConstraintDimension (M) = Dimension;

   -- Returns the symbol for x__index_subtype__n where x is the paraeter and n the dimension number. This is
   -- a symbol with discriminant ParameterConstraintSymbol and which is used to pass information about
   -- unconstrained parameters between the wffs and the VCG.  Pseudo annotation describes normal behaviour; an
   -- illegal SPARK program might result in calls being made which violate the "precondition", in which case
   -- the unknown type mark is returned.

   function GetSubprogramParameterConstraintDimension (TheConstraint : Symbol) return Positive;
   --# global in Dict;
   -- pre IsSubprogramParameterConstraint;

   function GetParameterAssociatedWithParameterConstraint (TheConstraint : Symbol) return Symbol;
   --# global in Dict;
   -- pre IsSubprogramParameterConstraint;

   function GetNumberOfSubprogramParameters (Subprogram : Symbol) return Natural;
   --# global in Dict;
   -- pre IsSubprogram (Subprogram) or IsTaskType (Subprogram);

   function GetSubprogramParameter (Subprogram : Symbol;
                                    Number     : Positive) return Symbol;
   --# global in Dict;
   -- pre IsSubprogram (Subprogram) and
   --     Number <= GetNumberOfSubprogramParameters (Subprogram);

   function GetNumberOfGlobalVariables (Abstraction : Abstractions;
                                        Subprogram  : Symbol) return Natural;
   --# global in Dict;
   -- pre IsSubprogram (Subprogram) or IsTaskType (SubprogSym);

   function GetGlobalMode (Abstraction : Abstractions;
                           Subprogram  : Symbol;
                           Variable    : Symbol) return Modes;
   --# global in Dict;
   --  pre IsSubprogram(Subprogram) and IsGlobalVariable(Variable);

   function GetOwnVariableMode (Variable : Symbol) return Modes;
   --# global in Dict;
   --  pre    IsOwnVariable (Variable);

   function GetOwnVariableProtected (Variable : Symbol) return Boolean;
   --# global in Dict;
   --  pre    IsOwnVariable (Variable);

   function GetOwnVariableIsInterruptStream (Variable : Symbol) return Boolean;
   --# global in Dict;
   -- pre     IsOwnVariable (Variable);

   function GetConstituentMode (Variable : Symbol) return Modes;
   --# global in Dict;
   --  pre    IsConstituent (Variable);

   function IsOwnVariableOrConstituentWithMode (Variable : Symbol) return Boolean;
   --# global in Dict;

   function GetOwnVariableOrConstituentMode (Variable : Symbol) return Modes;
   --# global in Dict;
   --  return M => (IsOwnVariable (Variable) and M = GetOwnVariableMode (Variable)) or
   --              (IsConstituent (Variable) and M = GetConstituentMode (Variable)) or
   --              (not IsConstituent (Variable) and not IsOwnVariable (Variable) and
   --               M = DefaultMode);

   function GetProtectedTypeOwnVariable (TheProtectedType : Symbol) return Symbol;
   --# global in Dict;
   --  pre IsType (TheProtectedType) and TypeIsProtected (TheProtectedType);
   --  return "the implicitly-declared abstract own variable that belongs to the PT"

   function IsUnmodedProtectedOwnVariable (Sym : Symbol) return Boolean;
   --# global in Dict;

   function GetProtectedImplicitInStream (TheProtectedOwnVar : Symbol) return Symbol;
   --# global in Dict;
   --  pre IsOwnVar (TheProtectedOwnVar) and
   --      GetOwnVariableProtected (TheProtectedOwnVar) and
   --      GetOwnVariableMode (TheProtectedOwnVar) = DefaultMode;
   -- Failure to observe precondtion will result in a null symbol being returned or dict crash.

   function IsProtectedImplicitInStream (Sym : Symbol) return Boolean;
   --# global in Dict;

   function GetProtectedTypeElementsHidden (TheProtectedType : in Symbol) return Boolean;
   --# global in Dict;
   --  pre IsType (TheProtectedType) and TypeIsProtected (TheProtectedType);

   function GetProtectedTypeHasEntry (TheProtectedType : Symbol) return Boolean;
   --# global in Dict;
   --  pre IsType (TheProtectedType) and TypeIsProtected (TheProtectedType);

   function GetTypeHasPragma (TheProtectedOrTaskType : in Symbol;
                              ThePragma              : in RavenscarPragmas) return Boolean;
   --# global in Dict;

   function GetTypePragmaValue (TheProtectedOrTaskType : in Symbol;
                                ThePragma              : in RavenscarPragmasWithValue)
                               return LexTokenManager.LexString;
   --# global in Dict;

   function GetTypePriority (TheProtectedOrTaskType : in Symbol)
                            return LexTokenManager.LexString;
   --# global in Dict;

   function GetActual (TheFormal  : Symbol;
                       TheSubtype : Symbol) return Symbol;
   --# global in Dict;
   -- pre IsKnownDiscriminant (TheFormal) and IsSubtype (TheSubtype);

   function GetPriorityProperty (OwnVariable : in Symbol)
                                return LexTokenManager.LexString;
   --# global in Dict;
   -- pre IsOwnVariable (OwnVariable);

   function GetIntegrityProperty (S : in Symbol)
                                return LexTokenManager.LexString;
   --# global in Dict;

   -- Implements information flow policy checking based on the Integrity
   -- property of the given import and export symbol.  CommandLineData is
   -- used to determine which info flow policy (if any) if being checked.
   function RelationViolatesInfoFlowPolicy (TheExport : in Symbol;
                                            TheImport : in Symbol) return Boolean;
   --# global in CommandLineData.Content;
   --#        in Dict;
   --#        in LexTokenManager.StringTable;

   function GetIsSuspendable (Variable : in Symbol) return Boolean;
   --# global in Dict;
   -- pre IsOwnVariable (Variable);

   function GetHasInterruptProperty (Variable : in Symbol) return Boolean;
   --# global in Dict;
   -- pre IsOwnVariable (Variable);

   function GetVirtualElementOwner (Variable : in Symbol) return Symbol;
   --# global in Dict;
   -- pre IsOwnVariable (Variable);

   function IsVirtualElement (Variable : in Symbol) return Boolean;
   --# global in Dict;
   -- pre IsOwnVariable (Variable);

   function IsVirtualElementForType (TheVariable      : in Symbol;
                                     TheProtectedType : in Symbol) return Boolean;
   --# global in Dict;
   -- pre IsOwnVariable (Variable) and
   --     IsType (TheProtectedType) and TypeIsProtected (TheProtectedType);

   function VirtualElementSeenByOwner (Variable : in Symbol) return Boolean;
   --# global in Dict;
   -- pre IsOwnVariable (Variable);

   function GetMainProgram return Symbol;
   --# global in Dict;
   -- pre MainProgramExists;

   function GetMainProgramPriority return LexTokenManager.LexString;
   --# global in Dict;
   -- pre MainProgramExists and MainProgramPriorityExists;

   function GetAbstraction (Subprogram : Symbol;
                            Scope      : Scopes) return Abstractions;
   --# global in Dict;

   function GetConstraintAbstraction (Subprogram : Symbol;
                                      Scope      : Scopes) return Abstractions;
   --# global in Dict;
   -- pre IsSubprogram (Subprogram);
   -- Similar above but selects which subprgoram constraint to use.  May return different
   -- result to GetAbstraction because a subprogram may have only one flow annotation
   -- but have two proof contexts (if private type refinement is involved).

   procedure AdjustTypeUpperBound (TypeMark : Symbol;
                                   NewBound : LexTokenManager.LexString);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   NewBound,
   --#                   TypeMark;
   -- pre TypeIsNumeric (TypeMark) and IsType (TypeMark)

   procedure AdjustTypeLowerBound (TypeMark : Symbol;
                                   NewBound : LexTokenManager.LexString);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   NewBound,
   --#                   TypeMark;
   -- pre TypeIsNumeric (TypeMark) and IsType (TypeMark)

   procedure AdjustTypeErrorBound (TypeMark : Symbol;
                                   NewBound : LexTokenManager.LexString);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   NewBound,
   --#                   TypeMark;
   -- pre TypeIsFloatingPoint (TypeMark) and IsType (TypeMark)

   -- **********************************************************************
   -- NB: the three above operations are not intended for general use,
   --     and are present for the benefit of the ConfigFile package.
   -- **********************************************************************

   --------------------------------------------------------------------------------
   --                              QUERY FUNCTIONS                               --
   --------------------------------------------------------------------------------

   function IsObject (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsTypeMark (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsDeclared (Item : Symbol) return Boolean;
   --# global in Dict;

   function IsConstant (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsVariable (TheSymbol : Symbol) return Boolean;
   --# global in Dict;
   -- Note, returns false for a record subcomponent

   function IsVariableOrSubcomponent (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function GetFirstRecordSubcomponent (TheSymbol : Symbol) return Symbol;
   --# global in Dict;
   --  pre IsVariableOrSubcomponent (TheSymbol);

   function GetNextRecordSubcomponent (TheSubcomponent : Symbol) return Symbol;
   --# global in Dict;
   --  pre IsSubcomponent (TheSubcomponent);

   function VariableIsInitialized (Variable : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsEntireVariable (Variable);

   function VariableHasAddressClause (Variable : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsEntireVariable (Variable);

   function VariableHasPragmaImport (Variable : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsEntireVariable (Variable);

   function TypeSizeAttribute (TypeMark : Symbol) return LexTokenManager.LexString;
   --# global in Dict;
   -- pre IsType (TypeMark);

   function VariableIsAliased (Variable : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsEntireVariable (Variable);

   procedure SetVariableMarkedValid (Variable   : in Symbol; Val : in Boolean);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Val,
   --#                   Variable;

   function VariableIsMarkedValid (TheVariable : Symbol) return Boolean;
   --# global in Dict;

   procedure SetSubcomponentMarkedValid (Subcomponent  : in Symbol; Val : in Boolean);
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Subcomponent,
   --#                   Val;

   function SubcomponentIsMarkedValid (TheSubcomponent : Symbol) return Boolean;
   --# global in Dict;

   function VariableOrSubcomponentIsMarkedValid (TheSym : Symbol) return Boolean;
   --# global in Dict;

   function IsAtomic (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsPragmaAtomic (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsDeferredConstant (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function ConstantIsDeferredHere (TheConstant : Symbol;
                                    Scope       : Scopes) return Boolean;
   --# global in Dict;

   function IsUnknownTypeMark (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsUniversalIntegerType (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsUniversalRealType (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   -- True iff IsTypeMark (TheSymbol) and ISN'T UnknownType and
   -- ISN'T a subtype
   function IsType (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsSubtype (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsTaskType (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsProtectedType (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsProtectedFunction (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function CompatibleTypes (Scope : Scopes;
                             Left  : Symbol;
                             Right : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (Left) and IsTypeMark (Right);

   function IsScalarTypeMark (TheSymbol : Symbol;
                              Scope     : Scopes) return Boolean;
   --# global in Dict;

   function IsDiscreteTypeMark (TheSymbol : Symbol;
                                Scope     : Scopes) return Boolean;
   --# global in Dict;

   function IsBooleanTypeMark (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsIntegerTypeMark (TheSymbol : Symbol;
                               Scope     : Scopes) return Boolean;
   --# global in Dict;

   function IsModularTypeMark (TheSymbol : Symbol;
                               Scope     : Scopes) return Boolean;
   --# global in Dict;

   function IsRealTypeMark (TheSymbol : Symbol;
                            Scope     : Scopes) return Boolean;
   --# global in Dict;

   function IsFixedPointTypeMark (TheSymbol : Symbol;
                                  Scope     : Scopes) return Boolean;
   --# global in Dict;

   function IsFloatingPointTypeMark (TheSymbol : Symbol;
                                     Scope     : Scopes) return Boolean;
   --# global in Dict;

   function IsArrayTypeMark (TheSymbol : Symbol;
                             Scope     : Scopes) return Boolean;
   --# global in Dict;

   function IsRecordTypeMark (TheSymbol : Symbol;
                              Scope     : Scopes) return Boolean;
   --# global in Dict;

   function TypeIsBoolean (TypeMark : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark);

   function TypeIsCharacter (TypeMark : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark);

   function TypeIsEnumeration (TypeMark : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark);

   function TypeIsInteger (TypeMark : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark);

   function TypeIsModular (TypeMark : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark);

   function TypeIsDiscrete (TypeMark : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark);

   function TypeIsFixedPoint (TypeMark : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark);

   function TypeIsFloatingPoint (TypeMark : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark);

   function TypeIsReal (TypeMark : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark);

   function TypeIsNumeric (TypeMark : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark);

   function TypeIsScalar (TypeMark : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark);

   function TypeIsRecord (TypeMark : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark);

   function TypeIsAbstractProof (TypeMark : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark);

   function TypeIsArray (TypeMark : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark);

   function TypeIsAnnounced (TheType : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TheType);

   function TypeIsPrivate (TheType : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsType (TheType);

   function TypeIsTagged (TheType : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsType (TheType);

   function TypeIsExtendedTagged (TheType : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsType (TheType);

   function TypeIsProtected (TheType : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsType (TheType);

   function TypeIsTask (TheType : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsType (TheType);

   function TypeIsAccess (TheType : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsType (TheType);

   function TypeIsGeneric (TheType : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsType (TheType);

   function TypeIsGenericArray (TheType : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsType (TheType);

   function ExtendedTaggedHasPrivateAncestors (TheType : Symbol;
                                               Scope   : Scopes) return Boolean;
   --# global in Dict;
   -- pre IsType (TheType) and TypeIsExtendedTagged (TheType);

   function IsAnExtensionOf (RootType, ExtendedType : Symbol) return Boolean;
   --# global in Dict;

   function NoFieldsBelowThisRecord (RecordSym : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsType (RecordSym) and TypeIsRecord (RecordSym);
   -- Returns true for an extended record if all its inherited records are null
   -- extensions or null records

   function RecordHasSomeFields (RecordSym : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsType (RecordSym) and TypeIsRecord (RecordSym);
   -- return true if a record either has soem fields itself or it inherits some from an
   -- ancestor type

   function TypeIsAbstract (TheType : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsType (TheType);

   function ContainsPrivate (TypeMark : Symbol; Scope : Scopes) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark) and IsScope (Scope);

   function ContainsFloat (TypeMark : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark);

   function EqualityDefined (TypeMark : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark);

   function GetEnclosingCompilationUnit (Scope : Scopes) return Symbol;
   --# global in Dict;
   -- pre IsLocalScope (Scope);

   function IsLoop (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function LoopHasName (TheLoop : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsLoop (TheLoop);

   function GetLoopNumber (TheLoop : Symbol) return Positive;
   --# global in Dict;
   -- pre IsLoop (TheLoop);

   function IsLoopParameter (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsPrivateType (TheSymbol : Symbol; Scope : Scopes) return Boolean;
   --# global in Dict;

   -- Returns True iff the given Type is INCOMPLETE
   -- the point of view of the given Scope.
   function TypeIsIncompleteHere (TypeMark : Symbol; Scope : Scopes) return Boolean;
   --# global in Dict;

   function IsLimitedPrivateType (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function TypeIsLimited (TypeMark : Symbol;
                           Scope    : Scopes) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark);

   function IsStatic (Item  : Symbol;
                      Scope : Scopes) return Boolean;
   --# global in Dict;
   -- pre IsConstant (Item) or IsTypeMark (Item);

   function IsScalarType (TheSymbol : Symbol;
                          Scope     : Scopes) return Boolean;
   --# global in Dict;

   function IsNumericType (TheSymbol : Symbol;
                           Scope     : Scopes) return Boolean;
   --# global in Dict;

   function IsEnumerationLiteral (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsIntegerType (TheSymbol : Symbol;
                           Scope     : Scopes) return Boolean;
   --# global in Dict;

   function IsModularType (TheSymbol : Symbol;
                           Scope     : Scopes) return Boolean;
   --# global in Dict;

   function IsFixedPointType (TheSymbol : Symbol;
                              Scope     : Scopes) return Boolean;
   --# global in Dict;

   function IsFloatingPointType (TheSymbol : Symbol;
                                 Scope     : Scopes) return Boolean;
   --# global in Dict;

   function IsConstrainedArrayType (TheSymbol : Symbol;
                                    Scope     : Scopes) return Boolean;
   --# global in Dict;

   function IsUnconstrainedArrayType (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsUnconstrainedTaskType (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsUnconstrainedProtectedType (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsRecordComponent (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsRecordSubcomponent (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function GetEnclosingObject (Object : Symbol) return Symbol;
   --# global in Dict;
   -- pre IsObject (Object);

   function GetMostEnclosingObject (Object : Symbol) return Symbol;
   --# global in Dict;
   -- pre IsObject (Object);

   function DefinedInPackageStandard (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsPredefined (TheSymbol : Symbol) return Boolean;
   --# global in CommandLineData.Content;
   --#        in Dict;
   -- Returns true if the scope of the symbol is the visible scope of any of the
   -- following predefined packages:
   -- STANDARD
   -- ASCII
   -- Ada                           -- Ada95
   -- Ada.Characters                -- Ada95
   -- Ada.Characters.Latin_1        -- Ada95
   -- Ada.Real_Time                 -- Ravenscar
   -- Ada.Synchronous_Task_Control, -- Ravenscar
   -- Ada.Interrupts                -- Ravenscar

   function IsPredefinedIntegerType (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsPredefinedFloatType (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsPredefinedCharacterType (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsPredefinedStringType (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsPredefinedTimeType (TheSymbol : Symbol) return Boolean;
   --# global in Dict;
   -- pre CommandLineData.RavenscarSelected;

   function IsPredefinedSuspensionObjectType (TheSymbol : Symbol) return Boolean;
   --# global in CommandLineData.Content;
   --#        in Dict;
   -- pre CommandLineData.RavenscarSelected;

   function IsPredefinedSuspendUntilTrueOperation (TheProcedure : Symbol) return Boolean;
   --# global in Dict;

   function IsPredefinedRealTimeClockOperation (TheProcedure : Symbol) return Boolean;
   --# global in Dict;

   function IsCompilationUnit (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsWithed (ThePackage : Symbol;
                      Scope      : Scopes) return Boolean;
   --# global in Dict;
   -- pre IsPackage (ThePackage);

   function IsWithedLocally (ThePackage : Symbol;
                             Scope      : Scopes) return Boolean;
   --# global in Dict;
   -- pre IsPackage (ThePackage);

   function IsUsedLocally (TheType : Symbol;
                           Scope   : Scopes) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TheType);

   function IsInherited (ThePackage, CompilationUnit : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsPackage (ThePackage) and
   --     IsCompilationUnit (CompilationUnit);

   function IsSubprogram (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsGenericSubprogram (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsFunction (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsEntry (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function GetSubprogramEntryBarrier (Subprogram  : Symbol) return Symbol;
   --# global in Dict;
   -- return Entries Booelan variable barrier symbol if there is one, else null sym

   function IsInterruptHandler (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsImplicitProofFunction (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsAnUncheckedConversion (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function UsesUncheckedConversion (TheUnit : Symbol) return Boolean;
   --# global in Dict;
   --pre IsCompilationUnit (TheUnit) or IsTaskType (TheUnit);

   function AssignsFromExternal (TheUnit : Symbol) return Boolean;
   --# global in Dict;
   --pre IsCompilationUnit (TheUnit) or IsTaskType (TheUnit);

   function IsProcedure (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsProofType (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsProofConstant (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsAdaFunction (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsProofFunction (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsRenamed (Subprogram : Symbol; Scope : Scopes) return Boolean;
   --# global in Dict;
   -- pre IsSubprogram (Subprogram);

   function IsQuantifiedVariable (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsKnownDiscriminant (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function SetsPriority (TheDiscriminant : Symbol) return Boolean;
   --# global in Dict;
   --  pre IsKnownDiscriminant (TheDiscriminant);
   --  return True if the discriminant appears in a priority pragma

   function IsSubprogramParameter (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsSubprogramParameterConstraint (TheSymbol : Symbol) return Boolean;
   --# global in Dict;
   -- A subprgoram parameter constraint is a special symbol associated with unconstrained formal parameters

   function IsFormalParameter (Subprogram, Parameter : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsSubprogram (Subprogram);

   function IsGenericFormalParameter (TheGeneric, Parameter : Symbol) return Boolean;
   --# global in Dict;

   function IsGenericParameterSymbol (Parameter : Symbol) return Boolean;
   --# global in Dict;
   -- Just tells us its a symbol of a generic parameter _somewhere_.  Previous call is more specific

   function HasBody (CompilationUnit : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsCompilationUnit (CompilationUnit);

   function HasBodyStub (CompilationUnit : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsCompilationUnit (CompilationUnit);

   function IsGlobalVariable (Abstraction : Abstractions;
                              Subprogram  : Symbol;
                              Variable    : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsSubprogram (Subprogram) and IsVariable (Variable);

   function IsImportExport (Abstraction  : Abstractions;
                            TheProcedure : Symbol;
                            Variable     : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsProcedure (TheProcedure) and IsVariable (Variable);

   function IsExport (Abstraction  : Abstractions;
                      TheProcedure : Symbol;
                      Variable     : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsProcedure (TheProcedure) and IsVariable (Variable);

   function IsImport (Abstraction  : Abstractions;
                      TheProcedure : Symbol;
                      Variable     : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsProcedure (TheProcedure) and IsVariable (Variable);

   function IsArrayAttribute (Name     : LexTokenManager.LexString;
                              TypeMark : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTypeMark (TypeMark) and
   --     for_some (Scope : Scopes, AttributeIsVisible (Name, TypeMark, Scope));

   function IsPackage (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsGenericPackage (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsEmbeddedPackage (TheSymbol : Symbol) return Boolean;
   --# global in Dict;

   function IsOwnVariable (Variable : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsVariable (Variable);

   function IsOwnTask (Variable : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsVariable (Variable);

   function IsAbstractOwnVariable (Variable : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsVariable (Variable);

   function OwnVariableHasType (OwnVariable : Symbol;
                                Scope       : Scopes) return Boolean;
   --# global in Dict;
   -- pre IsOwnVariable (Variable);

   function OwnVariableIsAnnounced (Variable : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsOwnVariable (Variable);

   function OwnVariableIsInitialized (Variable : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsOwnVariable (Variable);

   function OwnVariableIsAbstract (Variable : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsOwnVariable (Variable);

   function IsConcreteOwnVariable (Variable : Symbol) return Boolean;
   --# global in Dict;

   function OwnVariableIsConcrete (Variable : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsOwnVariable (Variable);

   function IsRefinement (Subject, Constituent : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsOwnVariable (Abstract, Subject) and IsVariable (Constituent);

   function IsRefinementConstituent (ThePackage, Variable : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsPackage (ThePackage) and IsVariable (Variable);

   function IsConstituent (Variable : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsVariable (Variable);

   function GetSubject (Constituent : Symbol) return Symbol;
   --# global in Dict;
   -- pre IsConstituent (Constituent);

   function HasDelayProperty (TheProcedure : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsProcedure (TheProcedure);

   function DelayPropertyIsAccountedFor (TheProcedure : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsProcedure (TheProcedure);

   function HasValidPriorityProperty (OwnVariable : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsOwnVariable (OwnVariable);

   function IsThread (Sym : in Symbol) return Boolean;
   --# global in Dict;

   function UsesUnprotectedVariables (Sym : in Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTaskType (Sym) or IsSubprogram (Sym);

   function GetUnprotectedReference (Variable : in Symbol) return Symbol;
   --# global in Dict;
   -- pre IsOwnVariable (Variable);

   function GetSuspendsReference (Variable : in Symbol) return Symbol;
   --# global in Dict;
   -- pre IsOwnVariable (Variable);

   function SuspendsOn (TheTaskOrProc : Symbol;
                        ThePOorSO     : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTask (TheTaskOrProc) or IsProcedure (TheTaskOrProc);

   function SuspendsListItemIsAccountedFor (TheTaskOrProc : Symbol;
                                            ThePOorSO     : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTaskType (TheTaskOrProc) or IsProcedure (TheTaskOrProc);

   function SuspendsListIsPropagated (FromProcedure : Symbol;
                                      ToTaskOrProc  : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsProcedure (FromProcedure) and
   --     (IsTaskType (ToTaskOrProc) or IsProcedure (ToTaskOrProc));

   function SubprogramMayBlock (Subprogram : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsSubprogram (Subprogram);

   function BodyIsHidden (Sym : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsTaskType (Sym) or IsProcedure (Sym);

   function IsMainProgram (Subprogram : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsSubprogram (Subprogram);
   -- post return true if is the main program or is the partition subprogram

   function IsThePartition (Subprogram : Symbol) return Boolean;
   --# global in Dict;
   -- pre IsSubprogram (Subprogram);

   -- Returns True iff Sym is a program unit as defined by
   -- Ada95 LRM Annex N.31.  Used to check the legality of
   -- adddress representation clauses.  This definition also
   -- seems to be consistent with that given in Ada83 LRM 13.1(4).
   function IsProgramUnit (Sym : Symbol) return Boolean;
   --# global in Dict;

   function MainProgramExists return Boolean;
   --# global in Dict;

   function MainProgramPrioritySupplied return Boolean;
   --# global in Dict;

   function PackageRequiresBody (ThePackage : Symbol) return Boolean;
   --# global in CommandLineData.Content;
   --#        in Dict;
   -- pre IsPackage (ThePackage);

   function GetInterruptStreamVariable
     (ProtectedObject  : in Symbol;
      InterruptHandler : in Symbol) return Symbol;
   --# global in Dict;

   function PartitionElaborationPolicyIsConcurrent return Boolean;
   -- This is a hook for when pragma Partition_Elaboration_Policy is supported
   -- in the Ada language as standard. It currently returns True.
   -- This forces the examiner to be more stringent in the initialisation of
   -- protected state.

   --------------------------------------------------------------------------------
   --                                 ITERATORS                                  --
   --------------------------------------------------------------------------------

   function CurrentSymbol (CurrentIterator : Iterator) return Symbol;
   --# global in Dict;

   function NextSymbol (Previous : Iterator) return Iterator;
   --# global in Dict;

   function IsNullIterator (Current : Iterator) return Boolean;

   function FirstDeferredConstant (ThePackage : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsPackage (ThePackage);

   function FirstArrayIndex (TypeMark : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsArrayTypeMark (TypeMark, GetScope (TypeMark));

   function FirstUndeclaredType (ThePackage : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsPackage (ThePackage);

   function FirstPrivateType (ThePackage : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsPackage (ThePackage);

   function FirstEnumerationLiteral (EnumerationType : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsEnumerationTypeMark (EnumerationType, GetScope (EnumerationType));

   function FirstRecordComponent (TheRecordType : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsRecordTypeMark (TheRecordType, GetScope (TheRecordType));
   -- N.B. This iterator works over a single record; if you want to iterate
   --      over a group of records modelling extension of tagged types then
   --      use the following function instead.

   function FirstExtendedRecordComponent (TheRecordType : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsRecordTypeMark (TheRecordType, GetScope (TheRecordType)) and
   --     TypeIsExtendedTagged (TheRecordType);

   function FirstVisibleSubprogram (ThePackageOrType : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsPackage (ThePackageOrType) or
   --     (IsType (ThePackageOrType) and TypeIsProtected (ThePackageOrType));

   function FirstPrivateSubprogram (ThePackage : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsPackage (ThePackageOrType);

   function FirstVisibleTaskType (ThePackage : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsPackage (ThePackage);

   function FirstPrivateTaskType (ThePackage : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsPackage (ThePackage);

   function FirstOwnTask (ThePackage : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsPackage (ThePackage);

   function FirstVisibleProtectedType (ThePackage : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsPackage (ThePackage);

   function FirstPrivateProtectedType (ThePackage : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsPackage (ThePackage);

   function FirstSubprogramParameter (Subprogram : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsSubprogram (Subprogram) or IsTaskType (Subprogram);
   -- return I => (IsTaskType (SubprogSym -> IsNullIterator (I) and
   --             (IsSubprogram (Subprogsym) -> "UsefulIterator"(I));

   function FirstGenericFormalParameter (TheGeneric : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsGenericSubprgoram (TheGeneric) or IsGenericPackage (TheGeneric);

   function GetNumberOfGenericFormalParameters (TheGeneric : Symbol) return Natural;
   --# global in Dict;
   -- pre IsGenericSubprgoram (TheGeneric) or IsGenericPackage (TheGeneric);

   function GetGenericFormalParameterKind (TheGenericFormalParameter : Symbol) return GenericParameterKind;
   --# global in Dict;
   -- pre IsGenericFormalParameter (TheGenericFormalParameter);

   function FirstKnownDiscriminant (ProtectedOrTaskType : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsType (ProtectedOrTaskType) and
   --        (TypeIsProtected (ProtectedOrTaskType) or TypeIsTask (ProtectedOrTaskType));

   function FirstDiscriminantConstraint (ProtectedOrTaskSubtype : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsType (GetParentType (ProtectedOrTaskType)) and
   --        (TypeIsProtected (GetParentType (ProtectedOrTaskType))
   --      or TypeIsTask (GetParentType (ProtectedOrTaskType));

   function FirstGlobalVariable (Abstraction : Abstractions;
                                 Subprogram  : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsSubprogram (Subprogram) or IsTaskType (SubprogSym);

   function FirstLocalVariable (Subprogram : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsSubprogram (Subprogram);

   function FirstInitializedVariable (Subprogram : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsSubprogram (Subprogram) or IsTaskType (Subprogram);

   function FirstProtectedElement (TheProtectedType : Symbol) return Iterator;
   --# global in Dict;
   -- pre (IsType (ThePackageOrType) and TypeIsProtected (ThePackageOrType));

   function FirstImportExport (Abstraction  : Abstractions;
                               TheProcedure : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsProcedure (TheProcedure);

   function FirstExport (Abstraction  : Abstractions;
                         TheProcedure : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsProcedure (TheProcedure) or is a task (TheProcedure);

   function FirstImport (Abstraction  : Abstractions;
                         TheProcedure : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsProcedure (TheProcedure);

   function FirstDependency (Abstraction  : Abstractions;
                             TheProcedure : Symbol;
                             TheExport    : Symbol) return Iterator;
   --# global in Dict;

   function FirstSuspendsListItem (TheTaskOrProc : Symbol) return Iterator;
   --# global in Dict;

   function FirstVirtualElement (TheProtectedType : Symbol) return Iterator;
   --# global in Dict;

   function FirstOwnedPackage (ThePackage : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsPackage (ThePackage);

   function FirstEmbeddedPackage (CompilationUnit : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsCompilationUnit (CompilationUnit);

   function FirstOwnVariable (ThePackage : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsPackage (ThePackage);

   function FirstInitializedOwnVariable (ThePackage : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsPackage (ThePackage);

   function FirstConstituent (Subject : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsOwnVariable (Abstract, Subject);

   function FirstInheritsClause (Sym : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsPackage (Sym) or IsMainProgram (Sym);

   function FirstInterruptStreamMapping (Sym : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsOwnVariable (Sym);

   function FirstLoopOnEntryVar (TheLoop : Symbol) return Iterator;
   --# global in Dict;
   -- pre IsLoop (TheLoop);
   -- returns first variable used in exit expression of for loop

   function GetInterruptStreamMappingHandler (TheMapping : in Symbol) return LexTokenManager.LexString;

   function GetInterruptStreamMappingStream (TheMapping : in Symbol) return LexTokenManager.LexString;

   --------------------------------------------------------------------------------

private
   --  We allocate SymbolTableSize entries for user-defined Symbols,
   --  plus sentinel value 0 for NullSymbol
   type Symbol is range 0 .. ExaminerConstants.SymbolTableSize;
   --# assert Symbol'Base is Integer;

   NullSymbol : constant Symbol := Symbol'First;

   type IteratorDiscriminant is (NullSymIterator,
                                 DeclarativeItemIterator,
                                 DeferredConstantIterator,
                                 ArrayIndexIterator,
                                 LoopIterator,
                                 UndeclaredTypeIterator,
                                 PrivateTypeIterator,
                                 EnumerationLiteralIterator,
                                 RecordComponentIterator,
                                 ExtendedRecordComponentIterator,
                                 LibraryUnitIterator,
                                 WithedPackageIterator,
                                 InheritedPackageIterator,
                                 VisibleSubprogramIterator,
                                 PrivateSubprogramIterator,
                                 TaskTypeIterator,
                                 OwnTaskIterator,
                                 ProtectedTypeIterator,
                                 SubprogramParameterIterator,
                                 GenericFormalParameterIterator,
                                 KnownDiscriminantIterator,
                                 DiscriminantConstraintIterator,
                                 ImplicitProofFunctionParameterIterator,
                                 ImplicitProofFunctionGlobalIterator,
                                 GlobalVariableIterator,
                                 LocalVariableIterator,
                                 InitializedVariableIterator,
                                 ImportExportIterator,
                                 ExportIterator,
                                 ImportIterator,
                                 DependencyIterator,
                                 InterruptStreamMappingIterator,
                                 SuspendsListItemIterator,
                                 VirtualElementIterator,
                                 OwnedPackageIterator,
                                 ProtectedElementIterator,
                                 LoopOnEntryVarIterator,
                                 EmbeddedPackageIterator,
                                 -- see Dictionary.CurrentSymbol before adding to this list, position matters
                                 OwnVariableIterator,
                                 InitializedOwnVariableIterator,
                                 AbstractOwnVariableIterator,
                                 -- see Dictionary.CurrentSymbol before adding to this list, position matters
                                 ConstituentIterator);

   type Iterator is
      record
         Discriminant : IteratorDiscriminant;
         Abstraction  : Abstractions;
         Current      : Symbol;
         Context      : Symbol;
      end record;

   NullIterator : constant Iterator := Iterator'(NullSymIterator,
                                                 IsAbstract,
                                                 NullSymbol,
                                                 NullSymbol);

   type Visibility is (Visible, Privat, Local);

   type Scopes is
      record
         TypeOfScope : Visibility;
         Region      : Symbol;
      end record;
   NullScope : constant Scopes := Scopes'(Visible, NullSymbol);
end Dictionary;
