-- $Id: dictionary-targetdata.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with CommandLineData,
     ELStrings,
     Ada.Characters.Handling,
     ScreenEcho,
     FileSystem,
     SystemErrors,
     XMLReport;

separate (Dictionary)
package body TargetData
is

   -- Types-------------------------------------------------------------------------------

   type ValStatus is (OkVal, MissingVal, IllegalVal);
   type ValSort   is (IntegerVal, RealVal);

   -- Local Subprograms-------------------------------------------------------------------

   procedure OpenFile (DataFile : out SPARK_IO.File_Type;
                       FileOk   : out Boolean)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives DataFile,
   --#         FileOk,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                SPARK_IO.File_Sys;
   is
      FileName       : EStrings.T;
      FileSpecStatus : FileSystem.TypFileSpecStatus;
      FileStatus     : SPARK_IO.File_Status;
      LocalFile      : SPARK_IO.File_Type;
   begin
      LocalFile := SPARK_IO.Null_File;

      --# accept Flow, 10, FileSpecStatus, "Expected ineffective assignment";
      FileSystem.FindFullFileName -- 782 expect flow error FileSpecStatus not used.
         (CommandLineData.Content.TargetDataFile,
          FileSpecStatus,  -- not used: flow error
          FileName);
      --# end accept;
      SPARK_IO.Open (LocalFile,
                     SPARK_IO.In_File,
                     FileName.Length,
                     FileName.Content,
                     "",
                     FileStatus);

      if FileStatus = SPARK_IO.Ok then
         FileOk := True;
      else
         FileOk := False;
         ScreenEcho.Put_String ("Cannot open file ");
         if CommandLineData.Content.PlainOutput then
            ScreenEcho.Put_ExaminerLine
               (EStrings.LowerCase
                (FileSystem.JustFile (FileName, True)));
         else
            ScreenEcho.Put_ExaminerLine (FileName);
         end if;
      end if;
      DataFile := LocalFile;
      --# accept Flow, 33, FileSpecStatus, "Expected to be neither referenced nor exported";
   end OpenFile;

   ------------------------------------------------

   procedure CloseFile (DataFile : in out SPARK_IO.File_Type)
      --# global in out SPARK_IO.File_Sys;
      --# derives DataFile,
      --#         SPARK_IO.File_Sys from *,
      --#                                DataFile;
   is
      FileStatus     : SPARK_IO.File_Status;
   begin
      --# accept Flow, 10, FileStatus, "Expected ineffective assignment";
      SPARK_IO.Close (DataFile, -- 782 Expect 2 ineffective assignments
                      FileStatus); -- not used
      --# accept Flow, 33, FileStatus, "Expected to be neither referenced nor exported";
   end CloseFile;

   ------------------------------------------------

   procedure GetString (File : in     SPARK_IO.File_Type;
                        Str  :    out EStrings.T)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys,
      --#         Str               from File,
      --#                                SPARK_IO.File_Sys;
   is
      CharOk   : Boolean;
      Ch       : Character;
      AppendOK : Boolean;
      EStr     : EStrings.T;

      procedure GetChar (File : in     SPARK_IO.File_Type;
                         Ch   :    out Character;
                         OK   :    out Boolean)
         --# global in out SPARK_IO.File_Sys;
         --# derives Ch,
         --#         OK,
         --#         SPARK_IO.File_Sys from File,
         --#                                SPARK_IO.File_Sys;
      is
      begin
         if SPARK_IO.End_Of_File (File) then
            OK := False;
            Ch := ' ';
         elsif SPARK_IO.End_Of_Line (File) then
            SPARK_IO.Skip_Line (File, 1);
            OK := True;
            Ch := ' ';
         else
            SPARK_IO.Get_Char (File, Ch);
            OK := True;
         end if;
      end GetChar;

   begin --GetString
      EStr := EStrings.EmptyString;
      --skip leading white space
      loop
         GetChar (File, Ch, CharOk);
         exit when Ch /= ' ';
         exit when not CharOk;
      end loop;

      if CharOk then
         loop
            EStrings.AppendChar (EStr, Ch, AppendOK);
            if not AppendOK then
               SystemErrors.FatalError (SystemErrors.WarningNameTooLong, "in TargetData.GetString");
            end if;
            GetChar (File, Ch, CharOk);
            exit when Ch = ' ';
            exit when not CharOk;
         end loop;
      end if;

      Str := EStr;
   end GetString;

   ----------------------------------------

   procedure CheckOptionName (OptName : in     EStrings.T;
                              Str     : in     String;
                              OK      :    out Boolean)
      --# derives OK from OptName,
      --#                 Str;
      --# pre OptName.Length > 0;
   is
      LOK : Boolean;
   begin --CheckOptionName
      if OptName.Length <= Str'Length then
         for I in EStrings.Lengths range 1 .. OptName.Length loop
            LOK := Ada.Characters.Handling.To_Lower (OptName.Content (I)) =
              Ada.Characters.Handling.To_Lower (Str (I));
            exit when not LOK;
         end loop;
         --# accept Flow, 501, LOK, "Always defined if precondition met";
         OK := LOK;  -- Assuming loop range non-empty, LOK always defined
         --# end accept;
      else
         OK := False;
      end if;
      --# accept Flow, 602, Ok, LOK, "Always defined if precondition met";
   end CheckOptionName;

   -------------------------------

   procedure SkipEquals (DataFile : SPARK_IO.File_Type)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                DataFile;
   is
      UnusedString : EStrings.T;
      pragma Unreferenced (UnusedString);
   begin
      --# accept Flow, 10, UnusedString, "Expected ineffective assignment" &
      --#        Flow, 33, UnusedString, "Expected to be neither referenced nor exported";
      GetString (DataFile,
                  -- to get
                 UnusedString);
   end SkipEquals;

   --------------------------------

   procedure GetDataValue (DataFile : in       SPARK_IO.File_Type;
                           Sort     : in       ValSort;
                           Val      :      out LexTokenManager.LexString;
                           Status   :      out ValStatus)
      --# global in out LexTokenManager.StringTable;
      --#        in out SPARK_IO.File_Sys;
      --# derives LexTokenManager.StringTable,
      --#         Status,
      --#         Val                         from DataFile,
      --#                                          LexTokenManager.StringTable,
      --#                                          Sort,
      --#                                          SPARK_IO.File_Sys &
      --#         SPARK_IO.File_Sys           from *,
      --#                                          DataFile;
   is
      ValueString : EStrings.T;
      LexVal      : LexTokenManager.LexString;
      TempLine    : EStrings.Line;
      ConvOk      : Maths.ErrorCode;
      Num         : Maths.Value;
      StartPos    : Integer;
      IsNegative  : Boolean;

   begin --GetDataValue
      SkipEquals (DataFile);
      GetString (DataFile,
                  -- to get
                 ValueString);
      if ValueString.Length /= 0 then
         if ValueString.Content (1) = '-' then
            StartPos := 2;
            IsNegative := True;
         else
            StartPos := 1;
            IsNegative := False;
         end if;

         --# accept Flow, 23, TempLine, "Array fully initialized by loop";
         for I in Integer range 1 .. ValueString.Length loop
            TempLine (I) := ValueString.Content (I);  --array initialisation flow error
         end loop;
         --# end accept;
         --# accept Flow, 504, TempLine, "Array fully initialized by loop";
         LexTokenManager.InsertLexString (TempLine, -- array initialisation flow error
                                          StartPos,
                                          ValueString.Length,
                                          LexVal);
         --# end accept;
         Maths.LiteralToValue (LexVal,
                                 -- to get
                               Num,
                               ConvOk);
         if ConvOk = Maths.NoError then
            if Sort = IntegerVal and not Maths.IsIntegerValue (Num) then
               Val := LexTokenManager.NullString;
               Status := IllegalVal;
            elsif Sort = RealVal and not Maths.IsRealValue (Num) then
               Val := LexTokenManager.NullString;
               Status := IllegalVal;
            else
               if IsNegative then
                  Maths.Negate (Num);
               end if;
               Maths.StorageRep (Num, Val);
               Status := OkVal;
            end if;
         else
            Val := LexTokenManager.NullString;
            Status := IllegalVal;
         end if;
      else
         Val := LexTokenManager.NullString;
         Status := MissingVal;
      end if;
      --# accept Flow, 602, LexTokenManager.StringTable, TempLine, "Fully initialized by loop" &
      --#        Flow, 602, VAL, TempLine, "Fully initialized by loop" &
      --#        Flow, 602, Status, TempLine, "Fully initialized by loop";
   end GetDataValue;

   -----------------------------------------------------------

   procedure EchoError (Status : in     ValStatus)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Status;
   is
   begin
      case Status is
         when OkVal =>
            null;
         when IllegalVal =>
            ScreenEcho.Put_Line ("Illegal value");
         when MissingVal =>
            ScreenEcho.Put_Line ("Value missing");
      end case;
   end EchoError;

   --Exported Subprograms-----------------------------------------------------------------

   procedure ReadTargetDataFile
   is
      Option      : EStrings.T;
      FileOk      : Boolean;
      DataFile    : SPARK_IO.File_Type;

      procedure InvalidOption (Opt : EStrings.T)
         --# global in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                Opt;
      is
      begin
         ScreenEcho.Put_String ("Invalid target compiler data item: ");
         ScreenEcho.Put_ExaminerLine (Opt);
      end InvalidOption;

      -------------------------------

      procedure ProcessOption (Opt : EStrings.T)
         --# global in     DataFile;
         --#        in out Dictionary.Dict;
         --#        in out LexTokenManager.StringTable;
         --#        in out SPARK_IO.File_Sys;
         --# derives Dictionary.Dict,
         --#         LexTokenManager.StringTable,
         --#         SPARK_IO.File_Sys           from *,
         --#                                          DataFile,
         --#                                          LexTokenManager.StringTable,
         --#                                          Opt,
         --#                                          SPARK_IO.File_Sys;
      is
         OptionMatch : Boolean;
         Val         : LexTokenManager.LexString;
         Status      : ValStatus;

      begin --ProcessOption
         OptionMatch := False;
         case Opt.Content (1) is
            when 'i' | 'I' =>
               case Opt.Content (9) is
                  when 'f' | 'F' =>
                     CheckOptionName (Opt, "integer'first", OptionMatch);
                     if OptionMatch then
                        GetDataValue (DataFile, IntegerVal, Val, Status);
                        EchoError (Status);
                        RawDict.SetTypeLower (Dictionary.GetPredefinedIntegerType, Val);
                     end if;

                  when 'l' | 'L' =>
                     CheckOptionName (Opt, "integer'last", OptionMatch);
                     if OptionMatch then
                        GetDataValue (DataFile, IntegerVal, Val, Status);
                        EchoError (Status);
                        RawDict.SetTypeUpper (Dictionary.GetPredefinedIntegerType, Val);
                        RawDict.SetTypeUpper (Dictionary.GetPredefinedPositiveSubtype, Val);
                        RawDict.SetTypeUpper (Dictionary.GetPredefinedNaturalSubtype, Val);
                     end if;

                  when others =>
                     null;

               end case;

            when 'l' | 'L' =>
               case Opt.Content (14) is
                  when 'f' | 'F' =>
                     CheckOptionName (Opt, "long_integer'first", OptionMatch);
                     if OptionMatch then
                        GetDataValue (DataFile, IntegerVal, Val, Status);
                        EchoError (Status);
                        RawDict.SetTypeLower (Dictionary.GetPredefinedLongIntegerType, Val);
                     end if;

                  when 'l' | 'L' =>
                     CheckOptionName (Opt, "long_integer'last", OptionMatch);
                     if OptionMatch then
                        GetDataValue (DataFile, IntegerVal, Val, Status);
                        EchoError (Status);
                        RawDict.SetTypeUpper (Dictionary.GetPredefinedLongIntegerType, Val);
                     end if;

                  when others =>
                     null;

               end case;

            when others =>
               null;
         end case;

         if not OptionMatch then
            InvalidOption (Opt);
         end if;
      end ProcessOption;

      -------------------------------

   begin --ReadTargetDataFile
      if CommandLineData.Content.TargetData then
         OpenFile (DataFile,
                     -- to get
                   FileOk);
         if FileOk then
            if CommandLineData.Content.Echo and not CommandLineData.Content.Brief then
               ScreenEcho.New_Line (1);
               ScreenEcho.Put_Line ("           Reading target compiler data ...");
            end if;
            loop
               GetString (DataFile,
                           -- to get
                          Option);
               exit when Option.Length = 0;
               ProcessOption (Option);
            end loop;
            --# accept Flow, 10, DataFile, "Expected ineffective assignment";
            CloseFile (DataFile); -- Expect ineff assignment here
            --# end accept;
         end if;
      end if;
   end ReadTargetDataFile;

   ------------------------------------------------------------------------------

   procedure OutputTargetDataFile (ToFile : in SPARK_IO.File_Type)
   is
      Option      : EStrings.T;
      FileOk      : Boolean;
      DataFile    : SPARK_IO.File_Type;

      --------------------------------

      procedure InvalidOption (ToFile : SPARK_IO.File_Type;
                               Opt : EStrings.T)
         --# global in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                Opt,
         --#                                ToFile;
      is
      begin
         SPARK_IO.Put_String (ToFile, "Invalid target compiler data item: ", 0);
         EStrings.PutString (ToFile, Opt);
         SPARK_IO.Put_Line (ToFile, " has been ignored", 0);
      end InvalidOption;

      -------------------------------

      procedure ProcessOption (Opt    : EStrings.T;
                               ToFile : SPARK_IO.File_Type)
      --# global in     CommandLineData.Content;
      --#        in     DataFile;
      --#        in out LexTokenManager.StringTable;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives LexTokenManager.StringTable from *,
      --#                                          CommandLineData.Content,
      --#                                          DataFile,
      --#                                          Opt,
      --#                                          SPARK_IO.File_Sys,
      --#                                          ToFile &
      --#         SPARK_IO.File_Sys           from *,
      --#                                          CommandLineData.Content,
      --#                                          DataFile,
      --#                                          LexTokenManager.StringTable,
      --#                                          Opt,
      --#                                          ToFile,
      --#                                          XMLReport.State &
      --#         XMLReport.State             from *,
      --#                                          CommandLineData.Content,
      --#                                          Opt;
      is
         OptionMatch : Boolean;
         --Val         : LexTokenManager.LexString;
         --Status      : ValStatus;

         procedure Margin
            --# global in     ToFile;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                ToFile;
         is
         begin
            SPARK_IO.Put_String (ToFile, "   ", 0);
         end Margin;

         ----------------------------------------

         procedure Separator
            --# global in     ToFile;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                ToFile;
         is
         begin
            SPARK_IO.Put_String (ToFile, " = ", 0);
         end Separator;

         ----------------------------------------

         procedure PutVal (ToFile : in     SPARK_IO.File_Type;
                           Val    : in     LexTokenManager.LexString;
                           Status : in     ValStatus)
            --# global in     LexTokenManager.StringTable;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                LexTokenManager.StringTable,
            --#                                Status,
            --#                                ToFile,
            --#                                Val;
         is
         begin
            case Status is
               when OkVal =>
                  ELStrings.PutLine (ToFile, Maths.ValueToString (Maths.ValueRep (Val)));
               when IllegalVal =>
                  SPARK_IO.Put_Line (ToFile, "Illegal value - ignored", 0);
               when MissingVal =>
                  SPARK_IO.Put_Line (ToFile, "Value missing - ignored", 0);
            end case;
         end PutVal;


         function GetVal (Val    : in LexTokenManager.LexString;
                          Status : in ValStatus) return ELStrings.T
         --# global in LexTokenManager.StringTable;
         is
            TmpString : ELStrings.T;
         begin
            case Status is
               when OkVal =>
                  TmpString := Maths.ValueToString (Maths.ValueRep (Val));
               when IllegalVal =>
                  ELStrings.CopyString (TmpString, "Illegal value - ignored");
               when MissingVal =>
                  ELStrings.CopyString (TmpString, "Value missing - ignored");
            end case;
            return TmpString;
         end GetVal;



         ---------------------------------------

         procedure OutputValue (DataFile : in     SPARK_IO.File_Type;
                                Opt      : in     EStrings.T;
                                Valid    : in     Boolean)
         --# global in     CommandLineData.Content;
         --#        in     ToFile;
         --#        in out LexTokenManager.StringTable;
         --#        in out SPARK_IO.File_Sys;
         --#        in out XMLReport.State;
         --# derives LexTokenManager.StringTable from *,
         --#                                          CommandLineData.Content,
         --#                                          DataFile,
         --#                                          Opt,
         --#                                          SPARK_IO.File_Sys,
         --#                                          ToFile,
         --#                                          Valid &
         --#         SPARK_IO.File_Sys           from *,
         --#                                          CommandLineData.Content,
         --#                                          DataFile,
         --#                                          LexTokenManager.StringTable,
         --#                                          Opt,
         --#                                          ToFile,
         --#                                          Valid,
         --#                                          XMLReport.State &
         --#         XMLReport.State             from *,
         --#                                          CommandLineData.Content,
         --#                                          Opt,
         --#                                          Valid;
         is
            Val    : LexTokenManager.LexString;
            Status : ValStatus;
         begin
            if Valid then
               if CommandLineData.Content.XML then
                  GetDataValue (DataFile, IntegerVal, Val, Status);
                  XMLReport.LongCompilerItem (Opt,
                                              GetVal (Val, Status),
                                              ToFile);
               else
                  Margin;
                  EStrings.PutString (ToFile, Opt);
                  Separator;
                  GetDataValue (DataFile, IntegerVal, Val, Status);
                  PutVal (ToFile, Val, Status);
               end if;
            end if;
         end OutputValue;

         ---------------------------------------

      begin --ProcessOption
         OptionMatch := False;
         case Opt.Content (1) is
            when 'i' | 'I' =>
               case Opt.Content (9) is
                  when 'f' | 'F' =>
                     CheckOptionName (Opt, "integer'first", OptionMatch);
                     OutputValue (DataFile, Opt, OptionMatch);

                  when 'l' | 'L' =>
                     CheckOptionName (Opt, "integer'last", OptionMatch);
                     OutputValue (DataFile, Opt, OptionMatch);

                  when others =>
                     null;
               end case;

            when 'l' | 'L' =>
               case Opt.Content (14) is
                  when 'f' | 'F' =>
                     CheckOptionName (Opt, "long_integer'first", OptionMatch);
                     OutputValue (DataFile, Opt, OptionMatch);

                  when 'l' | 'L' =>
                     CheckOptionName (Opt, "long_integer'last", OptionMatch);
                     OutputValue (DataFile, Opt, OptionMatch);

                  when others =>
                     null;
               end case;

            when others =>
               null;
         end case;

         if not OptionMatch then
            Margin;
            InvalidOption (ToFile, Opt);
         end if;
      end ProcessOption;

      -------------------------------

   begin --OutputTargetDataFile
      if CommandLineData.Content.TargetData then
         OpenFile (DataFile,
                   FileOk);
         if FileOk then
            if CommandLineData.Content.XML then
               XMLReport.StartSection (XMLReport.SCompilerData,
                                       ToFile);
               loop
                  GetString (DataFile,
                              -- to get
                             Option);
                  exit when Option.Length = 0;
                  ProcessOption (Option, ToFile);
               end loop;
               XMLReport.EndSection (XMLReport.SCompilerData,
                                     ToFile);
               --# accept Flow, 10, DataFile, "Expected ineffective assignment";
               CloseFile (DataFile); -- Expect ineff assignment here
               --# end accept;
            else
               SPARK_IO.New_Line (ToFile, 2);
               SPARK_IO.Put_Line (ToFile, "Target compiler data:", 0);
               loop
                  GetString (DataFile,
                              -- to get
                             Option);
                  exit when Option.Length = 0;
                  ProcessOption (Option, ToFile);
               end loop;
               --# accept Flow, 10, DataFile, "Expected ineffective assignment";
               CloseFile (DataFile); -- Expect ineff assignment here
               --# end accept;
            end if; -- XML
         end if;
      end if;
   end OutputTargetDataFile;

   ------------------------------------------------------------------------------

end TargetData;
