-- $Id: dictionary-lookupscope.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Dictionary)
procedure LookupScope (Name         : in     LexTokenManager.LexString;
                       StopAt       : in     LexTokenManager.LexString;
                       Scope        : in     Scopes;
                       CallingScope : in     Scopes;
                       Context      : in     Contexts;
                       Item         :    out Symbol;
                       IsVisible    :    out Boolean)
-- The extra CallingScope parameter is only used when LookUpScope is called from
-- LookUpSelectedItem because in this case only the scope we start the search in
-- is not the same as the scope where the search started (former is the visible
-- scope of the prefix package).  The extra parameter only affects the selection of
-- the correct implicit proof function that corresponds to an Ada function
is

   Region  : Symbol;
   TheItem : Symbol;

   --------------------------------------------------------------------------------

   function LookupDeclarations (Name    : LexTokenManager.LexString;
                                StopAt  : LexTokenManager.LexString;
                                Context : Contexts;
                                Head    : Symbol) return Symbol
      --# global in Dict;
   is

      Declaration, Item, DeclarativeItem : Symbol;
      DeclarativeItemName                : LexTokenManager.LexString;

      --------------------------------------------------------------------------------

      function LookupDeclaration (Name                : LexTokenManager.LexString;
                                  DeclarativeItem     : Symbol;
                                  DeclarativeItemName : LexTokenManager.LexString;
                                  Context             : Contexts;
                                  Declaration         : Symbol) return Symbol
         --# global in Dict;
      is

         Item : Symbol;

         --------------------------------------------------------------------------------

         function LookupEnumerationLiterals (Name    : LexTokenManager.LexString;
                                             TheType : Symbol) return Symbol
            --# global in Dict;
         is
            Literal : Symbol;
         begin
            TraceLexStr ("      In LookupEnumerationLiterals, seeking ", Name);

            Literal := RawDict.GetTypeFirstEnumerationLiteral (TheType);

            loop
               exit when Literal = NullSymbol;
               exit when RawDict.GetEnumerationLiteralName (Literal) = Name;
               Literal := RawDict.GetNextEnumerationLiteral (Literal);
            end loop;

            return Literal;

         end LookupEnumerationLiterals;

         --------------------------------------------------------------------------------

      begin
         if Context = ProgramContext and then
            RawDict.GetDeclarationContext (Declaration) = ProofContext then
            Item := NullSymbol;
         else

            if DeclarativeItemName = Name then
               Item := DeclarativeItem;
            elsif IsType (DeclarativeItem) and then
               TypeIsEnumeration (DeclarativeItem) then
               Item := LookupEnumerationLiterals (Name, DeclarativeItem);
            else
               Item := NullSymbol;
            end if;

         end if;

         return Item;

      end LookupDeclaration;

      --------------------------------------------------------------------------------

   begin
      TraceLexStr ("In LookupDeclarations, seeking ", Name);
      Item := NullSymbol;
      Declaration := Head;

      loop
         exit when Declaration = NullSymbol;
         DeclarativeItem := RawDict.GetDeclarationItem (Declaration);
         if RawDict.GetSymbolDiscriminant (DeclarativeItem) = OperatorSymbol then
            DeclarativeItemName := LexTokenManager.NullString;
         else
            DeclarativeItemName := GetSimpleName (DeclarativeItem);
         end if;
         TraceLexStr ("DeclarativeItemName is ", DeclarativeItemName);
         exit when StopAt /= LexTokenManager.NullString and then
           StopAt = DeclarativeItemName;
         Item := LookupDeclaration (Name,
                                    DeclarativeItem,
                                    DeclarativeItemName,
                                    Context,
                                    Declaration);
         exit when Item /= NullSymbol;
         Declaration := RawDict.GetNextDeclaration (Declaration);
      end loop;

      return Item;

   end LookupDeclarations;

   --------------------------------------------------------------------------------

   function LookupKnownDiscriminants (Name      : in LexTokenManager.LexString;
                                      TheRegion : in Symbol) return Symbol
   --# global in Dict;
   is
      TheDiscriminant : Symbol;
   begin
      TraceLexStr ("In LookupKnownDiscriminants, seeking ", Name);
      if RawDict.GetTypeDiscriminant (TheRegion) = ProtectedType then
         TheDiscriminant := RawDict.GetProtectedTypeFirstDiscriminant (TheRegion);
      else
         TheDiscriminant := RawDict.GetTaskTypeFirstDiscriminant (TheRegion);
      end if;

      loop
         exit when TheDiscriminant = NullSymbol;
         exit when RawDict.GetDiscriminantName (TheDiscriminant) = Name;
         TheDiscriminant := RawDict.GetNextDiscriminant (TheDiscriminant);
      end loop;
      return TheDiscriminant;
   end LookupKnownDiscriminants;

   --------------------------------------------------------------------------------

   procedure LookUpGenericFormalParameters (Name                : in     LexTokenManager.LexString;
                                            SubprogramOrPackage : in     Symbol;
                                            Parameter           :    out Symbol;
                                            IsVisible           :    out Boolean)
   --# global in Dict;
   --# derives IsVisible,
   --#         Parameter from Dict,
   --#                        Name,
   --#                        SubprogramOrPackage;
   is
      TheParameter : Symbol;

      function GetGenericParameterName (Sym : Symbol) return LexTokenManager.LexString
      --# global in Dict;
      is
         Result : LexTokenManager.LexString;
         TypeOrObjectSym : Symbol;

      begin -- GetGenericParameterName
         TypeOrObjectSym := RawDict.GetGenericParameterTypeOrObject (Sym);
         Result := GetSimpleName (TypeOrObjectSym);
         return Result;
      end GetGenericParameterName;

   begin -- LookUpGenericFormalParameters
      TraceLexStr ("      In LookupGenericFormalParameters, seeking ", Name);

      IsVisible := False;       -- default if not found
      Parameter := NullSymbol;  -- default if not found

      -- GenericTBD extend this to handle generic package formal later
      if IsPackage (SubprogramOrPackage) then
         TheParameter := RawDict.GetPackageFirstGenericParameter (SubprogramOrPackage);
      else
         TheParameter := RawDict.GetSubprogramFirstGenericParameter (SubprogramOrPackage);
      end if;

      loop
         exit when TheParameter = NullSymbol;
         if GetGenericParameterName (TheParameter) = Name then
            -- actually want the generic entity not the generic parameter
            Parameter := RawDict.GetGenericParameterTypeOrObject (TheParameter);
            IsVisible := True;
         end if;

         TheParameter := RawDict.GetGenericParameterNext (TheParameter);
      end loop;

   end LookUpGenericFormalParameters;

   --------------------------------------------------------------------------------

   function LookupVisibleDeclarations (Name       : LexTokenManager.LexString;
                                       TheRegion  : Symbol;
                                       Context    : Contexts) return Symbol
   --# global in Dict;
   is

      Item : Symbol;

      ------------------------------------------------------------------------------

      function LookupOwnVariables (Name       : LexTokenManager.LexString;
                                   ThePackage : Symbol) return Symbol
         --# global in Dict;
      is
         OwnVariable, Variable : Symbol;
      begin
         TraceLexStr ("   In LookupOwnVariables, seeking ", Name);

         OwnVariable := RawDict.GetPackageOwnVariables (ThePackage);

         loop
            if OwnVariable = NullSymbol then
               Variable := NullSymbol;
               exit;
            end if;
            Variable := RawDict.GetOwnVariableVariable (OwnVariable);
            exit when RawDict.GetVariableName (Variable) = Name;
            OwnVariable := RawDict.GetNextOwnVariable (OwnVariable);
         end loop;

         return Variable;

      end LookupOwnVariables;

      ------------------------------------------------------------------------------

      function LookupOwnTasks (Name       : LexTokenManager.LexString;
                               ThePackage : Symbol) return Symbol
         --# global in Dict;
      is
         OwnTask, Variable : Symbol;
      begin
         TraceLexStr ("   In LookupOwnTasks, seeking ", Name);

         OwnTask := RawDict.GetPackageTaskList (ThePackage);

         loop
            if OwnTask = NullSymbol then
               Variable := NullSymbol;
               exit;
            end if;
            Variable := RawDict.GetOwnTaskVariable (OwnTask);
            exit when RawDict.GetVariableName (Variable) = Name;
            OwnTask := RawDict.GetNextOwnTask (OwnTask);
         end loop;

         return Variable;

      end LookupOwnTasks;

      ------------------------------------------------------------------------------

      function LookUpPackageVisibleDeclarations (ThePackage : Symbol) return Symbol
      --# global in Context;
      --#        in Dict;
      --#        in Name;
      is
         Item : Symbol;
         Dummy : Boolean;
      begin
         TraceLexStr ("   In LookupPackageVisibleDeclarations, seeking ", Name);
         if GetSimpleName (ThePackage) = Name then
            Item := ThePackage;
         else
            Item := LookupDeclarations (Name,
                                        LexTokenManager.NullString,
                                        Context,
                                        RawDict.GetPackageFirstVisibleDeclaration (ThePackage));

            if Item = NullSymbol then

               Item := LookupDeclarations (Name,
                                           LexTokenManager.NullString,
                                           Context,
                                           RawDict.GetPackageVisibleRenamingDeclarations (ThePackage));

               if Item = NullSymbol and then
                 IsGenericPackage (ThePackage) then
                  --# accept Flow, 10, Dummy, "Expected ineffective assignment to Dummy";
                  LookUpGenericFormalParameters (Name, ThePackage, Item, Dummy);
                  --# end accept;
               end if;

               if Item = NullSymbol and then Context = ProofContext then
                  Item := LookupOwnVariables (Name, ThePackage);
                  if Item = NullSymbol then
                     Item := LookupOwnTasks (Name, ThePackage);
                  end if;
               end if;
            end if;
         end if;
         --# accept Flow, 33, Dummy, "Expected to be neither referenced nor exported Dummy";
         return Item;
      end LookUpPackageVisibleDeclarations;

      -----------------------------------

      function LookUpProtectedTypeVisibleDeclarations (TheProtectedType : Symbol) return Symbol
      --# global in Context;
      --#        in Dict;
      --#        in Name;
      is
         Item : Symbol;
      begin
         TraceLexStr ("   In LookupProtectedTypeVisibleDeclarations, seeking ", Name);
         if GetSimpleName (TheProtectedType) = Name then
            -- if we find PT (say) when looking from within PT then what we want to return is the
            -- implicitly declared abstract own variable of PT rather than the type itself
            Item := RawDict.GetOwnVariableVariable (RawDict.GetProtectedTypeOwnVariable (TheProtectedType));
         else
            Item := LookupDeclarations (Name,
                                        LexTokenManager.NullString,
                                        Context,
                                        RawDict.GetProtectedTypeFirstVisibleDeclaration (TheProtectedType));
         end if;

         -- If not found then check the known discriminants
         if Item = NullSymbol then
            Item := LookupKnownDiscriminants (Name, TheProtectedType);
         end if;

         --   -- if we don't find what we are looking for in the protected type we need to look again in
         --   -- the visible declarations of the enclosing package since these are also directly visible
         --   if Item = NullSymbol then
         --      Item := LookUpPackageVisibleDeclarations (GetEnclosingPackage (VisibleScope (TheProtectedType)));
         --   end if;

         return Item;
      end LookUpProtectedTypeVisibleDeclarations;

      -----------------------------------

      function LookUpTaskTypeVisibleDeclarations (TheTaskType : Symbol) return Symbol
      --# global in Dict;
      --#        in Name;
      is
         Item : Symbol;
      begin
         TraceLexStr ("   In LookupTaskTypeVisibleDeclarations, seeking ", Name);
         -- check the know discriminants
         Item := LookupKnownDiscriminants (Name, TheTaskType);

         -- if we don't find what we are looking for in the task type we need to look again in
         -- the visible declarations of the enclosing package since these are also directly visible
         -- if Item = NullSymbol then
         --   Item := LookUpPackageVisibleDeclarations (GetEnclosingPackage (VisibleScope (TheTaskType)));
         -- end if;

         return Item;
      end LookUpTaskTypeVisibleDeclarations;

      -----------------------------------

   begin --LookupVisibleDeclarations
      TraceLexStr ("In LookupVisibleDeclarations, seeking ", Name);

      if IsPackage (TheRegion) then
         Item := LookUpPackageVisibleDeclarations (TheRegion);
      elsif IsType (TheRegion) and then TypeIsProtected (TheRegion) then
         Item := LookUpProtectedTypeVisibleDeclarations (TheRegion);
      elsif IsType (TheRegion) and then TypeIsTask (TheRegion) then
         Item := LookUpTaskTypeVisibleDeclarations (TheRegion);
      else -- error, nothing else has visible scope
         SystemErrors.FatalError (SystemErrors.InvalidSymbolTable,
                                  "Invalid LookupVisibleDeclarations");
         Item := NullSymbol; -- can't be reached, there just to avoid DFerr
      end if;

      return Item;

   end LookupVisibleDeclarations;

   ------------------------------------------------------------------------------

   procedure LookupLocalDeclarations (Name      : in     LexTokenManager.LexString;
                                      StopAt    : in     LexTokenManager.LexString;
                                      Region    : in     Symbol;
                                      Context   : in     Contexts;
                                      ScopeType : in     Visibility;
                                      Item      :    out Symbol;
                                      IsVisible :    out Boolean)
      --# global in CallingScope;
      --#        in Dict;
      --# derives IsVisible,
      --#         Item      from CallingScope,
      --#                        Context,
      --#                        Dict,
      --#                        Name,
      --#                        Region,
      --#                        ScopeType,
      --#                        StopAt;
   is

      TheItem         : Symbol;
      TheDiscriminant : SymbolDiscriminant;

      ------------------------------------------------------------------------------

      function LookupPackageBody (Name       : LexTokenManager.LexString;
                                  StopAt     : LexTokenManager.LexString;
                                  ThePackage : Symbol;
                                  Context    : Contexts;
                                  ScopeType  : Visibility) return Symbol
         --# global in Dict;
      is

         Item : Symbol;

         ------------------------------------------------------------------------------

         function LookupRefinementConstituents (Name       : LexTokenManager.LexString;
                                                ThePackage : Symbol) return Symbol
            --# global in Dict;
         is

            AbstractOwnVariables : Iterator;
            Constituent          : Symbol;

            ------------------------------------------------------------------------------

            function LookupConstituents (Name    : LexTokenManager.LexString;
                                         Subject : Symbol) return Symbol
               --# global in Dict;
            is
               Constituents : Iterator;
               Constituent  : Symbol;
            begin
               TraceLexStr ("         In LookupConstituents, seeking ", Name);

               Constituents := FirstConstituent (Subject);

               loop
                  if IsNullIterator (Constituents) then
                     Constituent := NullSymbol;
                     exit;
                  end if;
                  Constituent := CurrentSymbol (Constituents);
                  exit when GetSimpleName (Constituent) = Name and then
                     not IsOwnVariable (Constituent);
                  Constituents := NextSymbol (Constituents);
               end loop;

               return Constituent;

            end LookupConstituents;

            ------------------------------------------------------------------------------

         begin
            TraceLexStr ("      In LookupRefinementConstituents, seeking ", Name);

            AbstractOwnVariables := FirstAbstractOwnVariable (ThePackage);
            Constituent := NullSymbol;

            loop
               exit when IsNullIterator (AbstractOwnVariables);
               Constituent := LookupConstituents (Name,
                                                  CurrentSymbol (AbstractOwnVariables));
               exit when Constituent /= NullSymbol;
               AbstractOwnVariables := NextSymbol (AbstractOwnVariables);
            end loop;

            return Constituent;

         end LookupRefinementConstituents;

         ------------------------------------------------------------------------------

      begin
         TraceLexStr ("In LookupPackageBody, seeking ", Name);
         if ScopeType = Privat then
            Item := LookupDeclarations (Name,
                                        StopAt,
                                        Context,
                                        RawDict.GetPackageFirstPrivateDeclaration (ThePackage));

         else
            Item := LookupDeclarations (Name,
                                        StopAt,
                                        Context,
                                        RawDict.GetPackageFirstLocalDeclaration (ThePackage));

            if Item = NullSymbol then
               Item := LookupDeclarations (Name,
                                           StopAt,
                                           Context,
                                           RawDict.GetPackageLocalRenamingDeclarations (ThePackage));
            end if;

            if Item = NullSymbol and then Context = ProofContext then
               Item := LookupRefinementConstituents (Name, ThePackage);
            end if;

            if Item = NullSymbol then
               Item := LookupDeclarations (Name,
                                           StopAt,
                                           Context,
                                           RawDict.GetPackageFirstPrivateDeclaration (ThePackage));
            end if;
         end if;

         if Item = NullSymbol then
            Item := LookupVisibleDeclarations (Name, ThePackage, Context);
         end if;

         return Item;

      end LookupPackageBody;

      ------------------------------------------------------------------------------

      procedure LookupSubprogramBody (Name       : in     LexTokenManager.LexString;
                                      StopAt     : in     LexTokenManager.LexString;
                                      Subprogram : in     Symbol;
                                      Context    : in     Contexts;
                                      Item       :    out Symbol;
                                      IsVisible  :    out Boolean)
         --# global in Dict;
         --# derives IsVisible,
         --#         Item      from Context,
         --#                        Dict,
         --#                        Name,
         --#                        StopAt,
         --#                        Subprogram;
      is

         TheItem : Symbol;

         ------------------------------------------------------------------------------

         procedure LookupSubprogramParameters (Name       : in     LexTokenManager.LexString;
                                               Subprogram : in     Symbol;
                                               Context    : in     Contexts;
                                               Parameter  :    out Symbol;
                                               IsVisible  :    out Boolean)
            --# global in Dict;
            --# derives IsVisible from Context,
            --#                        Dict,
            --#                        Name,
            --#                        Subprogram &
            --#         Parameter from Dict,
            --#                        Name,
            --#                        Subprogram;
         is

            TheParameter : Symbol;

            ------------------------------------------------------------------------------

            procedure LookupGlobalVariables (Name       : in LexTokenManager.LexString;
                                             Subprogram : in Symbol;
                                             Context    : in Contexts;
                                             Variable   :    out Symbol;
                                             IsVisible  :    out Boolean)
               --# global in Dict;
               --# derives IsVisible from Context,
               --#                        Dict,
               --#                        Name,
               --#                        Subprogram &
               --#         Variable  from Dict,
               --#                        Name,
               --#                        Subprogram;
            is
               GlobalVariable, TheVariable : Symbol;
            begin
               TraceLexStr ("         In LookupGlobalVariables, seeking ", Name);

               GlobalVariable := RawDict.GetSubprogramFirstGlobalVariable (GetAbstraction (Subprogram,
                                                                                           LocalScope (Subprogram)),
                                                                           Subprogram);

               loop
                  if GlobalVariable = NullSymbol then
                     Variable := NullSymbol;
                     IsVisible := False;
                     exit;
                  end if;
                  TheVariable := RawDict.GetGlobalVariableVariable (GlobalVariable);
                  if GetSimpleName (TheVariable) = Name and then
                     not RawDict.GetGlobalVariablePrefixNeeded (GlobalVariable) then
                     Variable := TheVariable;
                     IsVisible := Context = ProofContext or else
                        IsDeclared (TheVariable);
                     exit;
                  end if;
                  GlobalVariable := RawDict.GetNextGlobalVariable (GlobalVariable);
               end loop;

            end LookupGlobalVariables;

            ------------------------------------------------------------------------------

         begin
            TraceLexStr ("      In LookupSubprogramParameters, seeking ", Name);

            TheParameter := RawDict.GetSubprogramFirstParameter (Subprogram);

            loop
               exit when TheParameter = NullSymbol;
               exit when RawDict.GetSubprogramParameterName (TheParameter) = Name;
               TheParameter := RawDict.GetNextSubprogramParameter (TheParameter);
            end loop;

            if TheParameter = NullSymbol then
               LookupGlobalVariables (Name,
                                      Subprogram,
                                      Context,
                                      Parameter,
                                      IsVisible);
            else
               Parameter := TheParameter;
               IsVisible := True;
            end if;

         end LookupSubprogramParameters;

         ------------------------------------------------------------------------------

      begin
         TraceLexStr ("In LookupSubprogramBody, seeking ", Name);
         TheItem := LookupDeclarations (Name,
                                        StopAt,
                                        Context,
                                        RawDict.GetSubprogramFirstDeclaration (Subprogram));
         if TheItem = NullSymbol then
            TheItem := LookupDeclarations (Name,
                                           StopAt,
                                           Context,
                                           RawDict.GetSubprogramRenamingDeclarations (Subprogram));
         end if;

         if TheItem = NullSymbol then
            LookupSubprogramParameters (Name, Subprogram, Context, TheItem, IsVisible);
            if TheItem = NullSymbol and then
              IsGenericSubprogram (Subprogram) then
               LookUpGenericFormalParameters (Name, Subprogram, Item, IsVisible);
            else
               Item := TheItem;
            end if;
         else
            Item := TheItem;
            IsVisible := True;
         end if;

      end LookupSubprogramBody;

      ------------------------------------------------------------------------------

      function LookupLoop (Name    : LexTokenManager.LexString;
                           TheLoop : Symbol) return Symbol
         --# global in Dict;
      is

         Item : Symbol;

         -----------------------------------------------------------------------------

         function LookupLoopParameter (Name    : LexTokenManager.LexString;
                                       ForLoop : Symbol) return Symbol
            --# global in Dict;
         is
            Item : Symbol;
         begin
            TraceLexStr ("   In LookupLoopParameter, seeking ", Name);

            if GetSimpleName (GetLoopParameter (ForLoop)) = Name then
               Item := GetLoopParameter (ForLoop);
            else
               Item := NullSymbol;
            end if;

            return Item;

         end LookupLoopParameter;

         ------------------------------------------------------------------------------

      begin
         TraceLexStr ("In LookupLoop, seeking ", Name);

         if IsForLoop (TheLoop) then
            Item := LookupLoopParameter (Name, TheLoop);
         else
            Item := NullSymbol;
         end if;

         return Item;

      end LookupLoop;

      --------------------------------------------------------------------------------

      function LookupType (Name    : LexTokenManager.LexString;
                           Context : Contexts;
                           TheType : Symbol) return Symbol
         --# global in Dict;
         --#        in ScopeType;
      is

         Item : Symbol;

         --------------------------------------------------------------------------------

         function LookupRecordComponents (Name    : LexTokenManager.LexString;
                                          TheType : Symbol) return Symbol
            --# global in Dict;
         is
            Component      : Symbol;
            CurrentRecord  : Symbol;
            FirstComponent : Symbol;

         begin
            TraceLexStr ("      In LookupRecordComponents, seeking ", Name);
            CurrentRecord := TheType;
            loop
               FirstComponent := RawDict.GetTypeFirstRecordComponent (CurrentRecord);
               Component := FirstComponent;

               loop
                  exit when Component = NullSymbol; -- did not find

                  exit when RawDict.GetRecordComponentName (Component) = Name; -- found
                  Component := RawDict.GetNextRecordComponent (Component);
               end loop;
               exit when Component /= NullSymbol; -- carry success out of outer loop

               -- if we get here we failed to find field in local declarations
               -- so we search in inherited fields if there are any
               exit when not RecordComponentIsInherited (FirstComponent); -- no inherited fields

               -- restart search in inherited fields
               CurrentRecord := RawDict.GetRecordComponentType (FirstComponent);
            end loop;

            return Component;

         end LookupRecordComponents;

         --------------------------------------------------------------------------------

         function LookupProtectedElements (Name    : LexTokenManager.LexString;
                                           Context : Contexts;
                                           TheType : Symbol) return Symbol
         --# global in Dict;
         is
            Item        : Symbol;
            It          : Iterator;

            function LookupRefinementConstituents (Name       : LexTokenManager.LexString;
                                                   TheType    : Symbol) return Symbol
            --# global in Dict;
            is
               Constituents : Iterator;
               Constituent  : Symbol;
            begin
               Constituents := FirstConstituent (GetProtectedTypeOwnVariable (TheType));
               loop
                  if IsNullIterator (Constituents) then
                     Constituent := NullSymbol;
                     exit;
                  end if;
                  Constituent := CurrentSymbol (Constituents);
                  exit when GetSimpleName (Constituent) = Name;
                  Constituents := NextSymbol (Constituents);
               end loop;

               return Constituent;
            end LookupRefinementConstituents;

         begin
            TraceLexStr ("      In LookupProtectedElements, seeking ", Name);
            Item := NullSymbol;
            It := FirstProtectedElement (TheType);
            while not IsNullIterator (It) loop
               if GetSimpleName (CurrentSymbol (It)) = Name then
                  Item := CurrentSymbol (It);
                  exit;
               end if;
               It := NextSymbol (It);
            end loop;

            -- As we add each protected element to the Dictionary they first get added as
            -- refinement cosntiuents (in proof context) and then as variables.  The variable
            -- add must match each up with the refinement constituent already added so we
            -- need a special look up here to find them
            if Item = NullSymbol and then Context = ProofContext then
               Item := LookupRefinementConstituents (Name, TheType);
            end if;

            -- if Item is null at this point then we have not found a match in the private part
            -- of the private type but we must also look in the visible part since things here
            -- are also directly visible
            if Item = NullSymbol then
               Item := LookupVisibleDeclarations (Name, TheType, Context);
            end if;
            return Item;
         end LookupProtectedElements;

         --------------------------------------------------------------------------------

         function LookupProtectedLocalScope (Name    : LexTokenManager.LexString;
                                             Context : Contexts;
                                             TheType : Symbol) return Symbol
         --# global in Dict;
         --#        in ScopeType;
         is
            Item : Symbol;
         begin
            TraceLexStr ("      In LookupProtectedLocalScope, seeking ", Name);
            -- Search local declarations if we are in local scope.  If this fails (or if we
            -- are in private scope), then search protected elements etc., by calling
            -- LookupProtectedElements
            Item := NullSymbol;
            if ScopeType = Local then
               -- search starts in protected body where we seek local declarations
               Item := LookupDeclarations (Name,
                                           LexTokenManager.NullString,
                                           Context,
                                           RawDict.GetProtectedTypeFirstLocalDeclaration (TheType));

               if Item = NullSymbol then
                  -- we need to look at the part of the enclosing package body which is "above" the
                  -- protected type declaration
                  Item := LookupPackageBody (Name,
                                             GetSimpleName (TheType), -- don't look below the type declaration
                                             GetEnclosingPackage (LocalScope (TheType)),
                                             Context,
                                             Local);
               end if;
            end if;

            if Item = NullSymbol then
               -- we either failed to find the item in the protected body or we skipped
               -- it completely because ScopeType = Privat on entry; so now we search
               -- the private part of the protected type
               Item := LookupProtectedElements (Name, Context, TheType);
            end if;
            return Item;
         end LookupProtectedLocalScope;

         --------------------------------------------------------------------------------

         function LookupTaskLocalScope (Name    : LexTokenManager.LexString;
                                        Context : Contexts;
                                        TheType : Symbol) return Symbol
         --# global in Dict;
         is
            Item : Symbol;

            function LookupGlobalVariables (Name       : LexTokenManager.LexString;
                                            TheType    : Symbol;
                                            Context    : Contexts) return Symbol
            --# global in Dict;
            is
               GlobalVariable, TheVariable, Result : Symbol;
            begin
               TraceLexStr ("            In LookupGlobalVariables, seeking ", Name);
               GlobalVariable := RawDict.GetTaskTypeFirstGlobalVariable (GetAbstraction (TheType,
                                                                                         LocalScope (TheType)),
                                                                         TheType);

               loop
                  if GlobalVariable = NullSymbol then
                     Result := NullSymbol;
                     exit;
                  end if;
                  TheVariable := RawDict.GetGlobalVariableVariable (GlobalVariable);
                  if GetSimpleName (TheVariable) = Name and then
                    not RawDict.GetGlobalVariablePrefixNeeded (GlobalVariable) then
                     Result := TheVariable;
                     if Context = ProgramContext and then not IsDeclared (TheVariable) then
                        Result := NullSymbol;
                     end if;
                     exit;
                  end if;
                  GlobalVariable := RawDict.GetNextGlobalVariable (GlobalVariable);
               end loop;
               return Result;
            end LookupGlobalVariables;


         begin
            -- search starts in task body where we seek local declarations
            TraceLexStr ("         In LookupTaskLocalScope, seeking ", Name);
            Item := LookupDeclarations (Name,
                                        LexTokenManager.NullString,
                                        Context,
                                        RawDict.GetTaskTypeFirstLocalDeclaration (TheType));

            if Item = NullSymbol then
               -- look up globals
               Item := LookupGlobalVariables (Name,
                                              TheType,
                                              Context);
            end if;

            --pna-- Removed this clause because it is not the role of LookUpScope to step out into the
            --pna-- enclosing package; that should be done by LookUpItem if LookUpScope fails
            --pna--
            --pna-- if Item = NullSymbol then
            --pna--    -- we need to look at the part of the enclosing package body which is "above" the
            --pna--    -- task type declaration
            --pna--    Item := LookupPackageBody (Name,
            --pna--                               GetSimpleName (TheType), -- don't look below the type declaration
            --pna--                               GetEnclosingPackage (LocalScope (TheType)),
            --pna--                               Context,
            --pna--                               Local);
            --pna-- end if;

            if Item = NullSymbol then
               Item := LookupKnownDiscriminants (Name, TheType);
            end if;

            return Item;
         end LookupTaskLocalScope;

         --------------------------------------------------------------------------------

      begin
         TraceLexStr ("      In LookupType, seeking ", Name);
         case RawDict.GetTypeDiscriminant (TheType) is
            when RecordType =>
               Item := LookupRecordComponents (Name, TheType);
            when ProtectedType =>
               Item := LookupProtectedLocalScope (Name, Context, TheType);
            when TaskType =>
               Item := LookupTaskLocalScope (Name, Context, TheType);
               --Item := LookupVisibleDeclarations (Name, TheType, Context);
            when others =>
               Item := NullSymbol;
         end case;

         return Item;

      end LookupType;

      ------------------------------------------------------------------------------

--      function LookupSubcomponents (Name         : LexTokenManager.LexString;
--                                    Subcomponent : Symbol) return Symbol
--         --# global Dict;
--     is
--         Current           : Symbol;
--         CurrentRecord     : Symbol;
--         FirstSubComponent : Symbol;
--      begin
--         CurrentRecord := Subcomponent;
--         loop
--            FirstSubComponent := CurrentRecord;
--            Current := FirstSubComponent;
--            loop
--               exit when Current = NullSymbol;
--               exit when RawDict.GetRecordComponentName
--                 (RawDict.GetSubcomponentComponent (Current)) = Name;
--               Current := RawDict.GetNextSubcomponent (Current);
--            end loop;
--            exit when Current /= NullSymbol; -- carry success out of outer loop
--
--            -- if we get here then we failed to find the desired field at the first attempt
--            -- so we move to the first inherited field (if there is one) and try again
--            exit when not RecordComponentIsInherited (RawDict.GetSubcomponentComponent (FirstSubComponent));
--            CurrentRecord := RawDict.GetSubcomponentSubcomponents (FirstSubComponent);
--         end loop;
--         return Current;
--
--      end LookupSubcomponents;

      ------------------------------------------------------------------------------

      function LookupSubcomponents (Name         : LexTokenManager.LexString;
                                    Subcomponent : Symbol;
                                    CallingScope : Scopes) return Symbol
         --# global in Dict;
      is
         Current           : Symbol;
         CurrentRecord     : Symbol;
         FirstSubComponent : Symbol;
         Found             : Boolean := False;
         IsPrivateField    : Boolean := False;
      begin
         TraceLexStr ("      In LookupSubcomponents, seeking ", Name);
         CurrentRecord := Subcomponent;
         loop
            FirstSubComponent := CurrentRecord;
            Current := FirstSubComponent;
            if not IsPrivateField then
               loop -- search fields at current depth of current record
                  exit when Current = NullSymbol;
                  Found := RawDict.GetRecordComponentName
                    (RawDict.GetSubcomponentComponent (Current)) = Name;
                  exit when Found;
                  Current := RawDict.GetNextSubcomponent (Current);
               end loop;
            end if;
            exit when Found; -- carry success out of outer loop

            -- if we get here then we failed to find the desired field at the current depth
            -- so we move to the first inherited field (if there is one and if it not a private
            -- extension) and try again
            exit when not RecordComponentIsInherited (RawDict.GetSubcomponentComponent (FirstSubComponent));
            -- no more inherited fields
            IsPrivateField := IsPrivateType (GetType (CurrentRecord), CallingScope);
            CurrentRecord := RawDict.GetSubcomponentSubcomponents (FirstSubComponent);
         end loop;
         if not Found then
            Current := NullSymbol;
         end if;
         return Current;

      end LookupSubcomponents;

      ------------------------------------------------------------------------------

   begin  --LookupLocalDeclarations
      --first part of if traps things like declaring a variable P in a procedure P and also
      --handles implicit return variables in function proof annotations.
      --
      TraceLexStr ("   In LookupLocalDeclarations, seeking ", Name);
      TheDiscriminant := RawDict.GetSymbolDiscriminant (Region);


      if GetSimpleName (Region) = Name and then
         -- Trap finding of A in X.A in nested record case
         -- Note that this case also means that seeking PT from the private part or body of PT
         -- returns the type symbol not the implicit own variable of the protected type
         TheDiscriminant /= SubcomponentSymbol and then
         TheDiscriminant /= VariableSymbol and then
         TheDiscriminant /= SubprogramParameterSymbol and then
         -- This line added as a result of CFR 1738. It allows the rest of the procedure to be
         -- used in the sole situation where we are performing lookup on a member of a record
         -- that has the same name as its own type.
         (TheDiscriminant /= TypeSymbol or else RawDict.GetTypeDiscriminant (Region) /= RecordType)
      then
         Item := Region;
         IsVisible := True;
      else

         case RawDict.GetSymbolDiscriminant (Region) is
            when PackageSymbol =>
               TheItem := LookupPackageBody (Name, StopAt, Region, Context, ScopeType);
               Item := TheItem;
               IsVisible := TheItem /= NullSymbol;
            when SubprogramSymbol =>
               LookupSubprogramBody (Name, StopAt, Region, Context, Item, IsVisible);
            when LoopSymbol =>
               TheItem := LookupLoop (Name, Region);
               Item := TheItem;
               IsVisible := TheItem /= NullSymbol;
            when ImplicitReturnVariableSymbol | QuantifiedVariableSymbol =>
               Item := NullSymbol;
               IsVisible := False;
            when SubcomponentSymbol =>
               TheItem := LookupSubcomponents (Name,
                                               RawDict.GetSubcomponentSubcomponents (Region),
                                               CallingScope);
               Item := TheItem;
               IsVisible := TheItem /= NullSymbol;
            when VariableSymbol =>
               TheItem := LookupSubcomponents (Name,
                                               RawDict.GetVariableSubComponents (Region),
                                               CallingScope);
               Item := TheItem;
               IsVisible := TheItem /= NullSymbol;
            when SubprogramParameterSymbol =>
               TheItem := LookupSubcomponents (Name,
                                               RawDict.GetSubprogramParameterSubComponents (Region),
                                               CallingScope);
               Item := TheItem;
               IsVisible := TheItem /= NullSymbol;
            when others =>
               -- must be a type (record, task or protected)
               TheItem := LookupType (Name, Context, Region);
               Item := TheItem;
               IsVisible := TheItem /= NullSymbol;
         end case;
      end if;

   end LookupLocalDeclarations;

   ------------------------------------------------------------------------------

begin
   Region := GetRegion (Scope);

   TraceLexStr ("In LookupScope, seeking ", Name);
   TraceSym ("   in ", Region, Scope);
   TraceLexStr ("   with StopAt set to ", StopAt);

   case Scope.TypeOfScope is
      when Visible =>
         TheItem := LookupVisibleDeclarations (Name, Region, Context);
         IsVisible := TheItem /= NullSymbol;
      when Local | Privat =>
         LookupLocalDeclarations (Name,
                                  StopAt,
                                  Region,
                                  Context,
                                  Scope.TypeOfScope,
                                  TheItem,
                                  IsVisible);
   end case;

   if Context = ProofContext and then IsAdaFunction (TheItem) then
      TheItem := RawDict.GetSubprogramImplicitProofFunction (GetAbstraction (TheItem,
                                                                             CallingScope),
                                                             TheItem);
   end if;

   Item := TheItem;
   TraceSym ("Found in LookUpScope ", TheItem, Scope);

end LookupScope;
