-- $Id: dictionary-lookupitem.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Dictionary)
function LookupItem (Name    : LexTokenManager.LexString;
                     Scope   : Scopes;
                     Context : Contexts) return Symbol
is

   Item           : Symbol;
   IsVisible      : Boolean;
   CurrentScope   : Scopes;
   CurrentRegion  : Symbol;
   EnclosingScope : Scopes;
   InASubprogram  : Boolean;
   StopAt         : LexTokenManager.LexString;

   ------------------------------------------------------------------------------

   procedure LookupContextClauses (Name       : in     LexTokenManager.LexString;
                                   Scope      : in     Scopes;
                                   StartPos   : in     Scopes;
                                   Context    : in     Contexts;
                                   ThePackage :    out Symbol;
                                   IsVisible  :    out Boolean)
      --# global in Dict;
      --# derives IsVisible  from Context,
      --#                         Dict,
      --#                         Name,
      --#                         Scope,
      --#                         StartPos &
      --#         ThePackage from Dict,
      --#                         Name,
      --#                         Scope;
   is

      Region          : Symbol;
      InheritClause   : Symbol;
      CurrentPackage  : Symbol;
      ThePackageLocal : Symbol;

      --------------------------------------------------------------------------------

      function HasBeenWithed (ThePackage : Symbol;
                              Scope      : Scopes) return Boolean
         --# global in Dict;
      is
         Current  : Scopes;
         Last1    : Scopes;
         Ancestor : Symbol;
         Found    : Boolean;
      begin

         Current := Scope;
         Last1    := Current;

         loop
            exit when GetRegion (Current) = GetPredefinedPackageStandard;
            exit when IsWithed (ThePackage, Current);
            Last1 := Current;
            Current := GetEnclosingScope (Current);
         end loop;

         Found := GetRegion (Current) /= GetPredefinedPackageStandard;

         if not Found and then
            Last1 /= Current and then
            IsPackage (GetRegion (Last1))
         then -- search through ancestors
            Ancestor := RawDict.GetPackageParent (GetRegion (Last1));
            loop
               exit when Ancestor = NullSymbol;
               exit when IsWithed (ThePackage, VisibleScope (Ancestor));
               Ancestor := RawDict.GetPackageParent (Ancestor);
            end loop;
            Found := Ancestor /= NullSymbol;
         end if;

         return Found;

      end HasBeenWithed;

      --------------------------------------------------------------------------------

      function IsDirectlyVisible (ThePackageOrGeneric : Symbol;
                                  Scope               : Scopes) return Boolean
         --# global in Dict;
      is
         TheParent  : Symbol;
         LibPackage : Symbol;
         Result     : Boolean;
      begin
         if IsGenericSubprogram (ThePackageOrGeneric) then
            Result := True;
         else
            TheParent := RawDict.GetPackageParent (ThePackageOrGeneric);
            if TheParent = NullSymbol then
               Result := True;
            elsif not IsPackage (GetRegion (Scope)) then
               Result := False;
            else  --  ThePackage is a child and Scope is in another package
                  -- OK if Scope is (possibly embedded within) ThePackage's parent
                  -- or a descendent of the parent
               LibPackage := GetLibraryPackage (Scope);
               Result := (LibPackage = TheParent or else
                            IsProperDescendent (LibPackage, TheParent));
            end if;
         end if;

         return Result;

      end IsDirectlyVisible;

      --------------------------------------------------------------------------------

   begin
      TraceLexStr ("   In LookupContextClauses, seeking ", Name);

      Region := GetRegion (Scope);

      if IsPackage (Region) then
         InheritClause := RawDict.GetPackageInheritClauses (Region);
      elsif IsMainProgram (Region) then
         InheritClause := Dict.Main.InheritClauses;
      elsif IsType (Region) and then TypeIsProtected (Region) then
         Region := GetEnclosingPackage (Scope);
         InheritClause := RawDict.GetPackageInheritClauses (Region);
      else
         InheritClause := NullSymbol;
      end if;

      loop
         if InheritClause = NullSymbol then
            ThePackageLocal := NullSymbol;
            IsVisible := False;
            exit;
         end if;
         CurrentPackage := RawDict.GetContextClausePackage (InheritClause);
         if GetSimpleName (CurrentPackage) = Name
            and then IsDirectlyVisible (CurrentPackage, Scope)
         then
            ThePackageLocal := CurrentPackage;
            IsVisible := Context = ProofContext or else
              (IsPackage (CurrentPackage) and then IsEmbeddedPackage (CurrentPackage)) or else
               IsLocal (Scope, LocalScope (CurrentPackage)) or else
              (IsPackage (Region) and then IsPackage (CurrentPackage) and then
                IsProperDescendent (GetLibraryPackage (Scope),
                                    CurrentPackage))   or else
              HasBeenWithed (CurrentPackage, StartPos);
            exit;
         end if;
         InheritClause := RawDict.GetNextContextClause (InheritClause);
      end loop;
      TraceSym ("   found in  LookupContextClauses ", ThePackageLocal, Scope);
      ThePackage := ThePackageLocal;
   end LookupContextClauses;

   ------------------------------------------------------------------------------

   procedure LookupPrivateChildren (
                                    Name       : in     LexTokenManager.LexString;
                                    Region     : in     Symbol;
                                    Scope      : in     Scopes;
                                    Context    : in     Contexts;
                                    ThePackage :    out Symbol;
                                    IsVisible  :    out Boolean)
      --# global in Dict;
      --# derives IsVisible,
      --#         ThePackage from Context,
      --#                         Dict,
      --#                         Name,
      --#                         Region,
      --#                         Scope;
   is
      Result          : Symbol;
      CurrentPackage  : Symbol;

      -----------------------------------------

      function CheckIsWithed (PackSym : Symbol;
                              Scope   : Scopes;
                              Context : Contexts) return Symbol
         --# global in Dict;
      is
         Current : Scopes;
         Result  : Symbol;
      begin
         if Context = ProofContext then
            Result := PackSym;
         else
            Current := Scope;
            loop
               exit when GetRegion (Current) = GetPredefinedPackageStandard;
               exit when IsWithed (PackSym, Current);
               Current := GetEnclosingScope (Current);
            end loop;

            if GetRegion (Current) = GetPredefinedPackageStandard then
               Result := NullSymbol;
            else
               Result := PackSym;
            end if;
         end if;
         return Result;
      end CheckIsWithed;

      -----------------

   begin --LookUpPrivateChildren
      Result := NullSymbol;
      if IsPackage (Region) then
         CurrentPackage := RawDict.GetPackageFirstChild (Region, PrivateChild);
         loop
            exit when CurrentPackage = NullSymbol;

            if GetSimpleName (CurrentPackage) = Name then
               Result := CurrentPackage;
               exit;
            end if;

            CurrentPackage := RawDict.GetPackageSibling (CurrentPackage);
         end loop;

         if Result /= NullSymbol then
            Result := CheckIsWithed (Result, Scope, Context);
         end if;
      end if;

      ThePackage := Result;
      IsVisible := Result /= NullSymbol;

   end LookupPrivateChildren;

   ------------------------------------------------------------------------------

begin
   CurrentScope := Scope;
   CurrentRegion := GetRegion (CurrentScope);

   TraceMsg ("--------------------------------------------------------------------------");
   TraceLexStr ("In LookupItem, seeking ", Name);
   TraceSym ("   in ", CurrentRegion, Scope);

   loop
      LookupScope (Name,
                   LexTokenManager.NullString,
                   CurrentScope,
                   CurrentScope,
                   Context,
                   Item,
                   IsVisible);
      exit when Item /= NullSymbol;
      exit when IsCompilationUnit (CurrentRegion);
      exit when IsProtectedType (CurrentRegion);
      exit when IsTaskType (CurrentRegion);
      CurrentScope := GetEnclosingScope (CurrentScope);
      CurrentRegion := GetRegion (CurrentScope);
   end loop;

   if Item = NullSymbol then

      InASubprogram := IsSubprogram (CurrentRegion) or else
        IsTaskType (CurrentRegion);

      loop
         LookupPrivateChildren (Name,
                                CurrentRegion,
                                Scope,
                                Context,
                                Item,
                                IsVisible);
         exit when Item /= NullSymbol;

         LookupContextClauses (Name,
                               CurrentScope,
                               Scope,
                               Context,
                               Item,
                               IsVisible);
         exit when Item /= NullSymbol;
         exit when IsPackage (CurrentRegion);
         exit when IsMainProgram (CurrentRegion);
         EnclosingScope := GetEnclosingScope (CurrentScope);
         if IsSubprogram (CurrentRegion) and then
            IsLocalScope (EnclosingScope) then
            StopAt := GetSimpleName (CurrentRegion);
         elsif IsProtectedType (CurrentRegion) and then
           IsLocalScope (EnclosingScope) then
            StopAt := GetSimpleName (CurrentRegion);
         elsif IsTaskType (CurrentRegion) and then
           IsLocalScope (EnclosingScope) then
            StopAt := GetSimpleName (CurrentRegion);
         else
            StopAt := LexTokenManager.NullString;
         end if;
         LookupScope (Name,
                      StopAt,
                      EnclosingScope,
                      EnclosingScope,
                      Context,
                      Item,
                      IsVisible);
         if InASubprogram and then IsVariable (Item) then
            IsVisible := False;
         end if;
         exit when Item /= NullSymbol;
         CurrentScope := EnclosingScope;
         CurrentRegion := GetRegion (CurrentScope);
      end loop;

      if Item = NullSymbol and then
         CurrentRegion /= GetPredefinedPackageStandard then
         LookupScope (Name,
                      LexTokenManager.NullString,
                      PredefinedScope,
                      PredefinedScope,
                      Context,
                      Item,
                      IsVisible);
      end if;

   end if;

   if not IsVisible then
      Item := NullSymbol;
   end if;

   TraceSym ("Found in LookUpItem ", Item, Scope);
   TraceMsg ("--------------------------------------------------------------------------");
   return Item;

end LookupItem;
