-- $Id: dictionary-iscallable.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

----------------------------------------------------------------------------------------
-- Checks the following:
-- if not Dotted then SubProgSym has body and is not being called from within
-- itself
-- or, if not dotted but subprogram is declared remotely then it must be an
-- an inherited op associated with a tagged type and that is ok
-- if dotted then if proc in enclosing package then proc has body
--   else if
-- package embedded in something then package has body
--                             or, if package does not have body, does
--                             subprogram have body (might have pragma
--                             import/interface)
--   else if called from [descendent of] private child of proc's package
--        then error
--   else its Ok
--------------------------------------------------------------------------------

separate (Dictionary)
function IsCallable (Subprogram   : Symbol;
                     PrefixNeeded : Boolean;
                     Scope        : Scopes) return Boolean
is

   ThePackage : Symbol;
   Result     : Boolean;

   --------------------------------------------------------------------------------

   function BodyIsVisible (CompilationUnit : Symbol;
                           Scope           : Scopes) return Boolean
      --# global in Dict;
   is

      Region         : Symbol;
      StopAt         : Symbol;
      TheBody        : Symbol;
      Found          : Boolean;
      CurrentScope   : Scopes;
      EnclosingScope : Scopes;

      --------------------------------------------------------------------------------

      function GetBody (CompilationUnit : Symbol) return Symbol
         --# global in Dict;
      is
         TheBody : Symbol;
      begin

         case RawDict.GetSymbolDiscriminant (CompilationUnit) is
            when PackageSymbol =>
               TheBody := RawDict.GetPackageBody (CompilationUnit);
            when TypeSymbol =>
               if RawDict.GetTypeDiscriminant (CompilationUnit) = ProtectedType then
                  TheBody := RawDict.GetProtectedTypeBody (CompilationUnit);
               else -- Task
                  TheBody := RawDict.GetTaskTypeBody (CompilationUnit);
               end if;
            when SubprogramSymbol =>
               TheBody := RawDict.GetSubprogramBody (CompilationUnit);
            when others =>
               TheBody := NullSymbol; -- should be non-executable
         end case;

         return TheBody;

      end GetBody;

      --------------------------------------------------------------------------------

      function BodyIsDefined (TheBody : Symbol;
                              Scope   : Scopes;
                              StopAt  : Symbol) return Boolean
         --# global in Dict;
      is

         Found : Boolean;

         --------------------------------------------------------------------------------

         function LookupDeclarations (TheBody, Head, StopAt : Symbol) return Boolean
            --# global in Dict;
         is
            Declaration : Symbol;
         begin

            Declaration := Head;

            loop
               exit when Declaration = NullSymbol or else
                  Declaration = StopAt or else
                  Declaration = TheBody;
               Declaration := RawDict.GetNextDeclaration (Declaration);
            end loop;

            return Declaration = TheBody;

         end LookupDeclarations;

         ------------------------------------------------------------------------------

         function LookupLocalDeclarations (TheBody : Symbol;
                                           Region  : Symbol;
                                           StopAt  : Symbol) return Boolean
            --# global in Dict;
         is
            Found : Boolean;
         begin
            case RawDict.GetSymbolDiscriminant (Region) is
               when PackageSymbol =>
                  Found := LookupDeclarations (TheBody,
                                               RawDict.GetPackageFirstLocalDeclaration (Region),
                                               StopAt);
               when SubprogramSymbol =>
                  Found := LookupDeclarations (TheBody,
                                               RawDict.GetSubprogramFirstDeclaration (Region),
                                               StopAt);
               when TypeSymbol =>
                  -- must be protected or task type since these are the only types that could contain
                  -- a subprogram call.
                  if RawDict.GetTypeDiscriminant (Region) = ProtectedType then
                     Found := LookupDeclarations (TheBody,
                                                  RawDict.GetProtectedTypeFirstLocalDeclaration (Region),
                                                  StopAt);
                  elsif RawDict.GetTypeDiscriminant (Region) = TaskType then
                     Found := LookupDeclarations (TheBody,
                                                  RawDict.GetTaskTypeFirstLocalDeclaration (Region),
                                                  StopAt);
                  else
                     SystemErrors.FatalError (SystemErrors.OtherInternalError,
                                              "Seeking local declaration in a type which is not a task or protected type");
                     Found := False; -- unreachable, to avoid DF error
                  end if;
               when others =>
                  Found := False;
            end case;

            return Found;

         end LookupLocalDeclarations;

         ------------------------------------------------------------------------------

      begin
         case Scope.TypeOfScope is
            when Visible | Privat =>
               Found := False;
            when Local =>
               Found := LookupLocalDeclarations (TheBody, GetRegion (Scope), StopAt);
         end case;

         return Found;

      end BodyIsDefined;

      --------------------------------------------------------------------------------

   begin
      TheBody := GetBody (CompilationUnit);

      if TheBody = NullSymbol then
         Found := False;
      else

         Found := BodyIsDefined (TheBody, Scope, NullSymbol);

         if not Found then
            CurrentScope := Scope;

            loop
               exit when GetRegion (CurrentScope) = GetPredefinedPackageStandard;
               EnclosingScope := GetEnclosingScope (CurrentScope);
               Region := GetRegion (CurrentScope);
               if IsCompilationUnit (Region) then
                  StopAt := GetBody (Region);
               elsif IsType (Region) then -- Task or Protected  body subunit
                  StopAt := GetBody (Region);
               else
                  StopAt := NullSymbol;
               end if;
               Found := BodyIsDefined (TheBody, EnclosingScope, StopAt);
               exit when Found;
               CurrentScope := EnclosingScope;
            end loop;

         end if;

      end if;

      return Found;

   end BodyIsVisible;

   --------------------------------------------------------------------------------

   function DirectRecursion (Subprogram : Symbol; Scope : Scopes) return Boolean
      --# global in Dict;
   is
      CurrentScope  : Scopes;
      CurrentRegion : Symbol;
   begin

      CurrentScope := Scope;

      loop
         CurrentRegion := GetRegion (CurrentScope);
         exit when CurrentRegion = Subprogram;
         exit when CurrentRegion = GetPredefinedPackageStandard;
         CurrentScope := GetEnclosingScope (CurrentScope);
      end loop;

      return CurrentRegion = Subprogram;

   end DirectRecursion;

   --------------------------------------------------------------------------------

   function IsInheritedOperation (Subprogram : Symbol;
                                  Scope      : Scopes) return Boolean
   --# global in Dict;
   is
   begin
      -- a subprogram denoted by a simple name must be inherited if the
      -- library package in which it is declared is not the same as the
      -- package associated with the scope from which we are looking
      return GetLibraryPackage (GetScope (Subprogram)) /= GetLibraryPackage (Scope);
   end IsInheritedOperation;

   --------------------------------------------------------------------------------

   function SelectProtectedBody (Subprogram : Symbol;
                                 Scope      : Scopes) return Symbol
   --# global in Dict;
   is
      Result         : Symbol;
      DeclaredRegion : Symbol;
   begin
      -- If the subprogram is declared inside a protected type and we are not calling from inside the protected
      -- body itself then it is the body of the type we need to find not the body of the subprgoram
      -- itself.  Otherwise we return the Subprogram unchanged.
      Result := Subprogram;
      DeclaredRegion := GetRegion (GetScope (Subprogram));
      if IsProtectedType (DeclaredRegion) and then
        not IsLocal (Scope, LocalScope (DeclaredRegion)) then
         Result := DeclaredRegion;
      end if;
      return Result;
   end SelectProtectedBody;

   --------------------------------------------------------------------------------

begin
   SystemErrors.RTAssert (IsSubprogram (Subprogram),
                          SystemErrors.PreconditionFailure,
                          "In call to IsCallable");

   if PrefixNeeded or else IsRenamed (Subprogram, Scope) then
      ThePackage := GetRegion (GetScope (Subprogram));
      if IsLocal (Scope, LocalScope (ThePackage)) then
         Result := BodyIsVisible (Subprogram, Scope);
      elsif IsEmbeddedPackage (ThePackage) then
         Result := BodyIsVisible (ThePackage, Scope) or else
           BodyIsVisible (Subprogram, LocalScope (Subprogram));
      elsif IsDescendentOfPrivateChild (GetLibraryPackage (Scope), ThePackage) then
         Result := False;
      else
         Result := True;
      end if;
   else
      Result := (BodyIsVisible (SelectProtectedBody (Subprogram, Scope), Scope) and then
                   not DirectRecursion (Subprogram, Scope)) or else
        IsInheritedOperation (Subprogram, Scope);
   end if;

   return Result;

end IsCallable;
