-- $Id: declarations.adb 12812 2009-03-27 15:30:32Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with Pile,
     Clists,
     CStacks,
     Lists,
     EStrings,
     ExaminerConstants,
     Maths,
     SPSymbols,
     SystemErrors;

with Debug;

use type SPSymbols.SPSymbol;
use type EStrings.T;

package body Declarations
--# own State is UsedSymbols,
--#              AttributeList,
--#              ProcedureExportList,
--#              ReturnSymbol,
--#              BitwiseOpList,
--#              RootIntegerUsed;
is

   UsedSymbols         : Cells.Cell := Cells.Null_Cell;
   AttributeList       : Cells.Cell := Cells.Null_Cell;
   BitwiseOpList       : Cells.Cell := Cells.Null_Cell;
   ProcedureExportList : Cells.Cell := Cells.Null_Cell;
   ReturnSymbol        : Cells.Cell := Cells.Null_Cell;
   RootIntegerUsed     : Boolean    := False;

   ----------------------------------------------------------------

   procedure StartProcessing (Heap : in out Cells.Heap_Record)
   --# global in out Statistics.TableUsage;
   --#           out AttributeList;
   --#           out BitwiseOpList;
   --#           out ProcedureExportList;
   --#           out ReturnSymbol;
   --#           out RootIntegerUsed;
   --#           out UsedSymbols;
   --# derives AttributeList,
   --#         BitwiseOpList,
   --#         Heap,
   --#         ProcedureExportList   from Heap &
   --#         ReturnSymbol,
   --#         RootIntegerUsed,
   --#         UsedSymbols           from  &
   --#         Statistics.TableUsage from *,
   --#                                    Heap;
   is
   begin
      UsedSymbols  := Cells.Null_Cell;
      ReturnSymbol := Cells.Null_Cell;
      Cells.Create_Cell (Heap, AttributeList);
      Cells.Create_Cell (Heap, ProcedureExportList);
      Cells.Create_Cell (Heap, BitwiseOpList);
      RootIntegerUsed := False;
   end StartProcessing;

   ----------------------------------------------------------------

   -- New check to ensure that we don't end up with Ada and Implicit
   -- proof functions in list of used symbols
   procedure Add (Heap   : in out Cells.Heap_Record;
                  Symbol : in     Dictionary.Symbol)
   --# global in     Dictionary.Dict;
   --#        in out Statistics.TableUsage;
   --#        in out UsedSymbols;
   --# derives Heap,
   --#         Statistics.TableUsage,
   --#         UsedSymbols           from *,
   --#                                    Dictionary.Dict,
   --#                                    Heap,
   --#                                    Symbol,
   --#                                    UsedSymbols;
   is
   begin
      if not Dictionary.IsQuantifiedVariable (Symbol) then
         Pile.Insert (Heap, Symbol, Cells.Null_Cell, UsedSymbols);
      end if;
   end Add;

   ----------------------------------------------------------------

   procedure AddAttribute (Heap     : in out Cells.Heap_Record;
                           TickCell : in     Cells.Cell)
   --# global in     AttributeList;
   --#        in     Dictionary.Dict;
   --#        in out Statistics.TableUsage;
   --# derives Heap,
   --#         Statistics.TableUsage from *,
   --#                                    AttributeList,
   --#                                    Dictionary.Dict,
   --#                                    Heap,
   --#                                    TickCell;
   is
      InsertPtr : Cells.Cell;

      ------------------------------------

      function HasBase (TickCell : Cells.Cell) return Boolean
      --# global in Heap;
      is
      begin
         return Cells.Get_Kind (Heap, Cells.Get_A_Ptr (Heap, TickCell)) = Cells.Op;
      end HasBase;

      ------------------------------------

      function PrefixVal (TickCell : Cells.Cell) return Integer
      --# global in Heap;
      is
         PrefixCell : Cells.Cell;
      begin
         PrefixCell := Cells.Get_A_Ptr (Heap, TickCell);
         if Cells.Get_Kind (Heap, PrefixCell) = Cells.Op then -- Base found
            PrefixCell := Cells.Get_A_Ptr (Heap, PrefixCell);
         end if;
         return Cells.Get_Natural_Value (Heap, PrefixCell);
      end PrefixVal;

      ------------------------------------

      function IsEqual (TickCell1, TickCell2 : Cells.Cell) return Boolean
      --# global in Heap;
      is
      begin
         return (
                 (PrefixVal (TickCell1) = PrefixVal (TickCell2)) and then
                 (HasBase (TickCell1) = HasBase (TickCell2))     and then
                 (Cells.Get_Lex_Str (Heap, Cells.Get_B_Ptr (Heap, TickCell1)) =
                  Cells.Get_Lex_Str (Heap, Cells.Get_B_Ptr (Heap, TickCell2))) and then
                 (Cells.Get_Assoc_Var (Heap, Cells.Get_B_Ptr (Heap, TickCell1)) =
                  Cells.Get_Assoc_Var (Heap, Cells.Get_B_Ptr (Heap, TickCell2)))
                 );
      end IsEqual;

      ------------------------------------

      function IsGreater (TickCell1, TickCell2 : Cells.Cell) return Boolean
      --# global in Dictionary.Dict;
      --#        in Heap;
      is
         Result : Boolean;
         Val1,
         Val2   : Integer;
         ValSym1,
         ValSym2 : Dictionary.Symbol;
      begin
         if HasBase (TickCell1) = HasBase (TickCell2) then
            Val1 := PrefixVal (TickCell1);
            Val2 := PrefixVal (TickCell2);
            if Val1 /= Val2 then
               Result := Val1 > Val2;
            else
               Val1 := LexTokenManager.LexStringRef
                     (Cells.Get_Lex_Str (Heap, Cells.Get_B_Ptr (Heap, TickCell1)));
               Val2 := LexTokenManager.LexStringRef
                     (Cells.Get_Lex_Str (Heap, Cells.Get_B_Ptr (Heap, TickCell2)));
               if Val1 /= Val2 then
                  Result := Val1 > Val2;
               else
                  ValSym1 := Cells.Get_Assoc_Var (Heap, Cells.Get_B_Ptr (Heap, TickCell1));
                  ValSym2 := Cells.Get_Assoc_Var (Heap, Cells.Get_B_Ptr (Heap, TickCell2));
                  Val1 := LexTokenManager.LexStringRef (Dictionary.GetSimpleName (ValSym1));
                  Val2 := LexTokenManager.LexStringRef (Dictionary.GetSimpleName (ValSym2));
                  Result := Val1 > Val2;
               end if;
            end if;
         else
            Result := HasBase (TickCell1);
         end if;
         return Result;
      end IsGreater;

      ------------------------------------

      function NextTickCell (CurrentLink : Cells.Cell) return Cells.Cell
      --# global in Heap;
      --pre not Cells.Is_Null_Cell (Cells.Get_A_Ptr (Heap, CurrentLink));
      is
      begin
         return Cells.Get_C_Ptr (Heap, Cells.Get_A_Ptr (Heap, CurrentLink));
      end NextTickCell;

      ------------------------------------

      procedure InsertAfter (InsertPoint  : in Cells.Cell;
                             TickCell     : in Cells.Cell)
      --# global in out Heap;
      --#        in out Statistics.TableUsage;
      --# derives Heap                  from *,
      --#                                    InsertPoint,
      --#                                    TickCell &
      --#         Statistics.TableUsage from *,
      --#                                    Heap;
      is
         NewLink : Cells.Cell;
      begin
         Cells.Create_Cell (Heap, NewLink);
         Cells.Set_A_Ptr (Heap, NewLink, Cells.Get_A_Ptr (Heap, InsertPoint));
         Cells.Set_A_Ptr (Heap, InsertPoint, NewLink);
         Cells.Set_C_Ptr (Heap, NewLink, TickCell);
      end InsertAfter;

      ------------------------------------

   begin --AddAttribute
      if Cells.Get_Lex_Str (Heap, Cells.Get_B_Ptr (Heap, TickCell)) /=
         LexTokenManager.BaseToken then

         InsertPtr := AttributeList;
         loop
            if Cells.Is_Null_Cell (Cells.Get_A_Ptr (Heap, InsertPtr)) then
               InsertAfter (InsertPtr, TickCell);
               exit;
            end if;

            if IsGreater (TickCell, NextTickCell (InsertPtr)) then
               InsertAfter (InsertPtr, TickCell);
               exit;
            end if;

            if IsEqual (TickCell, NextTickCell (InsertPtr)) then
               exit;
            end if;

            InsertPtr := Cells.Get_A_Ptr (Heap, InsertPtr);
         end loop;
      end if;
   end AddAttribute;

   ----------------------------------------------------------------

   procedure AddBitwiseOp (Heap   : in out Cells.Heap_Record;
                             OpCell : in     Cells.Cell)
   --# global in     BitwiseOpList;
   --#        in     Dictionary.Dict;
   --#        in out Statistics.TableUsage;
   --#        in out UsedSymbols;
   --# derives Heap,
   --#         Statistics.TableUsage from *,
   --#                                    BitwiseOpList,
   --#                                    Dictionary.Dict,
   --#                                    Heap,
   --#                                    OpCell,
   --#                                    UsedSymbols &
   --#         UsedSymbols           from *,
   --#                                    Dictionary.Dict,
   --#                                    Heap,
   --#                                    OpCell;
   is
      InsertPtr : Cells.Cell;

      function IsEqual (OpCell1, OpCell2 : Cells.Cell) return Boolean
      --# global in Heap;
      is
      begin
         return ((Cells.Get_Natural_Value (Heap, OpCell1) =
                  Cells.Get_Natural_Value (Heap, OpCell2)) and then
                 (Cells.Get_Op_Symbol (Heap, OpCell1) =
                  Cells.Get_Op_Symbol (Heap, OpCell2)));
      end IsEqual;

      ------------------------------------

      function IsGreater (OpCell1, OpCell2 : Cells.Cell) return Boolean
      --# global in Heap;
      is
         Result : Boolean;
         Val1,
         Val2   : Integer;
      begin
         Val1 := Cells.Get_Natural_Value (Heap, OpCell1);
         Val2 := Cells.Get_Natural_Value (Heap, OpCell2);
         if Val1 = Val2 then
            Result := Cells.Get_Op_Symbol (Heap, OpCell1) > Cells.Get_Op_Symbol (Heap, OpCell2);
         else
            Result := Val1 > Val2;
         end if;
         return Result;
      end IsGreater;

      ------------------------------------

      function NextOpCell (CurrentLink : Cells.Cell) return Cells.Cell
      --# global in Heap;
      --pre not Cells.Is_Null_Cell (Cells.Get_A_Ptr (Heap, CurrentLink));
      is
      begin
         return Cells.Get_C_Ptr (Heap, Cells.Get_A_Ptr (Heap, CurrentLink));
      end NextOpCell;

      ------------------------------------

      procedure InsertAfter (InsertPoint : in Cells.Cell;
                             OpCell      : in Cells.Cell)
      --# global in out Heap;
      --#        in out Statistics.TableUsage;
      --# derives Heap                  from *,
      --#                                    InsertPoint,
      --#                                    OpCell &
      --#         Statistics.TableUsage from *,
      --#                                    Heap;
      is
         NewLink : Cells.Cell;
      begin
         Cells.Create_Cell (Heap, NewLink);
         Cells.Set_A_Ptr (Heap, NewLink, Cells.Get_A_Ptr (Heap, InsertPoint));
         Cells.Set_A_Ptr (Heap, InsertPoint, NewLink);
         Cells.Set_C_Ptr (Heap, NewLink, OpCell);
      end InsertAfter;

      ------------------------------------

   begin --AddBitwiseOp
      Add (Heap, Cells.Get_Symbol_Value (Heap, OpCell)); -- To get a type declaration
      InsertPtr := BitwiseOpList;
      loop
         if Cells.Is_Null_Cell (Cells.Get_A_Ptr (Heap, InsertPtr)) then
            InsertAfter (InsertPtr, OpCell);
            exit;
         end if;

         if IsGreater (OpCell, NextOpCell (InsertPtr)) then
            InsertAfter (InsertPtr, OpCell);
            exit;
         end if;

         if IsEqual (OpCell, NextOpCell (InsertPtr)) then
            exit;
         end if;

         InsertPtr := Cells.Get_A_Ptr (Heap, InsertPtr);
      end loop;
   end AddBitwiseOp;

   ----------------------------------------------------------------

   procedure AddProcedureExport (Heap       : in out Cells.Heap_Record;
                                 ExportCell : in     Cells.Cell)
   --# global in     Dictionary.Dict;
   --#        in     ProcedureExportList;
   --#        in out Statistics.TableUsage;
   --#        in out UsedSymbols;
   --# derives Heap,
   --#         Statistics.TableUsage,
   --#         UsedSymbols           from *,
   --#                                    Dictionary.Dict,
   --#                                    ExportCell,
   --#                                    Heap,
   --#                                    ProcedureExportList,
   --#                                    UsedSymbols;
   is
      NewLink : Cells.Cell;
      function AlreadyPresent return Boolean
      --# global in ExportCell;
      --#        in Heap;
      --#        in ProcedureExportList;
      is
         CurrentCell : Cells.Cell;
         Found       : Boolean;
      begin
         Found := False;
         CurrentCell := Cells.Get_A_Ptr (Heap, ProcedureExportList);
         while CurrentCell /= Cells.Null_Cell and not Found loop
            Found := (Cells.Get_Symbol_Value (Heap, CurrentCell) = Cells.Get_Symbol_Value (Heap, ExportCell)) and then
                     (Cells.Get_Lex_Str (Heap, CurrentCell) = Cells.Get_Lex_Str (Heap, ExportCell));
            CurrentCell := Cells.Get_A_Ptr (Heap, CurrentCell);
         end loop;

         return Found;
      end AlreadyPresent;

   begin
      if not AlreadyPresent then
         Add (Heap, Cells.Get_Symbol_Value (Heap, ExportCell)); -- Ensure we get a type decl

         Cells.Create_Cell (Heap, NewLink);
         -- put in linked list
         Cells.Set_A_Ptr (Heap, NewLink, Cells.Get_A_Ptr (Heap, ProcedureExportList));
         Cells.Set_A_Ptr (Heap, ProcedureExportList, NewLink);
         -- Copy in values to new list element
         Cells.Set_Symbol_Value (Heap,
                               NewLink,
                               Cells.Get_Symbol_Value (Heap, ExportCell));
         Cells.Set_Lex_Str (Heap,
                             NewLink,
                             Cells.Get_Lex_Str (Heap, ExportCell));
      end if;

   end AddProcedureExport;

   ----------------------------------------------------------------

   procedure AddReturnVar (Heap          : in out Cells.Heap_Record;
                           ReturnVarCell : in     Cells.Cell)
   --# global in     Dictionary.Dict;
   --#        in out Statistics.TableUsage;
   --#        in out UsedSymbols;
   --#           out ReturnSymbol;
   --# derives Heap,
   --#         Statistics.TableUsage,
   --#         UsedSymbols           from *,
   --#                                    Dictionary.Dict,
   --#                                    Heap,
   --#                                    ReturnVarCell,
   --#                                    UsedSymbols &
   --#         ReturnSymbol          from ReturnVarCell;
   is
   begin
      Add (Heap, Cells.Get_Symbol_Value (Heap, ReturnVarCell)); -- To get a type declaration
      ReturnSymbol := ReturnVarCell;
   end AddReturnVar;

   ----------------------------------------------------------------

   procedure AddUseOfRootInteger
   --# global out RootIntegerUsed;
   --# derives RootIntegerUsed from ;
   is
   begin
      RootIntegerUsed := True;
   end AddUseOfRootInteger;


   ----------------------------------------------------------------


   procedure FindCellContentsDeclarations (Heap        : in out Cells.Heap_Record;
                                           CellName    : in     Cells.Cell)
   --# global in     AttributeList;
   --#        in     BitwiseOpList;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     ProcedureExportList;
   --#        in out ReturnSymbol;
   --#        in out RootIntegerUsed;
   --#        in out Statistics.TableUsage;
   --#        in out UsedSymbols;
   --# derives Heap,
   --#         Statistics.TableUsage from *,
   --#                                    AttributeList,
   --#                                    BitwiseOpList,
   --#                                    CellName,
   --#                                    Dictionary.Dict,
   --#                                    Heap,
   --#                                    LexTokenManager.StringTable,
   --#                                    ProcedureExportList,
   --#                                    UsedSymbols &
   --#         ReturnSymbol,
   --#         RootIntegerUsed       from *,
   --#                                    CellName,
   --#                                    Heap &
   --#         UsedSymbols           from *,
   --#                                    CellName,
   --#                                    Dictionary.Dict,
   --#                                    Heap,
   --#                                    LexTokenManager.StringTable,
   --#                                    ProcedureExportList;
   is
      IdRef     : Dictionary.Symbol;

      -------------------------------------------------------

      procedure FindManifestConstantCellDeclarations (CellName : in Cells.Cell)
      --# global in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in out Heap;
      --#        in out Statistics.TableUsage;
      --#        in out UsedSymbols;
      --# derives Heap,
      --#         Statistics.TableUsage,
      --#         UsedSymbols           from *,
      --#                                    CellName,
      --#                                    Dictionary.Dict,
      --#                                    Heap,
      --#                                    LexTokenManager.StringTable,
      --#                                    UsedSymbols;
      is
         ExString : EStrings.T;
         LStr     : LexTokenManager.LexString;
      begin --FindManifestConstantCell
         LStr := Cells.Get_Lex_Str (Heap, CellName);
         LexTokenManager.LexStringToString (LStr,
                                            ExString);
         if ExString.Content (1)    = ''' then --character literal
            Add (Heap, Dictionary.GetPredefinedCharacterType);

         elsif ExString.Content (1) = '"' then --string literal
            Add (Heap, Dictionary.GetPredefinedCharacterType);
            Add (Heap, Dictionary.GetPredefinedStringType);

         else -- should be a numeric
            null;
         end if;
      end FindManifestConstantCellDeclarations;

      -------------------------------------------------------

   begin -- FindCellContentsDeclarations
      case Cells.Get_Kind (Heap, CellName) is

         when Cells.Manifest_Const          =>
            FindManifestConstantCellDeclarations (CellName);

         when Cells.Op                     =>
            if Cells.Get_Op_Symbol (Heap, CellName) = SPSymbols.apostrophe then
               AddAttribute (Heap, CellName);
            end if;

         when Cells.Return_Var =>
            AddReturnVar (Heap, CellName);

         when Cells.Named_Const =>
            IdRef := Cells.Get_Symbol_Value (Heap, CellName);
            Add (Heap, IdRef);

         when Cells.Declared_Function |
           Cells.Proof_Function       |
           Cells.Modified_Op          |
           Cells.Reference            |
           Cells.Constraining_Index   |
           Cells.Fixed_Var            |
           Cells.Mk_Aggregate         |
           Cells.Unconstrained_Attribute_Prefix =>

            IdRef := Cells.Get_Symbol_Value (Heap, CellName);

            if Cells.Get_Kind (Heap, CellName) = Cells.Mk_Aggregate and then
               Dictionary.IsSubtype (IdRef) then
               IdRef := Dictionary.GetRootType (IdRef);
            end if;

            Add (Heap, IdRef);

         when Cells.Root_Integer =>
            AddUseOfRootInteger;

         when Cells.Bitwise_Op =>
            if Dictionary.TypeIsArray (Cells.Get_Symbol_Value (Heap, CellName)) then
               AddBitwiseOp (Heap, CellName);
            end if;

         when Cells.Procedure_Export =>
            AddProcedureExport (Heap, CellName);

         when others =>
            null;

      end case;
   end FindCellContentsDeclarations;

   -------------------------------------------------------------------------

   procedure FindDagDeclarations (Heap        : in out Cells.Heap_Record;
                                  Root        : in     Cells.Cell)

   --# global in     AttributeList;
   --#        in     BitwiseOpList;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     ProcedureExportList;
   --#        in out ReturnSymbol;
   --#        in out RootIntegerUsed;
   --#        in out Statistics.TableUsage;
   --#        in out UsedSymbols;
   --# derives Heap,
   --#         ReturnSymbol,
   --#         RootIntegerUsed,
   --#         Statistics.TableUsage,
   --#         UsedSymbols           from *,
   --#                                    AttributeList,
   --#                                    BitwiseOpList,
   --#                                    Dictionary.Dict,
   --#                                    Heap,
   --#                                    LexTokenManager.StringTable,
   --#                                    ProcedureExportList,
   --#                                    Root,
   --#                                    UsedSymbols;

   is
      P,
      ParenthesisCell,
      SqBracketCell   : Cells.Cell;
      ParenthesisForm : SPSymbols.SPSymbol;
      ParReqd         : Boolean;
      S               : CStacks.Stack;

      --------------------------------------------------------------

      function IsLeaf (Node : Cells.Cell) return Boolean
      --# global in Heap;
      is
      begin
         return Cells.Is_Null_Cell (Cells.Get_B_Ptr (Heap, Node));
      end IsLeaf;

      --------------------------------------------------------------

      procedure Parenthesise (V               : in     Cells.Cell;
                              LeftTree        : in     Boolean;
                              ParReqd         :    out Boolean;
                              ParenthesisForm :    out SPSymbols.SPSymbol)
      --# global in Heap;
      --# derives ParenthesisForm from Heap,
      --#                              V &
      --#         ParReqd         from Heap,
      --#                              LeftTree,
      --#                              V;
      is

         VPrecedence,
         WPrecedence : Natural;
         Operand,
         W           : Cells.Cell;
         V_Kind      : Cells.Cell_Kind;

         --------------------------------------------

         function PrecedenceValue (C : Cells.Cell) return Natural
         --# global in Heap;
         is
            PrecVal : Natural;
         begin
            if Cells.Get_Kind (Heap, C) = Cells.FDL_Div_Op then
               PrecVal := 5;
            else
               case Cells.Get_Op_Symbol (Heap, C) is
                  when SPSymbols.RWand           |
                     SPSymbols.RWor            |
                     SPSymbols.RWandthen       |
                     SPSymbols.RWorelse        |
                     SPSymbols.implies         |
                     SPSymbols.RWnot           |
                     SPSymbols.is_equivalent_to  => PrecVal := 1;
                  when SPSymbols.equals          |
                     SPSymbols.not_equal       |
                     SPSymbols.less_than       |
                     SPSymbols.less_or_equal   |
                     SPSymbols.greater_than    |
                     SPSymbols.greater_or_equal  => PrecVal := 2;
                  when SPSymbols.plus            |
                     SPSymbols.minus           |
                     SPSymbols.ampersand         => PrecVal := 3;

                     -- arity is taken into account by examining node degrees of operator nodes
                     -- (see body of procedure Parenthesise).
                     --        when SPSymbols.unary_plus      |
                     --             SPSymbols.unary_minus       => PrecVal := 4;

                  when SPSymbols.multiply        |
                     SPSymbols.divide          |
                     SPSymbols.RWmod             => PrecVal := 5;
                  when SPSymbols.double_star       => PrecVal := 6;
                  when others                      => PrecVal := 7;
               end case;
            end if;
            return PrecVal;
         end PrecedenceValue;

         --------------------------------------------------------------------

      begin -- Parenthesise;
         ParReqd := False;
         ParenthesisForm := SPSymbols.left_paren;
         V_Kind := Cells.Get_Kind (Heap, V);
         if (V_Kind = Cells.Declared_Function) or
            (V_Kind = Cells.Proof_Function)    or
            (V_Kind = Cells.Attrib_Function)   or
            (V_Kind = Cells.Field_Access_Function)      or
            (V_Kind = Cells.Mk_Aggregate)      or
            (V_Kind = Cells.List_Function)     or
            (V_Kind = Cells.Element_Function)  or
            (V_Kind = Cells.Update_Function)   or
            (V_Kind = Cells.Pred_Function)     or
            (V_Kind = Cells.Succ_Function)     or
            (V_Kind = Cells.Abs_Function)      or
            (V_Kind = Cells.Trunc_Function)    or
            (V_Kind = Cells.Field_Update_Function)      or
            (V_Kind = Cells.Bitwise_Op)        then
            ParReqd := True;
            if (V_Kind = Cells.List_Function) then
               ParenthesisForm := SPSymbols.square_open;
            end if;
         elsif ((V_Kind = Cells.Op)
                  -- TEMPORARY FIX until right_paren given its own kind
                 and then ((Cells.Get_Op_Symbol (Heap, V) /= SPSymbols.right_paren)
                              -- END OF TEMPORARY FIX
                            and
                            (Cells.Get_Op_Symbol (Heap, V) /= SPSymbols.comma)))
            or else (V_Kind = Cells.FDL_Div_Op) then

            if Cells.Is_Null_Cell (Cells.Get_A_Ptr (Heap, V)) then
               -- V is a monadic operator;
               Operand := Cells.Get_B_Ptr (Heap, V);
               if not IsLeaf (Operand) then
                  ParReqd := True;
               end if;
            else
               if LeftTree then
                  W := Cells.Get_A_Ptr (Heap, V);
               else
                  W := Cells.Get_B_Ptr (Heap, V);
               end if;
               if not Cells.Is_Null_Cell (W) then
                  if (Cells.Get_Kind (Heap, W) = Cells.Op) or else
                     (Cells.Get_Kind (Heap, W) = Cells.FDL_Div_Op) then

                     VPrecedence := PrecedenceValue (V);
                     WPrecedence := PrecedenceValue (W);

                     -- general rule for constructing unambiguous expressions:
                     ParReqd := (VPrecedence > WPrecedence) or
                        ((VPrecedence = WPrecedence) and not LeftTree);

                     -- supplementary rules, to improve clarity:
                     if (VPrecedence = 1) or        -- v is a logical operation;
                        (WPrecedence = 2) then      -- subtree is a relation;
                        ParReqd := True;
                     end if;
                  end if;
               end if;
            end if;
         end if;
      end Parenthesise;

      -----------------------------------------------------------------------

   begin -- FindDagDeclarations
         -- Algorithm of D.E. Knuth, Fundamental Algorithms, p.317;
      CStacks.CreateStack (S);
      Cells.Create_Cell (Heap, ParenthesisCell);
      Cells.Set_Kind (Heap, ParenthesisCell, Cells.Op);
      Cells.Set_Op_Symbol (Heap, ParenthesisCell, SPSymbols.left_paren);
      Cells.Create_Cell (Heap, SqBracketCell);
      Cells.Set_Op_Symbol (Heap, SqBracketCell, SPSymbols.square_open);
      P := Root;
      loop
         loop
            exit when Cells.Is_Null_Cell (P);
            CStacks.Push (Heap, P, S);
            if IsLeaf (P) then
               P := Cells.Null_Cell;
            else
               if (not Cells.Is_Null_Cell (Cells.Get_A_Ptr (Heap, P))) then
                  Parenthesise (P, True, ParReqd, ParenthesisForm);
                  if ParReqd then
                     if ParenthesisForm = SPSymbols.left_paren then
                        CStacks.Push (Heap, ParenthesisCell, S);
                     else
                        CStacks.Push (Heap, SqBracketCell, S);
                     end if;
                  end if;
               end if;
               P := Cells.Get_A_Ptr (Heap, P);
            end if;
         end loop;
         exit when CStacks.IsEmpty (S);
         P := CStacks.Top (Heap, S);
         CStacks.Pop (Heap, S);

         FindCellContentsDeclarations (Heap, P);

         if IsLeaf (P) then
            P := Cells.Null_Cell;
            loop
               exit when
                  not ((Cells.Are_Identical (CStacks.Top (Heap, S),
                                            ParenthesisCell))     or
                       (Cells.Are_Identical (CStacks.Top (Heap, S),
                                            SqBracketCell)));
               CStacks.Pop (Heap, S);
            end loop;
         else
            Parenthesise (P, False, ParReqd, ParenthesisForm);
            if ParReqd then
               if ParenthesisForm = SPSymbols.left_paren then
                  CStacks.Push (Heap, ParenthesisCell, S);
               else
                  CStacks.Push (Heap, SqBracketCell, S);
               end if;
            end if;
            P := Cells.Get_B_Ptr (Heap, P);
         end if;
      end loop;
   end FindDagDeclarations;

   -------------------------------------------------------------------------

   procedure FindLogicalExpnDeclarations (Heap       : in out Cells.Heap_Record;
                                          Root       : in     Cells.Cell)
   --# global in     AttributeList;
   --#        in     BitwiseOpList;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     ProcedureExportList;
   --#        in out ReturnSymbol;
   --#        in out RootIntegerUsed;
   --#        in out Statistics.TableUsage;
   --#        in out UsedSymbols;
   --# derives Heap,
   --#         ReturnSymbol,
   --#         RootIntegerUsed,
   --#         Statistics.TableUsage,
   --#         UsedSymbols           from *,
   --#                                    AttributeList,
   --#                                    BitwiseOpList,
   --#                                    Dictionary.Dict,
   --#                                    Heap,
   --#                                    LexTokenManager.StringTable,
   --#                                    ProcedureExportList,
   --#                                    Root,
   --#                                    UsedSymbols;
   is
      SubExpnList : Cells.Cell;

      procedure Partition
      --# global in     Root;
      --#        in     SubExpnList;
      --#        in out Heap;
      --#        in out Statistics.TableUsage;
      --# derives Heap,
      --#         Statistics.TableUsage from *,
      --#                                    Heap,
      --#                                    Root,
      --#                                    SubExpnList;
      is
         P,
         SubExpn : Cells.Cell;
         S       : CStacks.Stack;
      begin
         CStacks.CreateStack (S);
         P := Root;
         loop
            loop
               exit when Cells.Is_Null_Cell (P);
               CStacks.Push (Heap, P, S);
               if (Cells.Get_Kind (Heap, P) = Cells.Op) and then
                  ((Cells.Get_Op_Symbol (Heap, P) = SPSymbols.RWand) or
                   (Cells.Get_Op_Symbol (Heap, P) = SPSymbols.RWandthen)) then
                  P := Cells.Get_A_Ptr (Heap, P);
               else
                  Cells.Create_Cell (Heap, SubExpn);
                  Cells.Set_B_Ptr (Heap, SubExpn, P);
                  Clists.AppendCell (Heap, SubExpn, SubExpnList);
                  P := Cells.Null_Cell;
               end if;
            end loop;
            exit when CStacks.IsEmpty (S);
            P := CStacks.Top (Heap, S);
            CStacks.Pop (Heap, S);
            if (Cells.Get_Kind (Heap, P) = Cells.Op) and then
               ((Cells.Get_Op_Symbol (Heap, P) = SPSymbols.RWand) or
                (Cells.Get_Op_Symbol (Heap, P) = SPSymbols.RWandthen)) then
               P := Cells.Get_B_Ptr (Heap, P);
            else
               P := Cells.Null_Cell;
            end if;
         end loop;
      end Partition;

      procedure FindListOfExpnsDeclarations
      --# global in     AttributeList;
      --#        in     BitwiseOpList;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in     ProcedureExportList;
      --#        in     SubExpnList;
      --#        in out Heap;
      --#        in out ReturnSymbol;
      --#        in out RootIntegerUsed;
      --#        in out Statistics.TableUsage;
      --#        in out UsedSymbols;
      --# derives Heap,
      --#         ReturnSymbol,
      --#         RootIntegerUsed,
      --#         Statistics.TableUsage,
      --#         UsedSymbols           from *,
      --#                                    AttributeList,
      --#                                    BitwiseOpList,
      --#                                    Dictionary.Dict,
      --#                                    Heap,
      --#                                    LexTokenManager.StringTable,
      --#                                    ProcedureExportList,
      --#                                    SubExpnList,
      --#                                    UsedSymbols;
      is
         ListMember : Cells.Cell;
      begin

         --this looks like the place to suppress multiple Trues in hypotheses
         --and do something with trues in conclusions

         ListMember := Clists.FirstCell (Heap, SubExpnList);
         loop
            FindDagDeclarations (Heap,
                                 Cells.Get_B_Ptr (Heap, ListMember));
            ListMember := Clists.NextCell (Heap, ListMember);
            exit when Cells.Is_Null_Cell (ListMember);
         end loop;
      end FindListOfExpnsDeclarations;

   begin -- FindLogicalExpnDeclarations
      Clists.CreateList (Heap, SubExpnList);
      Partition;
      FindListOfExpnsDeclarations;
      Clists.DisposeOfList (Heap, SubExpnList);
   end FindLogicalExpnDeclarations;

   -------------------------------------------------------------------------


   procedure FindLabelDeclarations (Heap       : in out Cells.Heap_Record;
                                    LabelName  : in     Labels.Label)
   --# global in     AttributeList;
   --#        in     BitwiseOpList;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     ProcedureExportList;
   --#        in out ReturnSymbol;
   --#        in out RootIntegerUsed;
   --#        in out Statistics.TableUsage;
   --#        in out UsedSymbols;
   --# derives Heap,
   --#         ReturnSymbol,
   --#         RootIntegerUsed,
   --#         Statistics.TableUsage,
   --#         UsedSymbols           from *,
   --#                                    AttributeList,
   --#                                    BitwiseOpList,
   --#                                    Dictionary.Dict,
   --#                                    Heap,
   --#                                    LabelName,
   --#                                    LexTokenManager.StringTable,
   --#                                    ProcedureExportList,
   --#                                    UsedSymbols;
   is
      CurrentPair : Pairs.Pair;

      -------------------------------------------------------

      procedure FindPairDeclarations (PairName   : in    Pairs.Pair)
      --# global in     AttributeList;
      --#        in     BitwiseOpList;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in     ProcedureExportList;
      --#        in out Heap;
      --#        in out ReturnSymbol;
      --#        in out RootIntegerUsed;
      --#        in out Statistics.TableUsage;
      --#        in out UsedSymbols;
      --# derives Heap,
      --#         ReturnSymbol,
      --#         RootIntegerUsed,
      --#         Statistics.TableUsage,
      --#         UsedSymbols           from *,
      --#                                    AttributeList,
      --#                                    BitwiseOpList,
      --#                                    Dictionary.Dict,
      --#                                    Heap,
      --#                                    LexTokenManager.StringTable,
      --#                                    PairName,
      --#                                    ProcedureExportList,
      --#                                    UsedSymbols;
      -- prints a predicate-action pair;
      is
         Action,
         ModCell,
         Predicate  : Cells.Cell;
      begin -- FindPairDeclarations
         if Pairs.IsTrue (Heap, PairName) then
            null;
         else
            Predicate := Cells.Get_B_Ptr (Heap, Pairs.PairHead (PairName));
            FindLogicalExpnDeclarations (Heap,
                                         Predicate);
         end if;
         if Pairs.IsUnitAction (Heap, PairName) then
            null;
         else
            -- print action;
            Action := Cells.Get_C_Ptr (Heap, Pairs.PairHead (PairName));
            ModCell := Clists.FirstCell (Heap, Action);

            FindCellContentsDeclarations (Heap,
                                          ModCell);
            FindDagDeclarations (Heap,
                                 Cells.Get_B_Ptr (Heap, ModCell));
            ModCell := Clists.NextCell (Heap, ModCell);

            loop
               exit when Cells.Is_Null_Cell (ModCell);

               FindCellContentsDeclarations (Heap,
                                             ModCell);
               FindDagDeclarations (Heap,
                                    Cells.Get_B_Ptr (Heap, ModCell));
               ModCell := Clists.NextCell (Heap, ModCell);
            end loop;
         end if;
      end FindPairDeclarations;

   begin -- FindLabelDeclarations
      CurrentPair := Labels.FirstPair (Heap, LabelName);
      loop
         exit when Pairs.IsNullPair (CurrentPair);
         FindPairDeclarations (CurrentPair);
         CurrentPair := Labels.NextPair (Heap, CurrentPair);
      end loop;
   end FindLabelDeclarations;

   -------------------------------------------------------------------------

   procedure FindVCFormulaDeclarations (Heap          : in out Cells.Heap_Record;
                                        PredicatePair : in     Pairs.Pair)
   --# global in     AttributeList;
   --#        in     BitwiseOpList;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     ProcedureExportList;
   --#        in out ReturnSymbol;
   --#        in out RootIntegerUsed;
   --#        in out Statistics.TableUsage;
   --#        in out UsedSymbols;
   --# derives Heap,
   --#         ReturnSymbol,
   --#         RootIntegerUsed,
   --#         Statistics.TableUsage,
   --#         UsedSymbols           from *,
   --#                                    AttributeList,
   --#                                    BitwiseOpList,
   --#                                    Dictionary.Dict,
   --#                                    Heap,
   --#                                    LexTokenManager.StringTable,
   --#                                    PredicatePair,
   --#                                    ProcedureExportList,
   --#                                    UsedSymbols;
   is
      ConclusionRoot,
      HypothesisRoot : Cells.Cell;

      function IsTriviallyTrue (DAG : Cells.Cell) return Boolean
      --# global in Dictionary.Dict;
      --#        in Heap;
      is
         CurrentCell : Cells.Cell;
         Result      : Boolean := True;

         function IsTrueCell (TheCell : Cells.Cell) return Boolean
         --# global in Dictionary.Dict;
         --#        in Heap;
         is
         begin
            return Cells.Get_Kind (Heap, TheCell) = Cells.Named_Const and then
               Cells.Get_Symbol_Value (Heap, TheCell) = Dictionary.GetTrue;
         end IsTrueCell;

         function AppropriateBinaryOperator (OpSym : SPSymbols.SPSymbol) return Boolean
         is
         begin
            return OpSym = SPSymbols.RWand or else
               OpSym = SPSymbols.RWandthen or else
               OpSym = SPSymbols.RWor or else
               OpSym = SPSymbols.RWorelse or else
               OpSym = SPSymbols.equals or else
               OpSym = SPSymbols.implies or else
               OpSym = SPSymbols.is_equivalent_to;
         end AppropriateBinaryOperator;

      begin --IsTriviallyTrue
         CurrentCell := DAG;
         loop
            exit when IsTrueCell (CurrentCell); --success condition

            --some expression other than an operator - fail
            if Cells.Get_Kind (Heap, CurrentCell) /= Cells.Op then
               Result := False;
               exit;
            end if;

            --inappropriate operator - fail
            if not AppropriateBinaryOperator (Cells.Get_Op_Symbol (Heap, CurrentCell)) then
               Result := False;
               exit;
            end if;

            --thing on left of operator is not true - fail
            if not IsTrueCell (Cells.Get_A_Ptr (Heap, CurrentCell)) then
               Result := False;
               exit;
            end if;

            --move down right hand chain of tree to get next sub-expression
            CurrentCell := Cells.Get_B_Ptr (Heap, CurrentCell);

            --fallen off the end - fail - (I think this check is redundant but safe)
            if Cells.Is_Null_Cell (CurrentCell) then
               Result := False;
               exit;
            end if;

         end loop;
         return Result;
      end IsTriviallyTrue;

   begin --FindVCFormula
      HypothesisRoot := Cells.Get_B_Ptr (Heap, Pairs.PairHead (PredicatePair));
      ConclusionRoot := Cells.Get_C_Ptr (Heap, Pairs.PairHead (PredicatePair));

      if IsTriviallyTrue (ConclusionRoot) then
         null;
      else
         FindLogicalExpnDeclarations (Heap,
                                      HypothesisRoot);
         FindLogicalExpnDeclarations (Heap,
                                      ConclusionRoot);
      end if;
   end FindVCFormulaDeclarations;



   procedure Initialize (It : out UsedSymbolIterator)
   --# global in UsedSymbols;
   --# derives It from UsedSymbols;
   is
   begin
      It := UsedSymbolIterator'(It => UsedSymbols);
   end Initialize;

   function CurrentNode (It : in UsedSymbolIterator) return Cells.Cell
   is
   begin
      return It.It;
   end CurrentNode;

   function NextNode (Heap : in Cells.Heap_Record;
                      It   : in UsedSymbolIterator) return UsedSymbolIterator
   is
   begin
      return UsedSymbolIterator'(It => Cells.Get_A_Ptr (Heap, It.It));
   end NextNode;

   function IsNullIterator (It : in UsedSymbolIterator) return Boolean
   is
   begin
      return It = NullIterator;
   end IsNullIterator;



   ----------------------------------------------------------------

   procedure PrintDeclarationTail (File : in SPARK_IO.File_Type)
   is
   begin
      SPARK_IO.New_Line (File, 1);
      SPARK_IO.Put_Line (File, "end;", 0);
   end PrintDeclarationTail;

   ----------------------------------------------------------------------


   procedure OutputDeclarations (Heap       : in out Cells.Heap_Record;
                                 File       : in     SPARK_IO.File_Type;
                                 RuleFile   : in     SPARK_IO.File_Type;
                                 Scope      : in     Dictionary.Scopes;
                                 WriteRules : in     Boolean;
                                 EndPosition : in     LexTokenManager.TokenPosition)
   --# global in     AttributeList;
   --#        in     BitwiseOpList;
   --#        in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     ProcedureExportList;
   --#        in     ReturnSymbol;
   --#        in     RootIntegerUsed;
   --#        in     UsedSymbols;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --# derives ErrorHandler.ErrorContext from *,
   --#                                        AttributeList,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EndPosition,
   --#                                        File,
   --#                                        Heap,
   --#                                        LexTokenManager.StringTable,
   --#                                        RuleFile,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        UsedSymbols,
   --#                                        WriteRules &
   --#         Heap,
   --#         Statistics.TableUsage     from *,
   --#                                        AttributeList,
   --#                                        Dictionary.Dict,
   --#                                        Heap,
   --#                                        Scope,
   --#                                        UsedSymbols,
   --#                                        WriteRules &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        AttributeList,
   --#                                        BitwiseOpList,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EndPosition,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        File,
   --#                                        Heap,
   --#                                        LexTokenManager.StringTable,
   --#                                        ProcedureExportList,
   --#                                        ReturnSymbol,
   --#                                        RootIntegerUsed,
   --#                                        RuleFile,
   --#                                        Scope,
   --#                                        UsedSymbols,
   --#                                        WriteRules;
   is separate;

end Declarations;
