-- $Id: declarations-outputdeclarations-printdeclarations.adb 12843 2009-03-31 17:29:05Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with DAG_IO,
     EStrings,
     ELStrings,
     LexTokenManager,
     AdjustFDL_RWs,
     CommandLineData;

with Debug;

separate (Declarations.OutputDeclarations)
procedure PrintDeclarations (Heap          : in out Cells.Heap_Record;
                             File          : in     SPARK_IO.File_Type;
                             RuleFile      : in     SPARK_IO.File_Type;
                             NeededSymbols : in     Cells.Cell;
                             Scope         : in     Dictionary.Scopes;
                             WriteRules    : in     Boolean;
                             EndPosition   : in     LexTokenManager.TokenPosition)
is
   Sym         : Dictionary.Symbol;
   DeclareList : Cells.Cell;
   Indent      : constant Integer := 3;
   MaxColumn   : constant Integer := 78;

   -- TypeList records types that require proof rules to be
   -- generated for them.  Prior to Examiner 7p2d08, only scalar
   -- type were added to this list, since these were the only
   -- types that could have replacement rules generated for them.
   -- Following 7p2d08, this list now records all types to allow
   -- for replacement rules for 'Size of non-scalar types.
   LHeap          : Lists.ListHeap;
   TypeList       : Lists.List;

   RuleFamilyName : EStrings.T;
   RuleCounter    : Natural;

   MaxRank : Cells.Cell_Rank;

   ----------------------------------------------------------------------
   procedure Debug_Rank_Sym (Msg : in String;
                             Sym : in Dictionary.Symbol)
   --# derives null from Msg,
   --#                   Sym;
   is
      --# hide Debug_Rank_Sym;
   begin
      if CommandLineData.Content.Debug.FDL_Ranking then
         Debug.PrintSym (Msg, Sym);
      end if;
   end Debug_Rank_Sym;

   procedure Debug_Rank_Int (Msg : in String;
                             N   : in Integer)
   --# derives null from Msg,
   --#                   N;
   is
      --# hide Debug_Rank_Int;
   begin
      if CommandLineData.Content.Debug.FDL_Ranking then
         Debug.PrintInt (Msg, N);
      end if;
   end Debug_Rank_Int;

   ----------------------------------------------------------------------

   function GetName (Sym   : Dictionary.Symbol;
                     Scope : Dictionary.Scopes) return EStrings.T
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --#        in LexTokenManager.StringTable;
   is
      Prefix        : EStrings.T;
      SimpleName    : EStrings.T;
      Result        : EStrings.T;

   begin
      -- if VCs/PFs are generated from code with semantic errors in type declarations then
      -- null symbols etc. might get as far as here and cause an internal error.  The first if
      -- provides protection against this.
      if Sym = Dictionary.NullSymbol then
         Result := EStrings.T'(Length => 22,
                                                   Content => EStrings.Contents'
                                                     ('u', 'n', 'e', 'x', 'p', 'e', 'c', 't', 'e', 'd', '_',
                                                      'n', 'u', 'l', 'l', '_', 's', 'y', 'm', 'b', 'o', 'l',
                                                      others => ' '));
      elsif Dictionary.IsUnknownTypeMark (Sym) then
         Result := EStrings.T'(Length => 12,
                                                   Content => EStrings.Contents'
                                                     ('u', 'n', 'k', 'n', 'o', 'w', 'n', '_',
                                                      't', 'y', 'p', 'e',
                                                      others => ' '));
      else
         if Dictionary.IsRecordComponent (Sym) then
            Prefix := EStrings.EmptyString;
         else
            Dictionary.GetAnyPrefixNeeded (Sym, Scope, "__", Prefix);
         end if;
         Dictionary.GenerateSimpleName (Sym, "__", SimpleName);

         if Prefix.Length > 0 then
            Result := Prefix;
            EStrings.AppendString (Result, "__");
            EStrings.AppendExaminerString (Result, SimpleName);
         else
            if CommandLineData.Content.FDLmangle /= EStrings.EmptyString then
               AdjustFDL_RWs.PossiblyAdjust (SimpleName, CommandLineData.Content.FDLmangle);
            end if;
            Result := SimpleName;
         end if;
         Result := EStrings.LowerCase (Result);
      end if;

      return Result;

   end GetName;

   ----------------------------------------------------------------------

   procedure PrintSymbol (File   : in SPARK_IO.File_Type;
                          Scope  : in Dictionary.Scopes;
                          Sym    : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives SPARK_IO.FILE_SYS from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                File,
   --#                                LexTokenManager.StringTable,
   --#                                Scope,
   --#                                Sym;
   is
      Name : EStrings.T;
   begin
      Name := GetName (Sym, Scope);

      if SPARK_IO.Col (File) /= Indent and then
         (((SPARK_IO.Col (File) + Name.Length) + 12) > MaxColumn)
      then
         SPARK_IO.New_Line (File, 1);
         SPARK_IO.Set_Col (File, 2 * Indent);
      end if;

      EStrings.PutString (File, Name);

   end PrintSymbol;

   ----------------------------------------------------------------------

   procedure PrintSymbolType (File  : in SPARK_IO.File_Type;
                              Scope : in Dictionary.Scopes;
                              Sym   : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives SPARK_IO.FILE_SYS from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                File,
   --#                                LexTokenManager.StringTable,
   --#                                Scope,
   --#                                Sym;
   is
      TypeSym : Dictionary.Symbol;
   begin
      if Dictionary.IsTypeMark (Sym) then
         TypeSym := Dictionary.GetRootType (Sym);
      else
         TypeSym := Dictionary.GetRootType (Dictionary.GetType (Sym));
      end if;

      if Dictionary.TypeIsInteger (TypeSym) or else
        Dictionary.TypeIsModular (TypeSym) -- modular types become integer in FDL
      then
         SPARK_IO.Put_String (File, "integer", 0);
      elsif Dictionary.TypeIsCharacter  (TypeSym) then
         SPARK_IO.Put_String (File, "character", 0);
      elsif Dictionary.TypeIsReal (TypeSym) then
         SPARK_IO.Put_String (File, "real", 0);
      elsif Dictionary.TypeIsBoolean (TypeSym) then
         SPARK_IO.Put_String (File, "boolean", 0);
      else
         PrintSymbol (File, Scope, TypeSym);
      end if;
   end PrintSymbolType;

   ----------------------------------------------------------------------
   -- routines to print rules go here because they need to be seen by
   -- PrintConstantDeclaration and they use PrintSymbol
   ----------------------------------------------------------------------

   procedure PrintRuleName (RuleFile : in SPARK_IO.File_Type)
   --# global in     RuleFamilyName;
   --#        in out RuleCounter;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives RuleCounter       from * &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                RuleCounter,
   --#                                RuleFamilyName,
   --#                                RuleFile;
   is
   begin
      RuleCounter := RuleCounter + 1;
      EStrings.PutString (RuleFile, RuleFamilyName);
      SPARK_IO.Put_Char (RuleFile, '(');
      SPARK_IO.Put_Integer (RuleFile, RuleCounter, 0, 10);
      SPARK_IO.Put_String (RuleFile, "): ", 0);
   end PrintRuleName;

   ----------------------------------------------

   procedure EndARule (RuleFile : in SPARK_IO.File_Type)
   --# global in out SPARK_IO.FILE_SYS;
   --# derives SPARK_IO.FILE_SYS from *,
   --#                                RuleFile;
   is
   begin
      SPARK_IO.Put_String (RuleFile, ".", 0);
      SPARK_IO.New_Line (RuleFile, 1);
   end EndARule;

   ----------------------------------------------

   function GetValue (StoreVal : LexTokenManager.LexString;
                      TypeMark : Dictionary.Symbol;
                      Scope    : Dictionary.Scopes) return ELStrings.T
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --#        in LexTokenManager.StringTable;
   is
      Str : EStrings.T;
      LStr : ELStrings.T;
   begin

      if StoreVal = LexTokenManager.TrueToken then
         ELStrings.CopyString (LStr, "true");
      elsif StoreVal = LexTokenManager.FalseToken then
         ELStrings.CopyString (LStr, "false");
      elsif Dictionary.TypeIsNumeric (TypeMark) or else
         Dictionary.TypeIsCharacter (TypeMark) then
         LStr := Maths.ValueToString (Maths.ValueRep (StoreVal));
      else
         Str := GetName (Dictionary.GetEnumerationLiteral (TypeMark, StoreVal),
                         Scope);
         LStr := ELStrings.ToExaminerLongString (Str);
      end if;

      return LStr;

   end GetValue;

   ----------------------------------------------

   procedure PrintReplacementRule (RuleFile : in     SPARK_IO.File_Type;
                                   StoreVal : in     LexTokenManager.LexString;
                                   TypeMark : in     Dictionary.Symbol;
                                   Scope    : in     Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives SPARK_IO.FILE_SYS from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.StringTable,
   --#                                RuleFile,
   --#                                Scope,
   --#                                StoreVal,
   --#                                TypeMark;
   is
   begin
      SPARK_IO.Put_String (RuleFile, " may_be_replaced_by ", 0);
      ELStrings.PutString (RuleFile, GetValue (StoreVal, TypeMark, Scope));
      EndARule (RuleFile);
   end PrintReplacementRule;

   ----------------------------------------------

   procedure PrintRuleHeader (WriteRules : in     Boolean;
                              RuleFile   : in     SPARK_IO.File_Type)
   --# global in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     Scope;
   --#        in out LHeap;
   --#        in out SPARK_IO.FILE_SYS;
   --#           out RuleCounter;
   --#           out RuleFamilyName;
   --#           out TypeList;
   --# derives LHeap,
   --#         TypeList          from LHeap &
   --#         RuleCounter       from  &
   --#         RuleFamilyName    from Dictionary.Dict,
   --#                                LexTokenManager.StringTable,
   --#                                Scope &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.StringTable,
   --#                                RuleFile,
   --#                                Scope,
   --#                                WriteRules;
   is separate;

   ----------------------------------------------
   procedure  PrintTypeRules (WriteRules : in     Boolean;
                              RuleFile   : in     SPARK_IO.File_Type)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     RuleFamilyName;
   --#        in     Scope;
   --#        in out LHeap;
   --#        in out RuleCounter;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out TypeList;
   --# derives LHeap,
   --#         TypeList          from LHeap,
   --#                                TypeList,
   --#                                WriteRules &
   --#         RuleCounter       from *,
   --#                                Dictionary.Dict,
   --#                                LHeap,
   --#                                TypeList,
   --#                                WriteRules &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.StringTable,
   --#                                LHeap,
   --#                                RuleCounter,
   --#                                RuleFamilyName,
   --#                                RuleFile,
   --#                                Scope,
   --#                                TypeList,
   --#                                WriteRules;
   is separate;

   ----------------------------------------------------------------------

   procedure PrintConstantRules (WriteRules  : in     Boolean;
                                 Sym         : in     Dictionary.Symbol;
                                 RuleFile    : in     SPARK_IO.File_Type;
                                 Scope       : in     Dictionary.Scopes;
                                 EndPosition : in     LexTokenManager.TokenPosition)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     RuleFamilyName;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out RuleCounter;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EndPosition,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        RuleCounter,
   --#                                        RuleFamilyName,
   --#                                        RuleFile,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        Sym,
   --#                                        WriteRules &
   --#         RuleCounter               from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.StringTable,
   --#                                        Scope,
   --#                                        Sym,
   --#                                        WriteRules;
   is separate;

   procedure PrintConstantReplacementRule (WriteRules : in     Boolean;
                                           Sym        : in     Dictionary.Symbol;
                                           DAG        : in     Cells.Cell;
                                           RuleFile   : in     SPARK_IO.File_Type;
                                           Scope      : in     Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     RuleFamilyName;
   --#        in out Heap;
   --#        in out RuleCounter;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --# derives Heap,
   --#         Statistics.TableUsage from *,
   --#                                    DAG,
   --#                                    Heap,
   --#                                    WriteRules &
   --#         RuleCounter           from *,
   --#                                    WriteRules &
   --#         SPARK_IO.FILE_SYS     from *,
   --#                                    CommandLineData.Content,
   --#                                    DAG,
   --#                                    Dictionary.Dict,
   --#                                    Heap,
   --#                                    LexTokenManager.StringTable,
   --#                                    RuleCounter,
   --#                                    RuleFamilyName,
   --#                                    RuleFile,
   --#                                    Scope,
   --#                                    Sym,
   --#                                    WriteRules;
   is
   begin
      if WriteRules then
         PrintRuleName (RuleFile);
         PrintSymbol (RuleFile, Scope, Sym);
         SPARK_IO.Put_String (RuleFile, " may_be_replaced_by ", 0);
         DAG_IO.PrintDag (Heap, RuleFile, DAG, Scope, DAG_IO.Default_Wrap_Limit);
         EndARule (RuleFile);
      end if;
   end PrintConstantReplacementRule;


   ----------------------------------------------------------------------
   -- Produces FDL rules that are common to ALL Examiner-generated rule
   -- files.
   procedure PrintStandardRules (WriteRules : in     Boolean;
                                 RuleFile   : in     SPARK_IO.File_Type)

   --# global in     RuleFamilyName;
   --#        in out RuleCounter;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives RuleCounter       from *,
   --#                                WriteRules &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                RuleCounter,
   --#                                RuleFamilyName,
   --#                                RuleFile,
   --#                                WriteRules;
   is
      --# hide PrintStandardRules;
      pragma Unreferenced (WriteRules);
      pragma Unreferenced (RuleFile);
   begin
      -- Currently, there are no such standard rules.
      -- This used to produce rules for Character'Pos and 'Val, but
      -- these are no longer needed following CFR 1804.
      -- We leave the contract of this unit as-was, though, in case
      -- of future re-instatement of this procedure.
      null;
   end PrintStandardRules;

   ----------------------------------------------------------------------

   procedure PrintDeclarationHead (File   : in SPARK_IO.File_Type;
                                   Scope  : in Dictionary.Scopes)
   --# global in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives SPARK_IO.FILE_SYS from *,
   --#                                Dictionary.Dict,
   --#                                File,
   --#                                LexTokenManager.StringTable,
   --#                                Scope;
   is
      SubProgSym    : Dictionary.Symbol;
      SubProgString : EStrings.T;
   begin
      -- Note that in the declarations header, we always print the
      -- _simple_ name of the subprogram, so it always matches the name
      -- output as the prefix of each VC generated by Graph.PrintVCs
      SubProgSym := Dictionary.GetRegion (Scope);
      LexTokenManager.LexStringToString
         (Dictionary.GetSimpleName (SubProgSym),
          SubProgString);
      SubProgString := EStrings.LowerCase (SubProgString);

      SPARK_IO.Put_String (File, "title ", 0);
      if Dictionary.IsFunction (SubProgSym) then
         SPARK_IO.Put_String (File, "function ", 0);
      elsif Dictionary.IsProcedure (SubProgSym) then
         SPARK_IO.Put_String (File, "procedure ", 0);
      else
         null;
      end if;
      EStrings.PutString (File, SubProgString);
      SPARK_IO.Put_Char (File, ';');
      SPARK_IO.New_Line (File, 2);
   end PrintDeclarationHead;

   ----------------------------------------------------------------------

   function FindOwnVarMatchingThisType (Sym   : Dictionary.Symbol;
                                        Scope : Dictionary.Scopes)
                                       return Dictionary.Symbol
   --# global in Dictionary.Dict;
   is
      It            : Dictionary.Iterator;
      Result        : Dictionary.Symbol := Dictionary.NullSymbol;
      CurrentOwnVar : Dictionary.Symbol;
   begin -- FindOwnVarMatchingThisType
      It := Dictionary.FirstOwnVariable (Dictionary.GetEnclosingPackage (Scope));
      while not Dictionary.IsNullIterator (It) loop
         CurrentOwnVar := Dictionary.CurrentSymbol (It);
         if Dictionary.GetType (CurrentOwnVar) = Sym then
            Result := CurrentOwnVar;
            exit;
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;
      return Result;
   end FindOwnVarMatchingThisType;

   ----------------------------------------------------------------------

   procedure PrintDeclaration (File   : in SPARK_IO.File_Type;
                               Scope  : in Dictionary.Scopes;
                               Sym    : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out Heap;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --# derives Heap,
   --#         Statistics.TableUsage from *,
   --#                                    Dictionary.Dict,
   --#                                    Heap,
   --#                                    Scope,
   --#                                    Sym &
   --#         SPARK_IO.FILE_SYS     from *,
   --#                                    CommandLineData.Content,
   --#                                    Dictionary.Dict,
   --#                                    File,
   --#                                    Heap,
   --#                                    LexTokenManager.StringTable,
   --#                                    Scope,
   --#                                    Sym;
   is

      procedure PrintVariableDeclaration (File   : in SPARK_IO.File_Type;
                                          Scope  : in Dictionary.Scopes;
                                          Sym    : in Dictionary.Symbol)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives SPARK_IO.FILE_SYS from *,
      --#                                CommandLineData.Content,
      --#                                Dictionary.Dict,
      --#                                File,
      --#                                LexTokenManager.StringTable,
      --#                                Scope,
      --#                                Sym;
      is
      begin
         SPARK_IO.Set_Col (File, Indent);
         SPARK_IO.Put_String (File, "var ", 0);
         PrintSymbol (File, Scope, Sym);
         SPARK_IO.Put_String (File, " : ", 0);
         PrintSymbolType (File, Scope, Sym);
         SPARK_IO.Put_Line (File, ";", 0);
      end PrintVariableDeclaration;

      ----------------------------------------------------------------------

      procedure PrintConstantDeclaration (File   : in SPARK_IO.File_Type;
                                          Scope  : in Dictionary.Scopes;
                                          Sym    : in Dictionary.Symbol)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives SPARK_IO.FILE_SYS from *,
      --#                                CommandLineData.Content,
      --#                                Dictionary.Dict,
      --#                                File,
      --#                                LexTokenManager.StringTable,
      --#                                Scope,
      --#                                Sym;
      is
      begin
         if not Dictionary.IsEnumerationLiteral (Sym) then
            SPARK_IO.Set_Col (File, Indent);
            SPARK_IO.Put_String (File, "const ", 0);
            PrintSymbol (File, Scope, Sym);
            SPARK_IO.Put_String (File, " : ", 0);
            PrintSymbolType (File, Scope, Sym);
            SPARK_IO.Put_String (File, " = pending", 0);
            SPARK_IO.Put_Line (File, ";", 0);
         end if;
      end PrintConstantDeclaration;

      ----------------------------------------------------------------------

      procedure PrintTypeDeclaration (File   : in SPARK_IO.File_Type;
                                      Scope  : in Dictionary.Scopes;
                                      Sym    : in Dictionary.Symbol)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives SPARK_IO.FILE_SYS from *,
      --#                                CommandLineData.Content,
      --#                                Dictionary.Dict,
      --#                                File,
      --#                                LexTokenManager.StringTable,
      --#                                Scope,
      --#                                Sym;
      is
         Typ : Dictionary.Symbol;

         --------------------------

         procedure PrintPrivateType (File   : in SPARK_IO.File_Type;
                                     Scope  : in Dictionary.Scopes;
                                     Sym    : in Dictionary.Symbol)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     LexTokenManager.StringTable;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS from *,
         --#                                CommandLineData.Content,
         --#                                Dictionary.Dict,
         --#                                File,
         --#                                LexTokenManager.StringTable,
         --#                                Scope,
         --#                                Sym;
         is
         begin
            SPARK_IO.Set_Col (File, Indent);
            SPARK_IO.Put_String (File, "type ", 0);
            PrintSymbol (File, Scope, Sym);
            SPARK_IO.Put_Line (File, " = pending;", 0);
         end PrintPrivateType;

         ------------------------------

         -- Same as PrintPrivateType for now, but kept separate in case we
         -- implement further support for proof of protected types
         -- in future.
         procedure PrintProtectedType (File   : in SPARK_IO.File_Type;
                                       Scope  : in Dictionary.Scopes;
                                       Sym    : in Dictionary.Symbol)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     LexTokenManager.StringTable;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS from *,
         --#                                CommandLineData.Content,
         --#                                Dictionary.Dict,
         --#                                File,
         --#                                LexTokenManager.StringTable,
         --#                                Scope,
         --#                                Sym;
         is
         begin
            SPARK_IO.Set_Col (File, Indent);
            SPARK_IO.Put_String (File, "type ", 0);
            PrintSymbol (File, Scope, Sym);
            SPARK_IO.Put_Line (File, " = pending;", 0);
         end PrintProtectedType;

         ------------------------------

         -- Same as PrintPrivateType for now, but kept separate for clarity
         procedure PrintIncompleteType (File   : in SPARK_IO.File_Type;
                                        Scope  : in Dictionary.Scopes;
                                        Sym    : in Dictionary.Symbol)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     LexTokenManager.StringTable;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS from *,
         --#                                CommandLineData.Content,
         --#                                Dictionary.Dict,
         --#                                File,
         --#                                LexTokenManager.StringTable,
         --#                                Scope,
         --#                                Sym;
         is
         begin
            SPARK_IO.Set_Col (File, Indent);
            SPARK_IO.Put_String (File, "type ", 0);
            PrintSymbol (File, Scope, Sym);
            SPARK_IO.Put_Line (File, " = pending;", 0);
         end PrintIncompleteType;

         ------------------------------

         procedure PrintSubType (File   : in SPARK_IO.File_Type;
                                 Scope  : in Dictionary.Scopes;
                                 Sym    : in Dictionary.Symbol)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     LexTokenManager.StringTable;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS from *,
         --#                                CommandLineData.Content,
         --#                                Dictionary.Dict,
         --#                                File,
         --#                                LexTokenManager.StringTable,
         --#                                Scope,
         --#                                Sym;
         is
         begin
            SPARK_IO.Set_Col (File, Indent);
            SPARK_IO.Put_String (File, "type ", 0);
            PrintSymbol (File, Scope, Sym);
            SPARK_IO.Put_String (File, " = ", 0);
            PrintSymbolType (File, Scope, Sym);
            SPARK_IO.Put_Line (File, ";", 0);
         end PrintSubType;

         --------------------------------------

         procedure PrintTimeType (File   : in SPARK_IO.File_Type;
                                  Scope  : in Dictionary.Scopes;
                                  Sym    : in Dictionary.Symbol)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     LexTokenManager.StringTable;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS from *,
         --#                                CommandLineData.Content,
         --#                                Dictionary.Dict,
         --#                                File,
         --#                                LexTokenManager.StringTable,
         --#                                Scope,
         --#                                Sym;
         is
         begin
            -- Ada95 LRM D.8 (27) says that time types (and their
            -- operators) behave as if they were integers.
            SPARK_IO.Set_Col (File, Indent);
            SPARK_IO.Put_String (File, "type ", 0);
            PrintSymbol (File, Scope, Sym);
            SPARK_IO.Put_Line (File, " = integer;", 0);
         end PrintTimeType;

         --------------------------------------

         procedure PrintEnumeratedType (File   : in SPARK_IO.File_Type;
                                        Scope  : in Dictionary.Scopes;
                                        Sym    : in Dictionary.Symbol)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     LexTokenManager.StringTable;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS from *,
         --#                                CommandLineData.Content,
         --#                                Dictionary.Dict,
         --#                                File,
         --#                                LexTokenManager.StringTable,
         --#                                Scope,
         --#                                Sym;
         is
            EnumIt    : Dictionary.Iterator;
            FirstOne  : Boolean;
         begin
            SPARK_IO.Set_Col (File, Indent);
            SPARK_IO.Put_String (File, "type ", 0);
            PrintSymbol (File, Scope, Sym);
            SPARK_IO.Put_String (File, " = ", 0);
            SPARK_IO.Put_Char (File, '(');
            if Dictionary.TypeIsGeneric (Sym) then
               -- we can't print the actual enumeration literals because there aren't any
               -- so we synthesise some to tell the proof tools this is an enumeration
               -- the synthetic literals will never appear in VCs and can't be used

               -- format is type T = (generic__t__lower, generic__t__upper);

               SPARK_IO.Put_String (File, "generic__", 0);
               PrintSymbol (File, Scope, Sym);
               SPARK_IO.Put_String (File, "__lower, ", 0);
               SPARK_IO.Put_String (File, "generic__", 0);
               PrintSymbol (File, Scope, Sym);
               SPARK_IO.Put_String (File, "__upper", 0);
            else
               -- not generic, get literals from Dictionary
               FirstOne := True;
               EnumIt := Dictionary.FirstEnumerationLiteral (Sym);
               while not Dictionary.IsNullIterator (EnumIt) loop
                  if FirstOne then
                     FirstOne := False;
                  else
                     SPARK_IO.Put_String (File, ", ", 0);
                  end if;
                  PrintSymbol (File, Scope, Dictionary.CurrentSymbol (EnumIt));
                  EnumIt := Dictionary.NextSymbol (EnumIt);
               end loop;
            end if;
            SPARK_IO.Put_Char (File, ')');
            SPARK_IO.Put_Line (File, ";", 0);
         end PrintEnumeratedType;

         ---------------------------

         procedure PrintCharacterType (File : in SPARK_IO.File_Type)
         --# global in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS from *,
         --#                                File;
         is
         begin
            SPARK_IO.Set_Col (File, Indent);
            SPARK_IO.Put_Line (File, "type character = integer;", 0);
         end PrintCharacterType;

         ---------------------------

         procedure PrintRecordType (File             : in SPARK_IO.File_Type;
                                    Scope            : in Dictionary.Scopes;
                                    Sym              : in Dictionary.Symbol;
                                    HasPrivateFields : in Boolean)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     LexTokenManager.StringTable;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS from *,
         --#                                CommandLineData.Content,
         --#                                Dictionary.Dict,
         --#                                File,
         --#                                HasPrivateFields,
         --#                                LexTokenManager.StringTable,
         --#                                Scope,
         --#                                Sym;
         is
            ComponentIt      : Dictionary.Iterator;
            FirstOne         : Boolean;

         begin
            -- If the root of an extended record is a null record then we don't want declaration
            if Dictionary.RecordHasSomeFields (Sym) then
               SPARK_IO.Set_Col (File, Indent);
               SPARK_IO.Put_String (File, "type ", 0);
               PrintSymbol (File, Scope, Sym);
               SPARK_IO.Put_String (File, " = ", 0);
               SPARK_IO.Put_Line (File, "record", 0);

               if HasPrivateFields then
                  -- just declare inherit field
                  SPARK_IO.Set_Col (File, 3*Indent);
                  SPARK_IO.Put_String (File, "inherit : ", 0);
                  PrintSymbolType (File,
                                   Scope,
                                   Dictionary.GetRootOfExtendedType (Sym));

               else
                  -- declare all fields
                  ComponentIt := Dictionary.FirstRecordComponent (Sym);

                  -- If all ancestors of an extended record are null records then we don't want
                  -- a declaration of an Inherit field referencing first of them.
                  if Dictionary.TypeIsExtendedTagged (Sym) and then
                    Dictionary.NoFieldsBelowThisRecord (Sym) then
                     --skip inherit field
                     ComponentIt := Dictionary.NextSymbol (ComponentIt);
                  end if;

                  FirstOne := True;
                  while not Dictionary.IsNullIterator (ComponentIt) loop
                     if FirstOne then
                        FirstOne := False;
                     else
                        SPARK_IO.Put_Line (File, ";", 0);
                     end if;
                     SPARK_IO.Set_Col (File, 3*Indent);
                     PrintSymbol (File,
                                  Scope,
                                  Dictionary.CurrentSymbol (ComponentIt));
                     SPARK_IO.Put_String (File, " : ", 0);
                     PrintSymbolType (File,
                                      Scope,
                                      Dictionary.CurrentSymbol (ComponentIt));
                     -- exit when HasPrivateFields; -- only print inherit field if rest private
                     ComponentIt := Dictionary.NextSymbol (ComponentIt);
                  end loop;
               end if;

               SPARK_IO.New_Line (File, 1);
               SPARK_IO.Set_Col (File, 2*Indent);
               SPARK_IO.Put_String (File, "end", 0);
               SPARK_IO.Put_Line (File, ";", 0);
            end if;
         end PrintRecordType;

         -------------------------------

         procedure PrintArrayType (File   : in SPARK_IO.File_Type;
                                   Scope  : in Dictionary.Scopes;
                                   Sym    : in Dictionary.Symbol)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     LexTokenManager.StringTable;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS from *,
         --#                                CommandLineData.Content,
         --#                                Dictionary.Dict,
         --#                                File,
         --#                                LexTokenManager.StringTable,
         --#                                Scope,
         --#                                Sym;
         is
            IndexIt  : Dictionary.Iterator;
            FirstOne : Boolean;
         begin
            SPARK_IO.Set_Col (File, Indent);
            SPARK_IO.Put_String (File, "type ", 0);
            PrintSymbol (File, Scope, Sym);
            SPARK_IO.Put_String (File, " = ", 0);
            SPARK_IO.Put_String (File, "array [", 0);
            IndexIt := Dictionary.FirstArrayIndex (Sym);
            FirstOne := True;
            while not Dictionary.IsNullIterator (IndexIt) loop
               if FirstOne then
                  FirstOne := False;
               else
                  SPARK_IO.Put_String (File, ",", 0);
               end if;
               PrintSymbolType (File,
                                Scope,
                                Dictionary.CurrentSymbol (IndexIt));
               IndexIt := Dictionary.NextSymbol (IndexIt);
            end loop;
            SPARK_IO.Put_String (File, "] of ", 0);
            PrintSymbolType (File,
                             Scope,
                             Dictionary.GetArrayComponent (Sym));
            SPARK_IO.Put_Line (File, ";", 0);

            -- Wherever type String is declared, we also produce a
            -- declaration of a deferred constant that represents the
            -- Null String literal ""
            if Dictionary.IsPredefinedStringType (Sym) then
               SPARK_IO.Set_Col (File, Indent);
               SPARK_IO.Put_String (File, "const ", 0);
               SPARK_IO.Put_String (File, DAG_IO.Null_String_Literal_Name, 0);
               SPARK_IO.Put_Line   (File, " : string = pending;", 0);
            end if;

         end PrintArrayType;

         -------------------------

         -- Procedure to print abstract types: either "pending"
         -- or and FDL record declaration is printed depending on scope
         procedure PrintAbstractType (File  : in SPARK_IO.File_Type;
                                      Scope : in Dictionary.Scopes;
                                      Sym   : in Dictionary.Symbol)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     LexTokenManager.StringTable;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS from *,
         --#                                CommandLineData.Content,
         --#                                Dictionary.Dict,
         --#                                File,
         --#                                LexTokenManager.StringTable,
         --#                                Scope,
         --#                                Sym;
         is
            OwnVarSym : Dictionary.Symbol;

            procedure PrintRefinementRecord (File  : in SPARK_IO.File_Type;
                                             Scope : in Dictionary.Scopes;
                                             Sym   : in Dictionary.Symbol)
            --# global in     CommandLineData.Content;
            --#        in     Dictionary.Dict;
            --#        in     LexTokenManager.StringTable;
            --#        in out SPARK_IO.FILE_SYS;
            --# derives SPARK_IO.FILE_SYS from *,
            --#                                CommandLineData.Content,
            --#                                Dictionary.Dict,
            --#                                File,
            --#                                LexTokenManager.StringTable,
            --#                                Scope,
            --#                                Sym;
            is
               ComponentIt  : Dictionary.Iterator;
               FirstOne     : Boolean;
            begin
               SPARK_IO.Set_Col (File, Indent);
               SPARK_IO.Put_String (File, "type ", 0);
               PrintSymbol (File, Scope, Dictionary.GetType (Sym));
               SPARK_IO.Put_String (File, " = ", 0);
               SPARK_IO.Put_Line (File, "record", 0);
               ComponentIt := Dictionary.FirstConstituent (Sym);
               FirstOne := True;
               while not Dictionary.IsNullIterator (ComponentIt)
               loop
                  if FirstOne then
                     FirstOne := False;
                  else
                     SPARK_IO.Put_Line (File, ";", 0);
                  end if;
                  SPARK_IO.Set_Col (File, 3*Indent);
                  PrintSymbol (File,
                               Scope,
                               Dictionary.CurrentSymbol (ComponentIt));
                  SPARK_IO.Put_String (File, " : ", 0);
                  PrintSymbolType (File,
                                   Scope,
                                   Dictionary.CurrentSymbol (ComponentIt));
                  ComponentIt := Dictionary.NextSymbol (ComponentIt);
               end loop;
               SPARK_IO.New_Line (File, 1);
               SPARK_IO.Set_Col (File, 2*Indent);
               SPARK_IO.Put_String (File, "end", 0);
               SPARK_IO.Put_Line (File, ";", 0);
            end PrintRefinementRecord;

            -------------------------

         begin  --PrintAbstractType
            OwnVarSym := FindOwnVarMatchingThisType (Sym, Scope);
            if IsLocalOwnVariableWithRefinement (OwnVarSym, Scope) then
               PrintRefinementRecord (File, Scope, OwnVarSym);
            else
               SPARK_IO.Set_Col (File, Indent);
               SPARK_IO.Put_String (File, "type ", 0);
               PrintSymbol (File, Scope, Sym);
               SPARK_IO.Put_Line (File, " = pending;", 0);
            end if;
         end PrintAbstractType;

         -------------------------

      begin --PrintTypeDeclaration

         -- Time types are private from the point of view of the
         -- type system, but need special handling in FDL so that
         -- their relational and arithmetic operators are well-defined,
         -- so we deal with them first.
         if Sym = Dictionary.GetPredefinedTimeType or
           Sym = Dictionary.GetPredefinedTimeSpanType then
            PrintTimeType (File, Scope, Sym);

         elsif Dictionary.IsPrivateType (Sym, Scope) and then
           not Dictionary.TypeIsExtendedTagged (Sym) then
            PrintPrivateType (File, Scope, Sym);

         elsif Dictionary.TypeIsAbstractProof (Sym) then
            PrintAbstractType (File, Scope, Sym);

         elsif Dictionary.TypeIsIncompleteHere (Sym, Scope) and then
           not Dictionary.TypeIsExtendedTagged (Sym) then
            PrintIncompleteType (File, Scope, Sym);

         elsif Dictionary.IsSubtype (Sym) then
            -- Subtypes are those which actually appear in
            -- the VC or PF, since only base types are ever added
            -- by generatesuccessors. These can only occur in attributes
            -- so we want the actual type not the base type.
            PrintSubType (File, Scope, Sym);

         elsif Dictionary.IsSubprogramParameterConstraint (Sym) then
            -- subprogram parameter constraint symbols are used to communicate information about
            -- unconstrained array formal parameters.  They behave as subtypes for VC declaration purposes
            PrintSubType (File, Scope, Sym);

         else
            Typ := Dictionary.GetRootType (Sym);

            if Dictionary.TypeIsCharacter (Typ) then
               PrintCharacterType (File);

            elsif Dictionary.IsPredefinedIntegerType (Typ) or
              Dictionary.IsPredefinedFloatType (Typ) or
              Dictionary.TypeIsBoolean (Typ) then
               -- No action, since "integer", "real" and "boolean" are
               -- predefined in FDL
               null;

            elsif Dictionary.TypeIsEnumeration (Typ) then
               PrintEnumeratedType (File, Scope, Typ);

            elsif Dictionary.TypeIsRecord (Typ) or
              Dictionary.TypeIsExtendedTagged (Sym) then
               PrintRecordType (File,
                                Scope,
                                Typ,
                                Dictionary.IsPrivateType (Sym, Scope));

            elsif Dictionary.TypeIsArray (Typ) then
               PrintArrayType (File, Scope, Typ);

            elsif Dictionary.TypeIsInteger (Typ) or
              Dictionary.TypeIsReal (Typ) or
              Dictionary.TypeIsModular (Typ) then

               PrintSubType (File, Scope, Typ);

            elsif Dictionary.IsProtectedType (Typ) then
               -- Protected types need at least a "pending" declaration in FDL,
               -- so that later proof functions and variables make sense.
               PrintProtectedType (File, Scope, Typ);

            else
               Debug_Rank_Sym ("PrintTypeDeclaration does not know how to deal with ", Typ);
               SystemErrors.RTAssert (False,
                                      SystemErrors.AssertionFailure,
                                      "Unknown type in PrintTypeDeclaration");
            end if;
         end if;
      end PrintTypeDeclaration;

      --------------------------------------------

      procedure PrintProcedureDeclaration (File   : in SPARK_IO.File_Type;
                                           Scope  : in Dictionary.Scopes;
                                           Sym    : in Dictionary.Symbol)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in out Heap;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out Statistics.TableUsage;
      --# derives Heap,
      --#         Statistics.TableUsage from *,
      --#                                    Dictionary.Dict,
      --#                                    Heap,
      --#                                    Scope,
      --#                                    Sym &
      --#         SPARK_IO.FILE_SYS     from *,
      --#                                    CommandLineData.Content,
      --#                                    Dictionary.Dict,
      --#                                    File,
      --#                                    Heap,
      --#                                    LexTokenManager.StringTable,
      --#                                    Scope,
      --#                                    Sym;
      is
         ExpIt,
         ImpIt       : Dictionary.Iterator;
         FirstOne    : Boolean;
         Abstraction : Dictionary.Abstractions;
         ExportSym,
         ImportSym   : Dictionary.Symbol;

         -- These stacks are needed to revsers order of both exports and imports
         ImportStack : CStacks.Stack;
         TempCell    : Cells.Cell; --stack member

         function ExportName (SubprogSym,
                              ExportSym  : Dictionary.Symbol;
                              Scope      : Dictionary.Scopes)
                             return EStrings.T
         --# global in CommandLineData.Content;
         --#        in Dictionary.Dict;
         --#        in LexTokenManager.StringTable;
         is
            Result : EStrings.T;
         begin
            if Dictionary.IsFormalParameter (SubprogSym, ExportSym) then
               -- No prefix needed
               LexTokenManager.LexStringToString (Dictionary.GetSimpleName (ExportSym),
                                                  Result);
            else -- Its global and a prefix may be needed
               Result := GetName (ExportSym, Scope);
            end if;
            return EStrings.LowerCase (Result);
         end ExportName;

         ------------------------------------------


      begin -- PrintProcedureDeclaration
            -- Use stack to reverse order of exports
         CStacks.CreateStack (ImportStack);

         Abstraction := Dictionary.GetAbstraction (Sym, Scope);
         ExpIt := Dictionary.FirstExport (Abstraction, Sym);
         while not Dictionary.IsNullIterator (ExpIt) loop
            ExportSym := Dictionary.CurrentSymbol (ExpIt);
            SPARK_IO.Set_Col (File, Indent);
            SPARK_IO.Put_String (File, "function ", 0);
            PrintSymbol (File, Scope, Sym);
            SPARK_IO.Put_String (File, "__", 0);
            EStrings.PutString (File, ExportName (Sym, ExportSym, Scope));

            ImpIt := Dictionary.FirstDependency (Abstraction,
                                                 Sym,
                                                 ExportSym);
            while not Dictionary.IsNullIterator (ImpIt) loop
               Cells.Create_Cell (Heap, TempCell);
               Cells.Set_Symbol_Value (Heap,
                                     TempCell,
                                     Dictionary.CurrentSymbol (ImpIt));
               CStacks.Push (Heap, TempCell, ImportStack);
               ImpIt := Dictionary.NextSymbol (ImpIt);
            end loop;

            --now pop them one at a time and build an argument list for function
            FirstOne := True;
            while not CStacks.IsEmpty (ImportStack) loop
               ImportSym := Cells.Get_Symbol_Value (Heap, CStacks.Top (Heap, ImportStack));
               CStacks.Pop (Heap, ImportStack);
               if FirstOne then
                  SPARK_IO.Put_String (File, "(", 0);
                  FirstOne := False;
               else
                  SPARK_IO.Put_String (File, ", ", 0);
               end if;
               PrintSymbolType (File, Scope, ImportSym);
            end loop;
            if not FirstOne then
               SPARK_IO.Put_String (File, ")", 0);
            end if;
            SPARK_IO.Put_String (File, " : ", 0);
            PrintSymbolType (File, Scope, ExportSym);
            SPARK_IO.Put_Line (File, ";", 0);
            ExpIt := Dictionary.NextSymbol (ExpIt);
         end loop;
      end PrintProcedureDeclaration;

      ---------------------------------------------------------

      procedure PrintFunctionDeclaration (File   : in SPARK_IO.File_Type;
                                          Scope  : in Dictionary.Scopes;
                                          Sym    : in Dictionary.Symbol)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives SPARK_IO.FILE_SYS from *,
      --#                                CommandLineData.Content,
      --#                                Dictionary.Dict,
      --#                                File,
      --#                                LexTokenManager.StringTable,
      --#                                Scope,
      --#                                Sym;
      is
         ArgumentIt  : Dictionary.Iterator;
         FirstOne    : Boolean;
      begin
         if Dictionary.GetRegion (Scope) = Sym then
            SPARK_IO.Set_Col (File, Indent);
            SPARK_IO.Put_String (File, "var ", 0);
            PrintSymbol (File, Scope, Sym);
            SPARK_IO.Put_String (File, " : ", 0);
            PrintSymbolType (File, Scope, Sym);
            SPARK_IO.Put_Line (File, ";", 0);
         else
            SPARK_IO.Set_Col (File, Indent);
            SPARK_IO.Put_String (File, "function ", 0);
            PrintSymbol (File, Scope, Sym);

            FirstOne := True;
            ArgumentIt := Dictionary.FirstSubprogramParameter (Sym);
            while not Dictionary.IsNullIterator (ArgumentIt) loop
               if FirstOne then
                  SPARK_IO.Put_String (File, "(", 0);
                  FirstOne := False;
               else
                  SPARK_IO.Put_String (File, ", ", 0);
               end if;
               PrintSymbolType (File, Scope, Dictionary.CurrentSymbol (ArgumentIt));
               ArgumentIt := Dictionary.NextSymbol (ArgumentIt);
            end loop;

            if not FirstOne then
               SPARK_IO.Put_String (File, ")", 0);
            end if;
            SPARK_IO.Put_String (File, " : ", 0);
            PrintSymbolType (File, Scope, Sym);
            SPARK_IO.Put_Line (File, ";", 0);
         end if;
      end PrintFunctionDeclaration;

      ----------------------------------------------

   begin --PrintDeclaration
      if Dictionary.IsVariable (Sym) then
         PrintVariableDeclaration (File, Scope, Sym);
      elsif Dictionary.IsConstant (Sym) then
         PrintConstantDeclaration (File, Scope, Sym);
      elsif Dictionary.IsKnownDiscriminant (Sym) then
         PrintConstantDeclaration (File, Scope, Sym);
      elsif Dictionary.IsTypeMark (Sym) then
         PrintTypeDeclaration (File, Scope, Sym);
      elsif Dictionary.IsProcedure (Sym) then
         PrintProcedureDeclaration (File, Scope, Sym);
      elsif Dictionary.IsFunction (Sym) then
         PrintFunctionDeclaration (File, Scope, Sym);
      else
         null;
      end if;
   end PrintDeclaration;

   ----------------------------------------------

   procedure PrintAttributeDeclarations (File  : in SPARK_IO.File_Type;
                                         Scope : in Dictionary.Scopes)
   --# global in     AttributeList;
   --#        in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     Heap;
   --#        in     LexTokenManager.StringTable;
   --#        in     RuleFamilyName;
   --#        in     RuleFile;
   --#        in     TypeList;
   --#        in     WriteRules;
   --#        in out LHeap;
   --#        in out RuleCounter;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives LHeap             from *,
   --#                                AttributeList,
   --#                                Heap,
   --#                                TypeList,
   --#                                WriteRules &
   --#         RuleCounter       from *,
   --#                                AttributeList,
   --#                                Dictionary.Dict,
   --#                                Heap,
   --#                                WriteRules &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                AttributeList,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                File,
   --#                                Heap,
   --#                                LexTokenManager.StringTable,
   --#                                RuleCounter,
   --#                                RuleFamilyName,
   --#                                RuleFile,
   --#                                Scope,
   --#                                WriteRules;
   is
      CurrentAttrib,
      PrevAttrib : Cells.Cell;

      ----------------------------

      procedure PrintOneAttribute (TickCell : in Cells.Cell;
                                   PrevCell : in Cells.Cell)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     File;
      --#        in     Heap;
      --#        in     LexTokenManager.StringTable;
      --#        in     RuleFamilyName;
      --#        in     RuleFile;
      --#        in     Scope;
      --#        in     TypeList;
      --#        in     WriteRules;
      --#        in out LHeap;
      --#        in out RuleCounter;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives LHeap             from *,
      --#                                Heap,
      --#                                TickCell,
      --#                                TypeList,
      --#                                WriteRules &
      --#         RuleCounter       from *,
      --#                                Dictionary.Dict,
      --#                                Heap,
      --#                                PrevCell,
      --#                                TickCell,
      --#                                WriteRules &
      --#         SPARK_IO.FILE_SYS from *,
      --#                                CommandLineData.Content,
      --#                                Dictionary.Dict,
      --#                                File,
      --#                                Heap,
      --#                                LexTokenManager.StringTable,
      --#                                PrevCell,
      --#                                RuleCounter,
      --#                                RuleFamilyName,
      --#                                RuleFile,
      --#                                Scope,
      --#                                TickCell,
      --#                                WriteRules;
      is

         procedure PrintAttributeConstant
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     File;
         --#        in     Heap;
         --#        in     LexTokenManager.StringTable;
         --#        in     Scope;
         --#        in     TickCell;
         --#        in     TypeList;
         --#        in     WriteRules;
         --#        in out LHeap;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives LHeap             from *,
         --#                                Heap,
         --#                                TickCell,
         --#                                TypeList,
         --#                                WriteRules &
         --#         SPARK_IO.FILE_SYS from *,
         --#                                CommandLineData.Content,
         --#                                Dictionary.Dict,
         --#                                File,
         --#                                Heap,
         --#                                LexTokenManager.StringTable,
         --#                                Scope,
         --#                                TickCell;
         is
            LexStr     : LexTokenManager.LexString;
            ExStr      : EStrings.T;
            PrefixCell : Cells.Cell;
            HasBase    : Boolean;
            AlreadyPresent,
            Ok         : Boolean;

            procedure PrintAttributeType (File   : in SPARK_IO.File_Type;
                                          Scope  : in Dictionary.Scopes;
                                          Sym    : in Dictionary.Symbol;
                                          Attrib : in LexTokenManager.LexString)
            --# global in     CommandLineData.Content;
            --#        in     Dictionary.Dict;
            --#        in     LexTokenManager.StringTable;
            --#        in out SPARK_IO.FILE_SYS;
            --# derives SPARK_IO.FILE_SYS from *,
            --#                                Attrib,
            --#                                CommandLineData.Content,
            --#                                Dictionary.Dict,
            --#                                File,
            --#                                LexTokenManager.StringTable,
            --#                                Scope,
            --#                                Sym;
            is
               TheType : Dictionary.Symbol;
            begin
               if Dictionary.IsTypeMark (Sym) then
                  TheType := Sym;
               else
                  TheType := Dictionary.GetType (Sym);
               end if;

               if Dictionary.IsArrayAttribute (Attrib, TheType) then
                  PrintSymbolType (File,
                                   Scope,
                                   Dictionary.GetArrayAttributeType (Attrib,
                                                                     TheType,
                                                                     1));
                  --note assumption of first index here pending expression evaluation
               else
                  PrintSymbolType (File,
                                   Scope,
                                   Dictionary.GetScalarAttributeType (Attrib,
                                                                      TheType));
               end if;
            end PrintAttributeType;

         begin --PrintAttributeConstant
            HasBase := False;
            PrefixCell := Cells.Get_A_Ptr (Heap, TickCell);
            if Cells.Get_Kind (Heap, PrefixCell) = Cells.Op then
               HasBase := True;
               PrefixCell := Cells.Get_A_Ptr (Heap, PrefixCell);
            end if;

            SPARK_IO.Set_Col (File, Indent);
            SPARK_IO.Put_String (File, "const ", 0);

            -- Make copy of symbol in list of type for later rule generation
            if WriteRules then
               --# accept Flow, 10, Ok, "Expected ineffective assignment to Ok" &
               --#        Flow, 10, AlreadyPresent, "Expected ineffective assignment to AlreadyPresent";

               --The Heap contains the full details of each referenced entity
               --(integer, myvariable, etc.), including information that
               --allows working out if the entity involves a base part
               --(__base) or the form of any any tick part (__first, __last,
               --etc.). With this fine grain information, FDL is generated
               --only for the specific entities that are referenced.
               --However, below, only the referenced entity name is recorded
               --and passed to rule generation (by referencing the cell
               --entity name in a list). With only this larger grain information,
               --every rule is generated for an entity (all base forms and
               --tick forms). Care needs to be taken to ensure that FDL is
               --generated for each entity that will be referenced in the
               --generated rules.
               Lists.AddValue (LHeap, -- Expect 2 ineffective assignments
                               TypeList,
                               Cells.Get_Natural_Value (Heap, PrefixCell),
                                 -- to get
                               AlreadyPresent,  --values not needed here
                               Ok);
               --# end accept;
            end if;

            PrintSymbol (File,
                         Scope,
                         Cells.Get_Symbol_Value (Heap, PrefixCell));
            if HasBase then
               SPARK_IO.Put_String (File, "__base", 0);
            end if;
            SPARK_IO.Put_String (File, "__", 0);
            LexStr := Cells.Get_Lex_Str (Heap, Cells.Get_B_Ptr (Heap, TickCell));
            LexTokenManager.LexStringToString (LexStr, ExStr);
            EStrings.PutString (File,
                                       EStrings.LowerCase (ExStr));
            SPARK_IO.Put_String (File, " : ", 0);
            PrintAttributeType (File,
                                Scope,
                                Cells.Get_Symbol_Value (Heap, PrefixCell),
                                LexStr);
            SPARK_IO.Put_Line (File, " = pending; ", 0);
            --# accept Flow, 33, Ok, "Expected Ok to be neither referenced nor exported" &
            --#        Flow, 33, AlreadyPresent, "Expected AlreadyPresent to be neither referenced nor exported";
         end PrintAttributeConstant; -- Expect 2 unused vars

         ------------------

         procedure PrintAttributeFunction
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     File;
         --#        in     Heap;
         --#        in     LexTokenManager.StringTable;
         --#        in     PrevCell;
         --#        in     RuleFamilyName;
         --#        in     RuleFile;
         --#        in     Scope;
         --#        in     TickCell;
         --#        in     WriteRules;
         --#        in out RuleCounter;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives RuleCounter       from *,
         --#                                Dictionary.Dict,
         --#                                Heap,
         --#                                PrevCell,
         --#                                TickCell,
         --#                                WriteRules &
         --#         SPARK_IO.FILE_SYS from *,
         --#                                CommandLineData.Content,
         --#                                Dictionary.Dict,
         --#                                File,
         --#                                Heap,
         --#                                LexTokenManager.StringTable,
         --#                                PrevCell,
         --#                                RuleCounter,
         --#                                RuleFamilyName,
         --#                                RuleFile,
         --#                                Scope,
         --#                                TickCell,
         --#                                WriteRules;
         is
            LexStr : LexTokenManager.LexString;
            ExStr  : EStrings.T;
            Selector  : EStrings.T;
            Brackets  : EStrings.T;
            RecVariable : Dictionary.Symbol;

            procedure FormatRecordSelector (FieldName : in     Dictionary.Symbol;
                                            Selector  :    out EStrings.T;
                                            Brackets  :    out EStrings.T;
                                            Variable  :    out Dictionary.Symbol)
            -- Analyse the fieldname, creating a field selector to access it,
            -- and a balancing set of close brackets
            -- also pick out the variable name
            -- This is also safe to call when the variable is not a record
            --# global in Dictionary.Dict;
            --#        in LexTokenManager.StringTable;
            --# derives Brackets,
            --#         Variable from Dictionary.Dict,
            --#                       FieldName &
            --#         Selector from Dictionary.Dict,
            --#                       FieldName,
            --#                       LexTokenManager.StringTable;
            is
               Field : Dictionary.Symbol;                -- the current field
               Sel   : EStrings.T;   -- the field selector
               Brk   : EStrings.T;   -- the close bracket list
               EStr  : EStrings.T;   -- LexString version of field name
               Token : LexTokenManager.LexString;        --
            begin
               Field := FieldName;
               Sel := EStrings.EmptyString;
               Brk := EStrings.EmptyString;
               loop
                  -- exit when we've reached the variable name
                  exit when not Dictionary.IsRecordSubcomponent (Field);
                  -- add the field selector to the end - note this serendipitously
                  -- generates the nested selectors in the corrrect order
                  EStrings.AppendString (Sel, "fld_");
                  Token := Dictionary.GetSimpleName (Field);
                  LexTokenManager.LexStringToString (Token, EStr);
                  EStr := EStrings.LowerCase (EStr);
                  EStrings.AppendExaminerString (Sel, EStr);
                  -- add a closing bracket, to match
                  EStrings.AppendString (Sel, "(");
                  EStrings.AppendString (Brk, ")");
                  -- get the enclosing field/variable
                  Field := Dictionary.GetEnclosingObject (Field);
               end loop;
               Selector := Sel;
               Brackets := Brk;
               Variable := Field;
            end FormatRecordSelector;

            procedure PrintFunctionHeader
            -- print the common function declaration header
            --# global in     CommandLineData.Content;
            --#        in     Dictionary.Dict;
            --#        in     ExStr;
            --#        in     File;
            --#        in     Heap;
            --#        in     LexTokenManager.StringTable;
            --#        in     Scope;
            --#        in     TickCell;
            --#        in out SPARK_IO.FILE_SYS;
            --# derives SPARK_IO.FILE_SYS from *,
            --#                                CommandLineData.Content,
            --#                                Dictionary.Dict,
            --#                                ExStr,
            --#                                File,
            --#                                Heap,
            --#                                LexTokenManager.StringTable,
            --#                                Scope,
            --#                                TickCell;
            is
            begin
               SPARK_IO.Set_Col (File, Indent);
               SPARK_IO.Put_String (File, "function ", 0);
               PrintSymbol (File,
                            Scope,
                            Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
               SPARK_IO.Put_String (File, "__", 0);
               EStrings.PutString (File,
                                          EStrings.LowerCase (ExStr));
            end PrintFunctionHeader;

         begin -- PrintAttributeFunction
            -- Function attributes will have had 'BASE eliminated in model build
            LexStr := Cells.Get_Lex_Str (Heap, Cells.Get_B_Ptr (Heap, TickCell));
            LexTokenManager.LexStringToString (LexStr, ExStr);

            -- Support for 'Valid in SPARK95
            if LexStr = LexTokenManager.ValidToken then
               PrintFunctionHeader;
               SPARK_IO.Put_Char (File, '(');
               PrintSymbolType (File,
                                Scope,
                                Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
               SPARK_IO.Put_String (File, ") : boolean;", 0);

               -- and associated rules
               if WriteRules then
                  -- first rule
                  PrintRuleName (RuleFile);
                  SPARK_IO.Put_String (RuleFile, "X <= ", 0);
                  PrintSymbol (RuleFile,
                               Scope,
                               Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
                  SPARK_IO.Put_Line (RuleFile, "__last may_be_deduced_from", 0);
                  SPARK_IO.Put_String (RuleFile, "     [ ", 0);
                  PrintSymbol (RuleFile,
                               Scope,
                               Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
                  SPARK_IO.Put_Line (RuleFile, "__valid(X) ].", 0);
                  -- second rule
                  PrintRuleName (RuleFile);
                  SPARK_IO.Put_String (RuleFile, "X >= ", 0);
                  PrintSymbol (RuleFile,
                               Scope,
                               Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
                  SPARK_IO.Put_Line (RuleFile, "__first may_be_deduced_from", 0);
                  SPARK_IO.Put_String (RuleFile, "     [ ", 0);
                  PrintSymbol (RuleFile,
                               Scope,
                               Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
                  SPARK_IO.Put_Line (RuleFile, "__valid(X) ].", 0);
               end if;

            -- Support for 'Always_Valid
            elsif LexStr = LexTokenManager.Always_ValidToken then

               -- Only output the declarations and generic rules if this is the
               -- first occurance of 'Always_Valid for this type
               -- The simplification rule is always output, since it contains
               -- the variable name, and possibly the field selectors
               if Cells.Is_Null_Cell (PrevCell) or else
                 (Cells.Get_Lex_Str (Heap, Cells.Get_B_Ptr (Heap, PrevCell))
                   /= LexTokenManager.Always_ValidToken) or else
                 (Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell))
                   /= Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, PrevCell))) then
                  PrintFunctionHeader;
                  SPARK_IO.Put_Char (File, '(');
                  PrintSymbolType (File,
                                   Scope,
                                   Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
                  SPARK_IO.Put_String (File, ") : boolean;", 0);
                  -- and the associated rules that must only be output once per type
                  -- MCA, could this be combined with the 'Valid code above?
                  if WriteRules then
                     -- first rule: X <= type__last may_be_deduced_from [ type__always_valid(X) ].
                     PrintRuleName (RuleFile);
                     SPARK_IO.Put_String (RuleFile, "X <= ", 0);
                     PrintSymbol (RuleFile,
                                  Scope,
                                  Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
                     SPARK_IO.Put_Line (RuleFile, "__last may_be_deduced_from", 0);
                     SPARK_IO.Put_String (RuleFile, "     [ ", 0);
                     PrintSymbol (RuleFile,
                                  Scope,
                                  Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
                     SPARK_IO.Put_Line (RuleFile, "__always_valid(X) ].", 0);
                     -- second rule: X >= type__first may_be_deduced_from [ type__always_valid(X) ].
                     PrintRuleName (RuleFile);
                     SPARK_IO.Put_String (RuleFile, "X >= ", 0);
                     PrintSymbol (RuleFile,
                                  Scope,
                                  Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
                     SPARK_IO.Put_Line (RuleFile, "__first may_be_deduced_from", 0);
                     SPARK_IO.Put_String (RuleFile, "     [ ", 0);
                     PrintSymbol (RuleFile,
                                  Scope,
                                  Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
                     SPARK_IO.Put_Line (RuleFile, "__always_valid(X) ].", 0);
                  end if;
               end if;

               if WriteRules then
                  -- third rule: type__always_valid(variable__tail(X))
                  --                       may_be_deduced_from type__always_valid(X).
                  FormatRecordSelector (Cells.Get_Assoc_Var (Heap, Cells.Get_A_Ptr (Heap, TickCell)),
                                        Selector,
                                        Brackets,
                                        RecVariable);

--                  Debug.PrintMsg ("FormatRecordSelector: ", False);
--                  EStrings.PutString (SPARK_IO.Standard_Output, Selector);
--                  Debug.PrintMsg ("", True);
--                  Debug.PrintSym ("FormatRecordSelector Sym: ", RecVariable);

                  PrintRuleName (RuleFile);
                  PrintSymbol (RuleFile,        -- type
                               Scope,
                               Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
                  SPARK_IO.Put_String (RuleFile, "__always_valid(", 0);
                  EStrings.PutString (RuleFile, Selector);
                  PrintSymbol (RuleFile,        -- variable
                               Scope,
                               RecVariable);

                  SPARK_IO.Put_String (RuleFile, "__tail(X))", 0);

                  EStrings.PutString (RuleFile, Brackets);
                  SPARK_IO.New_Line (RuleFile, 1);
                  SPARK_IO.Put_String (RuleFile, "     may_be_deduced_from [", 0);
                  PrintSymbol (RuleFile,        -- type
                               Scope,
                               Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
                  SPARK_IO.Put_String (RuleFile, "__always_valid(", 0);
                  EStrings.PutString (RuleFile, Selector);
                  SPARK_IO.Put_String (RuleFile, "X)", 0);

                  EStrings.PutString (RuleFile, Brackets);
                  SPARK_IO.Put_Line (RuleFile, "].", 0);
               end if;

            elsif LexStr = LexTokenManager.MinToken or else
                  LexStr = LexTokenManager.MaxToken then
               PrintFunctionHeader;
               SPARK_IO.Put_Char (File, '(');
               PrintSymbolType (File,
                                Scope,
                                Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
               SPARK_IO.Put_String (File, ", ", 0);
               PrintSymbolType (File,
                                Scope,
                                Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
               SPARK_IO.Put_String (File, ") : ", 0);
               PrintSymbolType (File,
                                Scope,
                                Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
               SPARK_IO.Put_String (File, ";", 0);

               -- now produce rules for Min and Max
               if WriteRules then
                  -- first case
                  PrintRuleName (RuleFile);
                  PrintSymbol (RuleFile,
                               Scope,
                               Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
                  SPARK_IO.Put_String (RuleFile, "__", 0);
                  LexStr := Cells.Get_Lex_Str (Heap, Cells.Get_B_Ptr (Heap, TickCell));
                  LexTokenManager.LexStringToString (LexStr, ExStr);
                  EStrings.PutString (RuleFile,
                                             EStrings.LowerCase (ExStr));
                  SPARK_IO.Put_String (RuleFile, "(X, Y) may_be_replaced_by ", 0);
                  if LexStr = LexTokenManager.MaxToken then
                     SPARK_IO.Put_Char (RuleFile, 'X');
                  else
                     SPARK_IO.Put_Char (RuleFile, 'Y');
                  end if;
                  SPARK_IO.Put_Line (RuleFile, " if [X >= Y].", 0);
                  -- second case
                  PrintRuleName (RuleFile);
                  PrintSymbol (RuleFile,
                               Scope,
                               Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
                  SPARK_IO.Put_String (RuleFile, "__", 0);
                  LexStr := Cells.Get_Lex_Str (Heap, Cells.Get_B_Ptr (Heap, TickCell));
                  LexTokenManager.LexStringToString (LexStr, ExStr);
                  EStrings.PutString (RuleFile,
                                             EStrings.LowerCase (ExStr));
                  SPARK_IO.Put_String (RuleFile, "(X, Y) may_be_replaced_by ", 0);
                  if LexStr = LexTokenManager.MaxToken then
                     SPARK_IO.Put_Char (RuleFile, 'Y');
                  else
                     SPARK_IO.Put_Char (RuleFile, 'X');
                  end if;
                  SPARK_IO.Put_Line (RuleFile, " if [Y >= X].", 0);
                  -- type range of output
                  -- lower bound
                  PrintRuleName (RuleFile);
                  PrintSymbol (RuleFile,
                               Scope,
                               Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
                  SPARK_IO.Put_String (RuleFile, "__", 0);
                  LexStr := Cells.Get_Lex_Str (Heap, Cells.Get_B_Ptr (Heap, TickCell));
                  LexTokenManager.LexStringToString (LexStr, ExStr);
                  EStrings.PutString (RuleFile,
                                             EStrings.LowerCase (ExStr));
                  SPARK_IO.Put_String (RuleFile, "(X, Y) >= ", 0);
                  PrintSymbol (RuleFile,
                               Scope,
                               Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
                  SPARK_IO.Put_Line (RuleFile, "__first may_be_deduced.", 0);
                  -- upper bound
                  PrintRuleName (RuleFile);
                  PrintSymbol (RuleFile,
                               Scope,
                               Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
                  SPARK_IO.Put_String (RuleFile, "__", 0);
                  LexStr := Cells.Get_Lex_Str (Heap, Cells.Get_B_Ptr (Heap, TickCell));
                  LexTokenManager.LexStringToString (LexStr, ExStr);
                  EStrings.PutString (RuleFile,
                                             EStrings.LowerCase (ExStr));
                  SPARK_IO.Put_String (RuleFile, "(X, Y) <= ", 0);
                  PrintSymbol (RuleFile,
                               Scope,
                               Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
                  SPARK_IO.Put_Line (RuleFile, "__last may_be_deduced.", 0);
               end if; -- Min and Max rules

            -- side effect modelling attributes for stream reads
            elsif LexStr =  LexTokenManager.TailToken then
               PrintFunctionHeader;
               SPARK_IO.Put_Char (File, '(');
               PrintSymbolType (File,
                                Scope,
                                Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
               SPARK_IO.Put_String (File, ") : ", 0);
               PrintSymbolType (File,
                                Scope,
                                Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
               SPARK_IO.Put_String (File, ";", 0);

            -- side effect modelling attributes for streams writes
            elsif LexStr = LexTokenManager.AppendToken then
               PrintFunctionHeader;
               SPARK_IO.Put_Char (File, '(');
               PrintSymbolType (File,
                                Scope,
                                Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
               SPARK_IO.Put_String (File, ", ", 0);
               PrintSymbolType (File,
                                Scope,
                                Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
               SPARK_IO.Put_String (File, ") : ", 0);
               PrintSymbolType (File,
                                Scope,
                                Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
               SPARK_IO.Put_String (File, ";", 0);

            elsif LexStr = LexTokenManager.PosToken then
               PrintFunctionHeader;

               SPARK_IO.Put_Char (File, '(');
               PrintSymbol (File,
                            Scope,
                            Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
               SPARK_IO.Put_String (File, ") : integer;", 0);
               if WriteRules then
                  if Dictionary.TypeIsInteger (Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell))) or else
                    Dictionary.TypeIsModular (Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell))) then
                     PrintRuleName (RuleFile);
                     PrintSymbol (RuleFile,
                                  Scope,
                                  Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
                     SPARK_IO.Put_Line (RuleFile, "__pos(X) may_be_replaced_by X .", 0);
                  end if;
               end if;
            elsif LexStr = LexTokenManager.CeilingToken or LexStr = LexTokenManager.FloorToken then
               PrintFunctionHeader;
               SPARK_IO.Put_String (File, "(real) : real;", 0);
            else -- Must be 'Val
               PrintFunctionHeader;
               SPARK_IO.Put_String (File, "(integer) : ", 0);
               PrintSymbol (File,
                            Scope,
                            Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
               SPARK_IO.Put_Char (File, ';');
               if WriteRules then
                  if Dictionary.TypeIsInteger (Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell))) or else
                    Dictionary.TypeIsModular (Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell))) then
                     PrintRuleName (RuleFile);
                     PrintSymbol (RuleFile,
                                  Scope,
                                  Cells.Get_Symbol_Value (Heap, Cells.Get_A_Ptr (Heap, TickCell)));
                     SPARK_IO.Put_Line (RuleFile, "__val(X) may_be_replaced_by X .", 0);
                  end if;
               end if;
            end if;
         end PrintAttributeFunction;

         ------------------

      begin --PrintOneAttribute
         if Cells.Get_Kind (Heap, Cells.Get_B_Ptr (Heap, TickCell)) =
            Cells.Attrib_Value then
            PrintAttributeConstant;
         else
            PrintAttributeFunction;
         end if;
      end PrintOneAttribute;

      ----------------------------

   begin --PrintAttributeDeclarations
      CurrentAttrib := AttributeList;
      PrevAttrib := Cells.Null_Cell;
      while not Cells.Is_Null_Cell (Cells.Get_A_Ptr (Heap, CurrentAttrib))
      loop
         CurrentAttrib := Cells.Get_A_Ptr (Heap, CurrentAttrib);
         PrintOneAttribute (Cells.Get_C_Ptr (Heap, CurrentAttrib),
                            Cells.Get_C_Ptr (Heap, PrevAttrib));
         PrevAttrib := CurrentAttrib;
      end loop;
   end PrintAttributeDeclarations;

   ----------------------------------------------

   procedure PrintBitwiseOpDeclarations (File,
                                         RuleFile   : in SPARK_IO.File_Type;
                                         WriteRules : in Boolean;
                                         Scope      : in Dictionary.Scopes)
   --# global in     BitwiseOpList;
   --#        in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     Heap;
   --#        in     LexTokenManager.StringTable;
   --#        in     RuleFamilyName;
   --#        in out RuleCounter;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives RuleCounter       from *,
   --#                                BitwiseOpList,
   --#                                Dictionary.Dict,
   --#                                Heap,
   --#                                WriteRules &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                BitwiseOpList,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                File,
   --#                                Heap,
   --#                                LexTokenManager.StringTable,
   --#                                RuleCounter,
   --#                                RuleFamilyName,
   --#                                RuleFile,
   --#                                Scope,
   --#                                WriteRules;
   is
      CurrentOp : Cells.Cell;

      ----------------------------

      procedure PrintOneBitwiseOp (OpCell : in Cells.Cell)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     File;
      --#        in     Heap;
      --#        in     LexTokenManager.StringTable;
      --#        in     RuleFamilyName;
      --#        in     RuleFile;
      --#        in     Scope;
      --#        in     WriteRules;
      --#        in out RuleCounter;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives RuleCounter       from *,
      --#                                Dictionary.Dict,
      --#                                Heap,
      --#                                OpCell,
      --#                                WriteRules &
      --#         SPARK_IO.FILE_SYS from *,
      --#                                CommandLineData.Content,
      --#                                Dictionary.Dict,
      --#                                File,
      --#                                Heap,
      --#                                LexTokenManager.StringTable,
      --#                                OpCell,
      --#                                RuleCounter,
      --#                                RuleFamilyName,
      --#                                RuleFile,
      --#                                Scope,
      --#                                WriteRules;
      is
         IndexSym : Dictionary.Symbol;
         IndexFirst,
         IndexLast : LexTokenManager.LexString;

         -- procedure to print the common part of rules such as "thetype__and(X,Y)"
         procedure PrintBitwiseOpFunction (TypeSym  : in Dictionary.Symbol;
                                           Operator : in SPSymbols.SPSymbol)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     LexTokenManager.StringTable;
         --#        in     RuleFile;
         --#        in     Scope;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS from *,
         --#                                CommandLineData.Content,
         --#                                Dictionary.Dict,
         --#                                LexTokenManager.StringTable,
         --#                                Operator,
         --#                                RuleFile,
         --#                                Scope,
         --#                                TypeSym;
         is
         begin -- PrintBitwiseOpFunction
            PrintSymbol (RuleFile,
                         Scope,
                         TypeSym);
            SPARK_IO.Put_String (RuleFile, "__", 0);
            case Operator is
               when SPSymbols.RWand => SPARK_IO.Put_String (RuleFile, "and(", 0);
               when SPSymbols.RWor  => SPARK_IO.Put_String (RuleFile, "or(",  0);
               when SPSymbols.RWxor => SPARK_IO.Put_String (RuleFile, "xor(", 0);
               when SPSymbols.RWnot => SPARK_IO.Put_String (RuleFile, "not(",  0);
               when others          => SPARK_IO.Put_String (RuleFile, "undef_op_value(", 0);
            end case;
            SPARK_IO.Put_String (RuleFile, "X", 0);
            if Operator /= SPSymbols.RWnot then
               SPARK_IO.Put_String (RuleFile, ", Y", 0);
            end if;
            SPARK_IO.Put_String (RuleFile, ") ", 0);
         end PrintBitwiseOpFunction;

      begin -- PrintOneBitwiseOp
         if Dictionary.TypeIsArray (Cells.Get_Symbol_Value (Heap, OpCell)) then
            SPARK_IO.Set_Col (File, Indent);
            SPARK_IO.Put_String (File, "function ", 0);
            PrintSymbol (File,
                         Scope,
                         Cells.Get_Symbol_Value (Heap, OpCell));
            SPARK_IO.Put_String (File, "__", 0);
            case Cells.Get_Op_Symbol (Heap, OpCell) is
               when SPSymbols.RWand => SPARK_IO.Put_String (File, "and(", 0);
               when SPSymbols.RWor  => SPARK_IO.Put_String (File, "or(",  0);
               when SPSymbols.RWxor => SPARK_IO.Put_String (File, "xor(", 0);
               when SPSymbols.RWnot => SPARK_IO.Put_String (File, "not(",  0);
               when others          => SPARK_IO.Put_String (File, "undef_op_value(", 0);
            end case;
            --# assert True;
            PrintSymbol (File,
                         Scope,
                         Cells.Get_Symbol_Value (Heap, OpCell));
            if Cells.Get_Op_Symbol (Heap, OpCell) /= SPSymbols.RWnot then
               SPARK_IO.Put_String (File, ", ", 0);
               PrintSymbol (File,
                            Scope,
                            Cells.Get_Symbol_Value (Heap, OpCell));
            end if;
            SPARK_IO.Put_String (File, ") :  ", 0);
            PrintSymbol (File,
                         Scope,
                         Cells.Get_Symbol_Value (Heap, OpCell));
            SPARK_IO.Put_Char (File, ';');
         end if;

         --# assert True;
         --now do rules for it (if not doing path functions)
         if WriteRules then
            -- we want rules for array bitwise ops
            if Dictionary.TypeIsArray (Cells.Get_Symbol_Value (Heap, OpCell)) then
               -- do array bitwise ops
               IndexSym := Dictionary.GetArrayIndex (Cells.Get_Symbol_Value (Heap, OpCell), 1);
               IndexFirst := Dictionary.GetScalarAttributeValue (False,
                                                                 LexTokenManager.FirstToken,
                                                                 IndexSym);
               IndexLast := Dictionary.GetScalarAttributeValue (False,
                                                                LexTokenManager.LastToken,
                                                                IndexSym);

               PrintRuleName (RuleFile);
               SPARK_IO.Put_String (RuleFile, "element(", 0);
               PrintBitwiseOpFunction (Cells.Get_Symbol_Value (Heap, OpCell),
                                       Cells.Get_Op_Symbol (Heap, OpCell));
               SPARK_IO.Put_Line (RuleFile, ", [I]) may_be_replaced_by ", 0);
               --# assert True;
               if Cells.Get_Op_Symbol (Heap, OpCell) = SPSymbols.RWnot then
                  SPARK_IO.Put_String (RuleFile, "     not element(X, [I]) ", 0);
               elsif Cells.Get_Op_Symbol (Heap, OpCell) = SPSymbols.RWxor then
                  SPARK_IO.Put_String (RuleFile,
                                       "     (element(X, [I]) or element(Y, [I])) " &
                                       "and (not (element(X, [I]) and element(Y, [I])))",
                                       0);
               else -- And or Or binary op directly reproducible in fdl
                  SPARK_IO.Put_String (RuleFile, "     element(X, [I]) ", 0);
                  case Cells.Get_Op_Symbol (Heap, OpCell) is
                     when SPSymbols.RWand => SPARK_IO.Put_String (RuleFile, "and ", 0);
                     when SPSymbols.RWor  => SPARK_IO.Put_String (RuleFile, "or ",  0);
                     when others          => SPARK_IO.Put_String (RuleFile, "undef_op_value ", 0);
                  end case;
                  SPARK_IO.Put_String (RuleFile, "element(Y, [I]) ", 0);
               end if;
               SPARK_IO.New_Line (RuleFile, 1);
               SPARK_IO.Put_String (RuleFile, "     if [", 0);
               ELStrings.PutString (RuleFile, GetValue (IndexFirst, IndexSym, Scope));
               SPARK_IO.Put_String (RuleFile, " <= I, I <= ", 0);
               ELStrings.PutString (RuleFile, GetValue (IndexLast, IndexSym, Scope));
               SPARK_IO.Put_String (RuleFile, "]", 0);
               EndARule (RuleFile);

            end if;
         end if;
      end PrintOneBitwiseOp;

      ----------------------------

   begin --PrintBitwiseOpDeclarations
      CurrentOp := BitwiseOpList;
      while not Cells.Is_Null_Cell (Cells.Get_A_Ptr (Heap, CurrentOp)) loop
         CurrentOp := Cells.Get_A_Ptr (Heap, CurrentOp);
         PrintOneBitwiseOp (Cells.Get_C_Ptr (Heap, CurrentOp));
      end loop;
   end PrintBitwiseOpDeclarations;

   ----------------------------------------------

   procedure PrintExportVariableDeclarations (File  : in SPARK_IO.File_Type;
                                              Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     Heap;
   --#        in     LexTokenManager.StringTable;
   --#        in     ProcedureExportList;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives SPARK_IO.FILE_SYS from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                File,
   --#                                Heap,
   --#                                LexTokenManager.StringTable,
   --#                                ProcedureExportList,
   --#                                Scope;
   is
      ListElement    : Cells.Cell;
      CountStr       : EStrings.T;
   begin
      ListElement := Cells.Get_A_Ptr (Heap, ProcedureExportList);
      while not Cells.Is_Null_Cell (ListElement) loop

         SPARK_IO.Set_Col (File, Indent);
         SPARK_IO.Put_String (File, "var ", 0);
         PrintSymbol (File,
                      Scope,
                      Cells.Get_Symbol_Value (Heap, ListElement));
         SPARK_IO.Put_String (File, "__", 0);
         LexTokenManager.LexStringToString (Cells.Get_Lex_Str (Heap, ListElement),
                                            CountStr);
         EStrings.PutString (File, CountStr);

         SPARK_IO.Put_String (File, " : ", 0);
         PrintSymbolType (File,
                          Scope,
                          Cells.Get_Symbol_Value (Heap, ListElement));
         SPARK_IO.Put_Line (File, ";", 0);

         ListElement := Cells.Get_A_Ptr (Heap, ListElement);
      end loop;
   end PrintExportVariableDeclarations;

   ----------------------------------------------

   procedure PrintReturnSymbol (File  : in SPARK_IO.File_Type;
                                Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     Heap;
   --#        in     LexTokenManager.StringTable;
   --#        in     ReturnSymbol;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives SPARK_IO.FILE_SYS from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                File,
   --#                                Heap,
   --#                                LexTokenManager.StringTable,
   --#                                ReturnSymbol,
   --#                                Scope;
   is
   begin
      if ReturnSymbol /= Cells.Null_Cell then
         SPARK_IO.Set_Col (File, Indent);
         SPARK_IO.Put_String (File, "var return : ", 0);
         PrintSymbolType (File,
                          Scope,
                          Cells.Get_Symbol_Value (Heap, ReturnSymbol));
         SPARK_IO.Put_Line (File, ";", 0);
      end if;
   end PrintReturnSymbol;

   ----------------------------------------------

   -- print pending constants for Min_Int and Max_Int but only if they appear in VC
   procedure PrintRootIntegerDeclaration (File  : in SPARK_IO.File_Type)
   --# global in     RootIntegerUsed;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives SPARK_IO.FILE_SYS from *,
   --#                                File,
   --#                                RootIntegerUsed;
   is
   begin
      if RootIntegerUsed then
         SPARK_IO.Set_Col (File, Indent);
         SPARK_IO.Put_Line (File, "const system__min_int : integer = pending;", 0);
         SPARK_IO.Set_Col (File, Indent);
         SPARK_IO.Put_Line (File, "const system__max_int : integer = pending;", 0);
      end if;
   end PrintRootIntegerDeclaration;

   ----------------------------------------------

   -- print replacement rules for Min_Int/Max_Int if they have been used
   procedure PrintRootIntegerRules (RuleFile   : in SPARK_IO.File_Type;
                                    WriteRules : in Boolean)
   --# global in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     RootIntegerUsed;
   --#        in     RuleFamilyName;
   --#        in out RuleCounter;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives RuleCounter       from *,
   --#                                Dictionary.Dict,
   --#                                RootIntegerUsed,
   --#                                WriteRules &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.StringTable,
   --#                                RootIntegerUsed,
   --#                                RuleCounter,
   --#                                RuleFamilyName,
   --#                                RuleFile,
   --#                                WriteRules;
   is
      SystemSym,
      MinMaxSym  : Dictionary.Symbol;

      procedure PrintRule (Sym      : in     Dictionary.Symbol;
                           MinOrMax : in     String)
      --# global in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in     RuleFamilyName;
      --#        in     RuleFile;
      --#        in out RuleCounter;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives RuleCounter       from *,
      --#                                Dictionary.Dict,
      --#                                Sym &
      --#         SPARK_IO.FILE_SYS from *,
      --#                                Dictionary.Dict,
      --#                                LexTokenManager.StringTable,
      --#                                MinOrMax,
      --#                                RuleCounter,
      --#                                RuleFamilyName,
      --#                                RuleFile,
      --#                                Sym;
      is
         StoreVal : LexTokenManager.LexString;
      begin
         if Sym /= Dictionary.NullSymbol then
            StoreVal := Dictionary.GetValue (Sym);
            if StoreVal /= LexTokenManager.NullString then
               PrintRuleName (RuleFile);
               SPARK_IO.Put_String (RuleFile,
                                    "system__",
                                    0);
               SPARK_IO.Put_String (RuleFile,
                                    MinOrMax,
                                    0);
               SPARK_IO.Put_String (RuleFile,
                                    "_int may_be_replaced_by ",
                                    0);
               ELStrings.PutString (RuleFile,
                                              Maths.ValueToString (Maths.ValueRep (StoreVal)));
               EndARule (RuleFile);
            end if;
         end if;
      end PrintRule;

   begin -- PrintRootIntegerRules
      if RootIntegerUsed and WriteRules then
         -- get symbol for Min_Int
         SystemSym := Dictionary.LookupItem (LexTokenManager.SystemToken,
                                             Dictionary.GlobalScope,
                                             Dictionary.ProgramContext);
         if SystemSym /= Dictionary.NullSymbol then
            -- Package System has been defined in a config file
            -- or a shadow specification, so look for Min_Int in it
            MinMaxSym := Dictionary.LookupItem (LexTokenManager.Min_IntToken,
                                                Dictionary.VisibleScope (SystemSym),
                                                Dictionary.ProgramContext);
            PrintRule (MinMaxSym, "min");
            -- and then for Max_Int
            MinMaxSym := Dictionary.LookupItem (LexTokenManager.Max_IntToken,
                                                Dictionary.VisibleScope (SystemSym),
                                                Dictionary.ProgramContext);
            PrintRule (MinMaxSym, "max");

         end if;
      end if;
   end PrintRootIntegerRules;



   procedure PrintConstantsInDeclarationOrder
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     EndPosition;
   --#        in     LexTokenManager.StringTable;
   --#        in     NeededSymbols;
   --#        in     RuleFamilyName;
   --#        in     RuleFile;
   --#        in     Scope;
   --#        in     WriteRules;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Heap;
   --#        in out RuleCounter;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EndPosition,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        Heap,
   --#                                        LexTokenManager.StringTable,
   --#                                        NeededSymbols,
   --#                                        RuleCounter,
   --#                                        RuleFamilyName,
   --#                                        RuleFile,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        WriteRules &
   --#         Heap,
   --#         Statistics.TableUsage     from *,
   --#                                        Dictionary.Dict,
   --#                                        Heap,
   --#                                        NeededSymbols,
   --#                                        WriteRules &
   --#         RuleCounter               from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        Heap,
   --#                                        LexTokenManager.StringTable,
   --#                                        NeededSymbols,
   --#                                        Scope,
   --#                                        WriteRules;
   is
      LSym         : Dictionary.Symbol;
      LDeclareList : Cells.Cell;
   begin
      LDeclareList := NeededSymbols;
      loop
         exit when Pile.IsNull (LDeclareList);
         LSym := Pile.NodeSymbol (Heap, LDeclareList);
         if Dictionary.IsConstant (LSym) or else
           Dictionary.IsKnownDiscriminant (LSym) then

            PrintConstantRules (WriteRules, LSym, RuleFile, Scope, EndPosition);

            if not Cells.Is_Null_Cell (Pile.DAG (Heap, LDeclareList)) then
               PrintConstantReplacementRule
                 (WriteRules, LSym, Pile.DAG (Heap, LDeclareList), RuleFile, Scope);
            end if;

         end if;
         LDeclareList := Pile.Sibling (Heap, LDeclareList);
      end loop;
   end PrintConstantsInDeclarationOrder;


   -- Same as above, but prints rules in reverse-declaration order.
   -- Contract is _identical_ to above.  Implementation is recursive,
   -- though, so hidden from SPARK!  This implementation is provided,
   -- but unused at present.
   procedure PrintConstantsInReverseOrder
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     EndPosition;
   --#        in     LexTokenManager.StringTable;
   --#        in     NeededSymbols;
   --#        in     RuleFamilyName;
   --#        in     RuleFile;
   --#        in     Scope;
   --#        in     WriteRules;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Heap;
   --#        in out RuleCounter;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EndPosition,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        Heap,
   --#                                        LexTokenManager.StringTable,
   --#                                        NeededSymbols,
   --#                                        RuleCounter,
   --#                                        RuleFamilyName,
   --#                                        RuleFile,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        WriteRules &
   --#         Heap,
   --#         Statistics.TableUsage     from *,
   --#                                        Dictionary.Dict,
   --#                                        Heap,
   --#                                        NeededSymbols,
   --#                                        WriteRules &
   --#         RuleCounter               from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        Heap,
   --#                                        LexTokenManager.StringTable,
   --#                                        NeededSymbols,
   --#                                        Scope,
   --#                                        WriteRules;
   is
      --# hide PrintConstantsInReverseOrder;

      -- Prints List C in reverse order - recursive algorithm,
      -- so hidden from SPARK.
      procedure PrintConstantsList (C : in Cells.Cell)
      is
         LSym : Dictionary.Symbol;
      begin
         if Pile.IsNull (C) then
            null;
         else
            PrintConstantsList (Pile.Sibling (Heap, C));

            LSym := Pile.NodeSymbol (Heap, C);
            if Dictionary.IsConstant (LSym) or else
              Dictionary.IsKnownDiscriminant (LSym) then

               PrintConstantRules (WriteRules, LSym, RuleFile, Scope, EndPosition);

               if not Cells.Is_Null_Cell (Pile.DAG (Heap, C)) then
                  PrintConstantReplacementRule
                    (WriteRules, LSym, Pile.DAG (Heap, C), RuleFile, Scope);
               end if;
            end if;
         end if;
      end PrintConstantsList;

   begin
      PrintConstantsList (NeededSymbols);
   end PrintConstantsInReverseOrder;
   pragma Unreferenced (PrintConstantsInReverseOrder);

   ----------------------------------------------

   procedure RankDeclarations (MaxRank : out Cells.Cell_Rank)
   --# global in     Dictionary.Dict;
   --#        in     NeededSymbols;
   --#        in     Scope;
   --#        in out Heap;
   --# derives Heap,
   --#         MaxRank from Dictionary.Dict,
   --#                      Heap,
   --#                      NeededSymbols,
   --#                      Scope;
   is
      AllDeclarationsRanked : Boolean;
      OverallMaxRank        : Cells.Cell_Rank;

      -----------------------------------------------------------------------------
      -- RankDeclarationList
      --
      -- Ranks as many cells in the Pile rooted at NeededSymbols as it can.
      -- The Pile is not guaranteed to be in strict declaration-before-use order,
      -- so the rank of some cells may depend on the ranks of other Cells which
      -- are currently unranked.  In this case, the offending Cell is left
      -- with UnknownRank and AllDeclarationsRanked returns as False,
      -- to be completed by a susequent iteration of RankDeclarations.
      -- When all Cells are ranked successfully, AlLDeclarationsRanked returns
      -- as True, and MaxOverallRank is set to the highest rank used in the Pile.
      -----------------------------------------------------------------------------
      procedure RankDeclarationList
      --# global in     Dictionary.Dict;
      --#        in     NeededSymbols;
      --#        in     Scope;
      --#        in out Heap;
      --#        in out OverallMaxRank;
      --#           out AllDeclarationsRanked;
      --# derives AllDeclarationsRanked from Dictionary.Dict,
      --#                                    Heap,
      --#                                    NeededSymbols,
      --#                                    Scope &
      --#         Heap,
      --#         OverallMaxRank        from *,
      --#                                    Dictionary.Dict,
      --#                                    Heap,
      --#                                    NeededSymbols,
      --#                                    Scope;
      is
         DeclareList : Cells.Cell;

         --------------------------------------------------------------------------
         -- RankDeclaration
         --
         -- Tries to set the rank of a single type.  Basically, scalar types have
         -- rank 1, array types have rank 1 more than the rank of their component
         -- type, and records have rank 1 more than the maximum rank of all their
         -- fields' types.  AllDeclarationsRanked and MaxOverallRank are set
         -- as described above.
         --------------------------------------------------------------------------
         procedure RankDeclaration
         --# global in     DeclareList;
         --#        in     Dictionary.Dict;
         --#        in     NeededSymbols;
         --#        in     Scope;
         --#        in out AllDeclarationsRanked;
         --#        in out Heap;
         --#        in out OverallMaxRank;
         --# derives AllDeclarationsRanked,
         --#         Heap,
         --#         OverallMaxRank        from *,
         --#                                    DeclareList,
         --#                                    Dictionary.Dict,
         --#                                    Heap,
         --#                                    NeededSymbols,
         --#                                    Scope;
         is
            Sym             : Dictionary.Symbol;
            OwnVarSym       : Dictionary.Symbol;
            NewRank         : Cells.Cell_Rank;
            MaxFieldRank    : Cells.Cell_Rank;
            RecordIt        : Dictionary.Iterator;
            ComponentIt     : Dictionary.Iterator;
            AllFieldsRanked : Boolean;

            -- Increment rank X, but saturate at CellRank'Last
            function IncRank (X : in Cells.Cell_Rank) return Cells.Cell_Rank
            is
               R : Cells.Cell_Rank;
            begin
               if X < Cells.Cell_Rank'Last then
                  R := X + 1;
               else
                  R := Cells.Cell_Rank'Last;
               end if;
               return R;
            end IncRank;

            -- Searches the entire Pile roorted at NeededSymbols and returns
            -- the Rank of the Cell containing the given Sym.  If no match is
            -- found, then UnknownRank is returned.
            function FindRankOf (Sym : in Dictionary.Symbol) return Cells.Cell_Rank
            --# global in Dictionary.Dict;
            --#        in Heap;
            --#        in NeededSymbols;
            is
               CurrentCell : Cells.Cell;
               Result      : Cells.Cell_Rank;
            begin
               if Sym = Dictionary.GetPredefinedBooleanType or else
                  Sym = Dictionary.GetUnknownTypeMark then
                  -- Special cases - Boolean is not entered in the Pile, so we
                  -- won't find it there.  It's scalar, so has rank 1.
                  -- Secondly, Unknown type can appear in the Pile following
                  -- a semantic error in a type declararation, so we need
                  -- to return something or we will fail to terminate.
                  Result := 1;
               elsif Dictionary.TypeIsScalar (Sym) then
                  -- All scalar types have rank 1, so don't bother to
                  -- search for them...
                  Result := 1;
               elsif Dictionary.IsProtectedType (Sym) then
                  -- Protected types always appear as "pending" in FDL,
                  -- so we give them rank 1
                  Result := 1;
               else
                  CurrentCell := NeededSymbols;
                  Result := Cells.Unknown_Rank;
                  loop
                     exit when Pile.IsNull (CurrentCell);
                     if Sym = Pile.NodeSymbol (Heap, CurrentCell) then
                        Result := Cells.Get_Rank (Heap, CurrentCell);
                        exit;
                     end if;
                     CurrentCell := Pile.Sibling (Heap, CurrentCell);
                  end loop;
               end if;

               return Result;
            end FindRankOf;

         begin
            if Cells.Get_Rank (Heap, DeclareList) = Cells.Unknown_Rank then
               Sym := Pile.NodeSymbol (Heap, DeclareList);
               if Dictionary.IsType (Sym) then

                  Debug_Rank_Sym ("Trying to rank Symbol ", Sym);

                  if Dictionary.IsPrivateType (Sym, Scope) then
                     -- All scalar types and abstract proof types have rank 1, since they
                     -- don't depend on any other type.  If a type is a private extension
                     -- of a tagged type, then its rank is one more than the rank of its
                     -- "inherit" field.
                     if Dictionary.TypeIsExtendedTagged (Sym) then
                        Sym := Dictionary.GetRootOfExtendedType (Sym);
                        Debug_Rank_Sym ("Is extended tagged with root type ", Sym);

                        NewRank := FindRankOf (Sym);

                        if NewRank = Cells.Unknown_Rank then
                           Debug_Rank_Int ("Extended tagged record, but parent has unknown rank",
                                           Integer (NewRank));
                           AllDeclarationsRanked := False;
                        else
                           NewRank := IncRank (NewRank);

                           Debug_Rank_Int ("Extended tagged record, so setting rank to",
                                           Integer (NewRank));
                           Cells.Set_Rank (Heap, DeclareList, NewRank);
                           if NewRank > OverallMaxRank then
                              OverallMaxRank := NewRank;
                           end if;
                        end if;
                     else
                        Debug_Rank_Int ("Private here, so setting rank to", 1);
                        Cells.Set_Rank (Heap, DeclareList, 1);
                     end if;

                  elsif Dictionary.TypeIsIncompleteHere (Sym, Scope) then

                        -- Not complete, so appears as "pending" in FDL,
                        -- so we give them rank 1
                        Debug_Rank_Int ("Incomplete type, so setting rank to", 1);
                        Cells.Set_Rank (Heap, DeclareList, 1);

                  elsif Dictionary.TypeIsScalar (Sym) then
                     -- All scalar types and abstract proof types have rank 1, since they
                     -- don't depend on any other type.
                     Debug_Rank_Int ("Scalar, so setting rank to", 1);
                     Cells.Set_Rank (Heap, DeclareList, 1);

                  elsif Dictionary.IsProtectedType (Sym) then
                     -- Protected types always appear as "pending" in FDL,
                     -- so we give them rank 1
                     Debug_Rank_Int ("Protected, so setting rank to", 1);
                     Cells.Set_Rank (Heap, DeclareList, 1);


                  elsif Dictionary.TypeIsAbstractProof (Sym) then
                     -- Abstract proof types are either "pending" or a record type
                     -- depending on the Scope.  Pending types don't depend on anything,
                     -- and so have rank 1.
                     OwnVarSym := FindOwnVarMatchingThisType (Sym, Scope);
                     if IsLocalOwnVariableWithRefinement (OwnVarSym, Scope) then
                        ComponentIt := Dictionary.FirstConstituent (OwnVarSym);
                        MaxFieldRank := 1;
                        AllFieldsRanked := True;
                        loop
                           exit when Dictionary.IsNullIterator (ComponentIt);
                           Sym := Dictionary.GetRootType
                             (Dictionary.GetType
                                (Dictionary.CurrentSymbol (ComponentIt)));
                           NewRank := FindRankOf (Sym);

                           if NewRank = Cells.Unknown_Rank then
                              AllDeclarationsRanked := False;
                              AllFieldsRanked := False;
                              exit;
                           end if;

                           if NewRank > MaxFieldRank then
                              MaxFieldRank := NewRank;
                           end if;

                           ComponentIt := Dictionary.NextSymbol (ComponentIt);
                        end loop;

                        if AllFieldsRanked then
                           NewRank := IncRank (MaxFieldRank);

                           Debug_Rank_Int ("Refinement record, so setting rank to", Integer (NewRank));
                           Cells.Set_Rank (Heap, DeclareList, NewRank);
                           if NewRank > OverallMaxRank then
                              OverallMaxRank := NewRank;
                           end if;
                        end if;

                     else
                        Debug_Rank_Int ("Pending abstract proof, so setting rank to", 1);
                        Cells.Set_Rank (Heap, DeclareList, 1);
                     end if;

                  elsif Dictionary.TypeIsRecord (Sym) then
                     -- The rank of record type is one more than the
                     -- maximum rank of all its field types.
                     MaxFieldRank := 1;
                     AllFieldsRanked := True;
                     RecordIt := Dictionary.FirstRecordComponent (Sym);
                     loop
                        exit when Dictionary.IsNullIterator (RecordIt);
                        Sym := Dictionary.GetRootType
                          (Dictionary.GetType
                             (Dictionary.CurrentSymbol (RecordIt)));
                        NewRank := FindRankOf (Sym);

                        if NewRank = Cells.Unknown_Rank then
                           AllDeclarationsRanked := False;
                           AllFieldsRanked := False;
                           exit;
                        end if;

                        if NewRank > MaxFieldRank then
                           MaxFieldRank := NewRank;
                        end if;

                        RecordIt := Dictionary.NextSymbol (RecordIt);
                     end loop;

                     if AllFieldsRanked then
                        NewRank := IncRank (MaxFieldRank);

                        Debug_Rank_Int ("Record, so setting rank to", Integer (NewRank));
                        Cells.Set_Rank (Heap, DeclareList, NewRank);
                        if NewRank > OverallMaxRank then
                           OverallMaxRank := NewRank;
                        end if;
                     end if;

                  elsif Dictionary.TypeIsArray (Sym) then
                     -- Array index types are always discrete, so will always
                     -- have rank 1.  The rank of an array type is therefore one
                     -- more than the rank of its component type.
                     NewRank := FindRankOf
                       (Dictionary.GetRootType (Dictionary.GetArrayComponent (Sym)));

                     if NewRank = Cells.Unknown_Rank then
                        AllDeclarationsRanked := False;
                     else
                        NewRank := IncRank (NewRank);
                        Debug_Rank_Int ("Array, so setting rank to", Integer (NewRank));
                        Cells.Set_Rank (Heap, DeclareList, NewRank);
                        if NewRank > OverallMaxRank then
                           OverallMaxRank := NewRank;
                        end if;

                     end if;

                  else
                     SystemErrors.RTAssert (False,
                                            SystemErrors.AssertionFailure,
                                            "RankDeclaration - unknown type");
                  end if;
               end if;
            end if;
         end RankDeclaration;

      begin
         AllDeclarationsRanked := True;
         DeclareList := NeededSymbols;
         loop
            exit when Pile.IsNull (DeclareList);
            RankDeclaration;
            DeclareList := Pile.Sibling (Heap, DeclareList);
         end loop;
      end RankDeclarationList;

   begin -- RankDeclarations
      OverallMaxRank := 1;
      loop
         RankDeclarationList;
         exit when AllDeclarationsRanked;
      end loop;
      MaxRank := OverallMaxRank;
   end RankDeclarations;

   procedure PrintTypeDeclarations (UpToRank : in Cells.Cell_Rank)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     File;
   --#        in     LexTokenManager.StringTable;
   --#        in     NeededSymbols;
   --#        in     Scope;
   --#        in out Heap;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --# derives Heap,
   --#         Statistics.TableUsage from *,
   --#                                    Dictionary.Dict,
   --#                                    Heap,
   --#                                    NeededSymbols,
   --#                                    Scope,
   --#                                    UpToRank &
   --#         SPARK_IO.FILE_SYS     from *,
   --#                                    CommandLineData.Content,
   --#                                    Dictionary.Dict,
   --#                                    File,
   --#                                    Heap,
   --#                                    LexTokenManager.StringTable,
   --#                                    NeededSymbols,
   --#                                    Scope,
   --#                                    UpToRank;
   is
      DeclareList : Cells.Cell;
      Sym         : Dictionary.Symbol;
   begin
      for CurrentRank in Cells.Cell_Rank range 1 .. UpToRank loop
         Debug_Rank_Int ("Printing type declarations at rank", Integer (CurrentRank));
         DeclareList := NeededSymbols;
         loop
            exit when Pile.IsNull (DeclareList);
            Sym := Pile.NodeSymbol (Heap, DeclareList);

            if Dictionary.IsType (Sym) then
               if Cells.Get_Rank (Heap, DeclareList) = CurrentRank then
                  Debug_Rank_Sym (" Printing type declaration for ", Sym);
                  PrintDeclaration (File, Scope, Sym);
               end if;
            end if;
            DeclareList := Pile.Sibling (Heap, DeclareList);
         end loop;

      end loop;

   end PrintTypeDeclarations;

begin -- PrintDeclarations;
   Lists.Init (LHeap);

   PrintDeclarationHead (File, Scope);

   SPARK_IO.Put_Line (File, "  function round__(real) : integer;", 0);

   PrintRuleHeader (WriteRules, RuleFile);
   PrintStandardRules (WriteRules, RuleFile);

   RankDeclarations (MaxRank);
   PrintTypeDeclarations (UpToRank => MaxRank);

   -- Print constants and discriminants declarations
   DeclareList := NeededSymbols;
   loop
      exit when Pile.IsNull (DeclareList);
      Sym := Pile.NodeSymbol (Heap, DeclareList);
      if Dictionary.IsConstant (Sym) or else
        Dictionary.IsKnownDiscriminant (Sym) then
         PrintDeclaration (File, Scope, Sym);
      end if;
      DeclareList := Pile.Sibling (Heap, DeclareList);
   end loop;

   -- Print rules for constants
   PrintConstantsInDeclarationOrder;

   -- It may be better to print constant replacement rules in reverse
   -- declaration order, so that composites that refer to scalars come
   -- out first, and this get replaced in that order by the Simplifier.
   -- Further study needed, though, in co-operation with further work
   -- on the Simplifier.
   -- PrintConstantsInReverseOrder;

   PrintRootIntegerDeclaration (File);
   PrintRootIntegerRules (RuleFile, WriteRules);
   PrintAttributeDeclarations (File, Scope);
   PrintBitwiseOpDeclarations (File, RuleFile, WriteRules, Scope);
   --# accept Flow, 10, LHeap, "Expected ineffective assignment to LHeap" &
   --#        Flow, 10, TypeList, "Expected ineffective assignment to TypeList" &
   --#        Flow, 10, RuleCounter, "Expected ineffective assignment to RuleCounter";
   PrintTypeRules (WriteRules, RuleFile); -- Expect 3 ineffective assignments.
                                          -- Changes to LHeap, RuleCounter and TypeList
                                          -- are not significant
   --# end accept;

   DeclareList := NeededSymbols;
   loop
      exit when Pile.IsNull (DeclareList);
      Sym := Pile.NodeSymbol (Heap, DeclareList);
      if Dictionary.IsVariable (Sym) then
         PrintDeclaration (File, Scope, Sym);
      end if;
      DeclareList := Pile.Sibling (Heap, DeclareList);
   end loop;

   -- Print declaration of return variable here, after other vars and
   -- before the function declarations
   PrintReturnSymbol (File, Scope);

   DeclareList := NeededSymbols;
   loop
      exit when Pile.IsNull (DeclareList);
      Sym := Pile.NodeSymbol (Heap, DeclareList);
      if Dictionary.IsSubprogram (Sym) then
         PrintDeclaration (File, Scope, Sym);
      end if;
      DeclareList := Pile.Sibling (Heap, DeclareList);
   end loop;

   PrintExportVariableDeclarations (File, Scope);

   PrintDeclarationTail (File);
end PrintDeclarations;
