-- $Id: declarations-outputdeclarations-printdeclarations-printtyperules.adb 11642 2008-11-07 15:54:58Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with Debug;
separate (Declarations.OutputDeclarations.PrintDeclarations)
procedure PrintTypeRules (WriteRules : in Boolean;
                          RuleFile   : in SPARK_IO.File_Type)
is
   Empty : Boolean;
   Ok    : Boolean;
   Val   : Natural;
   Sym   : Dictionary.Symbol;

   type Bounds is (LowerBound, UpperBound);

   procedure PutBase (WithBase : in Boolean)
   --# global in     RuleFile;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                RuleFile,
   --#                                WithBase;
   is
   begin
      if WithBase then
         SPARK_IO.Put_String (RuleFile, "__base", 0);
      end if;
   end PutBase;

   procedure PrintLowerLessThanUpperRule (Sym      : in Dictionary.Symbol;
                                          WithBase : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     RuleFamilyName;
   --#        in     RuleFile;
   --#        in     Scope;
   --#        in out RuleCounter;
   --#        in out SPARK_IO.File_Sys;
   --# derives RuleCounter       from *,
   --#                                WithBase &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.StringTable,
   --#                                RuleCounter,
   --#                                RuleFamilyName,
   --#                                RuleFile,
   --#                                Scope,
   --#                                Sym,
   --#                                WithBase;
   is
   begin
      PrintRuleName (RuleFile);
      PrintSymbol (RuleFile, Scope, Sym);
      PutBase (WithBase);
      SPARK_IO.Put_String (RuleFile, "__first <= ", 0);
      PrintSymbol (RuleFile, Scope, Sym);
      PutBase (WithBase);
      SPARK_IO.Put_Line (RuleFile, "__last may_be_deduced.", 0);

      if WithBase then --additional rule that base type is at least as big as type
         PrintRuleName (RuleFile);
         PrintSymbol (RuleFile, Scope, Sym);
         PutBase (WithBase);
         SPARK_IO.Put_String (RuleFile, "__first <= ", 0);
         PrintSymbol (RuleFile, Scope, Sym);
         SPARK_IO.Put_Line (RuleFile, "__first may_be_deduced.", 0);

         PrintRuleName (RuleFile);
         PrintSymbol (RuleFile, Scope, Sym);
         PutBase (WithBase);
         SPARK_IO.Put_String (RuleFile, "__last >= ", 0);
         PrintSymbol (RuleFile, Scope, Sym);
         SPARK_IO.Put_Line (RuleFile, "__last may_be_deduced.", 0);
      end if;
   end PrintLowerLessThanUpperRule;

   procedure PrintABound (WhichBound : in Bounds;
                          WithBase   : in Boolean;
                          StoreVal   : in LexTokenManager.LexString)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     RuleFamilyName;
   --#        in     RuleFile;
   --#        in     Scope;
   --#        in     Sym;
   --#        in out RuleCounter;
   --#        in out SPARK_IO.File_Sys;
   --# derives RuleCounter       from *,
   --#                                StoreVal,
   --#                                WhichBound,
   --#                                WithBase &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.StringTable,
   --#                                RuleCounter,
   --#                                RuleFamilyName,
   --#                                RuleFile,
   --#                                Scope,
   --#                                StoreVal,
   --#                                Sym,
   --#                                WhichBound,
   --#                                WithBase;
   is
   begin  --PrintABound
      if StoreVal /= LexTokenManager.NullString then
         PrintRuleName (RuleFile);
         PrintSymbol (RuleFile, Scope, Sym);
         PutBase (WithBase);
         if WhichBound = LowerBound then
            SPARK_IO.Put_String (RuleFile, "__first", 0);
         else
            SPARK_IO.Put_String (RuleFile, "__last", 0);
         end if;
         PrintReplacementRule (RuleFile, StoreVal, Sym, Scope);
      elsif WhichBound = UpperBound then
         --if we do not know the value of the upper bound then put
         --out the less precise rule that Lower <= Upper
         PrintLowerLessThanUpperRule (Sym, WithBase);
      end if;
   end PrintABound;

   procedure PrintBounds (WithBase : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     RuleFamilyName;
   --#        in     RuleFile;
   --#        in     Scope;
   --#        in     Sym;
   --#        in out RuleCounter;
   --#        in out SPARK_IO.File_Sys;
   --# derives RuleCounter       from *,
   --#                                Dictionary.Dict,
   --#                                Sym,
   --#                                WithBase &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.StringTable,
   --#                                RuleCounter,
   --#                                RuleFamilyName,
   --#                                RuleFile,
   --#                                Scope,
   --#                                Sym,
   --#                                WithBase;
   is
   begin
      PrintABound (LowerBound,
                   WithBase,
                   Dictionary.GetScalarAttributeValue (WithBase,
                                                       LexTokenManager.FirstToken,
                                                       Sym));
      PrintABound (UpperBound,
                   WithBase,
                   Dictionary.GetScalarAttributeValue (WithBase,
                                                       LexTokenManager.LastToken,
                                                       Sym));
   end PrintBounds;

   -- print a replacement rule for attribute T'Modulus where T is a modular type
   procedure PrintModulus (Sym : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     RuleFamilyName;
   --#        in     RuleFile;
   --#        in     Scope;
   --#        in out RuleCounter;
   --#        in out SPARK_IO.File_Sys;
   --# derives RuleCounter       from * &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.StringTable,
   --#                                RuleCounter,
   --#                                RuleFamilyName,
   --#                                RuleFile,
   --#                                Scope,
   --#                                Sym;
   is
   begin -- PrintModulus
      PrintRuleName (RuleFile);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_String (RuleFile, "__modulus", 0);
      PrintReplacementRule (RuleFile,
                            Dictionary.GetScalarAttributeValue (False,
                                                                LexTokenManager.ModulusToken,
                                                                Sym),
                            Sym,
                            Scope);
   end PrintModulus;

   procedure PrintSizeBoundsRule (Sym : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     RuleFamilyName;
   --#        in     RuleFile;
   --#        in     Scope;
   --#        in out RuleCounter;
   --#        in out SPARK_IO.File_Sys;
   --# derives RuleCounter       from * &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.StringTable,
   --#                                RuleCounter,
   --#                                RuleFamilyName,
   --#                                RuleFile,
   --#                                Scope,
   --#                                Sym;
   is
   begin
      -- We _can_ produce a rule that T'Size >= 0 for all types T.
      -- On the other hand, the upper bound of T'Size is implementation-
      -- dependent, so we cannot produce a rule for that.
      PrintRuleName (RuleFile);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_String (RuleFile, "__size >= 0 may_be_deduced", 0);

      EndARule (RuleFile);
   end PrintSizeBoundsRule;

   procedure PrintSizeReplacementRule (Sym : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     RuleFamilyName;
   --#        in     RuleFile;
   --#        in     Scope;
   --#        in out RuleCounter;
   --#        in out SPARK_IO.File_Sys;
   --# derives RuleCounter       from * &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.StringTable,
   --#                                RuleCounter,
   --#                                RuleFamilyName,
   --#                                RuleFile,
   --#                                Scope,
   --#                                Sym;
   is
      LexValue : LexTokenManager.LexString;
   begin
      -- convert value from Maths.Value to LexTokenManager.LexString
      LexValue := Dictionary.TypeSizeAttribute (Sym);
      PrintRuleName (RuleFile);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_String (RuleFile, "__size may_be_replaced_by ", 0);

      ELStrings.PutString (RuleFile,
                                     Maths.ValueToString (Maths.ValueRep (LexValue)));
      EndARule (RuleFile);
   end PrintSizeReplacementRule;

   procedure PrintEnumerationRules (Sym : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     RuleFamilyName;
   --#        in     RuleFile;
   --#        in     Scope;
   --#        in out RuleCounter;
   --#        in out SPARK_IO.File_Sys;
   --# derives RuleCounter       from *,
   --#                                Dictionary.Dict,
   --#                                Sym &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.StringTable,
   --#                                RuleCounter,
   --#                                RuleFamilyName,
   --#                                RuleFile,
   --#                                Scope,
   --#                                Sym;
   is
      It                 : Dictionary.Iterator;
      PositionNumber     : Natural;
      LastLiteral,
      FirstLiteral       : Dictionary.Symbol;

   begin --PrintEnumerationRules
         -- t__pos(t__first) may_be_replaced_by 0 .
      PrintRuleName (RuleFile);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_String (RuleFile, "__pos(", 0);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_String (RuleFile, "__first)", 0);
      SPARK_IO.Put_String (RuleFile, " may_be_replaced_by 0", 0);
      EndARule (RuleFile);

      -- pos and val rules for literals
      PositionNumber := 0;
      It := Dictionary.FirstEnumerationLiteral (Sym);
      FirstLiteral := Dictionary.CurrentSymbol (It);
      loop
         -- t__pos("a literal") may_be_replaced_by "its position number" .
         PrintRuleName (RuleFile);
         PrintSymbol (RuleFile, Scope, Sym);
         SPARK_IO.Put_String (RuleFile, "__pos(", 0);
         PrintSymbol (RuleFile, Scope, Dictionary.CurrentSymbol (It));
         SPARK_IO.Put_String (RuleFile, ") may_be_replaced_by ", 0);
         SPARK_IO.Put_Integer (RuleFile, PositionNumber, 0, 10);
         EndARule (RuleFile);

         -- t__val("a position number") may_be_replaced_by  "its associated literal".
         PrintRuleName (RuleFile);
         PrintSymbol (RuleFile, Scope, Sym);
         SPARK_IO.Put_String (RuleFile, "__val(", 0);
         SPARK_IO.Put_Integer (RuleFile, PositionNumber, 0, 10);
         SPARK_IO.Put_String (RuleFile, ") may_be_replaced_by ", 0);
         PrintSymbol (RuleFile, Scope, Dictionary.CurrentSymbol (It));
         EndARule (RuleFile);

         -- keep copy of the last literal before exiting the loop
         LastLiteral := Dictionary.CurrentSymbol (It);

         It := Dictionary.NextSymbol (It);
         exit when Dictionary.IsNullIterator (It);

         PositionNumber := PositionNumber + 1;
      end loop;
      -- on exit, PositionNumber holds the highest position nunber of the type and
      --          LastLiteral holds the symbols of the last literal

      -- t__pos(t__last) may_be_replaced_by "the right value" .
      PrintRuleName (RuleFile);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_String (RuleFile, "__pos(", 0);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_String (RuleFile, "__last)", 0);
      SPARK_IO.Put_String (RuleFile, " may_be_replaced_by ", 0);
      SPARK_IO.Put_Integer (RuleFile, PositionNumber, 0, 10);
      EndARule (RuleFile);

      -- t__pos(succ(X)) may_be_replaced_by t__pos(X) + 1 if ...
      PrintRuleName (RuleFile);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_String (RuleFile, "__pos(succ(X)) may_be_replaced_by ", 0);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_Line (RuleFile, "__pos(X) + 1", 0);
      SPARK_IO.Put_String (RuleFile, "     if [X <=", 0);
      PrintSymbol (RuleFile, Scope, LastLiteral);
      SPARK_IO.Put_String (RuleFile, ", X <> ", 0);
      PrintSymbol (RuleFile, Scope, LastLiteral);
      SPARK_IO.Put_String (RuleFile, "]", 0);
      EndARule (RuleFile);

      -- t__pos(pred(X)) may_be_replaced_by t__pos(X) - 1 if ...
      PrintRuleName (RuleFile);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_String (RuleFile, "__pos(pred(X)) may_be_replaced_by ", 0);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_Line (RuleFile, "__pos(X) - 1", 0);
      SPARK_IO.Put_String (RuleFile, "     if [X >=", 0);
      PrintSymbol (RuleFile, Scope, FirstLiteral);
      SPARK_IO.Put_String (RuleFile, ", X <> ", 0);
      PrintSymbol (RuleFile, Scope, FirstLiteral);
      SPARK_IO.Put_String (RuleFile, "]", 0);
      EndARule (RuleFile);

      -- colour__pos(X) >= 0 may_be_deduced ...
      PrintRuleName (RuleFile);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_Line (RuleFile, "__pos(X) >= 0 may_be_deduced_from", 0);
      SPARK_IO.Put_String (RuleFile, "     [", 0);
      PrintSymbol (RuleFile, Scope, FirstLiteral);
      SPARK_IO.Put_String (RuleFile, " <= X, X <= ", 0);
      PrintSymbol (RuleFile, Scope, LastLiteral);
      SPARK_IO.Put_String (RuleFile, "]", 0);
      EndARule (RuleFile);

      -- colour__pos(X) <= ? may_be_deduced ...
      PrintRuleName (RuleFile);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_String (RuleFile, "__pos(X) <= ", 0);
      SPARK_IO.Put_Integer (RuleFile, PositionNumber, 0, 10);
      SPARK_IO.Put_Line (RuleFile, " may_be_deduced_from", 0);
      SPARK_IO.Put_String (RuleFile, "     [", 0);
      PrintSymbol (RuleFile, Scope, FirstLiteral);
      SPARK_IO.Put_String (RuleFile, " <= X, X <= ", 0);
      PrintSymbol (RuleFile, Scope, LastLiteral);
      SPARK_IO.Put_String (RuleFile, "]", 0);
      EndARule (RuleFile);

      -- colour__val(X) >= first literal may_be_deduced ...
      PrintRuleName (RuleFile);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_String (RuleFile, "__val(X) >= ", 0);
      PrintSymbol (RuleFile, Scope, FirstLiteral);
      SPARK_IO.Put_Line (RuleFile, " may_be_deduced_from", 0);
      SPARK_IO.Put_String (RuleFile, "     [0 <= X, X <= ", 0);
      SPARK_IO.Put_Integer (RuleFile, PositionNumber, 0, 10);
      SPARK_IO.Put_String (RuleFile, "]", 0);
      EndARule (RuleFile);

      -- colour__val(X) <= last literal may_be_deduced ...
      PrintRuleName (RuleFile);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_String (RuleFile, "__val(X) <= ", 0);
      PrintSymbol (RuleFile, Scope, LastLiteral);
      SPARK_IO.Put_Line (RuleFile, " may_be_deduced_from", 0);
      SPARK_IO.Put_String (RuleFile, "     [0 <= X, X <= ", 0);
      SPARK_IO.Put_Integer (RuleFile, PositionNumber, 0, 10);
      SPARK_IO.Put_String (RuleFile, "]", 0);
      EndARule (RuleFile);

      -- succ(colour__val(X)) may_be_replace_by colour__val(X+1) if ...
      PrintRuleName (RuleFile);
      SPARK_IO.Put_String (RuleFile, "succ(", 0);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_String (RuleFile, "__val(X)) may_be_replaced_by ", 0);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_Line (RuleFile, "__val(X+1)", 0);
      SPARK_IO.Put_String (RuleFile, "     if [0 <= X, X < ", 0);
      SPARK_IO.Put_Integer (RuleFile, PositionNumber, 0, 10);
      SPARK_IO.Put_String (RuleFile, "]", 0);
      EndARule (RuleFile);

      -- pred(colour__val(X)) may_be_replace_by colour__val(X-1) if ...
      PrintRuleName (RuleFile);
      SPARK_IO.Put_String (RuleFile, "pred(", 0);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_String (RuleFile, "__val(X)) may_be_replaced_by ", 0);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_Line (RuleFile, "__val(X-1)", 0);
      SPARK_IO.Put_String (RuleFile, "     if [0 < X, X <= ", 0);
      SPARK_IO.Put_Integer (RuleFile, PositionNumber, 0, 10);
      SPARK_IO.Put_String (RuleFile, "]", 0);
      EndARule (RuleFile);

      -- Pos to Val reciprocity
      PrintRuleName (RuleFile);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_String (RuleFile, "__pos(", 0);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_Line (RuleFile, "__val(X)) may_be_replaced_by X", 0);
      SPARK_IO.Put_String (RuleFile, "     if [0 <= X, X <= ", 0);
      SPARK_IO.Put_Integer (RuleFile, PositionNumber, 0, 10);
      SPARK_IO.Put_String (RuleFile, "]", 0);
      EndARule (RuleFile);

      -- Val to pos reciprocity
      PrintRuleName (RuleFile);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_String (RuleFile, "__val(", 0);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_Line (RuleFile, "__pos(X)) may_be_replaced_by X", 0);
      SPARK_IO.Put_String (RuleFile, "     if [", 0);
      PrintSymbol (RuleFile, Scope, FirstLiteral);
      SPARK_IO.Put_String (RuleFile, " <= X, X <= ", 0);
      PrintSymbol (RuleFile, Scope, LastLiteral);
      SPARK_IO.Put_String (RuleFile, "]", 0);
      EndARule (RuleFile);

      -- Ordering equivalence (suggested by Phil Thornley, BAe)
      PrintRuleName (RuleFile);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_String (RuleFile, "__pos(X) <= ", 0);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_Line (RuleFile, "__pos(Y) & X <= Y are_interchangeable ", 0);
      SPARK_IO.Put_String (RuleFile, "     if [", 0);
      PrintSymbol (RuleFile, Scope, FirstLiteral);
      SPARK_IO.Put_String (RuleFile, " <= X, X <= ", 0);
      PrintSymbol (RuleFile, Scope, LastLiteral);
      SPARK_IO.Put_String (RuleFile, ", ", 0);
      PrintSymbol (RuleFile, Scope, FirstLiteral);
      SPARK_IO.Put_String (RuleFile, " <= Y, Y <= ", 0);
      PrintSymbol (RuleFile, Scope, LastLiteral);
      SPARK_IO.Put_String (RuleFile, "]", 0);
      EndARule (RuleFile);

      PrintRuleName (RuleFile);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_String (RuleFile, "__val(X) <= ", 0);
      PrintSymbol (RuleFile, Scope, Sym);
      SPARK_IO.Put_Line (RuleFile, "__val(Y) & X <= Y are_interchangeable ", 0);
      SPARK_IO.Put_String (RuleFile, "     if [0 <= X, X <= ", 0);
      SPARK_IO.Put_Integer (RuleFile, PositionNumber, 0, 10);
      SPARK_IO.Put_String (RuleFile, ", 0 <= Y, Y <= ", 0);
      SPARK_IO.Put_Integer (RuleFile, PositionNumber, 0, 10);
      SPARK_IO.Put_String (RuleFile, "]", 0);
      EndARule (RuleFile);

   end PrintEnumerationRules;

   procedure PrintConstraintRules (Sym : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     RuleFamilyName;
   --#        in     RuleFile;
   --#        in     Scope;
   --#        in out RuleCounter;
   --#        in out SPARK_IO.File_Sys;
   --# derives RuleCounter       from *,
   --#                                Dictionary.Dict,
   --#                                Sym &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.StringTable,
   --#                                RuleCounter,
   --#                                RuleFamilyName,
   --#                                RuleFile,
   --#                                Scope,
   --#                                Sym;
   is
   begin
      -- If the implcitly-declared constraint is associated with a formal parameter
      -- of type String then we know its lower bound must be 1 (SPARK95.doc (section 3.6.3))
      --
      -- NOTE here - a special case is needed here, since String is the only
      -- array type in SPARK that allows a null-range literal "".
      -- Where "" is passed as an actual parameter, we have the anomaly that
      -- S'First = 1 and S'Last = 0, so we need special rules here.
      if Dictionary.IsPredefinedStringType
        (Dictionary.GetType
           (Dictionary.GetParameterAssociatedWithParameterConstraint (Sym))) then

         -- S'First = 1
         PrintRuleName (RuleFile);
         PrintSymbol (RuleFile, Scope, Sym);
         SPARK_IO.Put_String (RuleFile, "__first may_be_replaced_by 1", 0);
         EndARule (RuleFile);

         -- S'Last <= Positive'Last
         PrintRuleName (RuleFile);
         PrintSymbol (RuleFile, Scope, Sym);
         SPARK_IO.Put_String (RuleFile, "__last <= ", 0);
         PrintSymbol (RuleFile, Scope, Dictionary.GetType (Sym));
         SPARK_IO.Put_String (RuleFile, "__last may_be_deduced", 0);
         EndARule (RuleFile);

         -- S'Last >= 0  -- NOT S'Last >= Positive'First - see above
         PrintRuleName (RuleFile);
         PrintSymbol (RuleFile, Scope, Sym);
         SPARK_IO.Put_String (RuleFile, "__last >= 0 may_be_deduced", 0);
         EndARule (RuleFile);

      else
         -- For formal array parameters that aren't String...
         --
         -- Subprogram constraint symbols are symbols representing the indexes of unconstrained
         -- objects as they are constrained by something at some point.  We typically do not know the
         -- actual bounds but we do know that the anonymous subtype represented by the symbol must
         -- at least fit within the type of the matching index of the unconstrained type declaration.
         -- i.e. for type A is array (Integer range <>) of T; and formal parameter X of type A then
         -- we will have a subprogram constraint symbol x__index_subtype__1 and we know that
         -- x__index_subtype__1__first >= integer__first and x__index_subtype__1__last <= integer__last

         PrintRuleName (RuleFile);
         PrintSymbol (RuleFile, Scope, Sym);
         SPARK_IO.Put_String (RuleFile, "__first >= ", 0);
         PrintSymbol (RuleFile, Scope, Dictionary.GetType (Sym));
         SPARK_IO.Put_String (RuleFile, "__first may_be_deduced", 0);
         EndARule (RuleFile);

         PrintRuleName (RuleFile);
         PrintSymbol (RuleFile, Scope, Sym);
         SPARK_IO.Put_String (RuleFile, "__last <= ", 0);
         PrintSymbol (RuleFile, Scope, Dictionary.GetType (Sym));
         SPARK_IO.Put_String (RuleFile, "__last may_be_deduced", 0);
         EndARule (RuleFile);

         -- and, as free bonus
         PrintLowerLessThanUpperRule (Sym, False);

         -- and by transitivity
         PrintRuleName (RuleFile);
         PrintSymbol (RuleFile, Scope, Sym);
         SPARK_IO.Put_String (RuleFile, "__last >= ", 0);
         PrintSymbol (RuleFile, Scope, Dictionary.GetType (Sym));
         SPARK_IO.Put_String (RuleFile, "__first may_be_deduced", 0);
         EndARule (RuleFile);

         PrintRuleName (RuleFile);
         PrintSymbol (RuleFile, Scope, Sym);
         SPARK_IO.Put_String (RuleFile, "__first <= ", 0);
         PrintSymbol (RuleFile, Scope, Dictionary.GetType (Sym));
         SPARK_IO.Put_String (RuleFile, "__last may_be_deduced", 0);
         EndARule (RuleFile);

      end if;
   end PrintConstraintRules;

begin --PrintTypeRules
   if WriteRules then
      loop
         --# accept Flow, 10, Ok, "Expected ineffective assignment to Ok";
         Lists.GetFirst
            (LHeap, TypeList, Val, Empty, Ok);
         --# end accept;

         exit when Empty;
         Sym := Dictionary.ConvertSymbolRef (ExaminerConstants.RefType (Val));
         -- Debug.PrintSym ("PrintTypeRule for ", Sym);

         if Dictionary.IsTypeMark (Sym) then

            PrintSizeBoundsRule (Sym);
            if Dictionary.TypeSizeAttribute (Sym) /= LexTokenManager.NullString then
               PrintSizeReplacementRule (Sym);
            end if;

            if Dictionary.TypeIsGeneric (Sym) then
               if Dictionary.TypeIsScalar (Sym) then
                  -- just output a lower <= upper may_be_deduced rule
                  PrintLowerLessThanUpperRule (Sym      => Sym,
                                               WithBase => False); -- TBD unsure whether we can have true here PNA 15/12/05
               end if;
               -- non-scalar generic types get no rules at all
            else
               if Dictionary.TypeIsScalar (Sym) then
                  PrintBounds (WithBase => False);

                  PrintBounds (WithBase => True);
                  if Dictionary.TypeIsModular (Sym) then
                     PrintModulus (Sym);
                  end if;

                  if Dictionary.TypeIsEnumeration (Sym) and then
                    Dictionary.GetRootType (Sym) = Sym and then
                    Sym /= Dictionary.GetPredefinedCharacterType then
                     PrintEnumerationRules (Sym);
                  end if;
               end if;
            end if;
         elsif Dictionary.IsSubprogramParameterConstraint (Sym) then
            PrintConstraintRules (Sym);
         end if;
      end loop;
   end if;
   --# accept Flow, 33, Ok, "Expected Ok to be neither referenced nor exported";
end PrintTypeRules;
