-- $Id: declarations-outputdeclarations-printdeclarations-printconstantrules.adb 12850 2009-04-01 12:34:33Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with SystemErrors;
with ErrorHandler;
separate (Declarations.OutputDeclarations.PrintDeclarations)
procedure PrintConstantRules (WriteRules  : in Boolean;
                              Sym         : in Dictionary.Symbol;
                              RuleFile    : in SPARK_IO.File_Type;
                              Scope       : in Dictionary.Scopes;
                              EndPosition : in     LexTokenManager.TokenPosition)
is

   --------------------------------------------------------------------------------

   procedure PrintScalarConstantRules (RuleFile : in SPARK_IO.File_Type;
                                       Sym      : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     RuleFamilyName;
   --#        in     Scope;
   --#        in out RuleCounter;
   --#        in out SPARK_IO.File_Sys;
   --# derives RuleCounter       from *,
   --#                                Dictionary.Dict,
   --#                                Sym &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.StringTable,
   --#                                RuleCounter,
   --#                                RuleFamilyName,
   --#                                RuleFile,
   --#                                Scope,
   --#                                Sym;
   is
      StoreVal : LexTokenManager.LexString;
      T        : Dictionary.Symbol;
   begin
      StoreVal := Dictionary.GetValue (Sym);
      if StoreVal /= LexTokenManager.NullString then
         -- Sym has a literal value, so print a replacement rule.
         PrintRuleName (RuleFile);
         PrintSymbol (RuleFile, Scope, Sym);
         PrintReplacementRule (RuleFile, StoreVal, Dictionary.GetType (Sym), Scope);
      else
         -- Sym doesn't have a literal value - could be a deferred
         -- constant with a hidden completion, or a known discriminant.
         --
         -- In the former case, the VCG will produce hypotheses giving the
         -- subtype membership of the constant, so no action here.
         --
         -- In the case of a known discriminant, we genrate a subtype membership
         -- rule, unless its type is Boolean.
         if Dictionary.IsKnownDiscriminant (Sym) then
            T := Dictionary.GetType (Sym);
            if not Dictionary.TypeIsBoolean (T) then
               PrintRuleName (RuleFile);
               PrintSymbol (RuleFile, Scope, Sym);
               SPARK_IO.Put_String (RuleFile, " >= ", 0);
               PrintSymbol (RuleFile, Scope, T);
               SPARK_IO.Put_Line (RuleFile, "__first may_be_deduced.", 0);

               PrintRuleName (RuleFile);
               PrintSymbol (RuleFile, Scope, Sym);
               SPARK_IO.Put_String (RuleFile, " <= ", 0);
               PrintSymbol (RuleFile, Scope, T);
               SPARK_IO.Put_Line (RuleFile, "__last may_be_deduced.", 0);
            end if;
         end if;
      end if;
   end PrintScalarConstantRules;

   --------------------------------------------------------------------------------

   procedure PrintStructuredConstantRules (RuleFile    : in SPARK_IO.File_Type;
                                           Sym         : in Dictionary.Symbol;
                                           EndPosition : in     LexTokenManager.TokenPosition)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     RuleFamilyName;
   --#        in     Scope;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out RuleCounter;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EndPosition,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        RuleCounter,
   --#                                        RuleFamilyName,
   --#                                        RuleFile,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        Sym &
   --#         RuleCounter               from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.StringTable,
   --#                                        Scope,
   --#                                        Sym;
   is

      Node         : Dictionary.Symbol;
      Name         : ELStrings.T;
      Constraints  : ELStrings.T;
      ConstraintOK : Boolean;

      ErrorsFound  : Boolean;

      --------------------------------------------------------------------------------

      --# inherit SPARK_IO,
      --#         SystemErrors;
      package Index
      --# own State : Letter;
      --# initializes State;
      is

         subtype IndexNameRange is Positive range 1 .. 1;
         subtype IndexNameType is String (IndexNameRange);

         subtype Letter is Character range 'A' .. 'Z';

         State : Letter;

         function Value return IndexNameType;
         --# global in State;
         procedure Next;
         --# global in out State;
         --# derives State from *;

         procedure Reset (Valu : in Letter);
         --# global out State;
         --# derives State from Valu;

      end Index;

      --------------------------------------------------------------------------------

      --# inherit Dictionary,
      --#         ELStrings,
      --#         EStrings,
      --#         Index,
      --#         SPARK_IO,
      --#         SystemErrors;
      package Stack
      --# own State : StackType;
      --# initializes State;
      is

         procedure Push (Iterator     : in Dictionary.Iterator;
                         Name         : in ELStrings.T;
                         Constraints  : in ELStrings.T;
                         ConstraintOK : in Boolean;
                         CurrentIndex : in Index.Letter);
         --# global in out State;
         --# derives State from *,
         --#                    ConstraintOK,
         --#                    Constraints,
         --#                    CurrentIndex,
         --#                    Iterator,
         --#                    Name;

         procedure Pop (Iterator     : out Dictionary.Iterator;
                        Name         : out ELStrings.T;
                        Constraints  : out ELStrings.T;
                        ConstraintOK : out Boolean;
                        CurrentIndex : out Index.Letter);
         --# global in out State;
         --# derives ConstraintOK,
         --#         Constraints,
         --#         CurrentIndex,
         --#         Iterator,
         --#         Name,
         --#         State        from State;


         function IsEmpty return Boolean;
         --# global in State;
      end Stack;

      --------------------------------------------------------------------------------

      package body Index
      is

         function Value return IndexNameType
         is
         begin
            return IndexNameType'(1 => State);
         end Value;

         procedure Next
         is
         begin
            if State = Letter'Last then
               SystemErrors.FatalError (SystemErrors.TooManyNestedArrays, "in PrintConstantRules");
            end if;
            State := Letter'Succ (State);
         end Next;

         procedure Reset (Valu : in Letter)
         is
         begin
            State := Valu;
         end Reset;

      begin
         State := 'I';
      end Index;

      --------------------------------------------------------------------------------

      package body Stack
      is

         type StackPointer is range 0 .. 50;
         --# assert StackPointer'Base is Short_Short_Integer; -- for GNAT

         subtype StackIndex is StackPointer range 1 .. StackPointer'Last;
         type StackElement is record
            Iterator     : Dictionary.Iterator;
            Name         : ELStrings.T;
            Constraints  : ELStrings.T;
            ConstraintOK : Boolean;
            CurrentIndex : Index.Letter;
         end record;
         type StackContents is array (StackIndex) of StackElement;
         type StackType is record
            Ptr      : StackPointer;
            Contents : StackContents;
         end record;

         State : StackType;

         procedure Push (Iterator     : in Dictionary.Iterator;
                         Name         : in ELStrings.T;
                         Constraints  : in ELStrings.T;
                         ConstraintOK : in Boolean;
                         CurrentIndex : in Index.Letter)
         is
         begin
            if State.Ptr = StackPointer'Last then
               SystemErrors.FatalError (SystemErrors.TooManyNestedRecords, "in PrintConstantRules");
            end if;
            State.Ptr := State.Ptr + 1;
            State.Contents (State.Ptr).Iterator := Iterator;
            State.Contents (State.Ptr).Name := Name;
            State.Contents (State.Ptr).Constraints := Constraints;
            State.Contents (State.Ptr).ConstraintOK := ConstraintOK;
            State.Contents (State.Ptr).CurrentIndex := CurrentIndex;
         end Push;

         procedure Pop (Iterator     : out Dictionary.Iterator;
                        Name         : out ELStrings.T;
                        Constraints  : out ELStrings.T;
                        ConstraintOK : out Boolean;
                        CurrentIndex : out Index.Letter)
         is
         begin
            Iterator := State.Contents (State.Ptr).Iterator;
            Name := State.Contents (State.Ptr).Name;
            Constraints := State.Contents (State.Ptr).Constraints;
            ConstraintOK := State.Contents (State.Ptr).ConstraintOK;
            CurrentIndex := State.Contents (State.Ptr).CurrentIndex;
            State.Ptr := State.Ptr - 1;
         end Pop;

         function IsEmpty return Boolean
         is
         begin
            return State.Ptr = 0;
         end IsEmpty;

      begin
         State.Ptr := 0;
         --# accept Flow, 32, State.Contents, "Init is partial but effective." &
         --#        Flow, 31, State.Contents, "Init is partial but effective." &
         --#        Flow, 602, State, State.Contents, "Init is partial but effective.";
      end Stack;

      --------------------------------------------------------------------------------

      procedure PushState (Iterator     : in Dictionary.Iterator;
                           Name         : in ELStrings.T;
                           Constraints  : in ELStrings.T;
                           ConstraintOK : in Boolean)
      --# global in     Dictionary.Dict;
      --#        in     Index.State;
      --#        in out Stack.State;
      --# derives Stack.State from *,
      --#                          ConstraintOK,
      --#                          Constraints,
      --#                          Dictionary.Dict,
      --#                          Index.State,
      --#                          Iterator,
      --#                          Name;
      is
         Next : Dictionary.Iterator;
      begin
         Next := Dictionary.NextSymbol (Iterator);
         if not Dictionary.IsNullIterator (Next) then
            Stack.Push (Next, Name, Constraints, ConstraintOK, Index.State);
         end if;
      end PushState;

      --------------------------------------------------------------------------------

      procedure WalkRecord (Sym          :    out Dictionary.Symbol;
                            Components   : in     Dictionary.Iterator;
                            Name         : in out ELStrings.T;
                            Constraints  : in     ELStrings.T;
                            ConstraintOK : in     Boolean)
      --# global in     Dictionary.Dict;
      --#        in     Index.State;
      --#        in     LexTokenManager.StringTable;
      --#        in out Stack.State;
      --# derives Name        from *,
      --#                          Components,
      --#                          Dictionary.Dict,
      --#                          LexTokenManager.StringTable &
      --#         Stack.State from *,
      --#                          Components,
      --#                          ConstraintOK,
      --#                          Constraints,
      --#                          Dictionary.Dict,
      --#                          Index.State,
      --#                          Name &
      --#         Sym         from Components,
      --#                          Dictionary.Dict;
      is
         Component     : Dictionary.Symbol;
         RecordName    : ELStrings.T;
         ComponentName : EStrings.T;
      begin
         Component := Dictionary.CurrentSymbol (Components);
         LexTokenManager.LexStringToString (Dictionary.GetSimpleName (Component),
                                            ComponentName);

         PushState (Components, Name, Constraints, ConstraintOK);

         RecordName := Name;
         ELStrings.CopyString (Name, "fld_");
         ELStrings.AppendExaminerString (Name,
                                                   EStrings.LowerCase (ComponentName));
         ELStrings.AppendString (Name, "(");
         ELStrings.AppendExaminerLongString (Name, RecordName);
         ELStrings.AppendString (Name, ")");

         Sym := Dictionary.GetType (Component);

      end WalkRecord;

      --------------------------------------------------------------------------------

      procedure TreeWalk (Sym          : in out Dictionary.Symbol;
                          Name         : in out ELStrings.T;
                          Constraints  : in out ELStrings.T;
                          ConstraintOK : in out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in     Scope;
      --#        in out Index.State;
      --#        in out Stack.State;
      --# derives ConstraintOK,
      --#         Index.State,
      --#         Sym          from *,
      --#                           Dictionary.Dict,
      --#                           Scope,
      --#                           Sym &
      --#         Constraints  from *,
      --#                           CommandLineData.Content,
      --#                           Dictionary.Dict,
      --#                           Index.State,
      --#                           LexTokenManager.StringTable,
      --#                           Scope,
      --#                           Sym &
      --#         Name         from *,
      --#                           Dictionary.Dict,
      --#                           Index.State,
      --#                           LexTokenManager.StringTable,
      --#                           Scope,
      --#                           Sym &
      --#         Stack.State  from *,
      --#                           CommandLineData.Content,
      --#                           ConstraintOK,
      --#                           Constraints,
      --#                           Dictionary.Dict,
      --#                           Index.State,
      --#                           LexTokenManager.StringTable,
      --#                           Name,
      --#                           Scope,
      --#                           Sym;
      is

         procedure WalkArray (Sym          : in out Dictionary.Symbol;
                              Name         : in out ELStrings.T;
                              Constraints  : in out ELStrings.T;
                              ConstraintOK : in out Boolean)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     LexTokenManager.StringTable;
         --#        in     Scope;
         --#        in out Index.State;
         --# derives ConstraintOK,
         --#         Index.State,
         --#         Sym          from *,
         --#                           Dictionary.Dict,
         --#                           Sym &
         --#         Constraints  from *,
         --#                           CommandLineData.Content,
         --#                           Dictionary.Dict,
         --#                           Index.State,
         --#                           LexTokenManager.StringTable,
         --#                           Scope,
         --#                           Sym &
         --#         Name         from *,
         --#                           Dictionary.Dict,
         --#                           Index.State,
         --#                           Sym;
         is

            ArrayName : ELStrings.T;
            Indices   : Dictionary.Iterator;

            procedure AppendIndexConstraints
               (Indices     : in     Dictionary.Iterator;
                Constraints : in out ELStrings.T)
            --# global in     CommandLineData.Content;
            --#        in     Dictionary.Dict;
            --#        in     Index.State;
            --#        in     LexTokenManager.StringTable;
            --#        in     Scope;
            --#        in out ConstraintOK;
            --# derives ConstraintOK from *,
            --#                           Dictionary.Dict,
            --#                           Indices &
            --#         Constraints  from *,
            --#                           CommandLineData.Content,
            --#                           Dictionary.Dict,
            --#                           Index.State,
            --#                           Indices,
            --#                           LexTokenManager.StringTable,
            --#                           Scope;
            is

               IndexType  : Dictionary.Symbol;
               Constraint : ELStrings.T;
               FirstValue : LexTokenManager.LexString;
               LastValue  : LexTokenManager.LexString;

               procedure NewConstraint
                  (Constraint  : in     ELStrings.T;
                   Constraints : in out ELStrings.T)
               --# derives Constraints from *,
               --#                          Constraint;
               is
               begin
                  if not ELStrings.EqString (Constraints,
                                                       ELStrings.EmptyString) then
                     ELStrings.AppendString (Constraints, ", ");
                  end if;
                  ELStrings.AppendExaminerLongString (Constraints, Constraint);
               end NewConstraint;

            begin

               IndexType := Dictionary.CurrentSymbol (Indices);

               if not Dictionary.IsUnknownTypeMark (IndexType) then
               -- Guard to prevent constraint being generated for boolean index
                  if not Dictionary.TypeIsBoolean (IndexType) then
                     FirstValue := Dictionary.GetScalarAttributeValue
                       (False,
                        LexTokenManager.FirstToken,
                        IndexType);
                     Constraint := GetValue (FirstValue, IndexType, Scope);
                     ELStrings.AppendString (Constraint, " <= ");
                     ELStrings.AppendString (Constraint, Index.Value);
                     NewConstraint (Constraint, Constraints);

                     ELStrings.CopyString (Constraint, Index.Value);
                     ELStrings.AppendString (Constraint, " <= ");
                     LastValue := Dictionary.GetScalarAttributeValue (False,
                                                                      LexTokenManager.LastToken,
                                                                      IndexType);
                     ELStrings.AppendExaminerLongString (Constraint,
                                                                   GetValue (LastValue,
                                                                             IndexType,
                                                                             Scope));
                     NewConstraint (Constraint, Constraints);
                  end if;
               else
                  ConstraintOK := False;
               end if;
            end AppendIndexConstraints;

            --------------------------------------------------------------------------------

         begin

            ArrayName := Name;
            ELStrings.CopyString (Name, "element(");
            ELStrings.AppendExaminerLongString (Name, ArrayName);
            ELStrings.AppendString (Name, ", [");
            ELStrings.AppendString (Name, Index.Value);
            Indices := Dictionary.FirstArrayIndex (Sym);

            AppendIndexConstraints (Indices, Constraints);
            Index.Next;

            loop
               Indices := Dictionary.NextSymbol (Indices);
               exit when Dictionary.IsNullIterator (Indices);
               ELStrings.AppendString (Name, ", ");
               ELStrings.AppendString (Name, Index.Value);
               AppendIndexConstraints (Indices, Constraints);
               Index.Next;
            end loop;

            ELStrings.AppendString (Name, "])");
            Sym := Dictionary.GetArrayComponent (Sym);

         end WalkArray;

      begin
         loop
            exit when Dictionary.IsUnknownTypeMark (Sym);
            exit when Dictionary.TypeIsScalar (Sym);
            exit when Dictionary.IsPrivateType (Sym, Scope);
            exit when Dictionary.TypeIsGeneric (Sym); -- no rules for generic types
            if Dictionary.TypeIsArray (Sym) then
               WalkArray (Sym, Name, Constraints, ConstraintOK);
            elsif Dictionary.TypeIsRecord (Sym) then
               -- Sym might denote a record subtype here, so...
               Sym := Dictionary.GetRootType (Sym);
               WalkRecord (Sym,
                           Dictionary.FirstRecordComponent (Sym),
                           Name,
                           Constraints,
                           ConstraintOK);
            else
               -- Should never be reached.  We have covered scalar, private, generic,
               -- unknown, array and record above
               -- and task and protected types can't appear in expressions.
               SystemErrors.FatalError (SystemErrors.OtherInternalError,
                                        "Unexpected type symbol in PrintConstantRules.TreeWalk");
            end if;
         end loop;
      end TreeWalk;

      --------------------------------------------------------------------------------

      procedure PrintTypeBounds (RuleFile    : in SPARK_IO.File_Type;
                                 Sym         : in Dictionary.Symbol;
                                 Name        : in ELStrings.T;
                                 Constraints : in ELStrings.T)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in     RuleFamilyName;
      --#        in     Scope;
      --#        in out RuleCounter;
      --#        in out SPARK_IO.File_Sys;
      --# derives RuleCounter       from *,
      --#                                Dictionary.Dict,
      --#                                Sym &
      --#         SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                Constraints,
      --#                                Dictionary.Dict,
      --#                                LexTokenManager.StringTable,
      --#                                Name,
      --#                                RuleCounter,
      --#                                RuleFamilyName,
      --#                                RuleFile,
      --#                                Scope,
      --#                                Sym;
      is

         procedure PrintLowerBound (RuleFile    : in SPARK_IO.File_Type;
                                    Sym         : in Dictionary.Symbol;
                                    Name        : in ELStrings.T;
                                    Constraints : in ELStrings.T)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     LexTokenManager.StringTable;
         --#        in     RuleFamilyName;
         --#        in     Scope;
         --#        in out RuleCounter;
         --#        in out SPARK_IO.File_Sys;
         --# derives RuleCounter       from * &
         --#         SPARK_IO.File_Sys from *,
         --#                                CommandLineData.Content,
         --#                                Constraints,
         --#                                Dictionary.Dict,
         --#                                LexTokenManager.StringTable,
         --#                                Name,
         --#                                RuleCounter,
         --#                                RuleFamilyName,
         --#                                RuleFile,
         --#                                Scope,
         --#                                Sym;
         is
         begin
            PrintRuleName (RuleFile);
            PrintSymbol (RuleFile, Scope, Sym);
            SPARK_IO.Put_String (RuleFile, "__first <= ", 0);
            ELStrings.PutString (RuleFile, Name);
            SPARK_IO.Put_String (RuleFile, " may_be_deduced", 0);
            if not ELStrings.EqString (Constraints,
                                                 ELStrings.EmptyString) then
               SPARK_IO.Put_String (RuleFile, "_from [", 0);
               ELStrings.PutString (RuleFile, Constraints);
               SPARK_IO.Put_String (RuleFile, "]", 0);
            end if;
            EndARule (RuleFile);
         end PrintLowerBound;

         procedure PrintUpperBound (RuleFile    : in SPARK_IO.File_Type;
                                    Sym         : in Dictionary.Symbol;
                                    Name        : in ELStrings.T;
                                    Constraints : in ELStrings.T)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     LexTokenManager.StringTable;
         --#        in     RuleFamilyName;
         --#        in     Scope;
         --#        in out RuleCounter;
         --#        in out SPARK_IO.File_Sys;
         --# derives RuleCounter       from * &
         --#         SPARK_IO.File_Sys from *,
         --#                                CommandLineData.Content,
         --#                                Constraints,
         --#                                Dictionary.Dict,
         --#                                LexTokenManager.StringTable,
         --#                                Name,
         --#                                RuleCounter,
         --#                                RuleFamilyName,
         --#                                RuleFile,
         --#                                Scope,
         --#                                Sym;
         is
         begin
            PrintRuleName (RuleFile);
            ELStrings.PutString (RuleFile, Name);
            SPARK_IO.Put_String (RuleFile, " <= ", 0);
            PrintSymbol (RuleFile, Scope, Sym);
            SPARK_IO.Put_String (RuleFile, "__last may_be_deduced", 0);
            if not ELStrings.EqString (Constraints,
                                                 ELStrings.EmptyString) then
               SPARK_IO.Put_String (RuleFile, "_from [", 0);
               ELStrings.PutString (RuleFile, Constraints);
               SPARK_IO.Put_String (RuleFile, "]", 0);
            end if;
            EndARule (RuleFile);
         end PrintUpperBound;

      begin
         -- Boolean types are scalar, but do not have "<=" or ">=" operators
         -- in FDL, so there's no range constraint for them.
         if not Dictionary.TypeIsBoolean (Sym) then
            PrintLowerBound (RuleFile, Sym, Name, Constraints);
            PrintUpperBound (RuleFile, Sym, Name, Constraints);
         end if;
      end PrintTypeBounds;

      procedure Backtrack (Sym          : out Dictionary.Symbol;
                           Name         : out ELStrings.T;
                           Constraints  : out ELStrings.T;
                           ConstraintOK : out Boolean)
      --# global in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in out Stack.State;
      --#           out Index.State;
      --# derives ConstraintOK,
      --#         Constraints,
      --#         Index.State  from Stack.State &
      --#         Name         from Dictionary.Dict,
      --#                           LexTokenManager.StringTable,
      --#                           Stack.State &
      --#         Stack.State,
      --#         Sym          from Dictionary.Dict,
      --#                           Stack.State;
      is
         CurrentIndex   : Index.Letter;
         TheSym         : Dictionary.Symbol;
         Components     : Dictionary.Iterator;
         TheName        : ELStrings.T;
         TheConstraints : ELStrings.T;
         TheConstraintOK : Boolean;
      begin
         Stack.Pop (Components, TheName, TheConstraints, TheConstraintOK, CurrentIndex);
         Index.Reset (CurrentIndex);
         WalkRecord (TheSym, Components, TheName, TheConstraints, TheConstraintOK);
         Sym := TheSym;
         Name := TheName;
         Constraints := TheConstraints;
         ConstraintOK := TheConstraintOK;
      end Backtrack;

      --------------------------------------------------------------------------------

   begin
      Node := Dictionary.GetType (Sym);
      Name := ELStrings.ToExaminerLongString (GetName (Sym, Scope));

      ConstraintOK := True;
      Constraints := ELStrings.EmptyString;

      ErrorsFound := False;
      -- Note: This accept annotation should be inside the loop around the TreeWalk call
      -- but currently this is rejected by the parser. See SEPR 2067
      --# accept Flow, 10, Index.State, "Expected ineffective assignment to Index.State";
      loop
         -- Expect ineffective assignment to Index.State, as this
         -- state is discarded when we leave PrintConstantRules
         TreeWalk (Node, Name, Constraints, ConstraintOK); -- 782 - Expect 1 ineffective assignment
         if not Dictionary.TypeIsPrivate (Node) then -- no bounds available for private types
            if not Dictionary.IsUnknownTypeMark (Node) then -- nor unknown types
               if ConstraintOK then
                  PrintTypeBounds (RuleFile, Node, Name, Constraints);
               else
                  ErrorsFound := True;
               end if;
            else
               ErrorsFound := True;
            end if;
         end if;
         exit when Stack.IsEmpty;
         Backtrack (Node, Name, Constraints, ConstraintOK);
      end loop;
      --# end accept;
      if ErrorsFound then
         ErrorHandler.SemanticWarningSym (314,
                                          EndPosition,
                                          Sym,
                                          Dictionary.GetScope (Sym));
      end if;
   end PrintStructuredConstantRules;

   --------------------------------------------------------------------------------

begin
   if WriteRules then
      if not Dictionary.IsPrivateType (Dictionary.GetType (Sym), Scope) then
         if Dictionary.TypeIsScalar (Dictionary.GetType (Sym)) then
            PrintScalarConstantRules (RuleFile, Sym);
         else
            PrintStructuredConstantRules (RuleFile, Sym, EndPosition);
         end if;
      end if;
   end if;
end PrintConstantRules;
