-- $Id: debug.adb 12812 2009-03-27 15:30:32Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with Text_IO,
  SPARK_IO,
  DAG_IO,
  EStrings,
  ExaminerConstants,
  GNAT.Traceback.Symbolic,
  SPSymbols;

package body Debug
is
   --# hide Debug;

   procedure PrintMsg (Msg     : in String;
                       NewLine : in Boolean)
   is
   begin
      Text_IO.Put (Msg);
      if NewLine then
         Text_IO.New_Line (1);
      end if;
   end PrintMsg;


   procedure Print_Sym_Raw (Sym : in Dictionary.Symbol)
   is
      Str : EStrings.T;
   begin
      if Sym = Dictionary.NullSymbol then
         SPARK_IO.Put_String (SPARK_IO.Standard_Output, "Null Symbol", 0);
      else
         Dictionary.GetAnyPrefixNeeded (Sym, Dictionary.GlobalScope, ".", Str);
         EStrings.PutString (SPARK_IO.Standard_Output, Str);
         if Str.Length > 0 then
            SPARK_IO.Put_Char (SPARK_IO.Standard_Output, '.');
         end if;
         Dictionary.GenerateSimpleName (Sym, ".", Str);
         EStrings.PutString (SPARK_IO.Standard_Output, Str);
      end if;
   end Print_Sym_Raw;

   procedure PrintSym (Msg      : in String;
                       Sym      : in Dictionary.Symbol)
   is
   begin
      SPARK_IO.Put_String (SPARK_IO.Standard_Output, Msg, 0);

      -- These statements put out the raw symbol number before its name
      -- SPARK_IO.Put_Char (SPARK_IO.Standard_Output, '(');
      -- Text_IO.Put (Integer'Image (Integer (Dictionary.SymbolRef (Sym))));
      -- SPARK_IO.Put_Char (SPARK_IO.Standard_Output, ')');
      -- end of numeric output lines

      Print_Sym_Raw (Sym);

      -- These statements put out the raw symbol Discriminant after its name
      -- Start of printing " (SymbolDiscriminant)"
      -- SPARK_IO.Put_Char (SPARK_IO.Standard_Output, ' ');
      -- SPARK_IO.Put_Char (SPARK_IO.Standard_Output, '(');
      -- EStrings.PutString (SPARK_IO.Standard_Output,
      --                            Dictionary.GetSymbolDiscriminant (Sym));
      -- SPARK_IO.Put_Char (SPARK_IO.Standard_Output, ')');
      -- End of printing " (SymbolDiscriminant)"

      SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
   end PrintSym;

   procedure PrintScope (Msg   : in String;
                         Scope : in Dictionary.Scopes)
   is
   begin
      SPARK_IO.Put_String (SPARK_IO.Standard_Output, Msg, 0);
      if Dictionary.IsVisibleScope (Scope) then
         SPARK_IO.Put_String (SPARK_IO.Standard_Output, "Visible scope of ", 0);
      elsif Dictionary.IsLocalScope (Scope) then
         SPARK_IO.Put_String (SPARK_IO.Standard_Output, "Local scope of   ", 0);
      else
         SPARK_IO.Put_String (SPARK_IO.Standard_Output, "Private scope of ", 0);
      end if;
      PrintSym ("", Dictionary.GetRegion (Scope));
      Text_IO.New_Line (1);
   end PrintScope;

   procedure PrintInt (Msg : in String;
                       I   : in Integer)
   is
   begin
      Text_IO.Put (Msg);
      Text_IO.Put_Line (Integer'Image (I));
   end PrintInt;

   procedure PrintBool (Msg : in String;
                        B   : in Boolean)
   is
   begin
      Text_IO.Put (Msg);
      Text_IO.Put_Line (Boolean'Image (B));
   end PrintBool;

   procedure PrintLexStr (Msg : in String;
                          L   : in LexTokenManager.LexString)
   is
      EStr : EStrings.T;
   begin
      LexTokenManager.LexStringToString (L, EStr);
      Text_IO.Put (Msg);
      EStrings.PutLine (SPARK_IO.Standard_Output, EStr);
   end PrintLexStr;

   -- needs DAG_IO.PrintDAG to be made visible
   procedure PrintDAG (Msg     : in String;
                       DAG     : in Cells.Cell;
                       TheHeap : in out Cells.Heap_Record;
                       Scope   : in Dictionary.Scopes)
   is
   begin
      Text_IO.Put (Msg);
      DAG_IO.PrintDag (TheHeap,
                       SPARK_IO.Standard_Output,
                       DAG,
                       Scope,
                       DAG_IO.Default_Wrap_Limit);
      Text_IO.New_Line;
   end PrintDAG;

   procedure PrintSymSeq (Msg     : in String;
                          Seq     : in SeqAlgebra.Seq;
                          TheHeap : in Heap.HeapRecord)
   is
      X   : SeqAlgebra.MemberOfSeq;
      Str : EStrings.T;
      Sym : Dictionary.Symbol;
      LaterItem : Boolean := False;
   begin
      Text_IO.Put (Msg);
      Text_IO.Put ("{ ");
      X := SeqAlgebra.FirstMember (TheHeap, Seq);
      while not SeqAlgebra.IsNullMember (X) loop
         if LaterItem then
            Text_IO.Put (", ");
         end if;
         LaterItem := True;
         Sym := Dictionary.ConvertSymbolRef
           (ExaminerConstants.RefType
              (SeqAlgebra.ValueOfMember (TheHeap,
                                         X)));
         Dictionary.GetAnyPrefixNeeded (Sym, Dictionary.GlobalScope, ".", Str);
         EStrings.PutString (SPARK_IO.Standard_Output, Str);
         if Str.Length > 0 then
            SPARK_IO.Put_Char (SPARK_IO.Standard_Output, '.');
         end if;
         Dictionary.GenerateSimpleName (Sym, ".", Str);
         EStrings.PutString (SPARK_IO.Standard_Output, Str);

         X := SeqAlgebra.NextMember (TheHeap, X);
      end loop;
      Text_IO.Put_Line (" }");
   end PrintSymSeq;

   procedure PrintSeq (Msg     : in String;
                       Seq     : in SeqAlgebra.Seq;
                       TheHeap : in Heap.HeapRecord)
   is
      X   : SeqAlgebra.MemberOfSeq;
      LaterItem : Boolean := False;
   begin
      Text_IO.Put (Msg);
      Text_IO.Put ("{ ");
      X := SeqAlgebra.FirstMember (TheHeap, Seq);
      while not SeqAlgebra.IsNullMember (X) loop
         if LaterItem then
            Text_IO.Put (", ");
         end if;
         LaterItem := True;
         Text_IO.Put (Integer'Image (SeqAlgebra.ValueOfMember (TheHeap, X)));
         X := SeqAlgebra.NextMember (TheHeap, X);
      end loop;
      Text_IO.Put_Line (" }");
   end PrintSeq;


   procedure PrintNode (Msg : in String;
                        N   : in STree.SyntaxNode)
   is
   begin
      Text_IO.Put (Msg);
      Text_IO.Put_Line (SPSymbols.SPSymbol'Image (STree.SyntaxNodeType (N)));
   end PrintNode;


   procedure PrintTraceback (Msg   : in String;
                             Depth : in Natural)
   is
      Traceback : GNAT.Traceback.Tracebacks_Array (1 .. Depth);
      Unused    : Natural;
   begin
      Text_IO.Put_Line (Msg);
      GNAT.Traceback.Call_Chain (Traceback, Unused);
      SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                           GNAT.Traceback.Symbolic.Symbolic_Traceback (Traceback),
                           0);
   end PrintTraceback;


end Debug;
