-- $Id: dag-buildgraph-modelassignmentstmt.adb 12696 2009-03-12 13:14:05Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (DAG.BuildGraph)
procedure ModelAssignmentStmt
is
   AssignedVarCell,
   AssignedVarRoot,
   DAGRoot,
   StmtCell,
   ModList               : Cells.Cell;
   ExpnNode,
   VariableComponentNode : STree.SyntaxNode;
   StmtLabel             : Labels.Label;
   StreamSymbol          : Dictionary.Symbol;

   procedure CreateAssignedVarCell
   --# global in     AssignedVarRoot;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --#           out AssignedVarCell;
   --# derives AssignedVarCell       from VCGHeap &
   --#         Statistics.TableUsage from *,
   --#                                    VCGHeap &
   --#         VCGHeap               from *,
   --#                                    AssignedVarRoot;
   is
      LocalCell : Cells.Cell;
   begin
      LocalCell := AssignedVarRoot;
      loop
         exit when (Cells.Get_Kind (VCGHeap, LocalCell) /= Cells.Op)    and
            (Cells.Get_Kind (VCGHeap, LocalCell) /= Cells.Element_Function)  and
            (Cells.Get_Kind (VCGHeap, LocalCell) /= Cells.Field_Access_Function);
         if    (Cells.Get_Kind (VCGHeap, LocalCell) = Cells.Op) then
            LocalCell := LeftPtr (VCGHeap, LocalCell);
         elsif (Cells.Get_Kind (VCGHeap, LocalCell) = Cells.Element_Function)  or
            (Cells.Get_Kind (VCGHeap, LocalCell) = Cells.Field_Access_Function)
         then
            LocalCell := RightPtr (VCGHeap, LocalCell);
         end if;
      end loop;
      Cells.Create_Cell (VCGHeap, AssignedVarCell);
      Cells.Copy_Contents (VCGHeap, LocalCell, AssignedVarCell);
      Cells.Set_Kind (VCGHeap, AssignedVarCell, Cells.Modified_Op);
   end CreateAssignedVarCell;

   -------------------------------------------------------------------------------

   -- Digs down to primary node where the wf_primary will have planted a symbol
   -- if the primary references a stream variable.  The returned symbol is either
   -- the stream variable itself if it is a direct assignement or the symbol of a
   -- function if it is a function that globally references one or more streams
   function AssignedStreamSymbol
      (ExpnNode : STree.SyntaxNode) return Dictionary.Symbol
   --# global in STree.Table;
   is
      LocalNode : STree.SyntaxNode;
      Result    : Dictionary.Symbol;
   begin
      Result := Dictionary.NullSymbol; --default answer
      LocalNode := ExpnNode;
      loop
         -- to have any chance of success the chain must lead to a primary
         if STree.SyntaxNodeType (LocalNode) = SPSymbols.primary then
            Result := STree.NodeSymbol (LocalNode);
            exit;
         end if;

         -- failure cases, if these are found it can't possibly be a simple stream or stream
         -- function assignment so we need to get out of the loop
         exit when
            STree.SyntaxNodeType (LocalNode) = SPSymbols.unary_adding_operator;
         exit when
            STree.SyntaxNodeType (LocalNode) = SPSymbols.RWabs;
         exit when
            STree.SyntaxNodeType (LocalNode) = SPSymbols.RWnot;

         LocalNode := STree.Child_Node (LocalNode);
      end loop;
      return Result;
   end AssignedStreamSymbol;

   -- Complete an assignment model LHS := RHS and chain it into graph
   procedure SetStreamAssignment (LHS, RHS : in Cells.Cell)
   --# global in out Graph.Table;
   --#        in out Statistics.TableUsage;
   --#        in out StmtStack.S;
   --#        in out VCGHeap;
   --# derives Graph.Table,
   --#         StmtStack.S,
   --#         VCGHeap               from Graph.Table,
   --#                                    LHS,
   --#                                    RHS,
   --#                                    StmtStack.S,
   --#                                    VCGHeap &
   --#         Statistics.TableUsage from *,
   --#                                    LHS,
   --#                                    RHS,
   --#                                    VCGHeap;
   is
      StmtLabel             : Labels.Label;
      StmtCell, ModList     : Cells.Cell;
   begin
      PrepareLabel (VCGHeap, StmtLabel, StmtCell);
      Clists.CreateList (VCGHeap, ModList);
      Clists.AppendCell (VCGHeap, LHS, ModList);
      SetRightArgument (LHS, RHS, VCGHeap);
      SetAuxPtr (StmtCell, ModList, VCGHeap);
      Chain (StmtLabel, VCGHeap);
   end SetStreamAssignment;

   -- Construct an attribute "PrefixSymbol'tail (ExpnDAG)" and return it as RHS
   procedure BuildStreamRHS (PrefixSymbol : in     Dictionary.Symbol;
                             ExpnDAG      : in     Cells.Cell;
                             RHS          :    out Cells.Cell)
   --# global in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives RHS                   from VCGHeap &
   --#         Statistics.TableUsage from *,
   --#                                    PrefixSymbol,
   --#                                    VCGHeap &
   --#         VCGHeap               from *,
   --#                                    ExpnDAG,
   --#                                    PrefixSymbol;
   is
      StreamFunction,
      StreamPrefix,
      StreamIdent : Cells.Cell;

   begin -- BuildStreamRHS
      -- create necessary cells
      CreateOpCell (StreamFunction, VCGHeap, SPSymbols.apostrophe);
      CreateFixedVarCell (StreamPrefix, VCGHeap, PrefixSymbol);

      CreateCellKind (StreamIdent, VCGHeap, Cells.Attrib_Function);
      Cells.Set_Lex_Str (VCGHeap,
                          StreamIdent,
                          LexTokenManager.TailToken);
      --assemble into a function attribute
      SetLeftArgument (StreamFunction, StreamPrefix, VCGHeap);
      SetRightArgument (StreamFunction, StreamIdent, VCGHeap);
      SetRightArgument (StreamIdent, ExpnDAG, VCGHeap);

      RHS := StreamFunction;
   end BuildStreamRHS;

   -- Build volatility model for a direct read of a stream variable
   procedure ModelStreamVariableSideEffect
   --# global in     StreamSymbol;
   --#        in out Graph.Table;
   --#        in out Statistics.TableUsage;
   --#        in out StmtStack.S;
   --#        in out VCGHeap;
   --# derives Graph.Table,
   --#         StmtStack.S,
   --#         VCGHeap               from Graph.Table,
   --#                                    StmtStack.S,
   --#                                    StreamSymbol,
   --#                                    VCGHeap &
   --#         Statistics.TableUsage from *,
   --#                                    StreamSymbol,
   --#                                    VCGHeap;
   is
      StreamTargetVar,
      StreamFunction,
      StreamExpn            : Cells.Cell;

   begin -- ModelStreamVariableSideEffect
      CreateReferenceCell (StreamExpn, VCGHeap, StreamSymbol);

      -- now create the proof attribute function
      BuildStreamRHS (StreamSymbol,
                      StreamExpn,
                        --to get
                      StreamFunction);

      CreateModifiedCell (StreamTargetVar, VCGHeap, StreamSymbol);
      --set up assignment
      SetStreamAssignment (StreamTargetVar, StreamFunction);
   end ModelStreamVariableSideEffect;

   -- Build a volatility model for an assignment of a function that globally
   -- references one or more stream variables
   procedure ModelStreamFunctionSideEffect
   --# global in     Dictionary.Dict;
   --#        in     LScope;
   --#        in     StreamSymbol;
   --#        in out Graph.Table;
   --#        in out Statistics.TableUsage;
   --#        in out StmtStack.S;
   --#        in out VCGHeap;
   --# derives Graph.Table,
   --#         Statistics.TableUsage,
   --#         StmtStack.S,
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    Graph.Table,
   --#                                    LScope,
   --#                                    StmtStack.S,
   --#                                    StreamSymbol,
   --#                                    VCGHeap;
   is
      ImportIt            : Dictionary.Iterator;
      ImportSym           : Dictionary.Symbol;
      StreamTargetVar,
      StreamFunction,
      StreamExpn          : Cells.Cell;

   begin -- ModelStreamFunctionSideEffect
      ImportIt := Dictionary.FirstGlobalVariable (Dictionary.GetAbstraction
                                                  (StreamSymbol,
                                                   LScope),
                                                  StreamSymbol);
      while not Dictionary.IsNullIterator (ImportIt) loop
         ImportSym := Dictionary.CurrentSymbol (ImportIt);
         if Dictionary.IsOwnVariableOrConstituentWithMode (ImportSym) then
            -- a side effect model is needed
            CreateModifiedCell (StreamTargetVar, VCGHeap, ImportSym);
            CreateReferenceCell (StreamExpn, VCGHeap, ImportSym);
            BuildStreamRHS (ImportSym,
                            StreamExpn,
                              -- to get
                            StreamFunction);
            SetStreamAssignment (StreamTargetVar, StreamFunction);
         end if;
         ImportIt := Dictionary.NextSymbol (ImportIt);
      end loop;
   end ModelStreamFunctionSideEffect;

   -- construct model of form StreamVar := StreamVar'Append (StreamVar, Expn);
   procedure ModelOutputStreamVolatility (AssignedVar : in     Dictionary.Symbol;
                                          DAGRoot     : in out Cells.Cell)
   --# global in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives DAGRoot               from VCGHeap &
   --#         Statistics.TableUsage from *,
   --#                                    AssignedVar,
   --#                                    VCGHeap &
   --#         VCGHeap               from *,
   --#                                    AssignedVar,
   --#                                    DAGRoot;
   is
      TickCell,
      PrefixCell,
      IdentCell,
      CommaCell,
      LHArgCell,
      RHArgCell : Cells.Cell;

   begin -- ModelOutputStreamVolatility
      CreateOpCell (TickCell, VCGHeap, SPSymbols.apostrophe);
      CreateOpCell (CommaCell, VCGHeap, SPSymbols.comma);
      CreateFixedVarCell (PrefixCell, VCGHeap, AssignedVar);

      CreateCellKind (IdentCell, VCGHeap, Cells.Attrib_Function);
      Cells.Set_Lex_Str (VCGHeap,
                          IdentCell,
                          LexTokenManager.AppendToken);

      -- function arguments
      RHArgCell := DAGRoot;
      CreateReferenceCell (LHArgCell, VCGHeap, AssignedVar);
      --assemble into a function attribute
      SetLeftArgument (TickCell, PrefixCell, VCGHeap);
      SetRightArgument (TickCell, IdentCell, VCGHeap);
      SetRightArgument (IdentCell, CommaCell, VCGHeap);
      SetLeftArgument (CommaCell, LHArgCell, VCGHeap);
      SetRightArgument (CommaCell, RHArgCell, VCGHeap);

      -- return build up function as new expression to be assigned
      DAGRoot := TickCell;
   end ModelOutputStreamVolatility;

   ---------------------------------------

   function AssignedVarIsAnExport (TheSubprogram, TheAssignedVar : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return Dictionary.IsProcedure (TheSubprogram) and then -- only procedures have exports
        (Dictionary.IsExport (Dictionary.IsAbstract, TheSubprogram, TheAssignedVar)
           or else
           Dictionary.IsExport (Dictionary.IsRefined, TheSubprogram, TheAssignedVar));
   end AssignedVarIsAnExport;

   ---------------------------------------

begin -- ModelAssignmentStmt
   PrepareLabel (VCGHeap, StmtLabel, StmtCell);
   Clists.CreateList (VCGHeap, ModList);

   VariableComponentNode := STree.Child_Node (Node);
   ExpnNode := STree.Next_Sibling (VariableComponentNode);

   BuildExpnDAG (OutputFile,
                 VariableComponentNode,
                 LScope,
                 Scope,
                 RedType,
                 LineNmbr,
                 True,
                 False,
                 LoopStack,
                 FlowHeap,
                 VCGHeap,
                 ContainsReals,
                 VCGFailure,
                 ShortCircuitStack,
                 CheckStack,
                 KindOfStackedCheck,
                  -- to get
                 AssignedVarRoot);

   BuildExpnDAG (OutputFile,
                 ExpnNode,
                 LScope,
                 Scope,
                 RedType,
                 LineNmbr,
                 True,
                 DoAssumeLocalRvalues,
                 LoopStack,
                 FlowHeap,
                 VCGHeap,
                 ContainsReals,
                 VCGFailure,
                 ShortCircuitStack,
                 CheckStack,
                 KindOfStackedCheck,
                  -- to get
                 DAGRoot);

   CreateAssignedVarCell; -- moved from below generation of RTC to make assigned var symbol available

   -- if the assigned expression represents a stream variable of mode in then
   -- wf_assignment_statement will have put its subtype into the syntax tree.
   -- If this subtype is the same as that of the variable assigned to we do not
   -- want to generate a RTC for the assignment.  wf_assignment_statement
   -- similarly plants the type for the results of an unchecked_conversion.
   --
   -- If the assigned variable is an export of the subprogram then we _do_ generate
   -- a check even if the subtypes are the same.  This is to prevent the result of
   -- an unchecked conversion escaping to the calling environment without any checks.
   -- There is a similar situation with the exporting of Ports; however, these generate
   -- a check in CheckTypeOfExports in IncorporateConstraints.  The modifications
   -- for unchecked conversion will result in an additional VC in the case streams.

   -- If we do need a check then use original rhs DAG structure before
   -- ConvertToEntireVariable

   if RedType.RTC and then
     (STree.NodeSymbol (ExpnNode) /= STree.NodeSymbol (Node) or else
        AssignedVarIsAnExport (Dictionary.GetRegion (Scope),
                               Cells.Get_Symbol_Value (VCGHeap, AssignedVarCell)))
   then
      CheckConstraintRunTimeError (STree.NodeSymbol (Node),
                                   DAGRoot,
                                   RedType,
                                   Scope,
                                   VCGHeap,
                                   ShortCircuitStack,
                                   CheckStack,
                                   ContainsReals);
   end if;
   UnStackRtcs (LineNmbr, RedType, VCGHeap, CheckStack, KindOfStackedCheck);

   -- Following line moved above preceding insertion of RTC
   --CreateAssignedVarCell;
   Clists.AppendCell (VCGHeap, AssignedVarCell, ModList);
   ConvertToEntireVariable (AssignedVarRoot, DAGRoot);

   -- if the assigned var is an output stream then we need to model volatility
   if Dictionary.IsOwnVariableOrConstituentWithMode
      (Cells.Get_Symbol_Value (VCGHeap, AssignedVarCell)) then
      ModelOutputStreamVolatility (Cells.Get_Symbol_Value (VCGHeap, AssignedVarCell),
                                   DAGRoot);
   end if;

   -- complete model of assignment
   SetRightArgument (AssignedVarCell, DAGRoot, VCGHeap);
   SetAuxPtr (StmtCell, ModList, VCGHeap);
   Chain (StmtLabel, VCGHeap);

   -- see if an assignment of stream is involved and model side effect if it is
   StreamSymbol := AssignedStreamSymbol (ExpnNode);
   if StreamSymbol /= Dictionary.NullSymbol then
      -- we must model side effect of stream assignment
      if Dictionary.IsAdaFunction (StreamSymbol) then
         ModelStreamFunctionSideEffect;
      else
         -- since it is not null and is not a function it must be a variable
         -- so create side-effect model for a stream variable assignment
         ModelStreamVariableSideEffect;
      end if;
   end if;
end ModelAssignmentStmt;
