-- $Id: dag-buildexpndag.adb 12812 2009-03-27 15:30:32Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with
  Debug,
  ExaminerConstants,
  EStrings,
  Labels,
  Maths,
  SeqAlgebra,
  SystemErrors;

separate (DAG)
procedure BuildExpnDAG (OutputFile         : in     SPARK_IO.File_Type;
                        StartNode          : in     STree.SyntaxNode;
                        ExpnScope          : in     Dictionary.Scopes;
                        Scope              : in     Dictionary.Scopes;
                        RedType            : in     CommandLineData.RedTypes;
                        LineNmbr           : in     Integer;
                        DoRtc              : in     Boolean;
                        AssumeRvalues      : in     Boolean;
                        LoopStack          : in     LoopContext.T;
                        FlowHeap           : in out Heap.HeapRecord;
                        VCGHeap            : in out Cells.Heap_Record;
                        ContainsReals      : in out Boolean;
                        VCGFailure         : in out Boolean;
                        ShortCircuitStack  : in out CStacks.Stack;
                        CheckStack         : in out CStacks.Stack;
                        KindOfStackedCheck : in out Graph.ProofContextType;
                        DAGRoot            :    out Cells.Cell)
   -- This procedure traverses a syntax tree of an expression, which may be
   --    - an expression of an assignment statement,
   --    - a condition of an if_statement (or elsif_part),
   --    - an expression of a case_statement,
   --    - a condition of an iteration scheme.
is
   DAGCell     : Cells.Cell;
   ExpnStack   : CStacks.Stack;
   NodeType    : SPSymbols.SPSymbol;
   LastNode,
   Node        : STree.SyntaxNode;

   ReferencedVars : SeqAlgebra.Seq; -- Set of rvalues of expression

   -- Populate set of r-values.  This procedure is called from ProcessIdentifier and
   -- ProcessSelectedComponent whenever a variable is found.  A set of referenced
   -- entire variables (R-values) is constructed by this means.
   procedure AddRvalueSymbol (TheHeap  : in out Heap.HeapRecord;
                              Sequ     : in     SeqAlgebra.Seq;
                              Sym      : in     Dictionary.Symbol)
   --# global in     AssumeRvalues;
   --#        in     Dictionary.Dict;
   --#        in     RedType;
   --#        in     Scope;
   --#        in out Statistics.TableUsage;
   --# derives Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    AssumeRvalues,
   --#                                    Dictionary.Dict,
   --#                                    RedType,
   --#                                    Scope,
   --#                                    Sequ,
   --#                                    Sym,
   --#                                    TheHeap;
   is
      function IsLocalVariable return Boolean
      --# global in Dictionary.Dict;
      --#        in Scope;
      --#        in Sym;
      is
      begin
         -- A variable is "local" if its scope if that of the current
         -- subprogram, and it's not a formal parameter.
         return Dictionary.IsVariable (Sym) and then
           Dictionary.GetScope (Sym) = Scope and then
           not Dictionary.IsSubprogramParameter (Sym);
      end IsLocalVariable;

      function IsOutModeFormalParameter return Boolean
      --# global in Dictionary.Dict;
      --#        in Scope;
      --#        in Sym;
      is
      begin
         return Dictionary.GetScope (Sym) = Scope and then
           Dictionary.IsSubprogramParameter (Sym) and then
           Dictionary.GetSubprogramParameterMode (Sym) = Dictionary.OutMode;
      end IsOutModeFormalParameter;

      function IsDeferredNonPrivateConstant return Boolean
      --# global in Dictionary.Dict;
      --#        in RedType;
      --#        in Scope;
      --#        in Sym;
      is
      begin
         -- If a non-private, scalar constant is referenced and we don't know its value
         -- then no rules will be generated.  In that specific case it is worth asserting
         -- that the value is in type because that is the best we can do unless the type is universal.
         return Dictionary.IsConstant (Sym) and then
           DiscreteTypeWithCheck (Dictionary.GetType (Sym), Scope, RedType) and then
           Dictionary.GetValue (Sym) = LexTokenManager.NullString and then -- i.e. no value known
           not (Dictionary.IsUniversalIntegerType (Dictionary.GetType (Sym)) or
                  Dictionary.IsUniversalRealType (Dictionary.GetType (Sym)));
      end IsDeferredNonPrivateConstant;

   begin -- AddRvalueSymbol
      if AssumeRvalues then
         -- Only add local variables or (in SPARK95) an "out" mode formal parameter or deferred constant
         if IsLocalVariable or IsOutModeFormalParameter or IsDeferredNonPrivateConstant then
            SeqAlgebra.AddMember (TheHeap,
                                  Sequ,
                                  Natural (Dictionary.SymbolRef (Sym)));
         end if;
      end if;
   end AddRvalueSymbol;

   ------------------------------------------------------------------------

   procedure ConjoinParamConstraint (Type_Sym :        Dictionary.Symbol;
                                     var_sym  :        Dictionary.Symbol;
                                     DAGCell  : in out Cells.Cell)
   --# global in     OutputFile;
   --#        in     RedType;
   --#        in     Scope;
   --#        in out ContainsReals;
   --#        in out Dictionary.Dict;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out VCGFailure;
   --#        in out VCGHeap;
   --# derives ContainsReals,
   --#         Dictionary.Dict,
   --#         LexTokenManager.StringTable,
   --#         VCGFailure                  from *,
   --#                                          Dictionary.Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          Type_Sym,
   --#                                          var_sym,
   --#                                          VCGHeap &
   --#         DAGCell,
   --#         Statistics.TableUsage,
   --#         VCGHeap                     from *,
   --#                                          DAGCell,
   --#                                          Dictionary.Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          Type_Sym,
   --#                                          var_sym,
   --#                                          VCGHeap &
   --#         SPARK_IO.FILE_SYS           from *,
   --#                                          Dictionary.Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          OutputFile,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          Type_Sym,
   --#                                          var_sym,
   --#                                          VCGHeap;
   is
      Constr,
      VarCell : Cells.Cell;
   begin
      -- ConjoinParamConstraint is only called if RedType.RTC
      CreateReferenceCell (VarCell, VCGHeap, var_sym);
      CreateStructConstraint (RedType,
                              OutputFile,
                              Type_Sym,
                              VarCell,
                              Scope,
                              Dictionary.NullSymbol,
                              VCGHeap,
                              ContainsReals,
                              VCGFailure,
                              Constr);

      if not Cells.Is_Null_Cell (Constr) then
         Conjoin (Constr, VCGHeap, DAGCell);
      end if;
   end ConjoinParamConstraint;

   -- Generate hypotheses that all variables in the
   -- ReferencedVars set are in their type
   procedure PlantRvalueAssumptions (ReferencedVars : in SeqAlgebra.Seq)
   --# global in     OutputFile;
   --#        in     RedType;
   --#        in     Scope;
   --#        in out ContainsReals;
   --#        in out Dictionary.Dict;
   --#        in out FlowHeap;
   --#        in out Graph.Table;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out StmtStack.S;
   --#        in out VCGFailure;
   --#        in out VCGHeap;
   --# derives ContainsReals,
   --#         Dictionary.Dict,
   --#         LexTokenManager.StringTable,
   --#         Statistics.TableUsage,
   --#         VCGFailure                  from *,
   --#                                          Dictionary.Dict,
   --#                                          FlowHeap,
   --#                                          LexTokenManager.StringTable,
   --#                                          RedType,
   --#                                          ReferencedVars,
   --#                                          Scope,
   --#                                          VCGHeap &
   --#         FlowHeap                    from *,
   --#                                          ReferencedVars &
   --#         Graph.Table,
   --#         StmtStack.S,
   --#         VCGHeap                     from Dictionary.Dict,
   --#                                          FlowHeap,
   --#                                          Graph.Table,
   --#                                          LexTokenManager.StringTable,
   --#                                          RedType,
   --#                                          ReferencedVars,
   --#                                          Scope,
   --#                                          StmtStack.S,
   --#                                          VCGHeap &
   --#         SPARK_IO.FILE_SYS           from *,
   --#                                          Dictionary.Dict,
   --#                                          FlowHeap,
   --#                                          LexTokenManager.StringTable,
   --#                                          OutputFile,
   --#                                          RedType,
   --#                                          ReferencedVars,
   --#                                          Scope,
   --#                                          VCGHeap;
   is
      DAGCell : Cells.Cell := Cells.Null_Cell;
      VarSym  : Dictionary.Symbol;
      TypeSym  : Dictionary.Symbol;
      X   : SeqAlgebra.MemberOfSeq;
      StmtLabel : Labels.Label;
      StmtCell  : Cells.Cell;
   begin
      -- ReferencedVars is a set of R-value leaves populated by BuildExpnDAG
      X := SeqAlgebra.FirstMember (FlowHeap, ReferencedVars);
      while not SeqAlgebra.IsNullMember (X) loop
         VarSym := Dictionary.ConvertSymbolRef
           (ExaminerConstants.RefType
              (SeqAlgebra.ValueOfMember (FlowHeap,
                                         X)));
         TypeSym := Dictionary.GetType (VarSym);
         ConjoinParamConstraint (TypeSym, VarSym, DAGCell);

         -- remove each element after it is used to recover heap space and prevent
         -- unnecessary repetition of hypothese
         SeqAlgebra.RemoveMember (FlowHeap, ReferencedVars, SeqAlgebra.ValueOfMember (FlowHeap, X));
         X := SeqAlgebra.FirstMember (FlowHeap, ReferencedVars);
         -- original method replaced two lines baove with one below, doen't recover heap space
         -- X := SeqAlgebra.NextMember (FlowHeap, X);
      end loop;

      -- DAGCell is now a complete list of constraints that we need to plant as a set of hypotheses
      PrepareLabel (VCGHeap, StmtLabel, StmtCell);
      SetRightArgument (StmtCell, DAGCell, VCGHeap);
      Chain (StmtLabel, VCGHeap);
   end PlantRvalueAssumptions;

   ------------------------------------------------------------------------

   -- Conditionally generate hypotheses that all variables in the
   -- ReferencedVars set are in their type
   procedure CheckPlantRvalueAssumptions
   --# global in     AssumeRvalues;
   --#        in     OutputFile;
   --#        in     RedType;
   --#        in     ReferencedVars;
   --#        in     Scope;
   --#        in out ContainsReals;
   --#        in out Dictionary.Dict;
   --#        in out FlowHeap;
   --#        in out Graph.Table;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out StmtStack.S;
   --#        in out VCGFailure;
   --#        in out VCGHeap;
   --# derives ContainsReals,
   --#         Dictionary.Dict,
   --#         LexTokenManager.StringTable,
   --#         Statistics.TableUsage,
   --#         VCGFailure                  from *,
   --#                                          AssumeRvalues,
   --#                                          Dictionary.Dict,
   --#                                          FlowHeap,
   --#                                          LexTokenManager.StringTable,
   --#                                          RedType,
   --#                                          ReferencedVars,
   --#                                          Scope,
   --#                                          VCGHeap &
   --#         FlowHeap                    from *,
   --#                                          AssumeRvalues,
   --#                                          ReferencedVars &
   --#         Graph.Table,
   --#         StmtStack.S,
   --#         VCGHeap                     from AssumeRvalues,
   --#                                          Dictionary.Dict,
   --#                                          FlowHeap,
   --#                                          Graph.Table,
   --#                                          LexTokenManager.StringTable,
   --#                                          RedType,
   --#                                          ReferencedVars,
   --#                                          Scope,
   --#                                          StmtStack.S,
   --#                                          VCGHeap &
   --#         SPARK_IO.FILE_SYS           from *,
   --#                                          AssumeRvalues,
   --#                                          Dictionary.Dict,
   --#                                          FlowHeap,
   --#                                          LexTokenManager.StringTable,
   --#                                          OutputFile,
   --#                                          RedType,
   --#                                          ReferencedVars,
   --#                                          Scope,
   --#                                          VCGHeap;
   is
   begin
      if AssumeRvalues then
         PlantRvalueAssumptions (ReferencedVars);
      end if;
   end CheckPlantRvalueAssumptions;

   -------------------------------------------------------------------

   procedure AssumeFnReturnType (FunctionSym  : in Dictionary.Symbol;
                                 FunctionCall : in Cells.Cell)
   --# global in     OutputFile;
   --#        in     RedType;
   --#        in     Scope;
   --#        in out ContainsReals;
   --#        in out Dictionary.Dict;
   --#        in out Graph.Table;
   --#        in out LexTokenManager.StringTable;
   --#        in out ShortCircuitStack;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out StmtStack.S;
   --#        in out VCGFailure;
   --#        in out VCGHeap;
   --# derives ContainsReals,
   --#         Dictionary.Dict,
   --#         LexTokenManager.StringTable,
   --#         VCGFailure                  from *,
   --#                                          Dictionary.Dict,
   --#                                          FunctionCall,
   --#                                          FunctionSym,
   --#                                          LexTokenManager.StringTable,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          VCGHeap &
   --#         Graph.Table,
   --#         StmtStack.S,
   --#         VCGHeap                     from Dictionary.Dict,
   --#                                          FunctionCall,
   --#                                          FunctionSym,
   --#                                          Graph.Table,
   --#                                          LexTokenManager.StringTable,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          ShortCircuitStack,
   --#                                          StmtStack.S,
   --#                                          VCGHeap &
   --#         ShortCircuitStack,
   --#         Statistics.TableUsage       from *,
   --#                                          Dictionary.Dict,
   --#                                          FunctionCall,
   --#                                          FunctionSym,
   --#                                          LexTokenManager.StringTable,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          ShortCircuitStack,
   --#                                          VCGHeap &
   --#         SPARK_IO.FILE_SYS           from *,
   --#                                          Dictionary.Dict,
   --#                                          FunctionCall,
   --#                                          FunctionSym,
   --#                                          LexTokenManager.StringTable,
   --#                                          OutputFile,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          VCGHeap;
   is
      StmtCell,
      RetCell : Cells.Cell;
      StmtLabel : Labels.Label;
   begin
      -- AssumeFnReturnType will only be called if RedType.VCs and DoRtc

      -- Don't assume return type is valid if it is an unchecked conversion
      if not Dictionary.IsAnUncheckedConversion (Dictionary.GetAdaFunction (FunctionSym)) then

         CreateStructConstraint (RedType,
                                 OutputFile,
                                 Dictionary.GetType (FunctionSym),
                                 FunctionCall,
                                 Scope,
                                 Dictionary.NullSymbol,
                                 VCGHeap,
                                 ContainsReals,
                                 VCGFailure,
                                 RetCell);

         if not Cells.Is_Null_Cell (RetCell) then
            AddAnySCImplications (RedType, VCGHeap, RetCell, ShortCircuitStack);
            PrepareLabel (VCGHeap, StmtLabel, StmtCell);
            SetRightArgument (StmtCell, RetCell, VCGHeap);
            Chain (StmtLabel, VCGHeap);
         end if;
      end if;
   end AssumeFnReturnType;

   ----------------------------------------------------------------------------------------------
   -- This is a direct copy of same routine in BuildGraph.  I had to copy it here for cisibility
   -- reasons.  A shared copy can't go in DAG.adb because VCGHeap is not visible there

   procedure InstantiateParameters (ConstraintRoot      : in Cells.Cell;
                                    InstantiatedSubProg : in Dictionary.Symbol)

   -- replace symbols in DAG which belong to a generic unit with the equivalent
   -- associated with the instantiated unit.  Substitutes generic formals/actuals
   -- and also parameters
   --# global in     Dictionary.Dict;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    ConstraintRoot,
   --#                                    Dictionary.Dict,
   --#                                    InstantiatedSubProg,
   --#                                    VCGHeap;
   is
      P              : Cells.Cell;
      S              : CStacks.Stack;
      SymToCheck     : Dictionary.Symbol;
      TheGeneric     : Dictionary.Symbol;

      function IsLeaf (Node : Cells.Cell) return Boolean
      --# global in VCGHeap;
      is
      begin
         return Cells.Is_Null_Cell (RightPtr (VCGHeap, Node));
      end IsLeaf;

   begin
      -- Debug.PrintMsg ("In InstantiateParameters", True);
      TheGeneric := Dictionary.GetGenericOfInstantiation (InstantiatedSubProg);

      -- DAG traversal algorithm of D.E. Knuth, Fundamental
      -- Algorithms, p.317;
      CStacks.CreateStack (S);
      P := ConstraintRoot;
      loop
         loop
            exit when Cells.Is_Null_Cell (P);
            CStacks.Push (VCGHeap, P, S);
            if IsLeaf (P) then
               P := Cells.Null_Cell;
            else
               P := LeftPtr (VCGHeap, P);
            end if;
         end loop;
         exit when CStacks.IsEmpty (S);
         P := CStacks.Top (VCGHeap, S);
         CStacks.Pop (VCGHeap, S);
         if IsLeaf (P) then
            SymToCheck := Cells.Get_Symbol_Value (VCGHeap, P);
            -- Debug.PrintSym ("Checking symbol ", SymToCheck);
            if Dictionary.IsFormalParameter (TheGeneric, SymToCheck) then
               Cells.Set_Symbol_Value (VCGHeap,
                                     P,
                                     Dictionary.ActualParameterOfGenericParameter (SymToCheck,
                                                                                   InstantiatedSubProg));
            elsif Dictionary.IsGenericFormalParameter (TheGeneric, SymToCheck) then
               Cells.Set_Symbol_Value (VCGHeap,
                                     P,
                                     Dictionary.ActualOfGenericFormal (SymToCheck,
                                                                       InstantiatedSubProg));
            elsif Dictionary.IsType (SymToCheck) and then Dictionary.TypeIsGeneric (SymToCheck) then
               Cells.Set_Symbol_Value (VCGHeap,
                                     P,
                                     Dictionary.ActualOfGenericFormal (SymToCheck,
                                                                       InstantiatedSubProg));
            end if;
            P := Cells.Null_Cell;
         else
            P := RightPtr (VCGHeap, P);
         end if;
      end loop;
   end InstantiateParameters;

   ---------------------------------------------------------------------

   procedure ModelFnConstraints (Abstraction : Dictionary.Abstractions)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     ExpnStack;
   --#        in     LoopStack;
   --#        in     RedType;
   --#        in     STree.Table;
   --#        in out CheckStack;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out KindOfStackedCheck;
   --#        in out LexTokenManager.StringTable;
   --#        in out ShortCircuitStack;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives CheckStack,
   --#         ShortCircuitStack,
   --#         Statistics.TableUsage       from *,
   --#                                          Abstraction,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ExpnStack,
   --#                                          LexTokenManager.StringTable,
   --#                                          LoopStack,
   --#                                          RedType,
   --#                                          ShortCircuitStack,
   --#                                          STree.Table,
   --#                                          VCGHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS           from Abstraction,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          ExpnStack,
   --#                                          LexTokenManager.StringTable,
   --#                                          LoopStack,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          STree.Table,
   --#                                          VCGHeap &
   --#         KindOfStackedCheck          from *,
   --#                                          Abstraction,
   --#                                          Dictionary.Dict,
   --#                                          ExpnStack,
   --#                                          VCGHeap &
   --#         LexTokenManager.StringTable from *,
   --#                                          Abstraction,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ExpnStack,
   --#                                          LoopStack,
   --#                                          STree.Table,
   --#                                          VCGHeap &
   --#         VCGHeap                     from *,
   --#                                          Abstraction,
   --#                                          CheckStack,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ExpnStack,
   --#                                          LexTokenManager.StringTable,
   --#                                          LoopStack,
   --#                                          RedType,
   --#                                          ShortCircuitStack,
   --#                                          STree.Table;
   is
      FunctionSym : Dictionary.Symbol;
      PreConNode  : STree.SyntaxNode;
      DAGCell     : Cells.Cell;
      GenericFunctionSym : Dictionary.Symbol;

      -------------------------

      function GetActual (ArgNo : in Positive) return Cells.Cell
      --# global in ExpnStack;
      --#        in VCGHeap;
      is
         -- Walks the data structure produced by SetUpFunctionCall to find the actual DAG
         -- associated with a particular function parameter number
         DAGCell : Cells.Cell;

      begin
         DAGCell := CStacks.Top (VCGHeap, ExpnStack);
         for i in Positive range 1 .. ArgNo loop
            DAGCell := RightPtr (VCGHeap, DAGCell);
         end loop;

         if Cells.Get_Kind (VCGHeap, DAGCell) = Cells.Op and then
            Cells.Get_Op_Symbol (VCGHeap, DAGCell) = SPSymbols.comma
         then
            DAGCell := LeftPtr (VCGHeap, DAGCell);
         end if;

         return DAGCell;
      end GetActual;

      --------------------------------

--        function GetActualConstraint (ArgNo : in Positive) return Cells.Cell
--        --# global VCGHeap,
--        --#        ExpnStack;
--        is
--           -- Similar to the above but returns the constraining index cell if present
--           DAGCell,
--           Result   : Cells.Cell;
--           ConstraintSym : Dictionary.Symbol; -- PNA

--        begin
--           DAGCell := GetActual (ArgNo);
--           Result := AuxPtr (VCGHeap, DAGCell);
--           if Cells.Is_Null_Cell (Result) then
--              Result := DAGCell; -- no constraint present

--           else
--              -- PNA experimental change.  Constraint is present.  Result contain either an array subtype symbol
--              -- PNA or the symbol of a subtype of positive if the actual parameter is a string literal.  We check
--              -- PNA for the array case and if we find it we repalce the array subtype with the symbol of an Index type.
--              -- PNA To maintain the Examiner's current broken behaviour we assume the first index.
--              ConstraintSym := Cells.Get_Symbol_Value (VCGHeap, Result);
--              if Dictionary.TypeIsArray (ConstraintSym) then
--                 -- PNA, note, temporary intro of function side effect here
--                 Cells.Set_Symbol_Value (VCGHeap,
--                                       Result,
--                                       Dictionary.GetArrayIndex (ConstraintSym, 1));
--              end if;
--              -- PNA end
--           end if;
--           return Result;
--        end GetActualConstraint;

      ---------------------------------------------------------

      procedure GetActualConstraint (FunctionSym : in     Dictionary.Symbol;
                                     Sym         : in     Dictionary.Symbol;
                                     Change      :    out Boolean;
                                     Result      :    out Cells.Cell)
      --# global in     Dictionary.Dict;
      --#        in     ExpnStack;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives Change                from Dictionary.Dict,
      --#                                    FunctionSym,
      --#                                    Sym &
      --#         Result                from Dictionary.Dict,
      --#                                    FunctionSym,
      --#                                    Sym,
      --#                                    VCGHeap &
      --#         Statistics.TableUsage from *,
      --#                                    Dictionary.Dict,
      --#                                    FunctionSym,
      --#                                    Sym,
      --#                                    VCGHeap &
      --#         VCGHeap               from *,
      --#                                    Dictionary.Dict,
      --#                                    ExpnStack,
      --#                                    FunctionSym,
      --#                                    Sym;
      is
         -- Similar to GetActual above but returns the constraining index type associated with
         -- a constrained actual parameter associated with an unconstrained formal parameter

         ArgNo          : Positive;
         LResult        : Cells.Cell;
         ActualCell     : Cells.Cell;
         ConstraintCell : Cells.Cell;
         ConstraintSym  : Dictionary.Symbol;

         ObjectSym : Dictionary.Symbol;
         ArrayDimension : Positive;
      begin
         -- Debug.PrintMsg ("in GetActualConstraint", True);
         -- Debug.PrintSym ("Sym passed in is ", Sym);
         -- The Sym passed to this routine will be a Dictionary.ParameterConstraintSymbol.
         -- From this we can obtain the object itself and the dimension of that object that appears
         -- in the expression we may be making substitutions to.


         ObjectSym := Dictionary.GetParameterAssociatedWithParameterConstraint (Sym);
         -- Debug.PrintSym ("Object sym is ", ObjectSym);

         if Dictionary.IsFormalParameter (FunctionSym, ObjectSym) then
            -- There may be something to do.  Only in the case of formal/actual
            -- parameter matching can constraints be introduced and constraint
            -- substitution requires.  If ObjectSym is global to FunctionSym (as
            -- it may be with nested subprogram calls) then the constraint will
            -- left unchanged
            -- Debug.PrintMsg ("Handling a formal parameter", True);
            Change := True;
            ArrayDimension := Dictionary.GetSubprogramParameterConstraintDimension (Sym);
            -- Debug.PrintInt ("ArrayDimension is ", ArrayDimension);

            ArgNo := Dictionary.GetSubprogramParameterNumber (ObjectSym);
            ActualCell := GetActual (ArgNo);
            ConstraintCell := AuxPtr (VCGHeap, ActualCell);

            Cells.Create_Cell (VCGHeap, LResult);
            if Cells.Is_Null_Cell (ConstraintCell) then
               Cells.Copy_Contents (VCGHeap,
                                   ActualCell, -- no constraint present
                                   LResult);
            else
               Cells.Copy_Contents (VCGHeap,
                                   ConstraintCell,
                                   LResult);
            end if;

            -- LResult contains either:
            -- (1) an array subtype symbol in the case where the actual paramater is of a constrained
            --     array subtype
            -- (2) a scalar index type symbol in the case of a string literal being passed to string
            -- (3) a symbol of a subprogram parameter in the case where the actual parameter is also
            --     an unconstrained array and no constraint has been planted (this final behaviour occurs
            --     because GetConstraintCell returns the actual parameter DAG if no constraint is present)
            ConstraintSym := Cells.Get_Symbol_Value (VCGHeap, LResult);
            -- Debug.PrintSym ("Constraint sym obtained from syntax tree ", ConstraintSym);
            if Dictionary.IsSubprogramParameter (ConstraintSym) then
               -- Case 3.  We substitute "actual__index__subtype__n" for "formal__index__subtype__n"
               -- Debug.PrintMsg ("Case 3", True);
               Cells.Set_Symbol_Value (VCGHeap,
                                     LResult,
                                     Dictionary.GetSubprogramParameterConstraint (ConstraintSym, ArrayDimension));

            elsif Dictionary.TypeIsArray (ConstraintSym) then
               -- Case 2. We substitute array index n of constraining subtype for "formal__index__subtype__n"
               -- Debug.PrintMsg ("Case 2", True);
               Cells.Set_Symbol_Value (VCGHeap,
                                     LResult,
                                     Dictionary.GetArrayIndex (ConstraintSym, ArrayDimension));
            else
               -- Case 1. we already have the constraining index directly
               -- Debug.PrintMsg ("Case 1", True);
               null;
            end if;

         else
            -- not a formal parameter so leave constraint unchanged
            LResult := Cells.Null_Cell;
            Change := False;
         end if;
         -- Debug.PrintSym ("Substituted symbol is ", Cells.Get_Symbol_Value (VCGHeap, LResult));
         -- Debug.PrintBool ("Change is ", Change);
         Result := LResult;
      end GetActualConstraint;

   ---------------------------------------------------------------------

      procedure SubstituteParameters (ConstraintRoot : in Cells.Cell)
      --# global in     Dictionary.Dict;
      --#        in     ExpnStack;
      --#        in     FunctionSym;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives Statistics.TableUsage,
      --#         VCGHeap               from *,
      --#                                    ConstraintRoot,
      --#                                    Dictionary.Dict,
      --#                                    ExpnStack,
      --#                                    FunctionSym,
      --#                                    VCGHeap;
         -- replace formal parameters by actual ones in a subprogram constraint;
      is
         P, CopyCell    : Cells.Cell;
         S              : CStacks.Stack;
         VarSym         : Dictionary.Symbol;
         Change         : Boolean;

         function IsLeaf (Node : Cells.Cell) return Boolean
         --# global in VCGHeap;
         is
         begin
            return Cells.Is_Null_Cell (RightPtr (VCGHeap, Node));
         end IsLeaf;

      begin -- SubstituteParameters
         -- DAG traversal algorithm of D.E. Knuth, Fundamental Algorithms, p.317;
         -- Debug.PrintMsg ("In SubstituteParameters", True);
         CStacks.CreateStack (S);
         P := ConstraintRoot;
         loop
            loop
               exit when Cells.Is_Null_Cell (P);
               CStacks.Push (VCGHeap, P, S);
               if IsLeaf (P) then
                  P := Cells.Null_Cell;
               else
                  P := LeftPtr (VCGHeap, P);
               end if;
            end loop;
            exit when CStacks.IsEmpty (S);
            P := CStacks.Top (VCGHeap, S);
            CStacks.Pop (VCGHeap, S);
            if IsLeaf (P) then
               VarSym := Cells.Get_Symbol_Value (VCGHeap, P);
               if Cells.Get_Kind (VCGHeap, P) = Cells.Reference then
                  -- Debug.PrintMsg ("   handling RefCell", True);
                  if Dictionary.IsFormalParameter (FunctionSym, VarSym) then
                     Structures.CopyStructure (VCGHeap,
                                               GetActual (Dictionary.GetSubprogramParameterNumber (VarSym)),
                                               CopyCell);
                     Cells.Copy_Contents (VCGHeap,
                                         CopyCell,
                                         P);
                  end if;
               elsif Cells.Get_Kind (VCGHeap, P) = Cells.Unconstrained_Attribute_Prefix then
                  -- Debug.PrintMsg ("   handling UnconstrainedAttributePrefix", True);
                  GetActualConstraint (FunctionSym,
                                       VarSym,
                                       -- to get
                                       Change,
                                       CopyCell);
                  if Change then
                     Cells.Copy_Contents (VCGHeap,
                                         CopyCell,
                                         P);
                  end if;
               end if;
               P := Cells.Null_Cell;
            else
               P := RightPtr (VCGHeap, P);
            end if;
         end loop;
      end SubstituteParameters;

      --------------------------------------------

   begin --ModelFnConstraints
         --assume that TOS is declared function with all its actual parameters
         --hanging off it;
      FunctionSym := Cells.Get_Symbol_Value (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));
      if Dictionary.IsInstantiation (FunctionSym) then
         -- for instantiations we go and get the constraint of the original generic
         GenericFunctionSym := Dictionary.GetGenericOfInstantiation (FunctionSym);
         PreConNode := STree.RefToNode
           (Dictionary.GetPrecondition (Dictionary.IsAbstract, GenericFunctionSym));
         if PreConNode /= STree.NullNode then
            --get generic precondition
            BuildAnnotationExpnDAG (PreConNode,
                                    Dictionary.LocalScope (GenericFunctionSym),
                                    Abstraction = Dictionary.IsAbstract,
                                    LoopStack,
                                    VCGHeap,
                                    DAGCell);
            -- then replace all generic formal symbols with their instantiated equivalent
            InstantiateParameters (DAGCell,
                                   FunctionSym);
            AddAnySCImplications (RedType, VCGHeap, DAGCell, ShortCircuitStack);
            SubstituteParameters (DAGCell);
            StackCheckStatement (DAGCell, VCGHeap, CheckStack);
            KindOfStackedCheck := Graph.PreconCheck;
         end if;
      else -- not generic
         PreConNode := STree.RefToNode (Dictionary.GetPrecondition (Abstraction,
                                                                                FunctionSym));
         if PreConNode /= STree.NullNode then
            --we need to plant actual pre-condition
            BuildAnnotationExpnDAG (PreConNode,
                                    Dictionary.LocalScope (FunctionSym),
                                    Abstraction = Dictionary.IsAbstract,
                                    LoopStack,
                                    VCGHeap,
                                    DAGCell);
            AddAnySCImplications (RedType, VCGHeap, DAGCell, ShortCircuitStack);
            SubstituteParameters (DAGCell);
            StackCheckStatement (DAGCell, VCGHeap, CheckStack);
            KindOfStackedCheck := Graph.PreconCheck;
         end if;
      end if;

      -- No need to check type of import globals because these must already
      -- be in type because of previous checks.

   end ModelFnConstraints;

   -------------------------------------------------------------------

   procedure SetUpFunctionCall (ThisScope : in Dictionary.Scopes;
                                Prefix    : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     DoRtc;
   --#        in     LineNmbr;
   --#        in     LoopStack;
   --#        in     OutputFile;
   --#        in     RedType;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out CheckStack;
   --#        in out ContainsReals;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out ExpnStack;
   --#        in out Graph.Table;
   --#        in out KindOfStackedCheck;
   --#        in out LexTokenManager.StringTable;
   --#        in out ShortCircuitStack;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out StmtStack.S;
   --#        in out VCGFailure;
   --#        in out VCGHeap;
   --# derives CheckStack,
   --#         KindOfStackedCheck          from *,
   --#                                          CheckStack,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ExpnStack,
   --#                                          LexTokenManager.StringTable,
   --#                                          LoopStack,
   --#                                          Prefix,
   --#                                          RedType,
   --#                                          ShortCircuitStack,
   --#                                          STree.Table,
   --#                                          ThisScope,
   --#                                          VCGHeap &
   --#         ContainsReals,
   --#         Dictionary.Dict,
   --#         Graph.Table,
   --#         LexTokenManager.StringTable,
   --#         ShortCircuitStack,
   --#         Statistics.TableUsage,
   --#         StmtStack.S,
   --#         VCGFailure,
   --#         VCGHeap                     from *,
   --#                                          CheckStack,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ExpnStack,
   --#                                          Graph.Table,
   --#                                          KindOfStackedCheck,
   --#                                          LexTokenManager.StringTable,
   --#                                          LineNmbr,
   --#                                          LoopStack,
   --#                                          Prefix,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          ShortCircuitStack,
   --#                                          StmtStack.S,
   --#                                          STree.Table,
   --#                                          ThisScope,
   --#                                          VCGHeap &
   --#         ErrorHandler.ErrorContext   from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ExpnStack,
   --#                                          LexTokenManager.StringTable,
   --#                                          LoopStack,
   --#                                          RedType,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          STree.Table,
   --#                                          ThisScope,
   --#                                          VCGHeap &
   --#         ExpnStack                   from *,
   --#                                          CheckStack,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          LexTokenManager.StringTable,
   --#                                          LoopStack,
   --#                                          RedType,
   --#                                          ShortCircuitStack,
   --#                                          STree.Table,
   --#                                          ThisScope,
   --#                                          VCGHeap &
   --#         SPARK_IO.FILE_SYS           from *,
   --#                                          CheckStack,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          ExpnStack,
   --#                                          Graph.Table,
   --#                                          KindOfStackedCheck,
   --#                                          LexTokenManager.StringTable,
   --#                                          LineNmbr,
   --#                                          LoopStack,
   --#                                          OutputFile,
   --#                                          Prefix,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          ShortCircuitStack,
   --#                                          StmtStack.S,
   --#                                          STree.Table,
   --#                                          ThisScope,
   --#                                          VCGHeap;
   is
      -- Creates a data structure into which DAGs of actual parameters can be slotted.
      -- We end up with (after the empty data structure is populated by ProcessPositionalArgumentAssociation (or the
      -- named equivalent)):
      -- function --- , --- , --- DAG
      --              |     |
      --             DAG   DAG
      --
      -- (where DAG is a DAG of an actual parameter expression, a down bar is  the A ptr and a right bar a B ptr).
      -- As well as the parameter DAGs, the structure will DAGs for any globals referenced by the function since, in
      -- the proof model, we don't have a concept of global variables.
      --
      -- In addition, each DAG cell may have hanging off its C ptr, a cell conating an index cosntraint.  This gets
      -- used in SubtituteParameters when dealing with attributes of unconstrained arrays where the actual
      -- parameter constrains the formal in some way.

      NumberOfParameters,
      NumberOfGlobals,
      TotalArguments : Natural;
      FunctionSym    : Dictionary.Symbol;
      Abstraction    : Dictionary.Abstractions;
      ConstraintAbstraction : Dictionary.Abstractions;

      ----------------------------

      procedure CopyInGlobals (ParamCount  : in Natural;
                               Sym         : in Dictionary.Symbol;
                               Abstraction : in Dictionary.Abstractions)
      --# global in     Dictionary.Dict;
      --#        in     ExpnStack;
      --#        in     Prefix;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives Statistics.TableUsage,
      --#         VCGHeap               from *,
      --#                                    Abstraction,
      --#                                    Dictionary.Dict,
      --#                                    ExpnStack,
      --#                                    ParamCount,
      --#                                    Prefix,
      --#                                    Sym,
      --#                                    VCGHeap;
      is
         GlobalCell,
         StartPoint : Cells.Cell;
         Unused     : Boolean;
         It         : Dictionary.Iterator;

         function SubstituteProtectedTypeSelfReference (Sym : Dictionary.Symbol) return Dictionary.Symbol
         --# global in Dictionary.Dict;
         --#        in Prefix;
         is
            Result : Dictionary.Symbol;
         begin
            -- if Sym is the implicitly-declared own variable of a protected type
            -- then we must replace it with the "current instance of the protected object"
            --
            -- Background: given protected type PT its operations will globally reference and
            -- derive PT meaning, in this case, "myself".
            -- If an object PO of type PT (or a subtype of PT) is declared then calls to its
            -- operations will take the form PO.Op and the calling environment will be annotated
            -- in terms of PO.  Therefore, when checking that the globals necessary for the call
            -- PO.Op are visible (for example), we need to replace all references to PT into
            -- references to PO before making the check.  The Prefix Symbol of the call is the
            -- symbol we need to substitute in.
            Result := Sym;
            if Prefix /= Dictionary.NullSymbol and then
              Dictionary.IsOwnVariable (Sym) and then
              Dictionary.IsProtectedType (Dictionary.GetOwner (Sym)) then
               Result := Prefix;
            end if;
            return Result;
         end SubstituteProtectedTypeSelfReference;

      begin -- CopyInGlobals
         --# accept F, 10, Unused, "Unused here OK" &
         --#        F, 33, Unused, "Unused here OK";
         CalculateInsertPoint (VCGHeap,
                               ExpnStack,
                               ParamCount,
                                 -- to get
                               StartPoint,
                               Unused);

         It := Dictionary.FirstGlobalVariable (Abstraction,
                                               Sym);
         while not Dictionary.IsNullIterator (It) loop
            CreateReferenceCell (GlobalCell,
                                 VCGHeap,
                                 SubstituteProtectedTypeSelfReference (Dictionary.CurrentSymbol (It)));

            if Cells.Is_Null_Cell (RightPtr (VCGHeap, StartPoint)) then
               SetRightArgument (StartPoint, GlobalCell, VCGHeap);
            else
               StartPoint := RightPtr (VCGHeap, StartPoint);
               SetLeftArgument (StartPoint, GlobalCell, VCGHeap);
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
      end CopyInGlobals;

      ------------------------------------------

   begin -- SetUpFunctionCall
      FunctionSym := Cells.Get_Symbol_Value (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));
      -- We need to know whether we are constructing the function call model using the
      -- abstract or refined signature for it.  Note that we need to consider this separately
      -- for flow annotations and proof annotations; this is because SEPR 1694 introduced the
      -- use of a second, refined constraint in the case of subprograms that manipulate a private
      -- type.  In such cases, weher there is no own variable refinement, we may use the asbtract flow
      -- annotation and the refined proof annotation - so two abstractions are invovled, thus:
      Abstraction := Dictionary.GetAbstraction (FunctionSym, ThisScope);
      ConstraintAbstraction := Dictionary.GetConstraintAbstraction (FunctionSym, ThisScope);
      NumberOfParameters := Dictionary.GetNumberOfSubprogramParameters (FunctionSym);
      if NumberOfParameters = 0 then
         Cells.Set_Kind (VCGHeap, CStacks.Top (VCGHeap, ExpnStack),
                           Cells.Proof_Function);
         if RedType.VCs and DoRtc then
            ModelFnConstraints (ConstraintAbstraction);
         end if;
      end if;
      NumberOfGlobals := Dictionary.GetNumberOfGlobalVariables (Abstraction,
                                                                FunctionSym);
      TotalArguments := NumberOfParameters + NumberOfGlobals;
      CreateEmptyList (TotalArguments, VCGHeap, ExpnStack);
      if NumberOfGlobals > 0 then
         CopyInGlobals (NumberOfParameters,
                        FunctionSym,
                        Abstraction);
      end if;

      -- replace function symbol with related implicit proof function if complete
      if NumberOfParameters = 0 then
         Cells.Set_Symbol_Value (VCGHeap,
                               CStacks.Top (VCGHeap, ExpnStack),
                               Dictionary.GetImplicitProofFunction (Abstraction,
                                                                    FunctionSym));
         FunctionSym := Cells.Get_Symbol_Value (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));
      end if;

      -- do this for parameterless functions AFTER any globals have
      -- been copied in.
      if RedType.VCs and DoRtc and NumberOfParameters = 0 then
         -- "CheckPlantRvalueAssumptions" not needed here because a parameterless function
         -- can't have any R-values;
         UnStackRtcs (LineNmbr, RedType, VCGHeap, CheckStack, KindOfStackedCheck);
         if RedType.RTC then
            AssumeFnReturnType (FunctionSym,
                                CStacks.Top (VCGHeap, ExpnStack));
         end if;
      end if;
   end SetUpFunctionCall;

   -----------------------------------------------------------------------

   procedure SetUpArrayAccess
   --# global in     Dictionary.Dict;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    VCGHeap;
   is
      NumberOfDimensions : Positive;
      --ExpressionCell,
      DAGCell            : Cells.Cell;
      TypeSym            : Dictionary.Symbol;

   begin
      TypeSym := Cells.Get_Symbol_Value (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));
      if Dictionary.IsTypeMark (TypeSym) then
         NumberOfDimensions := Dictionary.GetNumberOfDimensions (TypeSym);
      else
         NumberOfDimensions := Dictionary.GetNumberOfDimensions (Dictionary.GetType (TypeSym));
      end if;
      CreateCellKind (DAGCell, VCGHeap, Cells.List_Function);
      CStacks.Push (VCGHeap, DAGCell, ExpnStack);
      CreateEmptyList (NumberOfDimensions, VCGHeap, ExpnStack);
   end SetUpArrayAccess;

   -----------------------------------------------------------------------

   procedure ProcessPositionalArgumentAssociation (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     DoRtc;
   --#        in     RedType;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out CheckStack;
   --#        in out ContainsReals;
   --#        in out ExpnStack;
   --#        in out ShortCircuitStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives CheckStack,
   --#         ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    CheckStack,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    Scope,
   --#                                    ShortCircuitStack,
   --#                                    STree.Table,
   --#                                    VCGHeap &
   --#         ContainsReals         from *,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    STree.Table,
   --#                                    VCGHeap &
   --#         ShortCircuitStack     from *,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    Scope,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      ExpressionCell : Cells.Cell;
      TOSkind        : Cells.Cell_Kind;
      ConversionTargetType,
      ConversionSourceType : Dictionary.Symbol;
      ConstraintCell  : Cells.Cell;
      ConstraintIndex : Dictionary.Symbol;

   begin -- ProcessPositionalArgumentAssociation
      CStacks.PopOff (VCGHeap, ExpnStack, ExpressionCell);
      TOSkind := Cells.Get_Kind (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));

      if TOSkind = Cells.Pending_Function then
         if RedType.RTC and DoRtc then
            -- the wffs have provided the expected type.  We extract that and use it to
            -- constraint check the parameter.  If the function is an unchecked conversion
            -- and the wffs have determined that the subtype expected and given are identical,
            -- then no type symbol is planted and no check is generated.
            CheckConstraintRunTimeError (STree.NodeSymbol (Node),
                                         ExpressionCell,
                                         RedType,
                                         Scope,
                                         VCGHeap,
                                         ShortCircuitStack,
                                         CheckStack,
                                         ContainsReals);
         end if;

         -- We may need to convert the actual parameter by inserting some inherit
         -- derefences in front of it; conversion is required if we have called
         -- an inherited root function.  The parameter in this case must be an
         -- object.
         ConvertTaggedActualIfNecessary (Cells.Get_Symbol_Value
                                           (VCGHeap, CStacks.Top (VCGHeap, ExpnStack)),
                                         VCGHeap,
                                         ExpressionCell); -- function symbol

         -- If the formal parameter is unconstrained and the actual is a constrained subtype, then
         -- the wffs will have planted a constraining type at the expression node.  If we find such a
         -- type, we link it into the actual parameter expression DAG but using the expression's
         -- auxialliary (C) ptr.  Linking it in this way means that it is not part of the DAG itself and won't
         -- be printed; however, it will be available when we want to substitute actuals for formals in
         -- any check of the called function's precondition.
         ConstraintIndex := STree.NodeSymbol
           (STree.ExpressionFromPositionalArgumentAssociation (Node));
         if ConstraintIndex /= Dictionary.NullSymbol then
            CreateCellKind (ConstraintCell, VCGHeap, Cells.Constraining_Index);
            Cells.Set_Symbol_Value (VCGHeap,
                                  ConstraintCell,
                                  ConstraintIndex);
            SetAuxPtr (ExpressionCell, ConstraintCell, VCGHeap);
         end if;
         InsertParameterInNextFreeSlot (CStacks.Top (VCGHeap, ExpnStack),
                                        ExpressionCell,
                                        VCGHeap);
      elsif TOSkind = Cells.List_Function then
         if RedType.RTC and DoRtc then
            CheckConstraintRunTimeError (STree.NodeSymbol (Node),
                                         ExpressionCell,
                                         RedType,
                                         Scope,
                                         VCGHeap,
                                         ShortCircuitStack,
                                         CheckStack,
                                         ContainsReals);
         end if;
         InsertParameterInNextFreeSlot (CStacks.Top (VCGHeap, ExpnStack),
                                        ExpressionCell,
                                        VCGHeap);
      elsif TOSkind = Cells.Fixed_Var then
         ConversionSourceType := STree.NodeSymbol (Node);
         ConversionTargetType := Cells.Get_Symbol_Value (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));
         CStacks.Pop (VCGHeap, ExpnStack);                  -- get rid of type mark
         CStacks.Push (VCGHeap, ExpressionCell, ExpnStack); -- restore expression
         if IsRealType (ConversionSourceType) and then
            (IsIntegerType (ConversionTargetType) or else IsModularType (ConversionTargetType))
         then
            PushFunction (Cells.Trunc_Function, VCGHeap, ExpnStack);
         end if;

         if RedType.RTC and DoRtc then
            if IsScalarType (ConversionTargetType) then

               CheckConstraintRunTimeError (ConversionTargetType,
                                            CStacks.Top (VCGHeap, ExpnStack),
                                            RedType,
                                            Scope,
                                            VCGHeap,
                                            ShortCircuitStack,
                                            CheckStack,
                                            ContainsReals);
            end if;
         end if;
      else -- must be dealing with first indexed expression of array access
         if RedType.RTC and DoRtc then
            CheckConstraintRunTimeError (STree.NodeSymbol (Node),
                                         ExpressionCell,
                                         RedType,
                                         Scope,
                                         VCGHeap,
                                         ShortCircuitStack,
                                         CheckStack,
                                         ContainsReals);
         end if;
         SetUpArrayAccess;
         InsertParameterInNextFreeSlot (CStacks.Top (VCGHeap, ExpnStack),
                                        ExpressionCell,
                                        VCGHeap);
      end if;
   end ProcessPositionalArgumentAssociation;

   -----------------------------------------------------------------------

   procedure ProcessNamedArgumentAssociation (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     DoRtc;
   --#        in     RedType;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out CheckStack;
   --#        in out ContainsReals;
   --#        in out ExpnStack;
   --#        in out ShortCircuitStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives CheckStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    CheckStack,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    Scope,
   --#                                    ShortCircuitStack,
   --#                                    STree.Table,
   --#                                    VCGHeap &
   --#         ContainsReals         from *,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    Node,
   --#                                    RedType,
   --#                                    STree.Table &
   --#         ExpnStack             from *,
   --#                                    VCGHeap &
   --#         ShortCircuitStack     from *,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    Scope,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      InsertPoint,
      ExpressionCell  : Cells.Cell;
      FunctionSym     : Dictionary.Symbol;
      ParamPos        : Positive;
      LastOne         : Boolean;
      ConstraintCell  : Cells.Cell;
      ConstraintIndex : Dictionary.Symbol;

      ----------------------------------------------------------------

      function FindIdentifier (Node : STree.SyntaxNode)
                              return STree.SyntaxNode
      --# global in STree.Table;
      is
         IdentNode : STree.SyntaxNode;

      begin
         if STree.SyntaxNodeType
           (STree.Child_Node (Node)) = SPSymbols.simple_name
         then
            IdentNode := STree.Child_Node (STree.Child_Node (Node));
         else
            IdentNode := STree.Child_Node
              (STree.Next_Sibling
               (STree.Child_Node (Node)));
         end if;
         return IdentNode;
      end FindIdentifier;

      --------------------------------------------------------------

      procedure GetParamNumber (Name    : in     LexTokenManager.LexString;
                                FunSym  : in     Dictionary.Symbol;
                                ParamNo :    out Positive)
      --# global in Dictionary.Dict;
      --# derives ParamNo from Dictionary.Dict,
      --#                      FunSym,
      --#                      Name;
      is
         It    : Dictionary.Iterator;
         Sym   : Dictionary.Symbol;
      begin
         It := Dictionary.FirstSubprogramParameter (FunSym);
         SystemErrors.RTAssert
           (not Dictionary.IsNullIterator (It),
            SystemErrors.PreconditionFailure,
            "Can't find first subprogram parameter in BuildExpnDAG.GetParamNumber");
         loop
            Sym := Dictionary.CurrentSymbol (It);
            exit when Dictionary.GetSimpleName (Sym) = Name;
            It := Dictionary.NextSymbol (It);
            exit when Dictionary.IsNullIterator (It);
         end loop;
         ParamNo := Dictionary.GetSubprogramParameterNumber (Sym);
      end GetParamNumber;

      --------------------------------------------------------------

   begin --ProcessNamedArgumentAssociation
         --we must be dealing with a function call
      CStacks.PopOff (VCGHeap, ExpnStack, ExpressionCell);

      if RedType.RTC and DoRtc then
         CheckConstraintRunTimeError (STree.NodeSymbol (Node),
                                      ExpressionCell,
                                      RedType,
                                      Scope,
                                      VCGHeap,
                                      ShortCircuitStack,
                                      CheckStack,
                                      ContainsReals);
      end if;
      FunctionSym := Cells.Get_Symbol_Value (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));

      -- We may need to convert the actual parameter by inserting some inherit
      -- derefences in front of it; conversion is required if we have called
      -- an inherited root function.  The parameter in this case must be an
      -- object.
      ConvertTaggedActualIfNecessary (FunctionSym,
                                      VCGHeap,
                                      ExpressionCell);

      -- If the formal parameter is unconstrained and the actual is a constrained subtype, then
      -- the wffs will have planted a constraining type at the expression node.  If we find such a
      -- type, we link it into the actual parameter expression DAG but using the expression's
      -- auxialliary (C) ptr.  Linking it in this way means that it is not part of the DAG itself and won't
      -- be printed; however, it will be available when we want to substitute actuals for formals in
      -- any check of the called function's precondition.
      ConstraintIndex := STree.NodeSymbol
        (STree.ExpressionFromNamedArgumentAssociation (Node));
      if ConstraintIndex /= Dictionary.NullSymbol then
         CreateCellKind (ConstraintCell, VCGHeap, Cells.Constraining_Index);
         Cells.Set_Symbol_Value (VCGHeap,
                               ConstraintCell,
                               ConstraintIndex);
         SetAuxPtr (ExpressionCell, ConstraintCell, VCGHeap);
      end if;
      GetParamNumber (STree.NodeLexString (FindIdentifier (Node)),
                      FunctionSym,
                        -- to get
                      ParamPos);
      CalculateInsertPoint (VCGHeap,
                            ExpnStack,
                            ParamPos,
                              -- to get
                            InsertPoint,
                            LastOne);
      if LastOne then
         SetRightArgument (InsertPoint, ExpressionCell, VCGHeap);
      else
         SetLeftArgument (InsertPoint, ExpressionCell, VCGHeap);
      end if;
   end ProcessNamedArgumentAssociation;

   -----------------------------------------------------------------------

   procedure ProcessNameArgumentList
   --# global in     AssumeRvalues;
   --#        in     CommandLineData.Content;
   --#        in     DoRtc;
   --#        in     ExpnScope;
   --#        in     LineNmbr;
   --#        in     LoopStack;
   --#        in     OutputFile;
   --#        in     RedType;
   --#        in     ReferencedVars;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out CheckStack;
   --#        in out ContainsReals;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out ExpnStack;
   --#        in out FlowHeap;
   --#        in out Graph.Table;
   --#        in out KindOfStackedCheck;
   --#        in out LexTokenManager.StringTable;
   --#        in out ShortCircuitStack;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out StmtStack.S;
   --#        in out VCGFailure;
   --#        in out VCGHeap;
   --# derives CheckStack,
   --#         KindOfStackedCheck          from *,
   --#                                          AssumeRvalues,
   --#                                          CheckStack,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ExpnScope,
   --#                                          ExpnStack,
   --#                                          FlowHeap,
   --#                                          Graph.Table,
   --#                                          LexTokenManager.StringTable,
   --#                                          LoopStack,
   --#                                          RedType,
   --#                                          ReferencedVars,
   --#                                          Scope,
   --#                                          ShortCircuitStack,
   --#                                          StmtStack.S,
   --#                                          STree.Table,
   --#                                          VCGHeap &
   --#         ContainsReals,
   --#         Dictionary.Dict,
   --#         Graph.Table,
   --#         LexTokenManager.StringTable,
   --#         ShortCircuitStack,
   --#         Statistics.TableUsage,
   --#         StmtStack.S,
   --#         VCGFailure,
   --#         VCGHeap                     from *,
   --#                                          AssumeRvalues,
   --#                                          CheckStack,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ExpnScope,
   --#                                          ExpnStack,
   --#                                          FlowHeap,
   --#                                          Graph.Table,
   --#                                          KindOfStackedCheck,
   --#                                          LexTokenManager.StringTable,
   --#                                          LineNmbr,
   --#                                          LoopStack,
   --#                                          RedType,
   --#                                          ReferencedVars,
   --#                                          Scope,
   --#                                          ShortCircuitStack,
   --#                                          StmtStack.S,
   --#                                          STree.Table,
   --#                                          VCGHeap &
   --#         ErrorHandler.ErrorContext   from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ExpnScope,
   --#                                          ExpnStack,
   --#                                          LexTokenManager.StringTable,
   --#                                          LoopStack,
   --#                                          RedType,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          STree.Table,
   --#                                          VCGHeap &
   --#         ExpnStack                   from *,
   --#                                          VCGHeap &
   --#         FlowHeap                    from *,
   --#                                          AssumeRvalues,
   --#                                          DoRtc,
   --#                                          ExpnStack,
   --#                                          RedType,
   --#                                          ReferencedVars,
   --#                                          VCGHeap &
   --#         SPARK_IO.FILE_SYS           from *,
   --#                                          AssumeRvalues,
   --#                                          CheckStack,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          ExpnScope,
   --#                                          ExpnStack,
   --#                                          FlowHeap,
   --#                                          Graph.Table,
   --#                                          KindOfStackedCheck,
   --#                                          LexTokenManager.StringTable,
   --#                                          LineNmbr,
   --#                                          LoopStack,
   --#                                          OutputFile,
   --#                                          RedType,
   --#                                          ReferencedVars,
   --#                                          Scope,
   --#                                          ShortCircuitStack,
   --#                                          StmtStack.S,
   --#                                          STree.Table,
   --#                                          VCGHeap;
   is
      TOSkind     : Cells.Cell_Kind;
      Temp        : Cells.Cell;
      TypeSym     : Dictionary.Symbol;
      FunctionSym : Dictionary.Symbol;
      Abstraction : Dictionary.Abstractions;
      ConstraintAbstraction : Dictionary.Abstractions;

   begin
      TOSkind := Cells.Get_Kind (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));
      if TOSkind = Cells.Pending_Function then
         Cells.Set_Kind (VCGHeap, CStacks.Top (VCGHeap, ExpnStack),
                           Cells.Proof_Function);
         FunctionSym := Cells.Get_Symbol_Value (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));
         -- We need to know whether we are constructing the function call model using the
         -- abstract or refined signature for it.  Note that we need to consider this separately
         -- for flow annotations and proof annotations; this is because SEPR 1694 introduced the
         -- use of a second, refined constraint in the case of subprograms that manipulate a private
         -- type.  In such cases, weher there is no own variable refinement, we may use the asbtract flow
         -- annotation and the refined proof annotation - so two abstractions are invovled, thus:
         Abstraction := Dictionary.GetAbstraction (FunctionSym, ExpnScope);
         ConstraintAbstraction := Dictionary.GetConstraintAbstraction (FunctionSym, ExpnScope);

         if RedType.VCs and DoRtc then
            ModelFnConstraints (ConstraintAbstraction);
         end if; -- restructure like SetUpFnCall...

         -- function now complete so we can replace it with its implicit proof equivalent
         Cells.Set_Symbol_Value (VCGHeap,
                               CStacks.Top (VCGHeap, ExpnStack),
                               Dictionary.GetImplicitProofFunction (Abstraction,
                                                                    FunctionSym));

         if RedType.VCs and DoRtc then
            -- Before unstacking any RTCs associated with the now complete function call,
            -- we plant hypotheses to assume that all R-values (including the function's
            -- parameters are in type.
            CheckPlantRvalueAssumptions;
            -- then unstack checks
            UnStackRtcs (LineNmbr, RedType, VCGHeap, CheckStack, KindOfStackedCheck);
            if RedType.RTC then
               AssumeFnReturnType (Cells.Get_Symbol_Value (VCGHeap, CStacks.Top (VCGHeap, ExpnStack)),
                                   CStacks.Top (VCGHeap, ExpnStack));
            end if;
         end if;

      elsif TOSkind = Cells.List_Function then
         -- complete element model and store type so far in case of further
         -- indexing (to handle array of arrays case)
         CStacks.PopOff (VCGHeap, ExpnStack, Temp);
         TypeSym := Cells.Get_Symbol_Value (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));
         if Dictionary.IsTypeMark (TypeSym) then
            TypeSym := Dictionary.GetArrayComponent (TypeSym);
         else
            TypeSym := Dictionary.GetArrayComponent (Dictionary.GetType (TypeSym));
         end if;
         CStacks.Push (VCGHeap, Temp, ExpnStack);
         PushOperator (Binary, SPSymbols.comma, VCGHeap, ExpnStack);
         PushFunction (Cells.Element_Function, VCGHeap, ExpnStack);

         -- Note the TypeSym of the array component here.  This used later on
         -- in BuildGraph.ModelProcedureCall to get the type of an array element
         -- actual parameter
         Cells.Set_Symbol_Value (VCGHeap, CStacks.Top (VCGHeap, ExpnStack), TypeSym);

         -- elsif TOSkind = Cells.Fixed_Var
         -- then
         --  null; --type conversions not done yet
      end if;
   end ProcessNameArgumentList;

   -----------------------------------------------------------------------

   procedure ModelQualifiedExpression (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     DoRtc;
   --#        in     RedType;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out CheckStack;
   --#        in out ContainsReals;
   --#        in out ExpnStack;
   --#        in out ShortCircuitStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives CheckStack,
   --#         ShortCircuitStack,
   --#         Statistics.TableUsage from *,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    Scope,
   --#                                    ShortCircuitStack,
   --#                                    STree.Table,
   --#                                    VCGHeap &
   --#         ContainsReals         from *,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    STree.Table,
   --#                                    VCGHeap &
   --#         ExpnStack             from *,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap &
   --#         VCGHeap               from *,
   --#                                    CheckStack,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    Scope,
   --#                                    ShortCircuitStack,
   --#                                    STree.Table;
   is
      ExpnCell,
      TypeMarkCell : Cells.Cell;
   begin
      if STree.SyntaxNodeType
        (STree.Next_Sibling
         (STree.Child_Node (Node))) = SPSymbols.expression then
         -- discard type indication and return its argument to top of stack;
         CStacks.PopOff (VCGHeap, ExpnStack, ExpnCell);
         -- the topmost stack cell contains the typemark;
         CStacks.PopOff (VCGHeap, ExpnStack, TypeMarkCell);
         CStacks.Push (VCGHeap, ExpnCell, ExpnStack);

         if RedType.RTC and DoRtc then
            CheckConstraintRunTimeError (Cells.Get_Symbol_Value (VCGHeap, TypeMarkCell),
                                         ExpnCell,
                                         RedType,
                                         Scope,
                                         VCGHeap,
                                         ShortCircuitStack,
                                         CheckStack,
                                         ContainsReals);
         end if;
      end if;
   end ModelQualifiedExpression;

   ----------------------------------------------------------------------

   procedure UpProcessAggregateChoice (Node : in STree.SyntaxNode)
   --# global in     STree.Table;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      RangeNodeType   : SPSymbols.SPSymbol;
      RangeExpression : Cells.Cell;

   begin
      RangeNodeType := STree.SyntaxNodeType
        (STree.Next_Sibling
         (STree.Child_Node (Node)));
      if RangeNodeType = SPSymbols.simple_expression then
         PushOperator (Binary, SPSymbols.double_dot, VCGHeap, ExpnStack);
      elsif RangeNodeType = SPSymbols.range_constraint then
         TransformRangeConstraint  (VCGHeap, ExpnStack);
         CStacks.PopOff (VCGHeap, ExpnStack, RangeExpression);
         CStacks.Pop (VCGHeap, ExpnStack); --discard type mark part of range
         CStacks.Push (VCGHeap, RangeExpression, ExpnStack);
      elsif Cells.Get_Kind (VCGHeap, CStacks.Top (VCGHeap, ExpnStack)) = Cells.Fixed_Var then --type mark found
         TransformTypeName (VCGHeap, ExpnStack);
      end if;
   end UpProcessAggregateChoice;

   ---------------------------------------------------------------------

   procedure UpProcessNamedAssociationRep (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     DoRtc;
   --#        in     RedType;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out CheckStack;
   --#        in out ContainsReals;
   --#        in out ExpnStack;
   --#        in out ShortCircuitStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives CheckStack,
   --#         ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    CheckStack,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    Scope,
   --#                                    ShortCircuitStack,
   --#                                    STree.Table,
   --#                                    VCGHeap &
   --#         ContainsReals         from *,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    Node,
   --#                                    RedType,
   --#                                    STree.Table &
   --#         ShortCircuitStack     from *,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    Scope,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      AggExp : Cells.Cell;
   begin
      -- check components of array & record aggregates using named assoc
      if RedType.RTC and DoRtc then
         CheckConstraintRunTimeError (STree.NodeSymbol (Node),
                                      CStacks.Top (VCGHeap, ExpnStack),
                                      RedType,
                                      Scope,
                                      VCGHeap,
                                      ShortCircuitStack,
                                      CheckStack,
                                      ContainsReals);
      end if;
      PushOperator (Binary, SPSymbols.becomes, VCGHeap, ExpnStack);

      if DoingArrayAggregate (VCGHeap, ExpnStack) then
         if STree.SyntaxNodeType (STree.Child_Node (Node)) =
           SPSymbols.named_association_rep then
            PushOperator (Binary, SPSymbols.comma, VCGHeap, ExpnStack);
         end if;
      else -- record
         CStacks.PopOff (VCGHeap, ExpnStack, AggExp);
         InsertAssociation (CStacks.Top (VCGHeap, ExpnStack), AggExp, VCGHeap);
      end if;
   end UpProcessNamedAssociationRep;

   ---------------------------------------------------------------------

   procedure UpProcessNamedRecordComponentAssociation (Node : in     STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     DoRtc;
   --#        in     RedType;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out CheckStack;
   --#        in out ContainsReals;
   --#        in out ExpnStack;
   --#        in out ShortCircuitStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives CheckStack,
   --#         ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    CheckStack,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    Scope,
   --#                                    ShortCircuitStack,
   --#                                    STree.Table,
   --#                                    VCGHeap &
   --#         ContainsReals         from *,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    Node,
   --#                                    RedType,
   --#                                    STree.Table &
   --#         ShortCircuitStack     from *,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    Scope,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      AggExp : Cells.Cell;
   begin
      -- Node is named_record_component_association
      -- Direction is UP
      -- TOS is expression to be associated
      -- 2nd TOS is field name
      -- 3rd TOS is incomplete aggregate being constructed.

      -- check components of array & record aggregates using named assoc
      if RedType.RTC and DoRtc then
         CheckConstraintRunTimeError (STree.NodeSymbol (Node),
                                      CStacks.Top (VCGHeap, ExpnStack),
                                      RedType,
                                      Scope,
                                      VCGHeap,
                                      ShortCircuitStack,
                                      CheckStack,
                                      ContainsReals);
      end if;
      -- associated field name with expression
      PushOperator (Binary, SPSymbols.becomes, VCGHeap, ExpnStack);
      CStacks.PopOff (VCGHeap, ExpnStack, AggExp);
      InsertAssociation (CStacks.Top (VCGHeap, ExpnStack), AggExp, VCGHeap);
   end UpProcessNamedRecordComponentAssociation;

   ---------------------------------------------------------------------

   procedure UpProcessPositionalRecordComponentAssociation (Node : in     STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     DoRtc;
   --#        in     RedType;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out CheckStack;
   --#        in out ContainsReals;
   --#        in out ExpnStack;
   --#        in out ShortCircuitStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives CheckStack,
   --#         ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    CheckStack,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    Scope,
   --#                                    ShortCircuitStack,
   --#                                    STree.Table,
   --#                                    VCGHeap &
   --#         ContainsReals         from *,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    Node,
   --#                                    RedType,
   --#                                    STree.Table &
   --#         ShortCircuitStack     from *,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    Scope,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      AggExp,
      TypeCell : Cells.Cell;
   begin
      -- Node is positional_record_component_association
      -- Direction is UP
      -- TOS is expression to be associated
      -- 2nd TOS is incomplete aggregate being constructed.
      -- 3rd TOS is agggregate counter giving current field number

      -- check components for RTCs
      if RedType.RTC and DoRtc then
         CheckConstraintRunTimeError (STree.NodeSymbol (Node),
                                      CStacks.Top (VCGHeap, ExpnStack),
                                      RedType,
                                      Scope,
                                      VCGHeap,
                                      ShortCircuitStack,
                                      CheckStack,
                                      ContainsReals);
      end if;
      CreateFixedVarCell (TypeCell,
                          VCGHeap,
                          Dictionary.GetRecordComponent
                          (AggregateType (VCGHeap, ExpnStack),
                           CurrentFieldOrIndex (VCGHeap, ExpnStack)));
      CStacks.Push (VCGHeap, TypeCell, ExpnStack);
      SwitchAndPush (SPSymbols.becomes, VCGHeap, ExpnStack);
      IncCurrentFieldOrIndex (ExpnStack, VCGHeap);
      CStacks.PopOff (VCGHeap, ExpnStack, AggExp);
      InsertAssociation (CStacks.Top (VCGHeap, ExpnStack), AggExp, VCGHeap);
   end UpProcessPositionalRecordComponentAssociation;

   ---------------------------------------------------------------------

   procedure UpProcessAggregateOrExpression (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     DoRtc;
   --#        in     RedType;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out CheckStack;
   --#        in out ContainsReals;
   --#        in out ExpnStack;
   --#        in out LexTokenManager.StringTable;
   --#        in out ShortCircuitStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives CheckStack,
   --#         ShortCircuitStack           from *,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ExpnStack,
   --#                                          Node,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          ShortCircuitStack,
   --#                                          STree.Table,
   --#                                          VCGHeap &
   --#         ContainsReals               from *,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          Node,
   --#                                          RedType,
   --#                                          STree.Table &
   --#         ExpnStack,
   --#         LexTokenManager.StringTable,
   --#         Statistics.TableUsage,
   --#         VCGHeap                     from *,
   --#                                          CheckStack,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ExpnStack,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          ShortCircuitStack,
   --#                                          STree.Table,
   --#                                          VCGHeap;
   is
      IndexType     : Dictionary.Symbol;
      CounterCell,
      AttribCell,
      TypeCell      : Cells.Cell;
      CounterString : LexTokenManager.LexString;
      AggExp : Cells.Cell;

   begin --UpProcessAggregateOrExpression
      if STree.SyntaxNodeType (STree.ParentNode (Node)) =
         SPSymbols.positional_association_rep or else
         STree.Next_Sibling (Node) /=
         STree.NullNode
      then
         -- check components of array & record aggregates using
         -- positional association, but not the others part
         if RedType.RTC and DoRtc then
            CheckConstraintRunTimeError (STree.NodeSymbol (Node),
                                         CStacks.Top (VCGHeap, ExpnStack),
                                         RedType,
                                         Scope,
                                         VCGHeap,
                                         ShortCircuitStack,
                                         CheckStack,
                                         ContainsReals);

         end if;
         if DoingArrayAggregate (VCGHeap, ExpnStack) then
            CreateCellKind (TypeCell, VCGHeap, Cells.Fixed_Var);
            IndexType := Dictionary.GetArrayIndex (AggregateType (VCGHeap, ExpnStack), 1);

            Cells.Set_Symbol_Value (VCGHeap, TypeCell, IndexType);
            CStacks.Push (VCGHeap, TypeCell, ExpnStack);

            CreateAttribValueCell (AttribCell, VCGHeap, LexTokenManager.FirstToken);
            CStacks.Push (VCGHeap, AttribCell, ExpnStack);
            PushOperator (Binary, SPSymbols.apostrophe, VCGHeap, ExpnStack);

            if Dictionary.TypeIsEnumeration (IndexType) then
               for I in Integer range 2 .. CurrentFieldOrIndex (VCGHeap, ExpnStack) loop
                  --# accept F, 41, "Stable expression expected here";
                  if Dictionary.TypeIsBoolean (IndexType) then
                     PushOperator (Unary, SPSymbols.RWnot, VCGHeap, ExpnStack);
                  else
                     PushFunction (Cells.Succ_Function, VCGHeap, ExpnStack);
                  end if;
                  --# end accept;
               end loop;
            else --index type is numeric discrete
               if CurrentFieldOrIndex (VCGHeap, ExpnStack) > 1 then
                  LexTokenManager.InsertNat (CurrentFieldOrIndex (VCGHeap, ExpnStack) - 1,
                                             CounterString);
                  CreateManifestConstCell (CounterCell, VCGHeap, CounterString);
                  CStacks.Push (VCGHeap, CounterCell, ExpnStack);
                  PushOperator (Binary, SPSymbols.plus, VCGHeap, ExpnStack);
               end if;
            end if;
            PushFunction (Cells.List_Function, VCGHeap, ExpnStack);

         else --record aggregate
            CreateFixedVarCell (TypeCell,
                                VCGHeap,
                                Dictionary.GetRecordComponent
                                (AggregateType (VCGHeap, ExpnStack),
                                 CurrentFieldOrIndex (VCGHeap, ExpnStack)));
            CStacks.Push (VCGHeap, TypeCell, ExpnStack);
         end if;

         SwitchAndPush (SPSymbols.becomes, VCGHeap, ExpnStack);

         IncCurrentFieldOrIndex (ExpnStack, VCGHeap);

         if DoingArrayAggregate (VCGHeap, ExpnStack) then
            if STree.Next_Sibling (Node) =
              STree.NullNode
            then
               PushOperator (Binary, SPSymbols.comma, VCGHeap, ExpnStack);
            end if;
         else -- record
            CStacks.PopOff (VCGHeap, ExpnStack, AggExp);
            InsertAssociation (CStacks.Top (VCGHeap, ExpnStack), AggExp, VCGHeap);
         end if;
      end if;
   end UpProcessAggregateOrExpression;

   ---------------------------------------------------------------------

   procedure UpProcessComponentAssociation (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     DoRtc;
   --#        in     RedType;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out CheckStack;
   --#        in out ContainsReals;
   --#        in out ExpnStack;
   --#        in out ShortCircuitStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives CheckStack,
   --#         ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    CheckStack,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    Scope,
   --#                                    ShortCircuitStack,
   --#                                    STree.Table,
   --#                                    VCGHeap &
   --#         ContainsReals         from *,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    Node,
   --#                                    RedType,
   --#                                    STree.Table &
   --#         ShortCircuitStack     from *,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    Scope,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
   begin
      if STree.Next_Sibling
        (STree.Child_Node
         (STree.Child_Node (Node))) /=
        STree.NullNode then
         -- check the others part of an array aggregate with either
         -- named or positional association; but does not cover a
         -- lone others part.
         if RedType.RTC and DoRtc then
            CheckConstraintRunTimeError (STree.NodeSymbol (Node),
                                         CStacks.Top (VCGHeap, ExpnStack),
                                         RedType,
                                         Scope,
                                         VCGHeap,
                                         ShortCircuitStack,
                                         CheckStack,
                                         ContainsReals);
         end if;
         SwitchAndPush (SPSymbols.comma, VCGHeap, ExpnStack);
      end if;
   end UpProcessComponentAssociation;

   ---------------------------------------------------------------------

   procedure UpProcessAggregate
   --# global in     Dictionary.Dict;
   --#        in     DoRtc;
   --#        in     Node;
   --#        in     RedType;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out CheckStack;
   --#        in out ContainsReals;
   --#        in out ExpnStack;
   --#        in out ShortCircuitStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives CheckStack,
   --#         ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    CheckStack,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    Scope,
   --#                                    ShortCircuitStack,
   --#                                    STree.Table,
   --#                                    VCGHeap &
   --#         ContainsReals         from *,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    Node,
   --#                                    RedType,
   --#                                    STree.Table &
   --#         ShortCircuitStack     from *,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    Scope,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      TempAgg : Cells.Cell;
   begin
      --RTC--a lone others clause gets picked up here and will need a RTC.
      --     In this case TOS is the expn dag of the others expression and
      --     all that happens is that a mk_array get put on top of it so
      --     that (others => X) becomes mk_array (X).  The expected type
      --     will be found at the component_association node (1 down) and
      --     the check is only needed if the node 2 down is an agg_or_exp.

      --RTC--added if statement to control whether RTC on lone others needed
      if STree.SyntaxNodeType
        (STree.Child_Node
         (STree.Child_Node (Node))) =
        SPSymbols.aggregate_or_expression then
         --there is a lone others clause that needs a RTC and the expn is TOS
         if RedType.RTC and DoRtc then
            CheckConstraintRunTimeError
              (STree.NodeSymbol (STree.Child_Node (Node)),
               CStacks.Top (VCGHeap, ExpnStack),
               RedType,
               Scope,
               VCGHeap,
               ShortCircuitStack,
               CheckStack,
               ContainsReals);

         end if;
      end if;
      --RTC--end of new if clause

      -- Tidy up expression stack

      -- At this point the stack is rather confused (even for an ex-FORTH programmer.
      -- If we are doing a record then TOS is the IncompleteAggregate function and its arguments,
      --                           2nd TOS is the aggregate counter used for positional association.
      --
      -- If we are doing an array then TOS is the comma-delimited list of arguments to the MkAggregate func,
      --                           2nd TOS is the IncompleteAggregate function itself,
      --                           3rd TOS is the aggregate counter
      --
      CStacks.PopOff (VCGHeap, ExpnStack, TempAgg);  -- hold the aggregate expression or list
      if Cells.Get_Kind (VCGHeap, CStacks.Top (VCGHeap, ExpnStack)) =
        Cells.Aggregate_Counter
      then -- we are doing a record and just need to get rid of the counter
         CStacks.Pop (VCGHeap, ExpnStack); -- get rid of counter
      else -- we are doing an array and TOS is the Incomplete function which needs to be connected to
         -- the comma-delimited list
         SetRightArgument (CStacks.Top (VCGHeap, ExpnStack), TempAgg, VCGHeap);
         CStacks.PopOff (VCGHeap, ExpnStack, TempAgg);  -- hold the now complete aggregate expression
         -- and then get rid of the exposed counter
         CStacks.Pop (VCGHeap, ExpnStack);
      end if;
      -- Convert aggregate to a finished MkAggregate function
      Cells.Set_Kind (VCGHeap, TempAgg, Cells.Mk_Aggregate);
      -- Finally, restore aggregate DAG to TOS
      CStacks.Push (VCGHeap, TempAgg, ExpnStack);
   end UpProcessAggregate;

   ---------------------------------------------------------------------
   --                       Attribute Processing                      --
   ---------------------------------------------------------------------

   procedure DownProcessAttributeIdent (Node : in STree.SyntaxNode)
   --# global in     STree.Table;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      DAGCell : Cells.Cell;
   begin
      CreateAttribValueCell (DAGCell, VCGHeap, STree.NodeLexString (Node));
      CStacks.Push (VCGHeap, DAGCell, ExpnStack);
      PushOperator (Binary,
                    SPSymbols.apostrophe,
                    VCGHeap,
                    ExpnStack);
   end DownProcessAttributeIdent;

   ---------------------------------------------------------------------

   -- Note name here is different from similar subunit within BuildAnnotationExpnDAG
   -- do avoid clash with Ada83 "no identical subunit names" rule.
   procedure UpAttributeDesignator (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     DoRtc;
   --#        in     RedType;
   --#        in     STree.Table;
   --#        in out CheckStack;
   --#        in out ExpnStack;
   --#        in out LexTokenManager.StringTable;
   --#        in out ShortCircuitStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives CheckStack,
   --#         ShortCircuitStack           from *,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ExpnStack,
   --#                                          Node,
   --#                                          RedType,
   --#                                          ShortCircuitStack,
   --#                                          STree.Table,
   --#                                          VCGHeap &
   --#         ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap                     from *,
   --#                                          CheckStack,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ExpnStack,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          RedType,
   --#                                          ShortCircuitStack,
   --#                                          STree.Table,
   --#                                          VCGHeap &
   --#         LexTokenManager.StringTable from *,
   --#                                          Dictionary.Dict,
   --#                                          ExpnStack,
   --#                                          Node,
   --#                                          STree.Table,
   --#                                          VCGHeap;
   is separate;

   ---------------------------------------------------------------------
   --                 Identifier and Selected Components              --
   ---------------------------------------------------------------------

   procedure ProcessIdentifier (Node      : in STree.SyntaxNode;
                                ThisScope : in Dictionary.Scopes)
   --# global in     AssumeRvalues;
   --#        in     CommandLineData.Content;
   --#        in     DoRtc;
   --#        in     LineNmbr;
   --#        in     LoopStack;
   --#        in     OutputFile;
   --#        in     RedType;
   --#        in     ReferencedVars;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out CheckStack;
   --#        in out ContainsReals;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out ExpnStack;
   --#        in out FlowHeap;
   --#        in out Graph.Table;
   --#        in out KindOfStackedCheck;
   --#        in out LexTokenManager.StringTable;
   --#        in out ShortCircuitStack;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out StmtStack.S;
   --#        in out VCGFailure;
   --#        in out VCGHeap;
   --# derives CheckStack,
   --#         ExpnStack,
   --#         KindOfStackedCheck          from *,
   --#                                          CheckStack,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ExpnStack,
   --#                                          LexTokenManager.StringTable,
   --#                                          LoopStack,
   --#                                          Node,
   --#                                          RedType,
   --#                                          ShortCircuitStack,
   --#                                          STree.Table,
   --#                                          ThisScope,
   --#                                          VCGHeap &
   --#         ContainsReals,
   --#         Dictionary.Dict,
   --#         Graph.Table,
   --#         LexTokenManager.StringTable,
   --#         ShortCircuitStack,
   --#         StmtStack.S,
   --#         VCGFailure,
   --#         VCGHeap                     from *,
   --#                                          CheckStack,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ExpnStack,
   --#                                          Graph.Table,
   --#                                          KindOfStackedCheck,
   --#                                          LexTokenManager.StringTable,
   --#                                          LineNmbr,
   --#                                          LoopStack,
   --#                                          Node,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          ShortCircuitStack,
   --#                                          StmtStack.S,
   --#                                          STree.Table,
   --#                                          ThisScope,
   --#                                          VCGHeap &
   --#         ErrorHandler.ErrorContext   from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ExpnStack,
   --#                                          LexTokenManager.StringTable,
   --#                                          LoopStack,
   --#                                          Node,
   --#                                          RedType,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          STree.Table,
   --#                                          ThisScope,
   --#                                          VCGHeap &
   --#         FlowHeap                    from *,
   --#                                          AssumeRvalues,
   --#                                          Dictionary.Dict,
   --#                                          Node,
   --#                                          RedType,
   --#                                          ReferencedVars,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          ThisScope &
   --#         SPARK_IO.FILE_SYS           from *,
   --#                                          CheckStack,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          ExpnStack,
   --#                                          Graph.Table,
   --#                                          KindOfStackedCheck,
   --#                                          LexTokenManager.StringTable,
   --#                                          LineNmbr,
   --#                                          LoopStack,
   --#                                          Node,
   --#                                          OutputFile,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          ShortCircuitStack,
   --#                                          StmtStack.S,
   --#                                          STree.Table,
   --#                                          ThisScope,
   --#                                          VCGHeap &
   --#         Statistics.TableUsage       from *,
   --#                                          AssumeRvalues,
   --#                                          CheckStack,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ExpnStack,
   --#                                          FlowHeap,
   --#                                          Graph.Table,
   --#                                          KindOfStackedCheck,
   --#                                          LexTokenManager.StringTable,
   --#                                          LineNmbr,
   --#                                          LoopStack,
   --#                                          Node,
   --#                                          RedType,
   --#                                          ReferencedVars,
   --#                                          Scope,
   --#                                          ShortCircuitStack,
   --#                                          StmtStack.S,
   --#                                          STree.Table,
   --#                                          ThisScope,
   --#                                          VCGHeap;
   is
      Sym     : Dictionary.Symbol;
      DAGCell : Cells.Cell;
   begin
      Sym := Dictionary.LookupItem (STree.NodeLexString (Node),
                                    ThisScope,
                                    Dictionary.ProgramContext);

      -- If we call an inherited root function then the above call will fail
      -- to find it and returns a null symbol.  In this case we can check the
      -- syntax tree for the symbol of the root operation that will have been
      -- planted by StackIdentifier.
      if Sym = Dictionary.NullSymbol then
         Sym := STree.NodeSymbol (Node);
      end if;

      Cells.Create_Cell (VCGHeap, DAGCell);
      if Dictionary.IsVariable (Sym) then
         -- each time we find a referenced variable we add it to the set of referenced vars
         AddRvalueSymbol (FlowHeap, ReferencedVars, Sym);

         Cells.Set_Kind (VCGHeap, DAGCell, Cells.Reference);
         Cells.Set_Symbol_Value (VCGHeap, DAGCell, Sym);
         CStacks.Push (VCGHeap, DAGCell, ExpnStack);
      elsif Dictionary.IsFunction (Sym) then
         Cells.Set_Kind (VCGHeap, DAGCell, Cells.Pending_Function);
         Cells.Set_Symbol_Value (VCGHeap, DAGCell, Sym);
         --LexTokenManager.InsertNat (1, LexStr);
         --Cells.Set_Lex_Str (VCGHeap, DAGCell, LexStr);
         CStacks.Push (VCGHeap, DAGCell, ExpnStack);
         SetUpFunctionCall (ThisScope => ThisScope,
                            Prefix    => Dictionary.NullSymbol);
      elsif Dictionary.IsTypeMark (Sym) then

         -- If the identifier denotes a record subtype, then push its
         -- root type for subsequent VCG modelling.
         if Dictionary.TypeIsRecord (Sym) and then
           Dictionary.IsSubtype (Sym) then
            Sym := Dictionary.GetRootType (Sym);
         end if;

         Cells.Set_Kind (VCGHeap, DAGCell, Cells.Fixed_Var);
         Cells.Set_Symbol_Value (VCGHeap, DAGCell, Sym);
         CStacks.Push (VCGHeap, DAGCell, ExpnStack);
      else
         -- also check to see whether we need Rvalue for constants (see AddRValueSymbol for which sort)
         AddRvalueSymbol (FlowHeap, ReferencedVars, Sym);

         Cells.Set_Kind (VCGHeap, DAGCell, Cells.Named_Const);
         Cells.Set_Symbol_Value (VCGHeap, DAGCell, Sym);
         CStacks.Push (VCGHeap, DAGCell, ExpnStack);
      end if;
   end ProcessIdentifier;

   -------------------------------------------------------------------

   procedure ModelRecordComponent (RecordType, Sym : in Dictionary.Symbol)
   --# global in     Dictionary.Dict;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    RecordType,
   --#                                    Sym,
   --#                                    VCGHeap;
   is
      DAGCell  : Cells.Cell;
      ExpnCell : Cells.Cell;
   begin
      CStacks.PopOff (VCGHeap,
                      ExpnStack,
                      ExpnCell);
      -- ExpnCell is a DAG representing an expression which is a record field
      -- Insert one or more "fld_inherit (" before the expression
      ModelInheritedFieldsOfTaggedRecord (Dictionary.GetSimpleName (Sym),
                                          RecordType,
                                          VCGHeap,
                                          ExpnCell);
      -- Then prefix it with fld_? (
      CreateCellKind (DAGCell, VCGHeap, Cells.Field_Access_Function);
      Cells.Set_Symbol_Value (VCGHeap, DAGCell, Sym);
      Cells.Set_Lex_Str (VCGHeap, DAGCell, Dictionary.GetSimpleName (Sym));
      --SetRightArgument (DAGCell, CStacks.Top (VCGHeap, ExpnStack));
      --CStacks.Pop (VCGHeap, ExpnStack);
      SetRightArgument (DAGCell, ExpnCell, VCGHeap);
      CStacks.Push (VCGHeap, DAGCell, ExpnStack);
   end ModelRecordComponent;

   ------------------------------------------------------------------

   procedure ProcessSelectedComponent (Node      : in STree.SyntaxNode;
                                       ThisScope : in Dictionary.Scopes)
   --# global in     AssumeRvalues;
   --#        in     CommandLineData.Content;
   --#        in     DoRtc;
   --#        in     LineNmbr;
   --#        in     LoopStack;
   --#        in     OutputFile;
   --#        in     RedType;
   --#        in     ReferencedVars;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out CheckStack;
   --#        in out ContainsReals;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out ExpnStack;
   --#        in out FlowHeap;
   --#        in out Graph.Table;
   --#        in out KindOfStackedCheck;
   --#        in out LexTokenManager.StringTable;
   --#        in out ShortCircuitStack;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out StmtStack.S;
   --#        in out VCGFailure;
   --#        in out VCGHeap;
   --# derives CheckStack,
   --#         ExpnStack,
   --#         KindOfStackedCheck          from *,
   --#                                          CheckStack,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ExpnStack,
   --#                                          LexTokenManager.StringTable,
   --#                                          LoopStack,
   --#                                          Node,
   --#                                          RedType,
   --#                                          ShortCircuitStack,
   --#                                          STree.Table,
   --#                                          ThisScope,
   --#                                          VCGHeap &
   --#         ContainsReals,
   --#         Dictionary.Dict,
   --#         Graph.Table,
   --#         LexTokenManager.StringTable,
   --#         ShortCircuitStack,
   --#         StmtStack.S,
   --#         VCGFailure,
   --#         VCGHeap                     from *,
   --#                                          CheckStack,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ExpnStack,
   --#                                          Graph.Table,
   --#                                          KindOfStackedCheck,
   --#                                          LexTokenManager.StringTable,
   --#                                          LineNmbr,
   --#                                          LoopStack,
   --#                                          Node,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          ShortCircuitStack,
   --#                                          StmtStack.S,
   --#                                          STree.Table,
   --#                                          ThisScope,
   --#                                          VCGHeap &
   --#         ErrorHandler.ErrorContext   from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ExpnStack,
   --#                                          LexTokenManager.StringTable,
   --#                                          LoopStack,
   --#                                          Node,
   --#                                          RedType,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          STree.Table,
   --#                                          ThisScope,
   --#                                          VCGHeap &
   --#         FlowHeap                    from *,
   --#                                          AssumeRvalues,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ExpnStack,
   --#                                          Node,
   --#                                          RedType,
   --#                                          ReferencedVars,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          ThisScope,
   --#                                          VCGHeap &
   --#         SPARK_IO.FILE_SYS           from *,
   --#                                          CheckStack,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          ExpnStack,
   --#                                          Graph.Table,
   --#                                          KindOfStackedCheck,
   --#                                          LexTokenManager.StringTable,
   --#                                          LineNmbr,
   --#                                          LoopStack,
   --#                                          Node,
   --#                                          OutputFile,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          ShortCircuitStack,
   --#                                          StmtStack.S,
   --#                                          STree.Table,
   --#                                          ThisScope,
   --#                                          VCGHeap &
   --#         Statistics.TableUsage       from *,
   --#                                          AssumeRvalues,
   --#                                          CheckStack,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ExpnStack,
   --#                                          FlowHeap,
   --#                                          Graph.Table,
   --#                                          KindOfStackedCheck,
   --#                                          LexTokenManager.StringTable,
   --#                                          LineNmbr,
   --#                                          LoopStack,
   --#                                          Node,
   --#                                          RedType,
   --#                                          ReferencedVars,
   --#                                          Scope,
   --#                                          ShortCircuitStack,
   --#                                          StmtStack.S,
   --#                                          STree.Table,
   --#                                          ThisScope,
   --#                                          VCGHeap;
   is
      DAGCell   : Cells.Cell;
      Sym       : Dictionary.Symbol;
      IdentNode : STree.SyntaxNode;
      Prefix    : Dictionary.Symbol;

   begin
      DAGCell := CStacks.Top (VCGHeap, ExpnStack);
      IdentNode := STree.Child_Node (
                      STree.Child_Node (
                         STree.Next_Sibling (
                            STree.Child_Node (Node))));

      Prefix := GetTOStype (VCGHeap, ExpnStack);

      Sym := Dictionary.LookupSelectedItem (Prefix,              --GetTOStype,
                                            STree.NodeLexString (IdentNode),
                                            ThisScope,
                                            Dictionary.ProgramContext);

      -- If we call an inherited root function then the above call will fail
      -- to find it and returns a null symbol.  In this case we can check the
      -- syntax tree for the symbol of the root operation that will have been
      -- planted by StackIdentifier.
      if Sym = Dictionary.NullSymbol then
         Sym := STree.NodeSymbol (Node);
      end if;

      if Dictionary.IsRecordComponent (Sym) then
         ModelRecordComponent (Prefix, Sym);
      elsif Dictionary.IsVariable (Sym) then
         -- each time we find a referenced variable we add it to the set of referenced vars
         AddRvalueSymbol (FlowHeap, ReferencedVars, Sym);

         Cells.Set_Kind (VCGHeap, DAGCell, Cells.Reference);
         Cells.Set_Symbol_Value (VCGHeap, DAGCell, Sym);
      elsif Dictionary.IsFunction (Sym) then
         -- Before processing function, find the actual prefix symbol used
         Prefix := Cells.Get_Symbol_Value (VCGHeap, DAGCell);
         -- if Prefix is a protected object then we are handling a fucntion call
         -- of the form PO.F.  In this case we need to pass PO to SetUpFunctionCall
         -- so that it can replace globals of F expressed in terms of the type PT
         -- with the object PO
         if not (Dictionary.IsOwnVariable (Prefix) and then
                   Dictionary.GetOwnVariableProtected (Prefix)) then
            -- Prefix is NOT a protected object so we set it to null so that
            -- SetUpFunctionCall won't do any subtitutions.  If it is a PO
            -- we eleave it alone and it gets passed to SetUpFunctionCall
            Prefix := Dictionary.NullSymbol;
         end if;
         -- now replace top of stack with the function
         Cells.Set_Kind (VCGHeap, DAGCell, Cells.Pending_Function);
         Cells.Set_Symbol_Value (VCGHeap, DAGCell, Sym);
         -- set up the paramater list (and copy in the globals)
         SetUpFunctionCall (ThisScope => ThisScope,
                            Prefix    => Prefix);
      elsif Dictionary.IsTypeMark (Sym) then
         Cells.Set_Kind (VCGHeap, DAGCell, Cells.Fixed_Var);
         Cells.Set_Symbol_Value (VCGHeap, DAGCell, Sym);
      elsif Dictionary.IsPackage (Sym) then
         -- replace package symbolwith the child ready for next lookup
         Cells.Set_Symbol_Value (VCGHeap, DAGCell, Sym);
      else
         -- check to see whether we need Rvalue for constants (see AddRValueSymbol for which sort)
         AddRvalueSymbol (FlowHeap, ReferencedVars, Sym);

         Cells.Set_Kind (VCGHeap, DAGCell, Cells.Named_Const);
         Cells.Set_Symbol_Value (VCGHeap, DAGCell, Sym);
      end if;
   end ProcessSelectedComponent;

   ---------------------------------------------------------------------
   --                            Expressions                          --
   ---------------------------------------------------------------------

   -- procedure to model XOR iaw B manual para 3.1.5
   procedure ModelXorOperator
   --# global in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    ExpnStack,
   --#                                    VCGHeap;
   is
      DAGCell,
      Left,
      Right : Cells.Cell;
   begin
      CStacks.PopOff (VCGHeap, ExpnStack, Right);
      CStacks.PopOff (VCGHeap, ExpnStack, Left);

      CreateOpCell (DAGCell, VCGHeap, SPSymbols.RWor);
      SetRightArgument (DAGCell, Right, VCGHeap);
      SetLeftArgument (DAGCell, Left, VCGHeap);
      CStacks.Push (VCGHeap, DAGCell, ExpnStack);

      CreateOpCell (DAGCell, VCGHeap, SPSymbols.RWand);
      SetRightArgument (DAGCell, Right, VCGHeap);
      SetLeftArgument (DAGCell, Left, VCGHeap);
      CStacks.Push (VCGHeap, DAGCell, ExpnStack);

      PushOperator (Unary, SPSymbols.RWnot, VCGHeap, ExpnStack);
      PushOperator (Binary, SPSymbols.RWand, VCGHeap, ExpnStack);
   end ModelXorOperator;

   -------------------------------------------------------------

   procedure ProcessExpression (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     STree.Table;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      OpNode     : STree.SyntaxNode;
      Operator   : SPSymbols.SPSymbol;
      ResultType : Dictionary.Symbol;

      procedure ModelBitwiseOperation (Operator : in SPSymbols.SPSymbol;
                                       TypeSym  : in Dictionary.Symbol)

      --# global in out ExpnStack;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives ExpnStack,
      --#         Statistics.TableUsage,
      --#         VCGHeap               from *,
      --#                                    ExpnStack,
      --#                                    Operator,
      --#                                    TypeSym,
      --#                                    VCGHeap;
      is
         BoolOpCell : Cells.Cell;
      begin -- ModelBitwiseOperation
         CreateBoolOpCell (BoolOpCell,
                           VCGHeap,
                           TypeSym,
                           Operator);
         -- on the stack are the arguments we want for this new function
         PushOperator (Binary, SPSymbols.comma, VCGHeap, ExpnStack);

         -- tos now has comma cell joining the two arguments
         SetRightArgument (BoolOpCell, CStacks.Top (VCGHeap, ExpnStack), VCGHeap);
         CStacks.Pop (VCGHeap, ExpnStack);
         CStacks.Push (VCGHeap, BoolOpCell, ExpnStack);
         -- modelling function is now on TOS
      end ModelBitwiseOperation;

      ---------------------------------

   begin -- ProcessExpression
      OpNode := STree.Next_Sibling (STree.Child_Node (Node));
      if OpNode /= STree.NullNode then
         Operator := STree.SyntaxNodeType (OpNode);

         -- check to see if result type is an array and
         --     build special model if it is
         ResultType := STree.NodeSymbol (OpNode);
         if Dictionary.IsTypeMark (ResultType) and then
            Dictionary.TypeIsArray (ResultType)
         then -- must be a Boolean array operation
            ModelBitwiseOperation (Operator, ResultType);

         elsif IsModularBitwiseOp (Operator, ResultType) then
            ModelBitwiseOperation (Operator, ResultType);

         else -- procede as before for scalar bool ops

            if Operator = SPSymbols.RWxor then
               ModelXorOperator;
            elsif Operator = SPSymbols.RWandthen or
               Operator = SPSymbols.RWorelse
            then
               -- do nothing for AndThen's and OrElse's here as they have
               -- already been left-associated in ProcessRelation
               null;
            else
               PushOperator (Binary, Operator, VCGHeap, ExpnStack);
            end if;
         end if;
      end if;
   end ProcessExpression;

   -------------------------------------------------------------------

   procedure ModelInClause (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     STree.Table;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      LeftSideOfRange,
      RightSideOfRange,
      TypeMarkCell,
      AttribCell         : Cells.Cell;

      RelOperationLHS,
      RelOperationRHS,
      MiddleOperator    : SPSymbols.SPSymbol;

      InOperatorNode,
      RangeNode         : STree.SyntaxNode;

      type StaticResults is (IsTrue, IsFalse, IsUnknown);
      StaticResult     : StaticResults;

      type MembershipKinds is (Inside, Outside);
      MembershipKind : MembershipKinds;

      procedure CheckIfResultStaticallyKnown
      --# global in     Dictionary.Dict;
      --#        in     InOperatorNode;
      --#        in     STree.Table;
      --#           out StaticResult;
      --# derives StaticResult from Dictionary.Dict,
      --#                           InOperatorNode,
      --#                           STree.Table;
      is
         Sym : Dictionary.Symbol;
      begin
         Sym := STree.NodeSymbol (InOperatorNode);
         if Sym = Dictionary.GetTrue then
            StaticResult := IsTrue;
         elsif Sym = Dictionary.GetFalse then
            StaticResult := IsFalse;
         else
            StaticResult := IsUnknown;
         end if;
      end CheckIfResultStaticallyKnown;

      ----------------------

      procedure ModelStaticallyKnownResult
      --# global in     Dictionary.Dict;
      --#        in     StaticResult;
      --#        in out ExpnStack;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives ExpnStack             from Dictionary.Dict,
      --#                                    StaticResult,
      --#                                    VCGHeap &
      --#         Statistics.TableUsage from *,
      --#                                    Dictionary.Dict,
      --#                                    StaticResult,
      --#                                    VCGHeap &
      --#         VCGHeap               from *,
      --#                                    Dictionary.Dict,
      --#                                    ExpnStack,
      --#                                    StaticResult;
      is
         StaticResultCell : Cells.Cell;

      begin -- ModelStaticallyKnownResult
         CreateCellKind (StaticResultCell, VCGHeap, Cells.Named_Const);
         if StaticResult = IsTrue then
            Cells.Set_Symbol_Value (VCGHeap,
                                  StaticResultCell,
                                  Dictionary.GetTrue);
         else
            Cells.Set_Symbol_Value (VCGHeap,
                                  StaticResultCell,
                                  Dictionary.GetFalse);
         end if;
         CStacks.Push (VCGHeap, StaticResultCell, ExpnStack);
      end ModelStaticallyKnownResult;

      ----------------------

      procedure CompleteInequalityModel
      --# global in     LeftSideOfRange;
      --#        in     MiddleOperator;
      --#        in     RelOperationLHS;
      --#        in     RelOperationRHS;
      --#        in     RightSideOfRange;
      --#        in out ExpnStack;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives ExpnStack,
      --#         Statistics.TableUsage,
      --#         VCGHeap               from *,
      --#                                    ExpnStack,
      --#                                    LeftSideOfRange,
      --#                                    MiddleOperator,
      --#                                    RelOperationLHS,
      --#                                    RelOperationRHS,
      --#                                    RightSideOfRange,
      --#                                    VCGHeap;
      is
         LeftOperand : Cells.Cell;
      begin -- CompleteInequalityModel
         CStacks.PopOff (VCGHeap, ExpnStack, LeftOperand);
         -- restore stack keeping copy of LeftOperand
         CStacks.Push (VCGHeap, LeftOperand, ExpnStack);
         CStacks.Push (VCGHeap, LeftSideOfRange, ExpnStack);
         PushOperator (Binary, RelOperationLHS, VCGHeap, ExpnStack);
         CStacks.Push (VCGHeap, LeftOperand, ExpnStack);
         CStacks.Push (VCGHeap, RightSideOfRange, ExpnStack);
         PushOperator (Binary, RelOperationRHS, VCGHeap, ExpnStack);
         -- form conjunction of the two range constraints;
         PushOperator (Binary, MiddleOperator, VCGHeap, ExpnStack);
      end CompleteInequalityModel;

      ----------------------

      function IsBooleanMembership return Boolean
      --# global in Dictionary.Dict;
      --#        in InOperatorNode;
      --#        in STree.Table;
      is
         Sym : Dictionary.Symbol;
      begin
         Sym := STree.NodeSymbol (InOperatorNode);
         return Dictionary.IsType (Sym) and then
           Dictionary.TypeIsBoolean (Sym);
      end IsBooleanMembership;

      ----------------------

      procedure CompleteBooleanModel
      --# global in     LeftSideOfRange;
      --#        in     MembershipKind;
      --#        in     RightSideOfRange;
      --#        in out ExpnStack;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives ExpnStack,
      --#         Statistics.TableUsage,
      --#         VCGHeap               from *,
      --#                                    ExpnStack,
      --#                                    LeftSideOfRange,
      --#                                    MembershipKind,
      --#                                    RightSideOfRange,
      --#                                    VCGHeap;
      is
         LeftOperand : Cells.Cell;
      begin -- CompleteBooleanModel

         -- model: for X in L .. R create (X and R) or (not X and not L)
         --        negate entire model if operator is 'not in' rather than 'in'

         CStacks.PopOff (VCGHeap, ExpnStack, LeftOperand);

         -- create not L
         CStacks.Push (VCGHeap, LeftSideOfRange, ExpnStack);
         PushOperator (Unary, SPSymbols.RWnot, VCGHeap, ExpnStack);
         -- create not X (using copy of X)
         CStacks.Push (VCGHeap, LeftOperand, ExpnStack);
         PushOperator (Unary, SPSymbols.RWnot, VCGHeap, ExpnStack);
         -- conjoin
         PushOperator (Binary, SPSymbols.RWand, VCGHeap, ExpnStack);

         -- create X and R
         CStacks.Push (VCGHeap, RightSideOfRange, ExpnStack);
         CStacks.Push (VCGHeap, LeftOperand, ExpnStack);
         PushOperator (Binary, SPSymbols.RWand, VCGHeap, ExpnStack);

         -- disjoin above two subexpressions
         PushOperator (Binary, SPSymbols.RWor, VCGHeap, ExpnStack);

         -- finally, if outside rather than inside then invert answer
         if MembershipKind = Outside then
            PushOperator (Unary, SPSymbols.RWnot, VCGHeap, ExpnStack);
         end if;
      end CompleteBooleanModel;

      ----------------------

   begin -- ModelInClause
      InOperatorNode := STree.Next_Sibling (Node);
      if STree.SyntaxNodeType (InOperatorNode) =
         SPSymbols.inside then
         MembershipKind := Inside;
         RelOperationLHS := SPSymbols.greater_or_equal;
         RelOperationRHS := SPSymbols.less_or_equal;
         MiddleOperator  := SPSymbols.RWand;
      else
         MembershipKind := Outside;
         RelOperationLHS := SPSymbols.less_than;
         RelOperationRHS := SPSymbols.greater_than;
         MiddleOperator  := SPSymbols.RWor;
      end if;

      RangeNode := STree.Next_Sibling (InOperatorNode);
      if STree.SyntaxNodeType (RangeNode) =
         SPSymbols.arange then
         -- set is defined by a range, held in stack;
         if STree.SyntaxNodeType (STree.Child_Node (RangeNode)) =
            SPSymbols.attribute then
            -- range is defined by a range attribute on top of stack
            -- this has already been transformed by UpAttribute
            -- which has left Index'First .. Index'Last on stack
            LeftSideOfRange  := LeftPtr (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));
            RightSideOfRange := RightPtr (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));
            CStacks.Pop (VCGHeap, ExpnStack);  -- discard ..
         else
            -- range is defined by a pair of simple expressions;
            CStacks.PopOff (VCGHeap, ExpnStack, RightSideOfRange);
            CStacks.PopOff (VCGHeap, ExpnStack, LeftSideOfRange);
         end if;
         if IsBooleanMembership then
            CompleteBooleanModel;
         else
            CompleteInequalityModel;
         end if;
      else
         -- range is defined by a typemark on top of stack;
         -- form the right operands from this typemark, using FIRST and LAST;
         CheckIfResultStaticallyKnown; -- it will be static if type is non-scalar
         CStacks.PopOff (VCGHeap, ExpnStack, TypeMarkCell);
         if StaticResult = IsUnknown then
            -- not known so build attribute range from typemark
            CreateCellKind (AttribCell, VCGHeap, Cells.Attrib_Value);
            CreateOpCell (LeftSideOfRange, VCGHeap, SPSymbols.apostrophe);
            SetLeftArgument (LeftSideOfRange, TypeMarkCell, VCGHeap);
            SetRightArgument (LeftSideOfRange, AttribCell, VCGHeap);
            Structures.CopyStructure (VCGHeap, LeftSideOfRange, RightSideOfRange);
            Cells.Set_Lex_Str (VCGHeap, RightPtr (VCGHeap, LeftSideOfRange),
                                LexTokenManager.FirstToken);
            Cells.Set_Lex_Str (VCGHeap, RightPtr (VCGHeap, RightSideOfRange),
                                LexTokenManager.LastToken);
            CompleteInequalityModel;
         else -- it is known
            -- get rid of expression from TOS
            CStacks.Pop (VCGHeap, ExpnStack);
            -- put True or False literal cell on stack
            ModelStaticallyKnownResult;
         end if;
      end if;
   end ModelInClause;

   ------------------------------------------------------------------------

   procedure DownProcessRelation (Node : in STree.SyntaxNode)
   --# global in     DoRtc;
   --#        in     ExpnStack;
   --#        in     RedType;
   --#        in     STree.Table;
   --#        in out ShortCircuitStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ShortCircuitStack,
   --#         Statistics.TableUsage from *,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    STree.Table,
   --#                                    VCGHeap &
   --#         VCGHeap               from *,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    ShortCircuitStack,
   --#                                    STree.Table;
   is
      NotLeftHand,
      LeftHand : Cells.Cell;

   begin
      if (RedType.RTC and DoRtc) and then
         STree.SyntaxNodeType (STree.ParentNode (Node))
         = SPSymbols.expression_rep2
      then
         -- take a copy of the top of ExpnStack which is the LHS of
         -- the andthen; and push it on ShortCircuitStack
         Structures.CopyStructure (VCGHeap,
                                   CStacks.Top (VCGHeap, ExpnStack),
                                   LeftHand);
         CStacks.Push (VCGHeap, LeftHand, ShortCircuitStack);
      elsif (RedType.RTC and DoRtc) and then
         STree.SyntaxNodeType (STree.ParentNode (Node))
         = SPSymbols.expression_rep4
      then
         -- take a copy of the top of ExpnStack which is the LHS of
         -- the orelse and negate it; and push it on ShortCircuitStack
         Structures.CopyStructure (VCGHeap,
                                   CStacks.Top (VCGHeap, ExpnStack),
                                   LeftHand);
         CreateOpCell (NotLeftHand, VCGHeap, SPSymbols.RWnot);
         SetRightArgument (NotLeftHand, LeftHand, VCGHeap);

         CStacks.Push (VCGHeap, NotLeftHand, ShortCircuitStack);
      end if;

   end DownProcessRelation;

   procedure ModelAndThen
   --# global in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    ExpnStack,
   --#                                    VCGHeap;
   is
      DAGCell,
      Left,
      Right : Cells.Cell;
   begin
      -- continue to model AndThen as And for the moment

      CStacks.PopOff (VCGHeap, ExpnStack, Right);
      CStacks.PopOff (VCGHeap, ExpnStack, Left);

      CreateOpCell (DAGCell, VCGHeap, SPSymbols.RWand);
      SetRightArgument (DAGCell, Right, VCGHeap);
      SetLeftArgument (DAGCell, Left, VCGHeap);
      CStacks.Push (VCGHeap, DAGCell, ExpnStack);

   end ModelAndThen;

   procedure ModelOrElse
   --# global in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    ExpnStack,
   --#                                    VCGHeap;
   is
      DAGCell,
      Left,
      Right : Cells.Cell;
   begin
      -- continue to model OrElse as Or for the moment

      CStacks.PopOff (VCGHeap, ExpnStack, Right);
      CStacks.PopOff (VCGHeap, ExpnStack, Left);

      CreateOpCell (DAGCell, VCGHeap, SPSymbols.RWor);
      SetRightArgument (DAGCell, Right, VCGHeap);
      SetLeftArgument (DAGCell, Left, VCGHeap);
      CStacks.Push (VCGHeap, DAGCell, ExpnStack);

   end ModelOrElse;


   procedure ProcessRelation (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     DoRtc;
   --#        in     RedType;
   --#        in     STree.Table;
   --#        in out ExpnStack;
   --#        in out ShortCircuitStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         ShortCircuitStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    ShortCircuitStack,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      OpNode   : STree.SyntaxNode;

   begin

      OpNode := STree.Next_Sibling (STree.Child_Node (Node));
      if OpNode /= STree.NullNode then
         if STree.SyntaxNodeType (OpNode) =
            SPSymbols.relational_operator then
            PushOperator (Binary,
                          STree.SyntaxNodeType (STree.Child_Node (OpNode)),
                          VCGHeap,
                          ExpnStack);
         else
            ModelInClause (STree.Child_Node (Node));
         end if;
      end if;

      -- detect any short-circuit forms
      if STree.SyntaxNodeType (STree.ParentNode (Node))
         = SPSymbols.expression_rep2
      then
         if (RedType.RTC and DoRtc) then
            CStacks.Pop (VCGHeap, ShortCircuitStack);
         end if;
         -- left associate and then's
         ModelAndThen;
      elsif STree.SyntaxNodeType (STree.ParentNode (Node))
         = SPSymbols.expression_rep4
      then
         if (RedType.RTC and DoRtc) then
            CStacks.Pop (VCGHeap, ShortCircuitStack);
         end if;
         -- left associate orelse's
         ModelOrElse;
      end if;

   end ProcessRelation;


   -------------------------------------------------------------------

   procedure ProcessSimpleExpression (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     DoRtc;
   --#        in     RedType;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out CheckStack;
   --#        in out ContainsReals;
   --#        in out ExpnStack;
   --#        in out LexTokenManager.StringTable;
   --#        in out ShortCircuitStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives CheckStack,
   --#         ShortCircuitStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap                     from *,
   --#                                          CheckStack,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          ExpnStack,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          ShortCircuitStack,
   --#                                          STree.Table,
   --#                                          VCGHeap &
   --#         ContainsReals               from *,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          Node,
   --#                                          RedType,
   --#                                          STree.Table &
   --#         ExpnStack                   from *,
   --#                                          CheckStack,
   --#                                          Dictionary.Dict,
   --#                                          DoRtc,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          RedType,
   --#                                          ShortCircuitStack,
   --#                                          STree.Table,
   --#                                          VCGHeap &
   --#         LexTokenManager.StringTable from *,
   --#                                          Dictionary.Dict,
   --#                                          ExpnStack,
   --#                                          Node,
   --#                                          STree.Table,
   --#                                          VCGHeap;
   is
      OriginType : SPSymbols.SPSymbol;
      RelOpNode  : STree.SyntaxNode;
      OpNode     : STree.SyntaxNode;
      Op         : SPSymbols.SPSymbol;

      procedure CreateNonZeroConstraint (Expr       : in     Cells.Cell;
                                         Check_Cell :    out Cells.Cell)
      --# global in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives Check_Cell            from VCGHeap &
      --#         Statistics.TableUsage from *,
      --#                                    VCGHeap &
      --#         VCGHeap               from *,
      --#                                    Expr;
      is
         ZeroCell,
         NotEqualsCell : Cells.Cell;
      begin -- CreateNonZeroConstraint
         CreateManifestConstCell (ZeroCell,
                                  VCGHeap,
                                  LexTokenManager.ZeroValue);

         CreateOpCell (NotEqualsCell, VCGHeap, SPSymbols.not_equal);
         SetRightArgument (NotEqualsCell, ZeroCell, VCGHeap);
         SetLeftArgument (NotEqualsCell, Expr, VCGHeap);

         Check_Cell := NotEqualsCell;
      end CreateNonZeroConstraint;

      procedure CheckDivideByZero (RightArg : in Cells.Cell)
      --# global in     DoRtc;
      --#        in     RedType;
      --#        in out CheckStack;
      --#        in out ShortCircuitStack;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives CheckStack,
      --#         ShortCircuitStack,
      --#         Statistics.TableUsage from *,
      --#                                    DoRtc,
      --#                                    RedType,
      --#                                    RightArg,
      --#                                    ShortCircuitStack,
      --#                                    VCGHeap &
      --#         VCGHeap               from *,
      --#                                    CheckStack,
      --#                                    DoRtc,
      --#                                    RedType,
      --#                                    RightArg,
      --#                                    ShortCircuitStack;
      is
         Check_Cell,
         CpRightArg : Cells.Cell;
      begin
         if (RedType.RTC and DoRtc) then
            Structures.CopyStructure (VCGHeap, RightArg, CpRightArg);
            CreateNonZeroConstraint (CpRightArg, Check_Cell);
            PlantCheckStatement (Check_Cell,
                                 RedType,
                                 VCGHeap,
                                 ShortCircuitStack,
                                 CheckStack);

         end if;
      end CheckDivideByZero;

      ----------------------------------------------------

      procedure ModelDivide
      --# global in     Dictionary.Dict;
      --#        in     DoRtc;
      --#        in     OpNode;
      --#        in     RedType;
      --#        in     STree.Table;
      --#        in out CheckStack;
      --#        in out ExpnStack;
      --#        in out ShortCircuitStack;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives CheckStack,
      --#         ShortCircuitStack,
      --#         Statistics.TableUsage from *,
      --#                                    Dictionary.Dict,
      --#                                    DoRtc,
      --#                                    ExpnStack,
      --#                                    OpNode,
      --#                                    RedType,
      --#                                    ShortCircuitStack,
      --#                                    STree.Table,
      --#                                    VCGHeap &
      --#         ExpnStack             from *,
      --#                                    Dictionary.Dict,
      --#                                    OpNode,
      --#                                    STree.Table,
      --#                                    VCGHeap &
      --#         VCGHeap               from *,
      --#                                    CheckStack,
      --#                                    Dictionary.Dict,
      --#                                    DoRtc,
      --#                                    ExpnStack,
      --#                                    OpNode,
      --#                                    RedType,
      --#                                    ShortCircuitStack,
      --#                                    STree.Table;
      is
         OpCell : Cells.Cell;
      begin
         Cells.Create_Cell (VCGHeap, OpCell);
         if Dictionary.TypeIsReal (STree.NodeSymbol (OpNode)) then
            Cells.Set_Kind (VCGHeap, OpCell, Cells.Op);
            Cells.Set_Op_Symbol (VCGHeap, OpCell, SPSymbols.divide);
         else
            Cells.Set_Kind (VCGHeap, OpCell, Cells.FDL_Div_Op);
         end if;

         SetRightArgument (OpCell, CStacks.Top (VCGHeap, ExpnStack), VCGHeap);
         CStacks.Pop (VCGHeap, ExpnStack);
         SetLeftArgument (OpCell, CStacks.Top (VCGHeap, ExpnStack), VCGHeap);
         CStacks.Pop (VCGHeap, ExpnStack);
         CStacks.Push (VCGHeap, OpCell, ExpnStack);

         -- Should check for real types here?
         CheckDivideByZero (RightPtr (VCGHeap, CStacks.Top (VCGHeap, ExpnStack)));
      end ModelDivide;

      ----------------------------------------

      procedure ModelRem
      --# global in     DoRtc;
      --#        in     RedType;
      --#        in out CheckStack;
      --#        in out ExpnStack;
      --#        in out ShortCircuitStack;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives CheckStack,
      --#         ShortCircuitStack,
      --#         Statistics.TableUsage from *,
      --#                                    DoRtc,
      --#                                    ExpnStack,
      --#                                    RedType,
      --#                                    ShortCircuitStack,
      --#                                    VCGHeap &
      --#         ExpnStack             from *,
      --#                                    VCGHeap &
      --#         VCGHeap               from *,
      --#                                    CheckStack,
      --#                                    DoRtc,
      --#                                    ExpnStack,
      --#                                    RedType,
      --#                                    ShortCircuitStack;
      is
         DAGCell,
         Left,
         Right : Cells.Cell;
      begin
         -- modelling of I rem J as I - (I div J)  * J
         -- J is top of stack and I is 2nd TOS
         CStacks.PopOff (VCGHeap, ExpnStack, Right);
         CStacks.PopOff (VCGHeap, ExpnStack, Left);

         CreateCellKind (DAGCell, VCGHeap, Cells.FDL_Div_Op);
         SetRightArgument (DAGCell, Right, VCGHeap);
         SetLeftArgument (DAGCell, Left, VCGHeap);
         CStacks.Push (VCGHeap, DAGCell, ExpnStack);

         CreateOpCell (DAGCell, VCGHeap, SPSymbols.multiply);
         SetRightArgument (DAGCell, Right, VCGHeap);
         SetLeftArgument (DAGCell, CStacks.Top (VCGHeap, ExpnStack), VCGHeap);
         CStacks.Pop (VCGHeap, ExpnStack);
         CStacks.Push (VCGHeap, DAGCell, ExpnStack);

         CreateOpCell (DAGCell, VCGHeap, SPSymbols.minus);
         SetRightArgument (DAGCell, CStacks.Top (VCGHeap, ExpnStack), VCGHeap);
         CStacks.Pop (VCGHeap, ExpnStack);
         SetLeftArgument (DAGCell, Left, VCGHeap);
         CStacks.Push (VCGHeap, DAGCell, ExpnStack);

         CheckDivideByZero (Right);
      end ModelRem;

      ----------------------------------------

      procedure ModelCatenation
      --# global in     Dictionary.Dict;
      --#        in out ExpnStack;
      --#        in out LexTokenManager.StringTable;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives ExpnStack,
      --#         LexTokenManager.StringTable,
      --#         Statistics.TableUsage,
      --#         VCGHeap                     from *,
      --#                                          Dictionary.Dict,
      --#                                          ExpnStack,
      --#                                          LexTokenManager.StringTable,
      --#                                          VCGHeap;
      is
         Left,
         Right        : Cells.Cell;
         LeftString,
         RightString  : EStrings.T;
         LeftLength,
         RightLength  : EStrings.Lengths;
         NewString    : EStrings.Line;
         NewLength    : EStrings.Lengths;
         NewLexStr    : LexTokenManager.LexString;
         OkLeft,
         OkRight      : Boolean;

         procedure GetString (TheCell   : in     Cells.Cell;
                              TheString :    out EStrings.T;
                              ok        :    out Boolean)
         --# global in Dictionary.Dict;
         --#        in LexTokenManager.StringTable;
         --#        in VCGHeap;
         --# derives ok,
         --#         TheString from Dictionary.Dict,
         --#                        LexTokenManager.StringTable,
         --#                        TheCell,
         --#                        VCGHeap;
         is
            CharCode  : Integer;
            Unused    : Maths.ErrorCode;
         begin
            --# accept F, 10, Unused, "Unused here OK" &
            --#        F, 33, Unused, "Unused here OK";
            if Cells.Get_Kind (VCGHeap, TheCell) = Cells.Manifest_Const then
               LexTokenManager.LexStringToString (Cells.Get_Lex_Str (VCGHeap, TheCell),
                                                  TheString);
               ok := True;

            elsif Cells.Get_Kind (VCGHeap, TheCell) = Cells.Named_Const then
               Maths.ValueToInteger
                 (Maths.ValueRep (Dictionary.GetValue (Cells.Get_Symbol_Value (VCGHeap, TheCell))),
                  CharCode,
                  Unused);

               if CharCode = 0 then -- can't model nuls in strings
                  TheString := EStrings.EmptyString;
                  ok := False;

               else
                  TheString := EStrings.T'(
                                  Length  => 3,
                                  Content => EStrings.Contents'(
                                                1      => '"',
                                                2      => Character'Val (CharCode),
                                                3      => '"',
                                                others => ' '));
                  ok := True;
               end if;
            else
               TheString := EStrings.EmptyString; -- should not occur
               ok := False;
            end if;
         end GetString;

         -------------------------------------------------------


      begin --ModelCatenation
            -- get left and right strings to be concatenated
         CStacks.PopOff (VCGHeap, ExpnStack, Right);
         CStacks.PopOff (VCGHeap, ExpnStack, Left);
         if (Cells.Get_Kind (VCGHeap, Left) = Cells.Manifest_Const or
             Cells.Get_Kind (VCGHeap, Left) = Cells.Named_Const) and
            (Cells.Get_Kind (VCGHeap, Right) = Cells.Manifest_Const or
             Cells.Get_Kind (VCGHeap, Right) = Cells.Named_Const)
         then
            GetString (Left, LeftString, OkLeft);
            GetString (Right, RightString, OkRight);
            if OkLeft and OkRight then
               LeftLength  := LeftString.Length;
               RightLength := RightString.Length;

               -- New catenation of Left and Right (minus 2 quotation marks) must
               -- fit within an ExaminerString
               if (LeftLength + RightLength) - 2 <= EStrings.MaxStringLength then

                  -- build an examiner line with concatenated strings in it
                  NewString := EStrings.Line'(1 => '"', others => ' ');

                  for I in EStrings.Lengths range 2 .. LeftLength - 1 loop
                     NewString (I) := LeftString.Content (I);
                  end loop;
                  -- at this point we have " followed by left string and no terminating "
                  for I in EStrings.Lengths range 2 .. RightLength loop
                     NewString ((I + LeftLength) - 2) := RightString.Content (I);
                  end loop;
                  NewLength := (LeftLength + RightLength) - 2;
                  NewString (NewLength) := '"';
                  LexTokenManager.InsertCaseSensitiveLexString (NewString,
                                                                1,
                                                                NewLength,
                                                                  -- to get
                                                                NewLexStr);
                  Cells.Set_Lex_Str (VCGHeap,
                                      Left,
                                      NewLexStr);
                  CStacks.Push (VCGHeap, Left,  ExpnStack);
               else
                  -- Resulting string too long, so just re-push Left, Right and & op
                  CStacks.Push (VCGHeap, Left,  ExpnStack);
                  CStacks.Push (VCGHeap, Right, ExpnStack);
                  PushOperator (Binary, SPSymbols.ampersand, VCGHeap, ExpnStack);
               end if;
            else -- can't model string with nul in it so push & op instead
               CStacks.Push (VCGHeap, Left,  ExpnStack);
               CStacks.Push (VCGHeap, Right, ExpnStack);
               PushOperator (Binary, SPSymbols.ampersand, VCGHeap, ExpnStack);
            end if;

         else -- & operator shouldn't have got here so just push it as binop
            CStacks.Push (VCGHeap, Left,  ExpnStack);
            CStacks.Push (VCGHeap, Right, ExpnStack);
            PushOperator (Binary, SPSymbols.ampersand, VCGHeap, ExpnStack);
         end if;
      end ModelCatenation;

      ----------------------------------------

   begin -- ProcessSimpleExpression
      OpNode := STree.Child_Node (
                   STree.Next_Sibling (
                      STree.Child_Node (Node)));
      if OpNode /= STree.NullNode then
         -- detection of / and REM for special handling
         Op := STree.SyntaxNodeType (OpNode);
         if Op = SPSymbols.divide then
            ModelDivide;
         elsif Op = SPSymbols.RWrem then
            ModelRem;

         elsif Op = SPSymbols.ampersand then
            ModelCatenation;
         else
            PushOperator (Binary, Op, VCGHeap, ExpnStack);
            if Op = SPSymbols.RWmod then
               CheckDivideByZero (RightPtr (VCGHeap, CStacks.Top (VCGHeap, ExpnStack)));
            end if;
         end if;
         ModularizeIfNeeded (STree.NodeSymbol (OpNode), VCGHeap, ExpnStack);

         if RedType.ExpChecks and DoRtc then
            if STree.SyntaxNodeType (Node) =
               SPSymbols.simple_expression
            then
               OriginType :=
                  STree.SyntaxNodeType (
                     STree.ParentNode (-- relation
                        STree.ParentNode (-- expression
                           STree.ParentNode (Node))));
               -- check for relational operator
               RelOpNode := STree.Next_Sibling (
                               STree.Child_Node (
                                  STree.ParentNode (Node)));
            else -- SPSymbols.term
               OriginType :=
                  STree.SyntaxNodeType (
                     STree.ParentNode (-- simple_expression_opt
                        STree.ParentNode (-- simple_expression
                           STree.ParentNode (-- relation
                              STree.ParentNode (-- expression
                                 STree.ParentNode (Node))))));
               -- check for relational operator
               RelOpNode := STree.Next_Sibling (
                               STree.Child_Node (
                                  STree.ParentNode (
                                     STree.ParentNode (
                                        STree.ParentNode (Node)))));
            end if;
            if OriginType /= SPSymbols.assignment_statement or else
               RelOpNode /= STree.NullNode
            then
               -- suppress overflow check for outermost expression on rhs
               -- of assignment
               CheckOverflowRunTimeError (STree.NodeSymbol (OpNode),
                                          CStacks.Top (VCGHeap, ExpnStack),
                                          Scope,
                                          RedType,
                                          VCGHeap,
                                          ShortCircuitStack,
                                          ContainsReals,
                                          CheckStack);

            end if;
         end if;
      end if;
   end ProcessSimpleExpression;

   -------------------------------------------------------------------

   procedure ProcessSimpleExpressionOpt (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     DoRtc;
   --#        in     RedType;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out CheckStack;
   --#        in out ContainsReals;
   --#        in out ExpnStack;
   --#        in out ShortCircuitStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives CheckStack,
   --#         ShortCircuitStack,
   --#         Statistics.TableUsage from *,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    Scope,
   --#                                    ShortCircuitStack,
   --#                                    STree.Table,
   --#                                    VCGHeap &
   --#         ContainsReals         from *,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    Node,
   --#                                    RedType,
   --#                                    STree.Table &
   --#         ExpnStack             from *,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap &
   --#         VCGHeap               from *,
   --#                                    CheckStack,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    Scope,
   --#                                    ShortCircuitStack,
   --#                                    STree.Table;
   is
      OriginType : SPSymbols.SPSymbol;
      RelOpNode : STree.SyntaxNode;
      OpNode    : STree.SyntaxNode;

   begin
      OpNode := STree.Child_Node (Node);
      if STree.SyntaxNodeType (OpNode) =
         SPSymbols.unary_adding_operator then
         PushOperator (Unary,
                       STree.SyntaxNodeType
                         (STree.Child_Node (OpNode)),
                       VCGHeap,
                       ExpnStack);
         if RedType.ExpChecks and DoRtc then
            OriginType :=
               STree.SyntaxNodeType (
                  STree.ParentNode (-- simple_expression
                     STree.ParentNode (-- relation
                        STree.ParentNode (-- expression
                           STree.ParentNode (Node)))));

            -- check for relational operator
            RelOpNode := STree.Next_Sibling (
                            STree.Child_Node (
                               STree.ParentNode (
                                  STree.ParentNode (Node))));
            if OriginType /= SPSymbols.assignment_statement or else
               RelOpNode /= STree.NullNode
            then
               -- suppress overflow check for outermost expression on rhs
               -- of assignment
               CheckOverflowRunTimeError (STree.NodeSymbol (OpNode),
                                          CStacks.Top (VCGHeap, ExpnStack),
                                          Scope,
                                          RedType,
                                          VCGHeap,
                                          ShortCircuitStack,
                                          ContainsReals,
                                          CheckStack);
            end if;
         end if;
      end if;
   end ProcessSimpleExpressionOpt;

   -------------------------------------------------------------------

   procedure ProcessFactor (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     DoRtc;
   --#        in     RedType;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out CheckStack;
   --#        in out ContainsReals;
   --#        in out ExpnStack;
   --#        in out ShortCircuitStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives CheckStack,
   --#         ShortCircuitStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    CheckStack,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    RedType,
   --#                                    Scope,
   --#                                    ShortCircuitStack,
   --#                                    STree.Table,
   --#                                    VCGHeap &
   --#         ContainsReals         from *,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    Node,
   --#                                    RedType,
   --#                                    STree.Table &
   --#         ExpnStack             from *,
   --#                                    CheckStack,
   --#                                    Dictionary.Dict,
   --#                                    DoRtc,
   --#                                    Node,
   --#                                    RedType,
   --#                                    ShortCircuitStack,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      OriginType : SPSymbols.SPSymbol;
      RelOpNode,
      OpNextNode,
      OpNode     : STree.SyntaxNode;
      BoolOpCell : Cells.Cell;
      ResultType : Dictionary.Symbol;

      procedure CreateGeZeroConstraint (Expr       : in     Cells.Cell;
                                        Check_Cell :    out Cells.Cell)
      --# global in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives Check_Cell            from VCGHeap &
      --#         Statistics.TableUsage from *,
      --#                                    VCGHeap &
      --#         VCGHeap               from *,
      --#                                    Expr;
      is
         ZeroCell,
         GeCell : Cells.Cell;
      begin -- CreateGtZeroConstraint
         CreateManifestConstCell (ZeroCell,
                                  VCGHeap,
                                  LexTokenManager.ZeroValue);

         CreateOpCell (GeCell, VCGHeap, SPSymbols.greater_or_equal);
         SetRightArgument (GeCell, ZeroCell, VCGHeap);
         SetLeftArgument (GeCell, Expr, VCGHeap);

         Check_Cell := GeCell;
      end CreateGeZeroConstraint;

      procedure CreateEqZeroConstraint (Expr       : in     Cells.Cell;
                                        Check_Cell :    out Cells.Cell)
      --# global in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives Check_Cell            from VCGHeap &
      --#         Statistics.TableUsage from *,
      --#                                    VCGHeap &
      --#         VCGHeap               from *,
      --#                                    Expr;
      is
         ZeroCell,
         GeCell : Cells.Cell;
      begin -- CreateEqZeroConstraint
         CreateManifestConstCell (ZeroCell,
                                  VCGHeap,
                                  LexTokenManager.ZeroValue);

         CreateOpCell (GeCell, VCGHeap, SPSymbols.equals);
         SetRightArgument (GeCell, ZeroCell, VCGHeap);
         SetLeftArgument (GeCell, Expr, VCGHeap);

         Check_Cell := GeCell;
      end CreateEqZeroConstraint;


      procedure CheckExponentConstraint (LhsTypeSym : in Dictionary.Symbol;
                                         LeftArg    : in Cells.Cell;
                                         RightArg   : in Cells.Cell)
      --# global in     Dictionary.Dict;
      --#        in     DoRtc;
      --#        in     RedType;
      --#        in out CheckStack;
      --#        in out ShortCircuitStack;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives CheckStack,
      --#         ShortCircuitStack,
      --#         Statistics.TableUsage from *,
      --#                                    Dictionary.Dict,
      --#                                    DoRtc,
      --#                                    LeftArg,
      --#                                    LhsTypeSym,
      --#                                    RedType,
      --#                                    RightArg,
      --#                                    ShortCircuitStack,
      --#                                    VCGHeap &
      --#         VCGHeap               from *,
      --#                                    CheckStack,
      --#                                    Dictionary.Dict,
      --#                                    DoRtc,
      --#                                    LeftArg,
      --#                                    LhsTypeSym,
      --#                                    RedType,
      --#                                    RightArg,
      --#                                    ShortCircuitStack;
      is
         Check_Cell,
         EQcell,
         GEcell,
         CpLeftArg,
         CpRightArg : Cells.Cell;
      begin
         if RedType.RTC and DoRtc then
            if IsIntegerType (LhsTypeSym) or IsModularType (LhsTypeSym)  then
               -- LRM95 4.5.6(8) says that RHS for signed integer or modular
               -- types must be subtype Natural - i.e. >= 0, so...
               Structures.CopyStructure (VCGHeap, RightArg, CpRightArg);
               CreateGeZeroConstraint (CpRightArg, Check_Cell);
               PlantCheckStatement (Check_Cell,
                                    RedType,
                                    VCGHeap,
                                    ShortCircuitStack,
                                    CheckStack);

            elsif IsRealType (LhsTypeSym) then
               -- detect 0.0 to negative power case
               -- N.B. This is not guarded by the RealRTC switch because it
               -- effectively a division by zero and we already regard that as
               -- a special case in the RTC, even for reals
               Structures.CopyStructure (VCGHeap, RightArg, CpRightArg);
               CreateGeZeroConstraint (CpRightArg, GEcell);

               Structures.CopyStructure (VCGHeap, LeftArg, CpLeftArg);
               CreateEqZeroConstraint (CpLeftArg, EQcell);

               CreateOpCell (Check_Cell, VCGHeap, SPSymbols.implies);
               SetLeftArgument (Check_Cell, EQcell, VCGHeap);
               SetRightArgument (Check_Cell, GEcell, VCGHeap);
               PlantCheckStatement (Check_Cell,
                                    RedType,
                                    VCGHeap,
                                    ShortCircuitStack,
                                    CheckStack);
            end if;
         end if;
      end CheckExponentConstraint;

      -- Note, there is a similar version of this
      -- subprogram in BuildAnnotationExprDAG
      procedure ModelModularNotOperation
      --# global in     ResultType;
      --#        in out ExpnStack;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives ExpnStack,
      --#         Statistics.TableUsage,
      --#         VCGHeap               from *,
      --#                                    ExpnStack,
      --#                                    ResultType,
      --#                                    VCGHeap;
      is
         MinusOpCell,
         TickCell,
         PrefixCell,
         ModulusCell : Cells.Cell;
      begin
         ----------------------------------------------------
         -- LRM 4.5.6 (5) defines "not X" for a modular    --
         -- type T to be equivalent to T'Last - X.         --
         ----------------------------------------------------

         -- create ' operator
         CreateOpCell (TickCell, VCGHeap, SPSymbols.apostrophe);

         -- create Last attribute name
         CreateAttribValueCell (ModulusCell, VCGHeap, LexTokenManager.LastToken);

         -- Create prefix given by ResultType
         CreateFixedVarCell (PrefixCell, VCGHeap, ResultType);

         -- Assemble T'Last
         SetLeftArgument (TickCell, PrefixCell, VCGHeap);
         SetRightArgument (TickCell, ModulusCell, VCGHeap);

         -- create binary "-" operator
         CreateOpCell (MinusOpCell, VCGHeap, SPSymbols.minus);

         -- Construct T'Last - X, where X is on the top-of-stack
         SetRightArgument (MinusOpCell, CStacks.Top (VCGHeap, ExpnStack), VCGHeap);
         SetLeftArgument  (MinusOpCell, TickCell, VCGHeap);
         CStacks.Pop  (VCGHeap, ExpnStack);
         CStacks.Push (VCGHeap, MinusOpCell, ExpnStack);
      end ModelModularNotOperation;

   begin  -- ProcessFactor
      OpNode := STree.Child_Node (Node);
      if STree.SyntaxNodeType (OpNode) = SPSymbols.RWnot then
         -- check to see if result type is an array and
         -- build special model if it is
         ResultType := STree.NodeSymbol (OpNode);

         if Dictionary.IsTypeMark (ResultType) then

            if Dictionary.TypeIsArray (ResultType) then
               -- must be a Boolean array "not" operation
               CreateBoolOpCell (BoolOpCell,
                                 VCGHeap,
                                 ResultType,
                                 SPSymbols.RWnot);
               SetRightArgument (BoolOpCell, CStacks.Top (VCGHeap, ExpnStack), VCGHeap);
               CStacks.Pop (VCGHeap, ExpnStack);
               CStacks.Push (VCGHeap, BoolOpCell, ExpnStack);
            elsif Dictionary.TypeIsModular (ResultType) then
               -- must be a Modular "not" operation.
               ModelModularNotOperation;
            else -- proceed as before for scalar bool ops
               PushOperator (Unary, SPSymbols.RWnot, VCGHeap, ExpnStack);
            end if;

         else -- proceed as before for scalar bool ops
            PushOperator (Unary, SPSymbols.RWnot, VCGHeap, ExpnStack);
         end if;

         -- handle abs
      elsif STree.SyntaxNodeType (OpNode) = SPSymbols.RWabs then
         PushFunction (Cells.Abs_Function, VCGHeap, ExpnStack);
         if RedType.ExpChecks and DoRtc then
            OriginType :=
               STree.SyntaxNodeType (
                  STree.ParentNode (-- term
                     STree.ParentNode (-- simple_expression_opt
                        STree.ParentNode (-- simple_expression
                           STree.ParentNode (-- relation
                              STree.ParentNode (-- expression
                                 STree.ParentNode (Node)))))));
            -- check for relational operator
            RelOpNode := STree.Next_Sibling (
                            STree.Child_Node (
                               STree.ParentNode (
                                  STree.ParentNode (
                                     STree.ParentNode (
                                        STree.ParentNode (Node))))));
            if OriginType /= SPSymbols.assignment_statement or else
               RelOpNode /= STree.NullNode
            then
               -- suppress overflow check for outermost expression on rhs
               -- of assignment
               CheckOverflowRunTimeError (STree.NodeSymbol (OpNode),
                                          CStacks.Top (VCGHeap, ExpnStack),
                                          Scope,
                                          RedType,
                                          VCGHeap,
                                          ShortCircuitStack,
                                          ContainsReals,
                                          CheckStack);
            end if;
         end if;
      else
         OpNextNode := STree.Next_Sibling (OpNode);
         if OpNextNode /= STree.NullNode then
            PushOperator (Binary, SPSymbols.double_star, VCGHeap, ExpnStack);
            CheckExponentConstraint (STree.NodeSymbol (OpNode),
                                     LeftPtr (VCGHeap, CStacks.Top (VCGHeap, ExpnStack)),
                                     RightPtr (VCGHeap, CStacks.Top (VCGHeap, ExpnStack)));

            ModularizeIfNeeded (STree.NodeSymbol (OpNode), VCGHeap, ExpnStack);
            if RedType.ExpChecks and DoRtc then
               OriginType :=
                  STree.SyntaxNodeType (
                     STree.ParentNode (-- term
                        STree.ParentNode (-- simple_expression_opt
                           STree.ParentNode (-- simple_expression
                              STree.ParentNode (-- relation
                                 STree.ParentNode (-- expression
                                    STree.ParentNode (Node)))))));
               if OriginType /= SPSymbols.assignment_statement then
                  -- suppress overflow check for outermost expression on rhs
                  -- of assignment
                  CheckOverflowRunTimeError (STree.NodeSymbol (OpNode),
                                             CStacks.Top (VCGHeap, ExpnStack),
                                             Scope,
                                             RedType,
                                             VCGHeap,
                                             ShortCircuitStack,
                                             ContainsReals,
                                             CheckStack);
               end if;
            end if;
         end if;
      end if;
   end ProcessFactor;

   -------------------------------------------------------------------

   function ValidStartNode return Boolean
   --# global in StartNode;
   --#        in STree.Table;
   is
      NodeType : SPSymbols.SPSymbol;
   begin
      NodeType := STree.SyntaxNodeType (StartNode);
      return NodeType = SPSymbols.expression or else
        NodeType = SPSymbols.name or else
        NodeType = SPSymbols.selected_component or else
        NodeType = SPSymbols.simple_expression or else
        NodeType = SPSymbols.condition or else
        NodeType = SPSymbols.simple_name or else
        NodeType = SPSymbols.attribute;
   end ValidStartNode;

   -------------------------------------------------------------------

begin -- BuildExpnDAG
   -- Debug.PrintNode ("StartNode=", StartNode);

   SeqAlgebra.CreateSeq (FlowHeap, ReferencedVars);

   SystemErrors.RTAssert (ValidStartNode,
                          SystemErrors.InvalidSyntaxTree,
                          "Invalid start node passed to BuildExpnDAG");
   CStacks.CreateStack (ExpnStack);

   Node := StartNode;
   loop --------------------------------down loop
      LastNode := Node;
      NodeType := STree.SyntaxNodeType (Node);
      case NodeType is
         -- prune at selector nodes so that only left most idents found
         when SPSymbols.selector =>
            Node := STree.NullNode;

         when SPSymbols.numeric_literal =>
            CreateManifestConstCell (DAGCell,
                                     VCGHeap,
                                     STree.NodeLexString (
                                        STree.Child_Node (
                                           STree.Child_Node (Node))));
            CStacks.Push (VCGHeap, DAGCell, ExpnStack);
            Node := STree.NullNode;

         when SPSymbols.character_literal |
            SPSymbols.string_literal        =>
            CreateManifestConstCell (DAGCell,
                                     VCGHeap,
                                     STree.NodeLexString (Node));
            CStacks.Push (VCGHeap, DAGCell, ExpnStack);
            Node := STree.NullNode;

         when SPSymbols.attribute_ident       =>
            DownProcessAttributeIdent (Node);
            Node := STree.NullNode;

         when SPSymbols.identifier =>
            ProcessIdentifier (Node, ExpnScope);
            Node := STree.NullNode;

         when SPSymbols.simple_name =>
            if STree.SyntaxNodeType (STree.ParentNode (LastNode)) =
               SPSymbols.named_argument_association
            then
               -- do not want look at parameter or field identifier
               Node := STree.NullNode;
            else
               Node := STree.Child_Node (Node);
            end if;

         when SPSymbols.aggregate =>
            DownProcessAggregate (SPSymbols.qualified_expression, VCGHeap, Node, ExpnStack);

         when SPSymbols.aggregate_choice_rep   =>
            DownProcessAggregateChoiceRep (LastNode,
                                           ExpnScope,
                                           VCGHeap,
                                           ExpnStack,
                                             -- to get
                                           Node);

         when SPSymbols.record_component_selector_name   =>
            DownProcessRecordComponentSelectorName (LastNode,
                                                    ExpnScope,
                                                    VCGHeap,
                                                    ExpnStack,
                                                      -- to get
                                                    Node);

            -- detect any short-circuit forms
         when SPSymbols.relation                =>
            DownProcessRelation (Node);
            Node := STree.Child_Node (Node);

         when others                          =>
            Node := STree.Child_Node (Node);

      end case;
      if Node = STree.NullNode and LastNode /= StartNode then
         loop ------------------------up loop---------------------
            Node := STree.Next_Sibling (LastNode);
            exit when Node /= STree.NullNode;
            Node := STree.ParentNode (LastNode);
            exit when Node = STree.NullNode;
            NodeType := STree.SyntaxNodeType (Node);
            case NodeType is
               when SPSymbols.selected_component =>
                  ProcessSelectedComponent (Node, ExpnScope);

               when SPSymbols.qualified_expression    =>
                  ModelQualifiedExpression (Node);

               when SPSymbols.aggregate               =>
                  UpProcessAggregate;

               when SPSymbols.extension_aggregate     =>
                  UpProcessExtensionAggregate (VCGHeap, ExpnStack);

               when SPSymbols.ancestor_part =>
                  ProcessAncestorPart (Node, VCGHeap, ExpnStack);

               when SPSymbols.component_association   =>
                  UpProcessComponentAssociation (Node);

               when SPSymbols.named_association_rep       =>
                  UpProcessNamedAssociationRep (Node);

               when SPSymbols.named_record_component_association       =>
                  UpProcessNamedRecordComponentAssociation (Node);

               when SPSymbols.aggregate_choice_rep    =>
                  UpProcessAggregateChoiceRep (Node, VCGHeap, ExpnStack);

               when SPSymbols.aggregate_or_expression  =>
                  UpProcessAggregateOrExpression (Node);

               when SPSymbols.positional_record_component_association =>
                  UpProcessPositionalRecordComponentAssociation (Node);

               when SPSymbols.aggregate_choice        =>
                  UpProcessAggregateChoice (Node);

               when SPSymbols.expression            |
                  SPSymbols.expression_rep1       |
                  SPSymbols.expression_rep2       |
                  SPSymbols.expression_rep3       |
                  SPSymbols.expression_rep4       |
                  SPSymbols.expression_rep5         =>
                  ProcessExpression (Node);

               when SPSymbols.relation                =>
                  ProcessRelation (Node);

               when SPSymbols.simple_expression     |
                  SPSymbols.term                    =>
                  ProcessSimpleExpression (Node);

               when SPSymbols.simple_expression_opt   =>
                  ProcessSimpleExpressionOpt (Node);

               when SPSymbols.factor                  =>
                  ProcessFactor (Node);

               when SPSymbols.positional_argument_association =>
                  ProcessPositionalArgumentAssociation (Node);

               when SPSymbols.named_argument_association  =>
                  ProcessNamedArgumentAssociation (Node);

               when SPSymbols.name_argument_list      =>
                  ProcessNameArgumentList;

               when SPSymbols.attribute_designator    =>
                  UpAttributeDesignator (Node);

               when others                          =>
                  null;
            end case;
            exit when Node = StartNode;
            LastNode := Node;
         end loop; -----------------------------up loop--------------
      end if;
      exit when Node = STree.NullNode or Node = StartNode;
   end loop;  --------------------------down loop------------------

   --# accept F, 10, ExpnStack, "Ineffective assignment here OK";
   CStacks.PopOff (VCGHeap, ExpnStack, DAGRoot);
   --# end accept;

   -- at this point assume variables in set ReferencedVars are in their type
   CheckPlantRvalueAssumptions;

   if CommandLineData.Content.Debug.DAG then
      Debug_Print_DAG (StartNode, Scope, DAGRoot, VCGHeap);
   end if;
end BuildExpnDAG;
