-- $Id: dag-buildannotationexpndag.adb 12696 2009-03-12 13:14:05Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with
  EStrings,
  LexTokenManager,
  Maths,
  SPSymbols,
  Structures;

separate (DAG)
procedure BuildAnnotationExpnDAG (StartNode     : in     STree.SyntaxNode;
                                  Scope         : in     Dictionary.Scopes;
                                  ForceAbstract : in     Boolean;
                                  LoopStack     : in     LoopContext.T;
                                  VCGHeap       : in out Cells.Heap_Record;
                                  DAGRoot       :    out Cells.Cell)

   -- This procedure traverses a syntax tree of an annotation expression
is

   DAGCell   : Cells.Cell;
   ExpnStack : CStacks.Stack;
   NodeType  : SPSymbols.SPSymbol;
   LastNode  : STree.SyntaxNode;
   Node      : STree.SyntaxNode;
   LScope    : Dictionary.Scopes;

   -------------------------------------------------------------------

   procedure SetUpFunctionCall
   --# global in     Dictionary.Dict;
   --#        in     ForceAbstract;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    ForceAbstract,
   --#                                    VCGHeap;
   is
      NumberOfParameters : Natural;
      FunctionSym    : Dictionary.Symbol;

      ----------------------------

   begin -- SetUpFunctionCall
      FunctionSym := Cells.Get_Symbol_Value (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));
      -- regardless of which implicit proof function we have here make it the abstract one
      if ForceAbstract and then Dictionary.IsImplicitProofFunction (FunctionSym) then
         Cells.Set_Symbol_Value (VCGHeap,
                               CStacks.Top (VCGHeap, ExpnStack),
                               Dictionary.GetImplicitProofFunction
                               (Dictionary.IsAbstract,
                                Dictionary.GetAdaFunction (FunctionSym)));
         FunctionSym := Cells.Get_Symbol_Value (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));
      end if;
      NumberOfParameters := Dictionary.GetNumberOfSubprogramParameters (FunctionSym);
      if NumberOfParameters = 0 then
         Cells.Set_Kind (VCGHeap, CStacks.Top (VCGHeap, ExpnStack),
                           Cells.Proof_Function);
      end if;
      CreateEmptyList (NumberOfParameters, VCGHeap, ExpnStack);
   end SetUpFunctionCall;

   -----------------------------------------------------------------------

   procedure SetUpArrayAccess
   --# global in     Dictionary.Dict;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    VCGHeap;
   is
      NumberOfDimensions : Positive;
      DAGCell            : Cells.Cell;

   begin
      NumberOfDimensions := Dictionary.GetNumberOfDimensions (GetTOStype (VCGHeap, ExpnStack));
      CreateCellKind (DAGCell, VCGHeap, Cells.List_Function);
      CStacks.Push (VCGHeap, DAGCell, ExpnStack);
      CreateEmptyList (NumberOfDimensions, VCGHeap, ExpnStack);
   end SetUpArrayAccess;

   -----------------------------------------------------------------------

   procedure ProcessPositionalArgumentAssociation
   --# global in     Dictionary.Dict;
   --#        in     Node;
   --#        in     STree.Table;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      ExpressionCell : Cells.Cell;
      TOSkind        : Cells.Cell_Kind;
      ConversionTargetType,
      ConversionSourceType : Dictionary.Symbol;

   begin
      CStacks.PopOff (VCGHeap, ExpnStack, ExpressionCell);
      TOSkind := Cells.Get_Kind (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));
      if TOSkind = Cells.Pending_Function then
         -- We may need to convert the actual parameter by inserting some inherit
         -- derefences in front of it; conversion is required if we have called
         -- an inherited root function.  The parameter in this case must be an
         -- object.
         ConvertTaggedActualIfNecessary (Cells.Get_Symbol_Value (VCGHeap, CStacks.Top (VCGHeap, ExpnStack)),
                                         VCGHeap,
                                         ExpressionCell); -- function sym

         InsertParameterInNextFreeSlot (CStacks.Top (VCGHeap, ExpnStack), ExpressionCell, VCGHeap);
      elsif TOSkind = Cells.List_Function then
         InsertParameterInNextFreeSlot (CStacks.Top (VCGHeap, ExpnStack), ExpressionCell, VCGHeap);
      elsif TOSkind = Cells.Fixed_Var then
         ConversionSourceType := STree.NodeSymbol (Node);
         ConversionTargetType := Cells.Get_Symbol_Value (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));
         -- assume integer numeric conversion for now
         CStacks.Pop (VCGHeap, ExpnStack);                  -- get rid of type mark
         CStacks.Push (VCGHeap, ExpressionCell, ExpnStack); -- restore expression
                                                         -- insert trunc function if needed
         if Dictionary.TypeIsReal (ConversionSourceType) and then
            (Dictionary.TypeIsInteger (ConversionTargetType) or else IsModularType (ConversionTargetType))
         then
            PushFunction (Cells.Trunc_Function, VCGHeap, ExpnStack);
         end if;
      else --must be dealing with first indexed expression of array access
         SetUpArrayAccess;
         InsertParameterInNextFreeSlot (CStacks.Top (VCGHeap, ExpnStack), ExpressionCell, VCGHeap);
      end if;
   end ProcessPositionalArgumentAssociation;

   -----------------------------------------------------------------------

   procedure ProcessNamedArgumentAssociation (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     STree.Table;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack             from *,
   --#                                    VCGHeap &
   --#         Statistics.TableUsage from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    VCGHeap &
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    STree.Table;
   is
      InsertPoint,
      ExpressionCell : Cells.Cell;
      FunctionSym    : Dictionary.Symbol;
      ParamPos       : Positive;
      LastOne        : Boolean;

      ----------------------------------------------------------------

      function FindIdentifier (Node : STree.SyntaxNode)
                              return STree.SyntaxNode
      --# global in STree.Table;
      is
         IdentNode : STree.SyntaxNode;

      begin
         if STree.SyntaxNodeType (STree.Child_Node (Node)) =
               SPSymbols.annotation_simple_name
         then
            IdentNode := STree.Child_Node (STree.Child_Node (Node));
         else
            IdentNode := STree.Child_Node
              (STree.Next_Sibling
               (STree.Child_Node (Node)));
         end if;
         return IdentNode;
      end FindIdentifier;

      --------------------------------------------------------------

      procedure GetParamNumber (Name    : in     LexTokenManager.LexString;
                                FunSym  : in     Dictionary.Symbol;
                                ParamNo :    out Positive)
      --# global in Dictionary.Dict;
      --# derives ParamNo from Dictionary.Dict,
      --#                      FunSym,
      --#                      Name;
      -- This procedure has an implicit precondition, that the subprogram
      -- does have parameters and that the name passed identifies one of them
      -- this will be True because when VCs are generated, we know that the code
      -- is well-formed. Therefore the flow error can be ignored.
      is
         It    : Dictionary.Iterator;
         Sym   : Dictionary.Symbol;

         function ParameterNumber (FunSym, ParamSym : Dictionary.Symbol) return Positive
         --# global in Dictionary.Dict;
         is
            Current : Dictionary.Iterator;
            Number  : Positive;
         begin
            Current := Dictionary.FirstSubprogramParameter (FunSym);
            Number := 1;
            loop
               exit when Dictionary.CurrentSymbol (Current) = ParamSym;
               Current := Dictionary.NextSymbol (Current);
               Number := Number + 1;
            end loop;
            return Number;
         end ParameterNumber;

      begin  -- GetParamNumber
         It := Dictionary.FirstSubprogramParameter (FunSym);
         SystemErrors.RTAssert
           (not Dictionary.IsNullIterator (It),
            SystemErrors.PreconditionFailure,
            "Can't find first subprogram parameter in BuildAnnotationExpnDAG.GetParamNumber");
         loop
            Sym := Dictionary.CurrentSymbol (It);
            exit when Dictionary.GetSimpleName (Sym) = Name;
            It := Dictionary.NextSymbol (It);
            exit when Dictionary.IsNullIterator (It);
         end loop;
         ParamNo := ParameterNumber (FunSym, Sym);
      end GetParamNumber;

      --------------------------------------------------------------

   begin -- ProcessNamedArgumentAssociation
         -- we must be dealing with a function call
      CStacks.PopOff (VCGHeap, ExpnStack, ExpressionCell);
      FunctionSym := Cells.Get_Symbol_Value (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));
      -- We may need to convert the actual parameter by inserting some inherit
      -- derefences in front of it; conversion is required if we have called
      -- an inherited root function.  The parameter in this case must be an
      -- object.
      ConvertTaggedActualIfNecessary (FunctionSym, VCGHeap, ExpressionCell);

      GetParamNumber (STree.NodeLexString (FindIdentifier (Node)),
                      FunctionSym,
                        -- to get
                      ParamPos);
      CalculateInsertPoint (VCGHeap,
                            ExpnStack,
                            ParamPos,
                              -- to get
                            InsertPoint,
                            LastOne);
      if LastOne then
         SetRightArgument (InsertPoint, ExpressionCell, VCGHeap);
      else
         SetLeftArgument (InsertPoint, ExpressionCell, VCGHeap);
      end if;
   end ProcessNamedArgumentAssociation;

   -----------------------------------------------------------------------

   procedure ProcessNameArgumentList
   --# global in     Dictionary.Dict;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage from *,
   --#                                    ExpnStack,
   --#                                    VCGHeap &
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack;
   is
      TOSkind  : Cells.Cell_Kind;
      Temp     : Cells.Cell;
      TypeSym  : Dictionary.Symbol;

   begin
      TOSkind := Cells.Get_Kind (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));
      if TOSkind = Cells.Pending_Function then
         Cells.Set_Kind (VCGHeap, CStacks.Top (VCGHeap, ExpnStack),
                           Cells.Proof_Function);
      elsif TOSkind = Cells.List_Function then
         -- complete element model and store type so far in case of further
         -- indexing (to handle array of arrays or array of records case)
         CStacks.PopOff (VCGHeap, ExpnStack, Temp);
         TypeSym := Dictionary.GetArrayComponent (GetTOStype (VCGHeap, ExpnStack));
         CStacks.Push (VCGHeap, Temp, ExpnStack);
         PushOperator (Binary, SPSymbols.comma, VCGHeap, ExpnStack);
         PushFunction (Cells.Element_Function, VCGHeap, ExpnStack);
         Cells.Set_Symbol_Value (VCGHeap, CStacks.Top (VCGHeap, ExpnStack), TypeSym);
      end if;
   end ProcessNameArgumentList;

   -----------------------------------------------------------------------
   --            Handling Update Syntax in Annotations
   -----------------------------------------------------------------------

   procedure DownProcessStore
   --# global in     Dictionary.Dict;
   --#        in     LScope;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    LScope,
   --#                                    VCGHeap;
   is
      TypeSym            : Dictionary.Symbol;
   begin
      TypeSym := GetTOStype (VCGHeap, ExpnStack);
      -- Handle array and record updates differently, arrays need stuff for store-lists
      if Dictionary.IsArrayTypeMark (TypeSym, LScope) then
         SetUpArrayAccess;
         -- this leaves us with update subject on 2nd TOS and empty list on TOS
      end if;
      -- no action required for record
   end DownProcessStore;

   ------------------------------------------------------------------------

   procedure DownProcessStoreList (Node     : in     STree.SyntaxNode;
                                   NextNode :    out STree.SyntaxNode)
   --# global in ExpnStack;
   --#        in STree.Table;
   --#        in VCGHeap;
   --# derives NextNode from ExpnStack,
   --#                       Node,
   --#                       STree.Table,
   --#                       VCGHeap;
   is
   begin
      if Cells.Get_Kind (VCGHeap, CStacks.Top (VCGHeap, ExpnStack)) =
         Cells.List_Function
      then -- we are doing an array
         NextNode := STree.Child_Node (Node);
      else  --must be record so prune walk here
         NextNode := STree.NullNode;
      end if;
   end DownProcessStoreList;

   -----------------------------------------------------------------------

   procedure UpProcessStoreList
   --# global in out ExpnStack;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         VCGHeap   from ExpnStack,
   --#                        VCGHeap;
   is
      ExpressionCell : Cells.Cell;

   begin
      --will only be called if array being processed, earlier pruning
      --will stop us getting here for records

      CStacks.PopOff (VCGHeap, ExpnStack, ExpressionCell);
      InsertParameterInNextFreeSlot (CStacks.Top (VCGHeap, ExpnStack), ExpressionCell, VCGHeap);
   end UpProcessStoreList;

   -----------------------------------------------------------------------

   procedure UpProcessStore (Node : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LScope;
   --#        in     STree.Table;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    CommandLineData.Content,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    LScope,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      Temp,
      UpfCell,
      OriginalObjectBeingUpdated,
      LocalCopyOfObjectBeingUpdated,
      ObjectBeingUpdated,
      CommaCell  : Cells.Cell;
      TypeSym,
      FieldSym,
      FieldSymForInheritDerefLoop : Dictionary.Symbol;
      FieldName   : LexTokenManager.LexString;

   begin
      -- for an array update we have exp, list, updated_obj on stack
      -- for a record we have exp, updated_obj on stack

      CStacks.PopOff (VCGHeap, ExpnStack, Temp); --this is assigned expression

      if Cells.Get_Kind (VCGHeap, CStacks.Top (VCGHeap, ExpnStack)) =
         Cells.List_Function
      then -- we are doing an array
         CStacks.Push (VCGHeap, Temp, ExpnStack);
         PushOperator (Binary, SPSymbols.comma, VCGHeap, ExpnStack);
         -- now obtain type of whole composite object and store in update
         -- function cell so that updates of updates will work
         CStacks.PopOff (VCGHeap, ExpnStack, Temp); -- remove to get access to object
         TypeSym := GetTOStype (VCGHeap, ExpnStack);
         CStacks.Push (VCGHeap, Temp, ExpnStack);
         PushOperator (Binary, SPSymbols.comma, VCGHeap, ExpnStack);
         PushFunction (Cells.Update_Function, VCGHeap, ExpnStack);
         Cells.Set_Symbol_Value (VCGHeap, CStacks.Top (VCGHeap, ExpnStack), TypeSym);

      else -- we are doing a record -----------------------------------------------------

         -- Get the root type here in case the updated object is of a record
         -- subtype.
         TypeSym := Dictionary.GetRootType (GetTOStype (VCGHeap, ExpnStack));

         CStacks.PopOff (VCGHeap, ExpnStack, ObjectBeingUpdated);
         OriginalObjectBeingUpdated := ObjectBeingUpdated; -- because ObjectBeingUpdated changes later and we need a copy
         FieldName := STree.NodeLexString (STree.LastChildOf (Node));
         FieldSym := Dictionary.LookupSelectedItem (TypeSym,
                                                    FieldName,
                                                    LScope,
                                                    Dictionary.ProofContext);

         -- The field we are updating may be inherited from an earlier tagged types.
         -- So insert as many fld_inherit()s in front as needed
         ModelInheritedFieldsOfTaggedRecord (FieldName, TypeSym, VCGHeap, ObjectBeingUpdated);
         -- assemble upf_field (ObjectBeingUpdated, OriginalExpression)
         CStacks.Push (VCGHeap, ObjectBeingUpdated, ExpnStack);
         CStacks.Push (VCGHeap, Temp, ExpnStack);
         PushOperator (Binary, SPSymbols.comma, VCGHeap, ExpnStack);
         CreateUpfCell (UpfCell, VCGHeap, FieldSym, Dictionary.GetSimpleName (FieldSym));
         SetRightArgument (UpfCell, CStacks.Top (VCGHeap, ExpnStack), VCGHeap);
         CStacks.Pop (VCGHeap, ExpnStack);
         CStacks.Push (VCGHeap, UpfCell, ExpnStack);

         -- TOS now has an upf_field function that represents the most direct update of the field
         -- for example O[F=>exp] with no inheritance gives: upf_f (o, exp);
         -- if F is inherited one level we get:              upf_f (fld_inherit (o), exp)
         -- and two levels gives:                            upf_f (fld_inherit (fld_inherit (o), exp))
         --
         -- We now need to prefix this expression with some upf_ functions:
         -- First case required no prefix.
         -- Second case wants: "upf_inherit (o, "
         -- Third wants:       "upf_inherit (o, upf_inherit (fld_inherit (o), " etc.

         -- The number of prefixes required depends on ther inheritance depth at this point.
         -- Inner loop puts on the fld_inherits needed.  Loop not entered in no inheritance.
         -- After the inner loop we put on the upf_inherit function needed.

         -- We loop backwards so we can use I to tell us how many inherit derefs we need in an
         -- embedded loop.
         for I in reverse Integer range 1 .. Dictionary.GetInheritDepth (FieldName, TypeSym) loop
            -- Make copy of ObjectBeingUpdated because cell it is in gets changed each time we add inherit de-refs
            LocalCopyOfObjectBeingUpdated := OriginalObjectBeingUpdated;
            CreateOpCell (CommaCell, VCGHeap, SPSymbols.comma);
            SetRightArgument (CommaCell, CStacks.Top (VCGHeap, ExpnStack), VCGHeap);
            -- Insert n-1 inherit dereferences in front of LocalCopyOfObjectBeingUpdated
            FieldSymForInheritDerefLoop := TypeSym;
            for J in Integer range 1 .. (I - 1) loop
               FieldSymForInheritDerefLoop := Dictionary.GetType
                 (Dictionary.CurrentSymbol
                    (Dictionary.FirstRecordComponent
                       (FieldSymForInheritDerefLoop)));
               -- LocalCopyOfObjectBeingUpdated gets changed by following call
               InsertInheritDeReference (FieldSymForInheritDerefLoop, VCGHeap, LocalCopyOfObjectBeingUpdated);

            end loop;
            SetLeftArgument (CommaCell, LocalCopyOfObjectBeingUpdated, VCGHeap);

            -- Now put upf_inherit on front
            CreateUpfCell (UpfCell,
                           VCGHeap,
                           Dictionary.LookupSelectedItem (FieldSymForInheritDerefLoop,
                                                          LexTokenManager.InheritToken,
                                                          LScope,
                                                          Dictionary.ProofContext),
                           LexTokenManager.InheritToken);
            SetRightArgument (UpfCell, CommaCell, VCGHeap);
            CStacks.Pop (VCGHeap, ExpnStack);            -- old expression
            CStacks.Push (VCGHeap, UpfCell, ExpnStack);  -- expression with one level of prefix
         end loop;
      end if;
   end UpProcessStore;

   -----------------------------------------------------------------------

   -- only do this if down, right node is expression.
   procedure ModelQualifiedExpression (Node : in STree.SyntaxNode)
   --# global in     STree.Table;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      ExpnCell : Cells.Cell;
   begin
      if STree.SyntaxNodeType
        (STree.Next_Sibling
         (STree.Child_Node (Node))) =
        SPSymbols.annotation_expression then
         -- discard type indication and return its argument to top of stack;
         CStacks.PopOff (VCGHeap, ExpnStack, ExpnCell);
         -- the topmost stack cell contains the typemark;
         CStacks.Pop (VCGHeap, ExpnStack);
         CStacks.Push (VCGHeap, ExpnCell, ExpnStack);
      end if;
   end ModelQualifiedExpression;

   ----------------------------------------------------------------------

   procedure UpProcessAggregateChoice (Node : in STree.SyntaxNode)
   --# global in     STree.Table;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      RangeNodeType   : SPSymbols.SPSymbol;
      RangeExpression : Cells.Cell;

   begin
      RangeNodeType := STree.SyntaxNodeType (
                          STree.Next_Sibling (
                             STree.Child_Node (Node)));
      if RangeNodeType = SPSymbols.annotation_simple_expression then
         PushOperator (Binary, SPSymbols.double_dot, VCGHeap, ExpnStack);
      elsif RangeNodeType = SPSymbols.annotation_range_constraint then
         TransformRangeConstraint (VCGHeap, ExpnStack);
         CStacks.PopOff (VCGHeap, ExpnStack, RangeExpression);
         CStacks.Pop (VCGHeap, ExpnStack); -- discard type mark part of range
         CStacks.Push (VCGHeap, RangeExpression, ExpnStack);
      elsif Cells.Get_Kind (VCGHeap, CStacks.Top (VCGHeap, ExpnStack)) = Cells.Fixed_Var then
         -- type mark found
         TransformTypeName (VCGHeap, ExpnStack);
      end if;
   end UpProcessAggregateChoice;

   ---------------------------------------------------------------------

   procedure UpProcessNamedAssociationRep (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     STree.Table;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      AggExp : Cells.Cell;
   begin
      PushOperator (Binary, SPSymbols.becomes, VCGHeap, ExpnStack);
      if DoingArrayAggregate (VCGHeap, ExpnStack)  then
         if STree.SyntaxNodeType (STree.Child_Node (Node)) =
           SPSymbols.annotation_named_association_rep then
            PushOperator (Binary, SPSymbols.comma, VCGHeap, ExpnStack);
         end if;
      else -- record
         CStacks.PopOff (VCGHeap, ExpnStack, AggExp);
         InsertAssociation (CStacks.Top (VCGHeap, ExpnStack), AggExp, VCGHeap);
      end if;
   end UpProcessNamedAssociationRep;

   ---------------------------------------------------------------------

   procedure UpProcessNamedRecordComponentAssociation
   --# global in     Dictionary.Dict;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage from *,
   --#                                    ExpnStack,
   --#                                    VCGHeap &
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack;
   is
      AggExp : Cells.Cell;
   begin
      -- Node is named_record_component_association
      -- Direction is UP
      -- TOS is expression to be associated
      -- 2nd TOS is field name
      -- 3rd TOS is incomplete aggregate being constructed.

      -- associated field name with expression
      PushOperator (Binary, SPSymbols.becomes, VCGHeap, ExpnStack);
      CStacks.PopOff (VCGHeap, ExpnStack, AggExp);
      InsertAssociation (CStacks.Top (VCGHeap, ExpnStack), AggExp, VCGHeap);
   end UpProcessNamedRecordComponentAssociation;

   ---------------------------------------------------------------------

   procedure UpProcessPositionalRecordComponentAssociation
   --# global in     Dictionary.Dict;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    VCGHeap;
   is
      AggExp,
      TypeCell : Cells.Cell;
   begin
      -- Node is positional_record_component_association
      -- Direction is UP
      -- TOS is expression to be associated
      -- 2nd TOS is incomplete aggregate being constructed.
      -- 3rd TOS is agggregate counter giving current field number
      CreateFixedVarCell (TypeCell,
                          VCGHeap,
                          Dictionary.GetRecordComponent (AggregateType (VCGHeap, ExpnStack),
                                                         CurrentFieldOrIndex (VCGHeap, ExpnStack)));
      CStacks.Push (VCGHeap, TypeCell, ExpnStack);
      SwitchAndPush (SPSymbols.becomes, VCGHeap, ExpnStack);
      IncCurrentFieldOrIndex (ExpnStack, VCGHeap);
      CStacks.PopOff (VCGHeap, ExpnStack, AggExp);
      InsertAssociation (CStacks.Top (VCGHeap, ExpnStack), AggExp, VCGHeap);
   end UpProcessPositionalRecordComponentAssociation;

   ---------------------------------------------------------------------

   procedure UpProcessAggregateOrExpression (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     STree.Table;
   --#        in out ExpnStack;
   --#        in out LexTokenManager.StringTable;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         LexTokenManager.StringTable,
   --#         Statistics.TableUsage,
   --#         VCGHeap                     from *,
   --#                                          Dictionary.Dict,
   --#                                          ExpnStack,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          STree.Table,
   --#                                          VCGHeap;
   is
      IndexType     : Dictionary.Symbol;
      CounterCell,
      AttribCell,
      TypeCell      : Cells.Cell;
      CounterString : LexTokenManager.LexString;
      AggExp : Cells.Cell;

   begin -- UpProcessAggregateOrExpression
      if STree.SyntaxNodeType (STree.ParentNode (Node)) =
         SPSymbols.annotation_positional_association_rep or else
         STree.Next_Sibling (Node) /=
         STree.NullNode
      then
         if DoingArrayAggregate (VCGHeap, ExpnStack)  then
            CreateCellKind (TypeCell, VCGHeap, Cells.Fixed_Var);
            IndexType := Dictionary.GetArrayIndex (AggregateType (VCGHeap, ExpnStack), 1);
            Cells.Set_Symbol_Value (VCGHeap, TypeCell, IndexType);
            CStacks.Push (VCGHeap, TypeCell, ExpnStack);

            CreateAttribValueCell (AttribCell, VCGHeap, LexTokenManager.FirstToken);
            CStacks.Push (VCGHeap, AttribCell, ExpnStack);
            PushOperator (Binary, SPSymbols.apostrophe, VCGHeap, ExpnStack);

            if Dictionary.TypeIsEnumeration (IndexType) then
               for I in Integer range 2 .. CurrentFieldOrIndex (VCGHeap, ExpnStack) loop
                  --# accept F, 41, "Stable expression here OK";
                  if Dictionary.TypeIsBoolean (IndexType) then
                     PushOperator (Unary, SPSymbols.RWnot, VCGHeap, ExpnStack);
                  else
                     PushFunction (Cells.Succ_Function, VCGHeap, ExpnStack);
                  end if;
                  --# end accept;
               end loop;
            else -- index type is numeric discrete
               if CurrentFieldOrIndex (VCGHeap, ExpnStack) > 1 then
                  LexTokenManager.InsertNat (CurrentFieldOrIndex (VCGHeap, ExpnStack) - 1, CounterString);
                  CreateManifestConstCell (CounterCell, VCGHeap, CounterString);
                  CStacks.Push (VCGHeap, CounterCell, ExpnStack);
                  PushOperator (Binary, SPSymbols.plus, VCGHeap, ExpnStack);
               end if;
            end if;
            PushFunction (Cells.List_Function, VCGHeap, ExpnStack);

         else -- record aggregate
            CreateFixedVarCell (TypeCell,
                                VCGHeap,
                                Dictionary.GetRecordComponent (AggregateType (VCGHeap, ExpnStack),
                                                               CurrentFieldOrIndex (VCGHeap, ExpnStack)));
            CStacks.Push (VCGHeap, TypeCell, ExpnStack);
         end if;

         SwitchAndPush (SPSymbols.becomes, VCGHeap, ExpnStack);
         IncCurrentFieldOrIndex (ExpnStack, VCGHeap);
         if DoingArrayAggregate (VCGHeap, ExpnStack)  then
            if STree.Next_Sibling (Node) =
              STree.NullNode
            then
               PushOperator (Binary, SPSymbols.comma, VCGHeap, ExpnStack);
            end if;
         else -- record
            CStacks.PopOff (VCGHeap, ExpnStack, AggExp);
            InsertAssociation (CStacks.Top (VCGHeap, ExpnStack), AggExp, VCGHeap);
         end if;
      end if;
   end UpProcessAggregateOrExpression;

   ---------------------------------------------------------------------

   procedure UpProcessComponentAssociation (Node : in STree.SyntaxNode)
   --# global in     STree.Table;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
   begin
      if STree.Next_Sibling
        (STree.Child_Node
         (STree.Child_Node (Node))) /= STree.NullNode then
         SwitchAndPush (SPSymbols.comma, VCGHeap, ExpnStack);
      end if;
   end UpProcessComponentAssociation;

   ---------------------------------------------------------------------

   procedure UpProcessAggregate
   --# global in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    ExpnStack,
   --#                                    VCGHeap;
   is
      TempAgg : Cells.Cell;
   begin
      -- Tidy up expression stack

      -- At this point the stack is rather confused (even for an ex-FORTH programmer).
      -- If we are doing a record then TOS is the IncompleteAggregate function and its arguments,
      --                           2nd TOS is the aggregate counter used for positional association.
      --
      -- If we are doing an array then TOS is the comma-delimited list of arguments to the MkAggregate func,
      --                           2nd TOS is the IncompleteAggregate function itself,
      --                           3rd TOS is the aggregate counter
      --
      CStacks.PopOff (VCGHeap, ExpnStack, TempAgg);  -- hold the aggregate expression or list
      if Cells.Get_Kind (VCGHeap, CStacks.Top (VCGHeap, ExpnStack)) =
         Cells.Aggregate_Counter
      then -- we are doing a record and just need to get rid of the counter
         CStacks.Pop (VCGHeap, ExpnStack); -- get rid of counter
      else -- we are doing an array and TOS is the IncompleteArray function which needs to be connected to
         -- the comma-delimited list
         SetRightArgument (CStacks.Top (VCGHeap, ExpnStack), TempAgg, VCGHeap);
         CStacks.PopOff (VCGHeap, ExpnStack, TempAgg);  -- hold the now complete aggregate expression
         -- and then get rid of the exposed counter
         CStacks.Pop (VCGHeap, ExpnStack);
      end if;
      -- Convert aggregate to a finished MkAggregate function
      Cells.Set_Kind (VCGHeap, TempAgg, Cells.Mk_Aggregate);
      -- Finally, restore aggregate DAG to TOS
      CStacks.Push (VCGHeap, TempAgg, ExpnStack);
   end UpProcessAggregate;

   ---------------------------------------------------------------------
   --                       Attribute Processing                      --
   ---------------------------------------------------------------------

   procedure DownProcessAttributeIdent (Node : in STree.SyntaxNode)
   --# global in     STree.Table;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      DAGCell : Cells.Cell;
   begin
      CreateAttribValueCell (DAGCell, VCGHeap, STree.NodeLexString (Node));
      CStacks.Push (VCGHeap, DAGCell, ExpnStack);
      PushOperator (Binary, SPSymbols.apostrophe, VCGHeap, ExpnStack);
   end DownProcessAttributeIdent;

   ---------------------------------------------------------------------

   procedure UpProcessAttributeDesignator (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     STree.Table;
   --#        in out ExpnStack;
   --#        in out LexTokenManager.StringTable;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         LexTokenManager.StringTable,
   --#         Statistics.TableUsage,
   --#         VCGHeap                     from *,
   --#                                          Dictionary.Dict,
   --#                                          ExpnStack,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          STree.Table,
   --#                                          VCGHeap;
      is separate;

   ---------------------------------------------------------------------
   --                 Identifier and Selected Components              --
   ---------------------------------------------------------------------

   procedure ReplaceWithOnEntryVariable (DAGCell : in Cells.Cell;
                                         VarSym  : in Dictionary.Symbol)
   --# global in     Dictionary.Dict;
   --#        in     LoopStack;
   --#        in out VCGHeap;
   --# derives VCGHeap from *,
   --#                      DAGCell,
   --#                      Dictionary.Dict,
   --#                      LoopStack,
   --#                      VarSym;
   is
      OnEntryVar  : Dictionary.Symbol;
      CurrentLoop : Dictionary.Symbol;
   begin
      -- For a variable which appears in a for loop invariant in the form X%, replace X with the
      -- variable set up in BuildGraph as X_on_entry_to_the_loop.

      -- The variable we are seeking may appear in the exit condition of an enclosing for loop so we
      -- need to loop through any enclosing loops
      CurrentLoop := LoopContext.CurrentLoopSym (LoopStack, VCGHeap);
      loop
         OnEntryVar := Dictionary.GetLoopOnEntryVariable (VarSym,
                                                          CurrentLoop);
         -- success exit condition, sought variable is used in loop exit conditon
         exit when OnEntryVar /= Dictionary.NullSymbol;

         -- If we have a null symbol then the variable isn't used in the exit condition of the current loop
         -- so we need to get the enclosing loop and try again
         CurrentLoop := LoopContext.EnclosingLoopSym (LoopStack,
                                                      VCGHeap,
                                                      CurrentLoop);
         -- failure case, we have run out of loops without finding soughtvariable
         if CurrentLoop = Dictionary.NullSymbol then
            OnEntryVar := Dictionary.NullSymbol;
            exit;
         end if;
      end loop;

      -- If % is used on a variable that doesn't appear in any enclosing for loop exit condition then
      -- OnEntryVar will be still be a null symbol here.  Ideally we should prevent use of percent in this
      -- situation but the wffs for that would be very hard to write.  As a second best we simply
      -- don't make the substitution in this case.  In effect we say that X% = X is X doesn't appear
      -- in the for loop exit condition.
      if OnEntryVar /= Dictionary.NullSymbol then
         Cells.Set_Symbol_Value (VCGHeap,
                               DAGCell,
                               OnEntryVar);
      end if;
   end ReplaceWithOnEntryVariable;

   ---------------------------------------------------------------------

   procedure ProcessIdentifier (Node   : in STree.SyntaxNode;
                                LScope : in Dictionary.Scopes)
   --# global in     Dictionary.Dict;
   --#        in     ForceAbstract;
   --#        in     LoopStack;
   --#        in     STree.Table;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    ForceAbstract,
   --#                                    LoopStack,
   --#                                    LScope,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      Sym              : Dictionary.Symbol;
      EnclosingPackage : Dictionary.Symbol;
      DAGCell          : Cells.Cell;

      function GetEnclosingPackage (Scope : Dictionary.Scopes) return Dictionary.Symbol
      --# global in Dictionary.Dict;
      is
         Result         : Dictionary.Symbol;
         EnclosingScope : Dictionary.Scopes;
      begin
         EnclosingScope := Scope;
         loop
            EnclosingScope := Dictionary.GetEnclosingScope (EnclosingScope);

            Result := Dictionary.GetRegion (EnclosingScope);
            exit when Dictionary.IsPackage (Result);

            -- fail-safe exit if we hit "standard"
            if Dictionary.IsPredefinedScope (EnclosingScope) then
               Result := Dictionary.GetRegion (EnclosingScope);
               exit;
            end if;

         end loop;
         return Result;
      end GetEnclosingPackage;

   begin
      Sym := Dictionary.LookupItem (STree.NodeLexString (Node),
                                    LScope,
                                    Dictionary.ProofContext);

      -- if we are doing an abstract pre/post and we fail to find what we are
      -- expecting at the first attempt we need to re-search in the visible
      -- scope of the package where our subprogram is declared; this is to
      -- pick up abstract own variables that have been refined away
      if Sym = Dictionary.NullSymbol and then ForceAbstract then

         EnclosingPackage := GetEnclosingPackage (LScope);
         if Dictionary.IsPackage (EnclosingPackage) then
            Sym := Dictionary.LookupItem (STree.NodeLexString (Node),
                                          Dictionary.VisibleScope (EnclosingPackage),
                                          Dictionary.ProofContext);
         end if;
      end if;

      -- If we call an inherited root function then the above call will fail
      -- to find it and returns a null symbol.  In this case we can check the
      -- syntax tree for the symbol of the root operation that will have been
      -- planted by StackIdentifier.
      if Sym = Dictionary.NullSymbol then
         Sym := STree.NodeSymbol (Node);
      end if;

      Cells.Create_Cell (VCGHeap, DAGCell);
      if Dictionary.IsVariable (Sym) then
         Cells.Set_Kind (VCGHeap, DAGCell, Cells.Reference);
         Cells.Set_Symbol_Value (VCGHeap, DAGCell, Sym);
         if STree.IdentifierHasTildeSuffix (Node) then
            SetTilde (DAGCell, VCGHeap);
         elsif STree.IdentifierHasPercentSuffix (Node) then
            ReplaceWithOnEntryVariable (DAGCell, Sym);
         end if;
         CStacks.Push (VCGHeap, DAGCell, ExpnStack);
      elsif Dictionary.IsFunction (Sym) then
         Cells.Set_Kind (VCGHeap, DAGCell, Cells.Pending_Function);
         Cells.Set_Symbol_Value (VCGHeap, DAGCell, Sym);
         CStacks.Push (VCGHeap, DAGCell, ExpnStack);
         SetUpFunctionCall;
      elsif Dictionary.IsTypeMark (Sym) then

         -- If the identifier denotes a record subtype, then push its
         -- root type for subsequent VCG modelling.
         if Dictionary.TypeIsRecord (Sym) and then
           Dictionary.IsSubtype (Sym) then
            Sym := Dictionary.GetRootType (Sym);
         end if;

         Cells.Set_Kind (VCGHeap, DAGCell, Cells.Fixed_Var);
         Cells.Set_Symbol_Value (VCGHeap, DAGCell, Sym);
         CStacks.Push (VCGHeap, DAGCell, ExpnStack);
      else
         Cells.Set_Kind (VCGHeap, DAGCell, Cells.Named_Const);
         Cells.Set_Symbol_Value (VCGHeap, DAGCell, Sym);
         CStacks.Push (VCGHeap, DAGCell, ExpnStack);
      end if;
   end ProcessIdentifier;

   -------------------------------------------------------------------

   procedure ModelRecordComponent (RecordType, Sym : in Dictionary.Symbol)
   --# global in     Dictionary.Dict;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    RecordType,
   --#                                    Sym,
   --#                                    VCGHeap;
   is
      DAGCell : Cells.Cell;
      ExpnCell : Cells.Cell;
   begin
      CStacks.PopOff (VCGHeap,
                      ExpnStack,
                      ExpnCell);
      -- ExpnCell is a DAG representing an expression which is a record field
      -- Insert one or more "fld_inherit (" before the expression
      ModelInheritedFieldsOfTaggedRecord (Dictionary.GetSimpleName (Sym), RecordType, VCGHeap, ExpnCell);
      -- Then prefix it with fld_? (
      CreateCellKind (DAGCell, VCGHeap, Cells.Field_Access_Function);
      Cells.Set_Symbol_Value (VCGHeap, DAGCell, Sym);
      Cells.Set_Lex_Str (VCGHeap, DAGCell, Dictionary.GetSimpleName (Sym));
      --SetRightArgument (DAGCell, CStacks.Top (VCGHeap, ExpnStack));
      --CStacks.Pop (VCGHeap, ExpnStack);
      SetRightArgument (DAGCell, ExpnCell, VCGHeap);
      CStacks.Push (VCGHeap, DAGCell, ExpnStack);
   end ModelRecordComponent;

   ------------------------------------------------------------------

   procedure ProcessSelectedComponent (Node   : in STree.SyntaxNode;
                                       LScope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     ForceAbstract;
   --#        in     LoopStack;
   --#        in     STree.Table;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage from *,
   --#                                    CommandLineData.Content,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    ForceAbstract,
   --#                                    LScope,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap &
   --#         VCGHeap               from *,
   --#                                    CommandLineData.Content,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    ForceAbstract,
   --#                                    LoopStack,
   --#                                    LScope,
   --#                                    Node,
   --#                                    STree.Table;
   is
      DAGCell   : Cells.Cell;
      Sym       : Dictionary.Symbol;
      IdentNode : STree.SyntaxNode;
      Prefix    : Dictionary.Symbol;

   begin
      DAGCell := CStacks.Top (VCGHeap, ExpnStack);
      IdentNode := STree.Child_Node (
                      STree.Child_Node (
                         STree.Next_Sibling (
                           STree.Child_Node (Node))));
      Prefix := GetTOStype (VCGHeap, ExpnStack);
      Sym := Dictionary.LookupSelectedItem (Prefix, --GetTOStype,
                                            STree.NodeLexString (IdentNode),
                                            LScope,
                                            Dictionary.ProofContext);

      -- If we call an inherited root function then the above call will fail
      -- to find it and returns a null symbol.  In this case we can check the
      -- syntax tree for the symbol of the root operation that will have been
      -- planted by StackIdentifier.
      if Sym = Dictionary.NullSymbol then
         Sym := STree.NodeSymbol (Node);
      end if;

      if Dictionary.IsRecordComponent (Sym) then
         ModelRecordComponent (Prefix, Sym);
      elsif Dictionary.IsVariable (Sym) then
         Cells.Set_Kind (VCGHeap, DAGCell, Cells.Reference);
         Cells.Set_Symbol_Value (VCGHeap, DAGCell, Sym);
         if STree.IdentifierHasTildeSuffix (IdentNode) then
            SetTilde (DAGCell, VCGHeap);
         elsif STree.IdentifierHasPercentSuffix (IdentNode) then
            ReplaceWithOnEntryVariable (DAGCell, Sym);
         end if;
      elsif Dictionary.IsFunction (Sym) then
         Cells.Set_Kind (VCGHeap, DAGCell, Cells.Pending_Function);
         Cells.Set_Symbol_Value (VCGHeap, DAGCell, Sym);
         SetUpFunctionCall;
      elsif Dictionary.IsTypeMark (Sym) then
         Cells.Set_Kind (VCGHeap, DAGCell, Cells.Fixed_Var);
         Cells.Set_Symbol_Value (VCGHeap, DAGCell, Sym);
      else
         Cells.Set_Kind (VCGHeap, DAGCell, Cells.Named_Const);
         Cells.Set_Symbol_Value (VCGHeap, DAGCell, Sym);
      end if;
   end ProcessSelectedComponent;

   ---------------------------------------------------------------------
   --                            Expressions                          --
   ---------------------------------------------------------------------

   -- model XOR iaw B manual para 3.1.5
   procedure ModelXorOperator
   --# global in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    ExpnStack,
   --#                                    VCGHeap;
   is
      DAGCell,
      Left,
      Right,
      CopyOfLeft,
      CopyOfRight : Cells.Cell;
   begin
      -- Obtain operands and make copies of them so that we can construct a model
      -- that does not make multiple links to the Left and Right cells.  This
      -- change arises from CFR 1154 and affects only annotation expressions since
      -- it is to avoid problems when substituting for tilded globals in postconditions.
      CStacks.PopOff (VCGHeap, ExpnStack, Right);
      Structures.CopyStructure (VCGHeap, Right, CopyOfRight);
      CStacks.PopOff (VCGHeap, ExpnStack, Left);
      Structures.CopyStructure (VCGHeap, Left, CopyOfLeft);

      -- model OR part using original arguments
      CreateOpCell (DAGCell, VCGHeap, SPSymbols.RWor);
      SetRightArgument (DAGCell, Right, VCGHeap);
      SetLeftArgument (DAGCell, Left, VCGHeap);
      CStacks.Push (VCGHeap, DAGCell, ExpnStack);

      -- model AND part using copies
      CreateOpCell (DAGCell, VCGHeap, SPSymbols.RWand);
      SetRightArgument (DAGCell, CopyOfRight, VCGHeap);
      SetLeftArgument (DAGCell, CopyOfLeft, VCGHeap);
      CStacks.Push (VCGHeap, DAGCell, ExpnStack);
      -- negate AND parrt
      PushOperator (Unary, SPSymbols.RWnot, VCGHeap, ExpnStack);

      -- complete model by conjoining the OR and NOT AND parts
      PushOperator (Binary, SPSymbols.RWand, VCGHeap, ExpnStack);
   end ModelXorOperator;

   -------------------------------------------------------------

   procedure ProcessExpression (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     STree.Table;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      OpNode   : STree.SyntaxNode;
      Operator : SPSymbols.SPSymbol;
      ResultType : Dictionary.Symbol;

      procedure ModelBitwiseOperation (Operator : in SPSymbols.SPSymbol;
                                       TypeSym  : in Dictionary.Symbol)

      --# global in out ExpnStack;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives ExpnStack,
      --#         Statistics.TableUsage,
      --#         VCGHeap               from *,
      --#                                    ExpnStack,
      --#                                    Operator,
      --#                                    TypeSym,
      --#                                    VCGHeap;
      is
         BoolOpCell : Cells.Cell;
      begin -- ModelBitwiseOperation
         CreateBoolOpCell (BoolOpCell, VCGHeap, TypeSym, Operator);
         -- on the stack are the arguments we want for this new function
         PushOperator (Binary, SPSymbols.comma, VCGHeap, ExpnStack);

         -- tos now has comma cell joining the two arguments
         SetRightArgument (BoolOpCell, CStacks.Top (VCGHeap, ExpnStack), VCGHeap);
         CStacks.Pop (VCGHeap, ExpnStack);
         CStacks.Push (VCGHeap, BoolOpCell, ExpnStack);
         -- modelling function is now on TOS
      end ModelBitwiseOperation;

      ---------------------------------

   begin -- ProcessExpression
      OpNode := STree.Next_Sibling (STree.Child_Node (Node));
      if OpNode /= STree.NullNode then
         Operator := STree.SyntaxNodeType (OpNode);

         -- check to see if result type is an array and
         --     build special model if it is
         ResultType := STree.NodeSymbol (OpNode);
         if Dictionary.IsTypeMark (ResultType) and then
            Dictionary.TypeIsArray (ResultType)
         then -- must be a Boolean array operation
            ModelBitwiseOperation (Operator, ResultType);

         elsif IsModularBitwiseOp (Operator, ResultType) then
            ModelBitwiseOperation (Operator, ResultType);

         else -- proceed as before for scalar bool ops
            if Operator = SPSymbols.RWxor then
               ModelXorOperator;
            else
               PushOperator (Binary, Operator, VCGHeap, ExpnStack);
            end if;
         end if;
      end if;
   end ProcessExpression;

   -------------------------------------------------------------------

   procedure ModelInClause (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     STree.Table;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      LeftSideOfRange,
      RightSideOfRange,
      TypeMarkCell,
      AttribCell         : Cells.Cell;

      RelOperationLHS,
      RelOperationRHS,
      MiddleOperator    : SPSymbols.SPSymbol;

      InOperatorNode,
      RangeNode         : STree.SyntaxNode;

      type StaticResults is (IsTrue, IsFalse, IsUnknown);
      StaticResult     : StaticResults;

      type MembershipKinds is (Inside, Outside);
      MembershipKind : MembershipKinds;

      procedure CheckIfResultStaticallyKnown
      --# global in     Dictionary.Dict;
      --#        in     InOperatorNode;
      --#        in     STree.Table;
      --#           out StaticResult;
      --# derives StaticResult from Dictionary.Dict,
      --#                           InOperatorNode,
      --#                           STree.Table;
      is
         Sym : Dictionary.Symbol;
      begin
         Sym := STree.NodeSymbol (InOperatorNode);
         if Sym = Dictionary.GetTrue then
            StaticResult := IsTrue;
         elsif Sym = Dictionary.GetFalse then
            StaticResult := IsFalse;
         else
            StaticResult := IsUnknown;
         end if;
      end CheckIfResultStaticallyKnown;

      ----------------------

      procedure ModelStaticallyKnownResult
      --# global in     Dictionary.Dict;
      --#        in     StaticResult;
      --#        in out ExpnStack;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives ExpnStack             from Dictionary.Dict,
      --#                                    StaticResult,
      --#                                    VCGHeap &
      --#         Statistics.TableUsage from *,
      --#                                    Dictionary.Dict,
      --#                                    StaticResult,
      --#                                    VCGHeap &
      --#         VCGHeap               from *,
      --#                                    Dictionary.Dict,
      --#                                    ExpnStack,
      --#                                    StaticResult;
      is
         StaticResultCell : Cells.Cell;

      begin -- ModelStaticallyKnownResult
         CreateCellKind (StaticResultCell, VCGHeap, Cells.Named_Const);
         if StaticResult = IsTrue then
            Cells.Set_Symbol_Value (VCGHeap,
                                  StaticResultCell,
                                  Dictionary.GetTrue);
         else
            Cells.Set_Symbol_Value (VCGHeap,
                                  StaticResultCell,
                                  Dictionary.GetFalse);
         end if;
         CStacks.Push (VCGHeap, StaticResultCell, ExpnStack);
      end ModelStaticallyKnownResult;

      ----------------------

      procedure CompleteInequalityModel
      --# global in     LeftSideOfRange;
      --#        in     MiddleOperator;
      --#        in     RelOperationLHS;
      --#        in     RelOperationRHS;
      --#        in     RightSideOfRange;
      --#        in out ExpnStack;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives ExpnStack,
      --#         Statistics.TableUsage,
      --#         VCGHeap               from *,
      --#                                    ExpnStack,
      --#                                    LeftSideOfRange,
      --#                                    MiddleOperator,
      --#                                    RelOperationLHS,
      --#                                    RelOperationRHS,
      --#                                    RightSideOfRange,
      --#                                    VCGHeap;
      is
         LeftOperand,
         CopyOfLeftOperand : Cells.Cell;
      begin -- CompleteInequalityModel
         CStacks.PopOff (VCGHeap, ExpnStack, LeftOperand);
         -- Make deep copy of left operand so that we can construct the model
         -- without making multiple links to LeftOperand cell.  This change
         -- arises from CFR 1154 and is only needed in annotation expressions
         -- because of potential problems using tilded globals in postconditions
         -- that use IN operators
         Structures.CopyStructure (VCGHeap, LeftOperand, CopyOfLeftOperand);

         -- restore stack, model first inequality
         CStacks.Push (VCGHeap, LeftOperand, ExpnStack);
         CStacks.Push (VCGHeap, LeftSideOfRange, ExpnStack);
         PushOperator (Binary, RelOperationLHS, VCGHeap, ExpnStack);

         -- model second inequality using copy of LHS
         CStacks.Push (VCGHeap, CopyOfLeftOperand, ExpnStack);
         CStacks.Push (VCGHeap, RightSideOfRange, ExpnStack);
         PushOperator (Binary, RelOperationRHS, VCGHeap, ExpnStack);

         -- form conjunction of the two range constraints;
         PushOperator (Binary, MiddleOperator, VCGHeap, ExpnStack);
      end CompleteInequalityModel;

      ----------------------

      function IsBooleanMembership return Boolean
      --# global in Dictionary.Dict;
      --#        in InOperatorNode;
      --#        in STree.Table;
      is
         Sym : Dictionary.Symbol;
      begin
         Sym := STree.NodeSymbol (InOperatorNode);
         return Dictionary.IsType (Sym) and then
           Dictionary.TypeIsBoolean (Sym);
      end IsBooleanMembership;

      ----------------------

      procedure CompleteBooleanModel
      --# global in     LeftSideOfRange;
      --#        in     MembershipKind;
      --#        in     RightSideOfRange;
      --#        in out ExpnStack;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives ExpnStack,
      --#         Statistics.TableUsage,
      --#         VCGHeap               from *,
      --#                                    ExpnStack,
      --#                                    LeftSideOfRange,
      --#                                    MembershipKind,
      --#                                    RightSideOfRange,
      --#                                    VCGHeap;
      is
         LeftOperand,
         CopyOfLeftOperand : Cells.Cell;
      begin -- CompleteBooleanModel

         -- model: for X in L .. R create (X and R) or (not X and not L)
         --        negate entire model if operator is 'not in' rather than 'in'

         CStacks.PopOff (VCGHeap, ExpnStack, LeftOperand);
         -- Make deep copy of left operand so that we can construct the model
         -- without making multiple links to LeftOperand cell.  This change
         -- arises from CFR 1154 and is only needed in annotation expressions
         -- because of potential problems using tilded globals in postconditions
         -- that use IN operators
         Structures.CopyStructure (VCGHeap, LeftOperand, CopyOfLeftOperand);

         -- create not L
         CStacks.Push (VCGHeap, LeftSideOfRange, ExpnStack);
         PushOperator (Unary, SPSymbols.RWnot, VCGHeap, ExpnStack);
         -- create not X (using copy of X)
         CStacks.Push (VCGHeap, CopyOfLeftOperand, ExpnStack);
         PushOperator (Unary, SPSymbols.RWnot, VCGHeap, ExpnStack);
         -- conjoin
         PushOperator (Binary, SPSymbols.RWand, VCGHeap, ExpnStack);

         -- create X and R
         CStacks.Push (VCGHeap, RightSideOfRange, ExpnStack);
         CStacks.Push (VCGHeap, LeftOperand, ExpnStack);
         PushOperator (Binary, SPSymbols.RWand, VCGHeap, ExpnStack);

         -- disjoin above two subexpressions
         PushOperator (Binary, SPSymbols.RWor, VCGHeap, ExpnStack);

         -- finally, if outside rather than inside then invert answer
         if MembershipKind = Outside then
            PushOperator (Unary, SPSymbols.RWnot, VCGHeap, ExpnStack);
         end if;
      end CompleteBooleanModel;

      ----------------------

   begin --ModelInClause
      InOperatorNode := STree.Next_Sibling (Node);
      if STree.SyntaxNodeType (InOperatorNode) =
        SPSymbols.inside then
         MembershipKind := Inside;
         RelOperationLHS := SPSymbols.greater_or_equal;
         RelOperationRHS := SPSymbols.less_or_equal;
         MiddleOperator  := SPSymbols.RWand;
      else
         MembershipKind := Outside;
         RelOperationLHS := SPSymbols.less_than;
         RelOperationRHS := SPSymbols.greater_than;
         MiddleOperator  := SPSymbols.RWor;
      end if;

      RangeNode := STree.Next_Sibling (InOperatorNode);
      if STree.SyntaxNodeType (RangeNode) =
         SPSymbols.annotation_arange then
         -- set is defined by a range, held in stack;
         if STree.SyntaxNodeType (STree.Child_Node (RangeNode)) =
            SPSymbols.annotation_attribute then
            -- range is defined by a range attribute on top of stack
            -- this has already been transformed by UpProcessAttribute
            -- which has left Index'First .. Index'Last on stack
            LeftSideOfRange  := LeftPtr (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));
            RightSideOfRange := RightPtr (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));
            CStacks.Pop (VCGHeap, ExpnStack);  --discard ..
         else
            -- range is defined by a pair of simple expressions;
            CStacks.PopOff (VCGHeap, ExpnStack, RightSideOfRange);
            CStacks.PopOff (VCGHeap, ExpnStack, LeftSideOfRange);
         end if;
         if IsBooleanMembership then
            CompleteBooleanModel;
         else
            CompleteInequalityModel;
         end if;
      else
         -- range is defined by a typemark on top of stack;
         -- form the right operands from this typemark, using FIRST and LAST;
         CheckIfResultStaticallyKnown; -- it will be static if type is non-scalar
         CStacks.PopOff (VCGHeap, ExpnStack, TypeMarkCell);
         if StaticResult = IsUnknown then
            CreateCellKind (AttribCell, VCGHeap, Cells.Attrib_Value);
            CreateOpCell (LeftSideOfRange, VCGHeap, SPSymbols.apostrophe);
            SetLeftArgument (LeftSideOfRange, TypeMarkCell, VCGHeap);
            SetRightArgument (LeftSideOfRange, AttribCell, VCGHeap);
            Structures.CopyStructure (VCGHeap, LeftSideOfRange, RightSideOfRange);
            Cells.Set_Lex_Str (VCGHeap, RightPtr (VCGHeap, LeftSideOfRange), LexTokenManager.FirstToken);
            Cells.Set_Lex_Str (VCGHeap, RightPtr (VCGHeap, RightSideOfRange), LexTokenManager.LastToken);
            CompleteInequalityModel;
         else
            CStacks.Pop (VCGHeap, ExpnStack);
            ModelStaticallyKnownResult;
         end if;
      end if;
   end ModelInClause;

   ------------------------------------------------------------------------

   procedure ProcessRelation (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     STree.Table;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      OpNode   : STree.SyntaxNode;

   begin
      OpNode := STree.Next_Sibling (STree.Child_Node (Node));
      if OpNode /= STree.NullNode then
         if STree.SyntaxNodeType (OpNode) =
               SPSymbols.relational_operator then
            PushOperator (Binary,
                          STree.SyntaxNodeType (STree.Child_Node (OpNode)),
                          VCGHeap,
                          ExpnStack);
         else
            ModelInClause (STree.Child_Node (Node));
         end if;
      end if;
   end ProcessRelation;

   -------------------------------------------------------------------

   procedure ProcessSimpleExpression (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     STree.Table;
   --#        in out ExpnStack;
   --#        in out LexTokenManager.StringTable;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         LexTokenManager.StringTable,
   --#         Statistics.TableUsage,
   --#         VCGHeap                     from *,
   --#                                          Dictionary.Dict,
   --#                                          ExpnStack,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          STree.Table,
   --#                                          VCGHeap;
   is
      OpNode   : STree.SyntaxNode;
      Op       : SPSymbols.SPSymbol;
      ----------------------------------------------------

      procedure ModelDivide
      --# global in     Dictionary.Dict;
      --#        in     OpNode;
      --#        in     STree.Table;
      --#        in out ExpnStack;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives ExpnStack,
      --#         Statistics.TableUsage,
      --#         VCGHeap               from *,
      --#                                    Dictionary.Dict,
      --#                                    ExpnStack,
      --#                                    OpNode,
      --#                                    STree.Table,
      --#                                    VCGHeap;
      is
         OpCell : Cells.Cell;
      begin
         Cells.Create_Cell (VCGHeap, OpCell);
         if Dictionary.TypeIsReal (STree.NodeSymbol (OpNode)) then
            Cells.Set_Kind (VCGHeap, OpCell, Cells.Op);
            Cells.Set_Op_Symbol (VCGHeap, OpCell, SPSymbols.divide);
         else
            Cells.Set_Kind (VCGHeap, OpCell, Cells.FDL_Div_Op);
         end if;
         SetRightArgument (OpCell, CStacks.Top (VCGHeap, ExpnStack), VCGHeap);
         CStacks.Pop (VCGHeap, ExpnStack);
         SetLeftArgument (OpCell, CStacks.Top (VCGHeap, ExpnStack), VCGHeap);
         CStacks.Pop (VCGHeap, ExpnStack);
         CStacks.Push (VCGHeap, OpCell, ExpnStack);
      end ModelDivide;

      ----------------------------------------

      procedure ModelRem
      --# global in out ExpnStack;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives ExpnStack,
      --#         Statistics.TableUsage,
      --#         VCGHeap               from *,
      --#                                    ExpnStack,
      --#                                    VCGHeap;
      is
         DAGCell,
         Left,
         Right,
         CopyOfLeft,
         CopyOfRight : Cells.Cell;
      begin
         -- correct modelling of I rem J as I - (I div J)  * J
         -- J is top of stack and I is 2nd TOS
         CStacks.PopOff (VCGHeap, ExpnStack, Right);
         CStacks.PopOff (VCGHeap, ExpnStack, Left);
         -- Make deep copies of arguments so that we can construct a model with
         -- no sharing of the argument cells.  This change arises from CFR 1154
         -- and affects only annotation expression because of potential problems
         -- substituting tilded globals in post conditions if a cell is revisited.
         Structures.CopyStructure (VCGHeap, Left, CopyOfLeft);
         Structures.CopyStructure (VCGHeap, Right, CopyOfRight);

         -- make core DIV sub-model using original arguments
         CreateCellKind (DAGCell, VCGHeap, Cells.FDL_Div_Op);
         SetRightArgument (DAGCell, Right, VCGHeap);
         SetLeftArgument (DAGCell, Left, VCGHeap);
         CStacks.Push (VCGHeap, DAGCell, ExpnStack);

         -- multiply by copy of right rather than re-using rigth
         CreateOpCell (DAGCell, VCGHeap, SPSymbols.multiply);
         SetRightArgument (DAGCell, CopyOfRight, VCGHeap);
         SetLeftArgument (DAGCell, CStacks.Top (VCGHeap, ExpnStack), VCGHeap);
         CStacks.Pop (VCGHeap, ExpnStack);
         CStacks.Push (VCGHeap, DAGCell, ExpnStack);

         CreateOpCell (DAGCell, VCGHeap, SPSymbols.minus);
         SetRightArgument (DAGCell, CStacks.Top (VCGHeap, ExpnStack), VCGHeap);
         CStacks.Pop (VCGHeap, ExpnStack);
         SetLeftArgument (DAGCell, CopyOfLeft, VCGHeap); -- note use of copy
         CStacks.Push (VCGHeap, DAGCell, ExpnStack);
      end ModelRem;

      ----------------------------------------

      procedure ModelCatenation
      --# global in     Dictionary.Dict;
      --#        in out ExpnStack;
      --#        in out LexTokenManager.StringTable;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives ExpnStack,
      --#         LexTokenManager.StringTable,
      --#         Statistics.TableUsage,
      --#         VCGHeap                     from *,
      --#                                          Dictionary.Dict,
      --#                                          ExpnStack,
      --#                                          LexTokenManager.StringTable,
      --#                                          VCGHeap;
      is
         Left,
         Right        : Cells.Cell;
         LeftString,
         RightString  : EStrings.T;
         LeftLength,
         RightLength  : EStrings.Lengths;
         NewString    : EStrings.Line;
         NewLength    : EStrings.Lengths;
         NewLexStr    : LexTokenManager.LexString;
         OkLeft,
         OkRight      : Boolean;

         procedure GetString (TheCell   : in     Cells.Cell;
                              TheString :    out EStrings.T;
                              ok        :    out Boolean)
         --# global in Dictionary.Dict;
         --#        in LexTokenManager.StringTable;
         --#        in VCGHeap;
         --# derives ok,
         --#         TheString from Dictionary.Dict,
         --#                        LexTokenManager.StringTable,
         --#                        TheCell,
         --#                        VCGHeap;
         is
            CharCode  : Integer;
            Unused    : Maths.ErrorCode;
         begin
            --# accept F, 10, Unused, "Unused unused here" &
            --#        F, 33, Unused, "Unused unused here";
            if Cells.Get_Kind (VCGHeap, TheCell) = Cells.Manifest_Const then
               LexTokenManager.LexStringToString (Cells.Get_Lex_Str (VCGHeap, TheCell),
                                                  TheString);
               ok := True;

            elsif Cells.Get_Kind (VCGHeap, TheCell) = Cells.Named_Const then
               Maths.ValueToInteger (Maths.ValueRep (
                  Dictionary.GetValue (Cells.Get_Symbol_Value (VCGHeap, TheCell))),
                  CharCode,
                  Unused);
               if CharCode = 0 then -- can't model nuls in strings
                  TheString := EStrings.EmptyString;
                  ok := False;

               else
                  TheString := EStrings.T'(Length  => 3,
                                                               Content => EStrings.Contents'(
                                                                  1      => '"',
                                                                  2      => Character'Val (CharCode),
                                                                  3      => '"',
                                                                  others => ' '));
                  ok := True;
               end if;
            else
               TheString := EStrings.EmptyString; -- should not occur
               ok := False;
            end if;
         end GetString;

         -------------------------------------------------------

      begin -- ModelCatentation
            -- get left and right strings to be concatenated
         CStacks.PopOff (VCGHeap, ExpnStack, Right);
         CStacks.PopOff (VCGHeap, ExpnStack, Left);

         if (Cells.Get_Kind (VCGHeap, Left) = Cells.Manifest_Const or
             Cells.Get_Kind (VCGHeap, Left) = Cells.Named_Const) and
            (Cells.Get_Kind (VCGHeap, Right) = Cells.Manifest_Const or
             Cells.Get_Kind (VCGHeap, Right) = Cells.Named_Const)
         then
            GetString (Left, LeftString, OkLeft);
            GetString (Right, RightString, OkRight);
            if OkLeft and OkRight then
               LeftLength  := LeftString.Length;
               RightLength := RightString.Length;

               -- New catenation of Left and Right (minus 2 quotation marks) must
               -- fit within an ExaminerString
               if (LeftLength + RightLength) - 2 <= EStrings.MaxStringLength then

                  -- build an examiner line with concatenated strings in it
                  NewString := EStrings.Line'(others => ' '); -- Avoid flow err
                  NewString (1) := '"';
                  for I in EStrings.Lengths range 2 .. LeftLength - 1 loop
                     NewString (I) := LeftString.Content (I);
                  end loop;
                  -- at this point we have " followed by left string and no terminating "
                  for I in EStrings.Lengths range 2 .. RightLength loop
                     NewString ((I + LeftLength) - 2) := RightString.Content (I);
                  end loop;
                  NewLength := (LeftLength + RightLength) - 2;
                  NewString (NewLength) := '"';
                  LexTokenManager.InsertCaseSensitiveLexString (NewString,
                                                                1,
                                                                NewLength,
                                                                  -- to get
                                                                NewLexStr);
                  Cells.Set_Lex_Str (VCGHeap,
                                      Left,
                                      NewLexStr);
                  CStacks.Push (VCGHeap, Left,  ExpnStack);
               else
                  -- Resulting string too long, so just re-push Left, Right and & op
                  CStacks.Push (VCGHeap, Left,  ExpnStack);
                  CStacks.Push (VCGHeap, Right, ExpnStack);
                  PushOperator (Binary, SPSymbols.ampersand, VCGHeap, ExpnStack);
               end if;
            else -- can't model string with nul in it so push & op instead
               CStacks.Push (VCGHeap, Left,  ExpnStack);
               CStacks.Push (VCGHeap, Right, ExpnStack);
               PushOperator (Binary, SPSymbols.ampersand, VCGHeap, ExpnStack);
            end if;

         else -- & operator shouldn't have got here so just push it as binop
            CStacks.Push (VCGHeap, Left,  ExpnStack);
            CStacks.Push (VCGHeap, Right, ExpnStack);
            PushOperator (Binary, SPSymbols.ampersand, VCGHeap, ExpnStack);
         end if;
      end ModelCatenation;

      ----------------------------------------

   begin  -- ProcessSimpleExpression
      OpNode := STree.Child_Node (
                   STree.Next_Sibling (
                      STree.Child_Node (Node)));
      if OpNode /= STree.NullNode then
         -- detection of / and REM for special handling
         Op := STree.SyntaxNodeType (OpNode);
         if Op = SPSymbols.divide then
            ModelDivide;
         elsif Op = SPSymbols.RWrem then
            ModelRem;
         elsif Op = SPSymbols.ampersand then
            ModelCatenation;
         else
            PushOperator (Binary, Op, VCGHeap, ExpnStack);
         end if;
         ModularizeIfNeeded (STree.NodeSymbol (OpNode), VCGHeap, ExpnStack);
      end if;
   end ProcessSimpleExpression;

   -------------------------------------------------------------------

   procedure ProcessSimpleExpressionOpt (Node : in STree.SyntaxNode)
   --# global in     STree.Table;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      OpNode   : STree.SyntaxNode;

   begin
      OpNode := STree.Child_Node (Node);
      if STree.SyntaxNodeType (OpNode) =
         SPSymbols.unary_adding_operator then
         PushOperator (Unary,
                       STree.SyntaxNodeType (STree.Child_Node (OpNode)),
                       VCGHeap,
                       ExpnStack);
      end if;
   end ProcessSimpleExpressionOpt;

   -------------------------------------------------------------------

   procedure ProcessFactor (Node : in STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     STree.Table;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap;
   is
      OpNode   : STree.SyntaxNode;
      BoolOpCell : Cells.Cell;
      ResultType : Dictionary.Symbol;

      -- Note, there is a similar version of this
      -- subprogram in BuildExprDAG
      procedure ModelModularNotOperation
   --# global in     ResultType;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    ExpnStack,
   --#                                    ResultType,
   --#                                    VCGHeap;
      is
         MinusOpCell,
         TickCell,
         PrefixCell,
         ModulusCell : Cells.Cell;
      begin
         ----------------------------------------------------
         -- LRM 4.5.6 (5) defines "not X" for a modular    --
         -- type T to be equivalent to T'Last - X.         --
         ----------------------------------------------------

         -- create ' operator
         CreateOpCell (TickCell, VCGHeap, SPSymbols.apostrophe);

         -- create Last attribute name
         CreateAttribValueCell (ModulusCell, VCGHeap, LexTokenManager.LastToken);

         -- Create prefix given by ResultType
         CreateFixedVarCell (PrefixCell, VCGHeap, ResultType);

         -- Assemble T'Last
         SetLeftArgument (TickCell, PrefixCell, VCGHeap);
         SetRightArgument (TickCell, ModulusCell, VCGHeap);

         -- create binary "-" operator
         CreateOpCell (MinusOpCell, VCGHeap, SPSymbols.minus);

         -- Construct T'Last - X, where X is on the top-of-stack
         SetRightArgument (MinusOpCell, CStacks.Top (VCGHeap, ExpnStack), VCGHeap);
         SetLeftArgument  (MinusOpCell, TickCell, VCGHeap);
         CStacks.Pop  (VCGHeap, ExpnStack);
         CStacks.Push (VCGHeap, MinusOpCell, ExpnStack);
      end ModelModularNotOperation;

   begin
      OpNode := STree.Child_Node (Node);
      if STree.SyntaxNodeType (OpNode) = SPSymbols.RWnot then
         -- check to see if result type is an array and
         -- build special model if it is
         ResultType := STree.NodeSymbol (OpNode);

         if Dictionary.IsTypeMark (ResultType) then

            if Dictionary.TypeIsArray (ResultType) then
               -- must be a Boolean array "not" operation
               CreateBoolOpCell (BoolOpCell, VCGHeap, ResultType, SPSymbols.RWnot);
               SetRightArgument (BoolOpCell, CStacks.Top (VCGHeap, ExpnStack), VCGHeap);
               CStacks.Pop (VCGHeap, ExpnStack);
               CStacks.Push (VCGHeap, BoolOpCell, ExpnStack);
            elsif Dictionary.TypeIsModular (ResultType) then
               -- must be a Modular "not" operation.
               ModelModularNotOperation;
            else -- proceed as before for scalar bool ops
               PushOperator (Unary, SPSymbols.RWnot, VCGHeap, ExpnStack);
            end if;

         else -- proceed as before for scalar bool ops
            PushOperator (Unary, SPSymbols.RWnot, VCGHeap, ExpnStack);
         end if;

         -- handle abs
      elsif STree.SyntaxNodeType (OpNode) = SPSymbols.RWabs then
         PushFunction (Cells.Abs_Function, VCGHeap, ExpnStack);

      else
         OpNode := STree.Next_Sibling (OpNode);
         if OpNode /= STree.NullNode then
            PushOperator (Binary, SPSymbols.double_star, VCGHeap, ExpnStack);
            ModularizeIfNeeded (STree.NodeSymbol (OpNode), VCGHeap, ExpnStack);
         end if;
      end if;
   end ProcessFactor;

   -------------------------------------------------------------------

   procedure DownProcessQuantifier (Node     : in     STree.SyntaxNode;
                                    LScope   :    out Dictionary.Scopes;
                                    NextNode :    out STree.SyntaxNode)
   --# global in     Dictionary.Dict;
   --#        in     STree.Table;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack             from Dictionary.Dict,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap &
   --#         LScope,
   --#         NextNode              from Node,
   --#                                    STree.Table &
   --#         Statistics.TableUsage from *,
   --#                                    Dictionary.Dict,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap &
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    STree.Table;
   is
      IdentNode       : STree.SyntaxNode;
      QuantifierSym,
      TypeSym         : Dictionary.Symbol;

      VarDecl,
      QuantIdent,
      TypeIdent       : Cells.Cell;

   begin -- DownProcessQuantifier
      IdentNode := STree.Next_Sibling (STree.Child_Node (Node));

      -- continue tree walk from the range node if present or else the predicate node
      NextNode  :=  STree.Next_Sibling (STree.Next_Sibling (IdentNode));

      QuantifierSym := STree.NodeSymbol (IdentNode); --planted by wffs
      -- enter local scope of quantifier
      LScope := Dictionary.LocalScope (QuantifierSym);

      -- build quantifier and type declaration and stack it.  In FDL we want the base type
      TypeSym := Dictionary.GetRootType (Dictionary.GetType (QuantifierSym));
      CreateFixedVarCell (TypeIdent, VCGHeap, TypeSym);
      CreateFixedVarCell (QuantIdent, VCGHeap, QuantifierSym);
      -- create VarDecl as QuantIdent : TypeIdent
      CreateOpCell (VarDecl, VCGHeap, SPSymbols.colon);
      SetLeftArgument (VarDecl, QuantIdent, VCGHeap);
      SetRightArgument (VarDecl, TypeIdent, VCGHeap);

      -- stack for use on the way up
      CStacks.Push (VCGHeap, VarDecl, ExpnStack);
   end DownProcessQuantifier;

   -------------------------------------------------------------------

   procedure UpProcessQuantifier (Node   : in     STree.SyntaxNode;
                                  LScope : in out Dictionary.Scopes)
   --# global in     Dictionary.Dict;
   --#        in     STree.Table;
   --#        in out ExpnStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ExpnStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpnStack,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    VCGHeap &
   --#         LScope                from *,
   --#                                    Dictionary.Dict;
   is
      RangeFound      : Boolean;
      RangeNode       : STree.SyntaxNode;
      QuantifierKind  : SPSymbols.SPSymbol;
      Predicate,
      RangeData,
      LeftSideOfRange,
      RightSideOfRange,
      LeftOp,
      RightOp,
      Declaration,
      QuantIdentCell,
      QuantifiedExpression,
      CommaCell,
      ImpliesCell : Cells.Cell;

      QuantifierSym  : Dictionary.Symbol;
      QuantifierType : Dictionary.Symbol;

   begin -- UpProcessQuantifier
      RangeNode  := STree.Next_Sibling (
                       STree.Next_Sibling (
                          STree.Next_Sibling (
                             STree.Child_Node (Node))));
      RangeFound := STree.SyntaxNodeType (RangeNode) = SPSymbols.annotation_arange;
      QuantifierKind := STree.SyntaxNodeType (STree.Child_Node (Node));

      -- TOS is the DAG representing the predicate
      CStacks.PopOff (VCGHeap, ExpnStack, Predicate);

      -- 2nd TOS is range expression if its there
      if RangeFound then
         -- range is either an attribute or explicit range
         if STree.SyntaxNodeType (STree.Child_Node (RangeNode)) =
            SPSymbols.annotation_attribute then
            -- range is defined by a range attribute on top of stack
            -- this has already been transformed by UpProcessAttribute
            -- which has left Index'First .. Index'Last on stack
            LeftSideOfRange  := LeftPtr (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));
            RightSideOfRange := RightPtr (VCGHeap, CStacks.Top (VCGHeap, ExpnStack));
            CStacks.Pop (VCGHeap, ExpnStack);  -- discard ..
         else
            -- range is defined by a pair of simple expressions;
            CStacks.PopOff (VCGHeap, ExpnStack, RightSideOfRange);
            CStacks.PopOff (VCGHeap, ExpnStack, LeftSideOfRange);
         end if;
         -- we now have the bounds of the range which we just need to assemble into a pair
         -- of bounds checks

         -- first get the Declaration data
         CStacks.PopOff (VCGHeap, ExpnStack, Declaration);
         QuantIdentCell := LeftPtr (VCGHeap, Declaration);

         -- create left sub-tree
         CreateOpCell (LeftOp, VCGHeap, SPSymbols.greater_or_equal);
         SetLeftArgument (LeftOp, QuantIdentCell, VCGHeap);
         SetRightArgument (LeftOp, LeftSideOfRange, VCGHeap);

         -- create right subtree
         CreateOpCell (RightOp, VCGHeap, SPSymbols.less_or_equal);
         SetLeftArgument (RightOp, QuantIdentCell, VCGHeap);
         SetRightArgument (RightOp, RightSideOfRange, VCGHeap);

         -- and them together to for RangeData DAG
         CreateOpCell (RangeData, VCGHeap, SPSymbols.RWand);
         SetLeftArgument (RangeData, LeftOp, VCGHeap);
         SetRightArgument (RangeData, RightOp, VCGHeap);

      else -- no explicit range
         CStacks.PopOff (VCGHeap, ExpnStack, Declaration);
         -- create a range DAG here
         -- Declaration is the colon in "ident : type"
         QuantifierSym  := Cells.Get_Symbol_Value (VCGHeap, LeftPtr (VCGHeap, Declaration));
         QuantifierType := Dictionary.GetType (QuantifierSym);

         -- For Boolean, we _mustn't_ try to create a range constraint, since Boolean
         -- isn't ordered in FDL.  Sem-compunit-wf_arange forbids the use
         -- of explicit ranges with Boolean, so the only possibility here is full-range
         -- Boolean.
         --
         -- We can't emit (Sym >= False and Sym <= True), for the reason stated above.
         -- We really know that (Sym or not Sym), but that's just "True"!
         if Dictionary.TypeIsBoolean (QuantifierType) then
            CreateTrueCell (VCGHeap, RangeData);
         else
            CreateRangeConstraint (LeftPtr (VCGHeap, Declaration), QuantifierType, VCGHeap, RangeData);
         end if;
      end if;

      -- now assemble the quantifier expression
      CreateOpCell (QuantifiedExpression, VCGHeap, QuantifierKind);
      CreateOpCell (CommaCell, VCGHeap, SPSymbols.comma);
      if QuantifierKind = SPSymbols.RWforall then
         CreateOpCell (ImpliesCell, VCGHeap, SPSymbols.implies);
      else -- must for_some
         CreateOpCell (ImpliesCell, VCGHeap, SPSymbols.RWand);
      end if;

      SetLeftArgument (ImpliesCell, RangeData, VCGHeap);
      SetRightArgument (ImpliesCell, Predicate, VCGHeap);

      SetLeftArgument (CommaCell, Declaration, VCGHeap);
      SetRightArgument (CommaCell, ImpliesCell, VCGHeap);

      SetRightArgument (QuantifiedExpression, CommaCell, VCGHeap);

      CStacks.Push (VCGHeap, QuantifiedExpression, ExpnStack);

      -- leave local scope of quantifier
      LScope := Dictionary.GetEnclosingScope (LScope);
   end UpProcessQuantifier;

   -------------------------------------------------------------------

begin -- BuildAnnotationExpnDAG
   CStacks.CreateStack (ExpnStack);
   LScope := Scope;
   Node := StartNode;
   loop --------------------------------down loop

      LastNode := Node;
      NodeType := STree.SyntaxNodeType (Node);
      case NodeType is
         -- prune at selector nodes so that only left most idents found
         when SPSymbols.annotation_selector =>
            Node := STree.NullNode;

         when SPSymbols.numeric_literal =>
            CreateManifestConstCell (DAGCell,
                                     VCGHeap,
                                     STree.NodeLexString
                                       (STree.Child_Node
                                          (STree.Child_Node
                                             (Node))));
            CStacks.Push (VCGHeap, DAGCell, ExpnStack);
            Node := STree.NullNode;

         when SPSymbols.character_literal |
            SPSymbols.string_literal        =>
            CreateManifestConstCell (DAGCell, VCGHeap, STree.NodeLexString (Node));
            CStacks.Push (VCGHeap, DAGCell, ExpnStack);
            Node := STree.NullNode;

         when SPSymbols.attribute_ident       =>
            DownProcessAttributeIdent (Node);
            Node := STree.NullNode;

         when SPSymbols.identifier =>
            ProcessIdentifier (Node, LScope);
            Node := STree.NullNode;

         when SPSymbols.annotation_simple_name =>
            if STree.SyntaxNodeType (STree.ParentNode (LastNode)) =
                  SPSymbols.annotation_named_argument_association
            then
               -- do not want look at parameter or field identifier
               Node := STree.NullNode;
            else
               Node := STree.Child_Node (Node);
            end if;

         when SPSymbols.annotation_aggregate =>
            DownProcessAggregate (SPSymbols.annotation_qualified_expression, VCGHeap, Node, ExpnStack);

         when SPSymbols.annotation_aggregate_choice_rep   =>
            DownProcessAggregateChoiceRep (LastNode, LScope, VCGHeap, ExpnStack, Node);

         when SPSymbols.record_component_selector_name   =>
            DownProcessRecordComponentSelectorName (LastNode, LScope, VCGHeap, ExpnStack, Node);

         when SPSymbols.store_list =>
            DownProcessStoreList (LastNode,
                                    -- to get
                                  Node);

         when SPSymbols.store =>
            DownProcessStore;
            Node := STree.Child_Node (Node);

         when SPSymbols.quantified_expression =>
            DownProcessQuantifier (LastNode,
                                    -- to get
                                   LScope,
                                   Node);

         when others                          =>
            Node := STree.Child_Node (Node);

      end case;
      if Node = STree.NullNode and LastNode /= StartNode then
         loop ------------------------up loop---------------------
            Node := STree.Next_Sibling (LastNode);
            exit when Node /= STree.NullNode;
            Node := STree.ParentNode (LastNode);
            exit when Node = STree.NullNode;
            NodeType := STree.SyntaxNodeType (Node);
            case NodeType is
               when SPSymbols.annotation_selected_component =>
                  ProcessSelectedComponent (Node, LScope);

               when SPSymbols.annotation_qualified_expression    =>
                  ModelQualifiedExpression (Node);

               when SPSymbols.annotation_aggregate               =>
                  UpProcessAggregate;

               when SPSymbols.annotation_extension_aggregate     =>
                  UpProcessExtensionAggregate (VCGHeap, ExpnStack);

               when SPSymbols.annotation_ancestor_part =>
                  ProcessAncestorPart (Node, VCGHeap, ExpnStack);

               when SPSymbols.annotation_component_association   =>
                  UpProcessComponentAssociation (Node);

               when SPSymbols.annotation_named_association_rep    =>
                  UpProcessNamedAssociationRep (Node);

               when SPSymbols.annotation_named_record_component_association  =>
                  UpProcessNamedRecordComponentAssociation;

               when SPSymbols.annotation_positional_record_component_association  =>
                  UpProcessPositionalRecordComponentAssociation;

               when SPSymbols.annotation_aggregate_choice_rep   =>
                  UpProcessAggregateChoiceRep (Node, VCGHeap, ExpnStack);

               when SPSymbols.annotation_aggregate_or_expression =>
                  UpProcessAggregateOrExpression (Node);

               when SPSymbols.annotation_aggregate_choice        =>
                  UpProcessAggregateChoice (Node);

               when SPSymbols.annotation_expression          |
                  SPSymbols.annotation_expression_rep1       |
                  SPSymbols.annotation_expression_rep2       |
                  SPSymbols.annotation_expression_rep3       |
                  SPSymbols.annotation_expression_rep4       |
                  SPSymbols.annotation_expression_rep5       |
                  SPSymbols.annotation_expression_rep6       |
                  SPSymbols.annotation_expression_rep7         =>
                  ProcessExpression (Node);

               when SPSymbols.annotation_relation                =>
                  ProcessRelation (Node);

               when SPSymbols.annotation_simple_expression   |
                  SPSymbols.annotation_term                    =>
                  ProcessSimpleExpression (Node);

               when SPSymbols.annotation_simple_expression_opt   =>
                  ProcessSimpleExpressionOpt (Node);

               when SPSymbols.annotation_factor                  =>
                  ProcessFactor (Node);

               when SPSymbols.annotation_positional_argument_association =>
                  ProcessPositionalArgumentAssociation;

               when SPSymbols.annotation_named_argument_association =>
                  ProcessNamedArgumentAssociation (Node);

               when SPSymbols.annotation_name_argument_list      =>
                  ProcessNameArgumentList;

               when SPSymbols.annotation_attribute_designator    =>
                  UpProcessAttributeDesignator (Node);

               when SPSymbols.store =>
                  UpProcessStore (Node);

               when SPSymbols.store_list =>
                  UpProcessStoreList;

               when SPSymbols.quantified_expression =>
                  UpProcessQuantifier (Node,
                                       -- using and to get
                                       LScope);
               when others                          =>
                  null;
            end case;
            exit when Node = StartNode;
            LastNode := Node;
         end loop; -----------------------------up loop--------------
      end if;
      exit when Node = STree.NullNode or Node = StartNode;
   end loop;  --------------------------down loop------------------

   --# accept F, 10, ExpnStack, "Ineffective assignment here OK";
   CStacks.PopOff (VCGHeap, ExpnStack, DAGRoot);
   --# end accept;
end BuildAnnotationExpnDAG;
