-- $Id: contextmanager.adb 13056 2009-04-20 17:01:20Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with SystemErrors;

package body ContextManager
is

   type FileEntries is record
      Name            : EStrings.T;
      Status          : FileStatus;
      File            : SPARK_IO.File_Type;
      ListingReq      : Boolean;
      ListingFileName : EStrings.T;
      LineContext     : SparkLex.LineContext;
      ErrorContext    : ErrorHandler.ErrorContexts;
      UnitCount       : Natural;
      ErrsReported    : Boolean;
   end record;

   NullFileEntry : constant FileEntries :=
     FileEntries'(Name            => EStrings.EmptyString,
                  Status          => NoFileEntry,
                  File            => SPARK_IO.Null_File,
                  ListingReq      => False,
                  ListingFileName => EStrings.EmptyString,
                  LineContext     => SparkLex.NullLineContext,
                  ErrorContext    => ErrorHandler.NullErrorContext,
                  UnitCount       => Natural'First,
                  ErrsReported    => False);

   type UnitEntries is record
      Name           : LexTokenLists.Lists;
      UnitType       : UnitTypes;
      Status         : UnitStatus;
      FileDescriptor : FileDescriptors;
      ParseTree      : STree.SyntaxNode;
      RedType        : CommandLineData.RedTypes;
      UnitNumber     : Natural;
      CycleDetected  : Boolean;
   end record;

   NullUnitEntry : constant UnitEntries :=
      UnitEntries'(Name           => LexTokenLists.NullList,
                   UnitType       => PackageSpecification,
                   Status         => NoUnitEntry,
                   FileDescriptor => NullFile,
                   ParseTree      => STree.NullNode,
                   RedType        => CommandLineData.NoVCG,
                   UnitNumber     => 0,
                   CycleDetected  => False);

   subtype FilePointers is FileDescriptors range 1 .. FileDescriptors'Last;

   type FileHeapContents is array (FilePointers) of FileEntries;

   type FileHeaps is record
      Content  : FileHeapContents;
      LastUsed : FileDescriptors;
   end record;

   subtype UnitPointers is UnitDescriptors range 1 .. UnitDescriptors'Last;

   type UnitHeapContents is array (UnitPointers) of UnitEntries;

   type UnitHeaps is record
      Content  : UnitHeapContents;
      LastUsed : UnitDescriptors;
   end record;

   subtype StackHeights is Integer range
      0 .. ExaminerConstants.ContextManagerMaxUnits;

   subtype StackPointers is Integer range
      1 .. ExaminerConstants.ContextManagerMaxUnits;

   type StackContents is array (StackPointers) of UnitDescriptors;

   type UnitStacks is record
      Content : StackContents;
      Height  : StackHeights;
   end record;

   FileHeap  : FileHeaps;
   UnitHeap  : UnitHeaps;
   UnitStack : UnitStacks;

   function CurrentUnit return UnitDescriptors
   is
      Result : UnitDescriptors;
   begin
      if UnitStack.Height > 0 then
         Result := UnitStack.Content (UnitStack.Height);
      else
         Result := NullUnit;
      end if;
      return Result;
   end CurrentUnit;

   function GetUnitStatus (Descriptor : UnitDescriptors) return UnitStatus
   is
   begin
      return UnitHeap.Content (Descriptor).Status;
   end GetUnitStatus;

   procedure CreateUnitDescriptor (Descriptor : out UnitDescriptors)
   is
   begin
      if UnitHeap.LastUsed = UnitPointers'Last then
         SystemErrors.FatalError (SystemErrors.ContextUnitHeapOverflow, "");
      end if;
      UnitHeap.LastUsed := UnitHeap.LastUsed + 1;
      Descriptor := UnitHeap.LastUsed;
      UnitHeap.Content (UnitHeap.LastUsed) := NullUnitEntry;
   end CreateUnitDescriptor;

   procedure SetFileDescriptor (UnitDescriptor : in UnitDescriptors;
                                FileDescriptor : in FileDescriptors)
   is
   begin
      UnitHeap.Content (UnitDescriptor).FileDescriptor := FileDescriptor;
   end SetFileDescriptor;

   procedure GetFileDescriptor (UnitDescriptor : in     UnitDescriptors;
                                FileDescriptor :    out FileDescriptors)
   is
   begin
      FileDescriptor := UnitHeap.Content (UnitDescriptor).FileDescriptor;
   end GetFileDescriptor;

   procedure SetUnitStatus (Descriptor : in UnitDescriptors;
                            Status     : in UnitStatus)
   is
   begin
      UnitHeap.Content (Descriptor).Status := Status;
   end SetUnitStatus;

   procedure IncUnitCount (Descriptor : in FileDescriptors)
   is
   begin
      FileHeap.Content (Descriptor).UnitCount :=
         FileHeap.Content (Descriptor).UnitCount + 1;
   end IncUnitCount;

   procedure GetUnitCount (Descriptor : in     FileDescriptors;
                           Count      :    out Natural)
   is
   begin
      Count := FileHeap.Content (Descriptor).UnitCount;
   end GetUnitCount;

   procedure SetUnitNumber (Descriptor : in UnitDescriptors;
                            Count      : in Natural)
   is
   begin
      UnitHeap.Content (Descriptor).UnitNumber := Count;
   end SetUnitNumber;

   function GetUnitNumber (Descriptor : UnitDescriptors) return Natural
   is
   begin
      return UnitHeap.Content (Descriptor).UnitNumber;
   end GetUnitNumber;

   procedure MarkUnitInCycle (Descriptor : in UnitDescriptors)
   is
   begin
      UnitHeap.Content (Descriptor).CycleDetected := True;
   end MarkUnitInCycle;

   function UnitInCycle (Descriptor : UnitDescriptors) return Boolean
   is
   begin
      return UnitHeap.Content (Descriptor).CycleDetected;
   end UnitInCycle;

   function GetFileStatus (Descriptor : FileDescriptors) return FileStatus
   is
   begin
      return FileHeap.Content (Descriptor).Status;
   end GetFileStatus;

   procedure SetRedType (Descriptor : in UnitDescriptors;
                         RedType    : in CommandLineData.RedTypes)
   is
   begin
      UnitHeap.Content (Descriptor).RedType := RedType;
   end SetRedType;

   procedure GetRedType (Descriptor : in     UnitDescriptors;
                         RedType    :    out CommandLineData.RedTypes)
   is
   begin
      RedType := UnitHeap.Content (Descriptor).RedType;
   end GetRedType;

   procedure GetUnitByName (UnitName    : in     LexTokenLists.Lists;
                            UnitTypeSet : in     UnitTypeSets;
                            Descriptor  :    out UnitDescriptors)
   is
   begin
      Descriptor := NullUnit;
      for I in UnitPointers range 1 .. UnitHeap.LastUsed loop
         if LexTokenLists.EqUnit (UnitName, UnitHeap.Content (I).Name) and
            UnitTypeSet (UnitHeap.Content (I).UnitType)
         then
            Descriptor := I;
            exit;
         end if;
      end loop;
   end GetUnitByName;

   procedure SetUnitName (Descriptor : in UnitDescriptors;
                          UnitName   : in LexTokenLists.Lists;
                          UnitType   : in UnitTypes)
   is
   begin
      UnitHeap.Content (Descriptor).Name := UnitName;
      UnitHeap.Content (Descriptor).UnitType := UnitType;
   end SetUnitName;

   procedure GetUnitName (Descriptor : in     UnitDescriptors;
                          UnitName   :    out LexTokenLists.Lists;
                          UnitType   :    out UnitTypes)
   is
   begin
      UnitName := UnitHeap.Content (Descriptor).Name;
      UnitType := UnitHeap.Content (Descriptor).UnitType;
   end GetUnitName;

   procedure SetParseTree (Descriptor : in UnitDescriptors;
                           ParseTree  : in STree.SyntaxNode)
   is
   begin
      UnitHeap.Content (Descriptor).ParseTree := ParseTree;
   end SetParseTree;

   procedure GetParseTree (Descriptor : in     UnitDescriptors;
                           ParseTree  :    out STree.SyntaxNode)
   is
   begin
      ParseTree := UnitHeap.Content (Descriptor).ParseTree;
   end GetParseTree;

   function FirstUnitDescriptor return UnitDescriptors
   is
      Result : UnitDescriptors;
   begin
      if UnitHeap.LastUsed = NullUnit then
         Result := NullUnit;
      else
         Result := UnitPointers'First;
      end if;
      return Result;
   end FirstUnitDescriptor;

   function NextUnitDescriptor (Descriptor : UnitDescriptors)
                               return UnitDescriptors
   is
      Result : UnitDescriptors;
   begin
      if Descriptor = UnitHeap.LastUsed then
         Result :=  NullUnit;
      else
         Result :=  Descriptor + 1;
      end if;
      return Result;
   end NextUnitDescriptor;

   procedure PushUnit (Descriptor : in UnitDescriptors)
   is
   begin
      if UnitStack.Height = StackHeights'Last then
         SystemErrors.FatalError (SystemErrors.ContextUnitStackOverflow, "");
      end if;

      UnitStack.Height := UnitStack.Height + 1;
      UnitStack.Content (UnitStack.Height) := Descriptor;
   end PushUnit;

   procedure PopUnit (Descriptor : out UnitDescriptors)
   is
   begin
      if UnitStack.Height = 0 then
         SystemErrors.FatalError (SystemErrors.ContextUnitStackUnderflow, "");
      end if;

      Descriptor := UnitStack.Content (UnitStack.Height);
      UnitStack.Height := UnitStack.Height - 1;
   end PopUnit;

   procedure CreateFileDescriptor (Descriptor : out FileDescriptors)
   is
   begin
      if FileHeap.LastUsed = FilePointers'Last then
         SystemErrors.FatalError (SystemErrors.ContextFileHeapOverflow, "");
      end if;
      FileHeap.LastUsed := FileHeap.LastUsed + 1;
      Descriptor := FileHeap.LastUsed;
      FileHeap.Content (FileHeap.LastUsed).Status := NoFileEntry;
      FileHeap.Content (FileHeap.LastUsed).UnitCount := 0;
      FileHeap.Content (FileHeap.LastUsed).ErrsReported := False;
   end CreateFileDescriptor;

   procedure SetSourceFileName (Descriptor     : in FileDescriptors;
                                SourceFileName : in EStrings.T)
   is
   begin
      FileHeap.Content (Descriptor).Name := SourceFileName;
   end SetSourceFileName;

   procedure GetSourceFileName (Descriptor     : in     FileDescriptors;
                                SourceFileName :    out EStrings.T)
   is
   begin
      SourceFileName := FileHeap.Content (Descriptor).Name;
   end GetSourceFileName;

   procedure SetSourceFile (Descriptor : in FileDescriptors;
                            SourceFile : in SPARK_IO.File_Type)
   is
   begin
      FileHeap.Content (Descriptor).File := SourceFile;
   end SetSourceFile;

   procedure GetSourceFile (Descriptor : in     FileDescriptors;
                            SourceFile :    out SPARK_IO.File_Type)
   is
   begin
      SourceFile := FileHeap.Content (Descriptor).File;
   end GetSourceFile;

   function ListingReqt (Descriptor : FileDescriptors) return Boolean
   is
   begin
      return FileHeap.Content (Descriptor).ListingReq;
   end ListingReqt;

   function FirstFileDescriptor return FileDescriptors
   is
      Result : FileDescriptors;
   begin
      if FileHeap.LastUsed = NullFile then
         Result := NullFile;
      else
         Result := FilePointers'First;
      end if;
      return Result;
   end FirstFileDescriptor;

   function NextFileDescriptor (Descriptor : FileDescriptors)
                               return FileDescriptors
   is
      Result : FileDescriptors;
   begin
      if Descriptor = FileHeap.LastUsed then
         Result := NullFile;
      else
         Result := Descriptor + 1;
      end if;
      return Result;
   end NextFileDescriptor;

   procedure GetFileByName (FileName   : in     EStrings.T;
                            Descriptor :    out FileDescriptors)
   is
   begin
      Descriptor := NullFile;
      for I in FilePointers range 1 .. FileHeap.LastUsed loop
         if EStrings.EqString (FileName, FileHeap.Content (I).Name) then
            Descriptor := I;
            exit;
         end if;
      end loop;

   end GetFileByName;

   procedure SetFileStatus (Descriptor : in FileDescriptors;
                            Status     : in FileStatus)
   is
   begin
      FileHeap.Content (Descriptor).Status := Status;
   end SetFileStatus;

   procedure SetListingReq (Descriptor : in FileDescriptors;
                            Req        : in Boolean)
   is
   begin
      FileHeap.Content (Descriptor).ListingReq := Req;
   end SetListingReq;

   procedure SetLineContext (Descriptor  : in FileDescriptors;
                             FileContext : in SparkLex.LineContext)
   is
   begin
      FileHeap.Content (Descriptor).LineContext := FileContext;
   end SetLineContext;

   procedure GetLineContext (Descriptor  : in     FileDescriptors;
                             FileContext :    out SparkLex.LineContext)
   is
   begin
      FileContext := FileHeap.Content (Descriptor).LineContext;
   end GetLineContext;

   procedure SetErrorContext (Descriptor : in FileDescriptors;
                              Context    : in ErrorHandler.ErrorContexts)
   is
   begin
      FileHeap.Content (Descriptor).ErrorContext := Context;
   end SetErrorContext;

   procedure GetErrorContext (Descriptor : in     FileDescriptors;
                              Context    :    out ErrorHandler.ErrorContexts)
   is
   begin
      Context := FileHeap.Content (Descriptor).ErrorContext;
   end GetErrorContext;

   procedure SetListingFileName (Descriptor      : in FileDescriptors;
                                 ListingFileName : in EStrings.T)
   is
   begin
      FileHeap.Content (Descriptor).ListingFileName := ListingFileName;
   end SetListingFileName;

   procedure GetListingFileName (Descriptor      : in     FileDescriptors;
                                 ListingFileName :    out EStrings.T)
   is
   begin
      ListingFileName := FileHeap.Content (Descriptor).ListingFileName;
   end GetListingFileName;

   procedure SetErrorsReported (Descriptor : in FileDescriptors)
   is
   begin
      FileHeap.Content (Descriptor).ErrsReported := True;
   end SetErrorsReported;

   function ErrorsReported (Descriptor : FileDescriptors) return Boolean
   is
   begin
      return FileHeap.Content (Descriptor).ErrsReported;
   end ErrorsReported;

begin
   UnitHeap := UnitHeaps'(Content  => UnitHeapContents'(others => NullUnitEntry),
                          LastUsed => 0);

   UnitStack := UnitStacks'(Content => StackContents'(others => NullUnit),
                            Height  => StackHeights'First);

   --# accept F, 23, FileHeap.Content, "Partial initialization OK here";
   FileHeap.LastUsed := 0;
   FileHeap.Content (FilePointers'First) := NullFileEntry;

   --# accept F, 602, FileHeap, FileHeap.Content, "Partial initialization OK here";
end ContextManager;
