-- $Id: configfile.adb 13146 2009-04-24 17:27:57Z Trevor Jennings $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with LexTokenManager;
with Maths;
with Dictionary;
with EStrings;
with ExaminerConstants;
with SparkLex;
with SPSymbols;
with ErrorHandler;
with Error_Types;
with SPExpectedSymbols;
with CommandLineData;
with FileSystem;
with ScreenEcho;
with XMLReport;

use type SPSymbols.SPSymbol;
use type Maths.ErrorCode;
use type Maths.Value;
use type Dictionary.Symbol;
use type ErrorHandler.ErrorLevel;
use type LexTokenManager.LexString;

package body ConfigFile
--# own State is TheErrorContext;
is

   -- =====================================================================
   -- = TYPES AND CONSTANTS
   -- =====================================================================

   --------------------
   -- Declaration types
   --------------------

   type DeclarationType is (DecNamedInteger,
                            DecNamedReal,
                            DecIntegerSubtype,
                            DecIntegerType,
                            DecFloatingPointType,
                            DecPrivate,
                            DecTypedConstant);
   -- Depth of child package hierarchy that we support
   type ChildPackageDepth is range 1 .. 2;
   type PackageHierarchy is array (ChildPackageDepth) of LexTokenManager.LexValue;

   type Declaration is
      record
         MyType     : DeclarationType;
         EncPackage : PackageHierarchy;
         Name       : LexTokenManager.LexValue;
         BaseType   : LexTokenManager.LexValue;
         LowBound   : Maths.Value;
         HighBound  : Maths.Value;
         NumDigits  : Maths.Value;
         -- The value of a declartion can be numeric or an identifier.
         -- One of the following two fields is well-defined, depending on the
         -- value of MyType
         Value      : Maths.Value;
         ValueStr   : LexTokenManager.LexValue;
      end record;

   ---------------------------------
   -- Identifier types and constants
   ---------------------------------

   MaxNameLength : constant := 28;
   subtype NameLength is Positive range 1 .. MaxNameLength;
   subtype NameStringText is String (NameLength);

   type IdentifierString is
      record
         Content : NameStringText;
         Length  : NameLength;
      end record;

   type PossibleIdentifierRange is range 1 .. 21;
   --# assert PossibleIdentifierRange'Base is Short_Short_Integer; -- for GNAT

   type IdentifierStrings is array (PossibleIdentifierRange)
     of IdentifierString;

   PossibleIdentifiers : constant IdentifierStrings :=
     IdentifierStrings'(-- integers
                        1  => IdentifierString'("system.min_int              ", 14),
                        2  => IdentifierString'("system.max_int              ", 14),
                        3  => IdentifierString'("system.max_binary_modulus   ", 25),
                        4  => IdentifierString'("system.max_digits           ", 17),
                        5  => IdentifierString'("system.max_base_digits      ", 22),
                        6  => IdentifierString'("system.max_mantissa         ", 19),
                        7  => IdentifierString'("system.priority_last        ", 20),
                        8  => IdentifierString'("system.storage_unit         ", 19),
                        9  => IdentifierString'("system.word_size            ", 16),

                        -- reals
                        10 => IdentifierString'("system.fine_delta           ", 17),

                        -- subtypes
                        11 => IdentifierString'("system.any_priority         ", 19),
                        12 => IdentifierString'("system.priority             ", 15),
                        13 => IdentifierString'("system.interrupt_priority   ", 25),

                        -- types
                        14 => IdentifierString'("standard.integer            ", 16),
                        15 => IdentifierString'("standard.float              ", 14),
                        16 => IdentifierString'("standard.long_integer       ", 21),
                        17 => IdentifierString'("standard.long_float         ", 19),

                        -- private type
                        18 => IdentifierString'("system.address              ", 14),

                        -- Ravenscar type
                        19 => IdentifierString'("ada.real_time.seconds_count ", 27),
                        20 => IdentifierString'("ada.interrupts.interrupt_id ", 27),
                        -- typed constants
                        21 => IdentifierString'("system.default_bit_order    ", 24));

   type IdentifierTypeAry is array (PossibleIdentifierRange) of DeclarationType;
   IdentifierTypes : constant IdentifierTypeAry :=
     IdentifierTypeAry'(1 => DecNamedInteger,
                        2 => DecNamedInteger,
                        3 => DecNamedInteger,
                        4 => DecNamedInteger,
                        5 => DecNamedInteger,
                        6 => DecNamedInteger,
                        7 => DecNamedInteger,
                        8 => DecNamedInteger,
                        9 => DecNamedInteger,
                        10 => DecNamedReal,
                        11 => DecIntegerSubtype,
                        12 => DecIntegerSubtype,
                        13 => DecIntegerSubtype,
                        14 => DecIntegerType,
                        15 => DecFloatingPointType,
                        16 => DecIntegerType,
                        17 => DecFloatingPointType,
                        18 => DecPrivate,
                        19 => DecIntegerType,
                        20 => DecIntegerType,
                        21 => DecTypedConstant);

   -- special prefixes / suffixes
   StandardPrefix : constant IdentifierString :=
     IdentifierString'("standard.                   ",  9);
   FloatSuffix : constant IdentifierString :=
     IdentifierString'("_float                      ",  6);
   IntegerSuffix : constant IdentifierString :=
     IdentifierString'("_integer                    ",  8);

   -- acceptable packages
   type LibraryPackageRange is range 1 .. 3;
   type ChildPackageRange is range 1 .. 2;

   --# assert LibraryPackageRange'Base is Short_Short_Integer; -- for GNAT
   --# assert ChildPackageRange'Base is Short_Short_Integer; -- for GNAT
   --# assert ChildPackageDepth'Base is Short_Short_Integer; -- for GNAT

   type LibraryPackageStrings is array (LibraryPackageRange) of IdentifierString;
   type ChildPackageStrings is array (ChildPackageRange) of IdentifierString;

   PossibleLibraryPackages : constant LibraryPackageStrings :=
     LibraryPackageStrings'(IdentifierString'("standard                    ", 8),
                            IdentifierString'("system                      ", 6),
                            IdentifierString'("ada                         ", 3));
   PossibleChildPackages : constant ChildPackageStrings :=
     ChildPackageStrings'(1 => IdentifierString'("real_time                   ", 9),
                          2 => IdentifierString'("interrupts                  ", 10));

   -- special package(s) / type(s)
   PackageStandard          : constant LibraryPackageRange := 1;
   PackageSystem            : constant LibraryPackageRange := 2;
   PackageAda               : constant LibraryPackageRange := 3;
   PackageReal_Time         : constant ChildPackageRange := 1;
   PackageInterrupts        : constant ChildPackageRange := 2;

   MaxBinaryModulus         : constant PossibleIdentifierRange := 3;
   AnyPrioritySubtype       : constant PossibleIdentifierRange := 11;
   PrioritySubtype          : constant PossibleIdentifierRange := 12;
   InterruptPrioritySubtype : constant PossibleIdentifierRange := 13;
   PredefInteger            : constant PossibleIdentifierRange := 14;
   PredefFloat              : constant PossibleIdentifierRange := 15;
   PredefLongInteger        : constant PossibleIdentifierRange := 16;
   PredefLongFloat          : constant PossibleIdentifierRange := 17;
   SystemAddress            : constant PossibleIdentifierRange := 18;
   SecondsCount             : constant PossibleIdentifierRange := 19;
   InterruptID              : constant PossibleIdentifierRange := 20;
   DefaultBitOrder          : constant PossibleIdentifierRange := 21;

   -- 'renames' of commonly used entities from package Dictionary

   NullSymbol     : constant Dictionary.Symbol := Dictionary.NullSymbol;
   ProgramContext : constant Dictionary.Contexts := Dictionary.ProgramContext;
   ProofContext   : constant Dictionary.Contexts := Dictionary.ProofContext;

   -- error handling types and constants

   type SyntaxErrorTypes is (ExpSymbol, ExpDefn, ExpDigitsRange,
                             ExpMinusReal, ExpMinusInteger, ExpIntegerReal);

   type SyntaxErrors is
      record
         TheType        : SyntaxErrorTypes;
         TheSymbol      : SPSymbols.SPTerminal;
         ExpectedSymbol : SPSymbols.SPTerminal;
         TheLexval      : LexTokenManager.LexValue;
      end record;

   -- STATE

   TheErrorContext : ErrorHandler.ErrorContexts;

   procedure OpenFile (TheFile : out SPARK_IO.File_Type;
                       FileOk  : out Boolean)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives FileOk,
   --#         SPARK_IO.File_Sys,
   --#         TheFile           from CommandLineData.Content,
   --#                                SPARK_IO.File_Sys;
   is
      FileName       : EStrings.T;
      FileSpecStatus : FileSystem.TypFileSpecStatus;
      FileStatus     : SPARK_IO.File_Status;
      LocalFile      : SPARK_IO.File_Type;
   begin
      LocalFile := SPARK_IO.Null_File;

      --# accept F, 10, FileSpecStatus, "FileSpecStatus unused here";
      FileSystem.FindFullFileName
        (CommandLineData.Content.TargetConfigFile,
         FileSpecStatus,
         FileName);
      SPARK_IO.Open (LocalFile,
                     SPARK_IO.In_File,
                     FileName.Length,
                     FileName.Content,
                     "",
                     FileStatus);

      if FileStatus = SPARK_IO.Ok then
         FileOk := True;
      else
         FileOk := False;
         ScreenEcho.Put_String ("Cannot open file ");
         if CommandLineData.Content.PlainOutput then
            ScreenEcho.Put_ExaminerLine
              (EStrings.LowerCase
               (FileSystem.JustFile (FileName, True)));
         else
            ScreenEcho.Put_ExaminerLine (FileName);
         end if;
      end if;
      TheFile := LocalFile;
      --# accept F, 33, FileSpecStatus, "FileSpecStatus unused here";
   end OpenFile;

   procedure CloseFile (TheFile : in out SPARK_IO.File_Type)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys,
   --#         TheFile           from *,
   --#                                TheFile;
   is
      FileStatus     : SPARK_IO.File_Status;
   begin
      --# accept F, 10, FileStatus, "FileStatus unused here";
      SPARK_IO.Close (TheFile,
                      FileStatus);
      --# accept F, 33, FileStatus, "FileStatus unused here";
   end CloseFile;

   procedure ProcessConfigFile (TheConfigFile : in SPARK_IO.File_Type;
                                OverallStatus : out Boolean)
   --# global in     CommandLineData.Content;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#           out SparkLex.CurrLine;
   --# derives Dictionary.Dict,
   --#         ErrorHandler.ErrorContext,
   --#         LexTokenManager.StringTable,
   --#         OverallStatus,
   --#         SparkLex.CurrLine,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          SPARK_IO.File_Sys,
   --#                                          TheConfigFile;
   is
      -- =====================================================================
      -- = LOCAL STATE
      -- =====================================================================

      -------------------
      -- Pseudo-constants
      -------------------

      PredefinedScope : Dictionary.Scopes;

      ------------------------
      -- Important state items
      ------------------------

      CurrentScope       : Dictionary.Scopes;
      ValidPackage       : Boolean;
      CurrentDeclaration : Declaration;
      TheToken           : SPSymbols.SPTerminal;
      TheLexVal          : LexTokenManager.LexValue;
      ErrorsOccurred     : Boolean;

      -- Flag to reparse current lexeme (after parser state change)
      Lookahead : Boolean;

      AnyPrioritySym       : Dictionary.Symbol := Dictionary.NullSymbol;
      PrioritySym          : Dictionary.Symbol := Dictionary.NullSymbol;
      InterruptPrioritySym : Dictionary.Symbol := Dictionary.NullSymbol;
      LibPackageSymbol     : Dictionary.Symbol := Dictionary.NullSymbol;

      --------------------------------------------
      -- Integer expression parsing and evaluation
      --------------------------------------------

      UnaryMinus     : Boolean;
      IntSubtract    : Boolean;
      IntAdd         : Boolean;
      IntExponent    : Boolean;

      IntVal         : Maths.Value;
      IntExponentVal : Maths.Value;
      IntAddSubVal   : Maths.Value;

      ExprIntValue   : Maths.Value;

      ---------------------------------------
      -- Parsing and basic semantics checking
      ---------------------------------------

      type ParserStates is (Config_Defn_Start,
                            Next_Config_Defn_Start,
                            Defn,
                            Type_Unknown,
                            Type_Private,
                            Type_FP,
                            Type_Int,
                            Subtype_Int,
                            Expr_Int,
                            Const_Unknown,
                            Config_Defn_End,
                            FoundSyntaxError);
      subtype ParserStages is Positive range 1 .. 10;

      ParserState       : ParserStates;
      ParserStage       : ParserStages;
      SavedParserState  : ParserStates;
      SavedParserStage  : ParserStages;
      NextExpectedState : ParserStates;

      ParsingEnds : Boolean;

      ----------------------
      -- Dotted Name Parsing
      ----------------------
      PackageNameDepth : ChildPackageDepth := 1;
      PackageFullName,
      PackageEndName   : EStrings.T := EStrings.EmptyString;

      -- =====================================================================
      -- = LOCAL SUBPROGRAMS
      -- =====================================================================

      procedure NextStage
      --# global in out ParserStage;
      --# derives ParserStage from *;
      is
      begin
         ParserStage := ParserStage + 1;
      end NextStage;

      procedure NewState (The_Stage : in ParserStates)
      --# global out ParserStage;
      --#        out ParserState;
      --# derives ParserStage from  &
      --#         ParserState from The_Stage;
      is
      begin
         ParserStage := 1;
         ParserState := The_Stage;
      end NewState;

      -- Error / warning handling

      procedure GuardedLiteralToValue (Lex : in LexTokenManager.LexValue;
                                       Val : out Maths.Value)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.File_Sys         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        Lex,
      --#                                        LexTokenManager.StringTable,
      --#                                        SPARK_IO.File_Sys &
      --#         Val                       from Lex,
      --#                                        LexTokenManager.StringTable;
      is
         LocalErrorCode : Maths.ErrorCode;
      begin
         Maths.LiteralToValue (Lex.TokenStr, Val, LocalErrorCode);
         if LocalErrorCode /= Maths.NoError then
            ErrorHandler.SemanticWarning (200, Lex.Position, Lex.TokenStr);
         end if;
      end GuardedLiteralToValue;

      procedure RaiseSyntaxError (TheError : SyntaxErrors)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.StringTable;
      --#        in out ParserState;
      --#        in out SPARK_IO.File_Sys;
      --#           out ErrorsOccurred;
      --#           out NextExpectedState;
      --#           out ParserStage;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.File_Sys           from CommandLineData.Content,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.StringTable,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TheError &
      --#         ErrorsOccurred,
      --#         ParserStage,
      --#         ParserState                 from  &
      --#         LexTokenManager.StringTable from *,
      --#                                          TheError &
      --#         NextExpectedState           from ParserState;
      is
         NoOfTerminals : SPExpectedSymbols.SPEssSymRange;
         TerminalList : SPExpectedSymbols.SPExpSymList :=
           SPExpectedSymbols.SPExpSymList'(others => SPSymbols.SPDEFAULT);
      begin
         case TheError.TheType is
            when ExpSymbol =>
               NoOfTerminals := 1;
               TerminalList (1) := TheError.ExpectedSymbol;
            when ExpDefn =>
               NoOfTerminals := 3;
               TerminalList (1) := SPSymbols.identifier;
               TerminalList (2) := SPSymbols.RWtype;
               TerminalList (3) := SPSymbols.RWsubtype;
            when ExpDigitsRange =>
               NoOfTerminals := 3;
               TerminalList (1) := SPSymbols.RWdigits;
               TerminalList (2) := SPSymbols.RWrange;
               TerminalList (3) := SPSymbols.RWprivate;
            when ExpMinusReal =>
               NoOfTerminals := 2;
               TerminalList (1) := SPSymbols.minus;
               TerminalList (2) := SPSymbols.real_number;
            when ExpMinusInteger =>
               NoOfTerminals := 2;
               TerminalList (1) := SPSymbols.minus;
               TerminalList (2) := SPSymbols.integer_number;
            when ExpIntegerReal =>
               NoOfTerminals := 2;
               TerminalList (1) := SPSymbols.integer_number;
               TerminalList (2) := SPSymbols.real_number;
         end case;
         --# assert True;
         case ParserState is
            when Config_Defn_Start
              | Next_Config_Defn_Start =>
               NextExpectedState := ParserState;
            when others =>
               NextExpectedState := Defn;
         end case;
         NewState (FoundSyntaxError);
         ErrorHandler.SyntaxError
           (ErrorItem => TheError.TheLexval,
            CurrentSym => TheError.TheSymbol,
            EntrySymbol => TheError.TheSymbol,
            NoOfTerminals => NoOfTerminals,
            NoOfNonTerminals => 0,
            TerminalList => TerminalList,
            NonTerminalList => TerminalList);
         ErrorsOccurred := True;
      end RaiseSyntaxError;

      procedure RaiseSemanticError (ErrNum : in Natural;
                                    LexVal : in LexTokenManager.LexValue)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.File_Sys;
      --#           out ErrorsOccurred;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.File_Sys         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrNum,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.StringTable,
      --#                                        LexVal,
      --#                                        SPARK_IO.File_Sys &
      --#         ErrorsOccurred            from ;
      is
      begin
         ErrorHandler.SemanticError (ErrNum,
                                     ErrorHandler.NoReference,
                                     LexVal.Position,
                                     LexVal.TokenStr);
         ErrorsOccurred := True;
      end RaiseSemanticError;


      ---------------------------------------------------------
      -- String matching routines; whole-string, prefix, suffix
      ---------------------------------------------------------

      function MatchPrefix (Prefix : IdentifierString;
                            ToMatch : EStrings.T)
                           return Boolean is
         M : EStrings.T;
         Matched : Boolean := True;
      begin
         if Prefix.Length > ToMatch.Length then
            Matched := False;
         else
            M := EStrings.LowerCase (ToMatch);
            for I in Integer range 1 .. Prefix.Length loop
               if M.Content (I) /= Prefix.Content (I) then
                  Matched := False;
               end if;
               exit when not Matched;
            end loop;
         end if;
         return Matched;
      end MatchPrefix;

      function MatchSuffix (Suffix : IdentifierString;
                            ToMatch : EStrings.T)
                           return Boolean is
         M : EStrings.T;
         Matched : Boolean := True;
      begin
         if Suffix.Length > ToMatch.Length then
            Matched := False;
         else
            M := EStrings.LowerCase (ToMatch);
            for I in Integer range 1 .. Suffix.Length loop
               if M.Content ((M.Length - I) + 1) /=
                 Suffix.Content ((Suffix.Length - I) + 1) then
                  Matched := False;
               end if;
               exit when not Matched;
            end loop;
         end if;
         return Matched;
      end MatchSuffix;

      function MatchIdentifier (ID : IdentifierString;
                                ToMatch : EStrings.T)
                               return Boolean is
         Matched : Boolean;
      begin
         if ToMatch.Length /= ID.Length then
            Matched := False;
         else
            Matched := MatchPrefix (ID, ToMatch);
         end if;
         return Matched;
      end MatchIdentifier;

      ----------------------------------------------------------
      -- Semantic check on validity of 'new' package declaration
      ----------------------------------------------------------

      procedure ChangeToNewPackage (PackageName : LexTokenManager.LexValue)
      --# global in     CommandLineData.Content;
      --#        in     CurrentDeclaration;
      --#        in     LexTokenManager.StringTable;
      --#        in     PackageNameDepth;
      --#        in     PredefinedScope;
      --#        in out CurrentScope;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out ErrorsOccurred;
      --#        in out LibPackageSymbol;
      --#        in out SPARK_IO.File_Sys;
      --#        in out ValidPackage;
      --# derives CurrentScope              from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.StringTable,
      --#                                        LibPackageSymbol,
      --#                                        PackageName,
      --#                                        PackageNameDepth,
      --#                                        PredefinedScope &
      --#         Dictionary.Dict           from *,
      --#                                        LexTokenManager.StringTable,
      --#                                        PackageName,
      --#                                        PackageNameDepth &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.File_Sys         from CommandLineData.Content,
      --#                                        CurrentDeclaration,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.StringTable,
      --#                                        LibPackageSymbol,
      --#                                        PackageName,
      --#                                        PackageNameDepth,
      --#                                        SPARK_IO.File_Sys,
      --#                                        ValidPackage &
      --#         ErrorsOccurred,
      --#         ValidPackage              from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.StringTable,
      --#                                        LibPackageSymbol,
      --#                                        PackageName,
      --#                                        PackageNameDepth,
      --#                                        ValidPackage &
      --#         LibPackageSymbol          from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.StringTable,
      --#                                        PackageName,
      --#                                        PackageNameDepth;

      is
         NameMatch          : Boolean := False;
         LocalPackageName   : EStrings.T;
         LocalPackageSymbol : Dictionary.Symbol;
         AdaPackageSymbol   : Dictionary.Symbol;
         BitOrderTypeSymbol : Dictionary.Symbol;
      begin
         -- Get the symbol for package Ada now since we need it more than once later.
         -- For SPARK83 mode, this will return NullSymbol, but that doesn't matter
         -- since we only use the symbol in code specific to SPARK95.
         AdaPackageSymbol := Dictionary.LookupItem (LexTokenManager.AdaToken,
                                                    Dictionary.GlobalScope,
                                                    ProofContext);
         LexTokenManager.LexStringToString (PackageName.TokenStr,
                                            LocalPackageName);
         case PackageNameDepth is
            when 1 => -- Checking Library packages
               for I in LibraryPackageRange loop
                  if MatchIdentifier (PossibleLibraryPackages (I), LocalPackageName) then
                     NameMatch := True;
                     case I is
                        when PackageStandard =>
                           CurrentScope := PredefinedScope;
                           LibPackageSymbol := Dictionary.GetPredefinedPackageStandard;
                           ValidPackage := True;
                        when PackageSystem =>
                           --# accept F, 41, "Stable expression expected here";
                           if not CommandLineData.Content.Spark95 then
                              -- package System disallowed in Spark83 mode
                              -- don't complain about the other declarations here;
                              -- wood for trees etc.
                              RaiseSemanticError (779, PackageName);
                           end if;
                           --# end accept;
                           LibPackageSymbol := Dictionary.LookupItem (PackageName.TokenStr,
                                                                      Dictionary.GlobalScope,
                                                                      ProofContext);
                           if LibPackageSymbol /= NullSymbol then
                              -- redeclaration of existing package
                              ValidPackage := False;
                              RaiseSemanticError (10, PackageName);
                           else
                              -- create package
                              Dictionary.AddPackage (PackageName.TokenStr,
                                                     Dictionary.Location'(PackageName.Position,
                                                                          PackageName.Position),
                                                     Dictionary.GlobalScope,
                                                     LibPackageSymbol);
                              CurrentScope := Dictionary.VisibleScope (LibPackageSymbol);

                              -- Add Bit_Order, its enumeration literals, and
                              -- System.Default_Bit_Order as a deferred constant
                              -- (for now...the actual value might be supplied and
                              --  overridden in the config file).
                              Dictionary.AddEnumerationType
                                (Name            => LexTokenManager.Bit_OrderToken,
                                 Declaration     => Dictionary.Location'(PackageName.Position,
                                                                         PackageName.Position),
                                 Scope           => CurrentScope,
                                 Context         => Dictionary.ProgramContext,
                                 EnumerationType => BitOrderTypeSymbol);
                              Dictionary.AddEnumerationLiteral
                                (Name     => LexTokenManager.High_Order_FirstToken,
                                 Position => LexTokenManager.ZeroValue,
                                 TheType  => BitOrderTypeSymbol);
                              Dictionary.AddEnumerationLiteral
                                (Name     => LexTokenManager.Low_Order_FirstToken,
                                 Position => LexTokenManager.OneValue,
                                 TheType  => BitOrderTypeSymbol);
                              Dictionary.AddDeferredConstant
                                (Name          => LexTokenManager.Default_Bit_OrderToken,
                                 TypeMark      => BitOrderTypeSymbol,
                                 TypeReference => Dictionary.Location'(PackageName.Position,
                                                                       PackageName.Position),
                                 Declaration   => Dictionary.Location'(PackageName.Position,
                                                                       PackageName.Position),
                                 ThePackage    => LibPackageSymbol);

                              ValidPackage := True;
                           end if;
                        when PackageAda =>
                           --# accept F, 41, "Stable expression expected here";
                           if not CommandLineData.Content.Spark95 then
                              -- package Ada disallowed in Spark83 mode
                              RaiseSemanticError (779, PackageName);
                              ValidPackage := False;
                           else
                              LibPackageSymbol := AdaPackageSymbol;
                              CurrentScope := Dictionary.VisibleScope (LibPackageSymbol);
                              ValidPackage := True;
                           end if;
                           --# end accept;
                     end case;
                  end if;
                  exit when NameMatch;
               end loop;
            when 2 => -- Checking Child packages
               if CommandLineData.Content.Spark95 then
                  for I in ChildPackageRange loop
                     if MatchIdentifier (PossibleChildPackages (I), LocalPackageName) then
                        NameMatch := True;
                        case I is
                           when PackageInterrupts =>
                              LocalPackageSymbol := Dictionary.LookupSelectedItem
                                (Prefix => LibPackageSymbol,
                                 Selector => LexTokenManager.InterruptsToken,
                                 Scope => Dictionary.GlobalScope,
                                 Context => ProofContext);

                           when PackageReal_Time =>
                              LocalPackageSymbol := Dictionary.LookupSelectedItem
                                (Prefix => LibPackageSymbol,
                                 Selector => LexTokenManager.Real_TimeToken,
                                 Scope => Dictionary.GlobalScope,
                                 Context => ProofContext);
                        end case;
                        if LocalPackageSymbol = NullSymbol then
                           ValidPackage := False;
                           --# accept F, 41, "Stable expression expected here";
                           if LibPackageSymbol /= AdaPackageSymbol then
                              -- The child package is a child of a package other than Ada, so
                              -- name is not recognised.
                              NameMatch := False;
                           end if;
                           --# end accept;
                        else
                           CurrentScope := Dictionary.VisibleScope (LocalPackageSymbol);
                           ValidPackage := True;
                        end if;
                     end if;
                     exit when NameMatch;
                  end loop;
               else
                  -- Child packages not supported in Ada83 mode. Flag the error at the
                  -- library package name token, unless we have done so already.
                  if ValidPackage then -- No error flagged yet, so do it now.
                     RaiseSemanticError (779, CurrentDeclaration.EncPackage (1));
                     ValidPackage := False;
                  end if;
                  -- Don't rub it in by giving a second error (776) for an unmatched name. This
                  -- might be a bogus error message anyway since the name may be OK in Ada95.
                  NameMatch := True;
               end if;
         end case;

         --# assert True;
         if not NameMatch then
            RaiseSemanticError (776, PackageName);
            ValidPackage := False;
         end if;
      end ChangeToNewPackage;


      procedure CheckPackageEnd
      --# global in     AnyPrioritySym;
      --#        in     CommandLineData.Content;
      --#        in     CurrentDeclaration;
      --#        in     Dictionary.Dict;
      --#        in     InterruptPrioritySym;
      --#        in     PackageEndName;
      --#        in     PackageFullName;
      --#        in     PackageNameDepth;
      --#        in     PrioritySym;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out ErrorsOccurred;
      --#        in out LexTokenManager.StringTable;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.File_Sys           from AnyPrioritySym,
      --#                                          CommandLineData.Content,
      --#                                          CurrentDeclaration,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          InterruptPrioritySym,
      --#                                          LexTokenManager.StringTable,
      --#                                          PackageEndName,
      --#                                          PackageFullName,
      --#                                          PackageNameDepth,
      --#                                          PrioritySym,
      --#                                          SPARK_IO.File_Sys &
      --#         ErrorsOccurred              from *,
      --#                                          AnyPrioritySym,
      --#                                          InterruptPrioritySym,
      --#                                          PackageEndName,
      --#                                          PackageFullName,
      --#                                          PrioritySym &
      --#         LexTokenManager.StringTable from *,
      --#                                          PackageEndName,
      --#                                          PackageFullName,
      --#                                          PackageNameDepth;
      is
         FullNameString : EStrings.Line;
         FullNameTokenStr : LexTokenManager.LexString;
      begin
         -- Check that the (dotted) name after "end" matches the package name.
         if not EStrings.EqString (PackageEndName, PackageFullName) then
            -- don't use normal error handler here; need to pass different string from
            -- position
            if PackageNameDepth = 1 then
               FullNameTokenStr := CurrentDeclaration.EncPackage (1).TokenStr;
            else
               -- We must construct a new lex string from the library and child package names
               FullNameString := EStrings.Line'(EStrings.Lengths => ' ');
               for I in Positive range 1 .. PackageFullName.Length loop
                  FullNameString (I) := PackageFullName.Content (I);
               end loop;
               LexTokenManager.InsertLexString (FullNameString,
                                                1,
                                                PackageFullName.Length,
                                                -- to get
                                                FullNameTokenStr);
            end if;
            ErrorHandler.SemanticError (58,
                                        ErrorHandler.NoReference,
                                        CurrentDeclaration.Name.Position,
                                        FullNameTokenStr);
            ErrorsOccurred := True;
         end if;

         -- Check that the full set of priority decls are given if we have Any_Priority
         if MatchIdentifier (PossibleLibraryPackages (PackageSystem),
                             PackageFullName) then
            if AnyPrioritySym /= Dictionary.NullSymbol and then
              (PrioritySym = Dictionary.NullSymbol or
               InterruptPrioritySym = Dictionary.NullSymbol) then
               RaiseSemanticError (770, CurrentDeclaration.Name);
            end if;
         end if;

      end CheckPackageEnd;


      ----------------------------------------------
      -- Check semantic validity of new declaration,
      -- and update dictionary as appropriate
      ----------------------------------------------

      procedure CheckAndProcessDeclaration (TheDecl : in Declaration)
      --# global in     CommandLineData.Content;
      --#        in     CurrentScope;
      --#        in     PackageFullName;
      --#        in     ValidPackage;
      --#        in out AnyPrioritySym;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out ErrorsOccurred;
      --#        in out InterruptPrioritySym;
      --#        in out LexTokenManager.StringTable;
      --#        in out PrioritySym;
      --#        in out SPARK_IO.File_Sys;
      --# derives AnyPrioritySym,
      --#         Dictionary.Dict,
      --#         InterruptPrioritySym,
      --#         PrioritySym                 from AnyPrioritySym,
      --#                                          CommandLineData.Content,
      --#                                          CurrentScope,
      --#                                          Dictionary.Dict,
      --#                                          InterruptPrioritySym,
      --#                                          LexTokenManager.StringTable,
      --#                                          PackageFullName,
      --#                                          PrioritySym,
      --#                                          TheDecl,
      --#                                          ValidPackage &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.File_Sys           from AnyPrioritySym,
      --#                                          CommandLineData.Content,
      --#                                          CurrentScope,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          InterruptPrioritySym,
      --#                                          LexTokenManager.StringTable,
      --#                                          PackageFullName,
      --#                                          PrioritySym,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TheDecl,
      --#                                          ValidPackage &
      --#         ErrorsOccurred,
      --#         LexTokenManager.StringTable from *,
      --#                                          AnyPrioritySym,
      --#                                          CurrentScope,
      --#                                          Dictionary.Dict,
      --#                                          InterruptPrioritySym,
      --#                                          LexTokenManager.StringTable,
      --#                                          PackageFullName,
      --#                                          PrioritySym,
      --#                                          TheDecl,
      --#                                          ValidPackage;

      is
         type Failures is (NotValidName, WrongTypeFound, EmptyRange,
                           NoFailure, NotWithinValidPackage,
                           InvalidRedeclaration, ParentTypeDoesNotExist,
                           SubTypeRangeMismatch, ParentTypeNotInteger,
                           ParentTypeNotAP, PriorityRangeInsufficient,
                           AnyPrioFirst, AnyPrioLast, PrioMidPoint,
                           MustBePrivate, NotPositivePowerOf2, BitOrderWrongType,
                           BitOrderWrongValue);

         NullLocation : constant Dictionary.Location :=
           Dictionary.Location'(LexTokenManager.TokenPosition'(0, 0),
                                LexTokenManager.TokenPosition'(0, 0));

         FailureReason        : Failures := NoFailure;
         DottedName,
         TempString           : EStrings.T;
         NameMatch            : Boolean := False;
         MatchType            : DeclarationType;
         ParentType           : Dictionary.Symbol;
         IsAnyPriority        : Boolean := False;
         IsPriority           : Boolean := False;
         IsInterruptPriority  : Boolean := False;
         IsAddress            : Boolean := False;
         IsMaxBinaryModulus   : Boolean := False;
         IsPredefInteger      : Boolean := False;
         IsPredefLongInteger  : Boolean := False;
         IsPredefFloat        : Boolean := False;
         IsPredefLongFloat    : Boolean := False;
         IsSecondsCount       : Boolean := False;
         IsInterruptID        : Boolean := False;
         IsDefaultBitOrder    : Boolean := False;

         procedure CheckPriority
         --# global in     AnyPrioritySym;
         --#        in     Dictionary.Dict;
         --#        in     InterruptPrioritySym;
         --#        in     LexTokenManager.StringTable;
         --#        in     ParentType;
         --#        in     TheDecl;
         --#        in out FailureReason;
         --# derives FailureReason from *,
         --#                            AnyPrioritySym,
         --#                            Dictionary.Dict,
         --#                            InterruptPrioritySym,
         --#                            LexTokenManager.StringTable,
         --#                            ParentType,
         --#                            TheDecl;
         is
            HighMinusLow     : Maths.Value;
            LastPlusOne      : Maths.Value;
            BooleanCondition : Maths.Value;
            UnwantedME       : Maths.ErrorCode;
         begin
            --# accept F, 10, UnwantedME, "UnwantedME unused here";
            -- check parent type is Any_Priority
            if ParentType /= AnyPrioritySym then
               FailureReason := ParentTypeNotAP;
            else
               -- check range of at least 30 values - LRM D.1(26)
               Maths.Subtract (TheDecl.HighBound,
                               TheDecl.LowBound,
                               HighMinusLow,
                               UnwantedME);
               Maths.GreaterOrEqual (HighMinusLow,
                                     Maths.IntegerToValue (29),
                                     BooleanCondition,
                                     UnwantedME);
               -- ineffective assignment to UnwantedME expected here
               -- no possible error; lexical analysis confirms correct typing
               if BooleanCondition = Maths.BoolToValue (False) then
                  FailureReason := PriorityRangeInsufficient;
               else
                  -- check Priority'First = Any_Priority'First
                  if Maths.ValueRep (Dictionary.GetScalarAttributeValue (False,
                                                                         LexTokenManager.FirstToken,
                                                                         AnyPrioritySym)) /=
                    TheDecl.LowBound then
                     FailureReason := AnyPrioFirst;
                  else
                     if InterruptPrioritySym /= Dictionary.NullSymbol then
                        -- check that Priority'Last + 1 = Interrupt_Priority'First;
                        LastPlusOne := TheDecl.HighBound;
                        Maths.SuccOp (LastPlusOne, UnwantedME);
                        -- ineffective assignment to UnwantedME expected here
                        -- no possible error; lexical analysis confirms correct typing
                        if Maths.ValueRep (Dictionary.GetScalarAttributeValue (False,
                                                                               LexTokenManager.FirstToken,
                                                                               InterruptPrioritySym)) /=
                          LastPlusOne then
                           FailureReason := PrioMidPoint;
                        else
                           -- semantic checking complete
                           null;
                        end if;
                     else
                        -- semantic checking complete
                        null;
                     end if;
                  end if;
               end if;
            end if;
            --# accept F, 33, UnwantedME, "UnwantedME unused here";
         end CheckPriority;

         procedure CheckInterruptPriority
         --# global in     AnyPrioritySym;
         --#        in     Dictionary.Dict;
         --#        in     LexTokenManager.StringTable;
         --#        in     ParentType;
         --#        in     PrioritySym;
         --#        in     TheDecl;
         --#        in out FailureReason;
         --# derives FailureReason from *,
         --#                            AnyPrioritySym,
         --#                            Dictionary.Dict,
         --#                            LexTokenManager.StringTable,
         --#                            ParentType,
         --#                            PrioritySym,
         --#                            TheDecl;
         is
            LastPlusOne : Maths.Value;
            UnwantedME  : Maths.ErrorCode;
         begin
            --# accept F, 10, UnwantedME, "UnwantedME unused here";
            -- check parent type is Any_Priority
            if ParentType /= AnyPrioritySym then
               FailureReason := ParentTypeNotAP;
            else
               -- there must be at least one value (LRM D.1(25)), but this is
               -- checked by the bounds check in the general semantic checking
               -- for a subtype...

               -- check Interrupt_Priority'Last = Any_Priority'Last
               if TheDecl.HighBound /=
                 Maths.ValueRep (Dictionary.GetScalarAttributeValue (False,
                                                                     LexTokenManager.LastToken,
                                                                     AnyPrioritySym)) then
                  -- some other semantic error
                  FailureReason := AnyPrioLast;
               else
                  if PrioritySym /= Dictionary.NullSymbol then
                     -- check that Priority'Last + 1 = Interrupt_Priority'First;
                     LastPlusOne :=
                       Maths.ValueRep (Dictionary.GetScalarAttributeValue (False,
                                                                           LexTokenManager.LastToken,
                                                                           PrioritySym));
                     Maths.SuccOp (LastPlusOne, UnwantedME);
                     -- ineffective assignment to UnwantedME expected here
                     -- no possible error; lexical analysis confirms correct typing
                     if TheDecl.LowBound /= LastPlusOne then
                        -- some other semantic error
                        FailureReason := PrioMidPoint;
                     else
                        -- semantic checking complete
                        null;
                     end if;
                  else
                     -- semantic checking complete
                     null;
                  end if;
               end if;
            end if;
            --# accept F, 33, UnwantedME, "UnwantedME unused here";
         end CheckInterruptPriority;

         function CkFailed return Boolean
         --# global in FailureReason;
         is
         begin
            return FailureReason /= NoFailure;
         end CkFailed;

         procedure DoCkValidPackage
         --# global in     ValidPackage;
         --#        in out FailureReason;
         --# derives FailureReason from *,
         --#                            ValidPackage;
         is
         begin
            if not ValidPackage then
               FailureReason := NotWithinValidPackage;
            end if;
         end DoCkValidPackage;

         procedure DoCkSuffixName
         --# global in     DottedName;
         --#        in     FailureReason;
         --#        in out IsPredefLongFloat;
         --#        in out IsPredefLongInteger;
         --#        in out MatchType;
         --#        in out NameMatch;
         --# derives IsPredefLongFloat,
         --#         IsPredefLongInteger,
         --#         MatchType,
         --#         NameMatch           from *,
         --#                                  DottedName,
         --#                                  FailureReason;
         is
         begin
            if not CkFailed then
               if MatchPrefix (StandardPrefix, DottedName) then
                  if MatchSuffix (IntegerSuffix, DottedName) then
                     NameMatch := True;
                     MatchType := DecIntegerType;

                     -- We need to spot "Long_Integer" here since it's
                     -- predefined by Dictionary.Initialize
                     IsPredefLongInteger := MatchIdentifier
                       (PossibleIdentifiers (PredefLongInteger),
                        DottedName);

                  elsif MatchSuffix (FloatSuffix, DottedName) then
                     NameMatch := True;
                     MatchType := DecFloatingPointType;

                     -- We need to spot "Long_Float" here since it's
                     -- predefined by Dictionary.Initialize
                     IsPredefLongFloat := MatchIdentifier
                       (PossibleIdentifiers (PredefLongFloat),
                        DottedName);
                  end if;
               end if;
            end if;
         end DoCkSuffixName;

         procedure DoCkDefNameWithPriorityCk
         --# global in     DottedName;
         --#        in     NameMatch;
         --#        in out FailureReason;
         --#        in out IsAnyPriority;
         --#        in out IsInterruptPriority;
         --#        in out IsPriority;
         --#        in out MatchType;
         --# derives FailureReason,
         --#         IsAnyPriority,
         --#         IsInterruptPriority,
         --#         IsPriority,
         --#         MatchType           from *,
         --#                                  DottedName,
         --#                                  FailureReason,
         --#                                  NameMatch;
         is
            LocalNameMatch : Boolean := False;
         begin
            if not (NameMatch or CkFailed) then
               FailureReason := NotValidName; -- initially
               for I in PossibleIdentifierRange loop
                  if MatchIdentifier (PossibleIdentifiers (I),
                                      DottedName) then
                     FailureReason := NoFailure;
                     LocalNameMatch := False;
                     case I is
                        when AnyPrioritySubtype =>
                           IsAnyPriority := True;
                        when PrioritySubtype =>
                           IsPriority := True;
                        when InterruptPrioritySubtype =>
                           IsInterruptPriority := True;
                        when others =>
                           null;
                     end case;
                     MatchType := IdentifierTypes (I);
                  end if;
                  exit when LocalNameMatch;
               end loop;
            end if;
         end DoCkDefNameWithPriorityCk;

         procedure DoCkDefName
         --# global in     DottedName;
         --#        in     NameMatch;
         --#        in out FailureReason;
         --#        in out IsAddress;
         --#        in out IsDefaultBitOrder;
         --#        in out IsInterruptID;
         --#        in out IsMaxBinaryModulus;
         --#        in out IsPredefFloat;
         --#        in out IsPredefInteger;
         --#        in out IsSecondsCount;
         --#        in out MatchType;
         --# derives FailureReason,
         --#         IsAddress,
         --#         IsDefaultBitOrder,
         --#         IsInterruptID,
         --#         IsMaxBinaryModulus,
         --#         IsPredefFloat,
         --#         IsPredefInteger,
         --#         IsSecondsCount,
         --#         MatchType          from *,
         --#                                 DottedName,
         --#                                 FailureReason,
         --#                                 NameMatch;
         is
         begin
            if not (NameMatch or CkFailed) then
               FailureReason := NotValidName; -- initially
               for I in PossibleIdentifierRange loop
                  if MatchIdentifier (PossibleIdentifiers (I),
                                      DottedName) then
                     IsAddress          := I = SystemAddress;
                     IsMaxBinaryModulus := I = MaxBinaryModulus;
                     IsPredefInteger    := I = PredefInteger;
                     IsPredefFloat      := I = PredefFloat;
                     IsSecondsCount     := I = SecondsCount;
                     IsInterruptID      := I = InterruptID;
                     IsDefaultBitOrder  := I = DefaultBitOrder;
                     FailureReason      := NoFailure;
                     MatchType := IdentifierTypes (I);
                     exit;
                  end if;
               end loop;
            end if;
         end DoCkDefName;

         procedure DoCkExistingDecl
         --# global in     CurrentScope;
         --#        in     Dictionary.Dict;
         --#        in     IsDefaultBitOrder;
         --#        in     IsInterruptID;
         --#        in     IsSecondsCount;
         --#        in     LexTokenManager.StringTable;
         --#        in     TheDecl;
         --#        in out FailureReason;
         --# derives FailureReason from *,
         --#                            CurrentScope,
         --#                            Dictionary.Dict,
         --#                            IsDefaultBitOrder,
         --#                            IsInterruptID,
         --#                            IsSecondsCount,
         --#                            LexTokenManager.StringTable,
         --#                            TheDecl;
         is
            CheckSymbol : Dictionary.Symbol;
         begin
            if not CkFailed then
               CheckSymbol := Dictionary.LookupItem (TheDecl.Name.TokenStr,
                                                     CurrentScope,
                                                     ProgramContext);
               if CheckSymbol /= NullSymbol then
                  -- redeclaration of existing object ... but
                  if CheckSymbol = Dictionary.GetPredefinedIntegerType or else
                    CheckSymbol = Dictionary.GetPredefinedLongIntegerType or else
                    CheckSymbol = Dictionary.GetPredefinedFloatType or else
                    CheckSymbol = Dictionary.GetPredefinedLongFloatType or else
                    IsDefaultBitOrder or else
                    IsSecondsCount or else
                    IsInterruptID then

                     if IsDefaultBitOrder then
                        if not Dictionary.IsDeferredConstant (CheckSymbol) then
                           FailureReason := InvalidRedeclaration;
                        end if;
                     else
                        -- Must be a numeric constant
                        if (Maths.HasNoValue
                              (Maths.ValueRep (Dictionary.GetScalarAttributeValue (False,
                                                                                   LexTokenManager.FirstToken,
                                                                                   CheckSymbol)))) then
                           -- OK, not actually a redeclaration; initialised by Dictionary
                           null;
                        else
                           FailureReason := InvalidRedeclaration;
                        end if;
                     end if;

                  else
                     FailureReason := InvalidRedeclaration;
                  end if;
               end if;
            end if;
         end DoCkExistingDecl;

         procedure DoCkType
         --# global in     MatchType;
         --#        in     TheDecl;
         --#        in out FailureReason;
         --# derives FailureReason from *,
         --#                            MatchType,
         --#                            TheDecl;
         is
         begin
            if not CkFailed then
               if MatchType /= TheDecl.MyType then
                  if MatchType = DecPrivate then
                     FailureReason := MustBePrivate;
                  else
                     FailureReason := WrongTypeFound;
                  end if;
               end if;
            end if;

         end DoCkType;

         procedure DoCkBounds
         --# global in     TheDecl;
         --#        in out FailureReason;
         --# derives FailureReason from *,
         --#                            TheDecl;
         is
            TempValue : Maths.Value;
            LocalME   : Maths.ErrorCode;
         begin
            --# accept F, 10, LocalME, "LocalME unused here";
            if not CkFailed then
               Maths.Greater (TheDecl.LowBound,
                              TheDecl.HighBound,
                              TempValue,
                              LocalME);
               -- ineffective assignment to LocalME expected here
               -- no possible error; lexical analysis confirms correct typing

               if TempValue = Maths.BoolToValue (True) then
                  FailureReason := EmptyRange;
               end if;
            end if;
            --# accept F, 33, LocalME, "LocalME unused here";
         end DoCkBounds;

         procedure DoCkBaseType
         --# global in     CurrentScope;
         --#        in     Dictionary.Dict;
         --#        in     LexTokenManager.StringTable;
         --#        in     TheDecl;
         --#        in out FailureReason;
         --#        in out ParentType;
         --# derives FailureReason,
         --#         ParentType    from *,
         --#                            CurrentScope,
         --#                            Dictionary.Dict,
         --#                            FailureReason,
         --#                            LexTokenManager.StringTable,
         --#                            TheDecl;
         is
            CheckSymbol : Dictionary.Symbol;
            TempValue   : Maths.Value;
            LocalME     : Maths.ErrorCode;
         begin
            --# accept F, 10, LocalME, "LocalME unused here";
            if not CkFailed then
               CheckSymbol := Dictionary.LookupItem (TheDecl.BaseType.TokenStr,
                                                     CurrentScope,
                                                     ProgramContext);
               if CheckSymbol = NullSymbol then
                  -- parent type does not exist
                  FailureReason := ParentTypeDoesNotExist;
               else
                  Maths.Lesser (TheDecl.LowBound,
                                Maths.ValueRep (Dictionary.GetScalarAttributeValue (False,
                                                                                    LexTokenManager.FirstToken,
                                                                                    CheckSymbol)),
                                TempValue,
                                LocalME);
                  -- ineffective assignment to LocalME expected here
                  -- no possible error; lexical analysis confirms correct typing

                  if TempValue = Maths.BoolToValue (True) then
                     -- type'first less than type'base'first
                     FailureReason := SubTypeRangeMismatch;
                  else
                     Maths.Greater (TheDecl.HighBound,
                                    Maths.ValueRep (Dictionary.GetScalarAttributeValue (False,
                                                                                        LexTokenManager.LastToken,
                                                                                        CheckSymbol)),
                                    TempValue,
                                    LocalME);
                     -- ineffective assignment to LocalME expected here
                     -- no possible error; lexical analysis confirms correct typing
                     if TempValue = Maths.BoolToValue (True) then
                        -- type'last greater than type'base'last
                        FailureReason := SubTypeRangeMismatch;
                     else
                        ParentType := CheckSymbol;
                     end if;
                  end if;
               end if;
            end if;
            --# accept F, 33, LocalME, "LocalME unused here";
         end DoCkBaseType;

         procedure DoCkSystemPriority
         --# global in     AnyPrioritySym;
         --#        in     Dictionary.Dict;
         --#        in     InterruptPrioritySym;
         --#        in     IsAnyPriority;
         --#        in     IsInterruptPriority;
         --#        in     IsPriority;
         --#        in     LexTokenManager.StringTable;
         --#        in     ParentType;
         --#        in     PrioritySym;
         --#        in     TheDecl;
         --#        in out FailureReason;
         --# derives FailureReason from *,
         --#                            AnyPrioritySym,
         --#                            Dictionary.Dict,
         --#                            InterruptPrioritySym,
         --#                            IsAnyPriority,
         --#                            IsInterruptPriority,
         --#                            IsPriority,
         --#                            LexTokenManager.StringTable,
         --#                            ParentType,
         --#                            PrioritySym,
         --#                            TheDecl;
         is
         begin
            if not CkFailed then
               if IsAnyPriority then
                  -- check parent type is Integer
                  if ParentType /= Dictionary.GetPredefinedIntegerType then
                     FailureReason := ParentTypeNotInteger;
                  end if;
               elsif IsPriority then
                  CheckPriority;
               elsif IsInterruptPriority then
                  CheckInterruptPriority;
               end if;
            end if;
         end DoCkSystemPriority;

         procedure DoCkSpecialCases
         --# global in     IsDefaultBitOrder;
         --#        in     IsMaxBinaryModulus;
         --#        in     TheDecl;
         --#        in out FailureReason;
         --# derives FailureReason from *,
         --#                            IsDefaultBitOrder,
         --#                            IsMaxBinaryModulus,
         --#                            TheDecl;
         is
         begin
            if not CkFailed then
               -- Max_Binary_Modulus must be a positive power of 2
               if IsMaxBinaryModulus then

                  if not Maths.IsAPositivePowerOf2 (TheDecl.Value) then
                     FailureReason := NotPositivePowerOf2;
                  end if;

               elsif IsDefaultBitOrder then

                  if TheDecl.BaseType.TokenStr /= LexTokenManager.Bit_OrderToken then
                     FailureReason := BitOrderWrongType;
                  else
                     if TheDecl.ValueStr.TokenStr /= LexTokenManager.Low_Order_FirstToken and
                       TheDecl.ValueStr.TokenStr /= LexTokenManager.High_Order_FirstToken then
                        FailureReason := BitOrderWrongValue;
                     end if;
                  end if;

               end if;
            end if;
         end DoCkSpecialCases;


         procedure RaiseErrorsAndUpdateDictionary
         --# global in     CommandLineData.Content;
         --#        in     CurrentScope;
         --#        in     FailureReason;
         --#        in     IsAddress;
         --#        in     IsDefaultBitOrder;
         --#        in     IsInterruptID;
         --#        in     IsPredefFloat;
         --#        in     IsPredefInteger;
         --#        in     IsPredefLongFloat;
         --#        in     IsPredefLongInteger;
         --#        in     IsPriority;
         --#        in     IsSecondsCount;
         --#        in     ParentType;
         --#        in     TheDecl;
         --#        in out Dictionary.Dict;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out ErrorsOccurred;
         --#        in out LexTokenManager.StringTable;
         --#        in out SPARK_IO.File_Sys;
         --# derives Dictionary.Dict             from *,
         --#                                          CommandLineData.Content,
         --#                                          CurrentScope,
         --#                                          FailureReason,
         --#                                          IsAddress,
         --#                                          IsDefaultBitOrder,
         --#                                          IsInterruptID,
         --#                                          IsPredefFloat,
         --#                                          IsPredefInteger,
         --#                                          IsPredefLongFloat,
         --#                                          IsPredefLongInteger,
         --#                                          IsPriority,
         --#                                          IsSecondsCount,
         --#                                          LexTokenManager.StringTable,
         --#                                          ParentType,
         --#                                          TheDecl &
         --#         ErrorHandler.ErrorContext   from *,
         --#                                          CommandLineData.Content,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          FailureReason,
         --#                                          IsAddress,
         --#                                          IsPriority,
         --#                                          LexTokenManager.StringTable,
         --#                                          ParentType,
         --#                                          SPARK_IO.File_Sys,
         --#                                          TheDecl &
         --#         ErrorsOccurred              from *,
         --#                                          FailureReason &
         --#         LexTokenManager.StringTable from *,
         --#                                          FailureReason,
         --#                                          IsInterruptID,
         --#                                          IsPredefInteger,
         --#                                          IsPredefLongInteger,
         --#                                          IsPriority,
         --#                                          IsSecondsCount,
         --#                                          TheDecl &
         --#         SPARK_IO.File_Sys           from *,
         --#                                          CommandLineData.Content,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          FailureReason,
         --#                                          IsAddress,
         --#                                          IsInterruptID,
         --#                                          IsPredefFloat,
         --#                                          IsPredefInteger,
         --#                                          IsPredefLongFloat,
         --#                                          IsPredefLongInteger,
         --#                                          IsPriority,
         --#                                          IsSecondsCount,
         --#                                          LexTokenManager.StringTable,
         --#                                          ParentType,
         --#                                          TheDecl;
         is
            StorageRepValue,
            StorageRepLowBound,
            StorageRepHighBound,
            StorageRepDigits     : LexTokenManager.LexString;
            IntegerType          : Dictionary.Symbol;
            FloatType            : Dictionary.Symbol;
            LibPackageSym        : Dictionary.Symbol;
            ConstantSym          : Dictionary.Symbol;

            procedure AddNullAddress
            --# global in     CommandLineData.Content;
            --#        in     CurrentScope;
            --#        in     LexTokenManager.StringTable;
            --#        in     TheDecl;
            --#        in out Dictionary.Dict;
            --#        in out ErrorHandler.ErrorContext;
            --#        in out SPARK_IO.File_Sys;
            --# derives Dictionary.Dict           from *,
            --#                                        CurrentScope,
            --#                                        TheDecl &
            --#         ErrorHandler.ErrorContext,
            --#         SPARK_IO.File_Sys         from CommandLineData.Content,
            --#                                        CurrentScope,
            --#                                        Dictionary.Dict,
            --#                                        ErrorHandler.ErrorContext,
            --#                                        LexTokenManager.StringTable,
            --#                                        SPARK_IO.File_Sys,
            --#                                        TheDecl;
            is
            begin
               Dictionary.AddDeferredConstant
                 (LexTokenManager.Null_AddressToken,
                  Dictionary.LookupItem (TheDecl.Name.TokenStr,
                                         CurrentScope,
                                         ProgramContext),
                  Dictionary.Location'(TheDecl.Name.Position,
                                       TheDecl.Name.Position),
                  Dictionary.Location'(TheDecl.Name.Position,
                                       TheDecl.Name.Position),
                  Dictionary.GetRegion (CurrentScope));
               -- note to indicate implicit definition
               ErrorHandler.SemanticNote (3,
                                          TheDecl.Name.Position,
                                          TheDecl.Name.TokenStr);
            end AddNullAddress;

            procedure AddDefaultPriority
            --# global in     CommandLineData.Content;
            --#        in     CurrentScope;
            --#        in     TheDecl;
            --#        in out Dictionary.Dict;
            --#        in out ErrorHandler.ErrorContext;
            --#        in out LexTokenManager.StringTable;
            --#        in out SPARK_IO.File_Sys;
            --# derives Dictionary.Dict             from *,
            --#                                          CommandLineData.Content,
            --#                                          CurrentScope,
            --#                                          LexTokenManager.StringTable,
            --#                                          TheDecl &
            --#         ErrorHandler.ErrorContext,
            --#         SPARK_IO.File_Sys           from CommandLineData.Content,
            --#                                          CurrentScope,
            --#                                          Dictionary.Dict,
            --#                                          ErrorHandler.ErrorContext,
            --#                                          LexTokenManager.StringTable,
            --#                                          SPARK_IO.File_Sys,
            --#                                          TheDecl &
            --#         LexTokenManager.StringTable from *,
            --#                                          TheDecl;
            is
               TempVal1, TempVal2 : Maths.Value;
               UnwantedME         : Maths.ErrorCode;
               StorageRepAve      : LexTokenManager.LexString;
            begin
               --# accept F, 10, UnwantedME, "UnwantedME unused here";
               Maths.Add (TheDecl.LowBound,
                          TheDecl.HighBound,
                          TempVal1,
                          UnwantedME);
               Maths.Divide (TempVal1,
                             Maths.IntegerToValue (2),
                             TempVal2,
                             UnwantedME);
               Maths.StorageRep (TempVal2, StorageRepAve);
               Dictionary.AddScalarConstant
                 (LexTokenManager.Default_PriorityToken,
                  Dictionary.LookupItem (TheDecl.Name.TokenStr,
                                         CurrentScope,
                                         ProgramContext),
                  Dictionary.Location'(TheDecl.Name.Position,
                                       TheDecl.Name.Position),
                  StorageRepAve,
                  True,
                  ExaminerConstants.RefType (0),
                  True,
                  Dictionary.Location'(TheDecl.Name.Position,
                                       TheDecl.Name.Position),
                  CurrentScope,
                  ProgramContext);
               -- note to indicate implicit definition
               ErrorHandler.SemanticNote (4,
                                          TheDecl.Name.Position,
                                          TheDecl.Name.TokenStr);
               --# accept F, 33, UnwantedME, "UnwantedME unused here";
            end AddDefaultPriority;

         begin
            case FailureReason is
               when InvalidRedeclaration =>
                  RaiseSemanticError (10, TheDecl.Name);
               when WrongTypeFound =>
                  RaiseSemanticError (42, TheDecl.Name);
               when EmptyRange =>
                  RaiseSemanticError (409, TheDecl.Name);
               when SubTypeRangeMismatch =>
                  RaiseSemanticError (413, TheDecl.Name);
               when NotValidName | NotWithinValidPackage =>
                  RaiseSemanticError (778, TheDecl.Name);
               when ParentTypeDoesNotExist =>
                  RaiseSemanticError (63, TheDecl.BaseType);
               when ParentTypeNotAP =>
                  RaiseSemanticError (771, TheDecl.BaseType);
               when PriorityRangeInsufficient =>
                  RaiseSemanticError (772, TheDecl.BaseType);
               when AnyPrioFirst =>
                  RaiseSemanticError (773, TheDecl.BaseType);
               when AnyPrioLast =>
                  RaiseSemanticError (775, TheDecl.BaseType);
               when PrioMidPoint =>
                  RaiseSemanticError (774, TheDecl.BaseType);
               when ParentTypeNotInteger =>
                  RaiseSemanticError (777, TheDecl.BaseType);
               when MustBePrivate =>
                  RaiseSemanticError (780, TheDecl.Name);
               when NotPositivePowerOf2 =>
                  RaiseSemanticError (784, TheDecl.Name);
               when BitOrderWrongType =>
                  RaiseSemanticError (814, TheDecl.BaseType);
               when BitOrderWrongValue =>
                  RaiseSemanticError (815, TheDecl.ValueStr);

               when NoFailure =>
                  case TheDecl.MyType is
                     when DecNamedInteger =>
                        Maths.StorageRep (TheDecl.Value, StorageRepValue);
                        Dictionary.AddScalarConstant (TheDecl.Name.TokenStr,
                                                      Dictionary.GetUniversalIntegerType,
                                                      NullLocation,
                                                      StorageRepValue,
                                                      True,
                                                      ExaminerConstants.RefType (0),
                                                      True,
                                                      Dictionary.Location'(TheDecl.Name.Position,
                                                                           TheDecl.Name.Position),
                                                      CurrentScope,
                                                      ProgramContext);
                     when DecNamedReal =>
                        Maths.StorageRep (TheDecl.Value, StorageRepValue);
                        Dictionary.AddScalarConstant (TheDecl.Name.TokenStr,
                                                      Dictionary.GetUniversalRealType,
                                                      NullLocation,
                                                      StorageRepValue,
                                                      True,
                                                      ExaminerConstants.RefType (0),
                                                      True,
                                                      Dictionary.Location'(TheDecl.Name.Position,
                                                                           TheDecl.Name.Position),
                                                      CurrentScope,
                                                      ProgramContext);
                     when DecIntegerSubtype =>
                        Maths.StorageRep (TheDecl.LowBound, StorageRepLowBound);
                        Maths.StorageRep (TheDecl.HighBound, StorageRepHighBound);
                        Dictionary.AddIntegerSubtype (TheDecl.Name.TokenStr,
                                                      True,
                                                      ParentType,
                                                      NullLocation,
                                                      StorageRepLowBound,
                                                      StorageRepHighBound,
                                                      Dictionary.Location'(TheDecl.Name.Position,
                                                                           TheDecl.Name.Position),
                                                      CurrentScope,
                                                      ProgramContext);
                        if IsPriority then
                           AddDefaultPriority;
                        end if;
                     when DecIntegerType =>
                        Maths.StorageRep (TheDecl.LowBound, StorageRepLowBound);
                        Maths.StorageRep (TheDecl.HighBound, StorageRepHighBound);

                        if IsPredefLongInteger or IsInterruptID or IsSecondsCount then
                           -- An integer type which is pre-declared by
                           -- Dictionary.Initialize, and has no dependent
                           -- subtypes, so simply override the lower and upper bounds
                           IntegerType := Dictionary.LookupItem
                             (TheDecl.Name.TokenStr,
                              CurrentScope,
                              ProgramContext);
                           Dictionary.AdjustTypeLowerBound
                             (TypeMark => IntegerType,
                              NewBound => StorageRepLowBound);
                           Dictionary.AdjustTypeUpperBound
                             (TypeMark => IntegerType,
                              NewBound => StorageRepHighBound);
                        elsif IsPredefInteger then
                           -- Update the bounds for types Stbandard.Integer,
                           -- Standard.Natural and Standard.Positive in the dictionary
                           IntegerType := Dictionary.LookupItem
                             (TheDecl.Name.TokenStr,
                              CurrentScope,
                              ProgramContext);
                           Dictionary.AdjustTypeLowerBound
                             (TypeMark => IntegerType,
                              NewBound => StorageRepLowBound);
                           Dictionary.AdjustTypeUpperBound
                             (TypeMark => IntegerType,
                              NewBound => StorageRepHighBound);
                           Maths.StorageRep (Maths.ZeroInteger, StorageRepLowBound);
                           Dictionary.AdjustTypeLowerBound
                             (Dictionary.GetPredefinedNaturalSubtype,
                              StorageRepLowBound);
                           Dictionary.AdjustTypeUpperBound
                             (Dictionary.GetPredefinedNaturalSubtype,
                              StorageRepHighBound);
                           Maths.StorageRep (Maths.OneInteger, StorageRepLowBound);
                           Dictionary.AdjustTypeLowerBound
                             (Dictionary.GetPredefinedPositiveSubtype,
                              StorageRepLowBound);
                           Dictionary.AdjustTypeUpperBound
                             (Dictionary.GetPredefinedPositiveSubtype,
                              StorageRepHighBound);
                        else
                           -- A new predefined integer type which _isn't_
                           -- pre-declared by Dictionary.Initialize.
                           -- For example: Short_Integer or Long_Long_Integer
                           Dictionary.AddPreDefIntegerType (TheDecl.Name.TokenStr,
                                                            Dictionary.Location'(TheDecl.Name.Position,
                                                                                 TheDecl.Name.Position),
                                                            StorageRepLowBound,
                                                            StorageRepHighBound,
                                                            CurrentScope,
                                                            ProgramContext);
                        end if;


                     when DecFloatingPointType =>
                        Maths.StorageRep (TheDecl.LowBound, StorageRepLowBound);
                        Maths.StorageRep (TheDecl.HighBound, StorageRepHighBound);
                        Maths.StorageRep (TheDecl.NumDigits, StorageRepDigits);

                        if IsPredefFloat or IsPredefLongFloat then
                           -- A Floating-point type which is pre-declared by
                           -- Dictionary.Initialize, and has no dependent
                           -- subtypes, so simply override the accuracy,
                           -- lower and upper bounds
                           FloatType := Dictionary.LookupItem
                             (TheDecl.Name.TokenStr,
                              CurrentScope,
                              ProgramContext);
                           Dictionary.AdjustTypeLowerBound
                             (TypeMark => FloatType,
                              NewBound => StorageRepLowBound);
                           Dictionary.AdjustTypeUpperBound
                             (TypeMark => FloatType,
                              NewBound => StorageRepHighBound);
                           Dictionary.AdjustTypeErrorBound
                             (TypeMark => FloatType,
                              NewBound => StorageRepDigits);

                        else
                           -- A Floating-point type, which is _not_ pre-declared
                           -- by Dictionary.Initialize - e.g. Short_Float or
                           -- Long_Long_Float
                           Dictionary.AddPreDefFloatingPointType (TheDecl.Name.TokenStr,
                                                                  Dictionary.Location'(TheDecl.Name.Position,
                                                                                       TheDecl.Name.Position),
                                                                  StorageRepLowBound,
                                                                  StorageRepHighBound,
                                                                  StorageRepDigits,
                                                                  CurrentScope,
                                                                  ProgramContext);
                        end if;

                     when DecPrivate =>
                        Dictionary.AddPrivateType (TheDecl.Name.TokenStr,
                                                   Dictionary.Location'(TheDecl.Name.Position,
                                                                        TheDecl.Name.Position),
                                                   Dictionary.GetRegion (CurrentScope),
                                                   False, False, False, Dictionary.NullSymbol);
                        if IsAddress then
                           AddNullAddress;
                        end if;
                     when DecTypedConstant =>
                        if IsDefaultBitOrder then

                           LibPackageSym := Dictionary.LookupItem
                             (LexTokenManager.SystemToken,
                              Dictionary.GlobalScope,
                              ProgramContext);

                           ConstantSym := Dictionary.LookupItem
                             (TheDecl.Name.TokenStr,
                              CurrentScope,
                              ProgramContext);

                           -- In the Dictionary, we need to store the 'Pos of the
                           -- value, so...
                           if TheDecl.ValueStr.TokenStr = LexTokenManager.High_Order_FirstToken then
                              StorageRepValue := LexTokenManager.ZeroValue;
                           else
                              StorageRepValue := LexTokenManager.OneValue;
                           end if;

                           Dictionary.PromoteDeferredToFullConstant
                             (TheConstant => ConstantSym,
                              Value       => StorageRepValue,
                              ExpNode     => ExaminerConstants.RefType (0),
                              ThePackage  => LibPackageSym);

                        end if;
                  end case;
            end case;
         end RaiseErrorsAndUpdateDictionary;

         procedure UpdatePrioritySymbols
         --# global in     CurrentScope;
         --#        in     Dictionary.Dict;
         --#        in     FailureReason;
         --#        in     IsAnyPriority;
         --#        in     IsInterruptPriority;
         --#        in     IsPriority;
         --#        in     TheDecl;
         --#        in out AnyPrioritySym;
         --#        in out InterruptPrioritySym;
         --#        in out PrioritySym;
         --# derives AnyPrioritySym       from *,
         --#                                   CurrentScope,
         --#                                   Dictionary.Dict,
         --#                                   FailureReason,
         --#                                   IsAnyPriority,
         --#                                   TheDecl &
         --#         InterruptPrioritySym from *,
         --#                                   CurrentScope,
         --#                                   Dictionary.Dict,
         --#                                   FailureReason,
         --#                                   IsAnyPriority,
         --#                                   IsInterruptPriority,
         --#                                   IsPriority,
         --#                                   TheDecl &
         --#         PrioritySym          from *,
         --#                                   CurrentScope,
         --#                                   Dictionary.Dict,
         --#                                   FailureReason,
         --#                                   IsAnyPriority,
         --#                                   IsPriority,
         --#                                   TheDecl;
         is
            TempSym : Dictionary.Symbol;
         begin
            if not CkFailed then
               TempSym := Dictionary.LookupItem (TheDecl.Name.TokenStr,
                                                 CurrentScope,
                                                 ProgramContext);
               if IsAnyPriority then
                  AnyPrioritySym := TempSym;
               elsif IsPriority then
                  PrioritySym := TempSym;
               elsif IsInterruptPriority then
                  InterruptPrioritySym := TempSym;
               end if;
            end if;
         end UpdatePrioritySymbols;

      begin

         -- get full name of declared item
         DottedName := PackageFullName;
         EStrings.AppendString (DottedName, ".");
         LexTokenManager.LexStringToString (TheDecl.Name.TokenStr,
                                            TempString);
         EStrings.AppendExaminerString (DottedName, TempString);

         ParentType := NullSymbol;
         MatchType := DeclarationType'Last;

         -- perform whichever checks are appropriate
         case TheDecl.MyType is
            when DecNamedInteger =>
               DoCkValidPackage;
               DoCkDefName;
               DoCkExistingDecl;
               DoCkType;
               DoCkSpecialCases;
               RaiseErrorsAndUpdateDictionary;

            when DecNamedReal =>
               DoCkValidPackage;
               DoCkDefName;
               DoCkExistingDecl;
               DoCkType;
               DoCkSpecialCases;
               RaiseErrorsAndUpdateDictionary;

            when DecIntegerSubtype =>
               DoCkValidPackage;
               DoCkDefNameWithPriorityCk;
               DoCkExistingDecl;
               DoCkType;
               DoCkBounds;
               DoCkBaseType;
               DoCkSystemPriority;
               DoCkSpecialCases;
               RaiseErrorsAndUpdateDictionary;
               UpdatePrioritySymbols;

            when DecIntegerType =>
               DoCkValidPackage;
               DoCkSuffixName;
               DoCkDefName;
               DoCkExistingDecl;
               DoCkType;
               DoCkBounds;
               DoCkSpecialCases;
               RaiseErrorsAndUpdateDictionary;

            when DecFloatingPointType =>
               DoCkValidPackage;
               DoCkSuffixName;
               DoCkDefName;
               DoCkExistingDecl;
               DoCkType;
               DoCkBounds;
               DoCkSpecialCases;
               RaiseErrorsAndUpdateDictionary;

            when DecPrivate =>
               DoCkValidPackage;
               DoCkDefName;
               DoCkExistingDecl;
               DoCkType;
               DoCkSpecialCases;
               RaiseErrorsAndUpdateDictionary;

            when DecTypedConstant =>
               DoCkValidPackage;
               DoCkDefName;
               DoCkExistingDecl;
               DoCkType;
               DoCkSpecialCases;
               RaiseErrorsAndUpdateDictionary;
         end case;

      end CheckAndProcessDeclaration;

      procedure AssertToken (TheToken : in SPSymbols.SPTerminal;
                             ShouldBe : in SPSymbols.SPTerminal)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     TheLexVal;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out ErrorsOccurred;
      --#        in out LexTokenManager.StringTable;
      --#        in out NextExpectedState;
      --#        in out ParserStage;
      --#        in out ParserState;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.File_Sys           from CommandLineData.Content,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.StringTable,
      --#                                          ShouldBe,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TheLexVal,
      --#                                          TheToken &
      --#         ErrorsOccurred,
      --#         ParserStage,
      --#         ParserState                 from *,
      --#                                          ShouldBe,
      --#                                          TheToken &
      --#         LexTokenManager.StringTable from *,
      --#                                          ShouldBe,
      --#                                          TheLexVal,
      --#                                          TheToken &
      --#         NextExpectedState           from *,
      --#                                          ParserState,
      --#                                          ShouldBe,
      --#                                          TheToken;
     is
      begin
         if TheToken /= ShouldBe then
            RaiseSyntaxError (SyntaxErrors'(TheType => ExpSymbol,
                                            TheSymbol => TheToken,
                                            ExpectedSymbol => ShouldBe,
                                            TheLexval => TheLexVal));
         end if;
      end AssertToken;

      procedure BeginIntExprParse
      --# global in     UnaryMinus;
      --#        in out ParserStage;
      --#        in out ParserState;
      --#           out IntAdd;
      --#           out IntExponent;
      --#           out IntSubtract;
      --#           out SavedParserStage;
      --#           out SavedParserState;
      --# derives IntAdd,
      --#         IntExponent,
      --#         IntSubtract,
      --#         ParserState      from  &
      --#         ParserStage      from UnaryMinus &
      --#         SavedParserStage from ParserStage &
      --#         SavedParserState from ParserState;
      is
      begin
         SavedParserState := ParserState;
         SavedParserStage := ParserStage;
         NewState (Expr_Int);
         if UnaryMinus then
            -- cannot accept initial '-' if already had one
            NextStage;
         end if;
         IntSubtract := False;
         IntAdd := False;
         IntExponent := False;
      end BeginIntExprParse;

      procedure EndIntExprParse
      --# global in     SavedParserStage;
      --#        in     SavedParserState;
      --#           out ParserStage;
      --#           out ParserState;
      --# derives ParserStage from SavedParserStage &
      --#         ParserState from SavedParserState;
      is
      begin
         ParserState := SavedParserState;
         ParserStage := SavedParserStage;
      end EndIntExprParse;

      procedure CalcIntValue (Result : out Maths.Value)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     IntAdd;
      --#        in     IntAddSubVal;
      --#        in     IntExponent;
      --#        in     IntExponentVal;
      --#        in     IntSubtract;
      --#        in     IntVal;
      --#        in     LexTokenManager.StringTable;
      --#        in     TheLexVal;
      --#        in     UnaryMinus;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.File_Sys         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        IntAdd,
      --#                                        IntAddSubVal,
      --#                                        IntExponent,
      --#                                        IntExponentVal,
      --#                                        IntSubtract,
      --#                                        IntVal,
      --#                                        LexTokenManager.StringTable,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TheLexVal,
      --#                                        UnaryMinus &
      --#         Result                    from IntAdd,
      --#                                        IntAddSubVal,
      --#                                        IntExponent,
      --#                                        IntExponentVal,
      --#                                        IntSubtract,
      --#                                        IntVal,
      --#                                        UnaryMinus;
      is
         SoFar,
         TempVal    : Maths.Value;
         LocalME    : Maths.ErrorCode := Maths.NoError;
         HadAnError : Boolean := False;
      begin
         -- valid typing of the values is confirmed by correct lexical analysis,
         -- and therefore the only Maths.ErrorCode that can arise from the calculations
         -- here is Maths.Overflow; resulting in a semantic warning.
         if IntExponent then
            Maths.RaiseByPower (IntVal, IntExponentVal, SoFar, LocalME);
            if LocalME /= Maths.NoError then
               ErrorHandler.SemanticWarning (200, TheLexVal.Position, TheLexVal.TokenStr);
               HadAnError := True;
            end if;
         else
            SoFar := IntVal;
         end if;
         if UnaryMinus then
            Maths.Negate (SoFar);
         end if;
         --# assert True;
         if IntSubtract then
            Maths.Subtract (SoFar, IntAddSubVal, TempVal, LocalME);
            SoFar := TempVal;
         elsif IntAdd then
            Maths.Add (SoFar, IntAddSubVal, TempVal, LocalME);
            SoFar := TempVal;
         end if;
         if LocalME /= Maths.NoError and then not HadAnError then
            ErrorHandler.SemanticWarning (200, TheLexVal.Position, TheLexVal.TokenStr);
         end if;
         Result := SoFar;
      end CalcIntValue;

      procedure StateMachineInitialise
      --# global in     Dictionary.Dict;
      --#           out CurrentDeclaration;
      --#           out CurrentScope;
      --#           out ErrorsOccurred;
      --#           out ExprIntValue;
      --#           out IntAdd;
      --#           out IntAddSubVal;
      --#           out IntExponent;
      --#           out IntExponentVal;
      --#           out IntSubtract;
      --#           out IntVal;
      --#           out Lookahead;
      --#           out NextExpectedState;
      --#           out ParserStage;
      --#           out ParserState;
      --#           out ParsingEnds;
      --#           out PredefinedScope;
      --#           out SavedParserStage;
      --#           out SavedParserState;
      --#           out TheLexVal;
      --#           out TheToken;
      --#           out UnaryMinus;
      --#           out ValidPackage;
      --# derives CurrentDeclaration,
      --#         ErrorsOccurred,
      --#         ExprIntValue,
      --#         IntAdd,
      --#         IntAddSubVal,
      --#         IntExponent,
      --#         IntExponentVal,
      --#         IntSubtract,
      --#         IntVal,
      --#         Lookahead,
      --#         NextExpectedState,
      --#         ParserStage,
      --#         ParserState,
      --#         ParsingEnds,
      --#         SavedParserStage,
      --#         SavedParserState,
      --#         TheLexVal,
      --#         TheToken,
      --#         UnaryMinus,
      --#         ValidPackage       from  &
      --#         CurrentScope,
      --#         PredefinedScope    from Dictionary.Dict;
      is
      begin
         PredefinedScope := Dictionary.VisibleScope
           (Dictionary.GetPredefinedPackageStandard);
         CurrentScope := PredefinedScope;
         TheLexVal :=
           LexTokenManager.LexValue'(Position =>
                                       LexTokenManager.TokenPosition'(StartLineNo => 0,
                                                                      StartPos => 0),
                                     TokenStr =>
                                       LexTokenManager.NullString);
         CurrentDeclaration := Declaration'(Name       => TheLexVal,
                                            MyType     => DeclarationType'First,
                                            EncPackage =>
                                              PackageHierarchy'(ChildPackageDepth => TheLexVal),
                                            BaseType   => TheLexVal,
                                            LowBound   => Maths.NoValue,
                                            HighBound  => Maths.NoValue,
                                            NumDigits  => Maths.NoValue,
                                            Value      => Maths.NoValue,
                                            ValueStr   => TheLexVal);
         TheToken := SPSymbols.SPDEFAULT;
         UnaryMinus := False;
         IntSubtract := False;
         IntAdd := False;
         IntExponent := False;
         IntVal := Maths.NoValue;
         IntExponentVal := Maths.NoValue;
         IntAddSubVal := Maths.NoValue;
         ExprIntValue := Maths.NoValue;
         Lookahead := False;
         ParsingEnds := False;
         ValidPackage := False;
         SavedParserState := ParserStates'First;
         SavedParserStage := ParserStages'First;
         NextExpectedState := ParserStates'First;
         ParserState := ParserStates'First;
         ParserStage := ParserStages'First;
         ErrorsOccurred := False;
      end StateMachineInitialise;

      procedure StateMachineIterate
      --# global in     CommandLineData.Content;
      --#        in     PredefinedScope;
      --#        in     TheConfigFile;
      --#        in out AnyPrioritySym;
      --#        in out CurrentDeclaration;
      --#        in out CurrentScope;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out ErrorsOccurred;
      --#        in out ExprIntValue;
      --#        in out IntAdd;
      --#        in out IntAddSubVal;
      --#        in out InterruptPrioritySym;
      --#        in out IntExponent;
      --#        in out IntExponentVal;
      --#        in out IntSubtract;
      --#        in out IntVal;
      --#        in out LexTokenManager.StringTable;
      --#        in out LibPackageSymbol;
      --#        in out Lookahead;
      --#        in out NextExpectedState;
      --#        in out PackageEndName;
      --#        in out PackageFullName;
      --#        in out PackageNameDepth;
      --#        in out ParserStage;
      --#        in out ParserState;
      --#        in out PrioritySym;
      --#        in out SavedParserStage;
      --#        in out SavedParserState;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --#        in out TheLexVal;
      --#        in out TheToken;
      --#        in out UnaryMinus;
      --#        in out ValidPackage;
      --# derives AnyPrioritySym,
      --#         InterruptPrioritySym,
      --#         PrioritySym                 from AnyPrioritySym,
      --#                                          CommandLineData.Content,
      --#                                          CurrentDeclaration,
      --#                                          CurrentScope,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          ExprIntValue,
      --#                                          InterruptPrioritySym,
      --#                                          LexTokenManager.StringTable,
      --#                                          Lookahead,
      --#                                          PackageFullName,
      --#                                          ParserStage,
      --#                                          ParserState,
      --#                                          PrioritySym,
      --#                                          SparkLex.CurrLine,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TheConfigFile,
      --#                                          ValidPackage &
      --#         CurrentDeclaration          from *,
      --#                                          CommandLineData.Content,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          ExprIntValue,
      --#                                          LexTokenManager.StringTable,
      --#                                          Lookahead,
      --#                                          PackageNameDepth,
      --#                                          ParserStage,
      --#                                          ParserState,
      --#                                          SparkLex.CurrLine,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TheConfigFile,
      --#                                          TheLexVal,
      --#                                          TheToken,
      --#                                          UnaryMinus &
      --#         CurrentScope                from *,
      --#                                          CommandLineData.Content,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.StringTable,
      --#                                          LibPackageSymbol,
      --#                                          Lookahead,
      --#                                          PackageNameDepth,
      --#                                          ParserStage,
      --#                                          ParserState,
      --#                                          PredefinedScope,
      --#                                          SparkLex.CurrLine,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TheConfigFile,
      --#                                          TheLexVal,
      --#                                          TheToken &
      --#         Dictionary.Dict             from *,
      --#                                          AnyPrioritySym,
      --#                                          CommandLineData.Content,
      --#                                          CurrentDeclaration,
      --#                                          CurrentScope,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          ExprIntValue,
      --#                                          InterruptPrioritySym,
      --#                                          LexTokenManager.StringTable,
      --#                                          Lookahead,
      --#                                          PackageFullName,
      --#                                          PackageNameDepth,
      --#                                          ParserStage,
      --#                                          ParserState,
      --#                                          PrioritySym,
      --#                                          SparkLex.CurrLine,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TheConfigFile,
      --#                                          TheLexVal,
      --#                                          TheToken,
      --#                                          ValidPackage &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.File_Sys           from AnyPrioritySym,
      --#                                          CommandLineData.Content,
      --#                                          CurrentDeclaration,
      --#                                          CurrentScope,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          ExprIntValue,
      --#                                          IntAdd,
      --#                                          IntAddSubVal,
      --#                                          InterruptPrioritySym,
      --#                                          IntExponent,
      --#                                          IntExponentVal,
      --#                                          IntSubtract,
      --#                                          IntVal,
      --#                                          LexTokenManager.StringTable,
      --#                                          LibPackageSymbol,
      --#                                          Lookahead,
      --#                                          PackageEndName,
      --#                                          PackageFullName,
      --#                                          PackageNameDepth,
      --#                                          ParserStage,
      --#                                          ParserState,
      --#                                          PrioritySym,
      --#                                          SparkLex.CurrLine,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TheConfigFile,
      --#                                          TheLexVal,
      --#                                          TheToken,
      --#                                          UnaryMinus,
      --#                                          ValidPackage &
      --#         ErrorsOccurred              from *,
      --#                                          AnyPrioritySym,
      --#                                          CommandLineData.Content,
      --#                                          CurrentDeclaration,
      --#                                          CurrentScope,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          ExprIntValue,
      --#                                          InterruptPrioritySym,
      --#                                          LexTokenManager.StringTable,
      --#                                          LibPackageSymbol,
      --#                                          Lookahead,
      --#                                          PackageEndName,
      --#                                          PackageFullName,
      --#                                          PackageNameDepth,
      --#                                          ParserStage,
      --#                                          ParserState,
      --#                                          PrioritySym,
      --#                                          SparkLex.CurrLine,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TheConfigFile,
      --#                                          TheLexVal,
      --#                                          TheToken,
      --#                                          UnaryMinus,
      --#                                          ValidPackage &
      --#         ExprIntValue                from *,
      --#                                          CommandLineData.Content,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          IntAdd,
      --#                                          IntAddSubVal,
      --#                                          IntExponent,
      --#                                          IntExponentVal,
      --#                                          IntSubtract,
      --#                                          IntVal,
      --#                                          LexTokenManager.StringTable,
      --#                                          Lookahead,
      --#                                          ParserStage,
      --#                                          ParserState,
      --#                                          SparkLex.CurrLine,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TheConfigFile,
      --#                                          TheToken,
      --#                                          UnaryMinus &
      --#         IntAdd,
      --#         IntExponent,
      --#         IntSubtract,
      --#         Lookahead,
      --#         PackageNameDepth,
      --#         SavedParserStage,
      --#         SavedParserState,
      --#         UnaryMinus                  from *,
      --#                                          CommandLineData.Content,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.StringTable,
      --#                                          Lookahead,
      --#                                          ParserStage,
      --#                                          ParserState,
      --#                                          SparkLex.CurrLine,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TheConfigFile,
      --#                                          TheToken &
      --#         IntAddSubVal,
      --#         IntVal,
      --#         PackageEndName              from *,
      --#                                          CommandLineData.Content,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.StringTable,
      --#                                          Lookahead,
      --#                                          ParserStage,
      --#                                          ParserState,
      --#                                          SparkLex.CurrLine,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TheConfigFile,
      --#                                          TheLexVal,
      --#                                          TheToken &
      --#         IntExponentVal              from *,
      --#                                          CommandLineData.Content,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.StringTable,
      --#                                          Lookahead,
      --#                                          ParserStage,
      --#                                          ParserState,
      --#                                          SparkLex.CurrLine,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TheConfigFile,
      --#                                          TheLexVal &
      --#         LexTokenManager.StringTable from *,
      --#                                          AnyPrioritySym,
      --#                                          CommandLineData.Content,
      --#                                          CurrentDeclaration,
      --#                                          CurrentScope,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          ExprIntValue,
      --#                                          InterruptPrioritySym,
      --#                                          Lookahead,
      --#                                          PackageEndName,
      --#                                          PackageFullName,
      --#                                          PackageNameDepth,
      --#                                          ParserStage,
      --#                                          ParserState,
      --#                                          PrioritySym,
      --#                                          SparkLex.CurrLine,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TheConfigFile,
      --#                                          TheLexVal,
      --#                                          TheToken,
      --#                                          UnaryMinus,
      --#                                          ValidPackage &
      --#         LibPackageSymbol,
      --#         ValidPackage                from *,
      --#                                          CommandLineData.Content,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.StringTable,
      --#                                          LibPackageSymbol,
      --#                                          Lookahead,
      --#                                          PackageNameDepth,
      --#                                          ParserStage,
      --#                                          ParserState,
      --#                                          SparkLex.CurrLine,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TheConfigFile,
      --#                                          TheLexVal,
      --#                                          TheToken &
      --#         NextExpectedState           from *,
      --#                                          CommandLineData.Content,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.StringTable,
      --#                                          Lookahead,
      --#                                          PackageNameDepth,
      --#                                          ParserStage,
      --#                                          ParserState,
      --#                                          SparkLex.CurrLine,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TheConfigFile,
      --#                                          TheToken,
      --#                                          UnaryMinus &
      --#         PackageFullName             from *,
      --#                                          CommandLineData.Content,
      --#                                          CurrentDeclaration,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.StringTable,
      --#                                          Lookahead,
      --#                                          PackageNameDepth,
      --#                                          ParserStage,
      --#                                          ParserState,
      --#                                          SparkLex.CurrLine,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TheConfigFile,
      --#                                          TheToken &
      --#         ParserStage                 from *,
      --#                                          CommandLineData.Content,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.StringTable,
      --#                                          Lookahead,
      --#                                          PackageNameDepth,
      --#                                          ParserState,
      --#                                          SavedParserStage,
      --#                                          SparkLex.CurrLine,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TheConfigFile,
      --#                                          TheToken,
      --#                                          UnaryMinus &
      --#         ParserState                 from *,
      --#                                          CommandLineData.Content,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.StringTable,
      --#                                          Lookahead,
      --#                                          NextExpectedState,
      --#                                          PackageNameDepth,
      --#                                          ParserStage,
      --#                                          SavedParserState,
      --#                                          SparkLex.CurrLine,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TheConfigFile,
      --#                                          TheToken,
      --#                                          UnaryMinus &
      --#         SparkLex.CurrLine,
      --#         TheLexVal,
      --#         TheToken                    from *,
      --#                                          CommandLineData.Content,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.StringTable,
      --#                                          Lookahead,
      --#                                          SparkLex.CurrLine,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TheConfigFile;
      is
         UnwantedPunctToken : Boolean;

         procedure Do_Config_Defn_Start
         --# global in     CommandLineData.Content;
         --#        in     PredefinedScope;
         --#        in     TheLexVal;
         --#        in     TheToken;
         --#        in out CurrentDeclaration;
         --#        in out CurrentScope;
         --#        in out Dictionary.Dict;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out ErrorsOccurred;
         --#        in out LexTokenManager.StringTable;
         --#        in out LibPackageSymbol;
         --#        in out Lookahead;
         --#        in out NextExpectedState;
         --#        in out PackageFullName;
         --#        in out PackageNameDepth;
         --#        in out ParserStage;
         --#        in out ParserState;
         --#        in out SPARK_IO.File_Sys;
         --#        in out ValidPackage;
         --# derives CurrentDeclaration          from *,
         --#                                          PackageNameDepth,
         --#                                          ParserStage,
         --#                                          TheLexVal &
         --#         CurrentScope                from *,
         --#                                          CommandLineData.Content,
         --#                                          Dictionary.Dict,
         --#                                          LexTokenManager.StringTable,
         --#                                          LibPackageSymbol,
         --#                                          PackageNameDepth,
         --#                                          ParserStage,
         --#                                          PredefinedScope,
         --#                                          TheLexVal,
         --#                                          TheToken &
         --#         Dictionary.Dict,
         --#         LexTokenManager.StringTable from *,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageNameDepth,
         --#                                          ParserStage,
         --#                                          TheLexVal,
         --#                                          TheToken &
         --#         ErrorHandler.ErrorContext,
         --#         SPARK_IO.File_Sys           from CommandLineData.Content,
         --#                                          CurrentDeclaration,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          LexTokenManager.StringTable,
         --#                                          LibPackageSymbol,
         --#                                          PackageNameDepth,
         --#                                          ParserStage,
         --#                                          SPARK_IO.File_Sys,
         --#                                          TheLexVal,
         --#                                          TheToken,
         --#                                          ValidPackage &
         --#         ErrorsOccurred,
         --#         ValidPackage                from *,
         --#                                          CommandLineData.Content,
         --#                                          Dictionary.Dict,
         --#                                          LexTokenManager.StringTable,
         --#                                          LibPackageSymbol,
         --#                                          PackageNameDepth,
         --#                                          ParserStage,
         --#                                          TheLexVal,
         --#                                          TheToken,
         --#                                          ValidPackage &
         --#         LibPackageSymbol            from *,
         --#                                          CommandLineData.Content,
         --#                                          Dictionary.Dict,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageNameDepth,
         --#                                          ParserStage,
         --#                                          TheLexVal,
         --#                                          TheToken &
         --#         Lookahead                   from *,
         --#                                          ParserStage &
         --#         NextExpectedState           from *,
         --#                                          PackageNameDepth,
         --#                                          ParserStage,
         --#                                          ParserState,
         --#                                          TheToken &
         --#         PackageFullName             from *,
         --#                                          CurrentDeclaration,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageNameDepth,
         --#                                          ParserStage,
         --#                                          TheToken &
         --#         PackageNameDepth,
         --#         ParserStage,
         --#         ParserState                 from *,
         --#                                          PackageNameDepth,
         --#                                          ParserStage,
         --#                                          TheToken;
         is
            TempString : EStrings.T;
         begin
            case ParserStage is
               when 1 =>
                  AssertToken (TheToken, SPSymbols.RWpackage);
                  PackageNameDepth := 1;
                  NextStage;
               when 2 =>
                  AssertToken (TheToken, SPSymbols.identifier);
                  ChangeToNewPackage (TheLexVal);
                  CurrentDeclaration.EncPackage (PackageNameDepth) := TheLexVal;
                  NextStage;
               when 3 =>
                  case TheToken is
                     when SPSymbols.RWis =>
                        -- Evaluate and store the full name of the package
                        LexTokenManager.LexStringToString
                          (CurrentDeclaration.EncPackage (1).TokenStr, PackageFullName);
                        for I in ChildPackageDepth range 2 .. PackageNameDepth loop
                           EStrings.AppendString (PackageFullName, ".");
                           LexTokenManager.LexStringToString
                             (CurrentDeclaration.EncPackage (I).TokenStr, TempString);
                           EStrings.AppendExaminerString (PackageFullName, TempString);
                           --# assert True;
                        end loop;
                        NextStage;
                     when SPSymbols.point => -- Child unit
                        if PackageNameDepth = ChildPackageDepth'Last then
                           -- Too many child levels, or a spurious dot
                           AssertToken (TheToken, SPSymbols.RWis);
                        else
                           PackageNameDepth := PackageNameDepth + 1;
                           ParserStage := 2; -- Go back to process the child name
                        end if;
                     when others =>
                        -- expected a 'package <name> is'
                        AssertToken (TheToken, SPSymbols.RWis);
                  end case;
               when 4 =>
                  Lookahead := True;
                  NewState (Defn);
               when others =>
                  null;
            end case;
         end Do_Config_Defn_Start;

         procedure Do_Next_Config_Defn_Start
         --# global in     CommandLineData.Content;
         --#        in     PredefinedScope;
         --#        in     TheLexVal;
         --#        in     TheToken;
         --#        in out CurrentDeclaration;
         --#        in out CurrentScope;
         --#        in out Dictionary.Dict;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out ErrorsOccurred;
         --#        in out LexTokenManager.StringTable;
         --#        in out LibPackageSymbol;
         --#        in out Lookahead;
         --#        in out NextExpectedState;
         --#        in out PackageFullName;
         --#        in out PackageNameDepth;
         --#        in out ParserStage;
         --#        in out ParserState;
         --#        in out SPARK_IO.File_Sys;
         --#        in out ValidPackage;
         --# derives CurrentDeclaration          from *,
         --#                                          PackageNameDepth,
         --#                                          ParserStage,
         --#                                          TheLexVal &
         --#         CurrentScope                from *,
         --#                                          CommandLineData.Content,
         --#                                          Dictionary.Dict,
         --#                                          LexTokenManager.StringTable,
         --#                                          LibPackageSymbol,
         --#                                          PackageNameDepth,
         --#                                          ParserStage,
         --#                                          PredefinedScope,
         --#                                          TheLexVal,
         --#                                          TheToken &
         --#         Dictionary.Dict,
         --#         LexTokenManager.StringTable from *,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageNameDepth,
         --#                                          ParserStage,
         --#                                          TheLexVal,
         --#                                          TheToken &
         --#         ErrorHandler.ErrorContext,
         --#         SPARK_IO.File_Sys           from CommandLineData.Content,
         --#                                          CurrentDeclaration,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          LexTokenManager.StringTable,
         --#                                          LibPackageSymbol,
         --#                                          PackageNameDepth,
         --#                                          ParserStage,
         --#                                          SPARK_IO.File_Sys,
         --#                                          TheLexVal,
         --#                                          TheToken,
         --#                                          ValidPackage &
         --#         ErrorsOccurred,
         --#         ValidPackage                from *,
         --#                                          CommandLineData.Content,
         --#                                          Dictionary.Dict,
         --#                                          LexTokenManager.StringTable,
         --#                                          LibPackageSymbol,
         --#                                          PackageNameDepth,
         --#                                          ParserStage,
         --#                                          TheLexVal,
         --#                                          TheToken,
         --#                                          ValidPackage &
         --#         LibPackageSymbol            from *,
         --#                                          CommandLineData.Content,
         --#                                          Dictionary.Dict,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageNameDepth,
         --#                                          ParserStage,
         --#                                          TheLexVal,
         --#                                          TheToken &
         --#         Lookahead                   from *,
         --#                                          ParserStage &
         --#         NextExpectedState           from *,
         --#                                          PackageNameDepth,
         --#                                          ParserStage,
         --#                                          ParserState,
         --#                                          TheToken &
         --#         PackageFullName             from *,
         --#                                          CurrentDeclaration,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageNameDepth,
         --#                                          ParserStage,
         --#                                          TheToken &
         --#         PackageNameDepth,
         --#         ParserStage,
         --#         ParserState                 from *,
         --#                                          PackageNameDepth,
         --#                                          ParserStage,
         --#                                          TheToken;
         is
         begin
            case ParserStage is
               when 1 =>
                  case TheToken is
                     when SPSymbols.RWpackage =>
                        PackageNameDepth := 1;
                        NextStage;
                     when SPSymbols.SPEND =>
                        null;
                     when others =>
                        -- expected a 'package ... is'
                        AssertToken (TheToken,
                                     SPSymbols.RWpackage);
                  end case;
               when others =>
                  Do_Config_Defn_Start;
            end case;
         end Do_Next_Config_Defn_Start;

         procedure Do_Defn
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     TheLexVal;
         --#        in     TheToken;
         --#        in out CurrentDeclaration;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out ErrorsOccurred;
         --#        in out LexTokenManager.StringTable;
         --#        in out NextExpectedState;
         --#        in out ParserState;
         --#        in out SPARK_IO.File_Sys;
         --#           out ParserStage;
         --# derives CurrentDeclaration,
         --#         LexTokenManager.StringTable from *,
         --#                                          TheLexVal,
         --#                                          TheToken &
         --#         ErrorHandler.ErrorContext,
         --#         SPARK_IO.File_Sys           from CommandLineData.Content,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          LexTokenManager.StringTable,
         --#                                          SPARK_IO.File_Sys,
         --#                                          TheLexVal,
         --#                                          TheToken &
         --#         ErrorsOccurred              from *,
         --#                                          TheToken &
         --#         NextExpectedState           from *,
         --#                                          ParserState,
         --#                                          TheToken &
         --#         ParserStage,
         --#         ParserState                 from TheToken;
         is
         begin
            case TheToken is
               when SPSymbols.RWtype =>
                  NewState (Type_Unknown);
               when SPSymbols.RWsubtype =>
                  CurrentDeclaration.MyType := DecIntegerSubtype;
                  NewState (Subtype_Int);
               when SPSymbols.identifier =>

                  CurrentDeclaration.Name := TheLexVal;
                  NewState (Const_Unknown);
               when SPSymbols.RWend =>
                  NewState (Config_Defn_End);
               when others =>
                  -- expected a configuration definition
                  RaiseSyntaxError (SyntaxErrors'(TheType => ExpDefn,
                                                  TheSymbol => TheToken,
                                                  ExpectedSymbol => SPSymbols.SPDEFAULT,
                                                  TheLexval => TheLexVal));
            end case;
         end Do_Defn;

         procedure Do_Type_Unknown
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     TheLexVal;
         --#        in     TheToken;
         --#        in out CurrentDeclaration;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out ErrorsOccurred;
         --#        in out LexTokenManager.StringTable;
         --#        in out NextExpectedState;
         --#        in out ParserStage;
         --#        in out ParserState;
         --#        in out SPARK_IO.File_Sys;
         --# derives CurrentDeclaration,
         --#         LexTokenManager.StringTable from *,
         --#                                          ParserStage,
         --#                                          TheLexVal,
         --#                                          TheToken &
         --#         ErrorHandler.ErrorContext,
         --#         SPARK_IO.File_Sys           from CommandLineData.Content,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          LexTokenManager.StringTable,
         --#                                          ParserStage,
         --#                                          SPARK_IO.File_Sys,
         --#                                          TheLexVal,
         --#                                          TheToken &
         --#         ErrorsOccurred,
         --#         ParserStage,
         --#         ParserState                 from *,
         --#                                          ParserStage,
         --#                                          TheToken &
         --#         NextExpectedState           from *,
         --#                                          ParserStage,
         --#                                          ParserState,
         --#                                          TheToken;
         is
         begin
            case ParserStage is
               when 1 =>
                  AssertToken (TheToken, SPSymbols.identifier);
                  CurrentDeclaration.Name := TheLexVal;
                  NextStage;
               when 2 =>
                  AssertToken (TheToken, SPSymbols.RWis);
                  NextStage;
               when 3 =>
                  case TheToken is
                     when SPSymbols.RWprivate =>
                        CurrentDeclaration.MyType := DecPrivate;
                        NewState (Type_Private);
                     when SPSymbols.RWdigits =>
                        CurrentDeclaration.MyType := DecFloatingPointType;
                        NewState (Type_FP);
                     when SPSymbols.RWrange =>
                        CurrentDeclaration.MyType := DecIntegerType;
                        NewState (Type_Int);
                     when others =>
                        -- expected either 'digits', 'range' or 'private'
                        RaiseSyntaxError (SyntaxErrors'(TheType => ExpDigitsRange,
                                                        TheSymbol => TheToken,
                                                        ExpectedSymbol => SPSymbols.SPDEFAULT,
                                                        TheLexval => TheLexVal));
                  end case;
               when others =>
                  null;
            end case;
         end Do_Type_Unknown;

         procedure Do_Type_Private
         --# global in     CommandLineData.Content;
         --#        in     CurrentDeclaration;
         --#        in     CurrentScope;
         --#        in     PackageFullName;
         --#        in     TheLexVal;
         --#        in     TheToken;
         --#        in     ValidPackage;
         --#        in out AnyPrioritySym;
         --#        in out Dictionary.Dict;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out ErrorsOccurred;
         --#        in out InterruptPrioritySym;
         --#        in out LexTokenManager.StringTable;
         --#        in out Lookahead;
         --#        in out NextExpectedState;
         --#        in out ParserStage;
         --#        in out ParserState;
         --#        in out PrioritySym;
         --#        in out SPARK_IO.File_Sys;
         --# derives AnyPrioritySym,
         --#         Dictionary.Dict,
         --#         InterruptPrioritySym,
         --#         PrioritySym                 from AnyPrioritySym,
         --#                                          CommandLineData.Content,
         --#                                          CurrentDeclaration,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          InterruptPrioritySym,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageFullName,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          ValidPackage &
         --#         ErrorHandler.ErrorContext,
         --#         SPARK_IO.File_Sys           from AnyPrioritySym,
         --#                                          CommandLineData.Content,
         --#                                          CurrentDeclaration,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          InterruptPrioritySym,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageFullName,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          SPARK_IO.File_Sys,
         --#                                          TheLexVal,
         --#                                          TheToken,
         --#                                          ValidPackage &
         --#         ErrorsOccurred              from *,
         --#                                          AnyPrioritySym,
         --#                                          CurrentDeclaration,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          InterruptPrioritySym,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageFullName,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          TheToken,
         --#                                          ValidPackage &
         --#         LexTokenManager.StringTable from *,
         --#                                          AnyPrioritySym,
         --#                                          CurrentDeclaration,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          InterruptPrioritySym,
         --#                                          PackageFullName,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          TheLexVal,
         --#                                          TheToken,
         --#                                          ValidPackage &
         --#         Lookahead                   from *,
         --#                                          ParserStage &
         --#         NextExpectedState           from *,
         --#                                          ParserStage,
         --#                                          ParserState,
         --#                                          TheToken &
         --#         ParserStage,
         --#         ParserState                 from *,
         --#                                          ParserStage,
         --#                                          TheToken;
         is
         begin
            case ParserStage is
               when 1 =>
                  AssertToken (TheToken, SPSymbols.semicolon);
                  NextStage;
               when 2 =>
                  CheckAndProcessDeclaration (CurrentDeclaration);
                  Lookahead := True;
                  NewState (Defn);
               when others =>
                  null;
            end case;
         end Do_Type_Private;

         procedure Do_Type_FP
         --# global in     CommandLineData.Content;
         --#        in     CurrentScope;
         --#        in     PackageFullName;
         --#        in     TheLexVal;
         --#        in     TheToken;
         --#        in     ValidPackage;
         --#        in out AnyPrioritySym;
         --#        in out CurrentDeclaration;
         --#        in out Dictionary.Dict;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out ErrorsOccurred;
         --#        in out InterruptPrioritySym;
         --#        in out LexTokenManager.StringTable;
         --#        in out Lookahead;
         --#        in out NextExpectedState;
         --#        in out ParserStage;
         --#        in out ParserState;
         --#        in out PrioritySym;
         --#        in out SPARK_IO.File_Sys;
         --#        in out UnaryMinus;
         --# derives AnyPrioritySym,
         --#         Dictionary.Dict,
         --#         InterruptPrioritySym,
         --#         PrioritySym                 from AnyPrioritySym,
         --#                                          CommandLineData.Content,
         --#                                          CurrentDeclaration,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          InterruptPrioritySym,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageFullName,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          ValidPackage &
         --#         CurrentDeclaration          from *,
         --#                                          LexTokenManager.StringTable,
         --#                                          ParserStage,
         --#                                          TheLexVal,
         --#                                          TheToken,
         --#                                          UnaryMinus &
         --#         ErrorHandler.ErrorContext,
         --#         SPARK_IO.File_Sys           from AnyPrioritySym,
         --#                                          CommandLineData.Content,
         --#                                          CurrentDeclaration,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          InterruptPrioritySym,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageFullName,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          SPARK_IO.File_Sys,
         --#                                          TheLexVal,
         --#                                          TheToken,
         --#                                          UnaryMinus,
         --#                                          ValidPackage &
         --#         ErrorsOccurred              from *,
         --#                                          AnyPrioritySym,
         --#                                          CurrentDeclaration,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          InterruptPrioritySym,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageFullName,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          TheToken,
         --#                                          UnaryMinus,
         --#                                          ValidPackage &
         --#         LexTokenManager.StringTable from *,
         --#                                          AnyPrioritySym,
         --#                                          CurrentDeclaration,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          InterruptPrioritySym,
         --#                                          PackageFullName,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          TheLexVal,
         --#                                          TheToken,
         --#                                          UnaryMinus,
         --#                                          ValidPackage &
         --#         Lookahead                   from *,
         --#                                          ParserStage &
         --#         NextExpectedState           from *,
         --#                                          ParserStage,
         --#                                          ParserState,
         --#                                          TheToken,
         --#                                          UnaryMinus &
         --#         ParserStage,
         --#         ParserState,
         --#         UnaryMinus                  from *,
         --#                                          ParserStage,
         --#                                          TheToken,
         --#                                          UnaryMinus;
         is
         begin
            case ParserStage is
               when 1 =>
                  AssertToken (TheToken, SPSymbols.integer_number);
                  GuardedLiteralToValue (TheLexVal,
                                         CurrentDeclaration.NumDigits);
                  NextStage;
               when 2 =>
                  AssertToken (TheToken, SPSymbols.RWrange);
                  UnaryMinus := False;
                  NextStage;
               when 3 =>
                  case TheToken is
                     when SPSymbols.real_number =>
                        GuardedLiteralToValue (TheLexVal,
                                               CurrentDeclaration.LowBound);
                        if UnaryMinus then
                           Maths.Negate (CurrentDeclaration.LowBound);
                        end if;
                        NextStage;
                     when SPSymbols.minus =>
                        if UnaryMinus then
                           -- expected a number
                           AssertToken (TheToken,
                                        SPSymbols.real_number);
                        else
                           UnaryMinus := True;
                        end if;
                     when others =>
                        -- expected either either '-' or a number
                        RaiseSyntaxError (SyntaxErrors'(TheType => ExpMinusReal,
                                                        TheSymbol => TheToken,
                                                        ExpectedSymbol => SPSymbols.SPDEFAULT,
                                                        TheLexval => TheLexVal));
                  end case;
               when 4 =>
                  AssertToken (TheToken, SPSymbols.double_dot);
                  UnaryMinus := False;
                  NextStage;
               when 5 =>
                  case TheToken is
                     when SPSymbols.real_number =>
                        GuardedLiteralToValue (TheLexVal,
                                               CurrentDeclaration.HighBound);
                        if UnaryMinus then
                           Maths.Negate (CurrentDeclaration.HighBound);
                        end if;
                        NextStage;
                     when SPSymbols.minus =>
                        if UnaryMinus then
                           -- expected a number
                           AssertToken (TheToken,
                                        SPSymbols.real_number);
                        else
                           UnaryMinus := True;
                        end if;
                     when others =>
                        -- expected either '-' or a number
                        RaiseSyntaxError (SyntaxErrors'(TheType => ExpMinusReal,
                                                        TheSymbol => TheToken,
                                                        ExpectedSymbol => SPSymbols.SPDEFAULT,
                                                        TheLexval => TheLexVal));
                  end case;
               when 6 =>
                  AssertToken (TheToken, SPSymbols.semicolon);
                  NextStage;
               when 7 =>
                  CheckAndProcessDeclaration (CurrentDeclaration);
                  Lookahead := True;
                  NewState (Defn);
               when others =>
                  null;
            end case;

         end Do_Type_FP;

         procedure Do_Type_Int
         --# global in     CommandLineData.Content;
         --#        in     CurrentScope;
         --#        in     ExprIntValue;
         --#        in     PackageFullName;
         --#        in     TheLexVal;
         --#        in     TheToken;
         --#        in     ValidPackage;
         --#        in out AnyPrioritySym;
         --#        in out CurrentDeclaration;
         --#        in out Dictionary.Dict;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out ErrorsOccurred;
         --#        in out IntAdd;
         --#        in out InterruptPrioritySym;
         --#        in out IntExponent;
         --#        in out IntSubtract;
         --#        in out LexTokenManager.StringTable;
         --#        in out Lookahead;
         --#        in out NextExpectedState;
         --#        in out ParserStage;
         --#        in out ParserState;
         --#        in out PrioritySym;
         --#        in out SavedParserStage;
         --#        in out SavedParserState;
         --#        in out SPARK_IO.File_Sys;
         --#        in out UnaryMinus;
         --# derives AnyPrioritySym,
         --#         Dictionary.Dict,
         --#         InterruptPrioritySym,
         --#         PrioritySym                 from AnyPrioritySym,
         --#                                          CommandLineData.Content,
         --#                                          CurrentDeclaration,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          InterruptPrioritySym,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageFullName,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          ValidPackage &
         --#         CurrentDeclaration          from *,
         --#                                          ExprIntValue,
         --#                                          ParserStage &
         --#         ErrorHandler.ErrorContext,
         --#         SPARK_IO.File_Sys           from AnyPrioritySym,
         --#                                          CommandLineData.Content,
         --#                                          CurrentDeclaration,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          InterruptPrioritySym,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageFullName,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          SPARK_IO.File_Sys,
         --#                                          TheLexVal,
         --#                                          TheToken,
         --#                                          ValidPackage &
         --#         ErrorsOccurred              from *,
         --#                                          AnyPrioritySym,
         --#                                          CurrentDeclaration,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          InterruptPrioritySym,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageFullName,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          TheToken,
         --#                                          ValidPackage &
         --#         IntAdd,
         --#         IntExponent,
         --#         IntSubtract,
         --#         Lookahead,
         --#         SavedParserStage,
         --#         UnaryMinus                  from *,
         --#                                          ParserStage &
         --#         LexTokenManager.StringTable from *,
         --#                                          AnyPrioritySym,
         --#                                          CurrentDeclaration,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          InterruptPrioritySym,
         --#                                          PackageFullName,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          TheLexVal,
         --#                                          TheToken,
         --#                                          ValidPackage &
         --#         NextExpectedState           from *,
         --#                                          ParserStage,
         --#                                          ParserState,
         --#                                          TheToken &
         --#         ParserStage,
         --#         ParserState                 from *,
         --#                                          ParserStage,
         --#                                          TheToken &
         --#         SavedParserState            from *,
         --#                                          ParserStage,
         --#                                          ParserState;
         is
         begin
            case ParserStage is
               when 1 =>
                  Lookahead := True;
                  NextStage;
                  UnaryMinus := False;
                  BeginIntExprParse;
               when 2 =>
                  -- got the low bound, last time
                  CurrentDeclaration.LowBound := ExprIntValue;
                  AssertToken (TheToken, SPSymbols.double_dot);
                  NextStage;
               when 3 =>
                  Lookahead := True;
                  NextStage;
                  UnaryMinus := False;
                  BeginIntExprParse;
               when 4 =>
                  -- got the high bound, last time
                  CurrentDeclaration.HighBound := ExprIntValue;
                  AssertToken (TheToken, SPSymbols.semicolon);
                  NextStage;
               when 5 =>
                  CheckAndProcessDeclaration (CurrentDeclaration);
                  Lookahead := True;
                  NewState (Defn);
               when others =>
                  null;
            end case;
         end Do_Type_Int;


         procedure Do_Subtype_Int
         --# global in     CommandLineData.Content;
         --#        in     CurrentScope;
         --#        in     ExprIntValue;
         --#        in     PackageFullName;
         --#        in     TheLexVal;
         --#        in     TheToken;
         --#        in     ValidPackage;
         --#        in out AnyPrioritySym;
         --#        in out CurrentDeclaration;
         --#        in out Dictionary.Dict;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out ErrorsOccurred;
         --#        in out IntAdd;
         --#        in out InterruptPrioritySym;
         --#        in out IntExponent;
         --#        in out IntSubtract;
         --#        in out LexTokenManager.StringTable;
         --#        in out Lookahead;
         --#        in out NextExpectedState;
         --#        in out ParserStage;
         --#        in out ParserState;
         --#        in out PrioritySym;
         --#        in out SavedParserStage;
         --#        in out SavedParserState;
         --#        in out SPARK_IO.File_Sys;
         --#        in out UnaryMinus;
         --# derives AnyPrioritySym,
         --#         Dictionary.Dict,
         --#         InterruptPrioritySym,
         --#         PrioritySym                 from AnyPrioritySym,
         --#                                          CommandLineData.Content,
         --#                                          CurrentDeclaration,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          InterruptPrioritySym,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageFullName,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          ValidPackage &
         --#         CurrentDeclaration          from *,
         --#                                          ExprIntValue,
         --#                                          ParserStage,
         --#                                          TheLexVal &
         --#         ErrorHandler.ErrorContext,
         --#         SPARK_IO.File_Sys           from AnyPrioritySym,
         --#                                          CommandLineData.Content,
         --#                                          CurrentDeclaration,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          InterruptPrioritySym,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageFullName,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          SPARK_IO.File_Sys,
         --#                                          TheLexVal,
         --#                                          TheToken,
         --#                                          ValidPackage &
         --#         ErrorsOccurred              from *,
         --#                                          AnyPrioritySym,
         --#                                          CurrentDeclaration,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          InterruptPrioritySym,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageFullName,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          TheToken,
         --#                                          ValidPackage &
         --#         IntAdd,
         --#         IntExponent,
         --#         IntSubtract,
         --#         Lookahead,
         --#         SavedParserStage,
         --#         UnaryMinus                  from *,
         --#                                          ParserStage &
         --#         LexTokenManager.StringTable from *,
         --#                                          AnyPrioritySym,
         --#                                          CurrentDeclaration,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          InterruptPrioritySym,
         --#                                          PackageFullName,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          TheLexVal,
         --#                                          TheToken,
         --#                                          ValidPackage &
         --#         NextExpectedState           from *,
         --#                                          ParserStage,
         --#                                          ParserState,
         --#                                          TheToken &
         --#         ParserStage,
         --#         ParserState                 from *,
         --#                                          ParserStage,
         --#                                          TheToken &
         --#         SavedParserState            from *,
         --#                                          ParserStage,
         --#                                          ParserState;
         is
         begin
            case ParserStage is
               when 1 =>
                  AssertToken (TheToken, SPSymbols.identifier);
                  CurrentDeclaration.Name := TheLexVal;
                  NextStage;
               when 2 =>
                  AssertToken (TheToken, SPSymbols.RWis);
                  NextStage;
               when 3 =>
                  AssertToken (TheToken, SPSymbols.identifier);
                  CurrentDeclaration.BaseType := TheLexVal;
                  NextStage;
               when 4 =>
                  AssertToken (TheToken, SPSymbols.RWrange);
                  NextStage;
               when 5 =>
                  Lookahead := True;
                  NextStage;
                  UnaryMinus := False;
                  BeginIntExprParse;
               when 6 =>
                  -- got the low bound, last time
                  CurrentDeclaration.LowBound := ExprIntValue;
                  AssertToken (TheToken, SPSymbols.double_dot);
                  NextStage;
               when 7 =>
                  Lookahead := True;
                  NextStage;
                  UnaryMinus := False;
                  BeginIntExprParse;
               when 8 =>
                  -- got the high bound, last time
                  CurrentDeclaration.HighBound := ExprIntValue;
                  AssertToken (TheToken, SPSymbols.semicolon);
                  NextStage;
               when 9 =>
                  CheckAndProcessDeclaration (CurrentDeclaration);
                  Lookahead := True;
                  NewState (Defn);
               when others =>
                  null;
            end case;
         end Do_Subtype_Int;

         procedure Do_Const_Unknown
         --# global in     CommandLineData.Content;
         --#        in     CurrentScope;
         --#        in     ExprIntValue;
         --#        in     PackageFullName;
         --#        in     TheLexVal;
         --#        in     TheToken;
         --#        in     ValidPackage;
         --#        in out AnyPrioritySym;
         --#        in out CurrentDeclaration;
         --#        in out Dictionary.Dict;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out ErrorsOccurred;
         --#        in out IntAdd;
         --#        in out InterruptPrioritySym;
         --#        in out IntExponent;
         --#        in out IntSubtract;
         --#        in out LexTokenManager.StringTable;
         --#        in out Lookahead;
         --#        in out NextExpectedState;
         --#        in out ParserStage;
         --#        in out ParserState;
         --#        in out PrioritySym;
         --#        in out SavedParserStage;
         --#        in out SavedParserState;
         --#        in out SPARK_IO.File_Sys;
         --#        in out UnaryMinus;
         --# derives AnyPrioritySym,
         --#         Dictionary.Dict,
         --#         InterruptPrioritySym,
         --#         PrioritySym                 from AnyPrioritySym,
         --#                                          CommandLineData.Content,
         --#                                          CurrentDeclaration,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          ExprIntValue,
         --#                                          InterruptPrioritySym,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageFullName,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          ValidPackage &
         --#         CurrentDeclaration          from *,
         --#                                          ExprIntValue,
         --#                                          LexTokenManager.StringTable,
         --#                                          ParserStage,
         --#                                          TheLexVal,
         --#                                          TheToken,
         --#                                          UnaryMinus &
         --#         ErrorHandler.ErrorContext,
         --#         SPARK_IO.File_Sys           from AnyPrioritySym,
         --#                                          CommandLineData.Content,
         --#                                          CurrentDeclaration,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          ExprIntValue,
         --#                                          InterruptPrioritySym,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageFullName,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          SPARK_IO.File_Sys,
         --#                                          TheLexVal,
         --#                                          TheToken,
         --#                                          UnaryMinus,
         --#                                          ValidPackage &
         --#         ErrorsOccurred              from *,
         --#                                          AnyPrioritySym,
         --#                                          CurrentDeclaration,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          ExprIntValue,
         --#                                          InterruptPrioritySym,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageFullName,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          TheToken,
         --#                                          UnaryMinus,
         --#                                          ValidPackage &
         --#         IntAdd,
         --#         IntExponent,
         --#         IntSubtract,
         --#         Lookahead,
         --#         SavedParserStage            from *,
         --#                                          ParserStage,
         --#                                          TheToken &
         --#         LexTokenManager.StringTable from *,
         --#                                          AnyPrioritySym,
         --#                                          CurrentDeclaration,
         --#                                          CurrentScope,
         --#                                          Dictionary.Dict,
         --#                                          ExprIntValue,
         --#                                          InterruptPrioritySym,
         --#                                          PackageFullName,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          TheLexVal,
         --#                                          TheToken,
         --#                                          UnaryMinus,
         --#                                          ValidPackage &
         --#         NextExpectedState           from *,
         --#                                          ParserStage,
         --#                                          ParserState,
         --#                                          TheToken,
         --#                                          UnaryMinus &
         --#         ParserStage,
         --#         ParserState,
         --#         UnaryMinus                  from *,
         --#                                          ParserStage,
         --#                                          TheToken,
         --#                                          UnaryMinus &
         --#         SavedParserState            from *,
         --#                                          ParserStage,
         --#                                          ParserState,
         --#                                          TheToken;
         is
         begin
            case ParserStage is
               when 1 =>
                  AssertToken (TheToken, SPSymbols.colon);
                  NextStage;
               when 2 =>
                  AssertToken (TheToken, SPSymbols.RWconstant);
                  NextStage;
               when 3 =>
                  if TheToken = SPSymbols.identifier then
                     CurrentDeclaration.MyType   := DecTypedConstant;
                     CurrentDeclaration.BaseType := TheLexVal;
                     -- Stay in stage 3, since we now expect :=
                  else
                     AssertToken (TheToken, SPSymbols.becomes);
                     UnaryMinus := False;
                     NextStage;
                  end if;
               when 4 =>
                  case TheToken is
                     when SPSymbols.real_number =>
                        CurrentDeclaration.MyType := DecNamedReal;
                        GuardedLiteralToValue (TheLexVal,
                                               CurrentDeclaration.Value);
                        if UnaryMinus then
                           Maths.Negate (CurrentDeclaration.Value);
                        end if;
                        NextStage;
                     when SPSymbols.minus =>
                        if UnaryMinus then
                           -- expected a number
                           RaiseSyntaxError (SyntaxErrors'(TheType => ExpIntegerReal,
                                                           TheSymbol => TheToken,
                                                           ExpectedSymbol => SPSymbols.SPDEFAULT,
                                                           TheLexval => TheLexVal));
                        else
                           UnaryMinus := True;
                        end if;
                     when SPSymbols.identifier =>
                        CurrentDeclaration.ValueStr := TheLexVal;
                        NextStage;

                     when others =>
                        CurrentDeclaration.MyType := DecNamedInteger;
                        Lookahead := True;
                        NextStage;
                        BeginIntExprParse;
                  end case;

               when 5 =>
                  AssertToken (TheToken, SPSymbols.semicolon);
                  NextStage;

               when 6 =>
                  if CurrentDeclaration.MyType = DecNamedInteger then
                     CurrentDeclaration.Value := ExprIntValue;
                  end if;
                  CheckAndProcessDeclaration (CurrentDeclaration);
                  Lookahead := True;
                  NewState (Defn);
               when others =>
                  null;
            end case;
         end Do_Const_Unknown;

         procedure Do_Config_Defn_End
         --# global in     AnyPrioritySym;
         --#        in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     InterruptPrioritySym;
         --#        in     PackageFullName;
         --#        in     PackageNameDepth;
         --#        in     PrioritySym;
         --#        in     TheLexVal;
         --#        in     TheToken;
         --#        in out CurrentDeclaration;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out ErrorsOccurred;
         --#        in out LexTokenManager.StringTable;
         --#        in out Lookahead;
         --#        in out NextExpectedState;
         --#        in out PackageEndName;
         --#        in out ParserStage;
         --#        in out ParserState;
         --#        in out SPARK_IO.File_Sys;
         --# derives CurrentDeclaration          from *,
         --#                                          ParserStage,
         --#                                          TheLexVal &
         --#         ErrorHandler.ErrorContext,
         --#         SPARK_IO.File_Sys           from AnyPrioritySym,
         --#                                          CommandLineData.Content,
         --#                                          CurrentDeclaration,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          InterruptPrioritySym,
         --#                                          LexTokenManager.StringTable,
         --#                                          PackageEndName,
         --#                                          PackageFullName,
         --#                                          PackageNameDepth,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          SPARK_IO.File_Sys,
         --#                                          TheLexVal,
         --#                                          TheToken &
         --#         ErrorsOccurred              from *,
         --#                                          AnyPrioritySym,
         --#                                          InterruptPrioritySym,
         --#                                          PackageEndName,
         --#                                          PackageFullName,
         --#                                          ParserStage,
         --#                                          PrioritySym,
         --#                                          TheToken &
         --#         LexTokenManager.StringTable from *,
         --#                                          PackageEndName,
         --#                                          PackageFullName,
         --#                                          PackageNameDepth,
         --#                                          ParserStage,
         --#                                          TheLexVal,
         --#                                          TheToken &
         --#         Lookahead                   from *,
         --#                                          ParserStage &
         --#         NextExpectedState           from *,
         --#                                          ParserStage,
         --#                                          ParserState,
         --#                                          TheToken &
         --#         PackageEndName              from *,
         --#                                          LexTokenManager.StringTable,
         --#                                          ParserStage,
         --#                                          TheLexVal,
         --#                                          TheToken &
         --#         ParserStage,
         --#         ParserState                 from *,
         --#                                          ParserStage,
         --#                                          TheToken;
         is
            TempString : EStrings.T;
         begin
            case ParserStage is
               when 1 =>
                  AssertToken (TheToken, SPSymbols.identifier);
                  -- Build up the name after "end" in PackageEndName
                  LexTokenManager.LexStringToString (TheLexVal.TokenStr, PackageEndName);
                  CurrentDeclaration.Name := TheLexVal;
                  NextStage;
               when 2 =>
                  case TheToken is
                     when SPSymbols.semicolon =>
                        NextStage;
                     -- Note that we don't need to parse the name too strictly here
                     -- e.g. we won't (yet) fail <id> <id> .   The string that is
                     -- built won't match the package name unless it is well-formed,
                     -- and so a badly-formed name will fail later in CheckPackageEnd.
                     when SPSymbols.point =>
                        EStrings.AppendString (PackageEndName, ".");
                     when SPSymbols.identifier =>
                        LexTokenManager.LexStringToString (TheLexVal.TokenStr, TempString);
                        EStrings.AppendExaminerString (PackageEndName, TempString);
                     when others =>
                        -- expected a 'end <identifier>[.<identifier>] ;'
                        AssertToken (TheToken, SPSymbols.semicolon);
                  end case;
               when 3 =>
                  Lookahead := True;
                  CheckPackageEnd;
                  NewState (Next_Config_Defn_Start);
               when others =>
                  null;
            end case;
         end Do_Config_Defn_End;

         procedure Do_Expr_Int
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     SavedParserStage;
         --#        in     SavedParserState;
         --#        in     TheLexVal;
         --#        in     TheToken;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out ErrorsOccurred;
         --#        in out ExprIntValue;
         --#        in out IntAdd;
         --#        in out IntAddSubVal;
         --#        in out IntExponent;
         --#        in out IntExponentVal;
         --#        in out IntSubtract;
         --#        in out IntVal;
         --#        in out LexTokenManager.StringTable;
         --#        in out Lookahead;
         --#        in out NextExpectedState;
         --#        in out ParserStage;
         --#        in out ParserState;
         --#        in out SPARK_IO.File_Sys;
         --#        in out UnaryMinus;
         --# derives ErrorHandler.ErrorContext,
         --#         SPARK_IO.File_Sys           from CommandLineData.Content,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          IntAdd,
         --#                                          IntAddSubVal,
         --#                                          IntExponent,
         --#                                          IntExponentVal,
         --#                                          IntSubtract,
         --#                                          IntVal,
         --#                                          LexTokenManager.StringTable,
         --#                                          ParserStage,
         --#                                          SPARK_IO.File_Sys,
         --#                                          TheLexVal,
         --#                                          TheToken,
         --#                                          UnaryMinus &
         --#         ErrorsOccurred,
         --#         IntAdd,
         --#         IntExponent,
         --#         IntSubtract,
         --#         Lookahead,
         --#         UnaryMinus                  from *,
         --#                                          ParserStage,
         --#                                          TheToken &
         --#         ExprIntValue                from *,
         --#                                          IntAdd,
         --#                                          IntAddSubVal,
         --#                                          IntExponent,
         --#                                          IntExponentVal,
         --#                                          IntSubtract,
         --#                                          IntVal,
         --#                                          ParserStage,
         --#                                          TheToken,
         --#                                          UnaryMinus &
         --#         IntAddSubVal,
         --#         IntVal,
         --#         LexTokenManager.StringTable from *,
         --#                                          LexTokenManager.StringTable,
         --#                                          ParserStage,
         --#                                          TheLexVal,
         --#                                          TheToken &
         --#         IntExponentVal              from *,
         --#                                          LexTokenManager.StringTable,
         --#                                          ParserStage,
         --#                                          TheLexVal &
         --#         NextExpectedState           from *,
         --#                                          ParserStage,
         --#                                          ParserState,
         --#                                          TheToken &
         --#         ParserStage                 from *,
         --#                                          SavedParserStage,
         --#                                          TheToken &
         --#         ParserState                 from *,
         --#                                          ParserStage,
         --#                                          SavedParserState,
         --#                                          TheToken;
         is
         begin
            case ParserStage is
               when 1 =>
                  case TheToken is
                     when SPSymbols.minus =>
                        UnaryMinus := True;
                        ParserStage := 2;
                     when SPSymbols.integer_number =>
                        GuardedLiteralToValue (TheLexVal,
                                               IntVal);
                        ParserStage := 3;
                     when others =>
                        -- expected either '-' or an integer literal
                        RaiseSyntaxError (SyntaxErrors'(TheType => ExpMinusInteger,
                                                        TheSymbol => TheToken,
                                                        ExpectedSymbol => SPSymbols.SPDEFAULT,
                                                        TheLexval => TheLexVal));
                  end case;
               when 2 =>
                  GuardedLiteralToValue (TheLexVal,
                                         IntVal);
                  ParserStage := 3;
                  AssertToken (TheToken, SPSymbols.integer_number);
               when 3 =>
                  case TheToken is
                     when SPSymbols.minus =>
                        IntSubtract := True;
                        ParserStage := 6;
                     when SPSymbols.plus =>
                        IntAdd := True;
                        ParserStage := 6;
                     when SPSymbols.double_star =>
                        IntExponent := True;
                        ParserStage := 4;
                     when others =>
                        Lookahead := True;
                        CalcIntValue (ExprIntValue);
                        EndIntExprParse;
                  end case;
               when 4 =>
                  GuardedLiteralToValue (TheLexVal,
                                         IntExponentVal);
                  ParserStage := 5;
                  AssertToken (TheToken, SPSymbols.integer_number);
               when 5 =>
                  case TheToken is
                     when SPSymbols.minus =>
                        IntSubtract := True;
                        ParserStage := 6;
                     when SPSymbols.plus =>
                        IntAdd := True;
                        ParserStage := 6;
                     when others =>
                        Lookahead := True;
                        CalcIntValue (ExprIntValue);
                        EndIntExprParse;
                  end case;
               when 6 =>
                  AssertToken (TheToken, SPSymbols.integer_number);
                  GuardedLiteralToValue (TheLexVal,
                                         IntAddSubVal);
                  NextStage;
               when 7 =>
                  CalcIntValue (ExprIntValue);
                  Lookahead := True;
                  EndIntExprParse;
               when others =>
                  null;
            end case;
         end Do_Expr_Int;

      begin
         --# accept F, 10, UnwantedPunctToken, "UnwantedPunctToken unused here";
         if not Lookahead then
            SparkLex.ExaminerLex (TheConfigFile,
                                  TheToken,
                                  TheLexVal,
                                  UnwantedPunctToken);
         else
            Lookahead := False;
         end if;

         case ParserState is

            when Config_Defn_Start =>
               Do_Config_Defn_Start;
            when Next_Config_Defn_Start =>
               Do_Next_Config_Defn_Start;
            when Defn =>
               Do_Defn;
            when Type_Unknown =>
               Do_Type_Unknown;
            when Type_Private =>
               Do_Type_Private;
            when Type_FP =>
               Do_Type_FP;
            when Type_Int =>
               Do_Type_Int;
            when Subtype_Int =>
               Do_Subtype_Int;
            when Const_Unknown =>
               Do_Const_Unknown;
            when Config_Defn_End =>
               Do_Config_Defn_End;
            when Expr_Int =>
               Do_Expr_Int;
            when FoundSyntaxError =>
               if TheToken = SPSymbols.semicolon then
                  NewState (NextExpectedState);
               end if;

         end case;
         --# accept F, 33, UnwantedPunctToken, "UnwantedPunctToken unused here";
      end StateMachineIterate;

   begin
      SparkLex.ClearLineContext;
      StateMachineInitialise;
      loop
         StateMachineIterate;
         exit when TheToken = SPSymbols.SPEND or else ParsingEnds;
      end loop;
      OverallStatus := not ErrorsOccurred;

   end ProcessConfigFile;

   procedure ReadConfigFile (OpenedOK : out Boolean;
                             NoErrors : out Boolean)
   --# global in     CommandLineData.Content;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SparkLex.CurrLine;
   --#        in out SPARK_IO.File_Sys;
   --#           out TheErrorContext;
   --# derives Dictionary.Dict,
   --#         ErrorHandler.ErrorContext,
   --#         LexTokenManager.StringTable,
   --#         NoErrors,
   --#         SPARK_IO.File_Sys,
   --#         TheErrorContext             from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          SPARK_IO.File_Sys &
   --#         OpenedOK                    from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          SPARK_IO.File_Sys &
   --#         SparkLex.CurrLine           from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          SPARK_IO.File_Sys;
   is
      FileOpenedOK : Boolean;
      FileReadOK : Boolean;
      LocalConfigFile : SPARK_IO.File_Type;
      SavedErrorContext : ErrorHandler.ErrorContexts;

   begin
      FileOpenedOK := False;
      ErrorHandler.GetErrorContext (TheErrorContext);
      if CommandLineData.Content.TargetConfig then
         OpenFile (LocalConfigFile,
                   FileOpenedOK);
         if FileOpenedOK then
            if CommandLineData.Content.Echo and not CommandLineData.Content.Brief then
               ScreenEcho.New_Line (1);
               ScreenEcho.Put_Line ("           Reading target configuration file ...");
            end if;
            ErrorHandler.GetErrorContext (SavedErrorContext);
            ErrorHandler.ErrorInit (CommandLineData.Content.TargetConfigFile,
                                    CommandLineData.Content.Echo);
            ProcessConfigFile (LocalConfigFile, FileReadOK);
            ErrorHandler.GetErrorContext (TheErrorContext);
            ErrorHandler.SetErrorContext (SavedErrorContext);
            --# accept F, 10, LocalConfigFile, "LocalConfigFile unused here";
            CloseFile (LocalConfigFile);
            --# end accept;
         else
            FileReadOK := False;
         end if;
      else
         FileReadOK := True;
      end if;
      NoErrors := FileReadOK;
      OpenedOK := FileOpenedOK;
   end ReadConfigFile;

   procedure OutputConfigFile (ToFile : in SPARK_IO.File_Type)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     TheErrorContext;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --#        in out XMLReport.State;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys,
   --#         XMLReport.State           from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        SPARK_IO.File_Sys,
   --#                                        TheErrorContext,
   --#                                        ToFile,
   --#                                        XMLReport.State;
   is
      SavedErrorContext : ErrorHandler.ErrorContexts;
      ErrorSeverity     : ErrorHandler.ErrorLevel;

      procedure Check_And_Report_Errors
      --# global in     ErrorSeverity;
      --#        in     ToFile;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.ErrorContext from *,
      --#                                        ErrorSeverity &
      --#         SPARK_IO.File_Sys         from *,
      --#                                        ErrorSeverity,
      --#                                        ToFile;
      is
      begin
         if ErrorSeverity /= ErrorHandler.NoError then
            SPARK_IO.New_Line (ToFile, 1);
            SPARK_IO.Put_Line (ToFile, "Warning: analysis aborted due to errors in target configuration file", 0);
            ErrorHandler.SetFileOpenError;
         end if;
      end Check_And_Report_Errors;
   begin
      if CommandLineData.Content.TargetConfig then
         if CommandLineData.Content.XML then
            XMLReport.StartSection (XMLReport.STargetConfig,
                                    ToFile);

            XMLReport.Filename (CommandLineData.Content.TargetConfigFile,
                                ToFile);

            ErrorHandler.GetErrorContext (SavedErrorContext);
            ErrorHandler.SetErrorContext (TheErrorContext);
            ErrorHandler.GetErrorSeverity (ErrorSeverity);
            XMLReport.StartSection (XMLReport.SMessages,
                                    ToFile);
            ErrorHandler.AppendErrors (ToFile, Error_Types.ForReportIndexedFiles);
            XMLReport.EndSection (XMLReport.SMessages,
                                    ToFile);

            Check_And_Report_Errors;

            ErrorHandler.SetErrorContext (SavedErrorContext);

            XMLReport.EndSection (XMLReport.STargetConfig,
                                    ToFile);

         else
            SPARK_IO.New_Line (ToFile, 2);
            SPARK_IO.Put_Line (ToFile, "Target configuration file:", 0);
            ErrorHandler.GetErrorContext (SavedErrorContext);
            ErrorHandler.SetErrorContext (TheErrorContext);
            ErrorHandler.GetErrorSeverity (ErrorSeverity);
            ErrorHandler.PrintErrors (ToFile, Error_Types.ForReportIndexedFiles);

            Check_And_Report_Errors;

            ErrorHandler.SetErrorContext (SavedErrorContext);
         end if;
      end if;
   end OutputConfigFile;

end ConfigFile;
