-- $Id: commonstringutilities.adb 11889 2008-12-12 15:49:12Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with CommonString,
     Ada.Characters.Latin_1;

package body CommonStringUtilities
is

   ----------------------------------------------------------------------------
   -- Section:
   -- returns the specified subsection of the string
   -- if the subsection lies outside the string, empty string returned
   procedure Section (StrIn     : in     String;
                      StrLenIn  : in     Integer;
                      StartPos  : in     Integer;
                      Length    : in     Integer;
                      StrOut    :    out String;
                      StrLenOut :    out Integer)
   is
      Finished  : Boolean;
      Pos       : Integer;
   begin
      CommonString.InitString (StrOut);

      if (StartPos + Length) - 1 > StrLenIn then
         StrLenOut := 0;
      else
         Pos := StartPos;
         Finished := False;
         while not Finished and then Pos < StartPos + Length loop
            StrOut ((Pos + 1) - StartPos) := StrIn (Pos);
            if not (Pos = StrIn'Last) then
               Pos := Pos + 1;
            else
               Finished := True;
            end if;
         end loop;

         StrLenOut := Length;
      end if;
   end Section;

   ----------------------------------------------------------------------------
   -- Trim:
   -- removes ' ', Latin_1.HT, Latin_1.LF, Latin_1.CR from either end of the string
   procedure Trim (StrIn  : in     String;
                   LenIn  : in     Integer;
                   StrOut :    out String;
                   LenOut :    out Integer)
   is
      StartPos      : Integer;
      EndPos        : Integer;
      Finished      : Boolean;
      TrimmedLength : Integer;

      -----------------------------------------------------------------------
      function IsWhiteSpace (C : Character) return Boolean
      is
         Result : Boolean;
      begin
         if C = ' ' or else
            C = Ada.Characters.Latin_1.HT or else
            C = Ada.Characters.Latin_1.LF or else
            C = Ada.Characters.Latin_1.CR then

            Result := True;
         else
            Result := False;
         end if;

         return Result;
      end IsWhiteSpace;

   -----------------------------------------------------------------------

   begin -- Trim
   -- find first non space character

      StartPos := 1;
      Finished := False;

      while not Finished loop
         if StartPos > LenIn or else
            not IsWhiteSpace (StrIn (StartPos)) then

            Finished := True;
         else
            StartPos := StartPos + 1;
         end if;
      end loop;

      -- if end of string then return empty string
      if StartPos > LenIn then
         CommonString.InitString (StrOut);
         LenOut := 0;
      else
         -- find last non space character
         EndPos := LenIn;
         Finished := False;

         while not Finished loop
            if EndPos = StartPos or else
               not IsWhiteSpace (StrIn (EndPos)) then
               Finished := True;
            else
               EndPos := EndPos - 1;
            end if;
         end loop;

         -- calculate trimmed string length
         TrimmedLength := (EndPos - StartPos) + 1;

         Section (StrIn, LenIn, StartPos, TrimmedLength, StrOut, LenOut);

      end if;

   end Trim;

   -------------------------------------------------------------------------
   -- FindSubStringAfter:
   -- find the specified SearchString, starting at the specified position in
   -- the given String
   procedure FindSubStringAfter (Str          : in     String;
                                 Len          : in     Integer;
                                 SearchStart  : in     Integer;
                                 SearchString : in     String;
                                 SearchStrLen : in     Integer;
                                 StringFound  :    out Boolean;
                                 StringStart  :    out Integer)
   is

      CurrentStartPos      : Integer;
      CurrentCompareOffset : Integer;
      CouldBeHere          : Boolean;
      Found                : Boolean := False;
      LastStartPos         : Integer;

   begin
      CurrentStartPos := SearchStart;

      if SearchStrLen <= Len then

         LastStartPos := (Len + 1) - SearchStrLen;

         while not Found and then
               CurrentStartPos <= LastStartPos loop

            CurrentCompareOffset := 0;
            CouldBeHere := True;
            while not Found and then
                  CouldBeHere and then
                  CurrentCompareOffset < SearchStrLen loop

               if Str (CurrentStartPos + CurrentCompareOffset) /=
                  SearchString (1 + CurrentCompareOffset) then

                  CouldBeHere := False;

               else

                  if CurrentCompareOffset = SearchStrLen - 1 then
                     Found := True;
                  else
                     CurrentCompareOffset := CurrentCompareOffset + 1;
                  end if;
               end if;
            end loop;

            if not Found then
               CurrentStartPos := CurrentStartPos + 1;
            end if;
         end loop;
      end if;

      StringFound := Found;
      StringStart := CurrentStartPos;

   end FindSubStringAfter;

   ---------------------------------------------------------------------------
   -- FindCharAfter: find specified character in EStr, starting at specified
   -- position
   procedure FindCharAfter (Str         : in     String;
                            Len         : in     Integer;
                            SearchStart : in     Integer;
                            SearchChar  : in     Character;
                            CharFound   :    out Boolean;
                            CharPos     :    out Integer)
   is
      Position : Integer;
      Found    : Boolean := False;
      Finished : Boolean := False;
   begin
      Position := SearchStart;
      while not Finished loop
         if Position <= Len and then
            Str (Position) = SearchChar then

            Found := True;
            Finished := True;

         elsif Position < Len then
            Position := Position + 1;
         else
            Finished := True;
            Found := False;
         end if;
      end loop;

      CharPos := Position;
      CharFound := Found;

   end FindCharAfter;

   --------------------------------------------------------------------------

   function LexOrder (FirstName  : String;
                      FirstLen   : Integer;
                      SecondName : String;
                      SecondLen  : Integer)
       return OrderTypes
   is
      Pos            : Integer := 1;
      Finished       : Boolean := False;
      LexOrderResult : OrderTypes;

      RankOfNextCharInFirst  : Integer;
      RankOfNextCharInSecond : Integer;

   begin
      LexOrderResult := NeitherFirst; -- initialise to avoid flow error

      while not Finished loop
         -- if the remainder of only one of the strings is empty then
         -- that is the first in order
         -- if the remainder of both is empty then neither is first
         if Pos > FirstLen and Pos <= SecondLen then
            LexOrderResult := FirstOneFirst;
            Finished := True;
         elsif Pos > SecondLen and Pos <= FirstLen then
            LexOrderResult := SecondOneFirst;
            Finished := True;
         elsif Pos > FirstLen and Pos > SecondLen then
            LexOrderResult := NeitherFirst;
            Finished := True;
         else
            -- decide using the current character in the string
            RankOfNextCharInFirst  := Character'Pos (FirstName (Pos));
            RankOfNextCharInSecond := Character'Pos (SecondName (Pos));

            if RankOfNextCharInFirst < RankOfNextCharInSecond then
               LexOrderResult := FirstOneFirst;
               Finished := True;
            elsif RankOfNextCharInSecond < RankOfNextCharInFirst then
               LexOrderResult := SecondOneFirst;
               Finished := True;
            else
               if Pos < FirstName'Length and Pos < SecondName'Length then
                  Pos := Pos + 1;
               else
                  LexOrderResult := NeitherFirst;
                  Finished := True;
               end if;
            end if;
         end if;
      end loop;

      return LexOrderResult; -- warning OK, always defined
   end LexOrder; -- warning OK, always defined

end CommonStringUtilities;
