-- $Id: commonstring.adb 11889 2008-12-12 15:49:12Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with Ada.Characters.Handling,
     SystemErrors;
package body CommonString
is

   procedure FatalStringOverflowError
   --# derives ;
   is
      --# hide FatalStringOverflowError;
   begin
      SystemErrors.FatalError (SystemErrors.StringOverFlow, "");
   end FatalStringOverflowError;

   -- This procedure initialises a string to define its value.
   procedure InitString (S : out String)
   is
   begin
      --# accept F, 23,  S,    "Array update anomaly" &
      --#        F, 602, S, S, "Array update anomaly";
      for I in Natural range S'Range loop
         S (I) := ' ';  -- Flow error expected.
      end loop;
   end InitString;  -- Flow error expected; S is fully defined.

   -- This function returns the given string with any alphabetic
   -- characters converted to lower case.  A length is not returned
   -- the length of the returned string is equal to the length of the
   -- input string.
   procedure LowerCase (Str    : in     String;
                        Len    : in     Natural;
                        StrOut :    out String)
   is
   begin
      InitString (StrOut);

      for I in Natural range 1 .. Len loop
         StrOut (I) := Ada.Characters.Handling.To_Lower (Str (I));
      end loop;
   end LowerCase;

   ------------------------------------

   -- This function returns the given string with any alphabetic
   -- characters converted to lower case.  A length is not returned
   -- the length of the returned string is equal to the length of the
   -- input string.

   procedure UpperCase (Str    : in     String;
                        Len    : in     Natural;
                        StrOut :    out String)
   is
   begin
      InitString (StrOut);

      for I in Natural range 1 .. Len  loop
         StrOut (I) := Ada.Characters.Handling.To_Upper (Str (I));
      end loop;
   end UpperCase;

   -- Case insensitive comparison.  Returns true if and only if the strings
   -- are equal, given the specified lengths.
   function EqString (Str1    : String;
                      Length1 : Natural;
                      Str2    : String;
                      Length2 : Natural) return Boolean
   is
      Result : Boolean;
   begin
      if Length1 = Length2 then

         Result := True;

         for I in Natural range 1 .. Length1 loop

            Result := Result and Ada.Characters.Handling.To_Lower (Str1 (I)) =
                                 Ada.Characters.Handling.To_Lower (Str2 (I));

            exit when not Result;

         end loop;

      else

         Result := False;

      end if;

      return Result;

   end EqString;

   -- This procedure copies StrIn to StrOut.  If StrIn is longer than StrOut then
   -- it is truncated in StrOut.  Length is returned to show how much of StrIn was
   -- actually copied.
   procedure CopyString (StrOut :    out String;
                         Length :    out Natural;
                         StrIn  : in     String)
   is
   begin

      InitString (StrOut);

      --# accept Flow, 10, "Effective as triggers Fatal Error";
      if StrOut'Length < StrIn'Length then
         FatalStringOverflowError;
      end if;
      --# end accept;

      for I in Natural range 1 .. StrIn'Length loop
         StrOut (I) := StrIn ((I + StrIn'First) - 1); -- flow error OK, array init
      end loop;

      Length := StrIn'Length;
   end CopyString; -- warning OK, enough of Estr.Content defined

   procedure AppendString (Str1 : in out String;
                           Len1 : in out Natural;
                           Str2 : in     String;
                           Len2 : in     Natural)
   is
   begin
      --# accept Flow, 10, "Effective as triggers Fatal Error";
      if Str1'Length - Len1 < Len2 then
         FatalStringOverflowError;
      end if;
      --# end accept;

      for I in Natural range 1 .. Len2 loop
         Str1 (Len1 + I) := Str2 ((I + Str2'First) - 1);
      end loop;

      Len1 := Len1 + Len2;
   end AppendString;

   procedure AppendStringTruncate (Str1 : in out String;
                                   Len1 : in out Natural;
                                   Str2 : in     String;
                                   Len2 : in     Natural)
   is
      Limit           : Natural;
      Chars_To_Append : Natural;
   begin

      -- If the combined lengths exceed the maximum length
      -- of the result string then limit the length of the
      -- result and only copy as many characters as will fit,
      -- ie the difference between the current length of the
      -- result string and the max length of the result string.
      Limit := Len1 + Len2;
      if Limit > Str1'Length then
         Limit := Str1'Length;
         Chars_To_Append := Str1'Length - Len1;
      else
         Chars_To_Append := Len2;
      end if;

      for I in Natural range 1 .. Chars_To_Append loop
         Str1 (Len1 + I) := Str2 ((I + Str2'First) - 1);
      end loop;

      Len1 := Limit;

   end AppendStringTruncate;

   procedure PutString (File : in SPARK_IO.File_Type;
                        Str  : in String;
                        Len  : in Natural)
   is
   begin
      if Len /= 0 then
         SPARK_IO.Put_String (File, Str, Len);
      end if;
   end PutString;

   procedure PutLine (File : in SPARK_IO.File_Type;
                      Str  : in String;
                      Len  : in Natural)
   is
   begin
      if Len = 0 then
         SPARK_IO.New_Line (File, 1);
      else
         SPARK_IO.Put_Line (File, Str, Len);
      end if;
   end PutLine;

   procedure GetLine (File : in     SPARK_IO.File_Type;
                      Str  :    out String;
                      Len  :    out Natural)
   is
   begin
      SPARK_IO.Get_Line (File, Str, Len);
   end GetLine;

   procedure AppendChar (Str     : in out String;
                         Len     : in out Natural;
                         Ch      : in     Character;
                         Success :    out Boolean)
   is
   begin
      if Len = Str'Length then
         Success := False;
      else
         Success := True;
         Len := Len + 1;
         Str (Len) := Ch;
      end if;
   end AppendChar;

   procedure PopChar (Str  : in out String;
                      Len  : in out Natural;
                      Char :    out Character)
   is
   begin
      if Len = 0 then
         Char := ' ';
      else
         Char := Str (Str'First);
         for I in Natural range Str'First + 1 .. Str'Length loop
            Str (I - 1) := Str (I);
         end loop;
         Len := Len - 1;
      end if;
   end PopChar;

end CommonString;
