-- $Id: commandlinehandler.adb 12952 2009-04-09 13:13:58Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with CommandLineData,
     Ada.Command_Line,
     Ada.Characters.Handling,
     Ada.Characters.Latin_1,
     FileSystem,
     ScreenEcho,
     SPARK_IO,
     StringUtilities;

use type SPARK_IO.File_Status;
use type CommandLineData.InfoFlowPolicies;

package body CommandLineHandler
is

   type CommandLineErrors is
      (ESIndex, ESIndexName, ESNoIndex, ESSourceExt, ESListingExt, ESNoReport, ESReport, ESReportName, ESRules, ESRuleUnknown,
       ESNoEcho, ESNoDict, ESNoDuration, ESDict,
       ESInvalidOption, ESListingFile, ESNoComma, ESSource,
       EWOverlap, EWListing, EWIndex, EWReport, EWSource, ESReduction, ESErrorExplanations,
       EWtooMany,
       ESWarning, ESWarningName, ESNoWarning, EWWarning,
       ESOutputDir, ESOutputDirRepeated, ESOutputDirNotFound,
       SOFstyle,
       ESRealRTCs,
       ESsyntax,
       ESStatistics,
       ESNoStatistics,
       ESFDLoption,
       ESInformationFlow,
       ESTargetData, ESTargetDataName, ESNoTargetData, EWTarget,
       ESConfigFile, ESConfigFileName, ESNoConfigFile, EWConfig,
       ESToolVariant,
       ESDebug,
       ESanno,
       EShtml,
       ESPlainOutput,
       ESBrief,
       ESXML,
       ESXMLandHTML,
       ESXMLnorep,
       ESHTMLnorep,
       ESDataFlowAndInfoFlowPolicy,
       ESOriginalFlowErrors, ESErrorExplanationOption,
       ESJustificationRepeated,
       ESJustificationOption,
       ESProfile, ESProfileOption,
       ESNoListings,
       ESDataAndConfig);


   function IsWhiteSpace (SpaceChar : Character) return Boolean
   is
   begin
      return (SpaceChar = ' ') or
         (SpaceChar = Ada.Characters.Latin_1.HT) or
         (SpaceChar = Ada.Characters.Latin_1.CR);
   end IsWhiteSpace;

   procedure CopyString (F         : in     String;
                         StartFrom : in     Positive;
                         T         : in out String)
   is
      M,
      N : Integer;
   begin
      for I in Integer range StartFrom .. T'Last loop
         T (I) := ' ';
      end loop;
      M := F'First;
      N := StartFrom;
      loop
         exit when M > F'Last or N > T'Last;
         T (N) := F (M);
         N := N + 1;
         M := M + 1;
      end loop;
   end CopyString;

   procedure PutError (S : in String)
   is
   begin
      ScreenEcho.Put_String (S);
   end PutError;

   procedure PutWarning (S : in String)
   is
   begin
      ScreenEcho.Put_Line (S);
   end PutWarning;

   procedure OutputError (E : in CommandLineErrors)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                E;
   is
      CheckMsg : constant String := " (check default switch file)";
   begin
      case E is
         when ESIndex              =>
            PutError ("Index option already specified or NoIndex option already specified" & CheckMsg);

         when ESIndexName          =>
            PutError ("Index option incorrectly specified");

         when ESSource             =>
            PutError ("Source file incorrectly specified");

         when ESNoIndex            =>
            PutError ("NoIndex option already specified or Index option already specified" & CheckMsg);

         when ESSourceExt          =>
            PutError ("Source extension option already specified or incorrectly specified");

         when ESListingExt         =>
            PutError ("Listing extension option already specified or incorrectly specified");

         when ESNoReport           =>
            PutError ("NoReport option already specified, or contradictory report option found" & CheckMsg);

         when ESReport             =>
            PutError ("Report option already specified or NoReport option already specified" & CheckMsg);

         when ESReportName         =>
            PutError ("Report option incorrectly specified");

         when ESRules              =>
            PutError ("Rules option may not appear more than once" & CheckMsg);

         when ESRuleUnknown        =>
            PutError ("Rules option not recognised: legal values are: none, lazy, keen, all");

         when ESNoEcho             =>
            PutError ("NoEcho option may not appear more than once" & CheckMsg);

         when ESNoDict             =>
            PutError ("NoDictionary option already specified or Dictionary option already specified (check default switch file)");

         when ESNoDuration         =>
            PutError ("NoDuration option may not appear more than once" & CheckMsg);

         when ESDict               =>
            PutError ("Dictionary option already specified or NoDictionary option already specified (check default switch file)");

         when ESInvalidOption      =>
            PutError ("Invalid command line option:");

         when ESListingFile        =>
            PutError ("Listing file option incorrect");

         when ESNoComma            =>
            PutError ("Comma missing in line");

         when ESReduction          =>
            PutError ("PF, VC, RTC or EXP  option already specified, or NoFDL already specified" & CheckMsg);

         when ESErrorExplanations  =>
            PutError ("Error_Explanations option may not appear more than once" & CheckMsg);

         when EWOverlap            =>
            PutError ("File argument is repeated:");

         when EWListing            =>
            PutError ("Cannot create file");

         when EWIndex              =>
            PutError ("Cannot open index file");

         when EWReport             =>
            PutError ("Cannot create file");

         when EWSource             =>
            PutError ("Cannot open source file");

         when EWtooMany            =>
            PutError ("Too many source files on command line");

         when ESWarningName        =>
            PutError ("Warning option incorrectly specified");

         when ESWarning            =>
            PutError ("Warning option already specified" & CheckMsg);

         when ESNoWarning          =>
            PutError ("NoWarning option already specifies" & CheckMsg);

         when EWWarning            =>
            PutError ("Cannot open warning control file");

         when ESOutputDir          =>
            PutError ("Output directory option must have exactly one argument");

         when ESOutputDirRepeated  =>
            PutError ("Output directory option already specified" & CheckMsg);

         when ESOutputDirNotFound  =>
            PutError ("Cannot find or write to output directory");

         when SOFstyle             =>
            PutError ("-concat is no longer supported. -tree is now the default and only option" & CheckMsg);

         when ESRealRTCs           =>
            PutError ("RealRTCs option incorrect");

         when ESsyntax             =>
            PutError ("Syntax check option may not appear more than once" & CheckMsg);

         when ESStatistics         =>
            PutError ("Statistics option already specified, or NoStatistics option already specified" & CheckMsg);

         when ESNoStatistics       =>
            PutError ("NoStatistics option already specified, or Statistics option already specified" & CheckMsg);

         when ESFDLoption          =>
            PutError ("FDL identifier option already specified, or contradictory option found" & CheckMsg);

         when ESInformationFlow    =>
            PutError ("Flow analysis option already specified or incorrectly specified:");

         when ESTargetDataName     =>
            PutError ("Target compiler data option incorrectly specified");

         when ESTargetData         =>
            PutError ("Target compiler data option already specified or NoTarget compiler data option already specified" & CheckMsg);

         when ESNoTargetData       =>
            PutError ("NoTarget compiler data option already specified or Target compiler data option already specified" & CheckMsg);

         when EWTarget             =>
            PutError ("Cannot open target data file");

         when ESConfigFileName     =>
            PutError ("Configuration file incorrectly specified");

         when ESConfigFile         =>
            PutError ("Configuration file already specified or NoConfiguration file already specified" & CheckMsg);

         when ESNoConfigFile       =>
            PutError ("NoConfiguration file already specified or Configuration file already specified" & CheckMsg);

         when EWConfig             =>
            PutError ("Cannot open configuration file");

         when ESToolVariant        =>
            PutError ("Language variant option incorrect");

         when ESDebug              =>
            PutError ("Debugging option already specified, or incorrectly specified" & CheckMsg);

         when ESanno               =>
            PutError ("Annotation character option may not appear more than once" & CheckMsg);

         when EShtml               =>
            PutError ("HTML option may not appear more than once" & CheckMsg);

         when ESPlainOutput        =>
            PutError ("Plain output option may not appear more than once" & CheckMsg);

         when ESBrief              =>
            PutError ("Brief output option may not appear more than once" & CheckMsg);

         when ESOriginalFlowErrors =>
            PutError ("OriginalFlowErrors output option may not appear more than once" & CheckMsg);

         when ESProfile            =>
            PutError ("Analysis profile option may not appear more than once" & CheckMsg);

         when ESProfileOption      =>
            PutError ("Analysis profile option incorrect:");

         when ESErrorExplanationOption =>
            PutError ("Error_Explanation option incorrect:");

         when ESDataAndConfig      =>
            PutError ("Target data and config file cannot be specified together" & CheckMsg);

         when ESXML                =>
            PutError ("XML option may not appear more than once" & CheckMsg);

         when ESXMLandHTML         =>
            PutError ("XML and HTML cannot be specified together" & CheckMsg);

         when ESHTMLnorep          =>
            PutError ("HTML and noreport cannot be specified together" & CheckMsg);

         when ESXMLnorep           =>
            PutError ("XML and noreport cannot be specified together" & CheckMsg);

         when ESNoListings         =>
            PutError ("No listings option may not be specified more than once" & CheckMsg);

         when ESDataFlowAndInfoFlowPolicy =>
            PutError ("Information flow policy requires Flow_Analysis=Information");

         when ESJustificationRepeated =>
            PutError ("Justification_Option may not appear more than once" & CheckMsg);

         when ESJustificationOption   =>
            PutError ("Justification_Option value must be one of: Ignore, Full, or Brief");

      end case;
   end OutputError;

   procedure PossibleError (E : in CommandLineErrors)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                E;
   is
   begin
      if not CommandLineData.Content.Valid then
         OutputError (E);
         ScreenEcho.New_Line (1);
      end if;
   end PossibleError;

   procedure PossibleError2 (E : in CommandLineErrors;
                             F : in EStrings.T)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                E,
   --#                                F;
   is
   begin
      if not CommandLineData.Content.Valid then
         OutputError (E);
         ScreenEcho.Put_Char (' ');
         if CommandLineData.Content.PlainOutput then
            ScreenEcho.Put_ExaminerLine (EStrings.LowerCase (F));
         else
            ScreenEcho.Put_ExaminerLine (F);
         end if;
      end if;
   end PossibleError2;

   procedure ReadDefaultSwitches (DefaultSwitchesFound : out Boolean;
                                  CmdLine              : out FileSystem.TypCmdLine;
                                  CmdLinePtr           : out FileSystem.TypCmdLineRange)
   --# global in out SPARK_IO.File_Sys;
   --# derives CmdLine,
   --#         CmdLinePtr,
   --#         DefaultSwitchesFound,
   --#         SPARK_IO.File_Sys    from SPARK_IO.File_Sys;
   -- post (not DefaultSwitchesFound and CmdLinePtr = 1) or
   --      (DefaultSwitchesFound and CmdLinePtr = CmdLine'Last + 2);
   is

      SwitchFileFound : Boolean;
      SwitchFile      : SPARK_IO.File_Type;
      CmdLinePtrLocal : FileSystem.TypCmdLineRange;
      CmdLineLocal    : FileSystem.TypCmdLine;
      Unused          : SPARK_IO.File_Status;

      procedure OpenFile (SwitchFileFound : out Boolean;
                          SwitchFile      : out SPARK_IO.File_Type)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys,
      --#         SwitchFile,
      --#         SwitchFileFound   from SPARK_IO.File_Sys;
      is
         SwitchFileLocal : SPARK_IO.File_Type := SPARK_IO.Null_File;
         FileStatus : SPARK_IO.File_Status;
      begin
         SPARK_IO.Open (SwitchFileLocal,
                        SPARK_IO.In_File,
                        8,
                        "spark.sw",
                        "",
                        FileStatus);
         if FileStatus = SPARK_IO.Ok then
            SwitchFileFound := True;
            SwitchFile := SwitchFileLocal;
         else
            SwitchFileFound := False;
            SwitchFile := SPARK_IO.Null_File;
         end if;
      end OpenFile;

      procedure ProcessSwitchFile (SwitchFile : in     SPARK_IO.File_Type;
                                   Ptr        :    out FileSystem.TypCmdLineRange;
                                   CmdLine    : in out FileSystem.TypCmdLine)
      --# global in out SPARK_IO.File_Sys;
      --# derives CmdLine,
      --#         SPARK_IO.File_Sys from *,
      --#                                SPARK_IO.File_Sys,
      --#                                SwitchFile &
      --#         Ptr               from SPARK_IO.File_Sys,
      --#                                SwitchFile;
      is
         LocalPtr       : FileSystem.TypCmdLineRange;
         CurrentLine    : EStrings.T;
         InQuotedString : Boolean;

         function StripLeadingSpaces (S : EStrings.T)
                                     return EStrings.T
         is
            Ptr : EStrings.Positions := 1;
         begin
            loop
               exit when Ptr > S.Length;
               exit when not IsWhiteSpace (S.Content (Ptr));
               Ptr := Ptr + 1;
            end loop;
            return StringUtilities.Section (S, Ptr, (S.Length - Ptr) + 1);
         end StripLeadingSpaces;

         function StripComments (S : EStrings.T)
                               return EStrings.T
         is
            Res : EStrings.T;
         begin
            if S.Content (1) = '-' and then S.Content (2) = '-' then
               Res := EStrings.EmptyString;
            else
               Res := S;
            end if;
            return Res;
         end StripComments;

         function LineIsEmpty (S : EStrings.T) return Boolean
         is
         begin
            return S.Length = 0;
         end LineIsEmpty;

      begin --ProcessSwitchFile
         LocalPtr := FileSystem.TypCmdLineRange'First;
         InQuotedString := False;

         loop --look for non-empty lines in switch file and process them
            exit when  SPARK_IO.End_Of_File (SwitchFile);

            EStrings.GetLine (SwitchFile, -- to get
                                     CurrentLine);
            CurrentLine := StripComments (StripLeadingSpaces (CurrentLine));
            if not LineIsEmpty (CurrentLine) then
               for I in EStrings.Lengths range 1 .. CurrentLine.Length loop
                  --premature exit if comment found in line
                  exit when CurrentLine.Content (I) = '-' and then
                     I < CurrentLine.Length and then
                     CurrentLine.Content (I + 1) = '-';

                  if CurrentLine.Content (I) = Ada.Characters.Latin_1.Quotation then
                     InQuotedString := not InQuotedString;
                  end if;

                  if InQuotedString then
                     -- In a quoted string, we leave the character alone
                     CmdLine (LocalPtr) := CurrentLine.Content (I);
                  else
                     if IsWhiteSpace (CurrentLine.Content (I)) then
                        -- Not in a quoted string, so whitespace gets replaced
                        -- by the host's preferred argument separator character
                        CmdLine (LocalPtr) := FileSystem.ArgumentSeparator;
                     else
                        -- To make switch files platform independent, we substitute
                        -- a '-' that is the first character on the line, or
                        -- is preceeded by whitespace, by the host's preferred
                        -- switch character (which might be '-', so no change) or
                        -- might be something else, like '/'
                        --
                        -- This means that default switch files containing '-'
                        -- will work unchanged on all platforms.
                        if ((I = 1) and then (CurrentLine.Content (I) = '-')) or
                           ((I > 1) and then (CurrentLine.Content (I) = '-') and then
                            IsWhiteSpace (CurrentLine.Content (I - 1))) then

                           CmdLine (LocalPtr) := FileSystem.SwitchCharacter;
                        else
                           CmdLine (LocalPtr) := CurrentLine.Content (I);
                        end if;
                     end if;
                  end if;

                  LocalPtr := LocalPtr + 1;
               end loop;

               CmdLine (LocalPtr) := FileSystem.ArgumentSeparator;
               LocalPtr := LocalPtr + 1;
            end if;
         end loop;
         Ptr := LocalPtr + 2;
      end ProcessSwitchFile;


   begin --ReadDefaultSwitches
      CmdLinePtrLocal := FileSystem.TypCmdLineRange'First;
      CmdLineLocal := FileSystem.TypCmdLine'
         (others => FileSystem.ArgumentSeparator);

      OpenFile (SwitchFileFound, SwitchFile);
      if SwitchFileFound then
         ProcessSwitchFile (SwitchFile,
                            CmdLinePtrLocal,
                            CmdLineLocal);
         --# accept F, 10, Unused,     "Returned status not used here" &
         --#        F, 10, SwitchFile, "Returned handle not used here";
         SPARK_IO.Close (SwitchFile, Unused); --expect flow error
      end if;

      DefaultSwitchesFound := SwitchFileFound;
      CmdLinePtr := CmdLinePtrLocal;
      CmdLine    := CmdLineLocal;
      --# accept F, 33, Unused, "Unused not referenced here";
   end ReadDefaultSwitches;

   procedure SkipSpaces (CommandString : in out CommandStrings)
   is
   begin
      loop
         exit when CommandString.CurrentPosition >
           ExaminerConstants.MaxCommandStringLength; --477

         exit when not
           ((CommandString.Contents (CommandString.CurrentPosition) = ' ') or
              (CommandString.Contents (CommandString.CurrentPosition) =
                 FileSystem.ArgumentSeparator));

         CommandString.CurrentPosition := CommandString.CurrentPosition + 1;
      end loop;
   end SkipSpaces;

   procedure ReadTheString (CommandString : in out CommandStrings;
                            NextSymbol    : in out Symbols)
   is
      CurrentCh : Character;
      Ix        : EStrings.Positions;
   begin
      Ix := 1;

      NextSymbol.TheString.Length := EStrings.MaxStringLength;
      CurrentCh := CommandString.Contents (CommandString.CurrentPosition);

      loop
         -- Skip over quotes, but allow spaces in the string
         if CurrentCh /= Ada.Characters.Latin_1.Quotation then
            NextSymbol.TheString.Content (Ix) := CurrentCh;
            NextSymbol.TheString.Length := Ix;

            if Ix < EStrings.Positions'Last then
               Ix := Ix + 1;
            end if;
         end if;

         CommandString.CurrentPosition := CommandString.CurrentPosition + 1;
         CurrentCh := CommandString.Contents (CommandString.CurrentPosition);

         exit when CommandString.CurrentPosition >
           ExaminerConstants.MaxCommandStringLength or else
           FileSystem.IsAnArgumentTerminator (CurrentCh);
      end loop;
   end ReadTheString;

   procedure GetNextSymbol (CommandString : in out CommandStrings;
                            NextSymbol    :    out Symbols)
   --# derives CommandString,
   --#         NextSymbol    from CommandString;
   is
      LocalNextSymbol : Symbols;
   begin
      -- This procedure is intended to return NextSymbol; however, if the
      -- symbol is not a string then the string field is not set.  Although
      -- it is not used in these circumstances its lack of definition
      -- causes so many flow errors its is better to use an aggregate to
      -- initialize NextSymbol here, and then assign the final value
      -- to NextSymbol (for compatibility with the Ada83 "out parameter" rule
      LocalNextSymbol := Symbols'(Typ       => SEmpty,
                                  TheString => EStrings.EmptyString);

      SkipSpaces (CommandString);

      if CommandString.CurrentPosition <=
        ExaminerConstants.MaxCommandStringLength then
         case CommandString.Contents (CommandString.CurrentPosition) is
            when '=' =>
               LocalNextSymbol.Typ := SEqual;
               CommandString.CurrentPosition := CommandString.CurrentPosition + 1;
            when '/' =>
               --this condition is invariant for any particular system, we are actually
               --simulating conditional compilation for different target platforms.
               --Intended behaviour is correct despite flow error that will result.
               --# accept F, 22, "Stable expression here OK";
               if FileSystem.UseUnixCommandLine then
                  LocalNextSymbol.Typ := SString;
                  ReadTheString (CommandString, LocalNextSymbol);
               else -- Windows
                  LocalNextSymbol.Typ := SSlash;
                  CommandString.CurrentPosition :=
                    CommandString.CurrentPosition + 1;
               end if;
               --# end accept;
            when ',' =>
               --this condition is invariant for any particular system, we are actually
               --simulating conditional compilation for different target platforms.
               --Intended behaviour is correct despite flow error that will result.
               --# accept F, 22, "Stable expression here OK";
               if FileSystem.UseUnixCommandLine then
                  LocalNextSymbol.Typ := SString;
                  ReadTheString (CommandString, LocalNextSymbol); --675
               else -- Windows
                  LocalNextSymbol.Typ := SComma;
                  CommandString.CurrentPosition := CommandString.CurrentPosition + 1;
               end if;
               --# end accept;
            when '-' =>
               -- We allow '-' as a switch character on all platforms
               LocalNextSymbol.Typ := SSlash;
               CommandString.CurrentPosition :=
                 CommandString.CurrentPosition + 1;
            when others =>
               LocalNextSymbol.Typ := SString;
               ReadTheString (CommandString, LocalNextSymbol);
         end case;
      else
         -- Exceeded maximum command line length
         LocalNextSymbol.Typ := SEmpty;
      end if;

      NextSymbol := LocalNextSymbol;
   end GetNextSymbol;

   procedure ReadOption (OptName       :    out EStrings.T;
                         OptNameOK     :    out Boolean;
                         OptVal        :    out EStrings.T;
                         OptValOK      :    out Boolean;
                         CommandString : in out CommandStrings;
                         NextSymbol    :    out Symbols)
   --# derives CommandString,
   --#         NextSymbol,
   --#         OptName,
   --#         OptNameOK,
   --#         OptVal,
   --#         OptValOK      from CommandString;
   --  pre  NextSymbol.Typ = SSlash;
   is
      OptNameOkLocal  : Boolean;
      LocalNextSymbol : Symbols;
   begin
      OptVal := EStrings.EmptyString;
      GetNextSymbol (CommandString, LocalNextSymbol);
      OptNameOkLocal := LocalNextSymbol.Typ = SString;
      OptName := LocalNextSymbol.TheString;
      GetNextSymbol (CommandString, LocalNextSymbol);
      if OptNameOkLocal and LocalNextSymbol.Typ = SEqual then
         GetNextSymbol (CommandString, LocalNextSymbol);
         OptValOK := LocalNextSymbol.Typ = SString;
         OptVal := LocalNextSymbol.TheString;
         GetNextSymbol (CommandString, LocalNextSymbol);
      else
         OptValOK := False;
      end if;
      OptNameOK  := OptNameOkLocal;
      NextSymbol := LocalNextSymbol;
   end ReadOption;

   procedure CheckOptionName (OptName : in     EStrings.T;
                              Str     : in     String;
                              OK      :    out Boolean)
   is
      LOK : Boolean := False;
   begin
      if OptName.Length <= Str'Length then
         for I in EStrings.Lengths range 1 .. OptName.Length loop
            LOK := Ada.Characters.Handling.To_Lower (OptName.Content (I)) =
               Ada.Characters.Handling.To_Lower (Str (I));
            exit when not LOK;
         end loop;
         OK := LOK;
      else
         OK := False;
      end if;
   end CheckOptionName;

   procedure ParseCommandOptions (CommandString : in out CommandStrings;
                                  NextSymbol    : in out Symbols;
                                  DoRealRTCs    : in out Boolean)
   --# global in out CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives CommandLineData.Content,
   --#         CommandString,
   --#         DoRealRTCs,
   --#         NextSymbol,
   --#         SPARK_IO.File_Sys       from *,
   --#                                      CommandLineData.Content,
   --#                                      CommandString,
   --#                                      NextSymbol;
   is
      IndexFound,
      OutputDirectoryFound,
      WarningFound,
      NoWarningFound,
      SourceFound,
      ListingFound,
      OutputFound,
      ReportFound,
      ReductionFound,
      ErrorExplanationsFound,
      JustificationOptionFound,
      RulesFound,
      NoReportOptionFound,
      RealRTCsFound,
      SyntaxCheckFound,
      StatisticsFound,
      FDLreserveOptionFound,
      FDLignoreOptionFound,
      DoInformationFlowFound,
      TargetDataFound,
      ConfigFileFound,
      ToolVariantFound,
      DebugEnabledFound,
      AnnotationCharacterFound,
      HTMLFound,
      PlainOutputFound,
      NoDurationFound,
      BriefFound,
      XMLFound,
      OriginalFlowErrorsFound,
      NoListingsFound,
      DictFound,
      AnalysisProfileFound : Boolean := False;

      OptName,
      OptVal     : EStrings.T;
      OptNameOK,
      OptValOK   : Boolean;

      procedure Process_A
      --# global in     OptName;
      --#        in     OptVal;
      --#        in out AnnotationCharacterFound;
      --#        in out CommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --#        in out ToolVariantFound;
      --# derives AnnotationCharacterFound,
      --#         ToolVariantFound         from *,
      --#                                       OptName &
      --#         CommandLineData.Content  from *,
      --#                                       AnnotationCharacterFound,
      --#                                       OptName,
      --#                                       OptVal,
      --#                                       ToolVariantFound &
      --#         SPARK_IO.File_Sys        from *,
      --#                                       AnnotationCharacterFound,
      --#                                       CommandLineData.Content,
      --#                                       OptName,
      --#                                       ToolVariantFound;
      is
         OK : Boolean;
      begin
         --# accept W, 169, CommandLineData.Content.Valid,    "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Spark95,  "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.AnnoChar, "Direct updates OK here";
         case Ada.Characters.Handling.To_Lower (OptName.Content (2)) is
            when 'd' => -- 2nd letter
               CheckOptionName (OptName, "ada83", OK);
               if OK then
                  CommandLineData.Content.Valid := CommandLineData.IsSpark95 and
                    not ToolVariantFound and
                    not (CommandLineData.RavenscarSelected);
                  PossibleError (ESToolVariant);
                  ToolVariantFound := True;
                  CommandLineData.Content.Spark95 := False;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;

            when 'n' => -- 2nd letter
               CheckOptionName (OptName, "annotation_character", OK);
               if OK then
                  CommandLineData.Content.Valid := not AnnotationCharacterFound;
                  PossibleError (ESanno);
                  AnnotationCharacterFound := True;
                  CommandLineData.Content.AnnoChar := OptVal.Content (1);
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;

            when others =>
               CommandLineData.Content.Valid := False;
               PossibleError2 (ESInvalidOption, OptName);
         end case;
         --# end accept;
      end Process_A;

      procedure Process_B
      --# global in     OptName;
      --#        in out BriefFound;
      --#        in out CommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --# derives BriefFound              from *,
      --#                                      OptName &
      --#         CommandLineData.Content,
      --#         SPARK_IO.File_Sys       from *,
      --#                                      BriefFound,
      --#                                      CommandLineData.Content,
      --#                                      OptName;
      is
         OK : Boolean;
      begin
         --# accept W, 169, CommandLineData.Content.Valid, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Brief, "Direct updates OK here";
         CheckOptionName (OptName, "brief", OK);
         if OK then
            CommandLineData.Content.Valid := not BriefFound;
            PossibleError (ESBrief);
            BriefFound := True;
            CommandLineData.Content.Brief := True;
         else
            CommandLineData.Content.Valid := False;
            PossibleError2 (ESInvalidOption, OptName);
         end if;
         --# end accept;
      end Process_B;

      procedure Process_C
      --# global in     OptName;
      --#        in     OptVal;
      --#        in     OptValOK;
      --#        in     TargetDataFound;
      --#        in out CommandLineData.Content;
      --#        in out ConfigFileFound;
      --#        in out SPARK_IO.File_Sys;
      --# derives CommandLineData.Content from *,
      --#                                      ConfigFileFound,
      --#                                      OptName,
      --#                                      OptVal,
      --#                                      OptValOK,
      --#                                      TargetDataFound &
      --#         ConfigFileFound         from *,
      --#                                      OptName,
      --#                                      OptValOK,
      --#                                      TargetDataFound &
      --#         SPARK_IO.File_Sys       from *,
      --#                                      CommandLineData.Content,
      --#                                      ConfigFileFound,
      --#                                      OptName,
      --#                                      OptValOK,
      --#                                      TargetDataFound;
      is
         OK       : Boolean;
         FileName : EStrings.T;
      begin
         --# accept W, 169, CommandLineData.Content.Valid, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.TargetConfig, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.TargetConfigFile, "Direct updates OK here";

         case Ada.Characters.Handling.To_Lower (OptName.Content (2)) is
            when 'o' => -- 2nd letter
               case Ada.Characters.Handling.To_Lower (OptName.Content (3)) is -- 3rd letter
                  when 'n' =>
                     case Ada.Characters.Handling.To_Lower (OptName.Content (4)) is -- 4th letter
                        when 'f' =>
                           CheckOptionName (OptName, "config_file", OK);
                           if OK then
                              if TargetDataFound then
                                 CommandLineData.Content.Valid := False;
                                 PossibleError (ESDataAndConfig);
                              else
                                 CommandLineData.Content.Valid := OptValOK;
                                 PossibleError (ESConfigFileName);
                                 if OptValOK then
                                    CommandLineData.Content.Valid := not ConfigFileFound;
                                    PossibleError (ESConfigFile);
                                 end if;
                                 if CommandLineData.Content.Valid then
                                    ConfigFileFound := True;
                                    CommandLineData.Content.TargetConfig := True;
                                    FileName := OptVal;
                                    FileSystem.CheckExtension
                                      (FileName,
                                       FileSystem.CaseOfFileExtensions
                                         (CommandLineData.DEFAULT_CONFIG_EXTENSION));
                                    CommandLineData.Content.TargetConfigFile := FileName;
                                 end if;
                              end if;
                           else
                              CommandLineData.Content.Valid := False;
                              PossibleError2 (ESInvalidOption, OptName);
                           end if;
                        when 'c' => -- 4th letter
                           -- -concat is removed in release 8.1.1 and above, but
                           -- recognize it here and warn the user.
                           CheckOptionName (OptName, "concatenate", OK);
                           if OK then
                              CommandLineData.Content.Valid := False;
                              PossibleError (SOFstyle);
                           else
                              CommandLineData.Content.Valid := False;
                              PossibleError2 (ESInvalidOption, OptName);
                           end if;
                        when others =>
                           CommandLineData.Content.Valid := False;
                           PossibleError2 (ESInvalidOption, OptName);
                     end case;
                  when others =>
                     CommandLineData.Content.Valid := False;
                     PossibleError2 (ESInvalidOption, OptName);
               end case;
            when others =>
               CommandLineData.Content.Valid := False;
               PossibleError2 (ESInvalidOption, OptName);
         end case;
         --# end accept;
      end Process_C;

      procedure Process_D
      --# global in     OptName;
      --#        in     OptVal;
      --#        in     OptValOK;
      --#        in out CommandLineData.Content;
      --#        in out DebugEnabledFound;
      --#        in out DictFound;
      --#        in out SPARK_IO.File_Sys;
      --# derives CommandLineData.Content,
      --#         SPARK_IO.File_Sys       from *,
      --#                                      CommandLineData.Content,
      --#                                      DebugEnabledFound,
      --#                                      DictFound,
      --#                                      OptName,
      --#                                      OptVal,
      --#                                      OptValOK &
      --#         DebugEnabledFound       from *,
      --#                                      OptName &
      --#         DictFound               from *,
      --#                                      OptName,
      --#                                      OptValOK;
      is
         Default_Dict_Extension : constant EStrings.T :=
           EStrings.T'(3, EStrings.Contents'('d', 'i', 'c', others => ' '));

         OK       : Boolean;
         FileName : EStrings.T;
      begin
         --# accept W, 169, CommandLineData.Content.Valid, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.WriteDict, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.DictFileName, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Debug.Enabled,   "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Debug.HTML,   "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Debug.Expressions,   "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Debug.LookupTrace,   "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Debug.FileNames,   "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Debug.Units,   "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Debug.Invariants,   "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Debug.Components,   "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Debug.Rho,   "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Debug.FDL_Ranking, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Debug.VCG, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Debug.VCG_All, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Debug.DAG, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Debug.Parser,   "Direct updates OK here";
         case Ada.Characters.Handling.To_Lower (OptName.Content (2)) is
            when 'i' => -- 2nd letter
               CheckOptionName (OptName, "dictionary_file", OK);
               if OK then
                  CommandLineData.Content.Valid := OptValOK and not DictFound;
                  PossibleError (ESDict);

                  if CommandLineData.Content.Valid then
                     DictFound := True;
                     CommandLineData.Content.WriteDict := True;
                     FileName := OptVal;
                     FileSystem.CheckExtension
                       (FileName,
                        FileSystem.CaseOfFileExtensions (Default_Dict_Extension));
                     CommandLineData.Content.DictFileName := FileName;
                  end if;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;
            when 'e' => -- 2nd letter
               CheckOptionName (OptName, "debug", OK);

               if OK then
                  CommandLineData.Content.Valid := not DebugEnabledFound;
                  PossibleError (ESDebug);
                  DebugEnabledFound := True;
                  CommandLineData.Content.Debug.Enabled := True;

                  -- The -debug switch may also have a parameter which
                  -- is a list of character-codes for specific debug
                  -- options.  These are:
                  --   c  -  Print component manager state
                  --   e  -  Debug Expresion syntax and tree walking
                  --   f  -  Trace filename storage/open/create
                  --   h  -  Debug HTML generation
                  --   i  -  Print default loop invariants
                  --   k  -  Trace ranking and printing of FDL declarations
                  --   l  -  Trace dictionary Look-Ups
                  --   p  -  Print parser state on detection of syntax Error
                  --   r  -  Print computed Rho relation for each subprogram
                  --   u  -  Trace required units and index lookups
                  --   v  -  Print VCG State and BPG after DAG.BuildGraph
                  --   V  -  As v, but also print BPG during iteration of Graph.GenVCs
                  --   d  -  Print FDL DAG following BuildExpnDAG or
                  --         BuildAnnotationExpnDAG
                  -- These codes may be combined, so
                  --   -debug=eh and
                  --   -debug=he are both allowed and are equivalent

                  if OptValOK then
                     for I in Integer range 1 .. OptVal.Length loop
                        case OptVal.Content (I) is
                           when 'c' =>
                              CommandLineData.Content.Debug.Components  := True;
                           when 'd' =>
                              CommandLineData.Content.Debug.DAG         := True;
                           when 'e' =>
                              CommandLineData.Content.Debug.Expressions := True;
                           when 'f' =>
                              CommandLineData.Content.Debug.FileNames   := True;
                           when 'h' =>
                              CommandLineData.Content.Debug.HTML        := True;
                           when 'i' =>
                              CommandLineData.Content.Debug.Invariants  := True;
                           when 'k' =>
                              CommandLineData.Content.Debug.FDL_Ranking := True;
                           when 'l' =>
                              CommandLineData.Content.Debug.LookupTrace := True;
                           when 'p' =>
                              CommandLineData.Content.Debug.Parser      := True;
                           when 'r' =>
                              CommandLineData.Content.Debug.Rho         := True;
                           when 'u' =>
                              CommandLineData.Content.Debug.Units       := True;
                           when 'v' =>
                              CommandLineData.Content.Debug.VCG         := True;
                           when 'V' =>
                              CommandLineData.Content.Debug.VCG         := True;
                              CommandLineData.Content.Debug.VCG_All     := True;
                           when others =>
                              CommandLineData.Content.Valid := False;
                        end case;
                     end loop;
                     PossibleError (ESDebug);
                  end if;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;

            when others =>
               -- starts "d" but not valid
               CommandLineData.Content.Valid := False;
               PossibleError2 (ESInvalidOption, OptName);
         end case;
         --# end accept;
      end Process_D;

      procedure Process_E
      --# global in     FDLignoreOptionFound;
      --#        in     OptName;
      --#        in     OptVal;
      --#        in out CommandLineData.Content;
      --#        in out ErrorExplanationsFound;
      --#        in out ReductionFound;
      --#        in out SPARK_IO.File_Sys;
      --# derives CommandLineData.Content,
      --#         SPARK_IO.File_Sys       from *,
      --#                                      CommandLineData.Content,
      --#                                      ErrorExplanationsFound,
      --#                                      FDLignoreOptionFound,
      --#                                      OptName,
      --#                                      OptVal,
      --#                                      ReductionFound &
      --#         ErrorExplanationsFound,
      --#         ReductionFound          from *,
      --#                                      OptName;
      is
         OK : Boolean;
      begin
         --# accept W, 169, CommandLineData.Content.Valid, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.RedType, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.ErrorExplanation, "Direct updates OK here";
         case Ada.Characters.Handling.To_Lower (OptName.Content (2)) is
            when 'x' => -- 2nd letter
               CheckOptionName (OptName, "exp_checks", OK);
               if OK then
                  CommandLineData.Content.Valid :=
                    not (ReductionFound or
                           (FDLignoreOptionFound and
                              CommandLineData.Content.FDLmangle = EStrings.EmptyString));
                  PossibleError (ESReduction);
                  ReductionFound := True;
                  CommandLineData.Content.RedType := CommandLineData.DoExp;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;
            when 'r' => -- 2nd letter
               CheckOptionName (OptName, "error_explanations", OK);
               if OK then
                  CommandLineData.Content.Valid := not ErrorExplanationsFound;
                  PossibleError (ESErrorExplanations);
                  if CommandLineData.Content.Valid then
                     ErrorExplanationsFound := True;
                     -- now check argument - must be one of Off, FirstOccurrence, EveryOccurrence
                     case Ada.Characters.Handling.To_Lower (OptVal.Content (1)) is
                        when 'o' => -- 1st letter of option
                           CheckOptionName (OptVal, "off", OK);
                           if OK then
                              CommandLineData.Content.ErrorExplanation := CommandLineData.Off;
                           else -- error involving 'off'
                              CommandLineData.Content.Valid := False;
                              PossibleError2 (ESErrorExplanationOption, OptVal);
                           end if;
                        when 'f' => -- 1st letter of option
                           CheckOptionName (OptVal, "first_occurrence", OK);
                           if OK then
                              CommandLineData.Content.ErrorExplanation := CommandLineData.FirstOccurrence;
                           else -- error involving 'first_occurrence'
                              CommandLineData.Content.Valid := False;
                              PossibleError2 (ESErrorExplanationOption, OptVal);
                           end if;
                        when 'e' => -- 1st letter of option
                           CheckOptionName (OptVal, "every_occurrence", OK);
                           if OK then
                              CommandLineData.Content.ErrorExplanation := CommandLineData.EveryOccurrence;
                           else -- error involving 'every_occurrence'
                              CommandLineData.Content.Valid := False;
                              PossibleError2 (ESErrorExplanationOption, OptVal);
                           end if;
                        when others =>
                           CommandLineData.Content.Valid := False;
                           PossibleError2 (ESErrorExplanationOption, OptVal);
                     end case;
                  end if;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;
            when others =>
               CommandLineData.Content.Valid := False;
               PossibleError2 (ESInvalidOption, OptName);
         end case;
         --# end accept;
      end Process_E;


      procedure Process_F
      --# global in     OptName;
      --#        in     OptVal;
      --#        in     OptValOK;
      --#        in     ReductionFound;
      --#        in out CommandLineData.Content;
      --#        in out DoInformationFlowFound;
      --#        in out FDLignoreOptionFound;
      --#        in out FDLReserveOptionFound;
      --#        in out SPARK_IO.File_Sys;
      --# derives CommandLineData.Content,
      --#         SPARK_IO.File_Sys       from *,
      --#                                      CommandLineData.Content,
      --#                                      DoInformationFlowFound,
      --#                                      FDLignoreOptionFound,
      --#                                      FDLReserveOptionFound,
      --#                                      OptName,
      --#                                      OptVal,
      --#                                      OptValOK,
      --#                                      ReductionFound &
      --#         DoInformationFlowFound  from *,
      --#                                      OptName,
      --#                                      OptValOK &
      --#         FDLignoreOptionFound,
      --#         FDLReserveOptionFound   from *,
      --#                                      OptName,
      --#                                      OptVal,
      --#                                      OptValOK;
      is
         OK : Boolean;
      begin
         --# accept W, 169, CommandLineData.Content.Valid, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.FDLMangle,   "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.FDLreserved,   "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.DoInformationFlow,   "Direct updates OK here";
         case Ada.Characters.Handling.To_Lower (OptName.Content (2)) is
            when 'd' => -- 2nd letter
               CheckOptionName (OptName, "fdl_identifiers", OK);
               if OK then
                  CommandLineData.Content.Valid := not (FDLreserveOptionFound or FDLignoreOptionFound);
                  PossibleError (ESFDLoption);
                  if OptValOK then
                     CheckOptionName (OptVal, "accept", OK);
                     if OK then
                        CommandLineData.Content.FDLreserved := False; -- Equivalent to null;
                        CommandLineData.Content.Valid := not ReductionFound;
                        FDLignoreOptionFound := True;
                        PossibleError (ESFDLoption);
                     else
                        CheckOptionName (OptVal, "reject", OK);
                        if OK then
                           CommandLineData.Content.FDLreserved := True;
                           FDLreserveOptionFound := True;
                        else
                           CommandLineData.Content.FDLreserved := False; -- Will not reject fdl reserved words
                           CommandLineData.Content.FDLmangle := OptVal;  -- But will mangle them
                           FDLignoreOptionFound := True;
                        end if;
                     end if;
                  else
                     PutWarning ("Warning: The /fdl option is now deprecated. Please use /fdl=reject");
                     FDLreserveOptionFound := True;
                     CommandLineData.Content.FDLreserved := True;
                  end if;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;

            when 'l' => -- 2nd letter
               CheckOptionName (OptName, "flow_analysis", OK);
               if OK then
                  CommandLineData.Content.Valid := OptValOK and not DoInformationFlowFound;
                  PossibleError2 (ESInformationFlow, OptVal);
                  if CommandLineData.Content.Valid then --go on to check selection
                     DoInformationFlowFound := True;
                     case Ada.Characters.Handling.To_Lower (OptVal.Content (1)) is
                        when 'i' =>
                           CheckOptionName (OptVal, "information", OK);
                           if OK then
                              CommandLineData.Content.DoInformationFlow := True;
                           else  -- something which begins with 'i' other than information
                              CommandLineData.Content.Valid := False;
                              PossibleError2 (ESInformationFlow, OptVal);
                           end if;

                        when 'd' =>
                           CheckOptionName (OptVal, "data", OK);
                           if OK then
                              -- Here we have a potentially valid selection flow_analysis=data.
                              -- This is only allowed if information flow integrity checking has
                              -- not been selected
                              if (CommandLineData.Content.InfoFlowPolicy = CommandLineData.None) then
                                 CommandLineData.Content.DoInformationFlow := False;
                              else
                                 CommandLineData.Content.Valid := False;
                                 PossibleError (ESDataFlowAndInfoFlowPolicy);
                              end if;
                           else -- something which begins with 'd' other than data
                              CommandLineData.Content.Valid := False;
                              PossibleError2 (ESInformationFlow, OptVal);
                           end if;

                        when others => -- doesn't begin with 'i' or 'd'
                           CommandLineData.Content.Valid := False;
                           PossibleError2 (ESInformationFlow, OptVal);
                     end case;
                  end if;
               else -- begins with 'f' but not fdl... or flow...
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;

            when others =>
               CommandLineData.Content.Valid := False;
               PossibleError2 (ESInvalidOption, OptName);
         end case;
         --# end accept;
      end Process_F;

      procedure Process_H
      --# global in     NoReportOptionFound;
      --#        in     OptName;
      --#        in     OptVal;
      --#        in     OptValOK;
      --#        in     XMLFound;
      --#        in out CommandLineData.Content;
      --#        in out HTMLFound;
      --#        in out SPARK_IO.File_Sys;
      --# derives CommandLineData.Content from *,
      --#                                      HTMLFound,
      --#                                      NoReportOptionFound,
      --#                                      OptName,
      --#                                      OptVal,
      --#                                      OptValOK,
      --#                                      XMLFound &
      --#         HTMLFound               from *,
      --#                                      OptName &
      --#         SPARK_IO.File_Sys       from *,
      --#                                      CommandLineData.Content,
      --#                                      HTMLFound,
      --#                                      NoReportOptionFound,
      --#                                      OptName,
      --#                                      XMLFound;
      is
         OK       : Boolean;
      begin
         --# accept W, 169, CommandLineData.Content.Valid, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.HTML, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.HTMLDirectory, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.PlainOutput, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.HelpRequested, "Direct updates OK here";
         case Ada.Characters.Handling.To_Lower (OptName.Content (2)) is
            when 't' => -- 2nd letter
               CheckOptionName (OptName, "html", OK);
               if OK then
                  CommandLineData.Content.Valid :=
                    not (HTMLFound or NoReportOptionFound or XMLFound);

                  if XMLFound then
                     PossibleError (ESXMLandHTML);
                  elsif NoReportOptionFound then
                     PossibleError (ESHTMLnorep);
                  else
                     PossibleError (EShtml);
                  end if;

                  HTMLFound := True;
                  CommandLineData.Content.HTML := True;
                  if OptValOK then
                     CommandLineData.Content.HTMLDirectory := OptVal;
                  end if;

               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;
            when 'e' => -- 2nd letter
               CheckOptionName (OptName, "help", OK);
               if OK then
                  -- Examiner Help requested, so abandon all further
                  -- option processing by setting Valid = False.
                  CommandLineData.Content.Valid := False;
                  CommandLineData.Content.HelpRequested := True;
                  -- Override setting of PlainOutput so Help information
                  -- always appears with Examiner version number and date.
                  CommandLineData.Content.PlainOutput := False;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;
            when others =>
               CommandLineData.Content.Valid := False;
               PossibleError2 (ESInvalidOption, OptName);
         end case;
         --# end accept;
      end Process_H;

      procedure Process_I
      --# global in     OptName;
      --#        in     OptVal;
      --#        in     OptValOK;
      --#        in out CommandLineData.Content;
      --#        in out IndexFound;
      --#        in out SPARK_IO.File_Sys;
      --# derives CommandLineData.Content from *,
      --#                                      IndexFound,
      --#                                      OptName,
      --#                                      OptVal,
      --#                                      OptValOK &
      --#         IndexFound              from *,
      --#                                      OptName,
      --#                                      OptValOK &
      --#         SPARK_IO.File_Sys       from *,
      --#                                      CommandLineData.Content,
      --#                                      IndexFound,
      --#                                      OptName,
      --#                                      OptValOK;
      is
         OK       : Boolean;
         FileName : EStrings.T;
      begin
         --# accept W, 169, CommandLineData.Content.Valid, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Index, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.IndexFileName, "Direct updates OK here";
         CheckOptionName (OptName, "index_file", OK);
         if OK then
            CommandLineData.Content.Valid := OptValOK;
            PossibleError (ESIndexName);
            if OptValOK then
               CommandLineData.Content.Valid := not IndexFound;
               PossibleError (ESIndex);
            end if;

            if CommandLineData.Content.Valid then
               IndexFound := True;
               CommandLineData.Content.Index := True;
               FileName := OptVal;

               FileSystem.CheckExtension
                 (FileName,
                  FileSystem.CaseOfFileExtensions
                    (CommandLineData.DEFAULT_INDEX_EXTENSION));
               CommandLineData.Content.IndexFileName := FileName;
            end if;
         else
            CommandLineData.Content.Valid := False;
            PossibleError2 (ESInvalidOption, OptName);
         end if;
         --# end accept;
      end Process_I;

      procedure Process_J
      --# global in     OptName;
      --#        in     OptVal;
      --#        in out CommandLineData.Content;
      --#        in out JustificationOptionFound;
      --#        in out SPARK_IO.File_Sys;
      --# derives CommandLineData.Content,
      --#         SPARK_IO.File_Sys        from *,
      --#                                       CommandLineData.Content,
      --#                                       JustificationOptionFound,
      --#                                       OptName,
      --#                                       OptVal &
      --#         JustificationOptionFound from *,
      --#                                       OptName;
      is
         OK       : Boolean;
      begin
         --# accept W, 169, CommandLineData.Content.Valid, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.JustificationOption, "Direct updates OK here";
         CheckOptionName (OptName, "justification_option", OK);
         if OK then
            CommandLineData.Content.Valid := not JustificationOptionFound;
            PossibleError (ESJustificationRepeated);
            if CommandLineData.Content.Valid then
               JustificationOptionFound := True;
               -- now check argument - must be one of Ignore, Full, or Brief
               case Ada.Characters.Handling.To_Lower (OptVal.Content (1)) is
                  when 'i' => -- 1st letter of option
                     CheckOptionName (OptVal, "ignore", OK);
                     if OK then
                        CommandLineData.Content.JustificationOption := CommandLineData.Ignore;
                     else -- error involving 'ignore'
                        CommandLineData.Content.Valid := False;
                        PossibleError (ESJustificationOption);
                     end if;
                  when 'f' => -- 1st letter of option
                     CheckOptionName (OptVal, "full", OK);
                     if OK then
                        CommandLineData.Content.JustificationOption := CommandLineData.Full;
                     else -- error involving 'full'
                        CommandLineData.Content.Valid := False;
                        PossibleError (ESJustificationOption);
                     end if;
                  when 'b' => -- 1st letter of option
                     CheckOptionName (OptVal, "brief", OK);
                     if OK then
                        CommandLineData.Content.JustificationOption := CommandLineData.Brief;
                     else -- error involving 'brief'
                        CommandLineData.Content.Valid := False;
                        PossibleError (ESJustificationOption);
                     end if;
                  when others =>
                     CommandLineData.Content.Valid := False;
                     PossibleError (ESJustificationOption);
               end case;
            end if;
         else
            CommandLineData.Content.Valid := False;
            PossibleError2 (ESInvalidOption, OptName);
         end if;
         --# end accept;
      end Process_J;

      procedure Process_L
      --# global in     OptName;
      --#        in     OptVal;
      --#        in     OptValOK;
      --#        in out CommandLineData.Content;
      --#        in out ListingFound;
      --#        in out SPARK_IO.File_Sys;
      --# derives CommandLineData.Content from *,
      --#                                      ListingFound,
      --#                                      OptName,
      --#                                      OptVal,
      --#                                      OptValOK &
      --#         ListingFound            from *,
      --#                                      OptName &
      --#         SPARK_IO.File_Sys       from *,
      --#                                      CommandLineData.Content,
      --#                                      ListingFound,
      --#                                      OptName,
      --#                                      OptValOK;
      is
         OK       : Boolean;
      begin
         --# accept W, 169, CommandLineData.Content.Valid, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.ListingExtension, "Direct updates OK here";
         CheckOptionName (OptName, "listing_extension", OK);
         if OK then
            CommandLineData.Content.Valid := OptValOK and not ListingFound;
            PossibleError (ESListingExt);
            ListingFound := True;
            CommandLineData.Content.ListingExtension := OptVal;
         else
            CommandLineData.Content.Valid := False;
            PossibleError2 (ESInvalidOption, OptName);
         end if;
         --# end accept;
      end Process_L;


      procedure Process_N
      --# global in     FDLReserveOptionFound;
      --#        in     HTMLFound;
      --#        in     OptName;
      --#        in     ReductionFound;
      --#        in     XMLFound;
      --#        in out CommandLineData.Content;
      --#        in out ConfigFileFound;
      --#        in out DictFound;
      --#        in out FDLignoreOptionFound;
      --#        in out IndexFound;
      --#        in out NoDurationFound;
      --#        in out NoListingsFound;
      --#        in out NoReportOptionFound;
      --#        in out NoWarningFound;
      --#        in out OutputFound;
      --#        in out ReportFound;
      --#        in out SPARK_IO.File_Sys;
      --#        in out StatisticsFound;
      --#        in out TargetDataFound;
      --# derives CommandLineData.Content,
      --#         SPARK_IO.File_Sys       from *,
      --#                                      CommandLineData.Content,
      --#                                      ConfigFileFound,
      --#                                      DictFound,
      --#                                      FDLignoreOptionFound,
      --#                                      FDLReserveOptionFound,
      --#                                      HTMLFound,
      --#                                      IndexFound,
      --#                                      NoDurationFound,
      --#                                      NoListingsFound,
      --#                                      NoWarningFound,
      --#                                      OptName,
      --#                                      OutputFound,
      --#                                      ReductionFound,
      --#                                      ReportFound,
      --#                                      StatisticsFound,
      --#                                      TargetDataFound,
      --#                                      XMLFound &
      --#         ConfigFileFound,
      --#         DictFound,
      --#         FDLignoreOptionFound,
      --#         IndexFound,
      --#         NoDurationFound,
      --#         NoListingsFound,
      --#         NoReportOptionFound,
      --#         NoWarningFound,
      --#         OutputFound,
      --#         ReportFound,
      --#         StatisticsFound,
      --#         TargetDataFound         from *,
      --#                                      OptName;
      is
         OK       : Boolean;
      begin
         --# accept W, 169, CommandLineData.Content.Valid, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Index, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Warning, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Echo, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Report, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.NoDuration, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.WriteDict, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.WriteStatistics, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.FDLreserved, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.TargetData , "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.TargetConfig, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.NoListings, "Direct updates OK here";
         case Ada.Characters.Handling.To_Lower (OptName.Content (3)) is
            when 'i' => -- 3rd letter
               CheckOptionName (OptName, "noindex_file", OK);
               if OK then
                  CommandLineData.Content.Valid := not IndexFound;
                  PossibleError (ESNoIndex);
                  IndexFound := True;
                  CommandLineData.Content.Index := False;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;

            when 'w' => -- 3rd letter
               CheckOptionName (OptName, "nowarning_file", OK);
               if OK then
                  CommandLineData.Content.Valid := not NoWarningFound;
                  PossibleError (ESNoWarning);
                  NoWarningFound := True;
                  CommandLineData.Content.Warning := False;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;

            when 'e' => -- 3rd letter
               CheckOptionName (OptName, "noecho", OK);
               if OK then
                  CommandLineData.Content.Valid := not OutputFound;
                  PossibleError (ESNoEcho);
                  OutputFound := True;
                  CommandLineData.Content.Echo := False;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;
            when 'r' => -- 3rd letter
               CheckOptionName (OptName, "noreport_file", OK);
               if OK then
                  CommandLineData.Content.Valid :=
                    not (ReportFound or HTMLFound or XMLFound);
                  PossibleError (ESNoReport);
                  ReportFound := True;
                  NoReportOptionFound := True;
                  CommandLineData.Content.Report := False;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;
            when 'd' => -- 3rd letter
               case Ada.Characters.Handling.To_Lower (OptName.Content (4)) is
                  when 'i' => -- 4th letter
                     CheckOptionName (OptName, "nodictionary", OK);
                     if OK then
                        CommandLineData.Content.Valid := not DictFound;
                        PossibleError (ESNoDict);
                        DictFound := True;
                        CommandLineData.Content.WriteDict := False;
                     else
                        CommandLineData.Content.Valid := False;
                        PossibleError2 (ESInvalidOption, OptName);
                     end if;
                  when 'u' => -- 4th letter
                     CheckOptionName (OptName, "noduration", OK);
                     if OK then
                        CommandLineData.Content.Valid := not NoDurationFound;
                        PossibleError (ESNoDuration);
                        NoDurationFound := True;
                        CommandLineData.Content.NoDuration := True;
                     else
                        CommandLineData.Content.Valid := False;
                        PossibleError2 (ESInvalidOption, OptName);
                     end if;
                  when others =>
                     CommandLineData.Content.Valid := False;
                     PossibleError2 (ESInvalidOption, OptName);
               end case;
            when 's' => -- 3rd letter
               case Ada.Characters.Handling.To_Lower (OptName.Content (4)) is
                  when 't' => -- 4th letter
                     CheckOptionName (OptName, "nostatistics", OK);
                     if OK then
                        CommandLineData.Content.Valid := not StatisticsFound;
                        PossibleError (ESNoStatistics);
                        StatisticsFound := True;
                        CommandLineData.Content.WriteStatistics := False;
                     else
                        CommandLineData.Content.Valid := False;
                        PossibleError2 (ESInvalidOption, OptName);
                     end if;
                  when 'w' => -- 4th Letter
                     -- -noswitch
                     --
                     -- Already processed by Ignore_Default_Switch_File below,
                     -- so allow but ignore here.
                     CheckOptionName (OptName, "noswitch", OK);
                     if OK then
                        null;
                     else
                        CommandLineData.Content.Valid := False;
                        PossibleError2 (ESInvalidOption, OptName);
                     end if;
                  when others =>
                     CommandLineData.Content.Valid := False;
                     PossibleError2 (ESInvalidOption, OptName);
               end case;
            when 'f' => -- 3rd letter
               CheckOptionName (OptName, "nofdl_identifiers", OK);
               if OK then
                  CommandLineData.Content.Valid := not (FDLreserveOptionFound or
                                                          FDLignoreOptionFound or
                                                          ReductionFound);
                  PossibleError (ESFDLoption);
                  FDLignoreOptionFound := True;
                  CommandLineData.Content.FDLreserved := False;
                  PutWarning ("Warning: the /nodfl option is now deprecated. Please use /fdl=accept");
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;

            when 't' => -- 3rd letter
               CheckOptionName (OptName, "notarget_compiler_data", OK);
               if OK then
                  CommandLineData.Content.Valid := not TargetDataFound;
                  PossibleError (ESNoTargetData);
                  TargetDataFound := True;
                  CommandLineData.Content.TargetData := False;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;

            when 'c' => -- 3rd letter
               CheckOptionName (OptName, "noconfig_file", OK);
               if OK then
                  CommandLineData.Content.Valid := not ConfigFileFound;
                  PossibleError (ESNoConfigFile);
                  ConfigFileFound := True;
                  CommandLineData.Content.TargetConfig := False;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;

            when 'l' => -- 3rd letter
               CheckOptionName (OptName, "nolistings", OK);
               if OK then
                  CommandLineData.Content.Valid := not NoListingsFound;
                  PossibleError (ESNoListings);
                  NoListingsFound := True;
                  CommandLineData.Content.NoListings := True;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;

            when others =>
               CommandLineData.Content.Valid := False;
               PossibleError2 (ESInvalidOption, OptName);
         end case;
         --# end accept;
      end Process_N;

      procedure Process_O
      --# global in     OptName;
      --#        in     OptVal;
      --#        in     OptValOK;
      --#        in out CommandLineData.Content;
      --#        in out OriginalFlowErrorsFound;
      --#        in out OutputDirectoryFound;
      --#        in out SPARK_IO.File_Sys;
      --# derives CommandLineData.Content from *,
      --#                                      OptName,
      --#                                      OptVal,
      --#                                      OptValOK,
      --#                                      OriginalFlowErrorsFound,
      --#                                      OutputDirectoryFound &
      --#         OriginalFlowErrorsFound from *,
      --#                                      OptName &
      --#         OutputDirectoryFound    from *,
      --#                                      OptName,
      --#                                      OptValOK &
      --#         SPARK_IO.File_Sys       from *,
      --#                                      CommandLineData.Content,
      --#                                      OptName,
      --#                                      OptValOK,
      --#                                      OriginalFlowErrorsFound,
      --#                                      OutputDirectoryFound;
      is
         OK       : Boolean;
      begin
         --# accept W, 169, CommandLineData.Content.Valid, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.OutputDirectory, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.OutputDirectoryName, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.LegacyErrors, "Direct updates OK here";
         case Ada.Characters.Handling.To_Lower (OptName.Content (2)) is
            when 'r' => -- 2nd letter
               CheckOptionName (OptName, "original_flow_errors", OK);
               if OK then
                  CommandLineData.Content.Valid := not OriginalFlowErrorsFound;
                  PossibleError (ESOriginalFlowErrors);
                  OriginalFlowErrorsFound := True;
                  CommandLineData.Content.LegacyErrors := True;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;
            when 'u' => -- 2nd letter
               CheckOptionName (OptName, "output_directory", OK);
               if OK then
                  CommandLineData.Content.Valid := OptValOK;
                  PossibleError (ESOutputDir);
                  if OptValOK then
                     CommandLineData.Content.Valid := not OutputDirectoryFound;
                     PossibleError (ESOutputDirRepeated);
                  end if;

                  if CommandLineData.Content.Valid then
                     OutputDirectoryFound := True;
                     CommandLineData.Content.OutputDirectory := True;
                     CommandLineData.Content.OutputDirectoryName := OptVal;
                  end if;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;
            when others =>
               CommandLineData.Content.Valid := False;
               PossibleError2 (ESInvalidOption, OptName);
         end case;
         --# end accept;
      end Process_O;


      procedure Process_P
      --# global in     FDLignoreOptionFound;
      --#        in     OptName;
      --#        in     OptVal;
      --#        in out AnalysisProfileFound;
      --#        in out CommandLineData.Content;
      --#        in out PlainOutputFound;
      --#        in out ReductionFound;
      --#        in out SPARK_IO.File_Sys;
      --# derives AnalysisProfileFound,
      --#         PlainOutputFound,
      --#         ReductionFound          from *,
      --#                                      OptName &
      --#         CommandLineData.Content,
      --#         SPARK_IO.File_Sys       from *,
      --#                                      AnalysisProfileFound,
      --#                                      CommandLineData.Content,
      --#                                      FDLignoreOptionFound,
      --#                                      OptName,
      --#                                      OptVal,
      --#                                      PlainOutputFound,
      --#                                      ReductionFound;
      is
         OK       : Boolean;
      begin
         --# accept W, 169, CommandLineData.Content.Valid, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.RedType, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.PlainOutput, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.AnalysisProfile, "Direct updates OK here";
         case Ada.Characters.Handling.To_Lower (OptName.Content (2)) is
            when 'f' => -- 2nd letter
               CheckOptionName (OptName, "pfs", OK);
               if OK then
                  CommandLineData.Content.Valid :=
                    not (ReductionFound or (FDLignoreOptionFound and
                                              CommandLineData.Content.FDLmangle = EStrings.EmptyString)) and
                    CommandLineData.Content.DoInformationFlow;
                  PossibleError (ESReduction);
                  ReductionFound := True;
                  CommandLineData.Content.RedType := CommandLineData.JustPFs;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;

            when 'l' => -- 2nd letter
               CheckOptionName (OptName, "plain_output", OK);
               if OK then
                  CommandLineData.Content.Valid := not PlainOutputFound;
                  PossibleError (ESPlainOutput);
                  PlainOutputFound := True;
                  CommandLineData.Content.PlainOutput := True;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;

            when 'r' => -- 2nd letter
               CheckOptionName (OptName, "profile", OK);
               if OK then
                  CommandLineData.Content.Valid := not AnalysisProfileFound;
                  PossibleError (ESProfile);
                  if CommandLineData.Content.Valid then
                     AnalysisProfileFound := True;
                     case Ada.Characters.Handling.To_Lower (OptVal.Content (1)) is
                        when 'r' => -- 1st letter of option
                           CheckOptionName (OptVal, "ravenscar", OK);
                           if OK and then CommandLineData.Content.Spark95 then
                              CommandLineData.Content.AnalysisProfile :=
                                CommandLineData.Ravenscar;
                           else -- other error involving ravenscar
                              CommandLineData.Content.Valid := False;
                              PossibleError2 (ESProfileOption, OptVal);
                           end if;

                        when 's' => -- 1st letter of option
                           CheckOptionName (OptVal, "sequential", OK);
                           if OK then
                              CommandLineData.Content.AnalysisProfile :=
                                CommandLineData.Sequential;
                           else
                              CommandLineData.Content.Valid := False;
                              PossibleError2 (ESProfileOption, OptVal);
                           end if;

                        when others =>
                           CommandLineData.Content.Valid := False;
                           PossibleError2 (ESProfileOption, OptVal);
                     end case;
                  end if;
               else -- begins with "pr" but not "profile"
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;

               ------------------------------------------------
               -- -policy=safety|security
               --
               -- Currently unimplemented pending resolution of
               -- how integrity level of constants should be
               -- handled in the flow analyser
               ------------------------------------------------

               --  when 'o' => -- 2nd letter
               --     CheckOptionName (OptName, "policy", OK);
               --     if OK then
               --        CommandLineData.Content.Valid := CommandLineData.Content.DoInformationFlow;
               --        PossibleError (ESDataFlowAndInfoFlowPolicy);
               --        if CommandLineData.Content.Valid then
               --           case Ada.Characters.Handling.To_Lower (OptVal.Content (1)) is
               --              when 's' => -- 1st letter of option
               --                 case Ada.Characters.Handling.To_Lower (OptVal.Content (2)) is
               --                    when 'e' =>
               --                       CheckOptionName (OptVal, "security", OK);
               --                       if OK then
               --                          CommandLineData.Content.InfoFlowPolicy := CommandLineData.Security;
               --                       else
               --                          CommandLineData.Content.Valid := False;
               --                          PossibleError2 (ESProfileOption, OptVal);
               --                       end if;
               --                    when 'a' =>
               --                       CheckOptionName (OptVal, "safety", OK);
               --                       if OK then
               --                          CommandLineData.Content.InfoFlowPolicy := CommandLineData.Safety;
               --                       else
               --                          CommandLineData.Content.Valid := False;
               --                          PossibleError2 (ESProfileOption, OptVal);
               --                       end if;
               --                    when others =>
               --                       CommandLineData.Content.Valid := False;
               --                       PossibleError2 (ESProfileOption, OptVal);
               --                 end case;
               --              when others =>
               --                 CommandLineData.Content.Valid := False;
               --                 PossibleError2 (ESProfileOption, OptVal);
               --           end case;
               --        end if;
               --   end if;

               -- begin with "p" but not a valid option
            when others =>
               CommandLineData.Content.Valid := False;
               PossibleError2 (ESInvalidOption, OptName);
         end case;
         --# end accept;
      end Process_P;


      procedure Process_R
      --# global in     FDLignoreOptionFound;
      --#        in     OptName;
      --#        in     OptVal;
      --#        in     OptValOK;
      --#        in out CommandLineData.Content;
      --#        in out DoRealRTCs;
      --#        in out RealRTCsFound;
      --#        in out ReductionFound;
      --#        in out ReportFound;
      --#        in out RulesFound;
      --#        in out SPARK_IO.File_Sys;
      --# derives CommandLineData.Content,
      --#         SPARK_IO.File_Sys       from *,
      --#                                      CommandLineData.Content,
      --#                                      FDLignoreOptionFound,
      --#                                      OptName,
      --#                                      OptVal,
      --#                                      OptValOK,
      --#                                      RealRTCsFound,
      --#                                      ReductionFound,
      --#                                      ReportFound,
      --#                                      RulesFound &
      --#         DoRealRTCs,
      --#         RealRTCsFound           from *,
      --#                                      OptName,
      --#                                      RealRTCsFound &
      --#         ReductionFound,
      --#         RulesFound              from *,
      --#                                      OptName &
      --#         ReportFound             from *,
      --#                                      OptName,
      --#                                      OptValOK;
      is
         Default_Report_Extension : constant EStrings.T :=
           EStrings.T'(3, EStrings.Contents'('r', 'e', 'p', others => ' '));

         OK       : Boolean;
         FileName : EStrings.T;
      begin
         --# accept W, 169, CommandLineData.Content.Valid, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.RedType, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Report, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.ReportFileName, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.ConstantRules, "Direct updates OK here";
         case Ada.Characters.Handling.To_Lower (OptName.Content (2)) is
            when 'e' => -- 2nd letter
               case Ada.Characters.Handling.To_Lower (OptName.Content (3)) is
                  when 'p' => -- 3rd letter
                     CheckOptionName (OptName, "report_file", OK);
                     if OK then
                        CommandLineData.Content.Valid := OptValOK;
                        PossibleError (ESReportName);
                        if OptValOK then
                           CommandLineData.Content.Valid := not ReportFound;
                           PossibleError (ESReport);
                        end if;
                        if CommandLineData.Content.Valid then
                           ReportFound := True;
                           CommandLineData.Content.Report := True;
                           FileName := OptVal;
                           FileSystem.CheckExtension
                             (FileName,
                              FileSystem.CaseOfFileExtensions (Default_Report_Extension));
                           CommandLineData.Content.ReportFileName := FileName;
                        end if;
                     else -- starts "rep" but is not valid
                        CommandLineData.Content.Valid := False;
                        PossibleError2 (ESInvalidOption, OptName);
                     end if;

                  when 'a' => -- 3rd letter
                     CheckOptionName (OptName, "realrtcs", OK);
                     if OK then
                        if RealRTCsFound then -- duplicated
                           CommandLineData.Content.Valid := False;
                           PossibleError (ESRealRTCs);
                        else -- not duplicated, so valid
                           CommandLineData.Content.Valid := True;
                           RealRTCsFound := True;
                           DoRealRTCs := True;
                        end if;
                     else -- starts "rea" but not a valid option
                        CommandLineData.Content.Valid := False;
                        PossibleError2 (ESInvalidOption, OptName);
                     end if;

                  when others =>
                     -- starts "re" but isn't valid
                     CommandLineData.Content.Valid := False;
                     PossibleError2 (ESInvalidOption, OptName);
               end case; -- 3rd letter

            when 't' => -- 2nd letter
               CheckOptionName (OptName, "rtc", OK);
               if OK then
                  CommandLineData.Content.Valid :=
                    not (ReductionFound or (FDLignoreOptionFound and
                                              CommandLineData.Content.FDLmangle = EStrings.EmptyString));
                  PossibleError (ESReduction);
                  ReductionFound := True;
                  CommandLineData.Content.RedType := CommandLineData.DoRtc;
               else -- starts "rt" but isn't valid
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;

            when 'u' => -- 2nd letter u - should be "rules" switch
               CheckOptionName (OptName, "rules", OK);
               if OK then
                  if RulesFound then
                     CommandLineData.Content.Valid := False;
                     PossibleError (ESRules);
                  else
                     RulesFound := True;
                     CommandLineData.Content.Valid := OptValOK;
                     if CommandLineData.Content.Valid then
                        case Ada.Characters.Handling.To_Lower (OptVal.Content (1)) is
                           when 'n' => -- 1st letter of option
                              CheckOptionName (OptVal, "none", OK);
                              if OK then
                                 CommandLineData.Content.ConstantRules := CommandLineData.NoRules;
                              else
                                 CommandLineData.Content.Valid := False;
                                 PossibleError (ESRuleUnknown);
                              end if;

                           when 'l' => -- 1st letter of option
                              CheckOptionName (OptVal, "lazy", OK);
                              if OK then
                                 CommandLineData.Content.ConstantRules := CommandLineData.Lazy;
                              else
                                 CommandLineData.Content.Valid := False;
                                 PossibleError (ESRuleUnknown);
                              end if;

                           when 'k' => -- 1st letter of option
                              CheckOptionName (OptVal, "keen", OK);
                              if OK then
                                 CommandLineData.Content.ConstantRules := CommandLineData.Keen;
                              else
                                 CommandLineData.Content.Valid := False;
                                 PossibleError (ESRuleUnknown);
                              end if;

                           when 'a' => -- 1st letter of option
                              CheckOptionName (OptVal, "all", OK);
                              if OK then
                                 CommandLineData.Content.ConstantRules := CommandLineData.AllRules;
                              else
                                 CommandLineData.Content.Valid := False;
                                 PossibleError (ESRuleUnknown);
                              end if;
                           when others =>
                              CommandLineData.Content.Valid := False;
                              PossibleError (ESRuleUnknown);
                        end case;
                     else
                        CommandLineData.Content.Valid := False;
                        PossibleError (ESRuleUnknown);
                     end if;
                  end if;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;

            when others =>
               -- starts "r" but not valid
               CommandLineData.Content.Valid := False;
               PossibleError2 (ESInvalidOption, OptName);
         end case;
         --# end accept;
      end Process_R;


      procedure Process_S
      --# global in     OptName;
      --#        in     OptVal;
      --#        in     OptValOK;
      --#        in out CommandLineData.Content;
      --#        in out SourceFound;
      --#        in out SPARK_IO.File_Sys;
      --#        in out StatisticsFound;
      --#        in out SyntaxCheckFound;
      --# derives CommandLineData.Content from *,
      --#                                      OptName,
      --#                                      OptVal,
      --#                                      OptValOK,
      --#                                      SourceFound,
      --#                                      StatisticsFound,
      --#                                      SyntaxCheckFound &
      --#         SourceFound,
      --#         StatisticsFound,
      --#         SyntaxCheckFound        from *,
      --#                                      OptName &
      --#         SPARK_IO.File_Sys       from *,
      --#                                      CommandLineData.Content,
      --#                                      OptName,
      --#                                      OptValOK,
      --#                                      SourceFound,
      --#                                      StatisticsFound,
      --#                                      SyntaxCheckFound;
      is
         OK : Boolean;
      begin
         --# accept W, 169, CommandLineData.Content.WriteStatistics,   "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.SourceExtension,   "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.SyntaxOnly,        "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Valid,             "Direct updates OK here";
         case Ada.Characters.Handling.To_Lower (OptName.Content (2)) is
            when 'o' =>
               CheckOptionName (OptName, "source_extension", OK);
               if OK then
                  CommandLineData.Content.Valid := OptValOK and not SourceFound;
                  PossibleError (ESSourceExt);
                  SourceFound := True;
                  CommandLineData.Content.SourceExtension := OptVal;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;

            when 'y' =>
               CheckOptionName (OptName, "syntax_check", OK);
               if OK then
                  CommandLineData.Content.Valid := not SyntaxCheckFound;
                  PossibleError (ESsyntax);
                  SyntaxCheckFound := True;
                  CommandLineData.Content.SyntaxOnly := True;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;

            when 't' =>
               CheckOptionName (OptName, "statistics", OK);
               if OK then
                  CommandLineData.Content.Valid := not StatisticsFound;
                  PossibleError (ESStatistics);
                  StatisticsFound := True;
                  CommandLineData.Content.WriteStatistics := True;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;

            when others =>
               CommandLineData.Content.Valid := False;
               PossibleError2 (ESInvalidOption, OptName);
         end case;
         --# end accept;
      end Process_S;

      procedure Process_T
      --# global in     ConfigFileFound;
      --#        in     OptName;
      --#        in     OptVal;
      --#        in     OptValOK;
      --#        in out CommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --#        in out TargetDataFound;
      --# derives CommandLineData.Content from *,
      --#                                      ConfigFileFound,
      --#                                      OptName,
      --#                                      OptVal,
      --#                                      OptValOK,
      --#                                      TargetDataFound &
      --#         SPARK_IO.File_Sys       from *,
      --#                                      CommandLineData.Content,
      --#                                      ConfigFileFound,
      --#                                      OptName,
      --#                                      OptValOK,
      --#                                      TargetDataFound &
      --#         TargetDataFound         from *,
      --#                                      ConfigFileFound,
      --#                                      OptName,
      --#                                      OptValOK;
      is
         OK       : Boolean;
         FileName : EStrings.T;
      begin
         --# accept W, 169, CommandLineData.Content.Valid, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.TargetData, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.TargetDataFile, "Direct updates OK here";
         case Ada.Characters.Handling.To_Lower (OptName.Content (2)) is
            -- -tree is removed in release 8.1.1 and above, but it remains
            -- the default, so at least recognize it here and take no action
            -- for backward compatiblility with users' existing scripts.
            when 'r' => -- 2nd letter
               CheckOptionName (OptName, "tree", OK);
               if not OK then
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;

            when 'a' => -- 2nd letter
               CheckOptionName (OptName, "target_compiler_data", OK);
               if OK then
                  if ConfigFileFound then
                     CommandLineData.Content.Valid := False;
                     PossibleError (ESDataAndConfig);
                  else
                     CommandLineData.Content.Valid := OptValOK;
                     PossibleError (ESTargetDataName);
                     if OptValOK then
                        CommandLineData.Content.Valid := not TargetDataFound;
                        PossibleError (ESTargetData);
                     end if;

                     if CommandLineData.Content.Valid then
                        TargetDataFound := True;
                        CommandLineData.Content.TargetData := True;
                        FileName := OptVal;
                        FileSystem.CheckExtension
                          (FileName,
                           FileSystem.CaseOfFileExtensions
                             (CommandLineData.DEFAULT_DATA_EXTENSION));
                        CommandLineData.Content.TargetDataFile := FileName;
                     end if;
                  end if;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;

            when others =>
               CommandLineData.Content.Valid := False;
               PossibleError2 (ESInvalidOption, OptName);
         end case;
         --# end accept;
      end Process_T;

      procedure Process_V
      --# global in     FDLignoreOptionFound;
      --#        in     OptName;
      --#        in out CommandLineData.Content;
      --#        in out DoRealRTCs;
      --#        in out ReductionFound;
      --#        in out SPARK_IO.File_Sys;
      --# derives CommandLineData.Content,
      --#         SPARK_IO.File_Sys       from *,
      --#                                      CommandLineData.Content,
      --#                                      FDLignoreOptionFound,
      --#                                      OptName,
      --#                                      ReductionFound &
      --#         DoRealRTCs,
      --#         ReductionFound          from *,
      --#                                      OptName;
      is
         OK       : Boolean;
      begin
         --# accept W, 169, CommandLineData.Content.Valid, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.RedType, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.PlainOutput, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.VersionRequested, "Direct updates OK here";
         case Ada.Characters.Handling.To_Lower (OptName.Content (2)) is
            when 'c' => -- 2nd letter
               CheckOptionName (OptName, "vcg", OK);
               if OK then
                  CommandLineData.Content.Valid :=
                    not (ReductionFound or (FDLignoreOptionFound and
                                              CommandLineData.Content.FDLmangle = EStrings.EmptyString));
                  PossibleError (ESReduction);
                  ReductionFound := True;
                  -- "-vcg" is equivalent to "-exp -real", so...
                  CommandLineData.Content.RedType := CommandLineData.DoExp;
                  DoRealRTCs := True;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;
            when 'e' => -- 2nd letter
               CheckOptionName (OptName, "version", OK);
               if OK then
                  -- Examiner version requested, so abandon all further
                  -- option processing by setting Valid = False.
                  CommandLineData.Content.Valid := False;
                  CommandLineData.Content.VersionRequested := True;
                  CommandLineData.Content.PlainOutput := False;
               else
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;
            when others =>
               CommandLineData.Content.Valid := False;
               PossibleError2 (ESInvalidOption, OptName);
         end case;
         --# end accept;
      end Process_V;

      procedure Process_W
      --# global in     OptName;
      --#        in     OptVal;
      --#        in     OptValOK;
      --#        in out CommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --#        in out WarningFound;
      --# derives CommandLineData.Content from *,
      --#                                      OptName,
      --#                                      OptVal,
      --#                                      OptValOK,
      --#                                      WarningFound &
      --#         SPARK_IO.File_Sys       from *,
      --#                                      CommandLineData.Content,
      --#                                      OptName,
      --#                                      OptValOK,
      --#                                      WarningFound &
      --#         WarningFound            from *,
      --#                                      OptName,
      --#                                      OptValOK;
      is
         OK       : Boolean;
         FileName : EStrings.T;
      begin
         --# accept W, 169, CommandLineData.Content.Valid, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.Warning, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.WarningFileName, "Direct updates OK here";
         CheckOptionName (OptName, "warning_file", OK);
         if OK then
            CommandLineData.Content.Valid := OptValOK;
            PossibleError (ESWarningName);
            if OptValOK then
               CommandLineData.Content.Valid := not WarningFound;
               PossibleError (ESWarning);
            end if;
            if CommandLineData.Content.Valid then
               WarningFound := True;
               CommandLineData.Content.Warning := True;
               FileName := OptVal;
               FileSystem.CheckExtension
                 (FileName,
                  FileSystem.CaseOfFileExtensions
                    (CommandLineData.DEFAULT_WARNING_EXTENSION));
               CommandLineData.Content.WarningFileName := FileName;
            end if;
         else
            CommandLineData.Content.Valid := False;
            PossibleError2 (ESInvalidOption, OptName);
         end if;
         --# end accept;
      end Process_W;

      procedure Process_X
      --# global in     HTMLFound;
      --#        in     NoReportOptionFound;
      --#        in     OptName;
      --#        in out CommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLFound;
      --# derives CommandLineData.Content,
      --#         SPARK_IO.File_Sys       from *,
      --#                                      CommandLineData.Content,
      --#                                      HTMLFound,
      --#                                      NoReportOptionFound,
      --#                                      OptName,
      --#                                      XMLFound &
      --#         XMLFound                from *,
      --#                                      OptName;
      is
         OK       : Boolean;
      begin
         --# accept W, 169, CommandLineData.Content.Valid, "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.XML, "Direct updates OK here";
         CheckOptionName (OptName, "xml", OK);
         if OK then
            CommandLineData.Content.Valid := not (XMLFound or NoReportOptionFound or HTMLFound);
            if HTMLFound then
               PossibleError (ESXMLandHTML);
            elsif XMLFound then
               PossibleError (ESXML);
            else
               PossibleError (ESXMLnorep);
            end if;

            XMLFound := True;
            if CommandLineData.Content.Valid then
               CommandLineData.Content.XML := True;
            end if;
         else
            CommandLineData.Content.Valid := False;
            PossibleError2 (ESInvalidOption, OptName);
         end if;
         --# end accept;
      end Process_X;


   begin
      --# accept W, 169, CommandLineData.Content.Valid, "Direct updates OK here";
      loop
         exit when NextSymbol.Typ /= SSlash;
         ReadOption (OptName,
                     OptNameOK,
                     OptVal,
                     OptValOK,
                     CommandString,
                     NextSymbol);
         if OptNameOK then
            case Ada.Characters.Handling.To_Lower (OptName.Content (1)) is
               when 'a' =>
                  Process_A;
               when 'b' =>
                  Process_B;
               when 'c' =>
                  Process_C;
               when 'd' =>
                  Process_D;
               when 'e' =>
                  Process_E;
               when 'f' =>
                  Process_F;
               when 'h' =>
                  Process_H;
               when 'i' =>
                  Process_I;
               when 'j' =>
                  Process_J;
               when 'l' =>
                  Process_L;
               when 'n' =>
                  Process_N;
               when 'o' =>
                  Process_O;
               when 'p' =>
                  Process_P;
               when 'r' =>
                  Process_R;
               when 's' =>
                  Process_S;
               when 't' =>
                  Process_T;
               when 'v' =>
                  Process_V;
               when 'w' =>
                  Process_W;
               when 'x' =>
                  Process_X;
               when others =>
                  CommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
            end case;
         else
            CommandLineData.Content.Valid := False;
            PossibleError2 (ESInvalidOption, OptName);
         end if;
         exit when not CommandLineData.Content.Valid;
      end loop;
      --# end accept;
   end ParseCommandOptions;

   procedure ParseCommandLine (CommandString : in     CommandStrings;
                               DoRealRTCs    : in out Boolean)
   --# global in out CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives CommandLineData.Content,
   --#         DoRealRTCs,
   --#         SPARK_IO.File_Sys       from *,
   --#                                      CommandLineData.Content,
   --#                                      CommandString;
   is

      NextSymbol         : Symbols;
      LocalCommandString : CommandStrings;

      procedure ParseArguments (CommandString : in CommandStrings;
                                NextSymbol    : in Symbols)
      --# global in out CommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --# derives CommandLineData.Content,
      --#         SPARK_IO.File_Sys       from *,
      --#                                      CommandLineData.Content,
      --#                                      CommandString,
      --#                                      NextSymbol;
      --  pre NextSymbol.Typ in {SString, SEmpty};
      is

         LocalNextSymbol    : Symbols;
         LocalCommandString : CommandStrings;

         procedure ParseFileEntry (CommandString : in out CommandStrings;
                                   NextSymbol    : in out Symbols)
         --# global in out CommandLineData.Content;
         --#        in out SPARK_IO.File_Sys;
         --# derives CommandLineData.Content,
         --#         SPARK_IO.File_Sys       from *,
         --#                                      CommandLineData.Content,
         --#                                      CommandString,
         --#                                      NextSymbol &
         --#         CommandString,
         --#         NextSymbol              from CommandString,
         --#                                      NextSymbol;
         --  pre  NextSymbol.Typ = SString;
         --  post NextSymbol.Typ in {SComma, SEmpty};
         is

            FileName : EStrings.T;

            procedure ParseArgumentOption (CommandString : in out CommandStrings;
                                           NextSymbol    : in out Symbols)
            --# global in out CommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --# derives CommandLineData.Content,
            --#         SPARK_IO.File_Sys       from *,
            --#                                      CommandLineData.Content,
            --#                                      CommandString,
            --#                                      NextSymbol &
            --#         CommandString,
            --#         NextSymbol              from CommandString,
            --#                                      NextSymbol;
            is

               FileName : EStrings.T;
               DoList   : Boolean;
               Ok       : Boolean;

               procedure ReadArgumentOption (ListingFileName :    out EStrings.T;
                                             Listing         :    out Boolean;
                                             OK              :    out Boolean;
                                             CommandString   : in out CommandStrings;
                                             NextSymbol      :    out Symbols)
               --# global in out SPARK_IO.File_Sys;
               --# derives CommandString,
               --#         SPARK_IO.File_Sys from *,
               --#                                CommandString &
               --#         Listing,
               --#         ListingFileName,
               --#         NextSymbol,
               --#         OK                from CommandString;
               --  pre  NextSymbol.Typ = SSlash;
               --  post NextSymbol.Typ in {SComma, SEmpty};
               is
                  OptName   : EStrings.T;
                  OptVal    : EStrings.T;
                  OptNameOK : Boolean;
                  OptValOK  : Boolean;
                  LOK       : Boolean;
               begin
                  ListingFileName := EStrings.EmptyString;
                  Listing := False;

                  ReadOption (OptName, OptNameOK, OptVal, OptValOK,
                              CommandString, NextSymbol);
                  case Ada.Characters.Handling.To_Lower (OptName.Content (1)) is
                     when 'l' =>
                        CheckOptionName (OptName, "listing_file", LOK);
                        if LOK then
                           OK := OptNameOK and OptValOK;
                           Listing := True;
                           ListingFileName := OptVal;
                        else
                           OK := False;
                           Listing := False;
                           OutputError (ESInvalidOption);
                           ScreenEcho.New_Line (1);
                        end if;
                     when 'n' =>
                        CheckOptionName (OptName, "nolisting_file", LOK);
                        if LOK then
                           OK := OptNameOK and not OptValOK;
                        else
                           OK := False;
                           OutputError (ESInvalidOption);
                           ScreenEcho.New_Line (1);
                        end if;
                        Listing := False;
                     when others =>
                        OK := False;
                  end case;
               end ReadArgumentOption;

            begin
               --# accept W, 169, CommandLineData.Content.SourceFileList,   "Direct updates OK here" &
               --#        W, 169, CommandLineData.Content.Valid,   "Direct updates OK here";
               if NextSymbol.Typ = SSlash then
                  ReadArgumentOption (FileName, DoList, Ok,
                                      CommandString, NextSymbol);
                  CommandLineData.Content.SourceFileList
                    (CommandLineData.Content.NumberSource).ListingFileName := FileName;
                  CommandLineData.Content.SourceFileList
                    (CommandLineData.Content.NumberSource).Listing := DoList;
                  CommandLineData.Content.Valid := Ok;
                  PossibleError (ESListingFile);
               else
                  CommandLineData.Content.SourceFileList
                    (CommandLineData.Content.NumberSource).Listing := True;

                  CommandLineData.Content.SourceFileList
                    (CommandLineData.Content.NumberSource).ListingFileName :=
                    FileSystem.JustFile
                    (CommandLineData.Content.SourceFileList
                       (CommandLineData.Content.NumberSource).SourceFileName,
                     False);
               end if;

               if CommandLineData.Content.Valid and
                 CommandLineData.Content.SourceFileList
                 (CommandLineData.Content.NumberSource).Listing then

                  FileName := CommandLineData.Content.SourceFileList
                    (CommandLineData.Content.NumberSource).ListingFileName;

                  FileSystem.CheckListingExtension
                    (CommandLineData.Content.SourceFileList
                       (CommandLineData.Content.NumberSource).SourceFileName,
                     FileName,
                     CommandLineData.Content.ListingExtension);

                  CommandLineData.Content.SourceFileList
                    (CommandLineData.Content.NumberSource).ListingFileName := FileName;
               end if;
               --# end accept;
            end ParseArgumentOption;

         begin
            --# accept W, 169, CommandLineData.Content.SourceFileList,   "Direct updates OK here" &
            --#        W, 169, CommandLineData.Content.Valid,   "Direct updates OK here" &
            --#        W, 169, CommandLineData.Content.NumberSource,   "Direct updates OK here";
            if NextSymbol.Typ = SString then
               CommandLineData.Content.NumberSource :=
                 CommandLineData.Content.NumberSource + 1;

               FileName := NextSymbol.TheString;

               if FileName.Content (1) = '@' then
                  FileSystem.CheckExtension
                    (FileName,
                     FileSystem.CaseOfFileExtensions (CommandLineData.MetaFileExtension));
                  CommandLineData.Content.SourceFileList
                    (CommandLineData.Content.NumberSource).SourceFileName := FileName;
                  CommandLineData.Content.SourceFileList
                    (CommandLineData.Content.NumberSource).Listing := False;
                  GetNextSymbol (CommandString, NextSymbol);
               else
                  FileSystem.CheckExtension
                    (FileName,
                     CommandLineData.Content.SourceExtension);
                  CommandLineData.Content.SourceFileList
                    (CommandLineData.Content.NumberSource).SourceFileName := FileName;
                  GetNextSymbol (CommandString, NextSymbol);
                  ParseArgumentOption (CommandString, NextSymbol);
               end if;
            else
               CommandLineData.Content.Valid := False;
               PossibleError (ESSource);
            end if;
            --# end accept;
         end ParseFileEntry;

      begin
         --# accept W, 169, CommandLineData.Content.Valid,   "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.NumberSource,   "Direct updates OK here";
         LocalNextSymbol    := NextSymbol;
         LocalCommandString := CommandString;
         CommandLineData.Content.NumberSource := 0;
         loop
            ParseFileEntry (LocalCommandString, LocalNextSymbol);

            exit when LocalNextSymbol.Typ = SEmpty or
              not CommandLineData.Content.Valid;
            if CommandLineData.Content.NumberSource =
              ExaminerConstants.MaxFilesOnCommandLine
            then
               CommandLineData.Content.Valid := False;
               OutputError (EWtooMany);
               exit;
            end if;

            if FileSystem.UseWindowsCommandLine and
              LocalNextSymbol.Typ = SComma then
               GetNextSymbol (LocalCommandString, LocalNextSymbol);
            end if;

         end loop;
         --# end accept;
      end ParseArguments;

   begin
      LocalCommandString := CommandString;
      GetNextSymbol (LocalCommandString, NextSymbol);
      ParseCommandOptions (LocalCommandString, NextSymbol, DoRealRTCs);
      if CommandLineData.Content.Valid then
         ParseArguments (LocalCommandString, NextSymbol);
      end if;
   end ParseCommandLine;

   procedure WffCommandLine (DoRealRTCs : in Boolean)
   --# global in out CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives CommandLineData.Content,
   --#         SPARK_IO.File_Sys       from CommandLineData.Content,
   --#                                      DoRealRTCs,
   --#                                      SPARK_IO.File_Sys;
   is

      procedure WffNonOverlapping
      --# global in out CommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --# derives CommandLineData.Content,
      --#         SPARK_IO.File_Sys       from *,
      --#                                      CommandLineData.Content;
      is
         function ExistsIn (F         : EStrings.T;
                            L         : CommandLineData.SourceFileLists;
                            StartPos  : CommandLineData.SourceFilePositions;
                            FinishPos : CommandLineData.SourceFilePositions)
                           return Boolean
         is
            Result : Boolean;
         begin
            Result := False;
            for Ix in CommandLineData.SourceFilePositions range StartPos .. FinishPos loop
               Result := FileSystem.SameFile (F, L (Ix).SourceFileName);
               exit when Result;
               if L (Ix).Listing then
                  Result := FileSystem.SameFile (F, L (Ix).ListingFileName);
               end if;
               exit when Result;
            end loop;
            return Result;
         end ExistsIn;

         procedure IndexNonOverlapping
         --# global in out CommandLineData.Content;
         --#        in out SPARK_IO.File_Sys;
         --# derives CommandLineData.Content,
         --#         SPARK_IO.File_Sys       from *,
         --#                                      CommandLineData.Content;
         is
         begin
            --# accept W, 169, CommandLineData.Content.Valid,   "Direct updates OK here";
            if CommandLineData.Content.Index then
               CommandLineData.Content.Valid :=
                  not FileSystem.SameFile (CommandLineData.Content.IndexFileName,
                                           CommandLineData.Content.ReportFileName) and
                  not (CommandLineData.Content.Warning and then
                       FileSystem.SameFile (CommandLineData.Content.IndexFileName,
                                            CommandLineData.Content.WarningFileName)) and
                  not ExistsIn (CommandLineData.Content.IndexFileName,
                                CommandLineData.Content.SourceFileList,
                                1,
                                CommandLineData.Content.NumberSource);
               PossibleError2 (EWOverlap, CommandLineData.Content.IndexFileName);
            end if;
            --# end accept;
         end IndexNonOverlapping;

         procedure WarningNonOverlapping
         --# global in out CommandLineData.Content;
         --#        in out SPARK_IO.File_Sys;
         --# derives CommandLineData.Content,
         --#         SPARK_IO.File_Sys       from *,
         --#                                      CommandLineData.Content;
         is
         begin
            --# accept W, 169, CommandLineData.Content.Valid,   "Direct updates OK here";
            if CommandLineData.Content.Warning then
               CommandLineData.Content.Valid :=
                  not FileSystem.SameFile (CommandLineData.Content.WarningFileName,
                                           CommandLineData.Content.ReportFileName) and
                  not ExistsIn (CommandLineData.Content.WarningFileName,
                                CommandLineData.Content.SourceFileList,
                                1,
                                CommandLineData.Content.NumberSource);
               PossibleError2 (EWOverlap, CommandLineData.Content.WarningFileName);
            end if;
            --# end accept;
         end WarningNonOverlapping;

         procedure ReportNonOverlapping
         --# global in out CommandLineData.Content;
         --#        in out SPARK_IO.File_Sys;
         --# derives CommandLineData.Content,
         --#         SPARK_IO.File_Sys       from *,
         --#                                      CommandLineData.Content;
         is
         begin
            --# accept W, 169, CommandLineData.Content.Valid,   "Direct updates OK here";
            if CommandLineData.Content.Report then
               CommandLineData.Content.Valid :=
                  not ExistsIn (CommandLineData.Content.ReportFileName,
                                CommandLineData.Content.SourceFileList,
                                1,
                                CommandLineData.Content.NumberSource);
               PossibleError2 (EWOverlap, CommandLineData.Content.ReportFileName);
            end if;
            --# end accept;
         end ReportNonOverlapping;

         procedure SourceFileListNonOverlapping
         --# global in out CommandLineData.Content;
         --#        in out SPARK_IO.File_Sys;
         --# derives CommandLineData.Content,
         --#         SPARK_IO.File_Sys       from *,
         --#                                      CommandLineData.Content;
         is
         begin
            --# accept W, 169, CommandLineData.Content.Valid,   "Direct updates OK here";
            for Ix in CommandLineData.SourceFilePositions
               range 1 .. (CommandLineData.Content.NumberSource - 1) loop
               CommandLineData.Content.Valid :=
                  not ExistsIn (CommandLineData.Content.SourceFileList (Ix).SourceFileName,
                                CommandLineData.Content.SourceFileList,
                                Ix + 1,
                                CommandLineData.Content.NumberSource);
               PossibleError2 (EWOverlap,
                               CommandLineData.Content.SourceFileList (Ix).SourceFileName);
               exit when not CommandLineData.Content.Valid;
               if CommandLineData.Content.SourceFileList (Ix).Listing then
                  CommandLineData.Content.Valid := not FileSystem.SameFile
                     (CommandLineData.Content.SourceFileList (Ix).SourceFileName,
                      CommandLineData.Content.SourceFileList (Ix).ListingFileName) and
                     not ExistsIn (CommandLineData.Content.SourceFileList (Ix).SourceFileName,
                                   CommandLineData.Content.SourceFileList,
                                   Ix + 1,
                                   CommandLineData.Content.NumberSource);
                  PossibleError2 (EWOverlap,
                                  CommandLineData.Content.SourceFileList (Ix).SourceFileName);
               end if;
               exit when not CommandLineData.Content.Valid;
            end loop;
            if CommandLineData.Content.Valid and then
               CommandLineData.Content.NumberSource /= 0 then
               CommandLineData.Content.Valid := not FileSystem.SameFile
                  (CommandLineData.Content.SourceFileList
                   (CommandLineData.Content.NumberSource).SourceFileName,
                   CommandLineData.Content.SourceFileList
                   (CommandLineData.Content.NumberSource).ListingFileName);
               PossibleError2
                  (EWOverlap,
                   CommandLineData.Content.SourceFileList
                   (CommandLineData.Content.NumberSource).SourceFileName);
            end if;
            --# end accept;
         end SourceFileListNonOverlapping;

      begin
         IndexNonOverlapping;

         if CommandLineData.Content.Valid then
            WarningNonOverlapping;
         end if;

         if CommandLineData.Content.Valid then
            ReportNonOverlapping;
         end if;

         if CommandLineData.Content.Valid then
            SourceFileListNonOverlapping;
         end if;
      end WffNonOverlapping;

      function StripAnyAt (S : EStrings.T)
                          return EStrings.T
      is
         Result : EStrings.T;
      begin
         if S.Content (1) = '@' then
            Result :=  StringUtilities.Section (S, 2, S.Length - 1);
         else
            Result := S;
         end if;
         return Result;
      end StripAnyAt;

      function AdjustFilename (S : in EStrings.T) return EStrings.T
      --# global in CommandLineData.Content;
      is
         R : EStrings.T;
      begin
         if CommandLineData.Content.PlainOutput then
            R := EStrings.LowerCase (FileSystem.JustFile (S, True));
         else
            R := S;
         end if;
         return R;
      end AdjustFilename;

      procedure Check_Output_Directory_Exists
      --# global in out CommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --# derives CommandLineData.Content,
      --#         SPARK_IO.File_Sys       from CommandLineData.Content,
      --#                                      SPARK_IO.File_Sys;
      is
      begin
         --# accept W, 169, CommandLineData.Content.Valid,   "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content,         "Direct updates OK here";
         if CommandLineData.Content.Valid and CommandLineData.Content.OutputDirectory then

            if FileSystem.IsDirectory (CommandLineData.Content.OutputDirectoryName) then
               -- Make sure it ends with an EndOfPath separator, so it can be used
               -- later in a call to FileSystem.InterpretRelative
               FileSystem.AppendEndOfPathIfNeeded (CommandLineData.Content.OutputDirectoryName);
            else
               CommandLineData.Content.Valid := False;
               PossibleError2 (ESOutputDirNotFound, AdjustFilename (CommandLineData.Content.OutputDirectoryName));
            end if;

         end if;
         --# end accept;
      end Check_Output_Directory_Exists;

   begin
      --# accept W, 169, CommandLineData.Content.Valid,   "Direct updates OK here";
      if CommandLineData.Content.Valid and DoRealRTCs then
         if not (CommandLineData.Content.RedType.RTC or else
                 CommandLineData.Content.RedType.ExpChecks) then
            CommandLineData.Content.Valid := False;
            PossibleError (ESRealRTCs);
         end if;
      end if;
      --# assert True;

      -- Check index file readable
      if CommandLineData.Content.Valid and CommandLineData.Content.Index then
         CommandLineData.Content.Valid := FileSystem.IsReadableFile (CommandLineData.Content.IndexFileName);
         PossibleError2 (EWIndex, AdjustFilename (CommandLineData.Content.IndexFileName));
      end if;

      --# assert True;

      -- Check warning control file readable
      if CommandLineData.Content.Valid and CommandLineData.Content.Warning then
         CommandLineData.Content.Valid := FileSystem.IsReadableFile (CommandLineData.Content.WarningFileName);
         PossibleError2 (EWWarning, AdjustFilename (CommandLineData.Content.WarningFileName));
      end if;

      --# assert True;

      -- Check target data file readable
      if CommandLineData.Content.Valid and CommandLineData.Content.TargetData then
         CommandLineData.Content.Valid := FileSystem.IsReadableFile (CommandLineData.Content.TargetDataFile);
         PossibleError2 (EWTarget, AdjustFilename (CommandLineData.Content.TargetDataFile));
      end if;

      --# assert True;

      -- Check config file readable
      if CommandLineData.Content.Valid and CommandLineData.Content.TargetConfig then
         CommandLineData.Content.Valid := FileSystem.IsReadableFile (CommandLineData.Content.TargetConfigFile);
         PossibleError2 (EWConfig, AdjustFilename (CommandLineData.Content.TargetConfigFile));
      end if;

      --# assert True;

      -- Check source files readable
      if CommandLineData.Content.Valid then
         for Ix in CommandLineData.SourceFilePositions
            range 1 .. CommandLineData.Content.NumberSource loop

            CommandLineData.Content.Valid :=
               FileSystem.IsReadableFile
               (StripAnyAt
                (CommandLineData.Content.SourceFileList (Ix).SourceFileName));

            PossibleError2
              (EWSource,
               AdjustFilename (CommandLineData.Content.SourceFileList (Ix).SourceFileName));
            exit when not CommandLineData.Content.Valid;
         end loop;
      end if;

      --# assert True;

      Check_Output_Directory_Exists;

      if CommandLineData.Content.Valid then
         WffNonOverlapping;
      end if;
      --# end accept;
   end WffCommandLine;

   ----------------------------------------------------------------------
   -- Does a quick "look ahead" through all the command-line
   -- switches to check for the presence of "-noswitch", "/noswitch"
   -- or any unambiguous abbreviation thereof.
   --
   -- Uses Ada.Command_Line directly, so is hidden from SPARK.
   ----------------------------------------------------------------------
   function Ignore_Default_Switch_File return Boolean
   --# global in SPARK_IO.File_Sys;
   is
      --# hide Ignore_Default_Switch_File;
      Result   : Boolean;
      ArgCount : constant Natural := Ada.Command_Line.Argument_Count;
   begin
      Result := False;
      for I in Positive range 1 .. ArgCount loop
         declare
            Arg : constant String := Ada.Command_Line.Argument (I);
         begin
            if Arg'Length >= 5 then
               declare
                  Prefix : constant Character := Arg (1);
                  Switch : constant String    := String (Arg (2 .. Arg'Last));
                  PrefixLegal : Boolean;
               begin
                  -- '-' is always OK, but still allow '/' on Windows for now
                  PrefixLegal := (Prefix = '-') or
                    (FileSystem.UseWindowsCommandLine and Prefix = '/');

                  if PrefixLegal and
                    (Switch = "noswitch" or else
                       Switch = "noswitc" or else
                       Switch = "noswit" or else
                       Switch = "noswi" or else
                       Switch = "nosw") then

                     Result := True;
                     exit;
                  end if;
               end;
            end if;
         end;
      end loop;
      return Result;
   end Ignore_Default_Switch_File;

   procedure Process
   is

      CommandString : CommandStrings;
      DoRealRTCs    : Boolean := False;

      procedure ReadCommandLine (CommandString : out CommandStrings)
      --# global in out CommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --# derives CommandLineData.Content,
      --#         SPARK_IO.File_Sys       from *,
      --#                                      SPARK_IO.File_Sys &
      --#         CommandString           from SPARK_IO.File_Sys;
      is
         CmdLineFound         : Boolean;
         DefaultSwitchesFound : Boolean;
         SwitchLine           : FileSystem.TypCmdLine;
         SwitchPtr            : FileSystem.TypCmdLineRange;

         CmdLine              : FileSystem.TypCmdLine;
         CommandStringContent : CommandStringContents;
         CommandStringLocal   : CommandStrings;
      begin
         --# accept W, 169, CommandLineData.Content.DefaultSwitchFile,   "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.HelpRequested,   "Direct updates OK here" &
         --#        W, 169, CommandLineData.Content.PlainOutput,   "Direct updates OK here";
         CommandStringContent := CommandStringContents'
            (others => FileSystem.ArgumentSeparator);

         if Ignore_Default_Switch_File then
            CommandLineData.Content.DefaultSwitchFile := False;
            SwitchPtr := FileSystem.TypCmdLineRange'First;
         else
            ReadDefaultSwitches (DefaultSwitchesFound,
                                 SwitchLine,
                                 SwitchPtr);
            if DefaultSwitchesFound then
               CommandLineData.Content.DefaultSwitchFile := True;
               CopyString (SwitchLine, 1, CommandStringContent);
            end if;
         end if;

         FileSystem.ReadCmdLine (CmdLineFound, CmdLine);
         if CmdLineFound then
            CopyString (CmdLine, SwitchPtr, CommandStringContent);
            CommandStringLocal := CommandStrings'(1, CommandStringContent);
         else
            -- Nothing on command-line so abandon processing and
            -- indicate that the user needs Help!
            CommandLineData.Content.HelpRequested := True;
            -- Override setting of PlainOutput so Help information
            -- always appears with Examiner version number and date.
            CommandLineData.Content.PlainOutput := False;

            CommandStringLocal := CommandStrings'(SwitchPtr,
                                             CommandStringContent);
            CommandStringLocal.CurrentPosition := 1;
         end if;
         CommandString := CommandStringLocal;
         --# end accept;
      end ReadCommandLine;

   begin --Process
      --# accept W, 169, CommandLineData.Content.SyntaxOnly,   "Direct updates OK here" &
      --#        W, 169, CommandLineData.Content.FDLReserved,   "Direct updates OK here" &
      --#        W, 169, CommandLineData.Content.Spark95,   "Direct updates OK here" &
      --#        W, 169, CommandLineData.Content.AnnoChar,   "Direct updates OK here" &
      --#        W, 169, CommandLineData.Content.RedType.RealRTCs, "Direct updates OK here";
      CommandLineData.Content.SyntaxOnly  := False;
      CommandLineData.Content.AnnoChar    := '#';
      CommandLineData.Content.FDLreserved := True;
      CommandLineData.Content.Spark95     := True;

      ReadCommandLine  (CommandString);

      -- If there's no command line at all, then we assume Help
      -- is needed, so don't bother to parse the remainder or default switches.
      if not CommandLineData.Content.HelpRequested then
         ParseCommandLine (CommandString, DoRealRTCs);

         if CommandLineData.Content.Valid then
            WffCommandLine (DoRealRTCs);
         end if;

         -- Override setting of RedType.RealRTCs with user selection
         CommandLineData.Content.RedType.RealRTCs := DoRealRTCs;
      end if;
      --# end accept;
   end Process;

   procedure ProcessDefaultsFromSwitchFile
   is

      CommandString : CommandStrings;
      DoRealRTCs    : Boolean := False;

      procedure ReadSwitchFile (CommandString : out CommandStrings)
      --# global in out CommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --# derives CommandLineData.Content,
      --#         SPARK_IO.File_Sys       from *,
      --#                                      SPARK_IO.File_Sys &
      --#         CommandString           from SPARK_IO.File_Sys;
      is
         DefaultSwitchesFound : Boolean;
         SwitchLine           : FileSystem.TypCmdLine;
         SwitchPtr            : FileSystem.TypCmdLineRange;

         CommandStringContent : CommandStringContents;
      begin
         CommandStringContent := CommandStringContents'
            (others => FileSystem.ArgumentSeparator);

         ReadDefaultSwitches (DefaultSwitchesFound,
                              SwitchLine,
                              SwitchPtr);
         if DefaultSwitchesFound then
            --# accept W, 169, CommandLineData.Content.DefaultSwitchFile, "Direct updates OK here";
            CommandLineData.Content.DefaultSwitchFile := True;
            --# end accept;
            CopyString (SwitchLine, 1, CommandStringContent);
         end if;
         CommandString := CommandStrings'(SwitchPtr, CommandStringContent);
         CommandString.CurrentPosition := 1;
      end ReadSwitchFile;

      procedure ParseSwitchFile (CommandString : in     CommandStrings;
                                 DoRealRTCs    : in out Boolean)
      --# global in out CommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --# derives CommandLineData.Content,
      --#         DoRealRTCs,
      --#         SPARK_IO.File_Sys       from *,
      --#                                      CommandLineData.Content,
      --#                                      CommandString;
      is
         NextSymbol         : Symbols;
         LocalCommandString : CommandStrings;
      begin
         LocalCommandString := CommandString;
         GetNextSymbol (LocalCommandString, NextSymbol);
         --# accept F, 10, NextSymbol, "NextSymbol not used here" &
         --#        F, 10, LocalCommandString, "LocalCommandString not used here";
         ParseCommandOptions (LocalCommandString, NextSymbol, DoRealRTCs);
      end ParseSwitchFile;

   begin --ProcessDefaultsFromSwitchFile

      if Ignore_Default_Switch_File then
         null;
      else
         ReadSwitchFile (CommandString);
         ParseSwitchFile (CommandString, DoRealRTCs);

         -- Override setting of RedType.RealRTCs with user selection
         --# accept W, 169, CommandLineData.Content.RedType.RealRTCs, "Direct updates OK here";
         CommandLineData.Content.RedType.RealRTCs := DoRealRTCs;
         --# end accept;
      end if;
   end ProcessDefaultsFromSwitchFile;

end CommandLineHandler;
