-- $Id: cells.adb 12717 2009-03-13 15:05:15Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with SystemErrors,
     Statistics;

package body Cells
is
   Null_Cell_Content : constant Cell_Content :=
     Cell_Content'(A_Ptr     => Null_Cell,
                   B_Ptr     => Null_Cell,
                   C_Ptr     => Null_Cell,
                   Copy      => Null_Cell,
                   Free      => False,
                   Kind      => Unknown_Kind,
                   Rank      => Unknown_Rank,
                   Lex_Str   => LexTokenManager.NullString,
                   Marked    => False,
                   Op_Symbol => SPSymbols.RWnull,
                   Val       => 0,
                   Assoc_Var => Dictionary.NullSymbol);

   procedure Initialize (Heap :    out Heap_Record)
   is
   begin -- initialization;
      Heap.High_Mark      := Cell (0);
      Heap.Next_Free_Cell := Cell (0);

      --# accept F, 23, Heap.List_Of_Cells, "Partial but effective initialization";
      Heap.List_Of_Cells (Null_Cell) := Null_Cell_Content;

      --# accept F, 602, Heap, Heap.List_Of_Cells, "Partial but effective initialization";
   end Initialize;

   --------------------------------------------------------------------------

   function Are_Identical (Cell_1,
                           Cell_2 : Cell) return Boolean
   is
   begin
      return Cell_1 = Cell_2;
   end Are_Identical;

   --------------------------------------------------------------------------

   function Get_A_Ptr (Heap     : Heap_Record;
                       CellName : Cell) return Cell
   is
   begin
      return Heap.List_Of_Cells (CellName).A_Ptr;
   end Get_A_Ptr;

   --------------------------------------------------------------------------

   function Get_B_Ptr (Heap     : Heap_Record;
                       CellName : Cell) return Cell
   is
   begin
      return Heap.List_Of_Cells (CellName).B_Ptr;
   end Get_B_Ptr;

   --------------------------------------------------------------------------

   function Get_C_Ptr (Heap     : Heap_Record;
                       CellName : Cell) return Cell
   is
   begin
      return Heap.List_Of_Cells (CellName).C_Ptr;
   end Get_C_Ptr;

   --------------------------------------------------------------------------

   function Get_Natural_Value (Heap     : Heap_Record;
                               CellName : Cell) return Natural
   is
   begin
      return Heap.List_Of_Cells (CellName).Val;
   end Get_Natural_Value;

   --------------------------------------------------------------------------

   function Get_Rank (Heap     : Heap_Record;
                      CellName : Cell) return Cell_Rank
   is
   begin
      return Heap.List_Of_Cells (CellName).Rank;
   end Get_Rank;

   --------------------------------------------------------------------------

   function Get_Copy (Heap     : Heap_Record;
                      CellName : Cell) return Cell
   is
   begin
      return Heap.List_Of_Cells (CellName).Copy;
   end Get_Copy;

   --------------------------------------------------------------------------

   function Is_Free (Heap     : Heap_Record;
                     CellName : Cell) return Boolean
   is
   begin
      return Heap.List_Of_Cells (CellName).Free;
   end Is_Free;

   --------------------------------------------------------------------------

   function Is_Marked (Heap     : Heap_Record;
                       CellName : Cell) return Boolean
   is
   begin
      return Heap.List_Of_Cells (CellName).Marked;
   end Is_Marked;

   --------------------------------------------------------------------------

   function Is_Null_Cell (CellName : Cell) return Boolean
   is
   begin
      return CellName = Null_Cell;
   end Is_Null_Cell;

   --------------------------------------------------------------------------

   function Is_Const_Cell (Heap     : Heap_Record;
                           CellName : Cell) return Boolean
   is
   begin
      return (Heap.List_Of_Cells (CellName).Kind = Manifest_Const) or
         (Heap.List_Of_Cells (CellName).Kind = Named_Const);
   end Is_Const_Cell;

   --------------------------------------------------------------------------

   function Is_Reference_Cell (Heap     : Heap_Record;
                               CellName : Cell) return Boolean
   is
   begin
      return Heap.List_Of_Cells (CellName).Kind = Reference;
   end Is_Reference_Cell;

   --------------------------------------------------------------------------

   function Get_Kind (Heap     : Heap_Record;
                      CellName : Cell) return Cell_Kind
   is
   begin
      return Heap.List_Of_Cells (CellName).Kind;
   end Get_Kind;

   --------------------------------------------------------------------------

   function Get_Op_Symbol (Heap     : Heap_Record;
                           CellName : Cell) return SPSymbols.SPSymbol
   is
   begin
      return Heap.List_Of_Cells (CellName).Op_Symbol;
   end Get_Op_Symbol;

   --------------------------------------------------------------------------

   function Get_Lex_Str (Heap     : Heap_Record;
                         CellName : Cell) return LexTokenManager.LexString
   is
   begin
      return Heap.List_Of_Cells (CellName).Lex_Str;
   end Get_Lex_Str;

   --------------------------------------------------------------------------

   function Get_Symbol_Value      (Heap     : Heap_Record;
                           CellName : Cell) return Dictionary.Symbol
   is
   begin
      return Dictionary.ConvertSymbolRef
         (ExaminerConstants.RefType
          (Heap.List_Of_Cells (CellName).Val));
   end Get_Symbol_Value;

   --------------------------------------------------------------------------

   function Cell_Ref (Cell_Name : in Cell) return Natural
   is
   begin
      return Natural (Cell_Name);
   end Cell_Ref;


   --------------------------------------------------------------------------

   procedure Set_A_Ptr (Heap   : in out Heap_Record;
                        Cell_1,
                        Cell_2 : in     Cell)
   is
   begin
      Heap.List_Of_Cells (Cell_1).A_Ptr := Cell_2;
   end Set_A_Ptr;

   --------------------------------------------------------------------------

   procedure Set_B_Ptr (Heap   : in out Heap_Record;
                        Cell_1,
                        Cell_2 : in     Cell)
   is
   begin
      Heap.List_Of_Cells (Cell_1).B_Ptr := Cell_2;
   end Set_B_Ptr;

   --------------------------------------------------------------------------

   procedure Set_C_Ptr (Heap   : in out Heap_Record;
                        Cell_1,
                        Cell_2 : in     Cell)
   is
   begin
      Heap.List_Of_Cells (Cell_1).C_Ptr := Cell_2;
   end Set_C_Ptr;

   --------------------------------------------------------------------------

   procedure Copy_Contents  (Heap        : in out Heap_Record;
                             Source,
                             Destination : in     Cell)
   is
   begin
      Heap.List_Of_Cells (Destination) :=
        Cell_Content'(A_Ptr     => Heap.List_Of_Cells (Source).A_Ptr,
                      B_Ptr     => Heap.List_Of_Cells (Source).B_Ptr,
                      C_Ptr     => Heap.List_Of_Cells (Destination).C_Ptr,
                      Copy      => Heap.List_Of_Cells (Destination).Copy,
                      Free      => Heap.List_Of_Cells (Destination).Free,
                      Kind      => Heap.List_Of_Cells (Source).Kind,
                      Rank      => Heap.List_Of_Cells (Source).Rank,
                      Lex_Str   => Heap.List_Of_Cells (Source).Lex_Str,
                      Marked    => Heap.List_Of_Cells (Destination).Marked,
                      Op_Symbol => Heap.List_Of_Cells (Source).Op_Symbol,
                      Val       => Heap.List_Of_Cells (Source).Val,
                      Assoc_Var => Heap.List_Of_Cells (Source).Assoc_Var);
   end Copy_Contents;

   --------------------------------------------------------------------------

   procedure Create_Cell (Heap     : in out Heap_Record;
                          CellName :    out Cell)
   is
      NewCell : Cell;
   begin
      if Heap.Next_Free_Cell /= Null_Cell then
         -- There are cells in the returned free list
         NewCell := Heap.Next_Free_Cell;
         Heap.Next_Free_Cell := Heap.List_Of_Cells (Heap.Next_Free_Cell).A_Ptr;
         Heap.List_Of_Cells (NewCell) := Null_Cell_Content;
         CellName := NewCell;
      elsif Heap.High_Mark < Cell'Last then
         -- Free list empty but haven't used up array yet
         Heap.High_Mark := Heap.High_Mark + 1;
         NewCell := Heap.High_Mark;
         Heap.List_Of_Cells (NewCell) := Null_Cell_Content;
         CellName := NewCell;
      else
         -- Array and returned cells in free list both used up
         -- Set table use to 100%
         Statistics.SetTableUsage (Statistics.VCGHeap,
                                   Integer (Cell'Last));
         SystemErrors.FatalError (SystemErrors.VCGHeapIsExhausted, "");
         CellName := Null_Cell;
      end if;
   end Create_Cell;

   --------------------------------------------------------------------------

   procedure Create_Copy (Heap     : in out Heap_Record;
                          CellName : in     Cell)
   is
      NewCell : Cell;
   begin
      Create_Cell (Heap, NewCell);
      Heap.List_Of_Cells (NewCell).Kind      := Heap.List_Of_Cells (CellName).Kind;
      Heap.List_Of_Cells (NewCell).Rank      := Heap.List_Of_Cells (CellName).Rank;
      Heap.List_Of_Cells (NewCell).Lex_Str   := Heap.List_Of_Cells (CellName).Lex_Str;
      Heap.List_Of_Cells (NewCell).Op_Symbol := Heap.List_Of_Cells (CellName).Op_Symbol;
      Heap.List_Of_Cells (NewCell).Val       := Heap.List_Of_Cells (CellName).Val;
      Heap.List_Of_Cells (NewCell).Assoc_Var := Heap.List_Of_Cells (CellName).Assoc_Var;
      Heap.List_Of_Cells (CellName).Copy     := NewCell;
   end Create_Copy;

   --------------------------------------------------------------------------

   procedure Dispose_Of_Cell (Heap     : in out Heap_Record;
                              CellName : in     Cell)
   is
   begin
      if Heap.List_Of_Cells (CellName).Free then
         SystemErrors.FatalError (SystemErrors.VCGHeapIsCorrupted, "in DisposeOfCell");
      else
         Heap.List_Of_Cells (CellName).A_Ptr := Heap.Next_Free_Cell;
         Heap.List_Of_Cells (CellName).Free := True;
         Heap.Next_Free_Cell := CellName;
      end if;
   end Dispose_Of_Cell;

   --------------------------------------------------------------------------

   procedure Mark_Cell (Heap     : in out Heap_Record;
                        CellName : in     Cell)
   is
   begin
      Heap.List_Of_Cells (CellName).Marked := True;
   end Mark_Cell;

   --------------------------------------------------------------------------

   procedure UnMark_Cell (Heap     : in out Heap_Record;
                          CellName : in     Cell)
   is
   begin
      Heap.List_Of_Cells (CellName).Marked := False;
   end UnMark_Cell;

   --------------------------------------------------------------------------

   procedure Set_Kind (Heap      : in out Heap_Record;
                       CellName  : in     Cell;
                       KindConst : in     Cell_Kind)
   is
   begin
      Heap.List_Of_Cells (CellName).Kind := KindConst;
   end Set_Kind;

   --------------------------------------------------------------------------

   procedure Set_Rank (Heap      : in out Heap_Record;
                       CellName  : in     Cell;
                       Rank      : in     Cell_Rank)
   is
   begin
      Heap.List_Of_Cells (CellName).Rank := Rank;
   end Set_Rank;

   --------------------------------------------------------------------------

   procedure Set_Op_Symbol (Heap     : in out Heap_Record;
                            CellName : in     Cell;
                            Sym      : in     SPSymbols.SPSymbol)
   is
   begin
      Heap.List_Of_Cells (CellName).Op_Symbol := Sym;
   end Set_Op_Symbol;

   --------------------------------------------------------------------------

   procedure Set_Lex_Str (Heap     : in out Heap_Record;
                          CellName : in     Cell;
                          Str      : in     LexTokenManager.LexString)
   is
   begin
      Heap.List_Of_Cells (CellName).Lex_Str := Str;
   end Set_Lex_Str;

   --------------------------------------------------------------------------

   procedure Set_Natural_Value (Heap     : in out Heap_Record;
                                CellName : in     Cell;
                                Value    : in     Natural)
   is
   begin
      Heap.List_Of_Cells (CellName).Val := Value;
   end Set_Natural_Value;

   --------------------------------------------------------------------------

   procedure Set_Symbol_Value (Heap     : in out Heap_Record;
                               CellName : in     Cell;
                               Sym      : in     Dictionary.Symbol)
   is
   begin
      Set_Natural_Value (Heap,
                         CellName,
                         Natural (Dictionary.SymbolRef (Sym)));
   end Set_Symbol_Value;


   --------------------------------------------------------------------------
   procedure Report_Usage (TheHeap : in Heap_Record)
   is
   begin
      -- as the heap now uses the free list before increasing High_Mark,
      -- the max usage is High_Mark
      Statistics.SetTableUsage (Statistics.VCGHeap,
                                Integer (TheHeap.High_Mark));
   end Report_Usage;

   --------------------------------------------------------------------------
   procedure Set_Assoc_Var (Heap     : in out Heap_Record;
                            CellName : in     Cell;
                            VarSym   : in     Dictionary.Symbol)
   is
   begin
      Heap.List_Of_Cells (CellName).Assoc_Var := VarSym;
   end Set_Assoc_Var;

   --------------------------------------------------------------------------
   function Get_Assoc_Var (Heap     : in     Heap_Record;
                           CellName : in     Cell) return Dictionary.Symbol
   is
   begin
      return Heap.List_Of_Cells (CellName).Assoc_Var;
   end Get_Assoc_Var;

   --------------------------------------------------------------------------
   function VCG_Heap_Size return Natural
   is
      SU : constant := 8; -- bits
   begin
      return ((Heap_Record'Size + SU) - 1) / SU;
   end VCG_Heap_Size;

end Cells;
