%  $Id: listthm.pro 12104 2009-01-13 09:51:38Z Bill Ellis $
%-------------------------------------------------------------------------------
%  (C) Praxis High Integrity Systems Limited
%-------------------------------------------------------------------------------
% 
%  The SPARK toolset is free software; you can redistribute it and/or modify it
%  under terms of the GNU General Public License as published by the Free
%  Software Foundation; either version 3, or (at your option) any later
%  version. The SPARK toolset is distributed in the hope that it will be
%  useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
%  Public License for more details. You should have received a copy of the GNU
%  General Public License distributed with the SPARK toolset; see file
%  COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
%  the license.
% 
%===============================================================================


/******** LIST: top-level checker command ********/
list :-                 /* LIST THE WHOLE VC */
        \+ command_arg(list,_),
        max_hyp_no(HMAX),
        list(h,1-HMAX),
        write('-->'),
        nl,
        max_conc_no(CMAX),
        listconcs(1,CMAX),
        !.
list :-
        command_arg(list,VCPART),
        (
           VCPART=HC#N,
           list(HC,N)
        ;
           VCPART=HC#N-M,
           list(HC,N-M)
        ;                                                       /* CFR010 */
           VCPART = deleted,                                    /* CFR010 */
           list_deleted_hypotheses                              /* CFR010 */
        ;                                                       /* CFR010 */
           VCPART = forgotten,                                  /* CFR010 */
           list_forgotten_hypotheses                            /* CFR010 */
        ),
        fail.
list :- !.


list(HORC,N) :-           /* LIST A SINGLE HYPOTHESIS/CONCLUSION */
        integer(N),
        (
           HORC=h,
           HC=hyp
        ;
           HORC=c,
           HC=conc
        ),
        X=..[HC,N,Y],
        call(X),
        (
           HC=hyp,
           write('H')
        ;
           HC=conc,
           write('  C')
        ),
        print(N),
        put_code(58),  /* ":" */
        tab(2),
        print(Y),
        nl, !.

list(HORC,N-M) :-         /* LIST HYPOTHESES/CONCLUSIONS IN A RANGE */
        integer(N),
        integer(M),
        N=<M,
        (
           HORC=h,
           listhyps(N,M)
        ;
           HORC=c,
           listconcs(N,M)
        ),
        !.


/*** LISTHYPS(N,M) -- list non-forgotten hypotheses in range N-M ***/
listhyps(N,N) :-
        hyp(N,F),
        !,
        (
           forgotten(N)
        ;
           deleted(N)
        ;
           write('H'),
           print(N),
           write(':  '),
           print(F),
           nl
        ),
        !.
listhyps(N,M) :-
        N<M,
        !,
        listhyps(N,N),
        K is N+1,
        listhyps(K,M),
        !.
listhyps(_,_) :- !.


/*** LISTCONCS(N,M) -- list conclusions in range N-M ***/
listconcs(1,0) :-
        !,
        write('  NO CONCLUSIONS LEFT'),
        !.
listconcs(N,N) :-
        conc(N,F),
        !,
        write('  C'),
        print(N),
        write(':  '),
        print(F),
        nl,
        !.
listconcs(N,M) :-
        N<M,
        !,
        listconcs(N,N),
        K is N+1,
        listconcs(K,M),
        !.
listconcs(_,_) :- !.


/*** MAX_CONC_NO(MAX) -- highest used conclusion number ***/
max_conc_no(N) :-
        conc(N,_),
        \+ (( conc(M,_), M>N )),
        !.
max_conc_no(0).


/*** MAX_HYP_NO(MAX) -- highest used hypothesis number ***/
max_hyp_no(N) :-
        hyp(N,_),
        \+ (( hyp(M,_), M>N )),
        !.
max_hyp_no(0).


/*** PRINTVC - write VC out to a file. */
printvc :-
        current_vc(NAME, NUM),
        pvc_make_filename(NUM, FILENAME),
        !,
        write_vc_data(FILENAME, NAME),
        !,
        write('Printed VC to file: '),
        write(FILENAME),
        nl,
        !.


/* PVC_MAKE_FILENAME(NUM, NAME) -- create filename for writing VC to ***/
pvc_make_filename(N, FILE) :-
        name(N, NL),
        append(NL, ".LIS", TL),
        append("VC", TL, FL),
        name(FILE, FL),
        !.


/*** WRITE_VC_DATA(FILENAME, VC) -- write VC & status info out to file ***/
write_vc_data(FILE, VC) :-
        file_can_be_written(FILE),
        tell(FILE),
        write_vc_description(VC),
        !,
        list,
        !,
        write_forgotten_hyps,
        !,
        write_deleted_hyps,
        !,
        write_depth_information,
        !,
        told.
write_vc_data(FILE, _VC) :-
        \+ file_can_be_written(FILE),
        write('Warning: '),
        print(FILE),
        write(' cannot be written.'),
        nl,
        !,
        fail.


/*** WRITE_VC_DESCRIPTION(VC)  -- write its name and current step no. ***/
write_vc_description(VC) :-
        write('VC: '),
        print(VC),
        write(',        Step: '),
        step_number(N),
        print(N),
        put_code(46),
        nl, nl,
        !.


/*** WRITE_FORGOTTEN_HYPS ***/
write_forgotten_hyps :-
        \+ forgotten(_),
        !.
write_forgotten_hyps :-
        nl, nl,
        write('PLUS the following forgotten (hidden but useable) hypotheses:'),
        nl, nl,
        fail.
write_forgotten_hyps :-
        forgotten(N),
        list(h,N),
        fail.
write_forgotten_hyps :- !.


/*** WRITE_DELETED_HYPS ***/
write_deleted_hyps :-
        \+ deleted(_),
        !.
write_deleted_hyps :-
        nl, nl,
        write('PLUS the following deleted (currently unuseable) hypotheses:'),
        nl, nl,
        fail.
write_deleted_hyps :-
        deleted(N),
        write_deleted_hyp(N),
        fail.
write_deleted_hyps :- !.


/*** WRITE_DELETED_HYP(N): write it out */
write_deleted_hyp(N) :-           /* WRITE SINGLE DELETED HYPOTHESIS */
        deleted_hyp(N, Y),
        write('D'),
        print(N),
        put_code(58),             /* ":" */
        tab(2),
        print(Y),
        nl,
        !.


/*** WRITE_DEPTH_INFORMATION -- summarise current status ***/
write_depth_information :-
        case_pointer(0),
        !.
write_depth_information :-
        nl, nl,
        write('You are currently at depth '),
        case_pointer(DEPTH),
        print(DEPTH),
        write('.  Back-trace from here is:'),
        nl, nl,
        !,
        print_status_info_at_depth(DEPTH),
        !.


/*** PRINT_STATUS_INFO_AT_DEPTH(DEPTH) -- list previous depths back to top ***/
print_status_info_at_depth(0) :- !.
print_status_info_at_depth(DEPTH) :-
        DEPTH>0,
        !,
        NEWDEPTH is DEPTH-1,
        pvc_print_info_at_depth(DEPTH, NEWDEPTH),
        !,
        print_status_info_at_depth(NEWDEPTH),
        !.


/*** PVC_PRINT_INFO_AT_DEPTH(DEPTH, NEWDEPTH) -- info at this depth. ***/
pvc_print_info_at_depth(DEPTH, NEWDEPTH) :-
        subgoal_formula(DEPTH, _FORMULA, CONC_NO, METHOD),
        write('On exiting successfully from depth '),
        print(DEPTH),
        write(' you will have proved '),
        (
           integer(CONC_NO),
           write('C'),
           print(CONC_NO)
        ;
           write('<own-formula>')
        ),
        !,
        (
           NEWDEPTH = 0,
           write(' at the *TOP-LEVEL*, by ')
        ;
           write(' at the previous depth, by ')
        ),
        print(METHOD),
        put_code(46),
        nl,
        !,
        (
           METHOD = 'CASES',
           write('Progress so far on proof by cases:'),
           nl,
           list_case_status(DEPTH)
        ;
           true
        ),
        !.


/* list_deleted_hypotheses */                                   /* CFR010 */
list_deleted_hypotheses :-                                      /* CFR010 */
        \+ deleted(_),                                          /* CFR010 */
        !,                                                      /* CFR010 */
        write('No hypotheses have been deleted.'),              /* CFR010 */
        nl,                                                     /* CFR010 */
        fail.                                                   /* CFR010 */
list_deleted_hypotheses :-                                      /* CFR010 */
        write('DELETED HYPOTHESES :-'),                         /* CFR010 */
        nl,                                                     /* CFR010 */
        deleted(N),                                             /* CFR010 */
        write_deleted_hyp(N),                                   /* CFR010 */
        fail.                                                   /* CFR010 */
list_deleted_hypotheses :- !.                                   /* CFR010 */


/* list_forgotten_hypotheses */                                 /* CFR010 */
list_forgotten_hypotheses :-                                    /* CFR010 */
        \+ forgotten(_),                                        /* CFR010 */
        !,                                                      /* CFR010 */
        write('No hypotheses have been forgotten.'),            /* CFR010 */
        nl,                                                     /* CFR010 */
        fail.                                                   /* CFR010 */
list_forgotten_hypotheses :-                                    /* CFR010 */
        write('FORGOTTEN HYPOTHESES :-'),                       /* CFR010 */
        nl,                                                     /* CFR010 */
        forgotten(N),                                           /* CFR010 */
        list(h,N),                                              /* CFR010 */
        fail.                                                   /* CFR010 */
list_forgotten_hypotheses :- !.                                 /* CFR010 */
%###############################################################################
%END-OF-FILE
