-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with Maths, CommandLineData;

separate (Dictionary)
procedure Initialize (Write_To_File : in Boolean) is

   File   : SPARK_IO.File_Type := SPARK_IO.Null_File;
   Status : SPARK_IO.File_Status;

   procedure Assign_Initial_Value
   --# global out Dict;
   --# derives Dict from ;
   is
   begin
      Dict.File_Name       := E_Strings.Empty_String;
      Dict.TemporaryFile   := SPARK_IO.Null_File;
      Dict.Symbols.Current := NullSymbol;
      Dict.Main            :=
        MainProgramSet'
        (Subprogram     => NullSymbol,
         InheritClauses => NullSymbol,
         PriorityGiven  => False,
         ThePriority    => LexTokenManager.Null_String);
      Dict.Packages        :=
        PackageSet'(NullSymbol, NullSymbol, NullSymbol, NullSymbol, NullSymbol, NullSymbol, NullSymbol, NullSymbol);
      --# accept Flow, 32, Dict.Symbols.List, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 31, Dict.Symbols.List, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 602, Dict, Dict.Symbols.List, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 32, Dict.Types, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 31, Dict.Types, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 602, Dict, Dict.Types, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 32, Dict.Subprograms, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 31, Dict.Subprograms, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 602, Dict, Dict.Subprograms, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 32, Dict.NullVariable, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 31, Dict.NullVariable, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 602, Dict, Dict.NullVariable, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 32, Dict.ThePartition, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 31, Dict.ThePartition, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 602, Dict, Dict.ThePartition, "Flow errors due to undefined components of Dict OK";
   end Assign_Initial_Value;

   --------------------------------------------------------------------------------

   procedure Insert_Lex_String (Name  : in     String;
                                Token :    out LexTokenManager.Lex_String)
   --# global in out LexTokenManager.State;
   --# derives LexTokenManager.State,
   --#         Token                 from LexTokenManager.State,
   --#                                    Name;
   is
   begin
      LexTokenManager.Insert_Examiner_String (Str     => E_Strings.Copy_String (Str => Name),
                                              Lex_Str => Token);
   end Insert_Lex_String;

   --------------------------------------------------------------------------------

   procedure Add_Unknown_Type
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --# derives Dict,
   --#         LexTokenManager.State from *,
   --#                                    LexTokenManager.State;
   is
      Token    : LexTokenManager.Lex_String;
      The_Type : Symbol;
   begin
      Insert_Lex_String (Name  => "unknown",
                         Token => Token);
      RawDict.CreateType
        (Name      => Token,
         Comp_Unit => ContextManager.NullUnit,
         Loc       => LexTokenManager.Null_Token_Position,
         TypeMark  => The_Type);
      RawDict.SetTypeDiscriminant (The_Type, UnknownType);
      RawDict.SetTypeStatic (The_Type, True);
      Dict.Types.UnknownTypeMark := The_Type;
   end Add_Unknown_Type;

   --------------------------------------------------------------------------------

   procedure Add_Predefined_Package_Standard
   --# global in out Dict;
   --# derives Dict from *;
   is
      New_Package, Specification : Symbol;
   begin
      RawDict.CreatePackage
        (Name       => LexTokenManager.Standard_Token,
         Comp_Unit  => ContextManager.NullUnit,
         Loc        => LexTokenManager.Null_Token_Position,
         ThePackage => New_Package);
      RawDict.CreateDeclaration
        (Item        => New_Package,
         Context     => ProgramContext,
         Comp_Unit   => ContextManager.NullUnit,
         Loc         => LexTokenManager.Null_Token_Position,
         Declaration => Specification);
      RawDict.SetPackageSpecification (New_Package, Specification);
      Dict.Packages.PackageStandard := New_Package;
   end Add_Predefined_Package_Standard;

   --------------------------------------------------------------------------------

   procedure Add_Package_Standard
   --# global in     CommandLineData.Content;
   --#        in out Dict;
   --#        in out LexTokenManager.State;
   --# derives Dict,
   --#         LexTokenManager.State from *,
   --#                                    CommandLineData.Content,
   --#                                    LexTokenManager.State;
   is

      procedure Add_Predefined_Type (Token    : in     LexTokenManager.Lex_String;
                                     Context  : in     Contexts;
                                     The_Type :    out Symbol)
      --# global in out Dict;
      --# derives Dict     from *,
      --#                       Context,
      --#                       Token &
      --#         The_Type from Dict;
      is
         New_Type    : Symbol;
         Declaration : Symbol;
      begin
         RawDict.CreateType
           (Name      => Token,
            Comp_Unit => ContextManager.NullUnit,
            Loc       => LexTokenManager.Null_Token_Position,
            TypeMark  => New_Type);
         AddDeclaration
           (Item        => New_Type,
            Comp_Unit   => ContextManager.NullUnit,
            Loc         => Null_Location,
            Scope       => PredefinedScope,
            Context     => Context,
            Declaration => Declaration);
         RawDict.SetTypeDeclaration (New_Type, Declaration);
         The_Type := New_Type;
      end Add_Predefined_Type;

      --------------------------------------------------------------------------------

      procedure Add_Universal_Types
      --# global in out Dict;
      --# derives Dict from *;
      is

         procedure Add_Universal_Integer_Type
         --# global in out Dict;
         --# derives Dict from *;
         is
            The_Type : Symbol;
         begin
            Add_Predefined_Type (Token    => LexTokenManager.Universal_Integer_Token,
                                 Context  => ProofContext,
                                 The_Type => The_Type);
            AddIntegerTypeMark (The_Type, True, LexTokenManager.Null_String, LexTokenManager.Null_String);
            Dict.Types.TheUniversalTypes (UniversalIntegerType) := The_Type;
         end Add_Universal_Integer_Type;

         --------------------------------------------------------------------------------

         procedure Add_Universal_Real_Type
         --# global in out Dict;
         --# derives Dict from *;
         is
            The_Type : Symbol;
         begin
            Add_Predefined_Type (Token    => LexTokenManager.Universal_Real_Token,
                                 Context  => ProofContext,
                                 The_Type => The_Type);
            AddFloatingPointTypeMark (The_Type, True);
            Dict.Types.TheUniversalTypes (UniversalRealType) := The_Type;
         end Add_Universal_Real_Type;

         --------------------------------------------------------------------------------

         procedure Add_Universal_Fixed_Type
         --# global in out Dict;
         --# derives Dict from *;
         is
            The_Type : Symbol;
         begin
            Add_Predefined_Type (Token    => LexTokenManager.Universal_Fixed_Token,
                                 Context  => ProofContext,
                                 The_Type => The_Type);
            AddFixedPointTypeMark (The_Type, True);
            Dict.Types.TheUniversalTypes (UniversalFixedType) := The_Type;
         end Add_Universal_Fixed_Type;

      begin
         Add_Universal_Integer_Type;
         Add_Universal_Real_Type;
         Add_Universal_Fixed_Type;
      end Add_Universal_Types;

      --------------------------------------------------------------------------------

      procedure Add_Predefined_Type_Marks
      --# global in     CommandLineData.Content;
      --#        in out Dict;
      --#        in out LexTokenManager.State;
      --# derives Dict,
      --#         LexTokenManager.State from *,
      --#                                    CommandLineData.Content,
      --#                                    LexTokenManager.State;
      is

         procedure Add_Predefined_Character_Type
         --# global in     CommandLineData.Content;
         --#        in out Dict;
         --#        in out LexTokenManager.State;
         --# derives Dict,
         --#         LexTokenManager.State from *,
         --#                                    CommandLineData.Content,
         --#                                    LexTokenManager.State;
         is
            The_Type  : Symbol;
            Store_Rep : LexTokenManager.Lex_String;
         begin
            Add_Predefined_Type (Token    => LexTokenManager.Character_Token,
                                 Context  => ProgramContext,
                                 The_Type => The_Type);
            AddEnumerationTypeMark (The_Type, True);
            -- Set bounds for character type for use with 'first, 'last
            RawDict.SetTypeLower (The_Type, LexTokenManager.Zero_Value);

            case CommandLineData.Content.Language_Profile is
               when CommandLineData.SPARK83 =>
                  Maths.StorageRep (Maths.IntegerToValue (127), Store_Rep);
                  RawDict.SetTypeUpper (The_Type, Store_Rep);

               when CommandLineData.SPARK95 | CommandLineData.SPARK2005 =>
                  Maths.StorageRep (Maths.IntegerToValue (255), Store_Rep);
                  RawDict.SetTypeUpper (The_Type, Store_Rep);
            end case;

            Dict.Types.ThePredefinedTypes (PredefinedCharacterType) := The_Type;
         end Add_Predefined_Character_Type;

         --------------------------------------------------------------------------------

         procedure Add_Predefined_Boolean_Type
         --# global in out Dict;
         --#        in out LexTokenManager.State;
         --# derives Dict,
         --#         LexTokenManager.State from *,
         --#                                    LexTokenManager.State;
         is

            The_Type : Symbol;

            procedure Add_Predefined_Enumeration_Literal
              (Name     : in String;
               Value    : in LexTokenManager.Lex_String;
               The_Type : in Symbol)
            --# global in out Dict;
            --#        in out LexTokenManager.State;
            --# derives Dict                  from *,
            --#                                    LexTokenManager.State,
            --#                                    Name,
            --#                                    The_Type,
            --#                                    Value &
            --#         LexTokenManager.State from *,
            --#                                    Name;
            is
               Token : LexTokenManager.Lex_String;
            begin
               Insert_Lex_String (Name  => Name,
                                  Token => Token);
               AddEnumerationLiteral
                 (Name        => Token,
                  Comp_Unit   => ContextManager.NullUnit,
                  Declaration => Null_Location,
                  Position    => Value,
                  The_Type    => The_Type);
            end Add_Predefined_Enumeration_Literal;

         begin
            Add_Predefined_Type (Token    => LexTokenManager.Boolean_Token,
                                 Context  => ProgramContext,
                                 The_Type => The_Type);
            AddEnumerationTypeMark (The_Type, True);
            Add_Predefined_Enumeration_Literal (Name     => "False",
                                                Value    => LexTokenManager.False_Token,
                                                The_Type => The_Type);
            Add_Predefined_Enumeration_Literal (Name     => "True",
                                                Value    => LexTokenManager.True_Token,
                                                The_Type => The_Type);
            Dict.Types.ThePredefinedTypes (PredefinedBooleanType) := The_Type;
            Dict.Types.ThePredefinedLiterals (PredefinedFalse)    := RawDict.GetTypeFirstEnumerationLiteral (The_Type);
            Dict.Types.ThePredefinedLiterals (PredefinedTrue)     := RawDict.GetTypeLastEnumerationLiteral (The_Type);
         end Add_Predefined_Boolean_Type;

         --------------------------------------------------------------------------------

         procedure Add_Predefined_Integer_Type
         --# global in     CommandLineData.Content;
         --#        in out Dict;
         --#        in out LexTokenManager.State;
         --# derives Dict,
         --#         LexTokenManager.State from *,
         --#                                    CommandLineData.Content,
         --#                                    LexTokenManager.State;
         is
            Token    : LexTokenManager.Lex_String;
            The_Type : Symbol;
         begin
            Add_Predefined_Type (Token    => LexTokenManager.Integer_Token,
                                 Context  => ProgramContext,
                                 The_Type => The_Type);
            AddIntegerTypeMark (The_Type, True, LexTokenManager.Null_String, LexTokenManager.Null_String);
            SetTypeIsPredefined (The_Type);
            Dict.Types.ThePredefinedTypes (PredefinedIntegerType) := The_Type;
            if CommandLineData.Content.Target_Data then
               Insert_Lex_String (Name  => "Long_Integer",
                                  Token => Token);
               Add_Predefined_Type (Token    => Token,
                                    Context  => ProgramContext,
                                    The_Type => The_Type);
               AddIntegerTypeMark (The_Type, True, LexTokenManager.Null_String, LexTokenManager.Null_String);
               SetTypeIsPredefined (The_Type);
               Dict.Types.ThePredefinedTypes (PredefinedLongIntegerType) := The_Type;
            else
               Dict.Types.ThePredefinedTypes (PredefinedLongIntegerType) := NullSymbol;
            end if;
         end Add_Predefined_Integer_Type;

         --------------------------------------------------------------------------------

         procedure Add_Predefined_Float_Type
         --# global in     CommandLineData.Content;
         --#        in out Dict;
         --#        in out LexTokenManager.State;
         --# derives Dict,
         --#         LexTokenManager.State from *,
         --#                                    CommandLineData.Content,
         --#                                    LexTokenManager.State;
         is
            Token    : LexTokenManager.Lex_String;
            The_Type : Symbol;
         begin
            Add_Predefined_Type (Token    => LexTokenManager.Float_Token,
                                 Context  => ProgramContext,
                                 The_Type => The_Type);
            AddFloatingPointTypeMark (The_Type, True);
            SetTypeIsPredefined (The_Type);
            Dict.Types.ThePredefinedTypes (PredefinedFloatType) := The_Type;
            if CommandLineData.Content.Target_Data then
               Insert_Lex_String (Name  => "Long_Float",
                                  Token => Token);
               Add_Predefined_Type (Token    => Token,
                                    Context  => ProgramContext,
                                    The_Type => The_Type);
               AddFloatingPointTypeMark (The_Type, True);
               SetTypeIsPredefined (The_Type);
               Dict.Types.ThePredefinedTypes (PredefinedLongFloatType) := The_Type;
            else
               Dict.Types.ThePredefinedTypes (PredefinedLongFloatType) := NullSymbol;
            end if;
         end Add_Predefined_Float_Type;

         --------------------------------------------------------------------------------

         procedure Add_Duration
         --# global in out Dict;
         --# derives Dict from *;
         is
            The_Type : Symbol;
         begin
            Add_Predefined_Type (Token    => LexTokenManager.Duration_Token,
                                 Context  => ProgramContext,
                                 The_Type => The_Type);
            AddFixedPointTypeMark (The_Type, True);
            Dict.Types.ThePredefinedTypes (PredefinedDurationType) := The_Type;
         end Add_Duration;

         --------------------------------------------------------------------------------

         procedure Add_Predefined_String_Type
         --# global in out Dict;
         --# derives Dict from *;
         is
            The_Type : Symbol;
         begin
            Add_Predefined_Type (Token    => LexTokenManager.String_Token,
                                 Context  => ProgramContext,
                                 The_Type => The_Type);
            AddArrayTypeMark (The_Type, GetPredefinedCharacterType, False);
            AddNewArrayIndex
              (TheArrayType => The_Type,
               IndexType    => GetPredefinedPositiveSubtype,
               Comp_Unit    => ContextManager.NullUnit,
               Declaration  => Null_Location);
            Dict.Types.ThePredefinedTypes (PredefinedStringType) := The_Type;
         end Add_Predefined_String_Type;

         --------------------------------------------------------------------------------

         procedure Add_Predefined_Natural_Subtype
         --# global in out Dict;
         --# derives Dict from *;
         is
            The_Type : Symbol;
         begin
            Add_Predefined_Type (Token    => LexTokenManager.Natural_Token,
                                 Context  => ProgramContext,
                                 The_Type => The_Type);
            AddSubtype (The_Type, GetPredefinedIntegerType);
            AddIntegerTypeMark (The_Type, True, LexTokenManager.Zero_Value, LexTokenManager.Null_String);
            Dict.Types.ThePredefinedTypes (PredefinedNaturalSubtype) := The_Type;
         end Add_Predefined_Natural_Subtype;

         --------------------------------------------------------------------------------

         procedure Add_Predefined_Positive_Subtype
         --# global in out Dict;
         --# derives Dict from *;
         is
            The_Type : Symbol;
         begin
            Add_Predefined_Type (Token    => LexTokenManager.Positive_Token,
                                 Context  => ProgramContext,
                                 The_Type => The_Type);
            AddSubtype (The_Type, GetPredefinedIntegerType);
            AddIntegerTypeMark (The_Type, True, LexTokenManager.One_Value, LexTokenManager.Null_String);
            Dict.Types.ThePredefinedTypes (PredefinedPositiveSubtype) := The_Type;
         end Add_Predefined_Positive_Subtype;

      begin
         Add_Predefined_Boolean_Type;
         Add_Predefined_Integer_Type;
         Add_Predefined_Float_Type;
         Add_Predefined_Character_Type;
         Add_Predefined_Natural_Subtype;
         Add_Predefined_Positive_Subtype;
         Add_Predefined_String_Type;
         if not CommandLineData.Content.No_Duration then
            Add_Duration;
         end if;
      end Add_Predefined_Type_Marks;

      --------------------------------------------------------------------------------

      procedure Add_Null_Own_Variable
      --# global in out Dict;
      --#        in out LexTokenManager.State;
      --# derives Dict,
      --#         LexTokenManager.State from *,
      --#                                    LexTokenManager.State;
      is
         New_Variable, Own_Variable, Package_Standard : Symbol;
         Null_Str                                     : LexTokenManager.Lex_String;
      begin
         Package_Standard := Dict.Packages.PackageStandard;
         Insert_Lex_String (Name  => "null",
                            Token => Null_Str);
         RawDict.CreateVariable
           (Name      => Null_Str,
            Comp_Unit => ContextManager.NullUnit,
            Loc       => LexTokenManager.Null_Token_Position,
            Variable  => New_Variable);
         RawDict.CreateOwnVariable
           (Variable    => New_Variable,
            Owner       => Package_Standard,
            Comp_Unit   => ContextManager.NullUnit,
            Loc         => LexTokenManager.Null_Token_Position,
            OwnVariable => Own_Variable);
         RawDict.SetVariableOwnVariable (New_Variable, Own_Variable);
         RawDict.SetPackageOwnVariables (Package_Standard, Own_Variable);
         RawDict.SetOwnVariableMode (Own_Variable, OutMode);
         Dict.NullVariable := New_Variable;
      end Add_Null_Own_Variable;

   begin
      Add_Predefined_Package_Standard;
      Add_Universal_Types;
      Add_Predefined_Type_Marks;
      -- package standard has a mode out own variable called "Null" which is used
      -- as a "data sink" for operations that affect only things outside the SPARK
      -- boundary such as delay statements or calls to procedures which derive null
      -- from something.
      Add_Null_Own_Variable;
   end Add_Package_Standard;

   --------------------------------------------------------------------------------

   procedure Add_Predefined_Constant (Name        : in String;
                                      Asc_Code    : in Natural;
                                      The_Package : in Symbol)
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --# derives Dict                  from *,
   --#                                    Asc_Code,
   --#                                    LexTokenManager.State,
   --#                                    Name,
   --#                                    The_Package &
   --#         LexTokenManager.State from *,
   --#                                    Asc_Code,
   --#                                    Name;
   is
      Token        : LexTokenManager.Lex_String;
      The_Constant : Symbol;
      Declaration  : Symbol;
      Store_Rep    : LexTokenManager.Lex_String;
   begin
      Insert_Lex_String (Name  => Name,
                         Token => Token);
      RawDict.CreateConstant
        (Name        => Token,
         TypeMark    => GetPredefinedCharacterType,
         Static      => True,
         Comp_Unit   => ContextManager.NullUnit,
         Loc         => LexTokenManager.Null_Token_Position,
         TheConstant => The_Constant);
      -- Add values for predefined constants
      Maths.StorageRep (Maths.IntegerToValue (Asc_Code), Store_Rep);
      RawDict.SetConstantValue (The_Constant, Store_Rep);
      AddDeclaration
        (Item        => The_Constant,
         Comp_Unit   => ContextManager.NullUnit,
         Loc         => Null_Location,
         Scope       => VisibleScope (The_Package),
         Context     => ProgramContext,
         Declaration => Declaration);
      RawDict.SetConstantDeclaration (The_Constant, Declaration);
   end Add_Predefined_Constant;

   -------------------------------------

   -- Adds Character constants which are common to both
   -- SPARK83 and SPARK95 _AND_ have the same identifier - this
   -- is a subset of the first 127 Characters.
   procedure Add_Common_Characters (To_The_Package : in Symbol)
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --# derives Dict                  from *,
   --#                                    LexTokenManager.State,
   --#                                    To_The_Package &
   --#         LexTokenManager.State from *;
   is
   begin
      Add_Predefined_Constant (Name        => "NUL",
                               Asc_Code    => 0,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "SOH",
                               Asc_Code    => 1,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "STX",
                               Asc_Code    => 2,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "ETX",
                               Asc_Code    => 3,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "EOT",
                               Asc_Code    => 4,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "ENQ",
                               Asc_Code    => 5,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "ACK",
                               Asc_Code    => 6,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "BEL",
                               Asc_Code    => 7,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "BS",
                               Asc_Code    => 8,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "HT",
                               Asc_Code    => 9,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LF",
                               Asc_Code    => 10,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "VT",
                               Asc_Code    => 11,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "FF",
                               Asc_Code    => 12,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "CR",
                               Asc_Code    => 13,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "SO",
                               Asc_Code    => 14,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "SI",
                               Asc_Code    => 15,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "DLE",
                               Asc_Code    => 16,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "DC1",
                               Asc_Code    => 17,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "DC2",
                               Asc_Code    => 18,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "DC3",
                               Asc_Code    => 19,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "DC4",
                               Asc_Code    => 20,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "NAK",
                               Asc_Code    => 21,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "SYN",
                               Asc_Code    => 22,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "ETB",
                               Asc_Code    => 23,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "CAN",
                               Asc_Code    => 24,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "EM",
                               Asc_Code    => 25,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "SUB",
                               Asc_Code    => 26,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "ESC",
                               Asc_Code    => 27,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "FS",
                               Asc_Code    => 28,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "GS",
                               Asc_Code    => 29,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "RS",
                               Asc_Code    => 30,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "US",
                               Asc_Code    => 31,
                               The_Package => To_The_Package);

      --# assert True;

      Add_Predefined_Constant (Name        => "LC_A",
                               Asc_Code    => 97,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_B",
                               Asc_Code    => 98,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_C",
                               Asc_Code    => 99,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_D",
                               Asc_Code    => 100,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_E",
                               Asc_Code    => 101,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_F",
                               Asc_Code    => 102,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_G",
                               Asc_Code    => 103,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_H",
                               Asc_Code    => 104,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_I",
                               Asc_Code    => 105,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_J",
                               Asc_Code    => 106,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_K",
                               Asc_Code    => 107,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_L",
                               Asc_Code    => 108,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_M",
                               Asc_Code    => 109,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_N",
                               Asc_Code    => 110,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_O",
                               Asc_Code    => 111,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_P",
                               Asc_Code    => 112,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_Q",
                               Asc_Code    => 113,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_R",
                               Asc_Code    => 114,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_S",
                               Asc_Code    => 115,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_T",
                               Asc_Code    => 116,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_U",
                               Asc_Code    => 117,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_V",
                               Asc_Code    => 118,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_W",
                               Asc_Code    => 119,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_X",
                               Asc_Code    => 120,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_Y",
                               Asc_Code    => 121,
                               The_Package => To_The_Package);
      Add_Predefined_Constant (Name        => "LC_Z",
                               Asc_Code    => 122,
                               The_Package => To_The_Package);

      Add_Predefined_Constant (Name        => "DEL",
                               Asc_Code    => 127,
                               The_Package => To_The_Package);
   end Add_Common_Characters;

   --------------------------------------------------------------------------------

   procedure Add_Package_ASCII
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --# derives Dict,
   --#         LexTokenManager.State from *,
   --#                                    LexTokenManager.State;
   is
      Token         : LexTokenManager.Lex_String;
      Package_ASCII : Symbol;
   begin
      Insert_Lex_String (Name  => "ASCII",
                         Token => Token);
      AddNewPackage
        (Name        => Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Scope       => PredefinedScope,
         TheParent   => NullSymbol,
         ChildSort   => Public,
         ThePackage  => Package_ASCII);
      Dict.Packages.PackageASCII := Package_ASCII;

      Add_Common_Characters (To_The_Package => Package_ASCII);

      -- Ada83 Characters whose identiifers differ from those used in Ada95.
      Add_Predefined_Constant (Name        => "Exclam",
                               Asc_Code    => 33,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Quotation",
                               Asc_Code    => 34,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Sharp",
                               Asc_Code    => 35,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Dollar",
                               Asc_Code    => 36,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Percent",
                               Asc_Code    => 37,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Ampersand",
                               Asc_Code    => 38,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Colon",
                               Asc_Code    => 58,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Semicolon",
                               Asc_Code    => 59,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Query",
                               Asc_Code    => 63,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "At_Sign",
                               Asc_Code    => 64,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "L_Bracket",
                               Asc_Code    => 91,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Back_Slash",
                               Asc_Code    => 92,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "R_Bracket",
                               Asc_Code    => 93,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Circumflex",
                               Asc_Code    => 94,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Underline",
                               Asc_Code    => 95,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Grave",
                               Asc_Code    => 96,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "L_Brace",
                               Asc_Code    => 123,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Bar",
                               Asc_Code    => 124,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "R_Brace",
                               Asc_Code    => 125,
                               The_Package => Package_ASCII);
      Add_Predefined_Constant (Name        => "Tilde",
                               Asc_Code    => 126,
                               The_Package => Package_ASCII);
   end Add_Package_ASCII;

   --------------------------------------------------------------------------------

   procedure Add_Package_Ada
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --# derives Dict,
   --#         LexTokenManager.State from *,
   --#                                    LexTokenManager.State;
   is
      Token                                           : LexTokenManager.Lex_String;
      Package_Ada, Package_Characters, Package_Latin1 : Symbol;

      procedure Add_Lower_Half_Control_And_Graphic_Characters
      --# global in     Package_Latin1;
      --#        in out Dict;
      --#        in out LexTokenManager.State;
      --# derives Dict                  from *,
      --#                                    LexTokenManager.State,
      --#                                    Package_Latin1 &
      --#         LexTokenManager.State from *;
      is
      begin
         --ISO 6429 control characters
         Add_Predefined_Constant (Name        => "IS4",
                                  Asc_Code    => 28,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "IS3",
                                  Asc_Code    => 29,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "IS2",
                                  Asc_Code    => 30,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "IS1",
                                  Asc_Code    => 31,
                                  The_Package => Package_Latin1);

         --ISO 646 graphic characters
         Add_Predefined_Constant (Name        => "Space",
                                  Asc_Code    => 32,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Exclamation",
                                  Asc_Code    => 33,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Quotation",
                                  Asc_Code    => 34,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Number_Sign",
                                  Asc_Code    => 35,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Dollar_Sign",
                                  Asc_Code    => 36,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Percent_Sign",
                                  Asc_Code    => 37,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Ampersand",
                                  Asc_Code    => 38,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Apostrophe",
                                  Asc_Code    => 39,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Left_Parenthesis",
                                  Asc_Code    => 40,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Right_Parenthesis",
                                  Asc_Code    => 41,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Asterisk",
                                  Asc_Code    => 42,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Plus_Sign",
                                  Asc_Code    => 43,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Comma",
                                  Asc_Code    => 44,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Hyphen",
                                  Asc_Code    => 45,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Minus_Sign",
                                  Asc_Code    => 45,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Full_Stop",
                                  Asc_Code    => 46,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Solidus",
                                  Asc_Code    => 47,
                                  The_Package => Package_Latin1);

         --Decimal digits '0' through '9 are at positions 48 through 57

         Add_Predefined_Constant (Name        => "Colon",
                                  Asc_Code    => 58,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Semicolon",
                                  Asc_Code    => 59,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Less_Than_Sign",
                                  Asc_Code    => 60,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Equals_Sign",
                                  Asc_Code    => 61,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Greater_Than_Sign",
                                  Asc_Code    => 62,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Question",
                                  Asc_Code    => 63,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Commercial_At",
                                  Asc_Code    => 64,
                                  The_Package => Package_Latin1);

         --Letters 'A' through 'Z' are at positions 65 through 90

         Add_Predefined_Constant (Name        => "Left_Square_Bracket",
                                  Asc_Code    => 91,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Reverse_Solidus",
                                  Asc_Code    => 92,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Right_Square_Bracket",
                                  Asc_Code    => 93,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Circumflex",
                                  Asc_Code    => 94,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Low_Line",
                                  Asc_Code    => 95,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Grave",
                                  Asc_Code    => 96,
                                  The_Package => Package_Latin1);

         Add_Predefined_Constant (Name        => "Left_Curly_Bracket",
                                  Asc_Code    => 123,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Vertical_Line",
                                  Asc_Code    => 124,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Right_Curly_Bracket",
                                  Asc_Code    => 125,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Tilde",
                                  Asc_Code    => 126,
                                  The_Package => Package_Latin1);

      end Add_Lower_Half_Control_And_Graphic_Characters;

      --------------------------------------------------------------------------------

      procedure Add_Upper_Half_Ada95_Characters
      --# global in     Package_Latin1;
      --#        in out Dict;
      --#        in out LexTokenManager.State;
      --# derives Dict                  from *,
      --#                                    LexTokenManager.State,
      --#                                    Package_Latin1 &
      --#         LexTokenManager.State from *;
      is
      begin
         Add_Predefined_Constant (Name        => "Reserved_128",
                                  Asc_Code    => 128,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Reserved_129",
                                  Asc_Code    => 129,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "BPH",
                                  Asc_Code    => 130,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "NBH",
                                  Asc_Code    => 131,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Reserved_132",
                                  Asc_Code    => 132,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "NEL",
                                  Asc_Code    => 133,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "SSA",
                                  Asc_Code    => 134,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "ESA",
                                  Asc_Code    => 135,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "HTS",
                                  Asc_Code    => 136,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "HTJ",
                                  Asc_Code    => 137,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "VTS",
                                  Asc_Code    => 138,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "PLD",
                                  Asc_Code    => 139,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "PLU",
                                  Asc_Code    => 140,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "RI",
                                  Asc_Code    => 141,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "SS2",
                                  Asc_Code    => 142,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "SS3",
                                  Asc_Code    => 143,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "DCS",
                                  Asc_Code    => 144,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "PU1",
                                  Asc_Code    => 145,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "PU2",
                                  Asc_Code    => 146,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "STS",
                                  Asc_Code    => 147,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "CCH",
                                  Asc_Code    => 148,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "MW",
                                  Asc_Code    => 149,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "SPA",
                                  Asc_Code    => 150,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "EPA",
                                  Asc_Code    => 151,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "SOS",
                                  Asc_Code    => 152,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Reserved_153",
                                  Asc_Code    => 153,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "SCI",
                                  Asc_Code    => 154,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "CSI",
                                  Asc_Code    => 155,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "ST",
                                  Asc_Code    => 156,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "OSC",
                                  Asc_Code    => 157,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "PM",
                                  Asc_Code    => 158,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "APC",
                                  Asc_Code    => 159,
                                  The_Package => Package_Latin1);

         --# assert True;

         --other graphic characters
         --character positions 160 .. 175
         Add_Predefined_Constant (Name        => "No_Break_Space",
                                  Asc_Code    => 160,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "NBSP",
                                  Asc_Code    => 160,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Inverted_Exclamation",
                                  Asc_Code    => 161,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Cent_Sign",
                                  Asc_Code    => 162,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Pound_Sign",
                                  Asc_Code    => 163,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Currency_Sign",
                                  Asc_Code    => 164,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Yen_Sign",
                                  Asc_Code    => 165,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Broken_Bar",
                                  Asc_Code    => 166,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Section_Sign",
                                  Asc_Code    => 167,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Diaeresis",
                                  Asc_Code    => 168,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Copyright_Sign",
                                  Asc_Code    => 169,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Feminine_Ordinal_Indicator",
                                  Asc_Code    => 170,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Left_Angle_Quotation",
                                  Asc_Code    => 171,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Not_Sign",
                                  Asc_Code    => 172,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Soft_Hyphen",
                                  Asc_Code    => 173,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Registered_Trade_Mark_Sign",
                                  Asc_Code    => 174,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Macron",
                                  Asc_Code    => 175,
                                  The_Package => Package_Latin1);

         --# assert True;

         --character positions 176 .. 191
         Add_Predefined_Constant (Name        => "Degree_Sign",
                                  Asc_Code    => 176,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Ring_Above",
                                  Asc_Code    => 177,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Plus_Minus_Sign",
                                  Asc_Code    => 177,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Superscript_Two",
                                  Asc_Code    => 178,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Superscript_Three",
                                  Asc_Code    => 179,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Acute",
                                  Asc_Code    => 180,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Micro_Sign",
                                  Asc_Code    => 181,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Pilcrow_Sign",
                                  Asc_Code    => 182,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Paragraph_Sign",
                                  Asc_Code    => 182,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Middle_Dot",
                                  Asc_Code    => 183,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Cedilla",
                                  Asc_Code    => 184,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Superscript_One",
                                  Asc_Code    => 185,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Masculine_Ordinal_Indicator",
                                  Asc_Code    => 186,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Right_Angle_Quotation",
                                  Asc_Code    => 187,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Fraction_One_Quarter",
                                  Asc_Code    => 188,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Fraction_One_Half",
                                  Asc_Code    => 189,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Fraction_Three_Quarters",
                                  Asc_Code    => 190,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Inverted_Question",
                                  Asc_Code    => 191,
                                  The_Package => Package_Latin1);

         --# assert True;

         --character positions 192 .. 207
         Add_Predefined_Constant (Name        => "UC_A_Grave",
                                  Asc_Code    => 192,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_A_Acute",
                                  Asc_Code    => 193,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_A_Circumflex",
                                  Asc_Code    => 194,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_A_Tilde",
                                  Asc_Code    => 195,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_A_Diaeresis",
                                  Asc_Code    => 196,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_A_Ring",
                                  Asc_Code    => 197,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_AE_Diphthong",
                                  Asc_Code    => 198,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_C_Cedilla",
                                  Asc_Code    => 199,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_E_Grave",
                                  Asc_Code    => 200,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_E_Acute",
                                  Asc_Code    => 201,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_E_Circumflex",
                                  Asc_Code    => 202,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_E_Diaeresis",
                                  Asc_Code    => 203,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_I_Grave",
                                  Asc_Code    => 204,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_I_Acute",
                                  Asc_Code    => 205,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_I_Circumflex",
                                  Asc_Code    => 206,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_I_Diaeresis",
                                  Asc_Code    => 207,
                                  The_Package => Package_Latin1);

         --# assert True;

         --character positions 208 .. 223
         Add_Predefined_Constant (Name        => "UC_Icelandic_Eth",
                                  Asc_Code    => 208,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_N_Tilde",
                                  Asc_Code    => 209,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_O_Grave",
                                  Asc_Code    => 210,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_O_Acute",
                                  Asc_Code    => 211,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_O_Circumflex",
                                  Asc_Code    => 212,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_O_Tilde",
                                  Asc_Code    => 213,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_O_Diaeresis",
                                  Asc_Code    => 214,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Multiplication_Sign",
                                  Asc_Code    => 215,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_O_Oblique_Stroke",
                                  Asc_Code    => 216,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_U_Grave",
                                  Asc_Code    => 217,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_U_Acute",
                                  Asc_Code    => 218,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_U_Circumflex",
                                  Asc_Code    => 219,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_U_Diaeresis",
                                  Asc_Code    => 220,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_Y_Acute",
                                  Asc_Code    => 221,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "UC_Icelandic_Thorn",
                                  Asc_Code    => 222,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_German_Sharp_S",
                                  Asc_Code    => 223,
                                  The_Package => Package_Latin1);

         --# assert True;

         --character positions 224 .. 239
         Add_Predefined_Constant (Name        => "LC_A_Grave",
                                  Asc_Code    => 224,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_A_Acute",
                                  Asc_Code    => 225,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_A_Circumflex",
                                  Asc_Code    => 226,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_A_Tilde",
                                  Asc_Code    => 227,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_A_Diaeresis",
                                  Asc_Code    => 228,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_A_Ring",
                                  Asc_Code    => 229,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_AE_Diphthong",
                                  Asc_Code    => 230,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_C_Cedilla",
                                  Asc_Code    => 231,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_E_Grave",
                                  Asc_Code    => 232,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_E_Acute",
                                  Asc_Code    => 233,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_E_Circumflex",
                                  Asc_Code    => 234,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_E_Diaeresis",
                                  Asc_Code    => 235,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_I_Grave",
                                  Asc_Code    => 236,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_I_Acute",
                                  Asc_Code    => 237,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_I_Circumflex",
                                  Asc_Code    => 238,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_I_Diaeresis",
                                  Asc_Code    => 239,
                                  The_Package => Package_Latin1);

         --# assert True;

         --character positions 240 .. 255
         Add_Predefined_Constant (Name        => "LC_Icelandic_Eth",
                                  Asc_Code    => 240,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_N_Tilde",
                                  Asc_Code    => 241,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_O_Grave",
                                  Asc_Code    => 242,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_O_Acute",
                                  Asc_Code    => 243,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_O_Circumflex",
                                  Asc_Code    => 244,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_O_Tilde",
                                  Asc_Code    => 245,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_O_Diaeresis",
                                  Asc_Code    => 246,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "Division_Sign",
                                  Asc_Code    => 247,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_O_Oblique_Stroke",
                                  Asc_Code    => 248,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_U_Grave",
                                  Asc_Code    => 249,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_U_Acute",
                                  Asc_Code    => 250,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_U_Circumflex",
                                  Asc_Code    => 251,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_U_Diaeresis",
                                  Asc_Code    => 252,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_Y_Acute",
                                  Asc_Code    => 253,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_Icelandic_Thorn",
                                  Asc_Code    => 254,
                                  The_Package => Package_Latin1);
         Add_Predefined_Constant (Name        => "LC_Y_Diaeresis",
                                  Asc_Code    => 255,
                                  The_Package => Package_Latin1);
      end Add_Upper_Half_Ada95_Characters;

   begin --Add_Package_Ada

      Insert_Lex_String (Name  => "Ada",
                         Token => Token);
      AddNewPackage
        (Name        => Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Scope       => GlobalScope,
         TheParent   => NullSymbol,
         ChildSort   => Public,
         ThePackage  => Package_Ada);
      Dict.Packages.PackageAda := Package_Ada;
      Insert_Lex_String (Name  => "Characters",
                         Token => Token);
      AddNewPackage
        (Name        => Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Scope       => GlobalScope,
         TheParent   => Package_Ada,
         ChildSort   => Public,
         ThePackage  => Package_Characters);
      Dict.Packages.PackageAdaCharacters := Package_Characters;
      Insert_Lex_String (Name  => "Latin_1",
                         Token => Token);
      AddNewPackage
        (Name        => Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Scope       => GlobalScope,
         TheParent   => Package_Characters,
         ChildSort   => Public,
         ThePackage  => Package_Latin1);
      Dict.Packages.PackageAdaCharactersLatin1 := Package_Latin1;
      --control characters

      Add_Common_Characters (To_The_Package => Package_Latin1);

      Add_Lower_Half_Control_And_Graphic_Characters;

      Add_Upper_Half_Ada95_Characters;

   end Add_Package_Ada;

   --------------------------------------------------------------------------------

   procedure Add_Package_Real_Time
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         SPARK_IO.File_Sys     from *,
   --#                                    Dict,
   --#                                    LexTokenManager.State &
   --#         LexTokenManager.State from *;
   is
      Token, SC_Token, TS_Token                                                                                     :
        LexTokenManager.Lex_String;
      Package_Real_Time, Clock_Time, Time_Type, Time_Span_Type, Seconds_Count, T_Param, SC_Param, TS_Param, Subprog : Symbol;
      Package_Ada                                                                                                   : Symbol;
      Real_Time_Scope                                                                                               : Scopes;
      Arg                                                                                                           : Iterator;
   begin
      Package_Ada := Dict.Packages.PackageAda;
      --
      -- Create the package as a child of package Ada
      --
      AddNewPackage
        (Name        => LexTokenManager.Real_Time_Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Scope       => GlobalScope,
         TheParent   => Package_Ada,
         ChildSort   => Public,
         ThePackage  => Package_Real_Time);
      Dict.Packages.PackageRealTime := Package_Real_Time;
      Real_Time_Scope               := VisibleScope (Package_Real_Time);

      --
      -- Add type definitions for private types Time and Time_Span
      --
      Insert_Lex_String (Name  => "Time",
                         Token => Token);
      AddPrivateType
        (Name           => Token,
         Comp_Unit      => ContextManager.NullUnit,
         Declaration    => Null_Location,
         ThePackage     => Package_Real_Time,
         IsLimited      => False,
         IsTaggedType   => False,
         IsAbstractType => False,
         Extends        => NullSymbol);
      -- Annoyingly, the symbol is not returned so we must look it up
      Time_Type                                          :=
        LookupImmediateScope (Name    => Token,
                              Scope   => Real_Time_Scope,
                              Context => ProgramContext);
      Dict.Types.ThePredefinedTypes (PredefinedTimeType) := Time_Type;
      Insert_Lex_String (Name  => "Time_Span",
                         Token => Token);
      AddPrivateType
        (Name           => Token,
         Comp_Unit      => ContextManager.NullUnit,
         Declaration    => Null_Location,
         ThePackage     => Package_Real_Time,
         IsLimited      => False,
         IsTaggedType   => False,
         IsAbstractType => False,
         Extends        => NullSymbol);
      Time_Span_Type                                         :=
        LookupImmediateScope (Name    => Token,
                              Scope   => Real_Time_Scope,
                              Context => ProgramContext);
      Dict.Types.ThePredefinedTypes (PredefinedTimeSpanType) := Time_Span_Type;

      --
      -- Create annotation "own protected in ClockTime : Time" for external time returned by Clock.
      --
      -- Note that the type-announcement is needed here so that the signature
      -- of the proof function associated with function Clock is well-formed
      -- in FDL.
      Insert_Lex_String (Name  => "ClockTime",
                         Token => Token);
      AddOwnVariable
        (Name              => Token,
         ThePackage        => Package_Real_Time,
         Mode              => InMode,
         IsProtected       => True,
         IsInterruptStream => False,
         Comp_Unit         => ContextManager.NullUnit,
         Reference         => Null_Location,
         Variable          => Clock_Time);

      AddOwnVariableType (OwnVariable   => Clock_Time,
                          TypeMark      => Time_Type,
                          TypeReference => Null_Location);

      --
      -- Add the seven deferred constants. Note that Time_Unit is not supported.
      --
      Insert_Lex_String (Name  => "Time_First",
                         Token => Token);
      AddDeferredConstant
        (Name          => Token,
         TypeMark      => Time_Type,
         TypeReference => Null_Location,
         Comp_Unit     => ContextManager.NullUnit,
         Declaration   => Null_Location,
         ThePackage    => Package_Real_Time);
      Insert_Lex_String (Name  => "Time_Last",
                         Token => Token);
      AddDeferredConstant
        (Name          => Token,
         TypeMark      => Time_Type,
         TypeReference => Null_Location,
         Comp_Unit     => ContextManager.NullUnit,
         Declaration   => Null_Location,
         ThePackage    => Package_Real_Time);
      Insert_Lex_String (Name  => "Time_Span_First",
                         Token => Token);
      AddDeferredConstant
        (Name          => Token,
         TypeMark      => Time_Span_Type,
         TypeReference => Null_Location,
         Comp_Unit     => ContextManager.NullUnit,
         Declaration   => Null_Location,
         ThePackage    => Package_Real_Time);
      Insert_Lex_String (Name  => "Time_Span_Last",
                         Token => Token);
      AddDeferredConstant
        (Name          => Token,
         TypeMark      => Time_Span_Type,
         TypeReference => Null_Location,
         Comp_Unit     => ContextManager.NullUnit,
         Declaration   => Null_Location,
         ThePackage    => Package_Real_Time);
      Insert_Lex_String (Name  => "Time_Span_Zero",
                         Token => Token);
      AddDeferredConstant
        (Name          => Token,
         TypeMark      => Time_Span_Type,
         TypeReference => Null_Location,
         Comp_Unit     => ContextManager.NullUnit,
         Declaration   => Null_Location,
         ThePackage    => Package_Real_Time);
      Insert_Lex_String (Name  => "Time_Span_Unit",
                         Token => Token);
      AddDeferredConstant
        (Name          => Token,
         TypeMark      => Time_Span_Type,
         TypeReference => Null_Location,
         Comp_Unit     => ContextManager.NullUnit,
         Declaration   => Null_Location,
         ThePackage    => Package_Real_Time);
      Insert_Lex_String (Name  => "Tick",
                         Token => Token);
      AddDeferredConstant
        (Name          => Token,
         TypeMark      => Time_Span_Type,
         TypeReference => Null_Location,
         Comp_Unit     => ContextManager.NullUnit,
         Declaration   => Null_Location,
         ThePackage    => Package_Real_Time);
      --
      -- Add function Clock return Time.
      --
      Insert_Lex_String (Name  => "Clock",
                         Token => Token);
      AddSubprogram
        (Name          => Token,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location,
         Scope         => Real_Time_Scope,
         Context       => ProgramContext,
         Subprogram    => Subprog);
      AddReturnType
        (TheFunction   => Subprog,
         TypeMark      => Time_Type,
         Comp_Unit     => ContextManager.NullUnit,
         TypeReference => Null_Location);
      Dict.Subprograms.AdaRealTimeClock := Subprog;
      -- Clock requires annotation "global in ClockTime"
      AddGlobalVariable
        (Abstraction       => IsAbstract,
         Subprogram        => Subprog,
         Variable          => Clock_Time,
         Mode              => InMode,
         PrefixNeeded      => False,
         Comp_Unit         => ContextManager.NullUnit,
         VariableReference => Null_Location);
      --
      -- The arithmetic and relational operators are not in the dictionary.
      -- Functions To_Duration and To_Time_Span are not supported.
      --
      -- Add functions Nanoseconds / Microseconds / Milliseconds. These take a
      -- parameter of type Integer and return Time_Span.
      --
      Insert_Lex_String (Name  => "Nanoseconds",
                         Token => Token);
      AddSubprogram
        (Name          => Token,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location,
         Scope         => Real_Time_Scope,
         Context       => ProgramContext,
         Subprogram    => Subprog);
      Insert_Lex_String (Name  => "NS",
                         Token => Token);
      AddSubprogramParameter
        (Name          => Token,
         Subprogram    => Subprog,
         TypeMark      => GetPredefinedIntegerType,
         TypeReference => Null_Location,
         Mode          => InMode,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location);
      AddReturnType
        (TheFunction   => Subprog,
         TypeMark      => Time_Span_Type,
         Comp_Unit     => ContextManager.NullUnit,
         TypeReference => Null_Location);

      Insert_Lex_String (Name  => "Microseconds",
                         Token => Token);
      AddSubprogram
        (Name          => Token,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location,
         Scope         => Real_Time_Scope,
         Context       => ProgramContext,
         Subprogram    => Subprog);
      Insert_Lex_String (Name  => "US",
                         Token => Token);
      AddSubprogramParameter
        (Name          => Token,
         Subprogram    => Subprog,
         TypeMark      => GetPredefinedIntegerType,
         TypeReference => Null_Location,
         Mode          => InMode,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location);
      AddReturnType
        (TheFunction   => Subprog,
         TypeMark      => Time_Span_Type,
         Comp_Unit     => ContextManager.NullUnit,
         TypeReference => Null_Location);

      Insert_Lex_String (Name  => "Milliseconds",
                         Token => Token);
      AddSubprogram
        (Name          => Token,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location,
         Scope         => Real_Time_Scope,
         Context       => ProgramContext,
         Subprogram    => Subprog);
      Insert_Lex_String (Name  => "MS",
                         Token => Token);
      AddSubprogramParameter
        (Name          => Token,
         Subprogram    => Subprog,
         TypeMark      => GetPredefinedIntegerType,
         TypeReference => Null_Location,
         Mode          => InMode,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location);
      AddReturnType
        (TheFunction   => Subprog,
         TypeMark      => Time_Span_Type,
         Comp_Unit     => ContextManager.NullUnit,
         TypeReference => Null_Location);
      --
      -- Add type Seconds_Count. This is an integer type with implementation-
      -- defined range. The range is left blank at this stage, and may be
      -- overridden by values supplied in the configuration file.
      --
      AddIntegerType
        (Name        => LexTokenManager.Seconds_Count_Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Lower       => LexTokenManager.Null_String,
         Upper       => LexTokenManager.Null_String,
         Scope       => Real_Time_Scope,
         Context     => ProgramContext,
         Type_Symbol => Seconds_Count);
      --
      -- Add procedure Split (T:Time; SC:out Seconds_Count; TS:out Time_Span)
      --
      Insert_Lex_String (Name  => "Split",
                         Token => Token);
      AddSubprogram
        (Name          => Token,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location,
         Scope         => Real_Time_Scope,
         Context       => ProgramContext,
         Subprogram    => Subprog);
      Insert_Lex_String (Name  => "T",
                         Token => Token);
      AddSubprogramParameter
        (Name          => Token,
         Subprogram    => Subprog,
         TypeMark      => Time_Type,
         TypeReference => Null_Location,
         Mode          => InMode,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location);
      Insert_Lex_String (Name  => "SC",
                         Token => SC_Token); -- Remember the token for further use of "SC"
      AddSubprogramParameter
        (Name          => SC_Token,
         Subprogram    => Subprog,
         TypeMark      => Seconds_Count,
         TypeReference => Null_Location,
         Mode          => OutMode,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location);
      Insert_Lex_String (Name  => "TS",
                         Token => TS_Token); -- Remember the token for further use of "TS"
      AddSubprogramParameter
        (Name          => TS_Token,
         Subprogram    => Subprog,
         TypeMark      => Time_Span_Type,
         TypeReference => Null_Location,
         Mode          => OutMode,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location);

      -- Split requires annotation "derives SC, TS from T", so we need the
      -- symbols for the three parameters.
      Arg      := FirstSubprogramParameter (Subprog);
      T_Param  := CurrentSymbol (Arg);
      Arg      := NextSymbol (Arg);
      SC_Param := CurrentSymbol (Arg);
      Arg      := NextSymbol (Arg);
      TS_Param := CurrentSymbol (Arg);

      AddExport
        (Abstraction     => IsAbstract,
         TheProcedure    => Subprog,
         TheExport       => SC_Param,
         ExportReference => Null_Location,
         Annotation      => Null_Location);
      AddExport
        (Abstraction     => IsAbstract,
         TheProcedure    => Subprog,
         TheExport       => TS_Param,
         ExportReference => Null_Location,
         Annotation      => Null_Location);
      AddDependency
        (Abstraction     => IsAbstract,
         Comp_Unit       => ContextManager.NullUnit,
         TheProcedure    => Subprog,
         TheExport       => SC_Param,
         TheImport       => T_Param,
         ImportReference => Null_Location);
      AddDependency
        (Abstraction     => IsAbstract,
         Comp_Unit       => ContextManager.NullUnit,
         TheProcedure    => Subprog,
         TheExport       => TS_Param,
         TheImport       => T_Param,
         ImportReference => Null_Location);
      --
      -- Add function Time_Of (SC: Seconds_Count; TS: Time_Span) return Time
      --
      Insert_Lex_String (Name  => "Time_Of",
                         Token => Token);
      AddSubprogram
        (Name          => Token,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location,
         Scope         => Real_Time_Scope,
         Context       => ProgramContext,
         Subprogram    => Subprog);
      AddSubprogramParameter
        (Name          => SC_Token,
         Subprogram    => Subprog,
         TypeMark      => Seconds_Count,
         TypeReference => Null_Location,
         Mode          => InMode,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location);
      AddSubprogramParameter
        (Name          => TS_Token,
         Subprogram    => Subprog,
         TypeMark      => Time_Span_Type,
         TypeReference => Null_Location,
         Mode          => InMode,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location);
      AddReturnType
        (TheFunction   => Subprog,
         TypeMark      => Time_Type,
         Comp_Unit     => ContextManager.NullUnit,
         TypeReference => Null_Location);

   end Add_Package_Real_Time;

   --------------------------------------------------------------------------------

   procedure Add_Package_Synchronous_Task_Control
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         SPARK_IO.File_Sys     from *,
   --#                                    Dict,
   --#                                    LexTokenManager.State &
   --#         LexTokenManager.State from *;
   is
      Token, S_Token                                                             : LexTokenManager.Lex_String;
      Package_Synchronous_Task_Control, Suspension_Object_Type, Subprog, S_Param : Symbol;
      Package_Ada                                                                : Symbol;
      Synchronous_Task_Control_Scope                                             : Scopes;
      Arg                                                                        : Iterator;
   begin
      Package_Ada := Dict.Packages.PackageAda;
      --
      -- Create the package as a child of package Ada
      --
      AddNewPackage
        (Name        => LexTokenManager.Synchronous_Task_Control_Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Scope       => GlobalScope,
         TheParent   => Package_Ada,
         ChildSort   => Public,
         ThePackage  => Package_Synchronous_Task_Control);
      Dict.Packages.PackageSynchronousTaskControl := Package_Synchronous_Task_Control;
      Synchronous_Task_Control_Scope              := VisibleScope (Package_Synchronous_Task_Control);

      --
      -- Add type definitions for limited private type Suspension_Object
      --
      Insert_Lex_String (Name  => "Suspension_Object",
                         Token => Token);
      AddPrivateType
        (Name           => Token,
         Comp_Unit      => ContextManager.NullUnit,
         Declaration    => Null_Location,
         ThePackage     => Package_Synchronous_Task_Control,
         IsLimited      => True,
         IsTaggedType   => False,
         IsAbstractType => False,
         Extends        => NullSymbol);

      -- Annoyingly, the symbol is not returned so we must look it up
      Suspension_Object_Type :=
        LookupImmediateScope (Name    => Token,
                              Scope   => Synchronous_Task_Control_Scope,
                              Context => ProgramContext);

      -- Store the predefined type for easy lookup
      Dict.Types.ThePredefinedTypes (PredefinedSuspensionObjectType) := Suspension_Object_Type;

      -- Mark the type as Atomic
      SetTypeAtomic (Suspension_Object_Type);

      --
      -- Add procedure Set_True (S: in out Suspension_Object);
      --               --# derives S from ;
      --
      Insert_Lex_String (Name  => "Set_True",
                         Token => Token);
      AddSubprogram
        (Name          => Token,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location,
         Scope         => Synchronous_Task_Control_Scope,
         Context       => ProgramContext,
         Subprogram    => Subprog);
      Insert_Lex_String (Name  => "S",
                         Token => S_Token); -- used in all subroutine definitions for this package.
      AddSubprogramParameter
        (Name          => S_Token,
         Subprogram    => Subprog,
         TypeMark      => Suspension_Object_Type,
         TypeReference => Null_Location,
         Mode          => InOutMode,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location);

      -- The annotation
      Arg     := FirstSubprogramParameter (Subprog);
      S_Param := CurrentSymbol (Arg);

      AddExport
        (Abstraction     => IsAbstract,
         TheProcedure    => Subprog,
         TheExport       => S_Param,
         ExportReference => Null_Location,
         Annotation      => Null_Location);

      --
      -- Add procedure Set_False (S: in out Suspension_Object);
      --               --# derives S from ;
      --
      Insert_Lex_String (Name  => "Set_False",
                         Token => Token);
      AddSubprogram
        (Name          => Token,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location,
         Scope         => Synchronous_Task_Control_Scope,
         Context       => ProgramContext,
         Subprogram    => Subprog);
      AddSubprogramParameter
        (Name          => S_Token,
         Subprogram    => Subprog,
         TypeMark      => Suspension_Object_Type,
         TypeReference => Null_Location,
         Mode          => InOutMode,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location);
      -- The annotation
      Arg     := FirstSubprogramParameter (Subprog);
      S_Param := CurrentSymbol (Arg);

      AddExport
        (Abstraction     => IsAbstract,
         TheProcedure    => Subprog,
         TheExport       => S_Param,
         ExportReference => Null_Location,
         Annotation      => Null_Location);

      --
      -- Add procedure Suspend_Until_True (S: in out Suspension_Object);
      --               --# derives S from ;
      Insert_Lex_String (Name  => "Suspend_Until_True",
                         Token => Token);
      AddSubprogram
        (Name          => Token,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location,
         Scope         => Synchronous_Task_Control_Scope,
         Context       => ProgramContext,
         Subprogram    => Subprog);

      -- Record the symbol for later retrieval by wellformedness checker.
      Dict.Subprograms.STC_SuspendUntilTrue := Subprog;

      AddSubprogramParameter
        (Name          => S_Token,
         Subprogram    => Subprog,
         TypeMark      => Suspension_Object_Type,
         TypeReference => Null_Location,
         Mode          => InOutMode,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location);
      -- The annotations
      Arg     := FirstSubprogramParameter (Subprog);
      S_Param := CurrentSymbol (Arg);

      AddExport
        (Abstraction     => IsAbstract,
         TheProcedure    => Subprog,
         TheExport       => S_Param,
         ExportReference => Null_Location,
         Annotation      => Null_Location);

   end Add_Package_Synchronous_Task_Control;

   --------------------------------------------------------------------------------

   procedure Add_Package_Interrupts
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                LexTokenManager.State;
   is
      New_Symbol       : Symbol;
      Package_Ada      : Symbol;
      Interrupts_Scope : Scopes;
   begin
      Package_Ada := Dict.Packages.PackageAda;
      --
      -- Create the package as a child of package Ada
      --
      AddNewPackage
        (Name        => LexTokenManager.Interrupts_Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Scope       => GlobalScope,
         TheParent   => Package_Ada,
         ChildSort   => Public,
         ThePackage  => New_Symbol);
      Dict.Packages.PackageInterrupts := New_Symbol;
      Interrupts_Scope                := VisibleScope (New_Symbol);

      --
      -- Add type Interrupt_ID. This is an integer type with implementation-
      -- defined range. The range is left blank at this stage, and may be
      -- overridden by values supplied in the configuration file.
      --
      --# accept F, 10, New_Symbol, "Expected ineffective assignment to New_Symbol";
      AddIntegerType
        (Name        => LexTokenManager.Interrupt_ID_Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Lower       => LexTokenManager.Null_String,
         Upper       => LexTokenManager.Null_String,
         Scope       => Interrupts_Scope,
         Context     => ProgramContext,
         Type_Symbol => New_Symbol);
      --# end accept;
      pragma Warnings (Off, New_Symbol);
   end Add_Package_Interrupts;

   --------------------------------------------------------------------------------

   -- This procedure creates a subprogram record with which we can associate globals
   -- and dependencies from the partition annotation
   procedure Add_The_Partition
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         SPARK_IO.File_Sys     from *,
   --#                                    Dict,
   --#                                    LexTokenManager.State &
   --#         LexTokenManager.State from *;

   is
      Token : LexTokenManager.Lex_String;
      Sym   : Symbol;
   begin
      Insert_Lex_String (Name  => "main_program",
                         Token => Token); -- suitable reserved name for the subprog
      AddSubprogram
        (Name          => Token,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location,
         Scope         => GlobalScope,
         Context       => ProgramContext,
         Subprogram    => Sym);
      Dict.ThePartition := Sym;
   end Add_The_Partition;

   --------------------------------------------------------------------------------

   --     procedure AddGenericTypes
   --     --# global in out Dict;
   --     --#        in out LexTokenManager.State;
   --     --# derives Dict,
   --     --#         LexTokenManager.State from *,
   --     --#                                          LexTokenManager.State;
   --     is
   --        Sym : Symbol;

   --        procedure AddGenericType (Name    : in     String;
   --                                  Discrim : in     TypeDiscriminant;
   --                                  Sym     :    out Symbol)
   --        --# global in out Dict;
   --        --#        in out LexTokenManager.State;
   --        --# derives Dict                        from *,
   --        --#                                          LexTokenManager.State,
   --        --#                                          Name,
   --        --#                                          Discrim &
   --        --#         LexTokenManager.State from *,
   --        --#                                          Name &
   --        --#         Sym                         from Dict;
   --        is
   --           Token   : LexTokenManager.Lex_String;
   --           The_Type : Symbol;
   --        begin
   --           InsertLexString (Name, Token);
   --           RawDict.CreateType (Token, The_Type);
   --           RawDict.SetTypeDiscriminant (The_Type, Discrim);
   --           Sym := The_Type;
   --        end AddGenericType;

   --     begin -- AddGenericTypes
   --        AddGenericType ("GenericLimitedType", GenericLimitedType, Sym);
   --        Dict.Types.ThePredefinedGenericTypes (PredefinedGenericLimitedType) := Sym;
   --        AddGenericType ("GenericPrivateType", GenericPrivateType, Sym);
   --        Dict.Types.ThePredefinedGenericTypes (PredefinedGenericPrivateType) := Sym;
   --        AddGenericType ("GenericOrderedDiscreteType", GenericOrderedDiscreteType, Sym);
   --        Dict.Types.ThePredefinedGenericTypes (PredefinedGenericOrderedDiscreteType) := Sym;
   --        AddGenericType ("GenericUnorderedDiscreteType", GenericUnorderedDiscreteType, Sym);
   --        Dict.Types.ThePredefinedGenericTypes (PredefinedGenericUnorderedDiscreteType) := Sym;
   --        AddGenericType ("GenericIntegerType", GenericIntegerType, Sym);
   --        Dict.Types.ThePredefinedGenericTypes (PredefinedGenericIntegerType) := Sym;
   --        AddGenericType ("GenericModularType", GenericModularType, Sym);
   --        Dict.Types.ThePredefinedGenericTypes (PredefinedGenericModularType) := Sym;
   --        AddGenericType ("GenericFloatingPointType", GenericFloatingPointType, Sym);
   --        Dict.Types.ThePredefinedGenericTypes (PredefinedGenericFloatingPointType) := Sym;
   --        AddGenericType ("GenericFixedPointType", GenericFixedPointType, Sym);
   --        Dict.Types.ThePredefinedGenericTypes (PredefinedGenericFixedPointType) := Sym;
   --        AddGenericType ("GenericArrayType", GenericArrayType, Sym);
   --        Dict.Types.ThePredefinedGenericTypes (PredefinedGenericArrayType) := Sym;
   --     end AddGenericTypes;

   --------------------------------------------------------------------------------

   procedure Add_Unchecked_Conversion
   --# global in     CommandLineData.Content;
   --#        in out Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         SPARK_IO.File_Sys     from *,
   --#                                    CommandLineData.Content,
   --#                                    Dict,
   --#                                    LexTokenManager.State &
   --#         LexTokenManager.State from *;
   is
      Subprogram_Token, Source_Token, Target_Token, S_Token                  : LexTokenManager.Lex_String;
      Subprogram_Sym, Source_Type_Mark, Target_Type_Mark, Unused_Declaration : Symbol;
   begin
      -- add the function itself
      Insert_Lex_String (Name  => "Unchecked_Conversion",
                         Token => Subprogram_Token);
      AddSubprogram
        (Name          => Subprogram_Token,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location,
         Scope         => GlobalScope,
         Context       => ProgramContext,
         Subprogram    => Subprogram_Sym);
      -- mark predefined generic unit as having a body
      AddBody (CompilationUnit => Subprogram_Sym,
               Comp_Unit       => ContextManager.NullUnit,
               TheBody         => Null_Location,
               Hidden          => True);
      -- make it generic
      RawDict.SetSubprogramIsGeneric (Subprogram_Sym);
      Dict.Subprograms.UncheckedConversion := Subprogram_Sym;

      -- also add a declaration so that Ada.Unchecked_Conversion can be looked up
      case CommandLineData.Content.Language_Profile is
         when CommandLineData.SPARK83 =>
            null;

         when CommandLineData.SPARK95 | CommandLineData.SPARK2005 =>

            --# accept Flow, 10, Unused_Declaration, "Expected ineffective assignment to Unused_Declaration";
            AddDeclaration
              (Item        => Subprogram_Sym,
               Comp_Unit   => ContextManager.NullUnit,
               Loc         => Null_Location,
               Scope       => VisibleScope (Dict.Packages.PackageAda),
               Context     => ProgramContext,
               Declaration => Unused_Declaration);
            --# end accept;
      end case;

      -- create the generic types
      Insert_Lex_String (Name  => "Source",
                         Token => Source_Token);
      AddTypeDeclaration
        (Name        => Source_Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Scope       => LocalScope (Subprogram_Sym),
         Context     => ProgramContext,
         TypeMark    => Source_Type_Mark);
      RawDict.SetTypeDiscriminant (Source_Type_Mark, GenericLimitedType);
      -- PNA should we set Static false for generic type?

      Insert_Lex_String (Name  => "Target",
                         Token => Target_Token);
      AddTypeDeclaration
        (Name        => Target_Token,
         Comp_Unit   => ContextManager.NullUnit,
         Declaration => Null_Location,
         Scope       => LocalScope (Subprogram_Sym),
         Context     => ProgramContext,
         TypeMark    => Target_Type_Mark);
      RawDict.SetTypeDiscriminant (Target_Type_Mark, GenericLimitedType);
      -- PNA should we set Static false for generic type?

      -- create generic parameters
      AddGenericFormalParameter
        (Name                => LexTokenManager.Null_String,
         Comp_Unit           => ContextManager.NullUnit,
         Declaration         => Null_Location,
         SubprogramOrPackage => Subprogram_Sym,
         TypeMark            => Source_Type_Mark,
         Kind                => GenericTypeParameter);

      AddGenericFormalParameter
        (Name                => LexTokenManager.Null_String,
         Comp_Unit           => ContextManager.NullUnit,
         Declaration         => Null_Location,
         SubprogramOrPackage => Subprogram_Sym,
         TypeMark            => Target_Type_Mark,
         Kind                => GenericTypeParameter);

      -- add subprogram parameter
      Insert_Lex_String (Name  => "S",
                         Token => S_Token);
      AddSubprogramParameter
        (Name          => S_Token,
         Subprogram    => Subprogram_Sym,
         TypeMark      => Source_Type_Mark,
         TypeReference => Null_Location,
         Mode          => InMode,
         Comp_Unit     => ContextManager.NullUnit,
         Specification => Null_Location);

      -- add return type
      AddReturnType
        (TheFunction   => Subprogram_Sym,
         TypeMark      => Target_Type_Mark,
         Comp_Unit     => ContextManager.NullUnit,
         TypeReference => Null_Location);

      --# accept Flow, 33, Unused_Declaration, "Expected to be neither referenced nor exported";
   end Add_Unchecked_Conversion;

   --------------------------------------------------------------------------------

begin -- Initialize
   Assign_Initial_Value;
   Add_Unknown_Type;
   Add_Package_Standard;
   Add_Package_ASCII;

   case CommandLineData.Content.Language_Profile is
      when CommandLineData.SPARK83 =>

         Dict.Packages.PackageAda := NullSymbol;

      when CommandLineData.SPARK95 | CommandLineData.SPARK2005 =>

         Add_Package_Ada;
   end case;

   --AddGenericTypes;
   Add_Unchecked_Conversion;
   if CommandLineData.Ravenscar_Selected then
      Add_Package_Real_Time;
      Add_Package_Synchronous_Task_Control;
      Add_Package_Interrupts;
      Add_The_Partition;
   else
      Dict.Packages.PackageRealTime               := NullSymbol;
      Dict.Packages.PackageSynchronousTaskControl := NullSymbol;
      Dict.Packages.PackageInterrupts             := NullSymbol;
      Dict.ThePartition                           := NullSymbol;
   end if;
   if Write_To_File then
      SPARK_IO.Create (File, 0, "", "", Status);    -- Expected flow err
      if Status = SPARK_IO.Ok then
         Dict.TemporaryFile := File;
      else
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Init,
                                   Msg     => "in Dictionary.Initialize");
         Dict.TemporaryFile := SPARK_IO.Null_File;
      end if;
   else
      Dict.TemporaryFile := SPARK_IO.Null_File;
   end if;
   LexTokenManager.Set_Last_Token;
end Initialize;
