-- $Id: unit.adb 15612 2010-01-15 11:25:51Z Robin Messer $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with CommandLineData;
with EStrings.Not_Spark;
with SparkLex;
with SPARK_IO;
with SPSymbols;
with SparkMakeDebug;
with SparkMakeErrors;
with TokenManager;

use type SPSymbols.SPTerminal;
use type SPARK_IO.File_Status;
use type TokenManager.Token;
use type CommandLineData.LanguageProfiles;

package body Unit
is


   function KindToString (TheUnit : Kind) return EStrings.T
   is
      Result : EStrings.T;
   begin
      case TheUnit is

         when SpecificationUnit =>
            Result := EStrings.Copy_String (Str => "specification");

         when PackageBodyUnit =>
            Result := EStrings.Copy_String (Str => "body");

         when SeparateBodyUnit |
           MainProgramUnit =>
            Result := EStrings.Copy_String (Str => "subunit");

      end case;
      return Result;
   end KindToString;

   -----------------------------------------------------------------------------

   function AreEqual (L, R : Id) return Boolean
   is
   begin
      return EStrings.Eq_String (E_Str1 => L.TheName,
                                 E_Str2 => R.TheName) and then
        L.TheKind = R.TheKind;
   end AreEqual;

   ------------------------------------------------------------------------------

   function LessThan (L, R : Id) return Boolean
   is
      Result : Boolean;
      TheOrder : EStrings.Order_Types;
   begin
      TheOrder := EStrings.Lex_Order (First_Name  => L.TheName,
                                      Second_Name => R.TheName);
      case TheOrder is
         when EStrings.First_One_First =>
            Result := True;
         when EStrings.Second_One_First =>
            Result := False;
         when EStrings.Neither_First =>
            Result := L.TheKind < R.TheKind;
      end case;
      return Result;
   end LessThan;

   ------------------------------------------------------------------------------

   procedure Open (TheFileName : in     EStrings.T;
                   TheFileId   :    out SPARK_IO.File_Type;
                   Success     :    out Boolean)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys,
   --#         Success,
   --#         TheFileId         from SPARK_IO.File_Sys,
   --#                                TheFileName;
   is
      Status : SPARK_IO.File_Status;
   begin
      TheFileId := SPARK_IO.Null_File;
      EStrings.Open (File         => TheFileId,
                     Mode_Of_File => SPARK_IO.In_File,
                     Name_Of_File => TheFileName,
                     Form_Of_File => "",
                     Status       => Status);
      Success := Status = SPARK_IO.Ok;
   end Open;

   -----------------------------------------------------------------------------

   function Prefix (EStr : EStrings.T) return
     EStrings.T
   -- Returns the text up to but not including the final dot.
   -- E.g. Prefix (A.B.C.D) returns A.B.C
   is
      CharFound   : Boolean;
      CharPos     : EStrings.Positions;
      LastPos     : EStrings.Lengths := 0;
      Result      : EStrings.T;
   begin
      -- Look for the last dot.
      loop
         EStrings.Find_Char_After
           (E_Str        => EStr,
            Search_Start => LastPos + 1,
            Search_Char  => '.',
            Char_Found   => CharFound,
            Char_Pos     => CharPos);

         exit when not CharFound;

         if CharFound then
            LastPos := CharPos;
         end if;
      end loop;

      -- Last Pos contains the position of the last dot.

      if LastPos = 0 then
         -- There was not dot so return empty string.
         Result := EStrings.Empty_String;
      else
         -- Return the text up to but not including the last dot.
         Result := EStrings.Section
           (E_Str     => EStr,
            Start_Pos => 1,
            Length    => LastPos - 1);
      end if;
      return Result;
   end Prefix;

   -----------------------------------------------------------------------------

   function ConstructSpecUnitId
     (TheName   : EStrings.T;
      IsPrivate : Boolean) return Id
   is
      TheResult : Id;
   begin
      if EStrings.Is_Empty (E_Str => Prefix (TheName)) then
         TheResult := Id'(TheName => TheName,
                          TheKind => PackageSpecificationUnit);
      elsif IsPrivate then
         TheResult := Id'(TheName => TheName,
                          TheKind => PrivateChildPackageSpecificationUnit);
      else
         TheResult := Id'(TheName => TheName,
                          TheKind => PublicChildPackageSpecificationUnit);
      end if;
      return TheResult;
   end ConstructSpecUnitId;

   -----------------------------------------------------------------------------

   procedure GetUnit (InFile  : in     EStrings.T;
                      TheUnit :    out Object)
   is

      TokenIt           : TokenManager.Iterator;
      CurrentToken      : TokenManager.Token;
      TheFileId         : SPARK_IO.File_Type;
      Status            : SPARK_IO.File_Status;
      Success           : Boolean;
      Done              : Boolean := False;

      ------------------------------------------------------------------

      procedure OutputUnexpectedToken (Expected : in SPSymbols.SPSymbol;
                                       Found    : in TokenManager.Token)
      --# derives null from Expected,
      --#                   Found;
      is
         --# hide OutputUnexpectedToken;
         EStr : EStrings.T;
      begin
         EStr := TokenManager.ToString (Found);
         SparkMakeDebug.ReportCondText
           (Cond      => Expected /= SPSymbols.SPEND,
            TrueText  => "Unexpected token. " &
              "Expecting " & SPSymbols.SPSymbol'Image (Expected) & ", " &
              "but found " & EStrings.Not_Spark.Get_String (E_Str => EStr),
            FalseText => "Unexpected token. " &
              "Found " & EStrings.Not_Spark.Get_String (E_Str => EStr));
      end OutputUnexpectedToken;

      ------------------------------------------------------------------

      procedure AssertNextToken (IsSymbol : in SPSymbols.SPSymbol)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.State;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --#        in out TokenIt;
      --#           out Success;
      --# derives ErrorHandler.ErrorContext,
      --#         LexTokenManager.State,
      --#         SparkLex.CurrLine,
      --#         SPARK_IO.File_Sys,
      --#         TokenIt                   from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TokenIt &
      --#         Success                   from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        IsSymbol,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TokenIt;
      is
         CurrentToken : TokenManager.Token;
      begin
         -- Get the next token.
         TokenManager.Next (It => TokenIt);
         CurrentToken := TokenManager.Current (TokenIt);
         -- Is it what we were expecting?
         Success := CurrentToken.Kind = IsSymbol;
         if not Success then
            OutputUnexpectedToken (Expected => IsSymbol,
                                   Found => CurrentToken);
         end if;
      end AssertNextToken;

      ------------------------------------------------------------------

      procedure GetAndAssertNextToken (IsSymbol     : in     SPSymbols.SPSymbol;
                                       CurrentToken :    out TokenManager.Token)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.State;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --#        in out TokenIt;
      --#           out Success;
      --# derives CurrentToken,
      --#         Success                   from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        IsSymbol,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TokenIt &
      --#         ErrorHandler.ErrorContext,
      --#         LexTokenManager.State,
      --#         SparkLex.CurrLine,
      --#         SPARK_IO.File_Sys,
      --#         TokenIt                   from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TokenIt;
      is
      begin
         AssertNextToken (IsSymbol => IsSymbol);
         if Success then
            CurrentToken := TokenManager.Current (TokenIt);
         else
            CurrentToken := TokenManager.NullToken;
         end if;
      end GetAndAssertNextToken;

      ------------------------------------------------------------------

      procedure ProcessWithClause
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.State;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --#        in out TheUnit;
      --#        in out TokenIt;
      --#           out Success;
      --# derives ErrorHandler.ErrorContext,
      --#         LexTokenManager.State,
      --#         SparkLex.CurrLine,
      --#         SPARK_IO.File_Sys,
      --#         TheUnit,
      --#         TokenIt                   from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TokenIt &
      --#         Success                   from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TokenIt;
      is
         CurrentToken : TokenManager.Token;
      begin
         SparkMakeDebug.ReportText (Text  => "found with clause");
         GetAndAssertNextToken (IsSymbol     => SPSymbols.identifier,
                                CurrentToken => CurrentToken);
         while Success loop
            -- Add this unit to the list of withed units
            StringList.AddToFront (ToList  => TheUnit.TheWithedUnits,
                                   TheItem => CurrentToken.Value);
            -- What's next?
            TokenManager.Next (It => TokenIt);
            CurrentToken := TokenManager.Current (TokenIt);
            -- semicolon marks the end of the with clause.
            exit when CurrentToken.Kind = SPSymbols.semicolon;
            -- otherwise we must have a comma.
            Success := CurrentToken.Kind = SPSymbols.comma;
            if Success then
               GetAndAssertNextToken (IsSymbol     => SPSymbols.identifier,
                                      CurrentToken => CurrentToken);
            else
               OutputUnexpectedToken (Expected => SPSymbols.comma,
                                      Found => CurrentToken);
            end if;
         end loop;
         SparkMakeDebug.ReportList (Text => "The withed units are: ",
                                    List => TheUnit.TheWithedUnits);
      end ProcessWithClause;

      ------------------------------------------------------------------

      procedure ProcessInheritClause
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.State;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --#        in out TheUnit;
      --#        in out TokenIt;
      --#           out Success;
      --# derives ErrorHandler.ErrorContext,
      --#         LexTokenManager.State,
      --#         SparkLex.CurrLine,
      --#         SPARK_IO.File_Sys,
      --#         TheUnit,
      --#         TokenIt                   from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TokenIt &
      --#         Success                   from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TokenIt;
      is
         CurrentToken : TokenManager.Token;
      begin
         SparkMakeDebug.ReportText (Text  => "found inherit clause");
         GetAndAssertNextToken (IsSymbol     => SPSymbols.identifier,
                                CurrentToken => CurrentToken);
         while Success loop
            -- Add this unit to the list of inherited units
            StringList.AddToFront (ToList  => TheUnit.TheInheritedUnits,
                                   TheItem => CurrentToken.Value);
            -- What's next?
            TokenManager.Next (It => TokenIt);
            CurrentToken := TokenManager.Current (TokenIt);
            -- semicolon marks the end of the inherit clause.
            if CurrentToken.Kind = SPSymbols.semicolon then
               AssertNextToken (IsSymbol => SPSymbols.annotation_end);
               exit;
            end if;
            -- otherwise we must have a comma.
            Success := CurrentToken.Kind = SPSymbols.comma;
            if Success then
               GetAndAssertNextToken (IsSymbol     => SPSymbols.identifier,
                                      CurrentToken => CurrentToken);
            else
               OutputUnexpectedToken (Expected => SPSymbols.comma,
                                      Found => CurrentToken);
            end if;
         end loop;
         SparkMakeDebug.ReportList (Text => "The inherited units are: ",
                                    List => TheUnit.TheInheritedUnits);
      end ProcessInheritClause;

      ------------------------------------------------------------------

      procedure ProcessPackageSpecification (IsPrivate : in Boolean)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.State;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --#        in out TheUnit;
      --#        in out TokenIt;
      --#           out Done;
      --#           out Success;
      --# derives Done                      from  &
      --#         ErrorHandler.ErrorContext,
      --#         LexTokenManager.State,
      --#         SparkLex.CurrLine,
      --#         SPARK_IO.File_Sys,
      --#         Success,
      --#         TokenIt                   from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TokenIt &
      --#         TheUnit                   from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        IsPrivate,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TokenIt;
      is
         CurrentToken : TokenManager.Token;
      begin
         SparkMakeDebug.ReportCondText
           (Cond      => IsPrivate,
            TrueText  => "found private package specification",
            FalseText => "found package specification");
         -- Get the package name
         GetAndAssertNextToken (IsSymbol     => SPSymbols.identifier,
                                CurrentToken => CurrentToken);
         if Success then
            SparkMakeDebug.ReportTextEText (Text  => "with name ",
                                            EText => CurrentToken.Value);
            TheUnit.TheId :=
              ConstructSpecUnitId (TheName   => CurrentToken.Value,
                                   IsPrivate => IsPrivate);
         end if;
         Done := True;
      end ProcessPackageSpecification;

      ------------------------------------------------------------------

      procedure ProcessPackageBody
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.State;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --#        in out TheUnit;
      --#        in out TokenIt;
      --#           out Done;
      --#           out Success;
      --# derives Done                      from  &
      --#         ErrorHandler.ErrorContext,
      --#         LexTokenManager.State,
      --#         SparkLex.CurrLine,
      --#         SPARK_IO.File_Sys,
      --#         TheUnit,
      --#         TokenIt                   from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TokenIt &
      --#         Success                   from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TokenIt;
      is
         CurrentToken : TokenManager.Token;
      begin
         SparkMakeDebug.ReportText (Text  => "found package body");
         -- Get the package name.
         GetAndAssertNextToken (IsSymbol     => SPSymbols.identifier,
                                CurrentToken => CurrentToken);
         if Success then
            SparkMakeDebug.ReportTextEText (Text  => "with name ",
                                            EText => CurrentToken.Value);
            TheUnit.TheId := Id'(TheName => CurrentToken.Value,
                                 TheKind => PackageBodyUnit);
         end if;
         Done := True;
      end ProcessPackageBody;

      ------------------------------------------------------------------

      procedure ProcessPackage
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.State;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --#        in out TheUnit;
      --#        in out TokenIt;
      --#           out Done;
      --#           out Success;
      --# derives Done,
      --#         ErrorHandler.ErrorContext,
      --#         LexTokenManager.State,
      --#         SparkLex.CurrLine,
      --#         SPARK_IO.File_Sys,
      --#         Success,
      --#         TokenIt                   from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TokenIt &
      --#         TheUnit                   from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TokenIt;
      is
         CurrentToken : TokenManager.Token;
      begin
         TokenManager.LookAhead (TokenIt);
         CurrentToken := TokenManager.Current (TokenIt);
         -- Is this a package spec or body?
         if CurrentToken.Kind = SPSymbols.RWbody then
            -- Read over the reserved word body
            TokenManager.Next (It => TokenIt);
            ProcessPackageBody;
         else
            ProcessPackageSpecification (IsPrivate => False);
         end if;
      end ProcessPackage;

      ------------------------------------------------------------------

      procedure ProcessPrivateChildSpecification
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in out Done;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.State;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --#        in out TheUnit;
      --#        in out TokenIt;
      --#           out Success;
      --# derives Done,
      --#         ErrorHandler.ErrorContext,
      --#         LexTokenManager.State,
      --#         SparkLex.CurrLine,
      --#         SPARK_IO.File_Sys,
      --#         TheUnit,
      --#         TokenIt                   from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TokenIt &
      --#         Success                   from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TokenIt;
      is
      begin
         SparkMakeDebug.ReportText (Text  => "found reserved word private");
         -- Must be reserved word package
         AssertNextToken (IsSymbol => SPSymbols.RWpackage);
         if Success then
            ProcessPackageSpecification (IsPrivate => True);
         end if;
      end ProcessPrivateChildSpecification;

      ------------------------------------------------------------------

      procedure ProcessSeparate
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.State;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --#        in out TheUnit;
      --#        in out TokenIt;
      --#           out Done;
      --#           out Success;
      --# derives Done                      from  &
      --#         ErrorHandler.ErrorContext,
      --#         LexTokenManager.State,
      --#         SparkLex.CurrLine,
      --#         SPARK_IO.File_Sys,
      --#         TheUnit,
      --#         TokenIt                   from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TokenIt &
      --#         Success                   from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TokenIt;
      is
         CurrentToken : TokenManager.Token;
      begin
         SparkMakeDebug.ReportText (Text => "found separate");
         AssertNextToken (IsSymbol => SPSymbols.left_paren);
         if Success then
            -- Get the parent name.
            GetAndAssertNextToken (IsSymbol     => SPSymbols.identifier,
                                   CurrentToken => CurrentToken);
            if Success then
               SparkMakeDebug.ReportTextEText (Text  => "of enclosing unit ",
                                               EText => CurrentToken.Value);
               TheUnit.TheId.TheKind := SeparateBodyUnit;

               -- The parent
               TheUnit.TheId.TheName := CurrentToken.Value;

               AssertNextToken (IsSymbol => SPSymbols.right_paren);

               if Success then
                  TokenManager.Next (It => TokenIt);
                  CurrentToken := TokenManager.Current (TokenIt);

                  -- If language is SPARK 2005 then allow for an optional overriding_indicator
                  if CommandLineData.Content.LanguageProfile = CommandLineData.SPARK2005 then
                     if CurrentToken.Kind = SPSymbols.RWnot then
                        AssertNextToken (IsSymbol => SPSymbols.RWoverriding);
                        TokenManager.Next (TokenIt);
                        CurrentToken := TokenManager.Current (TokenIt);
                     elsif CurrentToken.Kind = SPSymbols.RWoverriding then
                        TokenManager.Next (TokenIt);
                        CurrentToken := TokenManager.Current (TokenIt);
                     end if;
                  end if;

                  -- Is this a procedure, function, package or task?
                  if CurrentToken.Kind = SPSymbols.RWpackage or
                    CurrentToken.Kind = SPSymbols.RWtask then
                     AssertNextToken (IsSymbol => SPSymbols.RWbody);
                  elsif CurrentToken.Kind /= SPSymbols.RWprocedure and
                    CurrentToken.Kind /= SPSymbols.RWfunction then
                     Success := False;
                     OutputUnexpectedToken (Expected => SPSymbols.SPEND,
                                            Found => CurrentToken);
                  end if;

                  if Success then
                     -- Get the unit name.
                     GetAndAssertNextToken (IsSymbol     => SPSymbols.identifier,
                                            CurrentToken => CurrentToken);
                     if Success then
                        EStrings.Append_String (E_Str => TheUnit.TheId.TheName,
                                                Str   => ".");
                        EStrings.Append_Examiner_String
                          (E_Str1 => TheUnit.TheId.TheName,
                           E_Str2 => CurrentToken.Value);
                        SparkMakeDebug.ReportTextEText (Text  => "Full unit name is ",
                                                        EText => TheUnit.TheId.TheName);
                     end if;
                  end if;
               end if;
            end if;
         end if;
         Done := True;
      end ProcessSeparate;

      ------------------------------------------------------------------

      procedure ProcessMainProgram
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.State;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --#        in out TheUnit;
      --#        in out TokenIt;
      --#           out Done;
      --#           out Success;
      --# derives Done                      from  &
      --#         ErrorHandler.ErrorContext,
      --#         LexTokenManager.State,
      --#         SparkLex.CurrLine,
      --#         SPARK_IO.File_Sys,
      --#         TheUnit,
      --#         TokenIt                   from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TokenIt &
      --#         Success                   from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TokenIt;
      is
         CurrentToken : TokenManager.Token;
      begin
         SparkMakeDebug.ReportText (Text  => "found main program");
         loop
            TokenManager.Next (It => TokenIt);
            CurrentToken := TokenManager.Current (TokenIt);
            exit when CurrentToken.Kind = SPSymbols.SPEND or
              CurrentToken.Kind = SPSymbols.RWprocedure;
         end loop;
         if CurrentToken.Kind = SPSymbols.RWprocedure then
            -- Get the main program name.
            GetAndAssertNextToken (IsSymbol     => SPSymbols.identifier,
                                   CurrentToken => CurrentToken);
            if Success then
               SparkMakeDebug.ReportTextEText (Text  => "main program is ",
                                               EText => CurrentToken.Value);
               TheUnit.TheId := Id'(TheName => CurrentToken.Value,
                                    TheKind => MainProgramUnit);
            end if;
         else
            Success := False;
            OutputUnexpectedToken (Expected => SPSymbols.RWprocedure,
                                   Found => CurrentToken);
         end if;
         Done := True;
      end ProcessMainProgram;

      ------------------------------------------------------------------

      procedure ProcessAnnotation
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in out Done;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.State;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Success;
      --#        in out TheUnit;
      --#        in out TokenIt;
      --# derives Done,
      --#         TheUnit                   from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TokenIt &
      --#         ErrorHandler.ErrorContext,
      --#         LexTokenManager.State,
      --#         SparkLex.CurrLine,
      --#         SPARK_IO.File_Sys,
      --#         Success,
      --#         TokenIt                   from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        Success,
      --#                                        TokenIt;
      is
         CurrentToken : TokenManager.Token;
      begin
         SparkMakeDebug.ReportText (Text  => "found annotation");
         TokenManager.Next (It => TokenIt);
         CurrentToken := TokenManager.Current (TokenIt);
         -- Is this the main program
         if CurrentToken.Kind = SPSymbols.RWmain_program then
            ProcessMainProgram;
         elsif CurrentToken.Kind = SPSymbols.RWinherit then
            ProcessInheritClause;
         else
            -- read until annotation end
            while Success loop
               TokenManager.Next (It => TokenIt);
               CurrentToken := TokenManager.Current (TokenIt);
               exit when CurrentToken.Kind = SPSymbols.annotation_end;
               if CurrentToken.Kind = SPSymbols.SPEND then
                  Success := False;
               end if;
            end loop;
         end if;
      end ProcessAnnotation;

      ------------------------------------------------------------------

   begin
      SparkLex.ClearLineContext;
      TheUnit := NullObject;

      Open (TheFileName => InFile,
            TheFileId   => TheFileId,
            Success     => Success);

      if Success then

         SparkMakeDebug.ReportTextEText (Text  => "Parsing file ",
                                         EText => InFile);

         TokenManager.GetFirstToken (FileId => TheFileId,
                                     It     => TokenIt);

         if TokenManager.IsNull (TokenIt) then
            Success := False;
         else

            while Success and then
              not Done loop

               CurrentToken := TokenManager.Current (TokenIt);

               case CurrentToken.Kind is

                  when SPSymbols.RWwith =>

                     ProcessWithClause;

                  when SPSymbols.RWprivate =>

                     ProcessPrivateChildSpecification;

                  when SPSymbols.RWpackage =>

                     ProcessPackage;

                  when SPSymbols.RWseparate =>

                     ProcessSeparate;

                  when SPSymbols.annotation_start =>

                     ProcessAnnotation;

                  when others  =>

                     -- ignore these tokens
                     null;

               end case;

               if TokenManager.IsNull (TokenIt) then
                  Success := False;
               else
                  TokenManager.Next (It => TokenIt);
               end if;

            end loop;
         end if;

         --# accept Flow, 10, TheFileId, "Closed file handle not used";
         SPARK_IO.Close (File   => TheFileId,
                         Status => Status);
         --# end accept;

         if Status /= SPARK_IO.Ok then
            SparkMakeErrors.Fatal ("Cannot close file");
         end if;
      else
         EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                            E_Str => InFile);
         SparkMakeErrors.Fatal ("Cannot open file");
      end if;
      if Success then
         TheUnit.TheFile := InFile;
      else
         TheUnit := NullObject;
      end if;
   end GetUnit;

   -----------------------------------------------------------------------------

   procedure OutputObject (TheUnit : in Object)
   is
      --# hide OutputObject;
   begin
      SparkMakeDebug.ReportText ("Unit debug:");
      SparkMakeDebug.ReportCondText
           (Cond      => TheUnit = NullObject,
            TrueText  => "Object is null",
            FalseText => "Object details below");
      if TheUnit /= NullObject then
         SparkMakeDebug.ReportTextEText ("File: ", TheUnit.TheFile);
         SparkMakeDebug.ReportTextEText ("Name: ", TheUnit.TheId.TheName);
         SparkMakeDebug.ReportText ("Kind: " & Kind'Image (TheUnit.TheId.TheKind));
         SparkMakeDebug.ReportList (Text => "The withed units: ",
                                    List => TheUnit.TheWithedUnits);
         SparkMakeDebug.ReportList (Text => "The inherited units: ",
                                    List => TheUnit.TheInheritedUnits);
      end if;
   end OutputObject;

   -----------------------------------------------------------------------------

   procedure OutputId (TheUnit : in Id)
   is
      --# hide OutputId;
   begin
      SparkMakeDebug.ReportText ("Unit debug:");
      SparkMakeDebug.ReportCondText
           (Cond      => TheUnit = NullId,
            TrueText  => "Id is null",
            FalseText => "Id details below");
      if TheUnit /= NullId then
         SparkMakeDebug.ReportTextEText ("Name: ", TheUnit.TheName);
         SparkMakeDebug.ReportText ("Kind: " & Kind'Image (TheUnit.TheKind));
      end if;
   end OutputId;

end Unit;
