-- $Id: sparkmakecommandline.adb 16522 2010-03-23 10:17:03Z Robin Messer $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with CommandLine;
with SPARK_IO;
with Directory_Operations;
with RegularExpression;
with SparkMakeDebug;
with SparkMakeErrors;
with SystemErrors;
with CommandLineData;
with FileSystem;
with Version;

package body SparkMakeCommandLine
--# own State is RootFile,
--#              Duplicates,
--#              Debug,
--#              TheIncRegExps,
--#              TheExcRegExps,
--#              TheDirectories,
--#              MetaFile,
--#              IndexFile,
--#              NoMeta,
--#              NoIndex,
--#              AnnoCharSpecified,
--#              Language_Found,
--#              Path;
is

   Path       : PathType;
   RootFile   : EStrings.T;
   MetaFile   : EStrings.T;
   IndexFile  : EStrings.T;
   Debug      : Boolean;
   Duplicates : Boolean;

   NoMeta     : Boolean;
   NoIndex    : Boolean;

   AnnoCharSpecified : Boolean;
   Language_Found    : Boolean;

   TheDirectories : StringList.Object;
   TheIncRegExps  : StringList.Object;
   TheExcRegExps  : StringList.Object;

   ------------------------------------------------------------------------------

   function NormaliseRootFileName (N : EStrings.T)
                                  return EStrings.T
   --
   -- Ensures any relative path is fully expanded.
   is
      Result  : EStrings.T;
   begin
      Result := Directory_Operations.NormalizePathName
        (Name      => N,
         Directory => Directory_Operations.CurrentDirectory);
      return Result;
   end NormaliseRootFileName;

   ------------------------------------------------------------------------------

   function NormaliseDirName (N : EStrings.T)
                             return EStrings.T
   --
   -- Ensures any relative path is fully expanded and that
   -- there is a directory separator at the end.
   is
      Result  : EStrings.T;
      Success : Boolean;
   begin
      Result := NormaliseRootFileName (N);
      if EStrings.Get_Length (E_Str => Result) > 0 and then
        EStrings.Get_Element (E_Str => Result,
                              Pos   => EStrings.Get_Length (E_Str => Result)) /=
        Directory_Operations.DirectorySeparator then
         EStrings.Append_Char (E_Str   => Result,
                               Ch      => Directory_Operations.DirectorySeparator,
                               Success => Success);
         if not Success then
            Result := EStrings.Empty_String;
         end if;
      else
         Result := EStrings.Empty_String;
      end if;
      return Result;
   end NormaliseDirName;

   ------------------------------------------------------------------------------

   function RegExp (ForFile : in EStrings.T) return
     EStrings.T
   --
   -- Returns a regulat expression that will exactly match the given file.
   is
      Result  : EStrings.T;
      Char    : Character;
      Success : Boolean := True;
      Escape : constant Character := '\';
   begin
      Result := EStrings.Empty_String;
      for I in EStrings.Positions range 1 .. EStrings.Get_Length (E_Str => ForFile) loop
         Char := EStrings.Get_Element (E_Str => ForFile,
                                       Pos   => I);
         if Char = '\' or else
           Char = '(' or else
           Char = ')' or else
           Char = '[' or else
           Char = ']' or else
           Char = '.' or else
           Char = '*' or else
           Char = '+' or else
           Char = '?' or else
           Char = '^' then
            -- We must escape these characters
            EStrings.Append_Char (E_Str   => Result,
                                  Ch      => Escape,
                                  Success => Success);
         end if;
         exit when not Success;
         EStrings.Append_Char (E_Str   => Result,
                               Ch      => Char,
                               Success => Success);
         exit when not Success;
      end loop;
      if not Success then
         Result := EStrings.Empty_String;
      end if;
      return Result;
   end RegExp;

   ------------------------------------------------------------------------------

   procedure ReportUsage
   --# global in out SPARK_IO.FILE_SYS;
   --# derives SPARK_IO.FILE_SYS from *;
   --
   -- Outputs the usage to the user.
   is
      --# hide ReportUsage;
   begin
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "Usage: sparkmake [option] [rootfile]",
                         0);
      SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "Options - all may be abbreviated to the shortest unique prefix",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "Options - all input options may be repeated as necessary",
                         0);
      SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "option         = input_option | output_option | behaviour_option | help_option",
                         0);
      SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "input_option   = dir_option | include_option | exclude_option",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "dir_option     = " &
                           "-" &
                           "directory=dirname - Look in and under dirname as well as cwd.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                      Default: none",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "include_option = " &
                           "-" &
                           "include=reg_exp   - Only include files if full path matches.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                      Default: *\.ad[bs]",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "exclude_option = " &
                           "-" &
                           "exclude=reg_exp   - Exclude files if full path matches.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                      Default: none",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "reg_exp        = term",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "term           = char",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "term           = elmt elmt ...     -- concatenation",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "term           = *                 -- any string of 0 or more characters",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "term           = ?                 -- matches any character",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "term           = [char char ...]   -- matches any character listed",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "term           = [char - char]     -- matches any character in given range",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "elmt           = nchr              -- matches given character",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "elmt           = [nchr nchr ...]   -- matches any character listed",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "elmt           = [^ nchr nchr ...] -- matches any character not listed",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "elmt           = [char - char]     -- matches chars in given range",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "elmt           = .                 -- matches any single character",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "elmt           = ( regexp )        -- parens used for grouping",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "char           = any character, including special characters",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "nchr           = any character except \()[].*+?^ or \char to match char",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "The characters '{' and '}' are NOT allowed to appear in any regular expression",
                         0);
      SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "behaviour_option = duplicates_option | annotation_option | language_option",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "duplicates_option = " &
                           "-" &
                           "duplicates_are_errors - Fail if duplicate units are found.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                             Default: false",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "annotation_option = " &
                           "-" &
                           "annotation_character  - Specify annotation character.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                             Default: #",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "language_option = " &
                           "-" &
                           "language=83 | 95 | 2005 - Specify language profile.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                             Default: 95",
                         0);
      SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "output_option  = index_option | meta_option | noindex_option | nometa_option |",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                 path_option",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "index_option   = " &
                           "-" &
                           "index=file-spec          - The index file.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                             Default rootfile.idx",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "meta_option    = " &
                           "-" &
                           "meta=file-spec           - The meta file.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                             Default rootfile.smf",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "noindex_option = " &
                           "-" &
                           "noindexfile              - Suppress generation of index file.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                             Default: false",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "nometa_option  = " &
                           "-" &
                           "nometafile               - Suppress generation of meta file.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                             Default: false",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "path_option    = " &
                           "-" &
                           "path=full | relative     - Produce relative or full pathnames.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                             Default: full",
                         0);
      SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "help_option    = helper_option | version_option",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "helper_option  = " &
                           "-" &
                           "help    - print off help information.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "version_option = " &
                           "-" &
                           "version - print off version information.",
                         0);
      SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "rootfile       = file-spec                 - The file to make.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                             Default: Produce index and metafile",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                             for analysis of all files in and",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                             under current directory.",
                         0);
      SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output, Version.Toolset_Support_Line1, 0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output, Version.Toolset_Support_Line2, 0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output, Version.Toolset_Support_Line3, 0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output, Version.Toolset_Support_Line4, 0);

   end ReportUsage;

   ------------------------------------------------------------------------------

   procedure ReportVersion
   --# global in out SPARK_IO.FILE_SYS;
   --# derives SPARK_IO.FILE_SYS from *;
   --
   -- Outputs the usage to the user.
   is
      --# hide ReportVersion;
   begin
      SPARK_IO.Put_Line
        (SPARK_IO.Standard_Output,
         "SPARKMake " & Version.Toolset_Banner_Line, 0);
      SPARK_IO.Put_Line
        (SPARK_IO.Standard_Output,
         Version.Toolset_Copyright, 0);
   end ReportVersion;

   ------------------------------------------------------------------------------

   procedure ProcessRootFileArgument (Arg     : in     EStrings.T;
                                      Success :    out Boolean)
   --# global in     Debug;
   --#        in out SPARK_IO.FILE_SYS;
   --#           out RootFile;
   --# derives RootFile,
   --#         Success           from Arg &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                Arg &
   --#         null              from Debug;
   is
   begin
      -- Expand the root file to its full path
      RootFile := NormaliseRootFileName (Arg);

      if Debug then
         SparkMakeDebug.ReportTextEText ("Found root file argument: ", Arg);
         SparkMakeDebug.ReportTextEText ("Expanded to: ", RootFile);
      end if;

      -- Does the file exist?
      if Directory_Operations.IsFile (RootFile) then
         Success := True;
      else
         Success := False;
         SparkMakeErrors.Report (TheFault => SparkMakeErrors.CannotFindFile,
                                 EStr1    => RootFile,
                                 EStr2    => EStrings.Empty_String,
                                 EStr3    => EStrings.Empty_String);
      end if;
   end ProcessRootFileArgument;

   ------------------------------------------------------------------------------

   procedure ProcessIncludeSwitch (Switch  : in     EStrings.T;
                                   Arg     : in     EStrings.T;
                                   Success :    out Boolean)
   --# global in     Debug;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out TheIncRegExps;
   --# derives SPARK_IO.FILE_SYS from *,
   --#                                Arg,
   --#                                Switch &
   --#         Success           from Arg &
   --#         TheIncRegExps     from *,
   --#                                Arg &
   --#         null              from Debug;
   is
   begin
      if Debug then
         SparkMakeDebug.ReportTextEText ("Found include switch: ", Arg);
      end if;

      if EStrings.Is_Empty (E_Str => Arg) or else
        RegularExpression.IsNull
        (RegularExpression.Create (Arg)) then
         Success := False;
         SparkMakeErrors.Report (TheFault => SparkMakeErrors.InvalidArgument,
                                 EStr1    => Arg,
                                 EStr2    => Switch,
                                 EStr3    => EStrings.Empty_String);
      else
         Success := True;
         StringList.AddToFront (ToList  => TheIncRegExps,
                                TheItem => Arg);
      end if;
   end ProcessIncludeSwitch;

   ------------------------------------------------------------------------------

   procedure ProcessExcludeSwitch (Switch  : in     EStrings.T;
                                   Arg     : in     EStrings.T;
                                   Success :    out Boolean)
   --# global in     Debug;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out TheExcRegExps;
   --# derives SPARK_IO.FILE_SYS from *,
   --#                                Arg,
   --#                                Switch &
   --#         Success           from Arg &
   --#         TheExcRegExps     from *,
   --#                                Arg &
   --#         null              from Debug;
   is
   begin
      if Debug then
         SparkMakeDebug.ReportTextEText ("Found exclude switch: ", Arg);
      end if;

      if EStrings.Is_Empty (E_Str => Arg) or else
        RegularExpression.IsNull
        (RegularExpression.Create (Arg)) then
         Success := False;
         SparkMakeErrors.Report (TheFault => SparkMakeErrors.InvalidArgument,
                                 EStr1    => Arg,
                                 EStr2    => Switch,
                                 EStr3    => EStrings.Empty_String);
      else
         Success := True;
         StringList.AddToFront (ToList  => TheExcRegExps,
                                TheItem => Arg);
      end if;
   end ProcessExcludeSwitch;

   ------------------------------------------------------------------------------

   procedure ProcessDirectorySwitch (Switch  : in     EStrings.T;
                                     Arg     : in     EStrings.T;
                                     Success :    out Boolean)
   --# global in     Debug;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out TheDirectories;
   --# derives SPARK_IO.FILE_SYS from *,
   --#                                Arg,
   --#                                Switch &
   --#         Success           from Arg &
   --#         TheDirectories    from *,
   --#                                Arg &
   --#         null              from Debug;
   is
      TheDir : EStrings.T;
   begin
      TheDir := NormaliseDirName (Arg);

      if Debug then
         SparkMakeDebug.ReportTextEText ("Found directory switch: ", Arg);
         SparkMakeDebug.ReportTextEText ("Normalised to: ", TheDir);
      end if;

      if Directory_Operations.IsDirectory (TheDir) then
         Success := True;
         -- the current directory is used by default so don't add it again
         if not EStrings.Eq_String (E_Str1 => TheDir,
                                    E_Str2 => Directory_Operations.CurrentDirectory) then
            StringList.AddToBack (ToList  => TheDirectories,
                                  TheItem => TheDir);
         end if;
      else
         Success := False;
         SparkMakeErrors.Report (TheFault => SparkMakeErrors.InvalidArgument,
                                 EStr1    => TheDir,
                                 EStr2    => Switch,
                                 EStr3    => EStrings.Empty_String);
      end if;
   end ProcessDirectorySwitch;

   ------------------------------------------------------------------------------

   procedure ProcessIndexSwitch (Switch  : in     EStrings.T;
                                 Arg     : in     EStrings.T;
                                 Success :    out Boolean)
   --# global in     Debug;
   --#        in out IndexFile;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives IndexFile         from *,
   --#                                Arg &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                IndexFile,
   --#                                Switch &
   --#         Success           from IndexFile &
   --#         null              from Debug;
   is
   begin
      if Debug then
         SparkMakeDebug.ReportTextEText ("Found index switch: ", Arg);
      end if;

      if EStrings.Is_Empty (E_Str => IndexFile) then
         Success := True;
         IndexFile := Arg;
         FileSystem.CheckExtension
           (IndexFile,
            EStrings.Copy_String (Str => CommandLineData.Default_Index_Extension));

      else
         Success := False;
         SparkMakeErrors.Report (TheFault => SparkMakeErrors.DuplicateSwitch,
                                 EStr1    => Switch,
                                 EStr2    => EStrings.Empty_String,
                                 EStr3    => EStrings.Empty_String);
      end if;
   end ProcessIndexSwitch;

   ------------------------------------------------------------------------------

   procedure ProcessMetaSwitch (Switch  : in     EStrings.T;
                                Arg     : in     EStrings.T;
                                Success :    out Boolean)
   --# global in     Debug;
   --#        in out MetaFile;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives MetaFile          from *,
   --#                                Arg &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                MetaFile,
   --#                                Switch &
   --#         Success           from MetaFile &
   --#         null              from Debug;
   is
   begin
      if Debug then
         SparkMakeDebug.ReportTextEText ("Found meta switch: ", Arg);
      end if;

      if EStrings.Is_Empty (E_Str => MetaFile) then
         Success := True;
         MetaFile := Arg;
         FileSystem.CheckExtension
           (MetaFile,
            EStrings.Copy_String (CommandLineData.Meta_File_Extension));
      else
         -- duplicate meta file switch
         Success := False;
         SparkMakeErrors.Report (TheFault => SparkMakeErrors.DuplicateSwitch,
                                 EStr1    => Switch,
                                 EStr2    => EStrings.Empty_String,
                                 EStr3    => EStrings.Empty_String);
      end if;
   end ProcessMetaSwitch;

   ------------------------------------------------------------------------------

   procedure ProcessPathSwitch (Switch  : in     EStrings.T;
                                Arg     : in     EStrings.T;
                                Success :    out Boolean)
   --# global in     Debug;
   --#        in out Path;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Path,
   --#         Success           from Arg,
   --#                                Path &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                Arg,
   --#                                Path,
   --#                                Switch &
   --#         null              from Debug;
   is
   begin
      if Debug then
         SparkMakeDebug.ReportTextEText ("Found path switch: ", Arg);
      end if;

      if Path = Undefined then

         if EStrings.Eq1_String (E_Str => Arg,
                                 Str   => "full") or else
           EStrings.Eq1_String (E_Str => Arg,
                                Str   => "ful") or else
           EStrings.Eq1_String (E_Str => Arg,
                                Str   => "fu") or else
           EStrings.Eq1_String (E_Str => Arg,
                                Str   => "f") then
            Success := True;
            Path := Full;
         elsif EStrings.Eq1_String (E_Str => Arg,
                                    Str   => "relative") or else
           EStrings.Eq1_String (E_Str => Arg,
                                Str   => "relativ") or else
           EStrings.Eq1_String (E_Str => Arg,
                                Str   => "relati") or else
           EStrings.Eq1_String (E_Str => Arg,
                                Str   => "relat") or else
           EStrings.Eq1_String (E_Str => Arg,
                                Str   => "rela") or else
           EStrings.Eq1_String (E_Str => Arg,
                                Str   => "rel") or else
           EStrings.Eq1_String (E_Str => Arg,
                                Str   => "re") or else
           EStrings.Eq1_String (E_Str => Arg,
                                Str   => "r")  then
            Success := True;
            Path := Relative;
         else
            Success := False;
            SparkMakeErrors.Report (TheFault => SparkMakeErrors.InvalidArgument,
                                    EStr1    => Arg,
                                    EStr2    => Switch,
                                    EStr3    => EStrings.Empty_String);
         end if;
      else
         -- duplicate path switch
         Success := False;
         SparkMakeErrors.Report (TheFault => SparkMakeErrors.DuplicateSwitch,
                                 EStr1    => Switch,
                                 EStr2    => EStrings.Empty_String,
                                 EStr3    => EStrings.Empty_String);
      end if;
   end ProcessPathSwitch;

   ------------------------------------------------------------------------------

   procedure Process_Language_Switch (Switch  : in     EStrings.T;
                                      Arg     : in     EStrings.T;
                                      Success :    out Boolean)
   --# global in     Debug;
   --#        in out Language_Found;
   --#        in out CommandLineData.Content;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Language_Found,
   --#         Success           from Arg,
   --#                                Language_Found &
   --#         CommandLineData.Content from *,
   --#                                      Arg,
   --#                                      Language_Found &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                Arg,
   --#                                Language_Found,
   --#                                Switch &
   --#         null              from Debug;
   is
   begin
      if Debug then
         SparkMakeDebug.ReportTextEText ("Found language switch: ", Arg);
      end if;

      if not Language_Found then

         if EStrings.Eq1_String (E_Str => Arg,
                                 Str   => "83") or else
           EStrings.Eq1_String (E_Str => Arg,
                                Str   => "8") then
            Success := True;
            Language_Found := True;
            CommandLineData.Content.LanguageProfile := CommandLineData.SPARK83;
         elsif EStrings.Eq1_String (E_Str => Arg,
                                    Str   => "95") or else
           EStrings.Eq1_String (E_Str => Arg,
                                Str   => "9") then
            Success := True;
            Language_Found := True;
            CommandLineData.Content.LanguageProfile := CommandLineData.SPARK95;
         elsif EStrings.Eq1_String (E_Str => Arg,
                                    Str   => "2005") or else
           EStrings.Eq1_String (E_Str => Arg,
                                Str   => "200") or else
           EStrings.Eq1_String (E_Str => Arg,
                                Str   => "20") or else
           EStrings.Eq1_String (E_Str => Arg,
                                Str   => "2")  then
            Success := True;
            Language_Found := True;
            CommandLineData.Content.LanguageProfile := CommandLineData.SPARK2005;
         else
            Success := False;
            SparkMakeErrors.Report (TheFault => SparkMakeErrors.InvalidArgument,
                                    EStr1    => Arg,
                                    EStr2    => Switch,
                                    EStr3    => EStrings.Empty_String);
         end if;
      else
         -- duplicate language switch
         Success := False;
         SparkMakeErrors.Report (TheFault => SparkMakeErrors.DuplicateSwitch,
                                 EStr1    => Switch,
                                 EStr2    => EStrings.Empty_String,
                                 EStr3    => EStrings.Empty_String);
      end if;
   end Process_Language_Switch;

   ------------------------------------------------------------------------------

   procedure ProcessAnnoChar (Switch  : in     EStrings.T;
                              Arg     : in     EStrings.T;
                              Success :    out Boolean)
   --# global in     Debug;
   --#        in out AnnoCharSpecified;
   --#        in out CommandLineData.Content;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives AnnoCharSpecified,
   --#         Success                 from AnnoCharSpecified,
   --#                                      Arg &
   --#         CommandLineData.Content from *,
   --#                                      AnnoCharSpecified,
   --#                                      Arg &
   --#         SPARK_IO.FILE_SYS       from *,
   --#                                      AnnoCharSpecified,
   --#                                      Arg,
   --#                                      Switch &
   --#         null                    from Debug;
   is
   begin

      if Debug then
         SparkMakeDebug.ReportTextEText ("Found annotation character argument: ", Arg);
      end if;

      if not AnnoCharSpecified  then

         if EStrings.Get_Length (E_Str => Arg) = 1 then

            Success := True;
            AnnoCharSpecified := True;
            -- set AnnoChar in the Examiner
            CommandLineData.Content.AnnoChar :=
              EStrings.Get_Element (E_Str => Arg,
                                    Pos   => 1);  -- expect warning here

         else

            Success := False;
            SparkMakeErrors.Report (TheFault => SparkMakeErrors.InvalidArgument,
                                    EStr1    => Arg,
                                    EStr2    => Switch,
                                    EStr3    => EStrings.Empty_String);
         end if;
      else
         -- duplicate annochar switch
         Success := False;
         SparkMakeErrors.Report (TheFault => SparkMakeErrors.DuplicateSwitch,
                                 EStr1    => Switch,
                                 EStr2    => EStrings.Empty_String,
                                 EStr3    => EStrings.Empty_String);
      end if;
   end ProcessAnnoChar;

   ------------------------------------------------------------------------------

   procedure Process (Success        : out Boolean;
                      HelpOrVerFound : out Boolean)
   --# global in     CommandLine.State;
   --#        in out CommandLineData.Content;
   --#        in out SPARK_IO.FILE_SYS;
   --#           out AnnoCharSpecified;
   --#           out Debug;
   --#           out Duplicates;
   --#           out IndexFile;
   --#           out MetaFile;
   --#           out NoIndex;
   --#           out NoMeta;
   --#           out Path;
   --#           out Language_Found;
   --#           out RootFile;
   --#           out TheDirectories;
   --#           out TheExcRegExps;
   --#           out TheIncRegExps;
   --# derives AnnoCharSpecified,
   --#         Debug,
   --#         Duplicates,
   --#         HelpOrVerFound,
   --#         IndexFile,
   --#         MetaFile,
   --#         NoIndex,
   --#         NoMeta,
   --#         Path,
   --#         Language_Found,
   --#         RootFile,
   --#         Success,
   --#         TheDirectories,
   --#         TheExcRegExps,
   --#         TheIncRegExps           from CommandLine.State &
   --#         CommandLineData.Content,
   --#         SPARK_IO.FILE_SYS       from *,
   --#                                      CommandLine.State;
   is
      Switch            : EStrings.T;
      Argument          : EStrings.T;
      TheRegExp         : EStrings.T;
      SwitchOrArgFound  : Boolean;
      Done              : Boolean;
      OptionOk          : Boolean;
   begin
      Path              := Undefined;
      RootFile          := EStrings.Empty_String;
      MetaFile          := EStrings.Empty_String;
      IndexFile         := EStrings.Empty_String;
      AnnoCharSpecified := False;
      Debug             := False;
      Duplicates        := False;
      TheIncRegExps     := StringList.NullObject;
      TheExcRegExps     := StringList.NullObject;
      Success           := True;
      Done              := False;
      HelpOrVerFound    := False;
      NoIndex           := False;
      NoMeta            := False;

      -- Default language profile is SPARK95
      Language_Found    := False;
      CommandLineData.Content.LanguageProfile := CommandLineData.SPARK95;

      -- Always allow FDL reserved words as identifiers. Leave it to the Examiner
      -- to reject them later if required.
      CommandLineData.Content.FDLreserved := False;

      -- The current directory is always assumed
      TheDirectories := StringList.NullObject;
      StringList.AddToFront (ToList  => TheDirectories,
                             TheItem => Directory_Operations.CurrentDirectory);

      -- Setup the command line
      CommandLine.Setup;

      -- Read options
      while not Done and not HelpOrVerFound loop

         CommandLine.Read (Switch => Switch,
                           Argument => Argument,
                           Success => SwitchOrArgFound);

         if SwitchOrArgFound then

            if EStrings.Is_Empty (E_Str => Switch) then
               -- ARGUMENT: root file
               ProcessRootFileArgument (Arg => Argument,
                                        Success => OptionOk);
            elsif EStrings.Eq1_String (E_Str => Switch,
                                       Str   => "help") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "hel") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "he") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "h") then
               -- SWITCH: help
               ReportUsage;
               HelpOrVerFound := True;
               OptionOk := True;
            elsif EStrings.Eq1_String (E_Str => Switch,
                                       Str   => "version") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "versio") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "versi") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "vers") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "ver") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "ve") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "v") then
               -- SWITCH: version
               ReportVersion;
               HelpOrVerFound := True;
               OptionOk := True;
            elsif EStrings.Eq1_String (E_Str => Switch,
                                       Str   => "language") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "languag") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "langua") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "langu") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "lang") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "lan") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "la") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "l") then
               -- SWITCH: language
               Process_Language_Switch (Switch  => Switch,
                                        Arg     => Argument,
                                        Success => OptionOk);
            elsif EStrings.Eq1_String (E_Str => Switch,
                                       Str   => "path") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "pat") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "pa") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "p") then
               -- SWITCH: path
               ProcessPathSwitch (Switch  => Switch,
                                  Arg     => Argument,
                                  Success => OptionOk);
            elsif EStrings.Eq1_String (E_Str => Switch,
                                       Str   => "directory") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "director") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "directo") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "direct") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "direc") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "dire") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "dir") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "di") then
               -- SWITCH: directory
               ProcessDirectorySwitch (Switch  => Switch,
                                       Arg     => Argument,
                                       Success => OptionOk);
            elsif EStrings.Eq1_String (E_Str => Switch,
                                       Str   => "include") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "includ") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "inclu") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "incl") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "inc") then
               -- SWITCH: include
               ProcessIncludeSwitch (Switch  => Switch,
                                     Arg     => Argument,
                                     Success => OptionOk);
            elsif EStrings.Eq1_String (E_Str => Switch,
                                       Str   => "exclude") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "exclud") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "exclu") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "excl") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "exc") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "ex") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "e") then
               -- SWITCH: exclude
               ProcessExcludeSwitch (Switch  => Switch,
                                     Arg     => Argument,
                                     Success => OptionOk);
            elsif EStrings.Eq1_String (E_Str => Switch,
                                       Str   => "meta") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "met") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "me") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "m") then
               -- SWITCH: meta
               ProcessMetaSwitch (Switch  => Switch,
                                  Arg     => Argument,
                                  Success => OptionOk);
            elsif EStrings.Eq1_String (E_Str => Switch,
                                       Str   => "index") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "inde") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "ind") then
               -- SWITCH: index
               ProcessIndexSwitch (Switch  => Switch,
                                   Arg     => Argument,
                                   Success => OptionOk);
            elsif EStrings.Eq1_String (E_Str => Switch,
                                       Str   => "noindexfile") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "noindexfil") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "noindexfi") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "noindexf") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "noindex") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "noinde") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "noind") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "noin") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "noi") then
               -- SWITCH: noindexfile
               NoIndex := True;
               OptionOk := True;
               SparkMakeDebug.ReportText ("Found noindexfile switch");
            elsif EStrings.Eq1_String (E_Str => Switch,
                                       Str   => "nometafile") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "nometafil") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "nometafi") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "nometaf") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "nometa") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "nomet") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "nome") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "nom") then
               -- SWITCH: nometafile
               NoMeta := True;
               OptionOk := True;
               SparkMakeDebug.ReportText ("Found nometafile switch");

            elsif EStrings.Eq1_String (E_Str => Switch,
                                       Str   => "debug") then
               -- SWITCH: debug
               Debug := True;
               OptionOk := True;
               SparkMakeDebug.ReportText ("Found debug switch");
            elsif EStrings.Eq1_String (E_Str => Switch,
                                       Str   => "duplicates_are_errors") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "duplicates_are_error") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "duplicates_are_erro") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "duplicates_are_err") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "duplicates_are_er") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "duplicates_are_e") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "duplicates_are_") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "duplicates_are") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "duplicates_ar") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "duplicates_a") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "duplicates_") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "duplicates") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "duplicate") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "duplicat") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "duplica") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "duplic") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "dupli") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "dupl") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "dup") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "du") then
               -- SWITCH: duplicates
               Duplicates := True;
               OptionOk := True;
               SparkMakeDebug.ReportText ("Found duplicates switch");
            elsif EStrings.Eq1_String (E_Str => Switch,
                                       Str   => "annotation_character") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "annotation_characte") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "annotation_charact") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "annotation_charac") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "annotation_chara") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "annotation_char") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "annotation_cha") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "annotation_ch") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "annotation_c") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "annotation_") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "annotation") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "annotatio") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "annotati") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "annotat") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "annota") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "annot") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "anno") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "ann") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "an") or else
              EStrings.Eq1_String (E_Str => Switch,
                                   Str   => "a") then
               ProcessAnnoChar (Switch  => Switch,
                                Arg     => Argument,
                                Success => OptionOk);
            else
               -- unrecognised switch
               OptionOk := False;
               SparkMakeErrors.Report (TheFault => SparkMakeErrors.InvalidSwitch,
                                       EStr1    => Switch,
                                       EStr2    => EStrings.Empty_String,
                                       EStr3    => EStrings.Empty_String);
            end if;
            Success := Success and OptionOk;
         else
            -- nothing more on the command line.
            Done := True;
         end if;
      end loop;

      -- If usage or version info requested then don't continue with any other processing
      if Success and not HelpOrVerFound then

         -- The command line has parsed OK

         -- Set non specified switches to default values.

         if Path = Undefined then
            Path := Full;
         end if;

         if EStrings.Is_Empty (E_Str => IndexFile) then
            -- Index file not specified so index file is <rootfile>.idx
            if EStrings.Is_Empty (E_Str => RootFile) then
               IndexFile := EStrings.Copy_String (Str => "spark");
            else
               IndexFile := RootFile;
            end if;
            Directory_Operations.SetExtension
              (Path => IndexFile,
               Ext  => EStrings.Copy_String (CommandLineData.Default_Index_Extension));
            if Debug then
               SparkMakeDebug.ReportTextEText ("Using default index file: ", IndexFile);
            end if;
         end if;

         if EStrings.Is_Empty (E_Str => MetaFile) then
            -- Meta file not specified so meta file is <rootfile>.smf
            if EStrings.Is_Empty (E_Str => RootFile) then
               MetaFile := EStrings.Copy_String (Str => "spark");
            else
               MetaFile := RootFile;
            end if;
            Directory_Operations.SetExtension
              (Path => MetaFile,
               Ext  => EStrings.Copy_String (CommandLineData.Meta_File_Extension));
            if Debug then
               SparkMakeDebug.ReportTextEText ("Using default meta file: ", MetaFile);
            end if;
         end if;

         if StringList.IsNull (StringList.GetFirst (TheIncRegExps)) then
            -- No include was specified so use the default (GNAT file naming convention).
            StringList.AddToFront (ToList  => TheIncRegExps,
                                   TheItem => EStrings.Copy_String (Str => "*\.ad[bs]"));
            if Debug then
               SparkMakeDebug.ReportText ("No include switch. Will use the GNAT naming convention");
            end if;
         end if;

         if not EStrings.Is_Empty (E_Str => RootFile) then
            -- Make sure the root file will be included.
            TheRegExp := RegExp (ForFile => RootFile);
            if EStrings.Is_Empty (E_Str => TheRegExp) then
               SystemErrors.FatalError
                 (SysErr => SystemErrors.OtherInternalError,
                  Msg => "Cannot generate regular expression for root file.");
            else
               SparkMakeDebug.ReportTextEText ("Root file will be included by ", TheRegExp);
               StringList.AddToFront (ToList  => TheIncRegExps,
                                      TheItem => TheRegExp);
            end if;
         end if;

      end if;
   end Process;


   function PathRequired return PathType
   --# global in Path;
   is
   begin
      return Path;
   end PathRequired;

   function MetaFileName return EStrings.T
   --# global in MetaFile;
   is
   begin
      return MetaFile;
   end MetaFileName;

   function IndexFileName return EStrings.T
   --# global in IndexFile;
   is
   begin
      return IndexFile;
   end IndexFileName;

   function RootFileName return EStrings.T
   --# global in RootFile;
   is
   begin
      return RootFile;
   end RootFileName;

   function DuplicatesError return Boolean
   --# global in Duplicates;
   is
   begin
      return Duplicates;
   end DuplicatesError;

   function DebugOn return Boolean
   --# global in Debug;
   is
   begin
      return Debug;
   end DebugOn;

   function NoIndexFile return Boolean
   --# global in NoIndex;
   is
   begin
      return NoIndex;
   end NoIndexFile;

   function NoMetaFile return Boolean
   --# global in NoMeta;
   is
   begin
      return NoMeta;
   end NoMetaFile;

   function TheDirectoryNames return StringList.Object
   --# global in TheDirectories;
   is
   begin
      return TheDirectories;
   end TheDirectoryNames;

   function TheIncFileRegExps return StringList.Object
   --# global in TheIncRegExps;
   is
   begin
      return TheIncRegExps;
   end TheIncFileRegExps;

   function TheExcFileRegExps return StringList.Object
   --# global in TheExcRegExps;
   is
   begin
      return TheExcRegExps;
   end TheExcFileRegExps;

end SparkMakeCommandLine;
