-- $Id: regularexpression.adb 14623 2009-10-28 13:54:11Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with EStrings.Not_Spark;
with GNAT.Regexp;
with SparkMakeDebug;
with SystemErrors;

package body RegularExpression
is

   ------------------------------------------------------
   -- Constructor
   ------------------------------------------------------

   function Create (EStr : EStrings.T) return Object
   --
   -- This function compiles the regular expression and then doesn't use it!
   -- This is to avoid having to with GNAT.Regexp in the specification and
   -- thus keeping the spec within the SPARK boundary.
   -- This function checks that a regular expression *can* be compiled for the
   -- given string. If it cannot then a null Object is returned that will
   -- cause a fatal error when passed to Matches.
   -- The Matches operation must then recompile the expression - but of course
   -- it knows it will succeed.
   -- This body is NOT within the SPARK boundary.
   is
      R : GNAT.Regexp.Regexp;
      Valid : Boolean := True;
      Result : Object;
      pragma Unreferenced (R);
   begin
      -- prohibit the {} characters
      for I in 1 .. EStrings.Get_Length (E_Str => EStr) loop
         if EStrings.Get_Element (E_Str => EStr,
                                  Pos   => I) = '{' or else
           EStrings.Get_Element (E_Str => EStr,
                                 Pos   => I) = '}' then
            Valid := False;
            exit;
         end if;
      end loop;
      if Valid then
         R := GNAT.Regexp.Compile
           (Pattern        => EStrings.Not_Spark.Get_String (E_Str => EStr),
            Glob           => True,
            Case_Sensitive => False);
         Result := Object'(TheRegExp => EStr,
                           IsNullExp => False);
      else
         SparkMakeDebug.ReportTextEText (Text => "Invalid regular expression",
                                         EText => EStr);
         Result := NullObject;
      end if;
      return Result;
   exception
      when GNAT.Regexp.Error_In_Regexp =>
         SparkMakeDebug.ReportTextEText (Text => "Invalid regular expression",
                                         EText => EStr);
         return NullObject;
   end Create;


   ------------------------------------------------------
   -- Accessors
   ------------------------------------------------------

   function IsNull (O : Object) return Boolean
   is
   begin
      return O.IsNullExp;
   end IsNull;


   function Matches (EStr      : EStrings.T;
                     TheRegExp : Object) return Boolean
   is
   begin
      if IsNull (TheRegExp) then
         SystemErrors.FatalError
           (SysErr => SystemErrors.OtherInternalError,
            Msg => "Illegal use of null regular expression");
         return False;
      else
         return GNAT.Regexp.Match
           (S => EStrings.Not_Spark.Get_String (E_Str => EStr),
            R => GNAT.Regexp.Compile
              (Pattern        => EStrings.Not_Spark.Get_String (E_Str => TheRegExp.TheRegExp),
               Glob           => True,
               Case_Sensitive => False));
      end if;
   end Matches;

end RegularExpression;
