-- $Id: directory_operations.adb 14623 2009-10-28 13:54:11Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with EStrings.Not_Spark;
with GNAT.Directory_Operations;
with GNAT.OS_Lib;
with SystemErrors;

package body Directory_Operations
is

   function Prefix (EStr             : EStrings.T;
                    Separator        : Character;
                    IncludeSeparator : Boolean) return EStrings.T
   -- Returns the text up to the last occurrance of the separator.
   -- If IncludeSeparator is true then the string returned ends in the separator
   -- otherwise it does not.
   is
      Result : EStrings.T := EStrings.Empty_String;
      Last   : Integer := 0;
   begin
      for I in reverse Integer range 1 .. EStrings.Get_Length (E_Str => EStr) loop
         if I /= EStrings.Get_Length (E_Str => EStr) and then
           EStrings.Get_Element (E_Str => EStr,
                                 Pos   => I) = Separator then
            Last := I;
            exit;
         end if;
      end loop;

      if Last /= 0 then
         if IncludeSeparator then
            Result := EStrings.Section (E_Str     => EStr,
                                        Start_Pos => 1,
                                        Length    => Last);
         else
            Result := EStrings.Section (E_Str     => EStr,
                                        Start_Pos => 1,
                                        Length    => Last - 1);
         end if;
      end if;
      return Result;
   end Prefix;

   --------------------------------------------------------------------------------

   function DirectorySeparator return Character
   is
   begin
      return GNAT.OS_Lib.Directory_Separator;
   end DirectorySeparator;

   --------------------------------------------------------------------------------

   function CurrentDirectory return EStrings.T
   is
   begin
      return EStrings.Copy_String (Str => GNAT.Directory_Operations.Get_Current_Dir);
   end CurrentDirectory;

   --------------------------------------------------------------------------------

   function FindFiles (Matching    : RegularExpression.Object;
                       InDirectory : EStrings.T;
                       Recursively : Boolean) return StringList.Object
   is

      TheResult : StringList.Object;

      procedure Scan_Directory (Dir : in GNAT.Directory_Operations.Dir_Name_Str)
      is
         D    : GNAT.Directory_Operations.Dir_Type;
         Str  : String (1 .. 1024);
         Last : Natural;
      begin
         GNAT.Directory_Operations.Open (D, Dir);
         loop
            GNAT.Directory_Operations.Read (D, Str, Last);
            exit when Last = 0;

            declare
               F : constant String := Dir & Str (1 .. Last);
               EStr : EStrings.T;
            begin
               if GNAT.OS_Lib.Is_Directory (F) then

                  --  Ignore "." and ".."
                  if ((Last = 1) and then (Str (1) = '.')) or
                    ((Last = 2) and then (Str (1) = '.' and
                                            Str (2) = '.')) then
                     null;
                  elsif Recursively then
                     --  Recurse here
                     Scan_Directory (F & GNAT.OS_Lib.Directory_Separator);
                  end if;
               else
                  -- Does this file match the regular expression?
                  EStr := EStrings.Copy_String (Str => F);
                  if RegularExpression.Matches (EStr      => EStr,
                                                TheRegExp => Matching) then
                     StringList.AddInLexOrder (ToList => TheResult,
                                               TheItem => EStr);
                  end if;
               end if;
            end;

         end loop;
         GNAT.Directory_Operations.Close (D);
      exception
         when others =>
            GNAT.Directory_Operations.Close (D);
            raise;
      end Scan_Directory;

   begin
      TheResult := StringList.NullObject;
      Scan_Directory (EStrings.Not_Spark.Get_String (E_Str => InDirectory));
      return TheResult;
   end FindFiles;

   --------------------------------------------------------------------------------

   function FileExtension (Path : EStrings.T) return EStrings.T
   is
   begin
      return EStrings.Copy_String
        (Str => GNAT.Directory_Operations.File_Extension
           (Path => EStrings.Not_Spark.Get_String (E_Str => Path)));
   end FileExtension;

   --------------------------------------------------------------------------------

   procedure SetExtension (Path : in out EStrings.T;
                           Ext  : in     EStrings.T)
   is
   begin
      if not EStrings.Is_Empty (E_Str => FileExtension (Path)) then
         -- Has an extension so remove it
         Path := Prefix (Path, '.', False);
      end if;
      -- Add the given extension.
      EStrings.Append_String (E_Str => Path,
                              Str   => ".");
      EStrings.Append_Examiner_String (E_Str1 => Path,
                                       E_Str2 => Ext);
   end SetExtension;

   --------------------------------------------------------------------------------

   function FileName (Path : EStrings.T) return EStrings.T
   is
   begin
      return EStrings.Copy_String
        (Str   => GNAT.Directory_Operations.File_Name
           (Path => EStrings.Not_Spark.Get_String (E_Str => Path)));
   end FileName;

   --------------------------------------------------------------------------------

   function IsDirectory (Path : EStrings.T) return Boolean
   is
   begin
      return GNAT.OS_Lib.Is_Directory
        (Name => EStrings.Not_Spark.Get_String (E_Str => Path));
   end IsDirectory;

   --------------------------------------------------------------------------------

   function IsFile (Path : EStrings.T) return Boolean
   is
   begin
      return GNAT.OS_Lib.Is_Regular_File
        (Name => EStrings.Not_Spark.Get_String (E_Str => Path));
   end IsFile;

   --------------------------------------------------------------------------------

   function NormalizePathName
     (Name      : EStrings.T;
      Directory : EStrings.T)
     return EStrings.T
   is
   begin
      return EStrings.Copy_String
        (Str   => GNAT.OS_Lib.Normalize_Pathname
           (Name          => EStrings.Not_Spark.Get_String (E_Str => Name),
            Directory     => EStrings.Not_Spark.Get_String (E_Str => Directory),
            Resolve_Links => False));
   end NormalizePathName;

   --------------------------------------------------------------------------------

   procedure NormaliseDir (D : in out EStrings.T)
   is
      Unused : Boolean;
   begin
      if not IsDirectory (D) then
         SystemErrors.FatalError
           (SysErr => SystemErrors.OtherInternalError,
            Msg => "Internal error.");
      elsif EStrings.Get_Element (E_Str => D,
                                  Pos   => EStrings.Get_Length (E_Str => D)) /= DirectorySeparator then
         EStrings.Append_Char (E_Str   => D,
                               Ch      => DirectorySeparator,
                               Success => Unused);
      end if;
   end NormaliseDir;

   --------------------------------------------------------------------------------

   function UpDir (D : EStrings.T) return EStrings.T
   is
      Result : EStrings.T := EStrings.Empty_String;
   begin
      if not IsDirectory (D) then
         SystemErrors.FatalError
           (SysErr => SystemErrors.OtherInternalError,
            Msg => "Internal error.");
      else
         Result := Prefix (EStr             => D,
                           Separator        => DirectorySeparator,
                           IncludeSeparator => True);
      end if;
      return Result;
   end UpDir;

   --------------------------------------------------------------------------------

   function RelativeName
     (OfThisFileOrDir : EStrings.T;
      ToThisDir       : EStrings.T) return EStrings.T
   is

      TheCommonBit : EStrings.T;
      OfThis       : EStrings.T;
      ToThis       : EStrings.T;
      Result       : EStrings.T;
      Unused       : Boolean;

      function CommonPath (P1, P2 : EStrings.T) return
        EStrings.T is
      --
      -- Returns the common directory.
      -- The last char is always the directory separator
         Result : EStrings.T := EStrings.Empty_String;
         Unused : Boolean;
      begin
         for I in Integer range 1 .. EStrings.Get_Length (E_Str => P1) loop
            exit when I > EStrings.Get_Length (E_Str => P2);
            exit when EStrings.Get_Element (E_Str => P1,
                                            Pos   => I) /=
              EStrings.Get_Element (E_Str => P2,
                                    Pos   => I);
            EStrings.Append_Char
              (E_Str   => Result,
               Ch      => EStrings.Get_Element (E_Str => P1,
                                                Pos   => I),
               Success => Unused);
         end loop;
         if EStrings.Get_Element (E_Str => Result,
                                  Pos   => EStrings.Get_Length (E_Str => Result)) /= DirectorySeparator then
            Result := Prefix (Result, DirectorySeparator, True);
         end if;
         return Result;
      end CommonPath;

   begin
      Result := EStrings.Empty_String;
      ToThis := ToThisDir;
      OfThis := OfThisFileOrDir;

      -- Check the input parameters make sense.
      if IsFile (OfThis) and not IsDirectory (OfThis) then
         OfThis := Prefix (OfThis, DirectorySeparator, True);
      end if;

      if IsDirectory (ToThis) then
         NormaliseDir (ToThis);
      end if;

      if not IsDirectory (OfThis) or not IsDirectory (ToThis) then
         SystemErrors.FatalError
           (SysErr => SystemErrors.OtherInternalError,
            Msg => "Internal error.");
      end if;

      TheCommonBit := CommonPath (OfThis, ToThis);

      if EStrings.Is_Empty (E_Str => TheCommonBit) then
         Result := OfThisFileOrDir;

      elsif EStrings.Eq_String (E_Str1 => TheCommonBit,
                                E_Str2 => ToThis) then
         Result := EStrings.Section
           (E_Str     => OfThisFileOrDir,
            Start_Pos => EStrings.Get_Length (E_Str => TheCommonBit) + 1,
            Length    => EStrings.Get_Length (E_Str => OfThisFileOrDir) - EStrings.Get_Length (E_Str => TheCommonBit));

      else
         loop
            ToThis := UpDir (ToThis);
            EStrings.Append_String (E_Str => Result,
                                    Str   => "..");
            EStrings.Append_Char (E_Str   => Result,
                                  Ch      => DirectorySeparator,
                                  Success => Unused);
            exit when EStrings.Eq_String (E_Str1 => ToThis,
                                          E_Str2 => TheCommonBit);
         end loop;
         EStrings.Append_Examiner_String
           (E_Str1 => Result,
            E_Str2 => EStrings.Section
              (E_Str     => OfThisFileOrDir,
               Start_Pos => EStrings.Get_Length (E_Str => TheCommonBit) + 1,
               Length    => EStrings.Get_Length (E_Str => OfThisFileOrDir) - EStrings.Get_Length (E_Str => TheCommonBit)));
      end if;
      return Result;
   end RelativeName;


end Directory_Operations;
