-- $Id: sparklalr_error.adb 14467 2009-10-07 10:07:14Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with Sparklalr_Char_Class;
use type Sparklalr_Char_Class.Char_Class;

package body Sparklalr_Error
   --# own State is
   --#   Err_Ptr,
   --#   Tail_Ptr,
   --#   Error_Set,
   --#   Prod_Err,
   --#   Sequenced,
   --#   Line_Out,
   --#   Blank_Line;
is

   Error_Table_Size : constant := 100;

   type Error_Set_T is array (Error_Range) of Boolean;

   Error_Set_False_Const : constant Error_Set_T := Error_Set_T'(others => False);

   subtype Pt_Error_Type is Natural range 0 .. Error_Table_Size;
   subtype Err_Array_Range is Positive range 1 .. Error_Table_Size;
   type Error_Type is record
      Used     : Boolean;
      Err_Col  : Err_Col_T;
      Error_No : Error_Range;
      Next     : Pt_Error_Type;
   end record;
   type Err_Array_T is array (Err_Array_Range) of Error_Type;

   subtype Line_Array_Range is Positive range 1 .. Sparklalr_Common.Line_Length;
   subtype Line_Array is String (Line_Array_Range);

   Err_Ptr    : Err_Array_T;
   Tail_Ptr   : Pt_Error_Type;
   Error_Set  : Error_Set_T;
   Prod_Err   : Boolean;
   Sequenced  : Boolean;
   Line_Out   : Line_Array;
   Blank_Line : Line_Array;

   -- Local procedures/functions
   procedure Skip_Seqno (F : in Sparklalr_IO.File_Type)
      --# global in     Sequenced;
      --#        in     Sparklalr_Char_Class.Charmap;
      --#        in out Sparklalr_IO.Inputs;
      --# derives Sparklalr_IO.Inputs from *,
      --#                                  F,
      --#                                  Sequenced,
      --#                                  Sparklalr_Char_Class.Charmap;
   is
      C : Character;
   begin
      Sparklalr_IO.Look_Ahead (F, C);
      if Sequenced and (Sparklalr_Char_Class.Get_Charmap (C) = Sparklalr_Char_Class.Digit) then
         Sparklalr_IO.Skip_Int (F, 0);
         Sparklalr_IO.Look_Ahead (F, C);
         if C = ' ' then
            --# accept F, 10, C, "Skipping whitespace, so value is discarded OK";
            Sparklalr_IO.Get_Char (F, C);
            --# end accept;
         end if;
      end if;
   end Skip_Seqno;

   procedure List_Errors (F : in Sparklalr_IO.File_Type)
      --# global in     Error_Set;
      --#        in out Err_Ptr;
      --#        in out Sparklalr_IO.Outputs;
      --# derives Err_Ptr              from * &
      --#         Sparklalr_IO.Outputs from *,
      --#                                   Error_Set,
      --#                                   Err_Ptr,
      --#                                   F;
   is

      procedure Eprint (F : in Sparklalr_IO.File_Type)
         --# global in out Err_Ptr;
         --#        in out Sparklalr_IO.Outputs;
         --# derives Err_Ptr              from * &
         --#         Sparklalr_IO.Outputs from *,
         --#                                   Err_Ptr,
         --#                                   F;
      is
         T           : Pt_Error_Type;
         Min_Err_Col : Err_Col_T;
         Continue    : Boolean;
         Err_Head    : Pt_Error_Type;
         Column      : Err_Col_T;
      begin
         Err_Head := 1;
         Column   := 0;
         Continue := True;
         while Continue loop
            while Err_Head /= 0 loop
               if Err_Ptr (Err_Head).Err_Col < Column then
                  Err_Head := Err_Ptr (Err_Head).Next;
               else
                  Sparklalr_Common.Put_N_Chars
                    (Sparklalr_IO.Standard_Output, '^', (Err_Ptr (Err_Head).Err_Col - Column) + 1);
                  Sparklalr_IO.Put_Int (Sparklalr_IO.Standard_Output, Err_Ptr (Err_Head).Error_No, 2);
                  Sparklalr_IO.Put_Char (Sparklalr_IO.Standard_Output, ' ');
                  Sparklalr_Common.Put_N_Chars (F, '^', (Err_Ptr (Err_Head).Err_Col - Column) + 1);
                  Sparklalr_IO.Put_Int (F, Err_Ptr (Err_Head).Error_No, 2);
                  Sparklalr_IO.Put_Char (F, ' ');
                  Column      := Err_Ptr (Err_Head).Err_Col + 4;
                  T           := Err_Head;
                  Err_Head    := Err_Ptr (Err_Head).Next;
                  Err_Ptr (T) := Error_Type'(Used => False, Err_Col => 0, Error_No => 0, Next => 0);
               end if;
            end loop;
            Sparklalr_IO.New_Line (Sparklalr_IO.Standard_Output);
            Sparklalr_IO.New_Line (F);
            Column := 0;

            Min_Err_Col := Sparklalr_Common.Line_Length;
            Continue    := False;
            for I in Err_Array_Range loop
               if Err_Ptr (I).Used then
                  if Min_Err_Col > Err_Ptr (I).Err_Col then
                     Min_Err_Col := Err_Ptr (I).Err_Col;
                     Err_Head    := I;
                  end if;
                  Continue := True;
               end if;
            end loop;
         end loop;
      end Eprint;

      procedure Emessages (F : in Sparklalr_IO.File_Type)
         --# global in     Error_Set;
         --#        in out Sparklalr_IO.Outputs;
         --# derives Sparklalr_IO.Outputs from *,
         --#                                   Error_Set,
         --#                                   F;
      is
      begin
         for I in Error_Range loop
            if Error_Set (I) then
               Sparklalr_IO.Put (F, "**ERROR**  :");
               Sparklalr_IO.Put_Int (F, I, 3);
               Sparklalr_IO.Put (F, "  ");
               case I is
               when 0 =>
                  Sparklalr_IO.Put_Line (F, "LINE TOO LONG - TRUNCATED");
               when 2 =>
                  Sparklalr_IO.Put_Line (F, "& NOT FOLLOWED BY KNOWN KEYWORD");
               when 3 =>
                  Sparklalr_IO.Put_Line (F, "UNEXPECTED SYMBOL - SKIPPING FORWARD");
               when 4 =>
                  Sparklalr_IO.Put_Line (F, "NO ACTION FOLLOWING ""=""");
               when 5 =>
                  Sparklalr_IO.Put_Line (F, "UNEXPECTED SYMBOL ON RHS");
               when 6 =>
                  Sparklalr_IO.Put_Line (F, "COLON EXPECTED ON LHS");
               when 7 =>
                  Sparklalr_IO.Put_Line (F, "LHS IDENTIFIER EXPECTED");
               when 8 =>
                  Sparklalr_IO.Put_Line (F, "BAD TERM SECTION SYNTAX");
               when 9 =>
                  Sparklalr_IO.Put_Line (F, "UNEXPECTED && - SKIPPED");
               when 10 =>
                  Sparklalr_IO.Put_Line (F, "INVALID SYMBOL ENCOUNTERED");
               when 12 =>
                  Sparklalr_IO.Put_Line (F, "IDENTIFIER NOT FOUND AFTER &PREC");
               when 13 =>
                  Sparklalr_IO.Put_Line (F, "NON-TERMINAL FOUND AFTER &PREC");
               when 14 =>
                  Sparklalr_IO.Put_Line (F, "TERMINAL SYMBOL ON LHS OF PRODUCTION");
               when 15 =>
                  Sparklalr_IO.Put_Line (F, "TOO MANY NON-TERMINAL SYMBOLS");
               when 16 =>
                  Sparklalr_IO.Put_Line (F, "TOO MANY TERMINAL SYMBOLS");
               when 17 =>
                  Sparklalr_IO.Put_Line (F, "INVALID SYMBOL AFTER ""\"" ");
               when 18 =>
                  Sparklalr_IO.Put_Line (F, """\"" Substitution is not allowed in this version");
               when others =>
                  null;
               end case;
            end if;
         end loop;
      end Emessages;

   begin
      Eprint (F);
      Emessages (Sparklalr_IO.Standard_Output);
      Emessages (F);
   end List_Errors;

   procedure List_Line (F : in Sparklalr_IO.File_Type; Col : in Err_Col_T)
      --# global in     Line_Out;
      --#        in out Sparklalr_IO.Outputs;
      --# derives Sparklalr_IO.Outputs from *,
      --#                                   Col,
      --#                                   F,
      --#                                   Line_Out;
   is
   begin
      Sparklalr_IO.Put_Char (F, ' ');
      if Col > 0 then
         for I in Integer range 1 .. Col loop
            Sparklalr_IO.Put_Char (F, Line_Out (I));
         end loop;
      end if;
      Sparklalr_IO.New_Line (F);
   end List_Line;
   -- End local procedures/functions

   procedure Initialise (F : in Sparklalr_IO.File_Type)
      --# global in     Sparklalr_Char_Class.Charmap;
      --#        in out Sparklalr_IO.Inputs;
      --#           out Blank_Line;
      --#           out Error_Set;
      --#           out Err_Ptr;
      --#           out Line_Out;
      --#           out Prod_Err;
      --#           out Sequenced;
      --#           out Tail_Ptr;
      --# derives Blank_Line,
      --#         Error_Set,
      --#         Err_Ptr,
      --#         Line_Out,
      --#         Prod_Err,
      --#         Tail_Ptr            from  &
      --#         Sequenced,
      --#         Sparklalr_IO.Inputs from F,
      --#                                  Sparklalr_Char_Class.Charmap,
      --#                                  Sparklalr_IO.Inputs;
   is
      C : Character;
   begin
      Err_Ptr    := Err_Array_T'(others => Error_Type'(Used => False, Err_Col => 0, Error_No => 0, Next => 0));
      Tail_Ptr   := 0;
      Error_Set  := Error_Set_T'(others => False);
      Prod_Err   := False;
      Blank_Line := Line_Array'(others => ' ');
      Line_Out   := Blank_Line;
      Sequenced  := True;
      Skip_Seqno (F);
      Sparklalr_IO.Look_Ahead (F, C);
      Sequenced := Sparklalr_Char_Class.Get_Charmap (C) = Sparklalr_Char_Class.Digit;
   end Initialise;

   procedure Syn_Error (Error_Num : in Error_Range; Col : in Err_Col_T)
      --# global in out Error_Set;
      --#        in out Err_Ptr;
      --#        in out Tail_Ptr;
      --#           out Prod_Err;
      --# derives Error_Set from *,
      --#                        Error_Num &
      --#         Err_Ptr   from *,
      --#                        Col,
      --#                        Error_Num,
      --#                        Tail_Ptr &
      --#         Prod_Err  from  &
      --#         Tail_Ptr  from *;
   is
      T : Pt_Error_Type;
   begin
      Prod_Err    := True;
      T           := Tail_Ptr + 1;
      Err_Ptr (T) := Error_Type'(Used => True, Err_Col => Col, Error_No => Error_Num, Next => 0);
      if Tail_Ptr /= 0 then
         Err_Ptr (Tail_Ptr).Next := T;
      end if;
      Tail_Ptr              := T;
      Error_Set (Error_Num) := True;
   end Syn_Error;

   procedure Error (F : in Sparklalr_IO.File_Type; N : in Integer)
      --# global in out Sparklalr_IO.Outputs;
      --#           out Prod_Err;
      --# derives Prod_Err             from  &
      --#         Sparklalr_IO.Outputs from *,
      --#                                   F,
      --#                                   N;
   is

      procedure Err_List (F : in Sparklalr_IO.File_Type; N : in Integer)
         --# global in out Sparklalr_IO.Outputs;
         --# derives Sparklalr_IO.Outputs from *,
         --#                                   F,
         --#                                   N;
      is
      begin
         Sparklalr_IO.Put (F, "***ERROR***");
         Sparklalr_IO.Put_Int (F, N, 4);
         Sparklalr_IO.Put (F, "  ");

         --# accept W, 303, "when others covers all cases here";
         case N is
            when 0 =>
               Sparklalr_IO.Put_Line (F, "LLAMA TERMINATED BEFORE END OF INPUT GRAMMAR FILE");
            when 1 =>
               Sparklalr_IO.Put_Line (F, "NO GRAMMAR SPECIFICATIONS IN INPUT FILE");
            when 30 =>
               Sparklalr_IO.Put (F, "UNDEFINED NONTERMINAL SYMBOL -");
            when 32 | 36 | 50 =>
               Sparklalr_IO.Put_Line (F, "INTERNAL LLAMA ERROR - CONSISTENCY CHECK");
            when others =>
               null;
         end case;
         --# end accept;
      end Err_List;

   begin
      Prod_Err := True;
      Err_List (F, N);
      Err_List (Sparklalr_IO.Standard_Output, N);
   end Error;

   procedure Write_The_Line (F, Echo : in Sparklalr_IO.File_Type; Col : in out Err_Col_T)
      --# global in     Line_Out;
      --#        in     Sequenced;
      --#        in     Sparklalr_Char_Class.Charmap;
      --#        in out Error_Set;
      --#        in out Err_Ptr;
      --#        in out Prod_Err;
      --#        in out Sparklalr_IO.Inputs;
      --#        in out Sparklalr_IO.Outputs;
      --#        in out Tail_Ptr;
      --# derives Col,
      --#         Error_Set            from  &
      --#         Err_Ptr              from *,
      --#                                   Col,
      --#                                   Error_Set,
      --#                                   Tail_Ptr &
      --#         Prod_Err,
      --#         Tail_Ptr             from *,
      --#                                   Col &
      --#         Sparklalr_IO.Inputs  from *,
      --#                                   F,
      --#                                   Sequenced,
      --#                                   Sparklalr_Char_Class.Charmap &
      --#         Sparklalr_IO.Outputs from *,
      --#                                   Col,
      --#                                   Echo,
      --#                                   Error_Set,
      --#                                   Err_Ptr,
      --#                                   F,
      --#                                   Line_Out,
      --#                                   Tail_Ptr;
   is
   begin
      if Col = Sparklalr_Common.Line_Length then
         Syn_Error (0, Col);
      end if;
      List_Line (Echo, Col);
      if Error_Set /= Error_Set_False_Const then
         List_Line (Sparklalr_IO.Standard_Output, Col);
         List_Errors (F);
      end if;
      Error_Set := Error_Set_False_Const;
      Col       := 0;
      Sparklalr_IO.Skip_Line (F, 1);
      if Sequenced then
         Skip_Seqno (F);
      end if;
   end Write_The_Line;

   procedure Set_Line_Out (I : in Err_Col_T; C : in Character)
      --# global in out Line_Out;
      --# derives Line_Out from *,
      --#                       C,
      --#                       I;
   is
   begin
      Line_Out (I) := C;
   end Set_Line_Out;

   procedure List_Line_Errors (F : in Sparklalr_IO.File_Type; Col : in Err_Col_T)
      --# global in     Error_Set;
      --#        in     Line_Out;
      --#        in out Err_Ptr;
      --#        in out Sparklalr_IO.Outputs;
      --# derives Err_Ptr              from *,
      --#                                   Error_Set &
      --#         Sparklalr_IO.Outputs from *,
      --#                                   Col,
      --#                                   Error_Set,
      --#                                   Err_Ptr,
      --#                                   F,
      --#                                   Line_Out;

   is
   begin
      if Error_Set /= Error_Set_False_Const then
         List_Line (F, Col);
         List_Errors (F);
      end if;
   end List_Line_Errors;

   function Get_Prod_Err return Boolean
      --# global in Prod_Err;
        is
   begin
      return Prod_Err;
   end Get_Prod_Err;

end Sparklalr_Error;
