-- $Id: command_line_options.adb 11998 2009-01-02 14:42:09Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

package body Command_Line_Options
   --# own State is
   --#   File_Name,
   --#   Verbose,
   --#   Debug,
   --#   Dump_Mem,
   --#   Parser,
   --#   Self_Pack,
   --#   Multi_Comp;
is

   type Debug_T is array (Debug_Level_Range) of Boolean;

   File_Name  : Sparklalr_IO.File_Name;
   Verbose    : Boolean;
   Debug      : Debug_T; -- DEBUG FLAGS
   Dump_Mem   : Boolean; -- FLAG FOR FINAL MEMORY DUMPING
   Parser     : Boolean;
   Self_Pack  : Boolean;
   Multi_Comp : Boolean;

   procedure Get_Options
      --# global in     Sparklalr_IO.State;
      --#           out Debug;
      --#           out Dump_Mem;
      --#           out File_Name;
      --#           out Multi_Comp;
      --#           out Parser;
      --#           out Self_Pack;
      --#           out Verbose;
      --# derives Debug,
      --#         Dump_Mem,
      --#         File_Name,
      --#         Multi_Comp,
      --#         Parser,
      --#         Self_Pack,
      --#         Verbose    from Sparklalr_IO.State;
   is

      Len_Switch   : constant := 10; -- length of a command line switch string
      Len_Cmd_Line : constant := 255; -- length of a command line string

      subtype Switch_Range is Positive range 1 .. Len_Switch;

      type Status_T is (Found, Missing, Invalid, Too_Long);
      subtype Switch_T is String (Switch_Range);
      subtype Cmd_Line_Range is Positive range 1 .. Len_Cmd_Line;

      Status        : Status_T;
      Switch        : Switch_T;
      Cmd_Line_Pntr : Cmd_Line_Range;

      procedure Cl_Next_Param
        (Cmd_Line_Pntr : in out Cmd_Line_Range;
         String_Var    : out Sparklalr_IO.Arg_String;
         Status        : out Status_T)
         --
         -- Get next parameter from command line and return it in conformant array 'String_Var'.
         -- Returns status which indicates result of operation in 'Status'.
         -- Pointer 'Cmd_Line_Pntr' is left pointing to next paramter to be read or last parameter in
         -- the command line.
         -- Where possible an syntactically invalid parameter is skipped so that subsequent
         -- calls to this procedure corectly return subsequent command line parameters.
         --

         --# global in Sparklalr_IO.State;
         --# derives Cmd_Line_Pntr,
         --#         Status,
         --#         String_Var    from Cmd_Line_Pntr,
         --#                            Sparklalr_IO.State;

      is
      begin
         if Cmd_Line_Pntr <= Sparklalr_IO.Argument_Count then
            String_Var    := Sparklalr_IO.Argument (Cmd_Line_Pntr);
            Status        := Found;
            Cmd_Line_Pntr := Cmd_Line_Pntr + 1;
         else
            String_Var := Sparklalr_IO.Arg_String'(others => ' ');
            Status     := Missing;
         end if;
      end Cl_Next_Param;

      function Cl_Empty (Cmd_Line_Pntr : in Cmd_Line_Range) return Boolean
         --
         -- Checks if the command line has any unread parameters on it, returns True if yes,
         -- False if not. May be called at any time after a call to CL_Read.
         -- Does not affect the value of Cmd_Line_Pntr.
         --

         --# global in Sparklalr_IO.State;

           is
      begin
         return Cmd_Line_Pntr > Sparklalr_IO.Argument_Count;
      end Cl_Empty;

      procedure Cl_File_Name
        (Cmd_Line_Pntr : in out Cmd_Line_Range;
         File_Name     : out Sparklalr_IO.File_Name;
         Status        : out Status_T)
         --# global in Sparklalr_IO.State;
         --# derives Cmd_Line_Pntr,
         --#         File_Name,
         --#         Status        from Cmd_Line_Pntr,
         --#                            Sparklalr_IO.State;
      is
         Arg_File_Name : Sparklalr_IO.Arg_String;

         procedure Check_File_Name (File_Name : in out Sparklalr_IO.File_Name; Status : in out Status_T)
            --
            -- Checks 'File_Name' for correct syntax and converts lower case letters to
            -- upper case.
            --

            --# derives File_Name,
            --#         Status    from File_Name,
            --#                        Status;

         is
         begin
            if Status = Found then -- check File_Name characters are valid
               for I in Sparklalr_IO.File_Name_Range loop
                  if (I = File_Name'First
                     and then not (File_Name (I) in 'a' .. 'z' or File_Name (I) in 'A' .. 'Z' or File_Name (I) in '0' .. '9')) or
                     (I /= File_Name'First
                     and then not (File_Name (I) in 'a' .. 'z' or
                                   File_Name (I) in 'A' .. 'Z' or
                                   File_Name (I) in '0' .. '9' or
                                   File_Name (I) = '_' or
                                   File_Name (I) = ' ')) then
                     Status := Invalid;
                  end if;
               end loop;
               if Status = Found then
                  Sparklalr_IO.To_Upper (File_Name);
               end if;
            end if;
         end Check_File_Name;

      begin
         File_Name := Sparklalr_IO.File_Name'(others => ' ');
         Cl_Next_Param (Cmd_Line_Pntr, Arg_File_Name, Status);
         if Status = Found then
            if Sparklalr_IO.Trim_Length (Arg_File_Name) <= File_Name'Length then
               for I in Sparklalr_IO.File_Name_Range loop
                  File_Name (I) := Arg_File_Name (I);
               end loop;
            else
               File_Name := Sparklalr_IO.File_Name'(others => ' ');
               Status    := Too_Long;
            end if;
         end if;
         Check_File_Name (File_Name, Status);
      end Cl_File_Name;

      procedure Cl_Switch
        (Cmd_Line_Pntr : in out Cmd_Line_Range;
         Switch        : out Switch_T;
         Status        : out Status_T)
         --# global in Sparklalr_IO.State;
         --# derives Cmd_Line_Pntr,
         --#         Status,
         --#         Switch        from Cmd_Line_Pntr,
         --#                            Sparklalr_IO.State;
      is
         Arg_Switch : Sparklalr_IO.Arg_String;

         procedure Check_Switch (Switch : in out Switch_T; Status : in out Status_T)
            --
            -- Checks Switch string for correct syntax, converting characters to upper case
            --

            --# derives Status,
            --#         Switch from Status,
            --#                     Switch;

         is
         begin
            if Status = Found then -- check Switch characters are valid
               for I in Switch_Range loop
                  if not (Switch (I) in 'a' .. 'z' or Switch (I) in 'A' .. 'Z' or Switch (I) = '-' or Switch (I) = ' ') then
                     Status := Invalid;
                  end if;
               end loop;
               if Status = Found then
                  Sparklalr_IO.To_Upper (Switch);
               end if;
            end if;
         end Check_Switch;

      begin
         Switch := Switch_T'(others => ' ');
         Cl_Next_Param (Cmd_Line_Pntr, Arg_Switch, Status);
         if Status = Found then
            if Sparklalr_IO.Trim_Length (Arg_Switch) <= Switch'Length then
               for I in Switch_Range loop
                  Switch (I) := Arg_Switch (I);
               end loop;
            else
               Switch := Switch_T'(others => ' ');
               Status := Too_Long;
            end if;
         end if;
         Check_Switch (Switch, Status);
      end Cl_Switch;

   begin
      Verbose       := False;
      Debug         := Debug_T'(others => False);
      Dump_Mem      := False;
      Parser        := False;
      Self_Pack     := False;
      Multi_Comp    := False;
      Cmd_Line_Pntr := 1;
      Cl_File_Name (Cmd_Line_Pntr, File_Name, Status);
      case Status is
         when Missing =>
            Sparklalr_IO.Exit_St ("No grammar file name supplied", Sparklalr_IO.Error);
         when Invalid =>
            Sparklalr_IO.Exit_St ("Grammar file name contains invalid characters", Sparklalr_IO.Error);
         when Too_Long =>
            Sparklalr_IO.Exit_St ("Grammar file name is too long", Sparklalr_IO.Error);
         when Found =>
            null;
      end case;
      while not Cl_Empty (Cmd_Line_Pntr) loop
         Cl_Switch (Cmd_Line_Pntr, Switch, Status);
         case Status is
            when Missing =>
               null;
            when Invalid =>
               Sparklalr_IO.Exit_St ("Switch name contains invalid characters", Sparklalr_IO.Error);
            when Too_Long =>
               Sparklalr_IO.Exit_St ("Switch name is too long", Sparklalr_IO.Error);
            when Found =>
               case Switch (2) is
                  when 'V' =>
                     Verbose := True;
                  when 'S' =>
                     Self_Pack := True;
                  when 'M' =>
                     Multi_Comp := True;
                  when 'D' =>
                     case Switch (3) is
                     when 'A' =>
                        Debug (1) := True;
                     when 'B' =>
                        Debug (2) := True;
                     when 'C' =>
                        Debug (3) := True;
                     when 'D' =>
                        Debug (4) := True;
                     when 'E' =>
                        Debug (5) := True;
                     when 'F' =>
                        Debug (6) := True;
                     when 'G' =>
                        Debug (7) := True;
                     when 'H' =>
                        Debug (8) := True;
                     when 'I' =>
                        Debug (9) := True;
                     when 'U' =>
                        Dump_Mem := True;
                     when others =>
                        Sparklalr_IO.Exit_St ("Invalid switch", Sparklalr_IO.Error);
                     end case;
                  when 'P' =>
                     Parser := True;
                  when others =>
                     Sparklalr_IO.Exit_St ("Invalid switch", Sparklalr_IO.Error);
               end case;
         end case;
      end loop;
   end Get_Options;

   function Get_File_Name return  Sparklalr_IO.File_Name
      --# global in File_Name;
        is
   begin
      return File_Name;
   end Get_File_Name;

   function Get_Verbose return Boolean
      --# global in Verbose;
        is
   begin
      return Verbose;
   end Get_Verbose;

   function Get_Debug_Level (Level : in Debug_Level_Range) return Boolean
      --# global in Debug;
        is
   begin
      return Debug (Level);
   end Get_Debug_Level;

   function Get_Dump_Mem return Boolean
      --# global in Dump_Mem;
        is
   begin
      return Dump_Mem;
   end Get_Dump_Mem;

   function Get_Parser return Boolean
      --# global in Parser;
        is
   begin
      return Parser;
   end Get_Parser;

   function Get_Self_Pack return Boolean
      --# global in Self_Pack;
        is
   begin
      return Self_Pack;
   end Get_Self_Pack;

   function Get_Multi_Comp return Boolean
      --# global in Multi_Comp;
        is
   begin
      return Multi_Comp;
   end Get_Multi_Comp;

end Command_Line_Options;
