-- $Id: sparkprogram.adb 15795 2010-01-26 14:30:59Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with Ada.Characters.Handling;
with Ada.Characters.Latin_1;
with CommandLineData;
with EStrings;
with Heap;
with LexTokenManager;
with LexTokenManager.Relation_Algebra;
with LexTokenManager.Relation_Algebra.String;
with LexTokenManager.Seq_Algebra;
with SeqAlgebra;
with SparkFormatCommandLineData;
with SparkLex;
with SPSymbols;

use type LexTokenManager.Str_Comp_Result;
use type SeqAlgebra.MemberOfSeq;
use type SparkFormatCommandLineData.ExpandOrCompress;
use type SPSymbols.SPTerminal;

package body SPARKProgram
is

   --# inherit Ada.Characters.Latin_1,
   --#         EStrings;
   package WhiteSpace
   is
      function IsWhiteSpace (Char : Character) return Boolean;

      procedure Skip (InputLine : in     EStrings.T;
                      Index     : in out EStrings.Lengths);
      --# derives Index from *,
      --#                    InputLine;
   end WhiteSpace;

   --# inherit CommandLineData,
   --#         Dictionary,
   --#         ErrorHandler,
   --#         EStrings,
   --#         LexTokenManager,
   --#         SparkLex,
   --#         SPARK_IO,
   --#         SPSymbols,
   --#         WhiteSpace;
   package File_IO
   is

      procedure Create (TemporaryFile : in out SPARK_IO.File_Type);
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys,
      --#         TemporaryFile     from SPARK_IO.File_Sys,
      --#                                TemporaryFile;

      procedure Reset (TemporaryFile : in out SPARK_IO.File_Type);
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys,
      --#         TemporaryFile     from *,
      --#                                TemporaryFile;

      procedure Delete (TemporaryFile : in out SPARK_IO.File_Type);
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys,
      --#         TemporaryFile     from *,
      --#                                TemporaryFile;

      procedure Lex (TemporaryFile : in     SPARK_IO.File_Type;
                     Token         :    out SPSymbols.SPTerminal;
                     LexVal        :    out LexTokenManager.Lex_Value);
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.State;
      --#        in out SparkLex.Currline;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.ErrorContext,
      --#         LexTokenManager.State,
      --#         LexVal,
      --#         SparkLex.Currline,
      --#         SPARK_IO.File_Sys,
      --#         Token                     from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        SparkLex.Currline,
      --#                                        SPARK_IO.File_Sys,
      --#                                        TemporaryFile;

   end File_IO;

   --# inherit Heap,
   --#         LexTokenManager,
   --#         LexTokenManager.Seq_Algebra,
   --#         SeqAlgebra,
   --#         WhiteSpace;
   package Iteration is
      type Iterator is private;

      procedure Initialise (TheHeap     : in     Heap.HeapRecord;
                            TheSeq      : in     LexTokenManager.Seq_Algebra.Seq;
                            TheIterator :    out Iterator);
      --# global in LexTokenManager.State;
      --# derives TheIterator from LexTokenManager.State,
      --#                          TheHeap,
      --#                          TheSeq;

      procedure Next (TheHeap     : in     Heap.HeapRecord;
                      TheIterator : in out Iterator);
      --# global in LexTokenManager.State;
      --# derives TheIterator from *,
      --#                          LexTokenManager.State,
      --#                          TheHeap;

      function Complete (TheIterator : Iterator) return Boolean;

      function CurrentString (TheIterator : Iterator)
                              return LexTokenManager.Lex_String;

      function CurrentMember (TheIterator : Iterator)
                              return LexTokenManager.Seq_Algebra.Member_Of_Seq;

   private

      type Iterator is record
         FirstMember   : LexTokenManager.Seq_Algebra.Member_Of_Seq;
         CurrentMember : LexTokenManager.Seq_Algebra.Member_Of_Seq;
         Placeholder   : LexTokenManager.Lex_String;
         Complete      : Boolean;
      end record;

   end Iteration;

   --# inherit Ada.Characters.Handling,
   --#         CommandLineData,
   --#         EStrings,
   --#         SparkFormatCommandLineData,
   --#         SPARK_IO,
   --#         WhiteSpace;
   package Annotations is
      type Anno_Type is private;

      procedure Initialize (This       :    out Anno_Type;
                            AnnoIntro  : in     String;
                            AnnoSucc   : in     String;
                            AnnoIndent : in     Natural);
      --# global in CommandLineData.Content;
      --# derives This from AnnoIndent,
      --#                   AnnoIntro,
      --#                   AnnoSucc,
      --#                   CommandLineData.Content;

      function Intro (This : Anno_Type) return EStrings.T;

      function Indent (This : Anno_Type) return Natural;

      procedure IsAnnoStart (This      : in     Anno_Type;
                             InputLine : in     EStrings.T;
                             Index     : in out EStrings.Lengths;
                             OK        :    out Boolean);
      --# derives Index,
      --#         OK    from Index,
      --#                    InputLine,
      --#                    This;

      function IsStartOf (This      : Anno_Type;
                          InputLine : EStrings.T)
                              return Boolean;

      function IsEndOf (This        : Anno_Type;
                        InputLine : EStrings.T) return Boolean;

      function ProperStartCol (StartCol : EStrings.Positions)
         return EStrings.Positions;

      procedure Write (This     : Anno_Type;
                       Output   : in SPARK_IO.File_Type;
                       StartCol : in EStrings.Positions);
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Output,
      --#                                StartCol,
      --#                                This;

      procedure WriteIntro (This     : in Anno_Type;
                            Output   : in SPARK_IO.File_Type;
                            StartCol : in EStrings.Positions);
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Output,
      --#                                StartCol,
      --#                                This;

      function Name1StartCol (This     : in Anno_Type;
                              StartCol : in EStrings.Positions)
                              return EStrings.Positions;

   private
      subtype AnnoStartBounds is Positive range 1 .. 3;
      subtype AnnoStart_Type is String (AnnoStartBounds);

      type Anno_Type is record
         AnnoStart   : AnnoStart_Type;
         AnnoIntro   : EStrings.T;
         AnnoSucc    : EStrings.T;
         AnnoIndent  : Natural;
      end record;
   end Annotations;

   --# inherit Ada.Characters.Handling,
   --#         Ada.Characters.Latin_1,
   --#         Annotations,
   --#         CommandLineData,
   --#         Dictionary,
   --#         ELStrings,
   --#         ErrorHandler,
   --#         EStrings,
   --#         File_IO,
   --#         Heap,
   --#         Iteration,
   --#         LexTokenManager,
   --#         LexTokenManager.Relation_Algebra,
   --#         LexTokenManager.Relation_Algebra.String,
   --#         LexTokenManager.Seq_Algebra,
   --#         RelationAlgebra,
   --#         SeqAlgebra,
   --#         SparkFormatCommandLineData,
   --#         SparkLex,
   --#         SPARKProgram,
   --#         SPARK_IO,
   --#         SPSymbols,
   --#         Statistics,
   --#         WhiteSpace;
   package Reformatter is
      Allow_Dotted_Names    : constant Boolean := True;
      Disallow_Dotted_Names : constant Boolean := False;

      type State is private;

      procedure Initialise (Anno        : in     Annotations.Anno_Type;
                            DottedNames : in     Boolean;
                            TheHeap     : in out Heap.HeapRecord;
                            This        :    out State);
      --# global in out Statistics.TableUsage;
      --# derives Statistics.TableUsage,
      --#         TheHeap               from *,
      --#                                    TheHeap &
      --#         This                  from Anno,
      --#                                    DottedNames,
      --#                                    TheHeap;

      procedure Parse (This          : in out State;
                       TheHeap       : in out Heap.HeapRecord;
                       TemporaryFile : in out SPARK_IO.File_Type);
      --# global in     CommandLineData.Content;
      --#        in out LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --# derives LexTokenManager.State,
      --#         SPARK_IO.File_Sys     from *,
      --#                                    CommandLineData.Content,
      --#                                    SPARK_IO.File_Sys,
      --#                                    TemporaryFile,
      --#                                    This &
      --#         Statistics.TableUsage,
      --#         TheHeap,
      --#         This                  from *,
      --#                                    CommandLineData.Content,
      --#                                    LexTokenManager.State,
      --#                                    SPARK_IO.File_Sys,
      --#                                    TemporaryFile,
      --#                                    TheHeap,
      --#                                    This &
      --#         TemporaryFile         from *;

      procedure Reformat (This          : in     State;
                          TheHeap       : in out Heap.HeapRecord;
                          TemporaryFile : in out SPARK_IO.File_Type;
                          Output        : in     SPARK_IO.File_Type;
                          Success       :    out Boolean);
      --# global in     CommandLineData.Content;
      --#        in     LexTokenManager.State;
      --#        in     SparkFormatCommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --# derives SPARK_IO.File_Sys     from *,
      --#                                    CommandLineData.Content,
      --#                                    LexTokenManager.State,
      --#                                    Output,
      --#                                    SparkFormatCommandLineData.Content,
      --#                                    TemporaryFile,
      --#                                    TheHeap,
      --#                                    This &
      --#         Statistics.TableUsage,
      --#         TheHeap               from *,
      --#                                    LexTokenManager.State,
      --#                                    SparkFormatCommandLineData.Content,
      --#                                    TheHeap,
      --#                                    This &
      --#         Success               from This &
      --#         TemporaryFile         from *,
      --#                                    This;

      procedure Finalise (This    : in out State;
                          TheHeap : in out Heap.HeapRecord);
      --# derives TheHeap,
      --#         This    from *,
      --#                      This;

   private
      type Relation_Type is record
         ModifierRel : LexTokenManager.Relation_Algebra.Relation;
         TypeRel     : LexTokenManager.Relation_Algebra.String.Relation;
         PropertyRel : LexTokenManager.Relation_Algebra.String.Relation;
      end record;

      type VariableModifier is (Unmoded, InMode, OutMode, InOutMode,
                                 ProtectedUnmoded, ProtectedIn, ProtectedOut,
                                 TaskModifier);
      type Modifier_Use is array (VariableModifier) of Boolean;

      type Statistics_Type is record
         StartCol            : EStrings.Positions;
         MaxModifierLength   : EStrings.Lengths;
         MaxPrimaryIdLength  : EStrings.Lengths;
         MaxTypeMarkLength   : EStrings.Lengths;
         ModifiersPresent    : Modifier_Use;
         TypeMarkPresent     : Boolean;
         PropertyListPresent : Boolean;
      end record;

      type State is record
         Anno             : Annotations.Anno_Type;
         Relations        : Relation_Type;
         ParseStats       : Statistics_Type;
         AllowDottedNames : Boolean;
         Success          : Boolean;
      end record;
   end Reformatter;

   package body WhiteSpace
   is

      function IsWhiteSpace (Char : Character) return Boolean
      is
      begin
         --Note that: Ada.Characters.Latin_1.HT is a tab.
         return Char = Ada.Characters.Latin_1.Space or Char = Ada.Characters.Latin_1.HT;
      end IsWhiteSpace;

      procedure Skip (InputLine : in     EStrings.T;
                      Index     : in out EStrings.Lengths)
      is
      begin
         loop
            exit when Index >= EStrings.Get_Length (E_Str => InputLine) or else
              not IsWhiteSpace (EStrings.Get_Element (E_Str => InputLine,
                                                      Pos   => Index));
            Index := Index + 1;
         end loop;
      end Skip;

   end WhiteSpace;

   package body File_IO
   is

      procedure Create (TemporaryFile : in out SPARK_IO.File_Type)
      is
         Unused : SPARK_IO.File_Status;
      begin
         --# accept Flow_Message, 10, Unused, "The variable Unused is not required" &
         --#        Flow_Message, 33, Unused, "The variable Unused is not required";
         SPARK_IO.Create (TemporaryFile, 0, "", "", Unused);
         -- Not sure what we can do if we can't create a temporary file
      end Create;

      procedure Reset (TemporaryFile : in out SPARK_IO.File_Type)
      is
         Unused : SPARK_IO.File_Status;
      begin
         --# accept Flow_Message, 10, Unused, "The variable Unused is not required" &
         --#        Flow_Message, 33, Unused, "The variable Unused is not required";
         SPARK_IO.Reset (TemporaryFile, SPARK_IO.In_File, Unused);
         -- not sure what we can do if we can't reset the file
      end Reset;

      procedure Delete (TemporaryFile : in out SPARK_IO.File_Type)
      is
         Unused : SPARK_IO.File_Status;
      begin
         --# accept Flow_Message, 10, Unused, "The variable Unused is not required" &
         --#        Flow_Message, 33, Unused, "The variable Unused is not required";
         SPARK_IO.Delete (TemporaryFile, Unused);
         -- not sure what we can do if we can't delete the file
      end Delete;

      procedure Lex (TemporaryFile : in     SPARK_IO.File_Type;
                     Token         :    out SPSymbols.SPTerminal;
                     LexVal        :    out LexTokenManager.Lex_Value)
      is
         Unused : Boolean;
      begin
         --# accept Flow_Message, 10, Unused, "The variable Unused is not required" &
         --#        Flow_Message, 33, Unused, "The variable Unused is not required";
         SparkLex.SPARKFormatLex (ProgText   => TemporaryFile,
                                  Token      => Token,
                                  LexVal     => LexVal,
                                  PunctToken => Unused);
      end Lex;

   end File_IO;

   procedure Copy (Input, Output : in SPARK_IO.File_Type)
   is
      InputLine : EStrings.T;
   begin
      while not SPARK_IO.End_Of_File (Input) loop
         EStrings.Get_Line (File  => Input,
                            E_Str => InputLine);
         EStrings.Put_Line (File  => Output,
                            E_Str => InputLine);
      end loop;
   end Copy;

   package body Iteration is separate;

   package body Annotations is separate;

   package body Reformatter is separate;

   procedure ReformatAnnotations (Input, Output : in SPARK_IO.File_Type)
   is

      type StateType is (ParsingSPARKCode, ParsingGlobalsList,
                         ParsingDerivesList, ParsingInheritClause,
                         ParsingOwnVariableClause,
                         ParsingRefinementDefinition,
                         ParsingInitializationSpecification);


      --# inherit Ada.Characters.Handling,
      --#         CommandLineData,
      --#         EStrings,
      --#         SPARK_IO,
      --#         WhiteSpace;
      package Annotation
      --# own StartString;
      is

         procedure Initialize;
         --# global in     CommandLineData.Content;
         --#           out StartString;
         --# derives StartString from CommandLineData.Content;

         procedure IsStart (InputLine : in     EStrings.T;
                            Index     : in out EStrings.Lengths;
                            OK        :    out Boolean);
         --# global in StartString;
         --# derives Index,
         --#         OK    from Index,
         --#                    InputLine,
         --#                    StartString;

         procedure Write (Output   : in SPARK_IO.File_Type;
                          StartCol : in EStrings.Positions);
         --# global in     StartString;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                Output,
         --#                                StartCol,
         --#                                StartString;

      end Annotation;

      --# inherit Ada.Characters.Handling,
      --#         Annotation,
      --#         EStrings,
      --#         WhiteSpace;
      package SPARKKeywordMatching
      is
         function Match (InputLine    : EStrings.T;
                         Index        : EStrings.Lengths;
                         MatchKeyword : String) return Boolean;

         function MatchAtStart (InputLine    : EStrings.T;
                                MatchKeyword : String) return Boolean;
         --# global in Annotation.StartString;
      end SPARKKeywordMatching;

      --# inherit Annotation,
      --#         EStrings,
      --#         SPARKKeywordMatching,
      --#         WhiteSpace;
      package AcceptList
      is
         function IsStart (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;
      end AcceptList;

      --# inherit Annotation,
      --#         EStrings,
      --#         SPARKKeywordMatching,
      --#         WhiteSpace;
      package EndList
      is
         function IsStart (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;
      end EndList;

      --# inherit Annotation,
      --#         EStrings,
      --#         SPARKKeywordMatching,
      --#         WhiteSpace;
      package ForList
      is
         function IsStart (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;
      end ForList;

      --# inherit Annotation,
      --#         EStrings,
      --#         SPARKKeywordMatching,
      --#         WhiteSpace;
      package FunctionList
      is
         function IsStart (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;
      end FunctionList;

      --# inherit Annotation,
      --#         EStrings,
      --#         SPARKKeywordMatching,
      --#         WhiteSpace;
      package TypeList
      is
         function IsStart (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;
      end TypeList;

      --# inherit Annotation,
      --#         EStrings,
      --#         SPARKKeywordMatching,
      --#         WhiteSpace;
      package AssertList
      is
         function IsStart (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;
      end AssertList;

      --# inherit Ada.Characters.Handling,
      --#         Annotation,
      --#         CommandLineData,
      --#         Dictionary,
      --#         ErrorHandler,
      --#         EStrings,
      --#         File_IO,
      --#         Heap,
      --#         LexTokenManager,
      --#         LexTokenManager.Relation_Algebra,
      --#         LexTokenManager.Relation_Algebra.String,
      --#         LexTokenManager.Seq_Algebra,
      --#         RelationAlgebra,
      --#         SeqAlgebra,
      --#         SparkFormatCommandLineData,
      --#         SPARKKeywordMatching,
      --#         SparkLex,
      --#         SPARKProgram,
      --#         SPARK_IO,
      --#         SPSymbols,
      --#         Statistics,
      --#         WhiteSpace;
      package DerivesList
      --# own State;
      is

         procedure Initialize;
         --# global in out LexTokenManager.State;
         --#           out State;
         --# derives LexTokenManager.State,
         --#         State                 from LexTokenManager.State;

         function IsStart (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;

         function IsEnd (InputLine : EStrings.T) return Boolean;

         procedure Parse (TemporaryFile : in out SPARK_IO.File_Type;
                          StartCol      :    out EStrings.Positions;
                          TheHeap       : in out Heap.HeapRecord;
                          Imports       :    out LexTokenManager.Seq_Algebra.Seq;
                          Exports       :    out LexTokenManager.Seq_Algebra.Seq;
                          Dependencies  :    out LexTokenManager.Relation_Algebra.String.Relation;
                          OK            :    out Boolean);
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     SparkFormatCommandLineData.Content;
         --#        in     State;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --#        in out Statistics.TableUsage;
         --#           out SparkLex.Currline;
         --# derives Dependencies,
         --#         Exports,
         --#         Imports                   from TheHeap &
         --#         ErrorHandler.ErrorContext,
         --#         LexTokenManager.State,
         --#         OK,
         --#         SparkLex.Currline,
         --#         SPARK_IO.File_Sys,
         --#         TheHeap                   from CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        LexTokenManager.State,
         --#                                        SparkFormatCommandLineData.Content,
         --#                                        SPARK_IO.File_Sys,
         --#                                        State,
         --#                                        TemporaryFile,
         --#                                        TheHeap &
         --#         StartCol                  from CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        LexTokenManager.State,
         --#                                        SPARK_IO.File_Sys,
         --#                                        TemporaryFile &
         --#         Statistics.TableUsage     from *,
         --#                                        CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        LexTokenManager.State,
         --#                                        SparkFormatCommandLineData.Content,
         --#                                        SPARK_IO.File_Sys,
         --#                                        State,
         --#                                        TemporaryFile,
         --#                                        TheHeap &
         --#         TemporaryFile             from *;

         procedure Reformat (OK            : in     Boolean;
                             TemporaryFile : in out SPARK_IO.File_Type;
                             Output        : in     SPARK_IO.File_Type;
                             StartCol      : in     EStrings.Positions;
                             TheHeap       : in out Heap.HeapRecord;
                             Exports       : in     LexTokenManager.Seq_Algebra.Seq;
                             Dependencies  : in     LexTokenManager.Relation_Algebra.String.Relation);
         --# global in     Annotation.StartString;
         --#        in     LexTokenManager.State;
         --#        in     SparkFormatCommandLineData.Content;
         --#        in     State;
         --#        in out SPARK_IO.File_Sys;
         --#        in out Statistics.TableUsage;
         --# derives SPARK_IO.File_Sys     from *,
         --#                                    Annotation.StartString,
         --#                                    Dependencies,
         --#                                    Exports,
         --#                                    LexTokenManager.State,
         --#                                    OK,
         --#                                    Output,
         --#                                    SparkFormatCommandLineData.Content,
         --#                                    StartCol,
         --#                                    State,
         --#                                    TemporaryFile,
         --#                                    TheHeap &
         --#         Statistics.TableUsage,
         --#         TheHeap               from *,
         --#                                    Dependencies,
         --#                                    Exports,
         --#                                    LexTokenManager.State,
         --#                                    OK,
         --#                                    SparkFormatCommandLineData.Content,
         --#                                    State,
         --#                                    TheHeap &
         --#         TemporaryFile         from *,
         --#                                    OK;

      end DerivesList;

      package Subprogram
      is
         type SubprogramType is (IsFunction, IsProcedure, IsUnknown);
      end Subprogram;

      --# inherit Ada.Characters.Handling,
      --#         Annotation,
      --#         CommandLineData,
      --#         DerivesList,
      --#         Dictionary,
      --#         ErrorHandler,
      --#         EStrings,
      --#         File_IO,
      --#         Heap,
      --#         LexTokenManager,
      --#         RelationAlgebra,
      --#         SeqAlgebra,
      --#         SparkFormatCommandLineData,
      --#         SPARKKeywordMatching,
      --#         SparkLex,
      --#         SPARKProgram,
      --#         SPARK_IO,
      --#         SPSymbols,
      --#         Statistics,
      --#         Subprogram,
      --#         WhiteSpace;
      package DeclareList
      is

         function IsStart (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;

         function IsEnd (InputLine : EStrings.T) return Boolean;
         pragma Unreferenced (IsEnd); -- for future use

      end DeclareList;

      --# inherit Ada.Characters.Handling,
      --#         Annotation,
      --#         CommandLineData,
      --#         DerivesList,
      --#         Dictionary,
      --#         ErrorHandler,
      --#         EStrings,
      --#         File_IO,
      --#         Heap,
      --#         LexTokenManager,
      --#         RelationAlgebra,
      --#         SeqAlgebra,
      --#         SparkFormatCommandLineData,
      --#         SPARKKeywordMatching,
      --#         SparkLex,
      --#         SPARKProgram,
      --#         SPARK_IO,
      --#         SPSymbols,
      --#         Statistics,
      --#         Subprogram,
      --#         WhiteSpace;
      package ProofList
      is

         function IsStart (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;

         function IsEnd (InputLine : EStrings.T) return Boolean;
         pragma Unreferenced (IsEnd); -- for future use

      end ProofList;

      --# inherit AcceptList,
      --#         Ada.Characters.Handling,
      --#         Annotation,
      --#         AssertList,
      --#         CommandLineData,
      --#         DeclareList,
      --#         DerivesList,
      --#         Dictionary,
      --#         EndList,
      --#         ErrorHandler,
      --#         EStrings,
      --#         File_IO,
      --#         ForList,
      --#         FunctionList,
      --#         Heap,
      --#         LexTokenManager,
      --#         LexTokenManager.Seq_Algebra,
      --#         ProofList,
      --#         RelationAlgebra,
      --#         SeqAlgebra,
      --#         SparkFormatCommandLineData,
      --#         SPARKKeywordMatching,
      --#         SparkLex,
      --#         SPARKProgram,
      --#         SPARK_IO,
      --#         SPSymbols,
      --#         Statistics,
      --#         Subprogram,
      --#         TypeList,
      --#         WhiteSpace;
      package GlobalsList
      --# own ConsecutiveEmptyLines;
      --# initializes ConsecutiveEmptyLines;
      is

         type Modes is (Unmoded, InMode, InOutMode, OutMode);
         type GlobalVariablesType is array (Modes) of LexTokenManager.Seq_Algebra.Seq;

         function IsStart (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;

         function IsEnd (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;

         procedure IncrementConsecutiveEmptyLines;
         --# global in out ConsecutiveEmptyLines;
         --# derives ConsecutiveEmptyLines from *;

         procedure ResetConsecutiveEmptyLines;
         --# global out ConsecutiveEmptyLines;
         --# derives ConsecutiveEmptyLines from ;

         function NoEmptyLines return Boolean;
         --# global in ConsecutiveEmptyLines;

         procedure Parse (TemporaryFile   : in out SPARK_IO.File_Type;
                          StartCol        :    out EStrings.Positions;
                          TheHeap         : in out Heap.HeapRecord;
                          GlobalVariables :    out GlobalVariablesType;
                          OK              :    out Boolean);
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --#        in out Statistics.TableUsage;
         --#           out SparkLex.Currline;
         --# derives ErrorHandler.ErrorContext,
         --#         LexTokenManager.State,
         --#         OK,
         --#         SparkLex.Currline,
         --#         SPARK_IO.File_Sys,
         --#         StartCol                  from CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        LexTokenManager.State,
         --#                                        SPARK_IO.File_Sys,
         --#                                        TemporaryFile &
         --#         GlobalVariables           from TheHeap &
         --#         Statistics.TableUsage,
         --#         TheHeap                   from *,
         --#                                        CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        LexTokenManager.State,
         --#                                        SPARK_IO.File_Sys,
         --#                                        TemporaryFile,
         --#                                        TheHeap &
         --#         TemporaryFile             from *;

         procedure Reformat (OK                   : in     Boolean;
                             TemporaryFile        : in out SPARK_IO.File_Type;
                             Output               : in     SPARK_IO.File_Type;
                             StartCol             : in     EStrings.Positions;
                             TheHeap              : in out Heap.HeapRecord;
                             FunctionOrProcedure  : in     Subprogram.SubprogramType;
                             GlobalVariables      : in     GlobalVariablesType;
                             Imports              : in     LexTokenManager.Seq_Algebra.Seq;
                             Exports              : in     LexTokenManager.Seq_Algebra.Seq);
         --# global in     Annotation.StartString;
         --#        in     LexTokenManager.State;
         --#        in     SparkFormatCommandLineData.Content;
         --#        in out ConsecutiveEmptyLines;
         --#        in out SPARK_IO.File_Sys;
         --#        in out Statistics.TableUsage;
         --# derives ConsecutiveEmptyLines from  &
         --#         SPARK_IO.File_Sys     from *,
         --#                                    Annotation.StartString,
         --#                                    ConsecutiveEmptyLines,
         --#                                    Exports,
         --#                                    FunctionOrProcedure,
         --#                                    GlobalVariables,
         --#                                    Imports,
         --#                                    LexTokenManager.State,
         --#                                    OK,
         --#                                    Output,
         --#                                    SparkFormatCommandLineData.Content,
         --#                                    StartCol,
         --#                                    TemporaryFile,
         --#                                    TheHeap &
         --#         Statistics.TableUsage,
         --#         TheHeap               from *,
         --#                                    Exports,
         --#                                    FunctionOrProcedure,
         --#                                    GlobalVariables,
         --#                                    Imports,
         --#                                    LexTokenManager.State,
         --#                                    OK,
         --#                                    SparkFormatCommandLineData.Content,
         --#                                    TheHeap &
         --#         TemporaryFile         from *,
         --#                                    OK;

      end GlobalsList;

      FunctionOrProcedure : Subprogram.SubprogramType;
      State               : StateType;
      InputLine           : EStrings.T;
      TheHeap             : Heap.HeapRecord;
      GlobalsListFile     : SPARK_IO.File_Type;
      DerivesListFile     : SPARK_IO.File_Type;
      InheritClauseFile   : SPARK_IO.File_Type;
      InitializeSpecFile  : SPARK_IO.File_Type;
      OwnVarClauseFile    : SPARK_IO.File_Type;
      GlobalVariables     : GlobalsList.GlobalVariablesType;
      Imports             : LexTokenManager.Seq_Algebra.Seq;
      Exports             : LexTokenManager.Seq_Algebra.Seq;
      Dependencies        : LexTokenManager.Relation_Algebra.String.Relation;
      StartCol            : EStrings.Positions;
      GlobalsListOK       : Boolean;
      DerivesListOK       : Boolean;
      InheritAnno         : Annotations.Anno_Type;
      InitializesAnno     : Annotations.Anno_Type;
      OwnVarAnno          : Annotations.Anno_Type;
      InheritClause       : Reformatter.State;
      InitializationSpec  : Reformatter.State;
      OwnVariableClause   : Reformatter.State;
      File_IO_Required    : Boolean;
      ReformatSuccessful  : Boolean;

      package body Annotation
      is

         StartString : EStrings.T;

         procedure Initialize
         is
            Unused : Boolean;
         begin
            StartString := EStrings.Copy_String (Str => "--");
            --# accept Flow_Message, 10, Unused, "The variable Unused is not required" &
            --#        Flow_Message, 33, Unused, "The variable Unused is not required";
            EStrings.Append_Char (E_Str   => StartString,
                                  Ch      => CommandLineData.Content.AnnoChar,
                                  Success => Unused);
            -- we know there's always room
         end Initialize;

         procedure IsStart (InputLine : in     EStrings.T;
                            Index     : in out EStrings.Lengths;
                            OK        :    out Boolean)
         is
         begin

            WhiteSpace.Skip (InputLine, Index);

            if EStrings.Get_Length (E_Str => InputLine) - Index >= 2 then
               if EStrings.Get_Element (E_Str => InputLine,
                                        Pos   => Index) =
                 EStrings.Get_Element (E_Str => StartString,
                                       Pos   => 1) and then
                 EStrings.Get_Element (E_Str => InputLine,
                                       Pos   => Index + 1) =
                 EStrings.Get_Element (E_Str => StartString,
                                       Pos   => 2) and then
                 EStrings.Get_Element (E_Str => InputLine,
                                       Pos   => Index + 2) =
                 EStrings.Get_Element (E_Str => StartString,
                                       Pos   => 3) then
                  Index := Index + 3;
                  OK := True;
               else
                  OK := False;
               end if;
            else
               OK := False;
            end if;

         end IsStart;

         procedure Write (Output   : in SPARK_IO.File_Type;
                          StartCol : in EStrings.Positions)
         is
         begin
            SPARK_IO.Set_Col (Output, StartCol);
            EStrings.Put_String (File  => Output,
                                 E_Str => StartString);
         end Write;

      end Annotation;

      package body SPARKKeywordMatching
      is

         -- InputLine    - A line from the spark input file.
         -- Index        - An index position, expected to inside InputLine.
         -- MatchKeyword - A keyword to match.
         --
         -- The function returns true where InputLine, at position index, contains
         -- the keyword MatchKeyword. For a match, the keyword must be terminated
         -- by a newline or white space.
         function Match (InputLine    : EStrings.T;
                         Index        : EStrings.Lengths;
                         MatchKeyword : String) return Boolean
         is
            IsMatch         : Boolean;
            CharFromLine    : Character;
            CharFromKeyword : Character;
         begin

            -- EStrings.Lengths'First is zero. This is used to describe empty strings.
            -- Can not find a match against an empty string.
            if Index = EStrings.Lengths'First then
               IsMatch := False;
            else
               IsMatch := True;
               for I in EStrings.Lengths range 1 .. MatchKeyword'Length loop
                  if (Index - 1) + I >= (EStrings.Get_Length (E_Str => InputLine) + 1) then
                     -- Reached or exceeded the end of line. Can not be a match.
                     IsMatch := False;
                     exit;
                  end if;

                  --# assert MatchKeyword=MatchKeyword% and
                  --#        Index > EStrings.Lengths'First and
                  --#        Index <= EStrings.Lengths'Last and
                  --#        I>=1 and I<=MatchKeyword'Length and
                  --#        (((Index - 1) + I) >= EStrings.Lengths'First) and
                  --#        (((Index - 1) + I) <= EStrings.Lengths'Last);

                  CharFromLine := Ada.Characters.Handling.To_Lower
                    (EStrings.Get_Element (E_Str => InputLine,
                                           Pos   => (Index - 1) + I));
                  CharFromKeyword := Ada.Characters.Handling.To_Lower (MatchKeyword (I));
                  if not (CharFromLine = CharFromKeyword) then
                     -- Are not the same at this index. Can not be a match.
                     IsMatch := False;
                     exit;
                  end if;
               end loop;

               -- If detected a keyword match, then investigate what appears after
               -- the keyword.
               if IsMatch then
                     -- If not terminated by line feed, is only a match if followed by
                     -- white space.
                  if ((Index + MatchKeyword'Length) <= EStrings.Get_Length (E_Str => InputLine)) then
                     IsMatch := WhiteSpace.IsWhiteSpace
                       (EStrings.Get_Element (E_Str => InputLine,
                                              Pos   => Index + MatchKeyword'Length));
                  end if;
               end if;
            end if;

            return IsMatch;
         end Match;

         -- InputLine    - A line from the spark input file.
         -- MatchKeyword - A sparkkwyword to match.
         --
         -- The function returns true where InputLine is an annotation line, starting
         -- with the keyword MatchKeyword.
         function MatchAtStart (InputLine    : EStrings.T;
                                MatchKeyword : String) return Boolean
         is
            Index : EStrings.Lengths;
            OK    : Boolean;
         begin
            if EStrings.Get_Length (E_Str => InputLine) > 0 then
               Index := 1;
               Annotation.IsStart (InputLine, Index, OK);
               if OK then
                  WhiteSpace.Skip (InputLine, Index);
                  OK := Match (InputLine, Index, MatchKeyword);
               end if;
            else
               OK := False;
            end if;

            return OK;
         end MatchAtStart;

      end SPARKKeywordMatching;

      package body AcceptList
      is
         Keyword : constant String := "accept";

         -- checks it's at the start of a line up to "--# accept"
         function IsStart (InputLine : EStrings.T) return Boolean
         is
         begin
            return SPARKKeywordMatching.MatchAtStart (InputLine, Keyword);
         end IsStart;

      end AcceptList;

      package body EndList
      is
         Keyword : constant String := "end";

         -- checks it's at the start of a line up to "--# end"
         function IsStart (InputLine : EStrings.T) return Boolean
         is
         begin
            return SPARKKeywordMatching.MatchAtStart (InputLine, Keyword);
         end IsStart;

      end EndList;

      package body ForList
      is
         Keyword : constant String := "for";

         -- checks it's at the start of a line up to "--# for"
         function IsStart (InputLine : EStrings.T) return Boolean
         is
         begin
            return SPARKKeywordMatching.MatchAtStart (InputLine, Keyword);
         end IsStart;

      end ForList;

      package body FunctionList
      is
         Keyword : constant String := "function";

         -- checks it's at the start of a line up to "--# function"
         function IsStart (InputLine : EStrings.T) return Boolean
         is
         begin
            return SPARKKeywordMatching.MatchAtStart (InputLine, Keyword);
         end IsStart;

      end FunctionList;

      package body TypeList
      is
         Keyword : constant String := "type";

         -- checks it's at the start of a line up to "--# type"
         function IsStart (InputLine : EStrings.T) return Boolean
         is
         begin
            return SPARKKeywordMatching.MatchAtStart (InputLine, Keyword);
         end IsStart;

      end TypeList;

      package body AssertList
      is
         Keyword : constant String := "assert";

         -- checks it's at the start of a line up to "--# assert"
         function IsStart (InputLine : EStrings.T) return Boolean
         is
         begin
            return SPARKKeywordMatching.MatchAtStart (InputLine, Keyword);
         end IsStart;

      end AssertList;

      package body DerivesList
      --# own State is MultiplyToken, RWNullToken;
      is
         Keyword : constant String := "derives";

         MultiplyToken, RWNullToken : LexTokenManager.Lex_String;

         procedure Initialize
         --# global in out LexTokenManager.State;
         --#           out MultiplyToken;
         --#           out RWNullToken;
         --# derives LexTokenManager.State,
         --#         MultiplyToken,
         --#         RWNullToken           from LexTokenManager.State;
         is
         begin
            LexTokenManager.Insert_Examiner_String (Str     => EStrings.Copy_String (Str => "*"),
                                                    Lex_Str => MultiplyToken);
            LexTokenManager.Insert_Examiner_String (Str     => EStrings.Copy_String (Str => "null"),
                                                    Lex_Str => RWNullToken);

         end Initialize;

         -- checks it's at the start of a line up to "--# derives"
         function IsStart (InputLine : EStrings.T) return Boolean
         is
         begin
            return SPARKKeywordMatching.MatchAtStart (InputLine, Keyword);
         end IsStart;

         -- checks for ";" at end of line
         function IsEnd (InputLine : EStrings.T) return Boolean
         is
            Result : Boolean;
         begin

            Result := False;

            for I in EStrings.Lengths range EStrings.Positions'First .. EStrings.Get_Length (E_Str => InputLine) loop
               if EStrings.Get_Element (E_Str => InputLine,
                                        Pos   => I) = ';' then
                  Result := True;
                  exit;
               end if;
            end loop;

            return Result;

         end IsEnd;

         procedure Parse (TemporaryFile : in out SPARK_IO.File_Type;
                          StartCol      :    out EStrings.Positions;
                          TheHeap       : in out Heap.HeapRecord;
                          Imports       :    out LexTokenManager.Seq_Algebra.Seq;
                          Exports       :    out LexTokenManager.Seq_Algebra.Seq;
                          Dependencies  :    out LexTokenManager.Relation_Algebra.String.Relation;
                          OK            :    out Boolean)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     MultiplyToken;
         --#        in     RWNullToken;
         --#        in     SparkFormatCommandLineData.Content;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --#        in out Statistics.TableUsage;
         --#           out SparkLex.Currline;
         --# derives Dependencies,
         --#         Exports,
         --#         Imports                   from TheHeap &
         --#         ErrorHandler.ErrorContext,
         --#         LexTokenManager.State,
         --#         OK,
         --#         SparkLex.Currline,
         --#         SPARK_IO.File_Sys,
         --#         TheHeap                   from CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        LexTokenManager.State,
         --#                                        MultiplyToken,
         --#                                        RWNullToken,
         --#                                        SparkFormatCommandLineData.Content,
         --#                                        SPARK_IO.File_Sys,
         --#                                        TemporaryFile,
         --#                                        TheHeap &
         --#         StartCol                  from CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        LexTokenManager.State,
         --#                                        SPARK_IO.File_Sys,
         --#                                        TemporaryFile &
         --#         Statistics.TableUsage     from *,
         --#                                        CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        LexTokenManager.State,
         --#                                        MultiplyToken,
         --#                                        RWNullToken,
         --#                                        SparkFormatCommandLineData.Content,
         --#                                        SPARK_IO.File_Sys,
         --#                                        TemporaryFile,
         --#                                        TheHeap &
         --#         TemporaryFile             from *;
         is

            Token       : SPSymbols.SPTerminal;
            LexVal      : LexTokenManager.Lex_Value;

            procedure ParseDependencyClauses (TemporaryFile : in     SPARK_IO.File_Type;
                                              Token         : in out SPSymbols.SPTerminal;
                                              LexVal        : in     LexTokenManager.Lex_Value;
                                              TheHeap       : in out Heap.HeapRecord;
                                              Exports       : in     LexTokenManager.Seq_Algebra.Seq;
                                              Imports       : in     LexTokenManager.Seq_Algebra.Seq;
                                              Dependencies  : in     LexTokenManager.Relation_Algebra.String.Relation;
                                              OK            :    out Boolean)
            --# global in     CommandLineData.Content;
            --#        in     Dictionary.Dict;
            --#        in     MultiplyToken;
            --#        in     RWNullToken;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out ErrorHandler.ErrorContext;
            --#        in out LexTokenManager.State;
            --#        in out SparkLex.Currline;
            --#        in out SPARK_IO.File_Sys;
            --#        in out Statistics.TableUsage;
            --# derives ErrorHandler.ErrorContext,
            --#         LexTokenManager.State,
            --#         SparkLex.Currline,
            --#         SPARK_IO.File_Sys,
            --#         Statistics.TableUsage,
            --#         TheHeap,
            --#         Token                     from *,
            --#                                        CommandLineData.Content,
            --#                                        Dependencies,
            --#                                        Dictionary.Dict,
            --#                                        ErrorHandler.ErrorContext,
            --#                                        Exports,
            --#                                        Imports,
            --#                                        LexTokenManager.State,
            --#                                        LexVal,
            --#                                        MultiplyToken,
            --#                                        RWNullToken,
            --#                                        SparkFormatCommandLineData.Content,
            --#                                        SparkLex.Currline,
            --#                                        SPARK_IO.File_Sys,
            --#                                        TemporaryFile,
            --#                                        TheHeap,
            --#                                        Token &
            --#         OK                        from CommandLineData.Content,
            --#                                        Dependencies,
            --#                                        Dictionary.Dict,
            --#                                        ErrorHandler.ErrorContext,
            --#                                        Exports,
            --#                                        Imports,
            --#                                        LexTokenManager.State,
            --#                                        LexVal,
            --#                                        MultiplyToken,
            --#                                        RWNullToken,
            --#                                        SparkFormatCommandLineData.Content,
            --#                                        SparkLex.Currline,
            --#                                        SPARK_IO.File_Sys,
            --#                                        TemporaryFile,
            --#                                        TheHeap,
            --#                                        Token;
            is

               SimilarExports       : LexTokenManager.Seq_Algebra.Seq;
               ImportsForThisExport : LexTokenManager.Seq_Algebra.Seq;
               NextLexVal           : LexTokenManager.Lex_Value;
               SomeExports          : Boolean;

               procedure ParseExportList (TemporaryFile  : in     SPARK_IO.File_Type;
                                          Token          : in out SPSymbols.SPTerminal;
                                          LexVal         : in out LexTokenManager.Lex_Value;
                                          TheHeap        : in out Heap.HeapRecord;
                                          Exports        : in     LexTokenManager.Seq_Algebra.Seq;
                                          SimilarExports : in     LexTokenManager.Seq_Algebra.Seq;
                                          OK             :    out Boolean)
               --# global in     CommandLineData.Content;
               --#        in     Dictionary.Dict;
               --#        in     RWNullToken;
               --#        in     SparkFormatCommandLineData.Content;
               --#        in out ErrorHandler.ErrorContext;
               --#        in out LexTokenManager.State;
               --#        in out SparkLex.Currline;
               --#        in out SPARK_IO.File_Sys;
               --#        in out Statistics.TableUsage;
               --# derives ErrorHandler.ErrorContext,
               --#         LexTokenManager.State,
               --#         LexVal,
               --#         OK,
               --#         SparkLex.Currline,
               --#         SPARK_IO.File_Sys,
               --#         Token                     from CommandLineData.Content,
               --#                                        Dictionary.Dict,
               --#                                        ErrorHandler.ErrorContext,
               --#                                        LexTokenManager.State,
               --#                                        LexVal,
               --#                                        SparkFormatCommandLineData.Content,
               --#                                        SparkLex.Currline,
               --#                                        SPARK_IO.File_Sys,
               --#                                        TemporaryFile,
               --#                                        Token &
               --#         Statistics.TableUsage,
               --#         TheHeap                   from *,
               --#                                        CommandLineData.Content,
               --#                                        Dictionary.Dict,
               --#                                        ErrorHandler.ErrorContext,
               --#                                        Exports,
               --#                                        LexTokenManager.State,
               --#                                        LexVal,
               --#                                        RWNullToken,
               --#                                        SimilarExports,
               --#                                        SparkFormatCommandLineData.Content,
               --#                                        SparkLex.Currline,
               --#                                        SPARK_IO.File_Sys,
               --#                                        TemporaryFile,
               --#                                        TheHeap,
               --#                                        Token;
               is
                  DottedSimpleName  : EStrings.T;
                  Export            : LexTokenManager.Lex_String;
                  NoExportsExcluded : Boolean;
               begin

                  OK := True;

                  --  CFR1753: Check to see if an export has been excluded.
                  NoExportsExcluded := EStrings.Is_Empty
                     (E_Str => SparkFormatCommandLineData.Content.ExcludeExport);

                  --  if NoExportsExcluded then
                  --     SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                  --        "ParseExportList: No Exclude Export Specified",
                  --        0);
                  --  else
                  --    SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                  --       "ParseExportList: Excluded Export specified",
                  --       0);
                  -- end if;


                  case Token is
                     when SPSymbols.identifier =>
                        loop
                           DottedSimpleName := LexTokenManager.Lex_String_To_String (Lex_Str => LexVal.Token_Str);
                           File_IO.Lex (TemporaryFile, Token, LexVal);
                           loop
                              exit when Token /= SPSymbols.point;
                              EStrings.Append_String (E_Str => DottedSimpleName,
                                                      Str   => ".");
                              File_IO.Lex (TemporaryFile, Token, LexVal);
                              if Token = SPSymbols.identifier then
                                 EStrings.Append_Examiner_String
                                   (E_Str1 => DottedSimpleName,
                                    E_Str2 => LexTokenManager.Lex_String_To_String (Lex_Str => LexVal.Token_Str));
                                 File_IO.Lex (TemporaryFile, Token, LexVal);
                              else
                                 OK := False;
                              end if;
                              exit when not OK;
                           end loop;
                           exit when not OK;

                           --  CFR 1753: Check to see if this export has to be
                           --  excluded.
                           if NoExportsExcluded or else not EStrings.Eq_String
                             (E_Str1 => DottedSimpleName,
                              E_Str2 => SparkFormatCommandLineData.Content.ExcludeExport)
                           then
                              --  This export is not excluded.
                              --  SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                              --     "ParseExportList: Export is included:",
                              --     0);
                              --  EStrings.PutLine (SPARK_IO.Standard_Output,
                              --     DottedSimpleName);
                              LexTokenManager.Insert_Examiner_String (Str     => DottedSimpleName,
                                                                      Lex_Str => Export);
                              LexTokenManager.Seq_Algebra.Add_Member (The_Heap    => TheHeap,
                                                                      S           => Exports,
                                                                      Given_Value => Export);
                              LexTokenManager.Seq_Algebra.Add_Member (The_Heap    => TheHeap,
                                                                      S           => SimilarExports,
                                                                      Given_Value => Export);
                           --  else
                           --     SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                           --        "ParseExportList: Export is excluded:",
                           --        0);
                           --     EStrings.PutLine (SPARK_IO.Standard_Output,
                           --        DottedSimpleName);
                           end if;
                           exit when Token = SPSymbols.RWfrom;
                           if Token = SPSymbols.comma then
                              File_IO.Lex (TemporaryFile, Token, LexVal);
                              if Token /= SPSymbols.identifier then
                                 OK := False;
                              end if;
                           else
                              OK := False;
                           end if;
                           exit when not OK;
                        end loop;
                     when SPSymbols.RWnull =>
                        LexTokenManager.Seq_Algebra.Add_Member (The_Heap    => TheHeap,
                                                                S           => Exports,
                                                                Given_Value => RWNullToken);
                        LexTokenManager.Seq_Algebra.Add_Member (The_Heap    => TheHeap,
                                                                S           => SimilarExports,
                                                                Given_Value => RWNullToken);
                        File_IO.Lex (TemporaryFile, Token, LexVal);
                     when SPSymbols.semicolon =>
                        null;
                     when others =>
                        OK := False;
                  end case;

               end ParseExportList;

               procedure ParseImportList (TemporaryFile        : in     SPARK_IO.File_Type;
                                          Token                : in out SPSymbols.SPTerminal;
                                          LexVal               : in out LexTokenManager.Lex_Value;
                                          TheHeap              : in out Heap.HeapRecord;
                                          SimilarExports       : in     LexTokenManager.Seq_Algebra.Seq;
                                          Imports              : in     LexTokenManager.Seq_Algebra.Seq;
                                          ImportsForThisExport : in     LexTokenManager.Seq_Algebra.Seq;
                                          OK                   :    out Boolean)
               --# global in     CommandLineData.Content;
               --#        in     Dictionary.Dict;
               --#        in     MultiplyToken;
               --#        in out ErrorHandler.ErrorContext;
               --#        in out LexTokenManager.State;
               --#        in out SparkLex.Currline;
               --#        in out SPARK_IO.File_Sys;
               --#        in out Statistics.TableUsage;
               --# derives ErrorHandler.ErrorContext,
               --#         LexTokenManager.State,
               --#         LexVal,
               --#         OK,
               --#         SparkLex.Currline,
               --#         SPARK_IO.File_Sys,
               --#         Token                     from CommandLineData.Content,
               --#                                        Dictionary.Dict,
               --#                                        ErrorHandler.ErrorContext,
               --#                                        LexTokenManager.State,
               --#                                        LexVal,
               --#                                        SimilarExports,
               --#                                        SparkLex.Currline,
               --#                                        SPARK_IO.File_Sys,
               --#                                        TemporaryFile,
               --#                                        TheHeap,
               --#                                        Token &
               --#         Statistics.TableUsage,
               --#         TheHeap                   from *,
               --#                                        CommandLineData.Content,
               --#                                        Dictionary.Dict,
               --#                                        ErrorHandler.ErrorContext,
               --#                                        Imports,
               --#                                        ImportsForThisExport,
               --#                                        LexTokenManager.State,
               --#                                        LexVal,
               --#                                        MultiplyToken,
               --#                                        SimilarExports,
               --#                                        SparkLex.Currline,
               --#                                        SPARK_IO.File_Sys,
               --#                                        TemporaryFile,
               --#                                        TheHeap,
               --#                                        Token;
               is
                  DottedSimpleName : EStrings.T;
                  Export           : LexTokenManager.Seq_Algebra.Member_Of_Seq;
                  ImportName       : LexTokenManager.Lex_String;
                  SomeExports      : Boolean;
               begin

                  OK := True;

                  --  CFR1753: There may be no exports in the export list if an
                  --  excluded export was the only variable in the list.
                  SomeExports := not LexTokenManager.Seq_Algebra.Is_Empty_Seq (The_Heap => TheHeap,
                                                                               S        => SimilarExports);
                  --  if SomeExports then
                  --     SPARK_IO.Put_line (SPARK_IO.Standard_Output,
                  --                  "ParseImportList: There are some exports",
                  --                  0);
                  --  end if;


                  loop
                     exit when Token = SPSymbols.semicolon or Token = SPSymbols.ampersand;
                     case Token is
                        when SPSymbols.identifier =>
                           DottedSimpleName := LexTokenManager.Lex_String_To_String (Lex_Str => LexVal.Token_Str);
                           File_IO.Lex (TemporaryFile, Token, LexVal);
                           loop
                              exit when Token /= SPSymbols.point;
                              EStrings.Append_String (E_Str => DottedSimpleName,
                                                      Str   => ".");
                              File_IO.Lex (TemporaryFile, Token, LexVal);
                              if Token = SPSymbols.identifier then
                                 EStrings.Append_Examiner_String
                                   (E_Str1 => DottedSimpleName,
                                    E_Str2 => LexTokenManager.Lex_String_To_String (Lex_Str => LexVal.Token_Str));
                                 File_IO.Lex (TemporaryFile, Token, LexVal);
                              else
                                 OK := False;
                              end if;
                              exit when not OK;
                           end loop;
                           --  CFR1753: Ensure that the export list is non-empty
                           --  Otherwise do not add imports to the import sets.
                           if OK and SomeExports then
                              --  SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                              --     "ParseImporList: Adding import:",
                              --     0);
                              --  EStrings.PutLine (SPARK_IO.Standard_Output,
                              --     DottedSimpleName);

                              LexTokenManager.Insert_Examiner_String (Str     => DottedSimpleName,
                                                                      Lex_Str => ImportName);
                              LexTokenManager.Seq_Algebra.Add_Member (The_Heap    => TheHeap,
                                                                      S           => Imports,
                                                                      Given_Value => ImportName);
                              LexTokenManager.Seq_Algebra.Add_Member (The_Heap    => TheHeap,
                                                                      S           => ImportsForThisExport,
                                                                      Given_Value => ImportName);
                           --  else
                           --     SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                           --        "ParseImporList: discarding import:",
                           --        0);
                           --     EStrings.PutLine (SPARK_IO.Standard_Output,
                           --        DottedSimpleName);
                           end if;
                        when SPSymbols.multiply =>
                           --  CFR1753: Ensure that the export list is non-empty
                           --  Otherwise do not add imports to the import sets.
                           --# accept F, 41, "Used to simplify loop structure";
                           if SomeExports then
                              Export := LexTokenManager.Seq_Algebra.First_Member (The_Heap => TheHeap,
                                                                                  S        => SimilarExports);
                              loop
                                 exit when LexTokenManager.Seq_Algebra.Is_Null_Member (M => Export);
                                 --  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                                 --     "ParseImporList: Adding import from *:",
                                 --     0);
                                 --  EStrings.PutLine (SPARK_IO.Standard_Output,
                                 --  DottedSimpleName);
                                 LexTokenManager.Seq_Algebra.Add_Member
                                   (The_Heap    => TheHeap,
                                    S           => Imports,
                                    Given_Value => LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                                                M        => Export));
                                 Export := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                                    M        => Export);
                              end loop;
                              LexTokenManager.Seq_Algebra.Add_Member (The_Heap    => TheHeap,
                                                                      S           => ImportsForThisExport,
                                                                      Given_Value => MultiplyToken);
                           --  else
                              --  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                              --     "ParseImporList: No exports discarding *",
                              --     0);
                           end if;
                           --# end accept;
                           File_IO.Lex (TemporaryFile, Token, LexVal);
                        when SPSymbols.comma =>
                           null;
                        when others =>
                           OK := False;
                     end case;
                     exit when not OK or Token = SPSymbols.semicolon or Token = SPSymbols.ampersand;
                     if Token = SPSymbols.comma then
                        File_IO.Lex (TemporaryFile, Token, LexVal);
                     else
                        OK := False;
                     end if;
                  end loop;

               end ParseImportList;

               procedure StoreDependencies (TheHeap      : in out Heap.HeapRecord;
                                            Exports      : in     LexTokenManager.Seq_Algebra.Seq;
                                            Imports      : in     LexTokenManager.Seq_Algebra.Seq;
                                            Dependencies : in     LexTokenManager.Relation_Algebra.String.Relation)
               --# global in     LexTokenManager.State;
               --#        in     MultiplyToken;
               --#        in out Statistics.TableUsage;
               --# derives Statistics.TableUsage,
               --#         TheHeap               from *,
               --#                                    Dependencies,
               --#                                    Exports,
               --#                                    Imports,
               --#                                    LexTokenManager.State,
               --#                                    MultiplyToken,
               --#                                    TheHeap;
               is
                  Export, Import       : LexTokenManager.Seq_Algebra.Member_Of_Seq;
                  ExportVal, ImportVal : LexTokenManager.Lex_String;
               begin
                  Export := LexTokenManager.Seq_Algebra.First_Member (The_Heap => TheHeap,
                                                                      S        => Exports);
                  loop
                     exit when LexTokenManager.Seq_Algebra.Is_Null_Member (M => Export);
                     ExportVal := LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                               M        => Export);
                     Import := LexTokenManager.Seq_Algebra.First_Member (The_Heap => TheHeap,
                                                                         S        => Imports);
                     loop
                        exit when LexTokenManager.Seq_Algebra.Is_Null_Member (M => Import);
                        ImportVal := LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                                  M        => Import);
                        if LexTokenManager.Lex_String_Case_Insensitive_Compare
                          (Lex_Str1 => ExportVal,
                           Lex_Str2 => ImportVal) = LexTokenManager.Str_Eq then
                           ImportVal := MultiplyToken;
                        end if;
                        LexTokenManager.Relation_Algebra.String.Insert_Pair (The_Heap => TheHeap,
                                                                             R        => Dependencies,
                                                                             I        => ExportVal,
                                                                             J        => ImportVal);
                        Import := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                           M        => Import);
                     end loop;
                     Export := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                        M        => Export);
                  end loop;
               end StoreDependencies;

            begin

               OK := True;
               NextLexVal := LexVal;

               loop
                  exit when Token = SPSymbols.semicolon;
                  LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                          S        => SimilarExports);
                  ParseExportList (TemporaryFile  => TemporaryFile,
                                   Token          => Token,
                                   LexVal         => NextLexVal,
                                   TheHeap        => TheHeap,
                                   Exports        => Exports,
                                   SimilarExports => SimilarExports,
                                   OK             => OK);
                  if OK then
                     --  CFR1753: There may be no exports in the export list if an
                     --  excluded export was the only variable in the list,
                     --  however, we still need to parse the input list.
                     SomeExports := not LexTokenManager.Seq_Algebra.Is_Empty_Seq (The_Heap => TheHeap,
                                                                                  S        => SimilarExports);
                     --  if SomeExports then
                     --     SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                     --        "ParseDependencies: There is at least one export",
                     --        0);
                     --  else
                     --     SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                     --        "ParseDependencies: There are no exports associated with this import list",
                     --        0);
                     -- end if;

                     if Token = SPSymbols.RWfrom then
                        File_IO.Lex (TemporaryFile, Token, NextLexVal);
                        LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                                S        => ImportsForThisExport);
                        ParseImportList (TemporaryFile        => TemporaryFile,
                                         Token                => Token,
                                         LexVal               => NextLexVal,
                                         TheHeap              => TheHeap,
                                         SimilarExports       => SimilarExports,
                                         Imports              => Imports,
                                         ImportsForThisExport => ImportsForThisExport,
                                         OK                   => OK);
                        --  CFR1753: Only add imports and store the dependencies
                        --  if there are some exported variables associated
                        --  this import list.
                        if SomeExports then
                           --  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                           --     "ParseDependencies: Adding imports associated with this non-null import list",
                           --     0);
                           LexTokenManager.Seq_Algebra.Augment_Seq (The_Heap => TheHeap,
                                                                    A        => Imports,
                                                                    B        => ImportsForThisExport);
                           StoreDependencies (TheHeap      => TheHeap,
                                              Exports      => SimilarExports,
                                              Imports      => ImportsForThisExport,
                                              Dependencies => Dependencies);
                        -- else
                           --  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                           --     "ParseDependencies: Null export list - discard imports",
                           --     0);
                        end if;
                        LexTokenManager.Seq_Algebra.Dispose_Of_Seq (The_Heap => TheHeap,
                                                                    S        => ImportsForThisExport);
                        if OK then
                           case Token is
                              when SPSymbols.ampersand =>
                                 File_IO.Lex (TemporaryFile, Token, NextLexVal);
                              when SPSymbols.semicolon =>
                                 null;
                              when others =>
                                 OK := False;
                           end case;
                        end if;
                     else
                        OK := False;
                     end if;
                  end if;
                  LexTokenManager.Seq_Algebra.Dispose_Of_Seq (The_Heap => TheHeap,
                                                              S        => SimilarExports);
                  exit when not OK;
               end loop;

            end ParseDependencyClauses;

         begin

            File_IO.Reset (TemporaryFile);
            SparkLex.ClearLineContext;
            LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                    S        => Exports);
            LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                    S        => Imports);
            LexTokenManager.Relation_Algebra.String.Create_Relation (The_Heap => TheHeap,
                                                                     R        => Dependencies);
            File_IO.Lex (TemporaryFile, Token, LexVal);

            if Token = SPSymbols.annotation_start then
               StartCol := LexVal.Position.Start_Pos; -- we know StartCol can't be zero because the line can't be empty
               --# accept Flow_Message, 10, LexVal, "The next symbol should be 'derives', so we don't need the LexVal";
               File_IO.Lex (TemporaryFile, Token, LexVal);
               --# end accept;
               if Token = SPSymbols.RWderives then
                  File_IO.Lex (TemporaryFile, Token, LexVal);
                  ParseDependencyClauses (TemporaryFile => TemporaryFile,
                                          Token         => Token,
                                          LexVal        => LexVal,
                                          TheHeap       => TheHeap,
                                          Exports       => Exports,
                                          Imports       => Imports,
                                          Dependencies  => Dependencies,
                                          OK            => OK);
                  OK := OK and Token = SPSymbols.semicolon;
               else
                  OK := False;
               end if;
            else
               OK := False;
            end if;
            --# accept Flow_Message, 602, StartCol, StartCol, "StartCol may not be set if not OK";
         end Parse;

         procedure Reformat (OK            : in     Boolean;
                             TemporaryFile : in out SPARK_IO.File_Type;
                             Output        : in     SPARK_IO.File_Type;
                             StartCol      : in     EStrings.Positions;
                             TheHeap       : in out Heap.HeapRecord;
                             Exports       : in     LexTokenManager.Seq_Algebra.Seq;
                             Dependencies  : in     LexTokenManager.Relation_Algebra.String.Relation)
         --# global in     Annotation.StartString;
         --#        in     LexTokenManager.State;
         --#        in     MultiplyToken;
         --#        in     RWNullToken;
         --#        in     SparkFormatCommandLineData.Content;
         --#        in out SPARK_IO.File_Sys;
         --#        in out Statistics.TableUsage;
         --# derives SPARK_IO.File_Sys     from *,
         --#                                    Annotation.StartString,
         --#                                    Dependencies,
         --#                                    Exports,
         --#                                    LexTokenManager.State,
         --#                                    MultiplyToken,
         --#                                    OK,
         --#                                    Output,
         --#                                    RWNullToken,
         --#                                    SparkFormatCommandLineData.Content,
         --#                                    StartCol,
         --#                                    TemporaryFile,
         --#                                    TheHeap &
         --#         Statistics.TableUsage,
         --#         TheHeap               from *,
         --#                                    Dependencies,
         --#                                    Exports,
         --#                                    LexTokenManager.State,
         --#                                    MultiplyToken,
         --#                                    OK,
         --#                                    RWNullToken,
         --#                                    SparkFormatCommandLineData.Content,
         --#                                    TheHeap &
         --#         TemporaryFile         from *,
         --#                                    OK;
         is

            FirstExport : Boolean;
            ExportWidth : EStrings.Lengths;

            function DerivesCol (StartCol : EStrings.Positions) return EStrings.Positions
            is
            begin
               return StartCol + 4; -- length of "--# "
            end DerivesCol;

            -- writes "derives " in correct position
            procedure WriteDerives (Output   : in SPARK_IO.File_Type;
                                    StartCol : in EStrings.Positions)
            --# global in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Output,
            --#                                SparkFormatCommandLineData.Content,
            --#                                StartCol;
            is
            begin
               SPARK_IO.Set_Col (Output, DerivesCol (StartCol));
               SPARK_IO.Put_String (Output, "derives", 0);
               -- If ExportIndent is not Inline, then don't put out a trailing space,
               -- since this will be rejected by GNAT -gnatyb style check mode.
               if SparkFormatCommandLineData.Content.ExportIndent = SparkFormatCommandLineData.Inline then
                  SPARK_IO.Put_String (Output, " ", 0);
               end if;
            end WriteDerives;

            function ExportCol (StartCol : EStrings.Positions) return EStrings.Positions
            --# global in SparkFormatCommandLineData.Content;
            is
               ExportColumn : EStrings.Positions;
            begin
               if SparkFormatCommandLineData.Content.ExportIndent = SparkFormatCommandLineData.Inline then
                  ExportColumn := DerivesCol (StartCol) + 8; -- length of "derives "
               else
                  ExportColumn := DerivesCol (StartCol) + (SparkFormatCommandLineData.Content.ExportIndent - 1);
               end if;
               return ExportColumn;
            end ExportCol;

            -- only called when there is a value for separator indent, returns starting column
            function SeparatorCol (StartCol : EStrings.Positions) return EStrings.Positions
            --# global in SparkFormatCommandLineData.Content;
            is
            begin
               return DerivesCol (StartCol) + (SparkFormatCommandLineData.Content.SeparatorIndent - 1); -- StartCol "is --# " but indent value is given from "--#"
            end SeparatorCol;

            procedure WriteExport (FirstExport : in out Boolean;
                                   Output      : in     SPARK_IO.File_Type;
                                   StartCol    : in     EStrings.Positions;
                                   TheHeap     : in     Heap.HeapRecord;
                                   Export      : in     LexTokenManager.Seq_Algebra.Member_Of_Seq)
            --# global in     Annotation.StartString;
            --#        in     LexTokenManager.State;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --# derives FirstExport       from * &
            --#         SPARK_IO.File_Sys from *,
            --#                                Annotation.StartString,
            --#                                Export,
            --#                                FirstExport,
            --#                                LexTokenManager.State,
            --#                                Output,
            --#                                SparkFormatCommandLineData.Content,
            --#                                StartCol,
            --#                                TheHeap;
            is
            begin
               if FirstExport then
                  FirstExport := False;
                  if SparkFormatCommandLineData.Content.ExportIndent /= SparkFormatCommandLineData.Inline then
                     Annotation.Write (Output, StartCol);
                  end if;
               else
                  if SparkFormatCommandLineData.Content.SeparatorIndent = SparkFormatCommandLineData.Inline then
                     SPARK_IO.Put_Line (Output, " &", 0);
                  else
                     Annotation.Write (Output, StartCol);
                     SPARK_IO.Set_Col (Output, SeparatorCol (StartCol));
                     SPARK_IO.Put_Line (Output, "&", 0);
                  end if;
                  Annotation.Write (Output, StartCol);
               end if;
               SPARK_IO.Set_Col (Output, ExportCol (StartCol));
               EStrings.Put_String
                 (File  => Output,
                  E_Str => LexTokenManager.Lex_String_To_String
                    (Lex_Str => LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                             M        => Export)));
            end WriteExport;

            function LongestExport (TheHeap : Heap.HeapRecord;
                                    Exports : LexTokenManager.Seq_Algebra.Seq) return EStrings.Lengths
            --# global in LexTokenManager.State;
            is
               MaxLength : Natural;
               Export    : LexTokenManager.Seq_Algebra.Member_Of_Seq;

               function Max (A, B : Natural) return Natural
               is
                  Result : Natural;
               begin

                  if A >= B then
                     Result := A;
                  else
                     Result := B;
                  end if;

                  return Result;

               end Max;

            begin
               MaxLength := 0;
               Export := LexTokenManager.Seq_Algebra.First_Member (The_Heap => TheHeap,
                                                                   S        => Exports);
               loop
                  exit when LexTokenManager.Seq_Algebra.Is_Null_Member (M => Export);
                  MaxLength := Max
                    (A => MaxLength,
                     B => EStrings.Get_Length (E_Str => LexTokenManager.Lex_String_To_String
                                                 (Lex_Str => LexTokenManager.Seq_Algebra.Value_Of_Member
                                                    (The_Heap => TheHeap,
                                                     M        => Export))));
                  Export := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                     M        => Export);
               end loop;
               return MaxLength;
            end LongestExport;

            function FromCol (StartCol    : EStrings.Positions;
                              ExportWidth : EStrings.Lengths) return EStrings.Positions
            --# global in SparkFormatCommandLineData.Content;
            is
               FromColumn : EStrings.Positions;
            begin
               if SparkFormatCommandLineData.Content.SeparatorIndent = SparkFormatCommandLineData.Inline then
                  FromColumn := (ExportCol (StartCol) + ExportWidth) + 1;
               else
                  FromColumn := SeparatorCol (StartCol);
               end if;
               return FromColumn;
            end FromCol;

            procedure WriteFrom (Output      : in SPARK_IO.File_Type;
                                 StartCol    : in EStrings.Positions;
                                 ExportWidth : in EStrings.Lengths)
            --# global in     Annotation.StartString;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Annotation.StartString,
            --#                                ExportWidth,
            --#                                Output,
            --#                                SparkFormatCommandLineData.Content,
            --#                                StartCol;
            is
            begin
               if SparkFormatCommandLineData.Content.SeparatorIndent /= SparkFormatCommandLineData.Inline then
                  Annotation.Write (Output, StartCol);
               end if;
               SPARK_IO.Set_Col (Output, FromCol (StartCol    => StartCol,
                                                  ExportWidth => ExportWidth));
               SPARK_IO.Put_String (Output, "from", 0);
               -- If ImportIndent is not Inline, then don't put out a trailing space,
               -- since this will be rejected by GNAT -gnatyb style check mode.
               if SparkFormatCommandLineData.Content.ImportIndent = SparkFormatCommandLineData.Inline then
                  SPARK_IO.Put_String (Output, " ", 0);
               end if;
            end WriteFrom;

            procedure WriteImport (FirstImport : in out Boolean;
                                   Output      : in     SPARK_IO.File_Type;
                                   StartCol    : in     EStrings.Positions;
                                   ExportWidth : in     EStrings.Lengths;
                                   TheHeap     : in     Heap.HeapRecord;
                                   Import      : in     LexTokenManager.Seq_Algebra.Member_Of_Seq)
            --# global in     Annotation.StartString;
            --#        in     LexTokenManager.State;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --# derives FirstImport       from * &
            --#         SPARK_IO.File_Sys from *,
            --#                                Annotation.StartString,
            --#                                ExportWidth,
            --#                                FirstImport,
            --#                                Import,
            --#                                LexTokenManager.State,
            --#                                Output,
            --#                                SparkFormatCommandLineData.Content,
            --#                                StartCol,
            --#                                TheHeap;
            is

               function ImportCol (StartCol    : EStrings.Positions;
                                   ExportWidth : EStrings.Lengths) return EStrings.Positions
               --# global in SparkFormatCommandLineData.Content;
               is
                  ImportColumn : EStrings.Positions;
               begin
                  if SparkFormatCommandLineData.Content.ImportIndent = SparkFormatCommandLineData.Inline then
                     ImportColumn := FromCol (StartCol    => StartCol,
                                              ExportWidth => ExportWidth) + 5; -- length of "from "
                  else
                     ImportColumn := DerivesCol (StartCol) + (SparkFormatCommandLineData.Content.ImportIndent - 1);
                  end if;
                  return ImportColumn;
               end ImportCol;

            begin

               if FirstImport then
                  FirstImport := False;
                  if SparkFormatCommandLineData.Content.ImportIndent /= SparkFormatCommandLineData.Inline then
                     Annotation.Write (Output, StartCol);
                  end if;
               else
                  SPARK_IO.Put_Line (Output, ",", 0);
                  Annotation.Write (Output, StartCol);
               end if;

               SPARK_IO.Set_Col (Output, ImportCol (StartCol    => StartCol,
                                                    ExportWidth => ExportWidth));
               EStrings.Put_String
                 (File  => Output,
                  E_Str => LexTokenManager.Lex_String_To_String
                    (Lex_Str => LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                             M        => Import)));
            end WriteImport;

            procedure WriteImports (Output      : in     SPARK_IO.File_Type;
                                    StartCol    : in     EStrings.Positions;
                                    ExportWidth : in     EStrings.Lengths;
                                    TheHeap     : in out Heap.HeapRecord;
                                    Imports     : in     LexTokenManager.Seq_Algebra.Seq)
            --# global in     Annotation.StartString;
            --#        in     LexTokenManager.State;
            --#        in     MultiplyToken;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Annotation.StartString,
            --#                                ExportWidth,
            --#                                Imports,
            --#                                LexTokenManager.State,
            --#                                MultiplyToken,
            --#                                Output,
            --#                                SparkFormatCommandLineData.Content,
            --#                                StartCol,
            --#                                TheHeap &
            --#         TheHeap           from *,
            --#                                Imports,
            --#                                LexTokenManager.State,
            --#                                MultiplyToken;
            is
               Import      : LexTokenManager.Seq_Algebra.Member_Of_Seq;
               FirstImport : Boolean;
               TheIterator : SPARKProgram.Iteration.Iterator;
            begin
               Import := LexTokenManager.Seq_Algebra.First_Member (The_Heap => TheHeap,
                                                                   S        => Imports);
               FirstImport := True;

               -- Imports might contain a "*", but it might not be the first
               -- member of the sequence.  SPARK _does_ require the "*" to appear
               -- first in the import list, so we scan for it alone on the first pass,
               -- write it out, then remove it from the sequence.
               -- Note that the sequence is disposed of after writing imports so it is
               -- OK to modify it here.
               loop
                  exit when LexTokenManager.Seq_Algebra.Is_Null_Member (M => Import);
                  if LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                              M        => Import),
                     Lex_Str2 => MultiplyToken) = LexTokenManager.Str_Eq then
                     WriteImport (FirstImport => FirstImport,
                                  Output      => Output,
                                  StartCol    => StartCol,
                                  ExportWidth => ExportWidth,
                                  TheHeap     => TheHeap,
                                  Import      => Import);
                     LexTokenManager.Seq_Algebra.Remove_Member (The_Heap    => TheHeap,
                                                                S           => Imports,
                                                                Given_Value => MultiplyToken);
                     exit;
                  end if;
                  Import := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                     M        => Import);
               end loop;

               -- If '*' was the only import in the seq then the seq is now empty so we don't
               -- need to write any more imports. Need to guard against this case as the
               -- Initialise routine has an implicit precondition that the sequence is not
               -- empty.
               if not LexTokenManager.Seq_Algebra.Is_Empty_Seq (The_Heap => TheHeap,
                                                                S        => Imports) then
                  SPARKProgram.Iteration.Initialise (TheHeap     => TheHeap,
                                                     TheSeq      => Imports,
                                                     TheIterator => TheIterator);
                  WriteImport (FirstImport => FirstImport,
                               Output      => Output,
                               StartCol    => StartCol,
                               ExportWidth => ExportWidth,
                               TheHeap     => TheHeap,
                               Import      => SPARKProgram.Iteration.CurrentMember (TheIterator));
                  loop
                     SPARKProgram.Iteration.Next (TheHeap, TheIterator);
                     exit when SPARKProgram.Iteration.Complete (TheIterator);
                     WriteImport (FirstImport => FirstImport,
                                  Output      => Output,
                                  StartCol    => StartCol,
                                  ExportWidth => ExportWidth,
                                  TheHeap     => TheHeap,
                                  Import      => SPARKProgram.Iteration.CurrentMember (TheIterator));
                  end loop;
               end if;
            end WriteImports;

            procedure SubstituteSelfForMultiply (TheHeap                     : in out Heap.HeapRecord;
                                                 Exports                     : in     LexTokenManager.Seq_Algebra.Seq;
                                                 DependenciesWithMultiply    : in     LexTokenManager.Relation_Algebra.String.Relation;
                                                 DependenciesWithoutMultiply : in     LexTokenManager.Relation_Algebra.String.Relation)
            --# global in     LexTokenManager.State;
            --#        in     MultiplyToken;
            --#        in out Statistics.TableUsage;
            --# derives Statistics.TableUsage,
            --#         TheHeap               from *,
            --#                                    DependenciesWithMultiply,
            --#                                    DependenciesWithoutMultiply,
            --#                                    Exports,
            --#                                    LexTokenManager.State,
            --#                                    MultiplyToken,
            --#                                    TheHeap;
            is
               Imports              : LexTokenManager.Seq_Algebra.Seq;
               Export, Import       : LexTokenManager.Seq_Algebra.Member_Of_Seq;
               ExportVal, ImportVal : LexTokenManager.Lex_String;
            begin
               Export := LexTokenManager.Seq_Algebra.First_Member (The_Heap => TheHeap,
                                                                   S        => Exports);
               loop
                  exit when LexTokenManager.Seq_Algebra.Is_Null_Member (M => Export);
                  ExportVal := LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                            M        => Export);
                  LexTokenManager.Relation_Algebra.String.Row_Extraction (The_Heap    => TheHeap,
                                                                          R           => DependenciesWithMultiply,
                                                                          Given_Index => ExportVal,
                                                                          S           => Imports);
                  Import := LexTokenManager.Seq_Algebra.First_Member (The_Heap => TheHeap,
                                                                      S        => Imports);
                  loop
                     exit when LexTokenManager.Seq_Algebra.Is_Null_Member (M => Import);
                     ImportVal := LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                               M        => Import);
                     if LexTokenManager.Lex_String_Case_Insensitive_Compare
                       (Lex_Str1 => ImportVal,
                        Lex_Str2 => MultiplyToken) = LexTokenManager.Str_Eq then
                        ImportVal := ExportVal;
                     end if;
                     LexTokenManager.Relation_Algebra.String.Insert_Pair (The_Heap => TheHeap,
                                                                          R        => DependenciesWithoutMultiply,
                                                                          I        => ExportVal,
                                                                          J        => ImportVal);
                     Import := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                        M        => Import);
                  end loop;
                  LexTokenManager.Seq_Algebra.Dispose_Of_Seq (The_Heap => TheHeap,
                                                              S        => Imports);
                  Export := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                     M        => Export);
               end loop;
            end SubstituteSelfForMultiply;

            procedure WriteSimilarClauses (Output       : in     SPARK_IO.File_Type;
                                           StartCol     : in     EStrings.Positions;
                                           ExportWidth  : in     EStrings.Lengths;
                                           FirstExport  : in out Boolean;
                                           TheHeap      : in out Heap.HeapRecord;
                                           Exports      : in     LexTokenManager.Seq_Algebra.Seq;
                                           Dependencies : in     LexTokenManager.Relation_Algebra.String.Relation)
            --# global in     Annotation.StartString;
            --#        in     LexTokenManager.State;
            --#        in     MultiplyToken;
            --#        in     RWNullToken;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --#        in out Statistics.TableUsage;
            --# derives FirstExport,
            --#         Statistics.TableUsage,
            --#         TheHeap               from *,
            --#                                    Dependencies,
            --#                                    Exports,
            --#                                    LexTokenManager.State,
            --#                                    MultiplyToken,
            --#                                    RWNullToken,
            --#                                    TheHeap &
            --#         SPARK_IO.File_Sys     from *,
            --#                                    Annotation.StartString,
            --#                                    Dependencies,
            --#                                    Exports,
            --#                                    ExportWidth,
            --#                                    FirstExport,
            --#                                    LexTokenManager.State,
            --#                                    MultiplyToken,
            --#                                    Output,
            --#                                    RWNullToken,
            --#                                    SparkFormatCommandLineData.Content,
            --#                                    StartCol,
            --#                                    TheHeap;
            is
               CompressedExports : LexTokenManager.Relation_Algebra.Relation;
               CompressedImports : LexTokenManager.Relation_Algebra.Relation;
               CompressedIndex   : Natural;

               procedure FindSimilarExports (TheHeap        : in out Heap.HeapRecord;
                                             Export         : in     LexTokenManager.Seq_Algebra.Member_Of_Seq;
                                             Dependencies   : in     LexTokenManager.Relation_Algebra.String.Relation;
                                             SimilarExports : in     LexTokenManager.Seq_Algebra.Seq;
                                             Imports        : in     LexTokenManager.Seq_Algebra.Seq)
               --# global in     LexTokenManager.State;
               --#        in     RWNullToken;
               --#        in out Statistics.TableUsage;
               --# derives Statistics.TableUsage,
               --#         TheHeap               from *,
               --#                                    Dependencies,
               --#                                    Export,
               --#                                    Imports,
               --#                                    LexTokenManager.State,
               --#                                    RWNullToken,
               --#                                    SimilarExports,
               --#                                    TheHeap;
               is
                  OtherImports   : LexTokenManager.Seq_Algebra.Seq;
                  OtherExport    : LexTokenManager.Seq_Algebra.Member_Of_Seq;
                  OtherExportVal : LexTokenManager.Lex_String;

                  function IdenticalSequences (TheHeap : Heap.HeapRecord;
                                               S       : LexTokenManager.Seq_Algebra.Seq;
                                               T       : LexTokenManager.Seq_Algebra.Seq) return Boolean
                  --# global in LexTokenManager.State;
                  is
                     M, N      : LexTokenManager.Seq_Algebra.Member_Of_Seq;
                     Identical : Boolean;
                  begin
                     M := LexTokenManager.Seq_Algebra.First_Member (The_Heap => TheHeap,
                                                                    S        => S);
                     N := LexTokenManager.Seq_Algebra.First_Member (The_Heap => TheHeap,
                                                                    S        => T);
                     Identical := True;
                     loop
                        exit when LexTokenManager.Seq_Algebra.Is_Null_Member (M => M) or else
                          LexTokenManager.Seq_Algebra.Is_Null_Member (M => N);
                        if LexTokenManager.Lex_String_Case_Insensitive_Compare
                          (Lex_Str1 => LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                                    M        => M),
                           Lex_Str2 => LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                                    M        => N)) /= LexTokenManager.Str_Eq then
                           Identical := False;
                           exit;
                        end if;
                        M := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                      M        => M);
                        N := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                      M        => N);
                     end loop;
                     return Identical and then
                       LexTokenManager.Seq_Algebra.Is_Null_Member (M => M) and then
                       LexTokenManager.Seq_Algebra.Is_Null_Member (M => N);
                  end IdenticalSequences;

               begin
                  OtherExport := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                          M        => Export);
                  loop
                     exit when LexTokenManager.Seq_Algebra.Is_Null_Member (M => OtherExport);
                     OtherExportVal := LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                                    M        => OtherExport);
                     if LexTokenManager.Lex_String_Case_Insensitive_Compare
                       (Lex_Str1 => OtherExportVal,
                        Lex_Str2 => RWNullToken) /= LexTokenManager.Str_Eq then
                        LexTokenManager.Relation_Algebra.String.Row_Extraction (The_Heap    => TheHeap,
                                                                                R           => Dependencies,
                                                                                Given_Index => OtherExportVal,
                                                                                S           => OtherImports);
                        if IdenticalSequences (TheHeap => TheHeap,
                                               S       => Imports,
                                               T       => OtherImports) then
                           LexTokenManager.Seq_Algebra.Add_Member (The_Heap    => TheHeap,
                                                                   S           => SimilarExports,
                                                                   Given_Value => OtherExportVal);
                        end if;
                        LexTokenManager.Seq_Algebra.Dispose_Of_Seq (The_Heap => TheHeap,
                                                                    S        => OtherImports);
                     end if;
                     OtherExport := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                             M        => OtherExport);
                  end loop;
               end FindSimilarExports;

               procedure SwapSequence (A, B : in out LexTokenManager.Seq_Algebra.Seq)
               --# derives A from B &
               --#         B from A;
               is
                  Temp : LexTokenManager.Seq_Algebra.Seq;
               begin
                  Temp := A;
                  A := B;
                  B := Temp;
               end SwapSequence;

               function SequenceLength (TheHeap : Heap.HeapRecord;
                                        S       : LexTokenManager.Seq_Algebra.Seq) return Natural
               is
                  M      : LexTokenManager.Seq_Algebra.Member_Of_Seq;
                  Length : Natural;
               begin
                  M := LexTokenManager.Seq_Algebra.First_Member (The_Heap => TheHeap,
                                                                 S        => S);
                  Length := 0;
                  loop
                     exit when LexTokenManager.Seq_Algebra.Is_Null_Member (M => M);
                     Length := Length + 1;
                     M := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                   M        => M);
                  end loop;
                  return Length;
               end SequenceLength;

               procedure BuildCompressedDependencies (TheHeap           : in out Heap.HeapRecord;
                                                      Dependencies      : in     LexTokenManager.Relation_Algebra.String.Relation;
                                                      Exports           : in     LexTokenManager.Seq_Algebra.Seq;
                                                      CompressedExports :    out LexTokenManager.Relation_Algebra.Relation;
                                                      CompressedImports :    out LexTokenManager.Relation_Algebra.Relation;
                                                      CompressedIndex   :    out Natural)

               --# global in     LexTokenManager.State;
               --#        in     MultiplyToken;
               --#        in     RWNullToken;
               --#        in out Statistics.TableUsage;
               --# derives CompressedExports,
               --#         CompressedImports     from TheHeap &
               --#         CompressedIndex,
               --#         TheHeap               from Dependencies,
               --#                                    Exports,
               --#                                    LexTokenManager.State,
               --#                                    MultiplyToken,
               --#                                    RWNullToken,
               --#                                    TheHeap &
               --#         Statistics.TableUsage from *,
               --#                                    Dependencies,
               --#                                    Exports,
               --#                                    LexTokenManager.State,
               --#                                    MultiplyToken,
               --#                                    RWNullToken,
               --#                                    TheHeap;

               is
                  ExportVal      : LexTokenManager.Lex_String;
                  Export         : LexTokenManager.Seq_Algebra.Member_Of_Seq;
                  NextExport     : LexTokenManager.Seq_Algebra.Member_Of_Seq;
                  SimilarExports : LexTokenManager.Seq_Algebra.Seq;
                  Imports        : LexTokenManager.Seq_Algebra.Seq;

                  procedure TryAddingSelfDependencies (TheHeap        : in out Heap.HeapRecord;
                                                       Export         : in     LexTokenManager.Seq_Algebra.Member_Of_Seq;
                                                       Dependencies   : in     LexTokenManager.Relation_Algebra.String.Relation;
                                                       SimilarExports : in out LexTokenManager.Seq_Algebra.Seq;
                                                       Imports        : in out LexTokenManager.Seq_Algebra.Seq)
                  --# global in     LexTokenManager.State;
                  --#        in     MultiplyToken;
                  --#        in     RWNullToken;
                  --#        in out Statistics.TableUsage;
                  --# derives Imports,
                  --#         SimilarExports,
                  --#         Statistics.TableUsage,
                  --#         TheHeap               from *,
                  --#                                    Dependencies,
                  --#                                    Export,
                  --#                                    Imports,
                  --#                                    LexTokenManager.State,
                  --#                                    MultiplyToken,
                  --#                                    RWNullToken,
                  --#                                    SimilarExports,
                  --#                                    TheHeap;
                  is
                     OtherExport : LexTokenManager.Seq_Algebra.Member_Of_Seq;

                     procedure TryAddingSelfDependency (TheHeap        : in out Heap.HeapRecord;
                                                        Export         : in     LexTokenManager.Seq_Algebra.Member_Of_Seq;
                                                        OtherExport    : in     LexTokenManager.Seq_Algebra.Member_Of_Seq;
                                                        Dependencies   : in     LexTokenManager.Relation_Algebra.String.Relation;
                                                        SimilarExports : in out LexTokenManager.Seq_Algebra.Seq;
                                                        Imports        : in out LexTokenManager.Seq_Algebra.Seq)
                     --# global in     LexTokenManager.State;
                     --#        in     RWNullToken;
                     --#        in out Statistics.TableUsage;
                     --# derives Imports,
                     --#         TheHeap               from Dependencies,
                     --#                                    Export,
                     --#                                    Imports,
                     --#                                    LexTokenManager.State,
                     --#                                    OtherExport,
                     --#                                    RWNullToken,
                     --#                                    SimilarExports,
                     --#                                    TheHeap &
                     --#         SimilarExports,
                     --#         Statistics.TableUsage from *,
                     --#                                    Dependencies,
                     --#                                    Export,
                     --#                                    LexTokenManager.State,
                     --#                                    OtherExport,
                     --#                                    RWNullToken,
                     --#                                    SimilarExports,
                     --#                                    TheHeap;
                     is
                        ExportVal, OtherExportVal  : LexTokenManager.Lex_String;
                        TrialSimilarExports        : LexTokenManager.Seq_Algebra.Seq;
                        TrialImports               : LexTokenManager.Seq_Algebra.Seq;
                        TrialDependencies          : LexTokenManager.Relation_Algebra.String.Relation;
                     begin
                        ExportVal := LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                                  M        => Export);
                        OtherExportVal := LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                                       M        => OtherExport);
                        LexTokenManager.Relation_Algebra.String.Create_Relation (The_Heap => TheHeap,
                                                                                 R        => TrialDependencies);
                        LexTokenManager.Relation_Algebra.String.Augment_Relation (The_Heap => TheHeap,
                                                                                  A        => TrialDependencies,
                                                                                  B        => Dependencies);
                        LexTokenManager.Relation_Algebra.String.Insert_Pair (The_Heap => TheHeap,
                                                                             R        => TrialDependencies,
                                                                             I        => OtherExportVal,
                                                                             J        => OtherExportVal);
                        LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                                S        => TrialSimilarExports);
                        LexTokenManager.Relation_Algebra.String.Row_Extraction (The_Heap    => TheHeap,
                                                                                R           => TrialDependencies,
                                                                                Given_Index => ExportVal,
                                                                                S           => TrialImports);
                        FindSimilarExports (TheHeap        => TheHeap,
                                            Export         => Export,
                                            Dependencies   => TrialDependencies,
                                            SimilarExports => TrialSimilarExports,
                                            Imports        => TrialImports);
                        if SequenceLength (TheHeap => TheHeap,
                                           S       => TrialSimilarExports) >
                          SequenceLength (TheHeap => TheHeap,
                                          S       => SimilarExports) then
                           SwapSequence (A => SimilarExports,
                                         B => TrialSimilarExports);
                           SwapSequence (A => Imports,
                                         B => TrialImports);
                           LexTokenManager.Relation_Algebra.String.Insert_Pair (The_Heap => TheHeap,
                                                                                R        => Dependencies,
                                                                                I        => OtherExportVal,
                                                                                J        => OtherExportVal);
                        end if;
                        LexTokenManager.Seq_Algebra.Dispose_Of_Seq (The_Heap => TheHeap,
                                                                    S        => TrialSimilarExports);
                        LexTokenManager.Seq_Algebra.Dispose_Of_Seq (The_Heap => TheHeap,
                                                                    S        => TrialImports);
                        LexTokenManager.Relation_Algebra.String.Dispose_Of_Relation (The_Heap => TheHeap,
                                                                                     R        => TrialDependencies);
                     end TryAddingSelfDependency;

                  begin
                     if LexTokenManager.Seq_Algebra.Is_Member (The_Heap    => TheHeap,
                                                               S           => Imports,
                                                               Given_Value => MultiplyToken) then
                        TryAddingSelfDependency (TheHeap        => TheHeap,
                                                 Export         => Export,
                                                 OtherExport    => Export,
                                                 Dependencies   => Dependencies,
                                                 SimilarExports => SimilarExports,
                                                 Imports        => Imports);
                        OtherExport := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                                M        => Export);
                        loop
                           exit when LexTokenManager.Seq_Algebra.Is_Null_Member (M => OtherExport);
                           TryAddingSelfDependency (TheHeap        => TheHeap,
                                                    Export         => Export,
                                                    OtherExport    => OtherExport,
                                                    Dependencies   => Dependencies,
                                                    SimilarExports => SimilarExports,
                                                    Imports        => Imports);
                           OtherExport := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                                   M        => OtherExport);
                        end loop;
                     end if;
                  end TryAddingSelfDependencies;

                  procedure TryWithoutMultiply (TheHeap        : in out Heap.HeapRecord;
                                                Export         : in     LexTokenManager.Seq_Algebra.Member_Of_Seq;
                                                Exports        : in     LexTokenManager.Seq_Algebra.Seq;
                                                Dependencies   : in     LexTokenManager.Relation_Algebra.String.Relation;
                                                SimilarExports : in out LexTokenManager.Seq_Algebra.Seq;
                                                Imports        : in out LexTokenManager.Seq_Algebra.Seq)
                  --# global in     LexTokenManager.State;
                  --#        in     MultiplyToken;
                  --#        in     RWNullToken;
                  --#        in out Statistics.TableUsage;
                  --# derives Imports,
                  --#         TheHeap               from Dependencies,
                  --#                                    Export,
                  --#                                    Exports,
                  --#                                    Imports,
                  --#                                    LexTokenManager.State,
                  --#                                    MultiplyToken,
                  --#                                    RWNullToken,
                  --#                                    SimilarExports,
                  --#                                    TheHeap &
                  --#         SimilarExports,
                  --#         Statistics.TableUsage from *,
                  --#                                    Dependencies,
                  --#                                    Export,
                  --#                                    Exports,
                  --#                                    LexTokenManager.State,
                  --#                                    MultiplyToken,
                  --#                                    RWNullToken,
                  --#                                    TheHeap;
                  is
                     TrialSimilarExports, TrialImports : LexTokenManager.Seq_Algebra.Seq;
                     ExportVal                         : LexTokenManager.Lex_String;
                     DependenciesWithoutMultiply       : LexTokenManager.Relation_Algebra.String.Relation;
                  begin
                     ExportVal := LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                               M        => Export);
                     LexTokenManager.Relation_Algebra.String.Create_Relation (The_Heap => TheHeap,
                                                                              R        => DependenciesWithoutMultiply);
                     SubstituteSelfForMultiply (TheHeap                     => TheHeap,
                                                Exports                     => Exports,
                                                DependenciesWithMultiply    => Dependencies,
                                                DependenciesWithoutMultiply => DependenciesWithoutMultiply);
                     LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                             S        => TrialSimilarExports);
                     LexTokenManager.Relation_Algebra.String.Row_Extraction (The_Heap    => TheHeap,
                                                                             R           => DependenciesWithoutMultiply,
                                                                             Given_Index => ExportVal,
                                                                             S           => TrialImports);
                     FindSimilarExports (TheHeap        => TheHeap,
                                         Export         => Export,
                                         Dependencies   => DependenciesWithoutMultiply,
                                         SimilarExports => TrialSimilarExports,
                                         Imports        => TrialImports);
                     if SequenceLength (TheHeap => TheHeap,
                                        S       => TrialSimilarExports) >
                       SequenceLength (TheHeap => TheHeap,
                                       S       => SimilarExports) then
                        SwapSequence (A => SimilarExports,
                                      B => TrialSimilarExports);
                        SwapSequence (A => Imports,
                                      B => TrialImports);
                     end if;
                     LexTokenManager.Seq_Algebra.Dispose_Of_Seq (The_Heap => TheHeap,
                                                                 S        => TrialSimilarExports);
                     LexTokenManager.Seq_Algebra.Dispose_Of_Seq (The_Heap => TheHeap,
                                                                 S        => TrialImports);
                     LexTokenManager.Relation_Algebra.String.Dispose_Of_Relation (The_Heap => TheHeap,
                                                                                  R        => DependenciesWithoutMultiply);
                  end TryWithoutMultiply;

                  procedure RemoveMembers (TheHeap : in out Heap.HeapRecord;
                                           A, B    : in     LexTokenManager.Seq_Algebra.Seq)
                  --# global in LexTokenManager.State;
                  --# derives TheHeap from *,
                  --#                      A,
                  --#                      B,
                  --#                      LexTokenManager.State;
                  is
                     M, Next  : LexTokenManager.Seq_Algebra.Member_Of_Seq;
                     ValueOfM : LexTokenManager.Lex_String;
                  begin
                     M := LexTokenManager.Seq_Algebra.First_Member (The_Heap => TheHeap,
                                                                    S        => A);
                     loop
                        exit when LexTokenManager.Seq_Algebra.Is_Null_Member (M => M);
                        ValueOfM := LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                                 M        => M);
                        if LexTokenManager.Seq_Algebra.Is_Member (The_Heap    => TheHeap,
                                                                  S           => B,
                                                                  Given_Value => ValueOfM) then
                           Next := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                            M        => M);
                           LexTokenManager.Seq_Algebra.Remove_Member (The_Heap    => TheHeap,
                                                                      S           => A,
                                                                      Given_Value => ValueOfM);
                           M := Next;
                        else
                           M := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                         M        => M);
                        end if;
                     end loop;
                  end RemoveMembers;

               begin
                  -- Create relations to store sequences of exports and their imports.
                  -- These two relations will be indexed by CompressedIndex, so for any given value
                  -- of CompressedIndex one relation gives the sequence of exports and the other
                  -- provides the sequence of imports from which they are derived.
                  LexTokenManager.Relation_Algebra.Create_Relation (The_Heap => TheHeap,
                                                                    R        => CompressedExports);
                  LexTokenManager.Relation_Algebra.Create_Relation (The_Heap => TheHeap,
                                                                    R        => CompressedImports);
                  CompressedIndex := Natural'First;
                  Export := LexTokenManager.Seq_Algebra.First_Member (The_Heap => TheHeap,
                                                                      S        => Exports);
                  -- Loop condition avoids potential RTE. CompressedIndex can never exceed bounds of
                  -- Natural in practice because it will never exceed the number of exports.
                  while CompressedIndex < Natural'Last loop
                     exit when LexTokenManager.Seq_Algebra.Is_Null_Member (M => Export);
                     -- Look for other exports with the same dependency list, and store them
                     -- in the sequence SimilarExports.
                     ExportVal := LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                               M        => Export);
                     if LexTokenManager.Lex_String_Case_Insensitive_Compare
                       (Lex_Str1 => ExportVal,
                        Lex_Str2 => RWNullToken) /= LexTokenManager.Str_Eq then -- null derives comes last so ignore it here
                        LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                                S        => SimilarExports);
                        LexTokenManager.Relation_Algebra.String.Row_Extraction (The_Heap    => TheHeap,
                                                                                R           => Dependencies,
                                                                                Given_Index => ExportVal,
                                                                                S           => Imports);
                        FindSimilarExports (TheHeap        => TheHeap,
                                            Export         => Export,
                                            Dependencies   => Dependencies,
                                            SimilarExports => SimilarExports,
                                            Imports        => Imports);
                        TryAddingSelfDependencies (TheHeap        => TheHeap,
                                                   Export         => Export,
                                                   Dependencies   => Dependencies,
                                                   SimilarExports => SimilarExports,
                                                   Imports        => Imports);
                        TryWithoutMultiply (TheHeap        => TheHeap,
                                            Export         => Export,
                                            Exports        => Exports,
                                            Dependencies   => Dependencies,
                                            SimilarExports => SimilarExports,
                                            Imports        => Imports);
                        -- Rather than write the dependencies out directly here (in whatever order they
                        -- happen to be in) we build a new, compressed, dependency relation to be written
                        -- out (in a given order) once it is complete.
                        --
                        -- Note that if an export has no similar exports it *is* added to the compressed
                        -- sequence but it is *not* removed from the sequence of exports. (The implementations
                        -- of FindSimilarExports, TryAddingSelfDependencies and TryWithoutMultiply depend on
                        -- this being the case.) Consequently, the sequence of exports should not be used
                        -- after calling this subprogram.
                        CompressedIndex := CompressedIndex + 1;
                        LexTokenManager.Relation_Algebra.Insert_Pair (The_Heap => TheHeap,
                                                                      R        => CompressedExports,
                                                                      I        => CompressedIndex,
                                                                      J        => ExportVal);
                        LexTokenManager.Relation_Algebra.Add_Row (The_Heap => TheHeap,
                                                                  R        => CompressedImports,
                                                                  I        => CompressedIndex,
                                                                  S        => Imports);

                        if LexTokenManager.Seq_Algebra.Is_Empty_Seq (The_Heap => TheHeap,
                                                                     S        => SimilarExports) then
                           Export := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                              M        => Export);
                        else
                           LexTokenManager.Relation_Algebra.Add_Row (The_Heap => TheHeap,
                                                                     R        => CompressedExports,
                                                                     I        => CompressedIndex,
                                                                     S        => SimilarExports);
                           RemoveMembers (TheHeap => TheHeap,
                                          A       => Exports,
                                          B       => SimilarExports);
                           NextExport := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                                  M        => Export);
                           LexTokenManager.Seq_Algebra.Remove_Member (The_Heap    => TheHeap,
                                                                      S           => Exports,
                                                                      Given_Value => ExportVal);
                           Export := NextExport;
                        end if;
                        LexTokenManager.Seq_Algebra.Dispose_Of_Seq (The_Heap => TheHeap,
                                                                    S        => SimilarExports);
                        LexTokenManager.Seq_Algebra.Dispose_Of_Seq (The_Heap => TheHeap,
                                                                    S        => Imports);
                     else
                        Export := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                           M        => Export); -- if null token move on to next item
                     end if;
                  end loop;
               end BuildCompressedDependencies;

               procedure WriteDependenciesAlphabetic (TheHeap           : in out Heap.HeapRecord;
                                                      Output            : in     SPARK_IO.File_Type;
                                                      StartCol          : in     EStrings.Positions;
                                                      ExportWidth       : in     EStrings.Lengths;
                                                      FirstExport       : in out Boolean;
                                                      CompressedExports : in     LexTokenManager.Relation_Algebra.Relation;
                                                      CompressedImports : in     LexTokenManager.Relation_Algebra.Relation;
                                                      CompressedIndex   : in     Natural)

               --# global in     Annotation.StartString;
               --#        in     LexTokenManager.State;
               --#        in     MultiplyToken;
               --#        in     SparkFormatCommandLineData.Content;
               --#        in out SPARK_IO.File_Sys;
               --#        in out Statistics.TableUsage;
               --# derives FirstExport,
               --#         Statistics.TableUsage,
               --#         TheHeap               from *,
               --#                                    CompressedExports,
               --#                                    CompressedImports,
               --#                                    CompressedIndex,
               --#                                    LexTokenManager.State,
               --#                                    MultiplyToken,
               --#                                    TheHeap &
               --#         SPARK_IO.File_Sys     from *,
               --#                                    Annotation.StartString,
               --#                                    CompressedExports,
               --#                                    CompressedImports,
               --#                                    CompressedIndex,
               --#                                    ExportWidth,
               --#                                    FirstExport,
               --#                                    LexTokenManager.State,
               --#                                    MultiplyToken,
               --#                                    Output,
               --#                                    SparkFormatCommandLineData.Content,
               --#                                    StartCol,
               --#                                    TheHeap;
               is
                  RelationsWritten : SeqAlgebra.Seq;
                  LocalIndex       : Natural;
                  RelationCount    : Natural;
                  FirstAlphaIndex  : Natural;
                  FirstAlphaExport : LexTokenManager.Lex_String;
                  ExportsToWrite   : LexTokenManager.Seq_Algebra.Seq;
                  ImportsToWrite   : LexTokenManager.Seq_Algebra.Seq;
                  TheIterator      : SPARKProgram.Iteration.Iterator;
                  CurrentString    : LexTokenManager.Lex_String;

                  procedure WriteDependencies (Output      : in     SPARK_IO.File_Type;
                                               StartCol    : in     EStrings.Positions;
                                               ExportWidth : in     EStrings.Lengths;
                                               FirstExport : in out Boolean;
                                               TheHeap     : in out Heap.HeapRecord;
                                               Exports     : in     LexTokenManager.Seq_Algebra.Seq;
                                               Imports     : in     LexTokenManager.Seq_Algebra.Seq)
                  --# global in     Annotation.StartString;
                  --#        in     LexTokenManager.State;
                  --#        in     MultiplyToken;
                  --#        in     SparkFormatCommandLineData.Content;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives FirstExport       from * &
                  --#         SPARK_IO.File_Sys from *,
                  --#                                Annotation.StartString,
                  --#                                Exports,
                  --#                                ExportWidth,
                  --#                                FirstExport,
                  --#                                Imports,
                  --#                                LexTokenManager.State,
                  --#                                MultiplyToken,
                  --#                                Output,
                  --#                                SparkFormatCommandLineData.Content,
                  --#                                StartCol,
                  --#                                TheHeap &
                  --#         TheHeap           from *,
                  --#                                Imports,
                  --#                                LexTokenManager.State,
                  --#                                MultiplyToken;
                  is
                     TheIterator   : SPARKProgram.Iteration.Iterator;

                     procedure WriteSimilarExport (Output   : in     SPARK_IO.File_Type;
                                                   StartCol : in     EStrings.Positions;
                                                   TheHeap  : in     Heap.HeapRecord;
                                                   Export   : in     LexTokenManager.Seq_Algebra.Member_Of_Seq)
                     --# global in     Annotation.StartString;
                     --#        in     LexTokenManager.State;
                     --#        in     SparkFormatCommandLineData.Content;
                     --#        in out SPARK_IO.File_Sys;
                     --# derives SPARK_IO.File_Sys from *,
                     --#                                Annotation.StartString,
                     --#                                Export,
                     --#                                LexTokenManager.State,
                     --#                                Output,
                     --#                                SparkFormatCommandLineData.Content,
                     --#                                StartCol,
                     --#                                TheHeap;
                     is
                     begin
                        SPARK_IO.Put_Line (Output, ",", 0);
                        Annotation.Write (Output, StartCol);
                        SPARK_IO.Set_Col (Output, ExportCol (StartCol));
                        EStrings.Put_String
                          (File  => Output,
                           E_Str => LexTokenManager.Lex_String_To_String
                             (Lex_Str => LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                                      M        => Export)));
                     end WriteSimilarExport;

                  begin
                     -- Dependency consists of <exports> from <imports>, so start with
                     -- the exports.
                     SPARKProgram.Iteration.Initialise (TheHeap     => TheHeap,
                                                        TheSeq      => Exports,
                                                        TheIterator => TheIterator);
                     WriteExport (FirstExport => FirstExport,
                                  Output      => Output,
                                  StartCol    => StartCol,
                                  TheHeap     => TheHeap,
                                  Export      => SPARKProgram.Iteration.CurrentMember (TheIterator));
                     SPARKProgram.Iteration.Next (TheHeap, TheIterator);
                     -- if there are more exports derived from the same imports, write them out
                     while not SPARKProgram.Iteration.Complete (TheIterator) loop
                        WriteSimilarExport (Output   => Output,
                                            StartCol => StartCol,
                                            TheHeap  => TheHeap,
                                            Export   => SPARKProgram.Iteration.CurrentMember (TheIterator));
                        SPARKProgram.Iteration.Next (TheHeap, TheIterator);
                     end loop;
                     -- from
                     WriteFrom (Output      => Output,
                                StartCol    => StartCol,
                                ExportWidth => ExportWidth);
                     -- the imports
                     WriteImports (Output      => Output,
                                   StartCol    => StartCol,
                                   ExportWidth => ExportWidth,
                                   TheHeap     => TheHeap,
                                   Imports     => Imports);
                  end WriteDependencies;

               begin
                  -- Need to find the alphabetical first member out of all the export sequences in
                  -- the ExportsToWrite relation. The corresponding dependency can then be written
                  -- out, after which it can be disposed of. Repeat until all relations have been
                  -- written.
                  SeqAlgebra.CreateSeq (TheHeap => TheHeap,
                                        S       => RelationsWritten);
                  RelationCount := CompressedIndex; -- Keep a record of the total number of relations
                  loop -- until all exports have been written
                     FirstAlphaIndex := Natural'First;
                     LocalIndex := RelationCount;
                     FirstAlphaExport := LexTokenManager.Null_String;
                     -- Go through all the export sequences (skipping those we have already written) looking
                     -- for the first export in alphabetical order.
                     while LocalIndex > Natural'First loop
                        if not SeqAlgebra.IsMember (TheHeap    => TheHeap,
                                                    S          => RelationsWritten,
                                                    GivenValue => LocalIndex) then
                           LexTokenManager.Relation_Algebra.Row_Extraction (The_Heap    => TheHeap,
                                                                            R           => CompressedExports,
                                                                            Given_Index => LocalIndex,
                                                                            S           => ExportsToWrite);
                           -- Find the first member of the current export sequence.
                           -- In this case we use initialise to find us the first member but we never go on to iterate
                           -- over the list.
                           SPARKProgram.Iteration.Initialise (TheHeap     => TheHeap,
                                                              TheSeq      => ExportsToWrite,
                                                              TheIterator => TheIterator);
                           -- If this is the first iteration then the item just returned must be the best match so far.
                           -- Otherwise, if it is better than our current best match then this export sequence becomes the one to write
                           CurrentString := SPARKProgram.Iteration.CurrentString (TheIterator);
                           if FirstAlphaIndex = Natural'First or else -- always do it on first iteration
                             LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => CurrentString,
                                                                                  Lex_Str2 => FirstAlphaExport) = LexTokenManager.Str_First then
                              FirstAlphaExport := CurrentString;
                              FirstAlphaIndex := LocalIndex;
                           end if;
                        end if;
                        LocalIndex := LocalIndex - 1;
                     end loop;
                     if FirstAlphaIndex = Natural'First then -- defensive - should always be false
                        exit;
                     end if;
                     -- FirstAlphaIndex gives us the relation index for the next dependency to be written
                     -- Write it out, then add it to the sequence of relations that have been written.
                     LexTokenManager.Relation_Algebra.Row_Extraction (The_Heap    => TheHeap,
                                                                      R           => CompressedExports,
                                                                      Given_Index => FirstAlphaIndex,
                                                                      S           => ExportsToWrite);
                     LexTokenManager.Relation_Algebra.Row_Extraction (The_Heap    => TheHeap,
                                                                      R           => CompressedImports,
                                                                      Given_Index => FirstAlphaIndex,
                                                                      S           => ImportsToWrite);
                     WriteDependencies (Output      => Output,
                                        StartCol    => StartCol,
                                        ExportWidth => ExportWidth,
                                        FirstExport => FirstExport,
                                        TheHeap     => TheHeap,
                                        Exports     => ExportsToWrite,
                                        Imports     => ImportsToWrite);
                     LexTokenManager.Seq_Algebra.Dispose_Of_Seq (The_Heap => TheHeap,
                                                                 S        => ExportsToWrite);
                     LexTokenManager.Seq_Algebra.Dispose_Of_Seq (The_Heap => TheHeap,
                                                                 S        => ImportsToWrite);
                     SeqAlgebra.AddMember (TheHeap    => TheHeap,
                                           S          => RelationsWritten,
                                           GivenValue => FirstAlphaIndex);
                     -- Exit if all relations have now been added to RelationsWritten
                     exit when SeqAlgebra.Length (TheHeap, RelationsWritten) >= RelationCount;
                  end loop; -- until all relations written
                  SeqAlgebra.DisposeOfSeq (TheHeap, RelationsWritten);
               end WriteDependenciesAlphabetic;

            begin -- WriteSimilarClauses
               -- Compress the dependencies if possible. Store the results in CompressedExports
               -- and CompressedImports, indexed by CompressedIndex.
               BuildCompressedDependencies (TheHeap           => TheHeap,
                                            Dependencies      => Dependencies,
                                            Exports           => Exports,
                                            CompressedExports => CompressedExports,
                                            CompressedImports => CompressedImports,
                                            CompressedIndex   => CompressedIndex);
               -- Extract the compressed dependencies and write them out.
               WriteDependenciesAlphabetic (TheHeap           => TheHeap,
                                            Output            => Output,
                                            StartCol          => StartCol,
                                            ExportWidth       => ExportWidth,
                                            FirstExport       => FirstExport,
                                            CompressedExports => CompressedExports,
                                            CompressedImports => CompressedImports,
                                            CompressedIndex   => CompressedIndex);
            end WriteSimilarClauses;

            procedure WriteClauses (Output       : in     SPARK_IO.File_Type;
                                    StartCol     : in     EStrings.Positions;
                                    ExportWidth  : in     EStrings.Lengths;
                                    FirstExport  : in out Boolean;
                                    TheHeap      : in out Heap.HeapRecord;
                                    Exports      : in     LexTokenManager.Seq_Algebra.Seq;
                                    Dependencies : in     LexTokenManager.Relation_Algebra.String.Relation)
            --# global in     Annotation.StartString;
            --#        in     LexTokenManager.State;
            --#        in     MultiplyToken;
            --#        in     RWNullToken;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --#        in out Statistics.TableUsage;
            --# derives FirstExport,
            --#         Statistics.TableUsage,
            --#         TheHeap               from *,
            --#                                    Dependencies,
            --#                                    Exports,
            --#                                    LexTokenManager.State,
            --#                                    MultiplyToken,
            --#                                    RWNullToken,
            --#                                    TheHeap &
            --#         SPARK_IO.File_Sys     from *,
            --#                                    Annotation.StartString,
            --#                                    Dependencies,
            --#                                    Exports,
            --#                                    ExportWidth,
            --#                                    FirstExport,
            --#                                    LexTokenManager.State,
            --#                                    MultiplyToken,
            --#                                    Output,
            --#                                    RWNullToken,
            --#                                    SparkFormatCommandLineData.Content,
            --#                                    StartCol,
            --#                                    TheHeap;
            is

               Imports                     : LexTokenManager.Seq_Algebra.Seq;
               ExportVal                   : LexTokenManager.Lex_String;
               DependenciesWithoutMultiply : LexTokenManager.Relation_Algebra.String.Relation;
               TheIterator                 : SPARKProgram.Iteration.Iterator;

            begin

               -- This writes out the clauses in uncompressed form, ie no lists of exports sharing the
               -- same list of imports, and no '*'.

               LexTokenManager.Relation_Algebra.String.Create_Relation (The_Heap => TheHeap,
                                                                        R        => DependenciesWithoutMultiply);
               SubstituteSelfForMultiply (TheHeap                     => TheHeap,
                                          Exports                     => Exports,
                                          DependenciesWithMultiply    => Dependencies,
                                          DependenciesWithoutMultiply => DependenciesWithoutMultiply);

               SPARKProgram.Iteration.Initialise (TheHeap     => TheHeap,
                                                  TheSeq      => Exports,
                                                  TheIterator => TheIterator);

               while not SPARKProgram.Iteration.Complete (TheIterator) loop

                  ExportVal := LexTokenManager.Seq_Algebra.Value_Of_Member
                    (The_Heap => TheHeap,
                     M        => SPARKProgram.Iteration.CurrentMember (TheIterator));

                  -- Null dependency clause must come last so don't write it here
                  if LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => ExportVal,
                     Lex_Str2 => RWNullToken) /= LexTokenManager.Str_Eq then

                     -- export
                     WriteExport (FirstExport => FirstExport,
                                  Output      => Output,
                                  StartCol    => StartCol,
                                  TheHeap     => TheHeap,
                                  Export      => SPARKProgram.Iteration.CurrentMember (TheIterator));

                     WriteFrom (Output      => Output,
                                StartCol    => StartCol,
                                ExportWidth => ExportWidth); --from

                     -- imports
                     LexTokenManager.Relation_Algebra.String.Row_Extraction (The_Heap    => TheHeap,
                                                                             R           => DependenciesWithoutMultiply,
                                                                             Given_Index => ExportVal,
                                                                             S           => Imports);
                     WriteImports (Output      => Output,
                                   StartCol    => StartCol,
                                   ExportWidth => ExportWidth,
                                   TheHeap     => TheHeap,
                                   Imports     => Imports);
                     LexTokenManager.Seq_Algebra.Dispose_Of_Seq (The_Heap => TheHeap,
                                                                 S        => Imports);
                  end if;

                  SPARKProgram.Iteration.Next (TheHeap, TheIterator);

               end loop;

               LexTokenManager.Relation_Algebra.String.Dispose_Of_Relation (The_Heap => TheHeap,
                                                                            R        => DependenciesWithoutMultiply);

            end WriteClauses;

            procedure WriteNullImportList (Output       : in     SPARK_IO.File_Type;
                                           StartCol     : in     EStrings.Positions;
                                           ExportWidth  : in     EStrings.Lengths;
                                           FirstExport  : in     Boolean;
                                           TheHeap      : in out Heap.HeapRecord;
                                           Dependencies : in     LexTokenManager.Relation_Algebra.String.Relation)
            --# global in     Annotation.StartString;
            --#        in     LexTokenManager.State;
            --#        in     RWNullToken;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --#        in out Statistics.TableUsage;
            --# derives SPARK_IO.File_Sys     from *,
            --#                                    Annotation.StartString,
            --#                                    Dependencies,
            --#                                    ExportWidth,
            --#                                    FirstExport,
            --#                                    LexTokenManager.State,
            --#                                    Output,
            --#                                    RWNullToken,
            --#                                    SparkFormatCommandLineData.Content,
            --#                                    StartCol,
            --#                                    TheHeap &
            --#         Statistics.TableUsage,
            --#         TheHeap               from *,
            --#                                    Dependencies,
            --#                                    LexTokenManager.State,
            --#                                    RWNullToken,
            --#                                    TheHeap;
            is
               Imports     : LexTokenManager.Seq_Algebra.Seq;
               Import      : LexTokenManager.Seq_Algebra.Member_Of_Seq;
               FirstImport : Boolean;
               TheIterator : SPARKProgram.Iteration.Iterator;

               procedure WriteNull (FirstExport : in Boolean;
                                    Output      : in SPARK_IO.File_Type;
                                    StartCol    : in EStrings.Positions)
               --# global in     Annotation.StartString;
               --#        in     SparkFormatCommandLineData.Content;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Annotation.StartString,
               --#                                FirstExport,
               --#                                Output,
               --#                                SparkFormatCommandLineData.Content,
               --#                                StartCol;
               is
               begin
                  if not FirstExport then
                     SPARK_IO.Put_Line (Output, " &", 0);
                     Annotation.Write (Output, StartCol);
                  elsif SparkFormatCommandLineData.Content.ExportIndent /= SparkFormatCommandLineData.Inline then
                     Annotation.Write (Output, StartCol);
                  end if;

                  SPARK_IO.Set_Col (Output, ExportCol (StartCol));
                  SPARK_IO.Put_String (Output, "null", 0);
               end WriteNull;

            begin

               LexTokenManager.Relation_Algebra.String.Row_Extraction (The_Heap    => TheHeap,
                                                                       R           => Dependencies,
                                                                       Given_Index => RWNullToken,
                                                                       S           => Imports);

               -- This procedure can be called even if there is no null import list. If the
               -- sequence is null then don't write it out.
               Import := LexTokenManager.Seq_Algebra.First_Member (The_Heap => TheHeap,
                                                                   S        => Imports);

               if not LexTokenManager.Seq_Algebra.Is_Null_Member (M => Import) then

                  SPARKProgram.Iteration.Initialise (TheHeap     => TheHeap,
                                                     TheSeq      => Imports,
                                                     TheIterator => TheIterator);

                  WriteNull (FirstExport => FirstExport,
                             Output      => Output,
                             StartCol    => StartCol); -- null
                  WriteFrom (Output      => Output,
                             StartCol    => StartCol,
                             ExportWidth => ExportWidth); -- from

                  -- Now write out the list of imports in the requested order
                  FirstImport := True;

                  WriteImport (FirstImport => FirstImport,
                               Output      => Output,
                               StartCol    => StartCol,
                               ExportWidth => ExportWidth,
                               TheHeap     => TheHeap,
                               Import      => SPARKProgram.Iteration.CurrentMember (TheIterator));

                  SPARKProgram.Iteration.Next (TheHeap, TheIterator);

                  while not SPARKProgram.Iteration.Complete (TheIterator) loop

                     WriteImport (FirstImport => FirstImport,
                                  Output      => Output,
                                  StartCol    => StartCol,
                                  ExportWidth => ExportWidth,
                                  TheHeap     => TheHeap,
                                  Import      => SPARKProgram.Iteration.CurrentMember (TheIterator));

                     SPARKProgram.Iteration.Next (TheHeap, TheIterator);

                  end loop;

               end if;

               LexTokenManager.Seq_Algebra.Dispose_Of_Seq (The_Heap => TheHeap,
                                                           S        => Imports);

            end WriteNullImportList;

         begin

            if OK then

               Annotation.Write (Output, StartCol);
               WriteDerives (Output, StartCol);
               FirstExport := True;
               ExportWidth := LongestExport (TheHeap => TheHeap,
                                             Exports => Exports);

               case SparkFormatCommandLineData.Content.Operation is

                  when SparkFormatCommandLineData.Compress =>

                     WriteSimilarClauses (Output       => Output,
                                          StartCol     => StartCol,
                                          ExportWidth  => ExportWidth,
                                          FirstExport  => FirstExport,
                                          TheHeap      => TheHeap,
                                          Exports      => Exports,
                                          Dependencies => Dependencies);

                  when SparkFormatCommandLineData.Expand =>

                     WriteClauses (Output       => Output,
                                   StartCol     => StartCol,
                                   ExportWidth  => ExportWidth,
                                   FirstExport  => FirstExport,
                                   TheHeap      => TheHeap,
                                   Exports      => Exports,
                                   Dependencies => Dependencies);

               end case;

               WriteNullImportList (Output       => Output,
                                    StartCol     => StartCol,
                                    ExportWidth  => ExportWidth,
                                    FirstExport  => FirstExport,
                                    TheHeap      => TheHeap,
                                    Dependencies => Dependencies);

               SPARK_IO.Put_Line (Output, ";", 0);

            else

               File_IO.Reset (TemporaryFile);
               SPARKProgram.Copy (TemporaryFile, Output);

            end if;

            LexTokenManager.Seq_Algebra.Dispose_Of_Seq (The_Heap => TheHeap,
                                                        S        => Exports);
            LexTokenManager.Relation_Algebra.String.Dispose_Of_Relation (The_Heap => TheHeap,
                                                                         R        => Dependencies);
            File_IO.Delete (TemporaryFile);

         end Reformat;

      end DerivesList;

      package body DeclareList
      is
         Keyword : constant String := "declare";

         -- checks it's at the start of a line up to "--# declare"
         function IsStart (InputLine : EStrings.T) return Boolean
         is
         begin
            return SPARKKeywordMatching.MatchAtStart (InputLine, Keyword);
         end IsStart;

         -- checks for ";" at end of line
         function IsEnd (InputLine : EStrings.T) return Boolean
         is
            Result : Boolean;
         begin

            Result := False;

            for I in EStrings.Lengths range EStrings.Positions'First .. EStrings.Get_Length (E_Str => InputLine) loop
               if EStrings.Get_Element (E_Str => InputLine,
                                        Pos   => I) = ';' then
                  Result := True;
                  exit;
               end if;
            end loop;

            return Result;

         end IsEnd;

      end DeclareList;

      package body ProofList
      is
         KeywordPre    : constant String := "pre";
         KeywordPost   : constant String := "post";
         KeywordReturn : constant String := "return";


         -- checks it's at the start of a line up to "--# pre" or "--# post" or "--# return"
         function IsStart (InputLine : EStrings.T) return Boolean
         is
         begin
            return SPARKKeywordMatching.MatchAtStart (InputLine, KeywordPre) or
                   SPARKKeywordMatching.MatchAtStart (InputLine, KeywordPost) or
                   SPARKKeywordMatching.MatchAtStart (InputLine, KeywordReturn);
         end IsStart;

         -- The 'IsEnd' for proof annotations below is not called anywhere, and thus has
         -- not been tested.

         -- checks for ";" at end of line
         function IsEnd (InputLine : EStrings.T) return Boolean
         is
            Result : Boolean;
         begin

            Result := False;

            for I in EStrings.Lengths range EStrings.Positions'First .. EStrings.Get_Length (E_Str => InputLine) loop
               if EStrings.Get_Element (E_Str => InputLine,
                                        Pos   => I) = ';' then
                  Result := True;
                  exit;
               end if;
            end loop;

            return Result;

         end IsEnd;

      end ProofList;

      package body GlobalsList
      is
         Keyword : constant String := "global";

         ConsecutiveEmptyLines : Natural := 0;

         -- checks it's at the start of a line up to "--# global"
         function IsStart (InputLine : EStrings.T) return Boolean
         is
         begin
            return SPARKKeywordMatching.MatchAtStart (InputLine, Keyword);
         end IsStart;

         --Here the end of a list of global declarations is detected.
         --Unfortunately, the SPARK syntax for globals does not have a unique terminator.
         --The ';' is used as a separator between different modes or as a terminator. Thus,
         --The end of a list of globals is detected by the end of a SPARK comment block, or
         --the start of a new comment item. Every possible following SPARK annotation item
         --needs to be considered, otherwise the real end of a global declaration may not be
         --detected. See SEPR: 2674
         function IsEnd (InputLine : EStrings.T) return Boolean
         is
            Result            : Boolean;
            IsAnnotationStart : Boolean;
            Index             : EStrings.Lengths;
         begin

            if DerivesList.IsStart (InputLine)  or
               DeclareList.IsStart (InputLine)  or
               ProofList.IsStart (InputLine)    or
               AcceptList.IsStart (InputLine)   or
               EndList.IsStart (InputLine)      or
               ForList.IsStart (InputLine)      or
               FunctionList.IsStart (InputLine) or
               TypeList.IsStart (InputLine)     or
               AssertList.IsStart (InputLine)   then
               Result := True;
            else
               Index  := 1;
               --# accept Flow_Message, 10, Index, "Final value of Index is not required";
               Annotation.IsStart (InputLine, Index, IsAnnotationStart);
               --# end accept;
               Result := not (IsAnnotationStart or
                              EStrings.Is_Empty (E_Str => EStrings.Trim (InputLine)));
            end if;

            return Result;

         end IsEnd;

         procedure IncrementConsecutiveEmptyLines
         is
         begin
            ConsecutiveEmptyLines := ConsecutiveEmptyLines + 1;
         end IncrementConsecutiveEmptyLines;

         procedure ResetConsecutiveEmptyLines
         is
         begin
            ConsecutiveEmptyLines := 0;
         end ResetConsecutiveEmptyLines;

         function NoEmptyLines return Boolean
         is
         begin
            return ConsecutiveEmptyLines = 0;
         end NoEmptyLines;

         procedure Parse (TemporaryFile   : in out SPARK_IO.File_Type;
                          StartCol        :    out EStrings.Positions;
                          TheHeap         : in out Heap.HeapRecord;
                          GlobalVariables :    out GlobalVariablesType;
                          OK              :    out Boolean)
         is

            Token  : SPSymbols.SPTerminal;
            LexVal : LexTokenManager.Lex_Value;

            procedure Create (TheHeap         : in out Heap.HeapRecord;
                              GlobalVariables :    out GlobalVariablesType)
            --# global in out Statistics.TableUsage;
            --# derives GlobalVariables       from TheHeap &
            --#         Statistics.TableUsage,
            --#         TheHeap               from *,
            --#                                    TheHeap;
            is
               UnmodedGlobalVariables,
               InGlobalVariables,
               InOutGlobalVariables,
               OutGlobalVariables     : LexTokenManager.Seq_Algebra.Seq;
            begin
               LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                       S        => UnmodedGlobalVariables);
               LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                       S        => InGlobalVariables);
               LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                       S        => InOutGlobalVariables);
               LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                       S        => OutGlobalVariables);
               GlobalVariables := GlobalVariablesType'(Unmoded   => UnmodedGlobalVariables,
                                                       InMode    => InGlobalVariables,
                                                       InOutMode => InOutGlobalVariables,
                                                       OutMode   => OutGlobalVariables);
            end Create;

            procedure ParseGlobalVariableClauses (TemporaryFile   : in     SPARK_IO.File_Type;
                                                  Token           : in     SPSymbols.SPTerminal;
                                                  LexVal          : in     LexTokenManager.Lex_Value;
                                                  TheHeap         : in out Heap.HeapRecord;
                                                  GlobalVariables : in     GlobalVariablesType;
                                                  OK              :    out Boolean)
            --# global in     CommandLineData.Content;
            --#        in     Dictionary.Dict;
            --#        in out ErrorHandler.ErrorContext;
            --#        in out LexTokenManager.State;
            --#        in out SparkLex.Currline;
            --#        in out SPARK_IO.File_Sys;
            --#        in out Statistics.TableUsage;
            --# derives ErrorHandler.ErrorContext,
            --#         LexTokenManager.State,
            --#         OK,
            --#         SparkLex.Currline,
            --#         SPARK_IO.File_Sys         from CommandLineData.Content,
            --#                                        Dictionary.Dict,
            --#                                        ErrorHandler.ErrorContext,
            --#                                        LexTokenManager.State,
            --#                                        LexVal,
            --#                                        SparkLex.Currline,
            --#                                        SPARK_IO.File_Sys,
            --#                                        TemporaryFile,
            --#                                        Token &
            --#         Statistics.TableUsage,
            --#         TheHeap                   from *,
            --#                                        CommandLineData.Content,
            --#                                        Dictionary.Dict,
            --#                                        ErrorHandler.ErrorContext,
            --#                                        GlobalVariables,
            --#                                        LexTokenManager.State,
            --#                                        LexVal,
            --#                                        SparkLex.Currline,
            --#                                        SPARK_IO.File_Sys,
            --#                                        TemporaryFile,
            --#                                        TheHeap,
            --#                                        Token;
            is

               NextToken  : SPSymbols.SPTerminal;
               NextLexVal : LexTokenManager.Lex_Value;

               procedure ParseGlobalVariableList (TemporaryFile   : in     SPARK_IO.File_Type;
                                                  Token           : in out SPSymbols.SPTerminal;
                                                  LexVal          : in out LexTokenManager.Lex_Value;
                                                  TheHeap         : in out Heap.HeapRecord;
                                                  GlobalVariables : in     GlobalVariablesType;
                                                  OK              :    out Boolean)
               --# global in     CommandLineData.Content;
               --#        in     Dictionary.Dict;
               --#        in out ErrorHandler.ErrorContext;
               --#        in out LexTokenManager.State;
               --#        in out SparkLex.Currline;
               --#        in out SPARK_IO.File_Sys;
               --#        in out Statistics.TableUsage;
               --# derives ErrorHandler.ErrorContext,
               --#         LexTokenManager.State,
               --#         LexVal,
               --#         OK,
               --#         SparkLex.Currline,
               --#         SPARK_IO.File_Sys,
               --#         Token                     from CommandLineData.Content,
               --#                                        Dictionary.Dict,
               --#                                        ErrorHandler.ErrorContext,
               --#                                        LexTokenManager.State,
               --#                                        LexVal,
               --#                                        SparkLex.Currline,
               --#                                        SPARK_IO.File_Sys,
               --#                                        TemporaryFile,
               --#                                        Token &
               --#         Statistics.TableUsage,
               --#         TheHeap                   from *,
               --#                                        CommandLineData.Content,
               --#                                        Dictionary.Dict,
               --#                                        ErrorHandler.ErrorContext,
               --#                                        GlobalVariables,
               --#                                        LexTokenManager.State,
               --#                                        LexVal,
               --#                                        SparkLex.Currline,
               --#                                        SPARK_IO.File_Sys,
               --#                                        TemporaryFile,
               --#                                        TheHeap,
               --#                                        Token;
               is
                  Mode             : Modes;
                  DottedSimpleName : EStrings.T;
                  GlobalVariable   : LexTokenManager.Lex_String;
               begin

                  OK := True;
                  Mode := Unmoded;

                  case Token is
                     when SPSymbols.RWin =>
                        File_IO.Lex (TemporaryFile, Token, LexVal);
                        case Token is
                           when SPSymbols.RWout =>
                              Mode := InOutMode;
                              File_IO.Lex (TemporaryFile, Token, LexVal);
                           when SPSymbols.identifier =>
                              Mode := InMode;
                           when others =>
                              OK := False;
                        end case;
                     when SPSymbols.RWout =>
                        File_IO.Lex (TemporaryFile, Token, LexVal);
                        Mode := OutMode;
                     when SPSymbols.identifier =>
                        Mode := Unmoded;
                     when others =>
                        OK := False;
                  end case;

                  loop
                     exit when not OK;
                     case Token is
                        when SPSymbols.identifier =>
                           DottedSimpleName := LexTokenManager.Lex_String_To_String (Lex_Str => LexVal.Token_Str);
                           File_IO.Lex (TemporaryFile, Token, LexVal);
                           loop
                              exit when Token /= SPSymbols.point;
                              EStrings.Append_String (E_Str => DottedSimpleName,
                                                      Str   => ".");
                              File_IO.Lex (TemporaryFile, Token, LexVal);
                              if Token = SPSymbols.identifier then
                                 EStrings.Append_Examiner_String
                                   (E_Str1 => DottedSimpleName,
                                    E_Str2 => LexTokenManager.Lex_String_To_String (Lex_Str => LexVal.Token_Str));
                                 File_IO.Lex (TemporaryFile, Token, LexVal);
                              else
                                 OK := False;
                              end if;
                              exit when not OK;
                           end loop;
                           if OK then
                              LexTokenManager.Insert_Examiner_String (Str     => DottedSimpleName,
                                                                      Lex_Str => GlobalVariable);
                              LexTokenManager.Seq_Algebra.Add_Member (The_Heap    => TheHeap,
                                                                      S           => GlobalVariables (Mode),
                                                                      Given_Value => GlobalVariable);
                           end if;
                        when SPSymbols.comma =>
                           null;
                        when others =>
                           OK := False;
                     end case;
                     exit when not OK or Token = SPSymbols.semicolon;
                     if Token = SPSymbols.comma then
                        File_IO.Lex (TemporaryFile, Token, LexVal);
                     else
                        OK := False;
                     end if;
                  end loop;

               end ParseGlobalVariableList;

            begin

               OK := True;
               NextToken := Token;
               NextLexVal := LexVal;

               loop
                  exit when NextToken = SPSymbols.annotation_end;
                  ParseGlobalVariableList (TemporaryFile, NextToken, NextLexVal, TheHeap, GlobalVariables, OK);
                  if OK then
                     File_IO.Lex (TemporaryFile, NextToken, NextLexVal);
                     case NextToken is
                        when SPSymbols.annotation_end | SPSymbols.RWin | SPSymbols.RWout | SPSymbols.identifier =>
                           null;
                        when others =>
                           OK := False;
                     end case;
                  end if;
                  exit when not OK;
               end loop;

            end ParseGlobalVariableClauses;

         begin

            File_IO.Reset (TemporaryFile);
            SparkLex.ClearLineContext;
            Create (TheHeap         => TheHeap,
                    GlobalVariables => GlobalVariables);
            File_IO.Lex (TemporaryFile, Token, LexVal);

            if Token = SPSymbols.annotation_start then
               StartCol := LexVal.Position.Start_Pos; -- we know StartCol can't be zero because the line can't be empty
               --# accept Flow_Message, 10, LexVal, "The next symbol should be 'global', so we don't need the LexVal";
               File_IO.Lex (TemporaryFile, Token, LexVal); -- Assignment to LexVal is ineffective
               --# end accept;
               if Token = SPSymbols.RWglobal then
                  File_IO.Lex (TemporaryFile, Token, LexVal);
                  ParseGlobalVariableClauses (TemporaryFile, Token, LexVal, TheHeap, GlobalVariables, OK);
               else
                  OK := False;
               end if;
            else
               OK := False;
            end if;
            --# accept Flow_Message, 602, StartCol, StartCol, "StartCol may not be set if not OK";
         end Parse;

         procedure Reformat (OK                   : in     Boolean;
                             TemporaryFile        : in out SPARK_IO.File_Type;
                             Output               : in     SPARK_IO.File_Type;
                             StartCol             : in     EStrings.Positions;
                             TheHeap              : in out Heap.HeapRecord;
                             FunctionOrProcedure  : in     Subprogram.SubprogramType;
                             GlobalVariables      : in     GlobalVariablesType;
                             Imports              : in     LexTokenManager.Seq_Algebra.Seq;
                             Exports              : in     LexTokenManager.Seq_Algebra.Seq)
         is

            RevisedGlobalVariables : GlobalVariablesType;
            FirstGlobalVariable    : Boolean;
            AnExportIsExcluded     : Boolean;

            function GlobalCol (StartCol : EStrings.Positions) return EStrings.Positions
            is
            begin
               return StartCol + 4; -- length of "--# "
            end GlobalCol;

            procedure WriteGlobal (Output   : in SPARK_IO.File_Type;
                                   StartCol : in EStrings.Positions)
            --# global in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Output,
            --#                                SparkFormatCommandLineData.Content,
            --#                                StartCol;
            is
            begin
               SPARK_IO.Set_Col (Output, GlobalCol (StartCol));
               SPARK_IO.Put_String (Output, "global", 0);
               -- If GlobalIndent is not Inline, then don't put out a trailing space,
               -- since this will be rejected by GNAT -gnatyb style check mode.
               if SparkFormatCommandLineData.Content.GlobalIndent = SparkFormatCommandLineData.Inline then
                  SPARK_IO.Put_String (Output, " ", 0);
               end if;
            end WriteGlobal;

            procedure RemoveExcludedExport (TheHeap         : in out Heap.HeapRecord;
                                            Imports         : in     LexTokenManager.Seq_Algebra.Seq;
                                            Exports         : in     LexTokenManager.Seq_Algebra.Seq;
                                            GlobalVariables : in     GlobalVariablesType)
            --# global in     LexTokenManager.State;
            --#        in out Statistics.TableUsage;
            --# derives Statistics.TableUsage,
            --#         TheHeap               from *,
            --#                                    Exports,
            --#                                    GlobalVariables,
            --#                                    Imports,
            --#                                    LexTokenManager.State,
            --#                                    TheHeap;
            is
               GlobalVariable    : LexTokenManager.Seq_Algebra.Member_Of_Seq;
               GlobalVariableVal : LexTokenManager.Lex_String;
               --  GlobalVariableStr : EStrings.T;
            begin
               -- SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
               --   "Entering RemoveExcludedExport",
               --    0);

               --  Iterate through all global variable modes to remove
               --  excluded export and all imported variables exclusively
               --  related to the excluded exported variable.

               --  An out mode global which is no longer exported should be removed
               GlobalVariable := LexTokenManager.Seq_Algebra.First_Member (The_Heap => TheHeap,
                                                                           S        => GlobalVariables (OutMode));

               while not LexTokenManager.Seq_Algebra.Is_Null_Member (M => GlobalVariable) loop
                  -- SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                  --   "RemoveExcludedExport: Checking for out mode globals",
                  --   0);
                  GlobalVariableVal := LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                                    M        => GlobalVariable);
                  GlobalVariable := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                             M        => GlobalVariable);

                  if not LexTokenManager.Seq_Algebra.Is_Member (The_Heap    => TheHeap,
                                                                S           => Exports,
                                                                Given_Value => GlobalVariableVal) then
                     --  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                     --     "RemoveExcludedExport: Removing variable from out mode globals",
                     --     0);
                     LexTokenManager.Seq_Algebra.Remove_Member (The_Heap    => TheHeap,
                                                                S           => GlobalVariables (OutMode),
                                                                Given_Value => GlobalVariableVal);
                  end if;
               end loop;

               --  An in out mode global which is no longer exported should
               --  be removed from the in out mode globals and if it is still
               --  and imported variable it should be added to the in mode globals
               GlobalVariable := LexTokenManager.Seq_Algebra.First_Member (The_Heap => TheHeap,
                                                                           S        => GlobalVariables (InOutMode));

               while not LexTokenManager.Seq_Algebra.Is_Null_Member (M => GlobalVariable) loop
                  --  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                  --     "RemoveExcludedExport: Checking for in out mode globals",
                  --     0);
                  GlobalVariableVal := LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                                    M        => GlobalVariable);
                  GlobalVariable := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                             M        => GlobalVariable);

                  if not LexTokenManager.Seq_Algebra.Is_Member (The_Heap    => TheHeap,
                                                                S           => Exports,
                                                                Given_Value => GlobalVariableVal) then
                     --  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                     --     "RemoveExcludedExport: Removing variable from in out mode globals",
                     --     0);
                     LexTokenManager.Seq_Algebra.Remove_Member (The_Heap    => TheHeap,
                                                                S           => GlobalVariables (InOutMode),
                                                                Given_Value => GlobalVariableVal);
                     if LexTokenManager.Seq_Algebra.Is_Member (The_Heap    => TheHeap,
                                                               S           => Imports,
                                                               Given_Value => GlobalVariableVal) then
                        --  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                        --     "RemoveExcludedExport: Adding variable to in mode globals",
                        --     0);
                        LexTokenManager.Seq_Algebra.Add_Member (The_Heap    => TheHeap,
                                                                S           => GlobalVariables (InMode),
                                                                Given_Value => GlobalVariableVal);
                     end if;
                  end if;
               end loop;

               --  An in mode global which is no longer imported should
               --  be removed from the in mode globals.
               GlobalVariable := LexTokenManager.Seq_Algebra.First_Member (The_Heap => TheHeap,
                                                                           S        => GlobalVariables (InMode));

               while not LexTokenManager.Seq_Algebra.Is_Null_Member (M => GlobalVariable) loop
                  --  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                  --     "RemoveExcludedExport: Checking for in mode globals",
                  --     0);
                  GlobalVariableVal := LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                                    M        => GlobalVariable);
                  GlobalVariable := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                             M        => GlobalVariable);

                  if not LexTokenManager.Seq_Algebra.Is_Member (The_Heap    => TheHeap,
                                                                S           => Imports,
                                                                Given_Value => GlobalVariableVal) then
                     --  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                     --     "RemoveExcludedExport: Removing variable from in mode globals",
                     --     0);
                     LexTokenManager.Seq_Algebra.Remove_Member (The_Heap    => TheHeap,
                                                                S           => GlobalVariables (InMode),
                                                                Given_Value => GlobalVariableVal);
                  end if;
               end loop;

               --  An unmoded global which is no longer exported or imported should
               --  be removed from the unmoded globals.
               GlobalVariable := LexTokenManager.Seq_Algebra.First_Member (The_Heap => TheHeap,
                                                                           S        => GlobalVariables (Unmoded));

               while not LexTokenManager.Seq_Algebra.Is_Null_Member (M => GlobalVariable) loop
                  --  SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                  --       "RemoveExcludedExport: Checking for unmoded globals: ",
                  --       0);

                  GlobalVariableVal := LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                                    M        => GlobalVariable);
                  GlobalVariable := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                             M        => GlobalVariable);

                  --  LexTokenManager.LexStringToString (
                  --     LexTokenManager.ConvertLexStringRef (GlobalVariableVal),
                  --     GlobalVariableStr);

                  --  EStrings.PutLine (SPARK_IO.Standard_Output, GlobalVariableStr);

                  if not LexTokenManager.Seq_Algebra.Is_Member (The_Heap    => TheHeap,
                                                                S           => Exports,
                                                                Given_Value => GlobalVariableVal) and then
                    not LexTokenManager.Seq_Algebra.Is_Member (The_Heap    => TheHeap,
                                                               S           => Imports,
                                                               Given_Value => GlobalVariableVal) then
                     --  SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                     --       "RemoveExcludedExport: Removing variable from unmoded globals: ",
                     --       0);
                     --  EStrings.PutLine (SPARK_IO.Standard_Output, GlobalVariableStr);
                     LexTokenManager.Seq_Algebra.Remove_Member (The_Heap    => TheHeap,
                                                                S           => GlobalVariables (Unmoded),
                                                                Given_Value => GlobalVariableVal);
                  end if;
               end loop;
            end RemoveExcludedExport;

            procedure AddModes (TheHeap             : in out Heap.HeapRecord;
                                FunctionOrProcedure : in     Subprogram.SubprogramType;
                                Imports             : in     LexTokenManager.Seq_Algebra.Seq;
                                Exports             : in     LexTokenManager.Seq_Algebra.Seq;
                                GlobalVariables     : in out GlobalVariablesType)
            --# global in     LexTokenManager.State;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out Statistics.TableUsage;
            --# derives GlobalVariables       from *,
            --#                                    FunctionOrProcedure,
            --#                                    SparkFormatCommandLineData.Content,
            --#                                    TheHeap &
            --#         Statistics.TableUsage,
            --#         TheHeap               from *,
            --#                                    Exports,
            --#                                    FunctionOrProcedure,
            --#                                    GlobalVariables,
            --#                                    Imports,
            --#                                    LexTokenManager.State,
            --#                                    SparkFormatCommandLineData.Content,
            --#                                    TheHeap;
            is

               procedure AddProcedureModes (TheHeap         : in out Heap.HeapRecord;
                                            Imports         : in     LexTokenManager.Seq_Algebra.Seq;
                                            Exports         : in     LexTokenManager.Seq_Algebra.Seq;
                                            GlobalVariables : in     GlobalVariablesType)
               --# global in     LexTokenManager.State;
               --#        in out Statistics.TableUsage;
               --# derives Statistics.TableUsage,
               --#         TheHeap               from *,
               --#                                    Exports,
               --#                                    GlobalVariables,
               --#                                    Imports,
               --#                                    LexTokenManager.State,
               --#                                    TheHeap;
               is
                  GlobalVariable    : LexTokenManager.Seq_Algebra.Member_Of_Seq;
                  GlobalVariableVal : LexTokenManager.Lex_String;
                  RevisedMode       : Modes;
               begin

                  GlobalVariable := LexTokenManager.Seq_Algebra.First_Member (The_Heap => TheHeap,
                                                                              S        => GlobalVariables (Unmoded));

                  loop

                     GlobalVariableVal := LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                                       M        => GlobalVariable);

                     if LexTokenManager.Seq_Algebra.Is_Member (The_Heap    => TheHeap,
                                                               S           => Imports,
                                                               Given_Value => GlobalVariableVal) then
                        if LexTokenManager.Seq_Algebra.Is_Member (The_Heap    => TheHeap,
                                                                  S           => Exports,
                                                                  Given_Value => GlobalVariableVal) then
                           RevisedMode := InOutMode;
                        else
                           RevisedMode := InMode;
                        end if;
                     elsif LexTokenManager.Seq_Algebra.Is_Member (The_Heap    => TheHeap,
                                                                  S           => Exports,
                                                                  Given_Value => GlobalVariableVal) then
                        RevisedMode := OutMode;
                     else
                        RevisedMode := Unmoded;
                     end if;

                     GlobalVariable := LexTokenManager.Seq_Algebra.Next_Member (The_Heap => TheHeap,
                                                                                M        => GlobalVariable);

                     if RevisedMode /= Unmoded then
                        LexTokenManager.Seq_Algebra.Remove_Member (The_Heap    => TheHeap,
                                                                   S           => GlobalVariables (Unmoded),
                                                                   Given_Value => GlobalVariableVal);
                        LexTokenManager.Seq_Algebra.Add_Member (The_Heap    => TheHeap,
                                                                S           => GlobalVariables (RevisedMode),
                                                                Given_Value => GlobalVariableVal);
                     end if;

                     exit when LexTokenManager.Seq_Algebra.Is_Null_Member (M => GlobalVariable);

                  end loop;

               end AddProcedureModes;

               procedure AddFunctionModes (TheHeap         : in out Heap.HeapRecord;
                                           GlobalVariables : in out GlobalVariablesType)
               --# global in     LexTokenManager.State;
               --#        in     SparkFormatCommandLineData.Content;
               --#        in out Statistics.TableUsage;
               --# derives Statistics.TableUsage,
               --#         TheHeap               from *,
               --#                                    GlobalVariables,
               --#                                    LexTokenManager.State,
               --#                                    SparkFormatCommandLineData.Content,
               --#                                    TheHeap &
               --#         GlobalVariables       from *,
               --#                                    GlobalVariables,
               --#                                    SparkFormatCommandLineData.Content,
               --#                                    TheHeap;
               is

                  procedure Copy (TheHeap         : in out Heap.HeapRecord;
                                  GlobalVariables : in out GlobalVariablesType;
                                  FromMode        : in Modes;
                                  ToMode          : in Modes)
                  --# global in     LexTokenManager.State;
                  --#        in out Statistics.TableUsage;
                  --# derives Statistics.TableUsage,
                  --#         TheHeap               from *,
                  --#                                    FromMode,
                  --#                                    GlobalVariables,
                  --#                                    LexTokenManager.State,
                  --#                                    TheHeap,
                  --#                                    ToMode &
                  --#         GlobalVariables       from *,
                  --#                                    FromMode,
                  --#                                    GlobalVariables,
                  --#                                    TheHeap,
                  --#                                    ToMode;
                  is
                     FromList    : LexTokenManager.Seq_Algebra.Seq;
                     ToList      : LexTokenManager.Seq_Algebra.Seq;
                     RevisedList : LexTokenManager.Seq_Algebra.Seq;
                  begin
                     FromList := GlobalVariables (FromMode);
                     ToList := GlobalVariables (ToMode);
                     LexTokenManager.Seq_Algebra.Union (The_Heap => TheHeap,
                                                        A        => FromList,
                                                        B        => ToList,
                                                        C        => RevisedList);
                     LexTokenManager.Seq_Algebra.Dispose_Of_Seq (The_Heap => TheHeap,
                                                                 S        => ToList);
                     LexTokenManager.Seq_Algebra.Reduction (The_Heap => TheHeap,
                                                            A        => FromList,
                                                            B        => RevisedList);
                     GlobalVariables (FromMode) := FromList;
                     GlobalVariables (ToMode) := RevisedList;
                  end Copy;

               begin

                  case SparkFormatCommandLineData.Content.DefaultFunctionMode is
                     when SparkFormatCommandLineData.Unmoded =>
                        Copy (TheHeap         => TheHeap,
                              GlobalVariables => GlobalVariables,
                              FromMode        => InMode,
                              ToMode          => Unmoded);
                     when SparkFormatCommandLineData.InMode =>
                        Copy (TheHeap         => TheHeap,
                              GlobalVariables => GlobalVariables,
                              FromMode        => Unmoded,
                              ToMode          => InMode);
                  end case;

               end AddFunctionModes;

            begin
               case FunctionOrProcedure is
                  when Subprogram.IsUnknown | Subprogram.IsProcedure =>
                     AddProcedureModes (TheHeap         => TheHeap,
                                        Imports         => Imports,
                                        Exports         => Exports,
                                        GlobalVariables => GlobalVariables);
                  when Subprogram.IsFunction =>
                     AddFunctionModes (TheHeap         => TheHeap,
                                       GlobalVariables => GlobalVariables);
               end case;
            end AddModes;

            procedure WriteGlobalVariableClauses (Output              : in     SPARK_IO.File_Type;
                                                  StartCol            : in     EStrings.Positions;
                                                  FirstGlobalVariable : in out Boolean;
                                                  TheHeap             : in     Heap.HeapRecord;
                                                  GlobalVariables     : in     GlobalVariablesType;
                                                  Mode                : in     Modes)
            --# global in     Annotation.StartString;
            --#        in     LexTokenManager.State;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --# derives FirstGlobalVariable from *,
            --#                                  GlobalVariables,
            --#                                  LexTokenManager.State,
            --#                                  Mode,
            --#                                  TheHeap &
            --#         SPARK_IO.File_Sys   from *,
            --#                                  Annotation.StartString,
            --#                                  FirstGlobalVariable,
            --#                                  GlobalVariables,
            --#                                  LexTokenManager.State,
            --#                                  Mode,
            --#                                  Output,
            --#                                  SparkFormatCommandLineData.Content,
            --#                                  StartCol,
            --#                                  TheHeap;
            is

               GlobalVariable : LexTokenManager.Seq_Algebra.Member_Of_Seq;
               TheIterator    : SPARKProgram.Iteration.Iterator;

               procedure WriteGlobalVariable (Output              : in     SPARK_IO.File_Type;
                                              StartCol            : in     EStrings.Positions;
                                              FirstGlobalVariable : in out Boolean;
                                              TheHeap             : in     Heap.HeapRecord;
                                              GlobalVariable      : in     LexTokenManager.Seq_Algebra.Member_Of_Seq;
                                              GlobalVariables     : in     GlobalVariablesType;
                                              Mode                : in     Modes)
               --# global in     Annotation.StartString;
               --#        in     LexTokenManager.State;
               --#        in     SparkFormatCommandLineData.Content;
               --#        in out SPARK_IO.File_Sys;
               --# derives FirstGlobalVariable from * &
               --#         SPARK_IO.File_Sys   from *,
               --#                                  Annotation.StartString,
               --#                                  FirstGlobalVariable,
               --#                                  GlobalVariable,
               --#                                  GlobalVariables,
               --#                                  LexTokenManager.State,
               --#                                  Mode,
               --#                                  Output,
               --#                                  SparkFormatCommandLineData.Content,
               --#                                  StartCol,
               --#                                  TheHeap;
               is

                  procedure WriteMode (Output          : in SPARK_IO.File_Type;
                                       StartCol        : in EStrings.Positions;
                                       TheHeap         : in Heap.HeapRecord;
                                       GlobalVariables : in GlobalVariablesType;
                                       Mode            : in Modes)
                  --# global in     SparkFormatCommandLineData.Content;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                GlobalVariables,
                  --#                                Mode,
                  --#                                Output,
                  --#                                SparkFormatCommandLineData.Content,
                  --#                                StartCol,
                  --#                                TheHeap;
                  is

                     function ModeCol (StartCol : EStrings.Positions) return EStrings.Positions
                     --# global in SparkFormatCommandLineData.Content;
                     is
                        ModeColumn : EStrings.Positions;
                     begin
                        if SparkFormatCommandLineData.Content.GlobalIndent >= 1 then
                           ModeColumn := GlobalCol (StartCol) + (SparkFormatCommandLineData.Content.GlobalIndent - 1); -- StartCol "is --# " but indent value is given from "--#"
                        else
                           ModeColumn := GlobalCol (StartCol) + 7; -- length of "global "
                        end if;
                        return ModeColumn;
                     end ModeCol;

                  begin

                     SPARK_IO.Set_Col (Output, ModeCol (StartCol));

                     case Mode is
                        when InMode =>
                           SPARK_IO.Put_String (Output, "in ", 0);
                           if not (LexTokenManager.Seq_Algebra.Is_Empty_Seq (The_Heap => TheHeap,
                                                                             S        => GlobalVariables (InOutMode)) and then
                                     LexTokenManager.Seq_Algebra.Is_Empty_Seq (The_Heap => TheHeap,
                                                                               S        => GlobalVariables (OutMode))) then
                              SPARK_IO.Put_String (Output, "    ", 0);
                           end if;
                        when OutMode =>
                           if not (LexTokenManager.Seq_Algebra.Is_Empty_Seq (The_Heap => TheHeap,
                                                                             S        => GlobalVariables (InMode)) and then
                                     LexTokenManager.Seq_Algebra.Is_Empty_Seq (The_Heap => TheHeap,
                                                                               S        => GlobalVariables (InOutMode))) then
                              SPARK_IO.Put_String (Output, "   ", 0);
                           end if;
                           SPARK_IO.Put_String (Output, "out ", 0);
                        when InOutMode =>
                           SPARK_IO.Put_String (Output, "in out ", 0);
                        when Unmoded =>
                           null;
                     end case;

                  end WriteMode;

               begin  -- WriteGlobalVariable

                  if FirstGlobalVariable then
                     FirstGlobalVariable := False;
                     if SparkFormatCommandLineData.Content.GlobalIndent >= 1 then
                        Annotation.Write (Output, StartCol);
                     end if;
                  else
                     case Mode is
                        when Unmoded =>
                           SPARK_IO.Put_Line (Output, ",", 0);
                        when InMode | OutMode | InOutMode =>
                           SPARK_IO.Put_Line (Output, ";", 0);
                     end case;
                     Annotation.Write (Output, StartCol);
                  end if;

                  WriteMode (Output, StartCol, TheHeap, GlobalVariables, Mode);
                  EStrings.Put_String
                    (File  => Output,
                     E_Str => LexTokenManager.Lex_String_To_String
                       (Lex_Str => LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                                M        => GlobalVariable)));
               end WriteGlobalVariable;

            begin -- WriteGlobalVariableClauses

               -- This subprogram is called for each mode in turn.
               -- Each sequence of global variables could be null, in which case we don't write anything
               -- for that mode.
               GlobalVariable := LexTokenManager.Seq_Algebra.First_Member (The_Heap => TheHeap,
                                                                           S        => GlobalVariables (Mode));

               if not LexTokenManager.Seq_Algebra.Is_Null_Member (M => GlobalVariable) then

                  SPARKProgram.Iteration.Initialise (TheHeap     => TheHeap,
                                                     TheSeq      => GlobalVariables (Mode),
                                                     TheIterator => TheIterator);

                  WriteGlobalVariable (Output              => Output,
                                       StartCol            => StartCol,
                                       FirstGlobalVariable => FirstGlobalVariable,
                                       TheHeap             => TheHeap,
                                       GlobalVariable      => SPARKProgram.Iteration.CurrentMember (TheIterator),
                                       GlobalVariables     => GlobalVariables,
                                       Mode                => Mode);

                  SPARKProgram.Iteration.Next (TheHeap, TheIterator);

                  while not SPARKProgram.Iteration.Complete (TheIterator) loop

                     WriteGlobalVariable (Output              => Output,
                                          StartCol            => StartCol,
                                          FirstGlobalVariable => FirstGlobalVariable,
                                          TheHeap             => TheHeap,
                                          GlobalVariable      => SPARKProgram.Iteration.CurrentMember (TheIterator),
                                          GlobalVariables     => GlobalVariables,
                                          Mode                => Mode);

                     SPARKProgram.Iteration.Next (TheHeap, TheIterator);

                  end loop;

               end if;

            end WriteGlobalVariableClauses;

            procedure Dispose (TheHeap         : in out Heap.HeapRecord;
                               GlobalVariables : in     GlobalVariablesType)
            --# derives TheHeap from *,
            --#                      GlobalVariables;
            is
            begin
               for Mode in Modes loop
                  LexTokenManager.Seq_Algebra.Dispose_Of_Seq (The_Heap => TheHeap,
                                                              S        => GlobalVariables (Mode));
               end loop;
            end Dispose;

         begin -- GlobalsList.Reformat

            RevisedGlobalVariables := GlobalVariables;

            if OK then

               --  CFR1753: Check to see if an export has been excluded.
               AnExportIsExcluded := not EStrings.Is_Empty
                  (E_Str => SparkFormatCommandLineData.Content.ExcludeExport);
               case FunctionOrProcedure is
                  when Subprogram.IsProcedure | Subprogram.IsUnknown =>
                     if AnExportIsExcluded then
                        RemoveExcludedExport (TheHeap         => TheHeap,
                                              Imports         => Imports,
                                              Exports         => Exports,
                                              GlobalVariables => RevisedGlobalVariables);
                     end if;
                  when Subprogram.IsFunction =>
                     null;
               end case;

               if not AnExportIsExcluded or else
                 not LexTokenManager.Seq_Algebra.Is_Empty_Seq (The_Heap => TheHeap,
                                                               S        => RevisedGlobalVariables (OutMode)) or else
                 not LexTokenManager.Seq_Algebra.Is_Empty_Seq (The_Heap => TheHeap,
                                                               S        => RevisedGlobalVariables (InOutMode)) or else
                 not LexTokenManager.Seq_Algebra.Is_Empty_Seq (The_Heap => TheHeap,
                                                               S        => RevisedGlobalVariables (InMode)) or else
                 not LexTokenManager.Seq_Algebra.Is_Empty_Seq (The_Heap => TheHeap,
                                                               S        => RevisedGlobalVariables (Unmoded)) then
                  Annotation.Write (Output, StartCol);
                  WriteGlobal (Output, StartCol);

                  if SparkFormatCommandLineData.Content.AddModes then
                     AddModes (TheHeap             => TheHeap,
                               FunctionOrProcedure => FunctionOrProcedure,
                               Imports             => Imports,
                               Exports             => Exports,
                               GlobalVariables     => RevisedGlobalVariables);
                  end if;

                  FirstGlobalVariable := True;

                  for Mode in Modes loop
                     WriteGlobalVariableClauses (Output              => Output,
                                                 StartCol            => StartCol,
                                                 FirstGlobalVariable => FirstGlobalVariable,
                                                 TheHeap             => TheHeap,
                                                 GlobalVariables     => RevisedGlobalVariables,
                                                 Mode                => Mode);
                  end loop;

                  SPARK_IO.Put_Line (Output, ";", 0);
               end if;
            else
               File_IO.Reset (TemporaryFile);
               SPARKProgram.Copy (TemporaryFile, Output);
            end if;

            Dispose (TheHeap         => TheHeap,
                     GlobalVariables => RevisedGlobalVariables);
            File_IO.Delete (TemporaryFile);

            if not NoEmptyLines then
               SPARK_IO.New_Line (Output, ConsecutiveEmptyLines);
            end if;
            ResetConsecutiveEmptyLines;
         end Reformat;

      end GlobalsList;

      function ContainsFunctionSpecification (InputLine : EStrings.T) return Boolean
      is
         Index  : EStrings.Positions;
      begin

         Index := EStrings.Positions'First;
         WhiteSpace.Skip (InputLine, Index);

         return EStrings.Get_Element (E_Str => InputLine,
                                      Pos   => Index) = 'f' and then
           EStrings.Get_Element (E_Str => InputLine,
                                 Pos   => Index + 1) = 'u' and then
           EStrings.Get_Element (E_Str => InputLine,
                                 Pos   => Index + 2) = 'n' and then
           EStrings.Get_Element (E_Str => InputLine,
                                 Pos   => Index + 3) = 'c' and then
           EStrings.Get_Element (E_Str => InputLine,
                                 Pos   => Index + 4) = 't' and then
           EStrings.Get_Element (E_Str => InputLine,
                                 Pos   => Index + 5) = 'i' and then
           EStrings.Get_Element (E_Str => InputLine,
                                 Pos   => Index + 6) = 'o' and then
           EStrings.Get_Element (E_Str => InputLine,
                                 Pos   => Index + 7) = 'n';

      end ContainsFunctionSpecification;

      function ContainsProcedureSpecification (InputLine : EStrings.T) return Boolean
      is
         Index  : EStrings.Positions;
      begin

         Index := EStrings.Positions'First;
         WhiteSpace.Skip (InputLine, Index);

         return EStrings.Get_Element (E_Str => InputLine,
                                      Pos   => Index) = 'p' and then
           EStrings.Get_Element (E_Str => InputLine,
                                 Pos   => Index + 1) = 'r' and then
           EStrings.Get_Element (E_Str => InputLine,
                                 Pos   => Index + 2) = 'o' and then
           EStrings.Get_Element (E_Str => InputLine,
                                 Pos   => Index + 3) = 'c' and then
           EStrings.Get_Element (E_Str => InputLine,
                                 Pos   => Index + 4) = 'e' and then
           EStrings.Get_Element (E_Str => InputLine,
                                 Pos   => Index + 5) = 'd' and then
           EStrings.Get_Element (E_Str => InputLine,
                                 Pos   => Index + 6) = 'u' and then
           EStrings.Get_Element (E_Str => InputLine,
                                 Pos   => Index + 7) = 'r' and then
           EStrings.Get_Element (E_Str => InputLine,
                                 Pos   => Index + 8) = 'e';

      end ContainsProcedureSpecification;

   begin  -- ReformatAnnotations

      State := ParsingSPARKCode;
      GlobalsListFile := SPARK_IO.Null_File;
      InheritClauseFile := SPARK_IO.Null_File;
      InitializeSpecFile := SPARK_IO.Null_File;
      OwnVarClauseFile := SPARK_IO.Null_File;
      DerivesListFile := SPARK_IO.Null_File;
      FunctionOrProcedure := Subprogram.IsUnknown;

      Heap.Initialize (TheHeap);
      SparkLex.ClearLineContext;
      Annotation.Initialize;
      DerivesList.Initialize;

      Annotations.Initialize (InheritAnno, "inherit", "main_program",
                              SparkFormatCommandLineData.Content.InheritIndent);

      Annotations.Initialize (InitializesAnno, "initializes", "",
                              SparkFormatCommandLineData.Content.InitializationIndent);

      Annotations.Initialize (OwnVarAnno, "own", "initializes",
                              SparkFormatCommandLineData.Content.OwnIndent);


      if not SPARK_IO.End_Of_File (Input) then
         EStrings.Get_Line (File  => Input,
                            E_Str => InputLine);
         File_IO_Required := False;

         while not (File_IO_Required and then SPARK_IO.End_Of_File (Input)) loop

            if File_IO_Required then
               EStrings.Get_Line (File  => Input,
                                  E_Str => InputLine);
            else
               File_IO_Required := True;
            end if;

            case State is
               when ParsingSPARKCode =>
                  if ContainsFunctionSpecification (InputLine) then
                     EStrings.Put_Line (File  => Output,
                                        E_Str => InputLine);
                     FunctionOrProcedure := Subprogram.IsFunction;
                  elsif ContainsProcedureSpecification (InputLine) then
                     EStrings.Put_Line (File  => Output,
                                        E_Str => InputLine);
                     FunctionOrProcedure := Subprogram.IsProcedure;
                  elsif GlobalsList.IsStart (InputLine) then
                     File_IO.Create (GlobalsListFile);
                     EStrings.Put_Line (File  => GlobalsListFile,
                                        E_Str => InputLine);
                     if SPARK_IO.End_Of_File (Input) then
                        -- This route only taken when global alone at end of file
                        GlobalsList.Parse (TemporaryFile   => GlobalsListFile,
                                           StartCol        => StartCol,
                                           TheHeap         => TheHeap,
                                           GlobalVariables => GlobalVariables,
                                           OK              => GlobalsListOK);
                        LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                                S        => Imports);
                        LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                                S        => Exports);
                        GlobalsList.Reformat (OK                  => GlobalsListOK,
                                              TemporaryFile       => GlobalsListFile,
                                              Output              => Output,
                                              StartCol            => StartCol,
                                              TheHeap             => TheHeap,
                                              FunctionOrProcedure => FunctionOrProcedure,
                                              GlobalVariables     => GlobalVariables,
                                              Imports             => Imports,
                                              Exports             => Exports);
                        FunctionOrProcedure := Subprogram.IsUnknown;
                     else
                        -- Normal route
                        State := ParsingGlobalsList;
                     end if;
                  elsif DerivesList.IsStart (InputLine) then
                     File_IO.Create (DerivesListFile);
                     EStrings.Put_Line (File  => DerivesListFile,
                                        E_Str => InputLine);
                     if DerivesList.IsEnd (InputLine) then
                        --# accept Flow_Message, 10, Imports, "There is no globals list, so no need for the list of imports.";
                        DerivesList.Parse (TemporaryFile => DerivesListFile,
                                           StartCol      => StartCol,
                                           TheHeap       => TheHeap,
                                           Imports       => Imports,
                                           Exports       => Exports,
                                           Dependencies  => Dependencies,
                                           OK            => DerivesListOK);
                        --# end accept;
                        DerivesList.Reformat (OK            => DerivesListOK,
                                              TemporaryFile => DerivesListFile,
                                              Output        => Output,
                                              StartCol      => StartCol,
                                              TheHeap       => TheHeap,
                                              Exports       => Exports,
                                              Dependencies  => Dependencies);
                        FunctionOrProcedure := Subprogram.IsUnknown;
                     elsif SPARK_IO.End_Of_File (Input) then
                        File_IO.Reset (DerivesListFile);
                        Copy (DerivesListFile, Output);
                        FunctionOrProcedure := Subprogram.IsUnknown;
                     else
                        State := ParsingDerivesList;
                     end if;
                  elsif Annotations.IsStartOf (InheritAnno, InputLine) then
                     File_IO.Create (InheritClauseFile);
                     State := ParsingInheritClause;
                     File_IO_Required := False;
                  elsif Annotations.IsStartOf (InitializesAnno, InputLine) then
                     File_IO.Create (InitializeSpecFile);
                     State := ParsingInitializationSpecification;
                     File_IO_Required := False;
                  elsif Annotations.IsStartOf (OwnVarAnno, InputLine) then
                     File_IO.Create (OwnVarClauseFile);
                     State := ParsingOwnVariableClause;
                     File_IO_Required := False;
                  else
                     EStrings.Put_Line (File  => Output,
                                        E_Str => InputLine);
                  end if;
               when ParsingGlobalsList =>
                  if GlobalsList.IsEnd (InputLine) then
                     if DerivesList.IsStart (InputLine) then
                        if GlobalsList.NoEmptyLines then
                           File_IO.Create (DerivesListFile);
                           EStrings.Put_Line (File  => DerivesListFile,
                                              E_Str => InputLine);
                           if DerivesList.IsEnd (InputLine) then
                              --# accept Flow_Message, 10, SparkLex.CurrLine, "Not used when both global and derives lists present" &
                              --#        Flow_Message, 10, StartCol, "If both globals and derives lists present, StartCol value from derives list is used";
                              GlobalsList.Parse (TemporaryFile   => GlobalsListFile,
                                                 StartCol        => StartCol,
                                                 TheHeap         => TheHeap,
                                                 GlobalVariables => GlobalVariables,
                                                 OK              => GlobalsListOK);
                              --# end accept;
                              DerivesList.Parse (TemporaryFile => DerivesListFile,
                                                 StartCol      => StartCol,
                                                 TheHeap       => TheHeap,
                                                 Imports       => Imports,
                                                 Exports       => Exports,
                                                 Dependencies  => Dependencies,
                                                 OK            => DerivesListOK);
                              GlobalsList.Reformat (OK                   => GlobalsListOK,
                                                    TemporaryFile        => GlobalsListFile,
                                                    Output               => Output,
                                                    StartCol             => StartCol,
                                                    TheHeap              => TheHeap,
                                                    FunctionOrProcedure  => FunctionOrProcedure,
                                                    GlobalVariables      => GlobalVariables,
                                                    Imports              => Imports,
                                                    Exports              => Exports);
                              DerivesList.Reformat (OK            => DerivesListOK,
                                                    TemporaryFile => DerivesListFile,
                                                    Output        => Output,
                                                    StartCol      => StartCol,
                                                    TheHeap       => TheHeap,
                                                    Exports       => Exports,
                                                    Dependencies  => Dependencies);
                              FunctionOrProcedure := Subprogram.IsUnknown;
                              State := ParsingSPARKCode;
                           else
                              State := ParsingDerivesList;
                           end if;
                        else
                           GlobalsList.Parse (TemporaryFile   => GlobalsListFile,
                                              StartCol        => StartCol,
                                              TheHeap         => TheHeap,
                                              GlobalVariables => GlobalVariables,
                                              OK              => GlobalsListOK);
                           LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                                   S        => Imports);
                           LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                                   S        => Exports);
                           GlobalsList.Reformat (OK                   => GlobalsListOK,
                                                 TemporaryFile        => GlobalsListFile,
                                                 Output               => Output,
                                                 StartCol             => StartCol,
                                                 TheHeap              => TheHeap,
                                                 FunctionOrProcedure  => FunctionOrProcedure,
                                                 GlobalVariables      => GlobalVariables,
                                                 Imports              => Imports,
                                                 Exports              => Exports);
                           State := ParsingSPARKCode;
                           File_IO_Required := False;
                        end if;
                     else
                        GlobalsList.Parse (TemporaryFile   => GlobalsListFile,
                                           StartCol        => StartCol,
                                           TheHeap         => TheHeap,
                                           GlobalVariables => GlobalVariables,
                                           OK              => GlobalsListOK);
                        LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                                S        => Imports);
                        LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                                S        => Exports);
                        GlobalsList.Reformat (OK                   => GlobalsListOK,
                                              TemporaryFile        => GlobalsListFile,
                                              Output               => Output,
                                              StartCol             => StartCol,
                                              TheHeap              => TheHeap,
                                              FunctionOrProcedure  => FunctionOrProcedure,
                                              GlobalVariables      => GlobalVariables,
                                              Imports              => Imports,
                                              Exports              => Exports);
                        FunctionOrProcedure := Subprogram.IsUnknown;
                        State := ParsingSPARKCode;
                        File_IO_Required := False; -- Have another look at line in case is a function
                     end if;
                  elsif SPARK_IO.End_Of_File (Input) then
                     EStrings.Put_Line (File  => GlobalsListFile,
                                        E_Str => InputLine);
                     GlobalsList.Parse (TemporaryFile   => GlobalsListFile,
                                        StartCol        => StartCol,
                                        TheHeap         => TheHeap,
                                        GlobalVariables => GlobalVariables,
                                        OK              => GlobalsListOK);
                     LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                             S        => Imports);
                     LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                             S        => Exports);
                     GlobalsList.Reformat (OK                   => GlobalsListOK,
                                           TemporaryFile        => GlobalsListFile,
                                           Output               => Output,
                                           StartCol             => StartCol,
                                           TheHeap              => TheHeap,
                                           FunctionOrProcedure  => FunctionOrProcedure,
                                           GlobalVariables      => GlobalVariables,
                                           Imports              => Imports,
                                           Exports              => Exports);
                     FunctionOrProcedure := Subprogram.IsUnknown;
                     State := ParsingSPARKCode;
                  else
                     EStrings.Put_Line (File  => GlobalsListFile,
                                        E_Str => InputLine);
                     if EStrings.Is_Empty (E_Str => EStrings.Trim (InputLine)) then
                        GlobalsList.IncrementConsecutiveEmptyLines;
                     else
                        GlobalsList.ResetConsecutiveEmptyLines;
                     end if;
                  end if;
               when ParsingDerivesList =>
                  EStrings.Put_Line (File  => DerivesListFile,
                                     E_Str => InputLine);
                  if DerivesList.IsEnd (InputLine) then
                     DerivesList.Parse (TemporaryFile => DerivesListFile,
                                        StartCol      => StartCol,
                                        TheHeap       => TheHeap,
                                        Imports       => Imports,
                                        Exports       => Exports,
                                        Dependencies  => Dependencies,
                                        OK            => DerivesListOK);
                     if SPARK_IO.Is_Open (GlobalsListFile) then
                        GlobalsList.Parse (TemporaryFile   => GlobalsListFile,
                                           StartCol        => StartCol,
                                           TheHeap         => TheHeap,
                                           GlobalVariables => GlobalVariables,
                                           OK              => GlobalsListOK);
                        GlobalsList.Reformat (OK                   => GlobalsListOK,
                                              TemporaryFile        => GlobalsListFile,
                                              Output               => Output,
                                              StartCol             => StartCol,
                                              TheHeap              => TheHeap,
                                              FunctionOrProcedure  => FunctionOrProcedure,
                                              GlobalVariables      => GlobalVariables,
                                              Imports              => Imports,
                                              Exports              => Exports);
                     end if;
                     DerivesList.Reformat (OK            => DerivesListOK,
                                           TemporaryFile => DerivesListFile,
                                           Output        => Output,
                                           StartCol      => StartCol,
                                           TheHeap       => TheHeap,
                                           Exports       => Exports,
                                           Dependencies  => Dependencies);
                     FunctionOrProcedure := Subprogram.IsUnknown;
                     State := ParsingSPARKCode;
                  elsif SPARK_IO.End_Of_File (Input) then
                     if SPARK_IO.Is_Open (GlobalsListFile) then
                        GlobalsList.Parse (TemporaryFile   => GlobalsListFile,
                                           StartCol        => StartCol,
                                           TheHeap         => TheHeap,
                                           GlobalVariables => GlobalVariables,
                                           OK              => GlobalsListOK);
                        LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                                S        => Imports);
                        LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                                S        => Exports);
                        GlobalsList.Reformat (OK                   => GlobalsListOK,
                                              TemporaryFile        => GlobalsListFile,
                                              Output               => Output,
                                              StartCol             => StartCol,
                                              TheHeap              => TheHeap,
                                              FunctionOrProcedure  => FunctionOrProcedure,
                                              GlobalVariables      => GlobalVariables,
                                              Imports              => Imports,
                                              Exports              => Exports);
                     end if;
                     File_IO.Reset (DerivesListFile);
                     Copy (DerivesListFile, Output);
                     FunctionOrProcedure := Subprogram.IsUnknown;
                     State := ParsingSPARKCode;
                  end if;
               when ParsingInheritClause =>
                  EStrings.Put_Line (File  => InheritClauseFile,
                                     E_Str => InputLine);

                  if Annotations.IsEndOf (InheritAnno, InputLine) or else
                     SPARK_IO.End_Of_File (Input)
                  then
                     Reformatter.Initialise (Anno        => InheritAnno,
                                             DottedNames => Reformatter.Allow_Dotted_Names,
                                             TheHeap     => TheHeap,
                                             This        => InheritClause);
                     Reformatter.Parse (InheritClause, TheHeap, InheritClauseFile);
                     Reformatter.Reformat (InheritClause,
                                           TheHeap,
                                           InheritClauseFile,
                                           Output,
                                           ReformatSuccessful);
                     --# accept Flow_Message, 10, InheritClause, "The state variable has been 'disposed' and should not be reused without initialisation";
                     Reformatter.Finalise (InheritClause, TheHeap);
                     --# end accept;
                     File_IO.Delete (InheritClauseFile);
                     State := ParsingSPARKCode;
                     File_IO_Required := not ReformatSuccessful;
                  end if;
               when ParsingInitializationSpecification =>
                  EStrings.Put_Line (File  => InitializeSpecFile,
                                     E_Str => InputLine);

                  if Annotations.IsEndOf (InitializesAnno, InputLine) or else
                     SPARK_IO.End_Of_File (Input)
                  then
                     Reformatter.Initialise (Anno        => InitializesAnno,
                                             DottedNames => Reformatter.Disallow_Dotted_Names,
                                             TheHeap     => TheHeap,
                                             This        => InitializationSpec);
                     Reformatter.Parse (InitializationSpec, TheHeap, InitializeSpecFile);
                     Reformatter.Reformat (InitializationSpec,
                                           TheHeap,
                                           InitializeSpecFile,
                                           Output,
                                           ReformatSuccessful);
                     --# accept Flow_Message, 10, InitializationSpec, "The state variable has been 'disposed' and should not be reused without initialisation";
                     Reformatter.Finalise (InitializationSpec, TheHeap);
                     --# end accept;
                     File_IO.Delete (InitializeSpecFile);
                     State := ParsingSPARKCode;
                     File_IO_Required := not ReformatSuccessful;
                  end if;
               when ParsingOwnVariableClause =>
                  EStrings.Put_Line (File  => OwnVarClauseFile,
                                     E_Str => InputLine);

                  if Annotations.IsEndOf (OwnVarAnno, InputLine) or else
                     SPARK_IO.End_Of_File (Input)
                  then
                     Reformatter.Initialise (Anno        => OwnVarAnno,
                                             DottedNames => Reformatter.Disallow_Dotted_Names,
                                             TheHeap     => TheHeap,
                                             This        => OwnVariableClause);
                     Reformatter.Parse (OwnVariableClause, TheHeap, OwnVarClauseFile);
                     Reformatter.Reformat (OwnVariableClause,
                                           TheHeap,
                                           OwnVarClauseFile,
                                           Output,
                                           ReformatSuccessful);
                     --# accept Flow_Message, 10, OwnVariableClause, "The state variable has been 'disposed' and should not be reused without initialisation";
                     Reformatter.Finalise (OwnVariableClause, TheHeap);
                     --# end accept;
                     File_IO.Delete (OwnVarClauseFile);
                     State := ParsingSPARKCode;
                     File_IO_Required := not ReformatSuccessful;
                  end if;

               when  ParsingRefinementDefinition =>
                  -- To be completed
                  State := ParsingSPARKCode;

            end case;

         end loop;
      end if;

   end ReformatAnnotations;

end SPARKProgram;
