-- $Id: sparkprogram-reformatter.adb 15795 2010-01-26 14:30:59Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with ELStrings;
with CommandLineData;
with SparkFormatCommandLineData;

separate (SPARKProgram)

package body Reformatter is
   UnmodedStr          : constant String := "";
   InModeStr           : constant String := "in";
   OutModeStr          : constant String := "out";
   InOutModeStr        : constant String := "in out";
   ProtectedUnmodedStr : constant String := "protected";
   ProtectedInStr      : constant String := "protected in";
   ProtectedOutStr     : constant String := "protected out";
   TaskModifierStr     : constant String := "task";

   type Format_Info is record
      StartCol            : EStrings.Positions;
      ModifierCol         : EStrings.Positions;
      PrimaryIdCol        : EStrings.Positions;
      TypeMarkCol         : EStrings.Positions;
      PropertiesCol       : EStrings.Positions;
      ModifiersPresent    : Modifier_Use;
      TypeMarkPresent     : Boolean;
      PropertyListPresent : Boolean;
   end record;

   -- A simple lexer is required so that the capitalisation of
   -- names - particularly predefined type_marks such as Natural
   -- are not converted to "NATURAL".  Also simplifies parsing of
   -- property lists.
   -- The lexer assumes that the input file contains only annotations
   -- and thus the annotation start and continuation symbols can be treated
   -- as whitespace.

   --# inherit Ada.Characters.Handling,
   --#         Ada.Characters.Latin_1,
   --#         Annotations,
   --#         CommandLineData,
   --#         ELStrings,
   --#         EStrings,
   --#         SPARK_IO,
   --#         WhiteSpace;
   package SimpleLex is
      type State is private;

      type Token_Type is (
         RWderives,
         RWglobal,
         RWinherit,
         RWinitializes,
         RWmain_program,
         RWown,
         RWfrom,
         RWin,
         RWis,
         RWout,
         RWprotected,
         RWtask,
         colon,
         comma,
         point,
         semicolon,
         other_punct, -- for simple annotations we do not need to know any others
         annotation_start,
         annotation_end,
         identifier,
         property_list);

      subtype annotation_signifiers is Token_Type range RWderives .. RWown;

      type Token_Record is record
         Token      : Token_Type;
         TokenValue : ELStrings.T;
      end record;

      procedure Initialise (InputFile : in      SPARK_IO.File_Type;
                            Anno      : in      Annotations.Anno_Type;
                            LexState  :     out State);
      --# derives LexState from Anno,
      --#                       InputFile;

      procedure Next (This      : in out State;
                      TokenRec  :    out Token_Record);
      --# global in     CommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys,
      --#         This,
      --#         TokenRec          from CommandLineData.Content,
      --#                                SPARK_IO.File_Sys,
      --#                                This;

      function GetColNo (This : State) return EStrings.Positions;

   private
      type State is record
         File         : SPARK_IO.File_Type;
         Anno         : Annotations.Anno_Type;
         Line         : EStrings.T;
         Index        : ELStrings.Positions;
         InAnnotation : Boolean;
      end record;
   end SimpleLex;

   function "=" (Left, Right : SimpleLex.Token_Type)
      return Boolean renames SimpleLex."=";


   package body SimpleLex is separate;

   procedure Initialise (Anno        : in     Annotations.Anno_Type;
                         DottedNames : in     Boolean;
                         TheHeap     : in out Heap.HeapRecord;
                         This        :    out State)
   is
      LocModifierRel  : LexTokenManager.Relation_Algebra.Relation;
      LocTypeRel      : LexTokenManager.Relation_Algebra.String.Relation;
      LocPropertyRel  : LexTokenManager.Relation_Algebra.String.Relation;
   begin
      LexTokenManager.Relation_Algebra.Create_Relation (The_Heap => TheHeap,
                                                        R        => LocModifierRel);
      LexTokenManager.Relation_Algebra.String.Create_Relation (The_Heap => TheHeap,
                                                               R        => LocTypeRel);
      LexTokenManager.Relation_Algebra.String.Create_Relation (The_Heap => TheHeap,
                                                               R        => LocPropertyRel);

      This := State'(Anno             => Anno,
                     Relations        =>
                       Relation_Type'
                       (ModifierRel => LocModifierRel,
                        TypeRel     => LocTypeRel,
                        PropertyRel => LocPropertyRel),
                     ParseStats       =>
                       Statistics_Type'
                       (StartCol            => EStrings.Positions'First,
                        MaxModifierLength   => 0,
                        MaxPrimaryIdLength  => 0,
                        MaxTypeMarkLength   => 0,
                        ModifiersPresent    => Modifier_Use'(others => False),
                        TypeMarkPresent     => False,
                        PropertyListPresent => False),
                     AllowDottedNames => DottedNames,
                     Success          => True);
   end Initialise;

   function ModifierLength (Modifier : VariableModifier) return Natural
   is
      Result : Natural;
   begin
      case Modifier is
         when Unmoded          => Result := UnmodedStr'Length;
         when InMode           => Result := InModeStr'Length;
         when OutMode          => Result := OutModeStr'Length;
         when InOutMode        => Result := InOutModeStr'Length;
         when ProtectedUnmoded => Result := ProtectedUnmodedStr'Length;
         when ProtectedIn      => Result := ProtectedInStr'Length;
         when ProtectedOut     => Result := ProtectedOutStr'Length;
         when TaskModifier     => Result := TaskModifierStr'Length;
      end case;
      return Result;
   end ModifierLength;

   function ModifierString (Modifier : VariableModifier)
      return EStrings.T
   is
      Result : EStrings.T;
   begin
      case Modifier is
         when Unmoded          => Result := EStrings.Copy_String (Str => UnmodedStr);
         when InMode           => Result := EStrings.Copy_String (Str => InModeStr);
         when OutMode          => Result := EStrings.Copy_String (Str => OutModeStr);
         when InOutMode        => Result := EStrings.Copy_String (Str => InOutModeStr);
         when ProtectedUnmoded => Result := EStrings.Copy_String (Str => ProtectedUnmodedStr);
         when ProtectedIn      => Result := EStrings.Copy_String (Str => ProtectedInStr);
         when ProtectedOut     => Result := EStrings.Copy_String (Str => ProtectedOutStr);
         when TaskModifier     => Result := EStrings.Copy_String (Str => TaskModifierStr);
      end case;
      return Result;
   end ModifierString;

   procedure ParseModifiers (Lex              : in out SimpleLex.State;
                             Token            : in out SimpleLex.Token_Record;
                             ModifierType     :    out VariableModifier)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives Lex,
   --#         ModifierType,
   --#         SPARK_IO.File_Sys,
   --#         Token             from CommandLineData.Content,
   --#                                Lex,
   --#                                SPARK_IO.File_Sys,
   --#                                Token;
   is
   begin
      case Token.Token is
         when SimpleLex.RWin =>
            ModifierType := InMode;
            SimpleLex.Next (Lex, Token);
         when SimpleLex.RWout =>
            ModifierType := OutMode;
            SimpleLex.Next (Lex, Token);
         when SimpleLex.RWprotected =>
            SimpleLex.Next (Lex, Token);
            case Token.Token is
               when SimpleLex.RWin =>
                  ModifierType := ProtectedIn;
                  SimpleLex.Next (Lex, Token);
               when SimpleLex.RWout =>
                  ModifierType := ProtectedOut;
                  SimpleLex.Next (Lex, Token);
               when others =>
                  ModifierType := ProtectedUnmoded;
                  -- The current token is still required in this case
            end case;
         when SimpleLex.RWtask =>
            ModifierType := TaskModifier;
            SimpleLex.Next (Lex, Token);
         when others =>
            ModifierType := Unmoded;
            -- The current token is still required in this case
      end case;
   end ParseModifiers;

   procedure ParseName (Lex                : in out SimpleLex.State;
                        AllowDottedNames   : in     Boolean;
                        Token              : in out SimpleLex.Token_Record;
                        Name               :    out ELStrings.T;
                        OK                 :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives Lex,
   --#         OK,
   --#         SPARK_IO.File_Sys,
   --#         Token             from AllowDottedNames,
   --#                                CommandLineData.Content,
   --#                                Lex,
   --#                                SPARK_IO.File_Sys &
   --#         Name              from AllowDottedNames,
   --#                                CommandLineData.Content,
   --#                                Lex,
   --#                                SPARK_IO.File_Sys,
   --#                                Token;
   is
      LocName    : ELStrings.T;
      LocOK      : Boolean;
   begin
      LocOK := True;
      LocName := Token.TokenValue;
      SimpleLex.Next (Lex, Token);
      if AllowDottedNames then
         loop
            exit when (not LocOK) or Token.Token /= SimpleLex.point;
            ELStrings.Append_String (E_Str => LocName,
                                     Str   => ".");
            SimpleLex.Next (Lex, Token);
            if Token.Token = SimpleLex.identifier then
               ELStrings.Append_Examiner_Long_String (E_Str1 => LocName,
                                                      E_Str2 => Token.TokenValue);
               SimpleLex.Next (Lex, Token);
            else
               LocOK := False;
            end if;
         end loop;
      end if;

      Name := LocName;
      OK := LocOK;
   end ParseName;

   procedure ParseNameList (Lex                : in out SimpleLex.State;
                            AllowDottedNames   : in     Boolean;
                            Token              : in out SimpleLex.Token_Record;
                            TheHeap            : in out Heap.HeapRecord;
                            TheSeq             : in     LexTokenManager.Seq_Algebra.Seq;
                            MaxNameLength      : in out EStrings.Lengths;
                            OK                 :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --# derives Lex,
   --#         LexTokenManager.State,
   --#         MaxNameLength,
   --#         SPARK_IO.File_Sys,
   --#         Token                 from *,
   --#                                    AllowDottedNames,
   --#                                    CommandLineData.Content,
   --#                                    Lex,
   --#                                    SPARK_IO.File_Sys,
   --#                                    Token &
   --#         OK                    from AllowDottedNames,
   --#                                    CommandLineData.Content,
   --#                                    Lex,
   --#                                    SPARK_IO.File_Sys,
   --#                                    Token &
   --#         Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    AllowDottedNames,
   --#                                    CommandLineData.Content,
   --#                                    Lex,
   --#                                    LexTokenManager.State,
   --#                                    SPARK_IO.File_Sys,
   --#                                    TheHeap,
   --#                                    TheSeq,
   --#                                    Token;
   is
      LocOK      : Boolean;
      Name       : ELStrings.T;
      LexName    : LexTokenManager.Lex_String;
   begin
      LocOK := Token.Token = SimpleLex.identifier;

      while LocOK and Token.Token = SimpleLex.identifier loop
         ParseName (Lex, AllowDottedNames, Token, Name, LocOK);
         if LocOK then
            LexTokenManager.Insert_Examiner_Long_String (Str     => Name,
                                                         Lex_Str => LexName);
            LexTokenManager.Seq_Algebra.Add_Member (The_Heap    => TheHeap,
                                                    S           => TheSeq,
                                                    Given_Value => LexName);
            if ELStrings.Get_Length (E_Str => Name) > MaxNameLength then
               MaxNameLength := ELStrings.Get_Length (E_Str => Name);
            end if;

            case Token.Token is
               when SimpleLex.comma =>
                  SimpleLex.Next (Lex, Token);
               when SimpleLex.identifier =>
                  -- Two successive identifiers are not syntactically correct
                  LocOK := False;
               when others =>
                  -- loop will terminate at current token
                  null;
            end case;
         end if;
      end loop;

      OK := LocOK;
   end ParseNameList;

   procedure ParseTypeMark (Lex         : in out SimpleLex.State;
                            Token       : in out SimpleLex.Token_Record;
                            TypeMark    :    out ELStrings.T;
                            OK          : in out Boolean)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives Lex,
   --#         OK,
   --#         SPARK_IO.File_Sys,
   --#         Token,
   --#         TypeMark          from CommandLineData.Content,
   --#                                Lex,
   --#                                OK,
   --#                                SPARK_IO.File_Sys,
   --#                                Token;
   is
      TypeMarkStr : ELStrings.T;
   begin
      if OK and Token.Token = SimpleLex.colon then
         SimpleLex.Next (Lex, Token);

         if Token.Token = SimpleLex.identifier then
            ParseName (Lex, Allow_Dotted_Names, Token, TypeMarkStr, OK);
         else
            OK := False;
            TypeMarkStr := ELStrings.Empty_String;
         end if;
      else
         TypeMarkStr := ELStrings.Empty_String;
      end if;

      TypeMark := TypeMarkStr;
   end ParseTypeMark;

   procedure ParseProperties (Lex              : in out SimpleLex.State;
                              Token            : in out SimpleLex.Token_Record;
                              PropertyList     :    out ELStrings.T;
                              OK               : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives Lex,
   --#         SPARK_IO.File_Sys,
   --#         Token             from CommandLineData.Content,
   --#                                Lex,
   --#                                OK,
   --#                                SPARK_IO.File_Sys,
   --#                                Token &
   --#         PropertyList      from OK,
   --#                                Token;
   is
   begin
      if OK and Token.Token = SimpleLex.property_list then
         PropertyList := Token.TokenValue;
         SimpleLex.Next (Lex, Token);
      else
         PropertyList := ELStrings.Empty_String;
      end if;
   end ParseProperties;

   procedure ParseTheAnnotation (Lex              : in     SimpleLex.State;
                                 Token            : in     SimpleLex.Token_Record;
                                 AllowDottedNames : in     Boolean;
                                 TheHeap          : in out Heap.HeapRecord;
                                 Relations        : in     Relation_Type;
                                 ParseStats       : in out Statistics_Type;
                                 OK               :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --# derives LexTokenManager.State,
   --#         SPARK_IO.File_Sys     from *,
   --#                                    AllowDottedNames,
   --#                                    CommandLineData.Content,
   --#                                    Lex,
   --#                                    SPARK_IO.File_Sys,
   --#                                    Token &
   --#         OK                    from AllowDottedNames,
   --#                                    CommandLineData.Content,
   --#                                    Lex,
   --#                                    SPARK_IO.File_Sys,
   --#                                    Token &
   --#         ParseStats,
   --#         Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    AllowDottedNames,
   --#                                    CommandLineData.Content,
   --#                                    Lex,
   --#                                    LexTokenManager.State,
   --#                                    Relations,
   --#                                    SPARK_IO.File_Sys,
   --#                                    TheHeap,
   --#                                    Token;
   is
      LocOK               : Boolean;
      LocLex              : SimpleLex.State;
      NextToken           : SimpleLex.Token_Record;
      NameList            : LexTokenManager.Seq_Algebra.Seq;
      FirstName           : LexTokenManager.Seq_Algebra.Member_Of_Seq;
      TypeMark            : ELStrings.T;
      PropertyList        : ELStrings.T;
      FirstNameLex        : LexTokenManager.Lex_String;
      TypeMarkLex         : LexTokenManager.Lex_String;
      PropertyListLex     : LexTokenManager.Lex_String;
      Modifier            : VariableModifier;
   begin -- ParseTheAnnotation
      LocOK := True;
      NextToken := Token;
      LocLex := Lex;

      while LocOK and then
         not (NextToken.Token in SimpleLex.annotation_signifiers or else
              NextToken.Token = SimpleLex.annotation_end)
      loop

         LexTokenManager.Seq_Algebra.Create_Seq (The_Heap => TheHeap,
                                                 S        => NameList);

         ParseModifiers (LocLex, NextToken, Modifier);
         ParseNameList  (Lex                => LocLex,
                         AllowDottedNames   => AllowDottedNames,
                         Token              => NextToken,
                         TheHeap            => TheHeap,
                         TheSeq             => NameList,
                         MaxNameLength      => ParseStats.MaxPrimaryIdLength,
                         OK                 => LocOK);
         ParseTypeMark   (LocLex, NextToken, TypeMark, LocOK);
         ParseProperties (LocLex, NextToken, PropertyList, LocOK);

         LocOK := LocOK and NextToken.Token = SimpleLex.semicolon;
         if LocOK then

            if not ELStrings.Is_Empty (E_Str => TypeMark) then
               -- A type_mark applies to all names in the list
               LexTokenManager.Insert_Examiner_Long_String (Str     => TypeMark,
                                                            Lex_Str => TypeMarkLex);
               LexTokenManager.Relation_Algebra.String.Add_Col (The_Heap => TheHeap,
                                                                R        => Relations.TypeRel,
                                                                J        => TypeMarkLex,
                                                                S        => NameList);
               ParseStats.TypeMarkPresent := True;
               if ELStrings.Get_Length (E_Str => TypeMark) > ParseStats.MaxTypeMarkLength then
                  ParseStats.MaxTypeMarkLength := ELStrings.Get_Length (E_Str => TypeMark);
               end if;
            end if;

            if not ELStrings.Is_Empty (E_Str => PropertyList) then
               -- A property_list applies to all names in the list
               LexTokenManager.Insert_Examiner_Long_String (Str     => PropertyList,
                                                            Lex_Str => PropertyListLex);
               LexTokenManager.Relation_Algebra.String.Add_Col (The_Heap => TheHeap,
                                                                R        => Relations.PropertyRel,
                                                                J        => PropertyListLex,
                                                                S        => NameList);
               ParseStats.PropertyListPresent := True;
            end if;

            ParseStats.ModifiersPresent (Modifier) := True;
            if ModifierLength (Modifier) > ParseStats.MaxModifierLength then
               ParseStats.MaxModifierLength := ModifierLength (Modifier);
            end if;
            if Modifier /= Unmoded then
               -- A modifier only applies to the first name in the list
               FirstName := LexTokenManager.Seq_Algebra.First_Member (The_Heap => TheHeap,
                                                                      S        => NameList);
               FirstNameLex := LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                            M        => FirstName);
               LexTokenManager.Seq_Algebra.Remove_Member (The_Heap    => TheHeap,
                                                          S           => NameList,
                                                          Given_Value => FirstNameLex);
               LexTokenManager.Relation_Algebra.Insert_Pair (The_Heap => TheHeap,
                                                             R        => Relations.ModifierRel,
                                                             I        => VariableModifier'Pos (Modifier),
                                                             J        => FirstNameLex);
            end if;
            -- The remaining names in the list are unmoded
            if not LexTokenManager.Seq_Algebra.Is_Empty_Seq (The_Heap => TheHeap,
                                                             S        => NameList) then
               LexTokenManager.Relation_Algebra.Add_Row (The_Heap => TheHeap,
                                                         R        => Relations.ModifierRel,
                                                         I        => VariableModifier'Pos (Unmoded),
                                                         S        => NameList);
               ParseStats.ModifiersPresent (Unmoded) := True;
            end if;

            SimpleLex.Next (LocLex, NextToken);
         end if;

         LexTokenManager.Seq_Algebra.Dispose_Of_Seq (The_Heap => TheHeap,
                                                     S        => NameList);

      end loop;

      OK := LocOK;

   end ParseTheAnnotation;

   procedure Parse (This          : in out State;
                    TheHeap       : in out Heap.HeapRecord;
                    TemporaryFile : in out SPARK_IO.File_Type)
   is
      LocOK       : Boolean;
      LocStartCol : EStrings.Positions;
      Token       : SimpleLex.Token_Record;
      Lex         : SimpleLex.State;
      ParseStats  : Statistics_Type;
   begin
      ParseStats := This.ParseStats;
      File_IO.Reset (TemporaryFile);
      SimpleLex.Initialise (TemporaryFile, This.Anno, Lex);
      SimpleLex.Next (Lex, Token);
      if Token.Token = SimpleLex.annotation_start then
         -- The start of an annotation has been located
         -- Take account of the length of the annotation_start symbol
         LocStartCol := SimpleLex.GetColNo (Lex);
         if LocStartCol >= 3 then
            LocStartCol := LocStartCol - 3;
         else
            LocStartCol := 1;
         end if;

         SimpleLex.Next (Lex, Token);

         if ELStrings.Eq_String (E_Str1 => Token.TokenValue,
                                 E_Str2 => ELStrings.ToExaminerLongString (Annotations.Intro (This.Anno)))
         then
            SimpleLex.Next (Lex, Token);
            ParseTheAnnotation (Lex,
                                Token,
                                This.AllowDottedNames,
                                TheHeap,
                                This.Relations,
                                ParseStats,
                                LocOK);
         else
            LocOK := False;
         end if;
      else
         LocOK := False;
         LocStartCol := 1;  -- Assume a StartCol of 0 if not start of anno
      end if;

      This.Success := LocOK;
      ParseStats.StartCol := LocStartCol;
      This.ParseStats := ParseStats;

   end Parse;

   procedure GetRelated (TheHeap     : in out Heap.HeapRecord;
                         Relation    : in     LexTokenManager.Relation_Algebra.String.Relation;
                         VarName     : in     LexTokenManager.Seq_Algebra.Member_Of_Seq;
                         RelatedStr  :    out ELStrings.T)
   --# global in     LexTokenManager.State;
   --#        in out Statistics.TableUsage;
   --# derives RelatedStr            from LexTokenManager.State,
   --#                                    Relation,
   --#                                    TheHeap,
   --#                                    VarName &
   --#         Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    LexTokenManager.State,
   --#                                    Relation,
   --#                                    TheHeap,
   --#                                    VarName;
   is
      Seq         : LexTokenManager.Seq_Algebra.Seq;
      FirstMember : LexTokenManager.Seq_Algebra.Member_Of_Seq;
      Result      : ELStrings.T;
   begin
      LexTokenManager.Relation_Algebra.String.Row_Extraction
        (The_Heap    => TheHeap,
         R           => Relation,
         Given_Index => LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                     M        => VarName),
         S           => Seq);

      -- There should be only zero or one related string with a variable
      FirstMember := LexTokenManager.Seq_Algebra.First_Member (The_Heap => TheHeap,
                                                               S        => Seq);
      if LexTokenManager.Seq_Algebra.Is_Null_Member (M => FirstMember) then
         Result := ELStrings.Empty_String;
      else
         Result := LexTokenManager.Lex_String_To_Long_String
           (Lex_Str => LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                    M        => FirstMember));
      end if;

      LexTokenManager.Seq_Algebra.Dispose_Of_Seq (The_Heap => TheHeap,
                                                  S        => Seq);

      RelatedStr :=  Result;
   end GetRelated;

   procedure WritePropertyList (Anno       : in     Annotations.Anno_Type;
                                Output     : in     SPARK_IO.File_Type;
                                Formatting : in     Format_Info;
                                Properties : in     ELStrings.T)
   --# global in     CommandLineData.Content;
   --#        in     SparkFormatCommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Anno,
   --#                                CommandLineData.Content,
   --#                                Formatting,
   --#                                Output,
   --#                                Properties,
   --#                                SparkFormatCommandLineData.Content;
   is
      I : ELStrings.Positions;
   begin
      I := 1;
      if SparkFormatCommandLineData.Content.PropertiesIndent /=
         SparkFormatCommandLineData.Inline then

         SPARK_IO.New_Line (Output, 1);
         Annotations.Write (Anno, Output, Formatting.StartCol);
      end if;
      SPARK_IO.Set_Col (Output, Formatting.PropertiesCol);
      while I <= ELStrings.Get_Length (E_Str => Properties) loop
         if I + 2 <= ELStrings.Get_Length (E_Str => Properties) and then
           ELStrings.Get_Element (E_Str => Properties,
                                  Pos   => I)     = '-' and then
           ELStrings.Get_Element (E_Str => Properties,
                                  Pos   => I + 1) = '-' and then
           ELStrings.Get_Element (E_Str => Properties,
                                  Pos   => I + 2) = CommandLineData.Content.AnnoChar then
            SPARK_IO.New_Line (Output, 1);
            Annotations.Write (Anno, Output, Formatting.StartCol);
            SPARK_IO.Set_Col (Output, Formatting.PropertiesCol);
            I := I + 3;
         else
            SPARK_IO.Put_Char (Output,
                               ELStrings.Get_Element (E_Str => Properties,
                                                      Pos   => I));
            I := I + 1;
         end if;
      end loop;
   end WritePropertyList;

   procedure WriteName (Anno        : in     Annotations.Anno_Type;
                        Output      : in     SPARK_IO.File_Type;
                        Formatting  : in     Format_Info;
                        FirstName   : in out Boolean;
                        MayUseComma : in out Boolean;
                        TheHeap     : in out Heap.HeapRecord;
                        Name        : in     LexTokenManager.Seq_Algebra.Member_Of_Seq;
                        Modifier    : in     VariableModifier;
                        Relations   : in     Relation_Type)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in     SparkFormatCommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --# derives FirstName             from * &
   --#         MayUseComma           from Formatting,
   --#                                    LexTokenManager.State,
   --#                                    Modifier,
   --#                                    Name,
   --#                                    Relations,
   --#                                    TheHeap &
   --#         SPARK_IO.File_Sys     from *,
   --#                                    Anno,
   --#                                    CommandLineData.Content,
   --#                                    FirstName,
   --#                                    Formatting,
   --#                                    LexTokenManager.State,
   --#                                    MayUseComma,
   --#                                    Modifier,
   --#                                    Name,
   --#                                    Output,
   --#                                    Relations,
   --#                                    SparkFormatCommandLineData.Content,
   --#                                    TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    Formatting,
   --#                                    LexTokenManager.State,
   --#                                    Name,
   --#                                    Relations,
   --#                                    TheHeap;
   is
      TypeMark       : ELStrings.T;
      PropertyList   : ELStrings.T;
      SimpleNameList : Boolean;
   begin
      if Formatting.TypeMarkPresent then
         GetRelated (TheHeap     => TheHeap,
                     Relation    => Relations.TypeRel,
                     VarName     => Name,
                     RelatedStr  => TypeMark);
      else
         TypeMark := ELStrings.Empty_String;
      end if;

      if Formatting.PropertyListPresent then
         GetRelated (TheHeap     => TheHeap,
                     Relation    => Relations.PropertyRel,
                     VarName     => Name,
                     RelatedStr  => PropertyList);
      else
         PropertyList := ELStrings.Empty_String;
      end if;

      SimpleNameList := Modifier = Unmoded and then
         ELStrings.Is_Empty (E_Str => TypeMark) and then
         ELStrings.Is_Empty (E_Str => PropertyList);

      if FirstName then
         FirstName := False;
         if Annotations.Indent (Anno) /= SparkFormatCommandLineData.Inline then
            SPARK_IO.New_Line (Output, 1);
            Annotations.Write (Anno, Output, Formatting.StartCol);
         end if;
      else
         if MayUseComma and SimpleNameList then
            SPARK_IO.Put_Line (Output, ",", 0);
         else
            SPARK_IO.Put_Line (Output, ";", 0);
         end if;
         Annotations.Write (Anno, Output, Formatting.StartCol);
      end if;
      --# assert True;
      if Modifier /= Unmoded then
         SPARK_IO.Set_Col (Output, Formatting.ModifierCol);
         EStrings.Put_String (File  => Output,
                              E_Str => ModifierString (Modifier));
      end if;

      SPARK_IO.Set_Col (Output, Formatting.PrimaryIdCol);
      ELStrings.Put_String
        (File  => Output,
         E_Str => LexTokenManager.Lex_String_To_Long_String
           (Lex_Str => LexTokenManager.Seq_Algebra.Value_Of_Member (The_Heap => TheHeap,
                                                                    M        => Name)));
      if not ELStrings.Is_Empty (E_Str => TypeMark) then
         SPARK_IO.Set_Col (Output, Formatting.TypeMarkCol);
         SPARK_IO.Put_String (Output, ": ", 0);
         ELStrings.Put_String (File  => Output,
                               E_Str => TypeMark);
      end if;

      if not ELStrings.Is_Empty (E_Str => PropertyList) then
         WritePropertyList (Anno, Output, Formatting, PropertyList);
      end if;

      MayUseComma := SimpleNameList;

   end WriteName;

   procedure WriteNameList (Anno       : in     Annotations.Anno_Type;
                            Output     : in     SPARK_IO.File_Type;
                            Formatting : in     Format_Info;
                            TheHeap    : in out Heap.HeapRecord;
                            Relations  : in     Relation_Type)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in     SparkFormatCommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --# derives SPARK_IO.File_Sys     from *,
   --#                                    Anno,
   --#                                    CommandLineData.Content,
   --#                                    Formatting,
   --#                                    LexTokenManager.State,
   --#                                    Output,
   --#                                    Relations,
   --#                                    SparkFormatCommandLineData.Content,
   --#                                    TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    Formatting,
   --#                                    LexTokenManager.State,
   --#                                    Relations,
   --#                                    TheHeap;
   is
      TheSeq       : LexTokenManager.Seq_Algebra.Seq;
      FirstName    : Boolean;
      MayUseComma  : Boolean;
      TheIterator  : Iteration.Iterator;

   begin
      FirstName := True;
      MayUseComma := False;

      for Modifier in VariableModifier loop

         if Formatting.ModifiersPresent (Modifier) then
            -- extract by modifier
            LexTokenManager.Relation_Algebra.Row_Extraction (The_Heap    => TheHeap,
                                                             R           => Relations.ModifierRel,
                                                             Given_Index => VariableModifier'Pos (Modifier),
                                                             S           => TheSeq);

            Iteration.Initialise (TheHeap     => TheHeap,
                                  TheSeq      => TheSeq,
                                  TheIterator => TheIterator);

            while not Iteration.Complete (TheIterator) loop

               WriteName (Anno, Output, Formatting, FirstName, MayUseComma, TheHeap,
                          Iteration.CurrentMember (TheIterator), Modifier, Relations);

               Iteration.Next (TheHeap, TheIterator);

            end loop;

            LexTokenManager.Seq_Algebra.Dispose_Of_Seq (The_Heap => TheHeap,
                                                        S        => TheSeq);
         end if;

      end loop;

   end WriteNameList;

   procedure Reformat (This          : in     State;
                       TheHeap       : in out Heap.HeapRecord;
                       TemporaryFile : in out SPARK_IO.File_Type;
                       Output        : in     SPARK_IO.File_Type;
                       Success       :    out Boolean)
   is
      Formatting    : Format_Info;
      ModifierCol   : EStrings.Positions;
      PrimaryIdCol  : EStrings.Positions;
      TypeMarkCol   : EStrings.Positions;
      PropertiesCol : EStrings.Positions;
   begin
      ModifierCol :=
         Annotations.Name1StartCol (This.Anno, This.ParseStats.StartCol);

      if This.ParseStats.MaxModifierLength > 0 then
         PrimaryIdCol := (ModifierCol + This.ParseStats.MaxModifierLength) + 1;
      else
         PrimaryIdCol := ModifierCol;
      end if;

      TypeMarkCol := (PrimaryIdCol + This.ParseStats.MaxPrimaryIdLength) + 1;

      if SparkFormatCommandLineData.Content.PropertiesIndent /=
         SparkFormatCommandLineData.Inline then

         PropertiesCol := SparkFormatCommandLineData.Content.PropertiesIndent + 4;
      else
         -- Extra 2 columns are for the ": " preceding the TypeMark
         PropertiesCol := (TypeMarkCol + This.ParseStats.MaxTypeMarkLength) + 3;
      end if;

      Formatting := Format_Info'(
         StartCol            => This.ParseStats.StartCol,
         ModifierCol         => ModifierCol,
         PrimaryIdCol        => PrimaryIdCol,
         TypeMarkCol         => TypeMarkCol,
         PropertiesCol       => PropertiesCol,
         ModifiersPresent    => This.ParseStats.ModifiersPresent,
         TypeMarkPresent     => This.ParseStats.TypeMarkPresent,
         PropertyListPresent => This.ParseStats.PropertyListPresent);

      if This.Success then
         Annotations.WriteIntro (This.Anno, Output, Formatting.StartCol);

         -- Writes inherits and own variables but not globals and derives
         WriteNameList (Anno       => This.Anno,
                        Output     => Output,
                        Formatting => Formatting,
                        TheHeap    => TheHeap,
                        Relations  => This.Relations);

         SPARK_IO.Put_Line (Output, ";", 0);

      else
         File_IO.Reset (TemporaryFile);
         SPARKProgram.Copy (TemporaryFile, Output);
      end if;

      Success := This.Success;

   end Reformat;

   procedure Finalise (This    : in out State;
                       TheHeap : in out Heap.HeapRecord)
   is
      LocRelations  : Relation_Type;
   begin
      LocRelations  := This.Relations;
      LexTokenManager.Relation_Algebra.Dispose_Of_Relation (The_Heap => TheHeap,
                                                            R        => This.Relations.ModifierRel);
      LexTokenManager.Relation_Algebra.String.Dispose_Of_Relation (The_Heap => TheHeap,
                                                                   R        => This.Relations.TypeRel);
      LexTokenManager.Relation_Algebra.String.Dispose_Of_Relation (The_Heap => TheHeap,
                                                                   R        => This.Relations.PropertyRel);
      This.Relations := LocRelations;
   end Finalise;

end Reformatter;
