-- $Id: sparkformatcommandlinehandler.adb 15795 2010-01-26 14:30:59Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with CommandLineData;
with CommandLineHandler;
with ExaminerConstants;
with EStrings;
with FileSystem;
with ScreenEcho;
with SparkFormatCommandLineData;

use type CommandLineHandler.STyps;

package body SparkFormatCommandLineHandler
is

   procedure Process
   is

      type CommandLineErrors is (ESInvalidOption,
                                 ESNoComma,
                                 ESSource,
                                 EWtooMany,
                                 ESanno,
                                 EScontradict,
                                 ESduplicate,
                                 ESindent,
                                 ESdefault,
                                 ESInvExclude);

      CommandString : CommandLineHandler.CommandStrings;

      procedure ReadCommandLine (CommandString : out CommandLineHandler.CommandStrings)
      --# global in out SPARK_IO.File_Sys;
      --# derives CommandString,
      --#         SPARK_IO.File_Sys from SPARK_IO.File_Sys;
      is
         CmdLineFound         : Boolean;
         CmdLine              : FileSystem.TypCmdLine;
         CommandStringContent : CommandLineHandler.CommandStringContents;
      begin
         CommandStringContent := CommandLineHandler.CommandStringContents'
            (others => FileSystem.ArgumentSeparator);

         FileSystem.ReadCmdLine (CmdLineFound, CmdLine);
         if CmdLineFound then
            CommandLineHandler.CopyString (CmdLine, 1, CommandStringContent);
            CommandString := CommandLineHandler.CommandStrings'(1, CommandStringContent);
         else
            CommandString := CommandLineHandler.CommandStrings'(1, CommandStringContent);
            CommandString.CurrentPosition := 1;
         end if;
      end ReadCommandLine;

      procedure OutputError (E : in CommandLineErrors)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                E;
      is
      begin
         case E is
            when ESInvalidOption => CommandLineHandler.PutError ("Invalid command line option");
            when ESSource        => CommandLineHandler.PutError ("Source file incorrectly specified");
            when ESNoComma       => CommandLineHandler.PutError ("Comma missing in line");
            when EWtooMany       => CommandLineHandler.PutError ("Too many source files on command line ");
            when ESanno          => CommandLineHandler.PutError ("Annotation character option incorrect");
            when EScontradict    => CommandLineHandler.PutError ("Command line option duplicated or contradictory options specified");
            when ESduplicate     => CommandLineHandler.PutError ("Command line options may only be specified once");
            when ESindent        => CommandLineHandler.PutError ("Indentation option incorrect");
            when ESdefault       => CommandLineHandler.PutError ("Default function mode option incorrect");
            when ESInvExclude    => CommandLineHandler.PutError ("Invalid syntax for excluded export");
         end case;
      end OutputError;

      procedure PossibleError (E : in CommandLineErrors)
      --# global in     SparkFormatCommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                E,
      --#                                SparkFormatCommandLineData.Content;
      is
      begin
         if not SparkFormatCommandLineData.Content.Valid then
            OutputError (E);
            ScreenEcho.New_Line (1);
         end if;
      end PossibleError;

      procedure PossibleError2 (E : in CommandLineErrors;
                                F : in EStrings.T)
      --# global in     SparkFormatCommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                E,
      --#                                F,
      --#                                SparkFormatCommandLineData.Content;
      is
      begin
         if not SparkFormatCommandLineData.Content.Valid then
            OutputError (E);
            ScreenEcho.Put_Char (' ');
            ScreenEcho.Put_ExaminerLine (F);
         end if;
      end PossibleError2;

      procedure ParseCommandLine (CommandString : in CommandLineHandler.CommandStrings)
      --# global in out CommandLineData.Content;
      --#        in out SparkFormatCommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --# derives CommandLineData.Content,
      --#         SparkFormatCommandLineData.Content,
      --#         SPARK_IO.File_Sys                  from *,
      --#                                                 CommandString,
      --#                                                 SparkFormatCommandLineData.Content;
      is

         NextSymbol         : CommandLineHandler.Symbols;
         LocalCommandString : CommandLineHandler.CommandStrings;

         procedure GetNextSymbol (CommandString : in out CommandLineHandler.CommandStrings;
                                  NextSymbol    :    out CommandLineHandler.Symbols)
         --# derives CommandString,
         --#         NextSymbol    from CommandString;
         is
         begin
            -- This procedure is intended to return NextSymbol; however, if the
            -- symbol is not a string then the string field is not set.  Although
            -- it is not used in these circcumstances its lack of definition
            -- causes so many flow errors its is better to use an aggregate to
            -- initialize NextSymbol here, and then assign the final value
            -- to NextSymbol (for compatibility with the Ada83 "out parameter" rule
            NextSymbol := CommandLineHandler.Symbols'(Typ       => CommandLineHandler.SEmpty,
                                                      TheString => EStrings.Empty_String);
            CommandLineHandler.SkipSpaces (CommandString);
            if CommandString.CurrentPosition <=
              ExaminerConstants.MaxCommandStringLength then
               case CommandString.Contents (CommandString.CurrentPosition) is
                  when '=' =>
                     NextSymbol.Typ := CommandLineHandler.SEqual;
                     CommandString.CurrentPosition := CommandString.CurrentPosition + 1;
                  when '/' =>
                     --this condition is invariant for any particular system, we are actually
                     --simulating conditional compilation for different target platforms.
                     --Intended behaviour is correct despite flow error that will result.
                     --# accept Flow_Message, 22, "Simulation of conditional compilation";
                     if FileSystem.UseUnixCommandLine then
                        --# end accept;
                        NextSymbol.Typ := CommandLineHandler.SString;
                        CommandLineHandler.ReadTheString (CommandString, NextSymbol);
                     else -- FileSystem.UseWindowsCommandLine assumed
                        NextSymbol.Typ := CommandLineHandler.SSlash;
                        CommandString.CurrentPosition :=
                          CommandString.CurrentPosition + 1;
                     end if;
                  when ',' =>
                     --this condition is invariant for any particular system, we are actually
                     --simulating conditional compilation for different target platforms.
                     --Intended behaviour is correct despite flow error that will result.
                     --# accept Flow_Message, 22, "Simulation of conditional compilation";
                     if FileSystem.UseUnixCommandLine then
                        --# end accept;
                        NextSymbol.Typ := CommandLineHandler.SString;
                        CommandLineHandler.ReadTheString (CommandString, NextSymbol); --675
                     else -- FileSystem.UseWindowsCommandLine assumed
                        NextSymbol.Typ := CommandLineHandler.SComma;
                        CommandString.CurrentPosition := CommandString.CurrentPosition + 1;
                     end if;
                  when '-' =>
                        -- We allow '-' as a switch character on all platforms
                        NextSymbol.Typ := CommandLineHandler.SSlash;
                        CommandString.CurrentPosition :=
                          CommandString.CurrentPosition + 1;
                  when others =>
                     NextSymbol.Typ := CommandLineHandler.SString;
                     CommandLineHandler.ReadTheString (CommandString, NextSymbol);
               end case;
            else
               -- Exceeded maximum command line length
               NextSymbol.Typ := CommandLineHandler.SEmpty;
            end if;
         end GetNextSymbol;

         procedure ParseCommandOptions (CommandString : in out CommandLineHandler.CommandStrings;
                                        NextSymbol    : in out CommandLineHandler.Symbols)
         --# global in out CommandLineData.Content;
         --#        in out SparkFormatCommandLineData.Content;
         --#        in out SPARK_IO.File_Sys;
         --# derives CommandLineData.Content,
         --#         CommandString,
         --#         NextSymbol,
         --#         SparkFormatCommandLineData.Content,
         --#         SPARK_IO.File_Sys                  from *,
         --#                                                 CommandString,
         --#                                                 NextSymbol,
         --#                                                 SparkFormatCommandLineData.Content;
         is

            AnnotationCharacterFound  : Boolean;
            CompressFound             : Boolean;
            ExpandFound               : Boolean;
            AddModesFound             : Boolean;
            GlobalIndentFound         : Boolean;
            ExportIndentFound         : Boolean;
            ImportIndentFound         : Boolean;
            InheritIndentFound        : Boolean;
            OwnIndentFound            : Boolean;
            RefinementIndentFound     : Boolean;
            ConstituentIndentFound    : Boolean;
            InitializationIndentFound : Boolean;
            SeparatorIndentFound      : Boolean;
            PropertiesIndentFound     : Boolean;
            AlphabeticOrderingFound   : Boolean;
            HelpFound                 : Boolean;
            VersionFound              : Boolean;
            DefaultFunctionModeFound  : Boolean;
            ExcludeExportFound        : Boolean;

            OptName   : EStrings.T;
            OptVal    : EStrings.T;
            OptNameOK : Boolean;
            OptValOK  : Boolean;

            OK : Boolean;

            IndentVal : Integer;

            -- reads in the option name and the value assigned to it
            procedure ReadOption (OptName       :    out EStrings.T;
                                  OptNameOK     :    out Boolean;
                                  OptVal        :    out EStrings.T;
                                  OptValOK      :    out Boolean;
                                  CommandString : in out CommandLineHandler.CommandStrings;
                                  NextSymbol    :    out CommandLineHandler.Symbols)
            --# derives CommandString,
            --#         NextSymbol,
            --#         OptName,
            --#         OptNameOK,
            --#         OptVal,
            --#         OptValOK      from CommandString;
            --  pre  NextSymbol.Typ = CommandLineHandler.SSlash;
            is
            begin
               OptVal := EStrings.Empty_String;
               GetNextSymbol (CommandString, NextSymbol);
               OptNameOK := NextSymbol.Typ = CommandLineHandler.SString;
               OptName := NextSymbol.TheString;
               GetNextSymbol (CommandString, NextSymbol);
               if OptNameOK and NextSymbol.Typ = CommandLineHandler.SEqual then
                  GetNextSymbol (CommandString, NextSymbol);
                  OptValOK := NextSymbol.Typ = CommandLineHandler.SString;
                  OptVal := NextSymbol.TheString;
                  GetNextSymbol (CommandString, NextSymbol);
               else
                  OptValOK := False;
               end if;
            end ReadOption;

            -- converts an ExaminerString to a Natural. S.Content(1) is
            -- assumes to be a member of '0' .. '9'
            procedure ExaminerStringToNatural (S  : in     EStrings.T;
                                               I  :    out Natural;
                                               OK :    out Boolean)
            --# derives I,
            --#         OK from S;
            is
               Stop : Natural;
            begin
               if EStrings.Get_Length (E_Str => S) >= 1 then
                  EStrings.Get_Int_From_String (Source   => S,
                                                Item     => I,
                                                Start_Pt => 1,
                                                Stop     => Stop);
                  if Stop = EStrings.Get_Length (E_Str => S) then
                     OK := True;
                  else
                     I  := 0;
                     OK := False;
                  end if;
               else
                  OK := False;
                  I := 0;
               end if;
            end ExaminerStringToNatural;

            procedure ParseIndentOption (OptValue    : in     EStrings.T;
                                         OptValueOK  : in     Boolean;
                                         IndentValue :    out Natural;
                                         OK          :    out Boolean)
            --# derives IndentValue,
            --#         OK          from OptValue,
            --#                          OptValueOK;
            is
               LocOK          : Boolean;
               LocIndentValue : Natural;
            begin
               if OptValueOK and EStrings.Get_Length (E_Str => OptValue) >= 1 then
                  case EStrings.Get_Element (E_Str => OptValue,
                                             Pos   => 1) is
                     -- if option is set to "inline"
                     when 'i' =>
                        CommandLineHandler.Check_Option_Name (Opt_Name => OptValue,
                                                              Str      => "inline",
                                                              OK       => LocOK);
                        LocIndentValue := SparkFormatCommandLineData.Inline;

                     -- check if valid number
                     when '1' .. '9' =>
                        ExaminerStringToNatural (OptValue, LocIndentValue, LocOK);

                     when others =>
                        LocOK := False;
                        -- assign default value
                        LocIndentValue := SparkFormatCommandLineData.Inline;
                  end case;
                  IndentValue := LocIndentValue;
                  OK := LocOK;
               else
                  IndentValue := SparkFormatCommandLineData.Inline;
                  OK := False;
               end if;
            end ParseIndentOption;

            --  CFR 1753: A crude syntax check for a dotted simple name as
            --  required by the /exclude_export.
            --  The name must include at least one dot.
            function CheckForDottedName (S : EStrings.T)
               return Boolean
            is
               type Symb_T is (Alpha, Digit, Dot, Under);
               LastSymb : Symb_T;
               DotFound : Boolean;
               Error    : Boolean;
            begin
               DotFound := False;
               Error  := False;
               if EStrings.Get_Length (E_Str => S) > 2 then  -- We require at least 3 chars, i.e., X.Y
                  if EStrings.Get_Element (E_Str => S,
                                           Pos   => 1) in 'A' .. 'Z' or else
                    EStrings.Get_Element (E_Str => S,
                                          Pos   => 1) in 'a' .. 'z' then
                     LastSymb := Alpha;
                     for I in EStrings.Positions range 2 .. EStrings.Get_Length (E_Str => S) loop
                        case EStrings.Get_Element (E_Str => S,
                                                   Pos   => I) is
                           when '.' =>
                              if LastSymb /= Dot and LastSymb /= Under and
                                 EStrings.Get_Length (E_Str => S) > I
                              then
                                 DotFound := True;
                                 LastSymb := Dot;
                              else
                                 Error := True;
                              end if;
                           when 'A' .. 'Z' | 'a' .. 'z' =>
                              LastSymb := Alpha;
                           when '0' .. '9' =>
                              if LastSymb /= Dot and LastSymb /= Under then
                                 LastSymb := Digit;
                              else
                                 Error := True;
                              end if;
                           when '_' =>
                              if LastSymb /= Dot and LastSymb /= Under and
                                 EStrings.Get_Length (E_Str => S) > I
                              then
                                 LastSymb := Under;
                              else
                                 Error := True;
                              end if;
                           when others =>
                              Error := True;
                        end case;

                        exit when Error;
                     end loop;
                  end if;
               end if;

               return DotFound and not Error;
            end CheckForDottedName;

         begin -- ParseCommandOptions

            AnnotationCharacterFound  := False;
            CompressFound             := False;
            ExpandFound               := False;
            AddModesFound             := False;
            GlobalIndentFound         := False;
            ExportIndentFound         := False;
            ImportIndentFound         := False;
            InheritIndentFound        := False;
            OwnIndentFound            := False;
            RefinementIndentFound     := False;
            ConstituentIndentFound    := False;
            InitializationIndentFound := False;
            SeparatorIndentFound      := False;
            PropertiesIndentFound     := False;
            AlphabeticOrderingFound   := False;
            HelpFound                 := False;
            VersionFound              := False;
            DefaultFunctionModeFound  := False;
            ExcludeExportFound        := False;

            loop
               exit when NextSymbol.Typ /= CommandLineHandler.SSlash;
               ReadOption (OptName,
                           OptNameOK,
                           OptVal,
                           OptValOK,
                           CommandString,
                           NextSymbol);
               if OptNameOK then
                  case EStrings.Get_Element (E_Str => OptName,
                                             Pos   => 1) is
                     when 'a' | 'A' =>
                        case EStrings.Get_Element (E_Str => OptName,
                                                   Pos   => 2) is
                           when 'd' | 'D' =>
                              CommandLineHandler.Check_Option_Name (Opt_Name => OptName,
                                                                    Str      => "add_modes",
                                                                    OK       => OK);
                              if OK then
                                 SparkFormatCommandLineData.Content.Valid := not AddModesFound;
                                 PossibleError (EScontradict);
                                 AddModesFound := True;
                                 SparkFormatCommandLineData.Content.AddModes := True;
                              else
                                 SparkFormatCommandLineData.Content.Valid := False;
                                 PossibleError2 (ESInvalidOption, OptName);
                              end if;
                           when 'n' | 'N' =>
                              CommandLineHandler.Check_Option_Name (Opt_Name => OptName,
                                                                    Str      => "annotation_character",
                                                                    OK       => OK);
                              if OK then
                                 SparkFormatCommandLineData.Content.Valid := OptValOK and
                                   EStrings.Get_Length (E_Str => OptVal) = 1 and not AnnotationCharacterFound;
                                 PossibleError (ESanno);
                                 AnnotationCharacterFound := True;
                                 CommandLineData.Content.AnnoChar := EStrings.Get_Element (E_Str => OptVal,
                                                                                           Pos   => 1);
                              else
                                 SparkFormatCommandLineData.Content.Valid := False;
                                 PossibleError2 (ESInvalidOption, OptName);
                              end if;
                           when others =>
                              SparkFormatCommandLineData.Content.Valid := False;
                              PossibleError2 (ESInvalidOption, OptName);
                        end case;

                     when 'c' | 'C' =>
                        case EStrings.Get_Element (E_Str => OptName,
                                                   Pos   => 3) is
                           when 'm' | 'M' =>
                              CommandLineHandler.Check_Option_Name (Opt_Name => OptName,
                                                                    Str      => "compress",
                                                                    OK       => OK);
                              if OK then
                                 SparkFormatCommandLineData.Content.Valid := not CompressFound;
                                 PossibleError (ESduplicate);
                                 CompressFound := True;
                                 SparkFormatCommandLineData.Content.Operation := SparkFormatCommandLineData.Compress;
                              else
                                 SparkFormatCommandLineData.Content.Valid := False;
                                 PossibleError2 (ESInvalidOption, OptName);
                              end if;
                           when 'n' | 'N' =>
                              CommandLineHandler.Check_Option_Name (Opt_Name => OptName,
                                                                    Str      => "constituent_indent",
                                                                    OK       => OK);
                              if OK then
                                 ParseIndentOption (OptVal, OptValOK, IndentVal, OK);
                                 if OK and not ConstituentIndentFound then
                                    SparkFormatCommandLineData.Content.Valid := True;
                                    SparkFormatCommandLineData.Content.ConstituentIndent := IndentVal;
                                 else
                                    SparkFormatCommandLineData.Content.Valid := False;
                                 end if;
                                 PossibleError (ESindent);
                                 ConstituentIndentFound := True;
                              else
                                 SparkFormatCommandLineData.Content.Valid := False;
                                 PossibleError2 (ESInvalidOption, OptName);
                              end if;
                           when others =>
                              SparkFormatCommandLineData.Content.Valid := False;
                              PossibleError2 (ESInvalidOption, OptName);
                        end case;


                     when 'd' | 'D' =>
                        CommandLineHandler.Check_Option_Name (Opt_Name => OptName,
                                                              Str      => "default_function_modes",
                                                              OK       => OK);
                        if OK then
                           SparkFormatCommandLineData.Content.Valid := OptValOK and not DefaultFunctionModeFound;
                           PossibleError (ESdefault);
                           if SparkFormatCommandLineData.Content.Valid then -- go on to check selection
                              DefaultFunctionModeFound := True;
                              case EStrings.Get_Element (E_Str => OptVal,
                                                         Pos   => 1) is
                                 when 'i' | 'I' =>
                                    CommandLineHandler.Check_Option_Name (Opt_Name => OptVal,
                                                                          Str      => "in_mode",
                                                                          OK       => OK);
                                    if OK then
                                       SparkFormatCommandLineData.Content.DefaultFunctionMode :=
                                         SparkFormatCommandLineData.InMode;
                                    else
                                       SparkFormatCommandLineData.Content.Valid := False;
                                       PossibleError (ESdefault);
                                    end if;
                                 when 'u' | 'U' =>
                                    CommandLineHandler.Check_Option_Name (Opt_Name => OptVal,
                                                                          Str      => "unmoded",
                                                                          OK       => OK);
                                    if OK then
                                       SparkFormatCommandLineData.Content.DefaultFunctionMode :=
                                         SparkFormatCommandLineData.Unmoded;
                                    else
                                       SparkFormatCommandLineData.Content.Valid := False;
                                       PossibleError (ESdefault);
                                    end if;
                                 when others =>
                                    SparkFormatCommandLineData.Content.Valid := False;
                                    PossibleError (ESdefault);
                              end case;
                           end if;
                        else
                           SparkFormatCommandLineData.Content.Valid := False;
                           PossibleError2 (ESInvalidOption, OptName);
                        end if;

                     when 'e' | 'E' =>
                        case EStrings.Get_Element (E_Str => OptName,
                                                   Pos   => 4) is
                           when 'a' | 'A' =>
                              CommandLineHandler.Check_Option_Name (Opt_Name => OptName,
                                                                    Str      => "expand",
                                                                    OK       => OK);
                              if OK then
                                 SparkFormatCommandLineData.Content.Valid := not ExpandFound;
                                 PossibleError (ESduplicate);
                                 ExpandFound := True;
                                 SparkFormatCommandLineData.Content.Operation := SparkFormatCommandLineData.Expand;
                              else
                                 SparkFormatCommandLineData.Content.Valid := False;
                                 PossibleError2 (ESInvalidOption, OptName);
                              end if;
                           when 'l' | 'L' =>
                              CommandLineHandler.Check_Option_Name (Opt_Name => OptName,
                                                                    Str      => "exclude_export",
                                                                    OK       => OK);
                              if OK then
                                 if not ExcludeExportFound then
                                    if OptValOK and then CheckForDottedName (OptVal) then
                                       SparkFormatCommandLineData.Content.Valid := True;
                                       SparkFormatCommandLineData.Content.ExcludeExport := OptVal;
                                    else
                                       SparkFormatCommandLineData.Content.Valid := False;
                                       PossibleError (ESInvExclude);
                                    end if;
                                 else
                                    SparkFormatCommandLineData.Content.Valid := False;
                                    PossibleError (ESduplicate);
                                 end if;
                                 ExcludeExportFound := True;
                              else
                                 SparkFormatCommandLineData.Content.Valid := False;
                                 PossibleError2 (ESInvalidOption, OptName);
                              end if;
                           when 'o' | 'O' =>
                              CommandLineHandler.Check_Option_Name (Opt_Name => OptName,
                                                                    Str      => "export_indent",
                                                                    OK       => OK);
                              if OK then
                                 ParseIndentOption (OptVal, OptValOK, IndentVal, OK);
                                 if OK and not ExportIndentFound then
                                    SparkFormatCommandLineData.Content.Valid := True;
                                    SparkFormatCommandLineData.Content.ExportIndent := IndentVal;
                                 else
                                    SparkFormatCommandLineData.Content.Valid := False;
                                 end if;
                                 PossibleError (ESindent);
                                 ExportIndentFound := True;
                              else
                                 SparkFormatCommandLineData.Content.Valid := False;
                                 PossibleError2 (ESInvalidOption, OptName);
                              end if;
                           when others =>
                              SparkFormatCommandLineData.Content.Valid := False;
                              PossibleError2 (ESInvalidOption, OptName);
                        end case;

                     when 'g' | 'G' =>
                        CommandLineHandler.Check_Option_Name (Opt_Name => OptName,
                                                              Str      => "global_indent",
                                                              OK       => OK);
                        if OK then
                           ParseIndentOption (OptVal, OptValOK, IndentVal, OK);
                           if OK and not GlobalIndentFound then
                              SparkFormatCommandLineData.Content.Valid := True;
                              SparkFormatCommandLineData.Content.GlobalIndent := IndentVal;
                           else
                              SparkFormatCommandLineData.Content.Valid := False;
                           end if;
                           PossibleError (ESindent);
                           GlobalIndentFound := True;
                        else
                           SparkFormatCommandLineData.Content.Valid := False;
                           PossibleError2 (ESInvalidOption, OptName);
                        end if;

                     when 'h' | 'H' =>
                        CommandLineHandler.Check_Option_Name (Opt_Name => OptName,
                                                              Str      => "help",
                                                              OK       => OK);
                        if OK then
                           SparkFormatCommandLineData.Content.Valid := not HelpFound;
                           PossibleError (ESduplicate);
                           HelpFound := True;
                           SparkFormatCommandLineData.Content.Help := True;
                        else
                           SparkFormatCommandLineData.Content.Valid := False;
                           PossibleError2 (ESInvalidOption, OptName);
                        end if;

                     when 'i' | 'I' =>
                        case EStrings.Get_Element (E_Str => OptName,
                                                   Pos   => 3) is
                           when 'h' | 'H' =>
                              CommandLineHandler.Check_Option_Name (Opt_Name => OptName,
                                                                    Str      => "inherit_indent",
                                                                    OK       => OK);
                              if OK then
                                 ParseIndentOption (OptVal, OptValOK, IndentVal, OK);
                                 if OK and not InheritIndentFound then
                                    SparkFormatCommandLineData.Content.Valid := True;
                                    SparkFormatCommandLineData.Content.InheritIndent := IndentVal;
                                 else
                                    SparkFormatCommandLineData.Content.Valid := False;
                                 end if;
                                 PossibleError (ESindent);
                                 InheritIndentFound := True;
                              else
                                 SparkFormatCommandLineData.Content.Valid := False;
                                 PossibleError2 (ESInvalidOption, OptName);
                              end if;
                           when 'i' | 'I' =>
                              CommandLineHandler.Check_Option_Name (Opt_Name => OptName,
                                                                    Str      => "initialization_indent",
                                                                    OK       => OK);
                              if OK then
                                 ParseIndentOption (OptVal, OptValOK, IndentVal, OK);
                                 if OK and not InitializationIndentFound then
                                    SparkFormatCommandLineData.Content.Valid := True;
                                    SparkFormatCommandLineData.Content.InitializationIndent := IndentVal;
                                 else
                                    SparkFormatCommandLineData.Content.Valid := False;
                                 end if;
                                 PossibleError (ESindent);
                                 InitializationIndentFound := True;
                              else
                                 SparkFormatCommandLineData.Content.Valid := False;
                                 PossibleError2 (ESInvalidOption, OptName);
                              end if;
                           when 'p' | 'P' =>
                              CommandLineHandler.Check_Option_Name (Opt_Name => OptName,
                                                                    Str      => "import_indent",
                                                                    OK       => OK);
                              if OK then
                                 ParseIndentOption (OptVal, OptValOK, IndentVal, OK);
                                 if OK and not ImportIndentFound then
                                    SparkFormatCommandLineData.Content.Valid := True;
                                    SparkFormatCommandLineData.Content.ImportIndent := IndentVal;
                                 else
                                    SparkFormatCommandLineData.Content.Valid := False;
                                 end if;
                                 PossibleError (ESindent);
                                 ImportIndentFound := True;
                              else
                                 SparkFormatCommandLineData.Content.Valid := False;
                                 PossibleError2 (ESInvalidOption, OptName);
                              end if;
                           when others =>
                              SparkFormatCommandLineData.Content.Valid := False;
                              PossibleError2 (ESInvalidOption, OptName);
                        end case;

                     when 'n' | 'N' =>
                        case EStrings.Get_Element (E_Str => OptName,
                                                   Pos   => 3) is
                           when 'a' | 'A' => -- 3rd letter
                              CommandLineHandler.Check_Option_Name (Opt_Name => OptName,
                                                                    Str      => "noadd_modes",
                                                                    OK       => OK);
                              if OK then
                                 SparkFormatCommandLineData.Content.Valid := not AddModesFound;
                                 PossibleError (EScontradict);
                                 AddModesFound := True;
                                 SparkFormatCommandLineData.Content.AddModes := False;
                              else
                                 SparkFormatCommandLineData.Content.Valid := False;
                                 PossibleError2 (ESInvalidOption, OptName);
                              end if;
                           when 's' | 'W' => -- 3rd letter
                              CommandLineHandler.Check_Option_Name (Opt_Name => OptName,
                                                                    Str      => "noswitch",
                                                                    OK       => OK);
                              if OK then
                                 null; -- already dealt with, so ignore here
                              else
                                 SparkFormatCommandLineData.Content.Valid := False;
                                 PossibleError2 (ESInvalidOption, OptName);
                              end if;
                           when others =>
                              SparkFormatCommandLineData.Content.Valid := False;
                              PossibleError2 (ESInvalidOption, OptName);
                        end case;

                     when 'o' | 'O' =>
                        case EStrings.Get_Element (E_Str => OptName,
                                                   Pos   => 2) is
                           when 'r' | 'R' =>
                              CommandLineHandler.Check_Option_Name (Opt_Name => OptName,
                                                                    Str      => "order",
                                                                    OK       => OK);
                              if OK then
                                 if OptValOK then
                                    case EStrings.Get_Element (E_Str => OptVal,
                                                               Pos   => 1) is
                                       when 'a' | 'A' =>
                                          CommandLineHandler.Check_Option_Name (Opt_Name => OptVal,
                                                                                Str      => "alphabetic",
                                                                                OK       => OK);
                                          if OK then
                                             SparkFormatCommandLineData.Content.Valid := not AlphabeticOrderingFound;
                                             PossibleError (EScontradict);
                                             AlphabeticOrderingFound := True;
                                          else
                                             SparkFormatCommandLineData.Content.Valid := False;
                                             PossibleError2 (ESInvalidOption, OptVal);
                                          end if;
                                       when others =>
                                          SparkFormatCommandLineData.Content.Valid := False;
                                          PossibleError2 (ESInvalidOption, OptVal);
                                    end case;
                                 else
                                    SparkFormatCommandLineData.Content.Valid := False;
                                    PossibleError2 (ESInvalidOption, OptVal);
                                 end if;
                              else
                                 SparkFormatCommandLineData.Content.Valid := False;
                                 PossibleError2 (ESInvalidOption, OptName);
                              end if;
                           when 'w' | 'W' =>
                              CommandLineHandler.Check_Option_Name (Opt_Name => OptName,
                                                                    Str      => "own_indent",
                                                                    OK       => OK);
                              if OK then
                                 ParseIndentOption (OptVal, OptValOK, IndentVal, OK);
                                 if OK and not OwnIndentFound then
                                    SparkFormatCommandLineData.Content.Valid := True;
                                    SparkFormatCommandLineData.Content.OwnIndent := IndentVal;
                                 else
                                    SparkFormatCommandLineData.Content.Valid := False;
                                 end if;
                                 PossibleError (ESindent);
                                 OwnIndentFound := True;
                              else
                                 SparkFormatCommandLineData.Content.Valid := False;
                                 PossibleError2 (ESInvalidOption, OptName);
                              end if;
                           when others =>
                              SparkFormatCommandLineData.Content.Valid := False;
                              PossibleError2 (ESInvalidOption, OptName);
                        end case;

                     when 'p' | 'P' =>
                        CommandLineHandler.Check_Option_Name (Opt_Name => OptName,
                                                              Str      => "properties_indent",
                                                              OK       => OK);
                        if OK then
                           ParseIndentOption (OptVal, OptValOK, IndentVal, OK);
                           if OK and not PropertiesIndentFound then
                              SparkFormatCommandLineData.Content.Valid := True;
                              SparkFormatCommandLineData.Content.PropertiesIndent := IndentVal;
                           else
                              SparkFormatCommandLineData.Content.Valid := False;
                           end if;
                           PossibleError (ESindent);
                           RefinementIndentFound := True;
                        else
                           SparkFormatCommandLineData.Content.Valid := False;
                           PossibleError2 (ESInvalidOption, OptName);
                        end if;

                     when 'r' | 'R' =>
                        CommandLineHandler.Check_Option_Name (Opt_Name => OptName,
                                                              Str      => "refinement_indent",
                                                              OK       => OK);
                        if OK then
                           ParseIndentOption (OptVal, OptValOK, IndentVal, OK);
                           if OK and not RefinementIndentFound then
                              SparkFormatCommandLineData.Content.Valid := True;
                              SparkFormatCommandLineData.Content.RefinementIndent := IndentVal;
                           else
                              SparkFormatCommandLineData.Content.Valid := False;
                           end if;
                           PossibleError (ESindent);
                           RefinementIndentFound := True;
                        else
                           SparkFormatCommandLineData.Content.Valid := False;
                           PossibleError2 (ESInvalidOption, OptName);
                        end if;

                     when 's' | 'S' =>
                        CommandLineHandler.Check_Option_Name (Opt_Name => OptName,
                                                              Str      => "separator_indent",
                                                              OK       => OK);
                        if OK then
                           ParseIndentOption (OptVal, OptValOK, IndentVal, OK);
                           if OK and not SeparatorIndentFound then
                              SparkFormatCommandLineData.Content.Valid := True;
                              SparkFormatCommandLineData.Content.SeparatorIndent := IndentVal;
                           else
                              SparkFormatCommandLineData.Content.Valid := False;
                           end if;
                           PossibleError (ESindent);
                           SeparatorIndentFound := True;
                        else
                           SparkFormatCommandLineData.Content.Valid := False;
                           PossibleError2 (ESInvalidOption, OptName);
                        end if;

                     when 'v' | 'V' =>
                        CommandLineHandler.Check_Option_Name (Opt_Name => OptName,
                                                              Str      => "version",
                                                              OK       => OK);
                        if OK then
                           SparkFormatCommandLineData.Content.Valid := not VersionFound;
                           PossibleError (ESduplicate);
                           VersionFound := True;
                           SparkFormatCommandLineData.Content.Version := True;
                        else
                           SparkFormatCommandLineData.Content.Valid := False;
                           PossibleError2 (ESInvalidOption, OptName);
                        end if;

                     when others =>
                        SparkFormatCommandLineData.Content.Valid := False;
                        PossibleError2 (ESInvalidOption, OptName);
                  end case;
               else
                  SparkFormatCommandLineData.Content.Valid := False;
                  PossibleError2 (ESInvalidOption, OptName);
               end if;
               exit when not SparkFormatCommandLineData.Content.Valid;
            end loop;
         end ParseCommandOptions;

         procedure ParseArguments (CommandString : in CommandLineHandler.CommandStrings;
                                   NextSymbol    : in CommandLineHandler.Symbols)
         --# global in out SparkFormatCommandLineData.Content;
         --#        in out SPARK_IO.File_Sys;
         --# derives SparkFormatCommandLineData.Content,
         --#         SPARK_IO.File_Sys                  from *,
         --#                                                 CommandString,
         --#                                                 NextSymbol,
         --#                                                 SparkFormatCommandLineData.Content;
         --  pre NextSymbol.Typ in {SString, SEmpty};
         is

            LocalNextSymbol    : CommandLineHandler.Symbols;
            LocalCommandString : CommandLineHandler.CommandStrings;

            procedure ParseFileEntry (CommandString : in out CommandLineHandler.CommandStrings;
                                      NextSymbol    : in out CommandLineHandler.Symbols)
            --# global in out SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --# derives CommandString,
            --#         SparkFormatCommandLineData.Content from *,
            --#                                                 NextSymbol &
            --#         NextSymbol                         from *,
            --#                                                 CommandString &
            --#         SPARK_IO.File_Sys                  from *,
            --#                                                 NextSymbol,
            --#                                                 SparkFormatCommandLineData.Content;
            --  pre  NextSymbol.Typ = CommandLineHandler.SString;
            --  post NextSymbol.Typ in {CommandLineHandler.SComma, CommandLineHandler.SEmpty};
            is
               FileName : EStrings.T;
            begin
               case NextSymbol.Typ is
                  when CommandLineHandler.SString =>
                     SparkFormatCommandLineData.Content.NumberSource :=
                       SparkFormatCommandLineData.Content.NumberSource + 1;

                     FileName := NextSymbol.TheString;

                     -- To be reinstated when metafile support implemented
--                     if FileName.Content (1) = '@' then
--                        FileSystem.CheckExtension
--                          (FileName,
--                           FileSystem.CaseOfFileExtensions (SparkFormatCommandLineData.MetaFileExtension));
--                        SparkFormatCommandLineData.Content.SourceFileList
--                          (SparkFormatCommandLineData.Content.NumberSource).SourceFileName := FileName;
--                        GetNextSymbol (CommandString, NextSymbol);
--                     else
                     FileSystem.CheckExtension
                       (FileName,
                        SparkFormatCommandLineData.Content.SourceExtension);
                     SparkFormatCommandLineData.Content.SourceFileList
                       (SparkFormatCommandLineData.Content.NumberSource).SourceFileName := FileName;
                     GetNextSymbol (CommandString, NextSymbol);
--                     end if;
                  when CommandLineHandler.SEmpty =>
                     null;
                  when others =>
                     SparkFormatCommandLineData.Content.Valid := False;
                     PossibleError (ESSource);
               end case;
            end ParseFileEntry;

         begin
            LocalNextSymbol    := NextSymbol;
            LocalCommandString := CommandString;
            SparkFormatCommandLineData.Content.NumberSource := 0;
            loop
               ParseFileEntry (LocalCommandString, LocalNextSymbol);

               exit when LocalNextSymbol.Typ = CommandLineHandler.SEmpty or
                 not SparkFormatCommandLineData.Content.Valid;
               if SparkFormatCommandLineData.Content.NumberSource =
                 ExaminerConstants.MaxFilesOnCommandLine
               then
                  SparkFormatCommandLineData.Content.Valid := False;
                  OutputError (EWtooMany);
                  exit;
               end if;
               if FileSystem.UseWindowsCommandLine and LocalNextSymbol.Typ = CommandLineHandler.SComma then
                  -- CFR 1824: Allow comma or space as separator on Windows
                  GetNextSymbol (LocalCommandString, LocalNextSymbol);
               end if;
            end loop;
         end ParseArguments;

      begin -- ParseCommandLine;
         LocalCommandString := CommandString;
         GetNextSymbol (LocalCommandString, NextSymbol);
         ParseCommandOptions (LocalCommandString, NextSymbol);
         if SparkFormatCommandLineData.Content.Valid then
            ParseArguments (LocalCommandString, NextSymbol);
         end if;
      end ParseCommandLine;

   begin --Process
      CommandLineHandler.ProcessDefaultsFromSwitchFile;
      ReadCommandLine  (CommandString);
      ParseCommandLine (CommandString);
   end Process;

end SparkFormatCommandLineHandler;
