-- $Id: vcs-printvcreport.adb 16062 2010-02-15 14:18:09Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Procedure to print the VC report                                            --
--                                                                            --
--------------------------------------------------------------------------------
with PathFormatter,
     SPARK_IO,
     VCDetails;

separate (VCS)

--------------------------------------------------------------------------
procedure PrintVCReport (VCFileName        : in ELStrings.T;
                         VCFileDateTime    : in EStrings.T;
                         SIVFileName       : in ELStrings.T;
                         SIVFileDateTime   : in EStrings.T;
                         PLGFileName       : in ELStrings.T;
                         PLGFileDateTime   : in EStrings.T;
                         SLGFileName       : in ELStrings.T;
                         VCError           : in Boolean;
                         VCErrorString     : in EStrings.T;
                         SIVError          : in Boolean;
                         SIVErrorString    : in EStrings.T;
                         PLGError          : in Boolean;
                         PLGErrorString    : in EStrings.T;
                         SLGError          : in Boolean;
                         SLGFileMissing    : in Boolean;
                         REVErrors         : in ReviewErrors;
                         ReportFile        : in SPARK_IO.File_Type;
                         TempFile          : in SPARK_IO.File_Type;
                         TempFalseFile     : in SPARK_IO.File_Type;
                         TempContraFile    : in SPARK_IO.File_Type;
                         TempUserFile      : in SPARK_IO.File_Type;
                         TempPRVerrFile    : in SPARK_IO.File_Type;
                         TempWarnErrorFile : in SPARK_IO.File_Type)
is
   Table_Pad_Width       : constant Integer := 44;

   Proved_By_Col_Width   : constant Integer := 20;
   Dead_Path_Col_Width   : constant Integer := 11;
   Status_Code_Str_Width : constant Integer := 1;

   subtype Proved_By_Str_Index is Integer range 1 .. Proved_By_Col_Width;
   subtype Dead_Path_Str_Index is Integer range 1 .. Dead_Path_Col_Width;

   subtype Status_Code_Str_Index is Integer range 1 .. Status_Code_Str_Width;

   subtype Proved_By_Str_T is String (Proved_By_Str_Index);
   subtype Dead_Path_Str_T is String (Dead_Path_Str_Index);
   subtype Status_Code_Str_T is String (Status_Code_Str_Index);

   type Proved_By_Str_Array is array (VCDetails.VC_State_T) of Proved_By_Str_T;
   type Dead_Path_Str_Array is array (VCDetails.DPC_State_T) of Dead_Path_Str_T;

   type Proved_By_Status_Code_Str_Array is array (VCDetails.VC_State_T) of Status_Code_Str_T;
   type Dead_Path_Status_Code_Str_Array is array (VCDetails.DPC_State_T) of Status_Code_Str_T;

   Proved_By_Str : constant Proved_By_Str_Array :=
     Proved_By_Str_Array'(
                          VCDetails.VC_Not_Present                   => " No VCG             ",
                          VCDetails.VC_SIV_Not_Present               => " No SIV             ",
                          VCDetails.VC_Undischarged                  => " Undischarged       ",
                          VCDetails.VC_Proved_By_Examiner            => " Examiner           ",
                          VCDetails.VC_Proved_By_Inference           => " Inference          ",
                          VCDetails.VC_Proved_By_Contradiction       => " Contradiction      ",
                          VCDetails.VC_Proved_Using_User_Proof_Rules => " User Rules         ",
                          VCDetails.VC_Proved_By_Checker             => " Checker            ",
                          VCDetails.VC_Proved_By_Review              => " Review             ",
                          VCDetails.VC_False                         => " False              ");

   Dead_Path_Str : constant Dead_Path_Str_Array :=
     Dead_Path_Str_Array'(VCDetails.DPC_Not_Present      => " No DPC    ",
                          VCDetails.DPC_SDP_Not_Present  => " No SDP    ",
                          VCDetails.DPC_Unchecked        => " Unchecked ",
                          VCDetails.DPC_Live             => " Live      ",
                          VCDetails.DPC_Dead             => " Dead      ");

   Proved_By_Status_Code : constant Proved_By_Status_Code_Str_Array :=
     Proved_By_Status_Code_Str_Array'(
                                      VCDetails.VC_Not_Present                     => "-",
                                      VCDetails.VC_SIV_Not_Present                 => "S",
                                      VCDetails.VC_Undischarged                    => "U",
                                      VCDetails.VC_Proved_By_Examiner              => "E",
                                      VCDetails.VC_Proved_By_Inference             => "I",
                                      VCDetails.VC_Proved_By_Contradiction         => "X",
                                      VCDetails.VC_Proved_By_Checker               => "C",
                                      VCDetails.VC_Proved_By_Review                => "R",
                                      VCDetails.VC_Proved_Using_User_Proof_Rules   => "P",
                                      VCDetails.VC_False                           => "F");

   Dead_Path_Status_Code : constant Dead_Path_Status_Code_Str_Array :=
     Dead_Path_Status_Code_Str_Array'(VCDetails.DPC_Not_Present      => "-",
                                      VCDetails.DPC_SDP_Not_Present  => "S",
                                      VCDetails.DPC_Unchecked        => "U",
                                      VCDetails.DPC_Live             => "L",
                                      VCDetails.DPC_Dead             => "D");

   VCSuccess             : Boolean;
   ThisVC                : Heap.Atom;
   NextVC                : Heap.Atom;
   VCNumber              : EStrings.T;
   TableLine             : EStrings.T;
   VCNamePrefix          : EStrings.T;
   VCNameTabulation      : Integer;
   VCPathStartTabulation : Integer;
   VCPathEndTabulation   : Integer;
   ThisVCName            : EStrings.T;
   ThisVCPathStart       : EStrings.T;
   ThisVCPathEnd         : EStrings.T;
   ThisVCEndType         : VCDetails.TerminalPointType;

   This_VC_State         : VCDetails.VC_State_T;
   This_DPC_State        : VCDetails.DPC_State_T;

   VCStat : XMLSummary.VCStatus;
   TmpErrorString : EStrings.T;

   NumUndischarged : Integer;
   NumFalse  : Integer;
   NumContra : Integer;
   NumUser   : Integer;

   Duplicated : Boolean;
   UnusedPos : Integer;
   NonDuplicatedErrors : Integer;

   --------------------------------------------------------------------------
   procedure AddPadding (Line    : in out EStrings.T;
                         Length  : in     Integer;
                         Padding : in     String)

   --# derives Line from *,
   --#                   Length,
   --#                   Padding;
   is
   begin -- AddPadding

      while EStrings.Get_Length (E_Str => Line) < Length loop
         EStrings.Append_String (E_Str => Line,
                                 Str   => Padding);
      end loop;

   end AddPadding;

   function NaturalToString (Number : in Natural) return String_10
   is
      TmpString : String_10;
   begin
      SPARK_IO.Put_Int_To_String (TmpString,
                                  Number,
                                  1,
                                  10);
      return TmpString;
   end NaturalToString;

begin --PrintVCReport
   NumUndischarged := 0;
   NumFalse := 0;
   NumContra := 0;
   NumUser := 0;
   -- Num_Of_Dead_Paths := 0;
   VCNamePrefix := VCHeap.GetVCNamePrefix;

   if CommandLine.Data.XML then
      -- Start the file
      XMLSummary.StartFile (VCFileName,
                            VCNamePrefix,
                            VCFileDateTime,
                            SIVFileDateTime,
                            PLGFileDateTime,
                            ReportFile);

      if VCError then
         XMLSummary.FileError (VCErrorString,
                               XMLSummary.VCGFile,
                               ReportFile);
      end if;

      --# assert True;

      if SIVError then
         XMLSummary.FileError (SIVErrorString,
                               XMLSummary.SIVFile,
                               ReportFile);
      end if;

      if PLGError then
         XMLSummary.FileError (PLGErrorString,
                               XMLSummary.PLGFile,
                               ReportFile);
      end if;

      --# assert True;
      if REVErrors.Errors then
         -- loop through the errors and generate a tag for each
         for CurrentError in Integer range ErrorsIndex'First .. REVErrors.LastError loop
            XMLSummary.FileError (REVErrors.ErrorList (CurrentError),
                                  XMLSummary.REVFile,
                                  ReportFile);
         end loop;

         -- Output a final message saying how many additional errors there were.
         if REVErrors.ExcessCount > 0 then
            TmpErrorString := EStrings.Copy_String (Str => "There were too many errors, ");
            EStrings.Append_String (E_Str => TmpErrorString,
                                    Str   => NaturalToString (REVErrors.ExcessCount));
            EStrings.Append_String (E_Str => TmpErrorString,
                                    Str   => " errors were not reported.");
            XMLSummary.FileError (TmpErrorString,
                                  XMLSummary.REVFile,
                                  ReportFile);
         end if;
      end if;


      --Only output detials if there were no errors for this proof object.
      if not (VCError or
                SIVError or
                  PLGError or
                    REVErrors.LastError > ErrorsIndex'First)
      then

         -- Now loop through the VCHeap and print one table line per VC
         ThisVC := VCHeap.FirstEntry;
         VCSuccess := True;

         while not Heap.IsNullPointer (ThisVC) and VCSuccess
         --#assert True;
         loop

            --# accept Flow, 10, This_DPC_State, "No XML output for DPC";

            VCHeap.Details (ThisVC,
                            ThisVCName,
                            ThisVCPathStart,
                            ThisVCPathEnd,
                            ThisVCEndType,
                            This_VC_State,
                            This_DPC_State);

            -- trim the VC name prefix from the VCName to get VC number
            -- as we only print the VC number in the table
            VCNumber := EStrings.Section
              (E_Str     => ThisVCName,
               Start_Pos => EStrings.Get_Length (E_Str => VCNamePrefix) + 2,
               Length    => EStrings.Get_Length (E_Str => ThisVCName) -
               (EStrings.Get_Length (E_Str => VCNamePrefix) + 1));

            --# assert True;

            -- Work out what the status of the vc is.
            if This_VC_State = VCDetails.VC_Proved_By_Examiner then
               VCStat := XMLSummary.VCG;
            elsif This_VC_State = VCDetails.VC_Proved_By_Inference or
               This_VC_State = VCDetails.VC_Proved_By_Contradiction or
               This_VC_State = VCDetails.VC_Proved_Using_User_Proof_Rules then
               VCStat := XMLSummary.SIV;
            elsif This_VC_State = VCDetails.VC_Proved_By_Checker then
               VCStat := XMLSummary.PLG;
            elsif This_VC_State = VCDetails.VC_Proved_By_Review then
               VCStat := XMLSummary.REV;
            elsif This_VC_State = VCDetails.VC_False then
               VCStat := XMLSummary.WRONG;
            else
               VCStat := XMLSummary.TODO;
            end if;

            --# assert True;

            if This_VC_State = VCDetails.VC_Proved_By_Contradiction then
               VCStat := XMLSummary.CONTRA;
            elsif This_VC_State = VCDetails.VC_Proved_Using_User_Proof_Rules then
               VCStat := XMLSummary.USER;
            end if;

            --# assert True;

            -- Generate the vc tag.
            XMLSummary.VC (VCNumber,
                           ThisVCPathStart,
                           XMLSummary.XStr (VCDetails.EndTypeImage (ThisVCEndType)),
                           ThisVCPathEnd,
                           VCStat,
                           ReportFile);

            VCHeap.Next (ThisVC,
                         VCSuccess,
                         NextVC);

            ThisVC := NextVC;
         end loop;
      end if;

      XMLSummary.EndFile (ReportFile);

   else

      -- If there any warnings or errors found in analysing the VCG,
      -- SIV or PLG file, record the filname of the errant file to a
      -- temporary warning and error file for reporting in the overall
      -- or short summary generated by the Total package.
      if VCError then
         SPARK_IO.Put_Integer (TempWarnErrorFile, 1, 4, 10);
         ELStrings.Put_Line (File  => TempWarnErrorFile,
                             E_Str => PathFormatter.Format (VCFileName));
      end if;

      --# assert True;
      if SIVError then
         SPARK_IO.Put_Integer (TempWarnErrorFile, 1, 4, 10);
         ELStrings.Put_Line (File  => TempWarnErrorFile,
                             E_Str => PathFormatter.Format (SIVFileName));
      end if;

      --# assert True;
      if PLGError then
         SPARK_IO.Put_Integer (TempWarnErrorFile, 1, 4, 10);
         ELStrings.Put_Line (File  => TempWarnErrorFile,
                             E_Str => PathFormatter.Format (PLGFileName));
      end if;

      --# assert True;
      if (SLGError or SLGFileMissing) then
         SPARK_IO.Put_Integer (TempWarnErrorFile, 1, 4, 10);
         ELStrings.Put_Line (File  => TempWarnErrorFile,
                             E_Str => PathFormatter.Format (SLGFileName));
      end if;

      --# assert True;
      if REVErrors.LastError /= ErrorsIndex'First then
         --  don't count any errors which are duplicated in review file
         NonDuplicatedErrors := REVErrors.LastError - 1;
         for I in Integer range (ErrorsIndex'First + 1) .. REVErrors.LastError loop
            --# accept F, 10, UnusedPos, "UnusedPos unused here";
            EStrings.Find_Sub_String (E_Str         => REVErrors.ErrorList (I),
                                      Search_String => "duplicated",
                                      String_Found  => Duplicated,
                                      String_Start  => UnusedPos);
            --# end accept;

            if Duplicated then
               NonDuplicatedErrors := NonDuplicatedErrors - 1;
            end if;
         end loop;

         --# assert True;
         --  if there are any (nonduplicated) errors in the file, print number and VC name
         if NonDuplicatedErrors >= 1 then
            if REVErrors.ExcessCount > 0 then
               SPARK_IO.Put_Line (ReportFile, "There were too many errors", 0);
            else
               SPARK_IO.Put_Integer (TempPRVerrFile, NonDuplicatedErrors, 4, 10);
            end if;
            ELStrings.Put_Line (File  => TempPRVerrFile,
                                E_Str => PathFormatter.Format (VCFileName));
         end if;
      end if;

      -- Only display the table if there were no errors for this proof object.
      -- Note than a missing SLG file is not treated as an error.
      -- Some organisations may deliberately delete SLG files, to
      -- save space.
      if not (VCError or
                SIVError or
                  SLGError or
                    PLGError or
                      REVErrors.LastError > ErrorsIndex'First)
      then

         SPARK_IO.New_Line (ReportFile, 1);
         TableLine := EStrings.Copy_String (Str => "VCs for ");
         EStrings.Append_Examiner_String (E_Str1 => TableLine,
                                          E_Str2 => VCNamePrefix);
         EStrings.Append_String (E_Str => TableLine,
                                 Str   => " :");
         EStrings.Put_Line (File  => ReportFile,
                            E_Str => TableLine);

         VCNameTabulation := VCHeap.GetLongestVCNameLength -
           EStrings.Get_Length (E_Str => VCNamePrefix);
         -- Give at least 5 columns (including a leading space)
         -- for the VC number, so subprograms with up to 9999
         -- VCs all come out looking the same.
         if VCNameTabulation < 5 then
            VCNameTabulation := 5;
         end if;

         VCPathStartTabulation := 3 + (VCNameTabulation +
                                         VCHeap.GetLongestVCStartLength);
         VCPathEndTabulation := 3 + (VCPathStartTabulation +
                                       VCHeap.GetLongestVCEndLength);

         -- Print the table header
         TableLine := EStrings.Copy_String (Str => " -");
         AddPadding (TableLine, VCPathEndTabulation + Table_Pad_Width, "-");

         EStrings.Put_Line (File  => ReportFile,
                            E_Str => TableLine);

         TableLine := EStrings.Copy_String (Str => "| # ");
         AddPadding (TableLine, VCNameTabulation, " ");
         EStrings.Append_String (E_Str => TableLine,
                                 Str   => " | From ");
         AddPadding (TableLine, VCPathStartTabulation, " ");
         EStrings.Append_String (E_Str => TableLine,
                                 Str   => " | To ");
         AddPadding (TableLine, VCPathEndTabulation, " ");

         EStrings.Append_String (E_Str => TableLine,
                                 Str   => "  | Proved By          | Dead Path | Status |");

         EStrings.Put_Line (File  => ReportFile,
                            E_Str => TableLine);

         TableLine := EStrings.Copy_String (Str => "|-");
         AddPadding (TableLine, VCPathEndTabulation + Table_Pad_Width, "-");

         EStrings.Put_Line (File  => ReportFile,
                            E_Str => TableLine);

         -- Now loop through the VCHeap and print one table line per VC
         ThisVC := VCHeap.FirstEntry;
         VCSuccess := True;

         while not Heap.IsNullPointer (ThisVC) and VCSuccess
         loop

            VCHeap.Details (ThisVC,
                            ThisVCName,
                            ThisVCPathStart,
                            ThisVCPathEnd,
                            ThisVCEndType,
                            This_VC_State,
                            This_DPC_State);

            -- trim the VC name prefix from the VCName to get VC number
            -- as we only print the VC number in the table

            VCNumber := EStrings.Section
              (E_Str     => ThisVCName,
               Start_Pos => EStrings.Get_Length (E_Str => VCNamePrefix) + 2,
               Length    => EStrings.Get_Length (E_Str => ThisVCName) -
               (EStrings.Get_Length (E_Str => VCNamePrefix) + 1));


            -- Start composing the table line for this VC
            TableLine := EStrings.Copy_String (Str => "| ");
            EStrings.Append_Examiner_String (E_Str1 => TableLine,
                                             E_Str2 => VCNumber);

            -- pad with spaces to longest VC number length
            AddPadding (TableLine, VCNameTabulation, " ");

            EStrings.Append_String (E_Str => TableLine,
                                    Str   => " | ");
            EStrings.Append_Examiner_String (E_Str1 => TableLine,
                                             E_Str2 => ThisVCPathStart);
            AddPadding (TableLine, VCPathStartTabulation, " ");

            EStrings.Append_String (E_Str => TableLine,
                                    Str   => " | ");

            EStrings.Append_String (E_Str => TableLine,
                                    Str   => VCDetails.EndTypeImage (ThisVCEndType));
            EStrings.Append_Examiner_String (E_Str1 => TableLine,
                                             E_Str2 => ThisVCPathEnd);
            AddPadding (TableLine, VCPathEndTabulation, " ");

            EStrings.Append_String (E_Str => TableLine,
                                    Str   => "  |");

            EStrings.Append_String (E_Str => TableLine,
                                    Str   => Proved_By_Str (This_VC_State));

            EStrings.Append_String (E_Str => TableLine,
                                    Str   => "|");

            EStrings.Append_String (E_Str => TableLine,
                                    Str   => Dead_Path_Str (This_DPC_State));

            EStrings.Append_String (E_Str => TableLine,
                                    Str   => "|   ");

            EStrings.Append_String (E_Str => TableLine,
                                    Str   => Proved_By_Status_Code (This_VC_State));

            EStrings.Append_String (E_Str => TableLine,
                                    Str   => Dead_Path_Status_Code (This_DPC_State));

            EStrings.Append_String (E_Str => TableLine,
                                    Str   => "   |");

            -- Now print the table line and get info for the next VC

            case This_VC_State is
               when VCDetails.VC_SIV_Not_Present | VCDetails.VC_Undischarged =>
                  NumUndischarged := NumUndischarged + 1;
               when VCDetails.VC_Proved_By_Contradiction =>
                  NumContra := NumContra + 1;
               when VCDetails.VC_Proved_Using_User_Proof_Rules =>
                  NumUser := NumUser + 1;
               when VCDetails.VC_False =>
                  NumFalse := NumFalse + 1;
               when others =>
                  null;
            end case;

            EStrings.Put_Line (File  => ReportFile,
                               E_Str => TableLine);

            VCHeap.Next (ThisVC,
                         VCSuccess,
                         NextVC);

            ThisVC := NextVC;

         end loop;

         --# assert True;
         if NumContra > 0 then
            SPARK_IO.Put_Integer (TempContraFile, NumContra, 4, 10);
            ELStrings.Put_Line (File  => TempContraFile,
                                E_Str => PathFormatter.Format (VCFileName));
         end if;

         --# assert True;
         if NumUser > 0 then
            SPARK_IO.Put_Integer (TempUserFile, NumUser, 4, 10);
            ELStrings.Put_Line (File  => TempUserFile,
                                E_Str => PathFormatter.Format (VCFileName));
         end if;

         --# assert True;
         if NumFalse > 0 then
            SPARK_IO.Put_Integer (TempFalseFile, NumFalse, 4, 10);
            ELStrings.Put_Line (File  => TempFalseFile,
                                E_Str => PathFormatter.Format (VCFileName));
         end if;

         --# assert True;
         -- if the subprogram contains any undischarged VCs add the name and number to TempFile
         if NumUndischarged > 0 then
            SPARK_IO.Put_Integer (TempFile, NumUndischarged, 4, 10);
            ELStrings.Put_Line (File  => TempFile,
                                E_Str => PathFormatter.Format (VCFileName));
         end if;

         -- Print the table footer if we are not doing XML.
         TableLine := EStrings.Copy_String (Str => " -");

         AddPadding (TableLine, VCPathEndTabulation + Table_Pad_Width, "-");

         EStrings.Put_Line (File  => ReportFile,
                            E_Str => TableLine);

         SPARK_IO.New_Line (ReportFile, 1);
      end if;
   end if;

   --# accept F, 33, UnusedPos, "UnusedPos unused here";
end PrintVCReport;
